/**
 * Enhanced Patient Billing Frontend
 * Connects to CodeIgniter backend API for real database operations
 */

class PatientBillingSystem {
    constructor() {
        this.apiBaseUrl = billingControllerURL; // Use billingControllerURL from billing_query.php
        this.currentPatient = null;
        this.currentBillingData = null;
        this.init();
    }

    init() {
        this.bindEvents();
        this.setupPrintStyles();
    }

    bindEvents() {
        // Search form
        document.getElementById('patientSearchForm').addEventListener('submit', (e) => {
            e.preventDefault();
            this.searchPatient();
        });

        // Clear search
        document.getElementById('clearSearch').addEventListener('click', () => {
            this.clearSearch();
        });

        // Print button
        document.getElementById('printBill').addEventListener('click', () => {
            this.printBill();
        });

        // Export button
        document.getElementById('exportBill').addEventListener('click', () => {
            this.exportBill();
        });

        // Real-time search as user types
        ['patient_id', 'patient_name', 'mobile_no'].forEach(fieldId => {
            document.getElementById(fieldId).addEventListener('input', 
                this.debounce(() => this.performLiveSearch(), 500)
            );
        });
    }

    // Debounce function for live search
    debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    async performLiveSearch() {
        const searchTerms = this.getSearchTerms();
        if (!searchTerms.trim()) {
            this.hideSearchSuggestions();
            return;
        }

        try {
            const patients = await this.searchPatientsAPI(searchTerms);
            this.showSearchSuggestions(patients);
        } catch (error) {
            console.error('Live search error:', error);
        }
    }

    getSearchTerms() {
        const patientId = document.getElementById('patient_id').value.trim();
        const patientName = document.getElementById('patient_name').value.trim();
        const mobileNo = document.getElementById('mobile_no').value.trim();
        
        return patientId || patientName || mobileNo;
    }

    showSearchSuggestions(patients) {
        // Remove existing suggestions
        this.hideSearchSuggestions();

        if (patients.length === 0) return;

        const suggestionsDiv = document.createElement('div');
        suggestionsDiv.id = 'searchSuggestions';
        suggestionsDiv.className = 'card mt-2';
        suggestionsDiv.style.position = 'absolute';
        suggestionsDiv.style.zIndex = '1000';
        suggestionsDiv.style.width = '100%';

        const suggestionsBody = document.createElement('div');
        suggestionsBody.className = 'card-body p-2';

        patients.slice(0, 5).forEach(patient => {
            const suggestionItem = document.createElement('div');
            suggestionItem.className = 'p-2 border-bottom cursor-pointer hover-bg-light';
            suggestionItem.style.cursor = 'pointer';
            suggestionItem.innerHTML = `
                <strong>${patient.patient_name}</strong> (${patient.patient_id})<br>
                <small class="text-muted">${patient.mobileno || 'No mobile'} | Age: ${patient.age}</small>
            `;
            
            suggestionItem.addEventListener('click', () => {
                this.selectPatientFromSuggestion(patient);
            });

            suggestionsBody.appendChild(suggestionItem);
        });

        suggestionsDiv.appendChild(suggestionsBody);
        
        const searchCard = document.querySelector('.card');
        searchCard.style.position = 'relative';
        searchCard.appendChild(suggestionsDiv);
    }

    hideSearchSuggestions() {
        const existing = document.getElementById('searchSuggestions');
        if (existing) {
            existing.remove();
        }
    }

    selectPatientFromSuggestion(patient) {
        // Fill form fields
        document.getElementById('patient_id').value = patient.patient_id;
        document.getElementById('patient_name').value = patient.patient_name;
        document.getElementById('mobile_no').value = patient.mobileno || '';
        
        // Hide suggestions and load patient data
        this.hideSearchSuggestions();
        this.currentPatient = patient;
        this.display	SetPatientInfo(patient);
        this.loadBillingData(patient.id);
    }

    async searchPatient() {
        const searchTerm = this.getSearchTerms();
        
        if (!searchTerm) {
            this.showAlert(langTranslations.no_patient_selected, 'warning');
            return;
        }

        try {
            this.showLoading(true);
            const patients = await this.searchPatientsAPI(searchTerm);
            
            if (patients.length === 0) {
                this.showAlert(langTranslations.patient_not_found, 'info');
                this.hideAllSections();
                return;
            }

            if (patients.length === 1) {
                // Auto-select if only one result
                this.currentPatient = patients[0];
                this.displayPatientInfo(patients[0]);
                await this.loadBillingData(patients[0].id);
            } else {
                // Show selection modal if multiple results
                this.showPatientSelectionModal(patients);
            }

        } catch (error) {
            this.showAlert(`${langTranslations.error_occurred}: ${error.message}`, 'danger');
        } finally {
            this.showLoading(false);
            this.hideSearchSuggestions();
        }
    }

    async searchPatientsAPI(searchTerm) {
        const response = await fetch(`${this.apiBaseUrl}search_patients?term=${encodeURIComponent(searchTerm)}`);
        if (response.status === 401 || response.status === 403) {
            this.showAlert('Session expired or access denied. Please log in.', 'danger');
            return [];
        }
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || langTranslations.error_occurred);
        }
        
        return data.data;
    }

    showPatientSelectionModal(patients) {
        const modalHtml = `
            <div class="modal fade" id="patientSelectionModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">${langTranslations.select_patient}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="table-responsive">
                                <table class="table table-hover">
                                    <thead>
                                        <tr>
                                            <th>${langTranslations.patient_id}</th>
                                            <th>${langTranslations.patient_name}</th>
                                            <th>${langTranslations.age}</th>
                                            <th>${langTranslations.mobile}</th>
                                            <th>${langTranslations.actions}</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        ${patients.map(patient => `
                                            <tr>
                                                <td>${patient.patient_id}</td>
                                                <td>${patient.patient_name}</td>
                                                <td>${patient.age}</td>
                                                <td>${patient.mobileno || '-'}</td>
                                                <td>
                                                    <button class="btn btn-primary btn-sm" 
                                                            onclick="billingSystem.selectPatient(${patient.id})">
                                                        ${langTranslations.select_patient}
                                                    </button>
                                                </td>
                                            </tr>
                                        `).join('')}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal
        const existingModal = document.getElementById('patientSelectionModal');
        if (existingModal) {
            existingModal.remove();
        }

        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('patientSelectionModal'));
        modal.show();

        // Store patients data for selection
        this.searchResults = patients;
    }

    async selectPatient(patientId) {
        const patient = this.searchResults.find(p => p.id == patientId);
        if (patient) {
            this.currentPatient = patient;
            this.displayPatientInfo(patient);
            await this.loadBillingData(patient.id);
            
            // Close modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('patientSelectionModal'));
            modal.hide();
        }
    }

    displayPatientInfo(patient) {
        document.getElementById('display_patient_name').textContent = patient.patient_name;
        document.getElementById('display_patient_id').textContent = getPatient(patient_id);
        document.getElementById('display_age').textContent = patient.age;
        document.getElementById('display_gender').textContent = patient.gender;
        document.getElementById('display_mobile').textContent = patient.mobileno || '-';
        document.getElementById('display_email').textContent = patient.email || '-';
        document.getElementById('display_blood_group').textContent = patient.blood_group || '-';
        document.getElementById('display_address').textContent = patient.address || '-';

        document.getElementById('patientInfoSection').style.display = 'block';
    }

    async loadBillingData(patientId) {
        try {
            this.showLoading(true);
            
            const response = await fetch(`${this.apiBaseUrl}get_billing?patient_id=${patientId}`);
            if (response.status === 401 || response.status === 403) {
                this.showAlert('Session expired or access denied. Please log in.', 'danger');
                return;
            }
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || langTranslations.error_occurred);
            }

            this.currentBillingData = data.data;
            this.updateSummaryCards(data.data.summary);
            this.updateBillingTable(data.data.bills);
            
            document.getElementById('billingSummarySection').style.display = 'block';
            document.getElementById('billingDetailsSection').style.display = 'block';

        } catch (error) {
            this.showAlert(`${langTranslations.error_occurred}: ${error.message}`, 'danger');
        } finally {
            this.showLoading(false);
        }
    }

    updateSummaryCards(summary) {
        // Update individual module cards
        Object.keys(summary.modules).forEach(module => {
            const data = summary.modules[module];
            const totalElement = document.getElementById(`${module}_total`);
            const statusElement = document.getElementById(`${module}_status`);
            
            if (totalElement) {
                totalElement.textContent = `Ksh.${data.total.toFixed(2)}`;
            }
            if (statusElement) {
                statusElement.textContent = data.status;
            }
        });

        // Update grand totals
        document.getElementById('grand_total').textContent = `Ksh.${summary.total_amount.toFixed(2)}`;
        document.getElementById('overall_status').textContent = this.getOverallStatus(summary);
    }

    getOverallStatus(summary) {
        if (summary.total_amount === 0) return langTranslations.no_bills;
        if (summary.total_pending === 0) return langTranslations.paid;
        if (summary.total_paid === 0) return langTranslations.pending;
        return langTranslations.partial;
    }

    updateBillingTable(bills) {
        const tbody = document.getElementById('billingTableBody');
        tbody.innerHTML = '';

        let totalAmount = 0, totalPaid = 0, totalPending = 0;

        bills.forEach(bill => {
            const row = document.createElement('tr');
            
            row.innerHTML = `
                <td>${new Date(bill.date).toLocaleDateString()}</td>
                <td><span class="badge bg-primary">${bill.module}</span></td>
                <td>${bill.bill_no}</td>
                <td>${bill.description}</td>
                <td>Ksh.${parseFloat(bill.amount).toFixed(2)}</td>
                <td>Ksh.${parseFloat(bill.paid).toFixed(2)}</td>
                <td>Ksh.${parseFloat(bill.pending).toFixed(2)}</td>
                <td>${this.getStatusBadge(bill.status)}</td>
                <td class="no-print">
                    <div class="btn-group btn-group-sm" role="group">
                        <button class="btn btn-outline-primary" onclick="billingSystem.viewBillDetails('${bill.bill_no}')" title="${langTranslations.view_details}">
                            <i class="fas fa-eye"></i>
                        </button>
                        ${parseFloat(bill.pending) > 0 ? `
                            <button class="btn btn-outline-success" onclick="billingSystem.makePayment('${bill.bill_no}', ${bill.pending})" title="${langTranslations.make_payment}">
                                <i class="fas fa-credit-card"></i>
                            </button>
                        ` : ''}
                    </div>
                </td>
            `;

            tbody.appendChild(row);

            totalAmount += parseFloat(bill.amount);
            totalPaid += parseFloat(bill.paid);
            totalPending += parseFloat(bill.pending);
        });

        // Update totals
        document.getElementById('total_amount').textContent = `Ksh.${totalAmount.toFixed(2)}`;
        document.getElementById('total_paid').textContent = `Ksh.${totalPaid.toFixed(2)}`;
        document.getElementById('total_pending').textContent = `Ksh.${totalPending.toFixed(2)}`;
    }

    getStatusBadge(status) {
        const badges = {
            'paid': `<span class="badge bg-success">${langTranslations.paid}</span>`,
            'pending': `<span class="badge bg-danger">${langTranslations.pending}</span>`,
            'partial': `<span class="badge bg-warning">${langTranslations.partial}</span>`
        };
        return badges[status] || '<span class="badge bg-secondary">Unknown</span>';
    }

    async viewBillDetails(billNo) {
        try {
            this.showLoading(true);
            
            const response = await fetch(`${this.apiBaseUrl}get_bill_details?bill_no=${encodeURIComponent(billNo)}`);
            if (response.status === 401 || response.status === 403) {
                this.showAlert('Session expired or access denied. Please log in.', 'danger');
                return;
            }
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || langTranslations.error_occurred);
            }

            this.showBillDetailsModal(billNo, data.data);

        } catch (error) {
            this.showAlert(`${langTranslations.error_occurred}: ${error.message}`, 'danger');
        } finally {
            this.showLoading(false);
        }
    }

    showBillDetailsModal(billNo, details) {
        const modalHtml = `
            <div class="modal fade" id="billDetailsModal" tabindex="-1">
                <div class="modal-dialog modal-lg">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">${langTranslations.detailed_billing_information} - ${billNo}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <h6>Patient Information</h6>
                                    <p><strong>${langTranslations.name}:</strong> ${details.patient_name || 'N/A'}</p>
                                    <p><strong>${langTranslations.date}:</strong> ${new Date(details.date || details.created_at).toLocaleDateString()}</p>
                                </div>
                                <div class="col-md-6">
                                    <h6>Bill Information</h6>
                                    <p><strong>${langTranslations.amount}:</strong> Ksh.${parseFloat(details.total_amount || details.amount || 0).toFixed(2)}</p>
                                    <p><strong>${langTranslations.status}:</strong> ${details.payment_status || 'N/A'}</p>
                                </div>
                            </div>
                            <hr>
                            <div class="row">
                                <div class="col-12">
                                    <h6>Additional Details</h6>
                                    <pre class="bg-light p-3 rounded">${JSON.stringify(details, null, 2)}</pre>
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">${langTranslations.close}</button>
                            <button type="button" class="btn btn-primary" onclick="billingSystem.printBillDetails('${billNo}')">
                                <i class="fas fa-print me-1"></i>${langTranslations.print}
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal
        const existingModal = document.getElementById('billDetailsModal');
        if (existingModal) {
            existingModal.remove();
        }

        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('billDetailsModal'));
        modal.show();
    }

    async makePayment(billNo, pendingAmount) {
        const modalHtml = `
            <div class="modal fade" id="paymentModal" tabindex="-1">
                <div class="modal-dialog">
                    <div class="modal-content">
                        <div class="modal-header">
                            <h5 class="modal-title">${langTranslations.process_payment} - ${billNo}</h5>
                            <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                        </div>
                        <div class="modal-body">
                            <form id="paymentForm">
                                <div class="mb-3">
                                    <label for="payment_amount" class="form-label">${langTranslations.payment_amount}</label>
                                    <input type="number" class="form-control" id="payment_amount" 
                                           value="${pendingAmount}" max="${pendingAmount}" min="0.01" step="0.01" required>
                                    <div class="form-text">Pending Amount: Ksh.${pendingAmount.toFixed(2)}</div>
                                </div>
                                <div class="mb-3">
                                    <label for="payment_mode" class="form-label">${langTranslations.payment_mode}</label>
                                    <select class="form-select" id="payment_mode" required>
                                        <option value="Cash">${langTranslations.cash}</option>
                                        <option value="Credit Card">${langTranslations.credit_card}</option>
                                        <option value="Debit Card">${langTranslations.debit_card}</option>
                                        <option value="Bank Transfer">${langTranslations.bank_transfer}</option>
                                        <option value="Check">${langTranslations.check}</option>
                                        <option value="Mobile Money">${langTranslations.mobile_money}</option>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="payment_notes" class="form-label">${langTranslations.notes} (Optional)</label>
                                    <textarea class="form-control" id="payment_notes" rows="3"></textarea>
                                </div>
                            </form>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">${langTranslations.cancel}</button>
                            <button type="button" class="btn btn-success" onclick="billingSystem.processPayment('${billNo}')">
                                <i class="fas fa-credit-card me-1"></i>${langTranslations.process_payment}
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;

        // Remove existing modal
        const existingModal = document.getElementById('paymentModal');
        if (existingModal) {
            existingModal.remove();
        }

        // Add modal to body
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        
        // Show modal
        const modal = new bootstrap.Modal(document.getElementById('paymentModal'));
        modal.show();
    }

    async processPayment(billNo) {
        try {
            const amount = parseFloat(document.getElementById('payment_amount').value);
            const paymentMode = document.getElementById('payment_mode').value;
            const notes = document.getElementById('payment_notes').value;

            if (amount <= 0) {
                this.showAlert(`${langTranslations.payment_amount} must be greater than 0`, 'warning');
                return;
            }

            this.showLoading(true);

            const response = await fetch(`${this.apiBaseUrl}process_payment`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    bill_no: billNo,
                    amount: amount,
                    payment_mode: paymentMode,
                    notes: notes,
                    [langTranslations.csrf_token_name]: langTranslations.csrf_token_value
                })
            });

            const data = await response.json();

            if (!data.success) {
                throw new Error(data.error || langTranslations.error_occurred);
            }

            // Close payment modal
            const modal = bootstrap.Modal.getInstance(document.getElementById('paymentModal'));
            modal.hide();

            // Show success message
            this.showAlert(langTranslations.payment_successful, 'success');

            // Reload billing data
            if (this.currentPatient) {
                await this.loadBillingData(this.currentPatient.id);
            }

            // Refresh CSRF token
            await this.refreshCsrfToken();

        } catch (error) {
            this.showAlert(`${langTranslations.error_occurred}: ${error.message}`, 'danger');
        } finally {
            this.showLoading(false);
        }
    }

    async refreshCsrfToken() {
        try {
            const response = await fetch(`${this.apiBaseUrl}get_csrf_token`);
            const data = await response.json();
            if (data.success) {
                langTranslations.csrf_token_value = data.csrf_token;
            }
        } catch (error) {
            console.error('Failed to refresh CSRF token:', error);
        }
    }

    clearSearch() {
        document.getElementById('patientSearchForm').reset();
        this.hideAllSections();
        this.hideSearchSuggestions();
        this.currentPatient = null;
        this.currentBillingData = null;
    }

    hideAllSections() {
        document.getElementById('patientInfoSection').style.display = 'none';
        document.getElementById('billingSummarySection').style.display = 'none';
        document.getElementById('billingDetailsSection').style.display = 'none';
    }

    printBill() {
        if (!this.currentPatient) {
            this.showAlert(langTranslations.no_patient_selected, 'warning');
            return;
        }

        this.preparePrintContent();
        window.print();
    }

    preparePrintContent() {
        const patientInfo = `
            <div class="row mb-4">
                <div class="col-6">
                    <h5>Patient Information</h5>
                    <p><strong>${langTranslations.name}:</strong> ${this.currentPatient.patient_name}</p>
                    <p><strong>${langTranslations.patient_id}:</strong> ${this.currentPatient.patient_id}</p>
                    <p><strong>${langTranslations.age}:</strong> ${this.currentPatient.age}</p>
                    <p><strong>${langTranslations.gender}:</strong> ${this.currentPatient.gender}</p>
                </div>
                <div class="col-6">
                    <p><strong>${langTranslations.mobile}:</strong> ${this.currentPatient.mobileno || '-'}</p>
                    <p><strong>${langTranslations.email}:</strong> ${this.currentPatient.email || '-'}</p>
                    <p><strong>${langTranslations.blood_group}:</strong> ${this.currentPatient.blood_group || '-'}</p>
                    <p><strong>${langTranslations.date}:</strong> ${new Date().toLocaleDateString()}</p>
                </div>
            </div>
        `;

        const billingTable = document.getElementById('billingTable').cloneNode(true);
        billingTable.querySelectorAll('.no-print').forEach(el => el.remove());

        document.getElementById('printPatientInfo').innerHTML = patientInfo;
        document.getElementById('printBillingDetails').innerHTML = '';
        document.getElementById('printBillingDetails').appendChild(billingTable);
    }

    exportBill() {
        if (!this.currentPatient || !this.currentBillingData) {
            this.showAlert('No billing data available for export', 'warning');
            return;
        }

        window.location.href = `${this.apiBaseUrl}export_csv?patient_id=${this.currentPatient.id}`;
    }

    printBillDetails(billNo) {
        const printWindow = window.open('', '_blank');
        const billDetailsContent = document.getElementById('billDetailsModal').querySelector('.modal-body').innerHTML;
        
        printWindow.document.write(`
            <!DOCTYPE html>
            <html>
            <head>
                <title>Bill Details - ${billNo}</title>
                <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/5.3.0/css/bootstrap.min.css" rel="stylesheet">
                <style>
                    body { padding: 20px; }
                    @media print {
                        .btn { display: none; }
                    }
                </style>
            </head>
            <body>
                <div class="container">
                    <h2 class="text-center mb-4">HEALTHCARE FACILITY</h2>
                    <h4 class="text-center mb-4">Bill Details - ${billNo}</h4>
                    <hr>
                    ${billDetailsContent}
                </div>
                <script>
                    window.onload = function() {
                        window.print();
                        window.onafterprint = function() {
                            window.close();
                        }
                    }
                </script>
            </body>
            </html>
        `);
        printWindow.document.close();
    }

    setupPrintStyles() {
        const printStyles = `
            @media print {
                .no-print { display: none !important; }
                .print-view { display: block !important; }
                .container-fluid { max-width: 100% !important; }
                body { background: white !important; }
                .card { border: 1px solid #000 !important; box-shadow: none !important; }
                .table th { background-color: #f8f9fa !important; color: #000 !important; }
                .billing-header { background: none !important; color: #000 !important; }
            }
        `;
        
        const styleSheet = document.createElement('style');
        styleSheet.textContent = printStyles;
        document.head.appendChild(styleSheet);
    }

    showLoading(show) {
        let loader = document.getElementById('loadingOverlay');
        
        if (show) {
            if (!loader) {
                loader = document.createElement('div');
                loader.id = 'loadingOverlay';
                loader.className = 'position-fixed top-0 start-0 w-100 h-100 d-flex justify-content-center align-items-center';
                loader.style.backgroundColor = 'rgba(0,0,0,0.5)';
                loader.style.zIndex = '9999';
                loader.innerHTML = `
                    <div class="bg-white p-4 rounded shadow">
                        <div class="d-flex align-items-center">
                            <div class="spinner-border text-primary me-3" role="status"></div>
                            <span>${langTranslations.loading}</span>
                        </div>
                    </div>
                `;
                document.body.appendChild(loader);
            }
            loader.style.display = 'flex';
        } else if (loader) {
            loader.style.display = 'none';
        }
    }

    showAlert(message, type = 'info') {
        const existingAlerts = document.querySelectorAll('.dynamic-alert');
        existingAlerts.forEach(alert => alert.remove());

        const alertDiv = document.createElement('div');
        alertDiv.className = `alert alert-${type} alert-dismissible fade show dynamic-alert mt-3`;
        alertDiv.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;

        const searchCard = document.querySelector('.card');
        searchCard.parentNode.insertBefore(alertDiv, searchCard.nextSibling);

        setTimeout(() => {
            if (alertDiv.parentNode) {
                alertDiv.remove();
            }
        }, 5000);
    }
}

// Initialize the billing system when the page loads
let billingSystem;
document.addEventListener('DOMContentLoaded', function() {
    billingSystem = new PatientBillingSystem();
});

// Export for global access
window.billingSystem = billingSystem;