<?php
// Include SecuGen WebAPI scripts at the top
?>
<script src="<?php echo base_url('WebAPI-JavaScript/secugen.js'); ?>"></script>

<div class="content-wrapper">
    <section class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-heartbeat"></i> Patient Triage with Department Routing & SecuGen Verification
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" id="addTriageBtn" class="btn btn-primary btn-sm">
                                <i class="fa fa-plus"></i> New Triage Assessment
                            </button>
                            <button type="button" id="departmentStatsBtn" class="btn btn-info btn-sm">
                                <i class="fa fa-bar-chart"></i> Department Stats
                            </button>
                        </div>
                    </div>

                    <ul class="nav nav-tabs navlistscroll">
                        <li class="active"><a href="#tab_patient_search" data-toggle="tab">
                            <i class="fa fa-search"></i> Patient Search & Verification
                        </a></li>
                        <li><a href="#tab_triage_list" data-toggle="tab">
                            <i class="fa fa-list"></i> Triage Records
                        </a></li>
                       
                    </ul>

                    <div class="tab-content">
                        <!-- Patient Search & Verification Tab -->
                        <div class="tab-pane active" id="tab_patient_search">
                            <div class="box-body">
                                <!-- Patient Search Section -->
                                <div class="row" style="margin-top: 20px;">
                                    <div class="col-md-10">
                                        <div class="form-group">
                                            <label>Search Patient for Triage Assessment</label>
                                            <select class="form-control" id="triage_patient_search" name="patient_search" style="width: 100%;">
                                                <option value="">Loading patients...</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-2">
                                        <label>&nbsp;</label><br>
                                        <button type="button" class="btn btn-success btn-block" onclick="$('#addPatientModal').modal('show')">
                                            <i class="fa fa-plus"></i> Add Patient
                                        </button>
                                    </div>
                                </div>

                                <!-- Patient Details Section -->
                                <div id="patientDetailsSection" style="display:none; margin-top: 20px;">
                                    <div class="panel panel-default">
                                        <div class="panel-heading">
                                            <h4 class="panel-title">
                                                <i class="fa fa-user"></i> Patient Information - Triage Assessment
                                                <span class="pull-right" id="verificationStatus"></span>
                                            </h4>
                                        </div>
                                        <div class="panel-body">
                                            <!-- Patient Basic Information -->
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="row">
                                                        <div class="col-md-6">
                                                            <p><strong>Name:</strong> <span id="patientName"></span></p>
                                                            <p><strong>Age:</strong> <span id="patientAge"></span></p>
                                                            <p><strong>Gender:</strong> <span id="patientGender"></span></p>
                                                        </div>
                                                        <div class="col-md-6">
                                                            <p><strong>Phone:</strong> <span id="patientPhone"></span></p>
                                                            <p><strong>Blood Group:</strong> <span id="patientBloodGroup"></span></p>
                                                            <p><strong>Allergies:</strong> <span id="patientAllergies"></span></p>
                                                        </div>
                                                    </div>
                                                </div>
                                                <div class="col-md-3">
                                                    <div class="text-center">
                                                        <img id="patientPhoto" src="<?php echo base_url('uploads/patient_images/no_image.png'); ?>" alt="Patient Photo" class="img-responsive" style="max-height: 120px; border: 1px solid #ddd; border-radius: 4px;">
                                                    </div>
                                                </div>
                                                <div class="col-md-3">
                                                    <!-- Current Patient Status -->
                                                    <div class="panel panel-warning">
                                                        <div class="panel-heading" style="padding: 5px 10px;">
                                                            <h6 class="panel-title">Current Status</h6>
                                                        </div>
                                                        <div class="panel-body" style="padding: 10px;">
                                                            <div id="currentPatientStatus">
                                                                <p class="text-muted">Select patient to view status</p>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Enhanced SecuGen Fingerprint Verification Section -->
                                            <div class="row" style="margin-top: 20px;">
                                                <div class="col-md-6">
                                                    <div class="panel panel-info">
                                                        <div class="panel-heading">
                                                            <h5 class="panel-title">
                                                                <i class="fa fa-fingerprint"></i> SecuGen Biometric Verification for Triage
                                                            </h5>
                                                        </div>
                                                        <div class="panel-body">
                                                            <!-- Service Status -->
                                                            <div id="secugen_service_status" class="mb-2"></div>
                                                            
                                                            <!-- Fingerprint Display -->
                                                            <div class="row mb-3">
                                                                <div class="col-md-6">
                                                                    <div class="panel panel-default">
                                                                        <div class="panel-heading" style="padding: 5px 10px;">
                                                                            <h6 class="panel-title">Stored Template</h6>
                                                                        </div>
                                                                        <div class="panel-body text-center" style="padding: 10px;">
                                                                            <img id="storedFingerprintImage" src="./Images/PlaceFinger.bmp" alt="Stored Fingerprint" 
                                                                                 style="height: 100px; width: 80px; border: 1px solid #ddd; border-radius: 4px; display: block; margin: 0 auto;">
                                                                            <div id="storedTemplateInfo" style="margin-top: 8px; font-size: 11px;">
                                                                                <div id="storedTemplateDetails" class="text-muted">No template loaded</div>
                                                                            </div>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                                <div class="col-md-6">
                                                                    <div class="panel panel-default">
                                                                        <div class="panel-heading" style="padding: 5px 10px;">
                                                                            <h6 class="panel-title">Live Capture</h6>
                                                                        </div>
                                                                        <div class="panel-body text-center" style="padding: 10px;">
                                                                            <img id="liveFingerprintImage" src="./Images/PlaceFinger.bmp" alt="Live Fingerprint" 
                                                                                 style="height: 100px; width: 80px; border: 1px solid #ddd; border-radius: 4px; display: block; margin: 0 auto;">
                                                                            <div id="liveTemplateInfo" style="margin-top: 8px; font-size: 11px;">
                                                                                <div id="liveTemplateDetails" class="text-muted">Ready for capture</div>
                                                                            </div>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                            </div>
                                                            
                                                            <!-- Verification Status -->
                                                            <div id="fingerprintStatus" class="text-center mb-3">
                                                                <div class="alert alert-info">
                                                                    <i class="fa fa-info-circle"></i> Ready for SecuGen biometric verification - Triage Module
                                                                </div>
                                                            </div>
                                                            
                                                            <!-- Verification Controls -->
                                                            <div class="text-center">
                                                                <div class="btn-group" style="margin-bottom: 10px;">
                                                                    <button type="button" id="manualVerifyBtn" class="btn btn-info">
                                                                        <i class="fa fa-hand-pointer-o"></i> Print Capture
                                                                    </button>
                                                                    <button type="button" id="verifyIdentity" class="btn btn-success" disabled>
                                                                        <i class="fa fa-shield"></i> Print Verify
                                                                    </button>
                                                                </div>
                                                                <br>
                                                                <div>
                                                                    <button type="button" id="skipVerificationBtn" class="btn btn-warning btn-sm">
                                                                        <i class="fa fa-user-shield"></i> Admin Override
                                                                    </button>
                                                                    <button type="button" id="clearCaptureData" class="btn btn-default btn-sm">
                                                                        <i class="fa fa-refresh"></i> Clear Data
                                                                    </button>
                                                                </div>
                                                            </div>
                                                            
                                                            <!-- Hidden Template Storage -->
                                                            <input type="hidden" id="patientStoredTemplate" value="">
                                                            <input type="hidden" id="liveScannedTemplate" value="">
                                                            <input type="hidden" id="lastMatchScore" value="0">
                                                            <input type="hidden" id="verificationQuality" value="">
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <!-- Patient Journey & History Panel -->
                                                <div class="col-md-6">
                                                    <div class="panel panel-success">
                                                        <div class="panel-heading">
                                                            <h5 class="panel-title">
                                                                <i class="fa fa-map-signs"></i> Patient Journey
                                                            </h5>
                                                        </div>
                                                        <div class="panel-body">
                                                            <div id="patientJourneyDisplay" style="max-height: 200px; overflow-y: auto;">
                                                                <p class="text-muted">Select a patient to view journey</p>
                                                            </div>
                                                            <div class="text-center" style="margin-top: 10px;">
                                                                <button type="button" id="refreshJourneyBtn" class="btn btn-info btn-xs" disabled>
                                                                    <i class="fa fa-refresh"></i> Refresh Journey
                                                                </button>
                                                            </div>
                                                        </div>
                                                    </div>
                                                    
                                                    <!-- Recent Triage Records -->
                                                    <div class="panel panel-primary">
                                                        <div class="panel-heading">
                                                            <h5 class="panel-title">
                                                                <i class="fa fa-history"></i> Recent Triage Records
                                                            </h5>
                                                        </div>
                                                        <div class="panel-body">
                                                            <div id="recentTriageRecords">
                                                                <p class="text-muted">Select a patient to view triage history</p>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>

                                            <!-- Start Triage Button -->
                                            <div class="text-center" style="margin-top: 20px;">
                                                <button type="button" id="startTriageBtn" class="btn btn-primary btn-lg" disabled>
                                                    <i class="fa fa-heartbeat"></i> Start Triage Assessment
                                                </button>
                                                <div id="verificationMessage" style="margin-top: 10px;"></div>
                                            </div>

                                            <!-- Hidden fields -->
                                            <input type="hidden" id="selectedPatientId" value="">
                                            <input type="hidden" id="verificationPassed" value="0">
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Triage List Tab -->
                        <div class="tab-pane" id="tab_triage_list">
                            <div class="box-body">
                                <div class="download_label">Triage Records with Priority & Department Assignment</div>
                                <div class="table-responsive">
                                    <table class="table table-striped table-bordered table-hover" id="triageListTable">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Date/Time</th>
                                                <th>Patient</th>
                                                <th>Priority</th>
                                                <th>Vitals Summary</th>
                                                <th>Department</th>
                                                <th>Status</th>
                                                <th>Staff</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Department Routing Tab -->
                        <div class="tab-pane" id="tab_department_routing">
                            <div class="box-body">
                                <div class="row">
                                    <div class="col-md-8">
                                        <div class="download_label">Department Routing & Patient Flow</div>
                                        <div class="table-responsive">
                                            <table class="table table-striped table-bordered table-hover" id="departmentRoutingTable">
                                                <thead>
                                                    <tr>
                                                        <th>Patient</th>
                                                        <th>From</th>
                                                        <th>To Department</th>
                                                        <th>Priority</th>
                                                        <th>Routed At</th>
                                                        <th>Status</th>
                                                        <th>Actions</th>
                                                    </tr>
                                                </thead>
                                                <tbody></tbody>
                                            </table>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <!-- Department Queue Summary -->
                                        <div class="panel panel-info">
                                            <div class="panel-heading">
                                                <h5 class="panel-title">
                                                    <i class="fa fa-dashboard"></i> Department Queue Status
                                                </h5>
                                            </div>
                                            <div class="panel-body">
                                                <div id="departmentQueueStatus">
                                                    <p class="text-muted">Loading department status...</p>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Quick Route Panel -->
                                        <div class="panel panel-success">
                                            <div class="panel-heading">
                                                <h5 class="panel-title">
                                                    <i class="fa fa-random"></i> Quick Route Patient
                                                </h5>
                                            </div>
                                            <div class="panel-body">
                                                <form id="quickRouteForm">
                                                    <div class="form-group">
                                                        <label>Patient</label>
                                                        <select class="form-control" id="quickRoutePatient">
                                                            <option value="">Select Patient</option>
                                                        </select>
                                                    </div>
                                                    <div class="form-group">
                                                        <label>Department</label>
                                                        <select class="form-control" id="quickRouteDepartment" required>
                                                            <option value="">Select Department</option>
                                                            <?php foreach($departments as $code => $name): ?>
                                                                <option value="<?php echo $code; ?>"><?php echo $name; ?></option>
                                                            <?php endforeach; ?>
                                                        </select>
                                                    </div>
                                                    <div class="form-group">
                                                        <label>Priority</label>
                                                        <select class="form-control" id="quickRoutePriority">
                                                            <option value="normal">Normal</option>
                                                            <option value="high">High</option>
                                                            <option value="urgent">Urgent</option>
                                                        </select>
                                                    </div>
                                                    <button type="submit" class="btn btn-primary btn-block">
                                                        <i class="fa fa-send"></i> Route Patient
                                                    </button>
                                                </form>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Patient Journey Tab -->
                        <div class="tab-pane" id="tab_patient_journey">
                            <div class="box-body">
                                <div class="row">
                                    <div class="col-md-12">
                                        <div class="download_label">Patient Journey Tracking</div>
                                        <div class="form-group">
                                            <label>Search Patient Journey</label>
                                            <select class="form-control" id="journeyPatientSearch" style="width: 100%;">
                                                <option value="">Select Patient to Track Journey</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                
                                <div id="patientJourneyTimeline" style="display: none;">
                                    <div class="panel panel-primary">
                                        <div class="panel-heading">
                                            <h4 class="panel-title">
                                                <i class="fa fa-user"></i> Patient Journey Timeline
                                                <span class="pull-right" id="journeyPatientInfo"></span>
                                            </h4>
                                        </div>
                                        <div class="panel-body">
                                            <div id="journeyTimelineDisplay">
                                                <!-- Timeline will be populated here -->
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Verification Logs Tab -->
                        <div class="tab-pane" id="tab_verification_logs">
                            <div class="box-body">
                                <div class="download_label">SecuGen Verification Logs - Triage Module</div>
                                <div class="table-responsive">
                                    <table class="table table-striped table-bordered table-hover" id="verificationLogsTable">
                                        <thead>
                                            <tr>
                                                <th>Timestamp</th>
                                                <th>Patient</th>
                                                <th>Staff</th>
                                                <th>Match Score</th>
                                                <th>Result</th>
                                                <th>Module</th>
                                                <th>Details</th>
                                            </tr>
                                        </thead>
                                        <tbody></tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Enhanced Triage Modal with Department Routing -->
<div class="modal fade" id="triageModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">&times;</button>
                <h4 class="modal-title">
                    <i class="fa fa-heartbeat"></i> <span id="modalTitle">Triage Assessment with Department Routing</span>
                </h4>
            </div>
            <form id="triageForm">
                <div class="modal-body">
                    <!-- Patient Info Display -->
                    <div class="alert alert-info">
                        <div class="row">
                            <div class="col-md-8">
                                <strong>Patient:</strong> <span id="modalPatientName"></span><br>
                                <small>ID: <span id="modalPatientId"></span></small>
                            </div>
                            <div class="col-md-4 text-right">
                                <span id="modalVerificationBadge"></span>
                                <div id="modalVerificationDetails" style="font-size: 11px; margin-top: 5px;"></div>
                            </div>
                        </div>
                    </div>

                    <!-- Vital Signs Section -->
                    <div class="panel panel-primary">
                        <div class="panel-heading">
                            <h5 class="panel-title"><i class="fa fa-heartbeat"></i> Vital Signs Assessment</h5>
                        </div>
                        <div class="panel-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Blood Pressure (mmHg) <span class="text-danger">*</span></label>
                                        <div class="row">
                                            <div class="col-xs-6">
                                                <input type="number" name="bp_systolic" id="bpSystolic" class="form-control" placeholder="Systolic" min="60" max="250">
                                            </div>
                                            <div class="col-xs-6">
                                                <input type="number" name="bp_diastolic" id="bpDiastolic" class="form-control" placeholder="Diastolic" min="30" max="150">
                                            </div>
                                        </div>
                                        <div id="bpCategory" style="margin-top: 5px;"></div>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Heart Rate (bpm) <span class="text-danger">*</span></label>
                                        <input type="number" name="heart_rate" id="heartRate" class="form-control" placeholder="Heart Rate" min="30" max="200">
                                        <div id="hrCategory" style="margin-top: 5px;"></div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Temperature (°C) <span class="text-danger">*</span></label>
                                        <input type="number" step="0.1" name="temperature" id="temperature" class="form-control" placeholder="Temperature" min="30" max="45">
                                        <div id="tempCategory" style="margin-top: 5px;"></div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Respiratory Rate (rpm)</label>
                                        <input type="number" name="respiratory_rate" id="respiratoryRate" class="form-control" placeholder="Respiratory Rate" min="8" max="60">
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Oxygen Saturation (%)</label>
                                        <input type="number" name="oxygen_saturation" id="oxygenSaturation" class="form-control" placeholder="SpO2" min="70" max="100">
                                        <div id="spo2Category" style="margin-top: 5px;"></div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Weight (kg)</label>
                                        <input type="number" step="0.1" name="weight" id="weight" class="form-control" placeholder="Weight">
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Height (cm)</label>
                                        <input type="number" name="height" id="height" class="form-control" placeholder="Height">
                                        <div id="bmiDisplay" style="margin-top: 5px;"></div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label>Pain Scale (0-10)</label>
                                        <select name="pain_scale" id="painScale" class="form-control">
                                            <option value="">Select Pain Level</option>
                                            <option value="0">0 - No Pain</option>
                                            <option value="1">1 - Minimal</option>
                                            <option value="2">2 - Mild</option>
                                            <option value="3">3 - Uncomfortable</option>
                                            <option value="4">4 - Moderate</option>
                                            <option value="5">5 - Distressing</option>
                                            <option value="6">6 - Severe</option>
                                            <option value="7">7 - Very Severe</option>
                                            <option value="8">8 - Intense</option>
                                            <option value="9">9 - Excruciating</option>
                                            <option value="10">10 - Unbearable</option>
                                        </select>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Clinical Assessment & Triage Priority -->
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Triage Priority <span class="text-danger">*</span></label>
                                <select name="triage_priority" id="triagePriority" class="form-control" required>
                                    <option value="">Select Priority</option>
                                    <?php foreach($triagePriorities as $priority => $details): ?>
                                        <option value="<?php echo $priority; ?>" style="color: <?php echo $details['color']; ?>; font-weight: bold;">
                                            <?php echo $priority; ?> - <?php echo $details['name']; ?>
                                        </option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">Priority determines urgency and department routing</small>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Assign to Department</label>
                                <select name="department_assigned" id="departmentAssigned" class="form-control">
                                    <option value="">Select Department (Optional)</option>
                                    <?php foreach($departments as $code => $name): ?>
                                        <option value="<?php echo $code; ?>"><?php echo $name; ?></option>
                                    <?php endforeach; ?>
                                </select>
                                <small class="text-muted">Auto-routing based on triage priority and condition</small>
                            </div>
                        </div>
                    </div>

                    <div class="form-group">
                        <label>Chief Complaint / Presenting Symptoms <span class="text-danger">*</span></label>
                        <textarea name="chief_complaint" id="chiefComplaint" class="form-control" rows="3" placeholder="Primary reason for visit and main symptoms..." required></textarea>
                    </div>

                    <div class="form-group">
                        <label>Assessment Notes & Recommendations</label>
                        <textarea name="assessment_notes" id="assessmentNotes" class="form-control" rows="4" placeholder="Clinical observations, assessment findings, recommendations, and routing rationale..."></textarea>
                    </div>

                    <!-- Department Routing Preview -->
                    <div id="routingPreview" style="display: none;">
                        <div class="panel panel-info">
                            <div class="panel-heading">
                                <h6 class="panel-title">
                                    <i class="fa fa-random"></i> Department Routing Preview
                                </h6>
                            </div>
                            <div class="panel-body">
                                <div id="routingPreviewContent"></div>
                            </div>
                        </div>
                    </div>

                    <!-- Hidden fields -->
                    <input type="hidden" name="patient_id" id="modalPatientIdHidden">
                    <input type="hidden" name="triage_id" id="modalTriageId">
                    <input type="hidden" name="verification_status" id="modalVerificationStatus" value="0">
                    <input type="hidden" name="verification_score" id="modalVerificationScore" value="0">
                    <input type="hidden" name="verification_quality" id="modalVerificationQuality" value="">
                    <input type="hidden" name="verification_method" id="modalVerificationMethod" value="">
                    <input type="hidden" name="vitals" id="modalVitalsJson" value="">
                </div>

                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                       <i class="fa fa-save"></i> Save Triage & Route Patient
                   </button>
               </div>
           </form>
       </div>
   </div>
</div>

<!-- Department Stats Modal -->
<div class="modal fade" id="departmentStatsModal" tabindex="-1" role="dialog">
   <div class="modal-dialog modal-lg" role="document">
       <div class="modal-content">
           <div class="modal-header">
               <button type="button" class="close" data-dismiss="modal">&times;</button>
               <h4 class="modal-title">
                   <i class="fa fa-bar-chart"></i> Department Statistics & Analytics
               </h4>
           </div>
           <div class="modal-body">
               <div class="row">
                   <div class="col-md-6">
                       <div class="form-group">
                           <label>Date From</label>
                           <input type="date" class="form-control" id="statsDateFrom" value="<?php echo date('Y-m-d'); ?>">
                       </div>
                   </div>
                   <div class="col-md-6">
                       <div class="form-group">
                           <label>Date To</label>
                           <input type="date" class="form-control" id="statsDateTo" value="<?php echo date('Y-m-d'); ?>">
                       </div>
                   </div>
               </div>
               <div class="text-center">
                   <button type="button" id="loadStatsBtn" class="btn btn-primary">
                       <i class="fa fa-refresh"></i> Load Statistics
                   </button>
               </div>
               <div id="departmentStatsDisplay" style="margin-top: 20px;">
                   <!-- Stats will be loaded here -->
               </div>
           </div>
       </div>
   </div>
</div>

<!-- Route Patient Modal -->
<div class="modal fade" id="routePatientModal" tabindex="-1" role="dialog">
   <div class="modal-dialog" role="document">
       <div class="modal-content">
           <div class="modal-header">
               <button type="button" class="close" data-dismiss="modal">&times;</button>
               <h4 class="modal-title">
                   <i class="fa fa-random"></i> Route Patient to Department
               </h4>
           </div>
           <form id="routePatientForm">
               <div class="modal-body">
                   <div id="routePatientInfo"></div>
                   
                   <div class="form-group">
                       <label>Department <span class="text-danger">*</span></label>
                       <select name="department" id="routeDepartment" class="form-control" required>
                           <option value="">Select Department</option>
                           <?php foreach($departments as $code => $name): ?>
                               <option value="<?php echo $code; ?>"><?php echo $name; ?></option>
                           <?php endforeach; ?>
                       </select>
                   </div>
                   
                   <div class="form-group">
                       <label>Priority</label>
                       <select name="priority" id="routePriority" class="form-control">
                           <option value="normal">Normal</option>
                           <option value="high">High</option>
                           <option value="urgent">Urgent</option>
                       </select>
                   </div>
                   
                   <div class="form-group">
                       <label>Routing Notes</label>
                       <textarea name="routing_notes" id="routingNotes" class="form-control" rows="3" placeholder="Additional notes for the receiving department..."></textarea>
                   </div>
                   
                   <input type="hidden" name="patient_id" id="routePatientId">
                   <input type="hidden" name="triage_id" id="routeTriageId">
               </div>
               <div class="modal-footer">
                   <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                   <button type="submit" class="btn btn-primary">
                       <i class="fa fa-send"></i> Route Patient
                   </button>
               </div>
           </form>
       </div>
   </div>
</div>

<!-- Add Patient Modal -->
<div class="modal fade" id="addPatientModal" tabindex="-1" role="dialog">
   <div class="modal-dialog modal-lg" role="document">
       <div class="modal-content">
           <div class="modal-header">
               <button type="button" class="close" data-dismiss="modal">&times;</button>
               <h4 class="modal-title">
                   <i class="fa fa-plus"></i> Add New Patient
               </h4>
           </div>
           <form id="addPatientForm">
               <div class="modal-body">
                   <div class="row">
                       <div class="col-md-6">
                           <div class="form-group">
                               <label>Name <span class="text-danger">*</span></label>
                               <input type="text" class="form-control" name="patientName" required>
                           </div>
                       </div>
                       <div class="col-md-6">
                           <div class="form-group">
                               <label>Age <span class="text-danger">*</span></label>
                               <input type="number" class="form-control" name="patientAge" required>
                           </div>
                       </div>
                   </div>
                   <div class="row">
                       <div class="col-md-6">
                           <div class="form-group">
                               <label>Gender <span class="text-danger">*</span></label>
                               <select class="form-control" name="patientGender" required>
                                   <option value="">Select Gender</option>
                                   <?php foreach($genderList as $key => $value): ?>
                                       <option value="<?php echo $key; ?>"><?php echo $value; ?></option>
                                   <?php endforeach; ?>
                               </select>
                           </div>
                       </div>
                       <div class="col-md-6">
                           <div class="form-group">
                               <label>Phone <span class="text-danger">*</span></label>
                               <input type="text" class="form-control" name="patientPhone" required>
                           </div>
                       </div>
                   </div>
               </div>
               <div class="modal-footer">
                   <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                   <button type="submit" class="btn btn-primary">
                       <i class="fa fa-save"></i> Save Patient
                   </button>
               </div>
           </form>
       </div>
   </div>
</div>

<script type="text/javascript">
'use strict';

$(document).ready(function() {
    console.log('🩺 Enhanced Triage System with Patient Routing & SecuGen WebAPI Integration Starting...');
    
    // =============================================
    // GLOBAL CONFIGURATION
    // =============================================
    
    // SecuGen WebAPI Configuration
    window.secugenConfig = {
       baseUrl: 'https://localhost:8443',
       httpUrl: 'http://localhost:8443',
       endpoints: {
           capture: '/SGIFPCapture',
           matchScore: '/SGIMatchScore', 
           getInfo: '/SGIFPGetInfo',
           getVersion: '/SGIFPGetVersion'
       },
       serviceActive: false,
       quality: 50,
       timeout: 20000,
       lastError: null
   };

   // Enhanced Triage System State
   window.triageState = {
       verificationPassed: false,
       selectedPatientId: null,
       selectedPatientData: null,
       storedTemplate: null,
       liveTemplate: null,
       lastVerificationResult: null,
       verificationHistory: [],
       currentMatchScore: 0,
       verificationQuality: 'Unknown',
       currentTriageId: null,
       routingPreview: null,
       departmentQueues: {}
   };

   // =============================================
   // UTILITY FUNCTIONS
   // =============================================
   
   function showMessage(message, type = 'info', duration = 5000) {
       const alertClass = type === 'success' ? 'alert-success' : 
                         type === 'warning' ? 'alert-warning' : 
                         type === 'error' ? 'alert-danger' : 'alert-info';
       
       const html = `
           <div class="alert ${alertClass} alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
               <button type="button" class="close" data-dismiss="alert">&times;</button>
               <strong>${type.toUpperCase()}:</strong> ${message}
           </div>
       `;
       
       $('body').append(html);
       
       setTimeout(() => {
           $('.alert').fadeOut(() => $('.alert').remove());
       }, duration);
   }
   
   window.showMessage = showMessage;

   function logDiagnostic(message, type = 'info') {
       const timestamp = new Date().toLocaleTimeString();
       const colorClass = type === 'error' ? 'text-danger' : 
                         type === 'success' ? 'text-success' : 
                         type === 'warning' ? 'text-warning' : 'text-info';
       
       const html = `<div class="${colorClass}">[${timestamp}] ${message}</div>`;
       $("#diagnosticResults").append(html);
       
       console.log(`[SecuGen-Triage] ${message}`);
       
       // Keep only last 15 messages
       const diagnosticLines = $("#diagnosticResults div");
       if (diagnosticLines.length > 15) {
           diagnosticLines.first().remove();
       }
   }

   // =============================================
   // PATIENT LOADING AND MANAGEMENT
   // =============================================

   function loadPatients() {
       console.log('Loading patients for triage...');
       
       $('#triage_patient_search').html('<option value="">Loading patients...</option>');
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/getAllPatients"); ?>',
           type: 'GET',
           dataType: 'json',
           timeout: 10000,
           success: function(response) {
               console.log('Patients response:', response);
               
               let patients = [];
               
               if (response && response.patients && Array.isArray(response.patients)) {
                   patients = response.patients;
               } else if (response && Array.isArray(response)) {
                   patients = response;
               } else if (response && response.count !== undefined) {
                   patients = response.patients || [];
               } else {
                   console.error('Unexpected response format:', response);
                   handlePatientLoadError('Invalid response format from server');
                   return;
               }
               
               if (patients.length > 0) {
                   populatePatientDropdown(patients);
                   showMessage(`Loaded ${patients.length} patients for triage`, 'success');
               } else {
                   handleNoPatients(response);
               }
           },
           error: function(xhr, status, error) {
               console.error('Error loading patients:', error);
               console.error('XHR Status:', xhr.status);
               console.error('Response Text:', xhr.responseText);
               
               let errorMessage = getPatientLoadErrorMessage(xhr, status, error);
               handlePatientLoadError(errorMessage);
           }
       });
   }

   function populatePatientDropdown(patients) {
       const select = $('#triage_patient_search');
       select.empty();
       select.append(`<option value="">Select Patient for Triage (${patients.length} total)</option>`);
       
       $.each(patients, function(index, patient) {
           let patientName = patient.patient_name || 
                            (patient.firstname ? `${patient.firstname} ${patient.lastname || ''}`.trim() : '') ||
                            patient.name || 
                            `Patient #${patient.id}`;
           
           let text = `${patientName} (${patient.id})`;
           
           let phone = patient.mobileno || patient.phone || patient.contact;
           if (phone) {
               text += ` - ${phone}`;
           }
           
           if (patient.age && patient.age !== 'Unknown') {
               text += ` - ${patient.age}y`;
           }
           
           // Add current location if available
           if (patient.current_location && patient.current_location.department !== 'Not tracked') {
               text += ` [${patient.current_location.department}]`;
           }
           
           if (patient.has_fingerprint) {
               text += ' 🔐';
           }
           
           select.append(`<option value="${patient.id}">${text}</option>`);
       });
       
       console.log(`Successfully loaded ${patients.length} patients for triage`);
       
       if (typeof $.fn.select2 !== 'undefined') {
           select.select2({
               placeholder: 'Search patients for triage...',
               allowClear: true,
               width: '100%'
           });
       }
   }

   function handleNoPatients(response) {
       $('#triage_patient_search').html('<option value="">No patients found</option>');
       
       if (response && response.debug && response.table_columns) {
           console.log('Available table columns:', response.table_columns);
           showMessage('No patients found. Check console for table structure details.', 'warning');
       } else {
           showMessage('No patients found in database', 'warning');
       }
   }

   function handlePatientLoadError(errorMessage) {
       $('#triage_patient_search').html('<option value="">Error loading patients</option>');
       showMessage(errorMessage, 'error');
       
       if (window.location.href.includes('localhost') || window.location.href.includes('127.0.0.1')) {
           const debugUrl = '<?php echo base_url("admin/triage/debug"); ?>';
           const debugButton = `<br><small><a href="${debugUrl}" target="_blank" class="btn btn-xs btn-info">Debug Table Structure</a></small>`;
           $('#triage_patient_search').parent().append(debugButton);
       }
   }

   function getPatientLoadErrorMessage(xhr, status, error) {
       if (xhr.status === 404) {
           return 'Patient loading endpoint not found (404). Please check the controller method exists.';
       } else if (xhr.status === 500) {
           return 'Server error loading patients (500). Check PHP error logs.';
       } else if (xhr.status === 0) {
           return 'Network error - check connection and server status.';
       } else if (status === 'timeout') {
           return 'Request timeout - server may be slow or unresponsive.';
       } else {
           return `Error loading patients: ${error} (Status: ${xhr.status})`;
       }
   }

   function loadPatientDetails(patientId) {
       if (!patientId) {
           $('#patientDetailsSection').hide();
           resetVerificationState();
           return;
       }
       
       window.triageState.selectedPatientId = patientId;
       $('#selectedPatientId').val(patientId);
       $('#patientDetailsSection').show();
       resetVerificationState();
       
       $('#patientName').text('Loading...');
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/getPatientDetails"); ?>',
           type: 'POST',
           data: { id: patientId },
           dataType: 'json',
           timeout: 10000,
           success: function(response) {
               console.log('Patient details response:', response);
               
               if (response && !response.error) {
                   window.triageState.selectedPatientData = response;
                   populatePatientDetails(response);
                   loadStoredFingerprint(response);
                   loadPatientTriageRecords(patientId);
                   loadPatientRoutingHistory(patientId);
               } else {
                   console.error('Invalid patient details response:', response);
                   const errorMsg = response.error || 'Failed to load patient details';
                   showMessage(errorMsg, 'error');
                   $('#patientName').text('Error loading patient');
               }
           },
           error: function(xhr, status, error) {
               console.error('Error loading patient details:', error);
               console.error('Response:', xhr.responseText);
               
               const errorMessage = `Error loading patient details: ${error}`;
               showMessage(errorMessage, 'error');
               $('#patientName').text('Error loading patient');
           }
       });
   }

   function populatePatientDetails(patient) {
       $('#patientName').text(patient.patient_name || 'Unknown Patient');
       $('#patientAge').text((patient.age || 'Unknown') + (patient.age !== 'Unknown' ? ' years' : ''));
       $('#patientGender').text(patient.gender || 'Unknown');
       $('#patientPhone').text(patient.mobileno || 'N/A');
       $('#patientBloodGroup').text(patient.blood_group_name || 'Unknown');
       $('#patientAllergies').text(patient.known_allergies || 'None');
       
       if (patient.image && patient.image !== 'uploads/patient_images/no_image.png') {
           $('#patientPhoto').attr('src', '<?php echo base_url(); ?>' + patient.image);
       } else {
           $('#patientPhoto').attr('src', '<?php echo base_url("uploads/patient_images/no_image.png"); ?>');
       }
       
       // Show current location if available
       if (patient.current_location) {
           updatePatientLocationDisplay(patient.current_location);
       }
   }

   function updatePatientLocationDisplay(location) {
       let html = '';
       if (location.department !== 'Not tracked') {
           const statusClass = location.status === 'completed' ? 'success' : 
                             location.status === 'in_progress' ? 'info' : 'warning';
           
           html = `
               <div class="alert alert-${statusClass}" style="margin-top: 10px; padding: 8px;">
                   <strong>Current Location:</strong> ${location.department}<br>
                   <small>Status: ${location.status} | Since: ${new Date(location.since).toLocaleString()}</small>
               </div>
           `;
       }
       
       $('#patientLocationInfo').html(html);
   }

   // =============================================
   // SECUGEN BIOMETRIC FUNCTIONS
   // =============================================
   
   function initializeSecuGenService() {
       $("#secugen_service_status").html(`
           <div class="alert alert-info">
               <i class="fa fa-info-circle"></i> <strong>SecuGen WebAPI Ready - Triage Module</strong><br>
               <small>Use Manual Print Capture to scan fingerprint, then Verify Print to authenticate</small>
           </div>
       `);
       
       $('#manualVerifyBtn').prop('disabled', false);
       $('#verifyIdentity').prop('disabled', true);
       
       logDiagnostic("SecuGen triage capture buttons enabled", 'info');
       
       setTimeout(() => {
           $.ajax({
               url: 'https://localhost:8443/SGIFPGetInfo',
               type: 'GET',
               timeout: 3000,
               success: function(data) {
                   logDiagnostic("✓ SecuGen service detected and ready for triage", 'success');
                   $("#secugen_service_status").html(`
                       <div class="alert alert-success">
                           <i class="fa fa-check-circle"></i> <strong>SecuGen WebAPI Connected - Triage Module</strong><br>
                           <small>Service active at https://localhost:8443</small>
                       </div>
                   `);
               },
               error: function() {
                   logDiagnostic("SecuGen service not detected - capture will try direct connection", 'warning');
                   $("#secugen_service_status").html(`
                       <div class="alert alert-warning">
                           <i class="fa fa-exclamation-triangle"></i> <strong>SecuGen Service Check - Triage</strong><br>
                           <small>Service not detected. Buttons will attempt direct connection.</small><br>
                           <small><a href="https://localhost:8443/SGIFPGetInfo" target="_blank">Click to accept SSL certificate</a></small>
                       </div>
                   `);
               }
           });
       }, 1000);
   }

   function captureFingerprint() {
       logDiagnostic("Starting Print Capture for triage", 'info');
       
       const captureUrl = 'https://localhost:8443/SGIFPCapture';
       
       $("#fingerprintStatus").html(`
           <div class="alert alert-info">
               <i class="fa fa-spinner fa-spin"></i> <strong>Place finger on scanner...</strong><br>
               <small>Print Capture in progress - Triage Module</small>
           </div>
       `);
       
       $("#manualVerifyBtn").prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Scanning...');
       
       const xhr = new XMLHttpRequest();
       xhr.open('POST', captureUrl, true);
       xhr.setRequestHeader('Content-Type', 'application/x-www-form-urlencoded');
       xhr.timeout = 25000;
       
       xhr.onreadystatechange = function() {
           if (xhr.readyState === 4) {
               $("#manualVerifyBtn").prop('disabled', false).html('<i class="fa fa-hand-pointer-o"></i> Manual Print Capture');
               
               if (xhr.status === 200) {
                   try {
                       const result = JSON.parse(xhr.responseText);
                       handleCaptureResponse(result);
                   } catch (e) {
                       handleCaptureError('Invalid response from Print Capture: ' + e.message);
                   }
               } else {
                   handleCaptureError(`Print Capture failed: HTTP ${xhr.status}`);
               }
           }
       };
       
       xhr.onerror = function() {
           $("#manualVerifyBtn").prop('disabled', false).html('<i class="fa fa-hand-pointer-o"></i> Manual Print Capture');
           handleCaptureError('Network error during Print Capture');
       };
       
       xhr.ontimeout = function() {
           $("#manualVerifyBtn").prop('disabled', false).html('<i class="fa fa-hand-pointer-o"></i> Manual Print Capture');
           handleCaptureError('Print Capture timeout - no finger detected');
       };
       
       const postData = `Quality=50&Timeout=20000&WSQCompression=2.25`;
       xhr.send(postData);
   }

   function handleCaptureResponse(result) {
       if (result.ErrorCode === 0) {
           logDiagnostic("✓ Print Capture successful for triage", 'success');
           
           if (result.TemplateBase64) {
               window.triageState.liveTemplate = result.TemplateBase64;
               $('#liveScannedTemplate').val(result.TemplateBase64);
               $('#liveTemplateDetails').text(`Template captured for triage`);
           }
           
           if (result.BMPBase64) {
               $('#liveFingerprintImage').attr('src', 'data:image/bmp;base64,' + result.BMPBase64);
           }
           
           $("#fingerprintStatus").html(`
               <div class="alert alert-success">
                   <i class="fa fa-check-circle"></i> <strong>Fingerprint captured successfully!</strong><br>
                   <small>Ready for comparison with stored template - Triage Module</small>
               </div>
           `);
           
           if (window.triageState.storedTemplate) {
               $('#verifyIdentity').prop('disabled', false);
               showMessage('Fingerprint captured. Click Verify Print to authenticate for triage.', 'success');
           } else {
               showMessage('Fingerprint captured, but no stored template for comparison.', 'warning');
           }
           
       } else {
           let errorDescription = 'Fingerprint capture failed';
           
           switch(result.ErrorCode) {
               case 1: errorDescription = 'Reader not found or driver error'; break;
               case 2: errorDescription = 'Function failed - check reader connection'; break;
               case 3: errorDescription = 'Invalid parameters'; break;
               case 54: errorDescription = 'Timeout - no finger detected'; break;
               case 55: errorDescription = 'Device not found'; break;
               default: 
                   errorDescription = result.ErrorDescription || `Error code: ${result.ErrorCode}`;
           }
           
           handleCaptureError(`SGIFPCapture Error ${result.ErrorCode}: ${errorDescription}`);
       }
   }

   function handleCaptureError(errorMsg) {
       $("#fingerprintStatus").html(`
           <div class="alert alert-danger">
               <i class="fa fa-exclamation-triangle"></i> <strong>Capture Failed:</strong><br>
               <small>${errorMsg}</small>
           </div>
       `);
       
       logDiagnostic(`✗ ${errorMsg}`, 'error');
       showMessage(errorMsg, 'error');
   }

   function performBiometricVerification() {
       if (!window.triageState.storedTemplate || !window.triageState.liveTemplate) {
           showMessage('Both stored and captured templates required for comparison', 'error');
           return;
       }
       
       logDiagnostic("Starting SGIMatchScore comparison for triage", 'info');
       
       const matchUrl = 'https://localhost:8443/SGIMatchScore';
       
       $("#fingerprintStatus").html(`
           <div class="alert alert-info">
               <i class="fa fa-spinner fa-spin"></i> <strong>Comparing fingerprints...</strong><br>
               <small>SGIMatchScore in progress - Triage Module</small>
           </div>
       `);
       
       $("#verifyIdentity").prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Matching...');
       
       $.ajax({
           url: matchUrl,
           type: 'POST',
           data: {
               Template1: window.triageState.storedTemplate,
               Template2: window.triageState.liveTemplate,
               TemplateFormat: 'ISO'
           },
           timeout: 15000,
           success: function(response) {
               $("#verifyIdentity").prop('disabled', false).html('<i class="fa fa-shield"></i> Verify Print');
               
               try {
                   const result = typeof response === 'string' ? JSON.parse(response) : response;
                   handleMatchingResponse(result);
               } catch (e) {
                   handleMatchingError('Invalid response from SGIMatchScore: ' + e.message);
               }
           },
           error: function(xhr, status, error) {
               $("#verifyIdentity").prop('disabled', false).html('<i class="fa fa-shield"></i> Verify Print');
               
               let errorMsg = 'SGIMatchScore failed';
               if (xhr.status === 0) {
                   errorMsg = 'Cannot connect to SecuGen service for matching';
               } else {
                   errorMsg = `HTTP ${xhr.status}: ${error}`;
               }
               
               handleMatchingError(errorMsg);
           }
       });
   }

   function handleMatchingResponse(result) {
       if (result.ErrorCode === 0) {
           const matchScore = parseInt(result.MatchingScore || 0);
           const threshold = 60;
           const isMatch = matchScore >= threshold;
           
           window.triageState.currentMatchScore = matchScore;
           
           let quality, qualityClass;
           if (matchScore >= 100) {
               quality = 'Excellent';
               qualityClass = 'label-success';
           } else if (matchScore >= 80) {
               quality = 'Very Good';
               qualityClass = 'label-success';
           } else if (matchScore >= threshold) {
               quality = 'Good';
               qualityClass = 'label-info';
           } else {
               quality = 'Poor';
               qualityClass = 'label-danger';
           }
           
           window.triageState.verificationQuality = quality;
           
           logDiagnostic(`SGIMatchScore for triage: ${matchScore}/199 (${quality})`, 'info');
           
           logVerificationToDatabase(matchScore, isMatch, quality);
           
           if (isMatch) {
               window.triageState.verificationPassed = true;
               $('#verificationPassed').val('1');
               
               $("#fingerprintStatus").html(`
                   <div class="alert alert-success">
                       <i class="fa fa-check-circle"></i> <strong>Identity Verified Successfully for Triage!</strong><br>
                       <small>Match Score: <strong>${matchScore}/199</strong> | Quality: <span class="label ${qualityClass}">${quality}</span></small><br>
                       <small class="text-success">Patient identity confirmed for triage assessment</small>
                   </div>
               `);
               
               $('#verificationStatus').html('<span class="label label-success"><i class="fa fa-shield"></i> Verified</span>');
               $('#startTriageBtn').prop('disabled', false);
               
               addToVerificationHistory('Success', matchScore, quality);
               showMessage(`Identity verified for triage! Match score: ${matchScore}/199 (${quality})`, 'success');
               
           } else {
               window.triageState.verificationPassed = false;
               
               $("#fingerprintStatus").html(`
                   <div class="alert alert-danger">
                       <i class="fa fa-times-circle"></i> <strong>Verification Failed!</strong><br>
                       <small>Match Score: <strong>${matchScore}/199</strong> (Required: ${threshold}+)</small><br>
                       <small>Quality: <span class="label ${qualityClass}">${quality}</span></small>
                   </div>
               `);
               
               $('#verificationStatus').html('<span class="label label-danger"><i class="fa fa-times"></i> Failed</span>');
               
               addToVerificationHistory('Failed', matchScore, quality);
               showMessage(`Verification failed. Score ${matchScore}/199 below threshold.`, 'error');
           }
           
       } else {
           const errorDescription = result.ErrorDescription || `Error code: ${result.ErrorCode}`;
           handleMatchingError(`SGIMatchScore Error ${result.ErrorCode}: ${errorDescription}`);
       }
   }

   function handleMatchingError(errorMsg) {
       $("#fingerprintStatus").html(`
           <div class="alert alert-danger">
               <i class="fa fa-exclamation-triangle"></i> <strong>Matching Error:</strong><br>
               <small>${errorMsg}</small>
           </div>
       `);
       
       logDiagnostic(`✗ ${errorMsg}`, 'error');
       showMessage(errorMsg, 'error');
   }

   function loadStoredFingerprint(patient) {
       if (patient.fingerprint_status && patient.fingerprint_status.enrolled) {
           $.ajax({
               url: '<?php echo base_url("admin/triage/getStoredFingerprintTemplate"); ?>',
               type: 'POST',
               data: { patient_id: patient.id },
               dataType: 'json',
               success: function(response) {
                   if (response.status === 'success') {
                       window.triageState.storedTemplate = response.fingerprint_template;
                       $('#patientStoredTemplate').val(response.fingerprint_template);
                       
                       if (response.fingerprint_image) {
                           $('#storedFingerprintImage').attr('src', 'data:image/bmp;base64,' + response.fingerprint_image);
                       }
                       
                       $('#storedTemplateDetails').text(`Template loaded (${response.template_size} chars)`);
                       
                       $("#fingerprintStatus").html(`
                           <div class="alert alert-info">
                               <i class="fa fa-fingerprint"></i> <strong>Biometric template on file for triage</strong><br>
                               <small>Use Manual Print Capture to scan fingerprint, then Verify Print to authenticate</small>
                           </div>
                       `);
                       
                       logDiagnostic("Stored fingerprint template loaded for triage successfully", 'success');
                   } else {
                       handleNoStoredFingerprint();
                   }
               },
               error: function() {
                   handleNoStoredFingerprint();
               }
           });
       } else {
           handleNoStoredFingerprint();
       }
   }

   function handleNoStoredFingerprint() {
       $("#fingerprintStatus").html(`
           <div class="alert alert-warning">
               <i class="fa fa-exclamation-triangle"></i> <strong>No biometric template on file</strong><br>
               <small>Admin verification only - fingerprint enrollment required for biometric verification</small>
           </div>
       `);
       
       $('#storedTemplateDetails').text('No template');
       window.triageState.storedTemplate = null;
       
       $('#skipVerificationBtn').prop('disabled', false);
       $('#manualVerifyBtn').prop('disabled', false);
       $('#verifyIdentity').prop('disabled', true);
   }

   // =============================================
   // PATIENT ROUTING FUNCTIONS
   // =============================================

   function updateRoutingPreview(priority, department = null) {
       let html = '';
       
       // Auto-suggest department based on priority
       if (!department) {
           switch(priority) {
               case '1':
               case '2':
                   department = 'emergency';
                   break;
               case '3':
                   department = 'consultation';
                   break;
               case '4':
               case '5':
                   department = 'consultation';
                   break;
           }
       }
       
       if (department) {
           const departments = {
               'consultation': 'Consultation/OPD',
               'emergency': 'Emergency Department',
               'pharmacy': 'Pharmacy',
               'laboratory': 'Laboratory',
               'radiology': 'Radiology/Imaging',
               'admission': 'IPD Admission'
           };
           
           html = `
               <div class="alert alert-info">
                   <i class="fa fa-arrow-right"></i> <strong>Routing Preview:</strong><br>
                   Patient will be routed to <strong>${departments[department] || department}</strong><br>
                   <small>Priority Level: ${priority} | Expected wait time: ~15-30 minutes</small>
               </div>
           `;
           
           // Auto-select department if not manually chosen
           if ($('#departmentAssigned').val() !== department) {
               $('#departmentAssigned').val(department);
           }
           
           // Store routing preview
           window.triageState.routingPreview = {
               department: department,
               departmentName: departments[department] || department,
               priority: priority
           };
       }
       
       $('#routingPreview').html(html);
   }

   function routePatient(patientId, triageId, department, priority, notes) {
       if (!patientId || !department) {
           showMessage('Patient ID and department are required for routing', 'error');
           return;
       }
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/routePatient"); ?>',
           type: 'POST',
           data: {
               patient_id: patientId,
               triage_id: triageId,
               department: department,
               priority: priority || 'normal',
               notes: notes || ''
           },
           dataType: 'json',
           success: function(response) {
               if (response.status === 'success') {
                   showMessage(`Patient successfully routed to ${response.department}`, 'success');
                   
                   // Update patient location display
                   updatePatientLocationDisplay({
                       department: response.department,
                       status: 'pending',
                       since: new Date().toISOString()
                   });
                   
                   // Refresh department queues
                   loadDepartmentQueues();
                   
                   // Close any open modals
                   $('#triageModal').modal('hide');
                   
               } else {
                   showMessage(response.message || 'Failed to route patient', 'error');
               }
           },
           error: function(xhr, status, error) {
               console.error('Routing error:', error);
               showMessage(`Error routing patient: ${error}`, 'error');
           }
       });
   }

   function loadDepartmentQueues() {
       $.ajax({
           url: '<?php echo base_url("admin/triage/getAllDepartmentQueues"); ?>',
           type: 'GET',
           dataType: 'json',
           success: function(response) {
               if (response.status === 'success') {
                   window.triageState.departmentQueues = response.queues;
                   updateDepartmentQueueDisplay(response.queues);
               }
           },
           error: function(xhr, status, error) {
               console.error('Error loading department queues:', error);
           }
       });
   }

   function updateDepartmentQueueDisplay(queues) {
       let html = '';
       
       Object.keys(queues).forEach(function(deptCode) {
           const dept = queues[deptCode];
           const statusIcon = dept.count > 0 ? 'fa-users' : 'fa-check-circle';
           const statusClass = dept.count > 0 ? 'text-warning' : 'text-success';
           
           html += `
               <div class="row" style="margin-bottom: 8px;">
                   <div class="col-xs-8">
                       <i class="fa ${statusIcon} ${statusClass}"></i> ${dept.name}
                   </div>
                   <div class="col-xs-4 text-right">
                       <span class="badge badge-${dept.count > 0 ? 'warning' : 'success'}">${dept.count}</span>
                    </div>
          `;
      });
      
      if (html === '') {
          html = '<p class="text-muted">No department data available</p>';
      }
      
      $('#departmentQueueStatus').html(html);
  }

  function loadPatientRoutingHistory(patientId) {
      $.ajax({
          url: '<?php echo base_url("admin/triage/getPatientRoutingHistory"); ?>',
          type: 'POST',
          data: { patient_id: patientId },
          dataType: 'json',
          success: function(response) {
              if (response && Array.isArray(response)) {
                  updateRoutingHistoryDisplay(response);
              }
          },
          error: function(xhr, status, error) {
              console.error('Error loading routing history:', error);
          }
      });
  }

  function updateRoutingHistoryDisplay(history) {
      let html = '';
      
      if (history.length === 0) {
          html = '<p class="text-muted">No routing history available</p>';
      } else {
          history.forEach(function(routing) {
              const statusClass = routing.status === 'completed' ? 'success' : 
                                routing.status === 'in_progress' ? 'info' : 'warning';
              
              html += `
                  <div class="panel panel-default" style="margin-bottom: 8px;">
                      <div class="panel-body" style="padding: 8px;">
                          <strong>${routing.to_department}</strong>
                          <span class="label label-${statusClass} pull-right">${routing.status}</span>
                          <br><small>${new Date(routing.routed_at).toLocaleString()}</small>
                      </div>
                  </div>
              `;
          });
      }
      
      $('#routingHistory').html(html);
  }

  // =============================================
  // TRIAGE MANAGEMENT
  // =============================================

  function startTriage() {
      const patientId = $('#selectedPatientId').val();
      if (!patientId) {
          showMessage('Please select a patient first', 'warning');
          return;
      }
      
      if (!window.triageState.verificationPassed) {
          showMessage('Patient identity verification required before triage assessment', 'warning');
          return;
      }
      
      $('#modalPatientIdHidden').val(patientId);
      $('#modalPatientId').text(patientId);
      $('#modalPatientName').text($('#patientName').text());
      $('#modalVerificationStatus').val('1');
      
      const verificationBadge = window.triageState.verificationPassed ? 
          '<span class="label label-success"><i class="fa fa-shield"></i> Biometric Verified</span>' : 
          '<span class="label label-warning"><i class="fa fa-exclamation-triangle"></i> Unverified</span>';
      
      $('#modalVerificationBadge').html(verificationBadge);
      
      if (window.triageState.verificationPassed) {
          $('#modalVerificationDetails').html(`
              <small>SecuGen Score: ${window.triageState.currentMatchScore}/199</small><br>
              <small>Quality: ${window.triageState.verificationQuality}</small>
          `);
      }
      
      $('#triageModal').modal('show');
  }

  function saveTriage() {
      const submitBtn = $('#triageForm').find('button[type="submit"]');
      submitBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Saving...');
      
      // Compile vitals into JSON format
      const vitals = {
          blood_pressure: {
              systolic: $('#bpSystolic').val() || null,
              diastolic: $('#bpDiastolic').val() || null
          },
          heart_rate: $('#heartRate').val() || null,
          temperature: $('#temperature').val() || null,
          respiratory_rate: $('#respiratoryRate').val() || null,
          oxygen_saturation: $('#oxygenSaturation').val() || null,
          weight: $('#weight').val() || null,
          height: $('#height').val() || null,
          pain_scale: $('#painScale').val() || null,
          triage_priority: $('#triagePriority').val() || null
      };
      
      // Set the vitals JSON
      $('#modalVitalsJson').val(JSON.stringify(vitals));
      
      // Combine chief complaint and assessment notes
      const notes = `Chief Complaint: ${$('#chiefComplaint').val()}\n\nAssessment Notes: ${$('#assessmentNotes').val()}`;
      
      // Get department assignment
      const departmentAssigned = $('#departmentAssigned').val();
      const priority = $('#triagePriority').val();
      
      // Prepare form data
      const formData = $('#triageForm').serialize() + 
                      `&vitals=${encodeURIComponent(JSON.stringify(vitals))}` +
                      `&notes=${encodeURIComponent(notes)}` +
                      `&department_assigned=${departmentAssigned}` +
                      `&priority=${priority}`;
      
      $.ajax({
          url: '<?php echo base_url("admin/triage/save"); ?>',
          type: 'POST',
          data: formData,
          dataType: 'json',
          success: function(response) {
              if (response.status === 'success') {
                  showMessage(response.message, 'success');
                  
                  // Store triage ID for routing
                  window.triageState.currentTriageId = response.triage_id;
                  
                  // If routing result is provided, show routing status
                  if (response.routing_result) {
                      if (response.routing_result.status === 'success') {
                          showMessage(`Patient routed to ${response.routing_result.department}`, 'success');
                      } else {
                          showMessage(`Triage saved but routing failed: ${response.routing_result.message}`, 'warning');
                      }
                  }
                  
                  $('#triageModal').modal('hide');
                  
                  const patientId = $('#selectedPatientId').val();
                  if (patientId) {
                      loadPatientTriageRecords(patientId);
                      loadPatientRoutingHistory(patientId);
                  }
                  
                  $('#triageForm')[0].reset();
                  
                  // Refresh department queues
                  loadDepartmentQueues();
                  
              } else {
                  showMessage(response.message || 'Failed to save triage assessment', 'error');
              }
          },
          error: function(xhr, status, error) {
              console.error('Save triage error:', xhr.responseText);
              showMessage(`Error saving triage assessment: ${error}`, 'error');
          },
          complete: function() {
              submitBtn.prop('disabled', false).html('<i class="fa fa-save"></i> Save Triage Assessment');
          }
      });
  }

  function addNewPatient() {
      const submitBtn = $('#addPatientForm').find('button[type="submit"]');
      submitBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Saving...');
      
      $.ajax({
          url: '<?php echo base_url("admin/triage/addPatient"); ?>',
          type: 'POST',
          data: $('#addPatientForm').serialize(),
          dataType: 'json',
          success: function(response) {
              if (response.status === 'success') {
                  showMessage(response.message, 'success');
                  $('#addPatientModal').modal('hide');
                  loadPatients();
                  $('#addPatientForm')[0].reset();
              } else {
                  showMessage(response.message || 'Failed to add patient', 'error');
              }
          },
          error: function(xhr, status, error) {
              showMessage(`Error adding patient: ${error}`, 'error');
          },
          complete: function() {
              submitBtn.prop('disabled', false).html('<i class="fa fa-save"></i> Save Patient');
          }
      });
  }

  function loadPatientTriageRecords(patientId) {
      // Placeholder for loading patient's previous triage records
      $('#recentTriageRecords').html('<p class="text-muted">Loading triage history...</p>');
      
      setTimeout(() => {
          $('#recentTriageRecords').html('<p class="text-muted">No previous triage records found</p>');
      }, 1000);
  }

  // =============================================
  // VERIFICATION CONTROLS
  // =============================================
  
  function skipVerification() {
      window.triageState.verificationPassed = true;
      $('#verificationPassed').val('1');
      
      $('#verificationStatus').html('<span class="label label-warning"><i class="fa fa-user-shield"></i> Admin Override</span>');
      
      $("#fingerprintStatus").html(`
          <div class="alert alert-warning">
              <i class="fa fa-shield"></i> <strong>Verification bypassed - Admin override active</strong><br>
              <small>Biometric verification skipped by administrator for triage</small>
          </div>
      `);
      
      $('#startTriageBtn').prop('disabled', false);
      
      addToVerificationHistory('Admin Override', 0, 'Bypassed');
      
      logDiagnostic("Triage verification skipped by admin override", 'warning');
      showMessage('Verification bypassed - Admin override active for triage', 'warning');
  }

  function clearCaptureData() {
      window.triageState.liveTemplate = null;
      $('#liveScannedTemplate').val('');
      $('#liveFingerprintImage').attr('src', './Images/PlaceFinger.bmp');
      $('#liveTemplateDetails').text('Ready for capture');
      
      $("#fingerprintStatus").html(`
          <div class="alert alert-info">
              <i class="fa fa-info-circle"></i> Capture data cleared - Ready for new verification
          </div>
      `);
      
      if (window.triageState.storedTemplate) {
          $('#verifyIdentity').prop('disabled', true);
          showMessage('Live capture cleared. Use Manual Print Capture to capture fingerprint again.', 'info');
      }
      
      logDiagnostic("Triage capture data cleared", 'info');
  }

  function resetVerificationState() {
      window.triageState.verificationPassed = false;
      window.triageState.storedTemplate = null;
      window.triageState.liveTemplate = null;
      window.triageState.currentMatchScore = 0;
      window.triageState.verificationQuality = 'Unknown';
      
      $('#verificationPassed').val('0');
      $('#verificationStatus').html('');
      
      $("#fingerprintStatus").html(`
          <div class="alert alert-info">
              <i class="fa fa-info-circle"></i> Ready for SecuGen biometric verification - Triage Module
          </div>
      `);
      
      $('#storedFingerprintImage, #liveFingerprintImage').attr('src', './Images/PlaceFinger.bmp');
      $('#storedTemplateDetails, #liveTemplateDetails').text('');
      $('#startTriageBtn').prop('disabled', true);
      $('#patientStoredTemplate, #liveScannedTemplate').val('');
      
      window.triageState.verificationHistory = [];
      updateVerificationHistoryDisplay();
  }

  // =============================================
  // VERIFICATION HISTORY
  // =============================================
  
  function addToVerificationHistory(result, score, quality) {
      const timestamp = new Date().toLocaleTimeString();
      const patientName = $('#patientName').text() || 'Unknown';
      
      const historyItem = {
          timestamp: timestamp,
          patient: patientName,
          result: result,
          score: score,
          quality: quality,
          method: 'SecuGen SGIMatchScore - Triage'
      };
      
      window.triageState.verificationHistory.unshift(historyItem);
      
      if (window.triageState.verificationHistory.length > 10) {
          window.triageState.verificationHistory.pop();
      }
      
      updateVerificationHistoryDisplay();
  }

  function updateVerificationHistoryDisplay() {
      let html = '';
      
      if (window.triageState.verificationHistory.length === 0) {
          html = '<p class="text-muted">No verification attempts yet</p>';
      } else {
          window.triageState.verificationHistory.forEach(item => {
              const resultClass = item.result === 'Success' ? 'text-success' : 'text-danger';
              const resultIcon = item.result === 'Success' ? 'fa-check' : 'fa-times';
              
              html += `
                  <div style="border-left: 3px solid ${item.result === 'Success' ? '#5cb85c' : '#d9534f'}; padding: 8px; margin-bottom: 8px; background-color: #f9f9f9; border-radius: 3px;">
                      <div class="${resultClass}">
                          <i class="fa ${resultIcon}"></i> <strong>${item.result}</strong> - ${item.timestamp}
                      </div>
                      <small>Score: ${item.score}/199 | Quality: ${item.quality}</small><br>
                      <small class="text-muted">${item.method}</small>
                  </div>
              `;
          });
      }
      
      $('#verificationHistory').html(html);
  }

  function logVerificationToDatabase(matchScore, isMatch, quality) {
      const patientId = $('#selectedPatientId').val();
      
      if (!patientId) return;
      
      $.ajax({
          url: '<?php echo base_url("admin/triage/verifyFingerprint"); ?>',
          type: 'POST',
          data: {
              patient_id: patientId,
              match_score: matchScore,
              is_match: isMatch,
              threshold: 60,
              service_url: window.secugenConfig.baseUrl,
              verification_method: 'SecuGen SGIMatchScore - Triage',
              quality_rating: quality
          },
          dataType: 'json',
          success: function(response) {
              logDiagnostic("Triage verification logged to database successfully", 'info');
          },
          error: function(xhr, status, error) {
              logDiagnostic(`Error logging triage verification: ${error}`, 'warning');
          }
      });
  }
  
  function populateTriageEditForm(triageData) {
       const vitals = triageData.vitals_parsed || {};
       
       $('#modalTitle').text('Edit Triage Assessment');
       $('#modalTriageId').val(triageData.id);
       $('#modalPatientId').text(triageData.patient_id);
       $('#modalPatientName').text(triageData.patient_name);
       
       $('#triagePriority').val(triageData.priority);
       $('#departmentAssigned').val(triageData.department_assigned);
       $('#chiefComplaint').val(triageData.chief_complaint);
       $('#assessmentNotes').val(triageData.assessment_notes);
       
       if (vitals.blood_pressure) {
           $('#bpSystolic').val(vitals.blood_pressure.systolic);
           $('#bpDiastolic').val(vitals.blood_pressure.diastolic);
       }
       $('#heartRate').val(vitals.heart_rate);
       $('#temperature').val(vitals.temperature);
       $('#respiratoryRate').val(vitals.respiratory_rate);
       $('#oxygenSaturation').val(vitals.oxygen_saturation);
       $('#weight').val(vitals.weight);
       $('#height').val(vitals.height);
       $('#painScale').val(vitals.pain_scale);
       
       $('#triageForm').off('submit').on('submit', function(e) {
           e.preventDefault();
           updateTriageRecord();
       });
       
       if (triageData.priority) {
           updateRoutingPreview(triageData.priority, triageData.department_assigned);
       }
   }

   function updateTriageRecord() {
       const submitBtn = $('#triageForm').find('button[type="submit"]');
       submitBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Updating...');
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/updateTriage"); ?>',
           type: 'POST',
           data: {
               triage_id: $('#modalTriageId').val(),
               patient_id: $('#modalPatientId').text(),
               priority: $('#triagePriority').val(),
               department_assigned: $('#departmentAssigned').val(),
               chief_complaint: $('#chiefComplaint').val(),
               assessment_notes: $('#assessmentNotes').val(),
               bp_systolic: $('#bpSystolic').val(),
               bp_diastolic: $('#bpDiastolic').val(),
               heart_rate: $('#heartRate').val(),
               temperature: $('#temperature').val(),
               respiratory_rate: $('#respiratoryRate').val(),
               oxygen_saturation: $('#oxygenSaturation').val(),
               weight: $('#weight').val(),
               height: $('#height').val(),
               pain_scale: $('#painScale').val()
           },
           dataType: 'json',
           success: function(response) {
               if (response.status === 'success') {
                   showMessage('Triage record updated successfully', 'success');
                   $('#triageModal').modal('hide');
                   
                   if ($('#tab_triage_list').hasClass('active')) {
                       loadTriageList();
                   }
               } else {
                   showMessage(response.message || 'Failed to update triage record', 'error');
               }
           },
           error: function(xhr, status, error) {
               showMessage(`Error updating triage record: ${error}`, 'error');
           },
           complete: function() {
               submitBtn.prop('disabled', false).html('<i class="fa fa-save"></i> Update Triage Assessment');
           }
       });
   }
   
   function displayTriageDetails(triageData) {
       const vitals = triageData.vitals_parsed || {};
       
       const modalContent = `
           <div class="modal fade" id="viewTriageModal" tabindex="-1" role="dialog">
               <div class="modal-dialog modal-lg" role="document">
                   <div class="modal-content">
                       <div class="modal-header">
                           <button type="button" class="close" data-dismiss="modal">&times;</button>
                           <h4 class="modal-title">
                               <i class="fa fa-eye"></i> Triage Details - ${triageData.patient_name}
                           </h4>
                       </div>
                       <div class="modal-body">
                           <div class="row">
                               <div class="col-md-6">
                                   <h5><i class="fa fa-user"></i> Patient Information</h5>
                                   <table class="table table-condensed">
                                       <tr><th>Name:</th><td>${triageData.patient_name || 'N/A'}</td></tr>
                                       <tr><th>Age:</th><td>${triageData.age || 'N/A'}</td></tr>
                                       <tr><th>Gender:</th><td>${triageData.gender || 'N/A'}</td></tr>
                                       <tr><th>Mobile:</th><td>${triageData.mobileno || 'N/A'}</td></tr>
                                   </table>
                               </div>
                               <div class="col-md-6">
                                   <h5><i class="fa fa-stethoscope"></i> Triage Information</h5>
                                   <table class="table table-condensed">
                                       <tr><th>Priority:</th><td><span class="label label-${getPriorityClass(triageData.priority)}">Priority ${triageData.priority}</span></td></tr>
                                       <tr><th>Staff:</th><td>${triageData.staff_name || 'N/A'}</td></tr>
                                       <tr><th>Date:</th><td>${formatDateTime(triageData.created_at)}</td></tr>
                                       <tr><th>Department:</th><td>${triageData.department_assigned || 'N/A'}</td></tr>
                                   </table>
                               </div>
                           </div>
                           
                           <div class="row">
                               <div class="col-md-12">
                                   <h5><i class="fa fa-heartbeat"></i> Vital Signs</h5>
                                   <div class="vital-signs-grid">
                                       <div class="vital-sign-item">
                                           <div class="vital-sign-label">Blood Pressure</div>
                                           <div class="vital-sign-value">
                                               ${vitals.blood_pressure ? `${vitals.blood_pressure.systolic}/${vitals.blood_pressure.diastolic} mmHg` : 'Not recorded'}
                                           </div>
                                       </div>
                                       <div class="vital-sign-item">
                                           <div class="vital-sign-label">Heart Rate</div>
                                           <div class="vital-sign-value">
                                               ${vitals.heart_rate ? `${vitals.heart_rate} bpm` : 'Not recorded'}
                                           </div>
                                       </div>
                                       <div class="vital-sign-item">
                                           <div class="vital-sign-label">Temperature</div>
                                           <div class="vital-sign-value">
                                               ${vitals.temperature ? `${vitals.temperature}°C` : 'Not recorded'}
                                           </div>
                                       </div>
                                       <div class="vital-sign-item">
                                           <div class="vital-sign-label">Pain Scale</div>
                                           <div class="vital-sign-value">
                                               ${vitals.pain_scale ? `${vitals.pain_scale}/10` : 'Not recorded'}
                                           </div>
                                       </div>
                                   </div>
                               </div>
                           </div>
                           
                           <div class="row" style="margin-top: 15px;">
                               <div class="col-md-12">
                                   <h5><i class="fa fa-notes-medical"></i> Clinical Notes</h5>
                                   <div class="well">
                                       <strong>Chief Complaint:</strong><br>
                                       ${triageData.chief_complaint || 'Not recorded'}
                                   </div>
                                   <div class="well">
                                       <strong>Assessment Notes:</strong><br>
                                       ${triageData.assessment_notes || 'Not recorded'}
                                   </div>
                               </div>
                           </div>
                       </div>
                       <div class="modal-footer">
                           <button type="button" class="btn btn-primary" onclick="editTriage(${triageData.id})">
                               <i class="fa fa-edit"></i> Edit Triage
                           </button>
                           <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                       </div>
                   </div>
               </div>
           </div>
       `;
       
       $('#viewTriageModal').remove();
       $('body').append(modalContent);
       $('#viewTriageModal').modal('show');
   }


  // =============================================
  // DATA TABLES INITIALIZATION
  // =============================================
  
  function loadTriageList() {
      if ($.fn.DataTable.isDataTable('#triageListTable')) {
          $('#triageListTable').DataTable().destroy();
      }
      
      $('#triageListTable').DataTable({
          "processing": true,
          "ajax": {
              "url": "<?php echo base_url('admin/triage/getTriageList'); ?>",
              "type": "GET",
              "dataSrc": "data"
          },
          "columns": [
              { "data": 0 }, // ID
              { "data": 1 }, // Date/Time
              { "data": 2 }, // Patient
              { "data": 3 }, // Staff
              { "data": 4 }, // Vitals/Priority
              { "data": 5 }, // Notes
              { "data": 6 }, // Status
              { "data": 7 }  // Actions
          ],
          "order": [[ 0, "desc" ]],
          "pageLength": 25,
          "responsive": true
      });
  }

// Function to initialize routing DataTable
function loadRoutingList() {
    console.log('Initializing routing DataTable...');
    
    // Destroy existing DataTable if it exists
    if ($.fn.DataTable.isDataTable('#departmentRoutingTable')) {
        $('#departmentRoutingTable').DataTable().destroy();
    }
    
    // Initialize new DataTable
    const routingTable = $('#departmentRoutingTable').DataTable({
        processing: true,
        serverSide: true,
        responsive: true,
        ajax: {
            url: baseUrl + 'admin/triage/getRoutingList',
            type: 'POST',
            data: function(d) {
                // Add any additional parameters if needed
                d.module = 'triage_routing';
                return d;
            },
            error: function(xhr, error, code) {
                console.error('DataTable Ajax Error:', {
                    status: xhr.status,
                    statusText: xhr.statusText,
                    responseText: xhr.responseText,
                    error: error,
                    code: code
                });
                
                // Show user-friendly error message
                showMessage('Failed to load routing data. Please check console for details.', 'error');
                
                // Try to parse error response
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.error) {
                        console.error('Server Error:', response.error);
                        showMessage('Server Error: ' + response.error, 'error');
                    }
                } catch (e) {
                    console.error('Could not parse error response');
                }
            }
        },
        columns: [
            {
                title: 'Patient',
                data: 0,
                orderable: true,
                searchable: true,
                width: '20%'
            },
            {
                title: 'From',
                data: 1,
                orderable: false,
                searchable: false,
                width: '12%'
            },
            {
                title: 'To Department',
                data: 2,
                orderable: false,
                searchable: true,
                width: '15%'
            },
            {
                title: 'Priority',
                data: 3,
                orderable: true,
                searchable: false,
                width: '10%'
            },
            {
                title: 'Routed At',
                data: 4,
                orderable: true,
                searchable: false,
                width: '15%'
            },
            {
                title: 'Status',
                data: 5,
                orderable: true,
                searchable: true,
                width: '10%'
            },
            {
                title: 'Actions',
                data: 6,
                orderable: false,
                searchable: false,
                width: '18%'
            }
        ],
        order: [[4, 'desc']], // Sort by routed_at descending
        pageLength: 25,
        lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
        language: {
            processing: '<i class="fa fa-spinner fa-spin"></i> Loading routing data...',
            emptyTable: 'No patient routing records found',
            zeroRecords: 'No matching routing records found',
            search: 'Search routing records:',
            lengthMenu: 'Show _MENU_ records per page',
            info: 'Showing _START_ to _END_ of _TOTAL_ routing records',
            infoEmpty: 'No routing records available',
            infoFiltered: '(filtered from _MAX_ total records)',
            paginate: {
                first: 'First',
                last: 'Last',
                next: 'Next',
                previous: 'Previous'
            }
        },
        dom: '<"row"<"col-sm-6"l><"col-sm-6"f>>' +
             '<"row"<"col-sm-12"tr>>' +
             '<"row"<"col-sm-5"i><"col-sm-7"p>>',
        drawCallback: function(settings) {
            console.log('DataTable draw completed. Records:', settings.json?.recordsTotal || 0);
            
            // Re-enable tooltips after each draw
            $('[title]').tooltip();
        },
        initComplete: function(settings, json) {
            console.log('Routing DataTable initialized successfully');
            console.log('Initial data:', json);
            
            // Add refresh button
            const wrapper = $('#departmentRoutingTable_wrapper');
            if (wrapper.find('.routing-refresh-btn').length === 0) {
                const refreshBtn = $('<button class="btn btn-primary btn-sm routing-refresh-btn" style="margin-left: 10px;">' +
                                   '<i class="fa fa-refresh"></i> Refresh</button>');
                refreshBtn.on('click', function() {
                    routingTable.ajax.reload(null, false);
                    loadDepartmentQueues();
                });
                wrapper.find('.dataTables_filter').append(refreshBtn);
            }
        }
    });
    
    // Store reference for later use
    window.routingDataTable = routingTable;
    
    return routingTable;
}

// Functions for routing actions
function viewRouting(routingId) {
    console.log('Viewing routing:', routingId);
    
    $.ajax({
        url: baseUrl + 'admin/triage/viewRouting',
        type: 'POST',
        data: { routing_id: routingId },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                displayRoutingDetails(response.data);
            } else {
                showMessage(response.message || 'Failed to load routing details', 'error');
            }
        },
        error: function(xhr, status, error) {
            console.error('Error viewing routing:', error);
            showMessage('Error loading routing details: ' + error, 'error');
        }
    });
}

function updateRoutingStatus(routingId, newStatus) {
    console.log('Updating routing status:', routingId, newStatus);
    
    const confirmMessage = `Are you sure you want to mark this routing as ${newStatus}?`;
    if (!confirm(confirmMessage)) {
        return;
    }
    
    $.ajax({
        url: baseUrl + 'admin/triage/updateRoutingStatus',
        type: 'POST',
        data: { 
            routing_id: routingId,
            status: newStatus,
            notes: `Status updated to ${newStatus} by triage staff`
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showMessage(response.message || 'Routing status updated successfully', 'success');
                
                // Refresh the DataTable
                if (window.routingDataTable) {
                    window.routingDataTable.ajax.reload(null, false);
                }
                
                // Refresh department queues
                loadDepartmentQueues();
            } else {
                showMessage(response.message || 'Failed to update routing status', 'error');
            }
        },
        error: function(xhr, status, error) {
            console.error('Error updating routing status:', error);
            showMessage('Error updating status: ' + error, 'error');
        }
    });
}

function displayRoutingDetails(routing) {
    let html = `
        <div class="modal fade" id="routingDetailsModal" tabindex="-1" role="dialog">
            <div class="modal-dialog modal-lg" role="document">
                <div class="modal-content">
                    <div class="modal-header">
                        <button type="button" class="close" data-dismiss="modal">&times;</button>
                        <h4 class="modal-title">Routing Details - ${routing.patient_name}</h4>
                    </div>
                    <div class="modal-body">
                        <div class="row">
                            <div class="col-md-6">
                                <h5>Patient Information</h5>
                                <table class="table table-condensed">
                                    <tr><td><strong>Name:</strong></td><td>${routing.patient_name}</td></tr>
                                    <tr><td><strong>ID:</strong></td><td>${routing.patient_id}</td></tr>
                                    <tr><td><strong>Age:</strong></td><td>${routing.age || 'N/A'}</td></tr>
                                    <tr><td><strong>Gender:</strong></td><td>${routing.gender || 'N/A'}</td></tr>
                                    <tr><td><strong>Phone:</strong></td><td>${routing.mobileno || 'N/A'}</td></tr>
                                </table>
                            </div>
                            <div class="col-md-6">
                                <h5>Routing Information</h5>
                                <table class="table table-condensed">
                                    <tr><td><strong>From:</strong></td><td>${routing.from_department || 'Triage'}</td></tr>
                                    <tr><td><strong>To:</strong></td><td>${routing.to_department}</td></tr>
                                    <tr><td><strong>Priority:</strong></td><td>${routing.priority || routing.triage_priority || 'Normal'}</td></tr>
                                    <tr><td><strong>Status:</strong></td><td>${routing.status}</td></tr>
                                    <tr><td><strong>Routed By:</strong></td><td>${routing.staff_name || 'Unknown Staff'}</td></tr>
                                    <tr><td><strong>Routed At:</strong></td><td>${routing.routed_at}</td></tr>
                                </table>
                            </div>
                        </div>
                        ${routing.routing_notes ? `
                        <div class="row">
                            <div class="col-md-12">
                                <h5>Notes</h5>
                                <p class="well">${routing.routing_notes}</p>
                            </div>
                        </div>
                        ` : ''}
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Remove existing modal and add new one
    $('#routingDetailsModal').remove();
    $('body').append(html);
    $('#routingDetailsModal').modal('show');
}

// Auto-refresh routing list every 30 seconds when tab is active
setInterval(function() {
    if ($('#tab_department_routing').hasClass('active') && window.routingDataTable) {
        window.routingDataTable.ajax.reload(null, false);
    }
}, 30000);

  // =============================================
  // EVENT HANDLERS
  // =============================================

  function initializeEvents() {
      // Patient selection
      $('#triage_patient_search').on('change', function() {
          const patientId = $(this).val();
          if (patientId) {
              loadPatientDetails(patientId);
          } else {
              $('#patientDetailsSection').hide();
              resetVerificationState();
          }
      });
      
      // SecuGen fingerprint operations
      $('#manualVerifyBtn').on('click', function(e) {
          e.preventDefault();
          captureFingerprint();
      });

      $('#verifyIdentity').on('click', function(e) {
          e.preventDefault();
          performBiometricVerification();
      });
      
      // Verification controls
      $('#skipVerificationBtn').on('click', function() {
          skipVerification();
      });
      
      $('#clearCaptureData').on('click', function() {
          clearCaptureData();
      });
      
      // Triage management
      $('#startTriageBtn').on('click', function() {
          startTriage();
      });
      
      // Form submissions
      $('#triageForm').on('submit', function(e) {
          e.preventDefault();
          saveTriage();
      });
      
      $('#addPatientForm').on('submit', function(e) {
          e.preventDefault();
          addNewPatient();
      });

      // Triage Priority change handler
      $('#triagePriority').on('change', function() {
          const priority = $(this).val();
          if (priority) {
              updateRoutingPreview(priority);
          }
      });

      // Department assignment change handler
      $('#departmentAssigned').on('change', function() {
          const department = $(this).val();
          const priority = $('#triagePriority').val();
          if (department && priority) {
              updateRoutingPreview(priority, department);
          }
      });

      // Diagnostic controls
      $('#testSecuGenConnection').on('click', function() {
          $('#diagnosticResults').html('');
          logDiagnostic("Manual connection test initiated for triage", 'info');
          initializeSecuGenService();
      });

      $('#openSecuGenDemo').on('click', function() {
          window.open('https://webapi.secugen.com/Demo1', '_blank');
          logDiagnostic("Opened SecuGen official demo", 'info');
      });

      $('#clearDiagnosticData').on('click', function() {
          $('#diagnosticResults').html('');
          logDiagnostic("Diagnostic logs cleared", 'info');
      });

      // Auto-calculate BMI when height and weight are entered
      $('#height, #weight').on('input', function() {
          const height = parseFloat($('#height').val());
          const weight = parseFloat($('#weight').val());
          
          if (height && weight && height > 0) {
              const heightInMeters = height / 100;
              const bmi = (weight / (heightInMeters * heightInMeters)).toFixed(1);
              
              if ($('#bmiDisplay').length === 0) {
                  $('#weight').parent().append(`<small id="bmiDisplay" class="text-info">BMI: ${bmi}</small>`);
              } else {
                  $('#bmiDisplay').text(`BMI: ${bmi}`);
              }
          }
      });

      // Vital signs validation and alerts
      $('#bpSystolic, #bpDiastolic').on('input', function() {
          const systolic = parseInt($('#bpSystolic').val());
          const diastolic = parseInt($('#bpDiastolic').val());
          
          if (systolic && diastolic) {
              let bpCategory = '';
              let alertClass = '';
              
              if (systolic >= 180 || diastolic >= 120) {
                  bpCategory = 'Hypertensive Crisis';
                  alertClass = 'text-danger';
              } else if (systolic >= 140 || diastolic >= 90) {
                  bpCategory = 'High Blood Pressure';
                  alertClass = 'text-warning';
              } else if (systolic >= 120 || diastolic >= 80) {
                  bpCategory = 'Elevated';
                  alertClass = 'text-info';
              } else {
                  bpCategory = 'Normal';
                  alertClass = 'text-success';
              }
              
              if ($('#bpCategory').length === 0) {
                  $('#bpDiastolic').parent().parent().append(`<small id="bpCategory" class="${alertClass}">${bpCategory}</small>`);
              } else {
                  $('#bpCategory').attr('class', alertClass).text(bpCategory);
              }
          }
      });

      // Tab switching handlers
      $('a[href="#tab_triage_list"]').on('shown.bs.tab', function() {
          loadTriageList();
      });
      
      $('a[href="#tab_department_routing"]').on('shown.bs.tab', function() {
          loadRoutingList();
          loadDepartmentQueues();
      });
  }

  // =============================================
  // MODAL RESET HANDLERS
  // =============================================

  $('#triageModal').on('hidden.bs.modal', function() {
      $('#triageForm')[0].reset();
      $('#modalTitle').text('New Triage Assessment');
      $('#modalTriageId').val('');
      $('#modalVerificationDetails').html('');
      $('#bmiDisplay, #bpCategory').remove();
      $('#routingPreview').html('');
  });

  $('#addPatientModal').on('hidden.bs.modal', function() {
      $('#addPatientForm')[0].reset();
  });

  // =============================================
  // GLOBAL HELPER FUNCTIONS
  // =============================================

  window.retrySecuGenConnection = function() {
      $('#diagnosticResults').html('');
      logDiagnostic("Retrying SecuGen service detection for triage...", 'info');
      initializeSecuGenService();
  };

  window.openCertificatePage = function() {
      window.open('https://localhost:8443/SGIFPGetInfo', '_blank');
      logDiagnostic("Opened certificate acceptance page", 'info');
      showMessage('Accept the security certificate and then retry connection', 'info');
  };

 window.viewTriage = function(triageId) {
       if (!triageId) {
           showMessage('Invalid triage ID', 'error');
           return;
       }
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/viewTriage"); ?>',
           type: 'POST',
           data: { triage_id: triageId },
           dataType: 'json',
           beforeSend: function() {
               showMessage('Loading triage details...', 'info');
           },
           success: function(response) {
               if (response.status === 'success') {
                   displayTriageDetails(response.data);
               } else {
                   showMessage(response.message || 'Failed to load triage details', 'error');
               }
           },
           error: function(xhr, status, error) {
               showMessage(`Error loading triage details: ${error}`, 'error');
           }
       });
   };

   window.editTriage = function(triageId) {
       if (!triageId) {
           showMessage('Invalid triage ID', 'error');
           return;
       }
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/editTriage"); ?>',
           type: 'POST',
           data: { triage_id: triageId },
           dataType: 'json',
           beforeSend: function() {
               showMessage('Loading triage for editing...', 'info');
           },
           success: function(response) {
               if (response.status === 'success') {
                   populateTriageEditForm(response.data);
                   $('#triageModal').modal('show');
               } else {
                   showMessage(response.message || 'Failed to load triage for editing', 'error');
               }
           },
           error: function(xhr, status, error) {
               showMessage(`Error loading triage for editing: ${error}`, 'error');
           }
       });
   };

   window.viewRouting = function(routingId) {
       if (!routingId) {
           showMessage('Invalid routing ID', 'error');
           return;
       }
       
       $.ajax({
           url: '<?php echo base_url("admin/triage/viewRouting"); ?>',
           type: 'POST',
           data: { routing_id: routingId },
           dataType: 'json',
           beforeSend: function() {
               showMessage('Loading routing details...', 'info');
           },
           success: function(response) {
               if (response.status === 'success') {
                   displayRoutingDetails(response.data);
               } else {
                   showMessage(response.message || 'Failed to load routing details', 'error');
               }
           },
           error: function(xhr, status, error) {
               showMessage(`Error loading routing details: ${error}`, 'error');
           }
       });
   };

  window.updateRoutingStatus = function(routingId, status) {
      console.log('Update routing status:', routingId, status);
      
      $.ajax({
          url: '<?php echo base_url("admin/triage/updateRoutingStatus"); ?>',
          type: 'POST',
          data: {
              routing_id: routingId,
              status: status
          },
          dataType: 'json',
          success: function(response) {
              if (response.status === 'success') {
                  showMessage('Routing status updated successfully', 'success');
                  loadRoutingList();
                  loadDepartmentQueues();
              } else {
                  showMessage(response.message || 'Failed to update routing status', 'error');
              }
          },
          error: function(xhr, status, error) {
              showMessage(`Error updating routing status: ${error}`, 'error');
          }
      });
  };

  // =============================================
  // DEBUG FUNCTIONS
  // =============================================

  window.debugTriageSystem = function() {
      console.log('=== Enhanced Triage System Debug Information ===');
      console.log('SecuGen Configuration:', window.secugenConfig);
      console.log('Triage State:', window.triageState);
      console.log('Service Active:', window.secugenConfig.serviceActive);
      console.log('Current Base URL:', window.secugenConfig.baseUrl);
      console.log('Selected Patient:', window.triageState.selectedPatientId);
      console.log('Verification Passed:', window.triageState.verificationPassed);
      console.log('Stored Template Size:', window.triageState.storedTemplate ? window.triageState.storedTemplate.length : 0);
      console.log('Live Template Size:', window.triageState.liveTemplate ? window.triageState.liveTemplate.length : 0);
      console.log('Last Match Score:', window.triageState.currentMatchScore);
      console.log('Verification History:', window.triageState.verificationHistory);
      console.log('Department Queues:', window.triageState.departmentQueues);
      
      // Test patient loading
      console.log('Testing patient loading for triage...');
      loadPatients();
  };

  window.testTriageRouting = function(patientId, department) {
      console.log(`Testing routing for patient ${patientId} to ${department}`);
      if (patientId && department) {
          routePatient(patientId, null, department, 'normal', 'Test routing from debug');
      } else {
          console.log('Please provide patientId and department parameters');
      }
  };

  window.testSecuGenCapture = function() {
      console.log('Testing SecuGen capture for triage...');
      captureFingerprint();
  };

  window.testSecuGenMatching = function() {
      if (!window.triageState.storedTemplate || !window.triageState.liveTemplate) {
          console.log('Both stored and live templates required for matching test');
          return;
      }
      
      console.log('Testing SecuGen matching for triage...');
      performBiometricVerification();
  };

  // =============================================
  // INITIALIZATION SEQUENCE
  // =============================================

  try {
      logDiagnostic("Starting Enhanced Triage System with SecuGen WebAPI", 'info');
      
      // Initialize SecuGen service
      initializeSecuGenService();
      
      // Load patients and set up events
      loadPatients();
      initializeEvents();
      
      // Load department queues
      loadDepartmentQueues();
      
      // Set up tab handlers
      $('a[href="#tab_triage_list"]').on('shown.bs.tab', function() {
          loadTriageList();
      });
      
      $('a[href="#tab_department_routing"]').on('shown.bs.tab', function() {
          loadRoutingList();
          loadDepartmentQueues();
      });
      
      console.log('🩺 Enhanced Triage System with Patient Routing Ready');
      logDiagnostic("Triage system initialization completed successfully", 'success');
      
      setTimeout(function() {
          showMessage('Enhanced Triage system ready - Capture, Verify and Patient Routing available', 'success');
      }, 1000);
      
  } catch (error) {
      console.error('Triage system initialization failed:', error);
      showMessage(`Failed to initialize triage system: ${error.message}`, 'error');
      logDiagnostic(`Triage system initialization failed: ${error.message}`, 'error');
  }

}); // End of $(document).ready

// =============================================
// CONSOLE HELP AND DOCUMENTATION
// =============================================

console.log(`
=== Enhanced Triage System with Advanced Patient Routing ===

🩺 COMPREHENSIVE TRIAGE FEATURES:
✓ SecuGen WebAPI fingerprint verification integration
✓ Advanced vital signs assessment with real-time validation
✓ Automated department routing based on triage priority
✓ Patient journey tracking across all departments
✓ BMI calculation and blood pressure categorization
✓ Pain scale assessment (0-10) with clinical correlation
✓ Real-time department queue monitoring
✓ Comprehensive routing history and analytics

🏥 DEPARTMENT ROUTING CAPABILITIES:
✓ Automatic routing to: Emergency, Consultation, Pharmacy, Laboratory, Radiology, IPD
✓ Priority-based routing logic (1=Emergency, 2-3=Urgent Care, 4-5=Standard)
✓ Queue management for each department
✓ Routing status tracking (pending, in_progress, completed)
✓ Department-specific record creation (OPD appointments, lab orders, etc.)

📊 ANALYTICS & REPORTING:
✓ Real-time department statistics
✓ Triage priority distribution analysis
✓ Department completion rates
✓ Patient flow analytics
✓ Verification audit trails

📋 AVAILABLE DEBUG COMMANDS:
- debugTriageSystem() - Show comprehensive system status
- testTriageRouting(patientId, department) - Test routing functionality
- testSecuGenCapture() - Test fingerprint capture
- testSecuGenMatching() - Test template matching
- retrySecuGenConnection() - Retry service detection

🔧 SYSTEM REQUIREMENTS:
1. SecuGen WebAPI installed and running on port 8443
2. SSL certificate accepted at https://localhost:8443/SGIFPGetInfo
3. Fingerprint scanner connected (Hamster Pro recommended)
4. Proper database tables for triage, patient_routing, and department queues

💡 ROUTING LOGIC:
- Priority 1 (Red) → Emergency Department (immediate)
- Priority 2 (Orange) → Emergency or Urgent Care
- Priority 3 (Yellow) → Consultation/OPD
- Priority 4 (Green) → Standard Consultation
- Priority 5 (Blue) → Fast Track/Minor Procedures

The system automatically creates appropriate records in destination departments
and maintains comprehensive audit trails for all patient movements.
`);
</script>   