<?php
/**
 * Integrated Patient Billing Query View
 * Path: application/views/admin/patient/billing_query.php
 * JavaScript integrated directly into the page for seamless operation
 */
?>

<div class="content-wrapper">
    <section class="content-header">
        <div class="container-fluid">
            <div class="row mb-2">
                <div class="col-sm-6">
                    <h1><i class="fas fa-receipt mr-2"></i><?php echo $this->lang->line('patient_billing_query'); ?></h1>
                </div>
                <div class="col-sm-6">
                    <ol class="breadcrumb float-sm-right">
                        <li class="breadcrumb-item"><a href="<?php echo base_url(); ?>admin/admin/dashboard"><?php echo $this->lang->line('dashboard'); ?></a></li>
                        <li class="breadcrumb-item active"><?php echo $this->lang->line('patient_billing_query'); ?></li>
                    </ol>
                </div>
            </div>
        </div>
    </section>

    <section class="content">
        <div class="container-fluid">
            
            <!-- Enhanced Search Section with Live Suggestions -->
            <div class="row">
                <div class="col-md-12">
                    <div class="card card-primary">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-search mr-2"></i><?php echo $this->lang->line('patient_search'); ?>
                            </h3>
                            <div class="card-tools">
                                <button class="btn btn-tool" type="button" data-card-widget="collapse">
                                    <i class="fas fa-minus"></i>
                                </button>
                            </div>
                        </div>
                        <div class="card-body">
                            <form id="patientSearchForm" class="row">
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="patient_id"><?php echo $this->lang->line('patient_id'); ?></label>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text"><i class="fas fa-id-card"></i></span>
                                            </div>
                                            <input type="text" class="form-control" id="patient_id" placeholder="<?php echo $this->lang->line('enter_patient_id'); ?>">
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="patient_name"><?php echo $this->lang->line('patient_name'); ?></label>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text"><i class="fas fa-user"></i></span>
                                            </div>
                                            <input type="text" class="form-control" id="patient_name" placeholder="<?php echo $this->lang->line('enter_patient_name'); ?>">
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="mobile_no"><?php echo $this->lang->line('mobile_number'); ?></label>
                                        <div class="input-group">
                                            <div class="input-group-prepend">
                                                <span class="input-group-text"><i class="fas fa-phone"></i></span>
                                            </div>
                                            <input type="text" class="form-control" id="mobile_no" placeholder="<?php echo $this->lang->line('enter_mobile_number'); ?>">
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-12">
                                    <button type="submit" class="btn btn-primary mr-2">
                                        <i class="fas fa-search mr-1"></i><?php echo $this->lang->line('search_patient'); ?>
                                    </button>
                                    <button type="button" class="btn btn-secondary mr-2" id="clearSearch">
                                        <i class="fas fa-times mr-1"></i><?php echo $this->lang->line('clear'); ?>
                                    </button>
                                    <button type="button" class="btn btn-info" id="advancedSearchToggle">
                                        <i class="fas fa-filter mr-1"></i>Advanced Search
                                    </button>
                                </div>
                            </form>

                            <!-- Live Search Suggestions Container -->
                            <div id="searchSuggestions" style="display: none;"></div>

                            <!-- Advanced Search Panel -->
                            <div id="advancedSearchPanel" style="display: none; margin-top: 20px;">
                                <div class="card card-secondary">
                                    <div class="card-header">
                                        <h3 class="card-title">Advanced Search Options</h3>
                                    </div>
                                    <div class="card-body">
                                        <div class="row">
                                            <div class="col-md-3">
                                                <div class="form-group">
                                                    <label>Date Range From</label>
                                                    <input type="date" class="form-control" id="date_from">
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="form-group">
                                                    <label>Date Range To</label>
                                                    <input type="date" class="form-control" id="date_to">
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="form-group">
                                                    <label>Payment Status</label>
                                                    <select class="form-control" id="payment_status_filter">
                                                        <option value="">All Statuses</option>
                                                        <option value="paid">Paid</option>
                                                        <option value="pending">Pending</option>
                                                        <option value="partial">Partial</option>
                                                    </select>
                                                </div>
                                            </div>
                                            <div class="col-md-3">
                                                <div class="form-group">
                                                    <label>Department</label>
                                                    <select class="form-control" id="department_filter">
                                                        <option value="">All Departments</option>
                                                        <option value="consultation">Consultation</option>
                                                        <option value="laboratory">Laboratory</option>
                                                        <option value="radiology">Radiology</option>
                                                        <option value="pharmacy">Pharmacy</option>
                                                        <option value="ipd">IPD</option>
                                                    </select>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Patient Information Section -->
            <div class="row" id="patientInfoSection" style="display: none;">
                <div class="col-md-12">
                    <div class="card card-info">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-user mr-2"></i><?php echo $this->lang->line('patient_information'); ?>
                            </h3>
                            <div class="card-tools">
                                <span id="lastUpdated" class="badge badge-light"></span>
                            </div>
                        </div>
                        <div class="card-body">
                            <div class="row">
                                <div class="col-md-3">
                                    <div class="text-center">
                                        <img id="patient_photo" src="<?php echo base_url('uploads/patient_images/no_image.png'); ?>" 
                                             alt="Patient Photo" class="img-circle img-fluid" style="max-width: 120px; border: 3px solid #ddd;">
                                        <h5 id="display_patient_name" class="mt-2">-</h5>
                                        <p class="text-muted" id="display_patient_id">-</p>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <table class="table table-borderless">
                                        <tr>
                                            <td><strong><i class="fas fa-birthday-cake text-primary"></i> <?php echo $this->lang->line('age'); ?>:</strong></td>
                                            <td><span id="display_age">-</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong><i class="fas fa-venus-mars text-info"></i> <?php echo $this->lang->line('gender'); ?>:</strong></td>
                                            <td><span id="display_gender">-</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong><i class="fas fa-phone text-success"></i> <?php echo $this->lang->line('mobile'); ?>:</strong></td>
                                            <td><span id="display_mobile">-</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong><i class="fas fa-envelope text-warning"></i> <?php echo $this->lang->line('email'); ?>:</strong></td>
                                            <td><span id="display_email">-</span></td>
                                        </tr>
                                    </table>
                                </div>
                                <div class="col-md-3">
                                    <table class="table table-borderless">
                                        <tr>
                                            <td><strong><i class="fas fa-tint text-danger"></i> <?php echo $this->lang->line('blood_group'); ?>:</strong></td>
                                            <td><span id="display_blood_group">-</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong><i class="fas fa-map-marker-alt text-secondary"></i> <?php echo $this->lang->line('address'); ?>:</strong></td>
                                            <td><span id="display_address">-</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong><i class="fas fa-clock text-info"></i> Last Visit:</strong></td>
                                            <td><span id="display_last_visit">-</span></td>
                                        </tr>
                                        <tr>
                                            <td><strong><i class="fas fa-calendar text-primary"></i> Registration:</strong></td>
                                            <td><span id="display_registration_date">-</span></td>
                                        </tr>
                                    </table>
                                </div>
                                <div class="col-md-2">
                                    <div class="text-center">
                                        <div class="info-box bg-light">
                                            <span class="info-box-icon"><i class="fas fa-history"></i></span>
                                            <div class="info-box-content">
                                                <span class="info-box-text">Total Visits</span>
                                                <span class="info-box-number" id="total_visits">0</span>
                                            </div>
                                        </div>
                                        <button class="btn btn-sm btn-outline-primary btn-block" id="viewPatientHistory">
                                            <i class="fas fa-eye"></i> View History
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Enhanced Billing Summary Cards -->
            <div class="row" id="billingSummarySection" style="display: none;">
                <div class="col-lg-2 col-6">
                    <div class="small-box bg-primary">
                        <div class="inner text-center">
                            <h4 id="consultation_total">$0.00</h4>
                            <p><?php echo $this->lang->line('consultation'); ?></p>
                            <small id="consultation_status"><?php echo $this->lang->line('no_bills'); ?></small>
                        </div>
                        <div class="icon">
                            <i class="fas fa-stethoscope"></i>
                        </div>
                        <div class="small-box-footer">
                            <span id="consultation_count">0 bills</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-2 col-6">
                    <div class="small-box bg-success">
                        <div class="inner text-center">
                            <h4 id="laboratory_total">$0.00</h4>
                            <p><?php echo $this->lang->line('laboratory'); ?></p>
                            <small id="laboratory_status"><?php echo $this->lang->line('no_bills'); ?></small>
                        </div>
                        <div class="icon">
                            <i class="fas fa-vials"></i>
                        </div>
                        <div class="small-box-footer">
                            <span id="laboratory_count">0 bills</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-2 col-6">
                    <div class="small-box bg-info">
                        <div class="inner text-center">
                            <h4 id="radiology_total">$0.00</h4>
                            <p><?php echo $this->lang->line('radiology'); ?></p>
                            <small id="radiology_status"><?php echo $this->lang->line('no_bills'); ?></small>
                        </div>
                        <div class="icon">
                            <i class="fas fa-x-ray"></i>
                        </div>
                        <div class="small-box-footer">
                            <span id="radiology_count">0 bills</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-2 col-6">
                    <div class="small-box bg-warning">
                        <div class="inner text-center">
                            <h4 id="pharmacy_total">$0.00</h4>
                            <p><?php echo $this->lang->line('pharmacy'); ?></p>
                            <small id="pharmacy_status"><?php echo $this->lang->line('no_bills'); ?></small>
                        </div>
                        <div class="icon">
                            <i class="fas fa-pills"></i>
                        </div>
                        <div class="small-box-footer">
                            <span id="pharmacy_count">0 bills</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-2 col-6">
                    <div class="small-box bg-danger">
                        <div class="inner text-center">
                            <h4 id="ipd_total">$0.00</h4>
                            <p><?php echo $this->lang->line('ipd'); ?></p>
                            <small id="ipd_status"><?php echo $this->lang->line('no_bills'); ?></small>
                        </div>
                        <div class="icon">
                            <i class="fas fa-bed"></i>
                        </div>
                        <div class="small-box-footer">
                            <span id="ipd_count">0 bills</span>
                        </div>
                    </div>
                </div>
                
                <div class="col-lg-2 col-6">
                    <div class="small-box bg-dark">
                        <div class="inner text-center">
                            <h4 id="grand_total">$0.00</h4>
                            <p><?php echo $this->lang->line('total'); ?></p>
                            <small id="overall_status"><?php echo $this->lang->line('no_bills'); ?></small>
                        </div>
                        <div class="icon">
                            <i class="fas fa-calculator"></i>
                        </div>
                        <div class="small-box-footer">
                            <button class="btn btn-sm btn-light" id="exportSummary">
                                <i class="fas fa-download"></i> Export
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Quick Actions Row -->
            <div class="row" id="quickActionsSection" style="display: none;">
                <div class="col-md-12">
                    <div class="card">
                        <div class="card-body">
                            <div class="btn-group" role="group">
                                <button class="btn btn-primary" id="makePaymentBtn">
                                    <i class="fas fa-credit-card mr-1"></i>Make Payment
                                </button>
                                <button class="btn btn-success" id="printAllBills">
                                    <i class="fas fa-print mr-1"></i>Print All Bills
                                </button>
                                <button class="btn btn-info" id="emailStatement">
                                    <i class="fas fa-envelope mr-1"></i>Email Statement
                                </button>
                                <button class="btn btn-warning" id="exportExcel">
                                    <i class="fas fa-file-excel mr-1"></i>Export Excel
                                </button>
                                <button class="btn btn-secondary" id="refreshData">
                                    <i class="fas fa-sync mr-1"></i>Refresh
                                </button>
                            </div>
                            <div class="float-right">
                                <div class="input-group" style="width: 300px;">
                                    <input type="text" class="form-control" placeholder="Search in bills..." id="billSearchInput">
                                    <div class="input-group-append">
                                        <button class="btn btn-outline-secondary" type="button" id="searchInBills">
                                            <i class="fas fa-search"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Enhanced Detailed Billing Table -->
            <div class="row" id="billingDetailsSection" style="display: none;">
                <div class="col-md-12">
                    <div class="card">
                        <div class="card-header">
                            <h3 class="card-title">
                                <i class="fas fa-table mr-2"></i><?php echo $this->lang->line('detailed_billing_information'); ?>
                            </h3>
                            <div class="card-tools">
                                <div class="btn-group">
                                    <button class="btn btn-primary btn-sm" id="printBill">
                                        <i class="fas fa-print mr-1"></i><?php echo $this->lang->line('print'); ?>
                                    </button>
                                    <button class="btn btn-success btn-sm" id="exportBill">
                                        <i class="fas fa-download mr-1"></i><?php echo $this->lang->line('export'); ?>
                                    </button>
                                    <button class="btn btn-info btn-sm" id="toggleColumns">
                                        <i class="fas fa-columns mr-1"></i>Columns
                                    </button>
                                </div>
                            </div>
                        </div>
                        <div class="card-body">
                            <!-- Filters Row -->
                            <div class="row mb-3">
                                <div class="col-md-3">
                                    <select class="form-control form-control-sm" id="moduleFilter">
                                        <option value="">All Modules</option>
                                        <option value="consultation">Consultation</option>
                                        <option value="laboratory">Laboratory</option>
                                        <option value="radiology">Radiology</option>
                                        <option value="pharmacy">Pharmacy</option>
                                        <option value="ipd">IPD</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <select class="form-control form-control-sm" id="statusFilter">
                                        <option value="">All Statuses</option>
                                        <option value="paid">Paid</option>
                                        <option value="pending">Pending</option>
                                        <option value="partial">Partial</option>
                                    </select>
                                </div>
                                <div class="col-md-3">
                                    <input type="date" class="form-control form-control-sm" id="dateFilter" placeholder="Filter by date">
                                </div>
                                <div class="col-md-3">
                                    <button class="btn btn-sm btn-secondary" id="clearFilters">
                                        <i class="fas fa-times"></i> Clear Filters
                                    </button>
                                </div>
                            </div>

                            <div class="table-responsive">
                                <table class="table table-bordered table-striped table-hover" id="billingTable">
                                    <thead class="bg-light">
                                        <tr>
                                            <th><input type="checkbox" id="selectAll"></th>
                                            <th><?php echo $this->lang->line('date'); ?> <i class="fas fa-sort"></i></th>
                                            <th><?php echo $this->lang->line('module'); ?></th>
                                            <th><?php echo $this->lang->line('bill_no'); ?></th>
                                            <th><?php echo $this->lang->line('description'); ?></th>
                                            <th><?php echo $this->lang->line('amount'); ?> <i class="fas fa-sort"></i></th>
                                            <th><?php echo $this->lang->line('paid'); ?></th>
                                            <th><?php echo $this->lang->line('pending'); ?></th>
                                            <th><?php echo $this->lang->line('status'); ?></th>
                                            <th class="no-print"><?php echo $this->lang->line('actions'); ?></th>
                                        </tr>
                                    </thead>
                                    <tbody id="billingTableBody">
                                        <!-- Dynamic content will be loaded here -->
                                    </tbody>
                                    <tfoot class="bg-light">
                                        <tr class="font-weight-bold">
                                            <td></td>
                                            <td colspan="4"><strong><?php echo strtoupper($this->lang->line('total')); ?></strong></td>
                                            <td><strong id="total_amount">$0.00</strong></td>
                                            <td><strong id="total_paid">$0.00</strong></td>
                                            <td><strong id="total_pending">$0.00</strong></td>
                                            <td colspan="2"></td>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>

                            <!-- Pagination -->
                            <div class="row mt-3">
                                <div class="col-md-6">
                                    <div class="dataTables_info">
                                        Showing <span id="showing_start">0</span> to <span id="showing_end">0</span> of <span id="total_records">0</span> entries
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <nav>
                                        <ul class="pagination pagination-sm float-right" id="pagination">
                                            <!-- Pagination will be generated here -->
                                        </ul>
                                    </nav>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Print View (Hidden) -->
            <div class="d-none" id="printContent">
                <div class="text-center mb-4">
                    <h2><?php echo $this->session->userdata('hospital_name') ?? 'HEALTHCARE FACILITY'; ?></h2>
                    <p><?php echo $this->lang->line('patient_billing_statement'); ?></p>
                    <hr>
                </div>
                <div id="printPatientInfo"></div>
                <div id="printBillingDetails"></div>
                <div class="text-center mt-4">
                    <p><small>Generated on: <span id="print_timestamp"></span></small></p>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Payment Processing Modal -->
<div class="modal fade" id="paymentModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-success text-white">
                <h4 class="modal-title">
                    <i class="fas fa-credit-card mr-2"></i>Process Payment
                </h4>
                <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
            </div>
            <form id="paymentForm">
                <div class="modal-body">
                    <div class="alert alert-info">
                        <div class="row">
                            <div class="col-md-8">
                                <strong>Patient:</strong> <span id="paymentPatientName"></span><br>
                                <strong>Bill No:</strong> <span id="paymentBillNo"></span>
                            </div>
                            <div class="col-md-4 text-right">
                                <strong>Outstanding:</strong><br>
                                <h4 class="text-primary">$<span id="outstandingAmount">0.00</span></h4>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Payment Amount <span class="text-danger">*</span></label>
                                <div class="input-group">
                                    <div class="input-group-prepend">
                                        <span class="input-group-text">$</span>
                                    </div>
                                    <input type="number" name="payment_amount" id="paymentAmount" class="form-control" step="0.01" min="0" required>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Payment Method <span class="text-danger">*</span></label>
                                <select name="payment_method" id="paymentMethod" class="form-control" required>
                                    <option value="">Select Payment Method</option>
                                    <option value="Cash">Cash</option>
                                    <option value="Credit Card">Credit Card</option>
                                    <option value="Debit Card">Debit Card</option>
                                    <option value="Mobile Money">Mobile Money</option>
                                    <option value="Bank Transfer">Bank Transfer</option>
                                    <option value="Check">Check</option>
                                    <option value="Insurance">Insurance</option>
                                </select>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Reference Number</label>
                                <input type="text" name="reference_number" id="referenceNumber" class="form-control" placeholder="Transaction/Reference number">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label>Payment Date</label>
                                <input type="date" name="payment_date" id="paymentDate" class="form-control" value="<?php echo date('Y-m-d'); ?>">
                            </div>
                        </div>
                    </div>
                    
                    <div class="form-group">
                        <label>Notes</label>
                        <textarea name="payment_notes" id="paymentFormNotes" class="form-control" rows="3" placeholder="Payment notes..."></textarea>
                    </div>
                    
                    <!-- Payment Summary -->
                    <div class="card bg-light">
                        <div class="card-body">
                            <h6>Payment Summary</h6>
                            <div class="row">
                                <div class="col-md-6">
                                    <small><strong>Original Amount:</strong> $<span id="originalAmount">0.00</span></small><br>
                                    <small><strong>Previous Payments:</strong> $<span id="previousPayments">0.00</span></small>
                                </div>
                                <div class="col-md-6">
                                    <small><strong>Current Payment:</strong> $<span id="currentPayment">0.00</span></small><br>
                                    <small><strong>Remaining Balance:</strong> $<span id="remainingBalance">0.00</span></small>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <input type="hidden" name="bill_id" id="paymentBillId">
                    <input type="hidden" name="patient_id" id="paymentPatientId">
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">
                        <i class="fas fa-times mr-1"></i>Cancel
                    </button>
                    <button type="submit" class="btn btn-success">
                        <i class="fas fa-credit-card mr-1"></i>Process Payment
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Bill Details Modal -->
<div class="modal fade" id="billDetailsModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-info text-white">
                <h4 class="modal-title">
                    <i class="fas fa-receipt mr-2"></i>Bill Details
                </h4>
                <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
            </div>
            <div class="modal-body" id="billDetailsContent">
                <!-- Content will be loaded dynamically -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-dismiss="modal">
                    <i class="fas fa-times mr-1"></i>Close
                </button>
                <button type="button" class="btn btn-primary" id="printBillDetails">
                    <i class="fas fa-print mr-1"></i>Print
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Patient Selection Modal -->
<div class="modal fade" id="patientSelectionModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary text-white">
                <h4 class="modal-title">
                    <i class="fas fa-users mr-2"></i>Select Patient
                </h4>
                <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
            </div>
            <div class="modal-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead>
                            <tr>
                                <th>Patient ID</th>
                                <th>Name</th>
                                <th>Age</th>
                                <th>Mobile</th>
                                <th>Last Visit</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody id="patientSelectionTableBody">
                            <!-- Dynamic content -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Patient History Modal -->
<div class="modal fade" id="patientHistoryModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-xl" role="document">
        <div class="modal-content">
            <div class="modal-header bg-secondary text-white">
                <h4 class="modal-title">
                    <i class="fas fa-history mr-2"></i>Patient History
                </h4>
                <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
            </div>
            <div class="modal-body" id="patientHistoryContent">
                <!-- Patient history content -->
            </div>
        </div>
    </div>
</div>

<!-- Enhanced CSS for Integrated Billing -->
<style>
/* Custom styling for integrated billing system */
.billing-summary-card {
    border-left: 4px solid #007bff;
    transition: all 0.3s ease;
}

.billing-summary-card:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}

.status-badge {
    font-size: 0.85em;
    padding: 0.4em 0.8em;
}

.small-box {
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
    transition: transform 0.2s;
}

.small-box:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.15);
}

.small-box .inner h4 {
    font-size: 1.5rem;
    font-weight: bold;
    margin-bottom: 5px;
}

.small-box .inner small {
    font-size: 0.8rem;
    opacity: 0.8;
}

#searchSuggestions {
    position: absolute;
    top: 100%;
    left: 0;
    right: 0;
    z-index: 1000;
    max-height: 300px;
    overflow-y: auto;
    border: 1px solid #ddd;
    border-radius: 4px;
    background: white;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.suggestion-item {
    padding: 12px;
    border-bottom: 1px solid #eee;
    cursor: pointer;
    transition: background-color 0.2s;
}

.suggestion-item:hover {
    background-color: #f8f9fa;
}

.suggestion-item:last-child {
    border-bottom: none;
}

.table-responsive {
    border-radius: 0.25rem;
}

.btn-group-sm > .btn, .btn-sm {
    padding: 0.25rem 0.5rem;
    font-size: 0.875rem;
}

@media print {
    .no-print { 
        display: none !important; 
    }
    .content-wrapper {
        margin: 0 !important;
        padding: 0 !important;
    }
    .card {
        border: 1px solid #000 !important;
        box-shadow: none !important;
    }
    .table th {
        background-color: #f8f9fa !important;
        color: #000 !important;
    }
    .small-box {
        page-break-inside: avoid;
    }
}

.info-box {
    box-shadow: 0 0 1px rgba(0,0,0,.125), 0 1px 3px rgba(0,0,0,.2);
    border-radius: 0.25rem;
    background-color: #fff;
    display: flex;
    margin-bottom: 1rem;
    min-height: 80px;
    padding: 0.5rem;
    position: relative;
    width: 100%;
}

.info-box .info-box-icon {
    border-radius: 0.25rem;
    align-items: center;
    display: flex;
    font-size: 1.875rem;
    justify-content: center;
    text-align: center;
    width: 70px;
}

.info-box .info-box-content {
    display: flex;
    flex-direction: column;
    justify-content: center;
    line-height: 1.8;
    margin-left: 0.5rem;
    padding: 0 0.5rem;
}

.card-tools .btn-group .btn {
    margin-left: 2px;
}

.pagination-sm .page-link {
    padding: 0.25rem 0.5rem;
    font-size: 0.875rem;
}

/* Loading overlay */
.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: flex;
    justify-content: center;
    align-items: center;
    z-index: 9999;
}

.loading-content {
    background: white;
    padding: 2rem;
    border-radius: 8px;
    text-align: center;
    box-shadow: 0 4px 6px rgba(0,0,0,0.1);
}

.spinner-border {
    width: 3rem;
    height: 3rem;
}

/* Alert styling */
.alert {
    border: none;
    border-radius: 0.375rem;
    margin-bottom: 1rem;
}

.alert-dismissible .close {
    padding: 0.75rem 1.25rem;
    color: inherit;
}

/* Table enhancements */
.table th {
    border-top: none;
    font-weight: 600;
    color: #495057;
    background-color: #f8f9fa;
}

.table-hover tbody tr:hover {
    background-color: rgba(0,0,0,0.075);
}

.badge {
    font-size: 0.75em;
    font-weight: 600;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .small-box .inner h4 {
        font-size: 1.2rem;
    }
    
    .card-tools {
        margin-top: 0.5rem;
    }
    
    .table-responsive {
        font-size: 0.875rem;
    }
    
    .btn-group {
        flex-direction: column;
        width: 100%;
    }
    
    .btn-group .btn {
        margin-bottom: 0.25rem;
    }
}

/* Form enhancements */
.input-group-text {
    background-color: #e9ecef;
    border-color: #ced4da;
}

.form-control:focus {
    border-color: #80bdff;
    box-shadow: 0 0 0 0.2rem rgba(0,123,255,0.25);
}

/* Button styling */
.btn {
    border-radius: 0.375rem;
    font-weight: 400;
}

.btn-outline-primary:hover {
    background-color: #007bff;
    border-color: #007bff;
}

/* Card hover effects */
.card {
    transition: box-shadow 0.2s ease;
}

.card:hover {
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}
</style>

<script type="text/javascript">
'use strict';

$(document).ready(function() {
    console.log('🏥 Enhanced Patient Billing System Starting...');
    
    // =============================================
    // GLOBAL CONFIGURATION
    // =============================================
    
    window.billingConfig = {
        apiBaseUrl: '<?php echo base_url("admin/patient_billing/"); ?>',
        currentPatient: null,
        currentBillingData: null,
        searchResults: [],
        currentPage: 1,
        recordsPerPage: 20,
        totalRecords: 0,
        filters: {
            module: '',
            status: '',
            date: '',
            search: ''
        }
    };

    // Language translations for JavaScript
    window.langTranslations = {
        'search_patient': '<?php echo $this->lang->line("search_patient"); ?>',
        'clear': '<?php echo $this->lang->line("clear"); ?>',
        'loading': '<?php echo $this->lang->line("loading"); ?>',
        'patient_not_found': '<?php echo $this->lang->line("patient_not_found"); ?>',
        'error_occurred': '<?php echo $this->lang->line("error_occurred"); ?>',
        'payment_successful': '<?php echo $this->lang->line("payment_successful"); ?>',
        'no_patient_selected': '<?php echo $this->lang->line("no_patient_selected"); ?>',
        'paid': '<?php echo $this->lang->line("paid"); ?>',
        'pending': '<?php echo $this->lang->line("pending"); ?>',
        'partial': '<?php echo $this->lang->line("partial"); ?>',
        'no_bills': '<?php echo $this->lang->line("no_bills"); ?>',
        'view_details': '<?php echo $this->lang->line("view_details"); ?>',
        'make_payment': '<?php echo $this->lang->line("make_payment"); ?>',
        'process_payment': '<?php echo $this->lang->line("process_payment"); ?>',
        'cancel': '<?php echo $this->lang->line("cancel"); ?>',
        'close': '<?php echo $this->lang->line("close"); ?>',
        'print': '<?php echo $this->lang->line("print"); ?>',
        'export': '<?php echo $this->lang->line("export"); ?>',
        'select_patient': '<?php echo $this->lang->line("select_patient"); ?>',
        'payment_amount': '<?php echo $this->lang->line("payment_amount"); ?>',
        'payment_mode': '<?php echo $this->lang->line("payment_mode"); ?>',
        'notes': '<?php echo $this->lang->line("notes"); ?>',
        'cash': '<?php echo $this->lang->line("cash"); ?>',
        'credit_card': '<?php echo $this->lang->line("credit_card"); ?>',
        'debit_card': '<?php echo $this->lang->line("debit_card"); ?>',
        'bank_transfer': '<?php echo $this->lang->line("bank_transfer"); ?>',
        'check': '<?php echo $this->lang->line("check"); ?>',
        'mobile_money': '<?php echo $this->lang->line("mobile_money"); ?>'
    };

    // =============================================
    // UTILITY FUNCTIONS
    // =============================================
    
    function showMessage(message, type = 'info', duration = 5000) {
        const alertClass = type === 'success' ? 'alert-success' : 
                          type === 'warning' ? 'alert-warning' : 
                          type === 'error' ? 'alert-danger' : 'alert-info';
        
        const html = `
            <div class="alert ${alertClass} alert-dismissible fade show" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                <strong>${type.toUpperCase()}:</strong> ${message}
            </div>
        `;
        
        $('body').append(html);
        
        setTimeout(() => {
            $('.alert').fadeOut(() => $('.alert').remove());
        }, duration);
    }

    function showLoading(show) {
        if (show) {
            if ($('#loadingOverlay').length === 0) {
                $('body').append(`
                    <div id="loadingOverlay" class="loading-overlay">
                        <div class="loading-content">
                            <div class="spinner-border text-primary mb-3" role="status"></div>
                            <div>${window.langTranslations.loading}...</div>
                        </div>
                    </div>
                `);
            }
            $('#loadingOverlay').show();
        } else {
            $('#loadingOverlay').hide();
        }
    }

    function debounce(func, wait) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                clearTimeout(timeout);
                func(...args);
            };
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
        };
    }

    function formatCurrency(amount) {
        return 'Ksh.' + parseFloat(amount || 0).toFixed(2);
    }

    function formatDate(dateString) {
        if (!dateString) return '-';
        const date = new Date(dateString);
        return date.toLocaleDateString();
    }

    // =============================================
    // PATIENT SEARCH FUNCTIONS
    // =============================================

    function getSearchTerms() {
        const patientId = $('#patient_id').val().trim();
        const patientName = $('#patient_name').val().trim();
        const mobileNo = $('#mobile_no').val().trim();
        
        return patientId || patientName || mobileNo;
    }

    async function performLiveSearch() {
        const searchTerms = getSearchTerms();
        if (!searchTerms || searchTerms.length < 2) {
            hideSearchSuggestions();
            return;
        }

        try {
            const patients = await searchPatientsAPI(searchTerms);
            showSearchSuggestions(patients);
        } catch (error) {
            console.error('Live search error:', error);
        }
    }

    async function searchPatientsAPI(searchTerm) {
        const response = await fetch(`${window.billingConfig.apiBaseUrl}search_patients?term=${encodeURIComponent(searchTerm)}`);
        
        if (response.status === 401 || response.status === 403) {
            showMessage('Session expired or access denied. Please log in.', 'error');
            return [];
        }
        
        const data = await response.json();
        
        if (!data.success) {
            throw new Error(data.error || window.langTranslations.error_occurred);
        }
        
        return data.data;
    }

    function showSearchSuggestions(patients) {
        hideSearchSuggestions();

        if (patients.length === 0) return;

        const suggestionsHtml = patients.slice(0, 5).map(patient => `
            <div class="suggestion-item" onclick="selectPatientFromSuggestion(${patient.id})">
                <div class="d-flex justify-content-between">
                    <div>
                        <strong>${patient.patient_name}</strong> (${patient.patient_id || patient.id})<br>
                        <small class="text-muted">${patient.mobileno || 'No mobile'} | Age: ${patient.age || 'N/A'}</small>
                    </div>
                    <div class="text-right">
                        <small class="badge badge-info">${patient.gender || ''}</small>
                        ${patient.has_pending_bills ? '<small class="badge badge-warning">Pending Bills</small>' : ''}
                    </div>
                </div>
            </div>
        `).join('');

        $('#searchSuggestions').html(suggestionsHtml).show();
    }

    function hideSearchSuggestions() {
        $('#searchSuggestions').hide().html('');
    }

    window.selectPatientFromSuggestion = async function(patientId) {
        const patient = window.billingConfig.searchResults.find(p => p.id == patientId) || 
                        await getPatientById(patientId);
        
        if (patient) {
            $('#patient_id').val(patient.patient_id || patient.id);
            $('#patient_name').val(patient.patient_name);
            $('#mobile_no').val(patient.mobileno || '');
            
            hideSearchSuggestions();
            window.billingConfig.currentPatient = patient;
            displayPatientInfo(patient);
            await loadBillingData(patient.id);
        }
    };

    async function getPatientById(patientId) {
        try {
            const response = await fetch(`${window.billingConfig.apiBaseUrl}get_patient?id=${patientId}`);
            const data = await response.json();
            return data.success ? data.data : null;
        } catch (error) {
            console.error('Error getting patient:', error);
            return null;
        }
    }

    async function searchPatient() {
        const searchTerm = getSearchTerms();
        
        if (!searchTerm) {
            showMessage(window.langTranslations.no_patient_selected, 'warning');
            return;
        }

        try {
            showLoading(true);
            const patients = await searchPatientsAPI(searchTerm);
            window.billingConfig.searchResults = patients;
            
            if (patients.length === 0) {
                showMessage(window.langTranslations.patient_not_found, 'info');
                hideAllSections();
                return;
            }

            if (patients.length === 1) {
                window.billingConfig.currentPatient = patients[0];
                displayPatientInfo(patients[0]);
                await loadBillingData(patients[0].id);
            } else {
                showPatientSelectionModal(patients);
            }

        } catch (error) {
            showMessage(`${window.langTranslations.error_occurred}: ${error.message}`, 'error');
        } finally {
            showLoading(false);
            hideSearchSuggestions();
        }
    }

    function showPatientSelectionModal(patients) {
        const tbody = $('#patientSelectionTableBody');
        tbody.empty();
        
        patients.forEach(patient => {
            const row = $(`
                <tr>
                    <td>${patient.patient_id || patient.id}</td>
                    <td>${patient.patient_name}</td>
                    <td>${patient.age || 'N/A'}</td>
                    <td>${patient.mobileno || '-'}</td>
                    <td>${formatDate(patient.last_visit)}</td>
                    <td>
                        <button class="btn btn-primary btn-sm" onclick="selectPatient(${patient.id})">
                            ${window.langTranslations.select_patient}
                        </button>
                    </td>
                </tr>
            `);
            tbody.append(row);
        });
        
        $('#patientSelectionModal').modal('show');
    }

    window.selectPatient = async function(patientId) {
        const patient = window.billingConfig.searchResults.find(p => p.id == patientId);
        if (patient) {
            window.billingConfig.currentPatient = patient;
            displayPatientInfo(patient);
            await loadBillingData(patient.id);
            
            $('#patientSelectionModal').modal('hide');
        }
    };

    // =============================================
    // PATIENT DISPLAY FUNCTIONS
    // =============================================

    function displayPatientInfo(patient) {
        $('#display_patient_name').text(patient.patient_name);
        $('#display_patient_id').text(patient.patient_id || patient.id);
        $('#display_age').text(patient.age || 'N/A');
        $('#display_gender').text(patient.gender || 'N/A');
        $('#display_mobile').text(patient.mobileno || '-');
        $('#display_email').text(patient.email || '-');
        $('#display_blood_group').text(patient.blood_group || '-');
        $('#display_address').text(patient.address || '-');
        $('#display_last_visit').text(formatDate(patient.last_visit));
        $('#display_registration_date').text(formatDate(patient.created_at || patient.admission_date));
        $('#total_visits').text(patient.total_visits || '0');
        
        if (patient.image) {
            $('#patient_photo').attr('src', '<?php echo base_url(); ?>' + patient.image);
        }

        $('#lastUpdated').text('Updated: ' + new Date().toLocaleTimeString());
        $('#patientInfoSection').show();
    }

    // =============================================
    // BILLING DATA FUNCTIONS
    // =============================================

    async function loadBillingData(patientId) {
        try {
            showLoading(true);
            
            const response = await fetch(`${window.billingConfig.apiBaseUrl}get_billing?patient_id=${patientId}`);
            if (response.status === 401 || response.status === 403) {
                showMessage('Session expired or access denied. Please log in.', 'error');
                return;
            }
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || window.langTranslations.error_occurred);
            }

            window.billingConfig.currentBillingData = data.data;
            updateSummaryCards(data.data.summary);
            updateBillingTable(data.data.bills);
            
            $('#billingSummarySection').show();
            $('#quickActionsSection').show();
            $('#billingDetailsSection').show();

            setupNotifications();

        } catch (error) {
            showMessage(`${window.langTranslations.error_occurred}: ${error.message}`, 'error');
        } finally {
            showLoading(false);
        }
    }

    function updateSummaryCards(summary) {
        const modules = ['consultation', 'laboratory', 'radiology', 'pharmacy', 'ipd'];
        
        modules.forEach(module => {
            const data = summary.modules?.[module] || { total: 0, status: window.langTranslations.no_bills, count: 0 };
            $(`#${module}_total`).text(formatCurrency(data.total));
            $(`#${module}_status`).text(data.status);
            $(`#${module}_count`).text(`${data.count} bills`);
        });

        $('#grand_total').text(formatCurrency(summary.total_amount || 0));
        $('#overall_status').text(getOverallStatus(summary));
    }

    function getOverallStatus(summary) {
        if (!summary || summary.total_amount === 0) return window.langTranslations.no_bills;
        if (summary.total_pending === 0) return window.langTranslations.paid;
        if (summary.total_paid === 0) return window.langTranslations.pending;
        return window.langTranslations.partial;
    }

    function updateBillingTable(bills) {
        const tbody = $('#billingTableBody');
        tbody.empty();

        let totalAmount = 0, totalPaid = 0, totalPending = 0;

        if (bills && bills.length > 0) {
            bills.forEach(bill => {
                const row = $(`
                    <tr>
                        <td><input type="checkbox" value="${bill.id}" class="bill-checkbox"></td>
                        <td>${formatDate(bill.date)}</td>
                        <td><span class="badge badge-primary">${bill.module}</span></td>
                        <td>${bill.bill_no}</td>
                        <td>${bill.description}</td>
                        <td>${formatCurrency(bill.amount)}</td>
                        <td>${formatCurrency(bill.paid)}</td>
                        <td>${formatCurrency(bill.pending)}</td>
                        <td>${getStatusBadge(bill.status)}</td>
                        <td class="no-print">
                            <div class="btn-group btn-group-sm" role="group">
                                <button class="btn btn-outline-info btn-sm" onclick="viewBillDetails('${bill.bill_no}')" title="${window.langTranslations.view_details}">
                                    <i class="fas fa-eye"></i>
                                </button>
                                ${parseFloat(bill.pending) > 0 ? `
                                    <button class="btn btn-outline-success btn-sm" onclick="makePayment('${bill.bill_no}', ${bill.pending}, '${bill.id}')" title="${window.langTranslations.make_payment}">
                                        <i class="fas fa-credit-card"></i>
                                    </button>
                                ` : ''}
                                <button class="btn btn-outline-secondary btn-sm" onclick="printSingleBill('${bill.bill_no}')" title="Print">
                                    <i class="fas fa-print"></i>
                                </button>
                            </div>
                        </td>
                    </tr>
                `);

                tbody.append(row);

                totalAmount += parseFloat(bill.amount || 0);
                totalPaid += parseFloat(bill.paid || 0);
                totalPending += parseFloat(bill.pending || 0);
            });
        } else {
            tbody.append('<tr><td colspan="10" class="text-center text-muted">No billing records found</td></tr>');
        }

        $('#total_amount').text(formatCurrency(totalAmount));
        $('#total_paid').text(formatCurrency(totalPaid));
        $('#total_pending').text(formatCurrency(totalPending));
        
        updatePaginationInfo(bills.length);
    }

    function getStatusBadge(status) {
        const badges = {
            'paid': `<span class="badge badge-success">${window.langTranslations.paid}</span>`,
            'pending': `<span class="badge badge-danger">${window.langTranslations.pending}</span>`,
            'partial': `<span class="badge badge-warning">${window.langTranslations.partial}</span>`
        };
        return badges[status] || '<span class="badge badge-secondary">Unknown</span>';
    }

    function updatePaginationInfo(recordsShown) {
        const start = window.billingConfig.currentPage === 1 ? 1 : 
                     (window.billingConfig.currentPage - 1) * window.billingConfig.recordsPerPage + 1;
        const end = Math.min(start + recordsShown - 1, window.billingConfig.totalRecords);
        
        $('#showing_start').text(start);
        $('#showing_end').text(end);
        $('#total_records').text(window.billingConfig.totalRecords);
    }

    // =============================================
    // PAYMENT PROCESSING FUNCTIONS
    // =============================================

    window.makePayment = function(billNo, pendingAmount, billId) {
        $('#paymentBillNo').text(billNo);
        $('#paymentPatientName').text($('#display_patient_name').text());
        $('#outstandingAmount').text(parseFloat(pendingAmount).toFixed(2));
        $('#paymentAmount').val(parseFloat(pendingAmount).toFixed(2));
        $('#paymentBillId').val(billId);
        $('#paymentPatientId').val(window.billingConfig.currentPatient?.id);
        
        updatePaymentSummary(pendingAmount, pendingAmount);
        
        $('#paymentModal').modal('show');
    };

    function updatePaymentSummary(originalAmount, currentPayment) {
        const original = parseFloat(originalAmount || 0);
        const current = parseFloat(currentPayment || 0);
        const previous = original - parseFloat($('#outstandingAmount').text() || 0);
        const remaining = parseFloat($('#outstandingAmount').text() || 0) - current;
        
        $('#originalAmount').text(original.toFixed(2));
        $('#previousPayments').text(previous.toFixed(2));
        $('#currentPayment').text(current.toFixed(2));
        $('#remainingBalance').text(Math.max(0, remaining).toFixed(2));
    }

    async function processPayment() {
        try {
            const formData = {
                bill_id: $('#paymentBillId').val(),
                patient_id: $('#paymentPatientId').val(),
                amount: parseFloat($('#paymentAmount').val()),
                payment_method: $('#paymentMethod').val(),
                reference_number: $('#referenceNumber').val(),
                payment_date: $('#paymentDate').val(),
                notes: $('#paymentFormNotes').val()
            };

            if (formData.amount <= 0) {
                showMessage('Payment amount must be greater than 0', 'warning');
                return;
            }

            if (!formData.payment_method) {
                showMessage('Please select a payment method', 'warning');
                return;
            }

            showLoading(true);

            const response = await fetch(`${window.billingConfig.apiBaseUrl}process_payment`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(formData)
            });

            const data = await response.json();

            if (!data.success) {
                throw new Error(data.error || window.langTranslations.error_occurred);
            }

            $('#paymentModal').modal('hide');
            showMessage(window.langTranslations.payment_successful, 'success');

            if (window.billingConfig.currentPatient) {
                await loadBillingData(window.billingConfig.currentPatient.id);
            }

        } catch (error) {
            showMessage(`${window.langTranslations.error_occurred}: ${error.message}`, 'error');
        } finally {
            showLoading(false);
        }
    }

    // =============================================
    // BILL DETAILS FUNCTIONS
    // =============================================

    window.viewBillDetails = async function(billNo) {
        try {
            showLoading(true);
            
            const response = await fetch(`${window.billingConfig.apiBaseUrl}get_bill_details?bill_no=${encodeURIComponent(billNo)}`);
            if (response.status === 401 || response.status === 403) {
                showMessage('Session expired or access denied. Please log in.', 'error');
                return;
            }
            const data = await response.json();
            
            if (!data.success) {
                throw new Error(data.error || window.langTranslations.error_occurred);
            }

            showBillDetailsModal(billNo, data.data);

        } catch (error) {
            showMessage(`${window.langTranslations.error_occurred}: ${error.message}`, 'error');
        } finally {
            showLoading(false);
        }
    };

    function showBillDetailsModal(billNo, details) {
        const modalContent = `
            <div class="row">
                <div class="col-md-6">
                    <h6><i class="fas fa-user text-primary"></i> Patient Information</h6>
                    <table class="table table-sm table-borderless">
                        <tr><td><strong>Name:</strong></td><td>${details.patient_name || 'N/A'}</td></tr>
                        <tr><td><strong>Patient ID:</strong></td><td>${details.patient_id || 'N/A'}</td></tr>
                        <tr><td><strong>Date:</strong></td><td>${formatDate(details.date || details.created_at)}</td></tr>
                        <tr><td><strong>Department:</strong></td><td>${details.department || details.module || 'N/A'}</td></tr>
                    </table>
                </div>
                <div class="col-md-6">
                    <h6><i class="fas fa-receipt text-success"></i> Bill Information</h6>
                    <table class="table table-sm table-borderless">
                        <tr><td><strong>Bill No:</strong></td><td>${billNo}</td></tr>
                        <tr><td><strong>Total Amount:</strong></td><td>${formatCurrency(details.total_amount || details.amount || 0)}</td></tr>
                        <tr><td><strong>Paid Amount:</strong></td><td>${formatCurrency(details.paid_amount || 0)}</td></tr>
                        <tr><td><strong>Pending:</strong></td><td>${formatCurrency(details.pending_amount || 0)}</td></tr>
                    </table>
                </div>
            </div>
            <hr>
            <div class="row">
                <div class="col-md-6">
                    <h6><i class="fas fa-list text-info"></i> Service Details</h6>
                    <div class="bg-light p-3 rounded">
                        <strong>Description:</strong><br>
                        ${details.description || 'No description available'}
                    </div>
                </div>
                <div class="col-md-6">
                    <h6><i class="fas fa-credit-card text-warning"></i> Payment Status</h6>
                    <div class="bg-light p-3 rounded">
                        <strong>Status:</strong> ${getStatusBadge(details.payment_status || details.status)}<br>
                        <strong>Last Payment:</strong> ${formatDate(details.last_payment_date)}<br>
                        <strong>Payment Method:</strong> ${details.payment_method || 'N/A'}
                    </div>
                </div>
            </div>
            ${details.payment_history ? `
                <hr>
                <h6><i class="fas fa-history text-secondary"></i> Payment History</h6>
                <div class="table-responsive">
                    <table class="table table-sm">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Amount</th>
                                <th>Method</th>
                                <th>Reference</th>
                            </tr>
                        </thead>
                        <tbody>
                            ${details.payment_history.map(payment => `
                                <tr>
                                    <td>${formatDate(payment.date)}</td>
                                    <td>${formatCurrency(payment.amount)}</td>
                                    <td>${payment.method}</td>
                                    <td>${payment.reference || '-'}</td>
                                </tr>
                            `).join('')}
                        </tbody>
                    </table>
                </div>
            ` : ''}
        `;

        $('#billDetailsContent').html(modalContent);
        $('#billDetailsModal').modal('show');
    }

    // =============================================
    // PRINT AND EXPORT FUNCTIONS
    // =============================================

    function printBill() {
        if (!window.billingConfig.currentPatient) {
            showMessage(window.langTranslations.no_patient_selected, 'warning');
            return;
        }

        preparePrintContent();
        window.print();
    }

    function preparePrintContent() {
        const patient = window.billingConfig.currentPatient;
        const patientInfo = `
            <div class="row mb-4">
                <div class="col-6">
                    <h5>Patient Information</h5>
                    <p><strong>Name:</strong> ${patient.patient_name}</p>
                    <p><strong>Patient ID:</strong> ${patient.patient_id || patient.id}</p>
                    <p><strong>Age:</strong> ${patient.age || 'N/A'}</p>
                    <p><strong>Gender:</strong> ${patient.gender || 'N/A'}</p>
                </div>
                <div class="col-6">
                    <p><strong>Mobile:</strong> ${patient.mobileno || '-'}</p>
                    <p><strong>Email:</strong> ${patient.email || '-'}</p>
                    <p><strong>Blood Group:</strong> ${patient.blood_group || '-'}</p>
                    <p><strong>Date:</strong> ${new Date().toLocaleDateString()}</p>
                </div>
            </div>
        `;

        const billingTable = document.getElementById('billingTable').cloneNode(true);
        $(billingTable).find('.no-print').remove();
        $(billingTable).find('input[type="checkbox"]').remove();

        $('#printPatientInfo').html(patientInfo);
        $('#printBillingDetails').html('').append(billingTable);
        $('#print_timestamp').text(new Date().toLocaleString());
    }

    function exportBill() {
        if (!window.billingConfig.currentPatient || !window.billingConfig.currentBillingData) {
            showMessage('No billing data available for export', 'warning');
            return;
        }

        window.location.href = `${window.billingConfig.apiBaseUrl}export_csv?patient_id=${window.billingConfig.currentPatient.id}`;
    }

    window.printSingleBill = function(billNo) {
        const printWindow = window.open('', '_blank');
        printWindow.document.write(`
            <!DOCTYPE html>
            <html>
            <head>
                <title>Bill - ${billNo}</title>
                <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/4.6.0/css/bootstrap.min.css" rel="stylesheet">
                <style>
                    body { padding: 20px; }
                    @media print { .btn { display: none; } }
                </style>
            </head>
            <body>
                <div class="container">
                    <h2 class="text-center mb-4"><?php echo $this->session->userdata('hospital_name') ?? 'HEALTHCARE FACILITY'; ?></h2>
                    <h4 class="text-center mb-4">Bill - ${billNo}</h4>
                    <hr>
                    <div id="billContent">Loading...</div>
                </div>
                <script>
                    fetch('${window.billingConfig.apiBaseUrl}get_bill_details?bill_no=${encodeURIComponent(billNo)}')
                        .then(response => response.json())
                        .then(data => {
                            if (data.success) {
                                document.getElementById('billContent').innerHTML = \`
                                    <div class="row">
                                        <div class="col-6">
                                            <h6>Patient: \${data.data.patient_name}</h6>
                                            <p>Date: \${data.data.date}</p>
                                        </div>
                                        <div class="col-6 text-right">
                                            <h6>Amount: \${data.data.total_amount}</h6>
                                            <p>Status: \${data.data.payment_status}</p>
                                        </div>
                                    </div>
                                    <p><strong>Description:</strong> \${data.data.description}</p>
                                \`;
                            }
                            window.print();
                        });
                </script>
            </body>
            </html>
        `);
        printWindow.document.close();
    };

    // =============================================
    // BULK OPERATIONS FUNCTIONS
    // =============================================

    function processSelectedBills() {
        const selectedBills = $('.bill-checkbox:checked').map(function() {
            return $(this).val();
        }).get();

        if (selectedBills.length === 0) {
            showMessage('Please select bills to process', 'warning');
            return;
        }

        const modalHtml = `
           <div class="modal fade" id="bulkProcessModal" tabindex="-1" role="dialog">
               <div class="modal-dialog" role="document">
                   <div class="modal-content">
                       <div class="modal-header bg-warning text-dark">
                           <h4 class="modal-title">
                               <i class="fas fa-tasks mr-2"></i>Bulk Process Bills (${selectedBills.length} selected)
                           </h4>
                           <button type="button" class="close" data-dismiss="modal">&times;</button>
                       </div>
                       <div class="modal-body">
                           <div class="btn-group-vertical w-100" role="group">
                               <button class="btn btn-outline-success" onclick="bulkPayment()">
                                   <i class="fas fa-credit-card mr-2"></i>Process Bulk Payment
                               </button>
                               <button class="btn btn-outline-info" onclick="bulkPrint()">
                                   <i class="fas fa-print mr-2"></i>Print Selected Bills
                               </button>
                               <button class="btn btn-outline-warning" onclick="bulkEmail()">
                                   <i class="fas fa-envelope mr-2"></i>Email Selected Bills
                               </button>
                               <button class="btn btn-outline-secondary" onclick="bulkExport()">
                                   <i class="fas fa-download mr-2"></i>Export Selected Bills
                               </button>
                           </div>
                       </div>
                   </div>
               </div>
           </div>
       `;

       $('#bulkProcessModal').remove();
       $('body').append(modalHtml);
       $('#bulkProcessModal').modal('show');
   }

   window.bulkPayment = function() {
       const selectedBills = $('.bill-checkbox:checked').map(function() {
           return $(this).val();
       }).get();

       let totalPending = 0;
       selectedBills.forEach(billId => {
           const row = $(`.bill-checkbox[value="${billId}"]`).closest('tr');
           const pendingAmount = parseFloat(row.find('td:nth-child(8)').text().replace('Ksh.', '')) || 0;
           totalPending += pendingAmount;
       });

       $('#paymentBillNo').text(`Bulk Payment (${selectedBills.length} bills)`);
       $('#paymentPatientName').text($('#display_patient_name').text());
       $('#outstandingAmount').text(totalPending.toFixed(2));
       $('#paymentAmount').val(totalPending.toFixed(2));
       $('#paymentBillId').val(selectedBills.join(','));
       
       $('#bulkProcessModal').modal('hide');
       $('#paymentModal').modal('show');
   };

   window.bulkPrint = function() {
       const selectedBills = $('.bill-checkbox:checked').map(function() {
           return $(this).val();
       }).get();

       const printWindow = window.open('', '_blank');
       printWindow.document.write(`
           <!DOCTYPE html>
           <html>
           <head>
               <title>Bulk Bills Print</title>
               <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/4.6.0/css/bootstrap.min.css" rel="stylesheet">
               <style>
                   body { padding: 20px; }
                   @media print { .btn { display: none; } .page-break { page-break-before: always; } }
               </style>
           </head>
           <body>
               <div class="container">
                   <h2 class="text-center mb-4"><?php echo $this->session->userdata('hospital_name') ?? 'HEALTHCARE FACILITY'; ?></h2>
                   <h4 class="text-center mb-4">Bulk Billing Statement</h4>
                   <div id="billsContent">Loading ${selectedBills.length} bills...</div>
               </div>
               <script>
                   const billIds = ${JSON.stringify(selectedBills)};
                   let loadedCount = 0;
                   let allContent = '';
                   
                   billIds.forEach((billId, index) => {
                       fetch('${window.billingConfig.apiBaseUrl}get_bill_details?bill_id=' + billId)
                           .then(response => response.json())
                           .then(data => {
                               if (data.success) {
                                   allContent += '<div class="' + (index > 0 ? 'page-break' : '') + '">';
                                   allContent += '<h5>Bill #' + data.data.bill_no + '</h5>';
                                   allContent += '<p>Amount: ' + data.data.total_amount + '</p>';
                                   allContent += '<p>Status: ' + data.data.payment_status + '</p>';
                                   allContent += '</div>';
                               }
                               loadedCount++;
                               if (loadedCount === billIds.length) {
                                   document.getElementById('billsContent').innerHTML = allContent;
                                   setTimeout(() => window.print(), 500);
                               }
                           });
                   });
               </script>
           </body>
           </html>
       `);
       printWindow.document.close();
       
       $('#bulkProcessModal').modal('hide');
   };

   window.bulkEmail = function() {
       showMessage('Bulk email functionality - Implementation depends on your email service', 'info');
       $('#bulkProcessModal').modal('hide');
   };

   window.bulkExport = function() {
       const selectedBills = $('.bill-checkbox:checked').map(function() {
           return $(this).val();
       }).get();

       window.location.href = `${window.billingConfig.apiBaseUrl}export_selected_bills?bill_ids=${selectedBills.join(',')}`;
       $('#bulkProcessModal').modal('hide');
   };

   // =============================================
   // EMAIL FUNCTIONALITY
   // =============================================

   function emailStatement() {
       if (!window.billingConfig.currentPatient) {
           showMessage('No patient selected for email', 'warning');
           return;
       }

       const modalHtml = `
           <div class="modal fade" id="emailModal" tabindex="-1" role="dialog">
               <div class="modal-dialog" role="document">
                   <div class="modal-content">
                       <div class="modal-header bg-info text-white">
                           <h4 class="modal-title">
                               <i class="fas fa-envelope mr-2"></i>Email Billing Statement
                           </h4>
                           <button type="button" class="close text-white" data-dismiss="modal">&times;</button>
                       </div>
                       <form id="emailForm">
                           <div class="modal-body">
                               <div class="form-group">
                                   <label>Patient Email <span class="text-danger">*</span></label>
                                   <input type="email" class="form-control" id="patientEmail" 
                                          value="${window.billingConfig.currentPatient.email || ''}" required>
                               </div>
                               <div class="form-group">
                                   <label>Additional Recipients (comma separated)</label>
                                   <input type="text" class="form-control" id="additionalEmails" 
                                          placeholder="email1@example.com, email2@example.com">
                               </div>
                               <div class="form-group">
                                   <label>Subject</label>
                                   <input type="text" class="form-control" id="emailSubject" 
                                          value="Billing Statement - ${window.billingConfig.currentPatient.patient_name}">
                               </div>
                               <div class="form-group">
                                   <label>Message</label>
                                   <textarea class="form-control" id="emailMessage" rows="4">Dear ${window.billingConfig.currentPatient.patient_name},

Please find attached your billing statement.

Thank you for choosing our healthcare services.

Best regards,
Healthcare Team</textarea>
                               </div>
                               <div class="form-group">
                                   <div class="custom-control custom-checkbox">
                                       <input type="checkbox" class="custom-control-input" id="includeHistory" checked>
                                       <label class="custom-control-label" for="includeHistory">Include payment history</label>
                                   </div>
                               </div>
                           </div>
                           <div class="modal-footer">
                               <button type="button" class="btn btn-secondary" data-dismiss="modal">Cancel</button>
                               <button type="submit" class="btn btn-info">
                                   <i class="fas fa-paper-plane mr-1"></i>Send Email
                               </button>
                           </div>
                       </form>
                   </div>
               </div>
           </div>
       `;

       $('#emailModal').remove();
       $('body').append(modalHtml);
       $('#emailModal').modal('show');

       $('#emailForm').on('submit', async function(e) {
           e.preventDefault();
           await sendEmailStatement();
       });
   }

   async function sendEmailStatement() {
       try {
           const emailData = {
               patient_id: window.billingConfig.currentPatient.id,
               to_email: $('#patientEmail').val(),
               additional_emails: $('#additionalEmails').val(),
               subject: $('#emailSubject').val(),
               message: $('#emailMessage').val(),
               include_history: $('#includeHistory').prop('checked')
           };

           showLoading(true);

           const response = await fetch(`${window.billingConfig.apiBaseUrl}send_billing_email`, {
               method: 'POST',
               headers: {
                   'Content-Type': 'application/json'
               },
               body: JSON.stringify(emailData)
           });

           const data = await response.json();

           if (!data.success) {
               throw new Error(data.error || 'Failed to send email');
           }

           $('#emailModal').modal('hide');
           showMessage('Billing statement sent successfully!', 'success');

       } catch (error) {
           showMessage(`Failed to send email: ${error.message}`, 'error');
       } finally {
           showLoading(false);
       }
   }

   // =============================================
   // QUICK ACTIONS FUNCTIONS
   // =============================================

   function refreshData() {
       if (window.billingConfig.currentPatient) {
           loadBillingData(window.billingConfig.currentPatient.id);
           showMessage('Data refreshed successfully', 'success');
       }
   }

   function exportSummary() {
       if (!window.billingConfig.currentBillingData) {
           showMessage('No data to export', 'warning');
           return;
       }

       const csvContent = generateCSVContent();
       downloadCSV(csvContent, `billing_summary_${window.billingConfig.currentPatient.patient_name}_${new Date().toISOString().split('T')[0]}.csv`);
   }

   function generateCSVContent() {
       const bills = window.billingConfig.currentBillingData.bills;
       const headers = ['Date', 'Module', 'Bill No', 'Description', 'Amount', 'Paid', 'Pending', 'Status'];
       
       let csv = headers.join(',') + '\n';
       
       bills.forEach(bill => {
           const row = [
               formatDate(bill.date),
               bill.module,
               bill.bill_no,
               `"${bill.description.replace(/"/g, '""')}"`,
               bill.amount,
               bill.paid,
               bill.pending,
               bill.status
           ];
           csv += row.join(',') + '\n';
       });
       
       return csv;
   }

   function downloadCSV(content, filename) {
       const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' });
       const link = document.createElement('a');
       if (link.download !== undefined) {
           const url = URL.createObjectURL(blob);
           link.setAttribute('href', url);
           link.setAttribute('download', filename);
           link.style.visibility = 'hidden';
           document.body.appendChild(link);
           link.click();
           document.body.removeChild(link);
       }
   }

   // =============================================
   // FILTERS AND SEARCH
   // =============================================

   function applyFilters() {
       const filters = {
           module: $('#moduleFilter').val(),
           status: $('#statusFilter').val(),
           date: $('#dateFilter').val(),
           search: $('#billSearchInput').val()
       };

       window.billingConfig.filters = filters;
       
       if (window.billingConfig.currentPatient) {
           loadBillingData(window.billingConfig.currentPatient.id);
       }
   }

   function clearFilters() {
       $('#moduleFilter').val('');
       $('#statusFilter').val('');
       $('#dateFilter').val('');
       $('#billSearchInput').val('');
       window.billingConfig.filters = { module: '', status: '', date: '', search: '' };
       
       if (window.billingConfig.currentPatient) {
           loadBillingData(window.billingConfig.currentPatient.id);
       }
   }

   function clearSearch() {
       $('#patientSearchForm')[0].reset();
       hideAllSections();
       hideSearchSuggestions();
       window.billingConfig.currentPatient = null;
       window.billingConfig.currentBillingData = null;
       window.billingConfig.searchResults = [];
   }

   function hideAllSections() {
       $('#patientInfoSection').hide();
       $('#billingSummarySection').hide();
       $('#quickActionsSection').hide();
       $('#billingDetailsSection').hide();
   }

   // =============================================
   // PATIENT HISTORY
   // =============================================

   function viewPatientHistory() {
       if (!window.billingConfig.currentPatient) {
           showMessage('No patient selected', 'warning');
           return;
       }

       loadPatientHistory(window.billingConfig.currentPatient.id);
   }

   async function loadPatientHistory(patientId) {
       try {
           showLoading(true);
           
           const response = await fetch(`${window.billingConfig.apiBaseUrl}get_patient_history?patient_id=${patientId}`);
           const data = await response.json();
           
           if (data.success) {
               showPatientHistoryModal(data.data);
           } else {
               showMessage('Failed to load patient history', 'error');
           }
       } catch (error) {
           showMessage(`Error loading history: ${error.message}`, 'error');
       } finally {
           showLoading(false);
       }
   }

   function showPatientHistoryModal(historyData) {
       const content = `
           <div class="row">
               <div class="col-md-12">
                   <ul class="nav nav-tabs" id="historyTabs">
                       <li class="nav-item">
                           <a class="nav-link active" data-toggle="tab" href="#consultations">Consultations</a>
                       </li>
                       <li class="nav-item">
                           <a class="nav-link" data-toggle="tab" href="#treatments">Treatments</a>
                       </li>
                       <li class="nav-item">
                           <a class="nav-link" data-toggle="tab" href="#payments">Payments</a>
                       </li>
                       <li class="nav-item">
                           <a class="nav-link" data-toggle="tab" href="#timeline">Timeline</a>
                       </li>
                   </ul>
                   <div class="tab-content mt-3">
                       <div class="tab-pane active" id="consultations">
                           <div class="table-responsive">
                               <table class="table table-sm">
                                   <thead>
                                       <tr>
                                           <th>Date</th>
                                           <th>Doctor</th>
                                           <th>Diagnosis</th>
                                           <th>Status</th>
                                       </tr>
                                   </thead>
                                   <tbody>
                                       ${historyData.consultations ? historyData.consultations.map(consultation => `
                                           <tr>
                                               <td>${formatDate(consultation.date)}</td>
                                               <td>${consultation.doctor_name}</td>
                                               <td>${consultation.diagnosis || '-'}</td>
                                               <td><span class="badge badge-success">Completed</span></td>
                                           </tr>
                                       `).join('') : '<tr><td colspan="4">No consultations found</td></tr>'}
                                   </tbody>
                               </table>
                           </div>
                       </div>
                       <div class="tab-pane" id="treatments">
                           <div class="table-responsive">
                               <table class="table table-sm">
                                   <thead>
                                       <tr>
                                           <th>Date</th>
                                           <th>Treatment</th>
                                           <th>Department</th>
                                           <th>Cost</th>
                                       </tr>
                                   </thead>
                                   <tbody>
                                       ${historyData.treatments ? historyData.treatments.map(treatment => `
                                           <tr>
                                               <td>${formatDate(treatment.date)}</td>
                                               <td>${treatment.description}</td>
                                               <td>${treatment.department}</td>
                                               <td>${formatCurrency(treatment.amount)}</td>
                                           </tr>
                                       `).join('') : '<tr><td colspan="4">No treatments found</td></tr>'}
                                   </tbody>
                               </table>
                           </div>
                       </div>
                       <div class="tab-pane" id="payments">
                           <div class="table-responsive">
                               <table class="table table-sm">
                                   <thead>
                                       <tr>
                                           <th>Date</th>
                                           <th>Amount</th>
                                           <th>Method</th>
                                           <th>Reference</th>
                                       </tr>
                                   </thead>
                                   <tbody>
                                       ${historyData.payments ? historyData.payments.map(payment => `
                                           <tr>
                                               <td>${formatDate(payment.date)}</td>
                                               <td>${formatCurrency(payment.amount)}</td>
                                               <td>${payment.method}</td>
                                               <td>${payment.reference || '-'}</td>
                                           </tr>
                                       `).join('') : '<tr><td colspan="4">No payments found</td></tr>'}
                                   </tbody>
                               </table>
                           </div>
                       </div>
                       <div class="tab-pane" id="timeline">
                           <div class="timeline">
                               ${historyData.timeline ? historyData.timeline.map(event => `
                                   <div class="timeline-item">
                                       <span class="timeline-badge bg-primary"><i class="fas fa-calendar"></i></span>
                                       <div class="timeline-panel">
                                           <div class="timeline-heading">
                                               <h6 class="timeline-title">${event.title}</h6>
                                               <small class="text-muted">${formatDate(event.date)}</small>
                                           </div>
                                           <div class="timeline-body">
                                               <p>${event.description}</p>
                                           </div>
                                       </div>
                                   </div>
                               `).join('') : '<p>No timeline events found</p>'}
                           </div>
                       </div>
                   </div>
               </div>
           </div>
       `;

       $('#patientHistoryContent').html(content);
       $('#patientHistoryModal').modal('show');
   }

   // =============================================
   // NOTIFICATION SYSTEM
   // =============================================

   function setupNotifications() {
       if (window.billingConfig.currentBillingData?.bills) {
           const overdueBills = window.billingConfig.currentBillingData.bills.filter(bill => {
               const billDate = new Date(bill.date);
               const daysDiff = (new Date() - billDate) / (1000 * 60 * 60 * 24);
               return bill.status === 'pending' && daysDiff > 30;
           });

           if (overdueBills.length > 0) {
               showMessage(`⚠️ ${overdueBills.length} bill(s) are overdue (>30 days)`, 'warning', 8000);
           }
       }

       if (window.billingConfig.currentBillingData?.bills) {
           const partialBills = window.billingConfig.currentBillingData.bills.filter(bill => bill.status === 'partial');
           if (partialBills.length > 0) {
               showMessage(`💰 ${partialBills.length} bill(s) have partial payments pending`, 'info', 6000);
           }
       }
   }

   // =============================================
   // BULK SELECTION MANAGEMENT
   // =============================================

   function updateBulkActionButtons() {
       const selectedCount = $('.bill-checkbox:checked').length;
       if (selectedCount > 0) {
           if ($('#bulkActionsBar').length === 0) {
               $('#quickActionsSection .card-body').append(`
                   <div id="bulkActionsBar" class="mt-3 p-3 bg-light border rounded">
                       <strong>${selectedCount} bill(s) selected</strong>
                       <div class="btn-group ml-3" role="group">
                           <button class="btn btn-sm btn-success" onclick="processSelectedBills()">
                               <i class="fas fa-tasks mr-1"></i>Bulk Actions
                           </button>
                           <button class="btn btn-sm btn-secondary" onclick="clearSelection()">
                               <i class="fas fa-times mr-1"></i>Clear
                           </button>
                       </div>
                   </div>
               `);
           } else {
               $('#bulkActionsBar strong').text(`${selectedCount} bill(s) selected`);
           }
       } else {
           $('#bulkActionsBar').remove();
       }
   }

   window.clearSelection = function() {
       $('.bill-checkbox, #selectAll').prop('checked', false);
       updateBulkActionButtons();
   };

   // =============================================
   // EVENT HANDLERS
   // =============================================

   function initializeEvents() {
       // Search form
       $('#patientSearchForm').on('submit', function(e) {
           e.preventDefault();
           searchPatient();
       });

       // Clear search
       $('#clearSearch').on('click', clearSearch);

       // Advanced search toggle
       $('#advancedSearchToggle').on('click', function() {
           $('#advancedSearchPanel').slideToggle();
       });

       // Live search
       $('#patient_id, #patient_name, #mobile_no').on('input', 
           debounce(performLiveSearch, 500)
       );

       // Click outside to hide suggestions
       $(document).on('click', function(e) {
           if (!$(e.target).closest('#searchSuggestions, .form-group').length) {
               hideSearchSuggestions();
           }
       });

       // Payment form
       $('#paymentForm').on('submit', function(e) {
           e.preventDefault();
           processPayment();
       });

       // Payment amount change
       $('#paymentAmount').on('input', function() {
           updatePaymentSummary($('#outstandingAmount').text(), $(this).val());
       });

       // Quick actions
       $('#makePaymentBtn').on('click', function() {
           if (window.billingConfig.currentBillingData?.bills?.length > 0) {
               const pendingBills = window.billingConfig.currentBillingData.bills.filter(bill => parseFloat(bill.pending) > 0);
               if (pendingBills.length > 0) {
                   makePayment(pendingBills[0].bill_no, pendingBills[0].pending, pendingBills[0].id);
               } else {
                   showMessage('No pending payments found', 'info');
               }
           } else {
               showMessage('No bills found for payment', 'warning');
           }
       });

       $('#printAllBills').on('click', printBill);
       $('#exportBill').on('click', exportBill);
       $('#exportSummary').on('click', exportSummary);
       $('#refreshData').on('click', refreshData);
       $('#viewPatientHistory').on('click', viewPatientHistory);
       $('#emailStatement').on('click', emailStatement);

       // Filters
       $('#moduleFilter, #statusFilter, #dateFilter').on('change', applyFilters);
       $('#billSearchInput').on('input', debounce(applyFilters, 500));
       $('#searchInBills').on('click', applyFilters);
       $('#clearFilters').on('click', clearFilters);

       // Table actions
       $('#selectAll').on('change', function() {
           $('.bill-checkbox').prop('checked', $(this).prop('checked'));
           updateBulkActionButtons();
       });

       $(document).on('change', '.bill-checkbox', updateBulkActionButtons);

       // Bill details print
       $('#printBillDetails').on('click', function() {
           const modalContent = $('#billDetailsContent').html();
           const printWindow = window.open('', '_blank');
           printWindow.document.write(`
               <!DOCTYPE html>
               <html>
               <head>
                   <title>Bill Details</title>
                   <link href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/4.6.0/css/bootstrap.min.css" rel="stylesheet">
                   <style>body { padding: 20px; }</style>
               </head>
               <body>
                   <div class="container">
                       <h2 class="text-center mb-4"><?php echo $this->session->userdata('hospital_name') ?? 'HEALTHCARE FACILITY'; ?></h2>
                       ${modalContent}
                   </div>
                   <script>window.print();</script>
               </body>
               </html>
           `);
           printWindow.document.close();
       });

       // Modal reset handlers
       $('#paymentModal').on('hidden.bs.modal', function() {
           $('#paymentForm')[0].reset();
       });

       $('#billDetailsModal').on('hidden.bs.modal', function() {
           $('#billDetailsContent').html('');
       });

       $('#patientSelectionModal').on('hidden.bs.modal', function() {
           $('#patientSelectionTableBody').empty();
       });

       // Keyboard shortcuts
       $(document).on('keydown', function(e) {
           // Ctrl/Cmd + F for search
           if ((e.ctrlKey || e.metaKey) && e.key === 'f') {
               e.preventDefault();
               $('#patient_name').focus();
           }
           
           // Ctrl/Cmd + P for print
           if ((e.ctrlKey || e.metaKey) && e.key === 'p') {
               e.preventDefault();
               printBill();
           }

           // Escape to close modals
           if (e.key === 'Escape') {
               $('.modal').modal('hide');
               hideSearchSuggestions();
           }
       });

       // Auto-save search preferences
       $('#patient_id, #patient_name, #mobile_no').on('change', function() {
           try {
               localStorage.setItem('lastSearch', JSON.stringify({
                   patient_id: $('#patient_id').val(),
                   patient_name: $('#patient_name').val(),
                   mobile_no: $('#mobile_no').val(),
                   timestamp: new Date().toISOString()
               }));
           } catch (e) {
               // Ignore localStorage errors
           }
       });

       // Restore last search on page load
       try {
           const lastSearch = JSON.parse(localStorage.getItem('lastSearch') || '{}');
           if (lastSearch.timestamp && new Date() - new Date(lastSearch.timestamp) < 24 * 60 * 60 * 1000) {
               $('#patient_id').val(lastSearch.patient_id || '');
               $('#patient_name').val(lastSearch.patient_name || '');
               $('#mobile_no').val(lastSearch.mobile_no || '');
           }
       } catch (e) {
           // Ignore localStorage errors
       }
   }

   // =============================================
   // INITIALIZATION
   // =============================================

   try {
       console.log('🏥 Enhanced Patient Billing System with Integrated JavaScript Ready');
       
       // Initialize all events
       initializeEvents();
       
       // Set today's date for filters
       const today = new Date().toISOString().split('T')[0];
       $('#date_to, #paymentDate').val(today);
       
       // Focus on first search field
       $('#patient_id').focus();
       
       console.log('✅ Patient Billing System initialized successfully');
       
       setTimeout(function() {
           showMessage('🏥 Patient Billing System ready - Search for a patient to begin', 'success', 3000);
       }, 1000);
       
   } catch (error) {
       console.error('❌ Billing system initialization failed:', error);
       showMessage(`Failed to initialize billing system: ${error.message}`, 'error');
   }

   // =============================================
   // GLOBAL HELPER FUNCTIONS
   // =============================================

   // Make functions available globally for onclick handlers
   window.selectPatientFromSuggestion = selectPatientFromSuggestion;
   window.selectPatient = selectPatient;
   window.viewBillDetails = viewBillDetails;
   window.makePayment = makePayment;
   window.printSingleBill = printSingleBill;
   window.processSelectedBills = processSelectedBills;
   window.bulkPayment = bulkPayment;
   window.bulkPrint = bulkPrint;
   window.bulkEmail = bulkEmail;
   window.bulkExport = bulkExport;
   window.clearSelection = clearSelection;

   // Debug function
   window.debugBillingSystem = function() {
       console.log('=== Enhanced Patient Billing System Debug Information ===');
       console.log('Configuration:', window.billingConfig);
       console.log('Current Patient:', window.billingConfig.currentPatient);
       console.log('Current Billing Data:', window.billingConfig.currentBillingData);
       console.log('Search Results:', window.billingConfig.searchResults);
       console.log('Filters:', window.billingConfig.filters);
       console.log('Language Translations:', window.langTranslations);
       
       if (window.billingConfig.currentPatient) {
           console.log('Testing patient data reload...');
           loadBillingData(window.billingConfig.currentPatient.id);
       }
   };

}); // End of $(document).ready

</script>