<?php
/**
 * Enhanced Laboratory Test Selection View with Patient Verification
 * File: application/views/admin/laboratory/test_selection.php
 * 
 * Features:
 * - SecuGen WebAPI fingerprint verification
 * - Patient photo display
 * - Real-time test selection with verification enforcement
 * - Comprehensive billing and order management
 */
?>
<!-- Include SecuGen WebAPI Script -->
<script src="<?php echo base_url('WebAPI-JavaScript/secugen.js'); ?>"></script>

<!-- ============================================= -->
<!-- CSS STYLES -->
<!-- ============================================= -->
<style>
/* =================================
   MAIN LAYOUT STYLES 
   ================================= */
.content-wrapper {
    background-color: #ecf0f5;
}

.test-category-card {
    border: 1px solid #ddd;
    border-radius: 5px;
    margin-bottom: 15px;
    transition: all 0.3s;
    background: #fff;
}

.test-category-card:hover {
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    border-color: #3c8dbc;
}

/* =================================
   PATIENT INFO & PHOTO STYLES
   ================================= */
.patient-info-card {
    background: linear-gradient(135deg, #d4edda, #c3e6cb);
    border: 1px solid #c3e6cb;
    border-radius: 6px;
    padding: 15px;
    margin-bottom: 20px;
    position: relative;
}

.patient-photo {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    border: 3px solid #fff;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    object-fit: cover;
    background: #f8f9fa;
}

.patient-photo-placeholder {
    width: 80px;
    height: 80px;
    border-radius: 50%;
    border: 3px solid #fff;
    box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    background: linear-gradient(135deg, #e9ecef, #dee2e6);
    display: flex;
    align-items: center;
    justify-content: center;
    color: #6c757d;
    font-size: 28px;
}

/* =================================
   VERIFICATION STATUS STYLES
   ================================= */
.verification-status {
    position: absolute;
    top: 15px;
    right: 15px;
    z-index: 10;
}

.verification-badge {
    padding: 5px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.verified {
    background: linear-gradient(135deg, #28a745, #20c997);
    color: white;
}

.pending-verification {
    background: linear-gradient(135deg, #ffc107, #fd7e14);
    color: #212529;
}

.verification-failed {
    background: linear-gradient(135deg, #dc3545, #e83e8c);
    color: white;
}

/* =================================
   FINGERPRINT SCANNER STYLES
   ================================= */
.fingerprint-container {
    background: #f8f9fa;
    border: 2px dashed #dee2e6;
    border-radius: 8px;
    padding: 20px;
    text-align: center;
    transition: all 0.3s;
    min-height: 200px;
    display: flex;
    flex-direction: column;
    justify-content: center;
    align-items: center;
}

.fingerprint-container.scanning {
    border-color: #007bff;
    background: linear-gradient(135deg, #e3f2fd, #bbdefb);
}

.fingerprint-container.success {
    border-color: #28a745;
    background: linear-gradient(135deg, #d4edda, #c3e6cb);
}

.fingerprint-container.failed {
    border-color: #dc3545;
    background: linear-gradient(135deg, #f8d7da, #f5c6cb);
}

.fingerprint-image {
    max-width: 150px;
    max-height: 150px;
    border: 2px solid #dee2e6;
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.quality-meter {
    background: #e9ecef;
    height: 8px;
    border-radius: 4px;
    margin: 10px 0;
    overflow: hidden;
}

.quality-bar {
    height: 100%;
    transition: width 0.3s, background-color 0.3s;
    border-radius: 4px;
}

.quality-excellent { background: #28a745; }
.quality-good { background: #28a745; }
.quality-fair { background: #ffc107; }
.quality-poor { background: #dc3545; }

.verification-info {
    background: #f8f9fa;
    border-left: 4px solid #007bff;
    padding: 15px;
    border-radius: 0 4px 4px 0;
}

/* =================================
   TEST SELECTION STYLES
   ================================= */
.category-header {
    background: linear-gradient(135deg, #3c8dbc, #367fa9);
    color: white;
    padding: 12px 15px;
    border-radius: 5px 5px 0 0;
    cursor: pointer;
    transition: all 0.3s;
    position: relative;
}

.category-header:hover {
    background: linear-gradient(135deg, #367fa9, #2e6da4);
}

.category-tests {
    padding: 0;
    max-height: 350px;
    overflow-y: auto;
    border-radius: 0 0 5px 5px;
}

.test-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 15px;
    border-bottom: 1px solid #f0f0f0;
    transition: all 0.2s;
    background: #fff;
}

.test-item:last-child {
    border-bottom: none;
}

.test-item:hover {
    background-color: #f8f9fa;
    padding-left: 20px;
}

.test-name {
    flex: 1;
    font-weight: 500;
    color: #333;
    font-size: 14px;
}

.test-price {
    color: #27ae60;
    font-weight: bold;
    margin-right: 15px;
    font-size: 14px;
    min-width: 100px;
    text-align: right;
}

.btn-add-test {
    background: linear-gradient(135deg, #28a745, #20a048);
    color: white;
    border: none;
    padding: 6px 12px;
    border-radius: 4px;
    font-size: 12px;
    font-weight: 500;
    transition: all 0.2s;
    min-width: 60px;
}

.btn-add-test:hover {
    background: linear-gradient(135deg, #218838, #1e7e34);
    color: white;
    transform: translateY(-1px);
}

.btn-add-test:disabled {
    background: #6c757d;
    cursor: not-allowed;
    transform: none;
}

.category-count {
    background: rgba(255,255,255,0.9);
    color: #3c8dbc;
    padding: 3px 10px;
    border-radius: 15px;
    font-size: 11px;
    font-weight: 600;
    margin-left: 10px;
    box-shadow: 0 1px 2px rgba(0,0,0,0.1);
}

/* =================================
   CART & BILLING STYLES
   ================================= */
.cart-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 12px 15px;
    border: 1px solid #e3e6f0;
    background: linear-gradient(135deg, #f8f9fa, #e9ecef);
    margin-bottom: 8px;
    border-radius: 5px;
    transition: all 0.2s;
}

.cart-item:hover {
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.cart-item-name {
    flex: 1;
    font-weight: 500;
    color: #495057;
    font-size: 13px;
}

.cart-item-price {
    color: #28a745;
    font-weight: bold;
    margin-right: 10px;
    font-size: 13px;
    min-width: 90px;
    text-align: right;
}

.btn-remove-test {
    background: linear-gradient(135deg, #dc3545, #c82333);
    color: white;
    border: none;
    padding: 4px 8px;
    border-radius: 3px;
    font-size: 11px;
    transition: all 0.2s;
}

.btn-remove-test:hover {
    background: linear-gradient(135deg, #c82333, #bd2130);
    transform: scale(1.05);
    color: white;
}

.empty-cart {
    text-align: center;
    color: #6c757d;
    padding: 40px 20px;
    font-style: italic;
}

.empty-cart i {
    color: #dee2e6;
    margin-bottom: 15px;
}

.billing-summary {
    background: linear-gradient(135deg, #e8f4fd, #d1ecf1);
    border: 2px solid #bee5eb;
    border-radius: 8px;
    padding: 20px;
    margin-top: 20px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.billing-summary h4 {
    color: #0c5460;
    margin-bottom: 15px;
    font-weight: 600;
}

.summary-row {
    display: flex;
    justify-content: space-between;
    margin-bottom: 10px;
    padding: 5px 0;
    font-size: 14px;
}

.summary-row span:first-child {
    color: #495057;
    font-weight: 500;
}

.summary-row span:last-child {
    color: #28a745;
    font-weight: 600;
}

.total-amount {
    font-size: 18px;
    font-weight: bold;
    color: #0c5460;
    border-top: 2px solid #3c8dbc;
    padding-top: 15px;
    margin-top: 15px;
    background: rgba(255,255,255,0.8);
    padding: 15px;
    border-radius: 5px;
}

/* =================================
   UTILITY STYLES
   ================================= */
.loading-overlay {
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(255,255,255,0.8);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 1000;
    border-radius: 5px;
}

.spinner {
    border: 3px solid #f3f3f4;
    border-top: 3px solid #3c8dbc;
    border-radius: 50%;
    width: 30px;
    height: 30px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.alert-custom {
    position: fixed;
    top: 80px;
    right: 20px;
    z-index: 9999;
    min-width: 350px;
    box-shadow: 0 4px 12px rgba(0,0,0,0.15);
    border: none;
    border-radius: 6px;
}

/* =================================
   RESPONSIVE DESIGN
   ================================= */
@media (max-width: 768px) {
    .patient-photo,
    .patient-photo-placeholder {
        width: 60px;
        height: 60px;
    }
    
    .fingerprint-container {
        min-height: 150px;
    }
    
    .test-item {
        flex-direction: column;
        align-items: flex-start;
        padding: 10px;
    }
    
    .test-price {
        margin: 5px 0;
        text-align: left;
    }
    
    .cart-item {
        flex-direction: column;
        align-items: flex-start;
    }
}
</style>

<!-- ============================================= -->
<!-- HTML STRUCTURE -->
<!-- ============================================= -->
<div class="content-wrapper">
    <!-- Content Header -->
    <section class="content-header">
        <h1>
            <i class="fa fa-flask"></i> Laboratory Test Selection
            <small>Enhanced with patient verification and biometric security</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="<?php echo base_url(); ?>admin/laboratory">Laboratory</a></li>
            <li class="active">Test Selection</li>
        </ol>
    </section>

    <!-- Main Content -->
    <section class="content">
        
        <!-- ========================================= -->
        <!-- PATIENT SELECTION SECTION -->
        <!-- ========================================= -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-user-circle"></i> Patient Selection & Information
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- Patient Search and Selection -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="patient_search">
                                        <i class="fa fa-search"></i> Search Patient
                                    </label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="patient_search" 
                                               placeholder="Search by name, phone, or patient ID">
                                        <span class="input-group-btn">
                                            <button class="btn btn-primary" type="button" id="search_patient_btn">
                                                <i class="fa fa-search"></i> Search
                                            </button>
                                        </span>
                                    </div>
                                    <small class="text-muted">Enter at least 2 characters to search</small>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="selected_patient">
                                        <i class="fa fa-user"></i> Selected Patient
                                    </label>
                                    <select class="form-control select2" id="selected_patient" name="patient_id">
                                        <option value="">-- Select a patient --</option>
                                        <?php if (!empty($patients)): ?>
                                            <?php foreach ($patients as $patient): ?>
                                                <option value="<?php echo $patient['id']; ?>" 
                                                        data-phone="<?php echo $patient['mobileno']; ?>"
                                                        data-age="<?php echo $patient['age']; ?>"
                                                        data-gender="<?php echo $patient['gender']; ?>"
                                                        data-unique-id="<?php echo $patient['patient_unique_id'] ?? ''; ?>"
                                                        data-photo="<?php echo $patient['patient_photo'] ?? ''; ?>"
                                                        data-fingerprint="<?php echo !empty($patient['fingerprint_template']) ? '1' : '0'; ?>"
                                                        data-workflow="<?php echo $patient['patient_workflow'] ?? 'general'; ?>">
                                                    <?php echo $patient['patient_name']; ?> - <?php echo $patient['mobileno']; ?>
                                                </option>
                                            <?php endforeach; ?>
                                        <?php endif; ?>
                                    </select>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Patient Info Display -->
                        <div id="patient_info" class="patient-info-card" style="display: none;">
                            <div class="verification-status">
                                <span id="verification_badge" class="verification-badge pending-verification">
                                    <i class="fa fa-clock-o"></i> Pending Verification
                                </span>
                                <button type="button" id="startVerificationBtn" class="btn btn-xs btn-warning" style="margin-left: 10px;">
                                    <i class="fa fa-fingerprint"></i> Verify Patient
                                </button>
                            </div>
                            
                            <div class="row">
                                <!-- Patient Photo -->
                                <div class="col-md-2">
                                    <div id="patient_photo_container">
                                        <!-- Photo will be loaded here -->
                                    </div>
                                </div>
                                
                                <!-- Patient Details -->
                                <div class="col-md-10">
                                    <div class="row">
                                        <div class="col-md-3">
                                            <strong><i class="fa fa-user"></i> Name:</strong><br>
                                            <span id="patient_name_display"></span>
                                        </div>
                                        <div class="col-md-2">
                                            <strong><i class="fa fa-phone"></i> Phone:</strong><br>
                                            <span id="patient_phone_display"></span>
                                        </div>
                                        <div class="col-md-2">
                                            <strong><i class="fa fa-birthday-cake"></i> Age:</strong><br>
                                            <span id="patient_age_display"></span>
                                        </div>
                                        <div class="col-md-2">
                                            <strong><i class="fa fa-venus-mars"></i> Gender:</strong><br>
                                            <span id="patient_gender_display"></span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong><i class="fa fa-id-card"></i> Patient ID:</strong><br>
                                            <span id="patient_unique_id_display"></span>
                                        </div>
                                    </div>
                                    
                                    <!-- Additional Info Row -->
                                    <div class="row" style="margin-top: 10px;">
                                        <div class="col-md-4">
                                            <strong><i class="fa fa-cogs"></i> Workflow:</strong><br>
                                            <span id="patient_workflow_display" class="label label-info"></span>
                                        </div>
                                        <div class="col-md-4">
                                            <strong><i class="fa fa-fingerprint"></i> Biometric:</strong><br>
                                            <span id="fingerprint_status_display" class="label label-default">No Data</span>
                                        </div>
                                        <div class="col-md-4">
                                            <strong><i class="fa fa-shield"></i> Last Verified:</strong><br>
                                            <span id="last_verification_display" class="text-muted">Never</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- ========================================= -->
        <!-- PATIENT VERIFICATION SECTION -->
        <!-- ========================================= -->
        <div id="patient_verification_section" class="row" style="display: none;">
            <div class="col-md-12">
                <div class="box box-warning">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-shield"></i> Patient Identity Verification
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <!-- Fingerprint Verification Column -->
                            <div class="col-md-6">
                                <h4><i class="fa fa-fingerprint"></i> Biometric Verification</h4>
                                
                                <!-- Fingerprint Scanner Area -->
                                <div class="fingerprint-container" id="fingerprint_container">
                                    <div id="fingerprint_status">
                                        <i class="fa fa-fingerprint fa-3x text-muted"></i>
                                        <h4>SecuGen WebAPI Ready</h4>
                                        <p class="text-muted">Click "Scan Fingerprint" to capture using SGIFPCapture</p>
                                    </div>
                                    
                                    <div id="fingerprint_display" style="display: none;">
                                        <img id="fingerprint_image" class="fingerprint-image" 
                                             src="<?php echo base_url(); ?>Images/PlaceFinger.bmp" 
                                             alt="Fingerprint">
                                        <div class="quality-meter" style="margin-top: 10px; display: none;">
                                            <div id="quality_bar" class="quality-bar" style="width: 0%;"></div>
                                        </div>
                                        <small id="quality_text" class="text-muted"></small>
                                    </div>
                                </div>
                                
                                <!-- Biometric Controls -->
                                <div class="text-center" style="margin-top: 15px;">
                                    <button type="button" class="btn btn-primary" id="scanFingerprintBtn">
                                        <i class="fa fa-fingerprint"></i> Scan Fingerprint
                                    </button>
                                    <button type="button" class="btn btn-success" id="verifyFingerprintBtn" style="display: none;">
                                        <i class="fa fa-check"></i> Verify Identity
                                    </button>
                                    <button type="button" class="btn btn-danger" id="clearFingerprintBtn" style="display: none;">
                                        <i class="fa fa-trash"></i> Clear
                                    </button>
                                </div>
                            </div>
                            
                            <!-- Verification Status Column -->
                            <div class="col-md-6">
                                <h4><i class="fa fa-info-circle"></i> Verification Status</h4>
                                
                                <div class="verification-info">
                                    <div id="verification_details">
                                        <p class="text-muted">Select a patient to begin verification process.</p>
                                    </div>
                                    
                                    <div id="manual_verification" style="display: none;">
                                        <hr>
                                        <h5><i class="fa fa-hand-paper-o"></i> Manual Verification</h5>
                                        <p class="text-muted">If fingerprint verification fails or is unavailable:</p>
                                        <button type="button" class="btn btn-warning" id="manualVerifyBtn">
                                            <i class="fa fa-check-square"></i> Manual Override
                                        </button>
                                    </div>
                                    
                                    <div id="verification_history" style="margin-top: 15px; display: none;">
                                        <h5><i class="fa fa-history"></i> Recent Verifications</h5>
                                        <div id="verification_list" class="text-muted">
                                            No recent verifications found.
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Hidden Form Fields -->
                        <input type="hidden" id="fingerprint_template" name="fingerprint_template">
                        <input type="hidden" id="fingerprint_image_data" name="fingerprint_image_data">
                        <input type="hidden" id="verification_status" name="verification_status" value="pending">
                        <input type="hidden" id="verification_method" name="verification_method">
                        <input type="hidden" id="verification_timestamp" name="verification_timestamp">
                        <input type="hidden" id="verification_score" name="verification_score">
                        <input type="hidden" id="fingerprint_template" name="fingerprint_template">
                    </div>
                </div>
            </div>
        </div>

        <!-- ========================================= -->
        <!-- TEST SELECTION & CART SECTION -->
        <!-- ========================================= -->
        <div class="row">
            <!-- Test Selection Panel -->
            <div class="col-md-8">
                <div class="nav-tabs-custom">
                    <ul class="nav nav-tabs pull-right">
                        <li class="active">
                            <a href="#tab_test_selection" data-toggle="tab">
                                <i class="fa fa-list-ul"></i> Select Tests
                            </a>
                        </li>
                        <li class="pull-left header">
                            <i class="fa fa-flask"></i> Available Laboratory Tests 
                            <span class="badge bg-blue" id="total_tests_count">0</span>
                        </li>
                    </ul>
                    <div class="tab-content">
                        <div class="tab-pane active" id="tab_test_selection">
                            <div class="box-body">
                                <!-- Verification Alert -->
                                <div id="verification_alert" class="alert alert-warning" style="display: none;">
                                    <i class="fa fa-exclamation-triangle"></i>
                                    <strong>Patient verification required!</strong>
                                    Please verify the patient's identity before proceeding with test selection.
                                </div>
                                
                                <!-- Search Bar -->
                                <div class="form-group">
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="testSearch" 
                                               placeholder="Search for specific tests...">
                                        <span class="input-group-addon">
                                            <i class="fa fa-search"></i>
                                        </span>
                                    </div>
                                    <small class="text-muted">Type to filter tests across all categories</small>
                                </div>

                                <!-- Loading Indicator -->
                                <div id="tests_loading" class="text-center" style="padding: 40px;">
                                    <div class="spinner"></div>
                                    <p style="margin-top: 10px;">Loading laboratory tests...</p>
                                </div>

                                <!-- Test Categories Container -->
                                <div id="testCategories" style="display: none;">
                                    <!-- Categories will be populated here via JavaScript -->
                                </div>

                                <!-- No Results Message -->
                                <div id="no_results" style="display: none; text-align: center; padding: 40px; color: #6c757d;">
                                    <i class="fa fa-search fa-3x" style="margin-bottom: 15px; opacity: 0.3;"></i>
                                    <h4>No tests found</h4>
                                    <p>Try adjusting your search terms</p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Cart and Billing Panel -->
            <div class="col-md-4">
                <!-- Selected Tests Cart -->
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-shopping-cart"></i> Selected Tests
                            <span class="badge bg-green" id="cartCount">0</span>
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body" style="position: relative;">
                        <!-- Loading Overlay -->
                        <div id="cart_loading" class="loading-overlay" style="display: none;">
                            <div class="spinner"></div>
                        </div>

                        <!-- Selected Tests Container -->
                        <div id="selectedTests">
                            <div class="empty-cart">
                                <i class="fa fa-shopping-cart fa-3x"></i>
                                <p>No tests selected yet</p>
                                <small class="text-muted">Add tests from the categories on the left</small>
                            </div>
                        </div>

                        <!-- Billing Summary -->
                        <div class="billing-summary" id="billingSummary" style="display: none;">
                            <h4><i class="fa fa-calculator"></i> Billing Summary</h4>
                            
                            <div class="summary-row">
                                <span>Subtotal:</span>
                                <span id="subtotal">KES 0</span>
                            </div>
                            
                            <div class="form-group" style="margin-top: 15px;">
                                <label for="discountPercent">
                                    <i class="fa fa-percent"></i> Discount (%)
                                </label>
                                <input type="number" class="form-control" id="discountPercent" 
                                       min="0" max="100" value="0" placeholder="0" step="0.1">
                            </div>
                            
                            <div class="summary-row">
                                <span>Discount Amount:</span>
                                <span id="discountAmount">KES 0</span>
                            </div>
                            
                            <div class="form-group">
                                <label for="taxPercent">
                                    <i class="fa fa-money"></i> Tax (%)
                                </label>
                                <input type="number" class="form-control" id="taxPercent" 
                                       min="0" max="100" value="16" placeholder="16" step="0.1">
                            </div>
                            
                            <div class="summary-row">
                                <span>Tax Amount:</span>
                                <span id="taxAmount">KES 0</span>
                            </div>
                            
                            <div class="summary-row total-amount">
                                <span><i class="fa fa-money"></i> Total Amount:</span>
                                <span id="totalAmount">KES 0</span>
                            </div>
                        </div>

                        <!-- Test Options -->
                        <div id="labOptions" style="display: none; margin-top: 20px;">
                            <h4><i class="fa fa-cogs"></i> Test Options</h4>
                            
                            <div class="form-group">
                                <label for="test_priority">
                                    <i class="fa fa-exclamation-triangle"></i> Priority
                                </label>
                                <select class="form-control" id="test_priority">
                                    <option value="normal">Normal</option>
                                    <option value="urgent">Urgent (+50% fee)</option>
                                    <option value="stat">STAT (+100% fee)</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="assigned_doctor">
                                    <i class="fa fa-user-md"></i> Requesting Doctor
                                </label>
                                <select class="form-control" id="assigned_doctor">
                                    <option value="">-- Select Doctor --</option>
                                    <?php if (!empty($doctors)): ?>
                                        <?php foreach ($doctors as $doctor): ?>
                                            <option value="<?php echo $doctor['id']; ?>">
                                                <?php 
                                                $doctor_name = isset($doctor['full_name']) ? $doctor['full_name'] : 
                                                              trim(($doctor['name'] ?? '') . ' ' . ($doctor['surname'] ?? ''));
                                                echo $doctor_name ?: 'Doctor ' . $doctor['id'];
                                                ?>
                                            </option>
                                        <?php endforeach; ?>
                                    <?php endif; ?>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="test_date">
                                    <i class="fa fa-calendar"></i> Test Date
                                </label>
                                <input type="date" class="form-control" id="test_date" 
                                       value="<?php echo date('Y-m-d'); ?>" min="<?php echo date('Y-m-d'); ?>">
                            </div>
                            
                            <div class="form-group">
                                <label for="payment_method">
                                    <i class="fa fa-credit-card"></i> Payment Method
                                </label>
                                <select class="form-control" id="payment_method">
                                    <option value="cash">Cash</option>
                                    <option value="card">Card</option>
                                    <option value="insurance">Insurance</option>
                                    <option value="credit">Credit</option>
                                </select>
                            </div>
                            
                            <div class="form-group">
                                <label for="lab_notes">
                                    <i class="fa fa-sticky-note"></i> Additional Notes
                                </label>
                                <textarea class="form-control" id="lab_notes" rows="3" 
                                          placeholder="Enter any special instructions or notes..."></textarea>
                            </div>
                        </div>

                        <!-- Action Buttons -->
                        <div style="margin-top: 20px;">
                            <button class="btn btn-warning btn-block" id="clearCart" style="display: none;">
                                <i class="fa fa-trash"></i> Clear All Tests
                            </button>
                            <button class="btn btn-success btn-block" id="saveLabOrder" 
                                    style="display: none; margin-top: 10px;" disabled>
                                <i class="fa fa-save"></i> Save Laboratory Order
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Quick Instructions -->
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-lightbulb-o"></i> Quick Guide
                        </h3>
                    </div>
                    <div class="box-body">
                        <div class="list-group">
                            <div class="list-group-item">
                                <i class="fa fa-check text-green"></i> 
                                <strong>Step 1:</strong> Select a patient
                            </div>
                            <div class="list-group-item">
                                <i class="fa fa-shield text-orange"></i> 
                                <strong>Step 2:</strong> Verify patient identity
                            </div>
                            <div class="list-group-item">
                                <i class="fa fa-flask text-blue"></i> 
                                <strong>Step 3:</strong> Browse tests by category
                            </div>
                            <div class="list-group-item">
                                <i class="fa fa-shopping-cart text-green"></i> 
                                <strong>Step 4:</strong> Add tests to cart
                            </div>
                            <div class="list-group-item">
                                <i class="fa fa-cog text-gray"></i> 
                                <strong>Step 5:</strong> Configure options
                            </div>
                            <div class="list-group-item">
                                <i class="fa fa-save text-purple"></i> 
                                <strong>Step 6:</strong> Save order
                            </div>
                        </div>
                        
                        <div style="margin-top: 15px; padding: 10px; background: #f8f9fa; border-radius: 4px;">
                            <small class="text-muted">
                                <i class="fa fa-info-circle"></i> 
                                <strong>Security:</strong> Patient verification is required before test selection
                            </small>
                        </div>
                    </div>
                </div>

                <!-- Session Statistics -->
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-bar-chart"></i> Session Stats
                        </h3>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <div class="col-xs-6">
                                <div class="description-block border-right">
                                    <span class="description-percentage text-green">
                                        <i class="fa fa-flask"></i>
                                    </span>
                                    <h5 class="description-header" id="session_tests">0</h5>
                                    <span class="description-text">Tests Added</span>
                                </div>
                            </div>
                            <div class="col-xs-6">
                                <div class="description-block">
                                    <span class="description-percentage text-blue">
                                        <i class="fa fa-money"></i>
                                    </span>
                                    <h5 class="description-header" id="session_amount">KES 0</h5>
                                    <span class="description-text">Total Value</span>
                                </div>
                            </div>
                        </div>
                        <div class="row" style="margin-top: 10px;">
                            <div class="col-xs-12">
                                <div class="description-block">
                                    <span class="description-percentage" id="verification_status_icon">
                                        <i class="fa fa-shield text-muted"></i>
                                    </span>
                                    <h6 class="description-header" id="verification_status_text">Not Verified</h6>
                                    <span class="description-text">Patient Status</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- ============================================= -->
<!-- MODAL DIALOGS -->
<!-- ============================================= -->

<!-- Order Confirmation Modal -->
<div class="modal fade" id="confirmModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-check-circle"></i> Confirm Laboratory Order
                </h4>
            </div>
            <div class="modal-body">
                <!-- Patient Verification Status -->
                <div class="alert alert-info">
                    <div class="row">
                        <div class="col-md-8">
                            <i class="fa fa-shield"></i> 
                            <strong>Patient Verification:</strong> 
                            <span id="modal_verification_status">Verified</span>
                        </div>
                        <div class="col-md-4 text-right">
                            <span id="modal_verification_method" class="label label-success">Fingerprint</span>
                        </div>
                    </div>
                </div>
                
                <div class="row">
                    <div class="col-md-6">
                        <h5><i class="fa fa-user"></i> Patient Information</h5>
                        <table class="table table-condensed">
                            <tr><td><strong>Name:</strong></td><td id="confirm_patient_name"></td></tr>
                            <tr><td><strong>Phone:</strong></td><td id="confirm_patient_phone"></td></tr>
                            <tr><td><strong>Age/Gender:</strong></td><td id="confirm_patient_details"></td></tr>
                            <tr><td><strong>Workflow:</strong></td><td id="confirm_patient_workflow"></td></tr>
                        </table>
                    </div>
                    <div class="col-md-6">
                        <h5><i class="fa fa-cog"></i> Order Details</h5>
                        <table class="table table-condensed">
                            <tr><td><strong>Tests:</strong></td><td id="confirm_test_count"></td></tr>
                            <tr><td><strong>Priority:</strong></td><td id="confirm_priority"></td></tr>
                            <tr><td><strong>Doctor:</strong></td><td id="confirm_doctor"></td></tr>
                            <tr><td><strong>Total Amount:</strong></td><td id="confirm_total" class="text-green"></td></tr>
                        </table>
                    </div>
                </div>
                
                <h5><i class="fa fa-list"></i> Selected Tests</h5>
                <div id="confirm_tests_list" style="max-height: 200px; overflow-y: auto;"></div>
                
                <div class="alert alert-info">
                    <i class="fa fa-info-circle"></i> 
                    <strong>Note:</strong> This order will be saved and can be processed in the laboratory section.
                    Patient identity has been verified for security.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    <i class="fa fa-times"></i> Cancel
                </button>
                <button type="button" class="btn btn-primary" id="finalConfirmBtn">
                    <i class="fa fa-check"></i> Confirm & Save Order
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Manual Verification Modal -->
<div class="modal fade" id="manualVerificationModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-warning">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-hand-paper-o"></i> Manual Patient Verification
                </h4>
            </div>
            <div class="modal-body">
                <div class="alert alert-warning">
                    <i class="fa fa-exclamation-triangle"></i> 
                    <strong>Manual Override Required:</strong> Fingerprint verification is not available or has failed.
                    Please manually verify the patient's identity using alternative methods.
                </div>
                
                <div class="form-group">
                    <label>Patient Name:</label>
                    <p class="form-control-static" id="manual_patient_name"></p>
                </div>
                
                <div class="form-group">
                    <label>Phone Number:</label>
                    <p class="form-control-static" id="manual_patient_phone"></p>
                </div>
                
                <div class="form-group">
                    <label for="manual_verification_method">Verification Method:</label>
                    <select class="form-control" id="manual_verification_method" required>
                        <option value="">-- Select Verification Method --</option>
                        <option value="national_id">National ID Card</option>
                        <option value="passport">Passport</option>
                        <option value="driving_license">Driving License</option>
                        <option value="phone_verification">Phone Number Verification</option>
                        <option value="guardian_confirmation">Guardian/Family Confirmation</option>
                        <option value="previous_visit">Previous Visit Records</option>
                        <option value="staff_recognition">Staff Recognition</option>
                        <option value="other">Other Method</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label for="verification_details">Verification Details:</label>
                    <textarea class="form-control" id="verification_details" rows="3" 
                              placeholder="Enter ID number, verification code, or detailed explanation of verification method used"
                              required></textarea>
                </div>
                
                <div class="form-group">
                    <label for="verified_by_staff">Verified By Staff:</label>
                    <input type="text" class="form-control" id="verified_by_staff" 
                           value="<?php echo $this->session->userdata('username') ?? 'Current User'; ?>" readonly>
                </div>
                
                <div class="checkbox">
                    <label>
                        <input type="checkbox" id="manual_verification_confirm" required> 
                        <strong>I confirm that I have manually verified the patient's identity using the above method 
                        and I am satisfied that this is the correct patient.</strong>
                    </label>
                </div>
                
                <div class="alert alert-info">
                    <i class="fa fa-info-circle"></i> 
                    <strong>Security Notice:</strong> Manual verification will be logged for audit purposes.
                    Ensure you are confident in the patient's identity before proceeding.
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    <i class="fa fa-times"></i> Cancel
                </button>
                <button type="button" class="btn btn-warning" id="confirmManualVerificationBtn">
                    <i class="fa fa-check"></i> Confirm Manual Verification
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Fingerprint Verification Modal -->
<div class="modal fade" id="fingerprintModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header bg-primary">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-fingerprint"></i> Patient Fingerprint Verification
                </h4>
            </div>
            <div class="modal-body">
                <div class="text-center">
                    <div id="secugen_status" class="alert alert-info">
                        <i class="fa fa-info-circle"></i> Initializing fingerprint scanner...
                    </div>
                    
                    <div style="margin: 20px 0;">
                        <div id="fingerprint_image" style="width: 200px; height: 200px; margin: 0 auto 20px; 
                            border: 2px dashed #ccc; display: flex; align-items: center; justify-content: center;">
                            <i class="fa fa-fingerprint" style="font-size: 80px; color: #ccc;"></i>
                        </div>
                        <p id="fingerprint_status">Ready to scan</p>
                    </div>
                    
                    <div class="btn-group" style="margin-bottom: 15px;">
                        <button type="button" id="captureFingerprintBtn" class="btn btn-primary" 
                            onclick="startFingerprintCapture()">
                            <i class="fa fa-hand-pointer-o"></i> Capture Fingerprint
                        </button>
                        <button type="button" id="verifyFingerprintBtn" class="btn btn-success" 
                            onclick="verifyPatientFingerprint()" disabled>
                            <i class="fa fa-check"></i> Verify
                        </button>
                    </div>
                    
                    <div class="alert alert-warning" style="margin-top: 15px;">
                        <small>
                            <i class="fa fa-info-circle"></i> 
                            Place finger on the scanner and click 'Capture Fingerprint'. 
                            Ensure finger is clean and dry for best results.
                        </small>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    <i class="fa fa-times"></i> Close
                </button>
            </div>
        </div>
    </div>
</div>

<!-- ============================================= -->
<!-- JAVASCRIPT APPLICATION -->
<!-- ============================================= -->
<script>
'use strict';

/* =========================================
   GLOBAL VARIABLES & CONFIGURATION
   ========================================= */
   
// Application State
var selectedTests = [];
var allTestsData = {};
var isLoading = false;

// Patient Verification State
var verificationState = {
    isVerified: false,
    verificationMethod: null,
    verificationTimestamp: null,
    patientId: null,
    verificationScore: 0,
    attempts: 0,
    maxAttempts: 3,
    requiresVerification: true
};

// SecuGen WebAPI Configuration (matching existing system)
window.secugenConfig = {
    baseUrl: 'https://localhost:8443',
    httpUrl: 'http://localhost:8443',
    endpoints: {
        capture: '/SGIFPCapture',
        matchScore: '/SGIMatchScore',
        getInfo: '/SGIFPGetInfo',
        getVersion: '/SGIFPGetVersion'
    },
    serviceActive: false,
    quality: 50,
    timeout: 20000,
    lastError: null
};

// Laboratory System State
window.laboratoryState = {
    verificationPassed: false,
    selectedPatientId: null,
    storedTemplate: null,
    liveTemplate: null,
    lastVerificationResult: null,
    verificationHistory: [],
    currentMatchScore: 0,
    verificationQuality: 'Unknown'
};

/* =========================================
   DOCUMENT READY & INITIALIZATION
   ========================================= */
   
$(document).ready(function() {
    try {
        console.log('🧪 Enhanced Laboratory Test Selection with SecuGen WebAPI Starting...');
        
        // Core Initialization
        initializeApplication();
        initializePatientVerification();
        initializeTestSelection();
        
        // Event Handlers
        bindPatientEvents();
        bindVerificationEvents();
        bindTestSelectionEvents();
        bindOrderEvents();
        
        // UI Enhancement
        initializeUIComponents();
        checkSecuGenAvailability();
        
        // Display Console Help
        displayConsoleHelp();
        
        console.log('✅ Laboratory Test Selection System Initialized Successfully');
        
    } catch (e) {
        console.error('❌ Critical initialization error:', e);
        showAlert('Application initialization failed. Please refresh the page.', 'error');
    }
});

/* =========================================
   INITIALIZATION FUNCTIONS
   ========================================= */
   
function initializeApplication() {
    // Reset all states
    selectedTests = [];
    allTestsData = {};
    isLoading = false;
    
    // Hide sections initially
    $('#patient_verification_section').hide();
    $('#labOptions').hide();
    $('#verification_alert').show();
    
    // Update UI
    updateSessionStats();
    updateTestSelectionAccess();
}

function initializePatientVerification() {
    // Reset verification state
    resetVerificationState();
    
    // Initialize verification UI
    updateVerificationBadge('pending');
    updateVerificationDetails('info', 'Patient verification required', 
        'Please verify patient identity before proceeding with test selection');
}

function initializeTestSelection() {
    // Load test categories
    loadTestCategories();
    
    // Initialize search
    $('#testSearch').val('');
    
    // Reset cart
    updateCartDisplay();
}

function initializeUIComponents() {
    // Initialize Select2 if available
    if ($.fn.select2) {
        try {
            $('#selected_patient').select2({
                placeholder: "Select a patient",
                allowClear: true
            });
        } catch (e) {
            console.warn('Select2 initialization failed:', e);
        }
    }
}

/* =========================================
   EVENT BINDING FUNCTIONS
   ========================================= */
   
function bindPatientEvents() {
    // Patient selection change
    $('#selected_patient').on('change', function() {
        try {
            handlePatientSelection();
        } catch (e) {
            console.error('Error in patient selection handler:', e);
            showAlert('Error selecting patient. Please try again.', 'error');
        }
    });
    
    // Patient search
    $('#search_patient_btn').on('click', function() {
        try {
            searchPatients();
        } catch (e) {
            console.error('Error in patient search:', e);
            showAlert('Error searching patients. Please try again.', 'error');
        }
    });
    
    $('#patient_search').on('keypress', function(e) {
        if (e.which === 13) {
            try {
                searchPatients();
            } catch (e) {
                console.error('Error in patient search keypress:', e);
            }
        }
    });
}

function bindVerificationEvents() {
    // Fingerprint verification events
    $('#scanFingerprintBtn').on('click', function() {
        try {
            startFingerprintCapture();
        } catch (e) {
            console.error('Error starting fingerprint capture:', e);
            showAlert('Error starting fingerprint capture. Please try again.', 'error');
        }
    });
    
    $('#verifyFingerprintBtn').on('click', function() {
        try {
            verifyPatientFingerprint();
        } catch (e) {
            console.error('Error verifying fingerprint:', e);
            showAlert('Error verifying fingerprint. Please try again.', 'error');
        }
    });
    
    $('#clearFingerprintBtn').on('click', function() {
        try {
            clearFingerprintData();
        } catch (e) {
            console.error('Error clearing fingerprint:', e);
        }
    });
    
    // Manual verification events
    $('#manualVerifyBtn').on('click', function() {
        try {
            showManualVerificationModal();
        } catch (e) {
            console.error('Error showing manual verification modal:', e);
            showAlert('Error opening verification dialog. Please try again.', 'error');
        }
    });
    
    $('#confirmManualVerificationBtn').on('click', function() {
        try {
            processManualVerification();
        } catch (e) {
            console.error('Error processing manual verification:', e);
            showAlert('Error processing verification. Please try again.', 'error');
        }
    });
}

function bindTestSelectionEvents() {
    // Test selection events (using event delegation)
    $(document).on('click', '.btn-add-test', function() {
        if ($(this).prop('disabled')) return;
        
        if (!verificationState.isVerified) {
            showAlert('Patient verification is required before adding tests', 'warning');
            return;
        }
        
        var testData = JSON.parse($(this).attr('data-test'));
        addTestToCart(testData);
        $(this).text('Added').prop('disabled', true).addClass('disabled');
    });

    $(document).on('click', '.btn-remove-test', function() {
        var testIndex = parseInt($(this).attr('data-index'));
        removeTestFromCart(testIndex);
    });

    // Search functionality
    $('#testSearch').on('input', function() {
        var searchTerm = $(this).val().toLowerCase();
        filterTests(searchTerm);
    });

    // Billing calculations
    $('#discountPercent, #taxPercent, #test_priority').on('input change', function() {
        updateBillingSummary();
    });

    // Cart management
    $('#clearCart').on('click', function() {
        if (confirm('Are you sure you want to clear all selected tests?')) {
            clearCart();
        }
    });
}

function bindOrderEvents() {
    // Save lab order
    $('#saveLabOrder').on('click', function() {
        try {
            if (verificationState.isVerified) {
                showConfirmationModal();
            } else {
                showAlert('Patient verification is required before saving orders', 'warning');
                $('#patient_verification_section')[0].scrollIntoView({ behavior: 'smooth' });
            }
        } catch (e) {
            console.error('Error saving lab order:', e);
            showAlert('Error preparing order. Please try again.', 'error');
        }
    });
    
    // Final confirmation
    $('#finalConfirmBtn').on('click', function() {
        try {
            saveLabOrder();
        } catch (e) {
            console.error('Error in final confirmation:', e);
            showAlert('Error saving order. Please try again.', 'error');
        }
    });
}

/* =========================================
   SECUGEN WEBAPI FUNCTIONS
   ========================================= */
   
function checkSecuGenAvailability() {
    logDiagnostic("Initializing SecuGen WebAPI - Laboratory Module", 'info');
    
    // Enable capture buttons by default (same as existing system)
    $('#scanFingerprintBtn').prop('disabled', false);
    $('#verifyFingerprintBtn').prop('disabled', true);
    
    updateFingerprintUI('ready');
    
    // Test basic connectivity
    setTimeout(() => {
        $.ajax({
            url: window.secugenConfig.baseUrl + window.secugenConfig.endpoints.getInfo,
            type: 'GET',
            timeout: 3000,
            success: function(data) {
                window.secugenConfig.serviceActive = true;
                logDiagnostic("✓ SecuGen service detected and ready for laboratory", 'success');
                updateFingerprintUI('ready');
            },
            error: function() {
                window.secugenConfig.serviceActive = false;
                logDiagnostic("SecuGen service not detected - capture will try direct connection", 'warning');
                updateFingerprintUI('ready'); // Still allow attempts
            }
        });
    }, 1000);
}

function startFingerprintCapture() {
    logDiagnostic("Starting SGIFPCapture (same as patient registration)", 'info');
    
    updateFingerprintUI('scanning');
    $('#scanFingerprintBtn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Scanning...');
    
    // Use the exact same approach as existing system
    const captureUrl = 'https://localhost:8443/SGIFPCapture';
    
    $.ajax({
        url: captureUrl,
        type: 'POST',
        data: {
            Quality: window.secugenConfig.quality,
            Timeout: window.secugenConfig.timeout,
            WSQCompression: 2.25
        },
        timeout: window.secugenConfig.timeout + 5000,
        success: function(response) {
            $('#scanFingerprintBtn').prop('disabled', false).html('<i class="fa fa-fingerprint"></i> Scan Fingerprint');
            
            try {
                const result = typeof response === 'string' ? JSON.parse(response) : response;
                handleFingerprintCaptureResponse(result);
            } catch (e) {
                handleFingerprintCaptureError('Invalid response from SGIFPCapture: ' + e.message);
            }
        },
        error: function(xhr, status, error) {
            $('#scanFingerprintBtn').prop('disabled', false).html('<i class="fa fa-fingerprint"></i> Scan Fingerprint');
            
            let errorMsg = 'SGIFPCapture failed';
            if (status === 'timeout') {
                errorMsg = 'Scanner timeout - no finger detected';
            } else if (xhr.status === 0) {
                errorMsg = 'Cannot connect to SecuGen service. Please check installation.';
            } else {
                errorMsg = `HTTP ${xhr.status}: ${error}`;
            }
            
            handleFingerprintCaptureError(errorMsg);
        },
        complete: function() {
            $('#verifyFingerprintBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Verify Identity');
        }
    });
}

function handleFingerprintCaptureResponse(result) {
    if (result.ErrorCode === 0 && result.TemplateBase64) {
        // Store templates in both old and new formats for compatibility
        $('#fingerprint_template').val(result.TemplateBase64);
        $('#fingerprint_image_data').val(result.BMPBase64);
        
        // Update laboratory state
        window.laboratoryState.liveTemplate = result.TemplateBase64;
        
        // Display fingerprint image
        if (result.BMPBase64) {
            $('#fingerprint_image').attr('src', 'data:image/bmp;base64,' + result.BMPBase64);
            $('#fingerprint_display').show();
            
            // Show quality indicator
            const quality = result.Quality || 50;
            updateQualityIndicator(quality);
            window.laboratoryState.verificationQuality = quality;
        }
        
        updateFingerprintUI('success');
        $('#verifyFingerprintBtn, #clearFingerprintBtn').show();
        
        logDiagnostic(`✓ SGIFPCapture successful - Template: ${result.TemplateBase64 ? result.TemplateBase64.length : 0} chars, Image: ${result.BMPBase64 ? result.BMPBase64.length : 0} chars`, 'success');
        showAlert('Fingerprint captured successfully', 'success');
        
    } else {
        let errorMsg = 'Fingerprint capture error';
        switch(result.ErrorCode) {
            case 1: errorMsg = 'Reader not found or driver error'; break;
            case 2: errorMsg = 'Function failed - check reader connection'; break;
            case 3: errorMsg = 'Invalid parameters'; break;
            case 5:
            case 6:
            case 7: errorMsg = 'Driver DLL load failed'; break;
            default: 
                errorMsg = result.ErrorDescription || `Error code: ${result.ErrorCode}`;
        }
        
        console.error('SGIFPCapture Error:', {
            errorCode: result.ErrorCode,
            errorDescription: result.ErrorDescription,
            message: errorMsg
        });
        
        handleFingerprintCaptureError(`${errorMsg} (Code: ${result.ErrorCode})`);
    }
}

function verifyPatientFingerprint() {
    if (!verificationState.patientId) {
        showAlert('No patient selected for verification', 'error');
        return;
    }
    
    const capturedTemplate = $('#fingerprint_template').val();
    if (!capturedTemplate) {
        showAlert('No fingerprint captured for verification', 'error');
        return;
    }
    
    $('#verifyFingerprintBtn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Verifying...');
    
    // Get stored fingerprint template from patient data
    var selectedOption = $('#selected_patient').find(':selected');
    var hasStoredFingerprint = selectedOption.data('fingerprint') == '1';
    
    if (!hasStoredFingerprint) {
        $('#verifyFingerprintBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Verify Identity');
        showFingerprintEnrollmentOption(capturedTemplate);
        return;
    }
    
    // Fetch the stored template from the database
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/getPatientFingerprint"); ?>',
        type: 'POST',
        data: { 
            patient_id: verificationState.patientId,
            action: 'get_template'
        },
        dataType: 'json',
        success: function(response) {
            console.log('Database Response:', response);
            
            if (response.status === 'success' && response.fingerprint_template) {
                // Store the template and proceed with matching
                window.laboratoryState.storedTemplate = response.fingerprint_template;
                
                logDiagnostic(`✓ Retrieved stored template (${response.fingerprint_template.length} chars)`, 'success');
                showAlert('Stored fingerprint retrieved, starting comparison...', 'info');
                
                // Now perform the actual matching using SGIMatchScore
                performSGIMatchScore(capturedTemplate, response.fingerprint_template);
                
            } else {
                $('#verifyFingerprintBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Verify Identity');
                showAlert(response.message || 'No stored fingerprint found for this patient', 'warning');
                showFingerprintEnrollmentOption(capturedTemplate);
            }
        },
        error: function(xhr, status, error) {
            $('#verifyFingerprintBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Verify Identity');
            console.error('Error fetching stored fingerprint:', xhr.responseText);
            showAlert('Error retrieving patient fingerprint data: ' + error, 'error');
        }
    });
}

function performSGIMatchScore(capturedTemplate, storedTemplate) {
    logDiagnostic("Starting SGIMatchScore verification", 'info');
    logDiagnostic(`Templates - Captured: ${capturedTemplate.length} chars, Stored: ${storedTemplate.length} chars`, 'info');
    
    // Use SGIMatchScore endpoint (same as existing system)
    const matchUrl = 'https://localhost:8443/SGIMatchScore';
    
    updateVerificationDetails('info', 'Comparing fingerprints...', 'SGIMatchScore algorithm in progress');
    
    $.ajax({
        url: matchUrl,
        type: 'POST',
        data: {
            Template1: capturedTemplate,
            Template2: storedTemplate,
            SecurityLevel: 4 // Same as existing system
        },
        timeout: 15000, // Increased timeout
        success: function(response) {
            try {
                const result = typeof response === 'string' ? JSON.parse(response) : response;
                console.log('SGIMatchScore Response:', result);
                handleSGIMatchScoreResult(result);
            } catch (e) {
                console.error('SGIMatchScore parsing error:', e);
                handleFingerprintMatchError('Invalid SGIMatchScore response format: ' + e.message);
            }
        },
        error: function(xhr, status, error) {
            let errorMsg = 'SGIMatchScore failed';
            if (status === 'timeout') {
                errorMsg = 'Match timeout - verification took too long';
            } else if (xhr.status === 0) {
                errorMsg = 'Cannot connect to SecuGen service. Please check WebAPI installation.';
            } else if (xhr.status === 404) {
                errorMsg = 'SGIMatchScore endpoint not found. Check SecuGen WebAPI installation.';
            } else {
                errorMsg = `HTTP ${xhr.status}: ${error}`;
            }
            console.error('SGIMatchScore AJAX Error:', {status, error, xhr: xhr.responseText});
            handleFingerprintMatchError(errorMsg);
        },
        complete: function() {
            $('#verifyFingerprintBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Verify Identity');
        }
    });
}

function handleSGIMatchScoreResult(result) {
    console.log('Processing SGIMatchScore Result:', result);
    
    if (result.ErrorCode === 0) {
        const matchScore = result.MatchingScore || 0;
        const isMatched = result.IsMatched || (matchScore >= 100);
        const threshold = 100; // SGIMatchScore typically uses 100 as threshold
        
        // Update laboratory state
        window.laboratoryState.currentMatchScore = matchScore;
        window.laboratoryState.lastVerificationResult = {
            score: matchScore,
            matched: isMatched,
            timestamp: new Date().toISOString(),
            errorCode: result.ErrorCode
        };
        
        logDiagnostic(`SGIMatchScore result: ${matchScore} (threshold: ${threshold}, matched: ${isMatched})`, isMatched ? 'success' : 'warning');
        
        if (isMatched) {
            // Verification successful
            logDiagnostic(`✓ Fingerprint verification SUCCESSFUL - Score: ${matchScore}%`, 'success');
            processSuccessfulVerification('fingerprint', matchScore);
        } else {
            // Verification failed
            logDiagnostic(`✗ Fingerprint verification FAILED - Score: ${matchScore}% (required: ${threshold}%)`, 'warning');
            processFailedVerification(matchScore, threshold);
        }
    } else {
        let errorMsg = 'SGIMatchScore error';
        switch(result.ErrorCode) {
            case 1: errorMsg = 'Reader not found or driver error'; break;
            case 2: errorMsg = 'Function failed - check reader connection'; break;
            case 3: errorMsg = 'Invalid parameters - check template format'; break;
            case 4: errorMsg = 'Invalid template data'; break;
            case 54: errorMsg = 'Timeout occurred during matching'; break;
            default: 
                errorMsg = result.ErrorDescription || `Unknown error code: ${result.ErrorCode}`;
        }
        
        console.error('SGIMatchScore Error:', {
            errorCode: result.ErrorCode,
            errorDescription: result.ErrorDescription,
            message: errorMsg
        });
        
        handleFingerprintMatchError(`${errorMsg} (Code: ${result.ErrorCode})`);
    }
}

/* =========================================
   PATIENT MANAGEMENT FUNCTIONS
   ========================================= */
   
function handlePatientSelection() {
    var selectedOption = $('#selected_patient').find(':selected');
    if (selectedOption.val()) {
        // Display patient information
        displayPatientInfo(selectedOption);
        
        // Show verification section
        $('#patient_verification_section').show();
        
        // Reset verification state for new patient
        resetVerificationState();
        verificationState.patientId = selectedOption.val();
        
        // Load patient verification history
        loadVerificationHistory(selectedOption.val());
        
        // Check if patient requires verification based on workflow
        var workflow = selectedOption.data('workflow') || 'general';
        verificationState.requiresVerification = (workflow === 'gcc') || true; // All patients require verification
        
        // Hide test selection until verified
        updateTestSelectionAccess();
        
    } else {
        $('#patient_info').slideUp();
        $('#patient_verification_section').hide();
        $('#labOptions').slideUp();
        resetVerificationState();
    }
}

function displayPatientInfo(selectedOption) {
    // Basic info
    $('#patient_name_display').text(selectedOption.text().split(' - ')[0]);
    $('#patient_phone_display').text(selectedOption.data('phone'));
    $('#patient_age_display').text(selectedOption.data('age'));
    $('#patient_gender_display').text(selectedOption.data('gender'));
    $('#patient_unique_id_display').text(selectedOption.data('unique-id') || 'N/A');
    
    // Workflow display
    var workflow = selectedOption.data('workflow') || 'general';
    $('#patient_workflow_display').text(workflow.toUpperCase()).removeClass().addClass('label label-info');
    
    // Fingerprint status
    var hasFingerprint = selectedOption.data('fingerprint') == '1';
    $('#fingerprint_status_display')
        .text(hasFingerprint ? 'Available' : 'Not Available')
        .removeClass()
        .addClass('label ' + (hasFingerprint ? 'label-success' : 'label-warning'));
    
    // Display patient photo
    displayPatientPhoto(selectedOption.data('photo'));
    
    $('#patient_info').slideDown();
}

function displayPatientPhoto(photoPath) {
    var photoContainer = $('#patient_photo_container');
    
    if (photoPath && photoPath.trim() !== '') {
        var fullPhotoPath = '<?php echo base_url(); ?>' + photoPath;
        photoContainer.html(
            '<img src="' + fullPhotoPath + '" class="patient-photo" alt="Patient Photo" ' +
            'onerror="this.style.display=\'none\'; this.nextElementSibling.style.display=\'flex\';">' +
            '<div class="patient-photo-placeholder" style="display: none;">' +
            '<i class="fa fa-user"></i></div>'
        );
    } else {
        photoContainer.html(
            '<div class="patient-photo-placeholder">' +
            '<i class="fa fa-user"></i></div>'
        );
    }
}

function searchPatients() {
    var searchTerm = $('#patient_search').val().trim();
    
    if (searchTerm.length < 2) {
        showAlert('Please enter at least 2 characters to search', 'warning');
        return;
    }
    
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/searchPatients"); ?>',
        type: 'POST',
        data: { search_term: searchTerm },
        dataType: 'json',
        beforeSend: function() {
            $('#search_patient_btn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Searching...');
        },
        success: function(response) {
            if (response.status === 'success') {
                updatePatientDropdown(response.data);
                showAlert('Found ' + response.count + ' patients', 'success');
            } else {
                showAlert(response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error searching patients', 'error');
        },
        complete: function() {
            $('#search_patient_btn').prop('disabled', false).html('<i class="fa fa-search"></i> Search');
        }
    });
}

function updatePatientDropdown(patients) {
    var patientSelect = $('#selected_patient');
    patientSelect.empty().append('<option value="">-- Select a patient --</option>');
    
    $.each(patients, function(index, patient) {
        patientSelect.append(
            $('<option>', {
                value: patient.id,
                text: patient.patient_name + ' - ' + patient.mobileno,
                'data-phone': patient.mobileno,
                'data-age': patient.age,
                'data-gender': patient.gender,
                'data-unique-id': patient.patient_unique_id,
                'data-photo': patient.patient_photo || '',
                'data-fingerprint': (patient.fingerprint_template && patient.fingerprint_template.trim() !== '') ? '1' : '0',
                'data-workflow': patient.patient_workflow || 'general'
            })
        );
    });
    
    // Refresh select2 if initialized
    if (patientSelect.hasClass('select2-hidden-accessible')) {
        patientSelect.trigger('change');
    }
}

/* =========================================
   VERIFICATION PROCESSING FUNCTIONS
   ========================================= */
   
function processSuccessfulVerification(method, score) {
    verificationState.isVerified = true;
    verificationState.verificationMethod = method;
    verificationState.verificationTimestamp = new Date().toISOString();
    verificationState.verificationScore = score;
    
    // Update UI
    updateVerificationBadge('verified');
    updateVerificationDetails('success', 'Patient identity verified successfully', 
        method + ' verification completed with ' + Math.round(score) + '% match');
    
    // Enable test selection
    updateTestSelectionAccess();
    
    // Save verification to database
    saveVerificationRecord();
    
    // Update session stats
    updateSessionStats();
    
    showAlert('Patient identity verified successfully (' + Math.round(score) + '% match)', 'success');
}

function processFailedVerification(score, threshold) {
    verificationState.attempts++;
    
    if (verificationState.attempts >= verificationState.maxAttempts) {
        updateVerificationDetails('error', 'Verification failed - Maximum attempts reached', 
            'Match score: ' + Math.round(score) + '% (required: ' + threshold + '%)');
        showManualVerificationModal();
    } else {
        updateVerificationDetails('warning', 'Verification failed - Please try again', 
            'Match score: ' + Math.round(score) + '% (required: ' + threshold + '%). ' +
            'Attempts remaining: ' + (verificationState.maxAttempts - verificationState.attempts));
        showAlert('Fingerprint verification failed. ' + 
            (verificationState.maxAttempts - verificationState.attempts) + ' attempts remaining.', 'warning');
    }
}

function showManualVerificationModal() {
    var patientData = $('#selected_patient option:selected');
    var patientName = patientData.text().split(' - ')[0];
    var patientPhone = patientData.data('phone');
    
    $('#manual_patient_name').text(patientName);
    $('#manual_patient_phone').text(patientPhone);
    
    // Clear previous form data
    $('#manual_verification_method').val('');
    $('#verification_details').val('');
    $('#manual_verification_confirm').prop('checked', false);
    
    $('#manualVerificationModal').modal('show');
}

function processManualVerification() {
    var method = $('#manual_verification_method').val();
    var details = $('#verification_details').val().trim();
    var confirmed = $('#manual_verification_confirm').is(':checked');
    
    if (!method) {
        showAlert('Please select a verification method', 'warning');
        return;
    }
    
    if (!details) {
        showAlert('Please provide verification details', 'warning');
        return;
    }
    
    if (!confirmed) {
        showAlert('Please confirm that you have verified the patient\'s identity', 'warning');
        return;
    }
    
    $('#confirmManualVerificationBtn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Processing...');
    
    // Save manual verification
    $.ajax({
        url: '<?php echo base_url("admin/patient/saveManualVerification"); ?>',
        type: 'POST',
        data: {
            patient_id: verificationState.patientId,
            verification_method: method,
            verification_details: details,
            verified_by: $('#verified_by_staff').val(),
            verification_timestamp: new Date().toISOString()
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                processSuccessfulVerification('manual', 100);
                $('#manualVerificationModal').modal('hide');
                showAlert('Manual verification completed successfully', 'success');
            } else {
                showAlert('Error saving verification: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error processing manual verification', 'error');
        },
        complete: function() {
            $('#confirmManualVerificationBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Confirm Manual Verification');
        }
    });
}

/* =========================================
   TEST SELECTION FUNCTIONS
   ========================================= */
   
function loadTestCategories() {
    $('#tests_loading').show();
    $('#testCategories').hide();
    
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/getLabTests"); ?>',
        type: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                allTestsData = response.data;
                renderTestCategories(response.data);
                $('#total_tests_count').text(response.count || 0);
            } else {
                showAlert('Error loading tests: ' + response.message, 'error');
                renderTestCategories(getStaticTestData());
            }
        },
        error: function() {
            showAlert('Failed to load tests, using offline data', 'warning');
            renderTestCategories(getStaticTestData());
        },
        complete: function() {
            $('#tests_loading').hide();
            $('#testCategories').show();
        }
    });
}

function getStaticTestData() {
    // Fallback static data
    return {
        "Hematology": [
            { "name": "Complete Blood Count (CBC)", "charge": 1200, "method": "Automated", "sample_type": "Blood" },
            { "name": "Blood Culture", "charge": 6000, "method": "Culture", "sample_type": "Blood" },
            { "name": "ESR", "charge": 500, "method": "Westergren", "sample_type": "Blood" },
            { "name": "Hemoglobin", "charge": 500, "method": "Spectrophotometry", "sample_type": "Blood" }
        ],
        "Endocrinology": [
            { "name": "TSH", "charge": 2000, "method": "ELISA", "sample_type": "Serum" },
            { "name": "FT3", "charge": 2000, "method": "ELISA", "sample_type": "Serum" },
            { "name": "FT4", "charge": 2000, "method": "ELISA", "sample_type": "Serum" }
        ],
        "Diabetes": [
            { "name": "HBA1C", "charge": 3000, "method": "HPLC", "sample_type": "Blood" },
            { "name": "Fasting Blood Sugar", "charge": 500, "method": "Enzymatic", "sample_type": "Serum" }
        ]
    };
}

function renderTestCategories(testData) {
    var container = $('#testCategories');
    container.empty();

    if ($.isEmptyObject(testData)) {
        container.html('<div class="alert alert-warning"><i class="fa fa-exclamation-triangle"></i> No test categories available</div>');
        return;
    }

    $.each(testData, function(category, tests) {
        var categoryCard = createCategoryCard(category, tests);
        container.append(categoryCard);
    });
    
    // Initialize collapse functionality
    container.find('.category-header').on('click', function() {
        var target = $(this).data('target');
        $(target).collapse('toggle');
        
        var icon = $(this).find('.fa-chevron-down, .fa-chevron-right');
        setTimeout(function() {
            if ($(target).hasClass('in')) {
                icon.removeClass('fa-chevron-right').addClass('fa-chevron-down');
            } else {
                icon.removeClass('fa-chevron-down').addClass('fa-chevron-right');
            }
        }, 350);
    });
}

function createCategoryCard(category, tests) {
    var categoryId = category.replace(/\s+/g, '_').toLowerCase();
    
    var categoryCard = $('<div>', {
        'class': 'test-category-card',
        'data-category': category
    });
    
    var header = $('<div>', {
        'class': 'category-header',
        'data-toggle': 'collapse',
        'data-target': '#' + categoryId,
        html: '<i class="fa fa-chevron-down pull-right"></i>' +
              '<strong>' + category + '</strong>' +
              '<span class="category-count">' + tests.length + ' tests</span>'
    });
    
    var collapseDiv = $('<div>', {
        'id': categoryId,
        'class': 'collapse'
    });
    
    var testsDiv = $('<div>', {
        'class': 'category-tests'
    });
    
    $.each(tests, function(index, test) {
        var testItem = createTestItem(test, category);
        testsDiv.append(testItem);
    });
    
    collapseDiv.append(testsDiv);
    categoryCard.append(header).append(collapseDiv);
    
    return categoryCard;
}

function createTestItem(test, category) {
    var testItem = $('<div>', {
        'class': 'test-item',
        'data-test-name': test.name.toLowerCase(),
        'data-category': category
    });
    
    var testName = $('<div>', {
        'class': 'test-name',
        html: test.name + (test.method ? '<br><small class="text-muted">' + test.method + ' | ' + test.sample_type + '</small>' : '')
    });
    
    var testPrice = $('<div>', {
        'class': 'test-price',
        text: 'KES ' + test.charge.toLocaleString()
    });
    
    var isSelected = selectedTests.some(selected => 
        selected.name === test.name && selected.category === category
    );
    
    var addButton = $('<button>', {
        'class': 'btn-add-test',
        text: isSelected ? 'Added' : 'Add',
        disabled: isSelected || !verificationState.isVerified,
        'data-test': JSON.stringify({...test, category: category})
    });
    
    testItem.append(testName).append(testPrice).append(addButton);
    return testItem;
}

function addTestToCart(test) {
    var existingIndex = selectedTests.findIndex(selected => 
        selected.name === test.name && selected.category === test.category
    );

    if (existingIndex === -1) {
        selectedTests.push({
            name: test.name,
            charge: test.charge,
            category: test.category,
            method: test.method || '',
            sample_type: test.sample_type || '',
            id: Date.now() + Math.random()
        });
        
        updateCartDisplay();
        updateBillingSummary();
        updateSessionStats();
        showAlert(test.name + ' added to cart', 'success');
    }
}

function removeTestFromCart(index) {
    var removedTest = selectedTests.splice(index, 1)[0];
    updateCartDisplay();
    updateBillingSummary();
    updateSessionStats();
    
    // Re-enable the add button
    $('.btn-add-test').each(function() {
        try {
            var testData = JSON.parse($(this).attr('data-test'));
            if (testData.name === removedTest.name && testData.category === removedTest.category) {
                $(this).text('Add').prop('disabled', !verificationState.isVerified).removeClass('disabled');
            }
        } catch (e) {
            // Ignore parsing errors
        }
    });
    
    showAlert(removedTest.name + ' removed from cart', 'info');
}

/* =========================================
   ORDER MANAGEMENT FUNCTIONS
   ========================================= */
   
function showConfirmationModal() {
    // Validate required fields
    if (!$('#selected_patient').val()) {
        showAlert('Please select a patient first', 'warning');
        return;
    }
    
    if (selectedTests.length === 0) {
        showAlert('Please select at least one test', 'warning');
        return;
    }
    
    if (!verificationState.isVerified) {
        showAlert('Patient verification is required before saving orders', 'warning');
        return;
    }
    
    // Populate confirmation modal
    var selectedPatient = $('#selected_patient option:selected');
    $('#confirm_patient_name').text(selectedPatient.text().split(' - ')[0]);
    $('#confirm_patient_phone').text(selectedPatient.data('phone'));
    $('#confirm_patient_details').text(selectedPatient.data('age') + ' years, ' + selectedPatient.data('gender'));
    $('#confirm_patient_workflow').text((selectedPatient.data('workflow') || 'general').toUpperCase());
    
    $('#confirm_test_count').text(selectedTests.length + ' tests');
    
    var priority = $('#test_priority').val();
    var priorityText = priority.charAt(0).toUpperCase() + priority.slice(1);
    if (priority !== 'normal') {
        priorityText += ' <span class="priority-badge priority-' + priority + '">' + priority.toUpperCase() + '</span>';
    }
    $('#confirm_priority').html(priorityText);
    
    // Doctor name
    var doctorText = $('#assigned_doctor option:selected').text() || 'Not assigned';
    $('#confirm_doctor').text(doctorText);
    
    var billingInfo = calculateBillingInfo();
    $('#confirm_total').text('KES ' + Math.round(billingInfo.totalAmount).toLocaleString());
    
    // Show selected tests
    var testsListHtml = '<div class="table-responsive"><table class="table table-condensed table-striped">';
    testsListHtml += '<thead><tr><th>Test Name</th><th>Category</th><th class="text-right">Price</th></tr></thead><tbody>';
    
    $.each(selectedTests, function(index, test) {
        testsListHtml += '<tr>' +
            '<td>' + test.name + '</td>' +
            '<td><small class="text-muted">' + test.category + '</small></td>' +
            '<td class="text-right">KES ' + test.charge.toLocaleString() + '</td>' +
            '</tr>';
    });
    
    testsListHtml += '</tbody></table></div>';
    $('#confirm_tests_list').html(testsListHtml);
    
    // Show verification status in modal
    $('#modal_verification_status').text('Verified ✓');
    $('#modal_verification_method').text(verificationState.verificationMethod.toUpperCase()).removeClass().addClass('label label-success');
    
    // Show modal
    $('#confirmModal').modal('show');
}

function saveLabOrder() {
    // Final validation
    if (!$('#selected_patient').val() || selectedTests.length === 0 || !verificationState.isVerified) {
        showAlert('Missing required information or verification', 'error');
        return;
    }
    
    // Calculate billing info
    var billingInfo = calculateBillingInfo();
    
    // Prepare data
    var orderData = {
        patient_id: $('#selected_patient').val(),
        selected_tests: JSON.stringify(selectedTests),
        billing_info: JSON.stringify(billingInfo),
        doctor_id: $('#assigned_doctor').val(),
        test_date: $('#test_date').val(),
        priority: $('#test_priority').val(),
        payment_method: $('#payment_method').val(),
        notes: $('#lab_notes').val(),
        verification_data: JSON.stringify({
            is_verified: verificationState.isVerified,
            verification_method: verificationState.verificationMethod,
            verification_timestamp: verificationState.verificationTimestamp,
            verification_score: verificationState.verificationScore
        })
    };
    
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/saveMultipleTests"); ?>',
        type: 'POST',
        data: orderData,
        dataType: 'json',
        beforeSend: function() {
            $('#finalConfirmBtn').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Saving Order...');
            $('#cart_loading').show();
        },
        success: function(response) {
            if (response.status === 'success') {
                $('#confirmModal').modal('hide');
                
                showAlert('Laboratory order saved successfully! Order includes ' + response.test_count + ' tests.', 'success');
                
                // Clear the cart after successful save
                setTimeout(function() {
                    clearCart();
                    resetForm();
                }, 2000);
                
                // Optionally redirect to lab management page
                if (confirm('Order saved successfully! Would you like to go to the Laboratory Management page?')) {
                    window.location.href = '<?php echo base_url("admin/laboratory"); ?>';
                }
            } else {
                showAlert('Error: ' + response.message, 'error');
            }
        },
        error: function(xhr, status, error) {
            var errorMsg = 'Error saving laboratory order';
            try {
                var response = JSON.parse(xhr.responseText);
                errorMsg = response.message || errorMsg;
            } catch (e) {
                errorMsg += ': ' + error;
            }
            showAlert(errorMsg, 'error');
        },
        complete: function() {
            $('#finalConfirmBtn').prop('disabled', false).html('<i class="fa fa-check"></i> Confirm & Save Order');
            $('#cart_loading').hide();
        }
    });
}

/* =========================================
   UTILITY FUNCTIONS
   ========================================= */
   
function updateCartDisplay() {
    var container = $('#selectedTests');
    var cartCount = $('#cartCount');
    var clearBtn = $('#clearCart');
    var saveBtn = $('#saveLabOrder');
    var billingSummary = $('#billingSummary');

    cartCount.text(selectedTests.length);

    if (selectedTests.length === 0) {
        container.html(
            '<div class="empty-cart">' +
            '<i class="fa fa-shopping-cart fa-3x"></i>' +
            '<p>No tests selected yet</p>' +
            '<small class="text-muted">Add tests from the categories on the left</small>' +
            '</div>'
        );
        clearBtn.hide();
        saveBtn.hide();
        billingSummary.hide();
    } else {
        var cartHtml = '';
        $.each(selectedTests, function(index, test) {
            cartHtml += 
                '<div class="cart-item">' +
                '<div class="cart-item-name">' + 
                test.name + 
                '<br><small class="text-muted">' + test.category + '</small>' +
                '</div>' +
                '<div class="cart-item-price">KES ' + test.charge.toLocaleString() + '</div>' +
                '<button class="btn btn-xs btn-danger btn-remove-test" data-index="' + index + '" title="Remove test">' +
                '<i class="fa fa-times"></i></button>' +
                '</div>';
        });

        container.html(cartHtml);
        clearBtn.show();
        saveBtn.show();
        billingSummary.show();
    }
}

function updateBillingSummary() {
    if (selectedTests.length === 0) return;

    var subtotal = selectedTests.reduce(function(sum, test) {
        return sum + test.charge;
    }, 0);
    
    // Apply priority fees
    var priority = $('#test_priority').val();
    var priorityMultiplier = 1.0;
    
    if (priority === 'urgent') {
        priorityMultiplier = 1.5;
    } else if (priority === 'stat') {
        priorityMultiplier = 2.0;
    }
    
    var priorityAdjustedTotal = subtotal * priorityMultiplier;
    
    var discountPercent = parseFloat($('#discountPercent').val()) || 0;
    var taxPercent = parseFloat($('#taxPercent').val()) || 0;

    var discountAmount = (priorityAdjustedTotal * discountPercent) / 100;
    var taxableAmount = priorityAdjustedTotal - discountAmount;
    var taxAmount = (taxableAmount * taxPercent) / 100;
    var totalAmount = taxableAmount + taxAmount;

    $('#subtotal').text('KES ' + Math.round(priorityAdjustedTotal).toLocaleString());
    $('#discountAmount').text('KES ' + Math.round(discountAmount).toLocaleString());
    $('#taxAmount').text('KES ' + Math.round(taxAmount).toLocaleString());
    $('#totalAmount').text('KES ' + Math.round(totalAmount).toLocaleString());
}

function updateSessionStats() {
    $('#session_tests').text(selectedTests.length);
    
    var totalAmount = selectedTests.reduce(function(sum, test) {
        return sum + test.charge;
    }, 0);
    
    $('#session_amount').text('KES ' + totalAmount.toLocaleString());
    
    // Update verification status in stats
    if (verificationState.isVerified) {
        $('#verification_status_icon').html('<i class="fa fa-shield text-success"></i>');
        $('#verification_status_text').text('Verified').removeClass().addClass('description-header text-success');
    } else {
        $('#verification_status_icon').html('<i class="fa fa-shield text-muted"></i>');
        $('#verification_status_text').text('Not Verified').removeClass().addClass('description-header text-muted');
    }
}

function calculateBillingInfo() {
    var subtotal = selectedTests.reduce(function(sum, test) {
        return sum + test.charge;
    }, 0);
    
    var priority = $('#test_priority').val();
    var priorityMultiplier = priority === 'urgent' ? 1.5 : priority === 'stat' ? 2.0 : 1.0;
    var priorityAdjustedTotal = subtotal * priorityMultiplier;
    
    var discountPercent = parseFloat($('#discountPercent').val()) || 0;
    var taxPercent = parseFloat($('#taxPercent').val()) || 0;

    var discountAmount = (priorityAdjustedTotal * discountPercent) / 100;
    var taxableAmount = priorityAdjustedTotal - discountAmount;
    var taxAmount = (taxableAmount * taxPercent) / 100;
    var totalAmount = taxableAmount + taxAmount;

    return {
        subtotal: priorityAdjustedTotal,
        discountPercent: discountPercent,
        discountAmount: discountAmount,
        taxPercent: taxPercent,
        taxAmount: taxAmount,
        totalAmount: totalAmount
    };
}

function showAlert(message, type) {
    var alertClass = type === 'success' ? 'alert-success' : 
                     type === 'warning' ? 'alert-warning' : 
                     type === 'error' ? 'alert-danger' : 'alert-info';
    
    var iconClass = type === 'success' ? 'fa-check-circle' :
                    type === 'warning' ? 'fa-exclamation-triangle' :
                    type === 'error' ? 'fa-times-circle' : 'fa-info-circle';
    
    var alert = $('<div>', {
        'class': 'alert ' + alertClass + ' alert-dismissible alert-custom',
        html: '<button type="button" class="close" data-dismiss="alert" aria-label="Close">' +
              '<span aria-hidden="true">&times;</span></button>' +
              '<i class="fa ' + iconClass + '"></i> ' + message
    });
    
    $('body').append(alert);
    
    // Auto-dismiss after 5 seconds
    setTimeout(function() {
        alert.fadeOut(400, function() {
            $(this).remove();
        });
    }, 5000);
}

function resetForm() {
    // Reset patient selection
    $('#selected_patient').val('').trigger('change');
    $('#patient_search').val('');
    $('#patient_info').hide();
    $('#patient_verification_section').hide();
    $('#labOptions').hide();
    
    // Reset verification state
    resetVerificationState();
    
    // Reset form fields
    $('#discountPercent').val(0);
    $('#taxPercent').val(16);
    $('#test_priority').val('normal');
    $('#assigned_doctor').val('');
    $('#test_date').val('<?php echo date('Y-m-d'); ?>');
    $('#payment_method').val('cash');
    $('#lab_notes').val('');
    
    // Clear search
    $('#testSearch').val('');
    filterTests('');
}

function clearCart() {
    selectedTests = [];
    updateCartDisplay();
    updateBillingSummary();
    updateSessionStats();
    
    // Re-enable all add buttons
    $('.btn-add-test').text('Add').prop('disabled', !verificationState.isVerified).removeClass('disabled');
    
    showAlert('Cart cleared successfully', 'info');
}

function filterTests(searchTerm) {
    if (!searchTerm) {
        $('.test-category-card').show();
        $('.test-item').show();
        $('#no_results').hide();
        return;
    }

    var hasResults = false;
    
    $('.test-category-card').each(function() {
        var $category = $(this);
        var $tests = $category.find('.test-item');
        var hasVisibleTests = false;

        $tests.each(function() {
            var testName = $(this).data('test-name');
            if (testName && testName.includes(searchTerm)) {
                $(this).show();
                hasVisibleTests = true;
                hasResults = true;
                
                // Highlight search term
                var $testNameDiv = $(this).find('.test-name');
                var originalText = $testNameDiv.text();
                var highlightedText = originalText.replace(
                    new RegExp('(' + searchTerm + ')', 'gi'),
                    '<span class="search-highlight" style="background-color: #fff3cd; padding: 2px 4px; border-radius: 3px;">$1</span>'
                );
                $testNameDiv.html(highlightedText);
            } else {
                $(this).hide();
            }
        });

        if (hasVisibleTests) {
            $category.show();
            $category.find('.collapse').addClass('in');
            $category.find('.fa-chevron-right').removeClass('fa-chevron-right').addClass('fa-chevron-down');
        } else {
            $category.hide();
        }
    });
    
    if (!hasResults) {
        $('#no_results').show();
    } else {
        $('#no_results').hide();
    }
}

/* =========================================
   UI UPDATE FUNCTIONS
   ========================================= */
   
function updateFingerprintUI(status) {
    var container = $('#fingerprint_container');
    var statusDiv = $('#fingerprint_status');
    
    container.removeClass('scanning success failed');
    
    switch (status) {
        case 'ready':
            statusDiv.html(
                '<i class="fa fa-fingerprint fa-3x text-primary"></i>' +
                '<h4>SecuGen WebAPI Ready</h4>' +
                '<p class="text-muted">Click "Scan Fingerprint" to capture using SGIFPCapture</p>'
            );
            $('#scanFingerprintBtn').prop('disabled', false);
            break;
            
        case 'unavailable':
            statusDiv.html(
                '<i class="fa fa-times-circle fa-3x text-danger"></i>' +
                '<h4>SecuGen Scanner Unavailable</h4>' +
                '<p class="text-muted">Please check SecuGen WebAPI installation and scanner connection</p>'
            );
            $('#scanFingerprintBtn').prop('disabled', true);
            $('#manual_verification').show();
            break;
            
        case 'scanning':
            container.addClass('scanning');
            statusDiv.html(
                '<i class="fa fa-spinner fa-spin fa-3x text-primary"></i>' +
                '<h4>SGIFPCapture in Progress...</h4>' +
                '<p class="text-muted">Please place finger on scanner</p>'
            );
            break;
            
        case 'success':
            container.addClass('success');
            statusDiv.html(
                '<i class="fa fa-check-circle fa-3x text-success"></i>' +
                '<h4>Fingerprint Captured Successfully</h4>' +
                '<p class="text-success">Ready for SGIMatchScore verification</p>'
            );
            break;
            
        case 'failed':
            container.addClass('failed');
            statusDiv.html(
                '<i class="fa fa-exclamation-triangle fa-3x text-danger"></i>' +
                '<h4>SGIFPCapture Failed</h4>' +
                '<p class="text-danger">Please try again or use manual verification</p>'
            );
            break;
    }
}

function updateVerificationBadge(status) {
    var badge = $('#verification_badge');
    var icon = badge.find('i');
    
    badge.removeClass('verified pending-verification verification-failed');
    
    switch (status) {
        case 'verified':
            badge.addClass('verified');
            icon.removeClass().addClass('fa fa-check-circle');
            badge.html('<i class="fa fa-check-circle"></i> Verified');
            break;
        case 'failed':
            badge.addClass('verification-failed');
            icon.removeClass().addClass('fa fa-times-circle');
            badge.html('<i class="fa fa-times-circle"></i> Failed');
            break;
        default:
            badge.addClass('pending-verification');
            icon.removeClass().addClass('fa fa-clock-o');
            badge.html('<i class="fa fa-clock-o"></i> Pending');
    }
}

function updateVerificationDetails(type, title, message) {
    var detailsDiv = $('#verification_details');
    var iconClass = type === 'success' ? 'fa-check-circle text-success' :
                   type === 'error' ? 'fa-times-circle text-danger' :
                   type === 'warning' ? 'fa-exclamation-triangle text-warning' : 
                   'fa-info-circle text-info';
    
    detailsDiv.html(
        '<div class="' + (type === 'success' ? 'text-success' : type === 'error' ? 'text-danger' : type === 'warning' ? 'text-warning' : 'text-info') + '">' +
        '<i class="fa ' + iconClass + '"></i> <strong>' + title + '</strong><br>' +
        '<small>' + message + '</small>' +
        '</div>'
    );
}

function updateTestSelectionAccess() {
    if (verificationState.isVerified) {
        $('#verification_alert').hide();
        $('#labOptions').slideDown();
        $('#saveLabOrder').prop('disabled', false);
        $('.test-item .btn-add-test').prop('disabled', false);
    } else {
        $('#verification_alert').show();
        $('#labOptions').slideUp();
        $('#saveLabOrder').prop('disabled', true);
        $('.test-item .btn-add-test').prop('disabled', true);
    }
}

function updateQualityIndicator(quality) {
    var qualityBar = $('#quality_bar');
    var qualityText = $('#quality_text');
    
    qualityBar.css('width', quality + '%');
    qualityText.text('Quality: ' + quality + '%');
    
    // Set quality color
    if (quality >= 80) {
        qualityBar.removeClass().addClass('quality-bar quality-excellent');
    } else if (quality >= 60) {
        qualityBar.removeClass().addClass('quality-bar quality-good');
    } else if (quality >= 40) {
        qualityBar.removeClass().addClass('quality-bar quality-fair');
    } else {
        qualityBar.removeClass().addClass('quality-bar quality-poor');
    }
    
    $('.quality-meter').show();
}

/* =========================================
   VERIFICATION HELPER FUNCTIONS
   ========================================= */
   
function resetVerificationState() {
    verificationState = {
        isVerified: false,
        verificationMethod: null,
        verificationTimestamp: null,
        patientId: null,
        verificationScore: 0,
        attempts: 0,
        maxAttempts: 3,
        requiresVerification: true
    };
    
    updateVerificationBadge('pending');
    updateVerificationDetails('info', 'Patient verification required', 
        'Please verify patient identity before proceeding with test selection');
    
    clearFingerprintData();
    $('#manual_verification').hide();
    $('#verification_history').hide();
    updateTestSelectionAccess();
}

function clearFingerprintData() {
    $('#fingerprint_template').val('');
    $('#fingerprint_image_data').val('');
    $('#fingerprint_image').attr('src', '<?php echo base_url(); ?>Images/PlaceFinger.bmp');
    $('#fingerprint_display').hide();
    $('.quality-meter').hide();
    
    updateFingerprintUI('ready');
    $('#verifyFingerprintBtn, #clearFingerprintBtn').hide();
    
    // Reset retry count
    verificationState.attempts = 0;
    $('#scanFingerprintBtn').prop('disabled', false);
}

function logDiagnostic(message, type = 'info') {
    const timestamp = new Date().toLocaleTimeString();
    const colorClass = type === 'error' ? 'text-danger' : 
                      type === 'success' ? 'text-success' : 
                      type === 'warning' ? 'text-warning' : 'text-info';
    
    console.log(`[SecuGen Laboratory] ${message}`);
    
    // Add to verification details if container exists
    if ($('#verification_details').length) {
        updateVerificationDetails(type, 'SecuGen Status', message);
    }
}

function handleFingerprintCaptureError(error) {
    updateFingerprintUI('failed');
    logDiagnostic(`✗ ${error}`, 'error');
    showAlert('Fingerprint capture failed: ' + error, 'error');
    $('#manual_verification').show();
    
    // Increment retry count
    verificationState.attempts++;
    
    if (verificationState.attempts >= verificationState.maxAttempts) {
        $('#scanFingerprintBtn').prop('disabled', true);
        showAlert('Maximum capture attempts reached. Please use manual verification.', 'warning');
    }
}

function handleFingerprintMatchError(error) {
    verificationState.attempts++;
    logDiagnostic(`✗ Match error: ${error}`, 'error');
    updateVerificationDetails('error', 'Verification error occurred', error);
    
    if (verificationState.attempts >= verificationState.maxAttempts) {
        showManualVerificationModal();
    } else {
        showAlert('Verification error: ' + error, 'error');
    }
}

function showFingerprintEnrollmentOption(capturedTemplate) {
    var patientName = $('#patient_name_display').text();
    
    if (confirm('No fingerprint found for ' + patientName + '. Would you like to enroll this fingerprint for future verification?')) {
        enrollFingerprint(capturedTemplate);
    } else {
        showManualVerificationModal();
    }
}

function enrollFingerprint(template) {
    var imageData = $('#fingerprint_image_data').val();
    
    $.ajax({
        url: '<?php echo base_url("admin/patient/saveFingerprintTemplate"); ?>',
        type: 'POST',
        data: {
            patient_id: verificationState.patientId,
            fingerprint_template: template,
            fingerprint_image: imageData,
            enrollment_date: new Date().toISOString()
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showAlert('Fingerprint enrolled successfully', 'success');
                // Now proceed with verification using the newly enrolled template
                setTimeout(() => {
                    performSGIMatchScore(template, template); // Self-match for enrollment verification
                }, 1000);
            } else {
                showAlert('Failed to enroll fingerprint: ' + response.message, 'error');
                showManualVerificationModal();
            }
        },
        error: function() {
            showAlert('Error enrolling fingerprint', 'error');
            showManualVerificationModal();
        }
    });
}

function loadVerificationHistory(patientId) {
    $.ajax({
        url: '<?php echo base_url("admin/patient/getVerificationHistory"); ?>',
        type: 'POST',
        data: { patient_id: patientId, limit: 5 },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success' && response.data.length > 0) {
                displayVerificationHistory(response.data);
            } else {
                $('#verification_history').hide();
            }
        },
        error: function() {
            $('#verification_history').hide();
        }
    });
}

function displayVerificationHistory(verifications) {
    var historyList = $('#verification_list');
    var html = '';
    
    $.each(verifications, function(index, verification) {
        var date = new Date(verification.verification_timestamp).toLocaleDateString();
        var time = new Date(verification.verification_timestamp).toLocaleTimeString();
        var method = verification.verification_method || 'Unknown';
        var result = verification.verification_result || 'Unknown';
        
        html += '<div class="verification-entry" style="margin-bottom: 5px; padding: 5px; background: #f8f9fa; border-radius: 3px;">' +
                '<small><strong>' + method.toUpperCase() + '</strong> - ' + result + '</small><br>' +
                '<small class="text-muted">' + date + ' at ' + time + '</small>' +
                '</div>';
    });
    
    if (html) {
        historyList.html(html);
        $('#verification_history').show();
        
        // Update last verification display
        var lastVerification = verifications[0];
        var lastDate = new Date(lastVerification.verification_timestamp).toLocaleDateString();
        $('#last_verification_display').text(lastDate + ' (' + lastVerification.verification_method + ')');
    }
}

function saveVerificationRecord() {
    $.ajax({
        url: '<?php echo base_url("admin/patient/saveVerificationRecord"); ?>',
        type: 'POST',
        data: {
            patient_id: verificationState.patientId,
            verification_method: verificationState.verificationMethod,
            verification_result: 'success',
            match_score: verificationState.verificationScore,
            department: 'laboratory',
            verified_by: '<?php echo $this->session->userdata("admin_id") ?? 1; ?>',
            verification_timestamp: verificationState.verificationTimestamp,
            device_info: navigator.userAgent,
            notes: 'Laboratory test order verification',
            ip_address: '<?php echo $this->input->ip_address(); ?>',
            session_id: '<?php echo $this->session->session_id; ?>'
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                console.log('Verification record saved successfully');
            }
        },
        error: function() {
            console.log('Failed to save verification record');
        }
    });
}

/* =========================================
   DEBUG & EXTERNAL API FUNCTIONS
   ========================================= */
   
// Debug functions for SecuGen troubleshooting
window.labDebugSecuGen = {
    testSGIFPCapture: function() {
        console.log('Testing SGIFPCapture...');
        startFingerprintCapture();
    },
    
    testSGIMatchScore: function() {
        if (!window.laboratoryState.storedTemplate || !window.laboratoryState.liveTemplate) {
            console.log('Both stored and live templates required for matching test');
            console.log('Stored Template:', window.laboratoryState.storedTemplate ? 'Available (' + window.laboratoryState.storedTemplate.length + ' chars)' : 'Missing');
            console.log('Live Template:', window.laboratoryState.liveTemplate ? 'Available (' + window.laboratoryState.liveTemplate.length + ' chars)' : 'Missing');
            return;
        }
        
        console.log('Testing SGIMatchScore...');
        performSGIMatchScore(window.laboratoryState.liveTemplate, window.laboratoryState.storedTemplate);
    },
    
    testPatientFingerprintRetrieval: function() {
        const patientId = verificationState.patientId;
        if (!patientId) {
            console.log('No patient selected');
            return;
        }
        
        console.log('Testing patient fingerprint retrieval for patient ID:', patientId);
        
        $.ajax({
            url: '<?php echo base_url("admin/laboratory/getPatientFingerprint"); ?>',
            type: 'POST',
            data: { 
                patient_id: patientId,
                action: 'get_template'
            },
            dataType: 'json',
            success: function(response) {
                console.log('Patient Fingerprint Response:', response);
                if (response.fingerprint_template) {
                    console.log('Template Length:', response.fingerprint_template.length);
                    console.log('Template Preview:', response.fingerprint_template.substring(0, 100) + '...');
                }
            },
            error: function(xhr, status, error) {
                console.error('Error retrieving patient fingerprint:', xhr.responseText);
            }
        });
    },
    
    checkCurrentState: function() {
        console.log('=== CURRENT LABORATORY STATE ===');
        console.log('Patient ID:', verificationState.patientId);
        console.log('Verification State:', verificationState);
        console.log('Laboratory State:', window.laboratoryState);
        console.log('Captured Template:', $('#fingerprint_template').val() ? 'Available (' + $('#fingerprint_template').val().length + ' chars)' : 'Missing');
        console.log('Selected Patient Fingerprint Flag:', $('#selected_patient option:selected').data('fingerprint'));
    },
    
    openCertificatePage: function() {
        window.open('https://localhost:8443/SGIFPGetInfo', '_blank');
        console.log('Opened certificate page - accept SSL certificate to enable SecuGen');
    },
    
    openSecuGenDemo: function() {
        window.open('https://webapi.secugen.com/Demo1', '_blank');
        console.log('Opened SecuGen official demo - test if scanner works there first');
    },
    
    testDirectMatchScore: function(template1, template2) {
        if (!template1 || !template2) {
            console.log('Usage: window.labDebugSecuGen.testDirectMatchScore(template1, template2)');
            return;
        }
        
        console.log('Testing direct SGIMatchScore with provided templates...');
        performSGIMatchScore(template1, template2);
    },
    
    getSystemState: function() {
        return {
            secugenConfig: window.secugenConfig,
            laboratoryState: window.laboratoryState,
            verificationState: verificationState,
            capturedTemplate: $('#fingerprint_template').val(),
            selectedPatient: $('#selected_patient option:selected').text(),
            hasStoredFingerprint: $('#selected_patient option:selected').data('fingerprint') == '1'
        };
    }
};

// External API for integration
window.labTestSelection = {
    getSelectedTests: function() {
        return selectedTests;
    },
    
    getBillingInfo: function() {
        return calculateBillingInfo();
    },
    
    clearSelection: function() {
        clearCart();
    },
    
    addTest: function(testName, charge, category) {
        if (verificationState.isVerified) {
            addTestToCart({ 
                name: testName, 
                charge: charge, 
                category: category || 'General Tests' 
            });
        } else {
            showAlert('Patient verification required before adding tests', 'warning');
        }
    },
    
    getTestCount: function() {
        return selectedTests.length;
    },
    
    getTotalAmount: function() {
        return calculateBillingInfo().totalAmount;
    },
    
    getVerificationStatus: function() {
        return verificationState;
    },
    
    isPatientVerified: function() {
        return verificationState.isVerified;
    }
};

/* =========================================
   ERROR HANDLING & CONSOLE HELP
   ========================================= */
   
function displayConsoleHelp() {
    setTimeout(() => {
        console.log(`
=== 🧪 Enhanced Laboratory Test Selection with SecuGen WebAPI Integration ===

🔐 SECUGEN WEBAPI INTEGRATION FEATURES:
✓ Official SGIFPCapture endpoint for fingerprint capture (same as patient registration)
✓ Official SGIMatchScore endpoint for template comparison  
✓ Comprehensive error handling with SecuGen error codes
✓ Manual verification fallback system
✓ Advanced diagnostics and troubleshooting tools

📋 AVAILABLE DEBUG COMMANDS:
- window.labDebugSecuGen.testSGIFPCapture() - Test fingerprint capture functionality
- window.labDebugSecuGen.testSGIMatchScore() - Test template matching (requires templates)
- window.labDebugSecuGen.openCertificatePage() - Open SSL certificate acceptance page
- window.labDebugSecuGen.openSecuGenDemo() - Open official SecuGen demo page
- window.labDebugSecuGen.getSystemState() - Show current system state

🔧 SECUGEN WEBAPI SETUP REQUIREMENTS:
1. Download and install SecuGen WebAPI from https://webapi.secugen.com/
2. Connect SecuGen fingerprint scanner (Hamster Pro, etc.)
3. Accept SSL certificate at https://localhost:8443/SGIFPGetInfo
4. Ensure SecuGen WebAPI service is running (default port 8443)

🏥 LABORATORY TEST SELECTION FEATURES:
✓ Patient search and selection with fingerprint status
✓ Biometric identity verification using SecuGen algorithms
✓ Test selection with verification requirement enforcement
✓ Real-time verification history and quality metrics
✓ Manual override capabilities for legacy workflows
✓ Comprehensive audit trail for all verification attempts

💡 TROUBLESHOOTING TIPS:
- If "Fingerprint scanner not available" appears, try window.labDebugSecuGen.openCertificatePage()
- Test basic functionality with window.labDebugSecuGen.testSGIFPCapture()
- Check browser console for detailed error messages
- Verify SecuGen WebAPI service is running on port 8443
- Ensure fingerprint scanner is properly connected

🚀 EXTERNAL API ACCESS:
- window.labTestSelection.getSelectedTests() - Get current cart
- window.labTestSelection.isPatientVerified() - Check verification status
- window.labTestSelection.getTotalAmount() - Get total billing amount

The system uses the same SecuGen endpoints as patient registration for consistency.
        `);
    }, 500);
}

// Enhanced error handling for global errors
window.onerror = function(msg, url, lineNo, columnNo, error) {
    console.error('JavaScript Error:', {
        message: msg,
        source: url,
        line: lineNo,
        column: columnNo,
        error: error,
        stack: error ? error.stack : 'No stack trace available'
    });
    
    // Show user-friendly error message
    showAlert('An unexpected error occurred. Please refresh the page if issues persist.', 'error');
    
    // Attempt to reset to a safe state
    try {
        resetVerificationState();
        clearCart();
    } catch (e) {
        console.error('Failed to reset to safe state:', e);
    }
    
    return false;
};

// Handle unhandled promise rejections
window.addEventListener('unhandledrejection', function(event) {
    console.error('Unhandled promise rejection:', event.reason);
    showAlert('A background operation failed. Please try again.', 'warning');
});

</script>