<?php
/**
 * Laboratory Results Management View
 * File: application/views/admin/laboratory/results.php
 */
?>

<style>
.content-wrapper {
    background-color: #f4f4f4;
    padding: 20px;
}

.box {
    border-top: 3px solid #3c8dbc;
    background: #fff;
    border-radius: 3px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.12);
}

.results-filters {
    background: #fff;
    padding: 15px;
    border-radius: 3px;
    margin-bottom: 20px;
    box-shadow: 0 1px 3px rgba(0,0,0,0.12);
}

.stats-widget {
    background: linear-gradient(45deg, #3c8dbc, #5bc0de);
    color: white;
    padding: 20px;
    border-radius: 5px;
    text-align: center;
    margin-bottom: 20px;
}

.stats-widget h3 {
    margin: 0 0 10px 0;
    font-size: 2.5em;
}

.stats-widget p {
    margin: 0;
    opacity: 0.9;
}

.status-badge {
    padding: 4px 8px;
    border-radius: 3px;
    font-size: 11px;
    font-weight: bold;
    text-transform: uppercase;
}

.status-pending { background: #f39c12; color: white; }
.status-in_progress { background: #3c8dbc; color: white; }
.status-completed { background: #00a65a; color: white; }
.status-abnormal { background: #dd4b39; color: white; }
.status-critical { background: #8B0000; color: white; }

.verification-unverified { background: #f39c12; color: white; }
.verification-verified { background: #00a65a; color: white; }
.verification-skipped { background: #95a5a6; color: white; }

.action-btn {
    padding: 2px 6px;
    margin: 1px;
    border: none;
    border-radius: 3px;
    cursor: pointer;
    font-size: 11px;
}

.btn-edit { background: #3c8dbc; color: white; }
.btn-print { background: #00a65a; color: white; }
.btn-verify { background: #f39c12; color: white; }

.patient-info {
    font-size: 12px;
}

.patient-name {
    font-weight: bold;
    color: #3c8dbc;
}

.result-text {
    max-width: 200px;
    overflow: hidden;
    text-overflow: ellipsis;
    white-space: nowrap;
}

.reference-range {
    font-size: 10px;
    color: #666;
    font-style: italic;
}

.critical-result {
    background: #ffebee !important;
    border-left: 4px solid #f44336;
}

.loading-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background: rgba(0,0,0,0.5);
    display: none;
    z-index: 9999;
}

.loading-content {
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    background: white;
    padding: 30px;
    border-radius: 5px;
    text-align: center;
}

.bulk-actions {
    margin-bottom: 15px;
    padding: 10px;
    background: #f9f9f9;
    border-radius: 3px;
    display: none;
}

#resultsTable_wrapper .dataTables_length,
#resultsTable_wrapper .dataTables_filter {
    margin-bottom: 10px;
}

.modal-lg {
    width: 90%;
    max-width: 1200px;
}

@media (max-width: 768px) {
    .stats-widget {
        margin-bottom: 10px;
    }
    
    .results-filters .form-group {
        margin-bottom: 10px;
    }
    
    .action-btn {
        display: block;
        margin: 2px 0;
        width: 100%;
    }
}
</style>

<div class="content-wrapper">
    <section class="content-header">
        <h1>
            <i class="fa fa-flask"></i> Laboratory Results Management
            <small>Manage and verify test results</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="#">Laboratory</a></li>
            <li class="active">Results Management</li>
        </ol>
    </section>

    <section class="content">
        <!-- Statistics Widgets -->
        <div class="row">
            <div class="col-lg-3 col-md-6">
                <div class="stats-widget">
                    <h3 id="todayTotal"><?php echo isset($stats['today_total']) ? $stats['today_total'] : 0; ?></h3>
                    <p><i class="fa fa-flask"></i> Tests Today</p>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="stats-widget" style="background: linear-gradient(45deg, #f39c12, #ff9800);">
                    <h3 id="todayPending"><?php echo isset($stats['today_pending']) ? $stats['today_pending'] : 0; ?></h3>
                    <p><i class="fa fa-clock-o"></i> Pending Results</p>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="stats-widget" style="background: linear-gradient(45deg, #00a65a, #4CAF50);">
                    <h3 id="todayCompleted"><?php echo isset($stats['today_completed']) ? $stats['today_completed'] : 0; ?></h3>
                    <p><i class="fa fa-check"></i> Completed Today</p>
                </div>
            </div>
            <div class="col-lg-3 col-md-6">
                <div class="stats-widget" style="background: linear-gradient(45deg, #dd4b39, #f44336);">
                    <h3 id="todayUnverified"><?php echo isset($stats['today_unverified']) ? $stats['today_unverified'] : 0; ?></h3>
                    <p><i class="fa fa-exclamation-triangle"></i> Unverified</p>
                </div>
            </div>
        </div>

        <!-- Main Results Management Box -->
        <div class="box box-primary">
            <div class="box-header with-border">
                <h3 class="box-title">
                    <i class="fa fa-list"></i> Laboratory Results
                </h3>
                <div class="box-tools pull-right">
                    <button type="button" class="btn btn-info btn-sm" onclick="exportResults('pdf')">
                        <i class="fa fa-file-pdf-o"></i> Export PDF
                    </button>
                    <button type="button" class="btn btn-success btn-sm" onclick="exportResults('excel')">
                        <i class="fa fa-file-ex excel-o"></i> Export Excel
                    </button>
                    <button type="button" class="btn btn-success btn-sm" onclick="printSelectedResults()">
                        <i class="fa fa-print"></i> Print Selected
                    </button>
                    <button type="button" class="btn btn-default btn-sm" onclick="refreshResults()">
                        <i class="fa fa-refresh"></i> Refresh
                    </button>
                </div>
            </div>

            <!-- Filters Section -->
            <div class="box-body">
                <div class="results-filters">
                    <div class="row">
                        <div class="col-md-2">
                            <div class="form-group">
                                <label for="statusFilter">Result Status</label>
                                <select class="form-control input-sm" id="statusFilter">
                                    <option value="">All Statuses</option>
                                    <?php foreach($result_statuses as $key => $status): ?>
                                    <option value="<?php echo $key; ?>"><?php echo $status; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="form-group">
                                <label for="verificationFilter">Verification</label>
                                <select class="form-control input-sm" id="verificationFilter">
                                    <option value="">All Verification</option>
                                    <?php foreach($verification_statuses as $key => $status): ?>
                                    <option value="<?php echo $key; ?>"><?php echo $status; ?></option>
                                    <?php endforeach; ?>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="form-group">
                                <label for="dateFrom">Date From</label>
                                <input type="date" class="form-control input-sm" id="dateFrom">
                            </div>
                        </div>
                        <div class="col-md-2">
                            <div class="form-group">
                                <label for="dateTo">Date To</label>
                                <input type="date" class="form-control input-sm" id="dateTo">
                            </div>
                        </div>
                        <div class="col-md-3">
                            <div class="form-group">
                                <label for="patientSearch">Patient Search</label>
                                <input type="text" class="form-control input-sm" id="patientSearch" 
                                       placeholder="Name, Phone, or ID">
                            </div>
                        </div>
                        <div class="col-md-1">
                            <div class="form-group">
                                <label>&nbsp;</label>
                                <button type="button" class="btn btn-primary btn-sm btn-block" onclick="applyFilters()">
                                    <i class="fa fa-search"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Bulk Actions -->
                <div class="bulk-actions" id="bulkActions">
                    <div class="row">
                        <div class="col-md-4">
                            <span id="selectedCount">0</span> results selected
                        </div>
                        <div class="col-md-8 text-right">
                            <button type="button" class="btn btn-success btn-sm" onclick="bulkVerify('verified')">
                                <i class="fa fa-check"></i> Mark as Verified
                            </button>
                            <button type="button" class="btn btn-warning btn-sm" onclick="bulkVerify('unverified')">
                                <i class="fa fa-times"></i> Mark as Unverified
                            </button>
                            <button type="button" class="btn btn-default btn-sm" onclick="clearSelection()">
                                <i class="fa fa-remove"></i> Clear Selection
                            </button>
                        </div>
                    </div>
                </div>

                <!-- Results Table -->
                <div class="table-responsive">
                    <table class="table table-striped table-bordered table-hover" id="resultsTable">
                        <thead>
                            <tr>
                                <th width="30">
                                    <input type="checkbox" id="selectAll">
                                </th>
                                <th width="60">ID</th>
                                <th width="150">Patient Info</th>
                                <th width="120">Test Details</th>
                                <th width="100">Test Date</th>
                                <th width="150">Result</th>
                                <th width="80">Status</th>
                                <th width="80">Verification</th>
                                <th width="100">Doctor</th>
                                <th width="80">Amount</th>
                                <th width="120">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be loaded via DataTables AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Edit Result Modal -->
<div class="modal fade" id="editResultModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">×</button>
                <h4 class="modal-title">
                    <i class="fa fa-edit"></i> Edit Test Result
                </h4>
            </div>
            <form id="editResultForm">
                <div class="modal-body">
                    <input type="hidden" id="editLabId" name="lab_id">
                    
                    <div class="row">
                        <!-- Patient Information -->
                        <div class="col-md-6">
                            <div class="panel panel-default">
                                <div class="panel-heading">
                                    <h5 class="panel-title">Patient Information</h5>
                                </div>
                                <div class="panel-body">
                                    <div id="editPatientInfo">
                                        <!-- Patient details will be loaded here -->
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <!-- Test Information -->
                        <div class="col-md-6">
                            <div class="panel panel-default">
                                <div class="panel-heading">
                                    <h5 class="panel-title">Test Information</h5>
                                </div>
                                <div class="panel-body">
                                    <div id="editTestInfo">
                                        <!-- Test details will be loaded here -->
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <!-- Result Input -->
                        <div class="col-md-12">
                            <div class="panel panel-primary">
                                <div class="panel-heading">
                                    <h5 class="panel-title">Test Result</h5>
                                </div>
                                <div class="panel-body">
                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="editTestResult">Test Result *</label>
                                                <textarea class="form-control" id="editTestResult" name="test_result" 
                                                         rows="3" placeholder="Enter test result..."></textarea>
                                            </div>
                                        </div>
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label for="editReferenceRange">Reference Range</label>
                                                <input type="text" class="form-control" id="editReferenceRange" 
                                                       name="reference_range" placeholder="e.g., 10-40 mg/dL">
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label for="editResultStatus">Result Status *</label>
                                                <select class="form-control" id="editResultStatus" name="result_status">
                                                    <option value="">Select Status</option>
                                                    <?php foreach($result_statuses as $key => $status): ?>
                                                    <option value="<?php echo $key; ?>"><?php echo $status; ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label for="editVerificationStatus">Verification Status</label>
                                                <select class="form-control" id="editVerificationStatus" name="verification_status">
                                                    <option value="">Select Verification</option>
                                                    <?php foreach($verification_statuses as $key => $status): ?>
                                                    <option value="<?php echo $key; ?>"><?php echo $status; ?></option>
                                                    <?php endforeach; ?>
                                                </select>
                                            </div>
                                        </div>
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label for="editNotes">Additional Notes</label>
                                                <input type="text" class="form-control" id="editNotes" name="notes" 
                                                       placeholder="Optional notes">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fa fa-save"></i> Update Result
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Loading Overlay -->
<div class="loading-overlay" id="loadingOverlay">
    <div class="loading-content">
        <i class="fa fa-spinner fa-spin fa-3x text-primary"></i>
        <h4>Processing...</h4>
        <p>Please wait while we process your request.</p>
    </div>
</div>

<script>
$(document).ready(function() {
    // Initialize DataTable
    initializeResultsTable();
    
    // Set up event handlers
    setupEventHandlers();
    
    // Set default date range (last 7 days)
    setDefaultDateRange();
});

let resultsTable;
let selectedResults = [];

function initializeResultsTable() {
    resultsTable = $('#resultsTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '<?php echo base_url("admin/laboratory/getLabResults"); ?>',
            type: 'POST',
            data: function(d) {
                d.status_filter = $('#statusFilter').val();
                d.verification_filter = $('#verificationFilter').val();
                d.date_from = $('#dateFrom').val();
                d.date_to = $('#dateTo').val();
                d.patient_search = $('#patientSearch').val();
            }
        },
        columns: [
            {
                data: null,
                orderable: false,
                render: function(data, type, row) {
                    return `<input type="checkbox" class="result-checkbox" value="${row.id}">`;
                }
            },
            { data: 'id' },
            {
                data: 'patient_info',
                render: function(data, type, row) {
                    return `
                        <div class="patient-info">
                            <div class="patient-name">${data.name}</div>
                            <small>${data.phone} </small>
                        </div>
                    `;
                }
            },
            {
                data: null,
                render: function(data, type, row) {
                    return `
                        <div>
                            <strong>${row.test_name}</strong><br>
                            <small class="text-muted">${row.category_name}</small>
                        </div>
                    `;
                }
            },
            {
                data: 'test_date',
                render: function(data, type, row) {
                    return moment(data).format('DD/MM/YYYY');
                }
            },
            {
                data: 'test_result',
                render: function(data, type, row) {
                    let result = data || '<em class="text-muted">Pending</em>';
                    let reference = row.reference_range ? `<br><small class="reference-range">Ref: ${row.reference_range}</small>` : '';
                    return `<div class="result-text" title="${data || 'No result yet'}">${result}${reference}</div>`;
                }
            },
            {
                data: 'result_status',
                render: function(data, type, row) {
                    let statusClass = `status-${data}`;
                    return `<span class="status-badge ${statusClass}">${data.replace('_', ' ')}</span>`;
                }
            },
            {
                data: 'verification_status',
                render: function(data, type, row) {
                    let verificationClass = `verification-${data}`;
                    return `<span class="status-badge ${verificationClass}">${data}</span>`;
                }
            },
            {
                data: 'doctor_name',
                render: function(data, type, row) {
                    return data || '<em class="text-muted">Not assigned</em>';
                }
            },
            {
                data: 'final_amount',
                render: function(data, type, row) {
                    return `KSh ${parseFloat(data || 0).toLocaleString()}`;
                }
            },
            {
                data: null,
                orderable: false,
                render: function(data, type, row) {
                    return `
                        <div class="btn-group-vertical btn-group-xs">
                            <button class="action-btn btn-edit" onclick="editResult(${row.id})" 
                                    title="Edit Result">
                                <i class="fa fa-edit"></i>
                            </button>
                            <button class="action-btn btn-print" onclick="printResult(${row.id})" 
                                    title="Print Result">
                                <i class="fa fa-print"></i>
                            </button>
                            <button class="action-btn btn-verify" onclick="toggleVerification(${row.id})" 
                                    title="Toggle Verification">
                                <i class="fa fa-shield"></i>
                            </button>
                        </div>
                    `;
                }
            }
        ],
        pageLength: 25,
        order: [[1, 'desc']],
        rowCallback: function(row, data) {
            if (data.result_status === 'critical') {
                $(row).addClass('critical-result');
            }
        },
        drawCallback: function() {
            updateSelectedCount();
            updateStatsWidgets();
        }
    });
}

function setupEventHandlers() {
    // Select all checkbox
    $('#selectAll').on('change', function() {
        let isChecked = $(this).is(':checked');
        $('.result-checkbox').prop('checked', isChecked);
        updateSelectedResults();
    });
    
    // Individual checkboxes
    $(document).on('change', '.result-checkbox', function() {
        updateSelectedResults();
    });
    
    // Edit form submission with validation
    $('#editResultForm').on('submit', function(e) {
        e.preventDefault();
        if (validateEditForm()) {
            updateTestResult();
        } else {
            showMessage('Please fill in all required fields', 'warning');
        }
    });
    
    // Filter change events
    $('#statusFilter, #verificationFilter, #dateFrom, #dateTo').on('change', function() {
        applyFilters();
    });
    
    // Patient search with debounce
    let searchTimeout;
    $('#patientSearch').on('keyup', function() {
        clearTimeout(searchTimeout);
        searchTimeout = setTimeout(applyFilters, 500);
    });
    
    // Enter key in patient search
    $('#patientSearch').on('keypress', function(e) {
        if (e.which === 13) {
            e.preventDefault();
            applyFilters();
        }
    });
    
    // Responsive handling for DataTable
    $(window).on('resize', function() {
        resultsTable.columns.adjust().draw();
    });
    
    // Initialize tooltips for action buttons
    $(document).on('mouseenter', '.action-btn', function() {
        $(this).tooltip({
            container: 'body',
            placement: 'top'
        });
    });
    
    // Clean up tooltips and form when modal is closed
    $('#editResultModal').on('hidden.bs.modal', function() {
        $('#editResultForm')[0].reset();
        $('.form-group').removeClass('has-error');
        $('.tooltip').remove();
    });
    
    // Handle double-click on table row to edit
    $('#resultsTable').on('dblclick', 'tr', function() {
        let data = resultsTable.row(this).data();
        if (data) {
            editResult(data.id);
        }
    });
}

function setDefaultDateRange() {
    let today = new Date();
    let lastWeek = new Date(today.getTime() - 7 * 24 * 60 * 60 * 1000);
    
    $('#dateTo').val(today.toISOString().split('T')[0]);
    $('#dateFrom').val(lastWeek.toISOString().split('T')[0]);
}

function applyFilters() {
    resultsTable.ajax.reload();
}

function refreshResults() {
    resultsTable.ajax.reload(null, false);
    showMessage('Results refreshed successfully', 'success');
}

function updateSelectedResults() {
    selectedResults = [];
    $('.result-checkbox:checked').each(function() {
        selectedResults.push($(this).val());
    });
    
    updateSelectedCount();
    
    if (selectedResults.length > 0) {
        $('#bulkActions').show();
    } else {
        $('#bulkActions').hide();
    }
    
    // Update select all checkbox state
    let totalCheckboxes = $('.result-checkbox').length;
    let checkedCheckboxes = $('.result-checkbox:checked').length;
    
    if (checkedCheckboxes === 0) {
        $('#selectAll').prop('indeterminate', false).prop('checked', false);
    } else if (checkedCheckboxes === totalCheckboxes) {
        $('#selectAll').prop('indeterminate', false).prop('checked', true);
    } else {
        $('#selectAll').prop('indeterminate', true);
    }
}

function updateSelectedCount() {
    $('#selectedCount').text(selectedResults.length);
}

function clearSelection() {
    $('.result-checkbox').prop('checked', false);
    $('#selectAll').prop('checked', false);
    updateSelectedResults();
}

function editResult(labId) {
    showLoading();
    
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/getTestDetails"); ?>',
        type: 'POST',
        data: { lab_id: labId },
        dataType: 'json',
        success: function(response) {
            hideLoading();
            
            if (response.status === 'success') {
                populateEditModal(response.data);
                $('#editResultModal').modal('show');
            } else {
                showMessage(response.message || 'Failed to load test details', 'error');
            }
        },
        error: function(xhr, status, error) {
            hideLoading();
            showMessage('Error loading test details: ' + error, 'error');
        }
    });
}

function populateEditModal(data) {
    // Set form data
    $('#editLabId').val(data.id);
    $('#editTestResult').val(data.test_result);
    $('#editReferenceRange').val(data.reference_range);
    $('#editResultStatus').val(data.result_status);
    $('#editVerificationStatus').val(data.verification_status);
    $('#editNotes').val(data.notes);
    
    // Patient information
    $('#editPatientInfo').html(`
        <table class="table table-condensed">
            <tr><td><strong>Name:</strong></td><td>${data.patient_name}</td></tr>
            <tr><td><strong>Phone:</strong></td><td>${data.mobileno}</td></tr>
            <tr><td><strong>Age/Gender:</strong></td><td>${data.age} years, ${data.gender}</td></tr>
            <tr><td><strong>Patient ID:</strong></td><td>${data.patient_unique_id || 'N/A'}</td></tr>
        </table>
    `);
    
    // Test information
    $('#editTestInfo').html(`
        <table class="table table-condensed">
            <tr><td><strong>Test:</strong></td><td>${data.test_name}</td></tr>
            <tr><td><strong>Category:</strong></td><td>${data.category_name}</td></tr>
            <tr><td><strong>Test Date:</strong></td><td>${moment(data.test_date).format('DD/MM/YYYY')}</td></tr>
            <tr><td><strong>Doctor:</strong></td><td>${data.doctor_name || 'Not assigned'}</td></tr>
            <tr><td><strong>Technician:</strong></td><td>${data.technician_name || 'Not assigned'}</td></tr>
            <tr><td><strong>Amount:</strong></td><td>KSh ${parseFloat(data.final_amount || 0).toLocaleString()}</td></tr>
        </table>
    `);
}

function updateTestResult() {
    let submitBtn = $('#editResultForm button[type="submit"]');
    submitBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Updating...');
    
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/updateTestResult"); ?>',
        type: 'POST',
        data: $('#editResultForm').serialize(),
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showMessage(response.message, 'success');
                $('#editResultModal').modal('hide');
                resultsTable.ajax.reload(null, false);
            } else {
                showMessage(response.message || 'Failed to update result', 'error');
            }
        },
        error: function(xhr, status, error) {
            showMessage('Error updating result: ' + error, 'error');
        },
        complete: function() {
            submitBtn.prop('disabled', false).html('<i class="fa fa-save"></i> Update Result');
        }
    });
}

function printResult(labId) {
    let printUrl = '<?php echo base_url("admin/laboratory/printLabResult"); ?>?lab_id=' + labId;
    window.open(printUrl, '_blank', 'width=800,height=600');
}

function printSelectedResults() {
    if (selectedResults.length === 0) {
        showMessage('Please select results to print', 'warning');
        return;
    }
    
    let printUrl = '<?php echo base_url("admin/laboratory/printMultipleResults"); ?>?lab_ids=' + selectedResults.join(',');
    window.open(printUrl, '_blank', 'width=800,height=600');
}

function toggleVerification(labId) {
    let currentRow = resultsTable.row(function(idx, data, node) {
        return data.id == labId;
    }).data();
    
    let newStatus = currentRow.verification_status === 'verified' ? 'unverified' : 'verified';
    
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/updateTestResult"); ?>',
        type: 'POST',
        data: {
            lab_id: labId,
            verification_status: newStatus
        },
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                showMessage(`Verification status updated to ${newStatus}`, 'success');
                resultsTable.ajax.reload(null, false);
            } else {
                showMessage(response.message || 'Failed to update verification', 'error');
            }
        },
        error: function(xhr, status, error) {
            showMessage('Error updating verification: ' + error, 'error');
        }
    });
}

function bulkVerify(status) {
    if (selectedResults.length === 0) {
        showMessage('Please select results to update', 'warning');
        return;
    }
    
    let confirmMessage = `Are you sure you want to mark ${selectedResults.length} result(s) as ${status}?`;
    
    if (confirm(confirmMessage)) {
        showLoading();
        
        $.ajax({
            url: '<?php echo base_url("admin/laboratory/bulkUpdateVerification"); ?>',
            type: 'POST',
            data: {
                lab_ids: selectedResults,
                verification_status: status
            },
            dataType: 'json',
            success: function(response) {
                hideLoading();
                if (response.status === 'success') {
                    showMessage(response.message || `Successfully updated ${selectedResults.length} result(s) to ${status}`, 'success');
                    resultsTable.ajax.reload(null, false);
                    clearSelection();
                } else {
                    showMessage(response.message || 'Failed to update verification status', 'error');
                }
            },
            error: function(xhr, status, error) {
                hideLoading();
                showMessage('Error updating verification status: ' + error, 'error');
            }
        });
    }
}

function showLoading() {
    $('#loadingOverlay').fadeIn(200);
}

function hideLoading() {
    $('#loadingOverlay').fadeOut(200);
}

function showMessage(message, type) {
    if (typeof toastr !== 'undefined') {
        toastr.options = {
            closeButton: true,
            progressBar: true,
            positionClass: 'toast-top-right',
            showDuration: 300,
            hideDuration: 1000,
            timeOut: 5000,
            extendedTimeOut: 1000
        };
        
        switch(type) {
            case 'success':
                toastr.success(message);
                break;
            case 'error':
                toastr.error(message);
                break;
            case 'warning':
                toastr.warning(message);
                break;
            default:
                toastr.info(message);
        }
    } else {
        alert(message);
    }
}

function validateEditForm() {
    let isValid = true;
    let testResult = $('#editTestResult').val().trim();
    let resultStatus = $('#editResultStatus').val();
    
    $('.form-group').removeClass('has-error');
    
    if (!testResult) {
        $('#editTestResult').closest('.form-group').addClass('has-error');
        isValid = false;
    }
    
    if (!resultStatus) {
        $('#editResultStatus').closest('.form-group').addClass('has-error');
        isValid = false;
    }
    
    return isValid;
}

function updateStatsWidgets() {
    $.ajax({
        url: '<?php echo base_url("admin/laboratory/getStats"); ?>',
        type: 'POST',
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                $('#todayTotal').text(response.data.today_total || 0);
                $('#todayPending').text(response.data.today_pending || 0);
                $('#todayCompleted').text(response.data.today_completed || 0);
                $('#todayUnverified').text(response.data.today_unverified || 0);
            }
        }
    });
}

function exportResults(format) {
    let url = '<?php echo base_url("admin/laboratory/exportResults"); ?>?' + 
              $.param({
                  status_filter: $('#statusFilter').val(),
                  verification_filter: $('#verificationFilter').val(),
                  date_from: $('#dateFrom').val(),
                  date_to: $('#dateTo').val(),
                  patient_search: $('#patientSearch').val(),
                  format: format
              });
    window.open(url, '_blank');
}
</script>