<?php
/**
 * Lab Reports Management Page
 * Path: application/views/admin/lab_reports/index.php
 */
?>

<div class="content-wrapper">
    <!-- Content Header -->
    <section class="content-header">
        <h1>
            <i class="fa fa-flask"></i> Lab Reports Management
            <small>Upload and manage patient lab reports</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Dashboard</a></li>
            <li class="active">Lab Reports</li>
        </ol>
    </section>

    <!-- Main content -->
    <section class="content">
        <!-- Statistics Row -->
        <div class="row">
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-aqua">
                    <div class="inner">
                        <h3 id="totalReports">0</h3>
                        <p>Total Reports</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-file-text"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-yellow">
                    <div class="inner">
                        <h3 id="pendingReports">0</h3>
                        <p>Pending Verification</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-clock-o"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-green">
                    <div class="inner">
                        <h3 id="verifiedReports">0</h3>
                        <p>Verified Today</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-check-circle"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-red">
                    <div class="inner">
                        <h3 id="criticalReports">0</h3>
                        <p>Critical Reports</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-exclamation-triangle"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Content Row -->
        <div class="row">
            <div class="col-md-12">
                <!-- Patient Search Box -->
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-search"></i> Patient Search</h3>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <div class="col-md-9">
                                <label>Search Patient</label>
                                <select id="patientSelect" class="form-control select2" style="width: 100%;">
                                    <option value="">Search by patient name, ID, or mobile number...</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label>&nbsp;</label>
                                <button type="button" class="btn btn-primary btn-block" onclick="loadPatientData()">
                                    <i class="fa fa-user"></i> Load Patient
                                </button>
                            </div>
                        </div>
                        
                        <!-- Selected Patient Info -->
                        <div id="selectedPatientInfo" class="row" style="display: none; margin-top: 15px;">
                            <div class="col-md-12">
                                <div class="alert alert-info">
                                    <h4><i class="fa fa-info-circle"></i> Selected Patient</h4>
                                    <div class="row">
                                        <div class="col-md-3">
                                            <strong>Name:</strong> <span id="patientName">-</span><br>
                                            <strong>ID:</strong> <span id="patientId">-</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Mobile:</strong> <span id="patientMobile">-</span><br>
                                            <strong>Age:</strong> <span id="patientAge">-</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Gender:</strong> <span id="patientGender">-</span><br>
                                            <strong>Reports:</strong> <span id="patientReportsCount">0</span>
                                        </div>
                                        <div class="col-md-3">
                                            <button class="btn btn-sm btn-info" onclick="viewPatientHistory()">
                                                <i class="fa fa-eye"></i> View All Reports
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Upload and Reports Row -->
        <div class="row">
            <!-- Upload Form -->
            <div class="col-md-8">
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-upload"></i> Upload Lab Report</h3>
                    </div>
                    <div class="box-body">
                        <form id="uploadForm" enctype="multipart/form-data">
                            <!-- Basic Information -->
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Report Type <span class="text-red">*</span></label>
                                        <select class="form-control" name="report_type" required>
                                            <option value="">Select type...</option>
                                            <option value="lab_report">Laboratory Report</option>
                                            <option value="imaging">Imaging/X-Ray</option>
                                            <option value="pathology">Pathology Report</option>
                                            <option value="microbiology">Microbiology</option>
                                            <option value="hematology">Hematology</option>
                                            <option value="biochemistry">Biochemistry</option>
                                            <option value="document">General Document</option>
                                            <option value="other">Other</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Department</label>
                                        <select class="form-control" name="department">
                                            <option value="laboratory">Laboratory</option>
                                            <option value="radiology">Radiology</option>
                                            <option value="pathology">Pathology</option>
                                            <option value="cardiology">Cardiology</option>
                                            <option value="hematology">Hematology</option>
                                            <option value="microbiology">Microbiology</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <div class="form-group">
                                <label>Report Title <span class="text-red">*</span></label>
                                <input type="text" class="form-control" name="report_title" 
                                       placeholder="e.g., Complete Blood Count" required>
                            </div>

                            <div class="form-group">
                                <label>Description</label>
                                <textarea class="form-control" name="report_description" rows="3" 
                                          placeholder="Brief description of the report"></textarea>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Test Names</label>
                                        <input type="text" class="form-control" name="test_names" 
                                               placeholder="e.g., Hemoglobin, WBC Count, Platelets">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Report Date <span class="text-red">*</span></label>
                                        <input type="date" class="form-control" name="report_date" 
                                               value="<?php echo date('Y-m-d'); ?>" required>
                                    </div>
                                </div>
                            </div>

                            <!-- Critical Flag -->
                            <div class="form-group">
                                <div class="checkbox">
                                    <label class="text-red">
                                        <input type="checkbox" name="is_critical" id="isCritical">
                                        <i class="fa fa-exclamation-triangle"></i> Mark as Critical Results
                                    </label>
                                </div>
                            </div>

                            <!-- File Upload -->
                            <div class="form-group">
                                <label>Upload File <span class="text-red">*</span></label>
                                <div class="file-upload-zone" onclick="document.getElementById('fileInput').click()">
                                    <input type="file" id="fileInput" name="lab_report_file" 
                                           accept=".pdf,.jpg,.jpeg,.png,.doc,.docx" style="display: none;" required>
                                    <div class="text-center">
                                        <i class="fa fa-cloud-upload fa-4x text-blue"></i>
                                        <h4>Click to select file or drag and drop</h4>
                                        <p class="text-muted">Supported: PDF, JPG, PNG, DOC, DOCX (Max: 10MB)</p>
                                    </div>
                                </div>
                                <div id="fileInfo" style="display: none; margin-top: 10px;"></div>
                            </div>

                            <!-- Additional Notes -->
                            <div class="form-group">
                                <label>Additional Notes</label>
                                <textarea class="form-control" name="notes" rows="2" 
                                          placeholder="Any additional notes or observations"></textarea>
                            </div>

                            <!-- Submit Button -->
                            <div class="form-group">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="fa fa-upload"></i> Upload Report
                                </button>
                                <button type="reset" class="btn btn-default btn-lg">
                                    <i class="fa fa-refresh"></i> Reset Form
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Patient Reports List -->
            <div class="col-md-4">
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-history"></i> Patient Reports</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" onclick="refreshPatientReports()">
                                <i class="fa fa-refresh"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body" style="height: 500px; overflow-y: auto;">
                        <div id="patientReportsList">
                            <div class="text-center text-muted" style="padding: 50px 20px;">
                                <i class="fa fa-file-text-o fa-3x"></i>
                                <p style="margin-top: 15px;">Select a patient to view reports</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- All Reports Table -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-list"></i> All Lab Reports</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-success btn-sm" onclick="exportReports()">
                                <i class="fa fa-download"></i> Export
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- Filters -->
                        <div class="row" style="margin-bottom: 15px;">
                            <div class="col-md-2">
                                <select class="form-control input-sm" id="filterStatus">
                                    <option value="">All Status</option>
                                    <option value="pending">Pending</option>
                                    <option value="completed">Completed</option>
                                    <option value="verified">Verified</option>
                                    <option value="cancelled">Cancelled</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-control input-sm" id="filterDepartment">
                                    <option value="">All Departments</option>
                                    <option value="laboratory">Laboratory</option>
                                    <option value="radiology">Radiology</option>
                                    <option value="pathology">Pathology</option>
                                    <option value="cardiology">Cardiology</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <input type="date" class="form-control input-sm" id="filterDateFrom" placeholder="From Date">
                            </div>
                            <div class="col-md-2">
                                <input type="date" class="form-control input-sm" id="filterDateTo" placeholder="To Date">
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-primary btn-sm" onclick="applyFilters()">
                                    <i class="fa fa-filter"></i> Filter
                                </button>
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-default btn-sm" onclick="clearFilters()">
                                    <i class="fa fa-times"></i> Clear
                                </button>
                            </div>
                        </div>

                        <!-- Reports Table -->
                        <table id="reportsTable" class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Patient</th>
                                    <th>Report Title</th>
                                    <th>Type</th>
                                    <th>Department</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- View Report Modal -->
<div class="modal fade" id="viewReportModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-eye"></i> View Lab Report
                </h4>
            </div>
            <div class="modal-body">
                <div id="reportPreview"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" id="downloadReportBtn">
                    <i class="fa fa-download"></i> Download
                </button>
                <button type="button" class="btn btn-success" id="verifyReportBtn" style="display: none;">
                    <i class="fa fa-check"></i> Verify Report
                </button>
            </div>
        </div>
    </div>
</div>

<!-- CSS Styles -->
<style>
.file-upload-zone {
    border: 2px dashed #3c8dbc;
    border-radius: 5px;
    padding: 30px;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    background: #f9f9f9;
    margin-bottom: 10px;
}

.file-upload-zone:hover {
    border-color: #2c6fbb;
    background: #f0f8ff;
}

.file-upload-zone.dragover {
    border-color: #00a65a;
    background: #f0fff0;
}

.report-item {
    border-left: 4px solid #3c8dbc;
    padding: 10px;
    margin-bottom: 10px;
    background: #fff;
    border-radius: 0 3px 3px 0;
    border: 1px solid #ddd;
}

.report-item.critical {
    border-left-color: #dd4b39;
    background: #fff5f5;
}

.report-item.verified {
    border-left-color: #00a65a;
    background: #f0fff0;
}

.report-item h5 {
    margin: 0 0 5px 0;
    font-size: 14px;
}

.report-item .report-meta {
    font-size: 11px;
    color: #666;
    margin-bottom: 8px;
}

.report-item .report-actions {
    text-align: right;
}

.report-item .report-actions .btn {
    margin-left: 3px;
}
</style>

<!-- JavaScript -->
<script>
let currentPatientId = null;
let currentPatientData = null;

$(document).ready(function() {
    initializeComponents();
    loadDashboardStats();
});

function initializeComponents() {
    // Initialize Select2
    $('#patientSelect').select2({
        placeholder: 'Search patient by name, ID, or mobile number',
        ajax: {
            url: '<?php echo base_url(); ?>admin/lab_reports/search_patients',
            dataType: 'json',
            delay: 250,
            data: function (params) {
                return { q: params.term, page: params.page };
            },
            processResults: function (data, params) {
                params.page = params.page || 1;
                return {
                    results: data.items,
                    pagination: { more: (params.page * 30) < data.total_count }
                };
            },
            cache: true
        },
        minimumInputLength: 2
    });

    // Initialize DataTable
    $('#reportsTable').DataTable({
        processing: true,
        serverSide: true,
        ajax: {
            url: '<?php echo base_url(); ?>admin/lab_reports/get_all_reports',
            type: 'POST',
            data: function(d) {
                d.status = $('#filterStatus').val();
                d.department = $('#filterDepartment').val();
                d.date_from = $('#filterDateFrom').val();
                d.date_to = $('#filterDateTo').val();
            }
        },
        columns: [
            { data: 'id', width: '50px' },
            { data: 'patient_name' },
            { data: 'report_title' },
            { data: 'report_type' },
            { data: 'department' },
            { data: 'report_date', width: '100px' },
            { data: 'status', width: '80px' },
            { data: 'actions', orderable: false, width: '120px' }
        ],
        order: [[0, 'desc']],
        pageLength: 25
    });

    setupFileUpload();
    setupFormHandlers();
}

function setupFileUpload() {
    const fileInput = document.getElementById('fileInput');
    const fileUploadZone = document.querySelector('.file-upload-zone');
    const fileInfo = document.getElementById('fileInfo');

    fileInput.addEventListener('change', function(e) {
        if (e.target.files.length > 0) {
            displayFileInfo(e.target.files[0]);
        }
    });

    fileUploadZone.addEventListener('dragover', function(e) {
        e.preventDefault();
        fileUploadZone.classList.add('dragover');
    });

    fileUploadZone.addEventListener('dragleave', function(e) {
        e.preventDefault();
        fileUploadZone.classList.remove('dragover');
    });

    fileUploadZone.addEventListener('drop', function(e) {
        e.preventDefault();
        fileUploadZone.classList.remove('dragover');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            fileInput.files = files;
            displayFileInfo(files[0]);
        }
    });

    function displayFileInfo(file) {
        const fileSize = (file.size / 1024 / 1024).toFixed(2);
        fileInfo.innerHTML = `
            <div class="alert alert-info" style="margin: 0;">
                <i class="fa fa-file"></i> <strong>Selected:</strong> ${file.name}<br>
                <i class="fa fa-hdd-o"></i> <strong>Size:</strong> ${fileSize} MB<br>
                <i class="fa fa-file-code-o"></i> <strong>Type:</strong> ${file.type}
            </div>
        `;
        fileInfo.style.display = 'block';
    }
}

function setupFormHandlers() {
    $('#uploadForm').on('submit', function(e) {
        e.preventDefault();
        if (!currentPatientId) {
            toastr.warning('Please select a patient first');
            return;
        }
        uploadLabReport();
    });
}

function loadPatientData() {
    const selectedPatient = $('#patientSelect').select2('data')[0];
    if (!selectedPatient) {
        toastr.warning('Please select a patient');
        return;
    }

    currentPatientId = selectedPatient.id;
    currentPatientData = selectedPatient;

    // Update patient info display
    $('#patientName').text(selectedPatient.patient_name);
    $('#patientId').text(selectedPatient.id);
    $('#patientMobile').text(selectedPatient.mobileno || 'N/A');
    $('#patientAge').text(selectedPatient.age || 'N/A');
    $('#patientGender').text(selectedPatient.gender || 'N/A');

    $('#selectedPatientInfo').show();
    
    loadPatientReports();
}

function loadPatientReports() {
    if (!currentPatientId) return;

    $.ajax({
        url: '<?php echo base_url(); ?>admin/lab_reports/get_patient_reports',
        method: 'GET',
        data: { patient_id: currentPatientId },
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                displayPatientReports(response.data);
                $('#patientReportsCount').text(response.data.length);
            } else {
                toastr.error('Failed to load patient reports');
            }
        },
        error: function() {
            toastr.error('Error loading patient reports');
        }
    });
}

function displayPatientReports(reports) {
    const container = $('#patientReportsList');
    
    if (!reports || reports.length === 0) {
        container.html(`
            <div class="text-center text-muted" style="padding: 50px 20px;">
                <i class="fa fa-file-text-o fa-2x"></i>
                <p style="margin-top: 15px;">No reports found</p>
            </div>
        `);
        return;
    }

    let html = '';
    reports.forEach(function(report) {
        const statusClass = getStatusClass(report.status);
        const criticalClass = report.is_critical == 1 ? 'critical' : '';
        const criticalIcon = report.is_critical == 1 ? '<i class="fa fa-exclamation-triangle text-red"></i> ' : '';

        html += `
            <div class="report-item ${criticalClass}">
                <h5>${criticalIcon}${report.report_title}</h5>
                <div class="report-meta">
                    ${report.report_type} • ${report.department} • ${report.report_date}
                </div>
                <div class="report-meta">
                    Status: <span class="label ${statusClass}">${report.status}</span>
                </div>
                <div class="report-actions">
                    <button class="btn btn-xs btn-primary" onclick="viewReport(${report.id})" title="View">
                        <i class="fa fa-eye"></i>
                    </button>
                    <button class="btn btn-xs btn-success" onclick="downloadReport(${report.id})" title="Download">
                        <i class="fa fa-download"></i>
                    </button>
                    ${report.status !== 'verified' ? 
                        `<button class="btn btn-xs btn-warning" onclick="verifyReport(${report.id})" title="Verify">
                            <i class="fa fa-check"></i>
                        </button>` : ''
                    }
                </div>
            </div>
        `;
    });

    container.html(html);
}

function getStatusClass(status) {
    const statusClasses = {
        'pending': 'label-warning',
        'completed': 'label-primary',
        'verified': 'label-success',
        'cancelled': 'label-danger'
    };
    return statusClasses[status] || 'label-default';
}

function uploadLabReport() {
    const formData = new FormData($('#uploadForm')[0]);
    formData.append('patient_id', currentPatientId);

    // Show loading
    const submitBtn = $('#uploadForm button[type="submit"]');
    const originalText = submitBtn.html();
    submitBtn.html('<i class="fa fa-spinner fa-spin"></i> Uploading...').prop('disabled', true);

    $.ajax({
        url: '<?php echo base_url(); ?>admin/lab_reports/upload_report',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            if (response.success) {
                toastr.success('Lab report uploaded successfully!');
                $('#uploadForm')[0].reset();
                $('#fileInfo').hide();
                loadPatientReports();
                loadDashboardStats();
                $('#reportsTable').DataTable().ajax.reload(null, false);
            } else {
                toastr.error('Upload failed: ' + response.message);
            }
        },
        error: function() {
            toastr.error('Error uploading file');
        },
        complete: function() {
            submitBtn.html(originalText).prop('disabled', false);
        }
    });
}

function viewReport(reportId) {
    $.ajax({
        url: '<?php echo base_url(); ?>admin/lab_reports/get_report/' + reportId,
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                const report = response.data;
                let previewHtml = '';

                if (report.file_type && report.file_type.includes('image')) {
                    previewHtml = `<img src="${report.file_path}" style="max-width: 100%; height: 400px;" alt="Lab Report">`;
                } else if (report.file_type && report.file_type.includes('pdf')) {
                    previewHtml = `<embed src="${report.file_path}" type="application/pdf" style="width: 100%; height: 400px;">`;
                } else {
                    previewHtml = `
                        <div class="text-center" style="padding: 40px;">
                            <i class="fa fa-file-text-o fa-4x text-muted"></i>
                            <h4>${report.report_title}</h4>
                            <p><strong>File:</strong> ${report.original_filename}</p>
                            <p><strong>Size:</strong> ${(report.file_size / 1024 / 1024).toFixed(2)} MB</p>
                            <p><strong>Type:</strong> ${report.file_type}</p>
                        </div>
                    `;
                }

                $('#reportPreview').html(previewHtml);
                $('#downloadReportBtn').attr('onclick', `downloadReport(${reportId})`);
                
                if (report.status !== 'verified') {
                    $('#verifyReportBtn').show().attr('onclick', `verifyReport(${reportId})`);
                } else {
                    $('#verifyReportBtn').hide();
                }
                
                $('#viewReportModal').modal('show');
            } else {
                toastr.error('Failed to load report');
            }
        },
        error: function() {
            toastr.error('Error loading report');
        }
    });
}

function downloadReport(reportId) {
    window.open('<?php echo base_url(); ?>admin/lab_reports/download_report/' + reportId, '_blank');
}

function verifyReport(reportId) {
    if (confirm('Are you sure you want to verify this report?')) {
        $.ajax({
            url: '<?php echo base_url(); ?>admin/lab_reports/verify_report',
            method: 'POST',
            data: { 
                report_id: reportId,
                notes: 'Report verified by staff'
            },
            dataType: 'json',
            success: function(response) {
                if (response.success) {
                    toastr.success('Report verified successfully!');
                    loadPatientReports();
                    loadDashboardStats();
                    $('#reportsTable').DataTable().ajax.reload(null, false);
                    $('#viewReportModal').modal('hide');
                } else {
                    toastr.error('Verification failed: ' + response.message);
                }
            },
            error: function() {
                toastr.error('Error verifying report');
            }
        });
    }
}

function loadDashboardStats() {
    $.ajax({
        url: '<?php echo base_url(); ?>admin/lab_reports/get_dashboard_stats',
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                const stats = response.data;
                $('#totalReports').text(stats.total || 0);
                $('#pendingReports').text(stats.pending || 0);
                $('#verifiedReports').text(stats.verified_today || 0);
                $('#criticalReports').text(stats.critical || 0);
            }
        },
        error: function() {
            console.log('Error loading dashboard stats');
        }
    });
}

function refreshPatientReports() {
    if (currentPatientId) {
        loadPatientReports();
    } else {
        toastr.info('Please select a patient first');
    }
}

function viewPatientHistory() {
    if (currentPatientId) {
        // Scroll to the all reports table and filter by current patient
        $('html, body').animate({
            scrollTop: $("#reportsTable").offset().top - 100
        }, 1000);
        
        // You can add additional filtering logic here if needed
        toastr.info('Showing all reports table below');
    }
}

function applyFilters() {
    $('#reportsTable').DataTable().ajax.reload();
}

function clearFilters() {
    $('#filterStatus, #filterDepartment, #filterDateFrom, #filterDateTo').val('');
    $('#reportsTable').DataTable().ajax.reload();
}

function exportReports() {
    const params = new URLSearchParams({
        status: $('#filterStatus').val(),
        department: $('#filterDepartment').val(),
        date_from: $('#filterDateFrom').val(),
        date_to: $('#filterDateTo').val()
    });
    
    window.open('<?php echo base_url(); ?>admin/lab_reports/export_reports?' + params.toString(), '_blank');
}

// Auto-refresh dashboard stats every 5 minutes
setInterval(function() {
    loadDashboardStats();
}, 300000);

// Initialize tooltips
$(document).ready(function() {
    $('[data-toggle="tooltip"]').tooltip();
});
</script>