<?php
$currency_symbol = $this->customlib->getHospitalCurrencyFormat();
?>

<div class="content-wrapper">
    <section class="content">
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-line-chart"></i> Financial Reports
                        </h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" data-widget="collapse">
                                <i class="fa fa-minus"></i>
                            </button>
                        </div>
                    </div>

                    <ul class="nav nav-tabs navlistscroll">
                        <li class="active">
                            <a href="#tab_income_statement" data-toggle="tab">
                                <i class="fa fa-bar-chart"></i> Income Statement
                            </a>
                        </li>
                        <li>
                            <a href="#tab_trial_balance" data-toggle="tab">
                                <i class="fa fa-balance-scale"></i> Trial Balance
                            </a>
                        </li>
                        <li>
                            <a href="#tab_daily_income" data-toggle="tab">
                                <i class="fa fa-calendar-o"></i> Daily Income
                            </a>
                        </li>
                    </ul>

                    <div class="tab-content">
                        <!-- Income Statement Tab -->
                        <div class="tab-pane active" id="tab_income_statement">
                            <div class="box-body">
                                <div class="row">
                                    <div class="col-md-12">
                                        <h4><i class="fa fa-bar-chart text-primary"></i> Income Statement</h4>
                                        
                                        <!-- Date Range Selection -->
                                        <div class="row" style="margin-bottom: 20px;">
                                            <div class="col-md-4">
                                                <div class="form-group">
                                                    <label>Start Date</label>
                                                    <input type="date" id="income_start_date" class="form-control" value="<?php echo $financial_year_start; ?>">
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="form-group">
                                                    <label>End Date</label>
                                                    <input type="date" id="income_end_date" class="form-control" value="<?php echo $financial_year_end; ?>">
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="form-group">
                                                    <label>&nbsp;</label><br>
                                                    <button type="button" id="generateIncomeStatement" class="btn btn-primary">
                                                        <i class="fa fa-refresh"></i> Generate Report
                                                    </button>
                                                    <div class="btn-group">
                                                        <button type="button" class="btn btn-info dropdown-toggle" data-toggle="dropdown">
                                                            <i class="fa fa-download"></i> Export <span class="caret"></span>
                                                        </button>
                                                        <ul class="dropdown-menu">
                                                            <li><a href="#" id="exportIncomeCSV"><i class="fa fa-file-text-o"></i> CSV</a></li>
                                                            <li><a href="#" id="exportIncomeExcel"><i class="fa fa-file-excel-o"></i> Excel</a></li>
                                                            <li><a href="#" id="exportIncomePDF"><i class="fa fa-file-pdf-o"></i> PDF</a></li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Income Statement Display -->
                                        <div id="incomeStatementContent">
                                            <div class="alert alert-info">
                                                <i class="fa fa-info-circle"></i> Click "Generate Report" to view the income statement.
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Trial Balance Tab -->
                        <div class="tab-pane" id="tab_trial_balance">
                            <div class="box-body">
                                <div class="row">
                                    <div class="col-md-12">
                                        <h4><i class="fa fa-balance-scale text-success"></i> Trial Balance</h4>
                                        
                                        <!-- Date Selection -->
                                        <div class="row" style="margin-bottom: 20px;">
                                            <div class="col-md-4">
                                                <div class="form-group">
                                                    <label>As of Date</label>
                                                    <input type="date" id="trial_balance_date" class="form-control" value="<?php echo date('Y-m-d'); ?>">
                                                </div>
                                            </div>
                                            <div class="col-md-8">
                                                <div class="form-group">
                                                    <label>&nbsp;</label><br>
                                                    <button type="button" id="generateTrialBalance" class="btn btn-success">
                                                        <i class="fa fa-refresh"></i> Generate Trial Balance
                                                    </button>
                                                    <div class="btn-group">
                                                        <button type="button" class="btn btn-info dropdown-toggle" data-toggle="dropdown">
                                                            <i class="fa fa-download"></i> Export <span class="caret"></span>
                                                        </button>
                                                        <ul class="dropdown-menu">
                                                            <li><a href="#" id="exportTrialCSV"><i class="fa fa-file-text-o"></i> CSV</a></li>
                                                            <li><a href="#" id="exportTrialExcel"><i class="fa fa-file-excel-o"></i> Excel</a></li>
                                                            <li><a href="#" id="exportTrialPDF"><i class="fa fa-file-pdf-o"></i> PDF</a></li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Trial Balance Display -->
                                        <div id="trialBalanceContent">
                                            <div class="alert alert-info">
                                                <i class="fa fa-info-circle"></i> Click "Generate Trial Balance" to view the trial balance.
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Daily Income Tab -->
                        <div class="tab-pane" id="tab_daily_income">
                            <div class="box-body">
                                <div class="row">
                                    <div class="col-md-12">
                                        <h4><i class="fa fa-calendar-o text-warning"></i> Daily Income Report</h4>
                                        
                                        <!-- Date Selection -->
                                        <div class="row" style="margin-bottom: 20px;">
                                            <div class="col-md-4">
                                                <div class="form-group">
                                                    <label>Select Date</label>
                                                    <input type="date" id="daily_income_date" class="form-control" value="<?php echo date('Y-m-d'); ?>">
                                                </div>
                                            </div>
                                            <div class="col-md-8">
                                                <div class="form-group">
                                                    <label>&nbsp;</label><br>
                                                    <button type="button" id="generateDailyIncome" class="btn btn-warning">
                                                        <i class="fa fa-refresh"></i> Generate Daily Report
                                                    </button>
                                                    <div class="btn-group">
                                                        <button type="button" class="btn btn-info dropdown-toggle" data-toggle="dropdown">
                                                            <i class="fa fa-download"></i> Export <span class="caret"></span>
                                                        </button>
                                                        <ul class="dropdown-menu">
                                                            <li><a href="#" id="exportDailyCSV"><i class="fa fa-file-text-o"></i> CSV</a></li>
                                                            <li><a href="#" id="exportDailyExcel"><i class="fa fa-file-excel-o"></i> Excel</a></li>
                                                            <li><a href="#" id="exportDailyPDF"><i class="fa fa-file-pdf-o"></i> PDF</a></li>
                                                        </ul>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>

                                        <!-- Daily Income Display -->
                                        <div id="dailyIncomeContent">
                                            <div class="alert alert-info">
                                                <i class="fa fa-info-circle"></i> Click "Generate Daily Report" to view daily income breakdown.
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<script type="text/javascript">
$(document).ready(function() {
    var currency_symbol = '<?php echo $currency_symbol; ?>';
    
    // =============================================
    // INCOME STATEMENT FUNCTIONS
    // =============================================
    
    $('#generateIncomeStatement').on('click', function() {
        var start_date = $('#income_start_date').val();
        var end_date = $('#income_end_date').val();
        
        if (!start_date || !end_date) {
            alert('Please select both start and end dates.');
            return;
        }
        
        if (start_date > end_date) {
            alert('Start date cannot be after end date.');
            return;
        }
        
        $(this).html('<i class="fa fa-spinner fa-spin"></i> Generating...').prop('disabled', true);
        
        // Use GET request with URL segments
        var url = '<?php echo base_url(); ?>admin/financial_reports/income_statement/' + start_date + '/' + end_date;
        
        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'json',
            cache: false,
            success: function(response) {
                console.log('Income Statement Response:', response);
                if (response.status === 'success') {
                    displayIncomeStatement(response.data);
                } else {
                    alert('Error generating income statement: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.log('Income Statement Error:', xhr.responseText);
                console.log('Status:', status);
                console.log('Error:', error);
                alert('Error communicating with server. Please check console for details.');
            },
            complete: function() {
                $('#generateIncomeStatement').html('<i class="fa fa-refresh"></i> Generate Report').prop('disabled', false);
            }
        });
    });
    
    function displayIncomeStatement(data) {
        var total_revenue = parseFloat(data.total_revenue);
        var total_expenses = parseFloat(data.total_expenses);
        var net_profit = parseFloat(data.net_profit);
        var profit_margin = parseFloat(data.profit_margin);
        
        var html = `
            <div class="panel panel-primary">
                <div class="panel-heading">
                    <h4 class="panel-title">Income Statement - ${data.period.start_date} to ${data.period.end_date}</h4>
                </div>
                <div class="panel-body">
                    <div class="row">
                        <div class="col-md-6">
                            <h5><strong>REVENUE</strong></h5>
                            <table class="table table-striped table-bordered">
                                <tr><td>OPD Consultations</td><td class="text-right">${currency_symbol}${formatNumber(data.revenue.opd_consultations)}</td></tr>
                                <tr><td>IPD Services</td><td class="text-right">${currency_symbol}${formatNumber(data.revenue.ipd_services)}</td></tr>
                                <tr><td>Pharmacy Sales</td><td class="text-right">${currency_symbol}${formatNumber(data.revenue.pharmacy_sales)}</td></tr>
                                <tr><td>Laboratory Tests</td><td class="text-right">${currency_symbol}${formatNumber(data.revenue.laboratory_tests)}</td></tr>
                                <tr><td>Ambulance Services</td><td class="text-right">${currency_symbol}${formatNumber(data.revenue.ambulance_services)}</td></tr>
                                <tr><td>Other Income</td><td class="text-right">${currency_symbol}${formatNumber(data.revenue.other_income)}</td></tr>
                                <tr class="success"><td><strong>TOTAL REVENUE</strong></td><td class="text-right"><strong>${currency_symbol}${formatNumber(total_revenue)}</strong></td></tr>
                            </table>
                        </div>
                        <div class="col-md-6">
                            <h5><strong>EXPENSES</strong></h5>
                            <table class="table table-striped table-bordered">
                                <tr><td>Staff Salaries</td><td class="text-right">${currency_symbol}${formatNumber(data.expenses.staff_salaries)}</td></tr>
                                <tr><td>Medical Supplies</td><td class="text-right">${currency_symbol}${formatNumber(data.expenses.medical_supplies)}</td></tr>
                                <tr><td>Utilities</td><td class="text-right">${currency_symbol}${formatNumber(data.expenses.utilities)}</td></tr>
                                <tr><td>Maintenance</td><td class="text-right">${currency_symbol}${formatNumber(data.expenses.maintenance)}</td></tr>
                                <tr><td>Administrative</td><td class="text-right">${currency_symbol}${formatNumber(data.expenses.administrative)}</td></tr>
                                <tr><td>Other Expenses</td><td class="text-right">${currency_symbol}${formatNumber(data.expenses.other_expenses)}</td></tr>
                                <tr class="danger"><td><strong>TOTAL EXPENSES</strong></td><td class="text-right"><strong>${currency_symbol}${formatNumber(total_expenses)}</strong></td></tr>
                            </table>
                        </div>
                    </div>
                    <div class="row">
                        <div class="col-md-12">
                            <div class="alert ${net_profit >= 0 ? 'alert-success' : 'alert-danger'}">
                                <div class="row">
                                    <div class="col-md-6">
                                        <h4><strong>NET ${net_profit >= 0 ? 'PROFIT' : 'LOSS'}: ${currency_symbol}${formatNumber(Math.abs(net_profit))}</strong></h4>
                                    </div>
                                    <div class="col-md-6 text-right">
                                        <h4><strong>Profit Margin: ${profit_margin}%</strong></h4>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Revenue Chart -->
                    <div class="row" style="margin-top: 20px;">
                        <div class="col-md-12">
                            <h5><strong>Revenue Breakdown</strong></h5>
                            <div class="progress" style="height: 30px;">
                                <div class="progress-bar progress-bar-info" style="width: ${(data.revenue.opd_consultations/total_revenue*100)}%">
                                    OPD ${((data.revenue.opd_consultations/total_revenue)*100).toFixed(1)}%
                                </div>
                                <div class="progress-bar progress-bar-success" style="width: ${(data.revenue.ipd_services/total_revenue*100)}%">
                                    IPD ${((data.revenue.ipd_services/total_revenue)*100).toFixed(1)}%
                                </div>
                                <div class="progress-bar progress-bar-warning" style="width: ${(data.revenue.pharmacy_sales/total_revenue*100)}%">
                                    Pharmacy ${((data.revenue.pharmacy_sales/total_revenue)*100).toFixed(1)}%
                                </div>
                                <div class="progress-bar progress-bar-danger" style="width: ${(data.revenue.laboratory_tests/total_revenue*100)}%">
                                    Lab ${((data.revenue.laboratory_tests/total_revenue)*100).toFixed(1)}%
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        $('#incomeStatementContent').html(html);
    }
    
    // =============================================
    // TRIAL BALANCE FUNCTIONS
    // =============================================
    
    $('#generateTrialBalance').on('click', function() {
        var as_of_date = $('#trial_balance_date').val();
        
        if (!as_of_date) {
            alert('Please select a date.');
            return;
        }
        
        $(this).html('<i class="fa fa-spinner fa-spin"></i> Generating...').prop('disabled', true);
        
        var url = '<?php echo base_url(); ?>admin/financial_reports/trial_balance/' + as_of_date;
        
        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'json',
            cache: false,
            success: function(response) {
                console.log('Trial Balance Response:', response);
                if (response.status === 'success') {
                    displayTrialBalance(response.data);
                } else {
                    alert('Error generating trial balance: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.log('Trial Balance Error:', xhr.responseText);
                alert('Error communicating with server.');
            },
            complete: function() {
                $('#generateTrialBalance').html('<i class="fa fa-refresh"></i> Generate Trial Balance').prop('disabled', false);
            }
        });
    });
    
    function displayTrialBalance(data) {
        var html = `
            <div class="panel panel-success">
                <div class="panel-heading">
                    <h4 class="panel-title">Trial Balance - As of ${data.as_of_date}</h4>
                </div>
                <div class="panel-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-bordered">
                            <thead>
                                <tr class="success">
                                    <th width="40%">Account Name</th>
                                    <th width="20%" class="text-center">Account Type</th>
                                    <th width="20%" class="text-right">Debit</th>
                                    <th width="20%" class="text-right">Credit</th>
                                </tr>
                            </thead>
                            <tbody>
        `;
        
        data.accounts.forEach(function(account) {
            html += `
                <tr>
                    <td>${account.account_name}</td>
                    <td class="text-center"><span class="label label-${getAccountTypeClass(account.account_type)}">${account.account_type}</span></td>
                    <td class="text-right">${account.debit > 0 ? currency_symbol + formatNumber(account.debit) : '-'}</td>
                    <td class="text-right">${account.credit > 0 ? currency_symbol + formatNumber(account.credit) : '-'}</td>
                </tr>
            `;
        });
        
        html += `
                            </tbody>
                            <tfoot>
                                <tr class="info">
                                    <td colspan="2"><strong>TOTALS</strong></td>
                                    <td class="text-right"><strong>${currency_symbol}${formatNumber(data.total_debits)}</strong></td>
                                    <td class="text-right"><strong>${currency_symbol}${formatNumber(data.total_credits)}</strong></td>
                                </tr>
                            </tfoot>
                        </table>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-6">
                            <div class="alert ${data.is_balanced ? 'alert-success' : 'alert-danger'}">
                                <h4><i class="fa fa-${data.is_balanced ? 'check' : 'exclamation-triangle'}"></i> 
                                Trial Balance is ${data.is_balanced ? 'BALANCED' : 'NOT BALANCED'}</h4>
                                ${!data.is_balanced ? `<p>Difference: ${currency_symbol}${formatNumber(Math.abs(data.difference))}</p>` : ''}
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="info-box">
                                <span class="info-box-icon bg-blue"><i class="fa fa-calculator"></i></span>
                                <div class="info-box-content">
                                    <span class="info-box-text">Total Accounts</span>
                                    <span class="info-box-number">${data.accounts.length}</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        $('#trialBalanceContent').html(html);
    }
    
    function getAccountTypeClass(type) {
        switch(type) {
            case 'Asset': return 'primary';
            case 'Liability': return 'danger';
            case 'Equity': return 'warning';
            case 'Revenue': return 'success';
            case 'Expense': return 'info';
            default: return 'default';
        }
    }
    
    // =============================================
    // DAILY INCOME FUNCTIONS
    // =============================================
    
    $('#generateDailyIncome').on('click', function() {
        var date = $('#daily_income_date').val();
        
        if (!date) {
            alert('Please select a date.');
            return;
        }
        
        $(this).html('<i class="fa fa-spinner fa-spin"></i> Generating...').prop('disabled', true);
        
        var url = '<?php echo base_url(); ?>admin/financial_reports/daily_income/' + date;
        
        $.ajax({
            url: url,
            type: 'GET',
            dataType: 'json',
            cache: false,
            success: function(response) {
                console.log('Daily Income Response:', response);
                if (response.status === 'success') {
                    displayDailyIncome(response.data);
                } else {
                    alert('Error generating daily income: ' + response.message);
                }
            },
            error: function(xhr, status, error) {
                console.log('Daily Income Error:', xhr.responseText);
                alert('Error communicating with server.');
            },
            complete: function() {
                $('#generateDailyIncome').html('<i class="fa fa-refresh"></i> Generate Daily Report').prop('disabled', false);
            }
        });
    });
    
    function displayDailyIncome(data) {
        var html = `
            <div class="panel panel-warning">
                <div class="panel-heading">
                    <h4 class="panel-title">Daily Income Report - ${data.date}</h4>
                </div>
                <div class="panel-body">
                    <div class="row">
                        <!-- Summary Cards -->
                        <div class="col-md-12" style="margin-bottom: 20px;">
                            <div class="row">
                                <div class="col-md-4">
                                    <div class="info-box">
                                        <span class="info-box-icon bg-yellow"><i class="fa fa-money"></i></span>
                                        <div class="info-box-content">
                                            <span class="info-box-text">Total Income</span>
                                            <span class="info-box-number">${currency_symbol}${formatNumber(data.total_income)}</span>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="info-box">
                                        <span class="info-box-icon bg-blue"><i class="fa fa-list"></i></span>
                                        <div class="info-box-content">
                                            <span class="info-box-text">Total Transactions</span>
                                            <span class="info-box-number">${data.total_transactions}</span>
                                        </div>
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="info-box">
                                        <span class="info-box-icon bg-green"><i class="fa fa-calculator"></i></span>
                                        <div class="info-box-content">
                                            <span class="info-box-text">Average per Transaction</span>
                                            <span class="info-box-number">${currency_symbol}${formatNumber(data.average_transaction)}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-8">
                            <h5><strong>SERVICE BREAKDOWN</strong></h5>
                            <div class="table-responsive">
                                <table class="table table-striped table-bordered">
                                    <thead>
                                        <tr class="warning">
                                            <th width="40%">Service</th>
                                            <th width="20%" class="text-center">Transactions</th>
                                            <th width="25%" class="text-right">Amount</th>
                                            <th width="15%" class="text-center">%</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <tr>
                                            <td><i class="fa fa-user-md text-primary"></i> OPD Consultations</td>
                                            <td class="text-center"><span class="badge bg-blue">${data.income_breakdown.opd_consultations.count}</span></td>
                                            <td class="text-right">${currency_symbol}${formatNumber(data.income_breakdown.opd_consultations.amount)}</td>
                                            <td class="text-center">${((data.income_breakdown.opd_consultations.amount/data.total_income)*100).toFixed(1)}%</td>
                                        </tr>
                                        <tr>
                                            <td><i class="fa fa-bed text-success"></i> IPD Services</td>
                                            <td class="text-center"><span class="badge bg-green">${data.income_breakdown.ipd_services.count}</span></td>
                                            <td class="text-right">${currency_symbol}${formatNumber(data.income_breakdown.ipd_services.amount)}</td>
                                            <td class="text-center">${((data.income_breakdown.ipd_services.amount/data.total_income)*100).toFixed(1)}%</td>
                                        </tr>
                                        <tr>
                                            <td><i class="fa fa-medkit text-warning"></i> Pharmacy Sales</td>
                                            <td class="text-center"><span class="badge bg-yellow">${data.income_breakdown.pharmacy_sales.count}</span></td>
                                            <td class="text-right">${currency_symbol}${formatNumber(data.income_breakdown.pharmacy_sales.amount)}</td>
                                            <td class="text-center">${((data.income_breakdown.pharmacy_sales.amount/data.total_income)*100).toFixed(1)}%</td>
                                        </tr>
                                        <tr>
                                            <td><i class="fa fa-flask text-info"></i> Laboratory Tests</td>
                                            <td class="text-center"><span class="badge bg-light-blue">${data.income_breakdown.laboratory_tests.count}</span></td>
                                            <td class="text-right">${currency_symbol}${formatNumber(data.income_breakdown.laboratory_tests.amount)}</td>
                                            <td class="text-center">${((data.income_breakdown.laboratory_tests.amount/data.total_income)*100).toFixed(1)}%</td>
                                        </tr>
                                        <tr>
                                            <td><i class="fa fa-ambulance text-danger"></i> Ambulance Services</td>
                                            <td class="text-center"><span class="badge bg-red">${data.income_breakdown.ambulance_services.count}</span></td>
                                            <td class="text-right">${currency_symbol}${formatNumber(data.income_breakdown.ambulance_services.amount)}</td>
                                            <td class="text-center">${((data.income_breakdown.ambulance_services.amount/data.total_income)*100).toFixed(1)}%</td>
                                        </tr>
                                        <tr>
                                            <td><i class="fa fa-plus text-muted"></i> Other Services</td>
                                            <td class="text-center"><span class="badge bg-gray">${data.income_breakdown.other_services.count}</span></td>
                                            <td class="text-right">${currency_symbol}${formatNumber(data.income_breakdown.other_services.amount)}</td>
                                            <td class="text-center">${((data.income_breakdown.other_services.amount/data.total_income)*100).toFixed(1)}%</td>
                                        </tr>
                                    </tbody>
                                    <tfoot>
                                        <tr class="success">
                                            <td><strong>TOTAL</strong></td>
                                            <td class="text-center"><strong><span class="badge bg-green">${data.total_transactions}</span></strong></td>
                                            <td class="text-right"><strong>${currency_symbol}${formatNumber(data.total_income)}</strong></td>
                                            <td class="text-center"><strong>100%</strong></td>
                                        </tr>
                                    </tfoot>
                                </table>
                            </div>
                        </div>
                        
                        <div class="col-md-4">
                            <h5><strong>PAYMENT METHODS</strong></h5>
        `;
        
        // Payment methods breakdown
        if (data.payment_methods && Object.keys(data.payment_methods).length > 0) {
            html += `
                            <div class="table-responsive">
                                <table class="table table-condensed table-bordered">
                                    <thead>
                                        <tr class="info">
                                            <th>Method</th>
                                            <th class="text-right">Amount</th>
                                        </tr>
                                    </thead>
                                    <tbody>
            `;
            for (var method in data.payment_methods) {
                var methodIcon = getPaymentMethodIcon(method);
                html += `
                    <tr>
                        <td><i class="fa ${methodIcon}"></i> ${capitalizeFirst(method)}</td>
                        <td class="text-right">${currency_symbol}${formatNumber(data.payment_methods[method])}</td>
                    </tr>
                `;
            }
            html += `
                                    </tbody>
                                </table>
                            </div>
            `;
        } else {
            html += `<p class="text-muted">No payment method data available</p>`;
        }
        
        // Hourly breakdown chart
        html += `
                            <h5 style="margin-top: 20px;"><strong>HOURLY BREAKDOWN</strong></h5>
                            <div style="max-height: 200px; overflow-y: auto;">
                                <table class="table table-condensed table-striped">
                                    <thead>
                                        <tr>
                                            <th>Hour</th>
                                            <th class="text-right">Income</th>
                                        </tr>
                                    </thead>
                                    <tbody>
        `;
        
        if (data.hourly_breakdown && Object.keys(data.hourly_breakdown).length > 0) {
            Object.keys(data.hourly_breakdown).forEach(function(hour) {
                if (data.hourly_breakdown[hour] > 0) {
                    html += `
                        <tr>
                            <td>${hour}</td>
                            <td class="text-right">${currency_symbol}${formatNumber(data.hourly_breakdown[hour])}</td>
                        </tr>
                    `;
                }
            });
        } else {
            html += '<tr><td colspan="2" class="text-muted text-center">No hourly data available</td></tr>';
        }
        
        html += `
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        `;
        
        $('#dailyIncomeContent').html(html);
    }
    
    function getPaymentMethodIcon(method) {
        switch(method.toLowerCase()) {
            case 'cash': return 'fa-money';
            case 'card': case 'credit_card': case 'debit_card': return 'fa-credit-card';
            case 'cheque': case 'check': return 'fa-file-text-o';
            case 'bank_transfer': case 'transfer': return 'fa-bank';
            case 'mobile_money': case 'mpesa': return 'fa-mobile';
            default: return 'fa-payment';
        }
    }
    
    function capitalizeFirst(str) {
        return str.charAt(0).toUpperCase() + str.slice(1).replace('_', ' ');
    }
    
    // =============================================
    // EXPORT FUNCTIONS
    // =============================================
    
    // Income Statement Exports
    $('#exportIncomeCSV').on('click', function(e) {
        e.preventDefault();
        var start_date = $('#income_start_date').val();
        var end_date = $('#income_end_date').val();
        if (start_date && end_date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_income_statement/' + start_date + '/' + end_date + '/csv', '_blank');
        } else {
            alert('Please generate the income statement first');
        }
    });
    
    $('#exportIncomeExcel').on('click', function(e) {
        e.preventDefault();
        var start_date = $('#income_start_date').val();
        var end_date = $('#income_end_date').val();
        if (start_date && end_date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_income_statement/' + start_date + '/' + end_date + '/excel', '_blank');
        } else {
            alert('Please generate the income statement first');
        }
    });
    
    $('#exportIncomePDF').on('click', function(e) {
        e.preventDefault();
        var start_date = $('#income_start_date').val();
        var end_date = $('#income_end_date').val();
        if (start_date && end_date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_income_statement/' + start_date + '/' + end_date + '/pdf', '_blank');
        } else {
            alert('Please generate the income statement first');
        }
    });
    
    // Trial Balance Exports
    $('#exportTrialCSV').on('click', function(e) {
        e.preventDefault();
        var as_of_date = $('#trial_balance_date').val();
        if (as_of_date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_trial_balance/' + as_of_date + '/csv', '_blank');
        } else {
            alert('Please generate the trial balance first');
        }
    });
    
    $('#exportTrialExcel').on('click', function(e) {
        e.preventDefault();
        var as_of_date = $('#trial_balance_date').val();
        if (as_of_date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_trial_balance/' + as_of_date + '/excel', '_blank');
        } else {
            alert('Please generate the trial balance first');
        }
    });
    
    $('#exportTrialPDF').on('click', function(e) {
        e.preventDefault();
        var as_of_date = $('#trial_balance_date').val();
        if (as_of_date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_trial_balance/' + as_of_date + '/pdf', '_blank');
        } else {
            alert('Please generate the trial balance first');
        }
    });
    
    // Daily Income Exports
    $('#exportDailyCSV').on('click', function(e) {
        e.preventDefault();
        var date = $('#daily_income_date').val();
        if (date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_daily_income/' + date + '/csv', '_blank');
        } else {
            alert('Please generate the daily report first');
        }
    });
    
    $('#exportDailyExcel').on('click', function(e) {
        e.preventDefault();
        var date = $('#daily_income_date').val();
        if (date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_daily_income/' + date + '/excel', '_blank');
        } else {
            alert('Please generate the daily report first');
        }
    });
    
    $('#exportDailyPDF').on('click', function(e) {
        e.preventDefault();
        var date = $('#daily_income_date').val();
        if (date) {
            window.open('<?php echo base_url(); ?>admin/financial_reports/export_daily_income/' + date + '/pdf', '_blank');
        } else {
            alert('Please generate the daily report first');
        }
    });
    
    // =============================================
    // UTILITY FUNCTIONS
    // =============================================
    
    function formatNumber(number) {
        if (isNaN(number) || number === null || number === undefined) {
            return '0.00';
        }
        return parseFloat(number).toLocaleString('en-US', {
            minimumFractionDigits: 2,
            maximumFractionDigits: 2
        });
    }
    
    function showMessage(message, type, duration) {
        duration = duration || 5000;
        var alertClass = 'alert-' + (type || 'info');
        var alertHtml = `
            <div class="alert ${alertClass} alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">
                <button type="button" class="close" data-dismiss="alert">&times;</button>
                <strong>${type.toUpperCase()}:</strong> ${message}
            </div>
        `;
        
        $('body').append(alertHtml);
        
        setTimeout(function() {
            $('.alert').fadeOut(function() {
                $(this).remove();
            });
        }, duration);
    }
    
    // =============================================
    // DATE PICKER IMPROVEMENTS
    // =============================================
    
    // Set financial year dates
    function setFinancialYearDates() {
        var currentDate = new Date();
        var currentYear = currentDate.getFullYear();
        var startYear, endYear;
        
        if (currentDate.getMonth() >= 3) { // April onwards
            startYear = currentYear;
            endYear = currentYear + 1;
        } else {
            startYear = currentYear - 1;
            endYear = currentYear;
        }
        
        $('#income_start_date').val(startYear + '-04-01');
        $('#income_end_date').val(endYear + '-03-31');
    }
    
    // Quick date range buttons
    $('<div class="btn-group" style="margin-left: 10px;">' +
        '<button type="button" class="btn btn-sm btn-default" id="thisMonth">This Month</button>' +
       '<button type="button" class="btn btn-sm btn-default" id="lastMonth">Last Month</button>' +
       '<button type="button" class="btn btn-sm btn-default" id="thisYear">This Year</button>' +
       '<button type="button" class="btn btn-sm btn-default" id="financialYear">Financial Year</button>' +
     '</div>').insertAfter('#generateIncomeStatement');
   
   // Quick date range handlers
   $('#thisMonth').on('click', function() {
       var now = new Date();
       var start = new Date(now.getFullYear(), now.getMonth(), 1);
       var end = new Date(now.getFullYear(), now.getMonth() + 1, 0);
       
       $('#income_start_date').val(formatDate(start));
       $('#income_end_date').val(formatDate(end));
   });
   
   $('#lastMonth').on('click', function() {
       var now = new Date();
       var start = new Date(now.getFullYear(), now.getMonth() - 1, 1);
       var end = new Date(now.getFullYear(), now.getMonth(), 0);
       
       $('#income_start_date').val(formatDate(start));
       $('#income_end_date').val(formatDate(end));
   });
   
   $('#thisYear').on('click', function() {
       var year = new Date().getFullYear();
       $('#income_start_date').val(year + '-01-01');
       $('#income_end_date').val(year + '-12-31');
   });
   
   $('#financialYear').on('click', function() {
       setFinancialYearDates();
   });
   
   function formatDate(date) {
       var d = new Date(date);
       var month = '' + (d.getMonth() + 1);
       var day = '' + d.getDate();
       var year = d.getFullYear();

       if (month.length < 2) month = '0' + month;
       if (day.length < 2) day = '0' + day;

       return [year, month, day].join('-');
   }
   
   // =============================================
   // KEYBOARD SHORTCUTS
   // =============================================
   
   $(document).keydown(function(e) {
       // Ctrl+1 for Income Statement
       if (e.ctrlKey && e.key === '1') {
           e.preventDefault();
           $('#generateIncomeStatement').click();
       }
       
       // Ctrl+2 for Trial Balance
       if (e.ctrlKey && e.key === '2') {
           e.preventDefault();
           $('#generateTrialBalance').click();
       }
       
       // Ctrl+3 for Daily Income
       if (e.ctrlKey && e.key === '3') {
           e.preventDefault();
           $('#generateDailyIncome').click();
       }
       
       // Ctrl+E for Export
       if (e.ctrlKey && e.key.toLowerCase() === 'e') {
           e.preventDefault();
           var activeTab = $('.nav-tabs .active a').attr('href');
           if (activeTab === '#tab_income_statement') {
               $('#exportIncomeCSV').click();
           } else if (activeTab === '#tab_trial_balance') {
               $('#exportTrialCSV').click();
           } else if (activeTab === '#tab_daily_income') {
               $('#exportDailyCSV').click();
           }
       }
   });
   
   // =============================================
   // DATA REFRESH FUNCTIONALITY
   // =============================================
   
   function refreshAllReports() {
       if ($('#incomeStatementContent .panel').length > 0) {
           $('#generateIncomeStatement').click();
       }
       if ($('#trialBalanceContent .panel').length > 0) {
           $('#generateTrialBalance').click();
       }
       if ($('#dailyIncomeContent .panel').length > 0) {
           $('#generateDailyIncome').click();
       }
   }
   
   // Add refresh button to each tab
   $('.nav-tabs li a').each(function() {
       var tabId = $(this).attr('href');
       var refreshBtn = '<button type="button" class="btn btn-xs btn-default pull-right refresh-tab" data-tab="' + tabId + '" style="margin-left: 10px;">' +
                       '<i class="fa fa-refresh"></i></button>';
       $(this).append(refreshBtn);
   });
   
   $('.refresh-tab').on('click', function(e) {
       e.preventDefault();
       e.stopPropagation();
       
       var tab = $(this).data('tab');
       if (tab === '#tab_income_statement') {
           $('#generateIncomeStatement').click();
       } else if (tab === '#tab_trial_balance') {
           $('#generateTrialBalance').click();
       } else if (tab === '#tab_daily_income') {
           $('#generateDailyIncome').click();
       }
   });
   
   // =============================================
   // PRINT FUNCTIONALITY
   // =============================================
   
   function printReport(containerId) {
       var printContent = document.getElementById(containerId).innerHTML;
       var originalContent = document.body.innerHTML;
       
       document.body.innerHTML = `
           <html>
           <head>
               <title>Financial Report</title>
               <style>
                   body { font-family: Arial, sans-serif; }
                   .panel { border: 1px solid #ddd; margin: 20px 0; }
                   .panel-heading { background: #f5f5f5; padding: 10px; border-bottom: 1px solid #ddd; }
                   .panel-body { padding: 15px; }
                   table { width: 100%; border-collapse: collapse; margin: 10px 0; }
                   th, td { border: 1px solid #ddd; padding: 8px; text-align: left; }
                   th { background-color: #f5f5f5; }
                   .text-right { text-align: right; }
                   .text-center { text-align: center; }
                   .alert { padding: 15px; margin: 20px 0; border-radius: 4px; }
                   .alert-success { background: #dff0d8; border-color: #d6e9c6; color: #3c763d; }
                   .alert-danger { background: #f2dede; border-color: #ebccd1; color: #a94442; }
                   @media print {
                       .no-print { display: none; }
                       .panel { page-break-inside: avoid; }
                   }
               </style>
           </head>
           <body>${printContent}</body>
           </html>
       `;
       
       window.print();
       document.body.innerHTML = originalContent;
       
       // Re-initialize event handlers after print
       setTimeout(function() {
           location.reload();
       }, 100);
   }
   
   // Add print buttons
   $('<button type="button" class="btn btn-default btn-sm" id="printIncome" style="margin-left: 5px;">' +
     '<i class="fa fa-print"></i> Print</button>').insertAfter('#exportIncomePDF');
   
   $('<button type="button" class="btn btn-default btn-sm" id="printTrial" style="margin-left: 5px;">' +
     '<i class="fa fa-print"></i> Print</button>').insertAfter('#exportTrialPDF');
   
   $('<button type="button" class="btn btn-default btn-sm" id="printDaily" style="margin-left: 5px;">' +
     '<i class="fa fa-print"></i> Print</button>').insertAfter('#exportDailyPDF');
   
   $('#printIncome').on('click', function() {
       if ($('#incomeStatementContent .panel').length > 0) {
           printReport('incomeStatementContent');
       } else {
           alert('Please generate the income statement first');
       }
   });
   
   $('#printTrial').on('click', function() {
       if ($('#trialBalanceContent .panel').length > 0) {
           printReport('trialBalanceContent');
       } else {
           alert('Please generate the trial balance first');
       }
   });
   
   $('#printDaily').on('click', function() {
       if ($('#dailyIncomeContent .panel').length > 0) {
           printReport('dailyIncomeContent');
       } else {
           alert('Please generate the daily report first');
       }
   });
   
   // =============================================
   // AUTO-REFRESH FUNCTIONALITY
   // =============================================
   
   var autoRefreshInterval;
   var autoRefreshEnabled = false;
   
   // Add auto-refresh toggle
   $('<div class="checkbox pull-right" style="margin-left: 15px;">' +
     '<label><input type="checkbox" id="autoRefresh"> Auto-refresh (5 min)</label></div>').insertAfter('.box-tools');
   
   $('#autoRefresh').on('change', function() {
       autoRefreshEnabled = $(this).is(':checked');
       
       if (autoRefreshEnabled) {
           autoRefreshInterval = setInterval(function() {
               refreshAllReports();
               showMessage('Reports auto-refreshed', 'info', 2000);
           }, 5 * 60 * 1000); // 5 minutes
           
           showMessage('Auto-refresh enabled (every 5 minutes)', 'success');
       } else {
           clearInterval(autoRefreshInterval);
           showMessage('Auto-refresh disabled', 'warning');
       }
   });
   
   // =============================================
   // COMPARISON FUNCTIONALITY
   // =============================================
   
   var comparisonData = {};
   
   function enableComparison() {
       // Add comparison buttons
       $('<button type="button" class="btn btn-info btn-sm" id="compareIncome" style="margin-left: 5px;">' +
         '<i class="fa fa-bar-chart"></i> Compare</button>').insertAfter('#printIncome');
       
       $('#compareIncome').on('click', function() {
           if ($('#incomeStatementContent .panel').length > 0) {
               showComparisonModal();
           } else {
               alert('Please generate an income statement first');
           }
       });
   }
   
   function showComparisonModal() {
       var modalHtml = `
           <div class="modal fade" id="comparisonModal" tabindex="-1" role="dialog">
               <div class="modal-dialog modal-lg" role="document">
                   <div class="modal-content">
                       <div class="modal-header">
                           <button type="button" class="close" data-dismiss="modal">&times;</button>
                           <h4 class="modal-title">Income Statement Comparison</h4>
                       </div>
                       <div class="modal-body">
                           <div class="row">
                               <div class="col-md-6">
                                   <h5>Compare with Previous Period</h5>
                                   <div class="form-group">
                                       <label>Start Date</label>
                                       <input type="date" id="compareStartDate" class="form-control">
                                   </div>
                                   <div class="form-group">
                                       <label>End Date</label>
                                       <input type="date" id="compareEndDate" class="form-control">
                                   </div>
                                   <button type="button" class="btn btn-primary" id="runComparison">Run Comparison</button>
                               </div>
                               <div class="col-md-6">
                                   <div id="comparisonResults"></div>
                               </div>
                           </div>
                       </div>
                   </div>
               </div>
           </div>
       `;
       
       $('body').append(modalHtml);
       $('#comparisonModal').modal('show');
       
       $('#comparisonModal').on('hidden.bs.modal', function() {
           $(this).remove();
       });
       
       $('#runComparison').on('click', function() {
           var startDate = $('#compareStartDate').val();
           var endDate = $('#compareEndDate').val();
           
           if (!startDate || !endDate) {
               alert('Please select both dates');
               return;
           }
           
           $(this).html('<i class="fa fa-spinner fa-spin"></i> Comparing...').prop('disabled', true);
           
           var url = '<?php echo base_url(); ?>admin/financial_reports/income_statement/' + startDate + '/' + endDate;
           
           $.ajax({
               url: url,
               type: 'GET',
               dataType: 'json',
               success: function(response) {
                   if (response.status === 'success') {
                       displayComparison(response.data);
                   } else {
                       alert('Error: ' + response.message);
                   }
               },
               complete: function() {
                   $('#runComparison').html('Run Comparison').prop('disabled', false);
               }
           });
       });
   }
   
   function displayComparison(compareData) {
       // This would require storing the current report data and comparing
       // Implementation would depend on specific comparison requirements
       $('#comparisonResults').html('<p>Comparison feature will show period-over-period changes here.</p>');
   }
   
   // =============================================
   // INITIALIZATION
   // =============================================
   
   // Set default dates
   setFinancialYearDates();
   
   // Enable comparison functionality
   enableComparison();
   
   // Auto-generate today's daily income on page load
   setTimeout(function() {
       $('#generateDailyIncome').click();
   }, 500);
   
   // Show keyboard shortcuts help
   $('<div class="alert alert-info alert-dismissible" style="margin-top: 10px;">' +
     '<button type="button" class="close" data-dismiss="alert">&times;</button>' +
     '<strong>Keyboard Shortcuts:</strong> Ctrl+1 (Income Statement), Ctrl+2 (Trial Balance), Ctrl+3 (Daily Income), Ctrl+E (Export)' +
     '</div>').prependTo('.box-body');
   
   // Console help message
   console.log('%c Financial Reports System Loaded Successfully! ', 'background: #4CAF50; color: white; padding: 5px 10px; border-radius: 3px;');
   console.log('Available functions:');
   console.log('- generateIncomeStatement()');
   console.log('- generateTrialBalance()'); 
   console.log('- generateDailyIncome()');
   console.log('- refreshAllReports()');
   console.log('- printReport(containerId)');
   
   // Make functions globally available for debugging
   window.generateIncomeStatement = function() { $('#generateIncomeStatement').click(); };
   window.generateTrialBalance = function() { $('#generateTrialBalance').click(); };
   window.generateDailyIncome = function() { $('#generateDailyIncome').click(); };
   window.refreshAllReports = refreshAllReports;
   window.printReport = printReport;
   window.formatNumber = formatNumber;
   
   // Error handling for missing elements
   if ($('#income_start_date').length === 0) {
       console.error('Income statement date inputs not found');
   }
   if ($('#trial_balance_date').length === 0) {
       console.error('Trial balance date input not found');
   }
   if ($('#daily_income_date').length === 0) {
       console.error('Daily income date input not found');
   }
   
   // Success message
   showMessage('Financial Reports system ready! Use Ctrl+1, Ctrl+2, Ctrl+3 for quick access.', 'success', 8000);
   
});
</script>