<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Chart of Accounts</title>
    <!-- AdminLTE styles -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/AdminLTE.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/skins/_all-skins.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/css/dataTables.bootstrap.min.css">
    
    <!-- jQuery and Bootstrap JS -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/js/bootstrap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/jquery.dataTables.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/dataTables.bootstrap.min.js"></script>
</head>
<body class="hold-transition skin-blue sidebar-mini" style="height: auto; min-height: 100vh; overflow-y: auto;">

<div class="content-wrapper">
    <!-- Content Header -->
    <section class="content-header">
        <h1>
            <i class="fa fa-list"></i> Chart of Accounts
            <small>Manage your account structure</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="#"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="#">Financial</a></li>
            <li class="active">Chart of Accounts</li>
        </ol>
    </section>

    <!-- Main content -->
    <section class="content">
        <!-- Flash Messages -->
        <div id="flashMessages"></div>

        <div class="row">
            <!-- Add Account Form -->
            <div class="col-md-4">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-plus"></i> Add New Account
                        </h3>
                    </div>
                    <form method="post" id="accountForm">
                        <div class="box-body">
                            <div class="form-group">
                                <label for="account_code">
                                    Account Code <span class="text-red">*</span>
                                </label>
                                <input type="text" 
                                       class="form-control" 
                                       name="account_code" 
                                       id="account_code" 
                                       placeholder="e.g., 1000" 
                                       required 
                                       maxlength="10">
                                <small class="help-block">Unique identifier for the account</small>
                            </div>

                            <div class="form-group">
                                <label for="account_name">
                                    Account Name <span class="text-red">*</span>
                                </label>
                                <input type="text" 
                                       class="form-control" 
                                       name="account_name" 
                                       id="account_name" 
                                       placeholder="e.g., Cash in Hand" 
                                       required 
                                       maxlength="100">
                            </div>

                            <div class="form-group">
                                <label for="account_type">
                                    Account Type <span class="text-red">*</span>
                                </label>
                                <select class="form-control" name="account_type" id="account_type" required>
                                    <option value="">Select Account Type</option>
                                    <option value="asset">Asset</option>
                                    <option value="liability">Liability</option>
                                    <option value="equity">Equity</option>
                                    <option value="revenue">Revenue</option>
                                    <option value="expense">Expense</option>
                                </select>
                            </div>

                            <div class="form-group">
                                <label for="parent_account_id">Parent Account</label>
                                <select class="form-control" name="parent_account_id" id="parent_account_id">
                                    <option value="">No Parent (Main Account)</option>
                                </select>
                                <small class="help-block">Optional: Select a parent account for hierarchy</small>
                            </div>

                            <div class="form-group">
                                <label for="description">Description</label>
                                <textarea class="form-control" 
                                          name="description" 
                                          id="description" 
                                          rows="3" 
                                          placeholder="Optional description for this account"></textarea>
                            </div>

                            <div class="form-group">
                                <label>
                                    <input type="checkbox" name="is_active" id="is_active" value="1" checked> 
                                    Active Account
                                </label>
                            </div>
                        </div>
                        <div class="box-footer">
                            <button type="submit" name="submit" class="btn btn-primary" id="submitAccount">
                                <i class="fa fa-save"></i> Add Account
                            </button>
                            <button type="button" class="btn btn-default" id="resetForm">
                                <i class="fa fa-refresh"></i> Reset
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Account Types Legend -->
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-info-circle"></i> Account Types
                        </h3>
                    </div>
                    <div class="box-body">
                        <div class="legend-item">
                            <span class="label label-success">Asset</span>
                            <small>Resources owned by the organization</small>
                        </div>
                        <div class="legend-item">
                            <span class="label label-warning">Liability</span>
                            <small>Debts owed to others</small>
                        </div>
                        <div class="legend-item">
                            <span class="label label-info">Equity</span>
                            <small>Owner's interest in the organization</small>
                        </div>
                        <div class="legend-item">
                            <span class="label label-primary">Revenue</span>
                            <small>Income from operations</small>
                        </div>
                        <div class="legend-item">
                            <span class="label label-danger">Expense</span>
                            <small>Costs of doing business</small>
                        </div>
                    </div>
                </div>

                <!-- Quick Stats -->
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-bar-chart"></i> Account Statistics
                        </h3>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <div class="col-xs-6">
                                <div class="description-block border-right">
                                    <span class="description-percentage text-green">
                                        <i class="fa fa-caret-up"></i> <span id="totalAccounts">0</span>
                                    </span>
                                    <h5 class="description-header">Total Accounts</h5>
                                    <span class="description-text">Active accounts</span>
                                </div>
                            </div>
                            <div class="col-xs-6">
                                <div class="description-block">
                                    <span class="description-percentage text-blue">
                                        <i class="fa fa-caret-up"></i> 5
                                    </span>
                                    <h5 class="description-header">Account Types</h5>
                                    <span class="description-text">Available types</span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Accounts List -->
            <div class="col-md-8">
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title">
                            <i class="fa fa-table"></i> Accounts List
                        </h3>
                        <div class="box-tools pull-right">
                            <div class="btn-group">
                                <button type="button" 
                                        class="btn btn-success btn-sm" 
                                        id="initializeAccounts"
                                        title="Create default account structure">
                                    <i class="fa fa-magic"></i> Initialize Default Accounts
                                </button>
                                <button type="button" 
                                        class="btn btn-default btn-sm" 
                                        id="exportAccounts"
                                        title="Export accounts to CSV">
                                    <i class="fa fa-download"></i> Export
                                </button>
                                <button type="button" 
                                        class="btn btn-info btn-sm" 
                                        id="refreshTable"
                                        title="Refresh table data">
                                    <i class="fa fa-refresh"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- Search and Filter -->
                        <div class="row" style="margin-bottom: 15px;">
                            <div class="col-md-4">
                                <div class="input-group">
                                    <input type="text" 
                                           class="form-control" 
                                           id="searchAccounts" 
                                           placeholder="Search accounts...">
                                    <span class="input-group-addon">
                                        <i class="fa fa-search"></i>
                                    </span>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <select class="form-control" id="filterAccountType">
                                    <option value="">All Account Types</option>
                                    <option value="asset">Asset</option>
                                    <option value="liability">Liability</option>
                                    <option value="equity">Equity</option>
                                    <option value="revenue">Revenue</option>
                                    <option value="expense">Expense</option>
                                </select>
                            </div>
                            <div class="col-md-4">
                                <select class="form-control" id="filterStatus">
                                    <option value="">All Status</option>
                                    <option value="1">Active Only</option>
                                    <option value="0">Inactive Only</option>
                                </select>
                            </div>
                        </div>

                        <!-- No Records State -->
                        <div class="alert alert-info text-center" id="noAccountsAlert">
                            <h4><i class="icon fa fa-info"></i> No Accounts Found</h4>
                            <p>No accounts found. Create your first account using the form on the left or initialize default accounts.</p>
                        </div>

                        <!-- Accounts Table -->
                        <div class="table-responsive" id="accountsTableContainer" style="display: none;">
                            <table class="table table-bordered table-striped" id="accountsTable">
                                <thead>
                                    <tr>
                                        <th>Code</th>
                                        <th>Account Name</th>
                                        <th>Type</th>
                                        <th>Parent</th>
                                        <th>Status</th>
                                        <th class="text-right">Balance</th>
                                        <th class="text-center">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="accountsTableBody">
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Edit Account Modal -->
<div class="modal fade" id="editAccountModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">Edit Account</h4>
            </div>
            <form id="editAccountForm">
                <div class="modal-body">
                    <input type="hidden" id="edit_account_id" name="account_id">
                    <div class="form-group">
                        <label for="edit_account_code">
                            Account Code <span class="text-red">*</span>
                        </label>
                        <input type="text" 
                               class="form-control" 
                               name="account_code" 
                               id="edit_account_code" 
                               required 
                               maxlength="10">
                    </div>
                    <div class="form-group">
                        <label for="edit_account_name">
                            Account Name <span class="text-red">*</span>
                        </label>
                        <input type="text" 
                               class="form-control" 
                               name="account_name" 
                               id="edit_account_name" 
                               required 
                               maxlength="100">
                    </div>
                    <div class="form-group">
                        <label for="edit_account_type">
                            Account Type <span class="text-red">*</span>
                        </label>
                        <select class="form-control" name="account_type" id="edit_account_type" required>
                            <option value="asset">Asset</option>
                            <option value="liability">Liability</option>
                            <option value="equity">Equity</option>
                            <option value="revenue">Revenue</option>
                            <option value="expense">Expense</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="edit_parent_account_id">Parent Account</label>
                        <select class="form-control" name="parent_account_id" id="edit_parent_account_id">
                            <option value="">No Parent (Main Account)</option>
                        </select>
                    </div>
                    <div class="form-group">
                        <label for="edit_description">Description</label>
                        <textarea class="form-control" 
                                  name="description" 
                                  id="edit_description" 
                                  rows="3"></textarea>
                    </div>
                    <div class="form-group">
                        <label>
                            <input type="checkbox" name="is_active" id="edit_is_active" value="1"> 
                            Active
                        </label>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">
                        Cancel
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fa fa-save"></i> Update Account
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- Delete Confirmation Modal -->
<div class="modal fade" id="deleteModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-sm" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">Confirm Delete</h4>
            </div>
            <div class="modal-body">
                <p>Are you sure you want to delete this account?</p>
                <p><strong>Note:</strong> Accounts with transactions will be deactivated instead of deleted.</p>
                <div id="deleteAccountInfo" class="alert alert-warning">
                    <strong>Account Details:</strong>
                    <span id="deleteAccountDetails"></span>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    Cancel
                </button>
                <button type="button" class="btn btn-danger" id="confirmDelete">
                    <i class="fa fa-trash"></i> Delete
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Account Details Modal -->
<div class="modal fade" id="accountDetailsModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">Account Details</h4>
            </div>
            <div class="modal-body" id="accountDetailsContent">
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">
                    Close
                </button>
                <button type="button" class="btn btn-primary" id="editFromDetails">
                    <i class="fa fa-edit"></i> Edit Account
                </button>
            </div>
        </div>
    </div>
</div>

<style>
.content-wrapper {
    min-height: 100vh !important;
    overflow-y: auto !important;
}
.legend-item {
    margin-bottom: 8px;
    padding: 5px 0;
    border-bottom: 1px solid #f0f0f0;
}
.legend-item:last-child {
    border-bottom: none;
}
.legend-item .label {
    margin-right: 10px;
    min-width: 60px;
    display: inline-block;
    text-align: center;
}
.description-block {
    text-align: center;
    padding: 10px 0;
}
.description-header {
    margin: 0;
    font-size: 18px;
    font-weight: bold;
}
.description-text {
    font-size: 12px;
    text-transform: uppercase;
    color: #999;
}
.description-percentage {
    font-size: 14px;
    font-weight: 600;
}
.border-right {
    border-right: 1px solid #dee2e6;
}
.text-red {
    color: #dd4b39;
}
.help-block {
    font-size: 12px;
    color: #777;
    margin-top: 5px;
}
.currency-amount {
    font-family: 'Courier New', monospace;
    font-weight: bold;
}
.form-group.has-error .form-control {
    border-color: #dd4b39;
    box-shadow: inset 0 1px 1px rgba(0, 0, 0, 0.075), 0 0 6px #ce8483;
}
.form-group.has-error .help-block {
    color: #dd4b39;
}
@media print {
    .box-tools, .btn, .breadcrumb, .modal {
        display: none !important;
    }
    .content-wrapper {
        margin: 0 !important;
        padding: 0 !important;
    }
    .box {
        box-shadow: none !important;
        border: 1px solid #ddd !important;
    }
    .table {
        font-size: 12px;
    }
    .col-md-4 {
        display: none !important;
    }
    .col-md-8 {
        width: 100% !important;
    }
}
@media (max-width: 768px) {
    .col-md-4, .col-md-8 {
        width: 100% !important;
        margin-bottom: 15px;
    }
    .btn-group .btn {
        margin-bottom: 5px;
        display: block;
        width: 100%;
    }
}
</style>

<script>
$(document).ready(function() {
    // Initialize variables for data storage
    var accounts = JSON.parse(localStorage.getItem('chartOfAccounts') || '[]');
    var accountIdCounter = parseInt(localStorage.getItem('accountIdCounter') || '0');
    var currentEditingId = null;
    var deleteAccountId = null;

    // System currency configuration
    var systemCurrency = {
        symbol: 'KSh',
        position: 'before',
        separator: ',',
        decimal: '.',
        precision: 2
    };

    // Default accounts structure
    var defaultAccounts = [
        {id: 1, account_code: '1000', account_name: 'Cash', account_type: 'asset', parent_account_id: null, description: 'Cash in hand', is_active: 1, current_balance: 0},
        {id: 2, account_code: '1100', account_name: 'Bank Account', account_type: 'asset', parent_account_id: null, description: 'Bank checking account', is_active: 1, current_balance: 0},
        {id: 3, account_code: '1200', account_name: 'Accounts Receivable', account_type: 'asset', parent_account_id: null, description: 'Money owed by customers', is_active: 1, current_balance: 0},
        {id: 4, account_code: '1300', account_name: 'Inventory', account_type: 'asset', parent_account_id: null, description: 'Goods for sale', is_active: 1, current_balance: 0},
        {id: 5, account_code: '1500', account_name: 'Equipment', account_type: 'asset', parent_account_id: null, description: 'Office equipment and furniture', is_active: 1, current_balance: 0},
        {id: 6, account_code: '2000', account_name: 'Accounts Payable', account_type: 'liability', parent_account_id: null, description: 'Money owed to suppliers', is_active: 1, current_balance: 0},
        {id: 7, account_code: '2100', account_name: 'Notes Payable', account_type: 'liability', parent_account_id: null, description: 'Loans and notes payable', is_active: 1, current_balance: 0},
        {id: 8, account_code: '2200', account_name: 'Accrued Expenses', account_type: 'liability', parent_account_id: null, description: 'Expenses incurred but not yet paid', is_active: 1, current_balance: 0},
        {id: 9, account_code: '3000', account_name: 'Owner Equity', account_type: 'equity', parent_account_id: null, description: 'Owner\'s investment in business', is_active: 1, current_balance: 0},
        {id: 10, account_code: '3100', account_name: 'Retained Earnings', account_type: 'equity', parent_account_id: null, description: 'Accumulated earnings', is_active: 1, current_balance: 0},
        {id: 11, account_code: '4000', account_name: 'Sales Revenue', account_type: 'revenue', parent_account_id: null, description: 'Revenue from sales', is_active: 1, current_balance: 0},
        {id: 12, account_code: '4100', account_name: 'Service Revenue', account_type: 'revenue', parent_account_id: null, description: 'Revenue from services', is_active: 1, current_balance: 0},
        {id: 13, account_code: '5000', account_name: 'Cost of Goods Sold', account_type: 'expense', parent_account_id: null, description: 'Direct costs of goods sold', is_active: 1, current_balance: 0},
        {id: 14, account_code: '6000', account_name: 'Rent Expense', account_type: 'expense', parent_account_id: null, description: 'Office rent expenses', is_active: 1, current_balance: 0},
        {id: 15, account_code: '6100', account_name: 'Utilities Expense', account_type: 'expense', parent_account_id: null, description: 'Electricity, water, internet', is_active: 1, current_balance: 0},
        {id: 16, account_code: '6200', account_name: 'Office Supplies Expense', account_type: 'expense', parent_account_id: null, description: 'Office supplies and materials', is_active: 1, current_balance: 0},
        {id: 17, account_code: '6300', account_name: 'Depreciation Expense', account_type: 'expense', parent_account_id: null, description: 'Depreciation of assets', is_active: 1, current_balance: 0}
    ];

    // Load initial data
    loadAccounts();
    populateParentAccounts();
    updateStatistics();

    // Fix scrolling issues
    setTimeout(function() {
        $(window).trigger('resize');
    }, 100);

    // Function to save data to localStorage
    function saveToStorage() {
        localStorage.setItem('chartOfAccounts', JSON.stringify(accounts));
        localStorage.setItem('accountIdCounter', accountIdCounter.toString());
    }

    // Function to format currency
    function formatCurrency(amount) {
        var value = parseFloat(amount || 0);
        var formattedAmount = value.toLocaleString('en-US', {
            minimumFractionDigits: systemCurrency.precision,
            maximumFractionDigits: systemCurrency.precision
        });

        // Replace separators if needed
        if (systemCurrency.separator !== ',' || systemCurrency.decimal !== '.') {
            formattedAmount = formattedAmount.replace(/,/g, 'TEMP_SEPARATOR');
            formattedAmount = formattedAmount.replace(/\./g, systemCurrency.decimal);
            formattedAmount = formattedAmount.replace(/TEMP_SEPARATOR/g, systemCurrency.separator);
        }

        return systemCurrency.position === 'before' 
            ? systemCurrency.symbol + formattedAmount 
            : formattedAmount + ' ' + systemCurrency.symbol;
    }

    // Function to get account type label class
    function getAccountTypeClass(type) {
        switch(type) {
            case 'asset': return 'label-success';
            case 'liability': return 'label-warning';
            case 'equity': return 'label-info';
            case 'revenue': return 'label-primary';
            case 'expense': return 'label-danger';
            default: return 'label-default';
        }
    }

    // Function to validate form data
    function validateAccountForm(formData, isEdit = false) {
        var errors = [];
        var formPrefix = isEdit ? '#edit_' : '#';

        // Clear previous validation errors
        $(`${formPrefix}account_code`).closest('.form-group').removeClass('has-error').find('.help-block.error').remove();
        $(`${formPrefix}account_name`).closest('.form-group').removeClass('has-error');
        $(`${formPrefix}account_type`).closest('.form-group').removeClass('has-error');
        $(`${formPrefix}parent_account_id`).closest('.form-group').removeClass('has-error');

        // Validate account code
        if (!formData.account_code || formData.account_code.trim() === '') {
            errors.push('Account code is required');
            $(`${formPrefix}account_code`).closest('.form-group').addClass('has-error');
        } else {
            // Check for duplicate account code
            var duplicateAccount = accounts.find(account => 
                account.account_code.toLowerCase() === formData.account_code.toLowerCase() &&
                (!isEdit || account.id != currentEditingId)
            );
            if (duplicateAccount) {
                errors.push('Account code already exists');
                $(`${formPrefix}account_code`).closest('.form-group').addClass('has-error');
                $(`${formPrefix}account_code`).after('<span class="help-block error">Account code already exists</span>');
            }
        }

        // Validate account name
        if (!formData.account_name || formData.account_name.trim() === '') {
            errors.push('Account name is required');
            $(`${formPrefix}account_name`).closest('.form-group').addClass('has-error');
        }

        // Validate account type
        if (!formData.account_type) {
            errors.push('Account type is required');
            $(`${formPrefix}account_type`).closest('.form-group').addClass('has-error');
        }

        // Validate parent account
        if (isEdit && formData.parent_account_id == currentEditingId) {
            errors.push('Account cannot be its own parent');
            $(`${formPrefix}parent_account_id`).closest('.form-group').addClass('has-error');
        }

        // Check for circular references
        if (formData.parent_account_id && isEdit) {
            if (hasCircularReference(currentEditingId, formData.parent_account_id)) {
                errors.push('Parent account would create a circular reference');
                $(`${formPrefix}parent_account_id`).closest('.form-group').addClass('has-error');
            }
        }

        return {
            isValid: errors.length === 0,
            errors: errors
        };
    }

    // Function to check for circular references
    function hasCircularReference(accountId, parentId) {
        if (!parentId) return false;
        let currentId = parentId;
        while (currentId) {
            if (currentId == accountId) return true;
            const account = getAccountById(currentId);
            currentId = account ? account.parent_account_id : null;
        }
        return false;
    }

    // Function to initialize DataTable
    function initializeDataTable() {
        if ($.fn.DataTable.isDataTable('#accountsTable')) {
            $('#accountsTable').DataTable().destroy();
        }

        var filteredAccounts = getFilteredAccounts();
        if (filteredAccounts.length === 0) {
            $('#noAccountsAlert').show();
            $('#accountsTableContainer').hide();
        } else {
            $('#noAccountsAlert').hide();
            $('#accountsTableContainer').show();
            populateAccountsTable(filteredAccounts);
            $('#accountsTable').DataTable({
                "order": [[0, "asc"]],
                "pageLength": 25,
                "responsive": true,
                "columnDefs": [
                    { "orderable": false, "targets": [6] }
                ],
                "searching": false // Managed by custom search
            });
        }
    }

    // Function to load and display accounts
    function loadAccounts() {
        initializeDataTable();
    }

    // Function to get filtered accounts
    function getFilteredAccounts() {
        var searchTerm = $('#searchAccounts').val().toLowerCase();
        var typeFilter = $('#filterAccountType').val();
        var statusFilter = $('#filterStatus').val();

        return accounts.filter(account => {
            var matchesSearch = !searchTerm || 
                account.account_code.toLowerCase().includes(searchTerm) ||
                account.account_name.toLowerCase().includes(searchTerm);
            var matchesType = !typeFilter || account.account_type === typeFilter;
            var matchesStatus = statusFilter === '' || account.is_active == statusFilter;
            return matchesSearch && matchesType && matchesStatus;
        });
    }

    // Function to populate accounts table
    function populateAccountsTable(accountsToShow) {
        var tbody = $('#accountsTableBody');
        tbody.empty();

        accountsToShow.forEach(function(account) {
            var parentName = account.parent_account_id 
                ? getAccountById(account.parent_account_id)?.account_name || '-' 
                : '-';
            var statusLabel = account.is_active == 1 
                ? '<span class="label label-success"><i class="fa fa-check"></i> Active</span>' 
                : '<span class="label label-default"><i class="fa fa-times"></i> Inactive</span>';
            var actionsHtml = '<div class="btn-group">' +
                `<button type="button" class="btn btn-xs btn-info view-account" data-id="${account.id}" title="View Details"><i class="fa fa-eye"></i></button>` +
                `<button type="button" class="btn btn-xs btn-warning edit-account" data-id="${account.id}" title="Edit Account"><i class="fa fa-edit"></i></button>` +
                (account.is_active == 1 
                    ? `<button type="button" class="btn btn-xs btn-default toggle-status" data-id="${account.id}" data-status="0" title="Deactivate"><i class="fa fa-pause"></i></button>`
                    : `<button type="button" class="btn btn-xs btn-success toggle-status" data-id="${account.id}" data-status="1" title="Activate"><i class="fa fa-play"></i></button>`) +
                `<button type="button" class="btn btn-xs btn-danger delete-account" data-id="${account.id}" title="Delete Account"><i class="fa fa-trash"></i></button>` +
                '</div>';

            var row = `<tr>
                <td><strong>${account.account_code}</strong></td>
                <td>${account.parent_account_id ? '<i class="fa fa-level-up fa-rotate-90 text-muted"></i> ' : ''}${account.account_name}</td>
                <td><span class="label ${getAccountTypeClass(account.account_type)}">${account.account_type.charAt(0).toUpperCase() + account.account_type.slice(1)}</span></td>
                <td><span class="text-muted">${parentName}</span></td>
                <td>${statusLabel}</td>
                <td class="text-right"><span class="currency-amount">${formatCurrency(account.current_balance)}</span></td>
                <td class="text-center">${actionsHtml}</td>
            </tr>`;
            tbody.append(row);
        });
    }

    // Function to get account by ID
    function getAccountById(id) {
        return accounts.find(account => account.id == id);
    }

    // Function to populate parent accounts dropdown
    function populateParentAccounts() {
        var parentOptions = '<option value="">No Parent (Main Account)</option>';
        accounts.filter(account => account.is_active == 1 && account.id != currentEditingId).forEach(function(account) {
            parentOptions += `<option value="${account.id}">${account.account_code} - ${account.account_name}</option>`;
        });
        $('#parent_account_id, #edit_parent_account_id').html(parentOptions);
    }

    // Function to update statistics
    function updateStatistics() {
        var activeAccounts = accounts.filter(account => account.is_active == 1).length;
        $('#totalAccounts').text(activeAccounts);
    }

    // Function to show alert messages
    function showAlert(type, message) {
        var alertHtml = `<div class="alert alert-${type} alert-dismissible">
            <button type="button" class="close" data-dismiss="alert">&times;</button>
            <i class="fa fa-${getAlertIcon(type)}"></i> ${message}
        </div>`;
        $('#flashMessages').html(alertHtml);
        setTimeout(function() {
            $('.alert').fadeOut();
        }, 5000);
    }

    function getAlertIcon(type) {
        switch(type) {
            case 'success': return 'check-circle';
            case 'danger': return 'times-circle';
            case 'warning': return 'exclamation-triangle';
            case 'info': return 'info-circle';
            default: return 'info-circle';
        }
    }

    // Form submission for adding accounts
    $('#accountForm').on('submit', function(e) {
        e.preventDefault();
        var formData = {
            account_code: $('#account_code').val().trim().toUpperCase(),
            account_name: $('#account_name').val().trim(),
            account_type: $('#account_type').val(),
            parent_account_id: $('#parent_account_id').val() || null,
            description: $('#description').val().trim(),
            is_active: $('#is_active').is(':checked') ? 1 : 0
        };

        var validation = validateAccountForm(formData);
        if (!validation.isValid) {
            showAlert('danger', 'Please fix the following errors:<br>' + validation.errors.join('<br>'));
            return;
        }

        $('#submitAccount').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Adding...');
        setTimeout(function() {
            try {
                var newAccount = {
                    id: ++accountIdCounter,
                    account_code: formData.account_code,
                    account_name: formData.account_name,
                    account_type: formData.account_type,
                    parent_account_id: formData.parent_account_id,
                    description: formData.description,
                    is_active: formData.is_active,
                    current_balance: 0,
                    created_at: new Date().toISOString()
                };
                accounts.push(newAccount);
                saveToStorage();
                showAlert('success', 'Account added successfully!');
                $('#accountForm')[0].reset();
                $('#is_active').prop('checked', true);
                loadAccounts();
                populateParentAccounts();
                updateStatistics();
            } catch (error) {
                showAlert('danger', 'An error occurred while adding the account: ' + error.message);
                console.error('Error adding account:', error);
            }
            $('#submitAccount').prop('disabled', false).html('<i class="fa fa-save"></i> Add Account');
        }, 500);
    });

    // Form submission for editing accounts
    $('#editAccountForm').on('submit', function(e) {
        e.preventDefault();
        var formData = {
            account_code: $('#edit_account_code').val().trim().toUpperCase(),
            account_name: $('#edit_account_name').val().trim(),
            account_type: $('#edit_account_type').val(),
            parent_account_id: $('#edit_parent_account_id').val() || null,
            description: $('#edit_description').val().trim(),
            is_active: $('#edit_is_active').is(':checked') ? 1 : 0
        };

        var validation = validateAccountForm(formData, true);
        if (!validation.isValid) {
            showAlert('danger', 'Please fix the following errors:<br>' + validation.errors.join('<br>'));
            return;
        }

        var accountIndex = accounts.findIndex(account => account.id === currentEditingId);
        if (accountIndex !== -1) {
            accounts[accountIndex] = {
                ...accounts[accountIndex],
                account_code: formData.account_code,
                account_name: formData.account_name,
                account_type: formData.account_type,
                parent_account_id: formData.parent_account_id,
                description: formData.description,
                is_active: formData.is_active,
                updated_at: new Date().toISOString()
            };
            saveToStorage();
            showAlert('success', 'Account updated successfully!');
            $('#editAccountModal').modal('hide');
            loadAccounts();
            populateParentAccounts();
            updateStatistics();
        }
    });

    // Search and filter functionality
    $('#searchAccounts, #filterAccountType, #filterStatus').on('input change', function() {
        loadAccounts();
    });

    // Reset form
    $('#resetForm').click(function() {
        $('#accountForm')[0].reset();
        $('#is_active').prop('checked', true);
        $('.form-group').removeClass('has-error').find('.help-block.error').remove();
    });

    // Refresh table
    $('#refreshTable').click(function() {
        loadAccounts();
        showAlert('info', 'Table refreshed');
    });

    // Initialize default accounts
    $('#initializeAccounts').click(function() {
        if (confirm('This will create a standard chart of accounts. Existing accounts will not be affected. Continue?')) {
            var addedCount = 0;
            defaultAccounts.forEach(function(defaultAccount) {
                if (!accounts.find(account => account.account_code === defaultAccount.account_code)) {
                    var newAccount = {
                        ...defaultAccount,
                        id: ++accountIdCounter,
                        created_at: new Date().toISOString()
                    };
                    accounts.push(newAccount);
                    addedCount++;
                }
            });
            if (addedCount > 0) {
                saveToStorage();
                showAlert('success', `${addedCount} default accounts initialized successfully!`);
                loadAccounts();
                populateParentAccounts();
                updateStatistics();
            } else {
                showAlert('info', 'All default accounts already exist.');
            }
        }
    });

    // Export accounts
    $('#exportAccounts').click(function() {
        if (accounts.length === 0) {
            showAlert('warning', 'No accounts to export.');
            return;
        }
        var csv = 'Account Code,Account Name,Account Type,Parent Code,Description,Status,Balance\n';
        accounts.forEach(function(account) {
            var parentCode = account.parent_account_id ? getAccountById(account.parent_account_id)?.account_code || '' : '';
            csv += `"${account.account_code}","${account.account_name.replace(/"/g, '""')}",` +
                   `"${account.account_type}","${parentCode}",` +
                   `"${(account.description || '').replace(/"/g, '""')}","${account.is_active ? 'Active' : 'Inactive'}",` +
                   `"${account.current_balance}"\n`;
        });
        var blob = new Blob([csv], { type: 'text/csv' });
        var url = window.URL.createObjectURL(blob);
        var a = document.createElement('a');
        a.href = url;
        a.download = `chart_of_accounts_${new Date().toISOString().slice(0, 10)}.csv`;
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
        showAlert('success', 'Chart of accounts exported successfully!');
    });

    // View account details
    $(document).on('click', '.view-account', function() {
        var accountId = parseInt($(this).data('id'));
        var account = getAccountById(accountId);
        if (account) {
            var parentAccount = account.parent_account_id ? getAccountById(account.parent_account_id) : null;
            var detailsHtml = `<div class="account-details">
                <h4>Account Information</h4>
                <table class="table table-bordered">
                    <tr><th style="width:30%">Account Code:</th><td>${account.account_code}</td></tr>
                    <tr><th>Account Name:</th><td>${account.account_name}</td></tr>
                    <tr><th>Account Type:</th><td><span class="label ${getAccountTypeClass(account.account_type)}">${account.account_type.charAt(0).toUpperCase() + account.account_type.slice(1)}</span></td></tr>
                    <tr><th>Parent Account:</th><td>${parentAccount ? parentAccount.account_code + ' - ' + parentAccount.account_name : 'None'}</td></tr>
                    <tr><th>Description:</th><td>${account.description || 'No description'}</td></tr>
                    <tr><th>Status:</th><td>${account.is_active ? '<span class="label label-success">Active</span>' : '<span class="label label-default">Inactive</span>'}</td></tr>
                    <tr><th>Current Balance:</th><td class="text-right"><strong>${formatCurrency(account.current_balance)}</strong></td></tr>
                    <tr><th>Created:</th><td>${new Date(account.created_at).toLocaleDateString()}</td></tr>
                    ${account.updated_at ? `<tr><th>Last Updated:</th><td>${new Date(account.updated_at).toLocaleDateString()}</td></tr>` : ''}
                </table>
            </div>`;
            $('#accountDetailsContent').html(detailsHtml);
            $('#accountDetailsModal').modal('show');
            $('#editFromDetails').data('id', accountId);
        }
    });

    // Edit account
    $(document).on('click', '.edit-account', function() {
        var accountId = parseInt($(this).data('id'));
        var account = getAccountById(accountId);
        if (account) {
            currentEditingId = accountId;
            $('#edit_account_id').val(account.id);
            $('#edit_account_code').val(account.account_code);
            $('#edit_account_name').val(account.account_name);
            $('#edit_account_type').val(account.account_type);
            $('#edit_parent_account_id').val(account.parent_account_id || '');
            $('#edit_description').val(account.description || '');
            $('#edit_is_active').prop('checked', account.is_active == 1);
            $('.form-group').removeClass('has-error').find('.help-block.error').remove();
            $('#editAccountModal').modal('show');
        }
    });

    // Edit from details modal
    $('#editFromDetails').click(function() {
        var accountId = $(this).data('id');
        $('#accountDetailsModal').modal('hide');
        $('.edit-account[data-id="' + accountId + '"]').click();
    });

    // Toggle account status
    $(document).on('click', '.toggle-status', function() {
        var accountId = parseInt($(this).data('id'));
        var newStatus = parseInt($(this).data('status'));
        var statusText = newStatus == 1 ? 'activate' : 'deactivate';
        if (confirm(`Are you sure you want to ${statusText} this account?`)) {
            var accountIndex = accounts.findIndex(account => account.id === accountId);
            if (accountIndex !== -1) {
                accounts[accountIndex].is_active = newStatus;
                accounts[accountIndex].updated_at = new Date().toISOString();
                saveToStorage();
                showAlert('success', 'Account status updated successfully!');
                loadAccounts();
                populateParentAccounts();
                updateStatistics();
            }
        }
    });

    // Delete account
    $(document).on('click', '.delete-account', function() {
        deleteAccountId = parseInt($(this).data('id'));
        var account = getAccountById(deleteAccountId);
        if (account) {
            // Check if account is used in journal entries
            var journalEntries = JSON.parse(localStorage.getItem('journalEntries') || '[]');
            var isUsed = journalEntries.some(entry => 
                entry.lines.some(line => line.account_id == deleteAccountId)
            );
            if (isUsed) {
                showAlert('warning', 'Cannot delete account with associated transactions. Please deactivate instead.');
                return;
            }
            $('#deleteAccountDetails').text(`${account.account_code} - ${account.account_name}`);
            $('#deleteModal').modal('show');
        }
    });

    // Confirm delete
    $('#confirmDelete').click(function() {
        if (deleteAccountId) {
            var accountIndex = accounts.findIndex(account => account.id === deleteAccountId);
            if (accountIndex !== -1) {
                accounts.splice(accountIndex, 1);
                saveToStorage();
                showAlert('success', 'Account deleted successfully!');
                $('#deleteModal').modal('hide');
                loadAccounts();
                populateParentAccounts();
                updateStatistics();
            }
        }
    });

    // Account code auto-generation
    $('#account_type, #edit_account_type').on('change', function() {
        var accountType = $(this).val();
        var targetInput = $(this).attr('id') === 'account_type' ? '#account_code' : '#edit_account_code';
        if (accountType && !$(targetInput).val()) {
            generateAccountCode(accountType, targetInput);
        }
    });

    // Function to generate account code
    function generateAccountCode(accountType, targetInput) {
        var baseCode = {
            'asset': '1',
            'liability': '2',
            'equity': '3',
            'revenue': '4',
            'expense': '5'
        }[accountType] || '9';
        var existingCodes = accounts
            .filter(account => account.account_code.startsWith(baseCode))
            .map(account => parseInt(account.account_code))
            .filter(code => !isNaN(code))
            .sort((a, b) => a - b);
        var nextCode = parseInt(baseCode + '000');
        for (var code of existingCodes) {
            if (code >= nextCode) nextCode = code + 100;
        }
        $(targetInput).val(nextCode.toString());
    }

    // Account code validation
    $('#account_code, #edit_account_code').on('input', function() {
        $(this).val($(this).val().toUpperCase());
    });

    // Keyboard shortcuts
    $(document).keydown(function(e) {
        if (e.ctrlKey && e.keyCode === 78) { // Ctrl+N
            e.preventDefault();
            $('#account_code').focus();
        }
        if (e.ctrlKey && e.keyCode === 70) { // Ctrl+F
            e.preventDefault();
            $('#searchAccounts').focus();
        }
        if (e.ctrlKey && e.keyCode === 82) { // Ctrl+R
            e.preventDefault();
            loadAccounts();
        }
    });

    // Global function to export accounts for journal entries
    window.getAccountsForJournal = function() {
        return accounts.filter(account => account.is_active == 1);
    };

    // Global function to update account balance
    window.updateAccountBalance = function(accountId, amount, isDebit) {
        var accountIndex = accounts.findIndex(account => account.id == accountId);
        if (accountIndex !== -1) {
            var account = accounts[accountIndex];
            if (isDebit) {
                account.current_balance += (account.account_type === 'asset' || account.account_type === 'expense') ? amount : -amount;
            } else {
                account.current_balance += (account.account_type === 'liability' || account.account_type === 'equity' || account.account_type === 'revenue') ? amount : -amount;
            }
            account.updated_at = new Date().toISOString();
            saveToStorage();
            loadAccounts();
            updateStatistics();
        }
    };

    // Global function to get system currency settings
    window.getSystemCurrency = function() {
        return systemCurrency;
    };

    // Global function to update system currency
    window.setSystemCurrency = function(newCurrency) {
        systemCurrency = { ...systemCurrency, ...newCurrency };
        loadAccounts();
    };

    // Initialize if no accounts exist
    if (accounts.length === 0) {
        showAlert('info', 'No accounts found. Click "Initialize Default Accounts" to create a standard chart of accounts.');
    }
});
</script>
</body>
</html>