<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Loyalty_points_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get patient's loyalty points balance
     */
    public function getPatientLoyaltyBalance($patient_id) {
        $this->db->select('
            COALESCE(SUM(CASE WHEN transaction_type = "earned" THEN points ELSE 0 END), 0) as total_earned,
            COALESCE(SUM(CASE WHEN transaction_type = "redeemed" THEN ABS(points) ELSE 0 END), 0) as total_redeemed,
            COALESCE(SUM(CASE WHEN transaction_type = "expired" THEN ABS(points) ELSE 0 END), 0) as total_expired,
            COALESCE(SUM(CASE 
                WHEN transaction_type = "earned" THEN points 
                WHEN transaction_type IN ("redeemed", "expired") THEN -ABS(points)
                WHEN transaction_type = "adjusted" THEN points
                ELSE 0 
            END), 0) as current_balance,
            COALESCE(SUM(CASE WHEN transaction_type = "earned" THEN amount_spent ELSE 0 END), 0) as total_spent
        ');
        $this->db->from('patient_loyalty_points');
        $this->db->where('patient_id', $patient_id);
        $this->db->where('is_active', 1);

        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get patient's loyalty history
     */
    public function getPatientLoyaltyHistory($patient_id, $limit = 20) {
        $this->db->select('
            plp.*, 
            s.name as staff_name,
            s.surname as staff_surname
        ');
        $this->db->from('patient_loyalty_points plp');
        $this->db->join('staff s', 's.id = plp.staff_id', 'left');
        $this->db->where('plp.patient_id', $patient_id);
        $this->db->where('plp.is_active', 1);
        $this->db->order_by('plp.created_at', 'DESC');
        $this->db->limit($limit);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Award loyalty points
     */
    public function awardPoints($data) {
        $this->db->insert('patient_loyalty_points', $data);
        return $this->db->insert_id();
    }

    /**
     * Redeem loyalty points
     */
    public function redeemPoints($patient_id, $points, $service, $value, $notes = '', $staff_id = null) {
        // Check if patient has enough points
        $current_balance = $this->getPatientCurrentBalance($patient_id);
        
        if ($current_balance < $points) {
            return ['success' => false, 'message' => 'Insufficient points balance'];
        }

        // Get minimum redemption requirement
        $min_redemption = $this->getSettingValue('minimum_redemption_points', 10);
        if ($points < $min_redemption) {
            return ['success' => false, 'message' => 'Minimum redemption is ' . $min_redemption . ' points'];
        }

        // Create redemption record
        $redemption_data = [
            'patient_id' => $patient_id,
            'transaction_type' => 'redeemed',
            'points' => -abs($points),
            'description' => 'Points redeemed for: ' . $service,
            'redeemed_service' => $service,
            'redeemed_value' => $value,
            'transaction_date' => date('Y-m-d'),
            'staff_id' => $staff_id,
            'notes' => $notes,
            'created_at' => date('Y-m-d H:i:s')
        ];

        $this->db->insert('patient_loyalty_points', $redemption_data);
        
        if ($this->db->affected_rows() > 0) {
            return ['success' => true, 'message' => 'Points redeemed successfully'];
        } else {
            return ['success' => false, 'message' => 'Failed to redeem points'];
        }
    }

    /**
     * Get patient's current balance (helper function)
     */
    public function getPatientCurrentBalance($patient_id) {
        $balance_data = $this->getPatientLoyaltyBalance($patient_id);
        return $balance_data['current_balance'];
    }

    /**
     * Get all loyalty activities with pagination
     */
    public function getAllLoyaltyActivities($limit = 10, $offset = 0, $filters = []) {
        $this->db->select('
            plp.*, 
            p.patient_name, 
            p.id as patient_id,
            s.name as staff_name,
            s.surname as staff_surname
        ');
        $this->db->from('patient_loyalty_points plp');
        $this->db->join('patients p', 'p.id = plp.patient_id', 'left');
        $this->db->join('staff s', 's.id = plp.staff_id', 'left');
        $this->db->where('plp.is_active', 1);

        // Apply filters
        if (!empty($filters['transaction_type'])) {
            $this->db->where('plp.transaction_type', $filters['transaction_type']);
        }
        if (!empty($filters['date_from'])) {
            $this->db->where('plp.transaction_date >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->db->where('plp.transaction_date <=', $filters['date_to']);
        }
        if (!empty($filters['patient_name'])) {
            $this->db->like('p.patient_name', $filters['patient_name']);
        }

        $this->db->order_by('plp.created_at', 'DESC');
        $this->db->limit($limit, $offset);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get total count of loyalty activities
     */
    public function getTotalLoyaltyActivitiesCount($filters = []) {
        $this->db->select('COUNT(*) as total');
        $this->db->from('patient_loyalty_points plp');
        $this->db->join('patients p', 'p.id = plp.patient_id', 'left');
        $this->db->where('plp.is_active', 1);

        // Apply filters
        if (!empty($filters['transaction_type'])) {
            $this->db->where('plp.transaction_type', $filters['transaction_type']);
        }
        if (!empty($filters['date_from'])) {
            $this->db->where('plp.transaction_date >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->db->where('plp.transaction_date <=', $filters['date_to']);
        }
        if (!empty($filters['patient_name'])) {
            $this->db->like('p.patient_name', $filters['patient_name']);
        }

        $query = $this->db->get();
        return $query->row()->total;
    }

    /**
     * Get loyalty activities for DataTables
     */
    public function getLoyaltyActivitiesForDataTables($request) {
        $columns = ['plp.transaction_date', 'p.patient_name', 'plp.transaction_type', 'plp.points', 'plp.description', 'staff_name', 'plp.reference_type'];
        
        $this->db->select('
            plp.*, 
            p.patient_name, 
            p.id as patient_id,
            CONCAT(s.name, " ", s.surname) as staff_name
        ');
        $this->db->from('patient_loyalty_points plp');
        $this->db->join('patients p', 'p.id = plp.patient_id', 'left');
        $this->db->join('staff s', 's.id = plp.staff_id', 'left');
        $this->db->where('plp.is_active', 1);

        // Apply filters from request
        if (!empty($request['transaction_type'])) {
            $this->db->where('plp.transaction_type', $request['transaction_type']);
        }
        if (!empty($request['date_from'])) {
            $this->db->where('plp.transaction_date >=', $request['date_from']);
        }
        if (!empty($request['date_to'])) {
            $this->db->where('plp.transaction_date <=', $request['date_to']);
        }
        if (!empty($request['patient_name'])) {
            $this->db->like('p.patient_name', $request['patient_name']);
        }

        // Search
        if (!empty($request['search']['value'])) {
            $search_value = $request['search']['value'];
            $this->db->group_start();
            $this->db->like('p.patient_name', $search_value);
            $this->db->or_like('plp.description', $search_value);
            $this->db->or_like('plp.redeemed_service', $search_value);
            $this->db->group_end();
        }

        // Count total records
        $total_records = $this->db->count_all_results('', FALSE);

        // Order
        if (isset($request['order'][0]['column'])) {
            $column_index = $request['order'][0]['column'];
            $order_dir = $request['order'][0]['dir'];
            if (isset($columns[$column_index])) {
                $this->db->order_by($columns[$column_index], $order_dir);
            }
        } else {
            $this->db->order_by('plp.created_at', 'DESC');
        }

        // Limit
        if ($request['length'] != -1) {
            $this->db->limit($request['length'], $request['start']);
        }

        $query = $this->db->get();
        $data = $query->result_array();

        return [
            'draw' => intval($request['draw']),
            'recordsTotal' => $total_records,
            'recordsFiltered' => $total_records,
            'data' => $data
        ];
    }

    /**
     * Get dashboard statistics
     */
    public function getDashboardStats() {
        $stats = [];

        // Active members (patients with points)
        $this->db->select('COUNT(DISTINCT patient_id) as active_members');
        $this->db->from('patient_loyalty_points');
        $this->db->where('is_active', 1);
        $this->db->where('points >', 0);
        $query = $this->db->get();
        $stats['active_members'] = $query->row()->active_members;

        // Points earned today
        $this->db->select('COALESCE(SUM(points), 0) as points_earned_today');
        $this->db->from('patient_loyalty_points');
        $this->db->where('transaction_type', 'earned');
        $this->db->where('transaction_date', date('Y-m-d'));
        $this->db->where('is_active', 1);
        $query = $this->db->get();
        $stats['points_earned_today'] = $query->row()->points_earned_today;

        // Points redeemed today
        $this->db->select('COALESCE(SUM(ABS(points)), 0) as points_redeemed_today');
        $this->db->from('patient_loyalty_points');
        $this->db->where('transaction_type', 'redeemed');
        $this->db->where('transaction_date', date('Y-m-d'));
        $this->db->where('is_active', 1);
        $query = $this->db->get();
        $stats['points_redeemed_today'] = $query->row()->points_redeemed_today;

        // Total points value
        $point_value = $this->getSettingValue('point_value_kes', 50);
        $this->db->select('COALESCE(SUM(CASE 
            WHEN transaction_type = "earned" THEN points 
            WHEN transaction_type = "redeemed" THEN -ABS(points)
            WHEN transaction_type = "adjusted" THEN points
            ELSE 0 
        END), 0) as total_points');
        $this->db->from('patient_loyalty_points');
        $this->db->where('is_active', 1);
        $query = $this->db->get();
        $total_points = $query->row()->total_points;
        $stats['total_points_value'] = $total_points * $point_value;

        return $stats;
    }

    /**
     * Get loyalty program settings
     */
    public function getSettings() {
        $this->db->select('setting_name, setting_value');
        $this->db->from('loyalty_program_settings');
        $this->db->where('is_active', 1);
        $query = $this->db->get();
        
        $settings = [];
        foreach ($query->result_array() as $row) {
            $settings[$row['setting_name']] = $row['setting_value'];
        }
        
        return $settings;
    }

    /**
     * Get specific setting value
     */
    public function getSettingValue($setting_name, $default = null) {
        $this->db->select('setting_value');
        $this->db->from('loyalty_program_settings');
        $this->db->where('setting_name', $setting_name);
        $this->db->where('is_active', 1);
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            return $query->row()->setting_value;
        }
        
        return $default;
    }

    /**
     * Update loyalty program settings
     */
    public function updateSettings($settings) {
        $this->db->trans_start();
        
        foreach ($settings as $setting_name => $setting_value) {
            $this->db->where('setting_name', $setting_name);
            $existing = $this->db->get('loyalty_program_settings');
            
            if ($existing->num_rows() > 0) {
                // Update existing setting
                $this->db->where('setting_name', $setting_name);
                $this->db->update('loyalty_program_settings', [
                    'setting_value' => $setting_value,
                    'updated_at' => date('Y-m-d H:i:s')
                ]);
            } else {
                // Insert new setting
                $this->db->insert('loyalty_program_settings', [
                    'setting_name' => $setting_name,
                    'setting_value' => $setting_value,
                    'is_active' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                ]);
            }
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    /**
     * Get loyalty leaderboard
     */
    public function getLeaderboard($limit = 10) {
        $this->db->select('
            p.id as patient_id,
            p.patient_name,
            COALESCE(SUM(CASE 
                WHEN plp.transaction_type = "earned" THEN plp.points 
                WHEN plp.transaction_type IN ("redeemed", "expired") THEN -ABS(plp.points)
                WHEN plp.transaction_type = "adjusted" THEN plp.points
                ELSE 0 
            END), 0) as total_points,
            COALESCE(SUM(CASE WHEN plp.transaction_type = "earned" THEN plp.amount_spent ELSE 0 END), 0) as total_spent
        ');
        $this->db->from('patients p');
        $this->db->join('patient_loyalty_points plp', 'p.id = plp.patient_id AND plp.is_active = 1', 'left');
        $this->db->where('p.is_active', 'yes');
        $this->db->group_by('p.id, p.patient_name');
        $this->db->having('total_points > 0');
        $this->db->order_by('total_points', 'DESC');
        $this->db->limit($limit);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Search patients for loyalty program
     */
    public function searchPatients($search_term, $limit = 10) {
        $this->db->select('
            p.id, 
            p.patient_name, 
            p.mobileno, 
            p.age, 
            p.gender,
            COALESCE(SUM(CASE 
                WHEN plp.transaction_type = "earned" THEN plp.points 
                WHEN plp.transaction_type IN ("redeemed", "expired") THEN -ABS(plp.points)
                WHEN plp.transaction_type = "adjusted" THEN plp.points
                ELSE 0 
            END), 0) as current_balance
        ');
        $this->db->from('patients p');
        $this->db->join('patient_loyalty_points plp', 'p.id = plp.patient_id AND plp.is_active = 1', 'left');
        $this->db->where('p.is_active', 'yes');
        
        if (!empty($search_term)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search_term);
            $this->db->or_like('p.mobileno', $search_term);
            $this->db->or_like('p.id', $search_term);
            $this->db->group_end();
        }
        
        $this->db->group_by('p.id, p.patient_name, p.mobileno, p.age, p.gender');
        $this->db->order_by('p.patient_name', 'ASC');
        $this->db->limit($limit);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Auto-award points based on payment
     */
    public function autoAwardPoints($patient_id, $amount_spent, $reference_type = 'general', $reference_id = null, $staff_id = null) {
        // Check if loyalty program is enabled
        $program_enabled = $this->getSettingValue('enable_program', 1);
        if (!$program_enabled) {
            return false;
        }

        // Get conversion rate
        $points_per_1000 = $this->getSettingValue('points_per_1000_kes', 1);
        $max_points_per_transaction = $this->getSettingValue('max_points_per_transaction', 100);
        
        // Calculate points
        $earned_points = floor($amount_spent / 1000 * $points_per_1000);
        
        // Apply maximum limit
        if ($earned_points > $max_points_per_transaction) {
            $earned_points = $max_points_per_transaction;
        }

        // Award points if significant enough
        if ($earned_points > 0) {
            $award_data = [
                'patient_id' => $patient_id,
                'transaction_type' => 'earned',
                'points' => $earned_points,
                'amount_spent' => $amount_spent,
                'conversion_rate' => $points_per_1000 / 1000,
                'description' => 'Points earned from ' . $reference_type . ' payment - ' . $earned_points . ' points for KES ' . number_format($amount_spent, 2),
                'reference_type' => $reference_type,
                'reference_id' => $reference_id,
                'transaction_date' => date('Y-m-d'),
                'staff_id' => $staff_id,
                'created_at' => date('Y-m-d H:i:s')
            ];

            return $this->awardPoints($award_data);
        }

        return false;
    }

    /**
     * Expire old points
     */
    public function expireOldPoints() {
        $expiry_enabled = $this->getSettingValue('enable_expiry', 1);
        if (!$expiry_enabled) {
            return false;
        }

        $expiry_months = $this->getSettingValue('points_expiry_months', 12);
        $expiry_date = date('Y-m-d', strtotime("-{$expiry_months} months"));

        // Get points that need to expire
        $this->db->select('id, patient_id, points, description');
        $this->db->from('patient_loyalty_points');
        $this->db->where('transaction_type', 'earned');
        $this->db->where('transaction_date <', $expiry_date);
        $this->db->where('is_active', 1);
        $this->db->where('expiry_date IS NULL OR expiry_date <=', date('Y-m-d'));
        $query = $this->db->get();

        $expired_points = $query->result_array();
        
        foreach ($expired_points as $expired_point) {
            // Create expiry record
            $expiry_data = [
                'patient_id' => $expired_point['patient_id'],
                'transaction_type' => 'expired',
                'points' => -abs($expired_point['points']),
                'description' => 'Points expired: ' . $expired_point['description'],
                'transaction_date' => date('Y-m-d'),
                'created_at' => date('Y-m-d H:i:s')
            ];

            $this->db->insert('patient_loyalty_points', $expiry_data);
        }

        return count($expired_points);
    }

    /**
     * Adjust patient points (admin function)
     */
    public function adjustPoints($patient_id, $points, $reason, $staff_id) {
        $adjustment_data = [
            'patient_id' => $patient_id,
            'transaction_type' => 'adjusted',
            'points' => $points,
            'description' => 'Points adjustment: ' . $reason,
            'transaction_date' => date('Y-m-d'),
            'staff_id' => $staff_id,
            'notes' => $reason,
            'created_at' => date('Y-m-d H:i:s')
        ];

        return $this->awardPoints($adjustment_data);
    }

    /**
     * Get points summary for a patient
     */
    public function getPatientPointsSummary($patient_id) {
        $balance = $this->getPatientLoyaltyBalance($patient_id);
        $recent_activity = $this->getPatientLoyaltyHistory($patient_id, 5);
        
        return [
            'balance' => $balance,
            'recent_activity' => $recent_activity,
            'point_value' => $this->getSettingValue('point_value_kes', 50)
        ];
    }
}