<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Laboratory_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Save a new laboratory test
     */
    public function saveLaboratoryTest($data) {
        $this->db->insert('laboratory', $data);
        return $this->db->insert_id();
    }

    /**
     * Update an existing laboratory test
     */
    public function updateLaboratoryTest($labId, $data) {
        $this->db->where('id', $labId);
        return $this->db->update('laboratory', $data);
    }

    /**
     * Get laboratory test details by ID
     */
    public function getLaboratoryDetails($labId) {
        $this->db->select('l.*, p.patient_name, p.mobileno as patient_phone, p.age, p.gender, 
                          CONCAT(s.name, " ", s.surname) as technician_name');
        
        // Add test_date if column exists
        if ($this->db->field_exists('test_date', 'laboratory')) {
            $this->db->select('l.test_date');
        }
        
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        $this->db->where('l.id', $labId);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get laboratory tests list for DataTable
     */
    public function getLaboratoryList($start = 0, $length = 10, $search = '', $orderColumn = 'id', $orderDir = 'desc') {
        $this->db->select('l.id, l.patient_id, l.test_name, l.test_result, l.notes, l.created_at,
                          p.patient_name, p.id as patient_id,
                          CONCAT(s.name, " ", s.surname) as technician_name');
        
        // Add verification_status if column exists
        if ($this->db->field_exists('verification_status', 'laboratory')) {
            $this->db->select('l.verification_status');
        }
        
        // Add tested_by if column exists
        if ($this->db->field_exists('tested_by', 'laboratory')) {
            $this->db->select('l.tested_by');
        }
        
        // Add test_date if column exists
        if ($this->db->field_exists('test_date', 'laboratory')) {
            $this->db->select('l.test_date');
        }
        
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search);
            $this->db->or_like('l.test_name', $search);
            $this->db->or_like('l.test_result', $search);
            $this->db->or_like('l.notes', $search);
            $this->db->or_like('CONCAT(s.name, " ", s.surname)', $search);
            $this->db->group_end();
        }
        
        $this->db->order_by($orderColumn, $orderDir);
        $this->db->limit($length, $start);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get total laboratory tests count
     */
    public function getLaboratoryCount($search = '') {
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search);
            $this->db->or_like('l.test_name', $search);
            $this->db->or_like('l.test_result', $search);
            $this->db->or_like('l.notes', $search);
            $this->db->or_like('CONCAT(s.name, " ", s.surname)', $search);
            $this->db->group_end();
        }
        
        return $this->db->count_all_results();
    }

    /**
     * Get patient's laboratory test history
     */
    public function getPatientLabHistory($patientId, $limit = 10) {
        $this->db->select('l.id, l.test_name, l.test_result, l.notes, l.created_at as date,
                          CONCAT(s.name, " ", s.surname) as technician_name');
        
        // Add verification_status if column exists
        if ($this->db->field_exists('verification_status', 'laboratory')) {
            $this->db->select('l.verification_status');
        }
        
        // Add test_date if column exists
        if ($this->db->field_exists('test_date', 'laboratory')) {
            $this->db->select('l.test_date');
        }
        
        $this->db->from('laboratory l');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        $this->db->where('l.patient_id', $patientId);
        $this->db->order_by('l.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Delete a laboratory test
     */
    public function deleteLaboratoryTest($labId) {
        $this->db->where('id', $labId);
        return $this->db->delete('laboratory');
    }

    /**
     * Get laboratory tests by date range
     */
    public function getLabTestsByDateRange($startDate, $endDate, $technicianId = null) {
        $this->db->select('l.*, p.patient_name, CONCAT(s.name, " ", s.surname) as technician_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        
        // Use test_date if available, otherwise created_at
        if ($this->db->field_exists('test_date', 'laboratory')) {
            $this->db->where('l.test_date >=', $startDate);
            $this->db->where('l.test_date <=', $endDate);
        } else {
            $this->db->where('DATE(l.created_at) >=', $startDate);
            $this->db->where('DATE(l.created_at) <=', $endDate);
        }
        
        if ($technicianId) {
            $this->db->where('l.tested_by', $technicianId);
        }
        
        $this->db->order_by('l.created_at', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get laboratory statistics
     */
    public function getLaboratoryStats($dateFrom = null, $dateTo = null) {
        if (!$dateFrom) $dateFrom = date('Y-m-01'); // First day of current month
        if (!$dateTo) $dateTo = date('Y-m-d'); // Today
        
        // Total lab tests
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $totalLabTests = $this->db->count_all_results('laboratory');
        
        // Tests by technician
        $this->db->select('s.name, s.surname, COUNT(l.id) as test_count');
        $this->db->from('laboratory l');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        $this->db->where('DATE(l.created_at) >=', $dateFrom);
        $this->db->where('DATE(l.created_at) <=', $dateTo);
        $this->db->group_by('l.tested_by');
        $this->db->order_by('test_count', 'DESC');
        $testsByTechnician = $this->db->get()->result_array();
        
        // Tests by category/type
        $this->db->select('test_name, COUNT(id) as count');
        $this->db->from('laboratory');
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $this->db->group_by('test_name');
        $this->db->order_by('count', 'DESC');
        $this->db->limit(10);
        $testsByType = $this->db->get()->result_array();
        
        // Daily lab tests for chart
        $this->db->select('DATE(created_at) as date, COUNT(id) as count');
        $this->db->from('laboratory');
        $this->db->where('DATE(created_at) >=', $dateFrom);
        $this->db->where('DATE(created_at) <=', $dateTo);
        $this->db->group_by('DATE(created_at)');
        $this->db->order_by('date', 'ASC');
        $dailyLabTests = $this->db->get()->result_array();
        
        // Verification status statistics (if column exists)
        $verificationStats = [];
        if ($this->db->field_exists('verification_status', 'laboratory')) {
            $this->db->select('verification_status, COUNT(id) as count');
            $this->db->from('laboratory');
            $this->db->where('DATE(created_at) >=', $dateFrom);
            $this->db->where('DATE(created_at) <=', $dateTo);
            $this->db->group_by('verification_status');
            $verificationStats = $this->db->get()->result_array();
        }
        
        return [
            'total_lab_tests' => $totalLabTests,
            'tests_by_technician' => $testsByTechnician,
            'tests_by_type' => $testsByType,
            'daily_lab_tests' => $dailyLabTests,
            'verification_stats' => $verificationStats
        ];
    }

    /**
     * Search laboratory tests
     */
    public function searchLabTests($searchTerm, $limit = 50) {
        $this->db->select('l.id, l.test_name, l.test_result, l.notes, l.created_at,
                          p.patient_name, p.id as patient_id,
                          CONCAT(s.name, " ", s.surname) as technician_name');
        
        // Add verification_status if column exists
        if ($this->db->field_exists('verification_status', 'laboratory')) {
            $this->db->select('l.verification_status');
        }
        
        // Add test_date if column exists
        if ($this->db->field_exists('test_date', 'laboratory')) {
            $this->db->select('l.test_date');
        }
        
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        
        $this->db->group_start();
        $this->db->like('p.patient_name', $searchTerm);
        $this->db->or_like('l.test_name', $searchTerm);
        $this->db->or_like('l.test_result', $searchTerm);
        $this->db->or_like('l.notes', $searchTerm);
        $this->db->or_like('p.id', $searchTerm);
        $this->db->group_end();
        
        $this->db->order_by('l.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get tests by category
     */
    public function getTestsByCategory($category, $limit = 50) {
        $this->db->select('l.*, p.patient_name, CONCAT(s.name, " ", s.surname) as technician_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->join('staff s', 's.id = l.tested_by', 'left');
        
        // Map category to test names
        $testCategories = [
            'Blood Tests' => ['CBC', 'FBC', 'ESR', 'Blood Sugar', 'HbA1c', 'Lipid Profile', 'Liver Function', 'Kidney Function', 'Thyroid Function'],
            'Urine Tests' => ['Urinalysis', 'Urine Culture', 'Urine Protein', 'Urine Glucose'],
            'Microbiology' => ['Blood Culture', 'Stool Culture', 'Throat Swab', 'Wound Swab'],
            'Serology' => ['HIV Test', 'Hepatitis B', 'Hepatitis C', 'VDRL', 'Widal Test'],
            'Others' => ['X-Ray', 'ECG', 'Ultrasound']
        ];
        
        if (isset($testCategories[$category])) {
           $this->db->where_in('l.test_name', $testCategories[$category]);
       } else {
           $this->db->like('l.test_name', $category);
       }
       
       $this->db->order_by('l.created_at', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get pending test results
    */
   public function getPendingTestResults($limit = 50) {
       $this->db->select('l.*, p.patient_name, p.mobileno, CONCAT(s.name, " ", s.surname) as technician_name');
       $this->db->from('laboratory l');
       $this->db->join('patients p', 'p.id = l.patient_id', 'left');
       $this->db->join('staff s', 's.id = l.tested_by', 'left');
       
       // Look for pending indicators in test_result or notes
       $this->db->group_start();
       $this->db->where('l.test_result', '');
       $this->db->or_where('l.test_result IS NULL');
       $this->db->or_like('l.test_result', 'pending', 'both');
       $this->db->or_like('l.notes', 'pending', 'both');
       $this->db->group_end();
       
       $this->db->order_by('l.created_at', 'ASC'); // Oldest first
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get critical/abnormal results
    */
   public function getCriticalResults($days = 7, $limit = 50) {
       $this->db->select('l.*, p.patient_name, p.mobileno, CONCAT(s.name, " ", s.surname) as technician_name');
       $this->db->from('laboratory l');
       $this->db->join('patients p', 'p.id = l.patient_id', 'left');
       $this->db->join('staff s', 's.id = l.tested_by', 'left');
       
       // Look for critical/abnormal indicators in test_result or notes
       $this->db->group_start();
       $this->db->like('l.test_result', 'critical', 'both');
       $this->db->or_like('l.test_result', 'abnormal', 'both');
       $this->db->or_like('l.test_result', 'high', 'both');
       $this->db->or_like('l.test_result', 'low', 'both');
       $this->db->or_like('l.notes', 'critical', 'both');
       $this->db->or_like('l.notes', 'abnormal', 'both');
       $this->db->or_like('l.notes', 'urgent', 'both');
       $this->db->group_end();
       
       $this->db->where('l.created_at >=', date('Y-m-d H:i:s', strtotime("-{$days} days")));
       $this->db->order_by('l.created_at', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get tests by specific test name
    */
   public function getTestsByName($testName, $dateFrom = null, $dateTo = null, $limit = 100) {
       $this->db->select('l.*, p.patient_name, p.age, p.gender, CONCAT(s.name, " ", s.surname) as technician_name');
       $this->db->from('laboratory l');
       $this->db->join('patients p', 'p.id = l.patient_id', 'left');
       $this->db->join('staff s', 's.id = l.tested_by', 'left');
       
       $this->db->like('l.test_name', $testName);
       
       if ($dateFrom) {
           $this->db->where('DATE(l.created_at) >=', $dateFrom);
       }
       
       if ($dateTo) {
           $this->db->where('DATE(l.created_at) <=', $dateTo);
       }
       
       $this->db->order_by('l.created_at', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get test turnaround time statistics
    */
   public function getTestTurnaroundStats($testName = null, $days = 30) {
       $this->db->select('l.test_name, 
                         AVG(TIMESTAMPDIFF(HOUR, l.created_at, l.updated_at)) as avg_hours,
                         MIN(TIMESTAMPDIFF(HOUR, l.created_at, l.updated_at)) as min_hours,
                         MAX(TIMESTAMPDIFF(HOUR, l.created_at, l.updated_at)) as max_hours,
                         COUNT(l.id) as total_tests');
       $this->db->from('laboratory l');
       $this->db->where('l.updated_at IS NOT NULL');
       $this->db->where('l.created_at >=', date('Y-m-d H:i:s', strtotime("-{$days} days")));
       
       if ($testName) {
           $this->db->like('l.test_name', $testName);
       }
       
       $this->db->group_by('l.test_name');
       $this->db->order_by('avg_hours', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get workload by technician
    */
   public function getTechnicianWorkload($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       $this->db->select('s.id, s.name, s.surname, 
                         COUNT(l.id) as total_tests,
                         COUNT(CASE WHEN l.test_result IS NOT NULL AND l.test_result != "" THEN 1 END) as completed_tests,
                         COUNT(CASE WHEN l.test_result IS NULL OR l.test_result = "" THEN 1 END) as pending_tests');
       $this->db->from('staff s');
       $this->db->join('laboratory l', 'l.tested_by = s.id', 'left');
       $this->db->where('DATE(l.created_at) >=', $dateFrom);
       $this->db->where('DATE(l.created_at) <=', $dateTo);
       $this->db->group_by('s.id, s.name, s.surname');
       $this->db->order_by('total_tests', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get test results summary for a patient
    */
   public function getPatientTestSummary($patientId, $months = 6) {
       $this->db->select('l.test_name, 
                         COUNT(l.id) as test_count,
                         MAX(l.created_at) as last_test_date,
                         GROUP_CONCAT(DISTINCT l.test_result ORDER BY l.created_at DESC SEPARATOR "; ") as recent_results');
       $this->db->from('laboratory l');
       $this->db->where('l.patient_id', $patientId);
       $this->db->where('l.created_at >=', date('Y-m-d H:i:s', strtotime("-{$months} months")));
       $this->db->group_by('l.test_name');
       $this->db->order_by('last_test_date', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Get tests requiring follow-up
    */
   public function getTestsRequiringFollowup($days = 30) {
       $this->db->select('l.*, p.patient_name, p.mobileno, CONCAT(s.name, " ", s.surname) as technician_name');
       $this->db->from('laboratory l');
       $this->db->join('patients p', 'p.id = l.patient_id', 'left');
       $this->db->join('staff s', 's.id = l.tested_by', 'left');
       
       // Look for follow-up indicators in notes or test results
       $this->db->group_start();
       $this->db->like('l.notes', 'follow', 'both');
       $this->db->or_like('l.notes', 'repeat', 'both');
       $this->db->or_like('l.test_result', 'follow', 'both');
       $this->db->or_like('l.test_result', 'repeat', 'both');
       $this->db->group_end();
       
       $this->db->where('l.created_at >=', date('Y-m-d H:i:s', strtotime("-{$days} days")));
       $this->db->order_by('l.created_at', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Parse test result notes for structured data
    */
   public function parseTestNotes($notes) {
       if (empty($notes)) {
           return [];
       }
       
       $parsed = [];
       $lines = explode("\n", $notes);
       
       foreach ($lines as $line) {
           $line = trim($line);
           if (empty($line)) continue;
           
           // Look for structured data patterns
           if (strpos($line, 'Test Result:') === 0) {
               $parsed['test_result'] = trim(str_replace('Test Result:', '', $line));
           } elseif (strpos($line, 'Reference Range:') === 0) {
               $parsed['reference_range'] = trim(str_replace('Reference Range:', '', $line));
           } elseif (strpos($line, 'Status:') === 0) {
               $parsed['status'] = trim(str_replace('Status:', '', $line));
           } elseif (strpos($line, 'Clinical Notes:') === 0) {
               $parsed['clinical_notes'] = trim(str_replace('Clinical Notes:', '', $line));
           }
       }
       
       return $parsed;
   }

   /**
    * Format test results for display
    */
   public function formatTestResultForDisplay($testResult, $notes = '') {
       $formatted = [];
       
       if (!empty($testResult)) {
           $formatted['Result'] = $testResult;
       }
       
       // Parse notes for additional structured data
       $parsedNotes = $this->parseTestNotes($notes);
       
       if (!empty($parsedNotes['reference_range'])) {
           $formatted['Reference Range'] = $parsedNotes['reference_range'];
       }
       
       if (!empty($parsedNotes['status'])) {
           $formatted['Status'] = $parsedNotes['status'];
       }
       
       if (!empty($parsedNotes['clinical_notes'])) {
           $formatted['Clinical Notes'] = $parsedNotes['clinical_notes'];
       }
       
       return $formatted;
   }

   /**
    * Get quality control statistics
    */
   public function getQualityControlStats($dateFrom = null, $dateTo = null) {
       if (!$dateFrom) $dateFrom = date('Y-m-01');
       if (!$dateTo) $dateTo = date('Y-m-d');
       
       // Tests with verification status
       $verificationStats = [];
       if ($this->db->field_exists('verification_status', 'laboratory')) {
           $this->db->select('verification_status, COUNT(id) as count, 
                             ROUND((COUNT(id) * 100.0 / (SELECT COUNT(*) FROM laboratory WHERE DATE(created_at) BETWEEN "' . $dateFrom . '" AND "' . $dateTo . '")), 2) as percentage');
           $this->db->from('laboratory');
           $this->db->where('DATE(created_at) >=', $dateFrom);
           $this->db->where('DATE(created_at) <=', $dateTo);
           $this->db->group_by('verification_status');
           $verificationStats = $this->db->get()->result_array();
       }
       
       // Tests by result status (if stored in notes)
       $this->db->select('CASE 
                          WHEN notes LIKE "%completed%" THEN "Completed"
                          WHEN notes LIKE "%pending%" THEN "Pending"
                          WHEN notes LIKE "%in_progress%" THEN "In Progress"
                          WHEN notes LIKE "%abnormal%" THEN "Abnormal"
                          WHEN notes LIKE "%critical%" THEN "Critical"
                          ELSE "Unknown"
                          END as status, COUNT(id) as count');
       $this->db->from('laboratory');
       $this->db->where('DATE(created_at) >=', $dateFrom);
       $this->db->where('DATE(created_at) <=', $dateTo);
       $this->db->group_by('status');
       $resultStatusStats = $this->db->get()->result_array();
       
       return [
           'verification_stats' => $verificationStats,
           'result_status_stats' => $resultStatusStats,
           'date_range' => ['from' => $dateFrom, 'to' => $dateTo]
       ];
   }

   /**
    * Get most common test types
    */
   public function getMostCommonTests($limit = 10, $days = 30) {
       $this->db->select('test_name, COUNT(id) as test_count, 
                         ROUND(AVG(CASE WHEN test_result IS NOT NULL AND test_result != "" THEN 1 ELSE 0 END) * 100, 2) as completion_rate');
       $this->db->from('laboratory');
       $this->db->where('created_at >=', date('Y-m-d H:i:s', strtotime("-{$days} days")));
       $this->db->group_by('test_name');
       $this->db->order_by('test_count', 'DESC');
       $this->db->limit($limit);
       
       $query = $this->db->get();
       return $query->result_array();
   }

   /**
    * Export laboratory data for reports
    */
   public function exportLabData($filters = []) {
       $this->db->select('l.id, l.patient_id, p.patient_name, p.age, p.gender,
                         l.test_name, l.test_result, l.notes, l.created_at,
                         CONCAT(s.name, " ", s.surname) as technician_name');
       
       // Add additional fields if they exist
       if ($this->db->field_exists('test_date', 'laboratory')) {
           $this->db->select('l.test_date');
       }
       
       if ($this->db->field_exists('verification_status', 'laboratory')) {
           $this->db->select('l.verification_status');
       }
       
       $this->db->from('laboratory l');
       $this->db->join('patients p', 'p.id = l.patient_id', 'left');
       $this->db->join('staff s', 's.id = l.tested_by', 'left');
       
       // Apply filters
       if (!empty($filters['date_from'])) {
           $this->db->where('DATE(l.created_at) >=', $filters['date_from']);
       }
       
       if (!empty($filters['date_to'])) {
           $this->db->where('DATE(l.created_at) <=', $filters['date_to']);
       }
       
       if (!empty($filters['test_name'])) {
           $this->db->like('l.test_name', $filters['test_name']);
       }
       
       if (!empty($filters['technician_id'])) {
           $this->db->where('l.tested_by', $filters['technician_id']);
       }
       
       if (!empty($filters['patient_id'])) {
           $this->db->where('l.patient_id', $filters['patient_id']);
       }
       
       $this->db->order_by('l.created_at', 'DESC');
       
       $query = $this->db->get();
       return $query->result_array();
   }
}