<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Lab_reports_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get all lab reports with pagination and filtering
     */
    public function getAllReports($limit = 10, $offset = 0, $search = '', $filters = []) {
        $this->db->select('
            plr.*, 
            p.patient_name, 
            p.id as patient_id, 
            p.mobileno,
            s1.name as uploaded_by_name,
            s1.surname as uploaded_by_surname,
            s2.name as verified_by_name,
            s2.surname as verified_by_surname,
            s3.name as technician_name,
            s3.surname as technician_surname
        ');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'p.id = plr.patient_id', 'left');
        $this->db->join('staff s1', 's1.id = plr.uploaded_by', 'left');
        $this->db->join('staff s2', 's2.id = plr.verified_by', 'left');
        $this->db->join('staff s3', 's3.id = plr.lab_technician', 'left');

        // Apply search
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search);
            $this->db->or_like('plr.report_title', $search);
            $this->db->or_like('plr.report_description', $search);
            $this->db->or_like('plr.test_names', $search);
            $this->db->group_end();
        }

        // Apply filters
        if (!empty($filters['status'])) {
            $this->db->where('plr.status', $filters['status']);
        }
        if (!empty($filters['department'])) {
            $this->db->where('plr.department', $filters['department']);
        }
        if (!empty($filters['report_type'])) {
            $this->db->where('plr.report_type', $filters['report_type']);
        }
        if (!empty($filters['date_from'])) {
            $this->db->where('plr.report_date >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->db->where('plr.report_date <=', $filters['date_to']);
        }
        if (!empty($filters['is_critical'])) {
            $this->db->where('plr.is_critical', 1);
        }

        $this->db->order_by('plr.created_at', 'DESC');
        $this->db->limit($limit, $offset);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get total count of reports for pagination
     */
    public function getTotalReportsCount($search = '', $filters = []) {
        $this->db->select('COUNT(*) as total');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'p.id = plr.patient_id', 'left');

        // Apply search
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $search);
            $this->db->or_like('plr.report_title', $search);
            $this->db->or_like('plr.report_description', $search);
            $this->db->or_like('plr.test_names', $search);
            $this->db->group_end();
        }

        // Apply filters
        if (!empty($filters['status'])) {
            $this->db->where('plr.status', $filters['status']);
        }
        if (!empty($filters['department'])) {
            $this->db->where('plr.department', $filters['department']);
        }
        if (!empty($filters['report_type'])) {
            $this->db->where('plr.report_type', $filters['report_type']);
        }
        if (!empty($filters['date_from'])) {
            $this->db->where('plr.report_date >=', $filters['date_from']);
        }
        if (!empty($filters['date_to'])) {
            $this->db->where('plr.report_date <=', $filters['date_to']);
        }
        if (!empty($filters['is_critical'])) {
            $this->db->where('plr.is_critical', 1);
        }

        $query = $this->db->get();
        return $query->row()->total;
    }

    /**
     * Get patient's lab reports
     */
    public function getPatientReports($patient_id, $limit = 20) {
        $this->db->select('
            plr.*, 
            s1.name as uploaded_by_name,
            s1.surname as uploaded_by_surname,
            s2.name as verified_by_name,
            s2.surname as verified_by_surname,
            s3.name as technician_name,
            s3.surname as technician_surname
        ');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('staff s1', 's1.id = plr.uploaded_by', 'left');
        $this->db->join('staff s2', 's2.id = plr.verified_by', 'left');
        $this->db->join('staff s3', 's3.id = plr.lab_technician', 'left');
        $this->db->where('plr.patient_id', $patient_id);
        $this->db->order_by('plr.created_at', 'DESC');
        $this->db->limit($limit);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get single report details
     */
    public function getReportById($report_id) {
        $this->db->select('
            plr.*, 
            p.patient_name, 
            p.id as patient_id, 
            p.mobileno,
            p.age,
            p.gender,
            s1.name as uploaded_by_name,
            s1.surname as uploaded_by_surname,
            s2.name as verified_by_name,
            s2.surname as verified_by_surname,
            s3.name as technician_name,
            s3.surname as technician_surname
        ');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'p.id = plr.patient_id', 'left');
        $this->db->join('staff s1', 's1.id = plr.uploaded_by', 'left');
        $this->db->join('staff s2', 's2.id = plr.verified_by', 'left');
        $this->db->join('staff s3', 's3.id = plr.lab_technician', 'left');
        $this->db->where('plr.id', $report_id);

        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Insert new lab report
     */
    public function insertReport($data) {
        $this->db->insert('patient_lab_reports', $data);
        return $this->db->insert_id();
    }

    /**
     * Update lab report
     */
    public function updateReport($report_id, $data) {
        $this->db->where('id', $report_id);
        return $this->db->update('patient_lab_reports', $data);
    }

    /**
     * Delete lab report
     */
    public function deleteReport($report_id) {
        $this->db->where('id', $report_id);
        return $this->db->delete('patient_lab_reports');
    }

    /**
     * Verify report
     */
    public function verifyReport($report_id, $verified_by, $notes = '') {
        $data = [
            'status' => 'verified',
            'verified_by' => $verified_by,
            'verified_at' => date('Y-m-d H:i:s'),
            'notes' => $notes,
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $this->db->where('id', $report_id);
        return $this->db->update('patient_lab_reports', $data);
    }

    /**
     * Get dashboard statistics
     */
    public function getDashboardStats() {
        $stats = [];

        // Total reports
        $this->db->select('COUNT(*) as total');
        $this->db->from('patient_lab_reports');
        $query = $this->db->get();
        $stats['total'] = $query->row()->total;

        // Pending reports
        $this->db->select('COUNT(*) as pending');
        $this->db->from('patient_lab_reports');
        $this->db->where('status', 'pending');
        $query = $this->db->get();
        $stats['pending'] = $query->row()->pending;

        // Verified today
        $this->db->select('COUNT(*) as verified_today');
        $this->db->from('patient_lab_reports');
        $this->db->where('status', 'verified');
        $this->db->where('DATE(verified_at)', date('Y-m-d'));
        $query = $this->db->get();
        $stats['verified_today'] = $query->row()->verified_today;

        // Critical reports
        $this->db->select('COUNT(*) as critical');
        $this->db->from('patient_lab_reports');
        $this->db->where('is_critical', 1);
        $this->db->where('status !=', 'verified');
        $query = $this->db->get();
        $stats['critical'] = $query->row()->critical;

        return $stats;
    }

    /**
     * Search patients for dropdown
     */
    public function searchPatients($search_term, $limit = 10) {
        $this->db->select('id, patient_name, mobileno, age, gender');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        
        if (!empty($search_term)) {
            $this->db->group_start();
            $this->db->like('patient_name', $search_term);
            $this->db->or_like('mobileno', $search_term);
            $this->db->or_like('id', $search_term);
            $this->db->group_end();
        }
        
        $this->db->order_by('patient_name', 'ASC');
        $this->db->limit($limit);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get reports for DataTables server-side processing
     */
    public function getReportsForDataTables($request) {
        $columns = ['plr.id', 'p.patient_name', 'plr.report_title', 'plr.report_type', 'plr.department', 'plr.report_date', 'plr.status'];
        
        $this->db->select('
            plr.*, 
            p.patient_name, 
            p.id as patient_id, 
            s1.name as uploaded_by_name,
            s1.surname as uploaded_by_surname
        ');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'p.id = plr.patient_id', 'left');
        $this->db->join('staff s1', 's1.id = plr.uploaded_by', 'left');

        // Apply filters from request
        if (!empty($request['status'])) {
            $this->db->where('plr.status', $request['status']);
        }
        if (!empty($request['department'])) {
            $this->db->where('plr.department', $request['department']);
        }
        if (!empty($request['date_from'])) {
            $this->db->where('plr.report_date >=', $request['date_from']);
        }
        if (!empty($request['date_to'])) {
            $this->db->where('plr.report_date <=', $request['date_to']);
        }

        // Search
        if (!empty($request['search']['value'])) {
            $search_value = $request['search']['value'];
            $this->db->group_start();
            $this->db->like('p.patient_name', $search_value);
            $this->db->or_like('plr.report_title', $search_value);
            $this->db->or_like('plr.report_description', $search_value);
            $this->db->group_end();
        }

        // Count total records (before limit)
        $total_query = clone $this->db;
        $total_records = $total_query->count_all_results('', FALSE);

        // Order
        if (isset($request['order'][0]['column'])) {
            $column_index = $request['order'][0]['column'];
            $order_dir = $request['order'][0]['dir'];
            if (isset($columns[$column_index])) {
                $this->db->order_by($columns[$column_index], $order_dir);
            }
        } else {
            $this->db->order_by('plr.created_at', 'DESC');
        }

        // Limit
        if (isset($request['length']) && $request['length'] != -1) {
            $this->db->limit($request['length'], $request['start']);
        }

        $query = $this->db->get();
        $data = $query->result_array();

        return [
            'draw' => intval($request['draw']),
            'recordsTotal' => $total_records,
            'recordsFiltered' => $total_records,
            'data' => $data
        ];
    }

    /**
     * Get patient reports count
     */
    public function getPatientReportsCount($patient_id) {
        $this->db->select('COUNT(*) as total');
        $this->db->from('patient_lab_reports');
        $this->db->where('patient_id', $patient_id);
        $query = $this->db->get();
        return $query->row()->total;
    }

    /**
     * Mark report as critical
     */
    public function markAsCritical($report_id, $is_critical = 1) {
        $data = [
            'is_critical' => $is_critical,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        $this->db->where('id', $report_id);
        return $this->db->update('patient_lab_reports', $data);
    }

    /**
     * Get reports by status
     */
    public function getReportsByStatus($status) {
        $this->db->select('
            plr.*, 
            p.patient_name, 
            p.id as patient_id
        ');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'p.id = plr.patient_id', 'left');
        $this->db->where('plr.status', $status);
        $this->db->order_by('plr.created_at', 'DESC');

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get recent reports
     */
    public function getRecentReports($limit = 10) {
        $this->db->select('
            plr.*, 
            p.patient_name, 
            p.id as patient_id
        ');
        $this->db->from('patient_lab_reports plr');
        $this->db->join('patients p', 'p.id = plr.patient_id', 'left');
        $this->db->order_by('plr.created_at', 'DESC');
        $this->db->limit($limit);

        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Check if table exists
     */
    public function checkTable() {
        return $this->db->table_exists('patient_lab_reports');
    }
}