<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Chartofaccounts_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Add new account
     */
    public function add($data) {
        if ($this->db->insert('chart_of_accounts', $data)) {
            return $this->db->insert_id();
        }
        return false;
    }

    /**
     * Update account
     */
    public function update($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('chart_of_accounts', $data);
    }

    /**
     * Get account by ID
     */
    public function get($id) {
        $this->db->select('
            coa.*,
            parent.account_name as parent_account_name
        ');
        $this->db->from('chart_of_accounts coa');
        $this->db->join('chart_of_accounts parent', 'parent.id = coa.parent_account_id', 'left');
        $this->db->where('coa.id', $id);
        
        return $this->db->get()->row_array();
    }

    /**
     * Get all accounts
     */
    public function getAll($active_only = true) {
        $this->db->select('
            coa.*,
            parent.account_name as parent_account_name
        ');
        $this->db->from('chart_of_accounts coa');
        $this->db->join('chart_of_accounts parent', 'parent.id = coa.parent_account_id', 'left');
        
        if ($active_only) {
            $this->db->where('coa.is_active', 1);
        }
        
        $this->db->order_by('coa.account_code');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get active accounts for dropdowns
     */
    public function getActiveAccounts() {
        $this->db->select('id, account_code, account_name, account_type');
        $this->db->from('chart_of_accounts');
        $this->db->where('is_active', 1);
        $this->db->order_by('account_code');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get parent accounts (for hierarchy)
     */
    public function getParentAccounts() {
        $this->db->select('id, account_code, account_name');
        $this->db->from('chart_of_accounts');
        $this->db->where('parent_account_id IS NULL');
        $this->db->where('is_active', 1);
        $this->db->order_by('account_code');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get accounts by type
     */
    public function getAccountsByType($account_type) {
        $this->db->select('id, account_code, account_name');
        $this->db->from('chart_of_accounts');
        $this->db->where('account_type', $account_type);
        $this->db->where('is_active', 1);
        $this->db->order_by('account_code');
        
        return $this->db->get()->result_array();
    }

    /**
     * Get account by code
     */
    public function getByCode($account_code) {
        $this->db->where('account_code', $account_code);
        $this->db->where('is_active', 1);
        
        return $this->db->get('chart_of_accounts')->row_array();
    }

    /**
     * Check if account code exists
     */
    public function isCodeExists($account_code, $exclude_id = null) {
        $this->db->where('account_code', $account_code);
        
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        
        $query = $this->db->get('chart_of_accounts');
        return $query->num_rows() > 0;
    }

    /**
     * Delete account (soft delete)
     */
    public function delete($id) {
        // Check if account has transactions
        $this->db->where('account_id', $id);
        $transaction_count = $this->db->count_all_results('journal_lines');
        
        if ($transaction_count > 0) {
            // Soft delete - deactivate instead of deleting
            return $this->update($id, ['is_active' => 0]);
        } else {
            // Hard delete if no transactions
            $this->db->where('id', $id);
            return $this->db->delete('chart_of_accounts');
        }
    }

    /**
     * Get account hierarchy (tree structure)
     */
    public function getAccountHierarchy() {
        $accounts = $this->getAll();
        return $this->buildTree($accounts);
    }

    /**
     * Build tree structure from flat array
     */
    private function buildTree($accounts, $parent_id = null) {
        $tree = [];
        
        foreach ($accounts as $account) {
            if ($account['parent_account_id'] == $parent_id) {
                $children = $this->buildTree($accounts, $account['id']);
                if ($children) {
                    $account['children'] = $children;
                }
                $tree[] = $account;
            }
        }
        
        return $tree;
    }

    /**
     * Get account balance
     */
    public function getAccountBalance($account_id, $as_of_date = null) {
        if (!$as_of_date) {
            $as_of_date = date('Y-m-d');
        }
        
        $this->db->select('
            SUM(jl.debit_amount) as total_debits,
            SUM(jl.credit_amount) as total_credits,
            coa.account_type
        ');
        $this->db->from('journal_lines jl');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
        $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id');
        $this->db->where('jl.account_id', $account_id);
        $this->db->where('jh.date <=', $as_of_date);
        
        $query = $this->db->get();
        $result = $query->row();
        
        if (!$result) {
            return 0;
        }
        
        $net_balance = $result->total_debits - $result->total_credits;
        
        // For liability, equity, and revenue accounts, credit balance is positive
        if (in_array($result->account_type, ['liability', 'equity', 'revenue'])) {
            return -$net_balance;
        }
        
        // For asset and expense accounts, debit balance is positive
        return $net_balance;
    }

    /**
     * Initialize default chart of accounts
     */
    public function initializeDefaultAccounts() {
        $default_accounts = [
            // Assets
            ['account_code' => '1000', 'account_name' => 'Cash', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1010', 'account_name' => 'Bank - Current Account', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1020', 'account_name' => 'Bank - Savings Account', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1050', 'account_name' => 'Petty Cash', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1200', 'account_name' => 'Accounts Receivable', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1300', 'account_name' => 'Inventory - Medical Supplies', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1310', 'account_name' => 'Inventory - Pharmacy', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1400', 'account_name' => 'Prepaid Expenses', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1500', 'account_name' => 'Medical Equipment', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1510', 'account_name' => 'Furniture & Fixtures', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1520', 'account_name' => 'Computer Equipment', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1600', 'account_name' => 'Building', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1700', 'account_name' => 'Accumulated Depreciation - Equipment', 'account_type' => 'asset', 'parent_account_id' => null],
            ['account_code' => '1710', 'account_name' => 'Accumulated Depreciation - Building', 'account_type' => 'asset', 'parent_account_id' => null],

            // Liabilities
            ['account_code' => '2000', 'account_name' => 'Accounts Payable', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2100', 'account_name' => 'Accrued Expenses', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2200', 'account_name' => 'Salaries Payable', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2300', 'account_name' => 'Taxes Payable', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2400', 'account_name' => 'Insurance Payable', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2500', 'account_name' => 'Short-term Loans', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2600', 'account_name' => 'Long-term Loans', 'account_type' => 'liability', 'parent_account_id' => null],
            ['account_code' => '2700', 'account_name' => 'Patient Deposits', 'account_type' => 'liability', 'parent_account_id' => null],

            // Equity
            ['account_code' => '3000', 'account_name' => 'Owner\'s Capital', 'account_type' => 'equity', 'parent_account_id' => null],
            ['account_code' => '3100', 'account_name' => 'Retained Earnings', 'account_type' => 'equity', 'parent_account_id' => null],
            ['account_code' => '3200', 'account_name' => 'Owner\'s Drawings', 'account_type' => 'equity', 'parent_account_id' => null],

            // Revenue
            ['account_code' => '4000', 'account_name' => 'OPD Consultation Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4100', 'account_name' => 'IPD Service Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4200', 'account_name' => 'Pathology Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4300', 'account_name' => 'Radiology Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4400', 'account_name' => 'Pharmacy Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4500', 'account_name' => 'Blood Bank Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4600', 'account_name' => 'Ambulance Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4700', 'account_name' => 'Surgery Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4800', 'account_name' => 'Other Medical Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],
            ['account_code' => '4900', 'account_name' => 'Insurance Revenue', 'account_type' => 'revenue', 'parent_account_id' => null],

            // Expenses
            ['account_code' => '5000', 'account_name' => 'Salaries & Wages', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5100', 'account_name' => 'Medical Supplies Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5200', 'account_name' => 'Pharmacy Purchases', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5300', 'account_name' => 'Utilities Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5400', 'account_name' => 'Rent Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5500', 'account_name' => 'Insurance Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5600', 'account_name' => 'Depreciation Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5700', 'account_name' => 'Maintenance & Repairs', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5800', 'account_name' => 'Professional Services', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '5900', 'account_name' => 'Office Supplies', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6000', 'account_name' => 'Travel & Transportation', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6100', 'account_name' => 'Training & Development', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6200', 'account_name' => 'Marketing & Advertising', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6300', 'account_name' => 'Legal & Professional Fees', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6400', 'account_name' => 'Bank Charges', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6500', 'account_name' => 'Interest Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6600', 'account_name' => 'Bad Debt Expense', 'account_type' => 'expense', 'parent_account_id' => null],
            ['account_code' => '6700', 'account_name' => 'Miscellaneous Expense', 'account_type' => 'expense', 'parent_account_id' => null]
        ];

        foreach ($default_accounts as $account) {
            // Check if account already exists
            if (!$this->isCodeExists($account['account_code'])) {
                $account['is_active'] = 1;
                $account['created_at'] = date('Y-m-d H:i:s');
                $this->add($account);
            }
        }

        return true;
    }

    /**
     * Get accounts for specific modules
     */
    public function getRevenueAccounts() {
        return $this->getAccountsByType('revenue');
    }

    public function getExpenseAccounts() {
        return $this->getAccountsByType('expense');
    }

    public function getAssetAccounts() {
        return $this->getAccountsByType('asset');
    }

    public function getLiabilityAccounts() {
        return $this->getAccountsByType('liability');
    }

    public function getEquityAccounts() {
        return $this->getAccountsByType('equity');
    }

    /**
     * Get cash and bank accounts
     */
    public function getCashAccounts() {
        $this->db->select('id, account_code, account_name');
        $this->db->from('chart_of_accounts');
        $this->db->where_in('account_code', ['1000', '1010', '1020', '1050']);
        $this->db->where('is_active', 1);
        $this->db->order_by('account_code');
        
        return $this->db->get()->result_array();
    }

    /**
     * Search accounts
     */
    public function searchAccounts($search_term) {
        $this->db->select('id, account_code, account_name, account_type');
        $this->db->from('chart_of_accounts');
        $this->db->where('is_active', 1);
        $this->db->group_start();
        $this->db->like('account_code', $search_term);
        $this->db->or_like('account_name', $search_term);
        $this->db->group_end();
        $this->db->order_by('account_code');
        $this->db->limit(20);
        
        return $this->db->get()->result_array();
    }

    /**
     * Get account summary with balance
     */
    public function getAccountSummary($as_of_date = null) {
        if (!$as_of_date) {
            $as_of_date = date('Y-m-d');
        }

        $this->db->select('
            coa.id,
            coa.account_code,
            coa.account_name,
            coa.account_type,
            COALESCE(SUM(jl.debit_amount), 0) as total_debits,
            COALESCE(SUM(jl.credit_amount), 0) as total_credits
        ');
        $this->db->from('chart_of_accounts coa');
        $this->db->join('journal_lines jl', 'jl.account_id = coa.id', 'left');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id', 'left');
        $this->db->where('coa.is_active', 1);
        $this->db->where('(jh.date IS NULL OR jh.date <=', $as_of_date . ')');
        $this->db->group_by('coa.id, coa.account_code, coa.account_name, coa.account_type');
        $this->db->order_by('coa.account_code');

        $query = $this->db->get();
        $results = $query->result_array();

        // Calculate balances
        foreach ($results as &$account) {
            $net_balance = $account['total_debits'] - $account['total_credits'];
            
            // For liability, equity, and revenue accounts, credit balance is positive
            if (in_array($account['account_type'], ['liability', 'equity', 'revenue'])) {
                $account['balance'] = -$net_balance;
            } else {
                // For asset and expense accounts, debit balance is positive
                $account['balance'] = $net_balance;
            }
        }

        return $results;
    }

    /**
     * Validate account hierarchy (prevent circular references)
     */
    public function validateHierarchy($account_id, $parent_id) {
        if (!$parent_id || $account_id == $parent_id) {
            return false;
        }

        // Check if parent_id is a descendant of account_id
        $current_parent = $parent_id;
        $depth = 0;
        $max_depth = 10; // Prevent infinite loops

        while ($current_parent && $depth < $max_depth) {
            $this->db->select('parent_account_id');
            $this->db->where('id', $current_parent);
            $query = $this->db->get('chart_of_accounts');
            
            if ($query->num_rows() == 0) {
                break;
            }

            $row = $query->row();
            
            if ($row->parent_account_id == $account_id) {
                return false; // Circular reference detected
            }

            $current_parent = $row->parent_account_id;
            $depth++;
        }

        return true;
    }

    /**
     * Get account usage statistics
     */
    public function getAccountUsage($account_id) {
        $this->db->select('
            COUNT(jl.id) as transaction_count,
            SUM(jl.debit_amount + jl.credit_amount) as total_activity,
            MIN(jh.date) as first_transaction,
            MAX(jh.date) as last_transaction
        ');
        $this->db->from('journal_lines jl');
        $this->db->join('journal_headers jh', 'jh.id = jl.journal_id');
        $this->db->where('jl.account_id', $account_id);

        return $this->db->get()->row_array();
    }
}
?>