<?php
if (!defined('BASEPATH')) exit('No direct script access allowed');

/**
 * Billing Model
 * Handles all billing operations for the HMS system
 * Supports OPD, IPD, Pathology, Radiology, Pharmacy billing
 */
class Billing_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================================================
    // BILL MANAGEMENT
    // =============================================================================

    /**
     * Create a new bill
     */
    public function createBill($data) {
        // Generate bill number if not provided
        if (!isset($data['bill_number'])) {
            $data['bill_number'] = $this->generateBillNumber($data['module_type'] ?? 'GENERAL');
        }

        // Set default values
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['status'] = $data['status'] ?? 'draft';
        
        // Calculate balance amount
        $data['balance_amount'] = $data['total_amount'] - ($data['paid_amount'] ?? 0);

        $this->db->insert('bills', $data);
        $bill_id = $this->db->insert_id();

        if ($bill_id) {
            log_message('info', "Bill created: ID {$bill_id}, Number {$data['bill_number']}");
        }

        return $bill_id;
    }

    /**
     * Update an existing bill
     */
    public function updateBill($bill_id, $data) {
        $data['updated_at'] = date('Y-m-d H:i:s');
        
        // Recalculate balance if amounts are updated
        if (isset($data['total_amount']) || isset($data['paid_amount'])) {
            $current_bill = $this->getBillById($bill_id);
            if ($current_bill) {
                $total = $data['total_amount'] ?? $current_bill['total_amount'];
                $paid = $data['paid_amount'] ?? $current_bill['paid_amount'];
                $data['balance_amount'] = $total - $paid;
                
                // Update status based on payment
                if ($data['balance_amount'] <= 0) {
                    $data['status'] = 'paid';
                } elseif ($paid > 0) {
                    $data['status'] = 'partial';
                }
            }
        }

        $this->db->where('id', $bill_id);
        return $this->db->update('bills', $data);
    }

    /**
     * Get bill by ID with patient details
     */
    public function getBillById($bill_id) {
        $this->db->select('
            b.*,
            p.patient_name,
            p.mobileno as patient_phone,
            p.email as patient_email,
            p.address as patient_address,
            p.age,
            p.gender,
            CONCAT(s.name, " ", s.surname) as created_by_name
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->join('staff s', 's.id = b.created_by', 'left');
        $this->db->where('b.id', $bill_id);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get bill by bill number
     */
    public function getBillByNumber($bill_number) {
        $this->db->select('
            b.*,
            p.patient_name,
            p.mobileno as patient_phone,
            p.email as patient_email
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->where('b.bill_number', $bill_number);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get bills by patient ID
     */
    public function getBillsByPatient($patient_id, $limit = null, $offset = 0) {
        $this->db->select('
            b.*,
            p.patient_name,
            (SELECT SUM(amount) FROM payments WHERE bill_id = b.id) as total_payments
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->where('b.patient_id', $patient_id);
        $this->db->order_by('b.bill_date', 'DESC');
        
        if ($limit) {
            $this->db->limit($limit, $offset);
        }
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get bills with filters and pagination
     */
    public function getBills($filters = [], $limit = 50, $offset = 0) {
        $this->db->select('
            b.*,
            p.patient_name,
            p.mobileno as patient_phone,
            CONCAT(s.name, " ", s.surname) as created_by_name,
            (SELECT SUM(amount) FROM payments WHERE bill_id = b.id) as total_payments
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->join('staff s', 's.id = b.created_by', 'left');

        // Apply filters
        if (!empty($filters['status'])) {
            $this->db->where('b.status', $filters['status']);
        }
        
        if (!empty($filters['module_type'])) {
            $this->db->where('b.module_type', $filters['module_type']);
        }
        
        if (!empty($filters['date_from'])) {
            $this->db->where('b.bill_date >=', $filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $this->db->where('b.bill_date <=', $filters['date_to']);
        }
        
        if (!empty($filters['patient_search'])) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $filters['patient_search']);
            $this->db->or_like('p.mobileno', $filters['patient_search']);
            $this->db->or_like('b.bill_number', $filters['patient_search']);
            $this->db->group_end();
        }

        $this->db->order_by('b.created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get bills count with filters
     */
    public function getBillsCount($filters = []) {
        $this->db->select('COUNT(*) as count');
        $this->db->from('bills b');
        
        if (!empty($filters['patient_search'])) {
            $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        }

        // Apply same filters as getBills method
        if (!empty($filters['status'])) {
            $this->db->where('b.status', $filters['status']);
        }
        
        if (!empty($filters['module_type'])) {
            $this->db->where('b.module_type', $filters['module_type']);
        }
        
        if (!empty($filters['date_from'])) {
            $this->db->where('b.bill_date >=', $filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $this->db->where('b.bill_date <=', $filters['date_to']);
        }
        
        if (!empty($filters['patient_search'])) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $filters['patient_search']);
            $this->db->or_like('p.mobileno', $filters['patient_search']);
            $this->db->or_like('b.bill_number', $filters['patient_search']);
            $this->db->group_end();
        }
        
        $query = $this->db->get();
        $result = $query->row_array();
        return $result['count'];
    }

    /**
     * Delete a bill and its related items
     */
    public function deleteBill($bill_id) {
        $this->db->trans_start();

        // Delete bill items first
        $this->db->where('bill_id', $bill_id);
        $this->db->delete('bill_items');

        // Delete payments (optional - you might want to keep payment history)
        // $this->db->where('bill_id', $bill_id);
        // $this->db->delete('payments');

        // Delete the bill
        $this->db->where('id', $bill_id);
        $this->db->delete('bills');

        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    // =============================================================================
    // BILL ITEMS MANAGEMENT
    // =============================================================================

    /**
     * Add item to bill
     */
    public function addBillItem($bill_id, $item_data) {
        $item_data['bill_id'] = $bill_id;
        $item_data['total_price'] = $item_data['quantity'] * $item_data['unit_price'];
        $item_data['created_at'] = date('Y-m-d H:i:s');

        $this->db->insert('bill_items', $item_data);
        $item_id = $this->db->insert_id();

        // Update bill totals
        $this->updateBillTotals($bill_id);

        return $item_id;
    }

    /**
     * Update bill item
     */
    public function updateBillItem($item_id, $item_data) {
        if (isset($item_data['quantity']) || isset($item_data['unit_price'])) {
            $current_item = $this->getBillItemById($item_id);
            if ($current_item) {
                $quantity = $item_data['quantity'] ?? $current_item['quantity'];
                $unit_price = $item_data['unit_price'] ?? $current_item['unit_price'];
                $item_data['total_price'] = $quantity * $unit_price;
            }
        }

        $this->db->where('id', $item_id);
        $result = $this->db->update('bill_items', $item_data);

        if ($result) {
            // Get bill_id to update totals
            $item = $this->getBillItemById($item_id);
            if ($item) {
                $this->updateBillTotals($item['bill_id']);
            }
        }

        return $result;
    }

    /**
     * Get bill item by ID
     */
    public function getBillItemById($item_id) {
        $this->db->select('*');
        $this->db->from('bill_items');
        $this->db->where('id', $item_id);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get items for a bill
     */
    public function getBillItems($bill_id) {
        $this->db->select('*');
        $this->db->from('bill_items');
        $this->db->where('bill_id', $bill_id);
        $this->db->order_by('id', 'ASC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Delete bill item
     */
    public function deleteBillItem($item_id) {
        $item = $this->getBillItemById($item_id);
        
        $this->db->where('id', $item_id);
        $result = $this->db->delete('bill_items');

        if ($result && $item) {
            $this->updateBillTotals($item['bill_id']);
        }

        return $result;
    }

    /**
     * Update bill totals based on items
     */
    private function updateBillTotals($bill_id) {
        // Calculate subtotal from items
        $this->db->select('SUM(total_price) as subtotal');
        $this->db->from('bill_items');
        $this->db->where('bill_id', $bill_id);
        $query = $this->db->get();
        $result = $query->row_array();
        $subtotal = $result['subtotal'] ?? 0;

        // Get current bill data for tax calculation
        $bill = $this->getBillById($bill_id);
        $tax_rate = 0; // You can implement tax calculation based on your requirements
        $tax_amount = $subtotal * ($tax_rate / 100);
        $total_amount = $subtotal + $tax_amount;

        // Update bill
        $update_data = [
            'subtotal' => $subtotal,
            'tax_amount' => $tax_amount,
            'total_amount' => $total_amount,
            'balance_amount' => $total_amount - ($bill['paid_amount'] ?? 0),
            'updated_at' => date('Y-m-d H:i:s')
        ];

        $this->db->where('id', $bill_id);
        return $this->db->update('bills', $update_data);
    }

    // =============================================================================
    // BILLING REPORTS AND ANALYTICS
    // =============================================================================

    /**
     * Get billing summary for a date range
     */
    public function getBillingSummary($date_from = null, $date_to = null, $module_type = null) {
        if (!$date_from) $date_from = date('Y-m-01');
        if (!$date_to) $date_to = date('Y-m-d');

        $this->db->select('
            COUNT(*) as total_bills,
            SUM(total_amount) as total_billed,
            SUM(paid_amount) as total_paid,
            SUM(balance_amount) as total_outstanding,
            AVG(total_amount) as average_bill_amount,
            COUNT(CASE WHEN status = "paid" THEN 1 END) as paid_bills,
            COUNT(CASE WHEN status = "unpaid" THEN 1 END) as unpaid_bills,
            COUNT(CASE WHEN status = "partial" THEN 1 END) as partial_bills
        ');
        $this->db->from('bills');
        $this->db->where('bill_date >=', $date_from);
        $this->db->where('bill_date <=', $date_to);
        
        if ($module_type) {
            $this->db->where('module_type', $module_type);
        }
        
        $query = $this->db->get();
        return $query->row_array();
    }

    /**
     * Get billing summary by module
     */
    public function getBillingSummaryByModule($date_from = null, $date_to = null) {
        if (!$date_from) $date_from = date('Y-m-01');
        if (!$date_to) $date_to = date('Y-m-d');

        $this->db->select('
            module_type,
            COUNT(*) as bill_count,
            SUM(total_amount) as total_amount,
            SUM(paid_amount) as paid_amount,
            SUM(balance_amount) as outstanding_amount
        ');
        $this->db->from('bills');
        $this->db->where('bill_date >=', $date_from);
        $this->db->where('bill_date <=', $date_to);
        $this->db->group_by('module_type');
        $this->db->order_by('total_amount', 'DESC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get daily billing report
     */
    public function getDailyBillingReport($date_from = null, $date_to = null) {
        if (!$date_from) $date_from = date('Y-m-01');
        if (!$date_to) $date_to = date('Y-m-d');

        $this->db->select('
            DATE(bill_date) as date,
            COUNT(*) as bill_count,
            SUM(total_amount) as daily_total,
            SUM(paid_amount) as daily_paid,
            AVG(total_amount) as average_bill
        ');
        $this->db->from('bills');
        $this->db->where('bill_date >=', $date_from);
        $this->db->where('bill_date <=', $date_to);
        $this->db->group_by('DATE(bill_date)');
        $this->db->order_by('date', 'ASC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get overdue bills
     */
    public function getOverdueBills($days_overdue = 30) {
        $overdue_date = date('Y-m-d', strtotime("-{$days_overdue} days"));
        
        $this->db->select('
            b.*,
            p.patient_name,
            p.mobileno as patient_phone,
            DATEDIFF(CURDATE(), b.due_date) as days_overdue
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->where('b.status !=', 'paid');
        $this->db->where('b.due_date <', $overdue_date);
        $this->db->where('b.balance_amount >', 0);
        $this->db->order_by('b.due_date', 'ASC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Get top patients by billing amount
     */
    public function getTopPatientsByBilling($limit = 10, $date_from = null, $date_to = null) {
        if (!$date_from) $date_from = date('Y-m-01');
        if (!$date_to) $date_to = date('Y-m-d');

        $this->db->select('
            p.id as patient_id,
            p.patient_name,
            p.mobileno,
            COUNT(b.id) as total_bills,
            SUM(b.total_amount) as total_billed,
            SUM(b.paid_amount) as total_paid,
            SUM(b.balance_amount) as total_outstanding
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->where('b.bill_date >=', $date_from);
        $this->db->where('b.bill_date <=', $date_to);
        $this->db->group_by('b.patient_id');
        $this->db->order_by('total_billed', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================================================
    // UTILITY METHODS
    // =============================================================================

    /**
     * Generate unique bill number
     */
    public function generateBillNumber($module_type = 'GENERAL') {
        $prefix = $this->getBillPrefix($module_type);
        $date_part = date('Ymd');
        
        // Get the last bill number for today
        $this->db->select('bill_number');
        $this->db->from('bills');
        $this->db->like('bill_number', $prefix . $date_part, 'after');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        
        $query = $this->db->get();
        $last_bill = $query->row_array();
        
        if ($last_bill) {
            // Extract sequence number and increment
            $last_number = $last_bill['bill_number'];
            $sequence_part = substr($last_number, -4);
            $new_sequence = str_pad((int)$sequence_part + 1, 4, '0', STR_PAD_LEFT);
        } else {
            $new_sequence = '0001';
        }
        
        return $prefix . $date_part . $new_sequence;
    }

    /**
     * Get bill prefix based on module type
     */
    private function getBillPrefix($module_type) {
        $prefixes = [
            'opd' => 'OPD',
            'ipd' => 'IPD',
            'pathology' => 'LAB',
            'radiology' => 'RAD',
            'pharmacy' => 'PHR',
            'ambulance' => 'AMB'
        ];
        
        return $prefixes[strtolower($module_type)] ?? 'GEN';
    }

    /**
     * Get bill status options
     */
    public function getBillStatusOptions() {
        return [
            'draft' => 'Draft',
            'sent' => 'Sent',
            'paid' => 'Paid',
            'unpaid' => 'Unpaid',
            'partial' => 'Partially Paid',
            'overdue' => 'Overdue',
            'cancelled' => 'Cancelled'
        ];
    }

    /**
     * Get payment method options
     */
    public function getPaymentMethodOptions() {
        return [
            'cash' => 'Cash',
            'card' => 'Card',
            'bank_transfer' => 'Bank Transfer',
            'cheque' => 'Cheque',
            'mobile_money' => 'Mobile Money',
            'insurance' => 'Insurance'
        ];
    }

    /**
     * Update bill status based on payments
     */
    public function updateBillStatus($bill_id) {
        $bill = $this->getBillById($bill_id);
        
        if (!$bill) return false;
        
        $total_payments = $this->getTotalPaymentsForBill($bill_id);
        $balance = $bill['total_amount'] - $total_payments;
        
        if ($balance <= 0) {
            $status = 'paid';
        } elseif ($total_payments > 0) {
            $status = 'partial';
        } elseif ($bill['due_date'] && strtotime($bill['due_date']) < time()) {
            $status = 'overdue';
        } else {
            $status = 'unpaid';
        }
        
        return $this->updateBill($bill_id, [
            'paid_amount' => $total_payments,
            'balance_amount' => $balance,
            'status' => $status
        ]);
    }

    /**
     * Get total payments for a bill
     */
    public function getTotalPaymentsForBill($bill_id) {
        $this->db->select('SUM(amount) as total');
        $this->db->from('payments');
        $this->db->where('bill_id', $bill_id);
        
        $query = $this->db->get();
        $result = $query->row_array();
        
        return $result['total'] ?? 0;
    }

    /**
     * Search bills
     */
    public function searchBills($search_term, $limit = 20) {
        $this->db->select('
            b.*,
            p.patient_name,
            p.mobileno as patient_phone
        ');
        $this->db->from('bills b');
        $this->db->join('patients p', 'p.id = b.patient_id', 'left');
        $this->db->group_start();
        $this->db->like('b.bill_number', $search_term);
        $this->db->or_like('p.patient_name', $search_term);
        $this->db->or_like('p.mobileno', $search_term);
        $this->db->group_end();
        $this->db->order_by('b.created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    /**
     * Create bill from module data (OPD, IPD, etc.)
     */
    public function createBillFromModule($module_type, $module_data) {
        $bill_data = [
            'patient_id' => $module_data['patient_id'],
            'bill_date' => date('Y-m-d'),
            'due_date' => date('Y-m-d', strtotime('+30 days')),
            'module_type' => $module_type,
            'module_id' => $module_data['id'],
            'subtotal' => $module_data['amount'] ?? 0,
            'tax_amount' => 0,
            'total_amount' => $module_data['amount'] ?? 0,
            'paid_amount' => 0,
            'balance_amount' => $module_data['amount'] ?? 0,
            'status' => 'unpaid',
            'notes' => $module_data['notes'] ?? "Bill generated from {$module_type}",
            'created_by' => $module_data['created_by'] ?? 1
        ];

        return $this->createBill($bill_data);
    }

    /**
     * Get module-specific billing data
     */
    public function getModuleBillingData($module_type, $module_id) {
        $this->db->select('*');
        $this->db->from('bills');
        $this->db->where('module_type', $module_type);
        $this->db->where('module_id', $module_id);
        
        $query = $this->db->get();
        return $query->result_array();
    }
}
?>