<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Radiology extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->model('radiology_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
        } catch (Exception $e) {
            log_message('error', 'Radiology controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'radiology');

        // Prepare data for view
        $data['title'] = 'Radiology & Imaging with Biometric Verification';
        
        // Get radiology staff (technicians, radiologists)
        try {
            $data['radiology_staff'] = $this->getRadiologyStaff();
        } catch (Exception $e) {
            log_message('error', 'Error loading radiology staff: ' . $e->getMessage());
            $data['radiology_staff'] = [];
        }
        
        // Get imaging test types
        $data['imaging_types'] = $this->getImagingTypes();
        
        // Get gender list
        try {
            if (method_exists($this->customlib, 'getGender_Patient')) {
                $data['genderList'] = $this->customlib->getGender_Patient();
            } else {
                $data['genderList'] = array(
                    'male' => 'Male',
                    'female' => 'Female',
                    'other' => 'Other'
                );
            }
        } catch (Exception $e) {
            $data['genderList'] = array('male' => 'Male', 'female' => 'Female');
        }
        
        // Load views
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/radiology/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'View loading error: ' . $e->getMessage());
            show_error('Failed to load radiology view: ' . $e->getMessage());
        }
    }

    /**
     * Get all patients for radiology selection
     */
    public function getAllPatients() {
        try {
            $patients = $this->radiology_model->getAllPatientsForRadiology();
            
            $formatted_patients = array();
            foreach ($patients as $patient) {
                $fingerprint_status = '';
                if (!empty($patient['fingerprint_template'])) {
                    $fingerprint_status = '<span class="badge badge-success"><i class="fa fa-fingerprint"></i> Verified</span>';
                } else {
                    $fingerprint_status = '<span class="badge badge-warning"><i class="fa fa-exclamation-triangle"></i> No Biometric</span>';
                }
                
                $age_display = $patient['age'];
                if (!empty($patient['month'])) {
                    $age_display .= ' years ' . $patient['month'] . ' months';
                } else {
                    $age_display .= ' years';
                }
                
                $formatted_patients[] = array(
                    'id' => $patient['id'],
                    'patient_name' => $patient['patient_name'],
                    'guardian_name' => $patient['guardian_name'],
                    'age' => $age_display,
                    'gender' => ucfirst($patient['gender']),
                    'mobileno' => $patient['mobileno'],
                    //'patient_unique_id' => $patient['patient_unique_id'],
                    'fingerprint_status' => $fingerprint_status,
                    'last_visit' => $patient['last_visit'] ?? 'Never',
                    'action' => '<button class="btn btn-primary btn-sm" onclick="selectPatientForRadiology(' . $patient['id'] . ')">
                                    <i class="fa fa-user-plus"></i> Select
                                </button>'
                );
            }
            
            echo json_encode(array('data' => $formatted_patients));
        } catch (Exception $e) {
            log_message('error', 'Error getting patients for radiology: ' . $e->getMessage());
            echo json_encode(array('data' => array(), 'error' => $e->getMessage()));
        }
    }

    /**
     * Get patient details for radiology workflow
     */
public function getPatientDetails() {
    /*if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }*/
    
    $patient_id = $this->input->post('id');
    
    if (empty($patient_id)) {
        echo json_encode(['success' => false, 'message' => 'Patient ID required']);
        return;
    }
    
    try {
        // Get patient details including fingerprint status
        $this->db->select('
            id, patient_name, patient_unique_id, guardian_name, gender, 
            age, month, mobileno, dob, fingerprint_template, 
            fingerprint_verified, created_at
        ');
        $this->db->where('id', $patient_id);
        $query = $this->db->get('patients');
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            // Add fingerprint status
            $patient['has_fingerprint'] = !empty($patient['fingerprint_template']);
            $patient['fingerprint_verified'] = $patient['fingerprint_verified'] == 1;
            
            // Remove actual template from response for security
            unset($patient['fingerprint_template']);
            
            echo json_encode([
                'success' => true,
                'patient' => $patient
            ]);
        } else {
            echo json_encode(['success' => false, 'message' => 'Patient not found']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    }
}

public function getPatientData($id = null)
{
    header('Content-Type: application/json');

    if (empty($id)) {
        echo json_encode([
            'success' => false,
            'message' => 'Patient ID is required.'
        ]);
        return;
    }

    // Validate that ID is numeric
    if (!is_numeric($id)) {
        echo json_encode([
            'success' => false,
            'message' => 'Invalid Patient ID format.'
        ]);
        return;
    }

    // Fetch patient data
    $this->db->select('
        id, patient_name, patient_unique_id, guardian_name, gender, 
        age, month, day, mobileno, dob, email, 
        blood_group, address, image, fingerprint_template, fingerprint_image
    ');
    $this->db->where('id', $id);
    $query = $this->db->get('patients');

    if ($query->num_rows() > 0) {
        $patient = $query->row_array();

        // Optionally remove sensitive fields
        unset($patient['fingerprint_template']);

        echo json_encode([
            'success' => true
        ] + $patient); // Flat JSON
    } else {
        echo json_encode([
            'success' => false,
            'message' => 'Patient not found'
        ]);
    }
}

// In application/controllers/admin/Radiology.php
public function get_patients() {
    if (!$this->rbac->hasPrivilege('radiology', 'can_view')) {
        // Return JSON error instead of access_denied() which returns HTML
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Access denied']);
        return;
    }

    $search = $this->input->get('search', TRUE);
    
    try {
        $patients = $this->radiology_model->searchPatients($search);
        
        // Ensure we return JSON
        header('Content-Type: application/json');
        echo json_encode($patients);
    } catch (Exception $e) {
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Database error: ' . $e->getMessage()]);
    }
}

// Add this test method to your Radiology controller
public function test_json() {
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'message' => 'JSON response working',
        'timestamp' => date('Y-m-d H:i:s')
    ]);
}

public function getStoredFingerprintTemplate() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Get fingerprint data
        $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            if (!empty($patient['fingerprint_template'])) {
                $template = trim($patient['fingerprint_template']);
                
                echo json_encode([
                    'status' => 'success',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'fingerprint_template' => $template,
                    'fingerprint_image' => $patient['fingerprint_image'] ?? '',
                    'template_size' => strlen($template)
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No fingerprint template stored for this patient'
                ]);
            }
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}
    
    /**
 * Get patient fingerprint template for verification
 */
public function getPatientTemplate() {
    if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }
    
    $patient_id = $this->input->post('patient_id');
    
    if (empty($patient_id)) {
        echo json_encode(['success' => false, 'message' => 'Patient ID required']);
        return;
    }
    
    try {
        // Get patient fingerprint template from database
        $this->db->select('fingerprint_template, fingerprint_verified');
        $this->db->where('id', $patient_id);
        $query = $this->db->get('patients');
        
        if ($query->num_rows() > 0) {
            $patient = $query->row();
            
            if (!empty($patient->fingerprint_template)) {
                echo json_encode([
                    'success' => true,
                    'template' => $patient->fingerprint_template,
                    'verified' => $patient->fingerprint_verified == 1
                ]);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'No fingerprint template found for this patient'
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Patient not found']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Database error: ' . $e->getMessage()]);
    }
}

/**
 * Verify patient identity using biometric matching
 */
public function verifyPatientIdentity() {
    if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }
    
    $patient_id = $this->input->post('patient_id');
    $scanned_template = $this->input->post('scanned_template');
    
    if (empty($patient_id) || empty($scanned_template)) {
        echo json_encode(['success' => false, 'message' => 'Patient ID and scanned template required']);
        return;
    }
    
    try {
        // Get stored template
        $this->db->select('fingerprint_template, patient_name, patient_unique_id');
        $this->db->where('id', $patient_id);
        $query = $this->db->get('patients');
        
        if ($query->num_rows() > 0) {
            $patient = $query->row();
            
            if (!empty($patient->fingerprint_template)) {
                // In a real implementation, you would call SecuGen matching service here
                // For now, we'll simulate a match score based on template comparison
                $match_score = $this->simulateBiometricMatch($scanned_template, $patient->fingerprint_template);
                
                $verification_result = [
                    'success' => true,
                    'verified' => $match_score >= 50, // 50% threshold
                    'match_score' => $match_score / 100, // Convert to decimal
                    'patient_name' => $patient->patient_name,
                    'patient_id' => $patient->patient_unique_id
                ];
                
                // Log verification attempt
                $this->logVerificationAttempt($patient_id, $match_score, $verification_result['verified']);
                
                echo json_encode($verification_result);
            } else {
                echo json_encode([
                    'success' => false,
                    'message' => 'No stored fingerprint template for verification'
                ]);
            }
        } else {
            echo json_encode(['success' => false, 'message' => 'Patient not found']);
        }
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Verification error: ' . $e->getMessage()]);
    }
}

/**
 * Log manual verification override
 */
public function logManualVerification() {
    if (!$this->session->userdata('admin_id')) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized access']);
        return;
    }
    
    $patient_id = $this->input->post('patient_id');
    $patient_name = $this->input->post('patient_name');
    $verification_type = $this->input->post('verification_type');
    $user_id = $this->session->userdata('admin_id');
    
    try {
        // Insert verification log
        $log_data = [
            'patient_id' => $patient_id,
            'patient_name' => $patient_name,
            'verification_type' => $verification_type,
            'verified_by' => $user_id,
            'verification_method' => 'manual_override',
            'match_score' => null,
            'created_at' => date('Y-m-d H:i:s'),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent()
        ];
        
        $this->db->insert('radiology_verification_log', $log_data);
        
        echo json_encode(['success' => true, 'message' => 'Manual verification logged']);
        
    } catch (Exception $e) {
        echo json_encode(['success' => false, 'message' => 'Logging error: ' . $e->getMessage()]);
    }
}

    /**
     * Create new radiology order
     */
public function createRadiologyOrder() {
        // ALWAYS set JSON header first
        header('Content-Type: application/json');
        
        // Log that method was called
        log_message('debug', 'createRadiologyOrder method called');
        log_message('debug', 'POST data: ' . print_r($_POST, true));
        
        try {
            // Get POST data
            $patient_id = $this->input->post('patient_id');
            $referring_doctor_id = $this->input->post('referring_doctor_id');
            $imaging_tests = $this->input->post('imaging_tests');
            
            // Debug what we received
            $debug_data = [
                'patient_id' => $patient_id,
                'referring_doctor_id' => $referring_doctor_id,
                'imaging_tests' => $imaging_tests,
                'all_post' => $this->input->post()
            ];
            log_message('debug', 'Received data: ' . json_encode($debug_data));
            
            // Basic validation
            if (empty($patient_id)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Patient ID is required',
                    'debug' => $debug_data
                ]);
                return;
            }
            
            if (empty($referring_doctor_id)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Referring doctor is required',
                    'debug' => $debug_data
                ]);
                return;
            }
            
            if (empty($imaging_tests) || !is_array($imaging_tests)) {
                echo json_encode([
                    'success' => false,
                    'message' => 'Imaging tests are required and must be an array',
                    'debug' => $debug_data
                ]);
                return;
            }
            
            // Check if required tables exist
            if (!$this->db->table_exists('radiology_orders')) {
                echo json_encode([
                    'success' => false,
                    'message' => 'radiology_orders table does not exist'
                ]);
                return;
            }
            
            if (!$this->db->table_exists('radiology_order_tests')) {
                echo json_encode([
                    'success' => false,
                    'message' => 'radiology_order_tests table does not exist'
                ]);
                return;
            }
            
            // Try to insert - SIMPLIFIED VERSION
            $order_data = [
                'patient_id' => intval($patient_id),
                'referring_doctor_id' => intval($referring_doctor_id),
                'clinical_notes' => $this->input->post('clinical_notes') ?: '',
                'priority' => $this->input->post('priority') ?: 'routine',
                'status' => 'pending',
                'order_date' => date('Y-m-d H:i:s'),
                'verification_status' => 'pending',
                'total_amount' => floatval($this->input->post('total_amount') ?: 0),
                'created_by' => $this->session->userdata('admin_id') ?: 1,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];
            
            log_message('debug', 'Inserting order data: ' . json_encode($order_data));
            
            // Insert the order
            $this->db->insert('radiology_orders', $order_data);
            $order_id = $this->db->insert_id();
            
            if (!$order_id) {
                $db_error = $this->db->error();
                log_message('error', 'Failed to insert radiology order: ' . json_encode($db_error));
                
                echo json_encode([
                    'success' => false,
                    'message' => 'Failed to create order record',
                    'db_error' => $db_error
                ]);
                return;
            }
            
            // Insert tests
            $test_count = 0;
            foreach ($imaging_tests as $test_id) {
                $test_data = [
                    'radiology_order_id' => $order_id,
                    'test_id' => intval($test_id),
                    'status' => 'pending',
                    'created_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('radiology_order_tests', $test_data);
                if ($this->db->affected_rows() > 0) {
                    $test_count++;
                }
            }
            
            log_message('debug', "Order created: ID=$order_id, Tests=$test_count");
            
            // Return success
            echo json_encode([
                'success' => true,
                'status' => 'success',
                'message' => 'Radiology order created successfully',
                'order_id' => $order_id,
                'test_count' => $test_count
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Exception in createRadiologyOrder: ' . $e->getMessage());
            log_message('error', 'Stack trace: ' . $e->getTraceAsString());
            
            echo json_encode([
                'success' => false,
                'message' => 'Server exception: ' . $e->getMessage(),
                'type' => 'exception'
            ]);
        }
    }


// Helper method to process test IDs
private function processTestIds($imaging_tests) {
    $test_ids = array();
    
    foreach ($imaging_tests as $test_identifier) {
        // If it's already a numeric ID, use it
        if (is_numeric($test_identifier)) {
            $test_ids[] = intval($test_identifier);
        } else {
            // If it's a test code, look up the ID
            $test_id = $this->getTestIdByCode($test_identifier);
            if ($test_id) {
                $test_ids[] = $test_id;
            }
        }
    }
    
    return array_unique($test_ids);
}

// Helper method to get test ID by code
private function getTestIdByCode($test_code) {
    $this->db->select('id');
    $this->db->from('radiology_tests');
    $this->db->where('test_code', $test_code);
    $this->db->or_where('id', $test_code); // In case it's already an ID
    $query = $this->db->get();
    
    if ($query->num_rows() > 0) {
        return $query->row()->id;
    }
    
    return null;
}

// Helper method to validate patient exists
private function validatePatientExists($patient_id) {
    $this->db->select('id');
    $this->db->from('patients');
    $this->db->where('id', $patient_id);
    $query = $this->db->get();
    
    return $query->num_rows() > 0;
}

// Helper method to validate doctor exists
private function validateDoctorExists($doctor_id) {
    $this->db->select('id');
    $this->db->from('staff');
    $this->db->where('id', $doctor_id);
    $query = $this->db->get();
    
    return $query->num_rows() > 0;
}

// Helper method to get current staff ID
private function getStaffId() {
    // Get from session or return default
    return $this->session->userdata('admin_id') ?: 1;
}

// Helper method to log radiology activity
private function logRadiologyActivity($patient_id, $activity_type, $description) {
    try {
        if ($this->db->table_exists('radiology_activity_log')) {
            $log_data = array(
                'patient_id' => $patient_id,
                'staff_id' => $this->getStaffId(),
                'activity_type' => $activity_type,
                'description' => $description,
                'timestamp' => date('Y-m-d H:i:s'),
                'department' => 'radiology',
                'ip_address' => $this->input->ip_address()
            );
            
            $this->db->insert('radiology_activity_log', $log_data);
        }
    } catch (Exception $e) {
        log_message('error', 'Failed to log radiology activity: ' . $e->getMessage());
        // Don't fail the main operation if logging fails
    }
}

    /**
     * Get test IDs by their keys/codes
     */
    private function getTestIdsByKeys($test_keys) {
        try {
            $test_ids = array();
            
            // Define mapping of keys to test codes
            $key_to_code_mapping = array(
                'xray' => 'XRAY_CHEST',
                'ct_scan' => 'CT_HEAD',
                'mri' => 'MRI_BRAIN',
                'ultrasound' => 'US_ABD',
                'mammography' => 'MAMMO'
            );
            
            foreach ($test_keys as $key) {
                // Check if it's already a numeric ID
                if (is_numeric($key)) {
                    $test_ids[] = intval($key);
                    continue;
                }
                
                // Map key to test code
                $test_code = isset($key_to_code_mapping[$key]) ? $key_to_code_mapping[$key] : $key;
                
                // Get test ID from database
                $this->db->select('id');
                $this->db->from('radiology_tests');
                $this->db->where('test_code', $test_code);
                $this->db->where('is_active', 1);
                $query = $this->db->get();
                
                if ($query->num_rows() > 0) {
                    $test_ids[] = $query->row()->id;
                } else {
                    // Try to find by test name if code doesn't match
                    $this->db->select('id');
                    $this->db->from('radiology_tests');
                    $this->db->like('test_name', $key);
                    $this->db->where('is_active', 1);
                    $query = $this->db->get();
                    
                    if ($query->num_rows() > 0) {
                        $test_ids[] = $query->row()->id;
                    }
                }
            }
            
            return array_unique($test_ids);
        } catch (Exception $e) {
            log_message('error', 'Error getting test IDs by keys: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Verify patient identity using fingerprint
     */
    /*public function verifyPatientIdentity() {
        $patient_id = $this->input->post('patient_id');
        $scanned_template = $this->input->post('scanned_template');
        
        if (!$patient_id || !$scanned_template) {
            echo json_encode(array('success' => false, 'message' => 'Patient ID and fingerprint template are required'));
            return;
        }
        
        try {
            $patient = $this->radiology_model->getPatientById($patient_id);
            
            if (!$patient || empty($patient['fingerprint_template'])) {
                echo json_encode(array('success' => false, 'message' => 'No fingerprint template found for this patient'));
                return;
            }
            
            // Simple template comparison (in production, use SecuGen SDK)
            $match_score = $this->compareFingerprints($scanned_template, $patient['fingerprint_template']);
            $is_verified = $match_score >= 0.8; // 80% threshold
            
            // Log verification attempt
            $verification_data = array(
                'patient_id' => $patient_id,
                'verification_method' => 'fingerprint',
                'verification_result' => $is_verified ? 'success' : 'failed',
                'match_score' => $match_score,
                'verified_by' => $this->getStaffId(),
                'verification_timestamp' => date('Y-m-d H:i:s'),
                'department' => 'radiology'
            );
            
            $this->radiology_model->logVerificationAttempt($verification_data);
            
            if ($is_verified) {
                echo json_encode(array(
                    'success' => true, 
                    'verified' => true,
                    'message' => 'Patient identity verified successfully',
                    'match_score' => $match_score
                ));
            } else {
                echo json_encode(array(
                    'success' => true, 
                    'verified' => false,
                    'message' => 'Fingerprint verification failed',
                    'match_score' => $match_score
                ));
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error verifying patient identity: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'Error during verification: ' . $e->getMessage()));
        }
    }*/

    /**
     * Get radiology queue/appointments
     */
    public function getRadiologyQueue() {
        try {
            $queue_data = $this->radiology_model->getRadiologyQueue();
            
            $formatted_queue = array();
            foreach ($queue_data as $item) {
                $priority_badge = $this->getPriorityBadge($item['priority']);
                $status_badge = $this->getStatusBadge($item['status']);
                
                $actions = '<div class="btn-group">';
                if ($item['status'] == 'pending') {
                    $actions .= '<button class="btn btn-success btn-sm" onclick="startRadiologyExam(' . $item['order_id'] . ')">
                                    <i class="fa fa-play"></i> Start
                                </button>';
                }
                if ($item['status'] == 'in_progress') {
                    $actions .= '<button class="btn btn-info btn-sm" onclick="completeRadiologyExam(' . $item['order_id'] . ')">
                                    <i class="fa fa-check"></i> Complete
                                </button>';
                }
                $actions .= '<button class="btn btn-primary btn-sm" onclick="viewRadiologyDetails(' . $item['order_id'] . ')">
                                <i class="fa fa-eye"></i> View
                            </button>';
                $actions .= '</div>';
                
                $formatted_queue[] = array(
                    'order_id' => $item['order_id'],
                    'patient_name' => $item['patient_name'],
                    'patient_id' => $item['patient_unique_id'],
                    'tests' => $item['test_names'],
                    'priority' => $priority_badge,
                    'status' => $status_badge,
                    'scheduled_time' => date('H:i', strtotime($item['scheduled_date'])),
                    'referring_doctor' => $item['doctor_name'],
                    'actions' => $actions
                );
            }
            
            echo json_encode(array('data' => $formatted_queue));
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology queue: ' . $e->getMessage());
            echo json_encode(array('data' => array(), 'error' => $e->getMessage()));
        }
    }

    /**
     * Update radiology order status
     */
    public function updateOrderStatus() {
        $order_id = $this->input->post('order_id');
        $new_status = $this->input->post('status');
        $notes = $this->input->post('notes');
        
        if (!$order_id || !$new_status) {
            echo json_encode(array('success' => false, 'message' => 'Order ID and status are required'));
            return;
        }
        
        try {
            $update_data = array(
                'status' => $new_status,
                'updated_at' => date('Y-m-d H:i:s'),
                'updated_by' => $this->getStaffId()
            );
            
            if ($new_status == 'in_progress') {
                $update_data['started_at'] = date('Y-m-d H:i:s');
                $update_data['technician_id'] = $this->getStaffId();
            } elseif ($new_status == 'completed') {
                $update_data['completed_at'] = date('Y-m-d H:i:s');
            }
            
            if ($notes) {
                $update_data['technician_notes'] = $notes;
            }
            
            $result = $this->radiology_model->updateRadiologyOrder($order_id, $update_data);
            
            if ($result) {
                // Log status change
                $this->logRadiologyActivity(null, 'status_changed', 'Order #' . $order_id . ' status changed to ' . $new_status);
                
                echo json_encode(array('success' => true, 'message' => 'Order status updated successfully'));
            } else {
                echo json_encode(array('success' => false, 'message' => 'Failed to update order status'));
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error updating order status: ' . $e->getMessage());
            echo json_encode(array('success' => false, 'message' => 'Error updating status: ' . $e->getMessage()));
        }
    }

    // =============================================
    // HELPER METHODS
    // =============================================

    /**
     * Get radiology staff (technicians, radiologists)
     */
    private function getRadiologyStaff() {
        try {
            $this->db->select('s.*, r.name as role_name');
            $this->db->from('staff s');
            $this->db->join('staff_roles sr', 'sr.staff_id = s.id', 'left');
            $this->db->join('roles r', 'r.id = sr.role_id', 'left');
            $this->db->where_in('r.name', array('Radiologist', 'Radiology Technician', 'Doctor'));
           // $this->db->or_where('s.department', 'Radiology');
            $this->db->where('s.is_active', 1);
            $this->db->order_by('s.name', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology staff: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get available imaging test types
     */
    private function getImagingTypes() {
        try {
            // Get actual tests from database
            $tests = $this->radiology_model->getRadiologyTests();
            $imaging_types = array();
            
            foreach ($tests as $test) {
                $imaging_types[$test['id']] = array(
                    'id' => $test['id'],
                    'name' => $test['test_name'],
                    'description' => $test['description'],
                    'duration' => $test['estimated_duration'] . ' minutes',
                    'preparation' => $test['preparation_instructions'],
                    'category' => $test['category'],
                    'price' => $test['price'],
                    'requires_fasting' => $test['requires_fasting'],
                    'requires_contrast' => $test['requires_contrast']
                );
            }
            
            // If no tests in database, return default set
            if (empty($imaging_types)) {
                return array(
                    '1' => array(
                        'name' => 'X-Ray Imaging',
                        'description' => 'Standard radiographic imaging',
                        'duration' => '15 minutes',
                        'preparation' => 'Remove metal objects'
                    ),
                    '2' => array(
                        'name' => 'CT Scan',
                        'description' => 'Computed Tomography',
                        'duration' => '30 minutes',
                        'preparation' => 'Fasting may be required'
                    ),
                    '3' => array(
                        'name' => 'MRI',
                        'description' => 'Magnetic Resonance Imaging',
                        'duration' => '45 minutes',
                        'preparation' => 'Remove all metal objects'
                    ),
                    '4' => array(
                        'name' => 'Ultrasound',
                        'description' => 'Ultrasonography',
                        'duration' => '20 minutes',
                        'preparation' => 'May require full bladder'
                    ),
                    '5' => array(
                        'name' => 'Mammography',
                        'description' => 'Breast imaging',
                        'duration' => '20 minutes',
                        'preparation' => 'No deodorant or powder'
                    )
                );
            }
            
            return $imaging_types;
        } catch (Exception $e) {
            log_message('error', 'Error getting imaging types: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Compare fingerprint templates
     */
    private function compareFingerprints($template1, $template2) {
        // Simplified comparison - in production use SecuGen SDK
        $similarity = similar_text($template1, $template2, $percent);
        return $percent / 100;
    }

    /**
     * Get priority badge HTML
     */
    private function getPriorityBadge($priority) {
        switch ($priority) {
            case 'urgent':
                return '<span class="badge badge-danger">Urgent</span>';
            case 'stat':
                return '<span class="badge badge-warning">STAT</span>';
            case 'routine':
                return '<span class="badge badge-info">Routine</span>';
            default:
                return '<span class="badge badge-secondary">Normal</span>';
        }
    }

    /**
     * Get status badge HTML
     */
    private function getStatusBadge($status) {
        switch ($status) {
            case 'pending':
                return '<span class="badge badge-warning">Pending</span>';
            case 'in_progress':
                return '<span class="badge badge-info">In Progress</span>';
            case 'completed':
                return '<span class="badge badge-success">Completed</span>';
            case 'cancelled':
                return '<span class="badge badge-danger">Cancelled</span>';
            default:
                return '<span class="badge badge-secondary">' . ucfirst($status) . '</span>';
        }
    }

    /**
     * Log radiology activities
     */
    /*private function logRadiologyActivity($patient_id, $activity_type, $description) {
        try {
            $log_data = array(
                'patient_id' => $patient_id,
                'activity_type' => $activity_type,
                'description' => $description,
                'staff_id' => $this->getStaffId(),
                'timestamp' => date('Y-m-d H:i:s'),
                'department' => 'radiology'
            );
            
            $this->radiology_model->logActivity($log_data);
        } catch (Exception $e) {
            log_message('error', 'Error logging radiology activity: ' . $e->getMessage());
        }
    }*/

    /**
     * Test controller functionality
     */
    public function test() {
        echo "<h1>Radiology Controller Test</h1>";
        echo "<p>Controller is working!</p>";
        echo "<p>Base URL: " . base_url() . "</p>";
        echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
        
        // Test database connection
        try {
            $this->db->select('COUNT(*) as patient_count');
            $this->db->from('patients');
            $query = $this->db->get();
            $result = $query->row();
            echo "<p>Patients in database: " . $result->patient_count . "</p>";
            
        } catch (Exception $e) {
            echo "<p>Database error: " . $e->getMessage() . "</p>";
        }
    }
}