<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * Procurement Controller
 * Hospital Management System - Procurement Module
 * Handles requisitions, purchase orders, and procurement management
 */
class Procurement extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('procurement_model');
            $this->load->model('supplier_model');
            $this->load->model('items_model');
            $this->load->model('staff_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
            
            // Load procurement helper if it exists
            if (file_exists(APPPATH . 'helpers/procurement_helper.php')) {
                $this->load->helper('procurement_helper');
            }
        } catch (Exception $e) {
            log_message('error', 'Procurement controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    // =============================================
    // MAIN DASHBOARD
    // =============================================

    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'dashboard');

        // Prepare data for dashboard
        $data['title'] = 'Procurement Dashboard';
        
        try {
            // Get dashboard statistics
            $data['stats'] = [
                'pending_requisitions' => $this->procurement_model->countPendingRequisitions(),
                'pending_orders' => $this->procurement_model->countPendingOrders(),
                'pending_deliveries' => $this->procurement_model->countPendingDeliveries(),
                'low_stock_items' => $this->items_model->countLowStockItems(),
                'monthly_spend' => $this->procurement_model->getMonthlySpend(),
                'budget_utilization' => $this->procurement_model->getBudgetUtilization()
            ];

            // Get recent activities
            $data['recent_requisitions'] = $this->procurement_model->getRecentRequisitions(5);
            $data['recent_orders'] = $this->getRecentOrders(5);
            $data['urgent_items'] = $this->items_model->getUrgentItems(10);

        } catch (Exception $e) {
            log_message('error', 'Dashboard data loading error: ' . $e->getMessage());
            $data['stats'] = [
                'pending_requisitions' => 0,
                'pending_orders' => 0,
                'pending_deliveries' => 0,
                'low_stock_items' => 0,
                'monthly_spend' => 0,
                'budget_utilization' => ['utilization_percentage' => 0]
            ];
            $data['recent_requisitions'] = [];
            $data['recent_orders'] = [];
            $data['urgent_items'] = [];
        }
        
        // Load views
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/procurement/dashboard', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'View loading error: ' . $e->getMessage());
            show_error('Failed to load procurement dashboard: ' . $e->getMessage());
        }
    }

    // =============================================
    // REQUISITIONS MANAGEMENT
    // =============================================

public function requisitions() {
    try {
        // Enable error reporting for debugging
        if (ENVIRONMENT === 'development') {
            ini_set('display_errors', 1);
            error_reporting(E_ALL);
        }
        
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'requisitions');

        $data['title'] = 'Purchase Requisitions';
        
        // Debug: Log that we're entering the method
        log_message('debug', 'Entering requisitions method');
        
        try {
            // Get departments for dropdown - with fallback
            $data['departments'] = $this->getDepartmentsList();
            log_message('debug', 'Departments loaded: ' . count($data['departments']));
            
            // Get staff members for requestor dropdown - with fallback
            if (method_exists($this->staff_model, 'getStaffList')) {
                $data['staff'] = $this->staff_model->getStaffList();
            } else {
                $data['staff'] = [];
            }
            log_message('debug', 'Staff loaded: ' . count($data['staff']));

            // Get item categories - with fallback
            if (method_exists($this->items_model, 'getActiveCategories')) {
                $data['categories'] = $this->items_model->getActiveCategories();
            } else {
                $data['categories'] = [];
            }
            log_message('debug', 'Categories loaded: ' . count($data['categories']));

        } catch (Exception $e) {
            log_message('error', 'Error loading requisitions data: ' . $e->getMessage());
            // Provide fallback data
            $data['departments'] = $this->getDefaultDepartments();
            $data['staff'] = [];
            $data['categories'] = [];
        }

        // Debug: Check if view file exists
        $view_path = APPPATH . 'views/admin/procurement/requisitions.php';
        if (!file_exists($view_path)) {
            log_message('error', 'Requisitions view file not found: ' . $view_path);
            show_error('Requisitions view file not found. Please create the file: ' . $view_path);
            return;
        }

        log_message('debug', 'Loading requisitions view');

        // Load views with error handling
        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/requisitions', $data);
        $this->load->view('layout/footer', $data);
        
        log_message('debug', 'Requisitions view loaded successfully');

    } catch (Exception $e) {
        log_message('error', 'Fatal error in requisitions method: ' . $e->getMessage());
        log_message('error', 'Stack trace: ' . $e->getTraceAsString());
        show_error('Error loading requisitions page: ' . $e->getMessage());
    }
}

public function setup() {
    echo "<h1>Procurement Setup</h1>";
    
    // Create departments table
    try {
        $this->createDepartmentsTable();
        echo "<p>✓ Departments table created/verified</p>";
    } catch (Exception $e) {
        echo "<p>✗ Error creating departments table: " . $e->getMessage() . "</p>";
    }
    
    // Test departments loading
    try {
        $departments = $this->getDepartmentsList();
        echo "<p>✓ Departments loaded: " . count($departments) . "</p>";
        foreach ($departments as $dept) {
            echo "<p>&nbsp;&nbsp;- " . $dept['department_name'] . "</p>";
        }
    } catch (Exception $e) {
        echo "<p>✗ Error loading departments: " . $e->getMessage() . "</p>";
    }
    
    // Test staff loading
    try {
        $staff = $this->getStaffListFallback();
        echo "<p>✓ Staff loaded: " . count($staff) . "</p>";
        foreach ($staff as $member) {
            $name = ($member['name'] ?? '') . ' ' . ($member['surname'] ?? '');
            echo "<p>&nbsp;&nbsp;- " . trim($name) . " (" . ($member['employee_id'] ?? 'N/A') . ")</p>";
        }
    } catch (Exception $e) {
        echo "<p>✗ Error loading staff: " . $e->getMessage() . "</p>";
    }
    
    echo "<p><a href='" . base_url('admin/procurement/requisitions') . "'>Test Requisitions Page</a></p>";
    echo "<p><a href='" . base_url('admin/procurement') . "'>Back to Dashboard</a></p>";
}

private function getStaffListFallback() {
    try {
        if ($this->db->table_exists('staff')) {
            $this->db->select('id, name, surname, employee_id');
            $this->db->from('staff');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('name', 'asc');
            $query = $this->db->get();
            return $query->result_array();
        } else {
            // Return default staff if table doesn't exist
            return [
                ['id' => 1, 'name' => 'Admin', 'surname' => 'User', 'employee_id' => 'EMP001']
            ];
        }
    } catch (Exception $e) {
        log_message('error', 'Error in staff fallback: ' . $e->getMessage());
        return [
            ['id' => 1, 'name' => 'Admin', 'surname' => 'User', 'employee_id' => 'EMP001']
        ];
    }
}

    /**
     * Save new requisition or update existing
     */
    public function saveRequisition() {
        header('Content-Type: application/json');
        
        try {
            // Validate input
            $department_id = $this->input->post('department_id');
            $required_date = $this->input->post('required_date');
            $priority = $this->input->post('priority');
            $purpose = $this->input->post('purpose');
            $items = $this->input->post('items'); // Array of items
            
            if (!$department_id || !$required_date || !$items) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Please fill all required fields'
                ]);
                return;
            }

            // Generate requisition number
            $requisition_no = $this->generateRequisitionNumber();

            // Calculate total estimated cost
            $total_cost = 0;
            if (is_array($items)) {
                foreach ($items as $item) {
                    $quantity = isset($item['quantity']) ? (int)$item['quantity'] : 0;
                    $cost = isset($item['estimated_cost']) ? (float)$item['estimated_cost'] : 0;
                    $total_cost += ($quantity * $cost);
                }
            }

            // Prepare requisition data
            $requisition_data = [
                'requisition_no' => $requisition_no,
                'requesting_department_id' => $department_id,
                'requested_by' => $this->customlib->getStaffID(),
                'request_date' => date('Y-m-d'),
                'required_date' => $required_date,
                'priority' => $priority,
                'purpose' => $purpose,
                'justification' => $this->input->post('justification'),
                'total_estimated_cost' => $total_cost,
                'status' => 'pending_approval',
                'created_by' => $this->customlib->getStaffID()
            ];

            // Start transaction
            $this->db->trans_start();

            // Save requisition
            $requisition_id = $this->procurement_model->saveRequisition($requisition_data);

            if (!$requisition_id) {
                throw new Exception('Failed to save requisition');
            }

            // Save requisition items
            if (is_array($items)) {
                foreach ($items as $item) {
                    $item_data = [
                        'requisition_id' => $requisition_id,
                        'item_id' => isset($item['item_id']) ? $item['item_id'] : null,
                        'quantity_requested' => isset($item['quantity']) ? $item['quantity'] : 0,
                        'unit_cost_estimated' => isset($item['estimated_cost']) ? $item['estimated_cost'] : 0,
                        'total_cost_estimated' => (isset($item['quantity']) ? $item['quantity'] : 0) * (isset($item['estimated_cost']) ? $item['estimated_cost'] : 0),
                        'urgency_notes' => isset($item['notes']) ? $item['notes'] : '',
                        'specifications' => isset($item['specifications']) ? $item['specifications'] : ''
                    ];

                    $this->procurement_model->saveRequisitionItem($item_data);
                }
            }

            // Complete transaction
            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Transaction failed');
            }

            // Send notification for approval workflow
            $this->notifyApprovers($requisition_id, 'requisition');

            echo json_encode([
                'status' => 'success',
                'message' => 'Requisition saved successfully',
                'requisition_id' => $requisition_id,
                'requisition_no' => $requisition_no
            ]);

        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Save requisition error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save requisition: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get requisitions list for DataTable
     */
    public function getRequisitionsList() {
        header('Content-Type: application/json');
        
        try {
            $start = $this->input->post('start') ?: 0;
            $length = $this->input->post('length') ?: 10;
            $search = $this->input->post('search')['value'] ?? '';
            $orderColumn = $this->input->post('order')[0]['column'] ?? 0;
            $orderDir = $this->input->post('order')[0]['dir'] ?? 'desc';

            $result = $this->procurement_model->getRequisitionsList($start, $length, $search, $orderColumn, $orderDir);

            echo json_encode([
                'draw' => intval($this->input->post('draw')),
                'recordsTotal' => $result['total'],
                'recordsFiltered' => $result['filtered'],
                'data' => $result['data']
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Approve or reject requisition
     */
    public function approveRequisition() {
        header('Content-Type: application/json');
        
        try {
            $requisition_id = $this->input->post('requisition_id');
            $action = $this->input->post('action'); // 'approve' or 'reject'
            $comments = $this->input->post('comments');

            if (!$requisition_id || !$action) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invalid parameters'
                ]);
                return;
            }

            $approval_data = [
                'approved_by' => $this->customlib->getStaffID(),
                'approved_date' => date('Y-m-d H:i:s'),
                'approval_comments' => $comments,
                'status' => $action === 'approve' ? 'approved' : 'rejected'
            ];

            $result = $this->procurement_model->updateRequisition($requisition_id, $approval_data);

            if ($result) {
                // Log approval in workflow table
                $this->logApproval('requisition', $requisition_id, $action, $comments);

                echo json_encode([
                    'status' => 'success',
                    'message' => 'Requisition ' . $action . 'd successfully'
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update requisition'
                ]);
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Approval failed: ' . $e->getMessage()
            ]);
        }
    }

    // =============================================
    // PURCHASE ORDERS (LPO) MANAGEMENT
    // =============================================

    public function purchaseOrders() {
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'purchase_orders');

        $data['title'] = 'Local Purchase Orders (LPO)';
        
        try {
            // Get suppliers for dropdown
            $data['suppliers'] = $this->supplier_model->getActiveSuppliers();
            
            // Get approved requisitions for conversion to PO
            $data['approved_requisitions'] = $this->getApprovedRequisitions();

        } catch (Exception $e) {
            log_message('error', 'Purchase orders view data error: ' . $e->getMessage());
            $data['suppliers'] = [];
            $data['approved_requisitions'] = [];
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/purchase_orders', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get purchase orders list for DataTable
     */
    public function getPurchaseOrdersList() {
        header('Content-Type: application/json');
        
        try {
            $start = $this->input->post('start') ?: 0;
            $length = $this->input->post('length') ?: 10;
            $search = $this->input->post('search')['value'] ?? '';

            // For now, return empty data since PO functionality is not fully implemented
            echo json_encode([
                'draw' => intval($this->input->post('draw')),
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => []
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Create LPO from approved requisition
     */
    public function createLPOFromRequisition() {
        header('Content-Type: application/json');
        
        try {
            $requisition_id = $this->input->post('requisition_id');
            $supplier_id = $this->input->post('supplier_id');
            $expected_delivery_date = $this->input->post('expected_delivery_date');
            $payment_terms = $this->input->post('payment_terms');
            $delivery_address = $this->input->post('delivery_address');

            if (!$requisition_id || !$supplier_id) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Requisition and supplier are required'
                ]);
                return;
            }

            // For now, return success message - full implementation requires purchase_orders table
            echo json_encode([
                'status' => 'success',
                'message' => 'LPO creation functionality will be available once database tables are set up',
                'lpo_id' => 0,
                'lpo_number' => 'LPO' . date('Ymd') . '001'
            ]);

        } catch (Exception $e) {
            log_message('error', 'Create LPO error: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to create LPO: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Generate and print LPO
     */
    public function printLPO($lpo_id) {
        try {
            // For now, show a simple message
            echo "<h1>LPO Print View</h1>";
            echo "<p>LPO ID: {$lpo_id}</p>";
            echo "<p>Print functionality will be implemented once LPO tables are set up.</p>";

        } catch (Exception $e) {
            log_message('error', 'Print LPO error: ' . $e->getMessage());
            show_error('Failed to generate LPO: ' . $e->getMessage());
        }
    }

    // =============================================
    // GOODS RECEIVED NOTES (GRN)
    // =============================================

    public function goodsReceived() {
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'goods_received');

        $data['title'] = 'Goods Received Notes (GRN)';
        
        try {
            // Get pending purchase orders for GRN creation
            $data['pending_orders'] = $this->getPendingOrders();

        } catch (Exception $e) {
            log_message('error', 'GRN view data error: ' . $e->getMessage());
            $data['pending_orders'] = [];
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/goods_received', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // SUPPLIERS MANAGEMENT
    // =============================================

    public function suppliers() {
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'suppliers');

        $data['title'] = 'Suppliers Management';
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/suppliers', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Save supplier
     */
    public function saveSupplier() {
        header('Content-Type: application/json');
        
        try {
            $supplier_data = [
                'supplier_name' => $this->input->post('supplier_name'),
                'supplier_code' => $this->input->post('supplier_code'),
                'contact_person' => $this->input->post('contact_person'),
                'email' => $this->input->post('email'),
                'phone' => $this->input->post('phone'),
                'address' => $this->input->post('address'),
                'city' => $this->input->post('city'),
                'state' => $this->input->post('state'),
                'postal_code' => $this->input->post('postal_code'),
                'country' => $this->input->post('country') ?: 'Kenya',
                'tax_number' => $this->input->post('tax_number'),
                'payment_terms' => $this->input->post('payment_terms'),
                'credit_limit' => $this->input->post('credit_limit') ?: 0,
                'supplier_category' => $this->input->post('supplier_category'),
                'created_by' => $this->customlib->getStaffID()
            ];

            $supplier_id = $this->supplier_model->saveSupplier($supplier_data);

            if ($supplier_id) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Supplier saved successfully',
                    'supplier_id' => $supplier_id
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to save supplier'
                ]);
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error saving supplier: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * AJAX method to get suppliers list
     */
    public function getSuppliersList() {
        header('Content-Type: application/json');
        
        try {
            $start = $this->input->post('start') ?: 0;
            $length = $this->input->post('length') ?: 10;
            $search = $this->input->post('search')['value'] ?? '';
            $orderColumn = $this->input->post('order')[0]['column'] ?? 0;
            $orderDir = $this->input->post('order')[0]['dir'] ?? 'desc';

            $result = $this->supplier_model->getSuppliersList($start, $length, $search, $orderColumn, $orderDir);

            echo json_encode([
                'draw' => intval($this->input->post('draw')),
                'recordsTotal' => $result['total'],
                'recordsFiltered' => $result['filtered'],
                'data' => $result['data']
            ]);

        } catch (Exception $e) {
            echo json_encode([
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ]);
        }
    }

    // =============================================
    // ITEMS MANAGEMENT
    // =============================================

    public function items() {
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'items');

        $data['title'] = 'Items Management';
        
        try {
            $data['categories'] = $this->items_model->getActiveCategories();
            $data['items'] = $this->items_model->getItemsForSelect(); // Get items for display
        } catch (Exception $e) {
            $data['categories'] = [];
            $data['items'] = [];
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/items', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Save new item
     */
    public function saveItem() {
        header('Content-Type: application/json');
        
        try {
            $item_data = [
                'item_name' => $this->input->post('item_name'),
                'item_code' => $this->input->post('item_code'),
                'description' => $this->input->post('description'),
                'category_id' => $this->input->post('category_id'),
                'unit_of_measure' => $this->input->post('unit_of_measure'),
                'unit_cost' => $this->input->post('unit_cost') ?: 0,
                'current_stock' => $this->input->post('current_stock') ?: 0,
                'minimum_stock' => $this->input->post('minimum_stock') ?: 0,
                'reorder_level' => $this->input->post('reorder_level') ?: 0,
                'created_by' => $this->customlib->getStaffID()
            ];

            $item_id = $this->items_model->saveItem($item_data);

            if ($item_id) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Item saved successfully',
                    'item_id' => $item_id
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to save item'
                ]);
            }

        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Error saving item: ' . $e->getMessage()
            ]);
        }
    }

    /**
     * Get items for select dropdown (AJAX endpoint)
     */
    public function getItemsForSelect() {
        header('Content-Type: application/json');
        
        try {
            $items = $this->items_model->getItemsForSelect();
            echo json_encode([
                'status' => 'success',
                'items' => $items
            ]);
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => $e->getMessage()
            ]);
        }
    }

    // =============================================
    // REPORTS
    // =============================================

    public function reports() {
        $this->session->set_userdata('top_menu', 'Procurement');
        $this->session->set_userdata('sub_menu', 'reports');

        $data['title'] = 'Procurement Reports';
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/reports', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Generate unique requisition number
     */
    private function generateRequisitionNumber() {
        $prefix = 'REQ';
        $year = date('Y');
        $month = date('m');
        
        // Get next sequence number
        try {
            $this->db->select('COUNT(*) + 1 as next_seq');
            $this->db->from('purchase_requisitions');
            $this->db->where('YEAR(request_date)', $year);
            $this->db->where('MONTH(request_date)', $month);
            $query = $this->db->get();
            $sequence = $query->row()->next_seq;
        } catch (Exception $e) {
            $sequence = 1;
        }
        
        return $prefix . $year . $month . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate unique LPO number
     */
    private function generateLPONumber() {
        $prefix = 'LPO';
        $year = date('Y');
        $month = date('m');
        
        // Get next sequence number
        try {
            $this->db->select('COUNT(*) + 1 as next_seq');
            $this->db->from('purchase_orders');
            $this->db->where('YEAR(order_date)', $year);
            $this->db->where('MONTH(order_date)', $month);
            $query = $this->db->get();
            $sequence = $query->row()->next_seq;
        } catch (Exception $e) {
            $sequence = 1;
        }
        
        return $prefix . $year . $month . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }

    /**
     * Generate unique GRN number
     */
    private function generateGRNNumber() {
        $prefix = 'GRN';
        $year = date('Y');
        $month = date('m');
        
        // Get next sequence number
        try {
            $this->db->select('COUNT(*) + 1 as next_seq');
            $this->db->from('goods_received_notes');
            $this->db->where('YEAR(received_date)', $year);
            $this->db->where('MONTH(received_date)', $month);
            $query = $this->db->get();
            $sequence = $query->row()->next_seq;
        } catch (Exception $e) {
            $sequence = 1;
        }
        
        return $prefix . $year . $month . str_pad($sequence, 4, '0', STR_PAD_LEFT);
    }
    
    private function createDepartmentsTable() {
    try {
        $sql = "CREATE TABLE IF NOT EXISTS `departments` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department_name` varchar(100) NOT NULL,
            `department_code` varchar(20) UNIQUE NOT NULL,
            `head_of_department` int(11) DEFAULT NULL,
            `budget_allocated` decimal(15,2) DEFAULT 0.00,
            `is_active` enum('yes','no') DEFAULT 'yes',
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        $this->db->query($sql);
        
        // Insert default departments
        $departments = [
            ['department_name' => 'Pharmacy', 'department_code' => 'PHARM'],
            ['department_name' => 'Laboratory', 'department_code' => 'LAB'],
            ['department_name' => 'Radiology', 'department_code' => 'RAD'],
            ['department_name' => 'Surgery', 'department_code' => 'SURG'],
            ['department_name' => 'Emergency', 'department_code' => 'EMER'],
            ['department_name' => 'Administration', 'department_code' => 'ADMIN'],
            ['department_name' => 'Maintenance', 'department_code' => 'MAINT']
        ];
        
        $this->db->insert_batch('departments', $departments);
        log_message('info', 'Departments table created and populated');
        
    } catch (Exception $e) {
        log_message('error', 'Failed to create departments table: ' . $e->getMessage());
    }
}

    /**
     * Get departments list
     */
    private function getDepartmentsList() {
    try {
        // Check if departments table exists
        if (!$this->db->table_exists('departments')) {
            log_message('info', 'Departments table does not exist, creating default departments');
            $this->createDepartmentsTable();
        }
        
        $this->db->select('id, department_name, department_code');
        $this->db->from('departments');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('department_name', 'ASC');
        $query = $this->db->get();
        
        $departments = $query->result_array();
        
        // If no departments found, return defaults
        if (empty($departments)) {
            return $this->getDefaultDepartments();
        }
        
        return $departments;
        
    } catch (Exception $e) {
        log_message('error', 'Error loading departments: ' . $e->getMessage());
        return $this->getDefaultDepartments();
    }
}

    /**
     * Get default departments if table doesn't exist
     */
    private function getDefaultDepartments() {
        return [
            ['id' => 1, 'department_name' => 'Pharmacy', 'department_code' => 'PHARM'],
            ['id' => 2, 'department_name' => 'Laboratory', 'department_code' => 'LAB'],
            ['id' => 3, 'department_name' => 'Radiology', 'department_code' => 'RAD'],
            ['id' => 4, 'department_name' => 'Surgery', 'department_code' => 'SURG'],
            ['id' => 5, 'department_name' => 'Emergency', 'department_code' => 'EMER'],
            ['id' => 6, 'department_name' => 'Administration', 'department_code' => 'ADMIN']
        ];
    }

    /**
     * Get approved requisitions
     */
    private function getApprovedRequisitions() {
        try {
            return $this->procurement_model->getApprovedRequisitions();
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Get recent orders
     */
    private function getRecentOrders($limit = 5) {
        try {
            return $this->procurement_model->getRecentOrders($limit);
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Get pending orders
     */
    private function getPendingOrders() {
        try {
            return $this->procurement_model->getPendingOrders();
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Notify approvers for workflow
     */
    private function notifyApprovers($document_id, $document_type) {
        // Implementation depends on your notification system
        // This could send emails, SMS, or internal notifications
        log_message('info', "Approval notification sent for {$document_type} ID: {$document_id}");
    }

    /**
     * Log approval action
     */
    private function logApproval($document_type, $document_id, $action, $comments) {
        try {
            if ($this->db->table_exists('procurement_approvals')) {
                $approval_data = [
                    'document_type' => $document_type,
                    'document_id' => $document_id,
                    'approval_level' => 1,
                    'approver_id' => $this->customlib->getStaffID(),
                    'approval_status' => $action === 'approve' ? 'approved' : 'rejected',
                    'approval_date' => date('Y-m-d H:i:s'),
                    'comments' => $comments
                ];

                $this->db->insert('procurement_approvals', $approval_data);
            }
        } catch (Exception $e) {
            log_message('error', 'Failed to log approval: ' . $e->getMessage());
        }
    }

    /**
     * Check and update PO status based on received quantities
     */
    private function checkAndUpdatePOStatus($po_id) {
        try {
            if ($this->db->table_exists('purchase_order_items')) {
                // Get PO items status
                $this->db->select('SUM(quantity_ordered) as total_ordered, SUM(quantity_received) as total_received');
                $this->db->from('purchase_order_items');
                $this->db->where('purchase_order_id', $po_id);
                $query = $this->db->get();
                $result = $query->row();

                if ($result && $result->total_received >= $result->total_ordered) {
                    // Fully received
                    $this->procurement_model->updatePurchaseOrder($po_id, ['status' => 'fully_received']);
                } else if ($result && $result->total_received > 0) {
                    // Partially received
                    $this->procurement_model->updatePurchaseOrder($po_id, ['status' => 'partially_received']);
               }
           }
       } catch (Exception $e) {
           log_message('error', 'Failed to update PO status: ' . $e->getMessage());
       }
   }

   /**
    * Get hospital details for printing
    */
   private function getHospitalDetails() {
       // Implementation depends on your system configuration
       return [
           'name' => $this->config->item('name') ?: 'Hospital Name',
           'address' => 'Hospital Address',
           'phone' => 'Hospital Phone',
           'email' => 'Hospital Email'
       ];
   }

   // =============================================
   // ADDITIONAL AJAX ENDPOINTS
   // =============================================

   /**
    * Get requisition items for a specific requisition
    */
   public function getRequisitionItems($requisition_id) {
       header('Content-Type: application/json');
       
       try {
           $items = $this->procurement_model->getRequisitionItems($requisition_id);
           echo json_encode([
               'status' => 'success',
               'items' => $items
           ]);
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => $e->getMessage()
           ]);
       }
   }

   /**
 * View requisition details
 */
public function viewRequisition($requisition_id) {
    try {
        if (!$requisition_id) {
            show_error('Requisition ID is required', 400);
            return;
        }

        $data['title'] = 'Requisition Details';
        
        // Get requisition details
        $data['requisition'] = $this->procurement_model->getRequisitionDetails($requisition_id);
        
        if (empty($data['requisition'])) {
            show_error('Requisition not found', 404);
            return;
        }
        
        // Get requisition items
        $data['requisition_items'] = $this->procurement_model->getRequisitionItems($requisition_id);

        $this->load->view('layout/header', $data);
        $this->load->view('admin/procurement/view_requisition', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Error viewing requisition: ' . $e->getMessage());
        show_error('Failed to load requisition details: ' . $e->getMessage());
    }
}

   /**
    * Print requisition
    */
   public function printRequisition($requisition_id) {
       try {
           $data['requisition'] = $this->procurement_model->getRequisitionDetails($requisition_id);
           $data['requisition_items'] = $this->procurement_model->getRequisitionItems($requisition_id);
           $data['hospital'] = $this->getHospitalDetails();

           $this->load->view('admin/procurement/print_requisition', $data);
       } catch (Exception $e) {
           show_error('Failed to print requisition: ' . $e->getMessage());
       }
   }

   /**
    * Get dashboard statistics (AJAX)
    */
   public function getDashboardStats() {
       header('Content-Type: application/json');
       
       try {
           $stats = [
               'pending_requisitions' => $this->procurement_model->countPendingRequisitions(),
               'pending_orders' => $this->procurement_model->countPendingOrders(),
               'pending_deliveries' => $this->procurement_model->countPendingDeliveries(),
               'low_stock_items' => $this->items_model->countLowStockItems(),
               'monthly_spend' => $this->procurement_model->getMonthlySpend()
           ];

           echo json_encode([
               'status' => 'success',
               'stats' => $stats
           ]);
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => $e->getMessage()
           ]);
       }
   }

   /**
    * Search items (AJAX autocomplete)
    */
   public function searchItems() {
       header('Content-Type: application/json');
       
       try {
           $search_term = $this->input->get('term');
           $items = $this->items_model->searchItems($search_term, 10);
           
           $result = [];
           foreach ($items as $item) {
               $result[] = [
                   'id' => $item['id'],
                   'label' => $item['item_code'] . ' - ' . $item['item_name'],
                   'value' => $item['item_name'],
                   'unit_cost' => $item['unit_cost'],
                   'unit_of_measure' => $item['unit_of_measure']
               ];
           }

           echo json_encode($result);
       } catch (Exception $e) {
           echo json_encode([]);
       }
   }

   /**
    * Get supplier details (AJAX)
    */
   public function getSupplierDetails($supplier_id) {
       header('Content-Type: application/json');
       
       try {
           $supplier = $this->supplier_model->getSupplierDetails($supplier_id);
           echo json_encode([
               'status' => 'success',
               'supplier' => $supplier
           ]);
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => $e->getMessage()
           ]);
       }
   }

   /**
    * Get item details (AJAX)
    */
   public function getItemDetails($item_id) {
       header('Content-Type: application/json');
       
       try {
           $item = $this->items_model->getItemDetails($item_id);
           echo json_encode([
               'status' => 'success',
               'item' => $item
           ]);
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => $e->getMessage()
           ]);
       }
   }

   // =============================================
   // BULK OPERATIONS
   // =============================================

   /**
    * Bulk approve requisitions
    */
   public function bulkApproveRequisitions() {
       header('Content-Type: application/json');
       
       try {
           $requisition_ids = $this->input->post('requisition_ids');
           $comments = $this->input->post('comments');

           if (!$requisition_ids || !is_array($requisition_ids)) {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'No requisitions selected'
               ]);
               return;
           }

           $approved_count = 0;
           foreach ($requisition_ids as $id) {
               $approval_data = [
                   'approved_by' => $this->customlib->getStaffID(),
                   'approved_date' => date('Y-m-d H:i:s'),
                   'approval_comments' => $comments,
                   'status' => 'approved'
               ];

               if ($this->procurement_model->updateRequisition($id, $approval_data)) {
                   $this->logApproval('requisition', $id, 'approve', $comments);
                   $approved_count++;
               }
           }

           echo json_encode([
               'status' => 'success',
               'message' => "{$approved_count} requisitions approved successfully"
           ]);

       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => 'Bulk approval failed: ' . $e->getMessage()
           ]);
       }
   }

   // =============================================
   // EXPORT FUNCTIONS
   // =============================================

   /**
    * Export requisitions to CSV
    */
   public function exportRequisitions() {
       try {
           $this->load->helper('download');
           
           $requisitions = $this->procurement_model->getRequisitionsList(0, 1000, '', 0, 'desc');
           
           $csv_data = "Requisition No,Department,Requested By,Date,Required Date,Priority,Status,Amount\n";
           
           foreach ($requisitions['data'] as $req) {
               $csv_data .= '"' . ($req['requisition_no'] ?? '') . '",';
               $csv_data .= '"' . ($req['department_name'] ?? '') . '",';
               $csv_data .= '"' . ($req['requested_by_name'] ?? '') . '",';
               $csv_data .= '"' . ($req['request_date'] ?? '') . '",';
               $csv_data .= '"' . ($req['required_date'] ?? '') . '",';
               $csv_data .= '"' . ($req['priority'] ?? '') . '",';
               $csv_data .= '"' . ($req['status'] ?? '') . '",';
               $csv_data .= '"' . ($req['total_estimated_cost'] ?? '0') . '"' . "\n";
           }
           
           $filename = 'requisitions_' . date('Y-m-d_H-i-s') . '.csv';
           force_download($filename, $csv_data);
           
       } catch (Exception $e) {
           show_error('Failed to export requisitions: ' . $e->getMessage());
       }
   }

   /**
    * Export suppliers to CSV
    */
   public function exportSuppliers() {
       try {
           $this->load->helper('download');
           
           $suppliers = $this->supplier_model->getSuppliersList(0, 1000, '', 0, 'desc');
           
           $csv_data = "Code,Name,Contact Person,Email,Phone,City,Category,Status\n";
           
           foreach ($suppliers['data'] as $supplier) {
               $csv_data .= '"' . ($supplier['supplier_code'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['supplier_name'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['contact_person'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['email'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['phone'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['city'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['supplier_category'] ?? '') . '",';
               $csv_data .= '"' . ($supplier['is_active'] ?? '') . '"' . "\n";
           }
           
           $filename = 'suppliers_' . date('Y-m-d_H-i-s') . '.csv';
           force_download($filename, $csv_data);
           
       } catch (Exception $e) {
           show_error('Failed to export suppliers: ' . $e->getMessage());
       }
   }

   // =============================================
   // SETTINGS AND CONFIGURATION
   // =============================================

   /**
    * Procurement settings page
    */
   public function settings() {
       $this->session->set_userdata('top_menu', 'Procurement');
       $this->session->set_userdata('sub_menu', 'settings');

       $data['title'] = 'Procurement Settings';
       
       $this->load->view('layout/header', $data);
       $this->load->view('admin/procurement/settings', $data);
       $this->load->view('layout/footer', $data);
   }

   /**
    * Update procurement settings
    */
   public function updateSettings() {
       header('Content-Type: application/json');
       
       try {
           // Implementation for updating procurement settings
           echo json_encode([
               'status' => 'success',
               'message' => 'Settings updated successfully'
           ]);
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => 'Failed to update settings: ' . $e->getMessage()
           ]);
       }
   }

   // =============================================
   // DEBUG AND TESTING METHODS
   // =============================================

   /**
    * Test procurement system
    */
   public function test() {
       echo "<h1>Procurement Module Test</h1>";
       echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
       
       // Test database connections
       echo "<h2>Database Tests</h2>";
       
       $tables = ['suppliers', 'items', 'item_categories', 'purchase_requisitions'];
       foreach ($tables as $table) {
           $exists = $this->db->table_exists($table);
           echo "<p>{$table} table: " . ($exists ? '✓ Exists' : '✗ Missing') . "</p>";
           
           if ($exists) {
               $count = $this->db->count_all($table);
               echo "<p>&nbsp;&nbsp;Records: {$count}</p>";
           }
       }
       
       // Test models
       echo "<h2>Model Tests</h2>";
       try {
           $suppliers = $this->supplier_model->getActiveSuppliers();
           echo "<p>✓ Supplier model working - " . count($suppliers) . " suppliers found</p>";
       } catch (Exception $e) {
           echo "<p>✗ Supplier model error: " . $e->getMessage() . "</p>";
       }
       
       try {
           $categories = $this->items_model->getActiveCategories();
           echo "<p>✓ Items model working - " . count($categories) . " categories found</p>";
       } catch (Exception $e) {
           echo "<p>✗ Items model error: " . $e->getMessage() . "</p>";
       }
       
       echo "<h2>Helper Functions</h2>";
       if (function_exists('format_currency')) {
           echo "<p>✓ Procurement helper loaded</p>";
           echo "<p>✓ Currency format test: " . format_currency(1500.75) . "</p>";
       } else {
           echo "<p>✗ Procurement helper not loaded</p>";
       }
       
       echo "<h2>Navigation</h2>";
       echo "<p><a href='" . base_url('admin/procurement') . "'>Go to Procurement Dashboard</a></p>";
       echo "<p><a href='" . base_url('admin/procurement/requisitions') . "'>Go to Requisitions</a></p>";
       echo "<p><a href='" . base_url('admin/procurement/suppliers') . "'>Go to Suppliers</a></p>";
       echo "<p><a href='" . base_url('admin/procurement/items') . "'>Go to Items</a></p>";
   }
}