<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Loyalty_points extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load required models and libraries
        $this->load->model('loyalty_points_model');
        $this->load->model('patient_model');
        $this->load->model('staff_model');
        $this->load->library('datatables');
        $this->load->library('customlib');
    }

    /**
     * Main loyalty points page
     */
    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'Loyalty Points');
        $this->session->set_userdata('sub_menu', 'loyalty_points');

        // Prepare data for view
        $data['title'] = 'Loyalty Points Management';
        
        // Get current user info
        $data['current_user'] = $this->getStaffInfo();
        
        // Load view
        $this->load->view('layout/header', $data);
        $this->load->view('admin/loyalty_points/index', $data);
        $this->load->view('layout/footer', $data);
    }

    /**
     * Get loyalty points for a patient
     */
    public function get_loyalty_points() {
        header('Content-Type: application/json');
        
        $patient_id = $this->input->get('patient_id');
        
        if (!$patient_id) {
            echo json_encode(['success' => false, 'message' => 'Patient ID required']);
            return;
        }

        $loyalty_data = $this->loyalty_points_model->getPatientLoyaltyBalance($patient_id);
        echo json_encode(['success' => true, 'data' => $loyalty_data]);
    }

    /**
     * Get loyalty history for a patient
     */
    public function get_loyalty_history() {
        header('Content-Type: application/json');
        
        $patient_id = $this->input->get('patient_id');
        $limit = $this->input->get('limit', 20);
        
        if (!$patient_id) {
            echo json_encode(['success' => false, 'message' => 'Patient ID required']);
            return;
        }

        $history = $this->loyalty_points_model->getPatientLoyaltyHistory($patient_id, $limit);
        echo json_encode(['success' => true, 'data' => $history]);
    }

    /**
     * Award loyalty points
     */
    public function award_points() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $points = $this->input->post('points');
            $reason = $this->input->post('reason');
            $reference_type = $this->input->post('reference_type');
            $reference_id = $this->input->post('reference_id');

            // Validate required fields
            if (empty($patient_id) || empty($points) || empty($reason)) {
                echo json_encode(['success' => false, 'message' => 'Please fill all required fields']);
                return;
            }

            // Validate points amount
            $max_points = $this->loyalty_points_model->getSettingValue('max_points_per_transaction', 100);
            if ($points > $max_points) {
                echo json_encode(['success' => false, 'message' => 'Maximum points per transaction is ' . $max_points]);
                return;
            }

            // Prepare data for awarding points
            $award_data = [
                'patient_id' => $patient_id,
                'transaction_type' => 'earned',
                'points' => abs($points),
                'description' => $reason,
                'reference_type' => $reference_type ?: 'general',
                'reference_id' => $reference_id,
                'transaction_date' => date('Y-m-d'),
                'staff_id' => $this->getStaffId(),
                'created_at' => date('Y-m-d H:i:s')
            ];

            // Award points
            $result = $this->loyalty_points_model->awardPoints($award_data);

            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Points awarded successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to award points']);
            }

        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Redeem loyalty points
     */
    public function redeem_points() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $points = $this->input->post('points');
            $service = $this->input->post('service');
            $value = $this->input->post('value');
            $notes = $this->input->post('notes');

            // Validate required fields
            if (empty($patient_id) || empty($points) || empty($service)) {
                echo json_encode(['success' => false, 'message' => 'Please fill all required fields']);
                return;
            }

            // Redeem points
            $result = $this->loyalty_points_model->redeemPoints(
                $patient_id, 
                $points, 
                $service, 
                $value, 
                $notes, 
                $this->getStaffId()
            );

            echo json_encode($result);

        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Get all loyalty activities for DataTables
     */
    public function get_all_activities() {
        header('Content-Type: application/json');
        
        $request = $this->input->post();
        $result = $this->loyalty_points_model->getLoyaltyActivitiesForDataTables($request);
        
        // Format data for DataTables
        $data = [];
        foreach ($result['data'] as $row) {
            $transaction_class = $this->getTransactionClass($row['transaction_type']);
            $points_display = $row['transaction_type'] == 'earned' ? '+' . $row['points'] : $row['points'];
            
            $data[] = [
                'transaction_date' => date('Y-m-d', strtotime($row['transaction_date'])),
                'patient_name' => $row['patient_name'],
                'transaction_type' => '<span class="label ' . $transaction_class . '">' . ucfirst($row['transaction_type']) . '</span>',
                'points' => '<span class="' . ($row['transaction_type'] == 'earned' ? 'text-success' : 'text-danger') . '">' . $points_display . '</span>',
                'description' => $row['description'],
                'staff_name' => $row['staff_name'] ?: 'System',
                'reference' => $row['reference_type'] . ($row['reference_id'] ? ' #' . $row['reference_id'] : '')
            ];
        }
        
        echo json_encode([
            'draw' => $result['draw'],
            'recordsTotal' => $result['recordsTotal'],
            'recordsFiltered' => $result['recordsFiltered'],
            'data' => $data
        ]);
    }

    /**
     * Get dashboard statistics
     */
    public function get_dashboard_stats() {
        header('Content-Type: application/json');
        
        $stats = $this->loyalty_points_model->getDashboardStats();
        echo json_encode(['success' => true, 'data' => $stats]);
    }

    /**
     * Search patients for loyalty program
     */
    public function search_patients() {
        header('Content-Type: application/json');
        
        $search_term = $this->input->get('q');
        $page = $this->input->get('page', 1);
        $per_page = 10;
        
        $patients = $this->loyalty_points_model->searchPatients($search_term, $per_page);
        
        $items = [];
        foreach ($patients as $patient) {
            $items[] = [
                'id' => $patient['id'],
                'text' => $patient['patient_name'] . ' (' . $patient['id'] . ') - ' . $patient['mobileno'],
                'patient_name' => $patient['patient_name'],
                'mobileno' => $patient['mobileno'],
                'age' => $patient['age'],
                'gender' => $patient['gender'],
                'current_balance' => $patient['current_balance']
            ];
        }
        
        echo json_encode([
            'items' => $items,
            'total_count' => count($items)
        ]);
    }

    /**
     * Get loyalty program settings
     */
    public function get_settings() {
        header('Content-Type: application/json');
        
        $settings = $this->loyalty_points_model->getSettings();
        echo json_encode(['success' => true, 'data' => $settings]);
    }

    /**
     * Update loyalty program settings
     */
    public function update_settings() {
        header('Content-Type: application/json');
        
        try {
            $settings = $this->input->post('settings');
            
            if (empty($settings)) {
                echo json_encode(['success' => false, 'message' => 'No settings provided']);
                return;
            }

            // Validate settings
            $valid_settings = [
                'points_per_1000_kes',
                'point_value_kes',
                'minimum_redemption_points',
                'max_points_per_transaction'
            ];

            $filtered_settings = [];
            foreach ($settings as $key => $value) {
                if (in_array($key, $valid_settings)) {
                    $filtered_settings[$key] = $value;
                }
            }

            // Update settings
            $result = $this->loyalty_points_model->updateSettings($filtered_settings);

            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Settings updated successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to update settings']);
            }

        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Get loyalty leaderboard
     */
    public function get_leaderboard() {
        header('Content-Type: application/json');
        
        $limit = $this->input->get('limit', 10);
        $leaderboard = $this->loyalty_points_model->getLeaderboard($limit);
        
        echo json_encode(['success' => true, 'data' => $leaderboard]);
    }

    /**
     * Export loyalty data
     */
    public function export_data() {
        $type = $this->input->get('type', 'activities');
        $filters = [
            'transaction_type' => $this->input->get('transaction_type'),
            'date_from' => $this->input->get('date_from'),
            'date_to' => $this->input->get('date_to'),
            'patient_name' => $this->input->get('patient_name')
        ];
        
        if ($type == 'activities') {
            $this->exportLoyaltyActivities($filters);
        } else {
            $this->exportLoyaltyLeaderboard();
        }
    }

    /**
     * Export loyalty activities
     */
    private function exportLoyaltyActivities($filters) {
        $activities = $this->loyalty_points_model->getAllLoyaltyActivities(1000, 0, $filters);
        
        // Set headers for CSV download
        $filename = 'loyalty_activities_' . date('Y-m-d') . '.csv';
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // CSV Headers
        fputcsv($output, [
            'Date', 'Patient Name', 'Transaction Type', 'Points', 'Description', 
            'Staff', 'Reference', 'Amount Spent', 'Service Value'
        ]);
        
        // CSV Data
        foreach ($activities as $activity) {
            fputcsv($output, [
                $activity['transaction_date'],
                $activity['patient_name'],
                ucfirst($activity['transaction_type']),
                $activity['points'],
                $activity['description'],
                $activity['staff_name'] . ' ' . $activity['staff_surname'],
                $activity['reference_type'] . ($activity['reference_id'] ? ' #' . $activity['reference_id'] : ''),
                $activity['amount_spent'] ?: '',
                $activity['redeemed_value'] ?: ''
            ]);
        }
        
        fclose($output);
    }

    /**
     * Export loyalty leaderboard
     */
    private function exportLoyaltyLeaderboard() {
        $leaderboard = $this->loyalty_points_model->getLeaderboard(50);
        
        // Set headers for CSV download
        $filename = 'loyalty_leaderboard_' . date('Y-m-d') . '.csv';
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment; filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        
        // CSV Headers
        fputcsv($output, ['Rank', 'Patient Name', 'Total Points', 'Total Spent (KES)']);
        
        // CSV Data
        $rank = 1;
        foreach ($leaderboard as $member) {
            fputcsv($output, [
                $rank++,
                $member['patient_name'],
                $member['total_points'],
                number_format($member['total_spent'], 2)
            ]);
        }
        
        fclose($output);
    }

    /**
     * Adjust patient points (admin function)
     */
    public function adjust_points() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $points = $this->input->post('points');
            $reason = $this->input->post('reason');

            // Validate required fields
            if (empty($patient_id) || empty($points) || empty($reason)) {
                echo json_encode(['success' => false, 'message' => 'Please fill all required fields']);
                return;
            }

            // Adjust points
            $result = $this->loyalty_points_model->adjustPoints(
                $patient_id, 
                $points, 
                $reason, 
                $this->getStaffId()
            );

            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Points adjusted successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to adjust points']);
            }

        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Process automatic points awarding from payment
     */
    public function process_payment_points() {
        header('Content-Type: application/json');
        
        try {
            $patient_id = $this->input->post('patient_id');
            $amount = $this->input->post('amount');
            $reference_type = $this->input->post('reference_type', 'payment');
            $reference_id = $this->input->post('reference_id');

            if (empty($patient_id) || empty($amount)) {
                echo json_encode(['success' => false, 'message' => 'Patient ID and amount required']);
                return;
            }

            // Auto-award points
            $result = $this->loyalty_points_model->autoAwardPoints(
                $patient_id, 
                $amount, 
                $reference_type, 
                $reference_id, 
                $this->getStaffId()
            );

            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Points awarded automatically']);
            } else {
                echo json_encode(['success' => false, 'message' => 'No points awarded (amount too small or program disabled)']);
            }

        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Get patient points summary
     */
    public function get_patient_summary() {
        header('Content-Type: application/json');
        
        $patient_id = $this->input->get('patient_id');
        
        if (!$patient_id) {
            echo json_encode(['success' => false, 'message' => 'Patient ID required']);
            return;
        }

        $summary = $this->loyalty_points_model->getPatientPointsSummary($patient_id);
        echo json_encode(['success' => true, 'data' => $summary]);
    }

    /**
     * Expire old points (admin function)
     */
    public function expire_old_points() {
        header('Content-Type: application/json');
        
        try {
            $expired_count = $this->loyalty_points_model->expireOldPoints();
            echo json_encode([
                'success' => true, 
                'message' => $expired_count . ' points expired',
                'count' => $expired_count
            ]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    // Helper methods
    private function getStaffId() {
        try {
            if (method_exists($this->customlib, 'getStaffID')) {
                return $this->customlib->getStaffID();
            } else {
                return $this->session->userdata('admin_id') ?: 1;
            }
        } catch (Exception $e) {
            return 1;
        }
    }

    private function getStaffInfo() {
        try {
            $staff_id = $this->getStaffId();
            return $this->staff_model->get($staff_id);
        } catch (Exception $e) {
            return null;
        }
    }

    private function getTransactionClass($transaction_type) {
        $classes = [
            'earned' => 'label-success',
            'redeemed' => 'label-warning',
            'expired' => 'label-danger',
            'adjusted' => 'label-info'
        ];
        return isset($classes[$transaction_type]) ? $classes[$transaction_type] : 'label-default';
    }
}