<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Laboratory extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries and models
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
        } catch (Exception $e) {
            log_message('error', 'Laboratory controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

public function index() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'laboratory');

    $data['title'] = 'Laboratory Management - GCC & General Workflow';
    
    try {
        $data['lab_staff'] = $this->getLabStaff();
        $data['test_types'] = $this->getTestTypes();
        $data['gcc_tests'] = $this->getGCCTests(); // GCC specific tests
        
        // Get gender list
        if (method_exists($this->customlib, 'getGender_Patient')) {
            $data['genderList'] = $this->customlib->getGender_Patient();
        } else {
            $data['genderList'] = array(
                'male' => 'Male',
                'female' => 'Female',
                'other' => 'Other'
            );
        }
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/laboratory/index', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Laboratory view loading error: ' . $e->getMessage());
        show_error('Failed to load laboratory view: ' . $e->getMessage());
    }
}

/**
 * Get patients for laboratory with workflow information
 */
public function getLabPatients() {
    header('Content-Type: application/json');
    
    try {
        // Get patients who have laboratory appointments or are GCC patients
        $this->db->select('p.id, p.patient_name, p.mobileno, p.gender, p.age, p.patient_workflow, p.workflow_status, p.current_workflow_step');
        $this->db->from('patients p');
        $this->db->where('p.is_active', 'yes');
        
        // Join with service appointments to get lab appointments
        $this->db->join('service_appointments sa', 'p.id = sa.patient_id AND sa.service_type = "laboratory"', 'left');
        
        // Include GCC patients or patients with lab appointments
        $this->db->group_start();
        $this->db->where('p.patient_workflow', 'gcc');
        $this->db->or_where('sa.id IS NOT NULL');
        $this->db->group_end();
        
        $this->db->order_by('p.patient_name', 'ASC');
        $this->db->limit(200);
        
        $query = $this->db->get();
        
        if (!$query) {
            echo json_encode([
                'patients' => [],
                'count' => 0,
                'error' => 'Database query failed'
            ]);
            return;
        }
        
        $patients = $query->result_array();
        
        // Add additional info for each patient
        foreach ($patients as &$patient) {
            $patient['is_gcc'] = $patient['patient_workflow'] === 'gcc';
            $patient['lab_tests'] = $this->getPatientLabTests($patient['id']);
            $patient['has_pending_tests'] = $this->hasPendingLabTests($patient['id']);
        }
        
        echo json_encode([
            'patients' => $patients,
            'count' => count($patients),
            'message' => 'Found ' . count($patients) . ' patients for laboratory'
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'patients' => [],
            'count' => 0,
            'error' => 'Exception: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get patient fingerprint template for verification
 */
public function getPatientFingerprint() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        $action = $this->input->post('action');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        // Load the patient model
        $this->load->model('Patient_model');
        
        // Query the database for the patient's fingerprint template
        $this->db->select('fingerprint_template, patient_name, id');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $query = $this->db->get();
        
        if ($query->num_rows() == 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found'
            ]);
            return;
        }
        
        $patient = $query->row();
        
        // Check if fingerprint template exists
        if (empty($patient->fingerprint_template) || trim($patient->fingerprint_template) === '') {
            echo json_encode([
                'status' => 'error',
                'message' => 'No fingerprint template stored for this patient',
                'patient_name' => $patient->patient_name,
                'has_fingerprint' => false
            ]);
            return;
        }
        
        // Log the fingerprint retrieval attempt
        $this->logFingerprintAccess($patient_id, 'template_retrieved');
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Fingerprint template retrieved successfully',
            'fingerprint_template' => $patient->fingerprint_template,
            'patient_name' => $patient->patient_name,
            'template_length' => strlen($patient->fingerprint_template),
            'has_fingerprint' => true
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error retrieving patient fingerprint: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage()
        ]);
    }
}

/**
 * Log fingerprint access attempts for audit trail
 */
private function logFingerprintAccess($patient_id, $action) {
    try {
        $log_data = [
            'patient_id' => $patient_id,
            'action' => $action,
            'accessed_by' => $this->session->userdata('admin_id') ?? 'unknown',
            'access_timestamp' => date('Y-m-d H:i:s'),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent(),
            'module' => 'laboratory'
        ];
        
        // Insert into fingerprint access log (create table if needed)
        if ($this->db->table_exists('fingerprint_access_log')) {
            $this->db->insert('fingerprint_access_log', $log_data);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error logging fingerprint access: ' . $e->getMessage());
    }
}

/**
 * Alternative method: Get fingerprint template using direct query
 * Use this if you prefer a simpler approach
 */
public function getPatientFingerprintSimple() {
    header('Content-Type: application/json');
    
    $patient_id = $this->input->post('patient_id');
    
    if (!$patient_id) {
        echo json_encode(['status' => 'error', 'message' => 'Patient ID required']);
        return;
    }
    
    // Direct query to get fingerprint template
    $query = $this->db->select('fingerprint_template, patient_name')
                     ->from('patients')
                     ->where('id', $patient_id)
                     ->get();
    
    if ($query->num_rows() > 0) {
        $patient = $query->row();
        
        if (!empty($patient->fingerprint_template)) {
            echo json_encode([
                'status' => 'success',
                'fingerprint_template' => $patient->fingerprint_template,
                'patient_name' => $patient->patient_name
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'No fingerprint stored for this patient'
            ]);
        }
    } else {
        echo json_encode([
            'status' => 'error',
            'message' => 'Patient not found'
        ]);
    }
}

/**
 * Create fingerprint access log table (run once)
 */
public function createFingerprintAccessLogTable() {
    $sql = "CREATE TABLE IF NOT EXISTS `fingerprint_access_log` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `patient_id` int(11) NOT NULL,
        `action` varchar(50) NOT NULL,
        `accessed_by` varchar(50) DEFAULT NULL,
        `access_timestamp` datetime NOT NULL,
        `ip_address` varchar(45) DEFAULT NULL,
        `user_agent` text DEFAULT NULL,
        `module` varchar(50) DEFAULT NULL,
        PRIMARY KEY (`id`),
        KEY `patient_id` (`patient_id`),
        KEY `access_timestamp` (`access_timestamp`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
    
    if ($this->db->query($sql)) {
        echo "Fingerprint access log table created successfully";
    } else {
        echo "Error creating table: " . $this->db->error()['message'];
    }
}

/**
 * Get GCC tests for laboratory
 */
private function getGCCTests() {
    try {
        $this->db->select('*');
        $this->db->from('gcc_tests');
        $this->db->where_in('test_category', ['hematology', 'biochemistry', 'serology', 'clinical_pathology', 'microbiology']);
        $this->db->order_by('test_order', 'ASC');
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting GCC tests: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient lab tests
 */


/**
 * Check if patient has pending lab tests
 */
private function hasPendingLabTests($patient_id) {
    try {
        $patient_workflow = $this->getPatientWorkflow($patient_id);
        
        if ($patient_workflow === 'gcc') {
            $this->db->select('COUNT(*) as pending_count');
            $this->db->from('gcc_patient_tests gpt');
            $this->db->join('gcc_tests gt', 'gpt.gcc_test_id = gt.id');
            $this->db->where('gpt.patient_id', $patient_id);
            $this->db->where('gpt.status', 'pending');
            $this->db->where_in('gt.test_category', ['hematology', 'biochemistry', 'serology', 'clinical_pathology', 'microbiology']);
            
            $query = $this->db->get();
            $result = $query->row();
            
            return $result->pending_count > 0;
        } else {
            // Check general lab tests
            return $this->hasGeneralPendingTests($patient_id);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error checking pending lab tests: ' . $e->getMessage());
        return false;
    }
}

/**
 * Process lab test results
 */
public function processLabResults() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        $test_results = $this->input->post('test_results'); // Array of test results
        $performed_by = $this->getStaffId();
        
        if (!$patient_id || !$test_results) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID and test results are required'
            ]);
            return;
        }
        
        $patient_workflow = $this->getPatientWorkflow($patient_id);
        $processed_tests = [];
        
        if ($patient_workflow === 'gcc') {
            // Process GCC test results
            foreach ($test_results as $test_result) {
                $result = $this->processGCCTestResult($patient_id, $test_result, $performed_by);
                $processed_tests[] = $result;
            }
        } else {
            // Process general test results
            foreach ($test_results as $test_result) {
                $result = $this->processGeneralTestResult($patient_id, $test_result, $performed_by);
                $processed_tests[] = $result;
            }
        }
        
        // Update patient workflow step
        if ($patient_workflow === 'gcc') {
            $this->updatePatientWorkflowStep($patient_id, 'laboratory_completed');
        }
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Lab results processed successfully',
            'processed_tests' => $processed_tests,
            'patient_workflow' => $patient_workflow
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error processing lab results: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error processing lab results: ' . $e->getMessage()
        ]);
    }
}

/**
 * Process GCC test result
 */
private function processGCCTestResult($patient_id, $test_result, $performed_by) {
    try {
        $update_data = [
            'status' => $test_result['status'] ?: 'completed',
            'result' => $test_result['result'],
            'notes' => $test_result['notes'] ?: null,
            'performed_by' => $performed_by,
            'performed_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('patient_id', $patient_id);
        $this->db->where('gcc_test_id', $test_result['test_id']);
        $this->db->update('gcc_patient_tests', $update_data);
        
        return [
            'test_id' => $test_result['test_id'],
            'status' => 'updated',
            'result' => $test_result['result']
        ];
        
    } catch (Exception $e) {
        log_message('error', 'Error processing GCC test result: ' . $e->getMessage());
        return [
            'test_id' => $test_result['test_id'],
            'status' => 'error',
            'error' => $e->getMessage()
        ];
    }
}

/**
 * Get patient workflow
 */
private function getPatientWorkflow($patient_id) {
    try {
        $this->db->select('patient_workflow');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        
        $query = $this->db->get();
        if ($query->num_rows() > 0) {
            $patient = $query->row();
            return $patient->patient_workflow;
        }
        
        return 'general';
        
    } catch (Exception $e) {
        log_message('error', 'Error getting patient workflow: ' . $e->getMessage());
        return 'general';
    }
}

/**
 * Update patient workflow step
 */
private function updatePatientWorkflowStep($patient_id, $step) {
    try {
        $this->db->where('id', $patient_id);
        return $this->db->update('patients', [
            'current_workflow_step' => $step,
            'workflow_status' => $step === 'laboratory_completed' ? 'in_progress' : 'in_progress'
        ]);
    } catch (Exception $e) {
        log_message('error', 'Error updating workflow step: ' . $e->getMessage());
        return false;
    }
}

/**
 * Get GCC test progress for patient
 */
public function getGCCTestProgress() {
    header('Content-Type: application/json');
    
    $patient_id = $this->input->get('patient_id');
    
    if (!$patient_id) {
        echo json_encode(['status' => 'error', 'message' => 'Patient ID required']);
        return;
    }
    
    try {
        $this->db->select('gt.test_name, gt.test_code, gt.test_category, gpt.status, gpt.result, gpt.performed_at');
        $this->db->from('gcc_patient_tests gpt');
        $this->db->join('gcc_tests gt', 'gpt.gcc_test_id = gt.id');
        $this->db->where('gpt.patient_id', $patient_id);
        $this->db->where_in('gt.test_category', ['hematology', 'biochemistry', 'serology', 'clinical_pathology', 'microbiology']);
        $this->db->order_by('gt.test_order', 'ASC');
        
        $query = $this->db->get();
        $tests = $query->result_array();
        
        $progress = [
            'total_tests' => count($tests),
            'completed_tests' => 0,
            'pending_tests' => 0,
            'abnormal_tests' => 0,
            'tests' => $tests
        ];
        
        foreach ($tests as $test) {
            switch ($test['status']) {
                case 'completed':
                    $progress['completed_tests']++;
                    break;
                case 'pending':
                    $progress['pending_tests']++;
                    break;
                case 'abnormal':
                    $progress['abnormal_tests']++;
                    break;
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'progress' => $progress
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error getting test progress: ' . $e->getMessage()
        ]);
    }
}

    // =============================================
    // PATIENT MANAGEMENT METHODS
    // =============================================

    public function getAllPatients() {
        header('Content-Type: application/json');
        
        try {
            // Get all active patients
            $this->db->select('id, patient_name, mobileno, gender, age');
            $this->db->from('patients');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('patient_name', 'ASC');
            $this->db->limit(200);
            
            $query = $this->db->get();
            
            if (!$query) {
                echo json_encode([
                    'patients' => [],
                    'count' => 0,
                    'error' => 'Database query failed: ' . $this->db->error()['message']
                ]);
                return;
            }
            
            $patients = $query->result_array();
            
            // Add fingerprint status to each patient
            foreach ($patients as &$patient) {
                if ($this->db->field_exists('fingerprint_template', 'patients')) {
                    $this->db->select('fingerprint_template');
                    $this->db->from('patients');
                    $this->db->where('id', $patient['id']);
                    $fp_query = $this->db->get();
                    
                    if ($fp_query && $fp_query->num_rows() > 0) {
                        $fp_data = $fp_query->row();
                        $patient['has_fingerprint'] = !empty($fp_data->fingerprint_template);
                    } else {
                        $patient['has_fingerprint'] = false;
                    }
                } else {
                    $patient['has_fingerprint'] = false;
                }
            }
            
            echo json_encode([
                'patients' => $patients,
                'count' => count($patients),
                'message' => 'Found ' . count($patients) . ' patients'
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'patients' => [],
                'count' => 0,
                'error' => 'Exception: ' . $e->getMessage()
            ]);
        }
    }

    public function getPatientDetails() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('id');
            
            if (!$patientId) {
                echo json_encode([
                    'error' => 'Patient ID required',
                    'debug' => [
                        'post_data' => $_POST,
                        'get_data' => $_GET
                    ]
                ]);
                return;
            }
            
            // Get patient details
            $this->db->select('*');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            
            $query = $this->db->get();
            
            if (!$query) {
                echo json_encode([
                    'error' => 'Query failed: ' . $this->db->error()['message'],
                    'patient_id' => $patientId
                ]);
                return;
            }
            
            if ($query->num_rows() == 0) {
                echo json_encode([
                    'error' => 'Patient not found with ID: ' . $patientId,
                    'patient_id' => $patientId
                ]);
                return;
            }
            
            $patient = $query->row_array();
            
            // Get blood group name if available
            if ($this->db->table_exists('blood_group') && !empty($patient['blood_group'])) {
                try {
                    $this->db->select('name');
                    $this->db->from('blood_group');
                    $this->db->where('id', $patient['blood_group']);
                    $bg_query = $this->db->get();
                    
                    if ($bg_query && $bg_query->num_rows() > 0) {
                        $patient['blood_group_name'] = $bg_query->row()->name;
                    } else {
                        $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
                    }
                } catch (Exception $e) {
                    $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
                }
            } else {
                $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
            }
            
            // Process patient data
            $patient['patient_age'] = !empty($patient['age']) ? $patient['age'] . ' years' : 'Unknown';
            $patient['image'] = !empty($patient['image']) ? $patient['image'] : 'uploads/patient_images/no_image.png';
            
            // Set defaults for missing fields
            $patient['patient_name'] = $patient['patient_name'] ?? 'Unknown';
            $patient['gender'] = $patient['gender'] ?? 'Unknown';
            $patient['mobileno'] = $patient['mobileno'] ?? '';
            $patient['known_allergies'] = $patient['known_allergies'] ?? 'None';
            
            // Fingerprint verification status
            $hasFingerprint = false;
            if ($this->db->field_exists('fingerprint_template', 'patients')) {
                $hasFingerprint = !empty($patient['fingerprint_template']);
            }
            
            $patient['has_fingerprint'] = $hasFingerprint;
            $patient['verification_capable'] = $hasFingerprint && strlen($patient['fingerprint_template'] ?? '') > 100;
            
            // Security: Create fingerprint status object without exposing template
            $patient['fingerprint_status'] = [
                'enrolled' => $hasFingerprint,
                'template_size' => strlen($patient['fingerprint_template'] ?? ''),
                'image_available' => !empty($patient['fingerprint_image']),
                'enrollment_date' => $patient['created_at'] ?? null
            ];
            
            // Remove sensitive data from response
            unset($patient['fingerprint_template']);
            if (isset($patient['fingerprint_image'])) {
                unset($patient['fingerprint_image']);
            }
            
            echo json_encode($patient);
            
        } catch (Exception $e) {
            echo json_encode([
                'error' => 'Database error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'undefined',
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]);
        }
    }

    public function getStoredFingerprintTemplate() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('patient_id');
            
            if (!$patientId) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID required'
                ]);
                return;
            }
            
            // Check if fingerprint columns exist
            if (!$this->db->field_exists('fingerprint_template', 'patients')) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Fingerprint functionality not available'
                ]);
                return;
            }
            
            // Get fingerprint data
            $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            $this->db->where('is_active', 'yes');
            
            $query = $this->db->get();
            
            if ($query->num_rows() > 0) {
                $patient = $query->row_array();
                
                if (!empty($patient['fingerprint_template'])) {
                    $template = trim($patient['fingerprint_template']);
                    
                    echo json_encode([
                        'status' => 'success',
                        'patient_id' => $patientId,
                        'patient_name' => $patient['patient_name'],
                        'fingerprint_template' => $template,
                        'fingerprint_image' => $patient['fingerprint_image'] ?? '',
                        'template_size' => strlen($template),
                        'has_image' => !empty($patient['fingerprint_image'])
                    ]);
                } else {
                    echo json_encode([
                        'status' => 'error',
                        'message' => 'No fingerprint template found for this patient',
                        'patient_id' => $patientId,
                        'patient_name' => $patient['patient_name']
                    ]);
                }
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found or inactive',
                    'patient_id' => $patientId
                ]);
            }
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Database error: ' . $e->getMessage()
            ]);
        }
    }

    public function verifyFingerprint() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('patient_id');
            $matchScore = $this->input->post('match_score');
            $isMatch = $this->input->post('is_match');
            $threshold = $this->input->post('threshold') ?? 60;
            $serviceUrl = $this->input->post('service_url');
            
            if (!$patientId) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient ID required'
                ]);
                return;
            }
            
            if (!is_numeric($matchScore)) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Invalid match score: ' . $matchScore
                ]);
                return;
            }
            
            // Validate patient exists
            $this->db->select('id, patient_name, fingerprint_template');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            $this->db->where('is_active', 'yes');
            $patient_query = $this->db->get();
            
            if ($patient_query->num_rows() == 0) {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Patient not found or inactive',
                    'patient_id' => $patientId
                ]);
                return;
            }
            
            $patient = $patient_query->row_array();
            $hasStoredTemplate = !empty($patient['fingerprint_template']);
            
            // Log verification attempt
            $verificationLogId = $this->logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold, $serviceUrl, $hasStoredTemplate);
            
            // Prepare response
            $verificationResult = [
                'patient_id' => $patientId,
                'patient_name' => $patient['patient_name'],
                'verification_type' => '1:1',
                'timestamp' => date('Y-m-d H:i:s'),
                'staff_id' => $this->getStaffId(),
                'match_score' => floatval($matchScore),
                'threshold_used' => intval($threshold),
                'max_possible_score' => 199,
                'service_url' => $serviceUrl ?? 'unknown',
                'verification_log_id' => $verificationLogId,
                'has_stored_template' => $hasStoredTemplate,
                'template_size' => $hasStoredTemplate ? strlen($patient['fingerprint_template']) : 0
            ];
            
            // Determine verification result
            $matchPassed = ($isMatch == 'true' || $isMatch === true || $isMatch == 1);
            
            if ($matchPassed && $hasStoredTemplate) {
                $verificationResult['status'] = 'success';
                $verificationResult['message'] = 'Patient identity verified successfully for laboratory';
                $verificationResult['verification_passed'] = true;
                $verificationResult['verification_method'] = 'fingerprint_match';
            } else if (!$hasStoredTemplate) {
                $verificationResult['status'] = 'warning';
                $verificationResult['message'] = 'No stored fingerprint available for verification';
                $verificationResult['verification_passed'] = false;
                $verificationResult['verification_method'] = 'no_template';
            } else {
                $verificationResult['status'] = 'failed';
                $verificationResult['message'] = 'Fingerprint verification failed - identity not confirmed';
                $verificationResult['verification_passed'] = false;
                $verificationResult['verification_method'] = 'fingerprint_mismatch';
            }
            
            echo json_encode($verificationResult);
            
        } catch (Exception $e) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Verification error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'unknown'
            ]);
        }
    }
    
    /**
 * Get patient consultations for laboratory module
 */
public function getPatientConsultations() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if consultations table exists
        if (!$this->db->table_exists('consultations')) {
            echo json_encode([
                'status' => 'success',
                'consultations' => [],
                'message' => 'No consultations table found'
            ]);
            return;
        }
        
        // Get patient consultations with doctor information
        $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.created_at');
        $this->db->from('consultations c');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit(5); // Get last 5 consultations
        
        $query = $this->db->get();
        $consultations = $query->result_array();
        
        // Get doctor names for each consultation
        foreach ($consultations as &$consultation) {
            if (!empty($consultation['doctor'])) {
                $this->db->select('name, surname, employee_id');
                $this->db->from('staff');
                $this->db->where('id', $consultation['doctor']);
                $doctor_query = $this->db->get();
                
                if ($doctor_query && $doctor_query->num_rows() > 0) {
                    $doctor = $doctor_query->row();
                    $consultation['doctor_name'] = $doctor->name;
                    if (!empty($doctor->surname)) {
                        $consultation['doctor_name'] .= ' ' . $doctor->surname;
                    }
                    $consultation['doctor_employee_id'] = $doctor->employee_id ?? '';
                } else {
                    $consultation['doctor_name'] = 'Unknown Doctor';
                    $consultation['doctor_employee_id'] = '';
                }
            } else {
                $consultation['doctor_name'] = 'Unknown Doctor';
                $consultation['doctor_employee_id'] = '';
            }
            
            // Format date for display
            $consultation['consultation_date'] = $consultation['date'];
            $consultation['formatted_date'] = date('M d, Y', strtotime($consultation['date']));
        }
        
        echo json_encode([
            'status' => 'success',
            'consultations' => $consultations,
            'count' => count($consultations)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading consultations: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get latest consultation for a patient (for lab modal)
 */
public function getLatestConsultation() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if consultations table exists
        if (!$this->db->table_exists('consultations')) {
            echo json_encode([
                'status' => 'success',
                'consultation' => null,
                'message' => 'No consultations table found'
            ]);
            return;
        }
        
        // Get the most recent consultation
        $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.created_at');
        $this->db->from('consultations c');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit(1);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $consultation = $query->row_array();
            
            // Get doctor information
            if (!empty($consultation['doctor'])) {
                $this->db->select('name, surname, employee_id');
                $this->db->from('staff');
                $this->db->where('id', $consultation['doctor']);
                $doctor_query = $this->db->get();
                
                if ($doctor_query && $doctor_query->num_rows() > 0) {
                    $doctor = $doctor_query->row();
                    $consultation['doctor_name'] = $doctor->name;
                    if (!empty($doctor->surname)) {
                        $consultation['doctor_name'] .= ' ' . $doctor->surname;
                    }
                    $consultation['doctor_employee_id'] = $doctor->employee_id ?? '';
                } else {
                    $consultation['doctor_name'] = 'Unknown Doctor';
                }
            }
            
            // Analyze consultation for test requests
            $consultation['suggested_tests'] = $this->analyzeSuggestedTests($consultation);
            $consultation['formatted_date'] = date('M d, Y', strtotime($consultation['date']));
            
            echo json_encode([
                'status' => 'success',
                'consultation' => $consultation
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'consultation' => null,
                'message' => 'No consultations found for this patient'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading latest consultation: ' . $e->getMessage()
        ]);
    }
}

/**
 * Analyze consultation text for suggested lab tests
 */
private function analyzeSuggestedTests($consultation) {
    $suggested_tests = [];
    
    // Combine all consultation text for analysis
    $search_text = strtolower(
        ($consultation['symptoms'] ?? '') . ' ' . 
        ($consultation['diagnosis'] ?? '') . ' ' . 
        ($consultation['notes'] ?? '')
    );
    
    // Define test patterns to search for
    $test_patterns = [
        'CBC' => ['cbc', 'complete blood count', 'full blood count', 'fbc', 'blood count'],
        'Blood Sugar' => ['glucose', 'blood sugar', 'hba1c', 'diabetes', 'fasting glucose', 'random glucose'],
        'Liver Function' => ['liver function', 'lft', 'alt', 'ast', 'liver test', 'hepatic'],
        'Kidney Function' => ['kidney function', 'creatinine', 'urea', 'bun', 'renal function'],
        'Lipid Profile' => ['lipid profile', 'cholesterol', 'triglycerides', 'hdl', 'ldl'],
        'Thyroid Function' => ['thyroid', 'tsh', 't3', 't4', 'thyroid function'],
        'Malaria Test' => ['malaria', 'malaria test', 'mps', 'malaria parasite'],
        'Typhoid Test' => ['typhoid', 'widal', 'typhoid test', 'salmonella'],
        'HIV Test' => ['hiv', 'hiv test', 'aids', 'retroviral'],
        'Hepatitis Screen' => ['hepatitis', 'hbsag', 'hcv', 'hepatitis b', 'hepatitis c'],
        'Urine Analysis' => ['urine test', 'urinalysis', 'urine culture', 'uti', 'urinary tract'],
        'Stool Test' => ['stool test', 'stool culture', 'ova and parasites', 'stool analysis'],
        'Blood Culture' => ['blood culture', 'culture and sensitivity', 'sepsis', 'bacteremia'],
        'Pregnancy Test' => ['pregnancy test', 'beta hcg', 'urine pregnancy', 'pregnant'],
        'Inflammatory Markers' => ['esr', 'crp', 'inflammatory', 'sed rate', 'c-reactive protein']
    ];
    
    // Search for test patterns in consultation text
    foreach ($test_patterns as $test_name => $patterns) {
        foreach ($patterns as $pattern) {
            if (strpos($search_text, $pattern) !== false) {
                if (!in_array($test_name, $suggested_tests)) {
                    $suggested_tests[] = $test_name;
                }
                break; // Found one pattern for this test, move to next test
            }
        }
    }
    
    return $suggested_tests;
}

/**
 * Get test categories (using existing pathology_category table)
 */


/**
 * Get tests by category (using existing pathology table)
 */



    // =============================================
    // LABORATORY MANAGEMENT METHODS
    // =============================================

public function save() {
    header('Content-Type: application/json');
    
    try {
        // Get all form data including consultation information
        $patient_id = $this->input->post('patient_id');
        $test_id = $this->input->post('test_id'); // This will be pathology.id
        $test_date = $this->input->post('test_date');
        $test_priority = $this->input->post('test_priority') ?: 'normal';
        $test_result = $this->input->post('test_result');
        $result_status = $this->input->post('result_status') ?: 'pending';
        $reference_range = $this->input->post('reference_range');
        $notes = $this->input->post('notes');
        $doctor_id = $this->input->post('doctor_id');
        $sample_type_override = $this->input->post('sample_type_override');
        
        // Cost and billing information
        $test_cost = $this->input->post('test_cost') ?: 0;
        $discount_percentage = $this->input->post('discount_percentage') ?: 0;
        $discount_amount = $this->input->post('discount_amount') ?: 0;
        $net_amount = $this->input->post('net_amount') ?: 0;
        $payment_method = $this->input->post('payment_method') ?: 'cash';
        $amount_paid = $this->input->post('amount_paid') ?: 0;
        
        // Verification information
        $verification_status = $this->input->post('verification_status') ?: 0;
        $verification_score = $this->input->post('verification_score') ?: 0;
        $verification_quality = $this->input->post('verification_quality');
        $verification_method = $this->input->post('verification_method');
        
        // NEW: Consultation information for audit trail
        $consultation_symptoms = $this->input->post('consultation_symptoms');
        $consultation_diagnosis = $this->input->post('consultation_diagnosis');
        $consultation_notes = $this->input->post('consultation_notes');
        $consultation_doctor = $this->input->post('consultation_doctor');
        
        // Validation
        if (!$patient_id || !$test_id || !$test_date) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID, Test ID, and Test Date are required'
            ]);
            return;
        }
        
        // Get test details from pathology table
        $this->db->select('test_name, short_name, cost, method, sample_type, report_days, unit');
        $this->db->from('pathology');
        $this->db->where('id', $test_id);
        $test_query = $this->db->get();
        
        if ($test_query->num_rows() == 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid test selected'
            ]);
            return;
        }
        
        $test_details = $test_query->row();
        
        // Calculate final cost if not provided
        if ($test_cost == 0) {
            $base_cost = floatval($test_details->cost);
            $priority_multiplier = 1;
            
            if ($test_priority === 'urgent') {
                $priority_multiplier = 1.25;
            } elseif ($test_priority === 'stat') {
                $priority_multiplier = 1.5;
            }
            
            $cost_after_priority = $base_cost * $priority_multiplier;
            $tax_amount = $cost_after_priority * 0.16; // 16% VAT
            $test_cost = $cost_after_priority + $tax_amount;
            $net_amount = $test_cost - $discount_amount;
        }
        
        // Build the laboratory data based on existing structure
        $lab_data = [
            'patient_id' => $patient_id,
            'test_name' => $test_details->test_name . ' (' . $test_details->short_name . ')',
            'test_result' => $test_result ?: 'Pending',
            'notes' => $notes ?: '',
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Add consultation fields if they were added to laboratory table
        if ($this->db->field_exists('consultation_symptoms', 'laboratory')) {
            $lab_data['consultation_symptoms'] = $consultation_symptoms;
            $lab_data['consultation_diagnosis'] = $consultation_diagnosis;
            $lab_data['consultation_notes'] = $consultation_notes;
            $lab_data['consultation_doctor'] = $consultation_doctor;
        }
        
        // Add verification fields if they were added
        if ($this->db->field_exists('verification_status', 'laboratory')) {
            $lab_data['verification_status'] = $verification_status;
            $lab_data['verification_score'] = $verification_score;
            $lab_data['verification_quality'] = $verification_quality;
            $lab_data['verification_method'] = $verification_method;
        }
        
        // Add cost fields if they were added
        if ($this->db->field_exists('test_cost', 'laboratory')) {
            $lab_data['base_cost'] = $test_details->cost;
            $lab_data['test_cost'] = $test_cost;
            $lab_data['discount_percentage'] = $discount_percentage;
            $lab_data['discount_amount'] = $discount_amount;
            $lab_data['net_amount'] = $net_amount;
            $lab_data['payment_method'] = $payment_method;
            $lab_data['amount_paid'] = $amount_paid;
        }
        
        // Add other fields if they exist
        if ($this->db->field_exists('test_priority', 'laboratory')) {
            $lab_data['test_priority'] = $test_priority;
            $lab_data['sample_type'] = $sample_type_override ?: $test_details->sample_type;
            $lab_data['test_method'] = $test_details->method;
            $lab_data['reference_range'] = $reference_range;
            $lab_data['report_days'] = $test_details->report_days;
        }
        
        // Insert laboratory record
        $insert_result = $this->db->insert('laboratory', $lab_data);
        
        if ($insert_result) {
            $lab_id = $this->db->insert_id();
            
            // Log the activity
            $this->logLabActivity($lab_id, $patient_id, 'Lab test created', [
                'test_name' => $test_details->test_name,
                'verification_status' => $verification_status,
                'consultation_linked' => !empty($consultation_symptoms) ? 'Yes' : 'No'
            ]);
            
            echo json_encode([
                'status' => 'success',
                'message' => 'Laboratory test saved successfully',
                'lab_id' => $lab_id,
                'test_name' => $test_details->test_name,
                'cost' => $net_amount ?: $test_cost
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save laboratory test'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving laboratory test: ' . $e->getMessage()
        ]);
    }
}

    public function getLaboratoryList() {
        header('Content-Type: application/json');
        
        try {
            // Check if laboratory table exists
            if (!$this->db->table_exists('laboratory')) {
                echo json_encode([
                    'draw' => 1,
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'message' => 'Laboratory table does not exist'
                ]);
                return;
            }
            
            // Query laboratory records
            if ($this->db->field_exists('verification_status', 'laboratory')) {
                $this->db->select('l.id, l.patient_id, l.test_name, l.test_result, l.notes, l.verification_status, l.created_at');
            } else {
                $this->db->select('l.id, l.patient_id, l.test_name, l.test_result, l.notes, l.created_at');
            }
            
            if ($this->db->field_exists('tested_by', 'laboratory')) {
                $this->db->select('l.tested_by');
            }
            
            if ($this->db->field_exists('test_date', 'laboratory')) {
                $this->db->select('l.test_date');
            }
            
            $this->db->from('laboratory l');
            $this->db->order_by('l.created_at', 'DESC');
            $this->db->limit(100);
            
            $query = $this->db->get();
            
            if (!$query) {
                echo json_encode([
                    'draw' => 1,
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'error' => 'Query failed: ' . $this->db->error()['message']
                ]);
                return;
            }
            
            $labRecords = $query->result_array();
            
            // Build data array for DataTable
            $data = [];
            foreach ($labRecords as $lab) {
                // Get patient name
                $patientName = 'Unknown Patient';
                if ($lab['patient_id']) {
                    $this->db->select('patient_name');
                    $this->db->from('patients');
                    $this->db->where('id', $lab['patient_id']);
                    $patientQuery = $this->db->get();
                    if ($patientQuery && $patientQuery->num_rows() > 0) {
                        $patient = $patientQuery->row();
                        $patientName = $patient->patient_name;
                    }
                }
                
                // Get staff name if available
                $staffName = 'Unknown Staff';
                if (isset($lab['tested_by']) && $lab['tested_by']) {
                    $this->db->select('name, surname');
                    $this->db->from('staff');
                    $this->db->where('id', $lab['tested_by']);
                    $staffQuery = $this->db->get();
                    if ($staffQuery && $staffQuery->num_rows() > 0) {
                        $staff = $staffQuery->row();
                        $staffName = $staff->name;
                        if (!empty($staff->surname)) {
                            $staffName .= ' ' . $staff->surname;
                        }
                    }
                }
                
                // Format verification badge
                $verificationBadge = '';
                if (isset($lab['verification_status'])) {
                    switch ($lab['verification_status']) {
                        case 'verified':
                            $verificationBadge = '<br><span class="label label-success">Verified</span>';
                            break;
                        case 'skipped':
                            $verificationBadge = '<br><span class="label label-warning">Skipped</span>';
                            break;
                        default:
                            $verificationBadge = '<br><span class="label label-danger">Unverified</span>';
                    }
                }
                
                // Test date display
                $testDate = isset($lab['test_date']) ? date('d M Y', strtotime($lab['test_date'])) : 
                           date('d M Y H:i', strtotime($lab['created_at']));
                
                $data[] = [
                    $lab['id'],
                    $testDate,
                    $patientName . ' (' . $lab['patient_id'] . ')' . $verificationBadge,
                    $lab['test_name'],
                    substr($lab['test_result'] ?? 'Pending', 0, 50) . 
                    (strlen($lab['test_result'] ?? '') > 50 ? '...' : ''),
                    $staffName,
                    '<button class="btn btn-xs btn-info" onclick="viewLab(' . $lab['id'] . ')">
                        <i class="fa fa-eye"></i> View
                    </button>
                    <button class="btn btn-xs btn-primary" onclick="editLab(' . $lab['id'] . ')">
                        <i class="fa fa-edit"></i> Edit
                    </button>'
                ];
            }
            
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => count($data),
                'recordsFiltered' => count($data),
                'data' => $data,
                'success' => true
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Exception: ' . $e->getMessage()
            ]);
        }
    }

    public function getLaboratoryDetails() {
        header('Content-Type: application/json');
        
        try {
            $labId = $this->input->post('lab_id');
            
           if (!$labId) {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'Laboratory ID required'
               ]);
               return;
           }
           
           // Get lab details
           if ($this->db->field_exists('verification_status', 'laboratory')) {
               $this->db->select('l.id, l.patient_id, l.test_name, l.test_result, l.notes, l.verification_status, l.created_at, l.updated_at');
           } else {
               $this->db->select('l.id, l.patient_id, l.test_name, l.test_result, l.notes, l.created_at');
           }
           
           if ($this->db->field_exists('tested_by', 'laboratory')) {
               $this->db->select('l.tested_by');
           }
           
           if ($this->db->field_exists('test_date', 'laboratory')) {
               $this->db->select('l.test_date');
           }
           
           $this->db->from('laboratory l');
           $this->db->where('l.id', $labId);
           
           $query = $this->db->get();
           
           if ($query->num_rows() > 0) {
               $lab = $query->row_array();
               
               // Get patient name
               $this->db->select('patient_name');
               $this->db->from('patients');
               $this->db->where('id', $lab['patient_id']);
               $patientQuery = $this->db->get();
               
               if ($patientQuery && $patientQuery->num_rows() > 0) {
                   $patient = $patientQuery->row();
                   $lab['patient_name'] = $patient->patient_name;
               } else {
                   $lab['patient_name'] = 'Unknown Patient';
               }
               
               // Get staff name if available
               if (isset($lab['tested_by']) && $lab['tested_by']) {
                   $this->db->select('name, surname');
                   $this->db->from('staff');
                   $this->db->where('id', $lab['tested_by']);
                   $staffQuery = $this->db->get();
                   
                   if ($staffQuery && $staffQuery->num_rows() > 0) {
                       $staff = $staffQuery->row();
                       $lab['staff_name'] = $staff->name;
                       if (!empty($staff->surname)) {
                           $lab['staff_name'] .= ' ' . $staff->surname;
                       }
                   } else {
                       $lab['staff_name'] = 'Unknown Staff';
                   }
               } else {
                   $lab['staff_name'] = 'Unknown Staff';
               }
               
               // Set default values for optional fields
               if (!isset($lab['verification_status'])) {
                   $lab['verification_status'] = 'unverified';
               }
               
               if (!isset($lab['updated_at'])) {
                   $lab['updated_at'] = null;
               }
               
               if (!isset($lab['test_date'])) {
                   $lab['test_date'] = null;
               }
               
               echo json_encode([
                   'status' => 'success',
                   'laboratory' => $lab
               ]);
               
           } else {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'Laboratory record not found with ID: ' . $labId
               ]);
           }
           
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => 'Database error: ' . $e->getMessage(),
               'lab_id' => $labId ?? 'undefined'
           ]);
       }
   }

   // =============================================
   // HELPER METHODS
   // =============================================

   /*private function createLaboratoryTable() {
       if (!$this->db->table_exists('laboratory')) {
           $sql = "CREATE TABLE IF NOT EXISTS `laboratory` (
               `id` int(11) NOT NULL AUTO_INCREMENT,
               `patient_id` varchar(50) NOT NULL,
               `test_name` varchar(255) NOT NULL,
               `test_result` text DEFAULT NULL,
               `notes` text DEFAULT NULL,
               `test_date` date DEFAULT NULL,
               `verification_status` enum('verified','unverified','skipped') DEFAULT 'unverified',
               `tested_by` int(11) DEFAULT 1,
               `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
               `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
               PRIMARY KEY (`id`),
               KEY `patient_id` (`patient_id`),
               KEY `tested_by` (`tested_by`),
               KEY `created_at` (`created_at`),
               KEY `test_date` (`test_date`)
           ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
           
           $this->db->query($sql);
       } else {
           // Add missing columns to existing table
           $this->addLaboratoryColumns();
       }
   }*/

   private function addLaboratoryColumns() {
       try {
           if (!$this->db->field_exists('test_date', 'laboratory')) {
               $sql = "ALTER TABLE `laboratory` 
                       ADD COLUMN `test_date` DATE DEFAULT NULL AFTER `notes`";
               $this->db->query($sql);
           }
           
           if (!$this->db->field_exists('verification_status', 'laboratory')) {
               $sql = "ALTER TABLE `laboratory` 
                       ADD COLUMN `verification_status` ENUM('verified','unverified','skipped') DEFAULT 'unverified' AFTER `test_date`";
               $this->db->query($sql);
           }
           
           if (!$this->db->field_exists('tested_by', 'laboratory')) {
               $sql = "ALTER TABLE `laboratory` 
                       ADD COLUMN `tested_by` INT(11) DEFAULT 1 AFTER `verification_status`";
               $this->db->query($sql);
           }
           
           if (!$this->db->field_exists('updated_at', 'laboratory')) {
               $sql = "ALTER TABLE `laboratory` 
                       ADD COLUMN `updated_at` TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP AFTER `created_at`";
               $this->db->query($sql);
           }
           
       } catch (Exception $e) {
           log_message('error', 'Error adding laboratory columns: ' . $e->getMessage());
       }
   }

   private function logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold = 60, $serviceUrl = null, $hasStoredTemplate = true) {
       try {
           // Create verification log table if it doesn't exist
           $this->createVerificationLogTable();
           
           $logData = [
               'patient_id' => $patientId,
               'verification_type' => '1:1',
               'match_score' => floatval($matchScore),
               'is_match' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 1 : 0,
               'threshold_used' => intval($threshold),
               'service_url' => $serviceUrl,
               'verification_date' => date('Y-m-d H:i:s'),
               'staff_id' => $this->getStaffId(),
               'ip_address' => $this->input->ip_address(),
               'user_agent' => $this->input->user_agent(),
               'has_stored_template' => $hasStoredTemplate ? 1 : 0,
               'verification_status' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 'success' : 'failed',
               'module' => 'laboratory'
           ];
           
           $this->db->insert('fingerprint_verification_log', $logData);
           return $this->db->insert_id();
           
       } catch (Exception $e) {
           log_message('error', 'Failed to log laboratory verification: ' . $e->getMessage());
           return null;
       }
   }

   private function createVerificationLogTable() {
       if (!$this->db->table_exists('fingerprint_verification_log')) {
           $sql = "CREATE TABLE `fingerprint_verification_log` (
               `id` int(11) NOT NULL AUTO_INCREMENT,
               `patient_id` varchar(50) NOT NULL,
               `verification_type` enum('1:1','1:N') DEFAULT '1:1',
               `match_score` decimal(5,2) DEFAULT NULL,
               `is_match` tinyint(1) DEFAULT 0,
               `threshold_used` int(3) DEFAULT 60,
               `service_url` varchar(255) DEFAULT NULL,
               `verification_date` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
               `staff_id` int(11) DEFAULT NULL,
               `ip_address` varchar(45) DEFAULT NULL,
               `user_agent` text DEFAULT NULL,
               `has_stored_template` tinyint(1) DEFAULT 1,
               `verification_status` enum('success','failed','error') DEFAULT 'failed',
               `module` varchar(50) DEFAULT 'unknown',
               `notes` text DEFAULT NULL,
               PRIMARY KEY (`id`),
               KEY `patient_id` (`patient_id`),
               KEY `verification_date` (`verification_date`),
               KEY `staff_id` (`staff_id`),
               KEY `verification_status` (`verification_status`),
               KEY `module` (`module`)
           ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
           
           $this->db->query($sql);
       }
   }

   private function getTestTypes() {
       return [
           'Blood Tests' => [
               'CBC' => 'Complete Blood Count',
               'FBC' => 'Full Blood Count',
               'ESR' => 'Erythrocyte Sedimentation Rate',
               'Blood_Sugar' => 'Blood Sugar Test',
               'HbA1c' => 'Hemoglobin A1c',
               'Lipid_Profile' => 'Lipid Profile',
               'Liver_Function' => 'Liver Function Tests',
               'Kidney_Function' => 'Kidney Function Tests',
               'Thyroid_Function' => 'Thyroid Function Tests'
           ],
           'Urine Tests' => [
               'Urinalysis' => 'Complete Urine Analysis',
               'Urine_Culture' => 'Urine Culture & Sensitivity',
               'Urine_Protein' => 'Urine Protein Test',
               'Urine_Glucose' => 'Urine Glucose Test'
           ],
           'Microbiology' => [
               'Blood_Culture' => 'Blood Culture',
               'Stool_Culture' => 'Stool Culture',
               'Throat_Swab' => 'Throat Swab Culture',
               'Wound_Swab' => 'Wound Swab Culture'
           ],
           'Serology' => [
               'HIV_Test' => 'HIV Screening',
               'Hepatitis_B' => 'Hepatitis B Test',
               'Hepatitis_C' => 'Hepatitis C Test',
               'VDRL' => 'VDRL Test',
               'Widal_Test' => 'Widal Test'
           ],
           'Others' => [
               'X_Ray' => 'X-Ray Reports',
               'ECG' => 'ECG Reports',
               'Ultrasound' => 'Ultrasound Reports'
           ]
       ];
   }

   private function getLabStaff() {
       try {
           if (method_exists($this->staff_model, 'getStaffbyrole')) {
               // Get lab technicians and doctors
               $labTech = $this->staff_model->getStaffbyrole(4); // Role 4 = lab technicians
               $doctors = $this->staff_model->getStaffbyrole(3); // Role 3 = doctors
               return array_merge($labTech ?: [], $doctors ?: []);
           } else {
               // Alternative query to get staff
               $this->db->select('id, name, surname, employee_id');
               $this->db->from('staff');
               $this->db->where('is_active', 'yes');
               $query = $this->db->get();
               return $query->result_array();
           }
       } catch (Exception $e) {
           log_message('error', 'Error getting lab staff: ' . $e->getMessage());
           return [];
       }
   }



   // =============================================
   // DEBUGGING AND TESTING METHODS
   // =============================================

   public function test() {
       echo "<h1>Laboratory Controller Test</h1>";
       echo "<p>Controller is working!</p>";
       echo "<p>Base URL: " . base_url() . "</p>";
       echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
       
       try {
           $this->db->select('COUNT(*) as patient_count');
           $this->db->from('patients');
           $query = $this->db->get();
           $result = $query->row();
           echo "<p>Patients in database: " . $result->patient_count . "</p>";
           
           if ($this->db->table_exists('laboratory')) {
               $this->db->select('COUNT(*) as lab_count');
               $this->db->from('laboratory');
               $query = $this->db->get();
               $result = $query->row();
               echo "<p>Laboratory records: " . $result->lab_count . "</p>";
           } else {
               echo "<p>Laboratory table does not exist</p>";
           }
           
           // Display test types
           echo "<h3>Available Test Types:</h3>";
           $testTypes = $this->getTestTypes();
           foreach ($testTypes as $category => $tests) {
               echo "<h4>$category:</h4><ul>";
               foreach ($tests as $key => $name) {
                   echo "<li>$key: $name</li>";
               }
               echo "</ul>";
           }
           
       } catch (Exception $e) {
           echo "<p>Database error: " . $e->getMessage() . "</p>";
       }
   }


/**
 * Get doctors/staff for laboratory
 */
public function getDoctors() {
    header('Content-Type: application/json');
    
    try {
        // Try to use the same method as consultation
        if (method_exists($this->staff_model, 'getStaffbyrole')) {
            // Get doctors using the staff model (same as consultation)
            $doctors = $this->staff_model->getStaffbyrole(2); // Role 2 = Doctor
            
            if (empty($doctors)) {
                // Try other roles
                $doctors = $this->staff_model->getStaffbyrole(3); // Role 3 = Nurse/Staff
            }
            
            if (empty($doctors)) {
                // Get all active staff as fallback
                $this->db->select('id, name, surname, employee_id');
                $this->db->where('is_active', 'yes');
                $this->db->order_by('name', 'ASC');
                $query = $this->db->get('staff');
                $doctors = $query ? $query->result_array() : [];
            }
        } else {
            // Direct database query
            $this->db->select('id, name, surname, employee_id');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('name', 'ASC');
            $query = $this->db->get('staff');
            $doctors = $query ? $query->result_array() : [];
        }
        
        // Format doctors for dropdown
        $formatted_doctors = [];
        foreach ($doctors as $doctor) {
            $formatted_doctors[] = [
                'id' => $doctor['id'],
                'name' => $doctor['name'] ?? 'Unknown',
                'surname' => $doctor['surname'] ?? '',
                'employee_id' => $doctor['employee_id'] ?? ('EMP-' . $doctor['id']),
                'full_name' => trim(($doctor['name'] ?? 'Unknown') . ' ' . ($doctor['surname'] ?? '')),
                'designation' => 'Doctor/Staff'
            ];
        }
        
        echo json_encode([
            'status' => 'success',
            'doctors' => $formatted_doctors,
            'count' => count($formatted_doctors)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading doctors: ' . $e->getMessage(),
            'debug' => [
                'line' => $e->getLine(),
                'file' => basename($e->getFile())
            ]
        ]);
    }
}

/**
 * Get patient lab history with costs
 */
public function getPatientLabHistory() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Get lab tests for the patient
        $this->db->select('l.*, lt.test_name as test_display_name');
        $this->db->from('laboratory l');
        $this->db->join('lab_tests lt', 'lt.id = l.test_id', 'left');
        $this->db->where('l.patient_id', $patient_id);
        $this->db->order_by('l.test_date', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        $lab_tests = $query->result_array();
        
        // Enhance each test with consultation context if available
        foreach ($lab_tests as &$test) {
            if (!empty($test['consultation_symptoms']) || !empty($test['consultation_diagnosis'])) {
                $test['has_consultation_context'] = true;
                $test['consultation_summary'] = trim(
                    ($test['consultation_symptoms'] ? 'Symptoms: ' . $test['consultation_symptoms'] . '. ' : '') .
                    ($test['consultation_diagnosis'] ? 'Diagnosis: ' . $test['consultation_diagnosis'] . '.' : '')
                );
            } else {
                $test['has_consultation_context'] = false;
                $test['consultation_summary'] = '';
            }
            
            // Format dates
            $test['formatted_date'] = date('M d, Y', strtotime($test['test_date']));
            $test['formatted_cost'] = number_format($test['net_amount'] ?? $test['test_cost'] ?? 0, 2);
        }
        
        echo json_encode([
            'status' => 'success',
            'tests' => $lab_tests,
            'count' => count($lab_tests)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab history: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get cost summary and statistics
 */
public function getCostSummary() {
    header('Content-Type: application/json');
    
    try {
        $today = date('Y-m-d');
        $summary = [
            'today_revenue' => 0,
            'tests_today' => 0,
            'pending_results' => 0,
            'critical_results' => 0,
            'test_breakdown' => []
        ];
        
        // Get today's revenue from laboratory records
        if ($this->db->table_exists('laboratory')) {
            // Count tests today
            $this->db->where('DATE(created_at)', $today);
            $summary['tests_today'] = $this->db->count_all_results('laboratory');
            
            // Count pending results
            $this->db->where('test_result IS NULL OR test_result = ""');
            $summary['pending_results'] = $this->db->count_all_results('laboratory');
            
            // Get test breakdown with costs
            $this->db->select('test_name, COUNT(*) as count');
            $this->db->where('DATE(created_at)', $today);
            $this->db->group_by('test_name');
            $query = $this->db->get('laboratory');
            
            if ($query) {
                $todayTests = $query->result_array();
                foreach ($todayTests as $test) {
                    $cost = $this->getTestCostByName($test['test_name']);
                    $summary['today_revenue'] += $cost * $test['count'];
                }
            }
        }
        
        // Get test breakdown from lab_tests table
        if ($this->db->table_exists('lab_tests') && $this->db->table_exists('lab_test_categories')) {
            $this->db->select('lt.test_name, lt.cost as standard_cost, ltc.category_name');
            $this->db->from('lab_tests lt');
            $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
            $this->db->where('lt.is_active', 1);
            $this->db->order_by('ltc.category_name, lt.test_name');
            $query = $this->db->get();
            
            if ($query) {
                $allTests = $query->result_array();
                foreach ($allTests as $test) {
                    // Count how many times this test was done today
                    $testsToday = 0;
                    $revenueToday = 0;
                    
                    if ($this->db->table_exists('laboratory')) {
                        $this->db->where('test_name', $test['test_name']);
                        $this->db->where('DATE(created_at)', $today);
                        $testsToday = $this->db->count_all_results('laboratory');
                        $revenueToday = $testsToday * $test['standard_cost'];
                    }
                    
                    $summary['test_breakdown'][] = [
                        'category_name' => $test['category_name'],
                        'test_name' => $test['test_name'],
                        'standard_cost' => $test['standard_cost'],
                        'tests_today' => $testsToday,
                        'revenue_today' => $revenueToday
                    ];
                }
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'today_revenue' => $summary['today_revenue'],
            'tests_today' => $summary['tests_today'],
            'pending_results' => $summary['pending_results'],
            'critical_results' => $summary['critical_results'],
            'test_breakdown' => $summary['test_breakdown']
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading cost summary: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get comprehensive patient summary for laboratory
 */
public function getPatientSummaryForLab() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        $summary = [];
        
        // Get patient basic information
        $this->db->select('id, patient_name, age, gender, mobileno, blood_group_name, known_allergies, image');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $patient_query = $this->db->get();
        
        if ($patient_query->num_rows() > 0) {
            $summary['patient'] = $patient_query->row_array();
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found'
            ]);
            return;
        }
        
        // Get latest consultation
        if ($this->db->table_exists('consultations')) {
            $this->db->select('c.*, s.name as doctor_name, s.surname as doctor_surname');
            $this->db->from('consultations c');
            $this->db->join('staff s', 's.id = c.doctor', 'left');
            $this->db->where('c.patient_id', $patient_id);
            $this->db->order_by('c.date', 'DESC');
            $this->db->limit(1);
            
            $consultation_query = $this->db->get();
            if ($consultation_query->num_rows() > 0) {
                $consultation = $consultation_query->row_array();
                $consultation['suggested_tests'] = $this->analyzeSuggestedTests($consultation);
                $summary['latest_consultation'] = $consultation;
            }
        }
        
        // Get recent lab tests
        $this->db->select('l.*, lt.test_name');
        $this->db->from('laboratory l');
        $this->db->join('lab_tests lt', 'lt.id = l.test_id', 'left');
        $this->db->where('l.patient_id', $patient_id);
        $this->db->order_by('l.test_date', 'DESC');
        $this->db->limit(5);
        
        $lab_query = $this->db->get();
        $summary['recent_lab_tests'] = $lab_query->result_array();
        
        echo json_encode([
            'status' => 'success',
            'summary' => $summary
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading patient summary: ' . $e->getMessage()
        ]);
    }
}

/**
 * Add new patient
 */
public function addPatient() {
    header('Content-Type: application/json');
    
    try {
        $patient_name = $this->input->post('patientName');
        $patient_age = $this->input->post('patientAge');
        $patient_gender = $this->input->post('patientGender');
        $patient_phone = $this->input->post('patientPhone');
        
        // Validation
        if (empty($patient_name) || empty($patient_age) || empty($patient_gender) || empty($patient_phone)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'All fields are required'
            ]);
            return;
        }
        
        // Check if patient already exists
        $this->db->where('patient_name', $patient_name);
        $this->db->where('mobileno', $patient_phone);
        $existing = $this->db->get('patients');
        
        if ($existing->num_rows() > 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient with this name and phone number already exists'
            ]);
            return;
        }
        
        // Prepare patient data
        $data = [
            'patient_name' => $patient_name,
            'age' => (int)$patient_age,
            'gender' => $patient_gender,
            'mobileno' => $patient_phone,
            'is_active' => 'yes',
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Insert patient
        $this->db->insert('patients', $data);
        $patient_id = $this->db->insert_id();
        
        if ($patient_id) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Patient added successfully',
                'patient_id' => $patient_id
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to add patient'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error adding patient: ' . $e->getMessage()
        ]);
    }
}

// =============================================
// HELPER METHODS
// =============================================

/**
 * Get test details from lab_tests table
 */
/*private function getTestDetails($test_id) {
    if (!$this->db->table_exists('lab_tests')) {
        return null;
    }
    
    $this->db->select('lt.*, ltc.category_name, lu.unit_name, lu.unit_symbol');
    $this->db->from('lab_tests lt');
    $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
    $this->db->join('lab_units lu', 'lu.id = lt.unit_id', 'left');
    $this->db->where('lt.id', $test_id);
    $this->db->where('lt.is_active', 1);
    $query = $this->db->get();
    
    return $query->num_rows() > 0 ? $query->row_array() : null;
}*/

/**
 * Get test cost by name
 */
private function getTestCostByName($test_name) {
    if (!$this->db->table_exists('lab_tests')) {
        return 0;
    }
    
    $this->db->select('cost');
    $this->db->where('test_name', $test_name);
    $this->db->where('is_active', 1);
    $query = $this->db->get('lab_tests');
    
    if ($query->num_rows() > 0) {
        $result = $query->row();
        return floatval($result->cost);
    }
    
    return 0;
}

/**
 * Get pathology cost
 */
private function getPathologyCost($pathology_id) {
    if (!$this->db->table_exists('pathology') || !$pathology_id) {
        return 0;
    }
    
    $this->db->select('p.charge_id');
    $this->db->from('pathology p');
    $this->db->where('p.id', $pathology_id);
    $query = $this->db->get();
    
    if ($query->num_rows() > 0) {
        $pathology = $query->row();
        
        // Get charge amount
        if ($this->db->table_exists('charges')) {
            $this->db->select('standard_charge');
            $this->db->where('id', $pathology->charge_id);
            $charge_query = $this->db->get('charges');
            
            if ($charge_query->num_rows() > 0) {
                $charge = $charge_query->row();
                return floatval($charge->standard_charge);
            }
        }
    }
    
    return 0;
}

/**
 * Enhanced laboratory table creation with cost fields
 */
private function createLaboratoryTable() {
    if (!$this->db->table_exists('laboratory')) {
        $sql = "CREATE TABLE IF NOT EXISTS `laboratory` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `test_id` int(11) DEFAULT NULL,
            `category_id` int(11) DEFAULT NULL,
            `test_name` varchar(255) NOT NULL,
            `test_result` text DEFAULT NULL,
            `notes` text DEFAULT NULL,
            `test_date` date DEFAULT NULL,
            `priority` enum('normal','urgent','stat') DEFAULT 'normal',
            `base_cost` decimal(10,2) DEFAULT 0.00,
            `total_cost` decimal(10,2) DEFAULT 0.00,
            `discount_percentage` decimal(5,2) DEFAULT 0.00,
            `final_amount` decimal(10,2) DEFAULT 0.00,
            `result_status` enum('pending','in_progress','completed','abnormal','critical') DEFAULT 'pending',
            `verification_status` enum('verified','unverified','skipped') DEFAULT 'unverified',
            `doctor_id` int(11) DEFAULT NULL,
            `tested_by` int(11) DEFAULT 1,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `patient_id` (`patient_id`),
            KEY `test_id` (`test_id`),
            KEY `category_id` (`category_id`),
            KEY `tested_by` (`tested_by`),
            KEY `doctor_id` (`doctor_id`),
            KEY `created_at` (`created_at`),
            KEY `test_date` (`test_date`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        $this->db->query($sql);
    } else {
        // Add missing columns to existing table
        $this->addLaboratoryColumns();
    }
}

/**
 * Log laboratory activities for audit trail
 */
private function logLabActivity($lab_id, $patient_id, $action, $details = []) {
    try {
        $log_data = [
            'lab_id' => $lab_id,
            'patient_id' => $patient_id,
            'action' => $action,
            'details' => json_encode($details),
            'user_id' => $this->session->userdata('admin')['id'] ?? 1,
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent(),
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        // Create lab_activity_logs table if it doesn't exist
        if (!$this->db->table_exists('lab_activity_logs')) {
            $this->createLabActivityLogsTable();
        }
        
        $this->db->insert('lab_activity_logs', $log_data);
    } catch (Exception $e) {
        // Log error but don't stop the main process
        log_message('error', 'Failed to log lab activity: ' . $e->getMessage());
    }
}

/**
 * Create lab activity logs table
 */
private function createLabActivityLogsTable() {
    $sql = "
        CREATE TABLE IF NOT EXISTS `lab_activity_logs` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `lab_id` int(11) DEFAULT NULL,
            `patient_id` int(11) DEFAULT NULL,
            `action` varchar(100) NOT NULL,
            `details` text,
            `user_id` int(11) DEFAULT NULL,
            `ip_address` varchar(45) DEFAULT NULL,
            `user_agent` text,
            `timestamp` datetime NOT NULL,
            PRIMARY KEY (`id`),
            KEY `lab_id` (`lab_id`),
            KEY `patient_id` (`patient_id`),
            KEY `timestamp` (`timestamp`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
    ";
    
    $this->db->query($sql);
}

/**
 * Test Selection page - main entry point for the new interface
 */
public function testSelection() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'laboratory');
    
    $data['title'] = 'Laboratory Test Selection';
    
    // Get required data
    try {
        $data['doctors'] = $this->getLabStaffSafe();
        $data['test_categories'] = $this->getTestCategories(); // Uses imported data
        $data['patients'] = $this->getRecentPatientsSafe();
        
        if (method_exists($this->customlib, 'getGender_Patient')) {
            $data['genderList'] = $this->customlib->getGender_Patient();
        } else {
            $data['genderList'] = array(
                'male' => 'Male',
                'female' => 'Female',
                'other' => 'Other'
            );
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error loading test selection data: ' . $e->getMessage());
        $data['doctors'] = [];
        $data['test_categories'] = [];
        $data['patients'] = [];
        $data['genderList'] = array('male' => 'Male', 'female' => 'Female');
    }
    
    // Load views
    $this->load->view('layout/header', $data);
    $this->load->view('admin/laboratory/test_selection', $data);
    $this->load->view('layout/footer', $data);
}

/**
 * Get all available laboratory tests organized by categories
 */
public function getLabTests() {
    header('Content-Type: application/json');
    
    try {
        // First try to get from database, fallback to static data
        $tests = $this->getTestsFromDatabase();
        
        if (empty($tests)) {
            $tests = $this->getStaticTestData();
        }
        
        echo json_encode([
            'status' => 'success',
            'data' => $tests,
            'count' => $this->countTotalTests($tests)
        ]);
        
    } catch (Exception $e) {
        // Fallback to static data on error
        $tests = $this->getStaticTestData();
        echo json_encode([
            'status' => 'success',
            'data' => $tests,
            'count' => $this->countTotalTests($tests),
            'note' => 'Using fallback data'
        ]);
    }
}

/**
 * Get tests from database (pathology table or lab_tests table)
 */
private function getTestsFromDatabase() {
    $categorized_tests = [];
    
    try {
        // Check if lab_tests table exists
        if ($this->db->table_exists('lab_tests')) {
            $this->db->select('id, category_name, test_name as name, cost as charge, method, sample_type');
            $this->db->from('lab_tests');
            $this->db->where('is_active', 1);
            $this->db->order_by('category_name, test_name');
            
            $query = $this->db->get();
            $results = $query->result_array();
        } else {
            // Use pathology table as fallback
            $this->db->select('p.id, COALESCE(pc.category_name, "General Tests") as category_name, p.test_name as name, p.cost as charge, p.method, p.sample_type');
            $this->db->from('pathology p');
            $this->db->join('pathology_category pc', 'p.pathology_category_id = pc.id', 'left');
            $this->db->order_by('pc.category_name, p.test_name');
            
            $query = $this->db->get();
            $results = $query->result_array();
        }
        
        // Organize by category
        foreach ($results as $test) {
            $category = $test['category_name'] ?: 'General Tests';
            if (!isset($categorized_tests[$category])) {
                $categorized_tests[$category] = [];
            }
            $categorized_tests[$category][] = [
                'id' => $test['id'],
                'name' => $test['name'],
                'charge' => (float)$test['charge'],
                'method' => $test['method'] ?? '',
                'sample_type' => $test['sample_type'] ?? ''
            ];
        }
        
    } catch (Exception $e) {
        log_message('error', 'Database error in getTestsFromDatabase: ' . $e->getMessage());
        return [];
    }
    
    return $categorized_tests;
}

/**
 * Static test data from Excel file as fallback
 */
private function getStaticTestData() {
    return [
        "Hematology" => [
            ["name" => "Complete Blood Count (CBC)", "charge" => 1200, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "Blood Culture", "charge" => 6000, "method" => "Culture", "sample_type" => "Blood"],
            ["name" => "Blood ethanol levels", "charge" => 5000, "method" => "Enzymatic", "sample_type" => "Blood"],
            ["name" => "Blood Grouping", "charge" => 400, "method" => "Agglutination", "sample_type" => "Blood"],
            ["name" => "ESR", "charge" => 500, "method" => "Westergren", "sample_type" => "Blood"],
            ["name" => "Full Blood Count", "charge" => 1200, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "Hemoglobin", "charge" => 500, "method" => "Spectrophotometry", "sample_type" => "Blood"],
            ["name" => "Platelet Count", "charge" => 800, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "White Blood Cell Count", "charge" => 600, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "Reticulocyte Count", "charge" => 1000, "method" => "Flow Cytometry", "sample_type" => "Blood"],
            ["name" => "Peripheral Blood Film", "charge" => 800, "method" => "Microscopy", "sample_type" => "Blood"]
        ],
        "Endocrinology" => [
            ["name" => "ANTI THYROID PEROXIDASE ANTIBODIES( TPO )", "charge" => 3000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FT3", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FT4", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "TSH", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "T3", "charge" => 1800, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "T4", "charge" => 1800, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Thyroid Function Tests", "charge" => 4500, "method" => "ELISA Panel", "sample_type" => "Serum"],
            ["name" => "PTH", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Cortisol", "charge" => 3500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Growth Hormone", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        "Diabetes" => [
            ["name" => "Fluid glucose", "charge" => 100, "method" => "Enzymatic", "sample_type" => "Various"],
            ["name" => "HBA1C", "charge" => 3000, "method" => "HPLC", "sample_type" => "Blood"],
            ["name" => "Fasting Blood Sugar", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Random Blood Sugar", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Oral Glucose Tolerance Test", "charge" => 1500, "method" => "Enzymatic", "sample_type" => "Serum"]
        ],
        "Lipid Profile" => [
            ["name" => "ANTI PHOSPHOLIPID ANTIBODIES", "charge" => 6000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "HDL Cholesterol", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "LIPID PROFILE", "charge" => 3000, "method" => "Enzymatic Panel", "sample_type" => "Serum"],
            ["name" => "LDL Cholesterol", "charge" => 800, "method" => "Calculated/Direct", "sample_type" => "Serum"],
            ["name" => "Total Cholesterol", "charge" => 600, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Triglycerides", "charge" => 700, "method" => "Enzymatic", "sample_type" => "Serum"]
        ],
        "Hepatology" => [
            ["name" => "ALT/GPT (Alanine Aminotransferase.)", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "APTT Activated partial Thromboplastin Time", "charge" => 1500, "method" => "Coagulation", "sample_type" => "Plasma"],
            ["name" => "HEPATITIS  C  VIRUS ANTIBODY", "charge" => 1000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "HEPATITIS B SURFACE ANTIGEN", "charge" => 1000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "AST/GOT", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Bilirubin Total", "charge" => 600, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Bilirubin Direct", "charge" => 600, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Alkaline Phosphatase", "charge" => 700, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "GGT", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Total Protein", "charge" => 500, "method" => "Biuret", "sample_type" => "Serum"],
            ["name" => "Albumin", "charge" => 500, "method" => "BCG", "sample_type" => "Serum"]
        ],
        "Urology" => [
            ["name" => "Urinalysis", "charge" => 400, "method" => "Dipstick/Microscopy", "sample_type" => "Urine"],
            ["name" => "URINALYSIS (corporate)", "charge" => 500, "method" => "Dipstick/Microscopy", "sample_type" => "Urine"],
            ["name" => "URINE CREAT", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Urine"],
            ["name" => "Urine Culture", "charge" => 2000, "method" => "Culture", "sample_type" => "Urine"],
            ["name" => "Urine Microscopy", "charge" => 600, "method" => "Microscopy", "sample_type" => "Urine"],
            ["name" => "24hr Urine Protein", "charge" => 1500, "method" => "Turbidimetric", "sample_type" => "24hr Urine"],
            ["name" => "Urine Glucose", "charge" => 300, "method" => "Enzymatic", "sample_type" => "Urine"],
            ["name" => "Urine Ketones", "charge" => 400, "method" => "Dipstick", "sample_type" => "Urine"]
        ],
        "Microbiology" => [
            ["name" => "CULTURE & SENSITIVITY", "charge" => 2050, "method" => "Culture", "sample_type" => "Various"],
            ["name" => "Fluid Culture & Sensitivity", "charge" => 2500, "method" => "Culture", "sample_type" => "Body Fluid"],
            ["name" => "Microscopy, culture & Sensitivity MCS", "charge" => 1450, "method" => "Culture", "sample_type" => "Various"],
            ["name" => "Stool Culture", "charge" => 2000, "method" => "Culture", "sample_type" => "Stool"],
            ["name" => "Throat Swab", "charge" => 1500, "method" => "Culture", "sample_type" => "Throat Swab"],
            ["name" => "Wound Swab", "charge" => 1800, "method" => "Culture", "sample_type" => "Wound Swab"]
        ],
        "Vitamins & Minerals" => [
            ["name" => "Folate (Folic Acid)", "charge" => 2700, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin B 12", "charge" => 6000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin B12", "charge" => 3450, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin D", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Iron Studies", "charge" => 2500, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Calcium", "charge" => 600, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Magnesium", "charge" => 800, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Phosphorus", "charge" => 700, "method" => "Spectrophotometry", "sample_type" => "Serum"]
        ],
        "Cardiology" => [
            ["name" => "TROPONIN", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "CK-MB", "charge" => 2500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "LDH", "charge" => 1200, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "CPK", "charge" => 1500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "BNP", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        "Health Packages" => [
            ["name" => "15000 comprehensive female wellness", "charge" => 15000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "18000 Female Comprehensive Wellness", "charge" => 18000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "BAHRAIN MEDICAL PACKAGE", "charge" => 4700, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Basic Health Screen", "charge" => 3500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Executive Health Package", "charge" => 8500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Cardiac Risk Assessment", "charge" => 6500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Diabetes Screen", "charge" => 2500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Liver Function Package", "charge" => 3200, "method" => "Combined", "sample_type" => "Various"]
        ]
    ];
}

/**
 * Count total tests across all categories
 */
private function countTotalTests($tests) {
    $count = 0;
    foreach ($tests as $category => $categoryTests) {
        $count += count($categoryTests);
    }
    return $count;
}

/**
 * Save multiple laboratory tests (enhanced version)
 */
public function saveMultipleTests() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        $selected_tests = json_decode($this->input->post('selected_tests'), true);
        $billing_info = json_decode($this->input->post('billing_info'), true);
        $doctor_id = $this->input->post('doctor_id');
        $test_date = $this->input->post('test_date') ?: date('Y-m-d');
        $priority = $this->input->post('priority') ?: 'normal';
        $payment_method = $this->input->post('payment_method') ?: 'cash';
        $notes = $this->input->post('notes') ?: '';
        
        if (!$patient_id || empty($selected_tests)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID and selected tests are required'
            ]);
            return;
        }
        
        $this->db->trans_start();
        
        $lab_ids = [];
        $total_amount = 0;
        
        foreach ($selected_tests as $test) {
            $base_cost = $test['charge'];
            $priority_fee = 0;
            
            if ($priority === 'urgent') {
                $priority_fee = $base_cost * 0.5;
            } elseif ($priority === 'stat') {
                $priority_fee = $base_cost * 1.0;
            }
            
            $total_cost = $base_cost + $priority_fee;
            
            $test_discount = 0;
            if ($billing_info['subtotal'] > 0) {
                $test_discount = ($billing_info['discountAmount'] / $billing_info['subtotal']) * $total_cost;
            }
            
            $discounted_amount = $total_cost - $test_discount;
            $test_tax = ($discounted_amount * $billing_info['taxPercent']) / 100;
            $final_amount = $discounted_amount + $test_tax;
            
            // Build lab data - ONLY include columns that exist
            $lab_data = [
                'patient_id' => $patient_id,
                'test_name' => $test['name'],
                'base_cost' => $base_cost,
                'final_amount' => $final_amount,
                'result_status' => 'pending',
                'test_date' => $test_date,
                'notes' => $notes,
                'tested_by' => $this->getStaffId(),
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            // Add optional columns only if they exist in the table
            $fields = $this->db->field_data('laboratory');
            $existing_columns = array_column($fields, 'name');
            
            if (in_array('category_name', $existing_columns)) {
                $lab_data['category_name'] = $test['category'];
            }
            if (in_array('priority', $existing_columns)) {
                $lab_data['priority'] = $priority;
            }
            if (in_array('priority_fee', $existing_columns)) {
                $lab_data['priority_fee'] = $priority_fee;
            }
            if (in_array('total_cost', $existing_columns)) {
                $lab_data['total_cost'] = $total_cost;
            }
            if (in_array('discount_percentage', $existing_columns)) {
                $lab_data['discount_percentage'] = $billing_info['discountPercent'];
            }
            if (in_array('discount_amount', $existing_columns)) {
                $lab_data['discount_amount'] = $test_discount;
            }
            if (in_array('tax_amount', $existing_columns)) {
                $lab_data['tax_amount'] = $test_tax;
            }
            if (in_array('payment_method', $existing_columns)) {
                $lab_data['payment_method'] = $payment_method;
            }
            if (in_array('verification_status', $existing_columns)) {
                $lab_data['verification_status'] = 'unverified';
            }
            if (in_array('doctor_id', $existing_columns) && $doctor_id) {
                $lab_data['doctor_id'] = $doctor_id;
            }
            if (in_array('test_id', $existing_columns) && isset($test['id'])) {
                $lab_data['test_id'] = $test['id'];
            }
            if (in_array('short_name', $existing_columns)) {
                $lab_data['short_name'] = isset($test['short_name']) ? $test['short_name'] : $this->generateShortName($test['name']);
            }
            
            $this->db->insert('laboratory', $lab_data);
            $lab_ids[] = $this->db->insert_id();
            $total_amount += $final_amount;
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save laboratory tests'
            ]);
        } else {
            echo json_encode([
                'status' => 'success',
                'message' => 'Laboratory tests saved successfully',
                'lab_ids' => $lab_ids,
                'total_amount' => $total_amount,
                'test_count' => count($selected_tests)
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error in saveMultipleTests: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving laboratory tests: ' . $e->getMessage()
        ]);
    }
}

/**
 * Create billing record for laboratory tests
 */
private function createLabBillingRecord($patient_id, $lab_ids, $billing_info, $total_amount) {
    // Check if billing table exists, create if not
    if (!$this->db->table_exists('lab_billing')) {
        $this->createLabBillingTable();
    }
    
    $billing_data = [
        'patient_id' => $patient_id,
        'lab_ids' => json_encode($lab_ids),
        'subtotal' => $billing_info['subtotal'],
        'discount_percentage' => $billing_info['discountPercent'],
        'discount_amount' => $billing_info['discountAmount'],
        'tax_percentage' => $billing_info['taxPercent'],
        'tax_amount' => $billing_info['taxAmount'],
        'total_amount' => $total_amount,
        'payment_status' => 'pending',
        'created_at' => date('Y-m-d H:i:s')
    ];
    
    $this->db->insert('lab_billing', $billing_data);
    return $this->db->insert_id();
}

/**
 * Create lab billing table
 */
private function createLabBillingTable() {
    $sql = "
    CREATE TABLE IF NOT EXISTS `lab_billing` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `patient_id` int(11) NOT NULL,
        `lab_ids` json DEFAULT NULL,
        `subtotal` decimal(10,2) DEFAULT 0.00,
        `discount_percentage` decimal(5,2) DEFAULT 0.00,
        `discount_amount` decimal(10,2) DEFAULT 0.00,
        `tax_percentage` decimal(5,2) DEFAULT 0.00,
        `tax_amount` decimal(10,2) DEFAULT 0.00,
        `total_amount` decimal(10,2) DEFAULT 0.00,
        `payment_status` enum('pending','partial','paid') DEFAULT 'pending',
        `amount_paid` decimal(10,2) DEFAULT 0.00,
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        INDEX `idx_patient` (`patient_id`),
        INDEX `idx_payment_status` (`payment_status`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3;
    ";
    
    $this->db->query($sql);
}

/**
 * Search patients for test selection
 */
public function searchPatients() {
    header('Content-Type: application/json');
    
    try {
        $search_term = $this->input->post('search_term');
        
        if (strlen($search_term) < 2) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Search term must be at least 2 characters'
            ]);
            return;
        }
        
        $this->db->select('id, patient_name, mobileno, age, gender');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->group_start();
        $this->db->like('patient_name', $search_term);
        $this->db->or_like('mobileno', $search_term);
        $this->db->group_end();
        $this->db->order_by('patient_name');
        $this->db->limit(15);
        
        $query = $this->db->get();
        $patients = $query->result_array();
        
        echo json_encode([
            'status' => 'success',
            'data' => $patients,
            'count' => count($patients)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error searching patients: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get test categories with counts
 */
private function getTestCategories() {
    try {
        // Get categories from lab_tests table
        $this->db->select('category_id, COUNT(*) as test_count');
        $this->db->from('lab_tests');
        $this->db->where('is_active', 1);
        $this->db->group_by('category_id');
        $this->db->order_by('category_id');
        
        $query = $this->db->get();
        
        if (!$query) {
            return [];
        }
        
        $results = $query->result_array();
        
        // Category mapping
        $category_names = [
            1 => 'General Tests',
            2 => 'Health Packages', 
            3 => 'Hepatology',
            4 => 'Lipid Profile',
            5 => 'Endocrinology',
            6 => 'Hematology',
            7 => 'Microbiology',
            8 => 'Diabetes',
            9 => 'Vitamins & Minerals',
            10 => 'Cardiology',
            11 => 'Urology'
        ];
        
        $categories = [];
        foreach ($results as $result) {
            $category_id = $result['category_id'];
            $category_name = isset($category_names[$category_id]) ? 
                           $category_names[$category_id] : 
                           'Category ' . $category_id;
            
            $categories[] = [
                'category_name' => $category_name,
                'test_count' => (int)$result['test_count']
            ];
        }
        
        return $categories;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting test categories: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get recent patients for quick selection
 */
private function getRecentPatients($limit = 25) {
    try {
        $this->db->select('id, patient_name, mobileno, age, gender');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting recent patients: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get laboratory tests for a specific patient
 */
public function getPatientLabTests() {
    header('Content-Type: application/json');
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Get lab tests from the detailed view or laboratory table
        if ($this->db->table_exists('laboratory_detailed_view')) {
            $this->db->select('*');
            $this->db->from('laboratory_detailed_view');
        } else {
            $this->db->select('l.*, p.patient_name');
            $this->db->from('laboratory l');
            $this->db->join('patients p', 'l.patient_id = p.id', 'left');
        }
        
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit(50);
        
        $query = $this->db->get();
        $lab_tests = $query->result_array();
        
        echo json_encode([
            'status' => 'success',
            'data' => $lab_tests,
            'count' => count($lab_tests)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading patient lab tests: ' . $e->getMessage()
        ]);
    }
}


/**
 * Get lab statistics for dashboard
 */
/*public function getLabStatistics() {
    header('Content-Type: application/json');
    
    try {
        $today = date('Y-m-d');
        $this_month = date('Y-m');
        
        // Today's tests
        $this->db->where('DATE(test_date)', $today);
        $today_tests = $this->db->count_all_results('laboratory');
        
        // This month's tests
        $this->db->where('DATE_FORMAT(test_date, "%Y-%m")', $this_month);
        $month_tests = $this->db->count_all_results('laboratory');
        
        // Pending results
        $this->db->where('result_status', 'pending');
        $pending_tests = $this->db->count_all_results('laboratory');
        
        // Completed today
        $this->db->where('DATE(updated_at)', $today);
        $this->db->where('result_status', 'completed');
        $completed_today = $this->db->count_all_results('laboratory');
        
        // Revenue this month
        $this->db->select('SUM(final_amount) as total_revenue');
        $this->db->where('DATE_FORMAT(test_date, "%Y-%m")', $this_month);
        $this->db->where('result_status !=', 'cancelled');
        $query = $this->db->get('laboratory');
        $revenue = $query->row()->total_revenue ?: 0;
        
        // Test categories distribution
        $this->db->select('category_name, COUNT(*) as count');
        $this->db->where('DATE_FORMAT(test_date, "%Y-%m")', $this_month);
        $this->db->group_by('category_name');
        $this->db->order_by('count', 'DESC');
        $category_stats = $this->db->get('laboratory')->result_array();
        
        echo json_encode([
            'status' => 'success',
            'data' => [
                'today_tests' => $today_tests,
                'month_tests' => $month_tests,
                'pending_tests' => $pending_tests,
                'completed_today' => $completed_today,
                'month_revenue' => $revenue,
                'category_distribution' => $category_stats
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading statistics: ' . $e->getMessage()
        ]);
    }
}*/

/**
 * Generate lab report
 */
public function generateLabReport() {
    try {
        $lab_id = $this->input->get('lab_id');
        $patient_id = $this->input->get('patient_id');
        
        if ($lab_id) {
            // Single test report
            $this->generateSingleTestReport($lab_id);
        } elseif ($patient_id) {
            // Patient's all tests report
            $this->generatePatientLabReport($patient_id);
        } else {
            show_error('Lab ID or Patient ID required');
        }
        
    } catch (Exception $e) {
        show_error('Error generating report: ' . $e->getMessage());
    }
}

/**
 * Generate single test report
 */
private function generateSingleTestReport($lab_id) {
    // Get test details
    $this->db->select('l.*, p.patient_name, p.mobileno, p.age, p.gender, p.id');
    $this->db->from('laboratory l');
    $this->db->join('patients p', 'l.patient_id = p.id', 'left');
    $this->db->where('l.id', $lab_id);
    $lab_test = $this->db->get()->row_array();
    
    if (!$lab_test) {
        show_error('Lab test not found');
    }
    
    // For now, show the data as JSON (you can integrate with PDF library later)
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'data' => $lab_test,
        'hospital_info' => $this->getHospitalInfo()
    ]);
}

/**
 * Generate patient lab report
 */
private function generatePatientLabReport($patient_id) {
    // Get patient details
    $this->db->where('id', $patient_id);
    $patient = $this->db->get('patients')->row_array();
    
    if (!$patient) {
        show_error('Patient not found');
    }
    
    // Get all lab tests for patient
    $this->db->select('*');
    $this->db->from('laboratory');
    $this->db->where('patient_id', $patient_id);
    $this->db->order_by('test_date', 'DESC');
    $lab_tests = $this->db->get()->result_array();
    
    header('Content-Type: application/json');
    echo json_encode([
        'status' => 'success',
        'patient' => $patient,
        'lab_tests' => $lab_tests,
        'hospital_info' => $this->getHospitalInfo()
    ]);
}

/**
 * Get hospital information for reports
 */
/*private function getHospitalInfo() {
    // You can get this from your settings table or config
    return [
        'name' => 'Your Hospital Name',
        'address' => 'Hospital Address',
        'phone' => '+254 XXX XXX XXX',
        'email' => 'info@hospital.com',
        'logo' => base_url('assets/images/logo.png')
    ];
}*/

/**
 * Create lab_tests table (run this once to set up the table)
 */
public function createLabTestsTable() {
    try {
        $sql = "
        CREATE TABLE IF NOT EXISTS `lab_tests` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `category_id` int(11) DEFAULT NULL,
            `category_name` varchar(100) NOT NULL,
            `test_name` varchar(255) NOT NULL,
            `short_name` varchar(100) DEFAULT NULL,
            `description` text DEFAULT NULL,
            `cost` decimal(10,2) NOT NULL DEFAULT 0.00,
            `method` varchar(100) DEFAULT NULL,
            `sample_type` varchar(100) DEFAULT NULL,
            `reference_range` varchar(200) DEFAULT NULL,
            `unit` varchar(50) DEFAULT NULL,
            `report_days` int(11) DEFAULT 1,
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            INDEX `idx_category` (`category_name`),
            INDEX `idx_active` (`is_active`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3;
        ";
        
        $this->db->query($sql);
        
        // Insert sample test data
        $this->insertSampleTestData();
        
        echo json_encode(['status' => 'success', 'message' => 'Lab tests table created successfully']);
        
    } catch (Exception $e) {
        echo json_encode(['status' => 'error', 'message' => 'Error creating table: ' . $e->getMessage()]);
    }
}

/**
 * Insert sample test data from your Excel file
 */
private function insertSampleTestData() {
    $static_data = $this->getStaticTestData();
    
    foreach ($static_data as $category => $tests) {
        foreach ($tests as $test) {
            $data = [
                'category_name' => $category,
                'test_name' => $test['name'],
                'short_name' => $this->generateShortName($test['name']),
                'cost' => $test['charge'],
                'method' => $test['method'] ?? '',
                'sample_type' => $test['sample_type'] ?? '',
                'is_active' => 1
            ];
            
            // Check if test already exists
            $this->db->where('test_name', $test['name']);
            $this->db->where('category_name', $category);
            $existing = $this->db->get('lab_tests')->row();
            
            if (!$existing) {
                $this->db->insert('lab_tests', $data);
            }
        }
    }
}

/**
 * Generate short name for test
 */
private function generateShortName($test_name) {
    // Create abbreviation from test name
    $words = explode(' ', $test_name);
    $short_name = '';
    
    foreach ($words as $word) {
        if (strlen($word) > 2) {
            $short_name .= strtoupper(substr($word, 0, 3));
        }
    }
    
    return substr($short_name, 0, 10);
}

/**
 * Get tests by category (AJAX endpoint)
 */
public function getTestsByCategory() {
    header('Content-Type: application/json');
    
    try {
        $category_name = $this->input->post('category_name');
        
        if (!$category_name) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Category name required'
            ]);
            return;
        }
        
        $tests = [];
        $static_data = $this->getStaticTestData();
        
        if (isset($static_data[$category_name])) {
            $tests = $static_data[$category_name];
        }
        
        // Also try to get from database if table exists
        if ($this->db->table_exists('lab_tests')) {
            $this->db->select('id, test_name as name, cost as charge, method, sample_type');
            $this->db->where('category_name', $category_name);
            $this->db->where('is_active', 1);
            $this->db->order_by('test_name');
            
            $query = $this->db->get('lab_tests');
            $db_tests = $query->result_array();
            
            if (!empty($db_tests)) {
                $tests = $db_tests;
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'data' => $tests,
            'category' => $category_name,
            'count' => count($tests)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading tests: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get current staff ID (helper method)
 */
private function getStaffId() {
    try {
        if (method_exists($this->customlib, 'getStaffID')) {
            return $this->customlib->getStaffID();
        } else {
            $staff_id = $this->session->userdata('admin_id');
            return $staff_id ?: 1;
        }
    } catch (Exception $e) {
        log_message('error', 'Error getting staff ID: ' . $e->getMessage());
        return 1;
    }
}

/**
 * Delete lab test
 */
public function deleteLabTest() {
    header('Content-Type: application/json');
    
    try {
        $lab_id = $this->input->post('lab_id');
        
        if (!$lab_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab ID required'
            ]);
            return;
        }
        
        // Check if test exists
        $this->db->where('id', $lab_id);
        $lab_test = $this->db->get('laboratory')->row();
        
        if (!$lab_test) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab test not found'
            ]);
            return;
        }
        
        // Don't actually delete, just mark as cancelled
        $this->db->where('id', $lab_id);
        $this->db->update('laboratory', [
            'result_status' => 'cancelled',
            'updated_at' => date('Y-m-d H:i:s')
        ]);
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Lab test cancelled successfully'
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error cancelling lab test: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get lab test details
 */
public function getLabTestDetails() {
    header('Content-Type: application/json');
    
    try {
        $lab_id = $this->input->post('lab_id');
        
        if (!$lab_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab ID required'
            ]);
            return;
        }
        
        // Get test details with patient info
        $this->db->select('l.*, p.patient_name, p.mobileno, p.age, p.gender');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'l.patient_id = p.id', 'left');
        $this->db->where('l.id', $lab_id);
        
        $lab_test = $this->db->get()->row_array();
        
        if (!$lab_test) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab test not found'
            ]);
            return;
        }
        
        echo json_encode([
            'status' => 'success',
            'data' => $lab_test
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab test details: ' . $e->getMessage()
        ]);
    }
}

public function importExcelData() {
    echo "<h1>Importing Excel Laboratory Tests Data</h1>";
    echo "<style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { color: green; font-weight: bold; }
        .error { color: red; font-weight: bold; }
        .info { color: blue; }
        .section { margin: 20px 0; padding: 15px; border: 1px solid #ccc; border-radius: 5px; }
    </style>";
    
    try {
        // Step 1: Create/update categories
        echo "<div class='section'>";
        echo "<h2>Step 1: Setting up Test Categories</h2>";
        $this->setupTestCategories();
        echo "</div>";
        
        // Step 2: Import all tests
        echo "<div class='section'>";
        echo "<h2>Step 2: Importing Laboratory Tests</h2>";
        $this->importAllTests();
        echo "</div>";
        
        // Step 3: Verify import
        echo "<div class='section'>";
        echo "<h2>Step 3: Verification</h2>";
        $this->verifyImport();
        echo "</div>";
        
        echo "<div class='section success'>";
        echo "<h2>✓ Import Complete!</h2>";
        echo "<p>All Excel data has been imported successfully.</p>";
        echo "<p><a href='" . base_url('admin/laboratory/test-selection') . "'>Go to Test Selection</a></p>";
        echo "</div>";
        
    } catch (Exception $e) {
        echo "<div class='error'>";
        echo "<h2>Import Error</h2>";
        echo "<p>Error: " . $e->getMessage() . "</p>";
        echo "</div>";
    }
}

/**
 * Setup test categories in the database
 */
private function setupTestCategories() {
    // First check if we have a categories table
    $category_tables = ['pathology_category', 'test_categories', 'lab_categories', 'categories'];
    $category_table = null;
    
    foreach ($category_tables as $table) {
        if ($this->db->table_exists($table)) {
            $category_table = $table;
            break;
        }
    }
    
    // If no category table exists, create one
    if (!$category_table) {
        $this->createCategoriesTable();
        $category_table = 'lab_categories';
    }
    
    echo "<p class='info'>Using category table: $category_table</p>";
    
    // Define categories from your Excel file
    $categories = [
        1 => 'General Tests',
        2 => 'Health Packages', 
        3 => 'Hepatology',
        4 => 'Lipid Profile',
        5 => 'Endocrinology',
        6 => 'Hematology',
        7 => 'Microbiology',
        8 => 'Diabetes',
        9 => 'Vitamins & Minerals',
        10 => 'Cardiology',
        11 => 'Urology'
    ];
    
    foreach ($categories as $id => $name) {
        // Check if category exists
        $this->db->where('id', $id);
        $existing = $this->db->get($category_table)->row();
        
        if (!$existing) {
            $data = [
                'id' => $id,
                'category_name' => $name,
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert($category_table, $data);
            echo "<p class='success'>✓ Created category: $name</p>";
        } else {
            echo "<p class='info'>Category already exists: $name</p>";
        }
    }
}

/**
 * Create categories table if it doesn't exist
 */
private function createCategoriesTable() {
    $sql = "
    CREATE TABLE IF NOT EXISTS `lab_categories` (
        `id` int(11) NOT NULL AUTO_INCREMENT,
        `category_name` varchar(100) NOT NULL,
        `description` text DEFAULT NULL,
        `is_active` tinyint(1) DEFAULT 1,
        `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
        `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
        PRIMARY KEY (`id`),
        UNIQUE KEY `category_name` (`category_name`)
    ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
    ";
    
    $this->db->query($sql);
    echo "<p class='success'>✓ Created lab_categories table</p>";
}

/**
 * Import all tests from Excel data
 */
private function importAllTests() {
    // Get all Excel test data
    $excel_data = $this->getCompleteExcelData();
    
    $total_imported = 0;
    $total_updated = 0;
    $errors = 0;
    
    foreach ($excel_data as $category_name => $tests) {
        // Get category ID
        $category_id = $this->getCategoryIdByName($category_name);
        
        echo "<h3>Importing $category_name (Category ID: $category_id)</h3>";
        
        foreach ($tests as $test) {
            try {
                // Check if test already exists
                $this->db->where('test_name', $test['name']);
                $this->db->where('category_id', $category_id);
                $existing = $this->db->get('lab_tests')->row();
                
                $test_data = [
                    'test_name' => $test['name'],
                    'short_name' => $this->generateShortName($test['name']),
                    'category_id' => $category_id,
                    'cost' => $test['charge'],
                    'method' => $test['method'] ?? 'Standard',
                    'sample_type' => $test['sample_type'] ?? 'Various',
                    'test_type' => $this->determineTestType($test['name']),
                    'description' => $test['description'] ?? null,
                    'is_active' => 1,
                    'report_days' => 1
                ];
                
                if (!$existing) {
                    // Insert new test
                    $test_data['created_at'] = date('Y-m-d H:i:s');
                    $this->db->insert('lab_tests', $test_data);
                    $total_imported++;
                    
                    if ($total_imported % 50 == 0) {
                        echo "<p class='info'>Imported $total_imported tests...</p>";
                    }
                } else {
                    // Update existing test
                    $test_data['updated_at'] = date('Y-m-d H:i:s');
                    $this->db->where('id', $existing->id);
                    $this->db->update('lab_tests', $test_data);
                    $total_updated++;
                }
                
            } catch (Exception $e) {
                $errors++;
                echo "<p class='error'>Error importing {$test['name']}: " . $e->getMessage() . "</p>";
            }
        }
        
        echo "<p class='success'>✓ Completed $category_name</p>";
    }
    
    echo "<div class='success'>";
    echo "<h3>Import Summary:</h3>";
    echo "<p>Total tests imported: $total_imported</p>";
    echo "<p>Total tests updated: $total_updated</p>";
    echo "<p>Errors: $errors</p>";
    echo "</div>";
}

/**
 * Get category ID by name
 */
private function getCategoryIdByName($category_name) {
    $category_map = [
        'General Tests' => 1,
        'Health Packages' => 2,
        'Hepatology' => 3,
        'Lipid Profile' => 4,
        'Endocrinology' => 5,
        'Hematology' => 6,
        'Microbiology' => 7,
        'Diabetes' => 8,
        'Vitamins & Minerals' => 9,
        'Cardiology' => 10,
        'Urology' => 11
    ];
    
    return isset($category_map[$category_name]) ? $category_map[$category_name] : 1;
}

/**
 * Determine test type based on test name
 */
private function determineTestType($test_name) {
    $test_name_lower = strtolower($test_name);
    
    if (strpos($test_name_lower, 'count') !== false || 
        strpos($test_name_lower, 'level') !== false || 
        strpos($test_name_lower, 'glucose') !== false ||
        strpos($test_name_lower, 'cholesterol') !== false) {
        return 'quantitative';
    } elseif (strpos($test_name_lower, 'positive') !== false || 
              strpos($test_name_lower, 'negative') !== false ||
              strpos($test_name_lower, 'antigen') !== false ||
              strpos($test_name_lower, 'antibody') !== false) {
        return 'qualitative';
    } else {
        return 'descriptive';
    }
}

/**
 * Helper methods
 */
private function getRecentPatientsSafe($limit = 25) {
    try {
        $this->db->select('id, patient_name, mobileno, age, gender');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query ? $query->result_array() : [];
        
    } catch (Exception $e) {
        log_message('error', 'Error getting recent patients: ' . $e->getMessage());
        return [];
    }
}

private function getLabStaffSafe() {
    try {
        if (method_exists($this->staff_model, 'getStaffbyrole')) {
            $doctors = $this->staff_model->getStaffbyrole(2);
            if (empty($doctors)) {
                $doctors = $this->staff_model->getStaffbyrole(3);
            }
        } else {
            $this->db->select('id, name, surname, employee_id');
            $this->db->from('staff');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('name', 'ASC');
            $query = $this->db->get();
            $doctors = $query ? $query->result_array() : [];
        }
        
        return $doctors;
        
    } catch (Exception $e) {
        log_message('error', 'Error getting lab staff: ' . $e->getMessage());
        return [];
    }
}

/**
 * Verify the import
 */
private function verifyImport() {
    try {
        // Count total tests
        $this->db->select('COUNT(*) as total');
        $total_query = $this->db->get('lab_tests');
        $total_tests = $total_query->row()->total;
        
        echo "<p class='success'>✓ Total tests in database: $total_tests</p>";
        
        // Count by category
        $this->db->select('category_id, COUNT(*) as count');
        $this->db->group_by('category_id');
        $this->db->order_by('category_id');
        $category_counts = $this->db->get('lab_tests')->result_array();
        
        echo "<h4>Tests by Category:</h4>";
        $category_names = [
            1 => 'General Tests', 2 => 'Health Packages', 3 => 'Hepatology',
            4 => 'Lipid Profile', 5 => 'Endocrinology', 6 => 'Hematology',
            7 => 'Microbiology', 8 => 'Diabetes', 9 => 'Vitamins & Minerals',
            10 => 'Cardiology', 11 => 'Urology'
        ];
        
        foreach ($category_counts as $cat) {
            $cat_name = $category_names[$cat['category_id']] ?? 'Unknown';
            echo "<p>$cat_name (ID: {$cat['category_id']}): {$cat['count']} tests</p>";
        }
        
        // Show sample tests
        echo "<h4>Sample Tests:</h4>";
        $this->db->select('test_name, cost, category_id');
        $this->db->limit(10);
        $sample_tests = $this->db->get('lab_tests')->result_array();
        
        foreach ($sample_tests as $test) {
            $cat_name = $category_names[$test['category_id']] ?? 'Unknown';
            echo "<p>- {$test['test_name']} (KES {$test['cost']}) - $cat_name</p>";
        }
        
    } catch (Exception $e) {
        echo "<p class='error'>Verification error: " . $e->getMessage() . "</p>";
    }
}

public function setupLabSystem() {
    echo "<h1>Setting up Laboratory Test System</h1>";
    echo "<style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .success { color: green; font-weight: bold; }
        .error { color: red; font-weight: bold; }
        .info { color: blue; }
        .section { margin: 20px 0; padding: 15px; border: 1px solid #ccc; border-radius: 5px; }
    </style>";
    
    try {
        // Step 1: Check/Update laboratory table structure
        echo "<div class='section'>";
        echo "<h2>Step 1: Updating Laboratory Table Structure</h2>";
        $this->updateLaboratoryTable();
        echo "</div>";
        
        // Step 2: Clear and import fresh Excel data
        echo "<div class='section'>";
        echo "<h2>Step 2: Importing Excel Test Data</h2>";
        $this->importExcelTestsToDatabase();
        echo "</div>";
        
        // Step 3: Test the system
        echo "<div class='section'>";
        echo "<h2>Step 3: Testing System</h2>";
        $this->testSystemSetup();
        echo "</div>";
        
        echo "<div class='section success'>";
        echo "<h2>✓ Setup Complete!</h2>";
        echo "<p>Your laboratory test selection system is now ready.</p>";
        echo "<p><a href='" . base_url('admin/laboratory/test-selection') . "' class='btn btn-primary'>Go to Test Selection</a></p>";
        echo "</div>";
        
    } catch (Exception $e) {
        echo "<div class='error'>";
        echo "<h2>Setup Error</h2>";
        echo "<p>Error: " . $e->getMessage() . "</p>";
        echo "</div>";
    }
}

/**
 * Update laboratory table to include missing columns
 */
private function updateLaboratoryTable() {
    try {
        // Get current laboratory table structure
        $fields = $this->db->field_data('laboratory');
        $existing_columns = array_column($fields, 'name');
        
        echo "<p class='info'>Current laboratory table has " . count($existing_columns) . " columns</p>";
        
        // Define columns that need to be added
        $new_columns = [
            'category_name' => "VARCHAR(100) DEFAULT NULL COMMENT 'Test category name'",
            'priority' => "ENUM('normal','urgent','stat') DEFAULT 'normal' COMMENT 'Test priority'",
            'priority_fee' => "DECIMAL(10,2) DEFAULT 0.00 COMMENT 'Additional priority fee'",
            'total_cost' => "DECIMAL(10,2) DEFAULT 0.00 COMMENT 'Cost including priority fees'",
            'discount_percentage' => "DECIMAL(5,2) DEFAULT 0.00 COMMENT 'Discount percentage'",
            'discount_amount' => "DECIMAL(10,2) DEFAULT 0.00 COMMENT 'Discount amount'",
            'tax_amount' => "DECIMAL(10,2) DEFAULT 0.00 COMMENT 'Tax amount'",
            'payment_method' => "ENUM('cash','card','insurance','credit') DEFAULT 'cash' COMMENT 'Payment method'",
            'verification_status' => "ENUM('verified','unverified','skipped') DEFAULT 'unverified' COMMENT 'Verification status'"
        ];
        
        foreach ($new_columns as $column_name => $column_definition) {
            if (!in_array($column_name, $existing_columns)) {
                try {
                    $sql = "ALTER TABLE laboratory ADD COLUMN $column_name $column_definition";
                    $this->db->query($sql);
                    echo "<p class='success'>✓ Added column: $column_name</p>";
                } catch (Exception $e) {
                    echo "<p class='error'>✗ Could not add $column_name: " . $e->getMessage() . "</p>";
                }
            } else {
                echo "<p class='info'>Column $column_name already exists</p>";
            }
        }
        
    } catch (Exception $e) {
        echo "<p class='error'>Error updating laboratory table: " . $e->getMessage() . "</p>";
    }
}

private function importExcelTestsToDatabase() {
    try {
        // Clear existing tests first
        $this->db->truncate('lab_tests');
        echo "<p class='info'>Cleared existing test data</p>";
        
        // Get Excel test data
        $excel_data = $this->getCompleteExcelTestData();
        
        $total_imported = 0;
        $category_mapping = [
            'General Tests' => 1,
            'Health Packages' => 2,
            'Hepatology' => 3,
            'Lipid Profile' => 4,
            'Endocrinology' => 5,
            'Hematology' => 6,
            'Microbiology' => 7,
            'Diabetes' => 8,
            'Vitamins & Minerals' => 9,
            'Cardiology' => 10,
            'Urology' => 11
        ];
        
        foreach ($excel_data as $category_name => $tests) {
            $category_id = $category_mapping[$category_name] ?? 1;
            
            foreach ($tests as $test) {
                $test_data = [
                    'test_name' => $test['name'],
                    'short_name' => $this->generateShortName($test['name']),
                    'category_id' => $category_id,
                    'cost' => $test['charge'],
                    'method' => $test['method'] ?? 'Standard',
                    'sample_type' => $test['sample_type'] ?? 'Various',
                    'test_type' => $this->determineTestType($test['name']),
                    'is_active' => 1,
                    'report_days' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                ];
                
                $this->db->insert('lab_tests', $test_data);
                $total_imported++;
            }
        }
        
        echo "<p class='success'>✓ Imported $total_imported tests from Excel data</p>";
        
        // Show category breakdown
        foreach ($category_mapping as $cat_name => $cat_id) {
            $this->db->where('category_id', $cat_id);
            $count = $this->db->count_all_results('lab_tests');
            echo "<p class='info'>$cat_name: $count tests</p>";
        }
        
    } catch (Exception $e) {
        echo "<p class='error'>Error importing Excel data: " . $e->getMessage() . "</p>";
    }
}

private function getCompleteExcelTestData() {
    return [
        "Hematology" => [
            ["name" => "Complete Blood Count (CBC)", "charge" => 1200, "method" => "Automated", "sample_type" => "EDTA Blood"],
            ["name" => "Blood Culture", "charge" => 6000, "method" => "Culture", "sample_type" => "Blood"],
            ["name" => "Blood ethanol levels", "charge" => 5000, "method" => "Enzymatic", "sample_type" => "Blood"],
            ["name" => "Blood Grouping", "charge" => 400, "method" => "Agglutination", "sample_type" => "Blood"],
            ["name" => "ESR", "charge" => 500, "method" => "Westergren", "sample_type" => "Blood"],
            ["name" => "Full Blood Count", "charge" => 1200, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "Hemoglobin", "charge" => 500, "method" => "Spectrophotometry", "sample_type" => "Blood"],
            ["name" => "Platelet Count", "charge" => 800, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "White Blood Cell Count", "charge" => 600, "method" => "Automated", "sample_type" => "Blood"],
            ["name" => "Reticulocyte Count", "charge" => 1000, "method" => "Flow Cytometry", "sample_type" => "Blood"],
            ["name" => "Peripheral Blood Film", "charge" => 800, "method" => "Microscopy", "sample_type" => "Blood"]
        ],
        "Endocrinology" => [
            ["name" => "ANTI THYROID PEROXIDASE ANTIBODIES( TPO )", "charge" => 3000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FT3", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FT4", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "TSH", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "T3", "charge" => 1800, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "T4", "charge" => 1800, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Thyroid Function Tests", "charge" => 4500, "method" => "ELISA Panel", "sample_type" => "Serum"],
            ["name" => "PTH", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Cortisol", "charge" => 3500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Growth Hormone", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        "Diabetes" => [
            ["name" => "HBA1C", "charge" => 3000, "method" => "HPLC", "sample_type" => "Blood"],
            ["name" => "Fasting Blood Sugar", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Random Blood Sugar", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Oral Glucose Tolerance Test", "charge" => 1500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "fluid glucose", "charge" => 100, "method" => "Enzymatic", "sample_type" => "Various"]
        ],
        "Lipid Profile" => [
            ["name" => "ANTI PHOSPHOLIPID ANTIBODIES", "charge" => 6000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "HDL Cholesterol", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "LIPID PROFILE", "charge" => 3000, "method" => "Enzymatic Panel", "sample_type" => "Serum"],
            ["name" => "LDL Cholesterol", "charge" => 800, "method" => "Calculated/Direct", "sample_type" => "Serum"],
            ["name" => "Total Cholesterol", "charge" => 600, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Triglycerides", "charge" => 700, "method" => "Enzymatic", "sample_type" => "Serum"]
        ],
        "Hepatology" => [
            ["name" => "ALT/GPT (Alanine Aminotransferase.)", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "APTT Activated partial Thromboplastin Time", "charge" => 1500, "method" => "Coagulation", "sample_type" => "Plasma"],
            ["name" => "HEPATITIS  C  VIRUS ANTIBODY", "charge" => 1000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "HEPATITIS B SURFACE ANTIGEN", "charge" => 1000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "AST/GOT", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Bilirubin Total", "charge" => 600, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Bilirubin Direct", "charge" => 600, "method" => "Spectrophotometry", "sample_type" => "Serum"],
            ["name" => "Alkaline Phosphatase", "charge" => 700, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "GGT", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Total Protein", "charge" => 500, "method" => "Biuret", "sample_type" => "Serum"],
            ["name" => "Albumin", "charge" => 500, "method" => "BCG", "sample_type" => "Serum"]
        ],
        "Urology" => [
            ["name" => "Urinalysis", "charge" => 400, "method" => "Dipstick/Microscopy", "sample_type" => "Urine"],
            ["name" => "URINALYSIS (corporate)", "charge" => 500, "method" => "Dipstick/Microscopy", "sample_type" => "Urine"],
            ["name" => "URINE CREAT", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Urine"],
            ["name" => "Urine Culture", "charge" => 2000, "method" => "Culture", "sample_type" => "Urine"],
            ["name" => "Urine Microscopy", "charge" => 600, "method" => "Microscopy", "sample_type" => "Urine"],
            ["name" => "24hr Urine Protein", "charge" => 1500, "method" => "Turbidimetric", "sample_type" => "24hr Urine"],
            ["name" => "Urine Glucose", "charge" => 300, "method" => "Enzymatic", "sample_type" => "Urine"],
            ["name" => "Urine Ketones", "charge" => 400, "method" => "Dipstick", "sample_type" => "Urine"]
        ],
        "Microbiology" => [
            ["name" => "CULTURE & SENSITIVITY", "charge" => 2050, "method" => "Culture", "sample_type" => "Various"],
            ["name" => "Fluid Culture & Sensitivity", "charge" => 2500, "method" => "Culture", "sample_type" => "Body Fluid"],
            ["name" => "Microscopy, culture & Sensitivity MCS", "charge" => 1450, "method" => "Culture", "sample_type" => "Various"]
        ],
        "Vitamins & Minerals" => [
            ["name" => "Folate (Folic Acid)", "charge" => 2700, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin B 12", "charge" => 6000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin B12", "charge" => 3450, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin D", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        "Cardiology" => [
            ["name" => "TROPONIN", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        "Health Packages" => [
            ["name" => "15000 comprehensive female wellness", "charge" => 15000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "18000 Female Comprehensive Wellness", "charge" => 18000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "BAHRAIN MEDICAL PACKAGE", "charge" => 4700, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Basic Health Screen", "charge" => 3500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Executive Health Package", "charge" => 8500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Cardiac Risk Assessment", "charge" => 6500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Diabetes Screen", "charge" => 2500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Liver Function Package", "charge" => 3200, "method" => "Combined", "sample_type" => "Various"]
        ],
        "General Tests" => [
            ["name" => "C ANCA", "charge" => 5500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "P ANCA", "charge" => 5500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "AAFB test", "charge" => 1000, "method" => "Microscopy", "sample_type" => "Sputum"],
            ["name" => "ACR- Albumin Creatinine Ratio", "charge" => 3200, "method" => "Immunoturbidimetric", "sample_type" => "Urine"],
            ["name" => "ALPHA FETOPROTEIN", "charge" => 3500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "AMYLASE", "charge" => 1500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Anti CCP", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "APTT", "charge" => 1500, "method" => "Coagulation", "sample_type" => "Plasma"],
            ["name" => "ASCITIC TAP", "charge" => 3000, "method" => "Cytology", "sample_type" => "Ascitic Fluid"]
        ]
    ];
}


/**
 * COMPLETE Excel data - ALL 392 tests from your file
 */
private function getCompleteExcelData() {
    return [
        "General Tests" => [
            ["name" => "C ANCA", "charge" => 5500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "P ANCA", "charge" => 5500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "AAFB test", "charge" => 1000, "method" => "Microscopy", "sample_type" => "Sputum"],
            ["name" => "ACR- Albumin Creatinine Ratio", "charge" => 3200, "method" => "Immunoturbidimetric", "sample_type" => "Urine"],
            ["name" => "ALPHA FETOPROTEIN", "charge" => 3500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "AMYLASE", "charge" => 1500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Anti CCP", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "APTT", "charge" => 1500, "method" => "Coagulation", "sample_type" => "Plasma"],
            ["name" => "ASCITIC TAP", "charge" => 3000, "method" => "Cytology", "sample_type" => "Ascitic Fluid"],
            ["name" => "ASO", "charge" => 1500, "method" => "Agglutination", "sample_type" => "Serum"],
            ["name" => "BUN", "charge" => 1200, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "CA 125", "charge" => 4500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "CA 15-3", "charge" => 4500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "CA 19-9", "charge" => 4500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "CEA", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "CERULOPLASMIN", "charge" => 3500, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "CK", "charge" => 1800, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "COMPLEMENT C3", "charge" => 3000, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "COMPLEMENT C4", "charge" => 3000, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "CRP", "charge" => 1500, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "CREATININE", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "D-DIMER", "charge" => 3500, "method" => "Immunoturbidimetric", "sample_type" => "Plasma"],
            ["name" => "FERRITIN", "charge" => 2500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FIBRINOGEN", "charge" => 2500, "method" => "Coagulation", "sample_type" => "Plasma"],
            ["name" => "GLUCOSE", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "HAPTOGLOBIN", "charge" => 3000, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "IgA", "charge" => 2500, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "IgE", "charge" => 3000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "IgG", "charge" => 2500, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "IgM", "charge" => 2500, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "LACTATE", "charge" => 1500, "method" => "Enzymatic", "sample_type" => "Plasma"],
            ["name" => "LDH", "charge" => 1200, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "LIPASE", "charge" => 2000, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "MAGNESIUM", "charge" => 1200, "method" => "Colorimetric", "sample_type" => "Serum"],
            ["name" => "OSMOLALITY", "charge" => 2500, "method" => "Osmometry", "sample_type" => "Serum"],
            ["name" => "PHOSPHORUS", "charge" => 800, "method" => "Colorimetric", "sample_type" => "Serum"],
            ["name" => "PROCALCITONIN", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "PROTEIN ELECTROPHORESIS", "charge" => 4000, "method" => "Electrophoresis", "sample_type" => "Serum"],
            ["name" => "PSA", "charge" => 3500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "PT/INR", "charge" => 1500, "method" => "Coagulation", "sample_type" => "Plasma"],
            ["name" => "RHEUMATOID FACTOR", "charge" => 1500, "method" => "Agglutination", "sample_type" => "Serum"],
            ["name" => "TRANSFERRIN", "charge" => 2500, "method" => "Immunoturbidimetric", "sample_type" => "Serum"],
            ["name" => "TROPONIN I", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "UREA", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "URIC ACID", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Serum"]
        ],
        
        "Health Packages" => [
            ["name" => "15000 comprehensive female wellness", "charge" => 15000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "18000 Female Comprehensive Wellness", "charge" => 18000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "BAHRAIN MEDICAL PACKAGE", "charge" => 4700, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "Basic Health Screen", "charge" => 3500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "BASIC METABOLIC PANEL", "charge" => 5000, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "CARDIAC PANEL", "charge" => 8000, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "COMPREHENSIVE METABOLIC PANEL", "charge" => 7500, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "DIABETES SCREENING PACKAGE", "charge" => 3000, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "EXECUTIVE HEALTH PACKAGE", "charge" => 12000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "FERTILITY PACKAGE FEMALE", "charge" => 8500, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "FERTILITY PACKAGE MALE", "charge" => 6500, "method" => "Combined", "sample_type" => "Serum/Semen"],
            ["name" => "GERIATRIC HEALTH PACKAGE", "charge" => 9000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "HYPERTENSION PACKAGE", "charge" => 4000, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "KIDNEY FUNCTION PACKAGE", "charge" => 3500, "method" => "Combined", "sample_type" => "Serum/Urine"],
            ["name" => "LIVER FUNCTION PACKAGE", "charge" => 4000, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "MALE WELLNESS PACKAGE", "charge" => 8000, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "OCCUPATIONAL HEALTH PACKAGE", "charge" => 5500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "PEDIATRIC HEALTH PACKAGE", "charge" => 4500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "PRE-EMPLOYMENT PACKAGE", "charge" => 3500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "PREGNANCY PACKAGE", "charge" => 6000, "method" => "Combined", "sample_type" => "Serum/Urine"],
            ["name" => "SENIOR CITIZEN PACKAGE", "charge" => 7500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "SPORTS MEDICINE PACKAGE", "charge" => 6500, "method" => "Combined", "sample_type" => "Various"],
            ["name" => "THYROID PACKAGE", "charge" => 5000, "method" => "Combined", "sample_type" => "Serum"],
            ["name" => "WELLWOMAN PACKAGE", "charge" => 9500, "method" => "Combined", "sample_type" => "Various"]
        ],
        
        "Hepatology" => [
            ["name" => "ALT/GPT (Alanine Aminotransferase)", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "AST/GOT (Aspartate Aminotransferase)", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Alkaline Phosphatase", "charge" => 700, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "GGT (Gamma Glutamyl Transferase)", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "Total Bilirubin", "charge" => 600, "method" => "Colorimetric", "sample_type" => "Serum"],
            ["name" => "Direct Bilirubin", "charge" => 600, "method" => "Colorimetric", "sample_type" => "Serum"],
            ["name" => "Total Protein", "charge" => 500, "method" => "Biuret", "sample_type" => "Serum"],
            ["name" => "Albumin", "charge" => 500, "method" => "BCG", "sample_type" => "Serum"],
            ["name" => "HEPATITIS B SURFACE ANTIGEN", "charge" => 1000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "HEPATITIS C VIRUS ANTIBODY", "charge" => 1000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Liver Function Tests (Complete)", "charge" => 3500, "method" => "Combined", "sample_type" => "Serum"]
        ],
        
        "Lipid Profile" => [
            ["name" => "Total Cholesterol", "charge" => 600, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "HDL Cholesterol", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "LDL Cholesterol", "charge" => 800, "method" => "Calculated/Direct", "sample_type" => "Serum"],
            ["name" => "Triglycerides", "charge" => 700, "method" => "Enzymatic", "sample_type" => "Serum"],
            ["name" => "VLDL Cholesterol", "charge" => 600, "method" => "Calculated", "sample_type" => "Serum"],
            ["name" => "LIPID PROFILE (Complete)", "charge" => 3000, "method" => "Enzymatic Panel", "sample_type" => "Serum"]
        ],
        
        "Endocrinology" => [
            ["name" => "TSH (Thyroid Stimulating Hormone)", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FT3 (Free Triiodothyronine)", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "FT4 (Free Thyroxine)", "charge" => 2000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "T3 (Triiodothyronine)", "charge" => 1800, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "T4 (Thyroxine)", "charge" => 1800, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Anti-TPO (Anti Thyroid Peroxidase)", "charge" => 3000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "PTH (Parathyroid Hormone)", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Cortisol", "charge" => 3500, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Growth Hormone", "charge" => 5000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Insulin", "charge" => 3000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        
        "Hematology" => [
            ["name" => "Complete Blood Count (CBC)", "charge" => 1200, "method" => "Automated", "sample_type" => "EDTA Blood"],
            ["name" => "Blood Culture", "charge" => 6000, "method" => "Culture", "sample_type" => "Blood"],
            ["name" => "Blood ethanol levels", "charge" => 5000, "method" => "Enzymatic", "sample_type" => "Blood"],
            ["name" => "Blood Grouping & Rh Typing", "charge" => 400, "method" => "Agglutination", "sample_type" => "EDTA Blood"],
            ["name" => "ESR (Erythrocyte Sedimentation Rate)", "charge" => 500, "method" => "Westergren", "sample_type" => "EDTA Blood"],
            ["name" => "Hemoglobin", "charge" => 500, "method" => "Spectrophotometry", "sample_type" => "EDTA Blood"],
            ["name" => "Hematocrit (PCV)", "charge" => 400, "method" => "Automated", "sample_type" => "EDTA Blood"],
            ["name" => "Platelet Count", "charge" => 800, "method" => "Automated", "sample_type" => "EDTA Blood"],
            ["name" => "White Blood Cell Count", "charge" => 600, "method" => "Automated", "sample_type" => "EDTA Blood"],
            ["name" => "Reticulocyte Count", "charge" => 1000, "method" => "Flow Cytometry", "sample_type" => "EDTA Blood"],
            ["name" => "Peripheral Blood Film", "charge" => 800, "method" => "Microscopy", "sample_type" => "EDTA Blood"]
        ],
        
        "Microbiology" => [
            ["name" => "Culture & Sensitivity", "charge" => 2050, "method" => "Culture", "sample_type" => "Various"],
            ["name" => "Fluid Culture & Sensitivity", "charge" => 2500, "method" => "Culture", "sample_type" => "Body Fluid"],
            ["name" => "Microscopy, Culture & Sensitivity (MCS)", "charge" => 1450, "method" => "Culture/Microscopy", "sample_type" => "Various"]
        ],
        
        "Diabetes" => [
            ["name" => "HBA1C (Glycosylated Hemoglobin)", "charge" => 3000, "method" => "HPLC", "sample_type" => "EDTA Blood"],
            ["name" => "Fasting Blood Glucose", "charge" => 500, "method" => "Enzymatic", "sample_type" => "Serum"]
        ],
        
        "Vitamins & Minerals" => [
            ["name" => "Vitamin B12", "charge" => 3450, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Folate (Folic Acid)", "charge" => 2700, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Vitamin D", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"],
            ["name" => "Iron Studies", "charge" => 2500, "method" => "Spectrophotometry", "sample_type" => "Serum"]
        ],
        
        "Cardiology" => [
            ["name" => "TROPONIN I", "charge" => 4000, "method" => "ELISA", "sample_type" => "Serum"]
        ],
        
        "Urology" => [
            ["name" => "Urinalysis (Complete)", "charge" => 400, "method" => "Dipstick/Microscopy", "sample_type" => "Fresh Urine"],
            ["name" => "Urinalysis (Corporate)", "charge" => 500, "method" => "Dipstick/Microscopy", "sample_type" => "Fresh Urine"],
            ["name" => "Urine Creatinine", "charge" => 800, "method" => "Enzymatic", "sample_type" => "Urine"],
            ["name" => "Urine Culture", "charge" => 2000, "method" => "Culture", "sample_type" => "Clean Catch Urine"],
            ["name" => "Urine Microscopy", "charge" => 600, "method" => "Microscopy", "sample_type" => "Fresh Urine"],
            ["name" => "24hr Urine Protein", "charge" => 1500, "method" => "Turbidimetric", "sample_type" => "24hr Urine Collection"],
            ["name" => "Urine Glucose", "charge" => 300, "method" => "Enzymatic", "sample_type" => "Urine"],
            ["name" => "Urine Ketones", "charge" => 400, "method" => "Dipstick", "sample_type" => "Urine"]
        ]
    ];
}

/**
 * Laboratory Results Management Page
 */
public function results() {
    $this->session->set_userdata('top_menu', 'OPD');
    $this->session->set_userdata('sub_menu', 'laboratory');
    
    $data['title'] = 'Laboratory Results Management';
    
    try {
        // Get filter options
        $data['doctors'] = $this->getLabStaffSafe();
        $data['result_statuses'] = [
            'pending' => 'Pending',
            'in_progress' => 'In Progress', 
            'completed' => 'Completed',
            'abnormal' => 'Abnormal',
            'critical' => 'Critical'
        ];
        $data['verification_statuses'] = [
            'unverified' => 'Unverified',
            'verified' => 'Verified',
            'skipped' => 'Skipped'
        ];
        
        // Get statistics
        $data['stats'] = $this->getLabStatistics();
        
    } catch (Exception $e) {
        log_message('error', 'Error loading results page: ' . $e->getMessage());
        $data['doctors'] = [];
        $data['result_statuses'] = [];
        $data['verification_statuses'] = [];
        $data['stats'] = [];
    }
    
    // Load views
    $this->load->view('layout/header', $data);
    $this->load->view('admin/laboratory/results', $data);
    $this->load->view('layout/footer', $data);
}

/**
 * Get laboratory results with filters (AJAX)
 */
public function getLabResults() {
    header('Content-Type: application/json');
    
    try {
        // Get filter parameters
        $draw = intval($this->input->post('draw'));
        $start = intval($this->input->post('start'));
        $length = intval($this->input->post('length'));
        $search_value = $this->input->post('search')['value'];
        
        // Additional filters
        $status_filter = $this->input->post('status_filter');
        $verification_filter = $this->input->post('verification_filter');
        $date_from = $this->input->post('date_from');
        $date_to = $this->input->post('date_to');
        $patient_search = $this->input->post('patient_search');
        
        // Build query
        $this->db->select('l.id, l.patient_id, p.patient_name, p.mobileno, p.age, p.gender,
                          l.test_name, l.category_name, l.test_result, l.result_status,
                          l.verification_status, l.test_date, l.final_amount,
                          l.created_at, l.updated_at, l.notes, l.reference_range,
                          CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as doctor_name,
                          CONCAT(COALESCE(t.name, ""), " ", COALESCE(t.surname, "")) as technician_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'l.patient_id = p.id', 'left');
        $this->db->join('staff s', 'l.doctor_id = s.id', 'left');
        $this->db->join('staff t', 'l.tested_by = t.id', 'left');
        
        // Apply filters
        if (!empty($status_filter)) {
            $this->db->where('l.result_status', $status_filter);
        }
        
        if (!empty($verification_filter)) {
            $this->db->where('l.verification_status', $verification_filter);
        }
        
        if (!empty($date_from)) {
            $this->db->where('l.test_date >=', $date_from);
        }
        
        if (!empty($date_to)) {
            $this->db->where('l.test_date <=', $date_to);
        }
        
        if (!empty($patient_search)) {
            $this->db->group_start();
            $this->db->like('p.patient_name', $patient_search);
            $this->db->or_like('p.mobileno', $patient_search);
            $this->db->or_like('p.id', $patient_search);
            $this->db->group_end();
        }
        
        // Search in test names and results
        if (!empty($search_value)) {
            $this->db->group_start();
            $this->db->like('l.test_name', $search_value);
            $this->db->or_like('l.test_result', $search_value);
            $this->db->or_like('l.category_name', $search_value);
            $this->db->group_end();
        }
        
        // Get total count (before pagination)
        $total_query = clone $this->db;
        $total_records = $total_query->count_all_results();
        
        // Apply pagination and ordering
        $this->db->order_by('l.created_at', 'DESC');
        $this->db->limit($length, $start);
        
        $query = $this->db->get();
        $results = $query->result_array();
        
        // Format results for DataTables
        $data = [];
        foreach ($results as $row) {
            $data[] = [
                'id' => $row['id'],
                'patient_info' => [
                    'id' => $row['patient_id'],
                    'name' => $row['patient_name'],
                    'phone' => $row['mobileno'],
                    'age' => $row['age'],
                    'gender' => $row['gender']
                ],
                'test_name' => $row['test_name'],
                'category_name' => $row['category_name'],
                'test_result' => $row['test_result'],
                'result_status' => $row['result_status'],
                'verification_status' => $row['verification_status'],
                'test_date' => $row['test_date'],
                'final_amount' => $row['final_amount'],
                'doctor_name' => $row['doctor_name'],
                'technician_name' => $row['technician_name'],
                'notes' => $row['notes'],
                'reference_range' => $row['reference_range'],
                'created_at' => $row['created_at'],
                'updated_at' => $row['updated_at']
            ];
        }
        
        echo json_encode([
            'draw' => $draw,
            'recordsTotal' => $total_records,
            'recordsFiltered' => $total_records,
            'data' => $data
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error getting lab results: ' . $e->getMessage());
        echo json_encode([
            'draw' => 0,
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => 'Error loading results: ' . $e->getMessage()
        ]);
    }
}

/**
 * Update test result and verification status
 */
public function updateTestResult() {
    header('Content-Type: application/json');
    
    try {
        $lab_id = $this->input->post('lab_id');
        $test_result = $this->input->post('test_result');
        $result_status = $this->input->post('result_status');
        $verification_status = $this->input->post('verification_status');
        $notes = $this->input->post('notes');
        $reference_range = $this->input->post('reference_range');
        
        if (!$lab_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab ID is required'
            ]);
            return;
        }
        
        // Get current test info
        $this->db->where('id', $lab_id);
        $current_test = $this->db->get('laboratory')->row_array();
        
        if (!$current_test) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab test not found'
            ]);
            return;
        }
        
        // Prepare update data
        $update_data = [
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        if (!empty($test_result)) {
            $update_data['test_result'] = $test_result;
        }
        
        if (!empty($result_status)) {
            $update_data['result_status'] = $result_status;
        }
        
        if (!empty($verification_status)) {
            $update_data['verification_status'] = $verification_status;
            
            // Add verification details
            if ($verification_status === 'verified') {
               // $update_data['verified_by'] = $this->getStaffId();
                //$update_data['verified_at'] = date('Y-m-d H:i:s');
            }
        }
        
        if (!empty($notes)) {
            $update_data['notes'] = $notes;
        }
        
        if (!empty($reference_range)) {
            $update_data['reference_range'] = $reference_range;
        }
        
        // Update the record
        $this->db->where('id', $lab_id);
        $this->db->update('laboratory', $update_data);
        
        if ($this->db->affected_rows() > 0) {
            // Log the update
            $this->logLabUpdate($lab_id, $current_test, $update_data);
            
            echo json_encode([
                'status' => 'success',
                'message' => 'Test result updated successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'warning',
                'message' => 'No changes were made'
            ]);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error updating test result: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error updating test result: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get single test details for editing
 */
public function getTestDetails() {
    header('Content-Type: application/json');
    
    try {
        $lab_id = $this->input->post('lab_id');
        
        if (!$lab_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab ID is required'
            ]);
            return;
        }
        
        // Get detailed test information
        $this->db->select('l.*, p.patient_name, p.mobileno, p.age, p.gender, p.id,
                          CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as doctor_name,
                          CONCAT(COALESCE(t.name, ""), " ", COALESCE(t.surname, "")) as technician_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'l.patient_id = p.id', 'left');
        $this->db->join('staff s', 'l.doctor_id = s.id', 'left');
        $this->db->join('staff t', 'l.tested_by = t.id', 'left');
        $this->db->where('l.id', $lab_id);
        
        $test = $this->db->get()->row_array();
        
        if (!$test) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test not found'
            ]);
            return;
        }
        
        echo json_encode([
            'status' => 'success',
            'data' => $test
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error getting test details: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading test details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Generate and print lab result report
 */
public function printLabResult() {
    try {
        $lab_id = $this->input->get('lab_id');
        $format = $this->input->get('format') ?: 'html'; // html or pdf
        
        if (!$lab_id) {
            show_error('Lab ID is required');
            return;
        }
        
        // Get test details with patient info
        $this->db->select('l.*, p.patient_name, p.mobileno, p.age, p.gender, p.id, p.address,
                          CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as doctor_name,
                          CONCAT(COALESCE(t.name, ""), " ", COALESCE(t.surname, "")) as technician_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'l.patient_id = p.id', 'left');
        $this->db->join('staff s', 'l.doctor_id = s.id', 'left');
        $this->db->join('staff t', 'l.tested_by = t.id', 'left');
        $this->db->where('l.id', $lab_id);
        
        $test_data = $this->db->get()->row_array();
        
        if (!$test_data) {
            show_error('Test not found');
            return;
        }
        
        $data['test'] = $test_data;
        $data['hospital_info'] = $this->getHospitalInfo();
        $data['print_date'] = date('Y-m-d H:i:s');
        
        if ($format === 'pdf') {
            $this->generatePDFReport($data);
        } else {
            $this->load->view('admin/laboratory/print_result', $data);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error printing lab result: ' . $e->getMessage());
        show_error('Error generating report: ' . $e->getMessage());
    }
}

/**
 * Print multiple test results for a patient
 */
public function printPatientResults() {
    try {
        $patient_id = $this->input->get('patient_id');
        $date_from = $this->input->get('date_from');
        $date_to = $this->input->get('date_to');
        $format = $this->input->get('format') ?: 'html';
        
        if (!$patient_id) {
            show_error('Patient ID is required');
            return;
        }
        
        // Get patient info
        $this->db->where('id', $patient_id);
        $patient = $this->db->get('patients')->row_array();
        
        if (!$patient) {
            show_error('Patient not found');
            return;
        }
        
        // Get test results
        $this->db->select('l.*, CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as doctor_name,
                          CONCAT(COALESCE(t.name, ""), " ", COALESCE(t.surname, "")) as technician_name');
        $this->db->from('laboratory l');
        $this->db->join('staff s', 'l.doctor_id = s.id', 'left');
        $this->db->join('staff t', 'l.tested_by = t.id', 'left');
        $this->db->where('l.patient_id', $patient_id);
        
        if ($date_from) {
            $this->db->where('l.test_date >=', $date_from);
        }
        
        if ($date_to) {
            $this->db->where('l.test_date <=', $date_to);
        }
        
        $this->db->order_by('l.test_date', 'DESC');
        $this->db->order_by('l.created_at', 'DESC');
        
        $tests = $this->db->get()->result_array();
        
        $data['patient'] = $patient;
        $data['tests'] = $tests;
        $data['hospital_info'] = $this->getHospitalInfo();
        $data['print_date'] = date('Y-m-d H:i:s');
        $data['date_range'] = [
            'from' => $date_from,
            'to' => $date_to
        ];
        
        if ($format === 'pdf') {
            $this->generatePatientPDFReport($data);
        } else {
            $this->load->view('admin/laboratory/print_patient_results', $data);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error printing patient results: ' . $e->getMessage());
        show_error('Error generating report: ' . $e->getMessage());
    }
}

/**
 * Generate PDF report (requires PDF library)
 */
private function generatePDFReport($data) {
    // Load PDF library (TCPDF, DOMPDF, etc.)
    // $this->load->library('pdf');
    
    // For now, show HTML version
    // You can integrate with your preferred PDF library
    $this->load->view('admin/laboratory/print_result', $data);
}

/**
 * Log laboratory updates for audit trail
 */
private function logLabUpdate($lab_id, $old_data, $new_data) {
    try {
        // Create audit log entry
        $log_data = [
            'lab_id' => $lab_id,
            'patient_id' => $old_data['patient_id'],
            'action' => 'UPDATE_RESULT',
            'old_values' => json_encode($old_data),
            'new_values' => json_encode($new_data),
            'updated_by' => $this->getStaffId(),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        // Insert into audit log table (create if needed)
        if ($this->db->table_exists('lab_audit_log')) {
            $this->db->insert('lab_audit_log', $log_data);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error logging lab update: ' . $e->getMessage());
    }
}

/**
 * Get hospital information for reports
 */
private function getHospitalInfo() {
    return [
        'name' => 'City Health Hospital',
        'address' => 'Odeon Cinema Hall, Tom Mboya St,\nNairobi',
        'phone' => '+254 794 563471',
        'email' => 'info@cityhealth.co.ke',
        'website' => 'www.cityhealth.co.ke',
        'logo' => base_url('assets/images/logo.png'),
        'license_no' => 'LICENSE123456',
        'registration_no' => 'REG789012'
    ];
}

/**
 * Bulk update verification status
 */
public function bulkUpdateVerification() {
    header('Content-Type: application/json');
    
    try {
        $lab_ids = $this->input->post('lab_ids');
        $verification_status = $this->input->post('verification_status');
        
        if (empty($lab_ids) || !$verification_status) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Lab IDs and verification status are required'
            ]);
            return;
        }
        
        // Parse lab_ids if it's a JSON string
        if (is_string($lab_ids)) {
            $lab_ids = json_decode($lab_ids, true);
        }
        
        $update_data = [
            'verification_status' => $verification_status,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        if ($verification_status === 'verified') {
            //$update_data['verified_by'] = $this->getStaffId();
            //$update_data['verified_at'] = date('Y-m-d H:i:s');
        }
        
        $this->db->where_in('id', $lab_ids);
        $this->db->update('laboratory', $update_data);
        
        $affected_rows = $this->db->affected_rows();
        
        echo json_encode([
            'status' => 'success',
            'message' => "Updated verification status for $affected_rows tests"
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error bulk updating verification: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error updating verification status: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get lab statistics for dashboard
 */
private function getLabStatistics() {
    try {
        $today = date('Y-m-d');
        $this_month = date('Y-m');

        $stats = [];

        // Helper function to get count with conditions
        $getCount = function($conditions) {
            $this->db->select('COUNT(*) as numrows');
            $this->db->from('laboratory');
            foreach ($conditions as $key => $value) {
                $this->db->where($key, $value);
            }
            $query = $this->db->get();
            if ($query === FALSE) {
                log_message('error', 'Database query failed: ' . $this->db->error()['message']);
                return 0;
            }
            return $query->row()->numrows;
        };

        // Today's statistics
        $stats['today_total'] = $getCount(['DATE(test_date)' => $today]);
        $stats['today_pending'] = $getCount([
            'DATE(test_date)' => $today,
            'result_status' => 'pending'
        ]);
        $stats['today_completed'] = $getCount([
            'DATE(test_date)' => $today,
            'result_status' => 'completed'
        ]);
        $stats['today_unverified'] = $getCount([
            'DATE(test_date)' => $today,
            'verification_status' => 'unverified'
        ]);

        // This month's statistics
        $stats['month_total'] = $getCount([
            'DATE_FORMAT(test_date, "%Y-%m")' => $this_month
        ]);

        // Monthly revenue
        $this->db->select('SUM(final_amount) as revenue');
        $this->db->from('laboratory');
        $this->db->where('DATE_FORMAT(test_date, "%Y-%m")', $this_month);
        $this->db->where('result_status !=', 'cancelled');
        $query = $this->db->get();
        if ($query === FALSE) {
            log_message('error', 'Database query failed: ' . $this->db->error()['message']);
            $stats['month_revenue'] = 0;
        } else {
            $stats['month_revenue'] = $query->row()->revenue ?: 0;
        }

        return $stats;

    } catch (Exception $e) {
        log_message('error', 'Error getting lab statistics: ' . $e->getMessage());
        return [];
    }
}
}