<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Lab_setup extends Admin_Controller {

    public function __construct() {
        parent::__construct();
        $this->load->model('lab_setup_model');
        $this->load->model('laboratory_model');
        $this->load->model('staff_model');
        $this->load->model('setting_model');
        $this->load->library('form_validation');
        $this->load->library('datatables');
        $this->load->helper('customfield_helper');
        $this->load->helper('custom');
        
        // Get system currency
        $this->setSystemCurrency();
    }

    private function setSystemCurrency() {
        try {
            $settings = $this->setting_model->get();
            $this->system_currency = isset($settings[0]['currency_symbol']) ? $settings[0]['currency_symbol'] : '$';
        } catch (Exception $e) {
            $this->system_currency = '$';
        }
    }

    // =============================================
    // DASHBOARD
    // =============================================

    public function index() {
        redirect('admin/lab_setup/dashboard');
    }

    public function dashboard() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/dashboard');

        $data['title'] = 'Laboratory Setup Dashboard';
        $data['currency_symbol'] = $this->system_currency;
        
        try {
            $data['stats'] = $this->lab_setup_model->getSetupStats();
        } catch (Exception $e) {
            $data['stats'] = array(
                'total_categories' => 0,
                'total_tests' => 0,
                'total_units' => 0,
                'total_reference_ranges' => 0,
                'total_lab_results' => 0,
                'expensive_tests' => array(),
                'tests_by_category' => array()
            );
        }

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/dashboard', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // CATEGORIES MANAGEMENT
    // =============================================

    public function categories() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/categories');

        $data['title'] = 'Lab Test Categories';
        $data['categories'] = $this->lab_setup_model->getTestCategories();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/categories', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add_category() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('category_name', 'Category Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('description', 'Description', 'trim|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'category_name' => form_error('category_name'),
                'description' => form_error('description'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'category_name' => $this->input->post('category_name'),
                'description' => $this->input->post('description'),
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            );
            
            $result = $this->lab_setup_model->addTestCategory($data);
            if ($result) {
                $array = array('status' => 'success', 'error' => '', 'message' => 'Category added successfully.');
            } else {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to add category.');
            }
        }
        echo json_encode($array);
    }

    public function edit_category($id) {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
            access_denied();
        }
        
        $data['category'] = $this->lab_setup_model->getTestCategory($id);
        $this->load->view('admin/lab_setup/edit_category_modal', $data);
    }

    public function update_category($id) {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
            access_denied();
        }

        $this->form_validation->set_rules('category_name', 'Category Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('description', 'Description', 'trim|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'category_name' => form_error('category_name'),
                'description' => form_error('description'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $existing = $this->lab_setup_model->checkCategoryExists($this->input->post('category_name'), $id);
            
            if ($existing) {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Category name already exists.');
            } else {
                $data = array(
                    'category_name' => $this->input->post('category_name'),
                    'description' => $this->input->post('description'),
                    'updated_at' => date('Y-m-d H:i:s')
                );
                
                $result = $this->lab_setup_model->updateTestCategory($id, $data);
                if ($result) {
                    $array = array('status' => 'success', 'error' => '', 'message' => 'Category updated successfully.');
                } else {
                    $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to update category.');
                }
            }
        }
        echo json_encode($array);
    }

    public function delete_category($id) {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_delete')) {
            access_denied();
        }

        $tests_count = $this->lab_setup_model->getTestsByCategoryCount($id);
        
        if ($tests_count > 0) {
            $array = array('status' => 'fail', 'message' => 'Cannot delete category. It is being used by ' . $tests_count . ' test(s).');
        } else {
            $result = $this->lab_setup_model->deleteTestCategory($id);
            if ($result) {
                $array = array('status' => 'success', 'message' => 'Category deleted successfully.');
            } else {
                $array = array('status' => 'fail', 'message' => 'Failed to delete category.');
            }
        }
        echo json_encode($array);
    }

    // =============================================
    // TESTS MANAGEMENT
    // =============================================

    public function tests() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/tests');

        $data['title'] = 'Lab Tests Setup';
        $data['categories'] = $this->lab_setup_model->getTestCategories();
        $data['units'] = $this->lab_setup_model->getUnits();
        $data['currency_symbol'] = $this->system_currency;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/tests', $data);
        $this->load->view('layout/footer', $data);
    }

   /* public function get_tests_datatable() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            echo json_encode(array('error' => 'Access denied'));
            return;
        }

        $requestData = $_REQUEST;
        
        $start = isset($requestData['start']) ? intval($requestData['start']) : 0;
        $length = isset($requestData['length']) ? intval($requestData['length']) : 10;
        $search = isset($requestData['search']['value']) ? $requestData['search']['value'] : '';
        
        try {
            $tests = $this->lab_setup_model->getTestsForDataTable($start, $length, $search);
            $totalTests = $this->lab_setup_model->getTestsCount($search);
            
            $data = array();
            foreach ($tests as $test) {
                $row = array();
                $row[] = htmlspecialchars($test['test_name']);
                $row[] = htmlspecialchars($test['short_name']);
                $row[] = htmlspecialchars($test['category_name']);
                $row[] = $test['unit_name'] ? htmlspecialchars($test['unit_name']) : '-';
                $row[] = $this->system_currency . number_format($test['cost'], 2);
                $row[] = $test['report_days'] ? $test['report_days'] . ' days' : '-';
                $row[] = $test['is_active'] ? '<span class="label label-success">Active</span>' : '<span class="label label-danger">Inactive</span>';
                
                $actions = '<div class="btn-group">';
                if ($this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
                    $actions .= '<a href="javascript:void(0)" onclick="editTest(' . $test['id'] . ')" class="btn btn-default btn-xs" title="Edit"><i class="fa fa-pencil"></i></a>';
                }
                if ($this->rbac->hasPrivilege('lab_setup', 'can_delete')) {
                    $actions .= '<a href="javascript:void(0)" onclick="deleteTest(' . $test['id'] . ')" class="btn btn-default btn-xs" title="Delete"><i class="fa fa-trash"></i></a>';
                }
                $actions .= '<a href="javascript:void(0)" onclick="viewTest(' . $test['id'] . ')" class="btn btn-default btn-xs" title="View"><i class="fa fa-eye"></i></a>';
                $actions .= '</div>';
                
                $row[] = $actions;
                $data[] = $row;
            }

            $output = array(
                "draw" => intval($requestData['draw']),
                "recordsTotal" => $totalTests,
                "recordsFiltered" => $totalTests,
                "data" => $data
            );
            
        } catch (Exception $e) {
            $output = array(
                "draw" => intval($requestData['draw']),
                "recordsTotal" => 0,
                "recordsFiltered" => 0,
                "data" => array(),
                "error" => $e->getMessage()
            );
        }
        
        echo json_encode($output);
    }*/

    /*public function add_test() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('test_name', 'Test Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('short_name', 'Short Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('category_id', 'Category', 'trim|required|numeric|xss_clean');
        $this->form_validation->set_rules('cost', 'Cost', 'trim|required|numeric|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'test_name' => form_error('test_name'),
                'short_name' => form_error('short_name'),
                'category_id' => form_error('category_id'),
                'cost' => form_error('cost'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $existing = $this->lab_setup_model->checkTestExists($this->input->post('test_name'));
            
            if ($existing) {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Test name already exists.');
            } else {
                $data = array(
                    'test_name' => $this->input->post('test_name'),
                    'short_name' => $this->input->post('short_name'),
                    'category_id' => $this->input->post('category_id'),
                    'unit_id' => $this->input->post('unit_id') ? $this->input->post('unit_id') : null,
                    'cost' => $this->input->post('cost'),
                    'report_days' => $this->input->post('report_days') ? $this->input->post('report_days') : null,
                    'method' => $this->input->post('method'),
                    'sample_type' => $this->input->post('sample_type'),
                    'test_type' => $this->input->post('test_type'),
                    'description' => $this->input->post('description'),
                    'is_active' => 1,
                    'created_at' => date('Y-m-d H:i:s')
                );
                
                $result = $this->lab_setup_model->addTest($data);
                if ($result) {
                    $array = array('status' => 'success', 'error' => '', 'message' => 'Test added successfully.');
                } else {
                    $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to add test.');
                }
            }
        }
        echo json_encode($array);
    }*/

    // =============================================
    // UNITS MANAGEMENT
    // =============================================

    public function units() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'lab');
        $this->session->set_userdata('sub_menu', 'lab_setup/units');

        $data['title'] = 'Lab Test Units';
        $data['units'] = $this->lab_setup_model->getUnits();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/lab_setup/units', $data);
        $this->load->view('layout/footer', $data);
    }

    public function add_unit() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        $this->form_validation->set_rules('unit_name', 'Unit Name', 'trim|required|xss_clean');
        $this->form_validation->set_rules('unit_symbol', 'Unit Symbol', 'trim|required|xss_clean');

        if ($this->form_validation->run() == FALSE) {
            $msg = array(
                'unit_name' => form_error('unit_name'),
                'unit_symbol' => form_error('unit_symbol'),
            );
            $array = array('status' => 'fail', 'error' => $msg, 'message' => '');
        } else {
            $data = array(
                'unit_name' => $this->input->post('unit_name'),
                'unit_symbol' => $this->input->post('unit_symbol'),
                'description' => $this->input->post('description'),
                'is_active' => 1,
                'created_at' => date('Y-m-d H:i:s')
            );
            
            $result = $this->lab_setup_model->addUnit($data);
            if ($result) {
                $array = array('status' => 'success', 'error' => '', 'message' => 'Unit added successfully.');
            } else {
                $array = array('status' => 'fail', 'error' => '', 'message' => 'Failed to add unit.');
            }
        }
        echo json_encode($array);
    }

    // =============================================
    // SYSTEM INITIALIZATION
    // =============================================

    public function initialize_system() {
        if (!$this->rbac->hasPrivilege('lab_setup', 'can_add')) {
            access_denied();
        }

        try {
            $this->lab_setup_model->createLabTables();
            $this->lab_setup_model->addDefaultCategories();
            $this->lab_setup_model->addDefaultUnits();
            $this->lab_setup_model->addDefaultTests();
            
            $array = array('status' => 'success', 'message' => 'Laboratory system initialized successfully!');
        } catch (Exception $e) {
            $array = array('status' => 'fail', 'message' => 'Failed to initialize system: ' . $e->getMessage());
        }
        
        echo json_encode($array);
    }

    public function test_setup() {
        echo "<h2>Laboratory Setup Test</h2>";
        
        try {
            echo "<p>✓ Database connection: OK</p>";
            
            $tables = array('lab_test_categories', 'lab_tests', 'lab_units', 'lab_reference_ranges', 'laboratory');
            
            foreach ($tables as $table) {
                if ($this->db->table_exists($table)) {
                    echo "<p>✓ Table '{$table}': EXISTS</p>";
                } else {
                    echo "<p>✗ Table '{$table}': MISSING</p>";
                }
            }
            
            $categories = $this->lab_setup_model->getTestCategories();
            echo "<p>✓ Categories found: " . count($categories) . "</p>";
            
            $units = $this->lab_setup_model->getUnits();
            echo "<p>✓ Units found: " . count($units) . "</p>";
            
            echo "<p><strong>System Status: Ready for setup</strong></p>";
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>✗ Error: " . $e->getMessage() . "</p>";
        }
        
        echo "<p><a href='" . base_url() . "admin/lab_setup/initialize_system' class='btn btn-primary'>Initialize System</a></p>";
        echo "<p><a href='" . base_url() . "admin/lab_setup/force_create_tables' class='btn btn-warning'>Force Create Tables</a></p>";
    }

    public function force_create_tables() {
        echo "<h2>Force Creating Laboratory Tables</h2>";
        
        try {
            $this->db->query("
                CREATE TABLE IF NOT EXISTS `lab_test_categories` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `category_name` varchar(100) NOT NULL,
                    `description` text,
                    `is_active` tinyint(1) DEFAULT 1,
                    `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `category_name` (`category_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ");
            echo "<p>✓ lab_test_categories table created</p>";

            $this->db->query("
                CREATE TABLE IF NOT EXISTS `lab_units` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `unit_name` varchar(50) NOT NULL,
                    `unit_symbol` varchar(10) NOT NULL,
                    `description` text,
                    `is_active` tinyint(1) DEFAULT 1,
                    `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `unit_name` (`unit_name`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ");
            echo "<p>✓ lab_units table created</p>";

            $this->db->query("
                CREATE TABLE IF NOT EXISTS `lab_tests` (
                    `id` int(11) NOT NULL AUTO_INCREMENT,
                    `test_name` varchar(150) NOT NULL,
                    `short_name` varchar(50) NOT NULL,
                    `category_id` int(11) NOT NULL,
                    `unit_id` int(11) DEFAULT NULL,
                    `cost` decimal(10,2) NOT NULL DEFAULT 0.00,
                    `report_days` int(3) DEFAULT NULL,
                    `method` varchar(100) DEFAULT NULL,
                    `sample_type` varchar(100) DEFAULT NULL,
                    `test_type` enum('quantitative','qualitative','descriptive') DEFAULT 'qualitative',
                    `description` text,
                    `is_active` tinyint(1) DEFAULT 1,
                    `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                    `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`),
                    UNIQUE KEY `test_name` (`test_name`),
                    KEY `category_id` (`category_id`),
                    KEY `unit_id` (`unit_id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
            ");
            echo "<p>✓ lab_tests table created</p>";

            $this->addDefaultData();
            echo "<p>✓ Default data added</p>";

            echo "<h3 style='color: green;'>✅ All tables created successfully!</h3>";
            echo "<p><a href='" . base_url() . "admin/lab_setup/test_setup' class='btn btn-success'>Test Again</a></p>";
            echo "<p><a href='" . base_url() . "admin/lab_setup/dashboard' class='btn btn-primary'>Go to Dashboard</a></p>";

        } catch (Exception $e) {
            echo "<p style='color: red;'>✗ Error: " . $e->getMessage() . "</p>";
        }
    }

    private function addDefaultData() {
        $categories = array(
            array('category_name' => 'Hematology', 'description' => 'Blood-related tests'),
            array('category_name' => 'Clinical Chemistry', 'description' => 'Chemical analysis tests'),
            array('category_name' => 'Microbiology', 'description' => 'Bacterial and viral tests'),
            array('category_name' => 'Immunology', 'description' => 'Immune system tests'),
            array('category_name' => 'Urine Analysis', 'description' => 'Urine-based tests'),
            array('category_name' => 'Serology', 'description' => 'Blood serum tests')
        );

        foreach ($categories as $category) {
            $this->db->where('category_name', $category['category_name']);
            $existing = $this->db->get('lab_test_categories')->num_rows();
            
            if ($existing == 0) {
                $category['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_test_categories', $category);
            }
        }

        $units = array(
            array('unit_name' => 'Grams per deciliter', 'unit_symbol' => 'g/dL'),
            array('unit_name' => 'Milligrams per deciliter', 'unit_symbol' => 'mg/dL'),
            array('unit_name' => 'Percentage', 'unit_symbol' => '%'),
            array('unit_name' => 'Count', 'unit_symbol' => 'count'),
            array('unit_name' => 'International Units per liter', 'unit_symbol' => 'IU/L')
        );

        foreach ($units as $unit) {
            $this->db->where('unit_name', $unit['unit_name']);
            $existing = $this->db->get('lab_units')->num_rows();
            
            if ($existing == 0) {
                $unit['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_units', $unit);
            }
        }
    }
    
    /**
 * Add new laboratory test
 */
public function add_test() {
    // Set content type for JSON response
    header('Content-Type: application/json');
    
    try {
        // Check if request is POST
        if ($this->input->method() !== 'post') {
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid request method'
            ]);
            return;
        }
        
        // Get POST data
        $test_name = trim($this->input->post('test_name'));
        $short_name = trim($this->input->post('short_name'));
        $category_id = $this->input->post('category_id');
        $unit_id = $this->input->post('unit_id');
        $cost = $this->input->post('cost');
        $report_days = $this->input->post('report_days');
        $test_type = $this->input->post('test_type');
        $method = trim($this->input->post('method'));
        $sample_type = $this->input->post('sample_type');
        $description = trim($this->input->post('description'));
        
        // Validation array for errors
        $errors = array();
        
        // Validate required fields
        if (empty($test_name)) {
            $errors['test_name'] = 'Test name is required';
        }
        
        if (empty($short_name)) {
            $errors['short_name'] = 'Short name is required';
        }
        
        if (empty($category_id)) {
            $errors['category_id'] = 'Category is required';
        }
        
        if (empty($cost) || !is_numeric($cost) || $cost < 0) {
            $errors['cost'] = 'Valid cost is required';
        }
        
        // Check if test name already exists
        if (!empty($test_name)) {
            if ($this->lab_setup_model->checkTestExists($test_name)) {
                $errors['test_name'] = 'Test name already exists';
            }
        }
        
        // If there are validation errors, return them
        if (!empty($errors)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Please correct the errors below',
                'error' => $errors
            ]);
            return;
        }
        
        // Prepare data for insertion
        $data = array(
            'test_name' => $test_name,
            'short_name' => $short_name,
            'category_id' => (int)$category_id,
            'unit_id' => !empty($unit_id) ? (int)$unit_id : null,
            'cost' => (float)$cost,
            'report_days' => !empty($report_days) ? (int)$report_days : null,
            'test_type' => !empty($test_type) ? $test_type : 'qualitative',
            'method' => !empty($method) ? $method : null,
            'sample_type' => !empty($sample_type) ? $sample_type : null,
            'description' => !empty($description) ? $description : null,
            'is_active' => 1,
            'created_at' => date('Y-m-d H:i:s')
        );
        
        // Insert test
        $test_id = $this->lab_setup_model->addTest($data);
        
        if ($test_id) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Laboratory test added successfully',
                'test_id' => $test_id
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to add laboratory test'
            ]);
        }
        
    } catch (Exception $e) {
        // Log the error
        log_message('error', 'Lab Setup Add Test Error: ' . $e->getMessage());
        
        echo json_encode([
            'status' => 'error',
            'message' => 'An error occurred while adding the test: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get tests data for DataTable
 */
public function get_tests_datatable() {
    header('Content-Type: application/json');
    
    try {
        $draw = intval($this->input->post("draw"));
        $start = intval($this->input->post("start"));
        $length = intval($this->input->post("length"));
        $search = $this->input->post("search")['value'];
        
        // Get data
        $tests = $this->lab_setup_model->getTestsForDataTable($start, $length, $search);
        $totalRecords = $this->lab_setup_model->getTestsCount();
        $totalDisplayRecords = $this->lab_setup_model->getTestsCount($search);
        
        $data = array();
        foreach ($tests as $test) {
            // Format status
            $status = $test['is_active'] == 1 ? 
                '<span class="label label-success">Active</span>' : 
                '<span class="label label-danger">Inactive</span>';
            
            // Format cost
            $cost = 'Ksh. ' . number_format($test['cost'], 2);
            
            // Format report days
            $report_days = $test['report_days'] ? $test['report_days'] . ' days' : '-';
            
            // Format unit
            $unit = $test['unit_name'] ? $test['unit_name'] : '-';
            
            // Action buttons
            $actions = '';
            if ($this->rbac->hasPrivilege('lab_setup', 'can_view')) {
                $actions .= '<button type="button" class="btn btn-info btn-xs" onclick="viewTest(' . $test['id'] . ')" title="View">
                                <i class="fa fa-eye"></i>
                            </button> ';
            }
            if ($this->rbac->hasPrivilege('lab_setup', 'can_edit')) {
                $actions .= '<button type="button" class="btn btn-warning btn-xs" onclick="editTest(' . $test['id'] . ')" title="Edit">
                                <i class="fa fa-edit"></i>
                            </button> ';
            }
            if ($this->rbac->hasPrivilege('lab_setup', 'can_delete')) {
                $actions .= '<button type="button" class="btn btn-danger btn-xs" onclick="deleteTest(' . $test['id'] . ')" title="Delete">
                                <i class="fa fa-trash"></i>
                            </button>';
            }
            
            $data[] = array(
                $test['test_name'],
                $test['short_name'],
                $test['category_name'],
                $unit,
                $cost,
                $report_days,
                $status,
                $actions
            );
        }
        
        $json_data = array(
            "draw" => $draw,
            "recordsTotal" => $totalRecords,
            "recordsFiltered" => $totalDisplayRecords,
            "data" => $data
        );
        
        echo json_encode($json_data);
        
    } catch (Exception $e) {
        echo json_encode([
            'error' => 'Error loading data: ' . $e->getMessage()
        ]);
    }
}
}