<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Lab_reports extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load required models and libraries
        $this->load->model('lab_reports_model');
        $this->load->model('patient_model');
        $this->load->model('staff_model');
        $this->load->library('datatables');
        $this->load->library('customlib');
        $this->load->helper('file');
        $this->load->helper('download');
    }

    /**
     * Main lab reports page
     */
    public function index() {
        // Set session data for navigation
        $this->session->set_userdata('top_menu', 'Lab Reports');
        $this->session->set_userdata('sub_menu', 'lab_reports');

        // Prepare data for view
        $data['title'] = 'Lab Reports Management';
        
        // Get current user info
        $data['current_user'] = $this->getStaffInfo();
        
        // Check if models are loaded properly
        if (!$this->load->is_loaded('lab_reports_model')) {
            $this->load->model('lab_reports_model');
        }
        
        // Load view with error handling
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/lab_reports/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'Lab Reports view loading error: ' . $e->getMessage());
            show_error('Failed to load lab reports page: ' . $e->getMessage());
        }
    }

    /**
     * Upload lab report
     */
    public function upload_report() {
        // Set JSON header
        header('Content-Type: application/json');
        
        try {
            // Get POST data
            $patient_id = $this->input->post('patient_id');
            $report_type = $this->input->post('report_type');
            $report_title = $this->input->post('report_title');
            $report_description = $this->input->post('report_description');
            $test_names = $this->input->post('test_names');
            $report_date = $this->input->post('report_date');
            $department = $this->input->post('department');
            $is_critical = $this->input->post('is_critical') ? 1 : 0;
            $notes = $this->input->post('notes');

            // Validate required fields
            if (empty($patient_id)) {
                echo json_encode(['success' => false, 'message' => 'Patient ID is required']);
                return;
            }
            
            if (empty($report_type)) {
                echo json_encode(['success' => false, 'message' => 'Report type is required']);
                return;
            }
            
            if (empty($report_title)) {
                echo json_encode(['success' => false, 'message' => 'Report title is required']);
                return;
            }
            
            if (empty($report_date)) {
                echo json_encode(['success' => false, 'message' => 'Report date is required']);
                return;
            }

            // Handle file upload
            $upload_result = $this->handleFileUpload();
            if (!$upload_result['success']) {
                echo json_encode($upload_result);
                return;
            }

            // Prepare data for insertion
            $report_data = [
                'patient_id' => $patient_id,
                'report_type' => $report_type,
                'report_title' => $report_title,
                'report_description' => $report_description,
                'file_name' => $upload_result['file_name'],
                'file_path' => $upload_result['file_path'],
                'file_size' => $upload_result['file_size'],
                'file_type' => $upload_result['file_type'],
                'original_filename' => $upload_result['original_filename'],
                'report_date' => $report_date,
                'test_names' => $test_names,
                'department' => $department ?: 'laboratory',
                'status' => 'pending',
                'is_critical' => $is_critical,
                'notes' => $notes,
                'uploaded_by' => $this->getStaffId(),
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            // Insert report
            $report_id = $this->lab_reports_model->insertReport($report_data);

            if ($report_id) {
                // Send notification if critical
                if ($is_critical) {
                    $this->sendCriticalReportNotification($report_id);
                }

                echo json_encode([
                    'success' => true, 
                    'message' => 'Lab report uploaded successfully',
                    'report_id' => $report_id
                ]);
            } else {
                // Clean up uploaded file if database insert failed
                if (file_exists($upload_result['file_path'])) {
                    unlink($upload_result['file_path']);
                }
                echo json_encode(['success' => false, 'message' => 'Failed to save report to database']);
            }

        } catch (Exception $e) {
            log_message('error', 'Lab report upload error: ' . $e->getMessage());
            echo json_encode(['success' => false, 'message' => 'Error: ' . $e->getMessage()]);
        }
    }

    /**
     * Handle file upload with comprehensive validation
     */
    private function handleFileUpload() {
        // Check if file was uploaded
        if (!isset($_FILES['lab_report_file'])) {
            return ['success' => false, 'message' => 'No file selected for upload'];
        }

        $file = $_FILES['lab_report_file'];

        // Check for upload errors
        if ($file['error'] != UPLOAD_ERR_OK) {
            $error_messages = [
                UPLOAD_ERR_INI_SIZE => 'File size exceeds server limit',
                UPLOAD_ERR_FORM_SIZE => 'File size exceeds form limit',
                UPLOAD_ERR_PARTIAL => 'File was only partially uploaded',
                UPLOAD_ERR_NO_FILE => 'No file was uploaded',
                UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder',
                UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk',
                UPLOAD_ERR_EXTENSION => 'File upload stopped by extension'
            ];
            
            $error_message = isset($error_messages[$file['error']]) ? 
                           $error_messages[$file['error']] : 
                           'Unknown upload error';
            
            return ['success' => false, 'message' => $error_message];
        }

        // File validation arrays
        $allowed_extensions = ['pdf', 'jpg', 'jpeg', 'png', 'doc', 'docx'];
        $allowed_mimes = [
            'application/pdf',
            'image/jpeg',
            'image/jpg', 
            'image/png',
            'application/msword',
            'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ];
        $max_size = 10 * 1024 * 1024; // 10MB

        // Check file size
        if ($file['size'] > $max_size) {
            return ['success' => false, 'message' => 'File size exceeds 10MB limit'];
        }

        // Check if file is empty
        if ($file['size'] == 0) {
            return ['success' => false, 'message' => 'File is empty'];
        }

        // Check file extension
        $file_ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        if (!in_array($file_ext, $allowed_extensions)) {
            return ['success' => false, 'message' => 'Invalid file type. Allowed: PDF, JPG, PNG, DOC, DOCX'];
        }

        // Check MIME type for additional security
        $file_mime = $file['type'];
        if (!in_array($file_mime, $allowed_mimes)) {
            // Try to detect MIME type using finfo if available
            if (function_exists('finfo_file')) {
                $finfo = finfo_open(FILEINFO_MIME_TYPE);
                $detected_mime = finfo_file($finfo, $file['tmp_name']);
                finfo_close($finfo);
                
                if (!in_array($detected_mime, $allowed_mimes)) {
                    return ['success' => false, 'message' => 'Invalid file content type'];
                }
                $file_mime = $detected_mime;
            }
        }

        // Create upload directory with year/month structure
        $upload_base = './uploads/lab_reports/';
        $upload_dir = $upload_base . date('Y/m/');
        
        // Create base directory if it doesn't exist
        if (!is_dir($upload_base)) {
            if (!mkdir($upload_base, 0755, true)) {
                return ['success' => false, 'message' => 'Failed to create upload base directory'];
            }
        }
        
        // Create year/month directory if it doesn't exist
        if (!is_dir($upload_dir)) {
            if (!mkdir($upload_dir, 0755, true)) {
                return ['success' => false, 'message' => 'Failed to create upload directory'];
            }
        }

        // Check if directory is writable
        if (!is_writable($upload_dir)) {
            return ['success' => false, 'message' => 'Upload directory is not writable'];
        }

        // Generate secure filename
        $file_name = 'lab_report_' . date('YmdHis') . '_' . uniqid() . '.' . $file_ext;
        $file_path = $upload_dir . $file_name;

        // Move uploaded file
        if (move_uploaded_file($file['tmp_name'], $file_path)) {
            // Set proper file permissions
            chmod($file_path, 0644);
            
            return [
                'success' => true,
                'file_name' => $file_name,
                'file_path' => $file_path,
                'file_size' => $file['size'],
                'file_type' => $file_mime,
                'original_filename' => $file['name']
            ];
        } else {
            return ['success' => false, 'message' => 'Failed to move uploaded file'];
        }
    }

    /**
     * Get patient reports
     */
    public function get_patient_reports() {
        header('Content-Type: application/json');
        
        $patient_id = $this->input->get('patient_id');
        
        if (!$patient_id) {
            echo json_encode(['success' => false, 'message' => 'Patient ID required']);
            return;
        }

        try {
            $reports = $this->lab_reports_model->getPatientReports($patient_id);
            echo json_encode(['success' => true, 'data' => $reports]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error fetching reports: ' . $e->getMessage()]);
        }
    }

    /**
     * Get single report details
     */
    public function get_report($report_id) {
        header('Content-Type: application/json');
        
        if (!$report_id) {
            echo json_encode(['success' => false, 'message' => 'Report ID required']);
            return;
        }

        try {
            $report = $this->lab_reports_model->getReportById($report_id);
            
            if ($report) {
                echo json_encode(['success' => true, 'data' => $report]);
            } else {
                echo json_encode(['success' => false, 'message' => 'Report not found']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error fetching report: ' . $e->getMessage()]);
        }
    }

    /**
     * Download report file
     */
    public function download_report($report_id) {
        if (!$report_id) {
            show_404();
            return;
        }

        try {
            $report = $this->lab_reports_model->getReportById($report_id);
            
            if (!$report) {
                show_404();
                return;
            }

            $file_path = $report['file_path'];
            
            if (!file_exists($file_path)) {
                show_error('File not found on server', 404);
                return;
            }

            // Force download with original filename
            $this->load->helper('download');
            force_download($report['original_filename'], file_get_contents($file_path));
            
        } catch (Exception $e) {
            log_message('error', 'Download error: ' . $e->getMessage());
            show_error('Download failed: ' . $e->getMessage());
        }
    }

    /**
     * Verify report
     */
    public function verify_report() {
        header('Content-Type: application/json');
        
        $report_id = $this->input->post('report_id');
        $notes = $this->input->post('notes');
        
        if (!$report_id) {
            echo json_encode(['success' => false, 'message' => 'Report ID required']);
            return;
        }

        try {
            $result = $this->lab_reports_model->verifyReport($report_id, $this->getStaffId(), $notes);
            
            if ($result) {
                echo json_encode(['success' => true, 'message' => 'Report verified successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to verify report']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error verifying report: ' . $e->getMessage()]);
        }
    }

    /**
     * Get all reports for DataTables
     */
    public function get_all_reports() {
        header('Content-Type: application/json');
        
        try {
            $request = $this->input->post();
            $result = $this->lab_reports_model->getReportsForDataTables($request);
            
            // Format data for DataTables
            $data = [];
            foreach ($result['data'] as $row) {
                $status_class = $this->getStatusClass($row['status']);
                $critical_badge = $row['is_critical'] ? ' <span class="label label-danger">Critical</span>' : '';
                
                $actions = '
                    <button class="btn btn-xs btn-primary" onclick="viewReport(' . $row['id'] . ')" title="View">
                        <i class="fa fa-eye"></i>
                    </button>
                    <button class="btn btn-xs btn-success" onclick="downloadReport(' . $row['id'] . ')" title="Download">
                        <i class="fa fa-download"></i>
                    </button>';
                
                if ($row['status'] != 'verified') {
                    $actions .= '
                        <button class="btn btn-xs btn-warning" onclick="verifyReport(' . $row['id'] . ')" title="Verify">
                            <i class="fa fa-check"></i>
                        </button>';
                }
                
                $data[] = [
                    'id' => $row['id'],
                    'patient_name' => $row['patient_name'],
                    'report_title' => $row['report_title'] . $critical_badge,
                    'report_type' => ucfirst(str_replace('_', ' ', $row['report_type'])),
                    'department' => ucfirst($row['department']),
                    'report_date' => date('Y-m-d', strtotime($row['report_date'])),
                    'status' => '<span class="label ' . $status_class . '">' . ucfirst($row['status']) . '</span>',
                    'actions' => $actions
                ];
            }
            
            echo json_encode([
                'draw' => $result['draw'],
                'recordsTotal' => $result['recordsTotal'],
                'recordsFiltered' => $result['recordsFiltered'],
                'data' => $data
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Get dashboard statistics
     */
    public function get_dashboard_stats() {
        header('Content-Type: application/json');
        
        try {
            $stats = $this->lab_reports_model->getDashboardStats();
            echo json_encode(['success' => true, 'data' => $stats]);
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error fetching stats: ' . $e->getMessage()]);
        }
    }

    /**
     * Search patients for dropdown
     */
    public function search_patients() {
        header('Content-Type: application/json');
        
        $search_term = $this->input->get('q');
        $page = $this->input->get('page', 1);
        $per_page = 10;
        
        try {
            $patients = $this->lab_reports_model->searchPatients($search_term, $per_page);
            
            $items = [];
            foreach ($patients as $patient) {
                $items[] = [
                    'id' => $patient['id'],
                    'text' => $patient['patient_name'] . ' (' . $patient['id'] . ') - ' . ($patient['mobileno'] ?: 'No phone'),
                    'patient_name' => $patient['patient_name'],
                    'mobileno' => $patient['mobileno'],
                    'age' => $patient['age'],
                    'gender' => $patient['gender']
                ];
            }
            
            echo json_encode([
                'items' => $items,
                'total_count' => count($items)
            ]);
            
        } catch (Exception $e) {
            echo json_encode([
                'items' => [],
                'total_count' => 0,
                'error' => $e->getMessage()
            ]);
        }
    }

    /**
     * Export reports to CSV
     */
    public function export_reports() {
        try {
            $filters = [
                'status' => $this->input->get('status'),
                'department' => $this->input->get('department'),
                'date_from' => $this->input->get('date_from'),
                'date_to' => $this->input->get('date_to')
            ];
            
            $reports = $this->lab_reports_model->getAllReports(1000, 0, '', $filters);
            
            // Set headers for CSV download
            $filename = 'lab_reports_' . date('Y-m-d_H-i-s') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            header('Pragma: no-cache');
            header('Expires: 0');
            
            $output = fopen('php://output', 'w');
            
            // CSV Headers
            fputcsv($output, [
                'ID', 'Patient Name', 'Report Title', 'Type', 'Department', 
                'Report Date', 'Status', 'Critical', 'Uploaded By', 'Created At'
            ]);
            
            // CSV Data
            foreach ($reports as $report) {
                fputcsv($output, [
                    $report['id'],
                    $report['patient_name'],
                    $report['report_title'],
                    ucfirst(str_replace('_', ' ', $report['report_type'])),
                    ucfirst($report['department']),
                    $report['report_date'],
                    ucfirst($report['status']),
                    $report['is_critical'] ? 'Yes' : 'No',
                    ($report['uploaded_by_name'] ?: '') . ' ' . ($report['uploaded_by_surname'] ?: ''),
                    $report['created_at']
                ]);
            }
            
            fclose($output);
            
        } catch (Exception $e) {
            log_message('error', 'Export error: ' . $e->getMessage());
            show_error('Export failed: ' . $e->getMessage());
        }
    }

    /**
     * Mark report as critical
     */
    public function mark_critical() {
        header('Content-Type: application/json');
        
        $report_id = $this->input->post('report_id');
        $is_critical = $this->input->post('is_critical') ? 1 : 0;
        
        if (!$report_id) {
            echo json_encode(['success' => false, 'message' => 'Report ID required']);
            return;
        }

        try {
            $result = $this->lab_reports_model->markAsCritical($report_id, $is_critical);
            
            if ($result) {
                if ($is_critical) {
                    $this->sendCriticalReportNotification($report_id);
                }
                echo json_encode(['success' => true, 'message' => 'Report updated successfully']);
            } else {
                echo json_encode(['success' => false, 'message' => 'Failed to update report']);
            }
        } catch (Exception $e) {
            echo json_encode(['success' => false, 'message' => 'Error updating report: ' . $e->getMessage()]);
        }
    }

    /**
     * Test endpoint to check if controller is working
     */
    public function test() {
        echo "<h1>Lab Reports Controller Test</h1>";
        echo "<p>Controller is working!</p>";
        echo "<p>Base URL: " . base_url() . "</p>";
        echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
        
        try {
            // Test database connection
            $this->db->select('COUNT(*) as patient_count');
            $this->db->from('patients');
            $query = $this->db->get();
            $result = $query->row();
            echo "<p>Patients in database: " . $result->patient_count . "</p>";
            
            // Test upload directory
            $upload_dir = './uploads/lab_reports/';
            if (is_dir($upload_dir)) {
                echo "<p>✅ Upload directory exists and is " . (is_writable($upload_dir) ? 'writable' : 'not writable') . "</p>";
            } else {
                echo "<p>❌ Upload directory does not exist</p>";
            }
            
            // Test model loading
            if ($this->load->is_loaded('lab_reports_model')) {
                echo "<p>✅ Lab reports model loaded successfully</p>";
            } else {
                echo "<p>❌ Lab reports model not loaded</p>";
            }
            
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
        }
    }

    // Helper methods
    
    /**
     * Get current staff ID
     */
    private function getStaffId() {
        try {
            if (method_exists($this->customlib, 'getStaffID')) {
                return $this->customlib->getStaffID();
            } else {
                $staff_id = $this->session->userdata('admin_id');
                return $staff_id ?: 1;
            }
        } catch (Exception $e) {
            log_message('error', 'Error getting staff ID: ' . $e->getMessage());
            return 1;
        }
    }

    /**
     * Get current staff information
     */
    private function getStaffInfo() {
        try {
            $staff_id = $this->getStaffId();
            if (method_exists($this->staff_model, 'get')) {
                return $this->staff_model->get($staff_id);
            } else {
                $this->db->select('*');
                $this->db->from('staff');
                $this->db->where('id', $staff_id);
                $query = $this->db->get();
                return $query->row_array();
            }
        } catch (Exception $e) {
            log_message('error', 'Error getting staff info: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get CSS class for status labels
     */
    private function getStatusClass($status) {
        $classes = [
            'pending' => 'label-warning',
            'completed' => 'label-primary',
            'verified' => 'label-success',
            'cancelled' => 'label-danger'
        ];
        return isset($classes[$status]) ? $classes[$status] : 'label-default';
    }

    /**
     * Send notification for critical reports
     */
    private function sendCriticalReportNotification($report_id) {
        try {
            $report = $this->lab_reports_model->getReportById($report_id);
            if ($report && $report['is_critical']) {
                // Log critical report for now
                // You can implement actual notification logic here
                log_message('info', 'Critical report uploaded - ID: ' . $report_id . ', Patient: ' . $report['patient_name']);
                
                // Example notification implementation:
                // $this->sendEmail($report);
                // $this->sendSMS($report);
                // $this->createSystemNotification($report);
            }
        } catch (Exception $e) {
            log_message('error', 'Failed to send critical report notification: ' . $e->getMessage());
        }
    }
}