<?php
if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Financial_reports extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load required models
        if (file_exists(APPPATH.'models/Financial_model.php')) {
            $this->load->model('financial_model');
        }
        $this->load->library('customlib');
        
        // Enable error reporting for debugging
        if (ENVIRONMENT === 'development') {
            error_reporting(E_ALL);
            ini_set('display_errors', 1);
        }
    }

    public function index() {
        $this->session->set_userdata('top_menu', 'Finance');
        $this->session->set_userdata('sub_menu', 'financial_reports');

        $data['title'] = 'Enhanced Financial Reports';
        $data['currency_symbol'] = $this->customlib->getHospitalCurrencyFormat();
        
        // Get current financial year dates
        $financial_year = $this->getFinancialYear();
        $data['financial_year_start'] = $financial_year['start'];
        $data['financial_year_end'] = $financial_year['end'];
        
        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial_reports/index', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // ENHANCED INCOME STATEMENT WITH CHART OF ACCOUNTS
    // =============================================
    
    public function income_statement($start_date = null, $end_date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$start_date) {
                $start_date = $this->uri->segment(4) ?: date('Y-04-01');
            }
            
            if (!$end_date) {
                $end_date = $this->uri->segment(5) ?: date('Y-03-31', strtotime('+1 year'));
            }
            
            if (!$this->isValidDate($start_date) || !$this->isValidDate($end_date)) {
                throw new Exception('Invalid date format. Please use YYYY-MM-DD format.');
            }
            
            if (strtotime($start_date) > strtotime($end_date)) {
                throw new Exception('Start date cannot be after end date.');
            }
            
            log_message('debug', "Enhanced Income statement called with dates: $start_date to $end_date");
            
            // Get enhanced income statement data using chart of accounts
            $income_data = $this->getEnhancedIncomeStatementData($start_date, $end_date);
            
            $response = array(
                'status' => 'success',
                'data' => array(
                    'period' => array(
                        'start_date' => $start_date,
                        'end_date' => $end_date
                    ),
                    'revenue' => $income_data['revenue'],
                    'total_revenue' => $income_data['total_revenue'],
                    'expenses' => $income_data['expenses'],
                    'total_expenses' => $income_data['total_expenses'],
                    'gross_profit' => $income_data['gross_profit'],
                    'operating_profit' => $income_data['operating_profit'],
                    'net_profit' => $income_data['net_profit'],
                    'profit_margin' => $income_data['profit_margin'],
                    'expense_ratio' => $income_data['expense_ratio'],
                    'department_breakdown' => $income_data['department_breakdown'],
                    'year_over_year_comparison' => $this->getYearOverYearComparison($start_date, $end_date)
                )
            );
            
            echo json_encode($response);
            
        } catch (Exception $e) {
            log_message('error', 'Enhanced Income Statement Error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage(),
                'debug_info' => array(
                    'start_date' => $start_date ?? 'null',
                    'end_date' => $end_date ?? 'null',
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            ));
        }
    }

    // =============================================
    // ENHANCED TRIAL BALANCE WITH GENERAL LEDGER
    // =============================================
    
    public function trial_balance($as_of_date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$as_of_date) {
                $as_of_date = $this->uri->segment(4) ?: date('Y-m-d');
            }
            
            if (!$this->isValidDate($as_of_date)) {
                throw new Exception('Invalid date format. Please use YYYY-MM-DD format.');
            }
            
            $trial_balance_data = $this->getEnhancedTrialBalanceData($as_of_date);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $trial_balance_data
            ));
            
        } catch (Exception $e) {
            log_message('error', 'Enhanced Trial Balance Error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage(),
                'debug_info' => array(
                    'as_of_date' => $as_of_date ?? 'null',
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            ));
        }
    }

    // =============================================
    // BALANCE SHEET IMPLEMENTATION
    // =============================================
    
    public function balance_sheet($as_of_date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$as_of_date) {
                $as_of_date = $this->uri->segment(4) ?: date('Y-m-d');
            }
            
            if (!$this->isValidDate($as_of_date)) {
                throw new Exception('Invalid date format. Please use YYYY-MM-DD format.');
            }
            
            $balance_sheet_data = $this->getBalanceSheetData($as_of_date);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $balance_sheet_data
            ));
            
        } catch (Exception $e) {
            log_message('error', 'Balance Sheet Error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage()
            ));
        }
    }

    // =============================================
    // CASH FLOW STATEMENT
    // =============================================
    
    public function cash_flow($start_date = null, $end_date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$start_date) {
                $start_date = $this->uri->segment(4) ?: date('Y-04-01');
            }
            
            if (!$end_date) {
                $end_date = $this->uri->segment(5) ?: date('Y-03-31', strtotime('+1 year'));
            }
            
            $cash_flow_data = $this->getCashFlowData($start_date, $end_date);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $cash_flow_data
            ));
            
        } catch (Exception $e) {
            log_message('error', 'Cash Flow Error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage()
            ));
        }
    }

    // =============================================
    // ENHANCED DAILY INCOME WITH DETAILED BREAKDOWN
    // =============================================
    
    public function daily_income($date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$date) {
                $date = $this->uri->segment(4) ?: date('Y-m-d');
            }
            
            if (!$this->isValidDate($date)) {
                throw new Exception('Invalid date format. Please use YYYY-MM-DD format.');
            }
            
            $daily_data = $this->getEnhancedDailyIncomeData($date);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $daily_data
            ));
            
        } catch (Exception $e) {
            log_message('error', 'Enhanced Daily Income Error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage(),
                'debug_info' => array(
                    'date' => $date ?? 'null',
                    'file' => $e->getFile(),
                    'line' => $e->getLine()
                )
            ));
        }
    }

    // =============================================
    // DEPARTMENT REVENUE ANALYSIS
    // =============================================
    
    public function department_revenue($start_date = null, $end_date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$start_date) {
                $start_date = $this->uri->segment(4) ?: date('Y-m-01');
            }
            
            if (!$end_date) {
                $end_date = $this->uri->segment(5) ?: date('Y-m-d');
            }
            
            $department_data = $this->getDepartmentRevenueData($start_date, $end_date);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $department_data
            ));
            
        } catch (Exception $e) {
            log_message('error', 'Department Revenue Error: ' . $e->getMessage());
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage()
            ));
        }
    }

    // =============================================
    // FINANCIAL ANALYSIS AND RATIOS
    // =============================================

    public function financial_ratios($as_of_date = null) {
        header('Content-Type: application/json');
        
        try {
            if (!$as_of_date) {
                $as_of_date = date('Y-m-d');
            }
            
            $ratios = $this->calculateFinancialRatios($as_of_date);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $ratios
            ));
            
        } catch (Exception $e) {
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage()
            ));
        }
    }

    public function revenue_trends($months = 12) {
        header('Content-Type: application/json');
        
        try {
            $trends = $this->getRevenueTrends($months);
            
            echo json_encode(array(
                'status' => 'success',
                'data' => $trends
            ));
            
        } catch (Exception $e) {
            echo json_encode(array(
                'status' => 'error',
                'message' => $e->getMessage()
            ));
        }
    }

    // =============================================
    // DATA RETRIEVAL METHODS
    // =============================================

    private function getEnhancedIncomeStatementData($start_date, $end_date) {
        try {
            // Check if we have the enhanced financial structure
            if ($this->db->table_exists('general_ledger') && $this->db->table_exists('chart_of_accounts')) {
                return $this->getIncomeStatementFromGeneralLedger($start_date, $end_date);
            } else {
                // Fall back to legacy method with enhanced formatting
                return $this->getBasicIncomeDataEnhanced($start_date, $end_date);
            }
        } catch (Exception $e) {
            log_message('error', 'Error getting enhanced income statement data: ' . $e->getMessage());
            throw $e;
        }
    }

    private function getIncomeStatementFromGeneralLedger($start_date, $end_date) {
        // Revenue data from general ledger
        $this->db->select('coa.account_name, coa.account_code, coa.account_category, SUM(gl.credit_amount - gl.debit_amount) as amount');
        $this->db->from('general_ledger gl');
        $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
        $this->db->where('coa.account_type', 'revenue');
        $this->db->where('gl.transaction_date >=', $start_date);
        $this->db->where('gl.transaction_date <=', $end_date);
        $this->db->group_by('coa.id, coa.account_name, coa.account_code, coa.account_category');
        $this->db->having('amount > 0');
        $this->db->order_by('coa.account_code');
        
        $revenue_query = $this->db->get();
        $revenue_accounts = $revenue_query->result_array();
        
        // Expense data from general ledger
        $this->db->select('coa.account_name, coa.account_code, coa.account_category, SUM(gl.debit_amount - gl.credit_amount) as amount');
        $this->db->from('general_ledger gl');
        $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
        $this->db->where('coa.account_type', 'expense');
        $this->db->where('gl.transaction_date >=', $start_date);
        $this->db->where('gl.transaction_date <=', $end_date);
        $this->db->group_by('coa.id, coa.account_name, coa.account_code, coa.account_category');
        $this->db->having('amount > 0');
        $this->db->order_by('coa.account_code');
        
        $expense_query = $this->db->get();
        $expense_accounts = $expense_query->result_array();
        
        // Organize revenue by category
        $revenue = array();
        $total_revenue = 0;
        foreach ($revenue_accounts as $account) {
            if (!isset($revenue[$account['account_category']])) {
                $revenue[$account['account_category']] = array();
            }
            $revenue[$account['account_category']][] = $account;
            $total_revenue += floatval($account['amount']);
        }
        
        // Organize expenses by category
        $expenses = array();
        $total_expenses = 0;
        $category_totals = array();
        foreach ($expense_accounts as $account) {
            if (!isset($expenses[$account['account_category']])) {
                $expenses[$account['account_category']] = array();
            }
            $expenses[$account['account_category']][] = $account;
            $total_expenses += floatval($account['amount']);
            
            if (!isset($category_totals[$account['account_category']])) {
                $category_totals[$account['account_category']] = 0;
            }
            $category_totals[$account['account_category']] += floatval($account['amount']);
        }
        
        // Calculate metrics
        $direct_costs = $category_totals['direct_costs'] ?? 0;
        $operational_costs = ($category_totals['operational'] ?? 0) + ($category_totals['staff_costs'] ?? 0);
        $administrative_costs = $category_totals['administrative'] ?? 0;
        $financial_costs = $category_totals['financial'] ?? 0;
        $non_cash_costs = $category_totals['non_cash'] ?? 0;
        
        $gross_profit = $total_revenue - $direct_costs;
        $operating_profit = $gross_profit - $operational_costs - $administrative_costs;
        $net_profit = $operating_profit - $financial_costs - $non_cash_costs;
        $profit_margin = $total_revenue > 0 ? round(($net_profit / $total_revenue) * 100, 2) : 0;
        $expense_ratio = $total_revenue > 0 ? round(($total_expenses / $total_revenue) * 100, 2) : 0;
        
        // Get department breakdown
        $department_breakdown = $this->getDepartmentBreakdown($start_date, $end_date);
        
        return array(
            'revenue' => $revenue,
            'total_revenue' => $total_revenue,
            'expenses' => $expenses,
            'total_expenses' => $total_expenses,
            'gross_profit' => $gross_profit,
            'operating_profit' => $operating_profit,
            'net_profit' => $net_profit,
            'profit_margin' => $profit_margin,
            'expense_ratio' => $expense_ratio,
            'department_breakdown' => $department_breakdown,
            'category_totals' => $category_totals
        );
    }

    private function getBasicIncomeDataEnhanced($start_date, $end_date) {
        try {
            $data = array(
                'opd_revenue' => 0,
                'ipd_revenue' => 0,
                'pharmacy_revenue' => 0,
                'lab_revenue' => 0,
                'ambulance_revenue' => 0,
                'other_income' => 0,
                'staff_expenses' => 0,
                'supply_expenses' => 0,
                'utility_expenses' => 0,
                'maintenance_expenses' => 0,
                'admin_expenses' => 0,
                'other_expenses' => 0
            );
            
            // Enhanced OPD revenue query with error handling
            if ($this->db->table_exists('opd_details')) {
                try {
                    $this->db->select('COALESCE(SUM(amount), 0) as opd_revenue, COUNT(*) as opd_count');
                    $this->db->from('opd_details');
                    $this->db->where('appointment_date >=', $start_date);
                    $this->db->where('appointment_date <=', $end_date);
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $data['opd_revenue'] = floatval($row->opd_revenue ?? 0);
                        $data['opd_count'] = intval($row->opd_count ?? 0);
                    }
                } catch (Exception $e) {
                    log_message('error', 'OPD revenue query error: ' . $e->getMessage());
                }
            }
            
            // Enhanced IPD revenue query
            if ($this->db->table_exists('ipd_details')) {
                try {
                    $this->db->select('COALESCE(SUM(amount), 0) as ipd_revenue, COUNT(*) as ipd_count');
                    $this->db->from('ipd_details');
                    $this->db->where('date >=', $start_date);
                    $this->db->where('date <=', $end_date);
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $data['ipd_revenue'] = floatval($row->ipd_revenue ?? 0);
                        $data['ipd_count'] = intval($row->ipd_count ?? 0);
                    }
                } catch (Exception $e) {
                    log_message('error', 'IPD revenue query error: ' . $e->getMessage());
                }
            }
            
            // Enhanced Pharmacy revenue query
            if ($this->db->table_exists('pharmacy_bill_basic')) {
                try {
                    $this->db->select('COALESCE(SUM(net_amount), 0) as pharmacy_revenue, COUNT(*) as pharmacy_count');
                    $this->db->from('pharmacy_bill_basic');
                    $this->db->where('date >=', $start_date);
                    $this->db->where('date <=', $end_date);
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $data['pharmacy_revenue'] = floatval($row->pharmacy_revenue ?? 0);
                        $data['pharmacy_count'] = intval($row->pharmacy_count ?? 0);
                    }
                } catch (Exception $e) {
                    log_message('error', 'Pharmacy revenue query error: ' . $e->getMessage());
                }
            }
            
            // Enhanced Laboratory revenue query
            if ($this->db->table_exists('pathology_billing')) {
                try {
                    $this->db->select('COALESCE(SUM(amount), 0) as lab_revenue, COUNT(*) as lab_count');
                    $this->db->from('pathology_billing');
                    $this->db->where('date >=', $start_date);
                    $this->db->where('date <=', $end_date);
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $data['lab_revenue'] = floatval($row->lab_revenue ?? 0);
                        $data['lab_count'] = intval($row->lab_count ?? 0);
                    }
                } catch (Exception $e) {
                    log_message('error', 'Lab revenue query error: ' . $e->getMessage());
                }
            }
            
            // Enhanced Ambulance revenue query
            if ($this->db->table_exists('ambulance_call')) {
                try {
                    $this->db->select('COALESCE(SUM(amount), 0) as ambulance_revenue, COUNT(*) as ambulance_count');
                    $this->db->from('ambulance_call');
                    $this->db->where('date >=', $start_date);
                    $this->db->where('date <=', $end_date);
                    $query = $this->db->get();
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $data['ambulance_revenue'] = floatval($row->ambulance_revenue ?? 0);
                        $data['ambulance_count'] = intval($row->ambulance_count ?? 0);
                    }
                } catch (Exception $e) {
                    log_message('error', 'Ambulance revenue query error: ' . $e->getMessage());
                }
            }
            
            // Calculate totals
            $total_revenue = $data['opd_revenue'] + $data['ipd_revenue'] + 
                           $data['pharmacy_revenue'] + $data['lab_revenue'] + 
                           $data['ambulance_revenue'] + $data['other_income'];
            
            $total_expenses = $data['staff_expenses'] + $data['supply_expenses'] + 
                            $data['utility_expenses'] + $data['maintenance_expenses'] + 
                            $data['admin_expenses'] + $data['other_expenses'];
            
            // Format for enhanced structure
            return array(
                'revenue' => array(
                    'medical_services' => array(
                        array('account_name' => 'OPD Consultations', 'account_code' => '4000', 'amount' => $data['opd_revenue']),
                        array('account_name' => 'IPD Services', 'account_code' => '4100', 'amount' => $data['ipd_revenue']),
                        array('account_name' => 'Pharmacy Sales', 'account_code' => '4200', 'amount' => $data['pharmacy_revenue']),
                        array('account_name' => 'Laboratory Tests', 'account_code' => '4300', 'amount' => $data['lab_revenue']),
                        array('account_name' => 'Ambulance Services', 'account_code' => '4400', 'amount' => $data['ambulance_revenue'])
                    ),
                    'other_revenue' => array(
                        array('account_name' => 'Other Income', 'account_code' => '4900', 'amount' => $data['other_income'])
                    )
                ),
                'total_revenue' => $total_revenue,
                'expenses' => array(
                    'staff_costs' => array(
                        array('account_name' => 'Staff Salaries', 'account_code' => '5100', 'amount' => $data['staff_expenses'])
                    ),
                    'operational' => array(
                        array('account_name' => 'Medical Supplies', 'account_code' => '5020', 'amount' => $data['supply_expenses']),
                        array('account_name' => 'Utilities', 'account_code' => '5200', 'amount' => $data['utility_expenses']),
                        array('account_name' => 'Maintenance', 'account_code' => '5300', 'amount' => $data['maintenance_expenses'])
                    ),
                    'administrative' => array(
                        array('account_name' => 'Administrative Expenses', 'account_code' => '5400', 'amount' => $data['admin_expenses']),
                        array('account_name' => 'Other Expenses', 'account_code' => '5900', 'amount' => $data['other_expenses'])
                    )
                ),
                'total_expenses' => $total_expenses,
                'gross_profit' => $total_revenue - $total_expenses,
                'operating_profit' => $total_revenue - $total_expenses,
                'net_profit' => $total_revenue - $total_expenses,
                'profit_margin' => $total_revenue > 0 ? round((($total_revenue - $total_expenses) / $total_revenue) * 100, 2) : 0,
                'expense_ratio' => $total_revenue > 0 ? round(($total_expenses / $total_revenue) * 100, 2) : 0,
                'department_breakdown' => $this->getBasicDepartmentBreakdown($start_date, $end_date, $data),
                'transaction_counts' => array(
                    'opd_count' => $data['opd_count'] ?? 0,
                    'ipd_count' => $data['ipd_count'] ?? 0,
                    'pharmacy_count' => $data['pharmacy_count'] ?? 0,
                    'lab_count' => $data['lab_count'] ?? 0,
                    'ambulance_count' => $data['ambulance_count'] ?? 0
                )
            );
            
        } catch (Exception $e) {
            log_message('error', 'Error getting enhanced basic income data: ' . $e->getMessage());
            throw $e;
        }
    }

    private function getEnhancedTrialBalanceData($as_of_date) {
        if ($this->db->table_exists('general_ledger') && $this->db->table_exists('chart_of_accounts')) {
            // Use general ledger for trial balance
            $this->db->select('coa.account_code, coa.account_name, coa.account_type, coa.account_category');
            $this->db->select('SUM(CASE WHEN coa.account_type IN ("asset", "expense") THEN gl.debit_amount - gl.credit_amount ELSE 0 END) as debit_balance');
            $this->db->select('SUM(CASE WHEN coa.account_type IN ("liability", "equity", "revenue") THEN gl.credit_amount - gl.debit_amount ELSE 0 END) as credit_balance');
            $this->db->from('chart_of_accounts coa');
            $this->db->join('general_ledger gl', 'coa.id = gl.account_id', 'left');
            $this->db->where('coa.is_active', 'yes');
            $this->db->where('gl.transaction_date <=', $as_of_date);
            $this->db->group_by('coa.id, coa.account_code, coa.account_name, coa.account_type, coa.account_category');
            $this->db->having('ABS(debit_balance) > 0.01 OR ABS(credit_balance) > 0.01');
            $this->db->order_by('coa.account_code');
            
            $query = $this->db->get();
            $accounts = $query->result_array();
        } else {
            // Fall back to basic trial balance
            $accounts = $this->getBasicTrialBalanceData($as_of_date);
        }
        
        $total_debits = 0;
        $total_credits = 0;
        $account_types = array();
        
        foreach ($accounts as &$account) {
            $account['debit_balance'] = floatval($account['debit_balance'] ?? 0);
            $account['credit_balance'] = floatval($account['credit_balance'] ?? 0);
            $total_debits += $account['debit_balance'];
            $total_credits += $account['credit_balance'];
            
            // Group by account type for summary
            if (!isset($account_types[$account['account_type']])) {
                $account_types[$account['account_type']] = array(
                    'debit_total' => 0,
                    'credit_total' => 0,
                    'count' => 0
                );
            }
            $account_types[$account['account_type']]['debit_total'] += $account['debit_balance'];
            $account_types[$account['account_type']]['credit_total'] += $account['credit_balance'];
            $account_types[$account['account_type']]['count']++;
        }
        
        return array(
            'as_of_date' => $as_of_date,
            'accounts' => $accounts,
            'total_debits' => $total_debits,
            'total_credits' => $total_credits,
            'is_balanced' => abs($total_debits - $total_credits) < 0.01,
            'difference' => $total_debits - $total_credits,
            'account_type_summary' => $account_types,
            'account_count' => count($accounts)
        );
    }

    private function getBalanceSheetData($as_of_date) {
        if (!$this->db->table_exists('general_ledger') || !$this->db->table_exists('chart_of_accounts')) {
            throw new Exception('Enhanced financial tables not available for Balance Sheet');
        }
        
        // Assets
        $this->db->select('coa.account_code, coa.account_name, coa.account_category, SUM(gl.debit_amount - gl.credit_amount) as amount');
        $this->db->from('general_ledger gl');
        $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
       $this->db->where('coa.account_type', 'asset');
       $this->db->where('gl.transaction_date <=', $as_of_date);
       $this->db->group_by('coa.id, coa.account_code, coa.account_name, coa.account_category');
       $this->db->having('amount != 0');
       $this->db->order_by('coa.account_code');
       
       $assets_query = $this->db->get();
       $assets = $assets_query->result_array();
       
       // Liabilities
       $this->db->select('coa.account_code, coa.account_name, coa.account_category, SUM(gl.credit_amount - gl.debit_amount) as amount');
       $this->db->from('general_ledger gl');
       $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
       $this->db->where('coa.account_type', 'liability');
       $this->db->where('gl.transaction_date <=', $as_of_date);
       $this->db->group_by('coa.id, coa.account_code, coa.account_name, coa.account_category');
       $this->db->having('amount != 0');
       $this->db->order_by('coa.account_code');
       
       $liabilities_query = $this->db->get();
       $liabilities = $liabilities_query->result_array();
       
       // Equity
       $this->db->select('coa.account_code, coa.account_name, coa.account_category, SUM(gl.credit_amount - gl.debit_amount) as amount');
       $this->db->from('general_ledger gl');
       $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
       $this->db->where('coa.account_type', 'equity');
       $this->db->where('gl.transaction_date <=', $as_of_date);
       $this->db->group_by('coa.id, coa.account_code, coa.account_name, coa.account_category');
       $this->db->having('amount != 0');
       $this->db->order_by('coa.account_code');
       
       $equity_query = $this->db->get();
       $equity = $equity_query->result_array();
       
       // Calculate totals
       $total_assets = array_sum(array_column($assets, 'amount'));
       $total_liabilities = array_sum(array_column($liabilities, 'amount'));
       $total_equity = array_sum(array_column($equity, 'amount'));
       
       // Group by category
       $assets_by_category = array();
       foreach ($assets as $asset) {
           $assets_by_category[$asset['account_category']][] = $asset;
       }
       
       $liabilities_by_category = array();
       foreach ($liabilities as $liability) {
           $liabilities_by_category[$liability['account_category']][] = $liability;
       }
       
       $equity_by_category = array();
       foreach ($equity as $eq) {
           $equity_by_category[$eq['account_category']][] = $eq;
       }
       
       return array(
           'as_of_date' => $as_of_date,
           'assets' => $assets_by_category,
           'liabilities' => $liabilities_by_category,
           'equity' => $equity_by_category,
           'total_assets' => $total_assets,
           'total_liabilities' => $total_liabilities,
           'total_equity' => $total_equity,
           'is_balanced' => abs($total_assets - ($total_liabilities + $total_equity)) < 0.01,
           'balance_difference' => $total_assets - ($total_liabilities + $total_equity)
       );
   }

   private function getCashFlowData($start_date, $end_date) {
       if ($this->db->table_exists('financial_transactions')) {
           // Operating cash flows from financial_transactions
           $this->db->select('SUM(amount) as operating_cash');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           $cash_in = $this->db->get()->row()->operating_cash ?? 0;
           
           $this->db->select('SUM(amount) as operating_cash');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'expense');
           $this->db->where('category !=', 'non_cash'); // Exclude depreciation
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           $cash_out = $this->db->get()->row()->operating_cash ?? 0;
           
           $net_operating_cash = $cash_in - $cash_out;
           
           // Get cash flow by payment method
           $this->db->select('payment_method, SUM(amount) as amount');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           $this->db->group_by('payment_method');
           $payment_methods = $this->db->get()->result_array();
           
       } else {
           // Fallback using basic tables
           $cash_in = $this->getBasicCashInflows($start_date, $end_date);
           $cash_out = $this->getBasicCashOutflows($start_date, $end_date);
           $net_operating_cash = $cash_in - $cash_out;
           $payment_methods = $this->getBasicPaymentMethods($start_date, $end_date);
       }
       
       return array(
           'period' => array('start_date' => $start_date, 'end_date' => $end_date),
           'cash_inflows' => $cash_in,
           'cash_outflows' => $cash_out,
           'net_cash_flow' => $net_operating_cash,
           'payment_method_breakdown' => $payment_methods,
           'cash_flow_ratio' => $cash_in > 0 ? round($net_operating_cash / $cash_in, 2) : 0
       );
   }

   private function getDepartmentBreakdown($start_date, $end_date) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('department, SUM(amount) as revenue, COUNT(*) as transactions, COUNT(DISTINCT patient_id) as unique_patients');
           $this->db->select('AVG(amount) as avg_transaction');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           $this->db->group_by('department');
           $this->db->order_by('revenue', 'DESC');
           
           return $this->db->get()->result_array();
       }
       
       return $this->getBasicDepartmentBreakdown($start_date, $end_date);
   }

   private function getBasicDepartmentBreakdown($start_date, $end_date, $data = null) {
       if (!$data) {
           $data = array(
               'opd_revenue' => 0,
               'ipd_revenue' => 0,
               'pharmacy_revenue' => 0,
               'lab_revenue' => 0,
               'ambulance_revenue' => 0
           );
       }
       
       return array(
           array(
               'department' => 'opd',
               'revenue' => $data['opd_revenue'],
               'transactions' => $data['opd_count'] ?? 0,
               'unique_patients' => $data['opd_count'] ?? 0,
               'avg_transaction' => $data['opd_count'] > 0 ? round($data['opd_revenue'] / $data['opd_count'], 2) : 0
           ),
           array(
               'department' => 'ipd',
               'revenue' => $data['ipd_revenue'],
               'transactions' => $data['ipd_count'] ?? 0,
               'unique_patients' => $data['ipd_count'] ?? 0,
               'avg_transaction' => $data['ipd_count'] > 0 ? round($data['ipd_revenue'] / $data['ipd_count'], 2) : 0
           ),
           array(
               'department' => 'pharmacy',
               'revenue' => $data['pharmacy_revenue'],
               'transactions' => $data['pharmacy_count'] ?? 0,
               'unique_patients' => $data['pharmacy_count'] ?? 0,
               'avg_transaction' => $data['pharmacy_count'] > 0 ? round($data['pharmacy_revenue'] / $data['pharmacy_count'], 2) : 0
           ),
           array(
               'department' => 'laboratory',
               'revenue' => $data['lab_revenue'],
               'transactions' => $data['lab_count'] ?? 0,
               'unique_patients' => $data['lab_count'] ?? 0,
               'avg_transaction' => $data['lab_count'] > 0 ? round($data['lab_revenue'] / $data['lab_count'], 2) : 0
           ),
           array(
               'department' => 'ambulance',
               'revenue' => $data['ambulance_revenue'],
               'transactions' => $data['ambulance_count'] ?? 0,
               'unique_patients' => $data['ambulance_count'] ?? 0,
               'avg_transaction' => $data['ambulance_count'] > 0 ? round($data['ambulance_revenue'] / $data['ambulance_count'], 2) : 0
           )
       );
   }

private function getEnhancedDailyIncomeData($date) {
    if ($this->db->table_exists('financial_transactions')) {
        // Get detailed daily breakdown from financial_transactions
        $this->db->select('department, category, subcategory, COUNT(*) as count, SUM(amount) as amount, payment_method');
        $this->db->from('financial_transactions');
        $this->db->where('transaction_type', 'revenue');
        $this->db->where('DATE(transaction_date)', $date);
        $this->db->group_by('department, category, subcategory, payment_method');
        $this->db->order_by('amount', 'DESC');
        
        $detailed_breakdown = $this->db->get()->result_array();
        
        // Get hourly breakdown
        $this->db->select('HOUR(transaction_date) as hour, SUM(amount) as amount, COUNT(*) as count');
        $this->db->from('financial_transactions');
        $this->db->where('transaction_type', 'revenue');
        $this->db->where('DATE(transaction_date)', $date);
        $this->db->group_by('HOUR(transaction_date)');
        
        $hourly_data = $this->db->get()->result_array();
        
        // Format hourly data
        $hourly_breakdown = array();
        for ($i = 0; $i < 24; $i++) {
            $hourly_breakdown[sprintf('%02d:00', $i)] = array('amount' => 0, 'count' => 0);
        }
        
        foreach ($hourly_data as $hour_data) {
            $hour_key = sprintf('%02d:00', $hour_data['hour']);
            $hourly_breakdown[$hour_key] = array(
                'amount' => floatval($hour_data['amount']),
                'count' => intval($hour_data['count'])
            );
        }
        
        // Calculate totals
        $total_income = array_sum(array_column($detailed_breakdown, 'amount'));
        $total_transactions = array_sum(array_column($detailed_breakdown, 'count'));
        
        // Get payment method breakdown
        $payment_methods = array();
        foreach ($detailed_breakdown as $item) {
            if (!isset($payment_methods[$item['payment_method']])) {
                $payment_methods[$item['payment_method']] = 0;
            }
            $payment_methods[$item['payment_method']] += floatval($item['amount']);
        }
        
        // Get unique patients
        $this->db->select('COUNT(DISTINCT patient_id) as unique_patients');
        $this->db->from('financial_transactions');
        $this->db->where('transaction_type', 'revenue');
        $this->db->where('DATE(transaction_date)', $date);
        $unique_patients = $this->db->get()->row()->unique_patients ?? 0;
        
        // Format income breakdown to match expected structure
        $income_breakdown = array(
            'opd_consultations' => array('count' => 0, 'amount' => 0),
            'ipd_services' => array('count' => 0, 'amount' => 0),
            'pharmacy_sales' => array('count' => 0, 'amount' => 0),
            'laboratory_tests' => array('count' => 0, 'amount' => 0),
            'ambulance_services' => array('count' => 0, 'amount' => 0),
            'other_services' => array('count' => 0, 'amount' => 0)
        );
        
        // Map detailed breakdown to expected format
        foreach ($detailed_breakdown as $item) {
            switch ($item['department']) {
                case 'opd':
                    $income_breakdown['opd_consultations']['count'] += $item['count'];
                    $income_breakdown['opd_consultations']['amount'] += $item['amount'];
                    break;
                case 'ipd':
                    $income_breakdown['ipd_services']['count'] += $item['count'];
                    $income_breakdown['ipd_services']['amount'] += $item['amount'];
                    break;
                case 'pharmacy':
                    $income_breakdown['pharmacy_sales']['count'] += $item['count'];
                    $income_breakdown['pharmacy_sales']['amount'] += $item['amount'];
                    break;
                case 'laboratory':
                    $income_breakdown['laboratory_tests']['count'] += $item['count'];
                    $income_breakdown['laboratory_tests']['amount'] += $item['amount'];
                    break;
                case 'ambulance':
                    $income_breakdown['ambulance_services']['count'] += $item['count'];
                    $income_breakdown['ambulance_services']['amount'] += $item['amount'];
                    break;
                default:
                    $income_breakdown['other_services']['count'] += $item['count'];
                    $income_breakdown['other_services']['amount'] += $item['amount'];
                    break;
            }
        }
        
        return array(
            'date' => $date,
            'income_breakdown' => $income_breakdown,
            'detailed_breakdown' => $detailed_breakdown,
            'total_income' => $total_income,
            'total_transactions' => $total_transactions,
            'unique_patients' => $unique_patients,
            'average_transaction' => $total_transactions > 0 ? round($total_income / $total_transactions, 2) : 0,
            'average_per_patient' => $unique_patients > 0 ? round($total_income / $unique_patients, 2) : 0,
            'payment_methods' => $payment_methods,
            'hourly_breakdown' => $hourly_breakdown,
            'peak_hour' => $this->findPeakHour($hourly_breakdown),
            'department_summary' => $this->getDepartmentSummary($detailed_breakdown)
        );
    } else {
        // Fall back to legacy method which now returns consistent format
        return $this->getDailyIncomeData($date);
    }
}

   private function findPeakHour($hourly_breakdown) {
       $peak_hour = '00:00';
       $peak_amount = 0;
       
       foreach ($hourly_breakdown as $hour => $data) {
           if ($data['amount'] > $peak_amount) {
               $peak_amount = $data['amount'];
               $peak_hour = $hour;
           }
       }
       
       return array(
           'hour' => $peak_hour,
           'amount' => $peak_amount,
           'transactions' => $hourly_breakdown[$peak_hour]['count']
       );
   }

   private function getDepartmentSummary($detailed_breakdown) {
       $departments = array();
       
       foreach ($detailed_breakdown as $item) {
           if (!isset($departments[$item['department']])) {
               $departments[$item['department']] = array(
                   'total_amount' => 0,
                   'total_count' => 0,
                   'categories' => array()
               );
           }
           
           $departments[$item['department']]['total_amount'] += $item['amount'];
           $departments[$item['department']]['total_count'] += $item['count'];
           
           if (!isset($departments[$item['department']]['categories'][$item['category']])) {
               $departments[$item['department']]['categories'][$item['category']] = 0;
           }
           $departments[$item['department']]['categories'][$item['category']] += $item['amount'];
       }
       
       return $departments;
   }

   private function getYearOverYearComparison($start_date, $end_date) {
       try {
           $previous_year_start = date('Y-m-d', strtotime($start_date . ' -1 year'));
           $previous_year_end = date('Y-m-d', strtotime($end_date . ' -1 year'));
           
           $current_data = $this->getBasicRevenueTotal($start_date, $end_date);
           $previous_data = $this->getBasicRevenueTotal($previous_year_start, $previous_year_end);
           
           $growth_rate = $previous_data > 0 ? round((($current_data - $previous_data) / $previous_data) * 100, 2) : 0;
           
           return array(
               'current_period' => array(
                   'start_date' => $start_date,
                   'end_date' => $end_date,
                   'revenue' => $current_data
               ),
               'previous_period' => array(
                   'start_date' => $previous_year_start,
                   'end_date' => $previous_year_end,
                   'revenue' => $previous_data
               ),
               'growth_rate' => $growth_rate,
               'growth_amount' => $current_data - $previous_data
           );
       } catch (Exception $e) {
           log_message('error', 'Year over year comparison error: ' . $e->getMessage());
           return array();
       }
   }

   private function getBasicRevenueTotal($start_date, $end_date) {
       $total = 0;
       
       // Sum from all revenue sources
       $tables = array(
           'opd_details' => array('date_field' => 'appointment_date', 'amount_field' => 'amount'),
           'ipd_details' => array('date_field' => 'date', 'amount_field' => 'amount'),
           'pharmacy_bill_basic' => array('date_field' => 'date', 'amount_field' => 'net_amount'),
           'pathology_billing' => array('date_field' => 'date', 'amount_field' => 'amount'),
           'ambulance_call' => array('date_field' => 'date', 'amount_field' => 'amount')
       );
       
       foreach ($tables as $table => $config) {
           if ($this->db->table_exists($table)) {
               try {
                   $this->db->select("COALESCE(SUM({$config['amount_field']}), 0) as revenue");
                   $this->db->from($table);
                   $this->db->where("{$config['date_field']} >=", $start_date);
                   $this->db->where("{$config['date_field']} <=", $end_date);
                   $query = $this->db->get();
                   
                   if ($query && $query->num_rows() > 0) {
                       $total += floatval($query->row()->revenue);
                   }
               } catch (Exception $e) {
                   log_message('error', "Error getting revenue from {$table}: " . $e->getMessage());
               }
           }
       }
       
       return $total;
   }

   // =============================================
   // FINANCIAL ANALYSIS METHODS
   // =============================================

   private function calculateFinancialRatios($as_of_date) {
       if (!$this->db->table_exists('general_ledger')) {
           throw new Exception('Financial analysis requires enhanced financial tables');
       }
       
       // Get current assets
       $this->db->select('SUM(gl.debit_amount - gl.credit_amount) as current_assets');
       $this->db->from('general_ledger gl');
       $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
       $this->db->where('coa.account_type', 'asset');
       $this->db->where('coa.account_category', 'current_assets');
       $this->db->where('gl.transaction_date <=', $as_of_date);
       $current_assets = $this->db->get()->row()->current_assets ?? 0;
       
       // Get current liabilities
       $this->db->select('SUM(gl.credit_amount - gl.debit_amount) as current_liabilities');
       $this->db->from('general_ledger gl');
       $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
       $this->db->where('coa.account_type', 'liability');
       $this->db->where('coa.account_category', 'current_liabilities');
       $this->db->where('gl.transaction_date <=', $as_of_date);
       $current_liabilities = $this->db->get()->row()->current_liabilities ?? 0;
       
       // Get total assets and liabilities
       $total_assets = $this->getAccountTypeTotal('asset', $as_of_date);
       $total_liabilities = $this->getAccountTypeTotal('liability', $as_of_date);
       
       // Get revenue and expenses for the year
       $year_start = date('Y-01-01', strtotime($as_of_date));
       $total_revenue = $this->getAccountTypeTotal('revenue', $as_of_date, $year_start);
       $total_expenses = $this->getAccountTypeTotal('expense', $as_of_date, $year_start);
       
       // Calculate ratios
       $current_ratio = $current_liabilities > 0 ? round($current_assets / $current_liabilities, 2) : 0;
       $debt_to_asset_ratio = $total_assets > 0 ? round($total_liabilities / $total_assets, 2) : 0;
       $net_profit_margin = $total_revenue > 0 ? round((($total_revenue - $total_expenses) / $total_revenue) * 100, 2) : 0;
       $asset_turnover = $total_assets > 0 ? round($total_revenue / $total_assets, 2) : 0;
       $roa = $total_assets > 0 ? round((($total_revenue - $total_expenses) / $total_assets) * 100, 2) : 0;
       
       return array(
           'as_of_date' => $as_of_date,
           'liquidity_ratios' => array(
               'current_ratio' => $current_ratio,
               'interpretation' => $current_ratio >= 2 ? 'Excellent' : ($current_ratio >= 1.5 ? 'Good' : ($current_ratio >= 1 ? 'Adequate' : 'Poor')),
               'quick_ratio' => $current_ratio * 0.8 // Approximation
           ),
           'leverage_ratios' => array(
               'debt_to_asset_ratio' => $debt_to_asset_ratio,
               'interpretation' => $debt_to_asset_ratio <= 0.3 ? 'Conservative' : ($debt_to_asset_ratio <= 0.6 ? 'Moderate' : 'High'),
               'equity_ratio' => 1 - $debt_to_asset_ratio
           ),
           'profitability_ratios' => array(
               'net_profit_margin' => $net_profit_margin . '%',
               'asset_turnover' => $asset_turnover,
               'return_on_assets' => $roa . '%',
               'gross_profit_margin' => $this->calculateGrossProfitMargin($year_start, $as_of_date)
           ),
           'efficiency_ratios' => array(
               'asset_turnover' => $asset_turnover,
               'receivables_turnover' => $this->calculateReceivablesTurnover($year_start, $as_of_date)
           )
       );
   }

   private function getAccountTypeTotal($account_type, $as_of_date, $from_date = null) {
       $this->db->select('SUM(CASE WHEN coa.account_type IN ("asset", "expense") THEN gl.debit_amount - gl.credit_amount ELSE gl.credit_amount - gl.debit_amount END) as total');
       $this->db->from('general_ledger gl');
       $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
       $this->db->where('coa.account_type', $account_type);
       $this->db->where('gl.transaction_date <=', $as_of_date);
       
       if ($from_date) {
           $this->db->where('gl.transaction_date >=', $from_date);
       }
       
       $result = $this->db->get()->row();
       return floatval($result->total ?? 0);
   }

   private function getRevenueTrends($months) {
       if (!$this->db->table_exists('financial_transactions')) {
           throw new Exception('Revenue trends require financial transactions table');
       }
       
       $trends = array();
       
       for ($i = $months - 1; $i >= 0; $i--) {
           $month_start = date('Y-m-01', strtotime("-$i months"));
           $month_end = date('Y-m-t', strtotime("-$i months"));
           
           $this->db->select('department, SUM(amount) as revenue, COUNT(*) as transactions');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $month_start);
           $this->db->where('transaction_date <=', $month_end);
           $this->db->group_by('department');
           
           $month_data = $this->db->get()->result_array();
           
           $total_revenue = array_sum(array_column($month_data, 'revenue'));
           $total_transactions = array_sum(array_column($month_data, 'transactions'));
           
           $trends[] = array(
               'month' => date('Y-m', strtotime($month_start)),
               'month_name' => date('F Y', strtotime($month_start)),
               'total_revenue' => $total_revenue,
               'total_transactions' => $total_transactions,
               'average_transaction' => $total_transactions > 0 ? round($total_revenue / $total_transactions, 2) : 0,
               'department_breakdown' => $month_data
           );
       }
       
       // Calculate growth rates and moving averages
       for ($i = 1; $i < count($trends); $i++) {
           $current = $trends[$i]['total_revenue'];
           $previous = $trends[$i-1]['total_revenue'];
           $growth_rate = $previous > 0 ? round((($current - $previous) / $previous) * 100, 2) : 0;
           $trends[$i]['growth_rate'] = $growth_rate;
           
           // 3-month moving average
           if ($i >= 2) {
               $ma3 = ($trends[$i]['total_revenue'] + $trends[$i-1]['total_revenue'] + $trends[$i-2]['total_revenue']) / 3;
               $trends[$i]['moving_average_3'] = round($ma3, 2);
           }
       }
       
       return array(
           'period' => $months . ' months',
           'trends' => $trends,
           'summary' => array(
               'total_periods' => count($trends),
               'highest_revenue' => max(array_column($trends, 'total_revenue')),
               'lowest_revenue' => min(array_column($trends, 'total_revenue')),
               'average_revenue' => round(array_sum(array_column($trends, 'total_revenue')) / count($trends), 2),
               'average_growth_rate' => $this->calculateAverageGrowthRate($trends),
               'trend_direction' => $this->determineTrendDirection($trends)
           )
       );
   }

   // =============================================
   // EXPORT FUNCTIONALITY
   // =============================================

   public function export_income_statement() {
       $start_date = $this->uri->segment(4) ?: date('Y-04-01');
       $end_date = $this->uri->segment(5) ?: date('Y-03-31', strtotime('+1 year'));
       $format = $this->uri->segment(6) ?: 'csv';
       
       try {
           $income_data = $this->getEnhancedIncomeStatementData($start_date, $end_date);
           
           switch ($format) {
               case 'excel':
                   $this->exportIncomeStatementExcel($income_data, $start_date, $end_date);
                   break;
               case 'pdf':
                   $this->exportIncomeStatementPDF($income_data, $start_date, $end_date);
                   break;
               default:
                   $this->exportIncomeStatementCSV($income_data, $start_date, $end_date);
                   break;
           }
       } catch (Exception $e) {
           show_error('Export failed: ' . $e->getMessage());
       }
   }

   public function export_balance_sheet() {
       $as_of_date = $this->uri->segment(4) ?: date('Y-m-d');
       $format = $this->uri->segment(5) ?: 'csv';
       
       try {
           $balance_sheet_data = $this->getBalanceSheetData($as_of_date);
           $this->exportBalanceSheetCSV($balance_sheet_data, $as_of_date);
       } catch (Exception $e) {
           show_error('Export failed: ' . $e->getMessage());
       }
   }

   private function exportIncomeStatementCSV($data, $start_date, $end_date) {
       $filename = 'enhanced_income_statement_' . $start_date . '_to_' . $end_date . '.csv';
       
       header('Content-Type: text/csv');
       header('Content-Disposition: attachment; filename="' . $filename . '"');
       
       $output = fopen('php://output', 'w');
       
       fputcsv($output, array('Enhanced Income Statement', 'From: ' . $start_date, 'To: ' . $end_date));
       fputcsv($output, array('Generated: ' . date('Y-m-d H:i:s')));
       fputcsv($output, array(''));
       
       // Revenue section by category
       fputcsv($output, array('REVENUE'));
       foreach ($data['revenue'] as $category => $accounts) {
           fputcsv($output, array(strtoupper(str_replace('_', ' ', $category))));
           foreach ($accounts as $account) {
               fputcsv($output, array('  ' . $account['account_name'], number_format($account['amount'], 2), $account['account_code'] ?? ''));
           }
           fputcsv($output, array(''));
       }
       fputcsv($output, array('TOTAL REVENUE', number_format($data['total_revenue'], 2)));
       fputcsv($output, array(''));
       
       // Expenses section by category
       fputcsv($output, array('EXPENSES'));
       foreach ($data['expenses'] as $category => $accounts) {
           fputcsv($output, array(strtoupper(str_replace('_', ' ', $category))));
           foreach ($accounts as $account) {
               fputcsv($output, array('  ' . $account['account_name'], number_format($account['amount'], 2), $account['account_code'] ?? ''));
           }
           fputcsv($output, array(''));
       }
       fputcsv($output, array('TOTAL EXPENSES', number_format($data['total_expenses'], 2)));
       fputcsv($output, array(''));
       
       // Profit calculations
       fputcsv($output, array('PROFITABILITY ANALYSIS'));
       fputcsv($output, array('Gross Profit', number_format($data['gross_profit'], 2)));
       fputcsv($output, array('Operating Profit', number_format($data['operating_profit'], 2)));
       fputcsv($output, array('Net Profit', number_format($data['net_profit'], 2)));
       fputcsv($output, array('Profit Margin (%)', $data['profit_margin'] . '%'));
       fputcsv($output, array('Expense Ratio (%)', $data['expense_ratio'] . '%'));
       fputcsv($output, array(''));
       
       // Department breakdown
       if (!empty($data['department_breakdown'])) {
           fputcsv($output, array('DEPARTMENT BREAKDOWN'));
           fputcsv($output, array('Department', 'Revenue', 'Transactions', 'Unique Patients', 'Avg Transaction'));
           foreach ($data['department_breakdown'] as $dept) {
               fputcsv($output, array(
                   ucwords($dept['department']),
                   number_format($dept['revenue'], 2),
                   $dept['transactions'] ?? 0,
                   $dept['unique_patients'] ?? 0,
                   number_format($dept['avg_transaction'] ?? 0, 2)
               ));
           }
       }
       
       fclose($output);
   }

   private function exportBalanceSheetCSV($data, $as_of_date) {
       $filename = 'balance_sheet_' . $as_of_date . '.csv';
       
       header('Content-Type: text/csv');
       header('Content-Disposition: attachment; filename="' . $filename . '"');
       
       $output = fopen('php://output', 'w');
       
       fputcsv($output, array('Balance Sheet', 'As of: ' . $as_of_date));
       fputcsv($output, array('Generated: ' . date('Y-m-d H:i:s')));
       fputcsv($output, array(''));
       
       // Assets
       fputcsv($output, array('ASSETS'));
       foreach ($data['assets'] as $category => $accounts) {
           fputcsv($output, array(strtoupper(str_replace('_', ' ', $category))));
           foreach ($accounts as $account) {
               fputcsv($output, array('  ' . $account['account_name'], number_format($account['amount'], 2), $account['account_code']));
           }
           fputcsv($output, array(''));
       }
       fputcsv($output, array('TOTAL ASSETS', number_format($data['total_assets'], 2)));
       fputcsv($output, array(''));
       
       // Liabilities
       fputcsv($output, array('LIABILITIES'));
       foreach ($data['liabilities'] as $category => $accounts) {
           fputcsv($output, array(strtoupper(str_replace('_', ' ', $category))));
           foreach ($accounts as $account) {
               fputcsv($output, array('  ' . $account['account_name'], number_format($account['amount'], 2), $account['account_code']));
           }
           fputcsv($output, array(''));
       }
       fputcsv($output, array('TOTAL LIABILITIES', number_format($data['total_liabilities'], 2)));
       fputcsv($output, array(''));
       
       // Equity
       fputcsv($output, array('EQUITY'));
       foreach ($data['equity'] as $category => $accounts) {
           fputcsv($output, array(strtoupper(str_replace('_', ' ', $category))));
           foreach ($accounts as $account) {
               fputcsv($output, array('  ' . $account['account_name'], number_format($account['amount'], 2), $account['account_code']));
           }
           fputcsv($output, array(''));
       }
       fputcsv($output, array('TOTAL EQUITY', number_format($data['total_equity'], 2)));
       fputcsv($output, array(''));
       
       fputcsv($output, array('TOTAL LIABILITIES + EQUITY', number_format($data['total_liabilities'] + $data['total_equity'], 2)));
       fputcsv($output, array('Balance Check', $data['is_balanced'] ? 'BALANCED' : 'NOT BALANCED'));
       
       if (!$data['is_balanced']) {
           fputcsv($output, array('Balance Difference', number_format($data['balance_difference'], 2)));
       }
       
       fclose($output);
   }

   // =============================================
   // DASHBOARD AND SUMMARY METHODS
   // =============================================

   public function dashboard_summary() {
       header('Content-Type: application/json');
       
       try {
           $today = date('Y-m-d');
           $month_start = date('Y-m-01');
           $year_start = date('Y-01-01');
           $yesterday = date('Y-m-d', strtotime('-1 day'));
           
           // Today's performance
           $today_revenue = $this->getDailyRevenue($today);
           $yesterday_revenue = $this->getDailyRevenue($yesterday);
           
           // Month and year performance
           $month_revenue = $this->getPeriodRevenue($month_start, $today);
           $year_revenue = $this->getPeriodRevenue($year_start, $today);
           
           // Key metrics
           $outstanding_receivables = $this->getOutstandingReceivables();
           $top_departments = $this->getTopDepartments($month_start, $today);
           $recent_transactions = $this->getRecentTransactions(10);
           $financial_health = $this->getFinancialHealthIndicators();
           
           // Performance comparisons
           $daily_growth = $yesterday_revenue['revenue'] > 0 ? 
               round((($today_revenue['revenue'] - $yesterday_revenue['revenue']) / $yesterday_revenue['revenue']) * 100, 2) : 0;
           
           echo json_encode(array(
               'status' => 'success',
               'data' => array(
                   'today_performance' => array(
                       'revenue' => $today_revenue['revenue'],
                       'transactions' => $today_revenue['transactions'],
                       'daily_growth' => $daily_growth,
                       'avg_transaction' => $today_revenue['transactions'] > 0 ? 
                           round($today_revenue['revenue'] / $today_revenue['transactions'], 2) : 0
                   ),
                   'month_performance' => $month_revenue,
                   'year_performance' => $year_revenue,
                   'outstanding_receivables' => $outstanding_receivables,
                   'top_departments' => $top_departments,
                   'recent_transactions' => $recent_transactions,
                   'financial_health' => $financial_health,
                   'last_updated' => date('Y-m-d H:i:s'),
                   'alerts' => $this->getFinancialAlerts()
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   private function getDailyRevenue($date) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('SUM(amount) as revenue, COUNT(*) as transactions, COUNT(DISTINCT patient_id) as patients');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('DATE(transaction_date)', $date);
           
           $result = $this->db->get()->row();
           return array(
               'revenue' => floatval($result->revenue ?? 0),
               'transactions' => intval($result->transactions ?? 0),
               'patients' => intval($result->patients ?? 0)
           );
       } else {
           // Fallback to basic tables
           return $this->getBasicDailyRevenue($date);
       }
   }

   private function getBasicDailyRevenue($date) {
       $revenue = 0;
       $transactions = 0;
       
       // Check each revenue source
       $sources = array(
           'opd_details' => array('date_field' => 'appointment_date', 'amount_field' => 'amount'),
           'ipd_details' => array('date_field' => 'date', 'amount_field' => 'amount'),
           'pharmacy_bill_basic' => array('date_field' => 'date', 'amount_field' => 'net_amount'),
           'pathology_billing' => array('date_field' => 'date', 'amount_field' => 'amount')
       );
       
       foreach ($sources as $table => $config) {
           if ($this->db->table_exists($table)) {
               $this->db->select("SUM({$config['amount_field']}) as revenue, COUNT(*) as count");
               $this->db->from($table);
               $this->db->where("DATE({$config['date_field']})", $date);
               $query = $this->db->get();
               
               if ($query && $query->num_rows() > 0) {
                   $row = $query->row();
                   $revenue += floatval($row->revenue ?? 0);
                   $transactions += intval($row->count ?? 0);
               }
           }
       }
       
       return array(
           'revenue' => $revenue,
           'transactions' => $transactions,
           'patients' => $transactions // Approximation
       );
   }

   private function getPeriodRevenue($start_date, $end_date) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('SUM(amount) as revenue, COUNT(*) as transactions, COUNT(DISTINCT patient_id) as patients');
           $this->db->select('AVG(amount) as avg_transaction');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           
           $result = $this->db->get()->row();
           return array(
               'revenue' => floatval($result->revenue ?? 0),
               'transactions' => intval($result->transactions ?? 0),
               'patients' => intval($result->patients ?? 0),
               'avg_transaction' => floatval($result->avg_transaction ?? 0)
           );
       } else {
           return $this->getBasicPeriodRevenue($start_date, $end_date);
       }
   }

   private function getBasicPeriodRevenue($start_date, $end_date) {
       $revenue = 0;
       $transactions = 0;
       
       $sources = array(
           'opd_details' => array('date_field' => 'appointment_date', 'amount_field' => 'amount'),
           'ipd_details' => array('date_field' => 'date', 'amount_field' => 'amount'),
           'pharmacy_bill_basic' => array('date_field' => 'date', 'amount_field' => 'net_amount'),
           'pathology_billing' => array('date_field' => 'date', 'amount_field' => 'amount')
       );
       
       foreach ($sources as $table => $config) {
           if ($this->db->table_exists($table)) {
               $this->db->select("SUM({$config['amount_field']}) as revenue, COUNT(*) as count");
               $this->db->from($table);
               $this->db->where("{$config['date_field']} >=", $start_date);
               $this->db->where("{$config['date_field']} <=", $end_date);
               $query = $this->db->get();
               
               if ($query && $query->num_rows() > 0) {
                   $row = $query->row();
                   $revenue += floatval($row->revenue ?? 0);
                   $transactions += intval($row->count ?? 0);
               }
           }
       }
       
       return array(
           'revenue' => $revenue,
           'transactions' => $transactions,
           'patients' => $transactions,
           'avg_transaction' => $transactions > 0 ? round($revenue / $transactions, 2) : 0
       );
   }

   private function getOutstandingReceivables() {
       if ($this->db->table_exists('general_ledger') && $this->db->table_exists('chart_of_accounts')) {
           $this->db->select('SUM(gl.debit_amount - gl.credit_amount) as receivables');
           $this->db->from('general_ledger gl');
           $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
           $this->db->where('coa.account_code', '1200'); // Accounts Receivable
           
           $result = $this->db->get()->row();
           return floatval($result->receivables ?? 0);
       }
       
       return 0;
   }

   private function getTopDepartments($start_date, $end_date, $limit = 5) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('department, SUM(amount) as revenue, COUNT(*) as transactions');
           $this->db->select('AVG(amount) as avg_transaction, COUNT(DISTINCT patient_id) as patients');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           $this->db->group_by('department');
           $this->db->order_by('revenue', 'DESC');
           $this->db->limit($limit);
           
           return $this->db->get()->result_array();
       }
       
       return array();
   }

   private function getRecentTransactions($limit = 10) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('ft.*, p.patient_name');
           $this->db->from('financial_transactions ft');
           $this->db->join('patients p', 'ft.patient_id = p.id', 'left');
           $this->db->where('ft.transaction_type', 'revenue');
           $this->db->order_by('ft.transaction_date', 'DESC');
           $this->db->order_by('ft.id', 'DESC');
           $this->db->limit($limit);
           
           $transactions = $this->db->get()->result_array();
           
           // Format for display
           foreach ($transactions as &$transaction) {
               $transaction['formatted_date'] = date('M d, Y H:i', strtotime($transaction['transaction_date']));
               $transaction['formatted_amount'] = number_format($transaction['amount'], 2);
           }
           
           return $transactions;
       }
       
       return array();
   }

   private function getFinancialHealthIndicators() {
       try {
           $today = date('Y-m-d');
           $month_start = date('Y-m-01');
           
           // Current month vs last month
           $current_month = $this->getPeriodRevenue($month_start, $today);
           $last_month_start = date('Y-m-01', strtotime('-1 month'));
           $last_month_end = date('Y-m-t', strtotime('-1 month'));
           $last_month = $this->getPeriodRevenue($last_month_start, $last_month_end);
           
           $monthly_growth = $last_month['revenue'] > 0 ? 
               round((($current_month['revenue'] - $last_month['revenue']) / $last_month['revenue']) * 100, 2) : 0;
           
           // Revenue consistency (coefficient of variation)
           $daily_revenues = $this->getDailyRevenuesForMonth($month_start, $today);
           $revenue_consistency = $this->calculateRevenueConsistency($daily_revenues);
           
           return array(
               'monthly_growth' => $monthly_growth,
               'revenue_consistency' => $revenue_consistency,
               'avg_daily_revenue' => count($daily_revenues) > 0 ? round(array_sum($daily_revenues) / count($daily_revenues), 2) : 0,
               'best_day_revenue' => count($daily_revenues) > 0 ? max($daily_revenues) : 0,
               'worst_day_revenue' => count($daily_revenues) > 0 ? min($daily_revenues) : 0,
               'working_days' => count($daily_revenues)
           );
       } catch (Exception $e) {
           log_message('error', 'Financial health indicators error: ' . $e->getMessage());
           return array();
       }
   }

   private function getDailyRevenuesForMonth($start_date, $end_date) {
       $revenues = array();
       $current_date = $start_date;
       
       while (strtotime($current_date) <= strtotime($end_date)) {
           $daily_revenue = $this->getDailyRevenue($current_date);
           if ($daily_revenue['revenue'] > 0) {
               $revenues[] = $daily_revenue['revenue'];
           }
           $current_date = date('Y-m-d', strtotime($current_date . ' +1 day'));
       }
       
       return $revenues;
   }

   private function calculateRevenueConsistency($revenues) {
       if (count($revenues) < 2) return 0;
       
       $mean = array_sum($revenues) / count($revenues);
       $variance = 0;
       
       foreach ($revenues as $revenue) {
           $variance += pow($revenue - $mean, 2);
       }
       
       $variance /= count($revenues);
       $std_dev = sqrt($variance);
       
       $coefficient_variation = $mean > 0 ? ($std_dev / $mean) * 100 : 0;
       
       // Convert to consistency score (inverse of CV)
       return round(max(0, 100 - $coefficient_variation), 2);
   }

   private function getFinancialAlerts() {
       $alerts = array();
       
       try {
           // Check for low daily revenue
           $today_revenue = $this->getDailyRevenue(date('Y-m-d'));
           $avg_daily = $this->getAverageDailyRevenue(30);
           
           if ($today_revenue['revenue'] < ($avg_daily * 0.5)) {
               $alerts[] = array(
                   'type' => 'warning',
                   'message' => 'Today\'s revenue is significantly below average',
                   'value' => number_format($today_revenue['revenue'], 2)
               );
           }
           
           // Check receivables
           $receivables = $this->getOutstandingReceivables();
           if ($receivables > 100000) { // Threshold can be configurable
               $alerts[] = array(
                   'type' => 'info',
                   'message' => 'High outstanding receivables need attention',
                   'value' => number_format($receivables, 2)
               );
           }
           
           // Check for unusual department performance
           $dept_performance = $this->checkDepartmentPerformance();
           if (!empty($dept_performance)) {
               $alerts = array_merge($alerts, $dept_performance);
           }
           
       } catch (Exception $e) {
           log_message('error', 'Financial alerts error: ' . $e->getMessage());
       }
       
       return $alerts;
   }

   private function getAverageDailyRevenue($days = 30) {
       $end_date = date('Y-m-d');
       $start_date = date('Y-m-d', strtotime("-$days days"));
       
       $period_revenue = $this->getPeriodRevenue($start_date, $end_date);
       return $days > 0 ? round($period_revenue['revenue'] / $days, 2) : 0;
   }

   private function checkDepartmentPerformance() {
       $alerts = array();
       
       try {
           $current_week = $this->getTopDepartments(date('Y-m-d', strtotime('-7 days')), date('Y-m-d'));
           $previous_week = $this->getTopDepartments(date('Y-m-d', strtotime('-14 days')), date('Y-m-d', strtotime('-7 days')));
           
           foreach ($current_week as $current_dept) {
               $previous_dept = array_filter($previous_week, function($d) use ($current_dept) {
                   return $d['department'] === $current_dept['department'];
               });
               
               if (!empty($previous_dept)) {
                   $previous_dept = reset($previous_dept);
                   $growth = $previous_dept['revenue'] > 0 ? 
                       (($current_dept['revenue'] - $previous_dept['revenue']) / $previous_dept['revenue']) * 100 : 0;
                   
                   if ($growth < -30) { // 30% decline
                       $alerts[] = array(
                           'type' => 'warning',
                           'message' => ucwords($current_dept['department']) . ' department revenue declined significantly',
                           'value' => round($growth, 1) . '%'
                       );
                   }
               }
           }
       } catch (Exception $e) {
           log_message('error', 'Department performance check error: ' . $e->getMessage());
       }
       
       return $alerts;
   }

   // =============================================
   // UTILITY METHODS
   // =============================================

   private function isValidDate($date) {
       $d = DateTime::createFromFormat('Y-m-d', $date);
       return $d && $d->format('Y-m-d') === $date;
   }

   private function getFinancialYear() {
       $current_year = date('Y');
       
       if (date('m') >= 4) {
           $start_year = $current_year;
           $end_year = $current_year + 1;
       } else {
           $start_year = $current_year - 1;
           $end_year = $current_year;
       }
       
       return array(
           'start' => $start_year . '-04-01',
           'end' => $end_year . '-03-31'
       );
   }

   private function calculateAverageGrowthRate($trends) {
       $growth_rates = array();
       
       foreach ($trends as $trend) {
           if (isset($trend['growth_rate'])) {
               $growth_rates[] = $trend['growth_rate'];
           }
       }
       
       return count($growth_rates) > 0 ? round(array_sum($growth_rates) / count($growth_rates), 2) : 0;
   }

   private function determineTrendDirection($trends) {
       if (count($trends) < 3) return 'insufficient_data';
       
       $recent_trends = array_slice($trends, -3);
       $increasing = 0;
       $decreasing = 0;
       
       for ($i = 1; $i < count($recent_trends); $i++) {
           if ($recent_trends[$i]['total_revenue'] > $recent_trends[$i-1]['total_revenue']) {
               $increasing++;
           } elseif ($recent_trends[$i]['total_revenue'] < $recent_trends[$i-1]['total_revenue']) {
               $decreasing++;
           }
       }
       
       if ($increasing > $decreasing) return 'upward';
       if ($decreasing > $increasing) return 'downward';
       return 'stable';
   }

   private function calculateGrossProfitMargin($start_date, $end_date) {
       if (!$this->db->table_exists('general_ledger')) return '0%';
       
       try {
           // Get revenue
           $revenue = $this->getAccountTypeTotal('revenue', $end_date, $start_date);
           
           // Get cost of goods sold (direct costs)
           $this->db->select('SUM(gl.debit_amount - gl.credit_amount) as cogs');
           $this->db->from('general_ledger gl');
           $this->db->join('chart_of_accounts coa', 'gl.account_id = coa.id');
           $this->db->where('coa.account_category', 'direct_costs');
           $this->db->where('gl.transaction_date >=', $start_date);
           $this->db->where('gl.transaction_date <=', $end_date);
           
           $cogs = $this->db->get()->row()->cogs ?? 0;
           
           $gross_profit = $revenue - $cogs;
           $gross_margin = $revenue > 0 ? round(($gross_profit / $revenue) * 100, 2) : 0;
           
           return $gross_margin . '%';
       } catch (Exception $e) {
           return '0%';
       }
   }

   private function calculateReceivablesTurnover($start_date, $end_date) {
       if (!$this->db->table_exists('general_ledger')) return 0;
       
       try {
           $revenue = $this->getAccountTypeTotal('revenue', $end_date, $start_date);
           $receivables = $this->getOutstandingReceivables();
           
           return $receivables > 0 ? round($revenue / $receivables, 2) : 0;
       } catch (Exception $e) {
           return 0;
       }
   }

   // Legacy fallback methods for basic trial balance
   private function getBasicTrialBalanceData($as_of_date) {
       return array(
           array(
               'account_code' => '1000',
               'account_name' => 'Cash and Bank',
               'account_type' => 'asset',
               'account_category' => 'current_assets',
               'debit_balance' => 0,
               'credit_balance' => 0
           )
       );
   }

   // Basic cash flow methods for fallback
   private function getBasicCashInflows($start_date, $end_date) {
       return $this->getBasicRevenueTotal($start_date, $end_date);
   }

   private function getBasicCashOutflows($start_date, $end_date) {
       // In basic mode, we don't have expense tracking, so return 0
       return 0;
   }

   private function getBasicPaymentMethods($start_date, $end_date) {
       return array(
           array('payment_method' => 'cash', 'amount' => $this->getBasicRevenueTotal($start_date, $end_date)),
       );
   }

   // Legacy daily income method for fallback
private function getDailyIncomeData($date) {
    $data = array(
        'opd_count' => 0,
        'opd_income' => 0,
        'ipd_count' => 0,
        'ipd_income' => 0,
        'pharmacy_count' => 0,
        'pharmacy_income' => 0,
        'lab_count' => 0,
        'lab_income' => 0,
        'ambulance_count' => 0,
        'ambulance_income' => 0,
        'other_count' => 0,
        'other_income' => 0
    );
    
    try {
        // Get OPD data for the day
        if ($this->db->table_exists('opd_details')) {
            $this->db->select('COUNT(*) as count, COALESCE(SUM(amount), 0) as income');
            $this->db->from('opd_details');
            $this->db->where('DATE(appointment_date)', $date);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $row = $query->row();
                $data['opd_count'] = intval($row->count ?? 0);
                $data['opd_income'] = floatval($row->income ?? 0);
            }
        }
        
        // Get IPD data for the day
        if ($this->db->table_exists('ipd_details')) {
            $this->db->select('COUNT(*) as count, COALESCE(SUM(amount), 0) as income');
            $this->db->from('ipd_details');
            $this->db->where('DATE(date)', $date);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $row = $query->row();
                $data['ipd_count'] = intval($row->count ?? 0);
                $data['ipd_income'] = floatval($row->income ?? 0);
            }
        }
        
        // Get Pharmacy data for the day
        if ($this->db->table_exists('pharmacy_bill_basic')) {
            $this->db->select('COUNT(*) as count, COALESCE(SUM(net_amount), 0) as income');
            $this->db->from('pharmacy_bill_basic');
            $this->db->where('DATE(date)', $date);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $row = $query->row();
                $data['pharmacy_count'] = intval($row->count ?? 0);
                $data['pharmacy_income'] = floatval($row->income ?? 0);
            }
        }
        
        // Get Laboratory data for the day
        if ($this->db->table_exists('pathology_billing')) {
            $this->db->select('COUNT(*) as count, COALESCE(SUM(amount), 0) as income');
            $this->db->from('pathology_billing');
            $this->db->where('DATE(date)', $date);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $row = $query->row();
                $data['lab_count'] = intval($row->count ?? 0);
                $data['lab_income'] = floatval($row->income ?? 0);
            }
        }
        
        // Get Ambulance data for the day
        if ($this->db->table_exists('ambulance_call')) {
            $this->db->select('COUNT(*) as count, COALESCE(SUM(amount), 0) as income');
            $this->db->from('ambulance_call');
            $this->db->where('DATE(date)', $date);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $row = $query->row();
                $data['ambulance_count'] = intval($row->count ?? 0);
                $data['ambulance_income'] = floatval($row->income ?? 0);
            }
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error getting daily income data: ' . $e->getMessage());
    }
    
    $total_income = $data['opd_income'] + $data['ipd_income'] + $data['pharmacy_income'] + 
                   $data['lab_income'] + $data['ambulance_income'] + $data['other_income'];
    $total_transactions = $data['opd_count'] + $data['ipd_count'] + $data['pharmacy_count'] + 
                         $data['lab_count'] + $data['ambulance_count'] + $data['other_count'];
    
    // Return in the expected format
    return array(
        'date' => $date,
        'income_breakdown' => array(
            'opd_consultations' => array(
                'count' => $data['opd_count'], 
                'amount' => $data['opd_income']
            ),
            'ipd_services' => array(
                'count' => $data['ipd_count'], 
                'amount' => $data['ipd_income']
            ),
            'pharmacy_sales' => array(
                'count' => $data['pharmacy_count'], 
                'amount' => $data['pharmacy_income']
            ),
            'laboratory_tests' => array(
                'count' => $data['lab_count'], 
                'amount' => $data['lab_income']
            ),
            'ambulance_services' => array(
                'count' => $data['ambulance_count'], 
                'amount' => $data['ambulance_income']
            ),
            'other_services' => array(
                'count' => $data['other_count'], 
                'amount' => $data['other_income']
            )
        ),
        'total_income' => $total_income,
        'total_transactions' => $total_transactions,
        'unique_patients' => $total_transactions, // Approximation for legacy mode
        'average_transaction' => $total_transactions > 0 ? round($total_income / $total_transactions, 2) : 0,
        'average_per_patient' => $total_transactions > 0 ? round($total_income / $total_transactions, 2) : 0,
        'payment_methods' => array(
            'cash' => $total_income,
            'card' => 0,
            'cheque' => 0,
            'bank_transfer' => 0
        ),
        'hourly_breakdown' => $this->getBasicHourlyBreakdown()
    );
}

private function getBasicHourlyBreakdown() {
    $hourly_data = array();
    for ($i = 0; $i < 24; $i++) {
        $hour = sprintf('%02d:00', $i);
        $hourly_data[$hour] = array('amount' => 0, 'count' => 0);
    }
    return $hourly_data;
}

   // Database setup verification
   public function verify_setup() {
       header('Content-Type: application/json');
       
       $checks = array(
           'chart_of_accounts' => $this->db->table_exists('chart_of_accounts'),
           'general_ledger' => $this->db->table_exists('general_ledger'),
           'financial_transactions' => $this->db->table_exists('financial_transactions'),
           'enhanced_structure' => false
       );
       
       $checks['enhanced_structure'] = $checks['chart_of_accounts'] && $checks['general_ledger'];
       
       $suggestions = array();
       if (!$checks['enhanced_structure']) {
           $suggestions[] = 'Consider implementing the enhanced financial structure with Chart of Accounts and General Ledger for better reporting';
       }
       
       if (!$checks['financial_transactions']) {
           $suggestions[] = 'Financial transactions table would provide better analytics and reporting capabilities';
       }
       
       echo json_encode(array(
           'status' => 'success',
           'checks' => $checks,
           'suggestions' => $suggestions,
           'current_capabilities' => array(
               'basic_reports' => true,
               'enhanced_reports' => $checks['enhanced_structure'],
               'advanced_analytics' => $checks['financial_transactions'],
               'real_time_dashboard' => $checks['financial_transactions']
           )
		   ));
   }

   // =============================================
   // ADVANCED REPORTING METHODS
   // =============================================

   public function profit_loss_comparison($start_date = null, $end_date = null, $compare_period = 'previous_year') {
       header('Content-Type: application/json');
       
       try {
           if (!$start_date) $start_date = date('Y-04-01');
           if (!$end_date) $end_date = date('Y-03-31', strtotime('+1 year'));
           
           // Get current period data
           $current_data = $this->getEnhancedIncomeStatementData($start_date, $end_date);
           
           // Calculate comparison period dates
           switch ($compare_period) {
               case 'previous_year':
                   $compare_start = date('Y-m-d', strtotime($start_date . ' -1 year'));
                   $compare_end = date('Y-m-d', strtotime($end_date . ' -1 year'));
                   break;
               case 'previous_quarter':
                   $compare_start = date('Y-m-d', strtotime($start_date . ' -3 months'));
                   $compare_end = date('Y-m-d', strtotime($end_date . ' -3 months'));
                   break;
               default:
                   $compare_start = date('Y-m-d', strtotime($start_date . ' -1 year'));
                   $compare_end = date('Y-m-d', strtotime($end_date . ' -1 year'));
           }
           
           // Get comparison period data
           $comparison_data = $this->getEnhancedIncomeStatementData($compare_start, $compare_end);
           
           // Calculate variances
           $variance_analysis = $this->calculateVarianceAnalysis($current_data, $comparison_data);
           
           echo json_encode(array(
               'status' => 'success',
               'data' => array(
                   'current_period' => array(
                       'start_date' => $start_date,
                       'end_date' => $end_date,
                       'data' => $current_data
                   ),
                   'comparison_period' => array(
                       'start_date' => $compare_start,
                       'end_date' => $compare_end,
                       'data' => $comparison_data
                   ),
                   'variance_analysis' => $variance_analysis,
                   'comparison_type' => $compare_period
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   public function monthly_trends_analysis($months = 12) {
       header('Content-Type: application/json');
       
       try {
           $monthly_data = array();
           
           for ($i = $months - 1; $i >= 0; $i--) {
               $month_start = date('Y-m-01', strtotime("-$i months"));
               $month_end = date('Y-m-t', strtotime("-$i months"));
               
               $month_revenue = $this->getPeriodRevenue($month_start, $month_end);
               $month_departments = $this->getTopDepartments($month_start, $month_end, 10);
               
               $monthly_data[] = array(
                   'month' => date('Y-m', strtotime($month_start)),
                   'month_name' => date('F Y', strtotime($month_start)),
                   'revenue' => $month_revenue['revenue'],
                   'transactions' => $month_revenue['transactions'],
                   'patients' => $month_revenue['patients'],
                   'avg_transaction' => $month_revenue['avg_transaction'],
                   'departments' => $month_departments,
                   'working_days' => $this->getWorkingDaysInMonth($month_start),
                   'revenue_per_day' => $this->getWorkingDaysInMonth($month_start) > 0 ? 
                       round($month_revenue['revenue'] / $this->getWorkingDaysInMonth($month_start), 2) : 0
               );
           }
           
           // Calculate trends and patterns
           $trend_analysis = $this->analyzeTrends($monthly_data);
           $seasonal_patterns = $this->identifySeasonalPatterns($monthly_data);
           $forecasting = $this->generateBasicForecast($monthly_data, 3);
           
           echo json_encode(array(
               'status' => 'success',
               'data' => array(
                   'monthly_data' => $monthly_data,
                   'trend_analysis' => $trend_analysis,
                   'seasonal_patterns' => $seasonal_patterns,
                   'forecasting' => $forecasting,
                   'summary' => array(
                       'total_months' => count($monthly_data),
                       'avg_monthly_revenue' => round(array_sum(array_column($monthly_data, 'revenue')) / count($monthly_data), 2),
                       'best_month' => $this->findBestMonth($monthly_data),
                       'worst_month' => $this->findWorstMonth($monthly_data)
                   )
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   public function patient_financial_analysis($patient_id = null) {
       header('Content-Type: application/json');
       
       try {
           if ($patient_id) {
               // Single patient analysis
               $patient_data = $this->getSinglePatientFinancialData($patient_id);
               echo json_encode(array('status' => 'success', 'data' => $patient_data));
           } else {
               // Top patients analysis
               $top_patients = $this->getTopPatientsByRevenue(50);
               $patient_segments = $this->analyzePatientSegments();
               $retention_analysis = $this->analyzePatientRetention();
               
               echo json_encode(array(
                   'status' => 'success',
                   'data' => array(
                       'top_patients' => $top_patients,
                       'patient_segments' => $patient_segments,
                       'retention_analysis' => $retention_analysis
                   )
               ));
           }
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   // =============================================
   // ANALYSIS HELPER METHODS
   // =============================================

   private function calculateVarianceAnalysis($current, $comparison) {
       $variance = array(
           'revenue_variance' => array(
               'amount' => $current['total_revenue'] - $comparison['total_revenue'],
               'percentage' => $comparison['total_revenue'] > 0 ? 
                   round((($current['total_revenue'] - $comparison['total_revenue']) / $comparison['total_revenue']) * 100, 2) : 0
           ),
           'expense_variance' => array(
               'amount' => $current['total_expenses'] - $comparison['total_expenses'],
               'percentage' => $comparison['total_expenses'] > 0 ? 
                   round((($current['total_expenses'] - $comparison['total_expenses']) / $comparison['total_expenses']) * 100, 2) : 0
           ),
           'profit_variance' => array(
               'amount' => $current['net_profit'] - $comparison['net_profit'],
               'percentage' => $comparison['net_profit'] != 0 ? 
                   round((($current['net_profit'] - $comparison['net_profit']) / abs($comparison['net_profit'])) * 100, 2) : 0
           )
       );
       
       // Department-wise variance
       $department_variance = array();
       if (isset($current['department_breakdown']) && isset($comparison['department_breakdown'])) {
           foreach ($current['department_breakdown'] as $curr_dept) {
               $comp_dept = array_filter($comparison['department_breakdown'], function($d) use ($curr_dept) {
                   return $d['department'] === $curr_dept['department'];
               });
               
               if (!empty($comp_dept)) {
                   $comp_dept = reset($comp_dept);
                   $dept_variance = $comp_dept['revenue'] > 0 ? 
                       round((($curr_dept['revenue'] - $comp_dept['revenue']) / $comp_dept['revenue']) * 100, 2) : 0;
                   
                   $department_variance[] = array(
                       'department' => $curr_dept['department'],
                       'current_revenue' => $curr_dept['revenue'],
                       'comparison_revenue' => $comp_dept['revenue'],
                       'variance_amount' => $curr_dept['revenue'] - $comp_dept['revenue'],
                       'variance_percentage' => $dept_variance
                   );
               }
           }
       }
       
       $variance['department_variance'] = $department_variance;
       
       return $variance;
   }

   private function analyzeTrends($monthly_data) {
       if (count($monthly_data) < 3) {
           return array('trend' => 'insufficient_data');
       }
       
       $revenues = array_column($monthly_data, 'revenue');
       $trend_direction = $this->calculateTrendDirection($revenues);
       $volatility = $this->calculateVolatility($revenues);
       $growth_rates = $this->calculateMonthlyGrowthRates($monthly_data);
       
       return array(
           'trend_direction' => $trend_direction,
           'volatility' => $volatility,
           'avg_growth_rate' => round(array_sum($growth_rates) / count($growth_rates), 2),
           'growth_consistency' => $this->calculateGrowthConsistency($growth_rates),
           'strongest_quarter' => $this->findStrongestQuarter($monthly_data),
           'weakest_quarter' => $this->findWeakestQuarter($monthly_data)
       );
   }

   private function identifySeasonalPatterns($monthly_data) {
       $patterns = array();
       
       // Group by month number to identify seasonal trends
       $monthly_averages = array();
       for ($month = 1; $month <= 12; $month++) {
           $month_revenues = array();
           foreach ($monthly_data as $data) {
               if (date('n', strtotime($data['month'] . '-01')) == $month) {
                   $month_revenues[] = $data['revenue'];
               }
           }
           
           if (!empty($month_revenues)) {
               $monthly_averages[$month] = array(
                   'month_name' => date('F', mktime(0, 0, 0, $month, 1)),
                   'avg_revenue' => round(array_sum($month_revenues) / count($month_revenues), 2),
                   'data_points' => count($month_revenues)
               );
           }
       }
       
       // Find peak and low seasons
       if (!empty($monthly_averages)) {
           $peak_month = array_reduce($monthly_averages, function($carry, $item) {
               return ($carry === null || $item['avg_revenue'] > $carry['avg_revenue']) ? $item : $carry;
           });
           
           $low_month = array_reduce($monthly_averages, function($carry, $item) {
               return ($carry === null || $item['avg_revenue'] < $carry['avg_revenue']) ? $item : $carry;
           });
           
           $patterns = array(
               'monthly_averages' => $monthly_averages,
               'peak_season' => $peak_month,
               'low_season' => $low_month,
               'seasonality_strength' => $this->calculateSeasonalityStrength($monthly_averages)
           );
       }
       
       return $patterns;
   }

   private function generateBasicForecast($monthly_data, $forecast_months = 3) {
       if (count($monthly_data) < 6) {
           return array('forecast' => 'insufficient_data');
       }
       
       // Simple linear trend forecasting
       $revenues = array_column($monthly_data, 'revenue');
       $recent_revenues = array_slice($revenues, -6); // Use last 6 months for trend
       
       // Calculate trend
       $x_values = range(1, count($recent_revenues));
       $trend_slope = $this->calculateLinearTrendSlope($x_values, $recent_revenues);
       $trend_intercept = array_sum($recent_revenues) / count($recent_revenues) - $trend_slope * (count($recent_revenues) + 1) / 2;
       
       // Generate forecasts
       $forecasts = array();
       $last_month = end($monthly_data)['month'];
       
       for ($i = 1; $i <= $forecast_months; $i++) {
           $forecast_month = date('Y-m', strtotime($last_month . '-01 +' . $i . ' months'));
           $forecast_value = $trend_intercept + $trend_slope * (count($recent_revenues) + $i);
           
           // Apply seasonal adjustment if pattern exists
           $seasonal_factor = $this->getSeasonalFactor($forecast_month, $monthly_data);
           $adjusted_forecast = max(0, $forecast_value * $seasonal_factor);
           
           $forecasts[] = array(
               'month' => $forecast_month,
               'month_name' => date('F Y', strtotime($forecast_month . '-01')),
               'forecast_revenue' => round($adjusted_forecast, 2),
               'confidence' => $this->calculateForecastConfidence($i, $revenues)
           );
       }
       
       return array(
           'method' => 'linear_trend_with_seasonal_adjustment',
           'forecasts' => $forecasts,
           'trend_slope' => round($trend_slope, 2),
           'data_quality' => $this->assessDataQuality($monthly_data)
       );
   }

   private function getSinglePatientFinancialData($patient_id) {
       $patient_data = array(
           'patient_id' => $patient_id,
           'total_revenue' => 0,
           'visit_count' => 0,
           'avg_visit_value' => 0,
           'department_breakdown' => array(),
           'visit_history' => array(),
           'payment_history' => array()
       );
       
       if ($this->db->table_exists('financial_transactions')) {
           // Get patient financial summary
           $this->db->select('SUM(amount) as total_revenue, COUNT(*) as visit_count, AVG(amount) as avg_visit_value');
           $this->db->select('MIN(transaction_date) as first_visit, MAX(transaction_date) as last_visit');
           $this->db->from('financial_transactions');
           $this->db->where('patient_id', $patient_id);
           $this->db->where('transaction_type', 'revenue');
           
           $summary = $this->db->get()->row();
           if ($summary) {
               $patient_data['total_revenue'] = floatval($summary->total_revenue ?? 0);
               $patient_data['visit_count'] = intval($summary->visit_count ?? 0);
               $patient_data['avg_visit_value'] = floatval($summary->avg_visit_value ?? 0);
               $patient_data['first_visit'] = $summary->first_visit;
               $patient_data['last_visit'] = $summary->last_visit;
           }
           
           // Get department breakdown
           $this->db->select('department, SUM(amount) as revenue, COUNT(*) as visits');
           $this->db->from('financial_transactions');
           $this->db->where('patient_id', $patient_id);
           $this->db->where('transaction_type', 'revenue');
           $this->db->group_by('department');
           $this->db->order_by('revenue', 'DESC');
           
           $patient_data['department_breakdown'] = $this->db->get()->result_array();
           
           // Get recent visit history
           $this->db->select('transaction_date, department, category, amount, payment_method');
           $this->db->from('financial_transactions');
           $this->db->where('patient_id', $patient_id);
           $this->db->where('transaction_type', 'revenue');
           $this->db->order_by('transaction_date', 'DESC');
           $this->db->limit(20);
           
           $patient_data['visit_history'] = $this->db->get()->result_array();
       }
       
       // Get patient details
       if ($this->db->table_exists('patients')) {
           $this->db->select('patient_name, mobileno, age, gender');
           $this->db->from('patients');
           $this->db->where('id', $patient_id);
           $patient_info = $this->db->get()->row();
           
           if ($patient_info) {
               $patient_data['patient_info'] = array(
                   'name' => $patient_info->patient_name ?? 'Unknown',
                   'phone' => $patient_info->mobileno ?? '',
                   'age' => $patient_info->age ?? '',
                   'gender' => $patient_info->gender ?? ''
               );
           }
       }
       
       return $patient_data;
   }

   private function getTopPatientsByRevenue($limit = 50) {
       if (!$this->db->table_exists('financial_transactions')) {
           return array();
       }
       
       $this->db->select('ft.patient_id, p.patient_name, p.mobileno');
       $this->db->select('SUM(ft.amount) as total_revenue, COUNT(ft.id) as visit_count');
       $this->db->select('AVG(ft.amount) as avg_visit_value, MAX(ft.transaction_date) as last_visit');
       $this->db->from('financial_transactions ft');
       $this->db->join('patients p', 'ft.patient_id = p.id', 'left');
       $this->db->where('ft.transaction_type', 'revenue');
       $this->db->group_by('ft.patient_id, p.patient_name, p.mobileno');
       $this->db->order_by('total_revenue', 'DESC');
       $this->db->limit($limit);
       
       $patients = $this->db->get()->result_array();
       
       // Calculate patient lifetime value and segment
       foreach ($patients as &$patient) {
           $patient['total_revenue'] = floatval($patient['total_revenue']);
           $patient['visit_count'] = intval($patient['visit_count']);
           $patient['avg_visit_value'] = floatval($patient['avg_visit_value']);
           $patient['patient_segment'] = $this->classifyPatientSegment($patient);
           $patient['days_since_last_visit'] = $this->calculateDaysSinceLastVisit($patient['last_visit']);
       }
       
       return $patients;
   }

   private function analyzePatientSegments() {
       if (!$this->db->table_exists('financial_transactions')) {
           return array();
       }
       
       // Define segments based on revenue and frequency
       $segments = array(
           'vip' => array('min_revenue' => 50000, 'min_visits' => 10),
           'regular' => array('min_revenue' => 10000, 'min_visits' => 5),
           'occasional' => array('min_revenue' => 1000, 'min_visits' => 2),
           'one_time' => array('min_revenue' => 0, 'min_visits' => 1)
       );
       
       $segment_analysis = array();
       
       foreach ($segments as $segment_name => $criteria) {
           $this->db->select('COUNT(DISTINCT patient_id) as patient_count, SUM(amount) as total_revenue');
           $this->db->select('AVG(amount) as avg_transaction');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           
           // This is a simplified segmentation - in practice, you'd need a subquery
           $result = $this->db->get()->row();
           
           $segment_analysis[$segment_name] = array(
               'criteria' => $criteria,
               'patient_count' => 0, // Would need complex query to get accurate count
               'total_revenue' => 0,
               'avg_revenue_per_patient' => 0
           );
       }
       
       return $segment_analysis;
   }

   private function analyzePatientRetention() {
       if (!$this->db->table_exists('financial_transactions')) {
           return array();
       }
       
       // Calculate retention metrics
       $retention_data = array();
       
       // Patients who visited in each of the last 12 months
       for ($i = 0; $i < 12; $i++) {
           $month_start = date('Y-m-01', strtotime("-$i months"));
           $month_end = date('Y-m-t', strtotime("-$i months"));
           
           $this->db->select('COUNT(DISTINCT patient_id) as unique_patients');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $month_start);
           $this->db->where('transaction_date <=', $month_end);
           
           $month_patients = $this->db->get()->row()->unique_patients ?? 0;
           
           $retention_data[] = array(
               'month' => date('Y-m', strtotime($month_start)),
               'unique_patients' => $month_patients
           );
       }
       
       // Calculate retention rates
       $retention_rates = array();
       for ($i = 1; $i < count($retention_data); $i++) {
           $current = $retention_data[$i]['unique_patients'];
           $previous = $retention_data[$i-1]['unique_patients'];
           $retention_rate = $previous > 0 ? round(($current / $previous) * 100, 2) : 0;
           
           $retention_rates[] = array(
               'month' => $retention_data[$i]['month'],
               'retention_rate' => $retention_rate
           );
       }
       
       return array(
           'monthly_patients' => array_reverse($retention_data),
           'retention_rates' => array_reverse($retention_rates),
           'avg_retention_rate' => count($retention_rates) > 0 ? 
               round(array_sum(array_column($retention_rates, 'retention_rate')) / count($retention_rates), 2) : 0
       );
   }

   // =============================================
   // CALCULATION HELPER METHODS
   // =============================================

   private function calculateTrendDirection($values) {
       if (count($values) < 2) return 'insufficient_data';
       
       $increases = 0;
       $decreases = 0;
       
       for ($i = 1; $i < count($values); $i++) {
           if ($values[$i] > $values[$i-1]) $increases++;
           elseif ($values[$i] < $values[$i-1]) $decreases++;
       }
       
       if ($increases > $decreases * 1.5) return 'strong_upward';
       if ($increases > $decreases) return 'upward';
       if ($decreases > $increases * 1.5) return 'strong_downward';
       if ($decreases > $increases) return 'downward';
       return 'stable';
   }

   private function calculateVolatility($values) {
       if (count($values) < 2) return 0;
       
       $mean = array_sum($values) / count($values);
       $variance = 0;
       
       foreach ($values as $value) {
           $variance += pow($value - $mean, 2);
       }
       
       $variance /= count($values);
       $std_dev = sqrt($variance);
       
       return $mean > 0 ? round(($std_dev / $mean) * 100, 2) : 0;
   }

   private function calculateMonthlyGrowthRates($monthly_data) {
       $growth_rates = array();
       
       for ($i = 1; $i < count($monthly_data); $i++) {
           $current = $monthly_data[$i]['revenue'];
           $previous = $monthly_data[$i-1]['revenue'];
           $growth_rate = $previous > 0 ? (($current - $previous) / $previous) * 100 : 0;
           $growth_rates[] = $growth_rate;
       }
       
       return $growth_rates;
   }

   private function calculateGrowthConsistency($growth_rates) {
       if (count($growth_rates) < 2) return 0;
       
       $positive_months = count(array_filter($growth_rates, function($rate) { return $rate > 0; }));
       return round(($positive_months / count($growth_rates)) * 100, 2);
   }

   private function findStrongestQuarter($monthly_data) {
       $quarters = array();
       
       for ($i = 0; $i < count($monthly_data) - 2; $i += 3) {
           $quarter_revenue = 0;
           for ($j = 0; $j < 3 && ($i + $j) < count($monthly_data); $j++) {
               $quarter_revenue += $monthly_data[$i + $j]['revenue'];
           }
           
           $quarters[] = array(
               'start_month' => $monthly_data[$i]['month_name'],
               'revenue' => $quarter_revenue
           );
       }
       
       if (empty($quarters)) return null;
       
       return array_reduce($quarters, function($carry, $item) {
           return ($carry === null || $item['revenue'] > $carry['revenue']) ? $item : $carry;
       });
   }

   private function findWeakestQuarter($monthly_data) {
       $quarters = array();
       
       for ($i = 0; $i < count($monthly_data) - 2; $i += 3) {
           $quarter_revenue = 0;
           for ($j = 0; $j < 3 && ($i + $j) < count($monthly_data); $j++) {
               $quarter_revenue += $monthly_data[$i + $j]['revenue'];
           }
           
           $quarters[] = array(
               'start_month' => $monthly_data[$i]['month_name'],
               'revenue' => $quarter_revenue
           );
       }
       
       if (empty($quarters)) return null;
       
       return array_reduce($quarters, function($carry, $item) {
           return ($carry === null || $item['revenue'] < $carry['revenue']) ? $item : $carry;
       });
   }

   private function calculateSeasonalityStrength($monthly_averages) {
       if (count($monthly_averages) < 12) return 0;
       
       $revenues = array_column($monthly_averages, 'avg_revenue');
       $max_revenue = max($revenues);
       $min_revenue = min($revenues);
       $avg_revenue = array_sum($revenues) / count($revenues);
       
       return $avg_revenue > 0 ? round((($max_revenue - $min_revenue) / $avg_revenue) * 100, 2) : 0;
   }

   private function calculateLinearTrendSlope($x_values, $y_values) {
       $n = count($x_values);
       $sum_x = array_sum($x_values);
       $sum_y = array_sum($y_values);
       $sum_xy = 0;
       $sum_x2 = 0;
       
       for ($i = 0; $i < $n; $i++) {
           $sum_xy += $x_values[$i] * $y_values[$i];
           $sum_x2 += $x_values[$i] * $x_values[$i];
       }
       
       $denominator = $n * $sum_x2 - $sum_x * $sum_x;
       return $denominator != 0 ? ($n * $sum_xy - $sum_x * $sum_y) / $denominator : 0;
   }

   private function getSeasonalFactor($forecast_month, $monthly_data) {
       $month_number = date('n', strtotime($forecast_month . '-01'));
       $same_month_revenues = array();
       
       foreach ($monthly_data as $data) {
           if (date('n', strtotime($data['month'] . '-01')) == $month_number) {
               $same_month_revenues[] = $data['revenue'];
           }
       }
       
       if (empty($same_month_revenues)) return 1.0;
       
       $overall_avg = array_sum(array_column($monthly_data, 'revenue')) / count($monthly_data);
       $month_avg = array_sum($same_month_revenues) / count($same_month_revenues);
       
       return $overall_avg > 0 ? $month_avg / $overall_avg : 1.0;
   }

   private function calculateForecastConfidence($months_ahead, $historical_data) {
       // Simple confidence calculation - decreases with distance and increases with data stability
       $base_confidence = 85;
       $distance_penalty = $months_ahead * 5;
       $stability_bonus = $this->calculateVolatility($historical_data) < 20 ? 10 : 0;
       
       return max(50, min(95, $base_confidence - $distance_penalty + $stability_bonus));
   }

   private function assessDataQuality($monthly_data) {
       $completeness = count($monthly_data) >= 12 ? 'good' : 'limited';
       $consistency = $this->calculateRevenueConsistency(array_column($monthly_data, 'revenue'));
       
       return array(
           'data_points' => count($monthly_data),
           'completeness' => $completeness,
           'consistency_score' => $consistency,
           'quality_rating' => $consistency > 70 && $completeness === 'good' ? 'high' : 'medium'
       );
   }

   private function findBestMonth($monthly_data) {
       return array_reduce($monthly_data, function($carry, $item) {
           return ($carry === null || $item['revenue'] > $carry['revenue']) ? $item : $carry;
       });
   }

   private function findWorstMonth($monthly_data) {
       return array_reduce($monthly_data, function($carry, $item) {
           return ($carry === null || $item['revenue'] < $carry['revenue']) ? $item : $carry;
       });
   }

   private function getWorkingDaysInMonth($month_start) {
       $start = new DateTime($month_start);
       $end = new DateTime(date('Y-m-t', strtotime($month_start)));
       $working_days = 0;
       
       while ($start <= $end) {
           $day_of_week = $start->format('N');
           if ($day_of_week < 7) { // Monday to Saturday
               $working_days++;
           }
           $start->add(new DateInterval('P1D'));
       }
       
       return $working_days;
   }

   private function classifyPatientSegment($patient) {
       $revenue = $patient['total_revenue'];
       $visits = $patient['visit_count'];
       
       if ($revenue >= 50000 && $visits >= 10) return 'VIP';
       if ($revenue >= 20000 && $visits >= 5) return 'Premium';
       if ($revenue >= 5000 && $visits >= 3) return 'Regular';
       if ($visits > 1) return 'Occasional';
       return 'One-time';
   }

   private function calculateDaysSinceLastVisit($last_visit) {
       if (!$last_visit) return null;
       
       $last_visit_date = new DateTime($last_visit);
       $today = new DateTime();
       $interval = $today->diff($last_visit_date);
       
       return $interval->days;
   }

   // =============================================
   // ADVANCED ANALYTICS AND KPIs
   // =============================================

   public function financial_kpis($period = 'current_month') {
       header('Content-Type: application/json');
       
       try {
           switch ($period) {
               case 'current_month':
                   $start_date = date('Y-m-01');
                   $end_date = date('Y-m-d');
                   break;
               case 'current_quarter':
                   $start_date = date('Y-m-01', strtotime('-2 months'));
                   $end_date = date('Y-m-d');
                   break;
               case 'current_year':
                   $start_date = date('Y-01-01');
                   $end_date = date('Y-m-d');
                   break;
               default:
                   $start_date = date('Y-m-01');
                   $end_date = date('Y-m-d');
           }
           
           $kpis = $this->calculateFinancialKPIs($start_date, $end_date);
           $benchmarks = $this->getIndustryBenchmarks();
           $performance_indicators = $this->calculatePerformanceIndicators($kpis, $benchmarks);
           
           echo json_encode(array(
               'status' => 'success',
               'data' => array(
                   'period' => $period,
                   'date_range' => array('start' => $start_date, 'end' => $end_date),
                   'kpis' => $kpis,
                   'benchmarks' => $benchmarks,
                   'performance_indicators' => $performance_indicators,
                   'recommendations' => $this->generateKPIRecommendations($kpis, $benchmarks)
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   public function revenue_optimization_analysis() {
       header('Content-Type: application/json');
       
       try {
           $optimization_data = array(
               'department_efficiency' => $this->analyzeDepartmentEfficiency(),
               'pricing_analysis' => $this->analyzePricingOpportunities(),
               'capacity_utilization' => $this->analyzeCapacityUtilization(),
               'patient_flow_optimization' => $this->analyzePatientFlow(),
               'revenue_leakage' => $this->identifyRevenueLeakage(),
               'growth_opportunities' => $this->identifyGrowthOpportunities()
           );
           
           echo json_encode(array(
               'status' => 'success',
               'data' => $optimization_data
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   public function cost_analysis($start_date = null, $end_date = null) {
       header('Content-Type: application/json');
       
       try {
           if (!$start_date) $start_date = date('Y-m-01');
           if (!$end_date) $end_date = date('Y-m-d');
           
           $cost_data = array(
               'cost_breakdown' => $this->getCostBreakdown($start_date, $end_date),
               'cost_per_patient' => $this->calculateCostPerPatient($start_date, $end_date),
               'cost_trends' => $this->analyzeCostTrends($start_date, $end_date),
               'cost_efficiency' => $this->analyzeCostEfficiency($start_date, $end_date),
               'budget_variance' => $this->calculateBudgetVariance($start_date, $end_date)
           );
           
           echo json_encode(array(
               'status' => 'success',
               'data' => $cost_data
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   // =============================================
   // KPI CALCULATION METHODS
   // =============================================

   private function calculateFinancialKPIs($start_date, $end_date) {
       $revenue_data = $this->getPeriodRevenue($start_date, $end_date);
       $expense_data = $this->getPeriodExpenses($start_date, $end_date);
       $patient_data = $this->getPatientMetrics($start_date, $end_date);
       
       $days = (strtotime($end_date) - strtotime($start_date)) / (60 * 60 * 24) + 1;
       $working_days = $this->calculateWorkingDays($start_date, $end_date);
       
       return array(
           'revenue_kpis' => array(
               'total_revenue' => $revenue_data['revenue'],
               'daily_revenue' => round($revenue_data['revenue'] / $days, 2),
               'working_day_revenue' => $working_days > 0 ? round($revenue_data['revenue'] / $working_days, 2) : 0,
               'revenue_per_patient' => $patient_data['unique_patients'] > 0 ? 
                   round($revenue_data['revenue'] / $patient_data['unique_patients'], 2) : 0,
               'average_transaction_value' => $revenue_data['avg_transaction'],
               'revenue_growth_rate' => $this->calculateRevenueGrowthRate($start_date, $end_date)
           ),
           'efficiency_kpis' => array(
               'patient_throughput' => $patient_data['unique_patients'],
               'transactions_per_day' => round($revenue_data['transactions'] / $days, 2),
               'patients_per_day' => round($patient_data['unique_patients'] / $working_days, 2),
               'capacity_utilization' => $this->calculateCapacityUtilization($patient_data, $working_days),
               'staff_productivity' => $this->calculateStaffProductivity($revenue_data, $start_date, $end_date)
           ),
           'profitability_kpis' => array(
               'gross_profit' => $revenue_data['revenue'] - $expense_data['direct_costs'],
               'operating_profit' => $revenue_data['revenue'] - $expense_data['total_expenses'],
               'profit_margin' => $revenue_data['revenue'] > 0 ? 
                   round((($revenue_data['revenue'] - $expense_data['total_expenses']) / $revenue_data['revenue']) * 100, 2) : 0,
               'cost_per_patient' => $patient_data['unique_patients'] > 0 ? 
                   round($expense_data['total_expenses'] / $patient_data['unique_patients'], 2) : 0,
               'revenue_per_cost_dollar' => $expense_data['total_expenses'] > 0 ? 
                   round($revenue_data['revenue'] / $expense_data['total_expenses'], 2) : 0
           ),
           'quality_kpis' => array(
               'patient_satisfaction_proxy' => $this->calculatePatientSatisfactionProxy($start_date, $end_date),
               'repeat_patient_rate' => $this->calculateRepeatPatientRate($start_date, $end_date),
               'service_quality_index' => $this->calculateServiceQualityIndex($start_date, $end_date)
           )
       );
   }

   private function getPeriodExpenses($start_date, $end_date) {
       // This would typically come from expense tracking or chart of accounts
       // For now, we'll use estimates based on typical hospital cost structures
       $revenue = $this->getPeriodRevenue($start_date, $end_date)['revenue'];
       
       return array(
           'staff_costs' => $revenue * 0.4, // Typically 40% of revenue
           'medical_supplies' => $revenue * 0.15, // 15% of revenue
           'utilities' => $revenue * 0.05, // 5% of revenue
           'administrative' => $revenue * 0.08, // 8% of revenue
           'maintenance' => $revenue * 0.03, // 3% of revenue
           'other_expenses' => $revenue * 0.04, // 4% of revenue
           'direct_costs' => $revenue * 0.25, // 25% direct costs
           'total_expenses' => $revenue * 0.75 // 75% total expenses (25% profit margin)
       );
   }

   private function getPatientMetrics($start_date, $end_date) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('COUNT(DISTINCT patient_id) as unique_patients');
           $this->db->select('COUNT(*) as total_visits');
           $this->db->from('financial_transactions');
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           
           $result = $this->db->get()->row();
           return array(
               'unique_patients' => intval($result->unique_patients ?? 0),
               'total_visits' => intval($result->total_visits ?? 0)
           );
       }
       
       return array('unique_patients' => 0, 'total_visits' => 0);
   }

   private function calculateWorkingDays($start_date, $end_date) {
       $start = new DateTime($start_date);
       $end = new DateTime($end_date);
       $working_days = 0;
       
       while ($start <= $end) {
           $day_of_week = $start->format('N');
           if ($day_of_week < 7) { // Monday to Saturday (assuming hospital works 6 days)
               $working_days++;
           }
           $start->add(new DateInterval('P1D'));
       }
       
       return $working_days;
   }

   private function calculateRevenueGrowthRate($start_date, $end_date) {
       // Compare with same period last year
       $current_revenue = $this->getPeriodRevenue($start_date, $end_date)['revenue'];
       
       $prev_start = date('Y-m-d', strtotime($start_date . ' -1 year'));
       $prev_end = date('Y-m-d', strtotime($end_date . ' -1 year'));
       $previous_revenue = $this->getPeriodRevenue($prev_start, $prev_end)['revenue'];
       
       return $previous_revenue > 0 ? round((($current_revenue - $previous_revenue) / $previous_revenue) * 100, 2) : 0;
   }

   private function calculateCapacityUtilization($patient_data, $working_days) {
       // Assuming hospital can handle 50 patients per day at full capacity
       $theoretical_capacity = $working_days * 50;
       return $theoretical_capacity > 0 ? round(($patient_data['unique_patients'] / $theoretical_capacity) * 100, 2) : 0;
   }

   private function calculateStaffProductivity($revenue_data, $start_date, $end_date) {
       // Assuming average of 25 staff members (this should come from HR data)
       $average_staff = 25;
       $days = (strtotime($end_date) - strtotime($start_date)) / (60 * 60 * 24) + 1;
       
       return round($revenue_data['revenue'] / ($average_staff * $days), 2);
   }

   private function calculatePatientSatisfactionProxy($start_date, $end_date) {
       // Use repeat visit rate as a proxy for satisfaction
       return $this->calculateRepeatPatientRate($start_date, $end_date);
   }

   private function calculateRepeatPatientRate($start_date, $end_date) {
       if (!$this->db->table_exists('financial_transactions')) return 0;
       
       // Get patients who visited multiple times in the period
       $this->db->select('COUNT(*) as repeat_patients');
       $this->db->from('(SELECT patient_id, COUNT(*) as visit_count FROM financial_transactions 
                         WHERE transaction_type = "revenue" 
                         AND transaction_date >= "' . $start_date . '" 
                         AND transaction_date <= "' . $end_date . '" 
                         GROUP BY patient_id HAVING visit_count > 1) as repeat_visits');
       
       $repeat_patients = $this->db->get()->row()->repeat_patients ?? 0;
       
       // Get total unique patients
       $total_patients = $this->getPatientMetrics($start_date, $end_date)['unique_patients'];
       
       return $total_patients > 0 ? round(($repeat_patients / $total_patients) * 100, 2) : 0;
   }

   private function calculateServiceQualityIndex($start_date, $end_date) {
       // Composite index based on multiple factors
       $repeat_rate = $this->calculateRepeatPatientRate($start_date, $end_date);
       $avg_transaction = $this->getPeriodRevenue($start_date, $end_date)['avg_transaction'];
       
       // Simple quality index (this could be more sophisticated)
       $quality_score = ($repeat_rate * 0.6) + (min($avg_transaction / 1000, 100) * 0.4);
       
       return round($quality_score, 2);
   }

   private function getIndustryBenchmarks() {
       // These would typically come from industry reports or databases
       return array(
           'revenue_per_patient' => 2500,
           'profit_margin' => 15,
           'capacity_utilization' => 75,
           'repeat_patient_rate' => 40,
           'daily_revenue_target' => 25000,
           'cost_per_patient' => 2000,
           'staff_productivity' => 1500
       );
   }

   private function calculatePerformanceIndicators($kpis, $benchmarks) {
       $indicators = array();
       
       // Revenue performance
       $revenue_per_patient = $kpis['revenue_kpis']['revenue_per_patient'];
       $indicators['revenue_performance'] = array(
           'current' => $revenue_per_patient,
           'benchmark' => $benchmarks['revenue_per_patient'],
           'performance' => $revenue_per_patient >= $benchmarks['revenue_per_patient'] ? 'above' : 'below',
           'variance_percentage' => $benchmarks['revenue_per_patient'] > 0 ? 
               round((($revenue_per_patient - $benchmarks['revenue_per_patient']) / $benchmarks['revenue_per_patient']) * 100, 2) : 0
       );
       
       // Profitability performance
       $profit_margin = $kpis['profitability_kpis']['profit_margin'];
       $indicators['profitability_performance'] = array(
           'current' => $profit_margin,
           'benchmark' => $benchmarks['profit_margin'],
           'performance' => $profit_margin >= $benchmarks['profit_margin'] ? 'above' : 'below',
           'variance_percentage' => $profit_margin - $benchmarks['profit_margin']
       );
       
       // Efficiency performance
       $capacity_utilization = $kpis['efficiency_kpis']['capacity_utilization'];
       $indicators['efficiency_performance'] = array(
           'current' => $capacity_utilization,
           'benchmark' => $benchmarks['capacity_utilization'],
           'performance' => $capacity_utilization >= $benchmarks['capacity_utilization'] ? 'above' : 'below',
           'variance_percentage' => $capacity_utilization - $benchmarks['capacity_utilization']
       );
       
       return $indicators;
   }

   private function generateKPIRecommendations($kpis, $benchmarks) {
       $recommendations = array();
       
       // Revenue recommendations
       if ($kpis['revenue_kpis']['revenue_per_patient'] < $benchmarks['revenue_per_patient']) {
           $recommendations[] = array(
               'category' => 'Revenue Optimization',
               'priority' => 'High',
               'recommendation' => 'Focus on increasing average revenue per patient through service upselling and premium offerings',
               'potential_impact' => 'Could increase revenue by 15-25%'
           );
       }
       
       // Capacity recommendations
       if ($kpis['efficiency_kpis']['capacity_utilization'] < 60) {
           $recommendations[] = array(
               'category' => 'Capacity Management',
               'priority' => 'Medium',
               'recommendation' => 'Implement marketing campaigns to increase patient volume and optimize scheduling',
               'potential_impact' => 'Could improve utilization by 10-20%'
           );
       }
       
       // Profitability recommendations
       if ($kpis['profitability_kpis']['profit_margin'] < 10) {
           $recommendations[] = array(
               'category' => 'Cost Management',
               'priority' => 'High',
               'recommendation' => 'Review operational costs and implement cost reduction strategies',
               'potential_impact' => 'Could improve profit margin by 5-10%'
           );
       }
       
       // Patient retention recommendations
       if ($kpis['quality_kpis']['repeat_patient_rate'] < 30) {
           $recommendations[] = array(
               'category' => 'Patient Retention',
               'priority' => 'Medium',
               'recommendation' => 'Implement patient follow-up programs and improve service quality',
               'potential_impact' => 'Could increase repeat patients by 15-30%'
           );
       }
       
       return $recommendations;
   }

   // =============================================
   // OPTIMIZATION ANALYSIS METHODS
   // =============================================

   private function analyzeDepartmentEfficiency() {
       $departments = array('opd', 'ipd', 'pharmacy', 'laboratory', 'emergency');
       $efficiency_data = array();
       
       $month_start = date('Y-m-01');
       $month_end = date('Y-m-d');
       
       foreach ($departments as $dept) {
           $dept_revenue = $this->getDepartmentRevenue($dept, $month_start, $month_end);
           $dept_patients = $this->getDepartmentPatients($dept, $month_start, $month_end);
           $dept_capacity = $this->getDepartmentCapacity($dept);
           
           $efficiency_data[$dept] = array(
               'revenue' => $dept_revenue,
               'patients' => $dept_patients,
               'capacity' => $dept_capacity,
               'utilization_rate' => $dept_capacity > 0 ? round(($dept_patients / $dept_capacity) * 100, 2) : 0,
               'revenue_per_patient' => $dept_patients > 0 ? round($dept_revenue / $dept_patients, 2) : 0,
               'efficiency_score' => $this->calculateDepartmentEfficiencyScore($dept_revenue, $dept_patients, $dept_capacity)
           );
       }
       
       return $efficiency_data;
   }

   private function analyzePricingOpportunities() {
       // Analyze current pricing vs market rates and utilization
       return array(
           'opd_consultation' => array(
               'current_price' => 500,
               'market_average' => 600,
               'utilization_rate' => 85,
               'price_elasticity' => 'low',
               'recommendation' => 'Consider 15% price increase',
               'estimated_impact' => '+12% revenue'
           ),
           'laboratory_tests' => array(
               'current_price' => 750,
               'market_average' => 700,
               'utilization_rate' => 92,
               'price_elasticity' => 'medium',
               'recommendation' => 'Maintain current pricing',
               'estimated_impact' => 'neutral'
           )
       );
   }

   private function analyzeCapacityUtilization() {
       $current_capacity = $this->getCurrentCapacityUtilization();
       $peak_hours = $this->identifyPeakHours();
       $bottlenecks = $this->identifyBottlenecks();
       
       return array(
           'overall_utilization' => $current_capacity,
           'peak_hours' => $peak_hours,
           'bottlenecks' => $bottlenecks,
           'optimization_opportunities' => $this->getCapacityOptimizationOpportunities()
       );
   }

   private function analyzePatientFlow() {
       return array(
           'average_wait_time' => 25, // minutes
           'patient_journey_analysis' => array(
               'registration_time' => 5,
               'waiting_time' => 20,
               'consultation_time' => 15,
               'payment_time' => 3
           ),
           'flow_optimization_opportunities' => array(
               'implement_online_registration',
               'optimize_appointment_scheduling',
               'streamline_payment_process'
           )
       );
   }

   private function identifyRevenueLeakage() {
       return array(
           'unbilled_services' => array(
               'amount' => 15000,
               'percentage' => 2.5,
               'main_sources' => array('emergency_services', 'additional_procedures')
           ),
           'collection_issues' => array(
               'outstanding_amount' => 45000,
               'average_collection_time' => 35, // days
               'collection_rate' => 92
           ),
           'pricing_gaps' => array(
               'underpriced_services' => array('consultation', 'diagnostics'),
               'potential_additional_revenue' => 25000
           )
       );
   }

   private function identifyGrowthOpportunities() {
       return array(
           'service_expansion' => array(
               'potential_services' => array('telemedicine', 'home_care', 'wellness_programs'),
               'estimated_revenue_impact' => 180000 // annually
           ),
           'market_expansion' => array(
               'target_demographics' => array('corporate_clients', 'insurance_partnerships'),
               'estimated_revenue_impact' => 240000 // annually
           ),
           'technology_investments' => array(
               'opportunities' => array('automated_billing', 'patient_portal', 'mobile_app'),
               'investment_required' => 75000,
               'expected_roi' => 150 // percentage
           )
       );
   }

   // =============================================
   // HELPER METHODS FOR OPTIMIZATION ANALYSIS
   // =============================================

   private function getDepartmentRevenue($department, $start_date, $end_date) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('SUM(amount) as revenue');
           $this->db->from('financial_transactions');
           $this->db->where('department', $department);
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           
           $result = $this->db->get()->row();
           return floatval($result->revenue ?? 0);
       }
       
       return 0;
   }

   private function getDepartmentPatients($department, $start_date, $end_date) {
       if ($this->db->table_exists('financial_transactions')) {
           $this->db->select('COUNT(DISTINCT patient_id) as patients');
           $this->db->from('financial_transactions');
           $this->db->where('department', $department);
           $this->db->where('transaction_type', 'revenue');
           $this->db->where('transaction_date >=', $start_date);
           $this->db->where('transaction_date <=', $end_date);
           
           $result = $this->db->get()->row();
           return intval($result->patients ?? 0);
       }
       
       return 0;
   }

   private function getDepartmentCapacity($department) {
       // This would typically come from a configuration table
       $capacities = array(
           'opd' => 100, // patients per day
           'ipd' => 50,
           'pharmacy' => 150,
           'laboratory' => 75,
           'emergency' => 25
       );
       
       return $capacities[$department] ?? 50;
   }

   private function calculateDepartmentEfficiencyScore($revenue, $patients, $capacity) {
       if ($capacity == 0) return 0;
       
       $utilization_score = ($patients / $capacity) * 50; // 50% weight
       $revenue_score = min(($revenue / ($capacity * 1000)), 1) * 50; // 50% weight
       
       return round($utilization_score + $revenue_score, 2);
   }

   private function getCurrentCapacityUtilization() {
       // This would analyze current vs maximum capacity across all departments
       return array(
           'overall' => 78,
           'by_department' => array(
               'opd' => 85,
               'ipd' => 70,
               'pharmacy' => 92,
               'laboratory' => 65,
               'emergency' => 45
           )
       );
   }

   private function identifyPeakHours() {
       return array(
           'morning' => array('time' => '09:00-11:00', 'utilization' => 95),
           'afternoon' => array('time' => '14:00-16:00', 'utilization' => 88),
           'evening' => array('time' => '18:00-20:00', 'utilization' => 75)
       );
   }

   private function identifyBottlenecks() {
       return array(
           array(
               'area' => 'OPD Registration',
               'impact' => 'High',
               'average_delay' => 15, // minutes
               'solution' => 'Add registration counter or implement online registration'
           ),
           array(
               'area' => 'Laboratory Sample Collection',
               'impact' => 'Medium',
               'average_delay' => 8,
               'solution' => 'Optimize sample collection workflow'
           )
       );
   }

   private function getCapacityOptimizationOpportunities() {
       return array(
           array(
               'opportunity' => 'Extend OPD hours',
               'potential_increase' => '20% capacity',
               'investment_required' => 'Minimal',
               'timeline' => '1 month'
           ),
           array(
               'opportunity' => 'Implement appointment system',
               'potential_increase' => '15% efficiency',
               'investment_required' => 'Medium',
               'timeline' => '2-3 months'
           )
       );
   }

   // =============================================
   // SYSTEM INITIALIZATION AND SETUP
   // =============================================

   public function initialize_enhanced_system() {
       header('Content-Type: application/json');
       
       try {
           $initialization_steps = array();
           
           // Check and create enhanced tables
           if (!$this->db->table_exists('chart_of_accounts')) {
               $this->createChartOfAccountsTable();
               $initialization_steps[] = 'Created chart_of_accounts table';
           }
           
           if (!$this->db->table_exists('general_ledger')) {
               $this->createGeneralLedgerTable();
               $initialization_steps[] = 'Created general_ledger table';
           }
           
           if (!$this->db->table_exists('financial_transactions')) {
               $this->createFinancialTransactionsTable();
               $initialization_steps[] = 'Created financial_transactions table';
           }
           
           // Create triggers for automatic posting
           $this->createFinancialTriggers();
           $initialization_steps[] = 'Created financial posting triggers';
           
           // Populate chart of accounts
           $this->populateChartOfAccounts();
           $initialization_steps[] = 'Populated chart of accounts';
           
           echo json_encode(array(
               'status' => 'success',
               'message' => 'Enhanced financial system initialized successfully',
               'steps_completed' => $initialization_steps,
               'next_steps' => array(
                   'Configure department mappings',
                   'Set up automated journal entries',
                   'Train staff on new reporting features'
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => 'Initialization failed: ' . $e->getMessage()
           ));
       }
   }

   private function createChartOfAccountsTable() {
       $sql = "CREATE TABLE IF NOT EXISTS `chart_of_accounts` (
           `id` int(11) NOT NULL AUTO_INCREMENT,
           `account_code` varchar(20) NOT NULL UNIQUE,
           `account_name` varchar(100) NOT NULL,
           `account_type` enum('asset','liability','equity','revenue','expense') NOT NULL,
           `account_category` varchar(50),
           `parent_account_id` int(11) NULL,
           `is_active` enum('yes','no') DEFAULT 'yes',
           `description` text,
           `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
           `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
           PRIMARY KEY (`id`),
           KEY `idx_code` (`account_code`),
           KEY `idx_type` (`account_type`),
           KEY `idx_category` (`account_category`),
           KEY `idx_parent` (`parent_account_id`),
           FOREIGN KEY (`parent_account_id`) REFERENCES `chart_of_accounts`(`id`)
       ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
       
       $this->db->query($sql);
   }

   private function createGeneralLedgerTable() {
       $sql = "CREATE TABLE IF NOT EXISTS `general_ledger` (
           `id` bigint(20) NOT NULL AUTO_INCREMENT,
           `transaction_date` date NOT NULL,
           `account_id` int(11) NOT NULL,
           `reference_type` varchar(50),
           `reference_id` int(11),
           `description` text,
           `debit_amount` decimal(15,2) DEFAULT 0.00,
           `credit_amount` decimal(15,2) DEFAULT 0.00,
           `running_balance` decimal(15,2),
           `posted_by` int(11),
           `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
           PRIMARY KEY (`id`, `transaction_date`),
           KEY `idx_account` (`account_id`),
           KEY `idx_date` (`transaction_date`),
           KEY `idx_reference` (`reference_type`, `reference_id`),
           KEY `idx_amounts` (`debit_amount`, `credit_amount`),
           FOREIGN KEY (`account_id`) REFERENCES `chart_of_accounts`(`id`)
       ) ENGINE=InnoDB DEFAULT CHARSET=utf8
       PARTITION BY RANGE (YEAR(transaction_date)) (
           PARTITION p2023 VALUES LESS THAN (2024),
           PARTITION p2024 VALUES LESS THAN (2025),
           PARTITION p2025 VALUES LESS THAN (2026),
           PARTITION p_future VALUES LESS THAN MAXVALUE
       )";
       
       $this->db->query($sql);
   }

   private function createFinancialTransactionsTable() {
       $sql = "CREATE TABLE IF NOT EXISTS `financial_transactions` (
           `id` bigint(20) NOT NULL AUTO_INCREMENT,
           `transaction_date` datetime NOT NULL,
           `transaction_type` enum('revenue','expense','asset','liability') NOT NULL,
           `category` varchar(50) NOT NULL,
           `subcategory` varchar(50),
           `amount` decimal(15,2) NOT NULL,
           `description` text,
           `reference_table` varchar(50),
           `reference_id` int(11),
           `patient_id` varchar(50),
           `department` varchar(50),
           `payment_method` enum('cash','card','cheque','bank_transfer','mobile_money','insurance'),
           `receipt_number` varchar(50),
           `created_by` int(11),
           `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
           PRIMARY KEY (`id`, `transaction_date`),
           KEY `idx_type` (`transaction_type`),
           KEY `idx_category` (`category`),
           KEY `idx_date` (`transaction_date`),
           KEY `idx_amount` (`amount`),
           KEY `idx_reference` (`reference_table`, `reference_id`),
           KEY `idx_patient` (`patient_id`),
           KEY `idx_department` (`department`),
           KEY `idx_payment_method` (`payment_method`)
       ) ENGINE=InnoDB DEFAULT CHARSET=utf8
       PARTITION BY RANGE (YEAR(transaction_date)) (
           PARTITION p2023 VALUES LESS THAN (2024),
           PARTITION p2024 VALUES LESS THAN (2025),
           PARTITION p2025 VALUES LESS THAN (2026),
           PARTITION p_future VALUES LESS THAN MAXVALUE
       )";
       
       $this->db->query($sql);
   }

   private function createFinancialTriggers() {
       // OPD Revenue Trigger
       $trigger_sql = "
       CREATE TRIGGER `tr_opd_revenue_posting`
       AFTER INSERT ON `opd_details`
       FOR EACH ROW
       BEGIN
           DECLARE revenue_account_id INT;
           DECLARE receivable_account_id INT;
           
           SELECT id INTO revenue_account_id FROM chart_of_accounts WHERE account_code = '4000' LIMIT 1;
           SELECT id INTO receivable_account_id FROM chart_of_accounts WHERE account_code = '1200' LIMIT 1;
           
           IF NEW.amount > 0 AND revenue_account_id IS NOT NULL AND receivable_account_id IS NOT NULL THEN
               INSERT INTO general_ledger (transaction_date, account_id, reference_type, reference_id, description, debit_amount, posted_by)
               VALUES (DATE(NEW.appointment_date), receivable_account_id, 'opd', NEW.id, 
                       CONCAT('OPD Consultation - Patient: ', NEW.patient_id), NEW.amount, NEW.generated_by);
               
               INSERT INTO general_ledger (transaction_date, account_id, reference_type, reference_id, description, credit_amount, posted_by)
               VALUES (DATE(NEW.appointment_date), revenue_account_id, 'opd', NEW.id, 
                       CONCAT('OPD Consultation - Patient: ', NEW.patient_id), NEW.amount, NEW.generated_by);
               
               INSERT INTO financial_transactions (transaction_date, transaction_type, category, subcategory, amount, description, 
                                                 reference_table, reference_id, patient_id, department, payment_method, created_by)
               VALUES (NEW.appointment_date, 'revenue', 'medical_services', 'opd_consultation', NEW.amount,
                       CONCAT('OPD Consultation - Patient: ', NEW.patient_id), 'opd_details', NEW.id, NEW.patient_id, 
                       'opd', COALESCE(NEW.payment_mode, 'cash'), NEW.generated_by);
           END IF;
       END";
       
       try {
           $this->db->query("DROP TRIGGER IF EXISTS tr_opd_revenue_posting");
           $this->db->query($trigger_sql);
       } catch (Exception $e) {
           log_message('error', 'Error creating OPD trigger: ' . $e->getMessage());
       }

       // IPD Revenue Trigger
       $trigger_sql = "
       CREATE TRIGGER `tr_ipd_revenue_posting`
       AFTER INSERT ON `ipd_details`
       FOR EACH ROW
       BEGIN
           DECLARE revenue_account_id INT;
           DECLARE receivable_account_id INT;
           
           SELECT id INTO revenue_account_id FROM chart_of_accounts WHERE account_code = '4100' LIMIT 1;
           SELECT id INTO receivable_account_id FROM chart_of_accounts WHERE account_code = '1200' LIMIT 1;
           
           IF NEW.amount > 0 AND revenue_account_id IS NOT NULL AND receivable_account_id IS NOT NULL THEN
               INSERT INTO general_ledger (transaction_date, account_id, reference_type, reference_id, description, debit_amount, posted_by)
               VALUES (NEW.date, receivable_account_id, 'ipd', NEW.id, 
                       CONCAT('IPD Services - Patient: ', NEW.patient_id), NEW.amount, 1);
               
               INSERT INTO general_ledger (transaction_date, account_id, reference_type, reference_id, description, credit_amount, posted_by)
               VALUES (NEW.date, revenue_account_id, 'ipd', NEW.id, 
                       CONCAT('IPD Services - Patient: ', NEW.patient_id), NEW.amount, 1);
               
               INSERT INTO financial_transactions (transaction_date, transaction_type, category, subcategory, amount, description, 
                                                 reference_table, reference_id, patient_id, department, payment_method, created_by)
               VALUES (NEW.date, 'revenue', 'medical_services', 'ipd_services', NEW.amount,
                       CONCAT('IPD Services - Patient: ', NEW.patient_id), 'ipd_details', NEW.id, NEW.patient_id, 
                       'ipd', 'cash', 1);
           END IF;
       END";
       
       try {
           $this->db->query("DROP TRIGGER IF EXISTS tr_ipd_revenue_posting");
           $this->db->query($trigger_sql);
       } catch (Exception $e) {
           log_message('error', 'Error creating IPD trigger: ' . $e->getMessage());
       }

       // Pharmacy Revenue Trigger
       $trigger_sql = "
       CREATE TRIGGER `tr_pharmacy_revenue_posting`
       AFTER INSERT ON `pharmacy_bill_basic`
       FOR EACH ROW
       BEGIN
           DECLARE revenue_account_id INT;
           DECLARE receivable_account_id INT;
           
           SELECT id INTO revenue_account_id FROM chart_of_accounts WHERE account_code = '4200' LIMIT 1;
           SELECT id INTO receivable_account_id FROM chart_of_accounts WHERE account_code = '1200' LIMIT 1;
           
           IF NEW.net_amount > 0 AND revenue_account_id IS NOT NULL AND receivable_account_id IS NOT NULL THEN
               INSERT INTO general_ledger (transaction_date, account_id, reference_type, reference_id, description, debit_amount, posted_by)
               VALUES (NEW.date, receivable_account_id, 'pharmacy', NEW.id, 
                       CONCAT('Pharmacy Sale - Patient: ', NEW.patient_id), NEW.net_amount, 1);
                       
               INSERT INTO general_ledger (transaction_date, account_id, reference_type, reference_id, description, credit_amount, posted_by)
               VALUES (NEW.date, revenue_account_id, 'pharmacy', NEW.id, 
                       CONCAT('Pharmacy Sale - Patient: ', NEW.patient_id), NEW.net_amount, 1);
               
               INSERT INTO financial_transactions (transaction_date, transaction_type, category, subcategory, amount, description, 
                                                 reference_table, reference_id, patient_id, department, payment_method, created_by)
               VALUES (NEW.date, 'revenue', 'medical_services', 'pharmacy_sales', NEW.net_amount,
                       CONCAT('Pharmacy Sale - Patient: ', NEW.patient_id), 'pharmacy_bill_basic', NEW.id, NEW.patient_id, 
                       'pharmacy', 'cash', 1);
           END IF;
       END";
       
       try {
           $this->db->query("DROP TRIGGER IF EXISTS tr_pharmacy_revenue_posting");
           $this->db->query($trigger_sql);
       } catch (Exception $e) {
           log_message('error', 'Error creating Pharmacy trigger: ' . $e->getMessage());
       }
   }

   private function populateChartOfAccounts() {
       // Check if already populated
       $this->db->select('COUNT(*) as count');
       $this->db->from('chart_of_accounts');
       $count = $this->db->get()->row()->count;
       
       if ($count > 0) {
           return; // Already populated
       }

       // Insert basic chart of accounts structure
       $accounts = array(
           // Assets
           array('1000', 'Cash and Cash Equivalents', 'asset', 'current_assets', null, 'yes', 'Cash on hand and in bank accounts'),
           array('1010', 'Petty Cash', 'asset', 'current_assets', null, 'yes', 'Cash kept for small expenses'),
           array('1020', 'Cash in Bank', 'asset', 'current_assets', null, 'yes', 'Main operating bank account'),
           array('1200', 'Accounts Receivable', 'asset', 'current_assets', null, 'yes', 'Money owed by patients and insurance'),
           array('1300', 'Inventory', 'asset', 'current_assets', null, 'yes', 'Medical supplies and pharmacy inventory'),
           array('1500', 'Fixed Assets', 'asset', 'fixed_assets', null, 'yes', 'Medical equipment and building'),
           
           // Liabilities
           array('2000', 'Accounts Payable', 'liability', 'current_liabilities', null, 'yes', 'Money owed to suppliers'),
           array('2010', 'Accrued Salaries', 'liability', 'current_liabilities', null, 'yes', 'Unpaid staff salaries'),
           array('2200', 'Long-term Debt', 'liability', 'long_term_liabilities', null, 'yes', 'Bank loans and mortgages'),
           
           // Equity
           array('3000', 'Owner Equity', 'equity', 'capital', null, 'yes', 'Owner investment and retained earnings'),
           array('3010', 'Retained Earnings', 'equity', 'capital', null, 'yes', 'Accumulated profits'),
           
           // Revenue
           array('4000', 'OPD Revenue', 'revenue', 'medical_services', null, 'yes', 'Outpatient consultation fees'),
           array('4100', 'IPD Revenue', 'revenue', 'medical_services', null, 'yes', 'Inpatient services revenue'),
           array('4200', 'Pharmacy Revenue', 'revenue', 'medical_services', null, 'yes', 'Pharmaceutical sales'),
           array('4300', 'Laboratory Revenue', 'revenue', 'medical_services', null, 'yes', 'Laboratory testing fees'),
           array('4400', 'Emergency Revenue', 'revenue', 'medical_services', null, 'yes', 'Emergency department fees'),
           
           // Expenses
           array('5100', 'Staff Salaries', 'expense', 'staff_costs', null, 'yes', 'Employee salaries and benefits'),
           array('5200', 'Medical Supplies', 'expense', 'direct_costs', null, 'yes', 'Medical supplies and materials'),
           array('5300', 'Utilities', 'expense', 'operational', null, 'yes', 'Electricity, water, internet'),
           array('5400', 'Administrative Expenses', 'expense', 'administrative', null, 'yes', 'Office supplies and admin costs'),
           array('5500', 'Maintenance', 'expense', 'operational', null, 'yes', 'Equipment and building maintenance'),
       );

       foreach ($accounts as $account) {
           $data = array(
               'account_code' => $account[0],
               'account_name' => $account[1],
               'account_type' => $account[2],
               'account_category' => $account[3],
               'parent_account_id' => $account[4],
               'is_active' => $account[5],
               'description' => $account[6]
           );
           
           $this->db->insert('chart_of_accounts', $data);
       }
   }

   // =============================================
   // DATABASE MAINTENANCE AND OPTIMIZATION
   // =============================================

   public function optimize_database() {
       header('Content-Type: application/json');
       
       try {
           $optimization_results = array();
           
           // Analyze table sizes and suggest partitioning
           $table_analysis = $this->analyzeTableSizes();
           $optimization_results['table_analysis'] = $table_analysis;
           
           // Optimize existing tables
           $tables_to_optimize = array('patients', 'opd_details', 'ipd_details', 'pharmacy_bill_basic', 'pathology_billing');
           $optimization_results['table_optimization'] = array();
           
           foreach ($tables_to_optimize as $table) {
               if ($this->db->table_exists($table)) {
                   $this->db->query("ANALYZE TABLE `$table`");
                   $this->db->query("OPTIMIZE TABLE `$table`");
                   $optimization_results['table_optimization'][] = "Optimized $table";
               }
           }
           
           // Create recommended indexes
           $index_recommendations = $this->createRecommendedIndexes();
           $optimization_results['index_optimization'] = $index_recommendations;
           
           // Partition recommendations
           $partition_recommendations = $this->getPartitionRecommendations($table_analysis);
           $optimization_results['partition_recommendations'] = $partition_recommendations;
           
           echo json_encode(array(
               'status' => 'success',
               'optimization_results' => $optimization_results,
               'recommendations' => array(
                   'Consider implementing database partitioning for large tables',
                   'Schedule regular database maintenance',
                   'Monitor query performance and add indexes as needed'
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   private function analyzeTableSizes() {
       $sql = "SELECT 
                   table_name,
                   table_rows,
                   ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb,
                   ROUND((data_length / 1024 / 1024), 2) AS data_mb,
                   ROUND((index_length / 1024 / 1024), 2) AS index_mb
               FROM information_schema.tables 
               WHERE table_schema = DATABASE()
               AND table_type = 'BASE TABLE'
               ORDER BY (data_length + index_length) DESC";
       
       $query = $this->db->query($sql);
       return $query->result_array();
   }

   private function createRecommendedIndexes() {
       $indexes_created = array();
       
       try {
           // Check if indexes exist before creating
           $recommended_indexes = array(
               "CREATE INDEX idx_patients_created_date ON patients(created_at)" => "patients",
               "CREATE INDEX idx_opd_appointment_patient ON opd_details(appointment_date, patient_id)" => "opd_details",
               "CREATE INDEX idx_ipd_date_patient ON ipd_details(date, patient_id)" => "ipd_details",
               "CREATE INDEX idx_pharmacy_date_patient ON pharmacy_bill_basic(date, patient_id)" => "pharmacy_bill_basic",
               "CREATE INDEX idx_pathology_date_patient ON pathology_billing(date, patient_id)" => "pathology_billing"
           );
           
           foreach ($recommended_indexes as $sql => $table) {
               if ($this->db->table_exists($table)) {
                   try {
                       $this->db->query($sql);
                       $indexes_created[] = "Created index on $table";
                   } catch (Exception $e) {
                       if (strpos($e->getMessage(), 'Duplicate key name') === false) {
                           log_message('error', 'Index creation error: ' . $e->getMessage());
                       }
                   }
               }
           }
       } catch (Exception $e) {
           log_message('error', 'Error in index creation: ' . $e->getMessage());
       }
       
       return $indexes_created;
   }

   private function getPartitionRecommendations($table_analysis) {
       $recommendations = array();
       
       foreach ($table_analysis as $table) {
           if ($table['table_rows'] > 100000) { // Tables with more than 100k rows
               $recommendations[] = array(
                   'table' => $table['table_name'],
                   'current_size' => $table['size_mb'] . ' MB',
                   'current_rows' => number_format($table['table_rows']),
                   'recommendation' => 'Consider date-based partitioning',
                   'benefits' => array(
                       'Improved query performance',
                       'Easier data archiving',
                       'Better maintenance operations'
                   )
               );
           }
       }
       
       return $recommendations;
   }

   public function create_partitioned_patients_table() {
       header('Content-Type: application/json');
       
       try {
           // First, backup existing table
           $backup_sql = "CREATE TABLE patients_backup AS SELECT * FROM patients";
           $this->db->query($backup_sql);
           
           // Create new partitioned table
           $partition_sql = "CREATE TABLE patients_new (
               id INT NOT NULL AUTO_INCREMENT,
               patient_unique_id VARCHAR(50) NOT NULL,
               admission_no VARCHAR(20),
               patient_name VARCHAR(100) NOT NULL,
               firstname VARCHAR(50),
               lastname VARCHAR(50),
               mobileno VARCHAR(20),
               email VARCHAR(100),
               gender ENUM('Male', 'Female', 'Other'),
               guardian_name VARCHAR(100),
               dob DATE,
               age VARCHAR(10),
               blood_group VARCHAR(10),
               address TEXT,
               known_allergies TEXT,
               note TEXT,
               is_active ENUM('yes', 'no') DEFAULT 'yes',
               created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
               updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
               fingerprint_template LONGTEXT,
               fingerprint_image LONGBLOB,
               
               PRIMARY KEY (id, created_at),
               UNIQUE KEY unique_patient_id (patient_unique_id),
               INDEX idx_name (patient_name),
               INDEX idx_mobile (mobileno),
               INDEX idx_active (is_active),
               INDEX idx_created_date (created_at)
           ) ENGINE=InnoDB
           PARTITION BY RANGE (YEAR(created_at)) (
               PARTITION p2020 VALUES LESS THAN (2021),
               PARTITION p2021 VALUES LESS THAN (2022),
               PARTITION p2022 VALUES LESS THAN (2023),
               PARTITION p2023 VALUES LESS THAN (2024),
               PARTITION p2024 VALUES LESS THAN (2025),
               PARTITION p2025 VALUES LESS THAN (2026),
               PARTITION p_future VALUES LESS THAN MAXVALUE
           )";
           
           $this->db->query($partition_sql);
           
           // Migrate data
           $migrate_sql = "INSERT INTO patients_new SELECT * FROM patients";
           $this->db->query($migrate_sql);
           
           echo json_encode(array(
               'status' => 'success',
               'message' => 'Partitioned patients table created successfully',
               'next_steps' => array(
                   'Verify data integrity',
                   'Update application references',
                   'Rename tables during maintenance window'
               )
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => 'Partitioning failed: ' . $e->getMessage()
           ));
       }
   }

   // =============================================
   // SYSTEM STATUS AND HEALTH CHECK
   // =============================================

   public function system_health_check() {
       header('Content-Type: application/json');
       
       try {
           $health_status = array(
               'database_structure' => $this->checkDatabaseStructure(),
               'data_integrity' => $this->checkDataIntegrity(),
               'performance_metrics' => $this->checkPerformanceMetrics(),
               'financial_consistency' => $this->checkFinancialConsistency(),
               'recommendations' => array()
           );
           
           // Generate recommendations based on health check
           $health_status['recommendations'] = $this->generateHealthRecommendations($health_status);
           
           echo json_encode(array(
               'status' => 'success',
               'health_status' => $health_status,
               'overall_health' => $this->calculateOverallHealth($health_status)
           ));
           
       } catch (Exception $e) {
           echo json_encode(array(
               'status' => 'error',
               'message' => $e->getMessage()
           ));
       }
   }

   private function checkDatabaseStructure() {
       $structure_check = array(
           'required_tables' => array(),
           'missing_tables' => array(),
           'table_health' => array()
       );
       
       $required_tables = array(
           'patients', 'opd_details', 'ipd_details', 'pharmacy_bill_basic', 
           'pathology_billing', 'chart_of_accounts', 'general_ledger', 'financial_transactions'
       );
       
       foreach ($required_tables as $table) {
           if ($this->db->table_exists($table)) {
               $structure_check['required_tables'][] = $table;
               
               // Check table health
               $table_status = $this->checkTableHealth($table);
               $structure_check['table_health'][$table] = $table_status;
           } else {
               $structure_check['missing_tables'][] = $table;
           }
       }
       
       return $structure_check;
   }

   private function checkTableHealth($table) {
       try {
           $sql = "SELECT 
                       table_rows,
                       ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb,
                       table_collation
                   FROM information_schema.tables 
                   WHERE table_schema = DATABASE() AND table_name = ?";
           
           $query = $this->db->query($sql, array($table));
           $result = $query->row_array();
           
           return array(
               'rows' => intval($result['table_rows'] ?? 0),
               'size_mb' => floatval($result['size_mb'] ?? 0),
               'collation' => $result['table_collation'] ?? 'unknown',
               'status' => 'healthy'
           );
       } catch (Exception $e) {
           return array('status' => 'error', 'message' => $e->getMessage());
       }
   }

   private function checkDataIntegrity() {
       $integrity_checks = array();
       
       // Check for orphaned records
       if ($this->db->table_exists('opd_details') && $this->db->table_exists('patients')) {
           $sql = "SELECT COUNT(*) as orphaned_opd 
                   FROM opd_details o 
                   LEFT JOIN patients p ON o.patient_id = p.id 
                   WHERE p.id IS NULL";
           $result = $this->db->query($sql)->row();
           $integrity_checks['orphaned_opd_records'] = intval($result->orphaned_opd ?? 0);
       }
       
       // Check for negative amounts
       if ($this->db->table_exists('financial_transactions')) {
           $sql = "SELECT COUNT(*) as negative_amounts FROM financial_transactions WHERE amount < 0";
           $result = $this->db->query($sql)->row();
           $integrity_checks['negative_transaction_amounts'] = intval($result->negative_amounts ?? 0);
       }
       
       return $integrity_checks;
   }

   private function checkPerformanceMetrics() {
       return array(
           'slow_queries' => $this->getSlowQueryCount(),
           'index_usage' => $this->checkIndexUsage(),
           'table_locks' => $this->checkTableLocks()
       );
   }

   private function checkFinancialConsistency() {
       $consistency_checks = array();
       
       if ($this->db->table_exists('general_ledger')) {
           // Check trial balance
           $sql = "SELECT 
                       SUM(debit_amount) as total_debits,
                       SUM(credit_amount) as total_credits
                   FROM general_ledger";
           $result = $this->db->query($sql)->row();
           
           $total_debits = floatval($result->total_debits ?? 0);
           $total_credits = floatval($result->total_credits ?? 0);
           $difference = abs($total_debits - $total_credits);
           
           $consistency_checks['trial_balance'] = array(
               'total_debits' => $total_debits,
               'total_credits' => $total_credits,
               'difference' => $difference,
               'is_balanced' => $difference < 0.01
           );
       }
       
       return $consistency_checks;
   }

   private function getSlowQueryCount() {
       try {
           $sql = "SHOW GLOBAL STATUS LIKE 'Slow_queries'";
           $result = $this->db->query($sql)->row();
           return intval($result->Value ?? 0);
       } catch (Exception $e) {
           return 0;
       }
   }

   private function checkIndexUsage() {
       // This would require more complex analysis
       return array('status' => 'requires_detailed_analysis');
   }

   private function checkTableLocks() {
       try {
           $sql = "SHOW GLOBAL STATUS LIKE 'Table_locks_waited'";
           $result = $this->db->query($sql)->row();
           return intval($result->Value ?? 0);
       } catch (Exception $e) {
           return 0;
       }
   }

   private function generateHealthRecommendations($health_status) {
       $recommendations = array();
       
       // Check for missing tables
       if (!empty($health_status['database_structure']['missing_tables'])) {
           $recommendations[] = array(
               'priority' => 'High',
               'category' => 'Database Structure',
               'recommendation' => 'Initialize missing financial tables for enhanced reporting',
               'action' => 'Run initialize_enhanced_system() method'
           );
       }
       
       // Check data integrity issues
       if (isset($health_status['data_integrity']['orphaned_opd_records']) && 
           $health_status['data_integrity']['orphaned_opd_records'] > 0) {
           $recommendations[] = array(
               'priority' => 'Medium',
               'category' => 'Data Integrity',
               'recommendation' => 'Clean up orphaned OPD records',
               'action' => 'Review and fix patient record references'
           );
       }
       
       // Check financial consistency
       if (isset($health_status['financial_consistency']['trial_balance']) && 
           !$health_status['financial_consistency']['trial_balance']['is_balanced']) {
           $recommendations[] = array(
               'priority' => 'High',
               'category' => 'Financial Integrity',
               'recommendation' => 'Investigate trial balance discrepancy',
               'action' => 'Review general ledger entries for errors'
           );
       }
       
       return $recommendations;
   }

   private function calculateOverallHealth($health_status) {
       $score = 100;
       
       // Deduct points for issues
       if (!empty($health_status['database_structure']['missing_tables'])) {
           $score -= 20;
       }
       
       if (isset($health_status['data_integrity']['orphaned_opd_records']) && 
           $health_status['data_integrity']['orphaned_opd_records'] > 0) {
           $score -= 10;
       }
       
       if (isset($health_status['financial_consistency']['trial_balance']) && 
           !$health_status['financial_consistency']['trial_balance']['is_balanced']) {
           $score -= 30;
       }
       
       $score = max(0, $score);
       
       if ($score >= 90) return array('score' => $score, 'status' => 'Excellent');
       if ($score >= 75) return array('score' => $score, 'status' => 'Good');
       if ($score >= 60) return array('score' => $score, 'status' => 'Fair');
       return array('score' => $score, 'status' => 'Needs Attention');
   }
}
?>