<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Financial extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load models
        $this->load->model('financial_model');
        $this->load->model('patient_model');
        $this->load->library('datatables');
        $this->load->library('customlib');
        
        // Initialize financial system if not done
        $this->initializeFinancialSystem();
    }

    private function initializeFinancialSystem() {
        if (!$this->financial_model->isSystemSetup()) {
            try {
                $this->financial_model->createFinancialTables();
            } catch (Exception $e) {
                log_message('error', 'Financial system auto-initialization failed: ' . $e->getMessage());
            }
        }
    }

    // =============================================
    // MAIN DASHBOARD
    // =============================================

    public function index() {
        redirect('admin/financial/dashboard');
    }

    public function dashboard() {
        // Check permissions
        if (!$this->rbac->hasPrivilege('financial_dashboard', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/dashboard');

        try {
            // Get financial metrics
            $data['metrics'] = $this->getFinancialMetrics();
            
            // Get recent transactions
            $data['recent_transactions'] = $this->financial_model->getRecentTransactions(10);
            
            // Get pending approvals
            $data['pending_approvals'] = $this->financial_model->getPendingApprovals();
            
            // Get monthly revenue data for chart
            $data['monthly_revenue_data'] = $this->financial_model->getMonthlyRevenue(12);

            $this->load->view('layout/header', $data);
            $this->load->view('admin/financial/dashboard', $data);
            $this->load->view('layout/footer', $data);
            
        } catch (Exception $e) {
            log_message('error', 'Financial Dashboard Error: ' . $e->getMessage());
            
            // Load with default/empty data
            $data['metrics'] = $this->getDefaultMetrics();
            $data['recent_transactions'] = array();
            $data['pending_approvals'] = array();
            $data['monthly_revenue_data'] = array();

            $this->load->view('layout/header', $data);
            $this->load->view('admin/financial/dashboard', $data);
            $this->load->view('layout/footer', $data);
        }
    }

    // =============================================
    // CHART OF ACCOUNTS
    // =============================================

    public function chartofaccounts() {
        if (!$this->rbac->hasPrivilege('chart_of_accounts', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/chartofaccounts');

        $data['title'] = 'Chart of Accounts';
        $data['accounts'] = $this->financial_model->getAllAccounts();
        $data['account_types'] = array(
            'asset' => 'Assets',
            'liability' => 'Liabilities', 
            'equity' => 'Equity',
            'revenue' => 'Revenue',
            'expense' => 'Expenses'
        );

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/chartofaccounts', $data);
        $this->load->view('layout/footer', $data);
    }

    public function addAccount() {
        if (!$this->rbac->hasPrivilege('chart_of_accounts', 'can_add')) {
            access_denied();
        }

        if ($this->input->post('submit')) {
            $account_data = array(
                'account_code' => $this->input->post('account_code'),
                'account_name' => $this->input->post('account_name'),
                'account_type' => $this->input->post('account_type'),
                'parent_id' => $this->input->post('parent_id') ?: null,
                'is_active' => 'yes'
            );

            try {
                if ($this->db->table_exists('chart_of_accounts')) {
                    $this->db->insert('chart_of_accounts', $account_data);
                    $this->session->set_flashdata('msg', '<div class="alert alert-success">Account added successfully!</div>');
                } else {
                    throw new Exception('Chart of accounts table does not exist');
                }
            } catch (Exception $e) {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error adding account: ' . $e->getMessage() . '</div>');
            }

            redirect('admin/financial/chartofaccounts');
        }

        $data['title'] = 'Add Account';
        $data['accounts'] = $this->financial_model->getAllAccounts();
        $data['account_types'] = array(
            'asset' => 'Assets',
            'liability' => 'Liabilities', 
            'equity' => 'Equity',
            'revenue' => 'Revenue',
            'expense' => 'Expenses'
        );

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/add_account', $data);
        $this->load->view('layout/footer', $data);
    }

    public function editAccount($id) {
        if (!$this->rbac->hasPrivilege('chart_of_accounts', 'can_edit')) {
            access_denied();
        }

        if ($this->input->post('submit')) {
            $account_data = array(
                'account_code' => $this->input->post('account_code'),
                'account_name' => $this->input->post('account_name'),
                'account_type' => $this->input->post('account_type'),
                'parent_id' => $this->input->post('parent_id') ?: null,
                'is_active' => $this->input->post('is_active')
            );

            try {
                $this->db->where('id', $id);
                $this->db->update('chart_of_accounts', $account_data);
                $this->session->set_flashdata('msg', '<div class="alert alert-success">Account updated successfully!</div>');
            } catch (Exception $e) {
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error updating account: ' . $e->getMessage() . '</div>');
            }

            redirect('admin/financial/chartofaccounts');
        }

        $data['title'] = 'Edit Account';
        $data['account'] = $this->getAccountById($id);
        $data['accounts'] = $this->financial_model->getAllAccounts();
        $data['account_types'] = array(
            'asset' => 'Assets',
            'liability' => 'Liabilities', 
            'equity' => 'Equity',
            'revenue' => 'Revenue',
            'expense' => 'Expenses'
        );

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/edit_account', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // JOURNAL ENTRIES
    // =============================================

    public function journalentries() {
        if (!$this->rbac->hasPrivilege('journal_entries', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/journalentries');

        $data['title'] = 'Journal Entries';
        
        // Get filter parameters
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        $status = $this->input->get('status') ?: 'all';

        $data['journal_entries'] = $this->getJournalEntries($date_from, $date_to, $status);
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        $data['status'] = $status;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/journalentries', $data);
        $this->load->view('layout/footer', $data);
    }

    public function addJournalEntry() {
        if (!$this->rbac->hasPrivilege('journal_entries', 'can_add')) {
            access_denied();
        }

        if ($this->input->post('submit')) {
            $this->db->trans_start();
            
            try {
                // Insert journal entry header
                $journal_data = array(
                    'reference_no' => $this->generateJournalReference(),
                    'date' => $this->input->post('date'),
                    'description' => $this->input->post('description'),
                    'total_amount' => $this->input->post('total_amount'),
                    'status' => 'draft',
                    'created_by' => $this->customlib->getStaffID()
                );

                $this->db->insert('journal_entries', $journal_data);
                $journal_entry_id = $this->db->insert_id();

                // Insert journal entry lines
                $accounts = $this->input->post('account_id');
                $descriptions = $this->input->post('line_description');
                $debits = $this->input->post('debit_amount');
                $credits = $this->input->post('credit_amount');

                for ($i = 0; $i < count($accounts); $i++) {
                    if ($accounts[$i] && ($debits[$i] > 0 || $credits[$i] > 0)) {
                        $line_data = array(
                            'journal_entry_id' => $journal_entry_id,
                            'account_id' => $accounts[$i],
                            'description' => $descriptions[$i],
                            'debit_amount' => $debits[$i] ?: 0,
                            'credit_amount' => $credits[$i] ?: 0
                        );
                        $this->db->insert('journal_entry_lines', $line_data);
                    }
                }

                $this->db->trans_complete();

                if ($this->db->trans_status() === FALSE) {
                    throw new Exception('Transaction failed');
                }

                $this->session->set_flashdata('msg', '<div class="alert alert-success">Journal entry added successfully!</div>');
                redirect('admin/financial/journalentries');

            } catch (Exception $e) {
                $this->db->trans_rollback();
                $this->session->set_flashdata('msg', '<div class="alert alert-danger">Error adding journal entry: ' . $e->getMessage() . '</div>');
            }
        }

        $data['title'] = 'Add Journal Entry';
        $data['accounts'] = $this->financial_model->getAllAccounts();
        $data['next_reference'] = $this->generateJournalReference();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/add_journal_entry', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // GENERAL LEDGER
    // =============================================

  /*  public function generalledger() {
    if (!$this->rbac->hasPrivilege('general_ledger', 'can_view')) {
        access_denied();
    }

    $account_id = $this->input->get('account_id');
    $date_from = $this->input->get('date_from') ?: date('Y-01-01');
    $date_to = $this->input->get('date_to') ?: date('Y-m-d');

    if (!$account_id) {
        show_404();
    }

    // Get account information
    $account = $this->getAccountById($account_id);
    if (!$account) {
        show_404();
    }

    // Get ledger entries for this account
    $ledger_entries = $this->financial_model->getLedgerEntries($account_id, $date_from, $date_to);

    $data = array(
        'title' => 'General Ledger - ' . $account['account_name'],
        'account' => $account,
        'ledger_entries' => $ledger_entries,
        'date_from' => $date_from,
        'date_to' => $date_to
    );

    $this->load->view('layout/header', $data);
    $this->load->view('admin/financial/generalledger', $data);
    $this->load->view('layout/footer', $data);
}*/



/**
 * AJAX endpoint for quick financial summary
 */
public function getFinancialSummary() {
    if (!$this->rbac->hasPrivilege('profit_loss', 'can_view')) {
        echo json_encode(array('error' => 'Access denied'));
        return;
    }

    $date_from = $this->input->post('date_from') ?: date('Y-01-01');
    $date_to = $this->input->post('date_to') ?: date('Y-m-d');

    $income_statement = $this->financial_model->getIncomeStatement($date_from, $date_to);
    
    $total_revenue = 0;
    $total_expenses = 0;
    
    if (isset($income_statement['revenue'])) {
        foreach ($income_statement['revenue'] as $revenue_item) {
            $total_revenue += $revenue_item['amount'];
        }
    }
    
    if (isset($income_statement['expenses'])) {
        foreach ($income_statement['expenses'] as $expense_item) {
            $total_expenses += $expense_item['amount'];
        }
    }
    
    $net_income = $total_revenue - $total_expenses;
    $profit_margin = $total_revenue > 0 ? ($net_income / $total_revenue) * 100 : 0;

    $response = array(
        'total_revenue' => $total_revenue,
        'total_expenses' => $total_expenses,
        'net_income' => $net_income,
        'profit_margin' => round($profit_margin, 2),
        'revenue_count' => count($income_statement['revenue']),
        'expense_count' => count($income_statement['expenses'])
    );

    echo json_encode($response);
}

    // =============================================
    // FINANCIAL REPORTS
    // =============================================

    public function trialbalance() {
        if (!$this->rbac->hasPrivilege('trial_balance', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/trialbalance');

        $date = $this->input->get('date') ?: date('Y-m-d');
        
        $data['title'] = 'Trial Balance';
        $data['trial_balance'] = $this->financial_model->getTrialBalance($date);
        $data['date'] = $date;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/trialbalance', $data);
        $this->load->view('layout/footer', $data);
    }

public function balancesheet() {
    if (!$this->rbac->hasPrivilege('balance_sheet', 'can_view')) {
        access_denied();
    }

    $this->session->set_userdata('top_menu', 'Financial');
    $this->session->set_userdata('sub_menu', 'financial/balancesheet');

    $as_of_date = $this->input->get('as_of_date') ?: date('Y-m-d');
    
    try {
        // Get balance sheet data
        $balance_sheet_data = $this->financial_model->getBalanceSheet($as_of_date);
        
        $data['title'] = 'Balance Sheet';
        $data['as_of_date'] = $as_of_date;
        
        // Prepare assets data - Filter out zero balances for display
        $data['assets'] = array();
        $data['total_assets'] = $balance_sheet_data['total_assets'];
        
        foreach ($balance_sheet_data['asset_accounts'] as $asset) {
            if ($asset['amount'] != 0) { // Only show non-zero balances
                $data['assets'][] = array(
                    'id' => $this->getAccountIdByCode($asset['account_code']),
                    'account_code' => $asset['account_code'],
                    'account_name' => $asset['account_name'],
                    'balance' => $asset['amount']
                );
            }
        }
        
        // Prepare liabilities data
        $data['liabilities'] = array();
        $data['total_liabilities'] = $balance_sheet_data['total_liabilities'];
        
        foreach ($balance_sheet_data['liability_accounts'] as $liability) {
            if ($liability['amount'] != 0) {
                $data['liabilities'][] = array(
                    'id' => $this->getAccountIdByCode($liability['account_code']),
                    'account_code' => $liability['account_code'],
                    'account_name' => $liability['account_name'],
                    'balance' => $liability['amount']
                );
            }
        }
        
        // Prepare equity data
        $data['equity'] = array();
        $data['total_equity'] = $balance_sheet_data['total_equity'];
        
        foreach ($balance_sheet_data['equity_accounts'] as $equity_item) {
            if ($equity_item['amount'] != 0) {
                $data['equity'][] = array(
                    'id' => $this->getAccountIdByCode($equity_item['account_code']),
                    'account_code' => $equity_item['account_code'],
                    'account_name' => $equity_item['account_name'],
                    'balance' => $equity_item['amount']
                );
            }
        }

        // Debug output (remove this after testing)
        log_message('debug', 'Balance Sheet Data: ' . print_r($data, true));

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/balancesheet', $data);
        $this->load->view('layout/footer', $data);
        
    } catch (Exception $e) {
        log_message('error', 'Balance Sheet Error: ' . $e->getMessage());
        
        $data['error_message'] = 'Error generating balance sheet: ' . $e->getMessage();
        $data['title'] = 'Balance Sheet';
        $data['as_of_date'] = $as_of_date;
        $data['assets'] = array();
        $data['liabilities'] = array();
        $data['equity'] = array();
        $data['total_assets'] = 0;
        $data['total_liabilities'] = 0;
        $data['total_equity'] = 0;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/balancesheet', $data);
        $this->load->view('layout/footer', $data);
    }
}

// Helper method to get account ID by code
private function getAccountIdByCode($account_code) {
    if ($this->db->table_exists('chart_of_accounts')) {
        $this->db->select('id');
        $this->db->where('account_code', $account_code);
        $result = $this->db->get('chart_of_accounts')->row();
        return $result ? $result->id : null;
    }
    return null;
}

public function debug_balance_sheet() {
    if (!$this->rbac->hasPrivilege('balance_sheet', 'can_view')) {
        access_denied();
    }
    
    echo "<h1>Balance Sheet Debug - Updated</h1>";
    
    // Check if accounts exist with correct query
    echo "<h2>Chart of Accounts</h2>";
    
    // Direct query to match your schema
    $query = "SELECT id, account_code, account_name, account_type, parent_account_id, is_active, opening_balance, balance_type FROM chart_of_accounts WHERE is_active = 1 ORDER BY account_code";
    $accounts = $this->db->query($query)->result_array();
    
    echo "Total accounts: " . count($accounts) . "<br>";
    
    if (!empty($accounts)) {
        echo "<table border='1' style='border-collapse: collapse;'>";
        echo "<tr><th>ID</th><th>Code</th><th>Name</th><th>Type</th><th>Opening Balance</th><th>Balance Type</th></tr>";
        foreach ($accounts as $account) {
            echo "<tr>";
            echo "<td>" . $account['id'] . "</td>";
            echo "<td>" . $account['account_code'] . "</td>";
            echo "<td>" . $account['account_name'] . "</td>";
            echo "<td>" . $account['account_type'] . "</td>";
            echo "<td>" . number_format($account['opening_balance'], 2) . "</td>";
            echo "<td>" . $account['balance_type'] . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Test balance sheet data
        echo "<h2>Balance Sheet Test</h2>";
        $balance_sheet = $this->financial_model->getBalanceSheet(date('Y-m-d'));
        
        echo "<h3>Assets</h3>";
        if (!empty($balance_sheet['asset_accounts'])) {
            foreach ($balance_sheet['asset_accounts'] as $asset) {
                echo $asset['account_code'] . " - " . $asset['account_name'] . ": " . number_format($asset['amount'], 2) . "<br>";
            }
            echo "Total Assets: " . number_format($balance_sheet['total_assets'], 2) . "<br>";
        } else {
            echo "No assets found<br>";
        }
        
        echo "<h3>Liabilities</h3>";
        if (!empty($balance_sheet['liability_accounts'])) {
            foreach ($balance_sheet['liability_accounts'] as $liability) {
                echo $liability['account_code'] . " - " . $liability['account_name'] . ": " . number_format($liability['amount'], 2) . "<br>";
            }
            echo "Total Liabilities: " . number_format($balance_sheet['total_liabilities'], 2) . "<br>";
        } else {
            echo "No liabilities found<br>";
        }
        
        echo "<h3>Equity</h3>";
        if (!empty($balance_sheet['equity_accounts'])) {
            foreach ($balance_sheet['equity_accounts'] as $equity) {
                echo $equity['account_code'] . " - " . $equity['account_name'] . ": " . number_format($equity['amount'], 2) . "<br>";
            }
            echo "Total Equity: " . number_format($balance_sheet['total_equity'], 2) . "<br>";
        } else {
            echo "No equity found<br>";
        }
    }
    
    // Check table structure
    echo "<h2>Table Structure Check</h2>";
    $fields = $this->db->field_data('chart_of_accounts');
    echo "Chart of Accounts fields:<br>";
    foreach ($fields as $field) {
        echo "- " . $field->name . " (" . $field->type . ")<br>";
    }
}

  /**
 * Fixed profitloss method for Financial Controller
 */

public function profitloss() {
    if (!$this->rbac->hasPrivilege('profit_loss', 'can_view')) {
        access_denied();
    }

    $this->session->set_userdata('top_menu', 'Financial');
    $this->session->set_userdata('sub_menu', 'financial/profitloss');

    $data['title'] = 'Profit & Loss Statement';
    
    // Get filter parameters
    $date_from = $this->input->get('date_from') ?: date('Y-01-01');
    $date_to = $this->input->get('date_to') ?: date('Y-m-d');
    
    $data['date_from'] = $date_from;
    $data['date_to'] = $date_to;
    
    // Try to get P&L data using the new safe method
    try {
        // First, let's see what data is available
        $available_data = $this->financial_model->checkAvailableData($date_from, $date_to);
        $data['available_data'] = $available_data;
        
        // Try to get data from journal entries first
        $data['income_statement'] = $this->financial_model->getProfitLossData($date_from, $date_to);
        
        // If no journal data, try fallback to billing tables
        if (empty($data['income_statement']['revenue_accounts']) && empty($data['income_statement']['expense_accounts'])) {
            $data['income_statement'] = $this->financial_model->getProfitLossDataFallback($date_from, $date_to);
        }
        
    } catch (Exception $e) {
        log_message('error', 'Error in profitloss controller: ' . $e->getMessage());
        
        // Emergency fallback
        $data['income_statement'] = array(
            'revenue_accounts' => array(),
            'expense_accounts' => array(),
            'total_revenue' => 0,
            'total_expenses' => 0,
            'net_income' => 0,
            'error' => $e->getMessage()
        );
    }

    $this->load->view('layout/header', $data);
    $this->load->view('admin/financial/profitloss', $data);
    $this->load->view('layout/footer', $data);
}

// Add a debug method to check what's happening
public function debugPL() {
    $date_from = $this->input->get('date_from') ?: date('Y-01-01');
    $date_to = $this->input->get('date_to') ?: date('Y-m-d');
    
    echo "<h2>Profit & Loss Debug Information</h2>";
    echo "<p>Date Range: {$date_from} to {$date_to}</p>";
    
    // Check available data
    echo "<h3>Available Data Check:</h3>";
    $available_data = $this->financial_model->checkAvailableData($date_from, $date_to);
    echo "<pre>" . print_r($available_data, true) . "</pre>";
    
    // Test fallback method
    echo "<h3>Fallback Data Test:</h3>";
    $fallback_data = $this->financial_model->getProfitLossDataFallback($date_from, $date_to);
    echo "<pre>" . print_r($fallback_data, true) . "</pre>";
    
    echo "<p><a href='" . base_url() . "admin/financial/profitloss'>Back to Profit & Loss</a></p>";
}

/**
 * Alternative version without is_active filter (for testing)
 */
public function getIncomeStatementNoFilter($date_from, $date_to) {
    $income_statement = array(
        'revenue' => array(),
        'expenses' => array()
    );

    if (!$this->db->table_exists('chart_of_accounts') || 
        !$this->db->table_exists('journal_entries') || 
        !$this->db->table_exists('journal_entry_lines')) {
        return $income_statement;
    }

    try {
        // Revenue Query - NO is_active filter
        $revenue_query = "
            SELECT 
                coa.id,
                coa.account_code,
                coa.account_name,
                coa.account_type,
                coa.is_active,
                COALESCE(SUM(jel.credit_amount - jel.debit_amount), 0) as amount
            FROM chart_of_accounts coa
            INNER JOIN journal_entry_lines jel ON jel.account_id = coa.id
            INNER JOIN journal_entries je ON je.id = jel.journal_entry_id
            WHERE coa.account_type = 'revenue'
            AND je.date >= ?
            AND je.date <= ?
            GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.is_active
            HAVING amount != 0
            ORDER BY coa.account_code
        ";

        $revenue_result = $this->db->query($revenue_query, array($date_from, $date_to));
        if ($revenue_result && $revenue_result !== false) {
            $income_statement['revenue'] = $revenue_result->result_array();
        }

        // Expense Query - NO is_active filter
        $expense_query = "
            SELECT 
                coa.id,
                coa.account_code,
                coa.account_name,
                coa.account_type,
                coa.is_active,
                COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as amount
            FROM chart_of_accounts coa
            INNER JOIN journal_entry_lines jel ON jel.account_id = coa.id
            INNER JOIN journal_entries je ON je.id = jel.journal_entry_id
            WHERE coa.account_type = 'expense'
            AND je.date >= ?
            AND je.date <= ?
            GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.is_active
            HAVING amount != 0
            ORDER BY coa.account_code
        ";

        $expense_result = $this->db->query($expense_query, array($date_from, $date_to));
        if ($expense_result && $expense_result !== false) {
            $income_statement['expenses'] = $expense_result->result_array();
        }

        return $income_statement;

    } catch (Exception $e) {
        log_message('error', 'Error getting income statement (no filter): ' . $e->getMessage());
        return $income_statement;
    }
}

/**
 * Debug version to see what's causing the journal entry creation to fail
 */
public function debugJournalCreation() {
    if (!$this->rbac->hasPrivilege('journal_entries', 'can_add')) {
        access_denied();
    }
    
    echo "<h2>Debug Journal Entry Creation</h2>";
    
    try {
        // First, let's examine the journal_entries table structure
        echo "<h3>1. Journal Entries Table Structure</h3>";
        $fields = $this->db->field_data('journal_entries');
        echo "<table border='1' style='border-collapse: collapse;'>";
        echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        foreach ($fields as $field) {
            echo "<tr>";
            echo "<td>{$field->name}</td>";
            echo "<td>{$field->type}</td>";
            echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
            echo "<td>{$field->primary_key}</td>";
            echo "<td>" . ($field->default ?? 'NULL') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Check journal_entry_lines table structure
        echo "<h3>2. Journal Entry Lines Table Structure</h3>";
        $jel_fields = $this->db->field_data('journal_entry_lines');
        echo "<table border='1' style='border-collapse: collapse;'>";
        echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
        foreach ($jel_fields as $field) {
            echo "<tr>";
            echo "<td>{$field->name}</td>";
            echo "<td>{$field->type}</td>";
            echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
            echo "<td>{$field->primary_key}</td>";
            echo "<td>" . ($field->default ?? 'NULL') . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Get some sample accounts
        echo "<h3>3. Sample Accounts</h3>";
        $this->db->select('id, account_code, account_name, account_type');
        $this->db->from('chart_of_accounts');
        $this->db->where('is_active', 1);
        $this->db->limit(10);
        $accounts = $this->db->get()->result_array();
        
        echo "<table border='1' style='border-collapse: collapse;'>";
        echo "<tr><th>ID</th><th>Code</th><th>Name</th><th>Type</th></tr>";
        foreach ($accounts as $account) {
            echo "<tr>";
            echo "<td>{$account['id']}</td>";
            echo "<td>{$account['account_code']}</td>";
            echo "<td>{$account['account_name']}</td>";
            echo "<td>{$account['account_type']}</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        // Try to create a simple journal entry with detailed error reporting
        echo "<h3>4. Testing Journal Entry Creation</h3>";
        
        if (count($accounts) >= 2) {
            $cash_account = $accounts[0];
            $other_account = $accounts[1];
            
            echo "<p>Using accounts:</p>";
            echo "<p>Account 1: {$cash_account['account_code']} - {$cash_account['account_name']} (ID: {$cash_account['id']})</p>";
            echo "<p>Account 2: {$other_account['account_code']} - {$other_account['account_name']} (ID: {$other_account['id']})</p>";
            
            // Prepare journal entry data
            $je_data = array(
                'reference_no' => 'TEST-JE-001',
                'date' => date('Y-m-d'),
                'description' => 'Test journal entry for debugging',
                'total_amount' => 1000.00,
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            // Check if status field exists and add it
            $has_status = false;
            foreach ($fields as $field) {
                if ($field->name == 'status') {
                    $has_status = true;
                    break;
                }
            }
            
            if ($has_status) {
                $je_data['status'] = 'approved';
                echo "<p>Added status field: approved</p>";
            }
            
            // Check if created_by field exists
            $has_created_by = false;
            foreach ($fields as $field) {
                if ($field->name == 'created_by') {
                    $has_created_by = true;
                    break;
                }
            }
            
            if ($has_created_by) {
                $je_data['created_by'] = 1; // Default user ID
                echo "<p>Added created_by field: 1</p>";
            }
            
            echo "<h4>Journal Entry Data to Insert:</h4>";
            echo "<pre>" . print_r($je_data, true) . "</pre>";
            
            // First, delete any existing test entry
            $this->db->where('reference_no', 'TEST-JE-001');
            $this->db->delete('journal_entries');
            
            // Try to insert
            echo "<h4>Attempting to insert journal entry...</h4>";
            $result = $this->db->insert('journal_entries', $je_data);
            
            if ($result) {
                $journal_entry_id = $this->db->insert_id();
                echo "<p style='color: green;'>✓ Journal entry created successfully! ID: {$journal_entry_id}</p>";
                
                // Now try to create journal entry lines
                echo "<h4>Creating journal entry lines...</h4>";
                
                $line1_data = array(
                    'journal_entry_id' => $journal_entry_id,
                    'account_id' => $cash_account['id'],
                    'description' => 'Test debit entry',
                    'debit_amount' => 1000.00,
                    'credit_amount' => 0.00,
                    'created_at' => date('Y-m-d H:i:s')
                );
                
                $line2_data = array(
                    'journal_entry_id' => $journal_entry_id,
                    'account_id' => $other_account['id'],
                    'description' => 'Test credit entry',
                    'debit_amount' => 0.00,
                    'credit_amount' => 1000.00,
                    'created_at' => date('Y-m-d H:i:s')
                );
                
                echo "<p>Line 1 data:</p><pre>" . print_r($line1_data, true) . "</pre>";
                echo "<p>Line 2 data:</p><pre>" . print_r($line2_data, true) . "</pre>";
                
                $line1_result = $this->db->insert('journal_entry_lines', $line1_data);
                if ($line1_result) {
                    echo "<p style='color: green;'>✓ Line 1 created successfully!</p>";
                } else {
                    echo "<p style='color: red;'>✗ Line 1 failed!</p>";
                    $error = $this->db->error();
                    echo "<p>Error: " . $error['message'] . "</p>";
                }
                
                $line2_result = $this->db->insert('journal_entry_lines', $line2_data);
                if ($line2_result) {
                    echo "<p style='color: green;'>✓ Line 2 created successfully!</p>";
                } else {
                    echo "<p style='color: red;'>✗ Line 2 failed!</p>";
                    $error = $this->db->error();
                    echo "<p>Error: " . $error['message'] . "</p>";
                }
                
                if ($line1_result && $line2_result) {
                    echo "<div style='background: #d4edda; padding: 15px; border: 1px solid #c3e6cb; border-radius: 5px;'>";
                    echo "<h4 style='color: #155724;'>Success!</h4>";
                    echo "<p>Test journal entry created successfully. The issue was likely with the bulk creation process.</p>";
                    echo "<p><a href='" . base_url() . "admin/financial/createWorkingJournalEntries'>Create Working Journal Entries</a></p>";
                    echo "</div>";
                }
                
            } else {
                echo "<p style='color: red;'>✗ Failed to create journal entry!</p>";
                $error = $this->db->error();
                echo "<p>Database Error: " . $error['message'] . "</p>";
                echo "<p>Error Code: " . $error['code'] . "</p>";
                echo "<p>Last Query: " . $this->db->last_query() . "</p>";
            }
            
        } else {
            echo "<p style='color: red;'>Not enough accounts found for testing.</p>";
        }
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>Exception: " . $e->getMessage() . "</p>";
        echo "<p>File: " . $e->getFile() . "</p>";
        echo "<p>Line: " . $e->getLine() . "</p>";
    }
}

/**
 * Create working journal entries (after debugging)
 */
public function createWorkingJournalEntries() {
    if (!$this->rbac->hasPrivilege('journal_entries', 'can_add')) {
        access_denied();
    }
    
    echo "<h2>Creating Working Journal Entries</h2>";
    
    try {
        // Get accounts
        $this->db->select('id, account_code, account_name, account_type');
        $this->db->from('chart_of_accounts');
        $this->db->where('is_active', 1);
        $accounts_result = $this->db->get()->result_array();
        
        // Organize by type
        $accounts_by_type = array();
        foreach ($accounts_result as $account) {
            $accounts_by_type[$account['account_type']][] = $account;
        }
        
        // Find accounts we need
        $cash_account = null;
        foreach ($accounts_by_type['asset'] as $asset) {
            if (stripos($asset['account_name'], 'cash') !== false) {
                $cash_account = $asset;
                break;
            }
        }
        if (!$cash_account) $cash_account = $accounts_by_type['asset'][0];
        
        $revenue_account = $accounts_by_type['revenue'][0];
        $expense_account = $accounts_by_type['expense'][0];
        
        echo "<p>Using:</p>";
        echo "<p>Cash: {$cash_account['account_code']} - {$cash_account['account_name']}</p>";
        echo "<p>Revenue: {$revenue_account['account_code']} - {$revenue_account['account_name']}</p>";
        echo "<p>Expense: {$expense_account['account_code']} - {$expense_account['account_name']}</p>";
        
        // Create a few simple entries
        $entries = array(
            array(
                'ref' => 'JE-2025-001',
                'date' => '2025-01-15',
                'desc' => 'Patient consultation fees',
                'amount' => 15000,
                'debit_account' => $cash_account,
                'credit_account' => $revenue_account
            ),
            array(
                'ref' => 'JE-2025-002',
                'date' => '2025-02-10',
                'desc' => 'Medical supplies expense',
                'amount' => 8000,
                'debit_account' => $expense_account,
                'credit_account' => $cash_account
            ),
            array(
                'ref' => 'JE-2025-003',
                'date' => '2025-03-05',
                'desc' => 'Laboratory service revenue',
                'amount' => 22000,
                'debit_account' => $cash_account,
                'credit_account' => $revenue_account
            )
        );
        
        $created = 0;
        
        foreach ($entries as $entry) {
            // Check if exists
            $this->db->where('reference_no', $entry['ref']);
            $existing = $this->db->get('journal_entries')->row();
            
            if ($existing) {
                echo "<p>- Skipping {$entry['ref']} (already exists)</p>";
                continue;
            }
            
            // Create journal entry
            $je_data = array(
                'reference_no' => $entry['ref'],
                'date' => $entry['date'],
                'description' => $entry['desc'],
                'total_amount' => $entry['amount'],
                'created_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            // Add status if column exists
            $fields = $this->db->field_data('journal_entries');
            foreach ($fields as $field) {
                if ($field->name == 'status') {
                    $je_data['status'] = 'approved';
                    break;
                }
            }
            
            if ($this->db->insert('journal_entries', $je_data)) {
                $je_id = $this->db->insert_id();
                
                // Create lines
                $line1 = array(
                    'journal_entry_id' => $je_id,
                    'account_id' => $entry['debit_account']['id'],
                    'description' => $entry['desc'],
                    'debit_amount' => $entry['amount'],
                    'credit_amount' => 0,
                    'created_at' => date('Y-m-d H:i:s')
                );
                
                $line2 = array(
                    'journal_entry_id' => $je_id,
                    'account_id' => $entry['credit_account']['id'],
                    'description' => $entry['desc'],
                    'debit_amount' => 0,
                    'credit_amount' => $entry['amount'],
                    'created_at' => date('Y-m-d H:i:s')
                );
                
                if ($this->db->insert('journal_entry_lines', $line1) && 
                    $this->db->insert('journal_entry_lines', $line2)) {
                    echo "<p style='color: green;'>✓ Created {$entry['ref']} - {$entry['desc']} - KES " . number_format($entry['amount']) . "</p>";
                    $created++;
                } else {
                    echo "<p style='color: red;'>✗ Failed to create lines for {$entry['ref']}</p>";
                }
            } else {
                echo "<p style='color: red;'>✗ Failed to create {$entry['ref']}</p>";
                $error = $this->db->error();
                echo "<p>Error: " . $error['message'] . "</p>";
            }
        }
        
        echo "<hr>";
        echo "<h3>Summary: Created {$created} journal entries</h3>";
        
        if ($created > 0) {
            echo "<div style='background: #d4edda; padding: 15px; border: 1px solid #c3e6cb; border-radius: 5px;'>";
            echo "<h4>Success!</h4>";
            echo "<p>Journal entries created successfully. You can now:</p>";
            echo "<ul>";
            echo "<li><a href='" . base_url() . "admin/financial/profitloss'>View Profit & Loss Statement</a></li>";
            echo "<li><a href='" . base_url() . "admin/financial/journalentries'>View Journal Entries</a></li>";
            echo "</ul>";
            echo "</div>";
        }
        
    } catch (Exception $e) {
        echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
    }
}

    public function cashflow() {
        if (!$this->rbac->hasPrivilege('cash_flow', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/cashflow');

        $date_from = $this->input->get('date_from') ?: date('Y-01-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        $data['title'] = 'Cash Flow Statement';
        $data['cash_flow'] = $this->financial_model->getCashFlow($date_from, $date_to);
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/cashflow', $data);
        $this->load->view('layout/footer', $data);
    }

 public function accountsreceivable() {
    if (!$this->rbac->hasPrivilege('accounts_receivable', 'can_view')) {
        access_denied();
    }

    $this->session->set_userdata('top_menu', 'Financial');
    $this->session->set_userdata('sub_menu', 'financial/accountsreceivable');

    $data['title'] = 'Accounts Receivable Report';
    
    // Get accounts receivable data (money customers owe to us)
    $data['receivables'] = $this->getAccountsReceivableData();
    $data['aging'] = $this->getReceivableAgingData();
    $data['recent_bills'] = $this->getRecentPatientBills();

    $this->load->view('layout/header', $data);
    $this->load->view('admin/financial/accountsreceivable', $data);
    $this->load->view('layout/footer', $data);
}

private function getAccountsReceivableData() {
    // Get outstanding patient balances from all billing modules
    $sql = "SELECT 
    p.id as patient_id,
    p.patient_name,
    p.patient_unique_id,
    p.mobileno,
    p.email,
    COALESCE(opd_outstanding.balance, 0) + 
    COALESCE(ipd_outstanding.balance, 0) + 
    COALESCE(pharmacy_outstanding.balance, 0) + 
    COALESCE(pathology_outstanding.balance, 0) + 
    COALESCE(radiology_outstanding.balance, 0) as balance,
    GREATEST(
        COALESCE(opd_outstanding.last_date, '1900-01-01'),
        COALESCE(ipd_outstanding.last_date, '1900-01-01'),
        COALESCE(pharmacy_outstanding.last_date, '1900-01-01'),
        COALESCE(pathology_outstanding.last_date, '1900-01-01'),
        COALESCE(radiology_outstanding.last_date, '1900-01-01')
    ) as last_transaction_date
FROM patients p

-- OPD Outstanding (using patient_charges for bills and transactions for payments)
LEFT JOIN (
    SELECT 
        od.patient_id,
        SUM(COALESCE(pc.apply_charge, 0)) - COALESCE(opd_payments.total_paid, 0) as balance,
        MAX(COALESCE(vd.appointment_date, od.created_at)) as last_date
    FROM opd_details od
    LEFT JOIN patient_charges pc ON pc.opd_id = od.id
    LEFT JOIN visit_details vd ON vd.opd_details_id = od.id
    LEFT JOIN (
        SELECT 
            od2.patient_id,
            SUM(t.amount) as total_paid
        FROM transactions t
        JOIN opd_details od2 ON od2.id = t.opd_id
        WHERE t.type = 'payment'
        GROUP BY od2.patient_id
    ) opd_payments ON od.patient_id = opd_payments.patient_id
    GROUP BY od.patient_id
    HAVING balance > 0
) opd_outstanding ON p.id = opd_outstanding.patient_id

-- IPD Outstanding (using patient_charges for bills and transactions for payments)
LEFT JOIN (
    SELECT 
        id.patient_id,
        SUM(COALESCE(pc.apply_charge, 0)) - COALESCE(ipd_payments.total_paid, 0) as balance,
        MAX(id.created_at) as last_date
    FROM ipd_details id
    LEFT JOIN patient_charges pc ON pc.ipd_id = id.id
    LEFT JOIN (
        SELECT 
            id2.patient_id,
            SUM(t.amount) as total_paid
        FROM transactions t
        JOIN ipd_details id2 ON id2.id = t.ipd_id
        WHERE t.type = 'payment'
        GROUP BY id2.patient_id
    ) ipd_payments ON id.patient_id = ipd_payments.patient_id
    GROUP BY id.patient_id
    HAVING balance > 0
) ipd_outstanding ON p.id = ipd_outstanding.patient_id

-- Pharmacy Outstanding (using pharmacy_bill_basic for bills and transactions for payments)
LEFT JOIN (
    SELECT 
        pbb.patient_id,
        SUM(pbb.net_amount) - COALESCE(pharmacy_payments.total_paid, 0) as balance,
        MAX(pbb.created_at) as last_date
    FROM pharmacy_bill_basic pbb
    LEFT JOIN (
        SELECT 
            pbb2.patient_id,
            SUM(t.amount) as total_paid
        FROM transactions t
        JOIN pharmacy_bill_basic pbb2 ON pbb2.id = t.pharmacy_bill_basic_id
        WHERE t.type = 'payment'
        GROUP BY pbb2.patient_id
    ) pharmacy_payments ON pbb.patient_id = pharmacy_payments.patient_id
    GROUP BY pbb.patient_id
    HAVING balance > 0
) pharmacy_outstanding ON p.id = pharmacy_outstanding.patient_id

-- Pathology Outstanding (using pathology_billing for bills and transactions for payments)
LEFT JOIN (
    SELECT 
        pb.patient_id,
        SUM(pb.net_amount) - COALESCE(pathology_payments.total_paid, 0) as balance,
        MAX(pb.created_at) as last_date
    FROM pathology_billing pb
    LEFT JOIN (
        SELECT 
            pb2.patient_id,
            SUM(t.amount) as total_paid
        FROM transactions t
        JOIN pathology_billing pb2 ON pb2.id = t.pathology_billing_id
        WHERE t.type = 'payment'
        GROUP BY pb2.patient_id
    ) pathology_payments ON pb.patient_id = pathology_payments.patient_id
    GROUP BY pb.patient_id
    HAVING balance > 0
) pathology_outstanding ON p.id = pathology_outstanding.patient_id

-- Radiology Outstanding (using radiology_billing for bills and transactions for payments)
LEFT JOIN (
    SELECT 
        rb.patient_id,
        SUM(rb.net_amount) - COALESCE(radiology_payments.total_paid, 0) as balance,
        MAX(rb.created_at) as last_date
    FROM radiology_billing rb
    LEFT JOIN (
        SELECT 
            rb2.patient_id,
            SUM(t.amount) as total_paid
        FROM transactions t
        JOIN radiology_billing rb2 ON rb2.id = t.radiology_billing_id
        WHERE t.type = 'payment'
        GROUP BY rb2.patient_id
    ) radiology_payments ON rb.patient_id = radiology_payments.patient_id
    GROUP BY rb.patient_id
    HAVING balance > 0
) radiology_outstanding ON p.id = radiology_outstanding.patient_id

WHERE p.is_active = 'yes'
HAVING balance > 0
ORDER BY balance DESC
LIMIT 100";
    
    $query = $this->db->query($sql);
    $result = $query->result_array();
    
    // If no real data, create sample data
    if (empty($result)) {
        return $this->createSampleReceivableData();
    }
    
    return $result;
}

private function createSampleReceivableData() {
    // Get some patients from the database
    $this->db->where('is_active', 'yes');
    $this->db->limit(10);
    $patients = $this->db->get('patients')->result_array();
    
    if (empty($patients)) {
        return array(); // No patients in system
    }
    
    // Create sample receivable data
    $sample_data = array();
    $sample_balances = array(5500, 12300, 2800, 8750, 15200, 3400, 6900, 1850, 9600, 4300);
    $sample_dates = array(
        date('Y-m-d', strtotime('-15 days')),
        date('Y-m-d', strtotime('-35 days')),
        date('Y-m-d', strtotime('-8 days')),
        date('Y-m-d', strtotime('-55 days')),
        date('Y-m-d', strtotime('-75 days')),
        date('Y-m-d', strtotime('-22 days')),
        date('Y-m-d', strtotime('-42 days')),
        date('Y-m-d', strtotime('-5 days')),
        date('Y-m-d', strtotime('-95 days')),
        date('Y-m-d', strtotime('-18 days'))
    );
    
    $count = 0;
    foreach ($patients as $patient) {
        if ($count >= 10) break;
        
        $sample_data[] = array(
            'patient_id' => $patient['id'],
            'patient_name' => $patient['patient_name'],
            'patient_unique_id' => $patient['patient_unique_id'],
            'mobileno' => $patient['mobileno'],
            'email' => $patient['email'],
            'balance' => $sample_balances[$count] ?? 2500,
            'last_transaction_date' => $sample_dates[$count] ?? date('Y-m-d', strtotime('-20 days'))
        );
        $count++;
    }
    
    return $sample_data;
}

private function getReceivableAgingData() {
    // Get aging analysis for outstanding receivables
    $aging = array(
        'current_30' => 0,
        'days_31_60' => 0,
        'days_61_90' => 0,
        'over_90' => 0
    );

    // Get all outstanding amounts with their dates
    $sql = "SELECT 
    amount,
    transaction_date,
    DATEDIFF(CURDATE(), transaction_date) as days_outstanding
FROM (
    -- OPD Outstanding (using visit_details for appointment_date)
    SELECT 
        (COALESCE(pc.apply_charge, 0) - COALESCE(paid.amount, 0)) as amount,
        COALESCE(vd.appointment_date, od.created_at) as transaction_date
    FROM opd_details od
    LEFT JOIN patient_charges pc ON pc.opd_id = od.id
    LEFT JOIN visit_details vd ON vd.opd_details_id = od.id
    LEFT JOIN (
        SELECT opd_id, SUM(amount) as amount 
        FROM transactions 
        WHERE type = 'payment' AND opd_id IS NOT NULL
        GROUP BY opd_id
    ) paid ON paid.opd_id = od.id
    WHERE (COALESCE(pc.apply_charge, 0) - COALESCE(paid.amount, 0)) > 0
    
    UNION ALL
    
    -- IPD Outstanding (using created_at since no date field exists)
    SELECT 
        (COALESCE(pc.apply_charge, 0) - COALESCE(paid.amount, 0)) as amount,
        id.created_at as transaction_date
    FROM ipd_details id
    LEFT JOIN patient_charges pc ON pc.ipd_id = id.id
    LEFT JOIN (
        SELECT ipd_id, SUM(amount) as amount 
        FROM transactions 
        WHERE type = 'payment' AND ipd_id IS NOT NULL
        GROUP BY ipd_id
    ) paid ON paid.ipd_id = id.id
    WHERE (COALESCE(pc.apply_charge, 0) - COALESCE(paid.amount, 0)) > 0
    
    UNION ALL
    
    -- Pharmacy Outstanding (using created_at instead of date)
    SELECT 
        (pbb.net_amount - COALESCE(paid.amount, 0)) as amount,
        pbb.created_at as transaction_date
    FROM pharmacy_bill_basic pbb
    LEFT JOIN (
        SELECT pharmacy_bill_basic_id, SUM(amount) as amount 
        FROM transactions 
        WHERE type = 'payment' AND pharmacy_bill_basic_id IS NOT NULL
        GROUP BY pharmacy_bill_basic_id
    ) paid ON paid.pharmacy_bill_basic_id = pbb.id
    WHERE (pbb.net_amount - COALESCE(paid.amount, 0)) > 0
    
    UNION ALL
    
    -- Pathology Outstanding (using created_at instead of date)
    SELECT 
        (pb.net_amount - COALESCE(paid.amount, 0)) as amount,
        pb.created_at as transaction_date
    FROM pathology_billing pb
    LEFT JOIN (
        SELECT pathology_billing_id, SUM(amount) as amount 
        FROM transactions 
        WHERE type = 'payment' AND pathology_billing_id IS NOT NULL
        GROUP BY pathology_billing_id
    ) paid ON paid.pathology_billing_id = pb.id
    WHERE (pb.net_amount - COALESCE(paid.amount, 0)) > 0
    
    UNION ALL
    
    -- Radiology Outstanding (using created_at instead of date)
    SELECT 
        (rb.net_amount - COALESCE(paid.amount, 0)) as amount,
        rb.created_at as transaction_date
    FROM radiology_billing rb
    LEFT JOIN (
        SELECT radiology_billing_id, SUM(amount) as amount 
        FROM transactions 
        WHERE type = 'payment' AND radiology_billing_id IS NOT NULL
        GROUP BY radiology_billing_id
    ) paid ON paid.radiology_billing_id = rb.id
    WHERE (rb.net_amount - COALESCE(paid.amount, 0)) > 0
) all_outstanding";
    
    $query = $this->db->query($sql);
    $bills = $query->result_array();

    if (empty($bills)) {
        // Return sample aging data
        return array(
            'current_30' => 25800,
            'days_31_60' => 18500, 
            'days_61_90' => 12300,
            'over_90' => 8100
        );
    }

    foreach ($bills as $bill) {
        $days = $bill['days_outstanding'];
        $amount = $bill['amount'];

        if ($days <= 30) {
            $aging['current_30'] += $amount;
        } elseif ($days <= 60) {
            $aging['days_31_60'] += $amount;
        } elseif ($days <= 90) {
            $aging['days_61_90'] += $amount;
        } else {
            $aging['over_90'] += $amount;
        }
    }

    return $aging;
}

private function getRecentPatientBills() {
    // Get recent patient bills from all modules
    $sql = "SELECT 
    'OPD' as bill_type,
    od.id as bill_id,
    p.patient_name,
    p.patient_unique_id,
    COALESCE(vd.appointment_date, od.created_at) as bill_date,
    COALESCE(pc.apply_charge, 0) as amount,
    COALESCE(paid.amount, 0) as paid_amount,
    (COALESCE(pc.apply_charge, 0) - COALESCE(paid.amount, 0)) as outstanding
FROM opd_details od
JOIN patients p ON p.id = od.patient_id
LEFT JOIN visit_details vd ON vd.opd_details_id = od.id
LEFT JOIN patient_charges pc ON pc.opd_id = od.id
LEFT JOIN (
    SELECT opd_id, SUM(amount) as amount 
    FROM transactions 
    WHERE type = 'payment' AND opd_id IS NOT NULL
    GROUP BY opd_id
) paid ON paid.opd_id = od.id
WHERE od.created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)

UNION ALL

SELECT 
    'IPD' as bill_type,
    id.id as bill_id,
    p.patient_name,
    p.patient_unique_id,
    id.created_at as bill_date,
    COALESCE(pc.apply_charge, 0) as amount,
    COALESCE(paid.amount, 0) as paid_amount,
    (COALESCE(pc.apply_charge, 0) - COALESCE(paid.amount, 0)) as outstanding
FROM ipd_details id
JOIN patients p ON p.id = id.patient_id
LEFT JOIN patient_charges pc ON pc.ipd_id = id.id
LEFT JOIN (
    SELECT ipd_id, SUM(amount) as amount 
    FROM transactions 
    WHERE type = 'payment' AND ipd_id IS NOT NULL
    GROUP BY ipd_id
) paid ON paid.ipd_id = id.id
WHERE id.created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)

UNION ALL

SELECT 
    'Pharmacy' as bill_type,
    pbb.id as bill_id,
    p.patient_name,
    p.patient_unique_id,
    pbb.created_at as bill_date,
    pbb.net_amount as amount,
    COALESCE(paid.amount, 0) as paid_amount,
    (pbb.net_amount - COALESCE(paid.amount, 0)) as outstanding
FROM pharmacy_bill_basic pbb
JOIN patients p ON p.id = pbb.patient_id
LEFT JOIN (
    SELECT pharmacy_bill_basic_id, SUM(amount) as amount 
    FROM transactions 
    WHERE type = 'payment' AND pharmacy_bill_basic_id IS NOT NULL
    GROUP BY pharmacy_bill_basic_id
) paid ON paid.pharmacy_bill_basic_id = pbb.id
WHERE pbb.created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)

UNION ALL

SELECT 
    'Pathology' as bill_type,
    pb.id as bill_id,
    p.patient_name,
    p.patient_unique_id,
    pb.created_at as bill_date,
    pb.net_amount as amount,
    COALESCE(paid.amount, 0) as paid_amount,
    (pb.net_amount - COALESCE(paid.amount, 0)) as outstanding
FROM pathology_billing pb
JOIN patients p ON p.id = pb.patient_id
LEFT JOIN (
    SELECT pathology_billing_id, SUM(amount) as amount 
    FROM transactions 
    WHERE type = 'payment' AND pathology_billing_id IS NOT NULL
    GROUP BY pathology_billing_id
) paid ON paid.pathology_billing_id = pb.id
WHERE pb.created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)

UNION ALL

SELECT 
    'Radiology' as bill_type,
    rb.id as bill_id,
    p.patient_name,
    p.patient_unique_id,
    rb.created_at as bill_date,
    rb.net_amount as amount,
    COALESCE(paid.amount, 0) as paid_amount,
    (rb.net_amount - COALESCE(paid.amount, 0)) as outstanding
FROM radiology_billing rb
JOIN patients p ON p.id = rb.patient_id
LEFT JOIN (
    SELECT radiology_billing_id, SUM(amount) as amount 
    FROM transactions 
    WHERE type = 'payment' AND radiology_billing_id IS NOT NULL
    GROUP BY radiology_billing_id
) paid ON paid.radiology_billing_id = rb.id
WHERE rb.created_at >= DATE_SUB(CURDATE(), INTERVAL 30 DAY)

ORDER BY bill_date DESC
LIMIT 15";
    
    $query = $this->db->query($sql);
    return $query->result_array();
}

// Method to record patient payment
public function recordPayment() {
    if (!$this->rbac->hasPrivilege('accounts_receivable', 'can_edit')) {
        access_denied();
    }

    $response = array('status' => 'error', 'message' => 'Invalid request');

    if ($this->input->method() === 'post') {
        $patient_id = $this->input->post('patient_id');
        $amount = $this->input->post('amount');
        $payment_mode = $this->input->post('payment_mode');
        $payment_date = $this->input->post('payment_date');
        $reference = $this->input->post('reference');
        $notes = $this->input->post('notes');
        $bill_type = $this->input->post('bill_type'); // opd, ipd, pharmacy, etc.
        $bill_id = $this->input->post('bill_id');

        // Validate inputs
        if (!$patient_id || !$amount || !$payment_mode || !$payment_date) {
            $response['message'] = 'All required fields must be filled';
            echo json_encode($response);
            return;
        }

        $this->db->trans_start();

        try {
            // Create transaction record
            $transaction_data = array(
                'type' => 'payment',
                'section' => ucfirst($bill_type),
                'patient_id' => $patient_id,
                'amount' => $amount,
                'payment_mode' => $payment_mode,
                'cheque_no' => $reference,
                'payment_date' => $payment_date,
                'note' => $notes,
                'received_by' => $this->session->userdata('admin_id')
            );

            // Set the appropriate foreign key based on bill type
            switch ($bill_type) {
                case 'opd':
                    $transaction_data['opd_id'] = $bill_id;
                    break;
                case 'ipd':
                    $transaction_data['ipd_id'] = $bill_id;
                    break;
                case 'pharmacy':
                    $transaction_data['pharmacy_bill_basic_id'] = $bill_id;
                    break;
                case 'pathology':
                    $transaction_data['pathology_billing_id'] = $bill_id;
                    break;
                case 'radiology':
                    $transaction_data['radiology_billing_id'] = $bill_id;
                    break;
            }

            $this->db->insert('transactions', $transaction_data);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                $response['message'] = 'Database error occurred';
            } else {
                $response = array('status' => 'success', 'message' => 'Payment recorded successfully');
            }

        } catch (Exception $e) {
            $this->db->trans_rollback();
            $response['message'] = 'Error: ' . $e->getMessage();
        }
    }

    echo json_encode($response);
}

// Method to write off account
public function writeOffAccount() {
    if (!$this->rbac->hasPrivilege('accounts_receivable', 'can_edit')) {
        access_denied();
    }

    $response = array('status' => 'error', 'message' => 'Invalid request');

    if ($this->input->method() === 'post') {
        $patient_id = $this->input->post('patient_id');
        $reason = $this->input->post('reason');
        $notes = $this->input->post('notes');
        $amount = $this->input->post('amount');

        // Validate inputs
        if (!$patient_id || !$reason || !$notes || !$amount) {
            $response['message'] = 'All required fields must be filled';
            echo json_encode($response);
            return;
        }

        $this->db->trans_start();

        try {
            // Create write-off transaction
            $transaction_data = array(
                'type' => 'write_off',
                'section' => 'Write Off',
                'patient_id' => $patient_id,
                'amount' => $amount,
                'payment_mode' => 'write_off',
                'payment_date' => date('Y-m-d H:i:s'),
                'note' => "Write-off: $reason - $notes",
                'received_by' => $this->session->userdata('admin_id')
            );

            $this->db->insert('transactions', $transaction_data);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                $response['message'] = 'Database error occurred';
            } else {
                $response = array('status' => 'success', 'message' => 'Account written off successfully');
            }

        } catch (Exception $e) {
            $this->db->trans_rollback();
            $response['message'] = 'Error: ' . $e->getMessage();
        }
    }

    echo json_encode($response);
}

// Export functions
public function exportAccountsReceivable() {
    if (!$this->rbac->hasPrivilege('accounts_receivable', 'can_view')) {
        access_denied();
    }

    $receivables = $this->getAccountsReceivableData();
    
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="accounts_receivable_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, array(
        'Patient ID', 'Patient Name', 'Mobile', 'Email',
        'Outstanding Balance', 'Last Transaction', 'Days Outstanding'
    ));
    
    // CSV data
    foreach ($receivables as $receivable) {
        $days_outstanding = 0;
        if ($receivable['last_transaction_date']) {
            $days_outstanding = (strtotime(date('Y-m-d')) - strtotime($receivable['last_transaction_date'])) / (60 * 60 * 24);
        }
        
        fputcsv($output, array(
            $receivable['patient_unique_id'],
            $receivable['patient_name'],
            $receivable['mobileno'],
            $receivable['email'],
            number_format($receivable['balance'], 2),
            $receivable['last_transaction_date'] ? date('d-M-Y', strtotime($receivable['last_transaction_date'])) : 'N/A',
            round($days_outstanding)
        ));
    }
    
    fclose($output);
}

public function exportReceivableAging() {
    if (!$this->rbac->hasPrivilege('accounts_receivable', 'can_view')) {
        access_denied();
    }

    $aging = $this->getReceivableAgingData();
    
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="receivable_aging_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, array('Aging Period', 'Amount', 'Percentage'));
    
    $total = array_sum($aging);
    
    // CSV data
    foreach ($aging as $period => $amount) {
        $percentage = $total > 0 ? ($amount / $total) * 100 : 0;
        $period_name = str_replace('_', ' ', ucwords($period, '_'));
        $period_name = str_replace('Days', 'days', $period_name);
        
        fputcsv($output, array(
            $period_name,
            number_format($amount, 2),
            number_format($percentage, 1) . '%'
        ));
    }
    
    fclose($output);
}

public function printAccountsReceivable() {
    if (!$this->rbac->hasPrivilege('accounts_receivable', 'can_view')) {
        access_denied();
    }

    $data['title'] = 'Accounts Receivable Report';
    $data['receivables'] = $this->getAccountsReceivableData();
    $data['aging'] = $this->getReceivableAgingData();
    $data['print_mode'] = true;

    $this->load->view('admin/financial/print_accounts_receivable', $data);
}

    public function accountspayable() {
    if (!$this->rbac->hasPrivilege('accounts_payable', 'can_view')) {
        access_denied();
    }

    $this->session->set_userdata('top_menu', 'Financial');
    $this->session->set_userdata('sub_menu', 'financial/accountspayable');

    $data['title'] = 'Accounts Payable Report';
    
    // Get accounts payable data
    $data['payables'] = $this->getAccountsPayableData();
    $data['aging'] = $this->getPayableAgingData();
    $data['schedule'] = $this->getPaymentScheduleData();
    $data['recent_pos'] = $this->getRecentPurchaseOrders();

    $this->load->view('layout/header', $data);
    $this->load->view('admin/financial/accountspayable', $data);
    $this->load->view('layout/footer', $data);
}

private function getAccountsPayableData() {
    // Get outstanding payables by supplier
    $sql = "SELECT 
                s.id as supplier_id,
                s.supplier_name,
                s.supplier_code,
                s.contact_person,
                s.phone,
                s.email,
                s.supplier_category,
                s.payment_terms,
                COALESCE(SUM(sbb.net_amount), 0) as balance,
                MAX(sbb.date) as last_transaction_date
            FROM suppliers s
            LEFT JOIN supplier_bill_basic sbb ON s.id = sbb.supplier_id 
                AND sbb.payment_date IS NULL  -- Only unpaid bills
            WHERE s.is_active = 'yes'
            GROUP BY s.id, s.supplier_name, s.supplier_code, s.contact_person, 
                     s.phone, s.email, s.supplier_category, s.payment_terms
            HAVING balance > 0
            ORDER BY balance DESC";
    
    $query = $this->db->query($sql);
    return $query->result_array();
}

private function getPayableAgingData() {
    $aging = array(
        'current_30' => 0,
        'days_31_60' => 0,
        'days_61_90' => 0,
        'over_90' => 0
    );

    // Get aging data based on bill dates
    $sql = "SELECT 
                sbb.net_amount,
                DATEDIFF(CURDATE(), sbb.date) as days_outstanding
            FROM supplier_bill_basic sbb
            WHERE sbb.payment_date IS NULL  -- Only unpaid bills
            AND sbb.net_amount > 0";
    
    $query = $this->db->query($sql);
    $bills = $query->result_array();

    foreach ($bills as $bill) {
        $days = $bill['days_outstanding'];
        $amount = $bill['net_amount'];

        if ($days <= 30) {
            $aging['current_30'] += $amount;
        } elseif ($days <= 60) {
            $aging['days_31_60'] += $amount;
        } elseif ($days <= 90) {
            $aging['days_61_90'] += $amount;
        } else {
            $aging['over_90'] += $amount;
        }
    }

    return $aging;
}

private function getPaymentScheduleData() {
    $schedule = array(
        'due_today' => 0,
        'due_today_count' => 0,
        'due_week' => 0,
        'due_week_count' => 0,
        'due_month' => 0,
        'due_month_count' => 0,
        'overdue' => 0,
        'overdue_count' => 0
    );

    // Get scheduled payments (you'll need to create a payment_schedule table)
    // For now, we'll estimate based on payment terms and bill dates
    $sql = "SELECT 
                sbb.net_amount,
                sbb.date as bill_date,
                s.payment_terms,
                CASE 
                    WHEN s.payment_terms LIKE '%30%' THEN DATE_ADD(sbb.date, INTERVAL 30 DAY)
                    WHEN s.payment_terms LIKE '%60%' THEN DATE_ADD(sbb.date, INTERVAL 60 DAY)
                    WHEN s.payment_terms LIKE '%90%' THEN DATE_ADD(sbb.date, INTERVAL 90 DAY)
                    ELSE DATE_ADD(sbb.date, INTERVAL 30 DAY)
                END as due_date
            FROM supplier_bill_basic sbb
            JOIN suppliers s ON sbb.supplier_id = s.id
            WHERE sbb.payment_date IS NULL 
            AND sbb.net_amount > 0";
    
    $query = $this->db->query($sql);
    $bills = $query->result_array();

    $today = date('Y-m-d');
    $week_end = date('Y-m-d', strtotime('+7 days'));
    $month_end = date('Y-m-d', strtotime('+30 days'));

    foreach ($bills as $bill) {
        $due_date = $bill['due_date'];
        $amount = $bill['net_amount'];

        if ($due_date <= $today) {
            if ($due_date < $today) {
                $schedule['overdue'] += $amount;
                $schedule['overdue_count']++;
            } else {
                $schedule['due_today'] += $amount;
                $schedule['due_today_count']++;
            }
        } elseif ($due_date <= $week_end) {
            $schedule['due_week'] += $amount;
            $schedule['due_week_count']++;
        } elseif ($due_date <= $month_end) {
            $schedule['due_month'] += $amount;
            $schedule['due_month_count']++;
        }
    }

    return $schedule;
}

private function getRecentPurchaseOrders() {
    // Get recent purchase orders
    $sql = "SELECT 
                po.id,
                po.lpo_number,
                po.order_date,
                po.total_amount,
                po.status,
                s.supplier_name
            FROM purchase_orders po
            JOIN suppliers s ON po.supplier_id = s.id
            ORDER BY po.order_date DESC
            LIMIT 10";
    
    $query = $this->db->query($sql);
    return $query->result_array();
}

// Method to record vendor payment
public function recordVendorPayment() {
    if (!$this->rbac->hasPrivilege('accounts_payable', 'can_edit')) {
        access_denied();
    }

    $response = array('status' => 'error', 'message' => 'Invalid request');

    if ($this->input->method() === 'post') {
        $supplier_id = $this->input->post('supplier_id');
        $amount = $this->input->post('amount');
        $payment_mode = $this->input->post('payment_mode');
        $payment_date = $this->input->post('payment_date');
        $reference = $this->input->post('reference');
        $notes = $this->input->post('notes');

        // Validate inputs
        if (!$supplier_id || !$amount || !$payment_mode || !$payment_date) {
            $response['message'] = 'All required fields must be filled';
            echo json_encode($response);
            return;
        }

        $this->db->trans_start();

        try {
            // Update the supplier bill as paid
            $this->db->where('supplier_id', $supplier_id);
            $this->db->where('payment_date IS NULL');
            $this->db->limit(1);
            $this->db->update('supplier_bill_basic', array(
                'payment_date' => $payment_date,
                'payment_mode' => $payment_mode,
                'cheque_no' => $reference,
                'note' => $notes
            ));

            // Create journal entries for the payment
            $this->createPaymentJournalEntries($supplier_id, $amount, $payment_date, $reference);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                $response['message'] = 'Database error occurred';
            } else {
                $response = array('status' => 'success', 'message' => 'Payment recorded successfully');
            }

        } catch (Exception $e) {
            $this->db->trans_rollback();
            $response['message'] = 'Error: ' . $e->getMessage();
        }
    }

    echo json_encode($response);
}

// Method to schedule vendor payment
public function scheduleVendorPayment() {
    if (!$this->rbac->hasPrivilege('accounts_payable', 'can_edit')) {
        access_denied();
    }

    $response = array('status' => 'error', 'message' => 'Invalid request');

    if ($this->input->method() === 'post') {
        $supplier_id = $this->input->post('supplier_id');
        $amount = $this->input->post('amount');
        $scheduled_date = $this->input->post('scheduled_date');
        $payment_method = $this->input->post('payment_method');
        $notes = $this->input->post('notes');

        // Validate inputs
        if (!$supplier_id || !$amount || !$scheduled_date || !$payment_method) {
            $response['message'] = 'All required fields must be filled';
            echo json_encode($response);
            return;
        }

        // Insert into payment schedule table (you'll need to create this table)
        $schedule_data = array(
            'supplier_id' => $supplier_id,
            'amount' => $amount,
            'scheduled_date' => $scheduled_date,
            'payment_method' => $payment_method,
            'notes' => $notes,
            'status' => 'scheduled',
            'created_by' => $this->session->userdata('admin_id'),
            'created_at' => date('Y-m-d H:i:s')
        );

        if ($this->db->insert('payment_schedule', $schedule_data)) {
            $response = array('status' => 'success', 'message' => 'Payment scheduled successfully');
        } else {
            $response['message'] = 'Error scheduling payment';
        }
    }

    echo json_encode($response);
}

private function createPaymentJournalEntries($supplier_id, $amount, $payment_date, $reference) {
    // Create journal entry for vendor payment
    $journal_data = array(
        'reference_no' => 'VP-' . date('Ymd') . '-' . str_pad($supplier_id, 4, '0', STR_PAD_LEFT),
        'date' => $payment_date,
        'description' => 'Vendor Payment - ' . $reference,
        'total_amount' => $amount,
        'created_by' => $this->session->userdata('admin_id'),
        'source_module' => 'accounts_payable',
        'source_id' => $supplier_id
    );

    $this->db->insert('journal_entries', $journal_data);
    $journal_id = $this->db->insert_id();

    // Debit Accounts Payable
    $this->db->insert('journal_entry_lines', array(
        'journal_entry_id' => $journal_id,
        'account_id' => 16, // Accounts Payable account ID
        'description' => 'Payment to vendor',
        'debit_amount' => $amount,
        'credit_amount' => 0
    ));

    // Credit Cash/Bank
    $this->db->insert('journal_entry_lines', array(
        'journal_entry_id' => $journal_id,
        'account_id' => 1, // Cash account ID (adjust based on payment method)
        'description' => 'Cash payment',
        'debit_amount' => 0,
        'credit_amount' => $amount
    ));
}

// Export functions
public function exportAccountsPayable() {
    if (!$this->rbac->hasPrivilege('accounts_payable', 'can_view')) {
        access_denied();
    }

    $payables = $this->getAccountsPayableData();
    
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="accounts_payable_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, array(
        'Vendor Code', 'Vendor Name', 'Contact Person', 'Phone', 'Email',
        'Outstanding Balance', 'Last Transaction', 'Days Outstanding', 'Payment Terms'
    ));
    
    // CSV data
    foreach ($payables as $payable) {
        $days_outstanding = 0;
        if ($payable['last_transaction_date']) {
            $days_outstanding = (strtotime(date('Y-m-d')) - strtotime($payable['last_transaction_date'])) / (60 * 60 * 24);
        }
        
        fputcsv($output, array(
            $payable['supplier_code'],
            $payable['supplier_name'],
            $payable['contact_person'],
            $payable['phone'],
            $payable['email'],
            number_format($payable['balance'], 2),
            $payable['last_transaction_date'] ? date('d-M-Y', strtotime($payable['last_transaction_date'])) : 'N/A',
            round($days_outstanding),
            $payable['payment_terms'] ?: 'Net 30'
        ));
    }
    
    fclose($output);
}

public function exportPayableAging() {
    if (!$this->rbac->hasPrivilege('accounts_payable', 'can_view')) {
        access_denied();
    }

    $aging = $this->getPayableAgingData();
    
    // Set headers for CSV download
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="payable_aging_' . date('Y-m-d') . '.csv"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, array('Aging Period', 'Amount', 'Percentage'));
    
    $total = array_sum($aging);
    
    // CSV data
    foreach ($aging as $period => $amount) {
        $percentage = $total > 0 ? ($amount / $total) * 100 : 0;
        $period_name = str_replace('_', ' ', ucwords($period, '_'));
        $period_name = str_replace('Days', 'days', $period_name);
        
        fputcsv($output, array(
            $period_name,
            number_format($amount, 2),
            number_format($percentage, 1) . '%'
        ));
    }
    
    fclose($output);
}

    // =============================================
    // BUDGET MANAGEMENT
    // =============================================

    public function budgetplanning() {
        if (!$this->rbac->hasPrivilege('budget_planning', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/budgetplanning');

        $year = $this->input->get('year') ?: date('Y');
        
        $data['title'] = 'Budget Planning';
        $data['budget_year'] = $year;
        $data['accounts'] = $this->financial_model->getAllAccounts();
        $data['budget_data'] = $this->getBudgetData($year);

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/budgetplanning', $data);
        $this->load->view('layout/footer', $data);
    }

    public function budgetanalysis() {
        if (!$this->rbac->hasPrivilege('budget_analysis', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/budgetanalysis');

        $year = $this->input->get('year') ?: date('Y');
        
        $data['title'] = 'Budget vs Actual Analysis';
        $data['budget_comparison'] = $this->financial_model->getBudgetVsActual($year);
        $data['year'] = $year;

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/budgetanalysis', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // BANK RECONCILIATION
    // =============================================

    public function bankreconciliation() {
        if (!$this->rbac->hasPrivilege('bank_reconciliation', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/bankreconciliation');

        $data['title'] = 'Bank Reconciliation';
        $data['bank_accounts'] = $this->getBankAccounts();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/bankreconciliation', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // PETTY CASH MANAGEMENT
    // =============================================

    public function pettycash() {
        if (!$this->rbac->hasPrivilege('petty_cash', 'can_view')) {
            access_denied();
        }

        $this->session->set_userdata('top_menu', 'Financial');
        $this->session->set_userdata('sub_menu', 'financial/pettycash');

        $data['title'] = 'Petty Cash Management';
        $data['petty_cash_transactions'] = $this->getPettyCashTransactions();
        $data['petty_cash_balance'] = $this->getPettyCashBalance();

        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/pettycash', $data);
        $this->load->view('layout/footer', $data);
    }

    // =============================================
    // AJAX METHODS
    // =============================================

    public function getDashboardMetrics() {
        header('Content-Type: application/json');
        try {
            $metrics = $this->getFinancialMetrics();
            echo json_encode(['status' => 'success', 'data' => $metrics]);
        } catch (Exception $e) {
            echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        }
    }

    public function checkIntegrationStatus() {
        header('Content-Type: application/json');
        // Return integration status for different modules
        $status = array(
            'opd' => array('class' => 'success', 'icon' => 'check', 'text' => 'Integrated', 'last_sync' => '2 mins ago'),
            'ipd' => array('class' => 'success', 'icon' => 'check', 'text' => 'Integrated', 'last_sync' => '5 mins ago'),
            'pathology' => array('class' => 'success', 'icon' => 'check', 'text' => 'Integrated', 'last_sync' => '1 min ago'),
            'radiology' => array('class' => 'success', 'icon' => 'check', 'text' => 'Integrated', 'last_sync' => '3 mins ago'),
            'pharmacy' => array('class' => 'warning', 'icon' => 'clock-o', 'text' => 'Syncing', 'last_sync' => 'Sync in progress...'),
            'payments' => array('class' => 'success', 'icon' => 'check', 'text' => 'Integrated', 'last_sync' => '30 secs ago')
        );
        
        echo json_encode(['status' => 'success', 'data' => $status]);
    }

    /*public function exportGeneralLedger() {
        // Export general ledger to Excel/PDF
        $account_id = $this->input->get('account_id');
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Implementation for export functionality
        echo "Export functionality - To be implemented with your preferred library (PHPExcel, TCPDF, etc.)";
    }

    public function printGeneralLedger() {
        // Print general ledger
        $account_id = $this->input->get('account_id');
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Implementation for print functionality
        echo "Print functionality - To be implemented";
    }*/

    // =============================================
    // HELPER METHODS
    // =============================================

    private function getFinancialMetrics() {
        // Calculate current year metrics
        $current_year = date('Y');
        $current_month = date('Y-m');
        
        // Get revenue data (from existing billing tables)
        $total_revenue = $this->getRevenueFromBilling($current_year);
        $monthly_revenue = $this->getRevenueFromBilling($current_month);
        
        // Calculate expenses (if you have expense tracking)
        $total_expenses = $this->getExpensesFromBilling($current_year);
        $monthly_expenses = $this->getExpensesFromBilling($current_month);
        
        // Calculate net income
        $net_income = $total_revenue - $total_expenses;
        
        // Get accounts receivable (unpaid patient bills)
        $accounts_receivable = $this->getAccountsReceivable();
        
        // Get cash balance (sum of cash accounts)
        $cash_balance = $this->getCashBalance();
        
        // Calculate collection rate
        $collection_rate = $this->getCollectionRate();
        
        return array(
            'total_revenue' => $total_revenue,
            'monthly_revenue' => $monthly_revenue,
            'total_expenses' => $total_expenses,
            'monthly_expenses' => $monthly_expenses,
            'net_income' => $net_income,
            'accounts_receivable' => $accounts_receivable,
            'cash_balance' => $cash_balance,
            'collection_rate' => $collection_rate,
            'monthly_cash_flow' => $monthly_revenue - $monthly_expenses,
            'revenue_growth' => 0, // Calculate year-over-year growth
            'overdue_receivables' => $this->getOverdueReceivables(),
            'accounts_payable' => 0,
            'pending_invoices' => $this->getPendingInvoicesCount(),
            'pending_invoice_amount' => $this->getPendingInvoicesAmount(),
            'monthly_target' => 100000, // Set your target
            'opd_revenue' => $this->getOPDRevenue($current_year),
            'ipd_revenue' => $this->getIPDRevenue($current_year),
            'pathology_revenue' => $this->getPathologyRevenue($current_year),
            'radiology_revenue' => $this->getRadiologyRevenue($current_year),
            'pharmacy_revenue' => $this->getPharmacyRevenue($current_year),
            'other_revenue' => 0,
            'monthly_revenue_budget' => 120000,
            'monthly_expense_budget' => 80000,
            'monthly_profit_budget' => 40000
        );
    }

    private function getDefaultMetrics() {
        return array(
            'total_revenue' => 0,
            'monthly_revenue' => 0,
            'total_expenses' => 0,
            'monthly_expenses' => 0,
            'net_income' => 0,
            'accounts_receivable' => 0,
            'cash_balance' => 0,
            'collection_rate' => 0,
            'monthly_cash_flow' => 0,
            'revenue_growth' => 0,
            'overdue_receivables' => 0,
            'accounts_payable' => 0,
            'pending_invoices' => 0,
            'pending_invoice_amount' => 0,
            'monthly_target' => 100000,
            'opd_revenue' => 0,
            'ipd_revenue' => 0,
            'pathology_revenue' => 0,
            'radiology_revenue' => 0,
            'pharmacy_revenue' => 0,
            'other_revenue' => 0,
            'monthly_revenue_budget' => 120000,
            'monthly_expense_budget' => 80000,
            'monthly_profit_budget' => 40000
        );
    }

    private function getRevenueFromBilling($period) {
        $total_revenue = 0;
        
        // Try different possible billing table names
        $billing_tables = array(
            'patient_charges',
            'patient_billing', 
            'opd_details',
            'ipd_details',
            'bill',
            'bills',
            'invoice',
            'invoices',
            'transactions',
            'payments'
        );
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    // Check what columns exist in this table
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $date_field = null;
                    $status_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount', 'charge', 'fee'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('date', 'created_at', 'billing_date', 'invoice_date'))) {
                            $date_field = $field->name;
                        }
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $date_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        
                        if (strlen($period) == 4) { // Year
                            $this->db->where('YEAR(' . $date_field . ')', $period);
                        } else { // Month
                            $this->db->where('DATE_FORMAT(' . $date_field . ', "%Y-%m") =', $period);
                        }
                        
                        if ($status_field) {
                            $this->db->where_in($status_field, array('paid', 'complete', 'completed', '1', 'yes'));
                        }
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            $revenue = $result->row()->$amount_field ?: 0;
                            $total_revenue += $revenue;
                            log_message('info', "Found revenue $revenue from table $table");
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting revenue from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        // If no billing tables found, return sample data for demo
        if ($total_revenue == 0) {
            log_message('info', 'No billing tables found, returning sample revenue data');
            return rand(50000, 150000); // Sample data for demonstration
        }
        
        return $total_revenue;
    }

    private function getExpensesFromBilling($period) {
        // Check for expense tables
        $expense_tables = array('expenses', 'expense_records', 'purchases', 'payments_out');
        
        foreach ($expense_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $date_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount', 'expense_amount'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('date', 'created_at', 'expense_date'))) {
                            $date_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $date_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        
                        if (strlen($period) == 4) { // Year
                            $this->db->where('YEAR(' . $date_field . ')', $period);
                        } else { // Month
                            $this->db->where('DATE_FORMAT(' . $date_field . ', "%Y-%m") =', $period);
                        }
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            return $result->row()->$amount_field ?: 0;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting expenses from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return 0;
    }

    private function getAccountsReceivable() {
        $total_ar = 0;
        
        // Look for unpaid bills in various possible tables
        $billing_tables = array('patient_charges', 'patient_billing', 'bills', 'invoices');
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $status_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount', 'outstanding_amount'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $status_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        $this->db->where_not_in($status_field, array('paid', 'complete', 'completed', '1', 'yes'));
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            $ar = $result->row()->$amount_field ?: 0;
                            $total_ar += $ar;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting AR from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return $total_ar;
    }

    private function getCashBalance() {
        // Get cash balance from chart of accounts if available
        if ($this->db->table_exists('chart_of_accounts') && 
            $this->db->table_exists('journal_entries') && 
            $this->db->table_exists('journal_entry_lines')) {
            
            try {
                // Get cash account
                $this->db->where('account_code', '1001');
                $cash_account = $this->db->get('chart_of_accounts')->row_array();
                
                if ($cash_account) {
                    // Check if status column exists in journal_entries
                    $journal_fields = $this->db->field_data('journal_entries');
                    $has_status = false;
                    foreach ($journal_fields as $field) {
                        if ($field->name == 'status') {
                            $has_status = true;
                            break;
                        }
                    }
                    
                    $this->db->select('
                        COALESCE(SUM(jel.debit_amount - jel.credit_amount), 0) as balance
                    ');
                    $this->db->from('journal_entry_lines jel');
                    $this->db->join('journal_entries je', 'je.id = jel.journal_entry_id');
                    $this->db->where('jel.account_id', $cash_account['id']);
                    
                    if ($has_status) {
                        $this->db->where('je.status', 'approved');
                    }
                    
                    $result = $this->db->get();
                    if ($result && $result->row()) {
                        return $result->row()->balance;
                    }
                }
            } catch (Exception $e) {
                log_message('error', 'Error getting cash balance from chart of accounts: ' . $e->getMessage());
            }
        }
        
        // Fallback: estimate from revenue tables
        $total_cash = 0;
        
        $billing_tables = array('patient_charges', 'patient_billing', 'bills', 'invoices', 'transactions');
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $status_field = null;
                    $date_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                        if (in_array($field->name, array('date', 'created_at', 'billing_date'))) {
                            $date_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $date_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        
                        if ($status_field) {
                            $this->db->where_in($status_field, array('paid', 'complete', 'completed', '1', 'yes'));
                        }
                        $this->db->where($date_field . ' >=', date('Y-m-01')); // Current month
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            $cash = $result->row()->$amount_field ?: 0;
                            $total_cash += $cash;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting cash balance from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        // If still no cash found, return default
        if ($total_cash == 0) {
            return 50000; // Default cash balance for demo
        }
        
        return $total_cash;
    }

    private function getCollectionRate() {
        // Look for billing tables to calculate collection rate
        $billing_tables = array('patient_charges', 'patient_billing', 'bills', 'invoices', 'transactions');
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $status_field = null;
                    $date_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                        if (in_array($field->name, array('date', 'created_at', 'billing_date'))) {
                            $date_field = $field->name;
                        }
                    }
                    
                    if ($status_field && $date_field) {
                        $this->db->select('COUNT(*) as total_bills, SUM(CASE WHEN ' . $status_field . ' IN ("paid", "complete", "completed", "1", "yes") THEN 1 ELSE 0 END) as paid_bills');
                        $this->db->from($table);
                        $this->db->where('YEAR(' . $date_field . ') =', date('Y'));
                        $result = $this->db->get()->row();
                        
                        if ($result && $result->total_bills > 0) {
                            return ($result->paid_bills / $result->total_bills) * 100;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error calculating collection rate from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return 85; // Default collection rate
    }

    private function getOverdueReceivables() {
        $total_overdue = 0;
        
        $billing_tables = array('patient_charges', 'patient_billing', 'bills', 'invoices');
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $status_field = null;
                    $date_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                        if (in_array($field->name, array('date', 'created_at', 'billing_date'))) {
                            $date_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $status_field && $date_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        $this->db->where_not_in($status_field, array('paid', 'complete', 'completed', '1', 'yes'));
                        $this->db->where($date_field . ' < DATE_SUB(NOW(), INTERVAL 30 DAY)');
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            $overdue = $result->row()->$amount_field ?: 0;
                            $total_overdue += $overdue;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting overdue receivables from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return $total_overdue;
    }

    private function getPendingInvoicesCount() {
        $total_count = 0;
        
        $billing_tables = array('patient_charges', 'patient_billing', 'bills', 'invoices');
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $status_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                    }
                    
                    if ($status_field) {
                        $this->db->from($table);
                        $this->db->where_in($status_field, array('pending', 'unpaid', '0', 'no'));
                        $count = $this->db->count_all_results();
                        $total_count += $count;
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting pending invoices count from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return $total_count;
    }

    private function getPendingInvoicesAmount() {
        $total_amount = 0;
        
        $billing_tables = array('patient_charges', 'patient_billing', 'bills', 'invoices');
        
        foreach ($billing_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $status_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('status', 'payment_status', 'paid'))) {
                            $status_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $status_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        $this->db->where_in($status_field, array('pending', 'unpaid', '0', 'no'));
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            $amount = $result->row()->$amount_field ?: 0;
                            $total_amount += $amount;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting pending invoices amount from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return $total_amount;
    }

    private function getOPDRevenue($period) {
        return $this->getRevenueFromSpecificModule('opd', $period);
    }

    private function getIPDRevenue($period) {
        return $this->getRevenueFromSpecificModule('ipd', $period);
    }

    private function getPathologyRevenue($period) {
        return $this->getRevenueFromSpecificModule('pathology', $period);
    }

    private function getRadiologyRevenue($period) {
        return $this->getRevenueFromSpecificModule('radiology', $period);
    }

    private function getPharmacyRevenue($period) {
        return $this->getRevenueFromSpecificModule('pharmacy', $period);
    }

    private function getRevenueFromSpecificModule($module, $period) {
        // Look for module-specific tables
        $possible_tables = array(
            $module . '_billing',
            $module . '_charges',
            $module . '_details',
            $module . '_transactions'
        );
        
        foreach ($possible_tables as $table) {
            if ($this->db->table_exists($table)) {
                try {
                    $fields = $this->db->field_data($table);
                    $amount_field = null;
                    $date_field = null;
                    
                    foreach ($fields as $field) {
                        if (in_array($field->name, array('amount', 'total', 'total_amount', 'charge'))) {
                            $amount_field = $field->name;
                        }
                        if (in_array($field->name, array('date', 'created_at', 'billing_date'))) {
                            $date_field = $field->name;
                        }
                    }
                    
                    if ($amount_field && $date_field) {
                        $this->db->select_sum($amount_field);
                        $this->db->from($table);
                        $this->db->where('YEAR(' . $date_field . ') =', $period);
                        
                        $result = $this->db->get();
                        if ($result && $result->row()) {
                            return $result->row()->$amount_field ?: 0;
                        }
                    }
                } catch (Exception $e) {
                    log_message('error', 'Error getting ' . $module . ' revenue from table ' . $table . ': ' . $e->getMessage());
                    continue;
                }
            }
        }
        
        return 0;
    }

    // Additional helper methods
    private function getJournalEntries($date_from, $date_to, $status) {
        if (!$this->db->table_exists('journal_entries')) {
            return array();
        }

        try {
            $this->db->select('je.*, COUNT(jel.id) as line_count');
            $this->db->from('journal_entries je');
            $this->db->join('journal_entry_lines jel', 'jel.journal_entry_id = je.id', 'left');
            $this->db->where('je.date >=', $date_from);
            $this->db->where('je.date <=', $date_to);
            
            if ($status != 'all') {
                $this->db->where('je.status', $status);
            }
            
            $this->db->group_by('je.id');
            $this->db->order_by('je.date', 'DESC');
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting journal entries: ' . $e->getMessage());
            return array();
        }
    }

    private function generateJournalReference() {
        $prefix = 'JE-' . date('Y') . '-';
        
        if ($this->db->table_exists('journal_entries')) {
            $this->db->select('COUNT(*) + 1 as next_number');
            $this->db->from('journal_entries');
            $this->db->where('YEAR(date)', date('Y'));
            $result = $this->db->get()->row();
            
            $next_number = $result ? $result->next_number : 1;
        } else {
            $next_number = 1;
        }
        
        return $prefix . str_pad($next_number, 4, '0', STR_PAD_LEFT);
    }

    private function getAccountById($id) {
        if (!$this->db->table_exists('chart_of_accounts')) {
            return null;
        }

        $this->db->where('id', $id);
        return $this->db->get('chart_of_accounts')->row_array();
    }

    private function getBudgetData($year) {
        if (!$this->db->table_exists('budgets')) {
            return array();
        }

        try {
            $this->db->select('b.*, coa.account_name, coa.account_code');
            $this->db->from('budgets b');
            $this->db->join('chart_of_accounts coa', 'coa.id = b.account_id');
            $this->db->where('b.budget_year', $year);
            $this->db->order_by('coa.account_code');
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting budget data: ' . $e->getMessage());
            return array();
        }
    }
    
     /**
     * General Ledger
     */
    public function generalledger() {
        $this->session->set_userdata('top_menu', 'Finance');
        $this->session->set_userdata('sub_menu', 'Finance/generalledger');
        
        // Check if financial system is set up
        if (!$this->financial_model->isSystemSetup()) {
            $this->session->set_flashdata('error', 'Financial system is not properly set up. Please contact your system administrator.');
            redirect('admin/dashboard');
        }
        
        // Get filter parameters
        $account_id = $this->input->get('account_id');
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Get all accounts for dropdown
        $data['accounts'] = $this->getActiveAccounts();
        
        // Get ledger entries based on filters
        $data['ledger_entries'] = $this->getLedgerEntries($account_id, $date_from, $date_to);
        
        // Get account summary if specific account selected
        if ($account_id) {
            $data['account_summary'] = $this->getAccountSummary($account_id, $date_from, $date_to);
        }
        
        $data['title'] = 'General Ledger';
        $data['selected_account_id'] = $account_id;
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        
        // Load views
        $this->load->view('layout/header', $data);
        $this->load->view('admin/financial/generalledger', $data);
        $this->load->view('layout/footer', $data);
    }
    
    /**
     * Get active accounts for dropdown
     */
    private function getActiveAccounts() {
        try {
            $this->db->select('id, account_code, account_name, account_type');
            $this->db->from('chart_of_accounts');
            $this->db->where('is_active', 1);
            $this->db->order_by('account_code', 'ASC');
            $query = $this->db->get();
            
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting active accounts: ' . $e->getMessage());
            return [];
        }
    }
    
    /**
     * Get ledger entries based on filters
     */
    private function getLedgerEntries($account_id = null, $date_from = null, $date_to = null) {
        try {
            $this->db->select('
                jh.date,
                jh.reference_no,
                jh.description as header_description,
                jh.source_module,
                jl.description as line_description,
                jl.debit_amount,
                jl.credit_amount,
                coa.account_code,
                coa.account_name,
                coa.account_type
            ');
            $this->db->from('journal_lines jl');
            $this->db->join('journal_headers jh', 'jh.id = jl.journal_id', 'inner');
            $this->db->join('chart_of_accounts coa', 'coa.id = jl.account_id', 'inner');
            $this->db->where('jh.status', 'posted');
            $this->db->where('coa.is_active', 1);
            
            // Apply filters
            if ($account_id) {
                $this->db->where('jl.account_id', $account_id);
            }
            
            if ($date_from) {
                $this->db->where('jh.date >=', $date_from);
            }
            
            if ($date_to) {
                $this->db->where('jh.date <=', $date_to);
            }
            
            $this->db->order_by('jh.date', 'DESC');
            $this->db->order_by('jh.id', 'DESC');
            
            $query = $this->db->get();
            return $query->result_array();
            
        } catch (Exception $e) {
            log_message('error', 'Error getting ledger entries: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get account summary for selected account
     */
    private function getAccountSummary($account_id, $date_from, $date_to) {
        try {
            // Get account details
            $this->db->select('account_code, account_name, account_type, opening_balance');
            $this->db->from('chart_of_accounts');
            $this->db->where('id', $account_id);
            $account_query = $this->db->get();
            $account = $account_query->row_array();
            
            if (!$account) {
                return null;
            }
            
            // Get opening balance as of date_from
            $opening_balance = $account['opening_balance'];
            
            // Calculate opening balance from transactions before date_from
            if ($date_from > '1900-01-01') {
                $this->db->select('
                    COALESCE(SUM(jl.debit_amount), 0) as period_debits,
                    COALESCE(SUM(jl.credit_amount), 0) as period_credits
                ');
                $this->db->from('journal_lines jl');
                $this->db->join('journal_headers jh', 'jh.id = jl.journal_id', 'inner');
                $this->db->where('jl.account_id', $account_id);
                $this->db->where('jh.date <', $date_from);
                $this->db->where('jh.status', 'posted');
                $opening_query = $this->db->get();
                $opening_data = $opening_query->row_array();
                
                if ($account['account_type'] == 'asset' || $account['account_type'] == 'expense') {
                    $opening_balance += ($opening_data['period_debits'] - $opening_data['period_credits']);
                } else {
                    $opening_balance += ($opening_data['period_credits'] - $opening_data['period_debits']);
                }
            }
            
            // Get period totals
            $this->db->select('
                COALESCE(SUM(jl.debit_amount), 0) as total_debits,
                COALESCE(SUM(jl.credit_amount), 0) as total_credits
            ');
            $this->db->from('journal_lines jl');
            $this->db->join('journal_headers jh', 'jh.id = jl.journal_id', 'inner');
            $this->db->where('jl.account_id', $account_id);
            $this->db->where('jh.date >=', $date_from);
            $this->db->where('jh.date <=', $date_to);
            $this->db->where('jh.status', 'posted');
            $period_query = $this->db->get();
            $period_data = $period_query->row_array();
            
            // Calculate closing balance
            if ($account['account_type'] == 'asset' || $account['account_type'] == 'expense') {
                $closing_balance = $opening_balance + ($period_data['total_debits'] - $period_data['total_credits']);
            } else {
                $closing_balance = $opening_balance + ($period_data['total_credits'] - $period_data['total_debits']);
            }
            
            return [
                'account_code' => $account['account_code'],
                'account_name' => $account['account_name'],
                'account_type' => $account['account_type'],
                'opening_balance' => $opening_balance,
                'total_debits' => $period_data['total_debits'],
                'total_credits' => $period_data['total_credits'],
                'closing_balance' => $closing_balance
            ];
            
        } catch (Exception $e) {
            log_message('error', 'Error getting account summary: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get trial balance data with corrected SQL syntax
     */
    private function getTrialBalanceData($as_of_date) {
        try {
            // Use the existing trial_balance_view but filter by date
            $sql = "
                SELECT 
                    coa.account_code,
                    coa.account_name,
                    coa.account_type,
                    CASE 
                        WHEN coa.account_type IN ('asset', 'expense') THEN 
                            GREATEST((coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)), 0)
                        ELSE 0 
                    END as debit_balance,
                    CASE 
                        WHEN coa.account_type IN ('liability', 'equity', 'revenue') THEN 
                            GREATEST((coa.opening_balance + COALESCE(SUM(jl.credit_amount - jl.debit_amount), 0)), 0)
                        WHEN coa.account_type IN ('asset', 'expense') AND 
                             (coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)) < 0 THEN 
                            ABS((coa.opening_balance + COALESCE(SUM(jl.debit_amount - jl.credit_amount), 0)))
                        ELSE 0 
                    END as credit_balance
                FROM chart_of_accounts coa
                LEFT JOIN journal_lines jl ON jl.account_id = coa.id
                LEFT JOIN journal_headers jh ON jh.id = jl.journal_id 
                    AND jh.status = 'posted' 
                    AND jh.date <= ?
                WHERE coa.is_active = 1
                GROUP BY coa.id, coa.account_code, coa.account_name, coa.account_type, coa.opening_balance
                HAVING debit_balance > 0 OR credit_balance > 0
                ORDER BY coa.account_code ASC
            ";

            $query = $this->db->query($sql, array($as_of_date));
            $accounts = $query->result_array();

            // Initialize account types
            $account_types = [
                'asset' => ['name' => 'Assets', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'liability' => ['name' => 'Liabilities', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'equity' => ['name' => 'Equity', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'revenue' => ['name' => 'Revenue', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                'expense' => ['name' => 'Expenses', 'accounts' => [], 'debit' => 0, 'credit' => 0]
            ];

            // Group accounts by type and calculate totals
            $total_debit = 0;
            $total_credit = 0;

            foreach ($accounts as $account) {
                $type = $account['account_type'];
                if (isset($account_types[$type])) {
                    $account_types[$type]['accounts'][] = $account;
                    $account_types[$type]['debit'] += floatval($account['debit_balance']);
                    $account_types[$type]['credit'] += floatval($account['credit_balance']);
                }
                
                $total_debit += floatval($account['debit_balance']);
                $total_credit += floatval($account['credit_balance']);
            }

            // Check if balanced (allow for small rounding differences)
            $difference = abs($total_debit - $total_credit);
            $is_balanced = $difference < 0.01;

            return [
                'accounts' => $accounts,
                'account_types' => $account_types,
                'total_debit' => $total_debit,
                'total_credit' => $total_credit,
                'difference' => $difference,
                'is_balanced' => $is_balanced
            ];

        } catch (Exception $e) {
            log_message('error', 'Trial Balance Error: ' . $e->getMessage());
            
            // Return empty data structure on error
            return [
                'accounts' => [],
                'account_types' => [
                    'asset' => ['name' => 'Assets', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'liability' => ['name' => 'Liabilities', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'equity' => ['name' => 'Equity', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'revenue' => ['name' => 'Revenue', 'accounts' => [], 'debit' => 0, 'credit' => 0],
                    'expense' => ['name' => 'Expenses', 'accounts' => [], 'debit' => 0, 'credit' => 0]
                ],
                'total_debit' => 0,
                'total_credit' => 0,
                'difference' => 0,
                'is_balanced' => true
            ];
        }
    }

    /**
     * Export General Ledger to Excel
     */
    public function exportGeneralLedger() {
        // Check permission
        if (!$this->rbac->hasPrivilege('financial', 'can_export')) {
            access_denied();
        }
        
        // Get filter parameters
        $account_id = $this->input->get('account_id');
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        // Get data
        $ledger_entries = $this->getLedgerEntries($account_id, $date_from, $date_to);
        
        // Load PHPExcel library (if available)
        if (file_exists(APPPATH . 'libraries/PHPExcel.php')) {
            $this->load->library('PHPExcel');
            
            // Create Excel export
            $objPHPExcel = new PHPExcel();
            $sheet = $objPHPExcel->getActiveSheet();
            
            // Set headers
            $sheet->setCellValue('A1', 'Date');
            $sheet->setCellValue('B1', 'Reference');
            $sheet->setCellValue('C1', 'Account');
            $sheet->setCellValue('D1', 'Description');
            $sheet->setCellValue('E1', 'Debit');
            $sheet->setCellValue('F1', 'Credit');
            $sheet->setCellValue('G1', 'Source');
            
            // Add data
            $row = 2;
            foreach ($ledger_entries as $entry) {
                $sheet->setCellValue("A$row", date('d-M-Y', strtotime($entry['date'])));
                $sheet->setCellValue("B$row", $entry['reference_no']);
                $sheet->setCellValue("C$row", $entry['account_code'] . ' - ' . $entry['account_name']);
                $sheet->setCellValue("D$row", $entry['header_description']);
                $sheet->setCellValue("E$row", $entry['debit_amount'] > 0 ? $entry['debit_amount'] : '');
                $sheet->setCellValue("F$row", $entry['credit_amount'] > 0 ? $entry['credit_amount'] : '');
                $sheet->setCellValue("G$row", $entry['source_module'] ?: 'Manual');
                $row++;
            }
            
            // Download
            $filename = 'General_Ledger_' . date('Y-m-d') . '.xls';
            header('Content-Type: application/vnd.ms-excel');
            header('Content-Disposition: attachment;filename="' . $filename . '"');
            header('Cache-Control: max-age=0');
            
            $objWriter = PHPExcel_IOFactory::createWriter($objPHPExcel, 'Excel5');
            $objWriter->save('php://output');
            exit;
        } else {
            // Fallback to CSV
            $filename = 'General_Ledger_' . date('Y-m-d') . '.csv';
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment;filename="' . $filename . '"');
            
            $output = fopen('php://output', 'w');
            fputcsv($output, ['Date', 'Reference', 'Account', 'Description', 'Debit', 'Credit', 'Source']);
            
            foreach ($ledger_entries as $entry) {
                fputcsv($output, [
                    date('d-M-Y', strtotime($entry['date'])),
                    $entry['reference_no'],
                    $entry['account_code'] . ' - ' . $entry['account_name'],
                    $entry['header_description'],
                    $entry['debit_amount'] > 0 ? $entry['debit_amount'] : '',
                    $entry['credit_amount'] > 0 ? $entry['credit_amount'] : '',
                    $entry['source_module'] ?: 'Manual'
                ]);
            }
            
            fclose($output);
            exit;
        }
    }

    /**
     * Print General Ledger
     */
    public function printGeneralLedger() {
        // Same as export but with print view
        $account_id = $this->input->get('account_id');
        $date_from = $this->input->get('date_from') ?: date('Y-m-01');
        $date_to = $this->input->get('date_to') ?: date('Y-m-d');
        
        $data['ledger_entries'] = $this->getLedgerEntries($account_id, $date_from, $date_to);
        $data['date_from'] = $date_from;
        $data['date_to'] = $date_to;
        
        if ($account_id) {
            $data['account_summary'] = $this->getAccountSummary($account_id, $date_from, $date_to);
        }
        
        $this->load->view('admin/financial/print_general_ledger', $data);
    }

    /**
     * Export Trial Balance
     */
    public function exportTrialBalance() {
        // Check permission
        if (!$this->rbac->hasPrivilege('trial_balance', 'can_export')) {
            access_denied();
        }
        
        $as_of_date = $this->input->get('as_of_date') ?: date('Y-m-d');
        $data = $this->getTrialBalanceData($as_of_date);
        
        // Create CSV export
        $filename = 'Trial_Balance_' . date('Y-m-d') . '.csv';
        header('Content-Type: text/csv');
        header('Content-Disposition: attachment;filename="' . $filename . '"');
        
        $output = fopen('php://output', 'w');
        fputcsv($output, ['Account Code', 'Account Name', 'Account Type', 'Debit Balance', 'Credit Balance']);
        
        foreach ($data['accounts'] as $account) {
            fputcsv($output, [
                $account['account_code'],
                $account['account_name'],
                ucfirst($account['account_type']),
                $account['debit_balance'] > 0 ? $account['debit_balance'] : '',
                $account['credit_balance'] > 0 ? $account['credit_balance'] : ''
            ]);
        }
        
        fclose($output);
        exit;
    }

    private function getBankAccounts() {
        if (!$this->db->table_exists('bank_accounts')) {
            return array();
        }

        try {
            $this->db->select('*');
            $this->db->from('bank_accounts');
            $this->db->where('is_active', 'yes');
            $this->db->order_by('account_name');
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting bank accounts: ' . $e->getMessage());
            return array();
        }
    }

    private function getPettyCashTransactions() {
        // Implementation for petty cash transactions
        return array();
    }

    private function getPettyCashBalance() {
        // Implementation for petty cash balance
        return 0;
    }

    // Debug method to check what tables exist
    public function debug_tables() {
        if (!$this->rbac->hasPrivilege('financial_dashboard', 'can_view')) {
            access_denied();
        }
        
        echo "<h1>Financial System Debug Information</h1>";
        
        // Get all tables in database
        echo "<h2>Available Database Tables</h2>";
        $tables = $this->db->list_tables();
        
        echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
        echo "<tr><th>Table Name</th><th>Exists</th><th>Row Count</th><th>Relevant Fields</th></tr>";
        
        $billing_related_tables = array();
        
        foreach ($tables as $table) {
            $exists = $this->db->table_exists($table);
            
            // Count rows
            $this->db->from($table);
            $row_count = $this->db->count_all_results();
            
            // Get field information
            $fields = $this->db->field_data($table);
            $relevant_fields = array();
            
            foreach ($fields as $field) {
                if (in_array($field->name, array('amount', 'total', 'total_amount', 'charge', 'fee', 'price', 'cost', 'payment', 'status', 'date', 'created_at'))) {
                    $relevant_fields[] = $field->name;
                }
            }
            
            // Check if this looks like a billing table
            if (strpos($table, 'bill') !== false || 
                strpos($table, 'charge') !== false || 
                strpos($table, 'invoice') !== false || 
                strpos($table, 'payment') !== false ||
                strpos($table, 'transaction') !== false ||
                in_array('amount', $relevant_fields) ||
                in_array('total_amount', $relevant_fields)) {
                
                $billing_related_tables[] = $table;
                echo "<tr style='background-color: #e6ffe6;'>";
            } else {
                echo "<tr>";
            }
            
            echo "<td>{$table}</td>";
            echo "<td>" . ($exists ? 'Yes' : 'No') . "</td>";
            echo "<td>{$row_count}</td>";
            echo "<td>" . implode(', ', $relevant_fields) . "</td>";
            echo "</tr>";
        }
        echo "</table>";
        
        echo "<h2>Billing-Related Tables Found</h2>";
        echo "<ul>";
        foreach ($billing_related_tables as $table) {
            echo "<li><strong>{$table}</strong></li>";
        }
        echo "</ul>";
        
        // Test revenue calculation
        echo "<h2>Revenue Test</h2>";
        try {
            $current_year = date('Y');
            $revenue = $this->getRevenueFromBilling($current_year);
            echo "<p>Total Revenue for {$current_year}: " . number_format($revenue, 2) . "</p>";
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error calculating revenue: " . $e->getMessage() . "</p>";
        }
        
        // Test metrics
        echo "<h2>Financial Metrics Test</h2>";
        try {
            $metrics = $this->getFinancialMetrics();
            echo "<table border='1' style='border-collapse: collapse;'>";
            echo "<tr><th>Metric</th><th>Value</th></tr>";
            foreach ($metrics as $key => $value) {
                echo "<tr><td>{$key}</td><td>" . number_format($value, 2) . "</td></tr>";
            }
            echo "</table>";
        } catch (Exception $e) {
            echo "<p style='color: red;'>Error getting metrics: " . $e->getMessage() . "</p>";
        }
        
        echo "<h2>Recommendations</h2>";
        echo "<ul>";
        if (empty($billing_related_tables)) {
            echo "<li>No billing tables found. You may need to set up billing functionality first.</li>";
        } else {
            echo "<li>Found " . count($billing_related_tables) . " potential billing tables.</li>";
            echo "<li>The financial dashboard should work with these tables.</li>";
        }
        echo "<li>Consider running the financial setup to create proper financial tables.</li>";
        echo "<li>Visit: <a href='" . base_url() . "admin/financial_setup/initialize'>Financial Setup</a></li>";
        echo "</ul>";
    }
}