<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Consultation extends Admin_Controller {

    function __construct() {
        parent::__construct();
        
        // Load essential libraries
        try {
            $this->load->model('patient_model');
            $this->load->model('staff_model');
            $this->load->library('datatables');
            $this->load->library('customlib');
        } catch (Exception $e) {
            log_message('error', 'Consultation controller error: ' . $e->getMessage());
            show_error('Failed to load required libraries: ' . $e->getMessage());
        }
    }

    public function index() {
        // Set session data
        $this->session->set_userdata('top_menu', 'OPD');
        $this->session->set_userdata('sub_menu', 'consultation');

        // Prepare data for view
        $data['title'] = 'Patient Consultation';
        
        // Get doctors
        try {
            if (method_exists($this->staff_model, 'getStaffbyrole')) {
                $data['doctors'] = $this->staff_model->getStaffbyrole(3); // Role 3 = doctors
            } else {
                // Alternative query to get doctors
                $this->db->select('s.*, r.name as role_name');
                $this->db->from('staff s');
                $this->db->join('staff_roles sr', 'sr.staff_id = s.id', 'left');
                $this->db->join('roles r', 'r.id = sr.role_id', 'left');
                $this->db->where('r.name', 'Doctor');
                $this->db->or_where('r.id', 3);
                $this->db->where('s.is_active', 'yes');
                $query = $this->db->get();
                $data['doctors'] = $query->result_array();
            }
        } catch (Exception $e) {
            // Fallback - get all active staff
            $this->db->select('id, name, surname, employee_id');
            $this->db->from('staff');
            $this->db->where('is_active', 'yes');
            $query = $this->db->get();
            $data['doctors'] = $query->result_array();
        }
        
        // Get gender list
        try {
            if (method_exists($this->customlib, 'getGender_Patient')) {
                $data['genderList'] = $this->customlib->getGender_Patient();
            } else {
                $data['genderList'] = array(
                    'male' => 'Male',
                    'female' => 'Female',
                    'other' => 'Other'
                );
            }
        } catch (Exception $e) {
            $data['genderList'] = array('male' => 'Male', 'female' => 'Female');
        }
        
        // Load views
        try {
            $this->load->view('layout/header', $data);
            $this->load->view('admin/consultation/index', $data);
            $this->load->view('layout/footer', $data);
        } catch (Exception $e) {
            log_message('error', 'View loading error: ' . $e->getMessage());
            show_error('Failed to load consultation view: ' . $e->getMessage());
        }
    }

    // =============================================
    // PATIENT MANAGEMENT METHODS
    // =============================================

    public function getAllPatients() {
    // CRITICAL: Clean output buffer and set headers
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    // Suppress any error output that could corrupt JSON
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        // Get all active patients with fingerprint info
        $this->db->select('id, patient_name, mobileno, gender, age');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('patient_name', 'ASC');
        $this->db->limit(100);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'patients' => [],
                'count' => 0,
                'error' => 'Database query failed: ' . $this->db->error()['message']
            ]);
            exit;
        }
        
        $patients = $query->result_array();
        
        // Add fingerprint status to each patient if fingerprint columns exist
        foreach ($patients as &$patient) {
            if ($this->db->field_exists('fingerprint_template', 'patients')) {
                // Get fingerprint status separately to avoid large data transfer
                $this->db->select('fingerprint_template');
                $this->db->from('patients');
                $this->db->where('id', $patient['id']);
                $fp_query = $this->db->get();
                
                if ($fp_query && $fp_query->num_rows() > 0) {
                    $fp_data = $fp_query->row();
                    $patient['has_fingerprint'] = !empty($fp_data->fingerprint_template);
                } else {
                    $patient['has_fingerprint'] = false;
                }
            } else {
                $patient['has_fingerprint'] = false;
            }
        }
        
        // Clean output buffer before sending JSON
        ob_clean();
        echo json_encode([
            'patients' => $patients,
            'count' => count($patients),
            'message' => 'Found ' . count($patients) . ' patients'
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'patients' => [],
            'count' => 0,
            'error' => 'Exception: ' . $e->getMessage()
        ]);
        exit;
    }
}
    
    public function getIcdDiagnoses() {
        header('Content-Type: application/json');
        
        try {
            // Load the consultation model if not already loaded
            if (!isset($this->consultation_model)) {
                $this->load->model('consultation_model');
            }
            
            $icdVersion = $this->input->get('version') ?: 'ICD10';
            $limit = $this->input->get('limit') ?: 1000;
            
            $diagnoses = $this->consultation_model->getIcdDiagnosisCodes($icdVersion, $limit);
            
            echo json_encode([
                'status' => 'success',
                'message' => 'ICD diagnoses loaded successfully',
                'diagnoses' => $diagnoses,
                'count' => count($diagnoses)
            ]);
            
        } catch (Exception $e) {
            log_message('error', 'Error loading ICD diagnoses: ' . $e->getMessage());
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to load ICD diagnoses: ' . $e->getMessage(),
                'diagnoses' => []
            ]);
        }
    }
    
/**
 * ENHANCED VERSION with error handling - Save consultation with financials
 */
public function saveWithFinancials() {
    // Set proper headers first
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    
    // Handle errors properly
    error_reporting(E_ALL);
    ini_set('display_errors', 0);
    
    // Start output buffering to catch any unwanted output
    ob_start();
    
    try {
        // Clear any previous output
        ob_clean();
        
        log_message('debug', 'saveWithFinancials started');
        log_message('debug', 'POST data: ' . json_encode($_POST));
        
        // Ensure tables exist
        $this->ensureTablesExist();
        
        // Validate required fields
        $patient_id = $this->input->post('patient_id');
        $symptoms = $this->input->post('symptoms');
        $doctor = $this->input->post('doctor');
        $consultation_date = $this->input->post('consultation_date'); // This comes from form
        $consultation_charge = $this->input->post('consultation_charge');
        
        if (!$patient_id || !$symptoms || !$doctor) { // Remove consultation_date requirement if optional
            ob_clean();
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Missing required fields: patient_id, symptoms, or doctor'
            ));
            ob_end_flush();
            return;
        }
        
        // Start transaction
        $this->db->trans_start();
        
        // Prepare consultation data - FIXED: Use correct database column names
        $consultationData = array(
            'patient_id' => (int)$patient_id,
            'doctor' => (int)$doctor,
            'date' => $consultation_date ?: date('Y-m-d'), // Use 'date' not 'consultation_date'
            'symptoms' => $symptoms,
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $this->input->post('verification_status') == '1' ? 'verified' : 'unverified',
            'created_by' => $this->session->userdata('admin_id') ?: 1,
            'created_at' => date('Y-m-d H:i:s')
        );
        
        // Handle diagnosis
        $diagnosisType = $this->input->post('diagnosis_type');
        if ($diagnosisType === 'custom' && $this->input->post('diagnosis_custom')) {
            $consultationData['diagnosis'] = $this->input->post('diagnosis_custom');
        } elseif ($diagnosisType === 'icd10' && $this->input->post('diagnosis_title')) {
            $diagnosisCode = $this->input->post('diagnosis_code');
            $diagnosisTitle = $this->input->post('diagnosis_title');
            $consultationData['diagnosis'] = $diagnosisCode . ' - ' . $diagnosisTitle;
        } else {
            $consultationData['diagnosis'] = '';
        }
        
        // Insert consultation
        $this->db->insert('consultations', $consultationData);
        $consultationId = $this->db->insert_id();
        
        if (!$consultationId) {
            throw new Exception('Failed to insert consultation');
        }
        
        // Handle financial data if provided
        if ($consultation_charge && floatval($consultation_charge) > 0) {
            $this->saveFinancialData($consultationId, $consultation_charge);
        }
        
        // Complete transaction
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }
        
        // Clear buffer and send success response
        ob_clean();
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Consultation saved successfully',
            'consultation_id' => $consultationId
        ));
        ob_end_flush();
        
    } catch (Exception $e) {
        // Rollback transaction
        $this->db->trans_rollback();
        
        log_message('error', 'saveWithFinancials error: ' . $e->getMessage());
        
        // Clear buffer and send error response
        ob_clean();
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage(),
            'debug' => array(
                'file' => $e->getFile(),
                'line' => $e->getLine()
            )
        ));
        ob_end_flush();
    }
}

/**
 * Save financial data separately
 */
private function saveFinancialData($consultationId, $consultationCharge) {
    $baseAmount = floatval($consultationCharge);
    $discountPercent = floatval($this->input->post('discount_percentage')) ?: 0;
    $taxPercent = floatval($this->input->post('tax_percentage')) ?: 0;
    
    // Calculate amounts
    $discountAmount = ($baseAmount * $discountPercent) / 100;
    $amountAfterDiscount = $baseAmount - $discountAmount;
    $taxAmount = ($amountAfterDiscount * $taxPercent) / 100;
    $totalAmount = $amountAfterDiscount + $taxAmount;
    
    $financialData = array(
        'consultation_id' => $consultationId,
        'base_amount' => $baseAmount,
        'discount_percentage' => $discountPercent,
        'discount_amount' => $discountAmount,
        'tax_percentage' => $taxPercent,
        'tax_amount' => $taxAmount,
        'total_amount' => $totalAmount,
        'payment_mode' => $this->input->post('payment_mode') ?: '',
        'payment_status' => $this->input->post('payment_status') ?: 'pending',
        'payment_notes' => $this->input->post('payment_notes') ?: '',
        'created_at' => date('Y-m-d H:i:s')
    );
    
    $this->db->insert('consultation_financials', $financialData);
    
    if ($this->db->error()['code'] != 0) {
        throw new Exception('Failed to insert financial data: ' . $this->db->error()['message']);
    }
}

public function saveWithLabTests() {
    // CRITICAL: Clean output buffer and set headers FIRST
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);

    try {
        // Log input data for debugging
        log_message('debug', 'Input data: ' . print_r($_POST, true));

        // Start database transaction
        $this->db->trans_start();

        // Save consultation first
        $consultation_data = [
            'patient_id' => $this->input->post('patient_id'),
            'symptoms' => $this->input->post('symptoms'),
            'doctor' => $this->input->post('doctor'),
            'date' => $this->input->post('consultation_date') ?: date('Y-m-d'),
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $this->input->post('verification_status') ?: 'unverified',
            'verification_score' => $this->input->post('verification_score') ?: 0,
            'verification_quality' => $this->input->post('verification_quality') ?: '',
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        // Insert consultation
        $this->db->insert('consultations', $consultation_data);

        // Check for database errors
        if ($this->db->error()['code'] !== 0) {
            throw new Exception('Database error: ' . $this->db->error()['message']);
        }

        $consultation_id = $this->db->insert_id();

        if ($consultation_id === 0) {
            throw new Exception('Failed to insert consultation data.');
        }

        // Save selected lab tests
        $selected_tests = json_decode($this->input->post('selected_tests'), true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('Invalid JSON for selected tests: ' . json_last_error_msg());
        }

        $lab_test_ids = [];

        if (!empty($selected_tests)) {
            foreach ($selected_tests as $test) {
                $lab_data = [
                    'patient_id' => $consultation_data['patient_id'],
                    'test_id' => isset($test['id']) ? $test['id'] : null,
                    'test_name' => $test['test_name'],
                    'short_name' => isset($test['short_name']) ? $test['short_name'] : '',
                    'test_result' => '',
                    'notes' => $this->input->post('test_instructions') ?: '',
                    'test_date' => date('Y-m-d'),
                    'priority' => $this->input->post('test_priority') ?: 'normal',
                    'base_cost' => floatval($test['cost']),
                    'total_cost' => floatval($test['cost']),
                    'final_amount' => floatval($test['cost']),
                    'result_status' => 'pending',
                    'doctor_id' => $consultation_data['doctor'],
                    'verification_status' => 'unverified',
                    'tested_by' => $this->getStaffId(),
                    'created_at' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('laboratory', $lab_data);

                if ($this->db->error()['code'] !== 0) {
                    throw new Exception('Database error: ' . $this->db->error()['message']);
                }

                $lab_test_ids[] = $this->db->insert_id();
            }
        }

        // Complete transaction
        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }

        // Clean output buffer before sending response
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'Consultation and lab tests saved successfully',
            'consultation_id' => $consultation_id,
            'lab_test_ids' => $lab_test_ids,
            'total_tests_saved' => count($lab_test_ids)
        ]);
        exit;

    } catch (Exception $e) {
        // Rollback transaction on error
        $this->db->trans_rollback();
        
        log_message('error', 'Error saving consultation: ' . $e->getMessage());
        
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage()
        ]);
        exit;
    }
}

/**
 * ALTERNATIVE: Create a link between consultations and lab tests using a separate table
 * This method creates a consultation_lab_tests table to link consultations with lab tests
 */
public function saveWithLabTestsLinked() {
    header('Content-Type: application/json');

    try {
        // Start database transaction
        $this->db->trans_start();

        // Create consultation_lab_tests table if it doesn't exist
        $this->createConsultationLabTestsTable();

        // Save consultation first
        $consultation_data = [
            'patient_id' => $this->input->post('patient_id'),
            'symptoms' => $this->input->post('symptoms'),
            'doctor' => $this->input->post('doctor'),
            'date' => $this->input->post('consultation_date') ?: date('Y-m-d'),
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $this->input->post('verification_status') ?: 'unverified',
            'created_by' => $this->getStaffId(),
            'created_at' => date('Y-m-d H:i:s')
        ];

        $this->db->insert('consultations', $consultation_data);
        $consultation_id = $this->db->insert_id();

        if (!$consultation_id) {
            throw new Exception('Failed to insert consultation data.');
        }

        // Save lab tests
        $selected_tests = json_decode($this->input->post('selected_tests'), true);
        $lab_test_ids = [];

        if (!empty($selected_tests)) {
            foreach ($selected_tests as $test) {
                // Insert into laboratory table
                $lab_data = [
                    'patient_id' => $consultation_data['patient_id'],
                    'test_id' => isset($test['id']) ? $test['id'] : null,
                    'test_name' => $test['test_name'],
                    'short_name' => isset($test['short_name']) ? $test['short_name'] : '',
                    'test_result' => '',
                    'notes' => $this->input->post('test_instructions') ?: '',
                    'test_date' => date('Y-m-d'),
                    'priority' => $this->input->post('test_priority') ?: 'normal',
                    'base_cost' => floatval($test['cost']),
                    'total_cost' => floatval($test['cost']),
                    'final_amount' => floatval($test['cost']),
                    'result_status' => 'pending',
                    'doctor_id' => $consultation_data['doctor'],
                    'verification_status' => 'unverified',
                    'tested_by' => $this->getStaffId(),
                    'created_at' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('laboratory', $lab_data);
                $lab_id = $this->db->insert_id();
                $lab_test_ids[] = $lab_id;

                // Create link between consultation and lab test
                $link_data = [
                    'consultation_id' => $consultation_id,
                    'lab_test_id' => $lab_id,
                    'created_at' => date('Y-m-d H:i:s')
                ];

                $this->db->insert('consultation_lab_tests', $link_data);
            }
        }

        // Complete transaction
        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            throw new Exception('Transaction failed');
        }

        echo json_encode([
            'status' => 'success',
            'message' => 'Consultation and lab tests saved successfully with links',
            'consultation_id' => $consultation_id,
            'lab_test_ids' => $lab_test_ids,
            'total_tests_saved' => count($lab_test_ids)
        ]);

    } catch (Exception $e) {
        $this->db->trans_rollback();
        
        log_message('error', 'Error saving consultation with lab tests: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage()
        ]);
    }
}

/**
 * Create consultation_lab_tests linking table
 */
private function createConsultationLabTestsTable() {
    if (!$this->db->table_exists('consultation_lab_tests')) {
        $sql = "CREATE TABLE `consultation_lab_tests` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `consultation_id` int(11) NOT NULL,
            `lab_test_id` int(11) NOT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `consultation_id` (`consultation_id`),
            KEY `lab_test_id` (`lab_test_id`),
            CONSTRAINT `fk_consultation_lab_consultation` FOREIGN KEY (`consultation_id`) REFERENCES `consultations` (`id`) ON DELETE CASCADE,
            CONSTRAINT `fk_consultation_lab_laboratory` FOREIGN KEY (`lab_test_id`) REFERENCES `laboratory` (`id`) ON DELETE CASCADE
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        $this->db->query($sql);
        
        if ($this->db->error()['code'] != 0) {
            log_message('error', 'Failed to create consultation_lab_tests table: ' . $this->db->error()['message']);
        } else {
            log_message('info', 'Created consultation_lab_tests table successfully');
        }
    }
}


/**
 * Save diagnosis relationship if consultation_diagnosis table exists
 */
private function saveDiagnosisRelation($consultationId, $diagnosisCodeId) {
    try {
        if ($this->db->table_exists('consultation_diagnosis')) {
            $diagnosisRelation = [
                'consultation_id' => $consultationId,
                'diagnosis_code_id' => (int)$diagnosisCodeId,
                'is_primary' => 1,
                'created_at' => date('Y-m-d H:i:s')
            ];
            
            $this->db->insert('consultation_diagnosis', $diagnosisRelation);
            log_message('debug', 'Diagnosis relationship saved for consultation: ' . $consultationId);
        }
    } catch (Exception $e) {
        // Don't fail the main transaction for diagnosis relationship issues
        log_message('warning', 'Could not save diagnosis relationship: ' . $e->getMessage());
    }
}

private function createFinancialTable() {
    if (!$this->db->table_exists('consultation_financials')) {
        $sql = "CREATE TABLE IF NOT EXISTS `consultation_financials` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `consultation_id` int(11) NOT NULL,
            `base_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `discount_percentage` decimal(5,2) NOT NULL DEFAULT 0.00,
            `discount_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `tax_percentage` decimal(5,2) NOT NULL DEFAULT 0.00,
            `tax_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `total_amount` decimal(10,2) NOT NULL DEFAULT 0.00,
            `payment_mode` varchar(50) DEFAULT NULL,
            `payment_status` enum('pending','paid','partial') DEFAULT 'pending',
            `payment_notes` text DEFAULT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `consultation_id` (`consultation_id`),
            KEY `payment_status` (`payment_status`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
        
        $this->db->query($sql);
        
        if ($this->db->error()['code'] != 0) {
            log_message('error', 'Failed to create consultation_financials table: ' . $this->db->error()['message']);
        } else {
            log_message('info', 'Created consultation_financials table successfully');
        }
    }
}

private function ensureDiagnosisCodesTables() {
    try {
        // Create diagnosis_codes table if it doesn't exist
        if (!$this->db->table_exists('diagnosis_codes')) {
            $sql = "CREATE TABLE IF NOT EXISTS `diagnosis_codes` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `icd_version` enum('ICD10','ICD11') NOT NULL DEFAULT 'ICD10',
                `code` varchar(20) NOT NULL,
                `title` varchar(500) NOT NULL,
                `description` text,
                `category` varchar(100) DEFAULT NULL,
                `subcategory` varchar(100) DEFAULT NULL,
                `is_active` tinyint(1) NOT NULL DEFAULT '1',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `unique_code_version` (`code`, `icd_version`),
                KEY `idx_icd_version` (`icd_version`),
                KEY `idx_category` (`category`),
                KEY `idx_active` (`is_active`),
                KEY `idx_code` (`code`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->db->query($sql);
            
            if ($this->db->error()['code'] == 0) {
                log_message('info', 'Created diagnosis_codes table successfully');
                $this->insertSampleDiagnosisCodes();
            }
        }
        
        // Create consultation_diagnosis table if it doesn't exist
        if (!$this->db->table_exists('consultation_diagnosis')) {
            $sql = "CREATE TABLE IF NOT EXISTS `consultation_diagnosis` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `consultation_id` int(11) NOT NULL,
                `diagnosis_code_id` int(11) NOT NULL,
                `is_primary` tinyint(1) NOT NULL DEFAULT '0',
                `notes` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `fk_consultation_diagnosis_consultation` (`consultation_id`),
                KEY `fk_consultation_diagnosis_code` (`diagnosis_code_id`),
                KEY `idx_consultation_primary` (`consultation_id`, `is_primary`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci";
            
            $this->db->query($sql);
            
            if ($this->db->error()['code'] == 0) {
                log_message('info', 'Created consultation_diagnosis table successfully');
            }
        }
        
    } catch (Exception $e) {
        log_message('warning', 'Could not create diagnosis tables: ' . $e->getMessage());
    }
}

/**
 * Search ICD-10 diagnosis codes
 */
public function searchIcdDiagnoses() {
    header('Content-Type: application/json');
    
    try {
        if (!isset($this->consultation_model)) {
            $this->load->model('consultation_model');
        }
        
        $searchTerm = $this->input->get('term') ?: $this->input->post('term');
        $category = $this->input->get('category');
        $icdVersion = $this->input->get('version') ?: 'ICD10';
        $limit = $this->input->get('limit') ?: 50;
        
        if (empty($searchTerm)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Search term required',
                'diagnoses' => []
            ]);
            return;
        }
        
        $diagnoses = $this->consultation_model->searchIcdDiagnosisCodes($searchTerm, $icdVersion, $category, $limit);
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Search completed',
            'diagnoses' => $diagnoses,
            'count' => count($diagnoses),
            'search_term' => $searchTerm
        ]);
        
    } catch (Exception $e) {
        log_message('error', 'Error searching ICD diagnoses: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Search failed: ' . $e->getMessage(),
            'diagnoses' => []
        ]);
    }
}

    /**
     * FIXED: Get patient details with proper error handling
     */
    public function getPatientDetails() {
        header('Content-Type: application/json');
        
        try {
            $patientId = $this->input->post('id');
            
            if (!$patientId) {
                echo json_encode([
                    'error' => 'Patient ID required',
                    'debug' => [
                        'post_data' => $_POST,
                        'get_data' => $_GET
                    ]
                ]);
                return;
            }
            
            // Check if patients table exists
            if (!$this->db->table_exists('patients')) {
                echo json_encode([
                    'error' => 'Patients table does not exist',
                    'patient_id' => $patientId
                ]);
                return;
            }
            
            // Start with basic patient query
            $this->db->select('*');
            $this->db->from('patients');
            $this->db->where('id', $patientId);
            
            $query = $this->db->get();
            
            if (!$query) {
                echo json_encode([
                    'error' => 'Query failed: ' . $this->db->error()['message'],
                    'patient_id' => $patientId,
                    'last_query' => $this->db->last_query()
                ]);
                return;
            }
            
            if ($query->num_rows() == 0) {
                echo json_encode([
                    'error' => 'Patient not found with ID: ' . $patientId,
                    'patient_id' => $patientId,
                    'last_query' => $this->db->last_query()
                ]);
                return;
            }
            
            $patient = $query->row_array();
            
            // Get blood group name if blood_group table exists
            if ($this->db->table_exists('blood_group') && !empty($patient['blood_group'])) {
                try {
                    $this->db->select('name');
                    $this->db->from('blood_group');
                    $this->db->where('id', $patient['blood_group']);
                    $bg_query = $this->db->get();
                    
                    if ($bg_query && $bg_query->num_rows() > 0) {
                        $patient['blood_group_name'] = $bg_query->row()->name;
                    } else {
                        $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
                    }
                } catch (Exception $e) {
                    $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
                }
            } else {
                $patient['blood_group_name'] = $patient['blood_group'] ?? 'Unknown';
            }
            
            // Process patient data
            $patient['patient_age'] = !empty($patient['age']) ? $patient['age'] . ' years' : 'Unknown';
            $patient['image'] = !empty($patient['image']) ? $patient['image'] : 'uploads/patient_images/no_image.png';
            
            // Set defaults for missing fields
            $patient['patient_name'] = $patient['patient_name'] ?? 'Unknown';
            $patient['gender'] = $patient['gender'] ?? 'Unknown';
            $patient['mobileno'] = $patient['mobileno'] ?? '';
            $patient['known_allergies'] = $patient['known_allergies'] ?? 'None';
            
            // Fingerprint verification status
            $hasFingerprint = false;
            if ($this->db->field_exists('fingerprint_template', 'patients')) {
                $hasFingerprint = !empty($patient['fingerprint_template']);
            }
            
            $patient['has_fingerprint'] = $hasFingerprint;
            $patient['verification_capable'] = $hasFingerprint && strlen($patient['fingerprint_template'] ?? '') > 100;
            
            // Security: Create fingerprint status object without exposing template
            $patient['fingerprint_status'] = [
                'enrolled' => $hasFingerprint,
                'template_size' => strlen($patient['fingerprint_template'] ?? ''),
                'image_available' => !empty($patient['fingerprint_image']),
                'enrollment_date' => $patient['created_at'] ?? null
            ];
            
            // Remove sensitive data from response - template will be fetched separately when needed
            unset($patient['fingerprint_template']);
            if (isset($patient['fingerprint_image'])) {
                unset($patient['fingerprint_image']);
            }
            
            echo json_encode($patient);
            
        } catch (Exception $e) {
            echo json_encode([
                'error' => 'Database error: ' . $e->getMessage(),
                'patient_id' => $patientId ?? 'undefined',
                'line' => $e->getLine(),
                'file' => $e->getFile(),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }

/**
 * ENHANCED: Get stored fingerprint template with better error handling
 */
public function getStoredFingerprintTemplate() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        // Check if fingerprint columns exist
        if (!$this->db->field_exists('fingerprint_template', 'patients')) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Fingerprint functionality not available - fingerprint_template column missing',
                'debug' => [
                    'table_exists' => $this->db->table_exists('patients'),
                    'available_fields' => $this->db->list_fields('patients')
                ]
            ]);
            return;
        }
        
        // Get only fingerprint data with better query
        $this->db->select('id, patient_name, fingerprint_template, fingerprint_image');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $patient = $query->row_array();
            
            if (!empty($patient['fingerprint_template'])) {
                // Clean the template data
                $template = trim($patient['fingerprint_template']);
                
                echo json_encode([
                    'status' => 'success',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'fingerprint_template' => $template,
                    'fingerprint_image' => $patient['fingerprint_image'] ?? '',
                    'template_size' => strlen($template),
                    'has_image' => !empty($patient['fingerprint_image']),
                    'template_type' => $this->detectTemplateType($template),
                    'debug' => [
                        'template_preview' => substr($template, 0, 50) . '...',
                        'encoding_detected' => $this->detectEncoding($template)
                    ]
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'No fingerprint template found for this patient',
                    'patient_id' => $patientId,
                    'patient_name' => $patient['patient_name'],
                    'debug' => [
                        'template_exists' => isset($patient['fingerprint_template']),
                        'template_empty' => empty($patient['fingerprint_template'])
                    ]
                ]);
            }
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or inactive',
                'patient_id' => $patientId,
                'debug' => [
                    'last_query' => $this->db->last_query()
                ]
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'debug' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

/**
 * ENHANCED: Verify fingerprint with comprehensive logging
 */
public function verifyFingerprint() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        $matchScore = $this->input->post('match_score');
        $isMatch = $this->input->post('is_match');
        $threshold = $this->input->post('threshold') ?? 60;
        $serviceUrl = $this->input->post('service_url');
        
        // Enhanced validation
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required',
                'received_data' => $_POST
            ]);
            return;
        }
        
        if (!is_numeric($matchScore)) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Invalid match score: ' . $matchScore,
                'received_data' => $_POST
            ]);
            return;
        }
        
        // Validate patient exists and get details
        $this->db->select('id, patient_name, fingerprint_template');
        $this->db->from('patients');
        $this->db->where('id', $patientId);
        $this->db->where('is_active', 'yes');
        $patient_query = $this->db->get();
        
        if ($patient_query->num_rows() == 0) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient not found or inactive',
                'patient_id' => $patientId
            ]);
            return;
        }
        
        $patient = $patient_query->row_array();
        
        // Check if patient has fingerprint template
        $hasStoredTemplate = !empty($patient['fingerprint_template']);
        
        // Log verification attempt with detailed information
        $verificationLogId = $this->logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold, $serviceUrl, $hasStoredTemplate);
        
        // Prepare detailed response
        $verificationResult = [
            'patient_id' => $patientId,
            'patient_name' => $patient['patient_name'],
            'verification_type' => '1:1',
            'timestamp' => date('Y-m-d H:i:s'),
            'staff_id' => $this->getStaffId(),
            'match_score' => floatval($matchScore),
            'threshold_used' => intval($threshold),
            'max_possible_score' => 199,
            'service_url' => $serviceUrl ?? 'unknown',
            'verification_log_id' => $verificationLogId,
            'has_stored_template' => $hasStoredTemplate,
            'template_size' => $hasStoredTemplate ? strlen($patient['fingerprint_template']) : 0
        ];
        
        // Determine verification result
        $matchPassed = ($isMatch == 'true' || $isMatch === true || $isMatch == 1);
        
        if ($matchPassed && $hasStoredTemplate) {
            $verificationResult['status'] = 'success';
            $verificationResult['message'] = 'Patient identity verified successfully';
            $verificationResult['verification_passed'] = true;
            $verificationResult['verification_method'] = 'fingerprint_match';
        } else if (!$hasStoredTemplate) {
            $verificationResult['status'] = 'warning';
            $verificationResult['message'] = 'No stored fingerprint available for verification';
            $verificationResult['verification_passed'] = false;
            $verificationResult['verification_method'] = 'no_template';
        } else {
            $verificationResult['status'] = 'failed';
            $verificationResult['message'] = 'Fingerprint verification failed - identity not confirmed';
            $verificationResult['verification_passed'] = false;
            $verificationResult['verification_method'] = 'fingerprint_mismatch';
        }
        
        // Add quality assessment
        $verificationResult['quality_assessment'] = $this->assessVerificationQuality($matchScore, $threshold, $hasStoredTemplate);
        
        echo json_encode($verificationResult);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Verification error: ' . $e->getMessage(),
            'patient_id' => $patientId ?? 'unknown',
            'debug' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

/**
 * ENHANCED: Log fingerprint verification with detailed tracking
 */
private function logFingerprintVerification($patientId, $matchScore, $isMatch, $threshold = 60, $serviceUrl = null, $hasStoredTemplate = true) {
    try {
        // Create verification log table if it doesn't exist
        $this->createVerificationLogTable();
        
        $logData = [
            'patient_id' => $patientId,
            'verification_type' => '1:1',
            'match_score' => floatval($matchScore),
            'is_match' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 1 : 0,
            'threshold_used' => intval($threshold),
            'service_url' => $serviceUrl,
            'verification_date' => date('Y-m-d H:i:s'),
            'staff_id' => $this->getStaffId(),
            'ip_address' => $this->input->ip_address(),
            'user_agent' => $this->input->user_agent(),
            'has_stored_template' => $hasStoredTemplate ? 1 : 0,
            'verification_status' => ($isMatch == 'true' || $isMatch === true || $isMatch == 1) ? 'success' : 'failed'
        ];
        
        $this->db->insert('fingerprint_verification_log', $logData);
        return $this->db->insert_id();
        
    } catch (Exception $e) {
        log_message('error', 'Failed to log verification: ' . $e->getMessage());
        return null;
    }
}

/**
 * Create enhanced verification log table
 */
private function createVerificationLogTable() {
    if (!$this->db->table_exists('fingerprint_verification_log')) {
        $sql = "CREATE TABLE `fingerprint_verification_log` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` varchar(50) NOT NULL,
            `verification_type` enum('1:1','1:N') DEFAULT '1:1',
            `match_score` decimal(5,2) DEFAULT NULL,
            `is_match` tinyint(1) DEFAULT 0,
            `threshold_used` int(3) DEFAULT 60,
            `service_url` varchar(255) DEFAULT NULL,
            `verification_date` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `staff_id` int(11) DEFAULT NULL,
            `ip_address` varchar(45) DEFAULT NULL,
            `user_agent` text DEFAULT NULL,
            `has_stored_template` tinyint(1) DEFAULT 1,
            `verification_status` enum('success','failed','error') DEFAULT 'failed',
            `notes` text DEFAULT NULL,
            PRIMARY KEY (`id`),
            KEY `patient_id` (`patient_id`),
            KEY `verification_date` (`verification_date`),
            KEY `staff_id` (`staff_id`),
            KEY `verification_status` (`verification_status`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
        
        $this->db->query($sql);
    } else {
        // Add missing columns to existing table
        $this->addVerificationLogColumns();
    }
}

/**
 * Add missing columns to verification log table
 */
private function addVerificationLogColumns() {
    try {
        if (!$this->db->field_exists('has_stored_template', 'fingerprint_verification_log')) {
            $sql = "ALTER TABLE `fingerprint_verification_log` 
                    ADD COLUMN `has_stored_template` TINYINT(1) DEFAULT 1 AFTER `user_agent`";
            $this->db->query($sql);
        }
        
        if (!$this->db->field_exists('verification_status', 'fingerprint_verification_log')) {
            $sql = "ALTER TABLE `fingerprint_verification_log` 
                    ADD COLUMN `verification_status` ENUM('success','failed','error') DEFAULT 'failed' AFTER `has_stored_template`";
            $this->db->query($sql);
        }
    } catch (Exception $e) {
        log_message('error', 'Error adding verification log columns: ' . $e->getMessage());
    }
}

/**
 * Assess verification quality
 */
private function assessVerificationQuality($matchScore, $threshold, $hasStoredTemplate) {
    if (!$hasStoredTemplate) {
        return [
            'quality' => 'no_template',
            'description' => 'No stored fingerprint template available',
            'recommendation' => 'Enroll patient fingerprint for future verifications'
        ];
    }
    
    $score = floatval($matchScore);
    
    if ($score >= ($threshold + 30)) {
        return [
            'quality' => 'excellent',
            'description' => 'Very high confidence match',
            'recommendation' => 'Proceed with confidence'
        ];
    } else if ($score >= ($threshold + 10)) {
        return [
            'quality' => 'good',
            'description' => 'High confidence match',
            'recommendation' => 'Proceed normally'
        ];
    } else if ($score >= $threshold) {
        return [
            'quality' => 'acceptable',
            'description' => 'Acceptable match above threshold',
            'recommendation' => 'Proceed with normal caution'
        ];
    } else if ($score >= ($threshold - 10)) {
        return [
            'quality' => 'low',
            'description' => 'Below threshold but close',
            'recommendation' => 'Consider retry or manual verification'
        ];
    } else {
        return [
            'quality' => 'poor',
            'description' => 'Well below threshold',
            'recommendation' => 'Retry verification or use alternative method'
        ];
    }
}

/**
 * Detect template type (ISO, ANSI, etc.)
 */
private function detectTemplateType($template) {
    if (empty($template)) {
        return 'none';
    }
    
    // Basic detection based on common patterns
    if (strpos($template, 'FMR') === 0) {
        return 'ISO_19794-2';
    } else if (strpos($template, 'ANSI') === 0) {
        return 'ANSI_378';
    } else if (strlen($template) > 100 && ctype_alnum(str_replace(['+', '/', '='], '', $template))) {
        return 'base64_encoded';
    } else {
        return 'unknown';
    }
}

/**
 * Detect encoding type
 */
private function detectEncoding($data) {
    if (empty($data)) {
        return 'none';
    }
    
    if (base64_encode(base64_decode($data, true)) === $data) {
        return 'base64';
    } else if (ctype_xdigit($data)) {
        return 'hexadecimal';
    } else {
        return 'binary_or_text';
    }
}

/**
 * Get verification history for a patient
 */
public function getVerificationHistory() {
    header('Content-Type: application/json');
    
    try {
        $patientId = $this->input->post('patient_id');
        $limit = $this->input->post('limit') ?? 10;
        
        if (!$patientId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            return;
        }
        
        if (!$this->db->table_exists('fingerprint_verification_log')) {
            echo json_encode([
                'status' => 'success',
                'verifications' => [],
                'message' => 'No verification history available'
            ]);
            return;
        }
        
        $this->db->select('v.*, s.name as staff_name, s.surname as staff_surname');
        $this->db->from('fingerprint_verification_log v');
        $this->db->join('staff s', 's.id = v.staff_id', 'left');
        $this->db->where('v.patient_id', $patientId);
        $this->db->order_by('v.verification_date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        $verifications = $query->result_array();
        
        // Format the data
        foreach ($verifications as &$verification) {
            $verification['staff_name_full'] = trim($verification['staff_name'] . ' ' . $verification['staff_surname']);
            $verification['verification_date_formatted'] = date('M j, Y g:i A', strtotime($verification['verification_date']));
            $verification['match_percentage'] = round(($verification['match_score'] / 199) * 100, 1);
        }
        
        echo json_encode([
            'status' => 'success',
            'verifications' => $verifications,
            'total_found' => count($verifications)
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error retrieving verification history: ' . $e->getMessage()
        ]);
    }
}

/**
 * Test fingerprint system connectivity
 */
public function testFingerprintSystem() {
    header('Content-Type: application/json');
    
    try {
        $results = [
            'timestamp' => date('Y-m-d H:i:s'),
            'database_checks' => [],
            'patient_fingerprint_stats' => [],
            'system_info' => []
        ];
        
        // Check database structure
        $results['database_checks']['patients_table_exists'] = $this->db->table_exists('patients');
        $results['database_checks']['fingerprint_template_column'] = $this->db->field_exists('fingerprint_template', 'patients');
        $results['database_checks']['fingerprint_image_column'] = $this->db->field_exists('fingerprint_image', 'patients');
        $results['database_checks']['verification_log_table'] = $this->db->table_exists('fingerprint_verification_log');
        
        // Get fingerprint statistics
        if ($results['database_checks']['patients_table_exists']) {
            $this->db->from('patients');
            $this->db->where('is_active', 'yes');
            $total_patients = $this->db->count_all_results();
            
            if ($results['database_checks']['fingerprint_template_column']) {
                $this->db->from('patients');
                $this->db->where('is_active', 'yes');
                $this->db->where('fingerprint_template IS NOT NULL');
                $this->db->where('fingerprint_template !=', '');
                $patients_with_fingerprints = $this->db->count_all_results();
            } else {
                $patients_with_fingerprints = 0;
            }
            
            $results['patient_fingerprint_stats'] = [
                'total_active_patients' => $total_patients,
                'patients_with_fingerprints' => $patients_with_fingerprints,
                'fingerprint_enrollment_rate' => $total_patients > 0 ? round(($patients_with_fingerprints / $total_patients) * 100, 1) : 0
            ];
        }
        
        // System information
        $results['system_info'] = [
            'php_version' => PHP_VERSION,
            'codeigniter_version' => CI_VERSION ?? 'Unknown',
            'server_time' => date('Y-m-d H:i:s'),
            'user_agent' => $this->input->user_agent(),
            'ip_address' => $this->input->ip_address()
        ];
        
        echo json_encode([
            'status' => 'success',
            'message' => 'System test completed',
            'results' => $results
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'System test failed: ' . $e->getMessage(),
            'debug' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}

    // =============================================
    // CONSULTATION MANAGEMENT METHODS  
    // =============================================

   /**
     * FIXED: Save consultation with proper field mapping
     */
public function save() {
    header('Content-Type: application/json');
    
    try {
        // Create consultations table if it doesn't exist
        $this->createConsultationTable();
        
        // Enhanced validation
        $patient_id = $this->input->post('patient_id');
        $symptoms = $this->input->post('symptoms');
        $doctor_id = $this->input->post('doctor'); // This maps to doctor field in database
        $consultation_date = $this->input->post('consultation_date');
        $consultation_id = $this->input->post('id');
        $verification_status = $this->input->post('verification_status');
        
        if (!$patient_id || !$symptoms || !$doctor_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Please fill all required fields',
                'required_fields' => [
                    'patient_id' => $patient_id ? 'present' : 'missing',
                    'symptoms' => $symptoms ? 'present' : 'missing',
                    'doctor' => $doctor_id ? 'present' : 'missing'
                ]
            ]);
            return;
        }
        
        // Handle diagnosis - ICD-10 or custom
        $diagnosisCodeId = $this->input->post('diagnosis_code_id');
        $customDiagnosis = $this->input->post('diagnosis_custom');
        $diagnosis = '';
        
        if ($diagnosisCodeId) {
            // ICD-10 diagnosis selected
            $diagnosisDetails = $this->getDiagnosisById($diagnosisCodeId);
            if ($diagnosisDetails) {
                $diagnosis = $diagnosisDetails['code'] . ' - ' . $diagnosisDetails['title'];
            }
        } elseif ($customDiagnosis) {
            // Custom diagnosis entered
            $diagnosis = $customDiagnosis;
        }
        
        // Prepare consultation data with CORRECT field names for your table structure
        $data = [
            'patient_id' => $patient_id,
            'doctor' => (int)$doctor_id,  // Using 'doctor' field name as per your table
            'date' => $consultation_date ?: date('Y-m-d'),  // Using 'date' field name as per your table
            'symptoms' => $symptoms,
            'diagnosis' => $diagnosis,
            'notes' => $this->input->post('notes') ?: '',
            'verification_status' => $verification_status == '1' ? 'verified' : 'unverified',
            'created_by' => $this->getStaffId()
        ];
        
        if ($consultation_id) {
            // Update existing consultation
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            $this->db->where('id', $consultation_id);
            $result = $this->db->update('consultations', $data);
            
            if ($result) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Consultation updated successfully',
                    'id' => $consultation_id
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to update consultation',
                    'db_error' => $this->db->error()
                ]);
            }
        } else {
            // Insert new consultation
            $data['created_at'] = date('Y-m-d H:i:s');
            
            $insertResult = $this->db->insert('consultations', $data);
            $consultationId = $this->db->insert_id();
            
            if ($insertResult && $consultationId) {
                echo json_encode([
                    'status' => 'success',
                    'message' => 'Consultation saved successfully',
                    'id' => $consultationId
                ]);
            } else {
                echo json_encode([
                    'status' => 'error',
                    'message' => 'Failed to save consultation',
                    'debug_info' => [
                        'insert_result' => $insertResult,
                        'consultation_id' => $consultationId,
                        'last_query' => $this->db->last_query(),
                        'db_error' => $this->db->error()
                    ]
                ]);
            }
        }
        
    } catch (Exception $e) {
        log_message('error', 'Consultation save error: ' . $e->getMessage());
        echo json_encode([
            'status' => 'error',
            'message' => 'Error: ' . $e->getMessage(),
            'debug_info' => [
                'exception' => $e->getMessage(),
                'line' => $e->getLine(),
                'file' => $e->getFile()
            ]
        ]);
    }
}


    /**
     * FIXED: Get patient consultations
     */
public function getPatientConsultations() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $patient_id = $this->input->post('patient_id');
        
        if (!$patient_id) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required'
            ]);
            exit;
        }
        
        // Check if consultations table exists
        if (!$this->db->table_exists('consultations')) {
            ob_clean();
            echo json_encode([
                'status' => 'success',
                'consultations' => [],
                'message' => 'No consultations table found'
            ]);
            exit;
        }
        
        // Query consultations
        $this->db->select('c.*, c.date as consultation_date');
        $this->db->from('consultations c');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit(5);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed'
            ]);
            exit;
        }
        
        $consultations = $query->result_array();
        
        // Get doctor names separately to avoid join issues
        foreach ($consultations as &$consultation) {
            if (!empty($consultation['doctor'])) {
                $this->db->select('name, surname');
                $this->db->from('staff');
                $this->db->where('id', $consultation['doctor']);
                $doctor_query = $this->db->get();
                
                if ($doctor_query && $doctor_query->num_rows() > 0) {
                    $doctor = $doctor_query->row();
                    $consultation['doctor_name'] = $doctor->name;
                    if (!empty($doctor->surname)) {
                        $consultation['doctor_name'] .= ' ' . $doctor->surname;
                    }
                } else {
                    $consultation['doctor_name'] = 'Unknown Doctor';
                }
            } else {
                $consultation['doctor_name'] = 'Unknown Doctor';
            }
        }
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'consultations' => $consultations
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

    /**
     * FIXED: Get consultation list for DataTable
     */
    public function getConsultationList() {
        header('Content-Type: application/json');
        
        try {
            // Check if consultations table exists
            if (!$this->db->table_exists('consultations')) {
                echo json_encode([
                    'draw' => 1,
                    'recordsTotal' => 0,
                    'recordsFiltered' => 0,
                    'data' => [],
                    'message' => 'Consultations table does not exist'
                ]);
                return;
            }
            
            // Query consultations with verification status if available
            if ($this->db->field_exists('verification_status', 'consultations')) {
                $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.verification_status, c.created_at');
            } else {
                $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.created_at');
            }
            
            $this->db->from('consultations c');
            $this->db->order_by('c.created_at', 'DESC');
            $this->db->limit(50);
            
            $query = $this->db->get();
            
            if (!$query) {
                echo json_encode([
                    'draw' => 1,
                    'recordsTotal' => 0,
                   'recordsFiltered' => 0,
                   'data' => [],
                   'error' => 'Query failed: ' . $this->db->error()['message']
               ]);
               return;
           }
           
           $consultations = $query->result_array();
           
           // Build data array for DataTable
           $data = [];
           foreach ($consultations as $consultation) {
               // Get patient name
               $patientName = 'Unknown Patient';
               if ($consultation['patient_id']) {
                   $this->db->select('patient_name');
                   $this->db->from('patients');
                   $this->db->where('id', $consultation['patient_id']);
                   $patientQuery = $this->db->get();
                   if ($patientQuery && $patientQuery->num_rows() > 0) {
                       $patient = $patientQuery->row();
                       $patientName = $patient->patient_name;
                   }
               }
               
               // Get doctor name
               $doctorName = 'Unknown Doctor';
               if ($consultation['doctor']) {
                   $this->db->select('name, surname');
                   $this->db->from('staff');
                   $this->db->where('id', $consultation['doctor']);
                   $doctorQuery = $this->db->get();
                   if ($doctorQuery && $doctorQuery->num_rows() > 0) {
                       $doctor = $doctorQuery->row();
                       $doctorName = 'Dr. ' . $doctor->name;
                       if (!empty($doctor->surname)) {
                           $doctorName .= ' ' . $doctor->surname;
                       }
                   }
               }
               
               // Format verification badge
               $verificationBadge = '';
               if (isset($consultation['verification_status'])) {
                   switch ($consultation['verification_status']) {
                       case 'verified':
                           $verificationBadge = '<br><span class="label label-success">Verified</span>';
                           break;
                       case 'skipped':
                           $verificationBadge = '<br><span class="label label-warning">Skipped</span>';
                           break;
                       default:
                           $verificationBadge = '<br><span class="label label-danger">Unverified</span>';
                   }
               }
               
               $data[] = [
                   $consultation['id'],
                   date('d M Y', strtotime($consultation['date'])),
                   $patientName . ' (' . $consultation['patient_id'] . ')' . $verificationBadge,
                   $doctorName,
                   substr($consultation['symptoms'] ?? 'No symptoms', 0, 50) . 
                   (strlen($consultation['symptoms'] ?? '') > 50 ? '...' : ''),
                   substr($consultation['diagnosis'] ?? 'No diagnosis', 0, 50),
                   '<button class="btn btn-xs btn-info" onclick="viewConsultation(' . $consultation['id'] . ')">
                       <i class="fa fa-eye"></i> View
                   </button>
                   <button class="btn btn-xs btn-primary" onclick="editConsultation(' . $consultation['id'] . ')">
                       <i class="fa fa-edit"></i> Edit
                   </button>'
               ];
           }
           
           echo json_encode([
               'draw' => 1,
               'recordsTotal' => count($data),
               'recordsFiltered' => count($data),
               'data' => $data,
               'success' => true
           ]);
           
       } catch (Exception $e) {
           echo json_encode([
               'draw' => 1,
               'recordsTotal' => 0,
               'recordsFiltered' => 0,
               'data' => [],
               'error' => 'Exception: ' . $e->getMessage() . ' on line ' . $e->getLine()
           ]);
       }
   }

   /**
    * FIXED: Get consultation details for viewing/editing
    */
public function getConsultationDetails() {
    header('Content-Type: application/json');
    
    try {
        $consultationId = $this->input->post('consultation_id');
        
        if (!$consultationId) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Consultation ID required'
            ]);
            return;
        }
        
        // Get consultation details - FIXED: Use correct column name 'date'
        if ($this->db->field_exists('verification_status', 'consultations')) {
            $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.verification_status, c.created_at, c.updated_at');
        } else {
            $this->db->select('c.id, c.patient_id, c.doctor, c.date, c.symptoms, c.diagnosis, c.notes, c.created_at');
        }
        
        $this->db->from('consultations c');
        $this->db->where('c.id', $consultationId);
        
        $query = $this->db->get();
        
        if ($query->num_rows() > 0) {
            $consultation = $query->row_array();
            
            // Get patient name separately
            $this->db->select('patient_name');
            $this->db->from('patients');
            $this->db->where('id', $consultation['patient_id']);
            $patientQuery = $this->db->get();
            
            if ($patientQuery && $patientQuery->num_rows() > 0) {
                $patient = $patientQuery->row();
                $consultation['patient_name'] = $patient->patient_name;
            } else {
                $consultation['patient_name'] = 'Unknown Patient';
            }
            
            // Get doctor name separately
            $this->db->select('name, surname');
            $this->db->from('staff');
            $this->db->where('id', $consultation['doctor']);
            $doctorQuery = $this->db->get();
            
            if ($doctorQuery && $doctorQuery->num_rows() > 0) {
                $doctor = $doctorQuery->row();
                $consultation['doctor_name'] = $doctor->name;
                if (!empty($doctor->surname)) {
                    $consultation['doctor_name'] .= ' ' . $doctor->surname;
                }
            } else {
                $consultation['doctor_name'] = 'Unknown Doctor';
            }
            
            // Format date for display - FIXED: Use correct column name 'date'
            $consultation['consultation_date'] = $consultation['date'];
            
            // Set default values for optional fields
            if (!isset($consultation['verification_status'])) {
                $consultation['verification_status'] = 'unverified';
            }
            
            if (!isset($consultation['updated_at'])) {
                $consultation['updated_at'] = null;
            }
            
            echo json_encode([
                'status' => 'success',
                'consultation' => $consultation
            ]);
            
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Consultation not found with ID: ' . $consultationId
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Database error: ' . $e->getMessage(),
            'line' => $e->getLine(),
            'consultation_id' => $consultationId ?? 'undefined'
        ]);
    }
}

   /**
    * FIXED: Add new patient
    */
   public function addPatient() {
       header('Content-Type: application/json');
       
       try {
           $patientName = $this->input->post('patientName');
           $patientAge = $this->input->post('patientAge');
           $patientGender = $this->input->post('patientGender');
           $patientPhone = $this->input->post('patientPhone');
           
           if (!$patientName || !$patientAge || !$patientGender || !$patientPhone) {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'Please fill all required fields'
               ]);
               return;
           }
           
           // Generate patient ID
           $patientId = $this->generatePatientId();
           
           $patientData = [
               'id' => $patientId,
               'patient_name' => $patientName,
               'age' => $patientAge,
               'gender' => strtolower($patientGender),
               'mobileno' => $patientPhone,
               'is_active' => 'yes',
               'created_at' => date('Y-m-d H:i:s')
           ];
           
           $this->db->insert('patients', $patientData);
           
           if ($this->db->affected_rows() > 0) {
               echo json_encode([
                   'status' => 'success',
                   'message' => 'Patient added successfully',
                   'patient_id' => $patientId
               ]);
           } else {
               echo json_encode([
                   'status' => 'error',
                   'message' => 'Failed to add patient'
               ]);
           }
           
       } catch (Exception $e) {
           echo json_encode([
               'status' => 'error',
               'message' => 'Error: ' . $e->getMessage()
           ]);
       }
   }

   /**
    * Generate unique patient ID
    */
   private function generatePatientId() {
       $currentYear = date('Y');
       
       // Get the last patient ID for current year
       $this->db->select('id');
       $this->db->from('patients');
       $this->db->like('id', $currentYear, 'after');
       $this->db->order_by('id', 'DESC');
       $this->db->limit(1);
       
       $query = $this->db->get();
       
       if ($query->num_rows() > 0) {
           $lastId = $query->row()->id;
           $lastNumber = (int)substr($lastId, 4);
           $newNumber = $lastNumber + 1;
       } else {
           $newNumber = 1;
       }
       
       return $currentYear . str_pad($newNumber, 4, '0', STR_PAD_LEFT);
   }

   /**
    * Get staff ID safely
    */
   private function getStaffId() {
       try {
           if (method_exists($this->customlib, 'getStaffID')) {
               return $this->customlib->getStaffID();
           } else {
               // Fallback - try to get from session
               $staffId = $this->session->userdata('admin_id');
               return $staffId ?: 1; // Default to 1 if no staff ID found
           }
       } catch (Exception $e) {
           log_message('error', 'Error getting staff ID: ' . $e->getMessage());
           return 1; // Default staff ID
       }
   }

   // =============================================
   // DEBUGGING AND TESTING METHODS
   // =============================================

   /**
    * Test controller functionality
    */
   public function test() {
       echo "<h1>Consultation Controller Test</h1>";
       echo "<p>Controller is working!</p>";
       echo "<p>Base URL: " . base_url() . "</p>";
       echo "<p>Current time: " . date('Y-m-d H:i:s') . "</p>";
       
       // Test database connection
       try {
           $this->db->select('COUNT(*) as patient_count');
           $this->db->from('patients');
           $query = $this->db->get();
           $result = $query->row();
           echo "<p>Patients in database: " . $result->patient_count . "</p>";
           
           // Test fingerprint data
           if ($this->db->field_exists('fingerprint_template', 'patients')) {
               $this->db->select('COUNT(*) as fingerprint_count');
               $this->db->from('patients');
               $this->db->where('fingerprint_template IS NOT NULL');
               $this->db->where('fingerprint_template !=', '');
               $query = $this->db->get();
               $result = $query->row();
               echo "<p>Patients with fingerprints: " . $result->fingerprint_count . "</p>";
           } else {
               echo "<p>Fingerprint columns not found</p>";
           }
           
       } catch (Exception $e) {
           echo "<p>Database error: " . $e->getMessage() . "</p>";
       }
   }

   /**
    * Debug patient details
    */
   public function debug_patient($patientId = null) {
       if (!$patientId) {
           $patientId = $this->input->get('id') ?: $this->input->post('id');
       }
       
       echo "<h2>Patient Details Debug</h2>";
       
       if (!$patientId) {
           echo "<p>No patient ID provided. Add ?id=PATIENT_ID to the URL</p>";
           return;
       }
       
       echo "<p><strong>Debugging Patient ID:</strong> " . $patientId . "</p>";
       
       try {
           // Check if patient exists
           $this->db->select('*');
           $this->db->from('patients');
           $this->db->where('id', $patientId);
           $query = $this->db->get();
           
           echo "<p><strong>SQL Query:</strong> " . $this->db->last_query() . "</p>";
           echo "<p><strong>Rows found:</strong> " . $query->num_rows() . "</p>";
           
           if ($query->num_rows() > 0) {
               $patient = $query->row_array();
               
               echo "<h3>Patient Data:</h3>";
               echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
               foreach ($patient as $key => $value) {
                   echo "<tr>";
                   echo "<td><strong>" . htmlspecialchars($key) . "</strong></td>";
                   echo "<td>" . htmlspecialchars($value ?? 'NULL') . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
               
               // Test the actual API call
               echo "<h3>API Response Test:</h3>";
               echo "<p><strong>Testing getPatientDetails API...</strong></p>";
               
               // Simulate the API call
               $_POST['id'] = $patientId;
               ob_start();
               $this->getPatientDetails();
               $apiResponse = ob_get_clean();
               
               echo "<p><strong>API Response:</strong></p>";
               echo "<pre>" . htmlspecialchars($apiResponse) . "</pre>";
               
           } else {
               echo "<p style='color: red;'>Patient not found!</p>";
               
               // Show some sample patients
               echo "<h3>Sample Patients:</h3>";
               $this->db->select('id, patient_name, mobileno');
               $this->db->from('patients');
               $this->db->limit(10);
               $query = $this->db->get();
               
               if ($query->num_rows() > 0) {
                  echo "<table border='1' style='border-collapse: collapse;'>";
                  echo "<tr><th>ID</th><th>Name</th><th>Phone</th></tr>";
                  foreach ($query->result_array() as $sample) {
                      echo "<tr>";
                      echo "<td><a href='?id=" . $sample['id'] . "'>" . $sample['id'] . "</a></td>";
                      echo "<td>" . $sample['patient_name'] . "</td>";
                      echo "<td>" . $sample['mobileno'] . "</td>";
                      echo "</tr>";
                  }
                  echo "</table>";
              } else {
                  echo "<p>No patients found in database!</p>";
              }
          }
          
      } catch (Exception $e) {
          echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
          echo "<p><strong>Last Query:</strong> " . $this->db->last_query() . "</p>";
      }
   }

   /**
    * Check current table structure
    */
   public function check_table_structure() {
       try {
           echo "<h2>Current Table Structure</h2>";
           
           // Check patients table
           if ($this->db->table_exists('patients')) {
               echo "<h3>Patients Table:</h3>";
               $fields = $this->db->field_data('patients');
               
               echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
               echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
               
               foreach ($fields as $field) {
                   echo "<tr>";
                   echo "<td>" . $field->name . "</td>";
                   echo "<td>" . $field->type . "</td>";
                   echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
                   echo "<td>" . ($field->primary_key ? 'PRI' : '') . "</td>";
                   echo "<td>" . ($field->default ?? 'NULL') . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
           } else {
               echo "<p style='color: red;'>Patients table does not exist!</p>";
           }
           
           // Check consultations table
           if ($this->db->table_exists('consultations')) {
               echo "<h3>Consultations Table:</h3>";
               $fields = $this->db->field_data('consultations');
               
               echo "<table border='1' style='border-collapse: collapse; width: 100%;'>";
               echo "<tr><th>Field</th><th>Type</th><th>Null</th><th>Key</th><th>Default</th></tr>";
               
               foreach ($fields as $field) {
                   echo "<tr>";
                   echo "<td>" . $field->name . "</td>";
                   echo "<td>" . $field->type . "</td>";
                   echo "<td>" . ($field->null ? 'YES' : 'NO') . "</td>";
                   echo "<td>" . ($field->primary_key ? 'PRI' : '') . "</td>";
                   echo "<td>" . ($field->default ?? 'NULL') . "</td>";
                   echo "</tr>";
               }
               echo "</table>";
           } else {
               echo "<p style='color: red;'>Consultations table does not exist!</p>";
           }
           
       } catch (Exception $e) {
           echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
       }
   }

   /**
    * Create test consultation for debugging
    */
   public function create_test_consultation() {
       header('Content-Type: application/json');
       
       try {
           // Create table first
           $this->createConsultationTable();
           
           // Get a sample patient
           $this->db->select('id');
           $this->db->from('patients');
           $this->db->limit(1);
           $patient = $this->db->get()->row();
           
           if (!$patient) {
               echo json_encode(['error' => 'No patients found in database']);
               return;
           }
           
           // Get a sample doctor
           $this->db->select('id');
           $this->db->from('staff');
           $this->db->limit(1);
           $doctor = $this->db->get()->row();
           
           if (!$doctor) {
               echo json_encode(['error' => 'No staff found in database']);
               return;
           }
           
           // Create test consultation
           $testData = [
               'patient_id' => $patient->id,
               'doctor' => $doctor->id,
               'date' => date('Y-m-d'),
               'symptoms' => 'Test symptoms for debugging',
               'diagnosis' => 'Test diagnosis',
               'notes' => 'Test consultation created for debugging',
               'verification_status' => 'unverified',
               'created_by' => 1,
               'created_at' => date('Y-m-d H:i:s')
           ];
           
           $result = $this->db->insert('consultations', $testData);
           $insertId = $this->db->insert_id();
           
           echo json_encode([
               'sample_patient_id' => $patient->id,
               'sample_doctor_id' => $doctor->id,
               'insert_result' => $result,
               'insert_id' => $insertId,
               'affected_rows' => $this->db->affected_rows(),
               'last_query' => $this->db->last_query(),
               'test_data' => $testData,
               'table_fields' => $this->db->list_fields('consultations')
           ], JSON_PRETTY_PRINT);
           
       } catch (Exception $e) {
           echo json_encode([
               'error' => $e->getMessage(),
               'line' => $e->getLine(),
               'file' => $e->getFile(),
               'last_query' => $this->db->last_query()
           ]);
       }
   }
   
       /**
     * Create consultations table if it doesn't exist
     */
    private function createConsultationTable() {
        if (!$this->db->table_exists('consultations')) {
            $sql = "CREATE TABLE IF NOT EXISTS `consultations` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` varchar(50) NOT NULL,
                `doctor` int(11) NOT NULL,
                `date` date NOT NULL,
                `symptoms` text NOT NULL,
                `diagnosis` text DEFAULT NULL,
                `notes` text DEFAULT NULL,
                `verification_status` enum('verified','unverified','skipped') DEFAULT 'unverified',
                `created_by` int(11) DEFAULT 1,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `doctor` (`doctor`),
                KEY `date` (`date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
        } else {
            // Add missing columns to existing table
            $this->addConsultationColumns();
        }
    }
   
   private function addConsultationColumns() {
        try {
            if (!$this->db->field_exists('verification_status', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `verification_status` ENUM('verified','unverified','skipped') DEFAULT 'unverified' AFTER `notes`";
                $this->db->query($sql);
            }
            
            if (!$this->db->field_exists('created_by', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `created_by` INT(11) DEFAULT 1 AFTER `verification_status`";
                $this->db->query($sql);
            }
            
            if (!$this->db->field_exists('created_at', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `created_at` TIMESTAMP NOT NULL DEFAULT CURRENT_TIMESTAMP AFTER `created_by`";
                $this->db->query($sql);
            }
            
            if (!$this->db->field_exists('updated_at', 'consultations')) {
                $sql = "ALTER TABLE `consultations` 
                        ADD COLUMN `updated_at` TIMESTAMP NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP AFTER `created_at`";
                $this->db->query($sql);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Error adding consultation columns: ' . $e->getMessage());
        }
    }
    
    /**
 * Debug method to check what's wrong
 */
public function debugSystem() {
    // Force JSON output and error reporting
    header('Content-Type: application/json');
    error_reporting(E_ALL);
    ini_set('display_errors', 0); // Don't display errors in HTML
    
    $debug_info = array();
    
    try {
        // Check if tables exist
        $debug_info['tables'] = array(
            'consultations' => $this->db->table_exists('consultations'),
            'consultation_financials' => $this->db->table_exists('consultation_financials'),
            'patients' => $this->db->table_exists('patients'),
            'staff' => $this->db->table_exists('staff'),
            'diagnosis_codes' => $this->db->table_exists('diagnosis_codes')
        );
        
        // Check database connection
        $debug_info['database'] = array(
            'connected' => $this->db->initialize(),
            'database' => $this->db->database,
            'hostname' => $this->db->hostname
        );
        
        // Check if methods exist
        $debug_info['methods'] = array(
            'saveWithFinancials' => method_exists($this, 'saveWithFinancials'),
            'getFinancialSummary' => method_exists($this, 'getFinancialSummary'),
            'generateFinancialReport' => method_exists($this, 'generateFinancialReport')
        );
        
        // Test simple query
        $debug_info['simple_query'] = $this->db->query("SELECT 1 as test")->row();
        
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Debug completed successfully',
            'debug_info' => $debug_info,
            'timestamp' => date('Y-m-d H:i:s')
        ));
        
    } catch (Exception $e) {
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Debug failed: ' . $e->getMessage(),
            'debug_info' => $debug_info,
            'error_details' => array(
                'file' => $e->getFile(),
                'line' => $e->getLine(),
                'trace' => $e->getTraceAsString()
            )
        ));
    }
}

/**
 * Simple financial summary
 */
public function getFinancialSummary() {
    header('Content-Type: application/json');
    ob_start();
    
    try {
        ob_clean();
        
        // Check if financial table exists
        if (!$this->db->table_exists('consultation_financials')) {
            echo json_encode(array(
                'status' => 'success',
                'summary' => array(
                    'total_revenue' => '0.00',
                    'paid_amount' => '0.00',
                    'pending_amount' => '0.00',
                    'outstanding_amount' => '0.00'
                ),
                'charts' => array(
                    'revenue_labels' => array(),
                    'revenue_data' => array(),
                    'payment_status_data' => array(0, 0, 0)
                ),
                'message' => 'Financial table not found'
            ));
            ob_end_flush();
            return;
        }
        
        // Get basic financial summary
        $this->db->select('
            COALESCE(SUM(total_amount), 0) as total_revenue,
            COALESCE(SUM(CASE WHEN payment_status = "paid" THEN total_amount ELSE 0 END), 0) as paid_amount,
            COALESCE(SUM(CASE WHEN payment_status = "pending" THEN total_amount ELSE 0 END), 0) as pending_amount,
            COALESCE(SUM(CASE WHEN payment_status = "partial" THEN total_amount ELSE 0 END), 0) as partial_amount
        ');
        $this->db->from('consultation_financials');
        $summary = $this->db->get()->row_array();
        
        $outstanding = floatval($summary['pending_amount']) + floatval($summary['partial_amount']);
        
        ob_clean();
        echo json_encode(array(
            'status' => 'success',
            'summary' => array(
                'total_revenue' => number_format($summary['total_revenue'], 2),
                'paid_amount' => number_format($summary['paid_amount'], 2),
                'pending_amount' => number_format($summary['pending_amount'], 2),
                'outstanding_amount' => number_format($outstanding, 2)
            ),
            'charts' => array(
                'revenue_labels' => array(),
                'revenue_data' => array(),
                'payment_status_data' => array(
                    floatval($summary['paid_amount']),
                    floatval($summary['pending_amount']),
                    floatval($summary['partial_amount'])
                )
            )
        ));
        ob_end_flush();
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Error loading financial summary: ' . $e->getMessage()
        ));
        ob_end_flush();
    }
}

/**
 * Ensure all required tables exist - ADD THIS METHOD TO YOUR CONTROLLER
 */
private function ensureTablesExist() {
    try {
        // Create consultation table
        $this->createConsultationTable();
        
        // Create financial table
        $this->createFinancialTable();
        
        // Create diagnosis tables
        $this->ensureDiagnosisCodesTables();
        
        log_message('debug', 'All required tables ensured');
        
    } catch (Exception $e) {
        log_message('error', 'Error ensuring tables exist: ' . $e->getMessage());
        throw new Exception('Failed to ensure required tables exist: ' . $e->getMessage());
    }
}
/**
 * Simple financial report
 */
public function generateFinancialReport() {
    header('Content-Type: application/json');
    ob_start();
    
    try {
        ob_clean();
        
        $dateFrom = $this->input->post('date_from');
        $dateTo = $this->input->post('date_to');
        
        if (!$dateFrom || !$dateTo) {
            echo json_encode(array(
                'status' => 'error',
                'message' => 'Date range is required'
            ));
            ob_end_flush();
            return;
        }
        
        // Simple report with basic data
        $report = array(
            'total_consultations' => 0,
            'total_revenue' => '0.00',
            'paid_amount' => '0.00',
            'outstanding_amount' => '0.00'
        );
        
        // Get consultations count
        $this->db->where('date >=', $dateFrom);
        $this->db->where('date <=', $dateTo);
        $this->db->from('consultations');
        $report['total_consultations'] = $this->db->count_all_results();
        
        // Get financial data if table exists
        if ($this->db->table_exists('consultation_financials')) {
            $this->db->select('
                COALESCE(SUM(total_amount), 0) as total_revenue,
                COALESCE(SUM(CASE WHEN payment_status = "paid" THEN total_amount ELSE 0 END), 0) as paid_amount
            ');
            $this->db->from('consultation_financials cf');
            $this->db->join('consultations c', 'c.id = cf.consultation_id');
            $this->db->where('c.date >=', $dateFrom);
            $this->db->where('c.date <=', $dateTo);
            $financial = $this->db->get()->row_array();
            
            if ($financial) {
                $report['total_revenue'] = number_format($financial['total_revenue'], 2);
                $report['paid_amount'] = number_format($financial['paid_amount'], 2);
                $outstanding = floatval($financial['total_revenue']) - floatval($financial['paid_amount']);
                $report['outstanding_amount'] = number_format($outstanding, 2);
            }
        }
        
        ob_clean();
        echo json_encode(array(
            'status' => 'success',
            'message' => 'Report generated successfully',
            'report' => $report
        ));
        ob_end_flush();
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode(array(
            'status' => 'error',
            'message' => 'Error generating report: ' . $e->getMessage()
        ));
        ob_end_flush();
    }
}

/*public function getLabTestsForConsultation() {
    header('Content-Type: application/json');
    
    try {
        // Check if lab_tests table exists, fallback to pathology
        if ($this->db->table_exists('lab_tests')) {
            $this->loadLabTestsFromLabTestsTable();
        } else {
            $this->loadLabTestsFromPathologyTable();
        }
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage()
        ]);
    }
}*/

public function getLabList() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        // Check if laboratory table exists
        if (!$this->db->table_exists('laboratory')) {
            ob_clean();
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'message' => 'Laboratory table does not exist'
            ]);
            exit;
        }
        
        // Query laboratory records
        $this->db->select('l.*, p.patient_name');
        $this->db->from('laboratory l');
        $this->db->join('patients p', 'p.id = l.patient_id', 'left');
        $this->db->order_by('l.created_at', 'DESC');
        $this->db->limit(50);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'draw' => 1,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => [],
                'error' => 'Query failed: ' . $this->db->error()['message']
            ]);
            exit;
        }
        
        $labs = $query->result_array();
        
        // Build data array for DataTable
        $data = [];
        foreach ($labs as $lab) {
            // Format verification badge
            $verificationBadge = $this->getVerificationBadge($lab['verification_status'] ?? 'unverified');
            
            // Get patient name
            $patientName = $lab['patient_name'] ?: 'Unknown Patient';
            
            // Get staff name if tested_by exists
            $staffName = 'Unknown Staff';
            if (!empty($lab['tested_by'])) {
                $this->db->select('name, surname');
                $this->db->from('staff');
                $this->db->where('id', $lab['tested_by']);
                $staffQuery = $this->db->get();
                if ($staffQuery && $staffQuery->num_rows() > 0) {
                    $staff = $staffQuery->row();
                    $staffName = $staff->name;
                    if (!empty($staff->surname)) {
                        $staffName .= ' ' . $staff->surname;
                    }
                }
            }
            
            $testDate = !empty($lab['test_date']) ? 
                       date('d M Y', strtotime($lab['test_date'])) : 
                       date('d M Y H:i', strtotime($lab['created_at']));
            
            $data[] = [
                $lab['id'],
                $testDate,
                $patientName . ' (' . $lab['patient_id'] . ')' . $verificationBadge,
                $lab['test_name'],
                substr($lab['test_result'] ?? 'Pending', 0, 50) . 
                (strlen($lab['test_result'] ?? '') > 50 ? '...' : ''),
                $staffName,
                '<button class="btn btn-xs btn-info" onclick="viewLab(' . $lab['id'] . ')">
                    <i class="fa fa-eye"></i> View
                </button>
                <button class="btn btn-xs btn-primary" onclick="editLab(' . $lab['id'] . ')">
                    <i class="fa fa-edit"></i> Edit
                </button>'
            ];
        }
        
        ob_clean();
        echo json_encode([
            'draw' => 1,
            'recordsTotal' => count($data),
            'recordsFiltered' => count($data),
            'data' => $data,
            'success' => true
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'draw' => 1,
            'recordsTotal' => 0,
            'recordsFiltered' => 0,
            'data' => [],
            'error' => 'Exception: ' . $e->getMessage() . ' on line ' . $e->getLine()
        ]);
        exit;
    }
}

public function getPatientLabTests() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        $patient_id = $this->input->post('patient_id') ?: $this->input->get('patient_id');
        
        if (!$patient_id) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Patient ID required',
                'lab_tests' => []
            ]);
            exit;
        }
        
        // Check if laboratory table exists
        if (!$this->db->table_exists('laboratory')) {
            ob_clean();
            echo json_encode([
                'status' => 'success',
                'lab_tests' => [],
                'message' => 'Laboratory table not found'
            ]);
            exit;
        }
        
        // Query lab tests for this patient
        $this->db->select('*');
        $this->db->from('laboratory');
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit(10);
        
        $query = $this->db->get();
        
        if (!$query) {
            ob_clean();
            echo json_encode([
                'status' => 'error',
                'message' => 'Database query failed',
                'lab_tests' => []
            ]);
            exit;
        }
        
        $lab_tests = $query->result_array();
        
        // Format the lab tests data
        foreach ($lab_tests as &$test) {
            $test['test_date_formatted'] = !empty($test['test_date']) ? 
                date('M j, Y', strtotime($test['test_date'])) : 
                date('M j, Y', strtotime($test['created_at']));
                
            $test['status_label'] = ucfirst($test['result_status'] ?? 'pending');
            $test['verification_badge'] = $this->getVerificationBadge($test['verification_status'] ?? 'unverified');
        }
        
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'lab_tests' => $lab_tests,
            'total_found' => count($lab_tests)
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage(),
            'lab_tests' => []
        ]);
        exit;
    }
}

private function loadLabTestsFromLabTestsTable() {
    // Get tests grouped by category
    $this->db->select('lt.*, ltc.category_name');
    $this->db->from('lab_tests lt');
    $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
    $this->db->where('lt.is_active', 1);
    $this->db->order_by('ltc.category_name, lt.test_name');
    
    $query = $this->db->get();
    $all_tests = $query->result_array();
    
    // Group by category
    $tests_by_category = [];
    $categories = [];
    
    foreach ($all_tests as $test) {
        $category_id = $test['category_id'] ?: 'other';
        $category_name = $test['category_name'] ?: 'Other Tests';
        
        if (!isset($tests_by_category[$category_id])) {
            $tests_by_category[$category_id] = [];
            $categories[$category_id] = $category_name;
        }
        
        $tests_by_category[$category_id][] = $test;
    }
    
    echo json_encode([
        'status' => 'success',
        'tests_by_category' => $tests_by_category,
        'categories' => $categories
    ]);
}

private function loadLabTestsFromPathologyTable() {
    // Fallback to pathology table
    $this->db->select('p.*, pc.category_name');
    $this->db->from('pathology p');
    $this->db->join('pathology_category pc', 'pc.id = p.pathology_category_id', 'left');
    $this->db->where('p.is_active', 1);
    $this->db->order_by('pc.category_name, p.test_name');
    
    $query = $this->db->get();
    $all_tests = $query->result_array();
    
    // Group by category and map to lab_tests format
    $tests_by_category = [];
    $categories = [];
    
    foreach ($all_tests as $test) {
        $category_id = $test['pathology_category_id'] ?: 'other';
        $category_name = $test['category_name'] ?: 'Other Tests';
        
        if (!isset($tests_by_category[$category_id])) {
            $tests_by_category[$category_id] = [];
            $categories[$category_id] = $category_name;
        }
        
        // Map pathology fields to lab_tests format
        $mapped_test = [
            'id' => $test['id'],
            'test_name' => $test['test_name'],
            'short_name' => $test['short_name'],
            'cost' => $test['cost'],
            'sample_type' => $test['sample_type'],
            'method' => $test['method'],
            'category_id' => $category_id
        ];
        
        $tests_by_category[$category_id][] = $mapped_test;
    }
    
    echo json_encode([
        'status' => 'success',
        'tests_by_category' => $tests_by_category,
        'categories' => $categories
    ]);
}

public function testJsonResponse() {
    ob_clean();
    header('Content-Type: application/json');
    header('Cache-Control: no-cache, must-revalidate');
    error_reporting(0);
    ini_set('display_errors', 0);
    
    try {
        ob_clean();
        echo json_encode([
            'status' => 'success',
            'message' => 'JSON response test successful',
            'timestamp' => date('Y-m-d H:i:s'),
            'method' => $_SERVER['REQUEST_METHOD'],
            'data' => [
                'test' => true,
                'number' => 123,
                'array' => [1, 2, 3]
            ]
        ]);
        exit;
        
    } catch (Exception $e) {
        ob_clean();
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
        exit;
    }
}

public function search() {
    try {
        $search = $this->input->get('search');
        $limit = $this->input->get('limit') ?: 20;
        
        if (strlen($search) < 2) {
            echo json_encode([]);
            return;
        }
        
        $patients = $this->db->select('id, patient_name as text, patient_name as name, mobileno as phone, age, gender')
                            ->like('patient_name', $search)
                            ->or_like('mobileno', $search)
                            ->or_like('id', $search)
                            ->order_by('patient_name')
                            ->limit($limit)
                            ->get('patients')
                            ->result_array();
        
        // Format for select2
        foreach ($patients as &$patient) {
            $patient['text'] = $patient['name'] . ' (ID: ' . $patient['id'] . ')';
        }
        
        echo json_encode($patients);
        
    } catch (Exception $e) {
        echo json_encode([]);
    }
}

/**
 * Save patient notes
 */
public function saveNotes() {
    try {
        $patient_id = $this->input->post('patient_id');
        $notes = $this->input->post('notes');
        
        if (!$patient_id) {
            echo json_encode([
                'status' => false,
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        // Check if notes exist
        $existing = $this->db->where('patient_id', $patient_id)->get('patient_notes')->row();
        
        if ($existing) {
            // Update existing notes
            $this->db->where('patient_id', $patient_id)
                    ->update('patient_notes', [
                        'notes' => $notes,
                        'updated_at' => date('Y-m-d H:i:s'),
                        'updated_by' => $this->session->userdata('user_id')
                    ]);
        } else {
            // Insert new notes
            $this->db->insert('patient_notes', [
                'patient_id' => $patient_id,
                'notes' => $notes,
                'created_at' => date('Y-m-d H:i:s'),
                'created_by' => $this->session->userdata('user_id')
            ]);
        }
        
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'status' => true,
                'message' => 'Notes saved successfully'
            ]);
        } else {
            echo json_encode([
                'status' => false,
                'message' => 'No changes made to notes'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => false,
            'message' => 'Error saving notes: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get consultations with lab tests
 */
public function getConsultationsWithLabTests() {
    try {
        $consultations = $this->db->select('c.*, p.patient_name, s.name as doctor_name, s.surname as doctor_surname,
                                           COUNT(ptr.id) as tests_count')
                                 ->from('consultations c')
                                 ->join('patients p', 'c.patient_id = p.id', 'left')
                                 ->join('staff s', 'c.doctor = s.id', 'left')
                                 ->join('pathology_test_reports ptr', 'c.id = ptr.consultation_id', 'left')
                                 ->group_by('c.id')
                                 ->order_by('c.consultation_date', 'DESC')
                                 ->limit(100)
                                 ->get()
                                 ->result_array();
        
        // Format data
        foreach ($consultations as &$consultation) {
            $consultation['doctor_name'] = trim(($consultation['doctor_name'] ?? '') . ' ' . ($consultation['doctor_surname'] ?? ''));
            $consultation['verification_status'] = $consultation['verification_status'] ? 'verified' : 'unverified';
        }
        
        echo json_encode([
            'status' => 'success',
            'consultations' => $consultations
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading consultations: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save consultation with lab tests
 */
/*public function saveWithLabTests() {
    try {
        $this->db->trans_start();
        
        // Get form data
        $patient_id = $this->input->post('patient_id');
        $consultation_date = $this->input->post('consultation_date');
        $doctor = $this->input->post('doctor');
        $symptoms = $this->input->post('symptoms');
        $notes = $this->input->post('notes');
        $selected_tests = $this->input->post('selected_tests');
        
        if (!$patient_id || !$consultation_date || !$doctor || !$symptoms) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Required fields are missing'
            ]);
            return;
        }
        
        // Save consultation
        $consultation_data = [
            'patient_id' => $patient_id,
            'consultation_date' => $consultation_date,
            'doctor' => $doctor,
            'symptoms' => $symptoms,
            'notes' => $notes,
            'verification_status' => $this->input->post('verification_status') ?: 0,
            'verification_score' => $this->input->post('verification_score') ?: 0,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Add diagnosis data if available
        if ($this->input->post('diagnosis_code')) {
            $consultation_data['diagnosis_code'] = $this->input->post('diagnosis_code');
            $consultation_data['diagnosis_title'] = $this->input->post('diagnosis_title');
        } elseif ($this->input->post('diagnosis_custom')) {
            $consultation_data['diagnosis_custom'] = $this->input->post('diagnosis_custom');
        }
        
        $this->db->insert('consultations', $consultation_data);
        $consultation_id = $this->db->insert_id();
        
        // Save lab tests if selected
        if ($selected_tests) {
            $tests = json_decode($selected_tests, true);
            if (is_array($tests) && !empty($tests)) {
                foreach ($tests as $test) {
                    $test_data = [
                        'patient_id' => $patient_id,
                        'consultation_id' => $consultation_id,
                        'test_id' => $test['id'],
                        'test_name' => $test['test_name'],
                        'test_category_id' => $test['category'],
                        'test_date' => $consultation_date,
                        'cost' => $test['cost'] ?: 0,
                        'priority' => $this->input->post('test_priority') ?: 'normal',
                        'instructions' => $this->input->post('test_instructions'),
                        'result_status' => 'pending',
                        'ordered_by' => $doctor,
                        'created_at' => date('Y-m-d H:i:s')
                    ];
                    
                    $this->db->insert('pathology_test_reports', $test_data);
                }
            }
        }
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save consultation and tests'
            ]);
        } else {
            $test_count = is_array(json_decode($selected_tests, true)) ? count(json_decode($selected_tests, true)) : 0;
            echo json_encode([
                'status' => 'success',
                'message' => "Consultation saved successfully with {$test_count} lab tests ordered",
                'consultation_id' => $consultation_id
            ]);
        }
        
    } catch (Exception $e) {
        $this->db->trans_rollback();
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving consultation: ' . $e->getMessage()
        ]);
    }
}*/

/**
 * Get patient review data for the review tab
 */
public function getPatientReview() {
    try {
        $patient_id = $this->input->post('patient_id');
        $time_period = $this->input->post('time_period', true) ?: 3; // Default 3 months
        
        if (!$patient_id) {
            echo json_encode([
                'status' => false,
                'message' => 'Patient ID is required'
            ]);
            return;
        }
        
        // Calculate date range
        $end_date = date('Y-m-d');
        if ($time_period == 0) {
            $start_date = '1900-01-01'; // All time
        } else {
            $start_date = date('Y-m-d', strtotime("-{$time_period} months"));
        }
        
        // Get patient basic information
        $patient = $this->db->select('id, patient_name as name, age, gender, mobileno as phone, blood_group_name, known_allergies')
                           ->where('id', $patient_id)
                           ->get('patients')
                           ->row_array();
        
        if (!$patient) {
            echo json_encode([
                'status' => false,
                'message' => 'Patient not found'
            ]);
            return;
        }
        
        // Get consultations count and data
        $consultations = $this->db->select('c.*, s.name as doctor_name, s.surname as doctor_surname')
                                 ->from('consultations c')
                                 ->join('staff s', 'c.doctor = s.id', 'left')
                                 ->where('c.patient_id', $patient_id)
                                 ->where('c.consultation_date >=', $start_date)
                                 ->where('c.consultation_date <=', $end_date)
                                 ->order_by('c.consultation_date', 'DESC')
                                 ->limit(10)
                                 ->get()
                                 ->result_array();
        
        // Format consultations
        foreach ($consultations as &$consultation) {
            $consultation['doctor'] = trim(($consultation['doctor_name'] ?? '') . ' ' . ($consultation['doctor_surname'] ?? ''));
            $consultation['date'] = date('M d, Y', strtotime($consultation['consultation_date']));
        }
        
        // Get triage records
        $triage_records = $this->db->select('tr.*, s.name as nurse_name, s.surname as nurse_surname')
                                  ->from('triage_records tr')
                                  ->join('staff s', 'tr.staff_id = s.id', 'left')
                                  ->where('tr.patient_id', $patient_id)
                                  ->where('tr.date >=', $start_date)
                                  ->where('tr.date <=', $end_date)
                                  ->order_by('tr.date', 'DESC')
                                  ->limit(10)
                                  ->get()
                                  ->result_array();
        
        // Format triage records
        foreach ($triage_records as &$record) {
            $record['nurse_name'] = trim(($record['nurse_name'] ?? '') . ' ' . ($record['nurse_surname'] ?? ''));
            $record['date'] = date('M d, Y', strtotime($record['date']));
            
            // Calculate BMI if weight and height available
            if ($record['weight'] && $record['height']) {
                $height_m = $record['height'] / 100;
                $record['bmi'] = round($record['weight'] / ($height_m * $height_m), 1);
            }
        }
        
        // Get medical history
        $medical_history = [];
        
        // Chronic conditions
        $medical_history['chronic_conditions'] = $this->db->select('*')
                                                         ->where('patient_id', $patient_id)
                                                         ->get('patient_chronic_conditions')
                                                         ->result_array();
        
        // Current medications
        $medical_history['medications'] = $this->db->select('pm.*, m.medicine_name as name')
                                                  ->from('patient_medications pm')
                                                  ->join('medicine_list m', 'pm.medicine_id = m.id', 'left')
                                                  ->where('pm.patient_id', $patient_id)
                                                  ->where('pm.status', 'active')
                                                  ->get()
                                                  ->result_array();
        
        // Allergies
        $medical_history['allergies'] = $this->db->select('*')
                                                ->where('patient_id', $patient_id)
                                                ->get('patient_allergies')
                                                ->result_array();
        
        // Past procedures
        $medical_history['procedures'] = $this->db->select('pp.*, s.name as doctor_name, s.surname as doctor_surname')
                                                 ->from('patient_procedures pp')
                                                 ->join('staff s', 'pp.doctor_id = s.id', 'left')
                                                 ->where('pp.patient_id', $patient_id)
                                                 ->order_by('pp.procedure_date', 'DESC')
                                                 ->get()
                                                 ->result_array();
        
        // Format procedures
        foreach ($medical_history['procedures'] as &$procedure) {
            $procedure['doctor'] = trim(($procedure['doctor_name'] ?? '') . ' ' . ($procedure['doctor_surname'] ?? ''));
            $procedure['date'] = date('M d, Y', strtotime($procedure['procedure_date']));
        }
        
        // Get patient notes
        $notes_result = $this->db->select('notes')
                               ->where('patient_id', $patient_id)
                               ->get('patient_notes')
                               ->row();
        $notes = $notes_result ? $notes_result->notes : '';
        
        // Calculate summary
        $summary = [
            'consultations' => count($consultations),
            'prescriptions' => $this->db->where('patient_id', $patient_id)
                                       ->where('date >=', $start_date)
                                       ->where('date <=', $end_date)
                                       ->count_all_results('prescriptions'),
            'lab_tests' => $this->db->where('patient_id', $patient_id)
                                   ->where('test_date >=', $start_date)
                                   ->where('test_date <=', $end_date)
                                   ->count_all_results('pathology_test_reports')
        ];
        
        echo json_encode([
            'status' => 'success',
            'data' => [
                'patient' => $patient,
                'summary' => $summary,
                'consultations' => $consultations,
                'triage_records' => $triage_records,
                'medical_history' => $medical_history,
                'notes' => $notes
            ]
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => false,
            'message' => 'Error loading patient review: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get lab summary for dashboard
 */
public function getLabSummary() {
    try {
        $today = date('Y-m-d');
        $this_month_start = date('Y-m-01');
        
        // Get summary counts
        $summary = [
            'total_tests' => $this->db->count_all_results('pathology_test_reports'),
            'pending_tests' => $this->db->where('result_status', 'pending')
                                       ->count_all_results('pathology_test_reports'),
            'completed_tests' => $this->db->where('result_status', 'completed')
                                         ->count_all_results('pathology_test_reports'),
            'critical_results' => $this->db->where('result_status', 'critical')
                                          ->count_all_results('pathology_test_reports')
        ];
        
        // Get chart data - tests by category
        $category_data = $this->db->select('pc.category_name, COUNT(ptr.id) as test_count')
                                 ->from('pathology_test_reports ptr')
                                 ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                                 ->group_by('ptr.test_category_id')
                                 ->get()
                                 ->result_array();
        
        $category_labels = [];
        $category_counts = [];
        foreach ($category_data as $data) {
            $category_labels[] = $data['category_name'] ?: 'Other';
            $category_counts[] = (int)$data['test_count'];
        }
        
        // Get status data
        $status_data = [
            $summary['pending_tests'],
            $summary['completed_tests'], 
            $summary['critical_results']
        ];
        
        $charts = [
            'category_labels' => $category_labels,
            'category_data' => $category_counts,
            'status_data' => $status_data
        ];
        
        echo json_encode([
            'status' => 'success',
            'summary' => $summary,
            'charts' => $charts
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab summary: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get recent lab tests for table
 */
public function getRecentLabTests() {
    try {
        $tests = $this->db->select('ptr.*, p.patient_name, pc.category_name, lt.test_name')
                         ->from('pathology_test_reports ptr')
                         ->join('patients p', 'ptr.patient_id = p.id', 'left')
                         ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                         ->join('lab_tests lt', 'ptr.test_id = lt.id', 'left')
                         ->order_by('ptr.test_date', 'DESC')
                         ->limit(50)
                         ->get()
                         ->result_array();
        
        // Format test dates
        foreach ($tests as &$test) {
            $test['test_date'] = date('M d, Y', strtotime($test['test_date']));
        }
        
        echo json_encode([
            'status' => 'success',
            'tests' => $tests
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get test details for results entry
 */
public function getTestDetails() {
    try {
        $test_id = $this->input->post('test_id');
        
        if (!$test_id) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test ID is required'
            ]);
            return;
        }
        
        $test = $this->db->select('ptr.*, p.patient_name, lt.test_name, pc.category_name')
                        ->from('pathology_test_reports ptr')
                        ->join('patients p', 'ptr.patient_id = p.id', 'left')
                        ->join('lab_tests lt', 'ptr.test_id = lt.id', 'left')
                        ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                        ->where('ptr.id', $test_id)
                        ->get()
                        ->row_array();
        
        if (!$test) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test not found'
            ]);
            return;
        }
        
        echo json_encode([
            'status' => 'success',
            'test' => $test
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading test details: ' . $e->getMessage()
        ]);
    }
}

/**
 * Save test results
 */
public function saveTestResults() {
    try {
        $test_id = $this->input->post('test_id');
        $test_result = $this->input->post('test_result');
        $result_status = $this->input->post('result_status');
        $reference_range = $this->input->post('reference_range');
        $lab_notes = $this->input->post('lab_notes');
        
        if (!$test_id || !$test_result) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Test ID and result are required'
            ]);
            return;
        }
        
        $update_data = [
            'test_result' => $test_result,
            'result_status' => $result_status ?: 'completed',
            'reference_range' => $reference_range,
            'notes' => $lab_notes,
            'result_date' => date('Y-m-d H:i:s'),
            'technician_id' => $this->session->userdata('user_id'),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $test_id)->update('pathology_test_reports', $update_data);
        
        if ($this->db->affected_rows() > 0) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Test results saved successfully'
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Failed to save test results'
            ]);
        }
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error saving test results: ' . $e->getMessage()
        ]);
    }
}

/**
 * Generate lab report
 */
public function generateLabReport() {
    try {
        $date_from = $this->input->post('date_from');
        $date_to = $this->input->post('date_to');
        $category_id = $this->input->post('category_id');
        $test_status = $this->input->post('test_status');
        
        if (!$date_from || !$date_to) {
            echo json_encode([
                'status' => 'error',
                'message' => 'Date range is required'
            ]);
            return;
        }
        
        // Build query
        $this->db->select('ptr.*, p.patient_name, pc.category_name, lt.test_name')
                ->from('pathology_test_reports ptr')
                ->join('patients p', 'ptr.patient_id = p.id', 'left')
                ->join('pathology_categories pc', 'ptr.test_category_id = pc.id', 'left')
                ->join('lab_tests lt', 'ptr.test_id = lt.id', 'left')
                ->where('ptr.test_date >=', $date_from)
                ->where('ptr.test_date <=', $date_to);
        
        if ($category_id) {
            $this->db->where('ptr.test_category_id', $category_id);
        }
        
        if ($test_status) {
            $this->db->where('ptr.result_status', $test_status);
        }
        
        $tests = $this->db->order_by('ptr.test_date', 'DESC')->get()->result_array();
        
        // Calculate report summary
        $total_tests = count($tests);
        $completed_tests = count(array_filter($tests, function($test) {
            return $test['result_status'] === 'completed';
        }));
        $pending_tests = count(array_filter($tests, function($test) {
            return $test['result_status'] === 'pending';
        }));
        $critical_tests = count(array_filter($tests, function($test) {
            return $test['result_status'] === 'critical';
        }));
        
        $report = [
            'total_tests' => $total_tests,
            'completed_tests' => $completed_tests,
            'pending_tests' => $pending_tests,
            'critical_tests' => $critical_tests,
            'tests' => $tests,
            'date_range' => [
                'from' => $date_from,
                'to' => $date_to
            ]
        ];
        
        echo json_encode([
            'status' => 'success',
            'report' => $report
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error generating report: ' . $e->getMessage()
        ]);
    }
}

/**
 * Get lab tests for consultation (for the modal)
 */
public function getLabTestsForConsultation() {
    try {
        // Get test categories
        $categories = $this->db->select('id, category_name')
                              ->where('is_active', 1)
                              ->get('pathology_categories')
                              ->result_array();
        
        $categories_list = [];
        foreach ($categories as $category) {
            $categories_list[$category['id']] = $category['category_name'];
        }
        
        // Get tests by category
        $tests_by_category = [];
        foreach ($categories as $category) {
            $tests = $this->db->select('id, test_name, short_name, cost, sample_type, method')
                             ->where('category_id', $category['id'])
                             ->where('is_active', 1)
                             ->order_by('test_name')
                             ->get('lab_tests')
                             ->result_array();
            
            if (!empty($tests)) {
                $tests_by_category[$category['id']] = $tests;
            }
        }
        
        echo json_encode([
            'status' => 'success',
            'categories' => $categories_list,
            'tests_by_category' => $tests_by_category
        ]);
        
    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => 'Error loading lab tests: ' . $e->getMessage()
        ]);
    }
}

}