<?php
/**
 * Queue Management System Setup Script
 * Run this file ONCE to initialize the queue management system
 * 
 * Usage: 
 * - Place this file in your CodeIgniter root directory
 * - Run via web browser: http://yourdomain.com/setup_queue_system.php
 * - Or run via CLI: php setup_queue_system.php
 */

// Set CodeIgniter environment
define('ENVIRONMENT', isset($_SERVER['CI_ENV']) ? $_SERVER['CI_ENV'] : 'production');

// Path to CodeIgniter's index.php
$system_path = 'system';
$application_folder = 'application';

define('BASEPATH', $system_path.DIRECTORY_SEPARATOR);
define('APPPATH', $application_folder.DIRECTORY_SEPARATOR);
define('VIEWPATH', $application_folder.DIRECTORY_SEPARATOR.'views'.DIRECTORY_SEPARATOR);

require_once BASEPATH.'core/Common.php';

// Initialize CodeIgniter
$CI = &get_instance();

// Load database
$CI->load->database();

echo "<h2>🏥 Queue Management System Setup</h2>";
echo "<p>Initializing queue management system for City Health HMS...</p>";

try {
    // =============================================
    // 1. CREATE QUEUE TABLES
    // =============================================
    echo "<h3>📊 Creating Database Tables</h3>";
    
    // Patient Queue Table
    if (!$CI->db->table_exists('patient_queue')) {
        echo "<p>Creating patient_queue table...</p>";
        
        $sql = "
        CREATE TABLE `patient_queue` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `queue_number` varchar(20) NOT NULL,
            `department` varchar(50) NOT NULL,
            `status` enum('waiting','in_progress','completed','skipped','transferred') DEFAULT 'waiting',
            `priority` enum('low','normal','high','urgent') DEFAULT 'normal',
            `queue_type` varchar(50) DEFAULT 'walk_in',
            `workflow_type` varchar(50) DEFAULT 'general',
            `estimated_wait_time` int(11) DEFAULT NULL COMMENT 'Estimated wait time in minutes',
            `actual_wait_time` int(11) DEFAULT NULL COMMENT 'Actual wait time in minutes',
            `called_at` datetime DEFAULT NULL,
            `completed_at` datetime DEFAULT NULL,
            `served_by` int(11) DEFAULT NULL COMMENT 'Staff ID who served the patient',
            `notes` text DEFAULT NULL,
            `next_queue` varchar(50) DEFAULT NULL COMMENT 'Next department if transferred',
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` datetime DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_patient_department` (`patient_id`,`department`),
            KEY `idx_department_status` (`department`,`status`),
            KEY `idx_queue_created` (`queue_number`,`created_at`),
            CONSTRAINT `fk_patient_queue_patient` FOREIGN KEY (`patient_id`) REFERENCES `patients` (`id`) ON DELETE CASCADE,
            CONSTRAINT `fk_patient_queue_staff` FOREIGN KEY (`served_by`) REFERENCES `staff` (`id`) ON DELETE SET NULL
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        $CI->db->query($sql);
        echo "<span style='color: green;'>✅ patient_queue table created</span><br>";
    } else {
        echo "<span style='color: orange;'>⚠️ patient_queue table already exists</span><br>";
    }
    
    // Queue Config Table
    if (!$CI->db->table_exists('queue_config')) {
        echo "<p>Creating queue_config table...</p>";
        
        $sql = "
        CREATE TABLE `queue_config` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department` varchar(50) NOT NULL,
            `queue_prefix` varchar(10) NOT NULL DEFAULT 'Q',
            `counter_current` int(11) DEFAULT 1,
            `counter_reset` enum('daily','weekly','monthly','never') DEFAULT 'daily',
            `average_service_time` int(11) DEFAULT 15 COMMENT 'Average service time in minutes',
            `max_queue_size` int(11) DEFAULT 100,
            `auto_call_enabled` tinyint(1) DEFAULT 0,
            `sound_enabled` tinyint(1) DEFAULT 1,
            `display_enabled` tinyint(1) DEFAULT 1,
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            `updated_at` datetime DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            UNIQUE KEY `uk_department` (`department`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        $CI->db->query($sql);
        echo "<span style='color: green;'>✅ queue_config table created</span><br>";
    } else {
        echo "<span style='color: orange;'>⚠️ queue_config table already exists</span><br>";
    }
    
    // Queue Display Settings Table
    if (!$CI->db->table_exists('queue_display_settings')) {
        echo "<p>Creating queue_display_settings table...</p>";
        
        $sql = "
        CREATE TABLE `queue_display_settings` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department` varchar(50) NOT NULL,
            `display_name` varchar(100) NOT NULL,
            `theme_color` varchar(20) DEFAULT '#007bff',
            `icon_class` varchar(50) DEFAULT 'fa-users',
            `display_order` int(11) DEFAULT 0,
            `is_visible` tinyint(1) DEFAULT 1,
            PRIMARY KEY (`id`),
            KEY `idx_display_order` (`display_order`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        $CI->db->query($sql);
        echo "<span style='color: green;'>✅ queue_display_settings table created</span><br>";
    } else {
        echo "<span style='color: orange;'>⚠️ queue_display_settings table already exists</span><br>";
    }
    
    // Queue Flow Rules Table
    if (!$CI->db->table_exists('queue_flow_rules')) {
        echo "<p>Creating queue_flow_rules table...</p>";
        
        $sql = "
        CREATE TABLE `queue_flow_rules` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `rule_name` varchar(100) NOT NULL,
            `from_department` varchar(50) NOT NULL,
            `to_department` varchar(50) NOT NULL,
            `condition_type` enum('automatic','manual','conditional') DEFAULT 'manual',
            `condition_data` longtext CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL CHECK (json_valid(`condition_data`)),
            `workflow_type` varchar(50) DEFAULT 'general',
            `priority_mapping` varchar(20) DEFAULT 'maintain' COMMENT 'maintain, increase, decrease, or specific priority',
            `is_active` tinyint(1) DEFAULT 1,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_from_condition` (`from_department`,`condition_type`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        $CI->db->query($sql);
        echo "<span style='color: green;'>✅ queue_flow_rules table created</span><br>";
    } else {
        echo "<span style='color: orange;'>⚠️ queue_flow_rules table already exists</span><br>";
    }
    
    // Queue Notifications Table
    if (!$CI->db->table_exists('queue_notifications')) {
        echo "<p>Creating queue_notifications table...</p>";
        
        $sql = "
        CREATE TABLE `queue_notifications` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `department` varchar(50) NOT NULL,
            `notification_type` varchar(50) NOT NULL,
            `patient_id` int(11) DEFAULT NULL,
            `queue_id` int(11) DEFAULT NULL,
            `message` text NOT NULL,
            `metadata` longtext CHARACTER SET utf8mb4 COLLATE utf8mb4_bin DEFAULT NULL CHECK (json_valid(`metadata`)),
            `is_read` tinyint(1) DEFAULT 0,
            `expires_at` datetime DEFAULT NULL,
            `created_at` datetime NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_department_read` (`department`,`is_read`),
            KEY `idx_expires` (`expires_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci
        ";
        
        $CI->db->query($sql);
        echo "<span style='color: green;'>✅ queue_notifications table created</span><br>";
    } else {
        echo "<span style='color: orange;'>⚠️ queue_notifications table already exists</span><br>";
    }
    
    // =============================================
    // 2. INSERT DEFAULT CONFIGURATION DATA
    // =============================================
    echo "<h3>⚙️ Inserting Default Configuration</h3>";
    
    // Default queue configuration
    $departments = [
        ['department' => 'registration', 'queue_prefix' => 'REG', 'avg_service_time' => 10],
        ['department' => 'triage', 'queue_prefix' => 'TRI', 'avg_service_time' => 15],
        ['department' => 'consultation', 'queue_prefix' => 'CON', 'avg_service_time' => 20],
        ['department' => 'pharmacy', 'queue_prefix' => 'PHM', 'avg_service_time' => 8],
        ['department' => 'laboratory', 'queue_prefix' => 'LAB', 'avg_service_time' => 12],
        ['department' => 'radiology', 'queue_prefix' => 'RAD', 'avg_service_time' => 25],
        ['department' => 'phlebotomy', 'queue_prefix' => 'PHL', 'avg_service_time' => 5],
        ['department' => 'emergency', 'queue_prefix' => 'EMR', 'avg_service_time' => 10]
    ];
    
    foreach ($departments as $dept) {
        // Check if config already exists
        $CI->db->where('department', $dept['department']);
        $exists = $CI->db->count_all_results('queue_config');
        
        if ($exists == 0) {
            $config = [
                'department' => $dept['department'],
                'queue_prefix' => $dept['queue_prefix'],
                'average_service_time' => $dept['avg_service_time'],
                'counter_current' => 1,
                'counter_reset' => 'daily',
                'max_queue_size' => 100,
                'sound_enabled' => 1,
                'display_enabled' => 1,
                'is_active' => 1
            ];
            
            $CI->db->insert('queue_config', $config);
            echo "<span style='color: green;'>✅ {$dept['department']} queue config added</span><br>";
        } else {
            echo "<span style='color: orange;'>⚠️ {$dept['department']} queue config already exists</span><br>";
        }
    }
    
    // Default display settings
    $display_settings = [
        ['department' => 'registration', 'display_name' => 'Registration', 'theme_color' => '#28a745', 'icon_class' => 'fa-user-plus', 'display_order' => 1],
        ['department' => 'triage', 'display_name' => 'Triage', 'theme_color' => '#17a2b8', 'icon_class' => 'fa-stethoscope', 'display_order' => 2],
        ['department' => 'consultation', 'display_name' => 'Consultation', 'theme_color' => '#007bff', 'icon_class' => 'fa-user-md', 'display_order' => 3],
        ['department' => 'pharmacy', 'display_name' => 'Pharmacy', 'theme_color' => '#fd7e14', 'icon_class' => 'fa-pills', 'display_order' => 4],
        ['department' => 'laboratory', 'display_name' => 'Laboratory', 'theme_color' => '#6f42c1', 'icon_class' => 'fa-vial', 'display_order' => 5],
        ['department' => 'radiology', 'display_name' => 'Radiology', 'theme_color' => '#20c997', 'icon_class' => 'fa-x-ray', 'display_order' => 6],
        ['department' => 'phlebotomy', 'display_name' => 'Phlebotomy', 'theme_color' => '#e83e8c', 'icon_class' => 'fa-syringe', 'display_order' => 7],
        ['department' => 'emergency', 'display_name' => 'Emergency', 'theme_color' => '#dc3545', 'icon_class' => 'fa-ambulance', 'display_order' => 8]
    ];
    
    foreach ($display_settings as $setting) {
        $CI->db->where('department', $setting['department']);
        $exists = $CI->db->count_all_results('queue_display_settings');
        
        if ($exists == 0) {
            $CI->db->insert('queue_display_settings', $setting);
            echo "<span style='color: green;'>✅ {$setting['department']} display settings added</span><br>";
        } else {
            echo "<span style='color: orange;'>⚠️ {$setting['department']} display settings already exist</span><br>";
        }
    }
    
    // Default flow rules
    $flow_rules = [
        [
            'rule_name' => 'Registration to Triage',
            'from_department' => 'registration',
            'to_department' => 'triage',
            'condition_type' => 'automatic',
            'workflow_type' => 'general',
            'priority_mapping' => 'maintain'
        ],
        [
            'rule_name' => 'Triage to Consultation',
            'from_department' => 'triage',
            'to_department' => 'consultation',
            'condition_type' => 'manual',
            'workflow_type' => 'general',
            'priority_mapping' => 'maintain'
        ]
    ];
    
    foreach ($flow_rules as $rule) {
        $CI->db->where('rule_name', $rule['rule_name']);
        $exists = $CI->db->count_all_results('queue_flow_rules');
        
        if ($exists == 0) {
            $CI->db->insert('queue_flow_rules', $rule);
            echo "<span style='color: green;'>✅ Flow rule '{$rule['rule_name']}' added</span><br>";
        } else {
            echo "<span style='color: orange;'>⚠️ Flow rule '{$rule['rule_name']}' already exists</span><br>";
        }
    }
    
    // =============================================
    // 3. CREATE CACHE DIRECTORY
    // =============================================
    echo "<h3>📁 Creating Cache Directory</h3>";
    
    $cache_dir = APPPATH . 'cache';
    if (!is_dir($cache_dir)) {
        mkdir($cache_dir, 0755, true);
        echo "<span style='color: green;'>✅ Cache directory created</span><br>";
    } else {
        echo "<span style='color: orange;'>⚠️ Cache directory already exists</span><br>";
    }
    
    // =============================================
    // 4. VERIFY QUEUE SYSTEM FILES
    // =============================================
    echo "<h3>📋 Verifying Queue System Files</h3>";
    
    $required_files = [
        'assets/js/multi_department_queue_integration.js' => 'Queue JavaScript Integration',
        'application/controllers/admin/Queue.php' => 'Queue Controller',
        'application/models/Queue_model.php' => 'Queue Model'
    ];
    
    foreach ($required_files as $file_path => $description) {
        if (file_exists($file_path)) {
            echo "<span style='color: green;'>✅ $description found</span><br>";
        } else {
            echo "<span style='color: red;'>❌ $description missing ($file_path)</span><br>";
        }
    }
    
    echo "<h3>🎉 Queue System Setup Complete!</h3>";
    echo "<div style='background: #d4edda; border: 1px solid #c3e6cb; color: #155724; padding: 15px; margin: 20px 0; border-radius: 5px;'>";
    echo "<h4>Next Steps:</h4>";
    echo "<ol>";
    echo "<li><strong>Test Patient Registration:</strong> Register a new patient and verify they are added to triage queue</li>";
    echo "<li><strong>Check Department Modules:</strong> Visit triage, consultation, pharmacy pages to see queue integration</li>";
    echo "<li><strong>Configure Notifications:</strong> Verify sound notifications work in each department</li>";
    echo "<li><strong>Customize Settings:</strong> Visit admin/queue to configure department-specific settings</li>";
    echo "<li><strong>Train Staff:</strong> Show staff how to use the queue management features</li>";
    echo "</ol>";
    echo "<p><strong>Access Queue Management:</strong> <a href='" . base_url() . "admin/queue' target='_blank'>Admin → Queue Management</a></p>";
    echo "</div>";
    
    echo "<hr><p><small>Queue Management System v1.0 - Integrated successfully!</small></p>";
    
} catch (Exception $e) {
    echo "<div style='background: #f8d7da; border: 1px solid #f5c6cb; color: #721c24; padding: 15px; margin: 20px 0; border-radius: 5px;'>";
    echo "<h4>❌ Setup Failed</h4>";
    echo "<p>Error: " . $e->getMessage() . "</p>";
    echo "<p>Please check your database connection and permissions.</p>";
    echo "</div>";
}
?>