// =============================================================================
// COMPLETE MULTI-DEPARTMENT QUEUE INTEGRATION
// File: assets/js/multi_department_queue_integration.js
// =============================================================================

/**
 * Universal Queue Integration for All Hospital Departments
 * This file provides queue integration for all your hospital modules
 */
class MultiDepartmentQueueIntegration {
    constructor() {
        this.currentDepartment = this.detectCurrentDepartment();
        this.queueEnabled = true;
        this.soundEnabled = localStorage.getItem('queueSoundEnabled') !== 'false';
        this.patientSelectionCallbacks = {};
        this.refreshInterval = null;
        this.notificationInterval = null;
        
        this.init();
    }
    
    init() {
        this.setupDepartmentIntegration();
        this.setupEventListeners();
        this.startQueuePolling();
        this.enhanceExistingInterface();
        
        console.log(`Queue integration initialized for: ${this.currentDepartment}`);
    }
    
    detectCurrentDepartment() {
        const path = window.location.pathname;
        const url = window.location.href;
        
        // Department detection based on URL patterns
        if (path.includes('/triage') || url.includes('triage')) return 'triage';
        if (path.includes('/pharmacy') || url.includes('pharmacy')) return 'pharmacy';
        if (path.includes('/laboratory') || url.includes('laboratory')) return 'laboratory';
        if (path.includes('/phlebotomy') || url.includes('phlebotomy')) return 'phlebotomy';
        if (path.includes('/radiology') || url.includes('radiology')) return 'radiology';
        if (path.includes('/consultation') || url.includes('consultation')) return 'consultation';
        if (path.includes('/registration') || url.includes('registration')) return 'registration';
        if (path.includes('/patients') || url.includes('patients')) return 'registration';
        if (path.includes('/ipd') || url.includes('ipd')) return 'ipd';
        if (path.includes('/emergency') || url.includes('emergency')) return 'emergency';
        if (path.includes('/nursing') || url.includes('nursing')) return 'nursing';
        if (path.includes('/operationtheatre') || url.includes('operationtheatre')) return 'operation_theatre';
        if (path.includes('/antenatal') || url.includes('antenatal')) return 'antenatal';
        if (path.includes('/pathology') || url.includes('pathology')) return 'pathology';
        
        return 'general';
    }
    
    setupDepartmentIntegration() {
        switch (this.currentDepartment) {
            case 'triage':
                this.setupTriageIntegration();
                break;
            case 'pharmacy':
                this.setupPharmacyIntegration();
                break;
            case 'laboratory':
            case 'phlebotomy':
                this.setupLaboratoryIntegration();
                break;
            case 'radiology':
                this.setupRadiologyIntegration();
                break;
            case 'consultation':
                this.setupConsultationIntegration();
                break;
            case 'registration':
                this.setupRegistrationIntegration();
                break;
            case 'ipd':
                this.setupIPDIntegration();
                break;
            case 'emergency':
                this.setupEmergencyIntegration();
                break;
            case 'nursing':
                this.setupNursingIntegration();
                break;
            case 'operation_theatre':
                this.setupOperationTheatreIntegration();
                break;
            case 'antenatal':
                this.setupAntenatalIntegration();
                break;
            case 'pathology':
                this.setupPathologyIntegration();
                break;
            default:
                this.setupGenericIntegration();
        }
    }
    
    // ===========================================
    // TRIAGE DEPARTMENT INTEGRATION
    // ===========================================
    setupTriageIntegration() {
        // Override existing loadPatients function if it exists
        if (typeof window.loadPatients === 'function') {
            this.originalLoadPatients = window.loadPatients;
        }
        
        // Replace with queue-aware version
        window.loadPatients = () => {
            this.loadTriageQueuePatients();
        };
        
        // Override patient selection if triage system exists
        if (typeof window.loadEnhancedPatientDetails === 'function') {
            this.originalLoadEnhancedPatientDetails = window.loadEnhancedPatientDetails;
            window.loadEnhancedPatientDetails = (patientId) => {
                this.handleTriagePatientSelection(patientId);
            };
        }
        
        // Override triage save function
        if (typeof window.saveEnhancedTriage === 'function') {
            this.originalSaveEnhancedTriage = window.saveEnhancedTriage;
            window.saveEnhancedTriage = () => {
                this.handleTriageSave();
            };
        }
    }
    
    async loadTriageQueuePatients() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getTriageQueue');
            
            if (response.status === 'success') {
                this.populateTriageDropdown(response.queue_patients);
            }
            
            // Call original function if it exists
            if (this.originalLoadPatients) {
                this.originalLoadPatients();
            }
        } catch (error) {
            console.error('Failed to load triage queue:', error);
            if (this.originalLoadPatients) {
                this.originalLoadPatients();
            }
        }
    }
    
    populateTriageDropdown(queuePatients) {
        const select = $('#triage_patient_search');
        if (!select.length) return;
        
        // Clear existing queue options
        select.find('optgroup[label*="Queue"]').remove();
        
        if (queuePatients.length > 0) {
            const queueGroup = $('<optgroup label="🔔 Patients in Triage Queue (Priority Order)">');
            
            queuePatients.forEach(patient => {
                const priorityIcon = this.getPriorityIcon(patient.priority);
                const waitTime = this.calculateWaitTime(patient.created_at);
                
                const text = `${priorityIcon} ${patient.queue_number} - ${patient.patient_name} ` +
                           `(${patient.age}y, ${patient.gender}) - Waiting: ${waitTime}`;
                
                const option = $('<option>', {
                    value: patient.patient_id,
                    text: text,
                    'data-queue-id': patient.queue_id,
                    'data-queue-number': patient.queue_number,
                    'data-priority': patient.priority,
                    'data-from-queue': 'true'
                });
                
                queueGroup.append(option);
            });
            
            select.prepend(queueGroup);
        }
    }
    
    handleTriagePatientSelection(patientId) {
        const selectedOption = $('#triage_patient_search').find(':selected');
        const isFromQueue = selectedOption.data('from-queue') === 'true';
        const queueId = selectedOption.data('queue-id');
        
        if (isFromQueue && queueId) {
            this.markQueueAsInProgress(queueId);
            
            // Store queue info for later use
            window.triageState = window.triageState || {};
            window.triageState.currentQueueId = queueId;
            window.triageState.fromQueue = true;
        }
        
        // Call original function
        if (this.originalLoadEnhancedPatientDetails) {
            this.originalLoadEnhancedPatientDetails(patientId);
        }
    }
    
    handleTriageSave() {
        // Call original save function first
        if (this.originalSaveEnhancedTriage) {
            this.originalSaveEnhancedTriage();
        }
        
        // Complete queue if patient came from queue
        if (window.triageState?.currentQueueId) {
            const departmentAssigned = $('#departmentAssigned').val() || null;
            this.completeQueueService(window.triageState.currentQueueId, departmentAssigned);
        }
    }
    
    // ===========================================
    // PHARMACY DEPARTMENT INTEGRATION
    // ===========================================
    setupPharmacyIntegration() {
        this.createPharmacyQueuePanel();
        this.loadPharmacyQueue();
        
        // Override any existing patient selection
        window.selectPatientForPharmacy = (patientId, queueId) => {
            this.handlePharmacyPatientSelection(patientId, queueId);
        };
    }
    
    createPharmacyQueuePanel() {
        const queuePanel = `
            <div class="panel panel-primary" id="pharmacyQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-pills"></i> Pharmacy Queue
                        <span class="badge" id="pharmacyQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="pharmacyQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        // Insert queue panel at the top of the main content area
        $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
    }
    
    async loadPharmacyQueue() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: 'pharmacy'
            });
            
            if (response.status === 'success') {
                this.renderPharmacyQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load pharmacy queue:', error);
        }
    }
    
    renderPharmacyQueue(queueList) {
        const container = $('#pharmacyQueueList');
        $('#pharmacyQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No patients in pharmacy queue</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer;"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handlePharmacyPatientSelection(${patient.patient_id}, ${patient.id})">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">${patient.mobileno || ''}</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handlePharmacyPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        
        // Load patient prescription data
        this.loadPatientPrescriptions(patientId);
        
        this.showNotification('Patient loaded for pharmacy service', 'success');
    }
    
    async loadPatientPrescriptions(patientId) {
        console.log(`Loading prescriptions for patient ${patientId}`);
        
        // Example: If you have a function to load prescriptions
        if (typeof window.loadPatientPrescriptions === 'function') {
            window.loadPatientPrescriptions(patientId);
        }
    }
    
    // ===========================================
    // LABORATORY DEPARTMENT INTEGRATION
    // ===========================================
    setupLaboratoryIntegration() {
        this.createLaboratoryQueuePanel();
        this.loadLaboratoryQueue();
        
        window.selectPatientForLab = (patientId, queueId) => {
            this.handleLaboratoryPatientSelection(patientId, queueId);
        };
    }
    
    createLaboratoryQueuePanel() {
        const queuePanel = `
            <div class="panel panel-info" id="laboratoryQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-vial"></i> Laboratory Queue
                        <span class="badge" id="laboratoryQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="laboratoryQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
    }
    
    async loadLaboratoryQueue() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: 'laboratory'
            });
            
            if (response.status === 'success') {
                this.renderLaboratoryQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load laboratory queue:', error);
        }
    }
    
    renderLaboratoryQueue(queueList) {
        const container = $('#laboratoryQueueList');
        $('#laboratoryQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No patients in laboratory queue</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer;"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handleLaboratoryPatientSelection(${patient.patient_id}, ${patient.id})">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">Lab Tests Required</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handleLaboratoryPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        
        // Load patient lab orders
        this.loadPatientLabOrders(patientId);
        
        this.showNotification('Patient loaded for laboratory service', 'success');
    }
    
    async loadPatientLabOrders(patientId) {
        console.log(`Loading lab orders for patient ${patientId}`);
        
        if (typeof window.loadPatientLabOrders === 'function') {
            window.loadPatientLabOrders(patientId);
        }
    }
    
    // ===========================================
    // CONSULTATION DEPARTMENT INTEGRATION
    // ===========================================
    setupConsultationIntegration() {
        this.createConsultationQueuePanel();
        this.loadConsultationQueue();
        
        window.loadPatientForConsultation = (patientId, queueId) => {
            this.handleConsultationPatientSelection(patientId, queueId);
        };
    }
    
    createConsultationQueuePanel() {
        const queuePanel = `
            <div class="panel panel-success" id="consultationQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-user-md"></i> Consultation Queue
                        <span class="badge" id="consultationQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="consultationQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
    }
    
    async loadConsultationQueue() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: 'consultation'
            });
            
            if (response.status === 'success') {
                this.renderConsultationQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load consultation queue:', error);
        }
    }
    
    renderConsultationQueue(queueList) {
        const container = $('#consultationQueueList');
        $('#consultationQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No patients in consultation queue</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer;"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handleConsultationPatientSelection(${patient.patient_id}, ${patient.id})">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">Ready for consultation</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handleConsultationPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        this.loadPatientConsultationData(patientId);
        this.showNotification('Patient loaded for consultation', 'success');
    }
    
    async loadPatientConsultationData(patientId) {
        console.log(`Loading consultation data for patient ${patientId}`);
        
        // Try common consultation functions
        if (typeof window.loadPatientForConsultation === 'function') {
            window.loadPatientForConsultation(patientId);
        } else if (typeof window.selectPatientConsultation === 'function') {
            window.selectPatientConsultation(patientId);
        }
    }
    
    // ===========================================
    // RADIOLOGY DEPARTMENT INTEGRATION
    // ===========================================
    setupRadiologyIntegration() {
        this.createRadiologyQueuePanel();
        this.loadRadiologyQueue();
        
        window.selectPatientForRadiology = (patientId, queueId) => {
            this.handleRadiologyPatientSelection(patientId, queueId);
        };
    }
    
    createRadiologyQueuePanel() {
        const queuePanel = `
            <div class="panel panel-warning" id="radiologyQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-x-ray"></i> Radiology Queue
                        <span class="badge" id="radiologyQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="radiologyQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
    }
    
    async loadRadiologyQueue() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: 'radiology'
            });
            
            if (response.status === 'success') {
                this.renderRadiologyQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load radiology queue:', error);
        }
    }
    
    renderRadiologyQueue(queueList) {
        const container = $('#radiologyQueueList');
        $('#radiologyQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No patients in radiology queue</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer;"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handleRadiologyPatientSelection(${patient.patient_id}, ${patient.id})">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">Imaging required</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handleRadiologyPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        this.loadPatientRadiologyOrders(patientId);
        this.showNotification('Patient loaded for radiology service', 'success');
    }
    
    async loadPatientRadiologyOrders(patientId) {
        console.log(`Loading radiology orders for patient ${patientId}`);
        
        if (typeof window.loadRadiologyOrders === 'function') {
            window.loadRadiologyOrders(patientId);
        }
    }
    
    // ===========================================
    // IPD DEPARTMENT INTEGRATION
    // ===========================================
    setupIPDIntegration() {
        this.createIPDQueuePanel();
        this.loadIPDQueue();
        
        window.selectPatientForIPD = (patientId, queueId) => {
            this.handleIPDPatientSelection(patientId, queueId);
        };
    }
    
    createIPDQueuePanel() {
        const queuePanel = `
            <div class="panel panel-default" id="ipdQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-bed"></i> IPD Admission Queue
                        <span class="badge" id="ipdQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="ipdQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
    }
    
    async loadIPDQueue() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: 'ipd'
            });
            
            if (response.status === 'success') {
                this.renderIPDQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load IPD queue:', error);
        }
    }
    
    renderIPDQueue(queueList) {
        const container = $('#ipdQueueList');
        $('#ipdQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No patients for IPD admission</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer;"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handleIPDPatientSelection(${patient.patient_id}, ${patient.id})">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">Awaiting admission</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handleIPDPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        this.loadPatientIPDData(patientId);
        this.showNotification('Patient loaded for IPD admission', 'success');
    }
    
    async loadPatientIPDData(patientId) {
        console.log(`Loading IPD data for patient ${patientId}`);
        
        if (typeof window.loadIPDAdmission === 'function') {
            window.loadIPDAdmission(patientId);
        }
    }
    
    // ===========================================
    // EMERGENCY DEPARTMENT INTEGRATION
    // ===========================================
    setupEmergencyIntegration() {
        this.createEmergencyQueuePanel();
        this.loadEmergencyQueue();
        
        window.selectEmergencyPatient = (patientId, queueId) => {
            this.handleEmergencyPatientSelection(patientId, queueId);
        };
    }
    
    createEmergencyQueuePanel() {
        const queuePanel = `
            <div class="panel panel-danger" id="emergencyQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-ambulance"></i> Emergency Queue
                        <span class="badge" id="emergencyQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="emergencyQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
    }
    
    async loadEmergencyQueue() {
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: 'emergency'
            });
            
            if (response.status === 'success') {
                this.renderEmergencyQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load emergency queue:', error);
        }
    }
    
    renderEmergencyQueue(queueList) {
        const container = $('#emergencyQueueList');
        $('#emergencyQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No emergency patients</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer; ${patient.priority === 'urgent' ? 'background-color: #ffebee;' : ''}"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handleEmergencyPatientSelection(${patient.patient_id}, ${patient.id})">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">Emergency case</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handleEmergencyPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        this.loadEmergencyPatientData(patientId);
        this.showNotification('Emergency patient loaded', 'warning');
    }
    
    async loadEmergencyPatientData(patientId) {
        console.log(`Loading emergency data for patient ${patientId}`);
        
        if (typeof window.loadEmergencyCase === 'function') {
            window.loadEmergencyCase(patientId);
        }
    }
    
    // ===========================================
    // REGISTRATION DEPARTMENT INTEGRATION
    // ===========================================
    setupRegistrationIntegration() {
        this.setupRegistrationQueueHandling();
        
        // Override patient save if it exists
        // Override patient save if it exists
        if (typeof window.savePatient === 'function') {
            this.originalSavePatient = window.savePatient;
            window.savePatient = (formData) => {
                this.handlePatientRegistrationSave(formData);
            };
        }
        
        // Monitor form submissions
        this.monitorPatientRegistration();
    }
    
    setupRegistrationQueueHandling() {
        // When a new patient is registered, add them to triage queue
        $(document).on('submit', '#patient_form, .patient-form', (e) => {
            setTimeout(() => {
                this.addNewPatientToTriageQueue();
            }, 1000);
        });
    }
    
    monitorPatientRegistration() {
        // Monitor for successful patient registration
        $(document).ajaxSuccess((event, xhr, options) => {
            if (options.url && options.url.includes('patients') && options.type === 'POST') {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.status === 'success' && response.patient_id) {
                        this.handleNewPatientRegistered(response.patient_id);
                    }
                } catch (e) {
                    // Not a JSON response, ignore
                }
            }
        });
    }
    
    async handleNewPatientRegistered(patientId) {
        try {
            await this.apiCall('POST', 'admin/queue/addToQueue', {
                patient_id: patientId,
                department: 'triage',
                priority: 'normal',
                workflow_type: 'general'
            });
            
            this.showNotification('New patient added to triage queue', 'success');
            this.playNotificationSound('new_patient');
        } catch (error) {
            console.error('Failed to add new patient to queue:', error);
        }
    }
    
    async addNewPatientToTriageQueue() {
        // Get the last registered patient ID from various possible sources
        const patientId = this.getLastRegisteredPatientId();
        
        if (patientId) {
            await this.handleNewPatientRegistered(patientId);
        }
    }
    
    getLastRegisteredPatientId() {
        // Try multiple ways to get the last registered patient ID
        
        // Check URL parameters
        const urlParams = new URLSearchParams(window.location.search);
        if (urlParams.get('patient_id')) {
            return urlParams.get('patient_id');
        }
        
        // Check global variables
        if (window.lastRegisteredPatientId) {
            return window.lastRegisteredPatientId;
        }
        
        // Check session storage
        if (sessionStorage.getItem('lastPatientId')) {
            return sessionStorage.getItem('lastPatientId');
        }
        
        // Check form data
        const patientIdInput = $('input[name="patient_id"], #patient_id').val();
        if (patientIdInput) {
            return patientIdInput;
        }
        
        return null;
    }
    
    handlePatientRegistrationSave(formData) {
        // Call original save function first
        if (this.originalSavePatient) {
            this.originalSavePatient(formData);
        }
        
        // Add delay to ensure patient is saved before adding to queue
        setTimeout(() => {
            this.addNewPatientToTriageQueue();
        }, 2000);
    }
    
    // ===========================================
    // GENERIC DEPARTMENT INTEGRATION
    // ===========================================
    setupGenericIntegration() {
        this.createGenericQueuePanel();
        this.loadGenericQueue();
    }
    
    createGenericQueuePanel() {
        if ($('#genericQueuePanel').length) return;
        
        const queuePanel = `
            <div class="panel panel-default" id="genericQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-users"></i> ${this.currentDepartment.charAt(0).toUpperCase() + this.currentDepartment.slice(1)} Queue
                        <span class="badge" id="genericQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="genericQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        // Try to insert in appropriate location
        if ($('.content-wrapper .row').length) {
            $('.content-wrapper .row').first().prepend(`<div class="col-md-4">${queuePanel}</div>`);
        } else if ($('.content-wrapper').length) {
            $('.content-wrapper').prepend(`<div class="row"><div class="col-md-4">${queuePanel}</div></div>`);
        } else if ($('.container-fluid').length) {
            $('.container-fluid').prepend(`<div class="row"><div class="col-md-4">${queuePanel}</div></div>`);
        }
    }
    
    async loadGenericQueue() {
        if (this.currentDepartment === 'general') return;
        
        try {
            const response = await this.apiCall('GET', 'admin/queue/getDepartmentQueue', {
                department: this.currentDepartment
            });
            
            if (response.status === 'success') {
                this.renderGenericQueue(response.data.queue_list || []);
            }
        } catch (error) {
            console.error('Failed to load generic queue:', error);
        }
    }
    
    renderGenericQueue(queueList) {
        const container = $('#genericQueueList');
        $('#genericQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            container.html('<div class="text-center text-muted">No patients in queue</div>');
            return;
        }
        
        const html = queueList.map(patient => `
            <div class="queue-item" style="padding: 10px; border-bottom: 1px solid #eee; cursor: pointer;"
                 data-patient-id="${patient.patient_id}" 
                 data-queue-id="${patient.id}"
                 onclick="multiDeptQueue.handleGenericPatientSelection(${patient.patient_id}, ${patient.id})"
                 title="Double-click to load patient in ${this.currentDepartment}">
                <div style="display: flex; justify-content: space-between; align-items: center;">
                    <div>
                        <strong>${patient.queue_number}</strong> - ${patient.patient_name || 'Unknown'}
                        <br><small class="text-muted">${patient.mobileno || ''}</small>
                    </div>
                    <div>
                        <span class="label label-${this.getPriorityLabelClass(patient.priority)}">${patient.priority}</span>
                        <br><small>${this.calculateWaitTime(patient.created_at)}</small>
                    </div>
                </div>
            </div>
        `).join('');
        
        container.html(html);
    }
    
    handleGenericPatientSelection(patientId, queueId) {
        this.markQueueAsInProgress(queueId);
        this.showNotification(`Patient loaded in ${this.currentDepartment}`, 'success');
        
        // Try to load patient in current module
        this.loadPatientInCurrentModule(patientId);
    }
    
    loadPatientInCurrentModule(patientId) {
        // Try common patient loading function names
        const loadFunctions = [
            'loadPatientDetails',
            'selectPatient', 
            'loadPatient',
            'getPatientData',
            'showPatientInfo',
            'displayPatientDetails',
            'openPatientRecord'
        ];
        
        for (const funcName of loadFunctions) {
            if (typeof window[funcName] === 'function') {
                window[funcName](patientId);
                console.log(`Called ${funcName}(${patientId})`);
                break;
            }
        }
        
        // Try jQuery-based selectors to populate patient data
        this.tryPopulatePatientFields(patientId);
    }
    
    async tryPopulatePatientFields(patientId) {
        try {
            // Try to get patient data
            const response = await this.apiCall('GET', 'admin/patients/getPatient', {
                id: patientId
            });
            
            if (response.status === 'success' && response.patient) {
                const patient = response.patient;
                
                // Try common field selectors
                $('input[name="patient_name"], #patient_name').val(patient.patient_name || '');
                $('input[name="patient_id"], #patient_id').val(patientId);
                $('input[name="phone"], #phone, input[name="mobile"], #mobile').val(patient.mobileno || '');
                $('select[name="gender"], #gender').val(patient.gender || '');
                $('input[name="age"], #age').val(patient.age || '');
                
                console.log('Patient fields populated automatically');
            }
        } catch (error) {
            console.error('Failed to populate patient fields:', error);
        }
    }
    
    // ===========================================
    // COMMON UTILITY METHODS
    // ===========================================
    
    setupEventListeners() {
        // Global double-click handler for queue items
        $(document).on('dblclick', '.queue-item', (e) => {
            const patientId = $(e.currentTarget).data('patient-id');
            const queueId = $(e.currentTarget).data('queue-id');
            
            if (patientId && queueId) {
                this.handlePatientDoubleClick(patientId, queueId);
            }
        });
        
        // Single click handler for mobile devices
        $(document).on('click', '.queue-item', (e) => {
            if (e.detail === 2) {  // Double click on mobile
                const patientId = $(e.currentTarget).data('patient-id');
                const queueId = $(e.currentTarget).data('queue-id');
                
                if (patientId && queueId) {
                    this.handlePatientDoubleClick(patientId, queueId);
                }
            }
        });
        
        // Refresh queue when page becomes visible
        document.addEventListener('visibilitychange', () => {
            if (!document.hidden) {
                this.refreshCurrentQueue();
            }
        });
        
        // Listen for custom events
        $(document).on('patientRegistered', (e, patientId) => {
            if (patientId) {
                this.handleNewPatientRegistered(patientId);
            }
        });
        
        $(document).on('queueRefreshNeeded', () => {
            this.refreshCurrentQueue();
        });
    }
    
    handlePatientDoubleClick(patientId, queueId) {
        this.playNotificationSound('patient_called');
        
        switch (this.currentDepartment) {
            case 'triage':
                this.handleTriagePatientSelection(patientId);
                break;
            case 'pharmacy':
                this.handlePharmacyPatientSelection(patientId, queueId);
                break;
            case 'laboratory':
            case 'phlebotomy':
                this.handleLaboratoryPatientSelection(patientId, queueId);
                break;
            case 'consultation':
                this.handleConsultationPatientSelection(patientId, queueId);
                break;
            case 'radiology':
                this.handleRadiologyPatientSelection(patientId, queueId);
                break;
            case 'ipd':
                this.handleIPDPatientSelection(patientId, queueId);
                break;
            case 'emergency':
                this.handleEmergencyPatientSelection(patientId, queueId);
                break;
            default:
                this.handleGenericPatientSelection(patientId, queueId);
        }
    }
    
    async markQueueAsInProgress(queueId) {
        try {
            await this.apiCall('POST', 'admin/queue/updateQueueStatus', {
                queue_id: queueId,
                status: 'in_progress'
            });
            
            console.log(`Queue ${queueId} marked as in-progress`);
            this.refreshCurrentQueue();
        } catch (error) {
            console.error('Failed to update queue status:', error);
        }
    }
    
    async completeQueueService(queueId, nextDepartment = null) {
        try {
            await this.apiCall('POST', 'admin/queue/completeService', {
                queue_id: queueId,
                next_department: nextDepartment,
                notes: `Service completed in ${this.currentDepartment}`
            });
            
            this.showNotification('Queue service completed', 'success');
            this.playNotificationSound('service_completed');
            this.refreshCurrentQueue();
        } catch (error) {
            console.error('Failed to complete queue service:', error);
        }
    }
    
    startQueuePolling() {
        // Refresh queue every 30 seconds
        this.refreshInterval = setInterval(() => {
            this.refreshCurrentQueue();
        }, 30000);
        
        // Check for notifications every 10 seconds
        this.notificationInterval = setInterval(() => {
            this.checkForNotifications();
        }, 10000);
    }
    
    async checkForNotifications() {
        try {
            const response = await this.apiCall('GET', `admin/${this.currentDepartment}/check_incoming_notifications`);
            
            if (response.status === 'success' && response.has_notification) {
                this.handleIncomingNotification(response.notification);
                
                if (response.sound) {
                    this.playNotificationSound('new_patient');
                }
            }
        } catch (error) {
            // Silently fail if endpoint doesn't exist
        }
    }
    
    handleIncomingNotification(notification) {
        this.showNotification(`New patient incoming from ${notification.from_department || 'registration'}`, 'info');
        this.refreshCurrentQueue();
    }
    
    refreshCurrentQueue() {
        switch (this.currentDepartment) {
            case 'triage':
                this.loadTriageQueuePatients();
                break;
            case 'pharmacy':
                this.loadPharmacyQueue();
                break;
            case 'laboratory':
            case 'phlebotomy':
                this.loadLaboratoryQueue();
                break;
            case 'consultation':
                this.loadConsultationQueue();
                break;
            case 'radiology':
                this.loadRadiologyQueue();
                break;
            case 'ipd':
                this.loadIPDQueue();
                break;
            case 'emergency':
                this.loadEmergencyQueue();
                break;
            default:
                this.loadGenericQueue();
        }
    }
    
    calculateWaitTime(createdAt) {
        const now = new Date();
        const created = new Date(createdAt);
        const diffMs = now - created;
        const diffMins = Math.floor(diffMs / 60000);
        
        if (diffMins < 60) {
            return `${diffMins}m`;
        } else {
            const hours = Math.floor(diffMins / 60);
            const mins = diffMins % 60;
            return `${hours}h ${mins}m`;
        }
    }
    
    getPriorityIcon(priority) {
        const icons = {
            urgent: '🚨',
            high: '⚠️',
            normal: '📋',
            low: '📄'
        };
        return icons[priority] || '📋';
    }
    
    getPriorityLabelClass(priority) {
        const classes = {
            urgent: 'danger',
            high: 'warning',
            normal: 'success',
            low: 'info'
        };
        return classes[priority] || 'default';
    }
    
    playNotificationSound(type = 'default') {
        if (!this.soundEnabled) return;
        
        // Create programmatic notification sounds
        try {
            const audioContext = new (window.AudioContext || window.webkitAudioContext)();
            const oscillator = audioContext.createOscillator();
            const gainNode = audioContext.createGain();
            
            oscillator.connect(gainNode);
            gainNode.connect(audioContext.destination);
            
            let frequency = 800;
            let duration = 200;
            
            switch (type) {
                case 'new_patient':
                    frequency = 600;
                    duration = 150;
                    break;
                case 'patient_called':
                    frequency = 1000;
                    duration = 300;
                    break;
                case 'service_completed':
                    frequency = 800;
                    duration = 200;
                    break;
                case 'urgent':
                    frequency = 1200;
                    duration = 400;
                    // Play twice for urgent
                    setTimeout(() => {
                        this.playNotificationSound('urgent_repeat');
                    }, 500);
                    break;
                case 'urgent_repeat':
                    frequency = 1200;
                    duration = 300;
                    break;
            }
            
            oscillator.frequency.value = frequency;
            oscillator.type = 'sine';
            
            gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
            gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + duration / 1000);
            
            oscillator.start(audioContext.currentTime);
            oscillator.stop(audioContext.currentTime + duration / 1000);
        } catch (error) {
            console.log('Sound generation failed:', error);
        }
    }
    
    showNotification(message, type = 'info') {
        const alertClass = {
            'success': 'alert-success',
            'error': 'alert-danger',
            'warning': 'alert-warning',
            'info': 'alert-info'
        }[type] || 'alert-info';
        
        const notification = $(`
            <div class="alert ${alertClass} alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px; max-width: 400px;">
                <button type="button" class="close" data-dismiss="alert">
                    <span>&times;</span>
                </button>
                <strong>${this.currentDepartment.charAt(0).toUpperCase() + this.currentDepartment.slice(1)} Queue:</strong> ${message}
            </div>
        `);
        
        $('body').append(notification);
        
        setTimeout(() => {
            notification.alert('close');
        }, 4000);
    }
    
    async apiCall(method, endpoint, data = {}) {
        const config = {
            method: method,
            headers: {
                'Content-Type': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        };

        let url = `${window.location.origin}/${endpoint}`;

        if (method === 'GET' && Object.keys(data).length > 0) {
            const params = new URLSearchParams(data);
            url += `?${params}`;
        } else if (method === 'POST') {
            config.body = JSON.stringify(data);
        }

        const response = await fetch(url, config);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        return await response.json();
    }
    
    // ===========================================
    // PUBLIC METHODS FOR EXTERNAL ACCESS
    // ===========================================
    
    /**
     * Manually add patient to queue from external modules
     */
    async addPatientToQueue(patientId, department = null, priority = 'normal') {
        const targetDepartment = department || this.currentDepartment;
        
        try {
            const response = await this.apiCall('POST', 'admin/queue/addToQueue', {
                patient_id: patientId,
                department: targetDepartment,
                priority: priority,
                workflow_type: 'general'
            });
            
            if (response.status === 'success') {
                this.showNotification(`Patient added to ${targetDepartment} queue`, 'success');
                this.playNotificationSound('new_patient');
                this.refreshCurrentQueue();
                return response;
            }
        } catch (error) {
            console.error('Failed to add patient to queue:', error);
            this.showNotification('Failed to add patient to queue', 'error');
        }
    }
    
    /**
     * Complete current patient service
     */
    async completeCurrentPatientService(queueId, nextDepartment = null) {
        return await this.completeQueueService(queueId, nextDepartment);
    }
    
    /**
     * Get current department queue count
     */
    getCurrentQueueCount() {
        const countElement = $(`#${this.currentDepartment}QueueCount, #genericQueueCount`);
        return parseInt(countElement.text()) || 0;
    }
    
    /**
     * Enable/disable sound notifications
     */
    toggleSound() {
        this.soundEnabled = !this.soundEnabled;
        localStorage.setItem('queueSoundEnabled', this.soundEnabled.toString());
        
        this.showNotification(`Queue sounds ${this.soundEnabled ? 'enabled' : 'disabled'}`, 'info');
        
        if (this.soundEnabled) {
            this.playNotificationSound('patient_called');
        }
        
        return this.soundEnabled;
    }
    
    /**
     * Get current department info
     */
    getCurrentDepartmentInfo() {
        return {
            department: this.currentDepartment,
            queueEnabled: this.queueEnabled,
            soundEnabled: this.soundEnabled,
            currentCount: this.getCurrentQueueCount()
        };
    }
    
    /**
     * Refresh queue manually
     */
    forceRefreshQueue() {
        this.refreshCurrentQueue();
    }
    
    /**
     * Set callback for patient selection
     */
    setPatientSelectionCallback(callback) {
        this.patientSelectionCallbacks[this.currentDepartment] = callback;
    }
    
    // ===========================================
    // HELPER METHODS FOR SPECIFIC INTEGRATIONS
    // ===========================================
    
    /**
     * Check if current page has specific elements (for better integration)
     */
    detectPageElements() {
        const elements = {
            patientForm: $('#patient_form, .patient-form').length > 0,
            patientDropdown: $('#patient_search, #triage_patient_search, select[name*="patient"]').length > 0,
            saveButton: $('button[type="submit"], .save-btn, #save-patient').length > 0,
            triageSystem: typeof window.loadEnhancedPatientDetails === 'function',
            pharmacySystem: $('#pharmacy_form, .pharmacy-panel').length > 0,
            labSystem: $('#lab_form, .lab-panel, #laboratory').length > 0
        };
        
        console.log('Page elements detected:', elements);
        return elements;
    }
    
    /**
     * Enhanced integration based on detected elements
     */
    enhanceExistingInterface() {
        const elements = this.detectPageElements();
        
        // Add queue button to existing forms
        if (elements.patientForm && !$('#addToQueueBtn').length) {
            this.addQueueButtonToForm();
        }
        
        // Enhance patient dropdowns with queue info
        if (elements.patientDropdown) {
            this.enhancePatientDropdowns();
        }
    }
    
    addQueueButtonToForm() {
        const queueButton = `
            <button type="button" class="btn btn-info btn-sm" id="addToQueueBtn" style="margin-left: 10px;">
                <i class="fa fa-plus"></i> Add to Queue
            </button>
        `;
        
        // Try to add near save button
        $('button[type="submit"]').first().after(queueButton);
        
        $('#addToQueueBtn').on('click', () => {
            const patientId = $('input[name="patient_id"], #patient_id').val();
            if (patientId) {
                this.addPatientToQueue(patientId);
            } else {
                this.showNotification('Please save patient first', 'warning');
            }
        });
    }
    
    enhancePatientDropdowns() {
        // Add queue indicator to patient options
        $('select[name*="patient"] option').each(function() {
            const option = $(this);
            const patientId = option.val();
            
            if (patientId) {
                // You could check if patient is in queue and add indicator
                // This would require additional API call
            }
        });
    }
    
    // ===========================================
    // CLEANUP AND DISPOSAL
    // ===========================================
    
    destroy() {
        // Clear intervals
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
        }
        
        if (this.notificationInterval) {
            clearInterval(this.notificationInterval);
        }
        
        // Remove event listeners
        $(document).off('dblclick', '.queue-item');
        $(document).off('click', '.queue-item');
        
        // Restore original functions
        if (this.originalLoadPatients) {
            window.loadPatients = this.originalLoadPatients;
        }
        
        if (this.originalLoadEnhancedPatientDetails) {
            window.loadEnhancedPatientDetails = this.originalLoadEnhancedPatientDetails;
        }
        
        if (this.originalSaveEnhancedTriage) {
            window.saveEnhancedTriage = this.originalSaveEnhancedTriage;
        }
        
        console.log('Multi-department queue integration cleaned up');
    }

    // ===========================================
    // NURSING DEPARTMENT INTEGRATION
    // ===========================================
    setupNursingIntegration() {
        this.createNursingQueuePanel();
        this.loadNursingQueue();
        
        window.selectNursingPatient = (patientId, queueId) => {
            this.handleNursingPatientSelection(patientId, queueId);
        };
    }
    
    createNursingQueuePanel() {
        const queuePanel = `
            <div class="panel panel-info" id="nursingQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-user-nurse"></i> Nursing Queue
                        <span class="badge" id="nursingQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="nursingQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('body').prepend(queuePanel);
    }
    
    loadNursingQueue() {
        $.get('/admin/nursing/get_queue_data', (data) => {
            if (data.status === 'success') {
                this.updateNursingQueueDisplay(data.queue_list || []);
            }
        });
    }
    
    updateNursingQueueDisplay(queueList) {
        const queueContainer = $('#nursingQueueList');
        $('#nursingQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            queueContainer.html('<div class="text-center text-muted">No patients in queue</div>');
            return;
        }
        
        let html = '';
        queueList.forEach(patient => {
            html += `
                <div class="queue-item priority-${patient.priority}" onclick="selectNursingPatient(${patient.patient_id}, ${patient.id})">
                    <span class="queue-number">${patient.queue_number}</span>
                    <span class="patient-name">${patient.patient_name}</span>
                    <span class="priority-badge">${patient.priority}</span>
                </div>
            `;
        });
        
        queueContainer.html(html);
    }
    
    handleNursingPatientSelection(patientId, queueId) {
        console.log(`Selected nursing patient: ${patientId}, Queue: ${queueId}`);
        this.loadPatientDetails(patientId, 'nursing');
    }

    // ===========================================
    // OPERATION THEATRE INTEGRATION
    // ===========================================
    setupOperationTheatreIntegration() {
        this.createOperationTheatreQueuePanel();
        this.loadOperationTheatreQueue();
        
        window.selectOperationPatient = (patientId, queueId) => {
            this.handleOperationTheatrePatientSelection(patientId, queueId);
        };
    }
    
    createOperationTheatreQueuePanel() {
        const queuePanel = `
            <div class="panel panel-warning" id="operationTheatreQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-procedures"></i> Operation Theatre Queue
                        <span class="badge" id="operationTheatreQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="operationTheatreQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('body').prepend(queuePanel);
    }
    
    loadOperationTheatreQueue() {
        $.get('/admin/operationtheatre/get_queue_data', (data) => {
            if (data.status === 'success') {
                this.updateOperationTheatreQueueDisplay(data.queue_list || []);
            }
        });
    }
    
    updateOperationTheatreQueueDisplay(queueList) {
        const queueContainer = $('#operationTheatreQueueList');
        $('#operationTheatreQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            queueContainer.html('<div class="text-center text-muted">No patients scheduled</div>');
            return;
        }
        
        let html = '';
        queueList.forEach(patient => {
            html += `
                <div class="queue-item priority-${patient.priority}" onclick="selectOperationPatient(${patient.patient_id}, ${patient.id})">
                    <span class="queue-number">${patient.queue_number}</span>
                    <span class="patient-name">${patient.patient_name}</span>
                    <span class="priority-badge">${patient.priority}</span>
                </div>
            `;
        });
        
        queueContainer.html(html);
    }
    
    handleOperationTheatrePatientSelection(patientId, queueId) {
        console.log(`Selected operation theatre patient: ${patientId}, Queue: ${queueId}`);
        this.loadPatientDetails(patientId, 'operation_theatre');
    }

    // ===========================================
    // ANTENATAL INTEGRATION
    // ===========================================
    setupAntenatalIntegration() {
        this.createAntenatalQueuePanel();
        this.loadAntenatalQueue();
        
        window.selectAntenatalPatient = (patientId, queueId) => {
            this.handleAntenatalPatientSelection(patientId, queueId);
        };
    }
    
    createAntenatalQueuePanel() {
        const queuePanel = `
            <div class="panel panel-success" id="antenatalQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-baby"></i> Antenatal Queue
                        <span class="badge" id="antenatalQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="antenatalQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('body').prepend(queuePanel);
    }
    
    loadAntenatalQueue() {
        $.get('/admin/antenatal/get_queue_data', (data) => {
            if (data.status === 'success') {
                this.updateAntenatalQueueDisplay(data.queue_list || []);
            }
        });
    }
    
    updateAntenatalQueueDisplay(queueList) {
        const queueContainer = $('#antenatalQueueList');
        $('#antenatalQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            queueContainer.html('<div class="text-center text-muted">No patients in queue</div>');
            return;
        }
        
        let html = '';
        queueList.forEach(patient => {
            html += `
                <div class="queue-item priority-${patient.priority}" onclick="selectAntenatalPatient(${patient.patient_id}, ${patient.id})">
                    <span class="queue-number">${patient.queue_number}</span>
                    <span class="patient-name">${patient.patient_name}</span>
                    <span class="priority-badge">${patient.priority}</span>
                </div>
            `;
        });
        
        queueContainer.html(html);
    }
    
    handleAntenatalPatientSelection(patientId, queueId) {
        console.log(`Selected antenatal patient: ${patientId}, Queue: ${queueId}`);
        this.loadPatientDetails(patientId, 'antenatal');
    }

    // ===========================================
    // PATHOLOGY INTEGRATION
    // ===========================================
    setupPathologyIntegration() {
        this.createPathologyQueuePanel();
        this.loadPathologyQueue();
        
        window.selectPathologyPatient = (patientId, queueId) => {
            this.handlePathologyPatientSelection(patientId, queueId);
        };
    }
    
    createPathologyQueuePanel() {
        const queuePanel = `
            <div class="panel panel-primary" id="pathologyQueuePanel">
                <div class="panel-heading">
                    <h4 class="panel-title">
                        <i class="fa fa-microscope"></i> Pathology Queue
                        <span class="badge" id="pathologyQueueCount">0</span>
                    </h4>
                </div>
                <div class="panel-body">
                    <div class="queue-list" id="pathologyQueueList">
                        <div class="text-center text-muted">Loading queue...</div>
                    </div>
                </div>
            </div>
        `;
        
        $('body').prepend(queuePanel);
    }
    
    loadPathologyQueue() {
        $.get('/admin/pathology/get_queue_data', (data) => {
            if (data.status === 'success') {
                this.updatePathologyQueueDisplay(data.queue_list || []);
            }
        });
    }
    
    updatePathologyQueueDisplay(queueList) {
        const queueContainer = $('#pathologyQueueList');
        $('#pathologyQueueCount').text(queueList.length);
        
        if (queueList.length === 0) {
            queueContainer.html('<div class="text-center text-muted">No tests pending</div>');
            return;
        }
        
        let html = '';
        queueList.forEach(patient => {
            html += `
                <div class="queue-item priority-${patient.priority}" onclick="selectPathologyPatient(${patient.patient_id}, ${patient.id})">
                    <span class="queue-number">${patient.queue_number}</span>
                    <span class="patient-name">${patient.patient_name}</span>
                    <span class="priority-badge">${patient.priority}</span>
                </div>
            `;
        });
        
        queueContainer.html(html);
    }
    
    handlePathologyPatientSelection(patientId, queueId) {
        console.log(`Selected pathology patient: ${patientId}, Queue: ${queueId}`);
        this.loadPatientDetails(patientId, 'pathology');
    }
}

// Initialize when DOM is ready
$(document).ready(function() {
    window.multiDeptQueue = new MultiDepartmentQueueIntegration();
    
    // Expose useful methods globally
    window.addPatientToQueue = function(patientId, department, priority) {
        if (window.multiDeptQueue) {
            return window.multiDeptQueue.addPatientToQueue(patientId, department, priority);
        }
    };
    
    window.completeQueueService = function(queueId, nextDepartment) {
        if (window.multiDeptQueue) {
            return window.multiDeptQueue.completeCurrentPatientService(queueId, nextDepartment);
        }
    };
    
    window.refreshQueue = function() {
        if (window.multiDeptQueue) {
            window.multiDeptQueue.forceRefreshQueue();
        }
    };
    
    window.toggleQueueSound = function() {
        if (window.multiDeptQueue) {
            return window.multiDeptQueue.toggleSound();
        }
    };
    
    // Debug helpers
    window.debugQueue = function() {
        if (window.multiDeptQueue) {
            console.log('Current Department:', window.multiDeptQueue.getCurrentDepartmentInfo());
            console.log('Queue Integration Object:', window.multiDeptQueue);
        }
    };
});

// Cleanup on page unload
window.addEventListener('beforeunload', function() {
    if (window.multiDeptQueue) {
        window.multiDeptQueue.destroy();
    }
});

// Export for use in other modules
window.MultiDepartmentQueueIntegration = MultiDepartmentQueueIntegration;