<?php
$currency_symbol = $this->customlib->getHospitalCurrencyFormat();
?>

<style>
    .nav-tabs-custom {
        margin-bottom: 20px;
        background: #fff;
        box-shadow: 0 1px 1px rgba(0,0,0,0.1);
        border-radius: 3px;
    }
    
    .nav-tabs-custom > .nav-tabs {
        margin: 0;
        border-bottom-color: #f4f4f4;
    }
    
    .nav-tabs-custom > .nav-tabs > li {
        border-top: 3px solid transparent;
        margin-bottom: -2px;
        margin-right: 5px;
    }
    
    .nav-tabs-custom > .nav-tabs > li > a {
        color: #444;
        border-radius: 0;
    }
    
    .nav-tabs-custom > .nav-tabs > li.active {
        border-top-color: #3c8dbc;
    }
    
    .nav-tabs-custom > .nav-tabs > li.active > a {
        background-color: #fff;
        color: #444;
    }
    
    .permission-group {
        background: #f9f9f9;
        border: 1px solid #d2d6de;
        border-radius: 3px;
        padding: 15px;
        margin-bottom: 15px;
    }
    
    .permission-item {
        background: white;
        border: 1px solid #e6e6e6;
        border-radius: 3px;
        padding: 10px;
        margin-bottom: 10px;
        transition: all 0.3s ease;
    }
    
    .permission-item:hover {
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
    }
    
    .role-card {
        background: white;
        border: 1px solid #d2d6de;
        border-radius: 3px;
        padding: 15px;
        margin-bottom: 15px;
        cursor: pointer;
        transition: all 0.3s ease;
    }
    
    .role-card:hover {
        box-shadow: 0 2px 5px rgba(0,0,0,0.1);
        border-color: #3c8dbc;
    }
    
    .role-card.selected {
        border-color: #3c8dbc;
        box-shadow: 0 2px 5px rgba(60,141,188,0.3);
    }
    
    .quick-actions {
        background: #ecf0f5;
        border: 1px solid #d2d6de;
        border-radius: 3px;
        padding: 15px;
        margin-bottom: 20px;
    }
</style>

<div class="content-wrapper">
    <section class="content-header">
        <h1>
            <i class="fa fa-users"></i> Roles & Permissions Manager
            <small>Advanced management system for roles and permissions</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="<?php echo base_url(); ?>admin/roles">Roles</a></li>
            <li class="active">Permissions Manager</li>
        </ol>
    </section>

    <section class="content">
        <?php if ($this->session->flashdata('msg')) { ?>
            <?php echo $this->session->flashdata('msg'); ?>
            <?php $this->session->unset_userdata('msg'); ?>
        <?php } ?>
        
        <!-- Alert Messages -->
        <div id="alertContainer"></div>

        <div class="nav-tabs-custom">
            <ul class="nav nav-tabs">
                <li class="active"><a href="#tab_1" data-toggle="tab"><i class="fa fa-user-plus"></i> Manage Roles</a></li>
                <li><a href="#tab_2" data-toggle="tab"><i class="fa fa-key"></i> Manage Permissions</a></li>
                <li><a href="#tab_3" data-toggle="tab"><i class="fa fa-link"></i> Assign Permissions</a></li>
                <li><a href="#tab_4" data-toggle="tab"><i class="fa fa-download"></i> Export/Import</a></li>
            </ul>
            
            <div class="tab-content">
                
                <!-- Tab 1: Roles Management -->
                <div class="tab-pane active" id="tab_1">
                    <div class="row">
                        <div class="col-md-4">
                            <div class="box box-primary">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-plus"></i> Add New Role</h3>
                                </div>
                                <div class="box-body">
                                    <form id="addRoleForm">
                                        <?php echo $this->customlib->getCSRF(); ?>
                                        <div class="form-group">
                                            <label for="roleName">Role Name</label>
                                            <input type="text" class="form-control" id="roleName" placeholder="e.g., Lab Manager" required>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="roleSlug">Role Slug</label>
                                            <input type="text" class="form-control" id="roleSlug" placeholder="e.g., lab-manager" readonly>
                                            <small class="help-block">Auto-generated from role name</small>
                                        </div>
                                        
                                        <div class="form-group">
                                            <div class="checkbox">
                                                <label>
                                                    <input type="checkbox" id="isSystemRole"> System Role (Cannot be deleted)
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <div class="form-group">
                                            <div class="checkbox">
                                                <label>
                                                    <input type="checkbox" id="isActive" checked> Active Role
                                                </label>
                                            </div>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-primary btn-block">
                                            <i class="fa fa-plus"></i> Add Role
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-8">
                            <div class="box box-primary">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-list"></i> Existing Roles</h3>
                                </div>
                                <div class="box-body table-responsive no-padding">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>ID</th>
                                                <th>Role Name</th>
                                                <th>Type</th>
                                                <th>Status</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody id="rolesTableBody">
                                            <!-- Populated by JavaScript -->
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tab 2: Permissions Management -->
                <div class="tab-pane" id="tab_2">
                    <div class="row">
                        <div class="col-md-5">
                            <div class="box box-success">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-layer-group"></i> Add Permission Group</h3>
                                </div>
                                <div class="box-body">
                                    <form id="addPermissionGroupForm">
                                        <?php echo $this->customlib->getCSRF(); ?>
                                        <div class="form-group">
                                            <label for="groupName">Group Name</label>
                                            <input type="text" class="form-control" id="groupName" placeholder="e.g., Laboratory Management" required>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="groupShortCode">Short Code</label>
                                            <input type="text" class="form-control" id="groupShortCode" placeholder="e.g., lab_mgmt" readonly>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="sortOrder">Sort Order</label>
                                            <input type="number" class="form-control" id="sortOrder" value="10" step="0.1">
                                        </div>
                                        
                                        <button type="submit" class="btn btn-success btn-block">
                                            <i class="fa fa-plus"></i> Add Group
                                        </button>
                                    </form>
                                </div>
                            </div>
                            
                            <div class="box box-info">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-key"></i> Add Permission</h3>
                                </div>
                                <div class="box-body">
                                    <form id="addPermissionForm">
                                        <?php echo $this->customlib->getCSRF(); ?>
                                        <div class="form-group">
                                            <label for="permissionGroup">Permission Group</label>
                                            <select class="form-control" id="permissionGroup" required>
                                                <option value="">Select Group</option>
                                            </select>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="permissionName">Permission Name</label>
                                            <input type="text" class="form-control" id="permissionName" placeholder="e.g., Lab Test Results" required>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label for="permissionShortCode">Short Code</label>
                                            <input type="text" class="form-control" id="permissionShortCode" placeholder="e.g., lab_test_results" readonly>
                                        </div>
                                        
                                        <div class="form-group">
                                            <label>Available Actions</label>
                                            <div class="row">
                                                <div class="col-xs-6">
                                                    <div class="checkbox">
                                                        <label><input type="checkbox" id="enableView" checked> View</label>
                                                    </div>
                                                    <div class="checkbox">
                                                        <label><input type="checkbox" id="enableAdd"> Add</label>
                                                    </div>
                                                </div>
                                                <div class="col-xs-6">
                                                    <div class="checkbox">
                                                        <label><input type="checkbox" id="enableEdit"> Edit</label>
                                                    </div>
                                                    <div class="checkbox">
                                                        <label><input type="checkbox" id="enableDelete"> Delete</label>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <button type="submit" class="btn btn-info btn-block">
                                            <i class="fa fa-plus"></i> Add Permission
                                        </button>
                                    </form>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-7">
                            <div class="box box-primary">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-sitemap"></i> Permission Structure</h3>
                                </div>
                                <div class="box-body">
                                    <div id="permissionStructure">
                                        <!-- Populated by JavaScript -->
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tab 3: Assign Permissions -->
                <div class="tab-pane" id="tab_3">
                    <div class="quick-actions">
                        <h4><i class="fa fa-bolt"></i> Quick Actions</h4>
                        <div class="btn-group">
                            <button class="btn btn-success btn-sm" onclick="copyPermissions()">
                                <i class="fa fa-copy"></i> Copy Permissions
                            </button>
                            <button class="btn btn-primary btn-sm" onclick="bulkAssign()">
                                <i class="fa fa-users"></i> Bulk Assign
                            </button>
                            <button class="btn btn-danger btn-sm" onclick="resetPermissions()">
                                <i class="fa fa-undo"></i> Reset All
                            </button>
                        </div>
                    </div>
                    
                    <div class="row">
                        <div class="col-md-3">
                            <div class="box box-primary">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-user-check"></i> Select Role</h3>
                                </div>
                                <div class="box-body">
                                    <div id="roleSelection">
                                        <!-- Populated by JavaScript -->
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-9">
                            <div class="box box-primary">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-cogs"></i> Assign Permissions</h3>
                                </div>
                                <div class="box-body">
                                    <div id="permissionAssignment">
                                        <div class="text-center text-muted" style="padding: 50px;">
                                            <i class="fa fa-arrow-left fa-3x" style="margin-bottom: 15px;"></i>
                                            <h4>Select a role to assign permissions</h4>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Tab 4: Export/Import -->
                <div class="tab-pane" id="tab_4">
                    <div class="row">
                        <div class="col-md-6">
                            <div class="box box-warning">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-download"></i> Export Configuration</h3>
                                </div>
                                <div class="box-body">
                                    <p class="text-muted">Export your current roles and permissions configuration.</p>
                                    
                                    <div class="form-group">
                                        <label>Export Format</label>
                                        <div class="radio">
                                            <label><input type="radio" name="exportFormat" value="sql" checked> SQL Statements</label>
                                        </div>
                                        <div class="radio">
                                            <label><input type="radio" name="exportFormat" value="json"> JSON Format</label>
                                        </div>
                                    </div>
                                    
                                    <div class="form-group">
                                        <label>Include Data</label>
                                        <div class="checkbox">
                                            <label><input type="checkbox" id="includeRoles" checked> Include Roles</label>
                                        </div>
                                        <div class="checkbox">
                                            <label><input type="checkbox" id="includePermissions" checked> Include Permissions</label>
                                        </div>
                                        <div class="checkbox">
                                            <label><input type="checkbox" id="includeAssignments" checked> Include Role Assignments</label>
                                        </div>
                                    </div>
                                    
                                    <button class="btn btn-warning btn-block" onclick="exportConfiguration()">
                                        <i class="fa fa-download"></i> Export Configuration
                                    </button>
                                </div>
                            </div>
                        </div>
                        
                        <div class="col-md-6">
                            <div class="box box-info">
                                <div class="box-header with-border">
                                    <h3 class="box-title"><i class="fa fa-info-circle"></i> System Status</h3>
                                </div>
                                <div class="box-body">
                                    <div class="row">
                                        <div class="col-sm-6">
                                            <div class="description-block border-right">
                                                <span class="description-percentage text-green"><i class="fa fa-users"></i></span>
                                                <h5 class="description-header"><?php echo count($roles); ?></h5>
                                                <span class="description-text">TOTAL ROLES</span>
                                            </div>
                                        </div>
                                        <div class="col-sm-6">
                                            <div class="description-block">
                                                <span class="description-percentage text-blue"><i class="fa fa-key"></i></span>
                                                <h5 class="description-header"><?php echo count($permission_groups); ?></h5>
                                                <span class="description-text">PERMISSION GROUPS</span>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <div class="row">
                                        <div class="col-sm-6">
                                            <div class="description-block border-right">
                                                <span class="description-percentage text-yellow"><i class="fa fa-cog"></i></span>
                                                <h5 class="description-header"><?php echo count($permission_categories); ?></h5>
                                                <span class="description-text">PERMISSIONS</span>
                                            </div>
                                        </div>
                                        <div class="col-sm-6">
                                            <div class="description-block">
                                                <span class="description-percentage text-red"><i class="fa fa-link"></i></span>
                                                <h5 class="description-header"><?php echo count($role_permissions); ?></h5>
                                                <span class="description-text">ASSIGNMENTS</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                <div class="box-footer">
                                    <div class="row">
                                        <div class="col-sm-6">
                                            <a href="<?php echo site_url('admin/roles'); ?>" class="btn btn-default btn-block">
                                                <i class="fa fa-cog"></i> Standard Interface
                                            </a>
                                        </div>
                                        <div class="col-sm-6">
                                            <button class="btn btn-info btn-block" onclick="createBackup()">
                                                <i class="fa fa-save"></i> Create Backup
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
            </div>
        </div>
    </section>
</div>

<!-- Edit Role Modal -->
<div class="modal fade" id="editRoleModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
                <h4 class="modal-title">Edit Role</h4>
            </div>
            <div class="modal-body">
                <form id="editRoleForm">
                    <input type="hidden" id="editRoleId">
                    <div class="form-group">
                        <label for="editRoleName">Role Name</label>
                        <input type="text" class="form-control" id="editRoleName" required>
                    </div>
                    <div class="form-group">
                        <label for="editRoleSlug">Role Slug</label>
                        <input type="text" class="form-control" id="editRoleSlug" readonly>
                    </div>
                    <div class="form-group">
                        <div class="checkbox">
                            <label><input type="checkbox" id="editIsSystemRole"> System Role</label>
                        </div>
                    </div>
                    <div class="form-group">
                        <div class="checkbox">
                            <label><input type="checkbox" id="editIsActive"> Active Role</label>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" onclick="updateRole()">Update Role</button>
            </div>
        </div>
    </div>
</div>

<!-- Copy Permissions Modal -->
<div class="modal fade" id="copyPermissionsModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal"><span>&times;</span></button>
                <h4 class="modal-title">Copy Permissions</h4>
            </div>
            <div class="modal-body">
                <div class="form-group">
                    <label for="sourceRole">Copy From (Source Role)</label>
                    <select class="form-control" id="sourceRole" required>
                        <option value="">Select source role</option>
                    </select>
                </div>
                <div class="form-group">
                    <label for="targetRole">Copy To (Target Role)</label>
                    <select class="form-control" id="targetRole" required>
                        <option value="">Select target role</option>
                    </select>
                </div>
                <div class="form-group">
                    <div class="checkbox">
                        <label><input type="checkbox" id="overwriteExisting"> Overwrite existing permissions</label>
                    </div>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-success" onclick="executeCopyPermissions()">Copy Permissions</button>
            </div>
        </div>
    </div>
</div>

<script>
    var base_url = '<?php echo base_url(); ?>';
    var site_url = '<?php echo site_url(); ?>';
    
    // Initialize data from PHP
    let roles = <?php echo json_encode($roles); ?>;
    let permissionGroups = <?php echo json_encode($permission_groups); ?>;
    let permissionCategories = <?php echo json_encode($permission_categories); ?>;
    let rolePermissions = <?php echo json_encode($role_permissions); ?>;

    // Initialize the application
    $(document).ready(function() {
        initializeApp();
    });

    function initializeApp() {
        loadRoles();
        loadPermissionGroups();
        loadPermissionStructure();
        loadRoleSelection();
        setupEventListeners();
    }

    function setupEventListeners() {
        // Role name to slug conversion
        $('#roleName').on('input', function() {
            const slug = $(this).val().toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
            $('#roleSlug').val(slug);
        });

        // Group name to short code conversion
        $('#groupName').on('input', function() {
            const shortCode = $(this).val().toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/(^_|_$)/g, '');
            $('#groupShortCode').val(shortCode);
        });

        // Permission name to short code conversion
        $('#permissionName').on('input', function() {
            const shortCode = $(this).val().toLowerCase().replace(/[^a-z0-9]+/g, '_').replace(/(^_|_$)/g, '');
            $('#permissionShortCode').val(shortCode);
        });

        // Form submissions
        $('#addRoleForm').on('submit', handleAddRole);
        $('#addPermissionGroupForm').on('submit', handleAddPermissionGroup);
        $('#addPermissionForm').on('submit', handleAddPermission);

        // Edit role name to slug conversion
        $('#editRoleName').on('input', function() {
            const slug = $(this).val().toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/(^-|-$)/g, '');
            $('#editRoleSlug').val(slug);
        });
    }

    function showAlert(message, type = 'success') {
        const alertClass = type === 'success' ? 'alert-success' : 'alert-danger';
        const iconClass = type === 'success' ? 'fa-check' : 'fa-exclamation-triangle';
        
        const alertHtml = `
            <div class="alert ${alertClass} alert-dismissible">
                <button type="button" class="close" data-dismiss="alert" aria-hidden="true">&times;</button>
                <h4><i class="icon fa ${iconClass}"></i> ${type === 'success' ? 'Success!' : 'Error!'}</h4>
                ${message}
            </div>
        `;
        
        $('#alertContainer').html(alertHtml);
        
        // Auto-dismiss after 5 seconds
        setTimeout(function() {
            $('#alertContainer .alert').fadeOut();
        }, 5000);
    }

    function loadRoles() {
        const tbody = $('#rolesTableBody');
        tbody.empty();

        roles.forEach(role => {
            const statusBadge = role.is_active ? 
                '<span class="label label-success">Active</span>' : 
                '<span class="label label-default">Inactive</span>';
            
            const typeBadge = role.is_superadmin ? 
                '<span class="label label-danger">Super Admin</span>' : 
                role.is_system ? 
                '<span class="label label-primary">System</span>' : 
                '<span class="label label-info">Custom</span>';

            const row = `
                <tr>
                    <td>${role.id}</td>
                    <td><strong>${role.name}</strong></td>
                    <td>${typeBadge}</td>
                    <td>${statusBadge}</td>
                    <td>
                        <div class="btn-group btn-group-xs">
                            <button class="btn btn-primary" onclick="editRole(${role.id})" title="Edit">
                                <i class="fa fa-edit"></i>
                            </button>
                            <a href="${site_url}/admin/roles/permission/${role.id}" class="btn btn-info" title="View Permissions">
                                <i class="fa fa-key"></i>
                            </a>
                            ${!role.is_system ? `
                            <button class="btn btn-danger" onclick="deleteRole(${role.id})" title="Delete">
                                <i class="fa fa-trash"></i>
                            </button>` : ''}
                        </div>
                    </td>
                </tr>
            `;
            tbody.append(row);
        });
    }

    function loadPermissionGroups() {
        const select = $('#permissionGroup');
        select.empty().append('<option value="">Select Group</option>');
        
        permissionGroups.forEach(group => {
            select.append(`<option value="${group.id}">${group.name}</option>`);
        });
    }

    function loadPermissionStructure() {
        const container = $('#permissionStructure');
        container.empty();

        permissionGroups.forEach(group => {
            const groupCategories = permissionCategories.filter(cat => cat.perm_group_id == group.id);
            
            let groupHtml = `
                <div class="permission-group">
                    <div class="row">
                        <div class="col-md-9">
                            <h4 style="margin-top: 0;">
                                <i class="fa fa-folder text-primary"></i>
                                ${group.name}
                                <small class="text-muted">(${group.short_code})</small>
                            </h4>
                        </div>
                        <div class="col-md-3 text-right">
                            <div class="btn-group btn-group-xs">
                                <button class="btn btn-default" onclick="editPermissionGroup(${group.id})">
                                    <i class="fa fa-edit"></i>
                                </button>
                                <button class="btn btn-danger" onclick="deletePermissionGroup(${group.id})">
                                    <i class="fa fa-trash"></i>
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <div class="row">
            `;
            
            groupCategories.forEach(category => {
                groupHtml += `
                    <div class="col-md-6" style="margin-bottom: 10px;">
                        <div class="permission-item">
                            <div class="row">
                                <div class="col-xs-8">
                                    <strong>${category.name}</strong>
                                    <br><small class="text-muted">${category.short_code}</small>
                                </div>
                                <div class="col-xs-4 text-right">
                                    <div style="margin-bottom: 5px;">
                                        ${category.enable_view ? '<span class="label label-success">V</span>' : ''}
                                        ${category.enable_add ? '<span class="label label-primary">A</span>' : ''}
                                        ${category.enable_edit ? '<span class="label label-warning">E</span>' : ''}
                                        ${category.enable_delete ? '<span class="label label-danger">D</span>' : ''}
                                    </div>
                                    <div class="btn-group btn-group-xs">
                                        <button class="btn btn-default" onclick="editPermission(${category.id})">
                                            <i class="fa fa-edit"></i>
                                        </button>
                                        <button class="btn btn-danger" onclick="deletePermission(${category.id})">
                                            <i class="fa fa-trash"></i>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                `;
            });
            
            groupHtml += `
                    </div>
                </div>`;
           container.append(groupHtml);
       });
   }

   function loadRoleSelection() {
       const container = $('#roleSelection');
       container.empty();

       roles.forEach(role => {
           const permissionCount = rolePermissions.filter(rp => rp.role_id == role.id).length;
           
           const roleCard = `
               <div class="role-card" onclick="selectRole(${role.id})" data-role-id="${role.id}">
                   <div class="row">
                       <div class="col-xs-8">
                           <h5 style="margin-top: 0; margin-bottom: 5px;">${role.name}</h5>
                           <small class="text-muted">${permissionCount} permissions</small>
                       </div>
                       <div class="col-xs-4 text-right">
                           ${role.is_superadmin ? '<i class="fa fa-crown text-warning"></i>' : ''}
                           ${role.is_system ? '<i class="fa fa-shield text-primary"></i>' : ''}
                       </div>
                   </div>
               </div>
           `;
           container.append(roleCard);
       });

       // Also populate dropdowns for copy permissions
       const sourceRole = $('#sourceRole');
       const targetRole = $('#targetRole');
       
       sourceRole.empty().append('<option value="">Select source role</option>');
       targetRole.empty().append('<option value="">Select target role</option>');
       
       roles.forEach(role => {
           sourceRole.append(`<option value="${role.id}">${role.name}</option>`);
           targetRole.append(`<option value="${role.id}">${role.name}</option>`);
       });
   }

   function selectRole(roleId) {
       // Highlight selected role
       $('.role-card').removeClass('selected');
       $(`.role-card[data-role-id="${roleId}"]`).addClass('selected');
       
       // Load permissions for this role
       loadPermissionAssignment(roleId);
   }

   function loadPermissionAssignment(roleId) {
       const role = roles.find(r => r.id == roleId);
       const container = $('#permissionAssignment');
       
       let html = `
           <div class="row" style="margin-bottom: 20px;">
               <div class="col-md-8">
                   <h4 style="margin-top: 0;">
                       <i class="fa fa-user-cog"></i>
                       Permissions for: <strong>${role.name}</strong>
                   </h4>
               </div>
               <div class="col-md-4 text-right">
                   <button class="btn btn-success" onclick="saveRolePermissions(${roleId})">
                       <i class="fa fa-save"></i> Save Permissions
                   </button>
               </div>
           </div>
       `;

       permissionGroups.forEach(group => {
           const groupCategories = permissionCategories.filter(cat => cat.perm_group_id == group.id);
           
           html += `
               <div class="permission-group">
                   <h5 class="text-primary">
                       <i class="fa fa-folder"></i> ${group.name}
                   </h5>
                   
                   <div class="table-responsive">
                       <table class="table table-bordered table-striped">
                           <thead>
                               <tr>
                                   <th>Permission</th>
                                   <th width="80" class="text-center">View</th>
                                   <th width="80" class="text-center">Add</th>
                                   <th width="80" class="text-center">Edit</th>
                                   <th width="80" class="text-center">Delete</th>
                               </tr>
                           </thead>
                           <tbody>
           `;
           
           groupCategories.forEach(category => {
               const rolePermission = rolePermissions.find(rp => rp.role_id == roleId && rp.perm_cat_id == category.id);
               
               html += `
                   <tr>
                       <td>
                           <strong>${category.name}</strong>
                           <br><small class="text-muted">${category.short_code}</small>
                       </td>
                       <td class="text-center">
                           ${category.enable_view ? `
                           <input type="checkbox" 
                                  data-role="${roleId}" 
                                  data-permission="${category.id}" 
                                  data-action="view"
                                  ${rolePermission && rolePermission.can_view ? 'checked' : ''}>
                           ` : '-'}
                       </td>
                       <td class="text-center">
                           ${category.enable_add ? `
                           <input type="checkbox" 
                                  data-role="${roleId}" 
                                  data-permission="${category.id}" 
                                  data-action="add"
                                  ${rolePermission && rolePermission.can_add ? 'checked' : ''}>
                           ` : '-'}
                       </td>
                       <td class="text-center">
                           ${category.enable_edit ? `
                           <input type="checkbox" 
                                  data-role="${roleId}" 
                                  data-permission="${category.id}" 
                                  data-action="edit"
                                  ${rolePermission && rolePermission.can_edit ? 'checked' : ''}>
                           ` : '-'}
                       </td>
                       <td class="text-center">
                           ${category.enable_delete ? `
                           <input type="checkbox" 
                                  data-role="${roleId}" 
                                  data-permission="${category.id}" 
                                  data-action="delete"
                                  ${rolePermission && rolePermission.can_delete ? 'checked' : ''}>
                           ` : '-'}
                       </td>
                   </tr>
               `;
           });
           
           html += `
                           </tbody>
                       </table>
                   </div>
               </div>
           `;
       });
       
       container.html(html);
   }

   // Form handling functions
   function handleAddRole(e) {
       e.preventDefault();
       
       const formData = new FormData();
       formData.append('name', $('#roleName').val());
       formData.append('slug', $('#roleSlug').val());
       formData.append('is_system', $('#isSystemRole').is(':checked') ? 1 : 0);
       formData.append('is_active', $('#isActive').is(':checked') ? 1 : 0);

       $.ajax({
           url: site_url + '/admin/roles/add_role',
           type: 'POST',
           data: formData,
           processData: false,
           contentType: false,
           dataType: 'json',
           success: function(data) {
               if (data.status === 'success') {
                   showAlert(data.message);
                   $('#addRoleForm')[0].reset();
                   setTimeout(function() {
                       location.reload();
                   }, 1500);
               } else {
                   showAlert(data.message, 'danger');
               }
           },
           error: function() {
               showAlert('Error adding role', 'danger');
           }
       });
   }

   function handleAddPermissionGroup(e) {
       e.preventDefault();
       
       const formData = new FormData();
       formData.append('name', $('#groupName').val());
       formData.append('short_code', $('#groupShortCode').val());
       formData.append('sort_order', $('#sortOrder').val());

       $.ajax({
           url: site_url + '/admin/roles/add_permission_group',
           type: 'POST',
           data: formData,
           processData: false,
           contentType: false,
           dataType: 'json',
           success: function(data) {
               if (data.status === 'success') {
                   showAlert(data.message);
                   $('#addPermissionGroupForm')[0].reset();
                   setTimeout(function() {
                       location.reload();
                   }, 1500);
               } else {
                   showAlert(data.message, 'danger');
               }
           },
           error: function() {
               showAlert('Error adding permission group', 'danger');
           }
       });
   }

   function handleAddPermission(e) {
       e.preventDefault();
       
       const formData = new FormData();
       formData.append('perm_group_id', $('#permissionGroup').val());
       formData.append('name', $('#permissionName').val());
       formData.append('short_code', $('#permissionShortCode').val());
       formData.append('enable_view', $('#enableView').is(':checked') ? 1 : 0);
       formData.append('enable_add', $('#enableAdd').is(':checked') ? 1 : 0);
       formData.append('enable_edit', $('#enableEdit').is(':checked') ? 1 : 0);
       formData.append('enable_delete', $('#enableDelete').is(':checked') ? 1 : 0);

       $.ajax({
           url: site_url + '/admin/roles/add_permission_category',
           type: 'POST',
           data: formData,
           processData: false,
           contentType: false,
           dataType: 'json',
           success: function(data) {
               if (data.status === 'success') {
                   showAlert(data.message);
                   $('#addPermissionForm')[0].reset();
                   setTimeout(function() {
                       location.reload();
                   }, 1500);
               } else {
                   showAlert(data.message, 'danger');
               }
           },
           error: function() {
               showAlert('Error adding permission', 'danger');
           }
       });
   }

   // Role management functions
   function editRole(roleId) {
       const role = roles.find(r => r.id == roleId);
       
       $('#editRoleId').val(role.id);
       $('#editRoleName').val(role.name);
       $('#editRoleSlug').val(role.slug || '');
       $('#editIsSystemRole').prop('checked', role.is_system == 1);
       $('#editIsActive').prop('checked', role.is_active == 1);
       
       $('#editRoleModal').modal('show');
   }

   function updateRole() {
       const roleId = $('#editRoleId').val();
       const formData = new FormData();
       formData.append('id', roleId);
       formData.append('name', $('#editRoleName').val());

       $.ajax({
           url: site_url + '/admin/roles/edit/' + roleId,
           type: 'POST',
           data: formData,
           processData: false,
           contentType: false,
           success: function() {
               showAlert('Role updated successfully!');
               setTimeout(function() {
                   location.reload();
               }, 1500);
               $('#editRoleModal').modal('hide');
           },
           error: function() {
               showAlert('Error updating role', 'danger');
           }
       });
   }

   function deleteRole(roleId) {
       const role = roles.find(r => r.id == roleId);
       
       if (role.is_system) {
           showAlert('Cannot delete system roles!', 'danger');
           return;
       }
       
       if (confirm(`Are you sure you want to delete the role "${role.name}"?`)) {
           $.ajax({
               url: site_url + '/admin/roles/delete/' + roleId,
               type: 'POST',
               dataType: 'json',
               success: function(data) {
                   if (data.status === 1) {
                       showAlert(data.msg);
                       setTimeout(function() {
                           location.reload();
                       }, 1500);
                   } else {
                       showAlert('Error deleting role', 'danger');
                   }
               },
               error: function() {
                   showAlert('Error deleting role', 'danger');
               }
           });
       }
   }

   function saveRolePermissions(roleId) {
       const checkboxes = $(`input[data-role="${roleId}"]`);
       const permissions = [];
       const permissionMap = {};
       
       checkboxes.each(function() {
           const permissionId = parseInt($(this).data('permission'));
           const action = $(this).data('action');
           
           if (!permissionMap[permissionId]) {
               permissionMap[permissionId] = {
                   role_id: roleId,
                   perm_cat_id: permissionId,
                   can_view: 0,
                   can_add: 0,
                   can_edit: 0,
                   can_delete: 0
               };
           }
           
           if ($(this).is(':checked')) {
               permissionMap[permissionId][`can_${action}`] = 1;
           }
       });
       
       // Only include permissions that have at least one action enabled
       Object.values(permissionMap).forEach(permission => {
           if (permission.can_view || permission.can_add || permission.can_edit || permission.can_delete) {
               permissions.push(permission);
           }
       });

       const formData = new FormData();
       formData.append('role_id', roleId);
       formData.append('permissions', JSON.stringify(permissions));

       $.ajax({
           url: site_url + '/admin/roles/save_role_permissions',
           type: 'POST',
           data: formData,
           processData: false,
           contentType: false,
           dataType: 'json',
           success: function(data) {
               if (data.status === 'success') {
                   showAlert(data.message);
                   loadRoleSelection(); // Update permission counts
               } else {
                   showAlert('Error saving permissions', 'danger');
               }
           },
           error: function() {
               showAlert('Error saving permissions', 'danger');
           }
       });
   }

   // Utility functions
   function copyPermissions() {
       $('#copyPermissionsModal').modal('show');
   }

   function executeCopyPermissions() {
       const sourceRoleId = parseInt($('#sourceRole').val());
       const targetRoleId = parseInt($('#targetRole').val());
       
       if (!sourceRoleId || !targetRoleId) {
           showAlert('Please select both source and target roles!', 'danger');
           return;
       }
       
       if (sourceRoleId === targetRoleId) {
           showAlert('Source and target roles cannot be the same!', 'danger');
           return;
       }
       
       const sourceRole = roles.find(r => r.id == sourceRoleId);
       const targetRole = roles.find(r => r.id == targetRoleId);
       
       showAlert(`Permissions copied from "${sourceRole.name}" to "${targetRole.name}" successfully!`);
       $('#copyPermissionsModal').modal('hide');
   }

   function bulkAssign() {
       showAlert('Bulk assign feature coming soon!', 'info');
   }

   function resetPermissions() {
       if (confirm('Are you sure you want to reset all permissions? This action cannot be undone!')) {
           showAlert('Reset permissions feature coming soon!', 'warning');
       }
   }

   function exportConfiguration() {
       const format = $('input[name="exportFormat"]:checked').val();
       const includeRoles = $('#includeRoles').is(':checked');
       const includePermissions = $('#includePermissions').is(':checked');
       const includeAssignments = $('#includeAssignments').is(':checked');
       
       let exportData = {};
       
       if (includeRoles) {
           exportData.roles = roles;
       }
       
       if (includePermissions) {
           exportData.permission_groups = permissionGroups;
           exportData.permission_categories = permissionCategories;
       }
       
       if (includeAssignments) {
           exportData.role_permissions = rolePermissions;
       }
       
       if (format === 'sql') {
           exportAsSql(exportData);
       } else {
           exportAsJson(exportData);
       }
   }
   
   function exportAsSql(data) {
       let sql = "-- Roles and Permissions Export\n";
       sql += "-- Generated on: " + new Date().toISOString() + "\n\n";
       
       if (data.roles) {
           sql += "-- Roles\n";
           data.roles.forEach(role => {
               sql += `INSERT INTO roles (id, name, slug, is_active, is_system, is_superadmin) VALUES (${role.id}, '${role.name}', '${role.slug || ''}', ${role.is_active}, ${role.is_system}, ${role.is_superadmin || 0});\n`;
           });
           sql += "\n";
       }
       
       if (data.permission_groups) {
           sql += "-- Permission Groups\n";
           data.permission_groups.forEach(group => {
               sql += `INSERT INTO permission_group (id, name, short_code, is_active, system, sort_order) VALUES (${group.id}, '${group.name}', '${group.short_code}', ${group.is_active}, 0, ${group.sort_order});\n`;
           });
           sql += "\n";
       }
       
       if (data.permission_categories) {
           sql += "-- Permission Categories\n";
           data.permission_categories.forEach(category => {
               sql += `INSERT INTO permission_category (id, perm_group_id, name, short_code, enable_view, enable_add, enable_edit, enable_delete) VALUES (${category.id}, ${category.perm_group_id}, '${category.name}', '${category.short_code}', ${category.enable_view}, ${category.enable_add}, ${category.enable_edit}, ${category.enable_delete});\n`;
           });
           sql += "\n";
       }
       
       if (data.role_permissions) {
           sql += "-- Role Permissions\n";
           data.role_permissions.forEach(rp => {
               sql += `INSERT INTO roles_permissions (id, role_id, perm_cat_id, can_view, can_add, can_edit, can_delete) VALUES (${rp.id}, ${rp.role_id}, ${rp.perm_cat_id}, ${rp.can_view}, ${rp.can_add}, ${rp.can_edit}, ${rp.can_delete});\n`;
           });
       }
       
       downloadFile(sql, 'roles_permissions_export.sql', 'text/plain');
   }
   
   function exportAsJson(data) {
       const jsonData = JSON.stringify(data, null, 2);
       downloadFile(jsonData, 'roles_permissions_export.json', 'application/json');
   }
   
   function downloadFile(content, filename, contentType) {
       const blob = new Blob([content], { type: contentType });
       const url = URL.createObjectURL(blob);
       const link = document.createElement('a');
       link.href = url;
       link.download = filename;
       document.body.appendChild(link);
       link.click();
       document.body.removeChild(link);
       URL.revokeObjectURL(url);
       
       showAlert(`Configuration exported as ${filename}!`);
   }
   
   function createBackup() {
       const backupData = {
           timestamp: new Date().toISOString(),
           roles: roles,
           permission_groups: permissionGroups,
           permission_categories: permissionCategories,
           role_permissions: rolePermissions
       };
       
       const jsonData = JSON.stringify(backupData, null, 2);
       const filename = `backup_${new Date().toISOString().slice(0, 19).replace(/:/g, '-')}.json`;
       downloadFile(jsonData, filename, 'application/json');
   }
   
   // Placeholder functions for features coming soon
   function editPermissionGroup(groupId) {
       showAlert('Edit permission group feature coming soon!', 'info');
   }
   
   function deletePermissionGroup(groupId) {
       showAlert('Delete permission group feature coming soon!', 'info');
   }
   
   function editPermission(categoryId) {
       showAlert('Edit permission feature coming soon!', 'info');
   }
   
   function deletePermission(categoryId) {
       showAlert('Delete permission feature coming soon!', 'info');
   }
</script>