<!DOCTYPE html>
<html>
<head>
    <title>Radiology Tests Management</title>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.11.5/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <div class="container-fluid mt-4">
        <!-- Header Section -->
        <div class="row mb-4">
            <div class="col-md-8">
                <h2><i class="fas fa-x-ray"></i> Radiology Tests Management</h2>
                <p class="text-muted">Manage radiology tests, categories, and parameters</p>
            </div>
            <div class="col-md-4 text-end">
                <button type="button" class="btn btn-primary" data-bs-toggle="modal" data-bs-target="#addRadiologyTestModal">
                    <i class="fas fa-plus"></i> Add New Test
                </button>
                <button type="button" class="btn btn-secondary" onclick="setupBasicData()">
                    <i class="fas fa-cog"></i> Setup Basic Data
                </button>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="row mb-4">
            <div class="col-md-3">
                <div class="card border-primary">
                    <div class="card-body text-center">
                        <i class="fas fa-vials fa-2x text-primary mb-2"></i>
                        <h5 class="card-title">Total Tests</h5>
                        <h3 class="text-primary" id="totalTests">0</h3>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card border-success">
                    <div class="card-body text-center">
                        <i class="fas fa-list fa-2x text-success mb-2"></i>
                        <h5 class="card-title">Categories</h5>
                        <h3 class="text-success" id="totalCategories">0</h3>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card border-info">
                    <div class="card-body text-center">
                        <i class="fas fa-sliders-h fa-2x text-info mb-2"></i>
                        <h5 class="card-title">Parameters</h5>
                        <h3 class="text-info" id="totalParameters">0</h3>
                    </div>
                </div>
            </div>
            <div class="col-md-3">
                <div class="card border-warning">
                    <div class="card-body text-center">
                        <i class="fas fa-money-bill fa-2x text-warning mb-2"></i>
                        <h5 class="card-title">Active Charges</h5>
                        <h3 class="text-warning" id="totalCharges">0</h3>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Table -->
        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-table"></i> Radiology Tests List</h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table id="radiologyTestsTable" class="table table-striped table-bordered" style="width:100%">
                        <thead>
                            <tr>
                                <th>Test Name</th>
                                <th>Short Name</th>
                                <th>Test Type</th>
                                <th>Category</th>
                                <th>Sub Category</th>
                                <th>Report Days</th>
                                <th>Standard Charge</th>
                                <th>Tax (%)</th>
                                <th>Total Amount</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <!-- Data will be loaded via AJAX -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Add/Edit Modal -->
    <div class="modal fade" id="addRadiologyTestModal" tabindex="-1" aria-labelledby="addRadiologyTestModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="addRadiologyTestModalLabel">Add New Radiology Test</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <form id="radiologyTestForm">
                    <div class="modal-body">
                        <input type="hidden" id="testId" name="testId">
                        
                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="testName" class="form-label">Test Name *</label>
                                    <input type="text" class="form-control" id="testName" name="test_name" required>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="shortName" class="form-label">Short Name *</label>
                                    <input type="text" class="form-control" id="shortName" name="short_name" required>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="testType" class="form-label">Test Type</label>
                                    <select class="form-select" id="testType" name="test_type">
                                        <option value="">Select Test Type</option>
                                        <option value="X-Ray">X-Ray</option>
                                        <option value="CT Scan">CT Scan</option>
                                        <option value="MRI">MRI</option>
                                        <option value="Ultrasound">Ultrasound</option>
                                        <option value="Mammography">Mammography</option>
                                        <option value="Fluoroscopy">Fluoroscopy</option>
                                        <option value="Nuclear Medicine">Nuclear Medicine</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="categoryId" class="form-label">Category *</label>
                                    <select class="form-select" id="categoryId" name="radiology_category_id" required>
                                        <option value="">Select Category</option>
                                    </select>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="subCategory" class="form-label">Sub Category</label>
                                    <input type="text" class="form-control" id="subCategory" name="sub_category">
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="reportDays" class="form-label">Report Days *</label>
                                    <input type="number" class="form-control" id="reportDays" name="report_days" min="0" required>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="chargeCategoryId" class="form-label">Charge Category *</label>
                                    <select class="form-select" id="chargeCategoryId" name="charge_category_id" required>
                                        <option value="">Select Charge Category</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="mb-3">
                                    <label for="chargeCode" class="form-label">Charge Code *</label>
                                    <input type="text" class="form-control" id="chargeCode" name="code" required>
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="standardCharge" class="form-label">Standard Charge *</label>
                                    <input type="number" class="form-control" id="standardCharge" name="standard_charge" step="0.01" min="0" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="tax" class="form-label">Tax (%) *</label>
                                    <input type="number" class="form-control" id="tax" name="tax" step="0.01" min="0" max="100" required>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="mb-3">
                                    <label for="totalAmount" class="form-label">Total Amount</label>
                                    <input type="number" class="form-control" id="totalAmount" name="amount" step="0.01" readonly>
                                </div>
                            </div>
                        </div>

                        <!-- Parameters Section -->
                        <div class="mb-3">
                            <label class="form-label">Test Parameters</label>
                            <div id="parametersContainer">
                                <div class="parameter-row mb-2">
                                    <div class="row">
                                        <div class="col-md-4">
                                            <select class="form-select parameter-select" name="parameter_name[]">
                                                <option value="">Select Parameter</option>
                                            </select>
                                        </div>
                                        <div class="col-md-3">
                                            <input type="text" class="form-control" name="reference_range[]" placeholder="Reference Range">
                                        </div>
                                        <div class="col-md-3">
                                            <input type="text" class="form-control" name="radio_unit[]" placeholder="Unit">
                                        </div>
                                        <div class="col-md-2">
                                            <button type="button" class="btn btn-success btn-sm" onclick="addParameterRow()">
                                                <i class="fas fa-plus"></i>
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div class="modal-footer">
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                        <button type="submit" class="btn btn-primary">Save Test</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Setup Modal -->
    <div class="modal fade" id="setupModal" tabindex="-1" aria-labelledby="setupModalLabel" aria-hidden="true">
        <div class="modal-dialog modal-lg">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="setupModalLabel">Setup Basic Data</h5>
                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                </div>
                <div class="modal-body">
                    <div class="alert alert-info">
                        <i class="fas fa-info-circle"></i> This will create basic lab categories, charge categories, and sample data required for the radiology system.
                    </div>
                    <div id="setupProgress" class="d-none">
                        <div class="progress mb-3">
                            <div class="progress-bar" role="progressbar" style="width: 0%"></div>
                        </div>
                        <div id="setupLog"></div>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" onclick="runSetup()">Run Setup</button>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.1.3/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.11.5/js/dataTables.bootstrap5.min.js"></script>

    <script>
        let radiologyTable;
        const baseUrl = window.location.origin + '/';

        $(document).ready(function() {
            initializeDataTable();
            loadCategories();
            loadChargeCategories();
            loadParameters();
            updateStatistics();
            
            // Calculate total amount when charges change
            $('#standardCharge, #tax').on('input', calculateTotalAmount);
        });

        function initializeDataTable() {
            radiologyTable = $('#radiologyTestsTable').DataTable({
                processing: true,
                serverSide: true,
                ajax: {
                    url: baseUrl + 'admin/radio/getradiologytestDatatable',
                    type: 'POST'
                },
                columns: [
                    { data: 'test_name' },
                    { data: 'short_name' },
                    { data: 'test_type' },
                    { data: 'lab_name' },
                    { data: 'sub_category' },
                    { data: 'report_days' },
                    { 
                        data: 'standard_charge',
                        render: function(data) {
                            return '$' + parseFloat(data).toFixed(2);
                        }
                    },
                    { data: 'percentage' },
                    { 
                        data: null,
                        render: function(row) {
                            const total = parseFloat(row.standard_charge) + (parseFloat(row.standard_charge) * parseFloat(row.percentage) / 100);
                            return '$' + total.toFixed(2);
                        }
                    },
                    {
                        data: null,
                        orderable: false,
                        render: function(row) {
                            return `
                                <div class="btn-group" role="group">
                                    <button type="button" class="btn btn-sm btn-outline-primary" onclick="viewTest(${row.id})" title="View">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-warning" onclick="editTest(${row.id})" title="Edit">
                                        <i class="fas fa-edit"></i>
                                    </button>
                                    <button type="button" class="btn btn-sm btn-outline-danger" onclick="deleteTest(${row.id})" title="Delete">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            `;
                        }
                    }
                ],
                pageLength: 25,
                responsive: true,
                language: {
                    emptyTable: "No radiology tests found. Click 'Add New Test' to create your first test."
                }
            });
        }

        function loadCategories() {
            $.get(baseUrl + 'admin/radio/getCategories', function(data) {
                $('#categoryId').empty().append('<option value="">Select Category</option>');
                if (data.categories && data.categories.length > 0) {
                    data.categories.forEach(function(category) {
                        $('#categoryId').append(`<option value="${category.id}">${category.lab_name}</option>`);
                    });
                }
            });
        }

        function loadChargeCategories() {
            $.get(baseUrl + 'admin/radio/getChargeCategories', function(data) {
                $('#chargeCategoryId').empty().append('<option value="">Select Charge Category</option>');
                if (data.categories && data.categories.length > 0) {
                    data.categories.forEach(function(category) {
                        $('#chargeCategoryId').append(`<option value="${category.id}">${category.name}</option>`);
                    });
                }
            });
        }

        function loadParameters() {
            $.get(baseUrl + 'admin/radio/getParameters', function(data) {
                $('.parameter-select').empty().append('<option value="">Select Parameter</option>');
                if (data.parameters && data.parameters.length > 0) {
                    data.parameters.forEach(function(parameter) {
                        $('.parameter-select').append(`<option value="${parameter.id}">${parameter.parameter_name}</option>`);
                    });
                }
            });
        }

        function calculateTotalAmount() {
            const charge = parseFloat($('#standardCharge').val()) || 0;
            const tax = parseFloat($('#tax').val()) || 0;
            const total = charge + (charge * tax / 100);
            $('#totalAmount').val(total.toFixed(2));
        }

        function addParameterRow() {
            const newRow = `
                <div class="parameter-row mb-2">
                    <div class="row">
                        <div class="col-md-4">
                            <select class="form-select parameter-select" name="parameter_name[]">
                                <option value="">Select Parameter</option>
                            </select>
                        </div>
                        <div class="col-md-3">
                            <input type="text" class="form-control" name="reference_range[]" placeholder="Reference Range">
                        </div>
                        <div class="col-md-3">
                            <input type="text" class="form-control" name="radio_unit[]" placeholder="Unit">
                        </div>
                        <div class="col-md-2">
                            <button type="button" class="btn btn-danger btn-sm" onclick="removeParameterRow(this)">
                                <i class="fas fa-minus"></i>
                            </button>
                        </div>
                    </div>
                </div>
            `;
            $('#parametersContainer').append(newRow);
            loadParameters(); // Reload parameters for new row
        }

        function removeParameterRow(button) {
            $(button).closest('.parameter-row').remove();
        }

        function setupBasicData() {
            $('#setupModal').modal('show');
        }

        function runSetup() {
            $('#setupProgress').removeClass('d-none');
            const progressBar = $('.progress-bar');
            const log = $('#setupLog');
            
            // Simulate setup process
            let progress = 0;
            const interval = setInterval(function() {
                progress += 20;
                progressBar.css('width', progress + '%');
                
                if (progress === 20) {
                    log.append('<p>✓ Creating lab categories...</p>');
                } else if (progress === 40) {
                    log.append('<p>✓ Setting up charge categories...</p>');
                } else if (progress === 60) {
                    log.append('<p>✓ Creating sample charges...</p>');
                } else if (progress === 80) {
                    log.append('<p>✓ Setting up parameters...</p>');
                } else if (progress === 100) {
                    log.append('<p>✓ Setup completed successfully!</p>');
                    clearInterval(interval);
                    setTimeout(function() {
                        $('#setupModal').modal('hide');
                        location.reload();
                    }, 2000);
                }
            }, 1000);
        }

        function updateStatistics() {
            // Update statistics cards
            $.get(baseUrl + 'admin/radio/getStatistics', function(data) {
                $('#totalTests').text(data.total_tests || 0);
                $('#totalCategories').text(data.total_categories || 0);
                $('#totalParameters').text(data.total_parameters || 0);
                $('#totalCharges').text(data.total_charges || 0);
            });
        }

        function viewTest(id) {
            // Implementation for viewing test details
            alert('View test functionality - ID: ' + id);
        }

        function editTest(id) {
            // Implementation for editing test
            alert('Edit test functionality - ID: ' + id);
        }

        function deleteTest(id) {
            if (confirm('Are you sure you want to delete this test?')) {
                // Implementation for deleting test
                alert('Delete test functionality - ID: ' + id);
            }
        }

        // Form submission
        $('#radiologyTestForm').on('submit', function(e) {
            e.preventDefault();
            
            const formData = $(this).serialize();
            
            $.post(baseUrl + 'admin/radio/add', formData, function(response) {
                if (response.status === 'success') {
                    $('#addRadiologyTestModal').modal('hide');
                    radiologyTable.ajax.reload();
                    updateStatistics();
                    alert('Test saved successfully!');
                } else {
                    alert('Error saving test: ' + (response.message || 'Unknown error'));
                }
            }, 'json');
        });
    </script>
</body>
</html>