<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo ucfirst($department); ?> Queue Display</title>
    
    <!-- Bootstrap CSS -->
    <link rel="stylesheet" href="https://maxcdn.bootstrapcdn.com/bootstrap/3.3.7/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">
    
    <style>
        body {
            font-family: 'Arial', sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            margin: 0;
            padding: 20px;
            color: white;
            overflow-x: hidden;
        }

        .display-header {
            text-align: center;
            margin-bottom: 30px;
            padding: 20px;
            background: rgba(255, 255, 255, 0.1);
            border-radius: 15px;
            backdrop-filter: blur(10px);
        }

        .display-header h1 {
            font-size: 3.5em;
            margin: 0;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.3);
        }

        .display-header .subtitle {
            font-size: 1.5em;
            opacity: 0.9;
            margin-top: 10px;
        }

        .current-serving {
            background: rgba(255, 255, 255, 0.15);
            border-radius: 20px;
            padding: 40px;
            margin-bottom: 30px;
            text-align: center;
            backdrop-filter: blur(10px);
            border: 2px solid rgba(255, 255, 255, 0.2);
        }

        .current-serving h2 {
            font-size: 2.5em;
            margin-bottom: 20px;
            color: #fff;
        }

        .current-number {
            font-size: 6em;
            font-weight: bold;
            color: #FFD700;
            text-shadow: 3px 3px 6px rgba(0,0,0,0.4);
            display: block;
            animation: pulse 2s ease-in-out infinite alternate;
        }

        @keyframes pulse {
            from { transform: scale(1); }
            to { transform: scale(1.05); }
        }

        .patient-info {
            font-size: 1.8em;
            margin-top: 20px;
            opacity: 0.9;
        }

        .waiting-queue {
            background: rgba(255, 255, 255, 0.1);
            border-radius: 15px;
            padding: 30px;
            backdrop-filter: blur(10px);
        }

        .waiting-queue h3 {
            font-size: 2.2em;
            margin-bottom: 25px;
            text-align: center;
            color: #fff;
        }

        .queue-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-top: 20px;
        }

        .queue-item {
            background: rgba(255, 255, 255, 0.2);
            border-radius: 12px;
            padding: 20px;
            text-align: center;
            border: 1px solid rgba(255, 255, 255, 0.3);
            transition: all 0.3s ease;
        }

        .queue-item:hover {
            transform: translateY(-5px);
            background: rgba(255, 255, 255, 0.25);
        }

        .queue-number {
            font-size: 2.5em;
            font-weight: bold;
            color: #FFD700;
            margin-bottom: 10px;
        }

        .queue-patient {
            font-size: 1.3em;
            margin-bottom: 5px;
        }

        .queue-wait-time {
            font-size: 1.1em;
            opacity: 0.8;
        }

        .priority-urgent {
            border-left: 5px solid #dc3545;
            background: rgba(220, 53, 69, 0.2);
        }

        .priority-high {
            border-left: 5px solid #ffc107;
            background: rgba(255, 193, 7, 0.2);
        }

        .priority-normal {
            border-left: 5px solid #17a2b8;
            background: rgba(23, 162, 184, 0.2);
        }

        .priority-low {
            border-left: 5px solid #6c757d;
            background: rgba(108, 117, 125, 0.2);
        }

        .stats-row {
            display: flex;
            justify-content: space-around;
            margin: 30px 0;
            flex-wrap: wrap;
        }

        .stat-box {
            background: rgba(255, 255, 255, 0.15);
            border-radius: 15px;
            padding: 25px;
            text-align: center;
            min-width: 200px;
            margin: 10px;
            backdrop-filter: blur(10px);
        }

        .stat-number {
            font-size: 3em;
            font-weight: bold;
            color: #FFD700;
            display: block;
        }

        .stat-label {
            font-size: 1.2em;
            opacity: 0.9;
            margin-top: 10px;
        }

        .no-queue {
            text-align: center;
            padding: 60px;
            font-size: 1.8em;
            opacity: 0.7;
        }

        .last-updated {
            position: fixed;
            bottom: 20px;
            right: 20px;
            background: rgba(0, 0, 0, 0.5);
            padding: 10px 15px;
            border-radius: 8px;
            font-size: 0.9em;
        }

        .marquee {
            background: rgba(255, 255, 255, 0.1);
            padding: 15px;
            margin: 20px 0;
            border-radius: 10px;
            overflow: hidden;
            white-space: nowrap;
        }

        .marquee-content {
            display: inline-block;
            animation: scroll 30s linear infinite;
        }

        @keyframes scroll {
            0% { transform: translate3d(100%, 0, 0); }
            100% { transform: translate3d(-100%, 0, 0); }
        }

        .emergency-alert {
            background: #dc3545;
            color: white;
            padding: 15px;
            border-radius: 10px;
            margin: 20px 0;
            text-align: center;
            font-size: 1.5em;
            animation: blink 1s ease-in-out infinite alternate;
        }

        @keyframes blink {
            from { opacity: 1; }
            to { opacity: 0.7; }
        }

        @media (max-width: 768px) {
            .display-header h1 { font-size: 2.5em; }
            .current-number { font-size: 4em; }
            .queue-grid { grid-template-columns: repeat(auto-fit, minmax(200px, 1fr)); }
            .stats-row { flex-direction: column; align-items: center; }
        }
    </style>
</head>
<body>
    <div class="container-fluid">
        <!-- Header -->
        <div class="display-header">
            <h1>
                <i class="fa fa-hospital-o"></i> 
                <?php echo ucfirst($department); ?> Department
            </h1>
            <div class="subtitle">Queue Management System</div>
            <div class="marquee">
                <div class="marquee-content">
                    Welcome to <?php echo ucfirst($department); ?> Department • Please wait for your number to be called • Thank you for your patience
                </div>
            </div>
        </div>

        <!-- Emergency Alert (if any urgent patients) -->
        <div id="emergency-alert" class="emergency-alert" style="display: none;">
            <i class="fa fa-exclamation-triangle"></i> 
            URGENT PATIENTS IN QUEUE - PRIORITY SERVICE REQUIRED
        </div>

        <!-- Current Serving Section -->
        <div class="current-serving">
            <h2><i class="fa fa-user"></i> Now Serving</h2>
            <div id="current-serving-content">
                <span class="current-number" id="current-number">-</span>
                <div class="patient-info" id="current-patient-info">
                    Please wait for the next call
                </div>
            </div>
        </div>

        <!-- Statistics Row -->
        <div class="stats-row" id="stats-row">
            <div class="stat-box">
                <span class="stat-number" id="waiting-count">0</span>
                <div class="stat-label">
                    <i class="fa fa-clock-o"></i> Waiting
                </div>
            </div>
            <div class="stat-box">
                <span class="stat-number" id="served-today">0</span>
                <div class="stat-label">
                    <i class="fa fa-check"></i> Served Today
                </div>
            </div>
            <div class="stat-box">
                <span class="stat-number" id="avg-wait">0</span>
                <div class="stat-label">
                    <i class="fa fa-clock-o"></i> Avg Wait (min)
                </div>
            </div>
        </div>

        <!-- Waiting Queue -->
        <div class="waiting-queue">
            <h3><i class="fa fa-list"></i> Next in Queue</h3>
            <div id="queue-container">
                <div class="queue-grid" id="queue-grid">
                    <!-- Queue items will be loaded here -->
                </div>
                <div class="no-queue" id="no-queue" style="display: none;">
                    <i class="fa fa-check-circle fa-3x"></i>
                    <br><br>
                    No patients waiting in queue
                </div>
            </div>
        </div>

        <!-- Last Updated -->
        <div class="last-updated" id="last-updated">
            Last updated: <span id="update-time">-</span>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://ajax.googleapis.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script>
        const DEPARTMENT = '<?php echo $department; ?>';
        const REFRESH_INTERVAL = <?php echo isset($refresh_interval) ? $refresh_interval * 1000 : 30000; ?>;
        
        let isFirstLoad = true;
        let previousServingNumber = null;

        $(document).ready(function() {
            loadQueueDisplay();
            
            // Auto-refresh
            setInterval(loadQueueDisplay, REFRESH_INTERVAL);
            
            // Update time every second
            setInterval(updateCurrentTime, 1000);
            
            // Initial time update
            updateCurrentTime();
        });

        function loadQueueDisplay() {
            $.ajax({
                url: '<?php echo base_url("admin/queue/get_queue_status"); ?>',
                method: 'GET',
                data: { department: DEPARTMENT },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        updateCurrentServing(response.current_serving);
                        updateWaitingQueue(response.queue_list);
                        updateStatistics(response.statistics);
                        updateLastUpdated();
                        
                        // Check for emergency patients
                        checkEmergencyPatients(response.queue_list);
                        
                        isFirstLoad = false;
                    }
                },
                error: function() {
                    console.error('Failed to load queue data');
                    $('#current-number').text('ERROR');
                    $('#current-patient-info').text('Connection failed - please contact IT support');
                }
            });
        }

        function updateCurrentServing(servingPatient) {
            if (servingPatient) {
                const queueNumber = servingPatient.queue_number;
                $('#current-number').text(queueNumber);
                $('#current-patient-info').html(`
                    <strong>${servingPatient.patient_name || 'Patient'}</strong>
                    <br>Please proceed to the service counter
                `);

                // Play announcement sound if number changed
                if (!isFirstLoad && previousServingNumber !== queueNumber) {
                    playAnnouncementSound();
                    announceNumber(queueNumber);
                }
                
                previousServingNumber = queueNumber;
            } else {
                $('#current-number').text('-');
                $('#current-patient-info').text('Please wait for the next call');
                previousServingNumber = null;
            }
        }

        function updateWaitingQueue(queueList) {
            const queueGrid = $('#queue-grid');
            const noQueue = $('#no-queue');
            
            queueGrid.empty();
            
            if (queueList && queueList.length > 0) {
                noQueue.hide();
                
                // Show next 12 patients in queue
                const displayCount = Math.min(queueList.length, 12);
                
                for (let i = 0; i < displayCount; i++) {
                    const patient = queueList[i];
                    if (patient.status === 'waiting') {
                        const waitTime = calculateWaitTime(patient.created_at);
                        const priorityClass = `priority-${patient.priority}`;
                        
                        const queueItem = `
                            <div class="queue-item ${priorityClass}">
                                <div class="queue-number">${patient.queue_number}</div>
                                <div class="queue-patient">${patient.patient_name || 'Patient'}</div>
                                <div class="queue-wait-time">
                                    <i class="fa fa-clock-o"></i> ${waitTime}
                                </div>
                            </div>
                        `;
                        
                        queueGrid.append(queueItem);
                    }
                }
                
                if (queueGrid.children().length === 0) {
                    noQueue.show();
                }
            } else {
                noQueue.show();
            }
        }

        function updateStatistics(stats) {
            if (stats) {
                $('#waiting-count').text(stats.waiting_count || 0);
                $('#served-today').text(stats.completed_today || 0);
                $('#avg-wait').text(Math.round(stats.avg_wait_time || 0));
            }
        }

        function checkEmergencyPatients(queueList) {
            const hasUrgent = queueList && queueList.some(p => p.priority === 'urgent' && p.status === 'waiting');
            
            if (hasUrgent) {
                $('#emergency-alert').show();
            } else {
                $('#emergency-alert').hide();
            }
        }

        function updateLastUpdated() {
            const now = new Date();
            const timeString = now.toLocaleTimeString();
            $('#update-time').text(timeString);
        }

        function updateCurrentTime() {
            const now = new Date();
            const timeString = now.toLocaleString();
            // You can add a clock display if needed
        }

        function calculateWaitTime(createdAt) {
            const created = new Date(createdAt);
            const now = new Date();
            const diffMinutes = Math.floor((now - created) / 60000);
            
            if (diffMinutes < 60) {
                return diffMinutes + ' min';
            } else {
                const hours = Math.floor(diffMinutes / 60);
                const minutes = diffMinutes % 60;
                return hours + 'h ' + minutes + 'm';
            }
        }

        function playAnnouncementSound() {
            try {
                // Create a pleasant notification sound
                const audioContext = new (window.AudioContext || window.webkitAudioContext)();
                const oscillator = audioContext.createOscillator();
                const gainNode = audioContext.createGain();
                
                oscillator.connect(gainNode);
                gainNode.connect(audioContext.destination);
                
                oscillator.frequency.setValueAtTime(800, audioContext.currentTime);
                oscillator.frequency.setValueAtTime(600, audioContext.currentTime + 0.1);
                oscillator.frequency.setValueAtTime(800, audioContext.currentTime + 0.2);
                
                gainNode.gain.setValueAtTime(0.3, audioContext.currentTime);
                gainNode.gain.exponentialRampToValueAtTime(0.01, audioContext.currentTime + 0.3);
                
                oscillator.start(audioContext.currentTime);
                oscillator.stop(audioContext.currentTime + 0.3);
            } catch (e) {
                console.log('Audio not supported');
            }
        }

        function announceNumber(queueNumber) {
            // Text-to-speech announcement (if supported)
            if ('speechSynthesis' in window) {
                const utterance = new SpeechSynthesisUtterance(`Now serving number ${queueNumber}`);
                utterance.rate = 0.8;
                utterance.volume = 0.8;
                speechSynthesis.speak(utterance);
            }
        }

        // Handle fullscreen for display screens
        function toggleFullscreen() {
            if (!document.fullscreenElement) {
                document.documentElement.requestFullscreen();
            } else {
                document.exitFullscreen();
            }
        }

        // Double-click to toggle fullscreen
        $(document).dblclick(function() {
            toggleFullscreen();
        });

        // Prevent context menu for kiosk mode
        $(document).contextmenu(function(e) {
            e.preventDefault();
        });

        // Auto-fullscreen on certain screens
        $(window).on('load', function() {
            // Uncomment the next line for automatic fullscreen on large displays
            // if (window.innerWidth > 1200) toggleFullscreen();
        });
    </script>
</body>
</html>