<?php
/**
 * Queue Configuration List View
 * File: application/views/admin/queue/config_list.php
 */
?>

<div class="content-wrapper">
    <section class="content-header">
        <h1>
            <i class="fa fa-cogs"></i> <?php echo $this->lang->line('queue_configuration'); ?>
            <small><?php echo $this->lang->line('manage_queue_settings'); ?></small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/dashboard"><i class="fa fa-dashboard"></i> <?php echo $this->lang->line('dashboard'); ?></a></li>
            <li><a href="<?php echo base_url(); ?>admin/queue"><?php echo $this->lang->line('queue_management'); ?></a></li>
            <li class="active"><?php echo $this->lang->line('configuration'); ?></li>
        </ol>
    </section>

    <section class="content">
        <div class="row">
            <!-- Queue Settings Overview -->
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-list"></i> <?php echo $this->lang->line('department_configurations'); ?></h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-success btn-sm" onclick="resetAllCounters()">
                                <i class="fa fa-refresh"></i> Reset All Counters
                            </button>
                            <button type="button" class="btn btn-warning btn-sm" onclick="clearOldQueues()">
                                <i class="fa fa-trash"></i> Clear Old Queues
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <div class="table-responsive">
                            <table class="table table-striped table-bordered table-hover" id="queue-config-table">
                                <thead>
                                    <tr>
                                        <th><?php echo $this->lang->line('department'); ?></th>
                                        <th><?php echo $this->lang->line('queue_prefix'); ?></th>
                                        <th><?php echo $this->lang->line('current_counter'); ?></th>
                                        <th><?php echo $this->lang->line('avg_service_time'); ?></th>
                                        <th><?php echo $this->lang->line('max_queue_size'); ?></th>
                                        <th><?php echo $this->lang->line('working_hours'); ?></th>
                                        <th><?php echo $this->lang->line('status'); ?></th>
                                        <th class="text-right"><?php echo $this->lang->line('action'); ?></th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php 
                                    $default_departments = [
                                        'registration' => 'Registration Counter',
                                        'triage' => 'Triage Department', 
                                        'consultation' => 'Consultation Rooms',
                                        'laboratory' => 'Laboratory Services',
                                        'radiology' => 'Radiology Department',
                                        'pharmacy' => 'Pharmacy Counter',
                                        'phlebotomy' => 'Phlebotomy',
                                        'emergency' => 'Emergency Department',
                                        'ipd' => 'In-Patient Department'
                                    ];
                                    
                                    foreach ($default_departments as $dept_code => $dept_name): 
                                        // Check if configuration exists for this department
                                        $config = null;
                                        if (isset($departments) && is_array($departments)) {
                                            foreach ($departments as $dept_data) {
                                                if (isset($dept_data['department']) && $dept_data['department'] == $dept_code) {
                                                    $config = $dept_data;
                                                    break;
                                                }
                                            }
                                        }
                                        
                                        // Set default values if no config exists
                                        $queue_prefix = $config['queue_prefix'] ?? strtoupper(substr($dept_code, 0, 3));
                                        $counter_current = $config['counter_current'] ?? 1;
                                        $avg_service_time = $config['average_service_time'] ?? 15;
                                        $max_queue_size = $config['max_queue_size'] ?? 50;
                                        $working_hours_start = $config['working_hours_start'] ?? '08:00';
                                        $working_hours_end = $config['working_hours_end'] ?? '17:00';
                                        $is_active = $config['is_active'] ?? 1;
                                    ?>
                                    <tr data-department="<?php echo $dept_code; ?>">
                                        <td>
                                            <strong><?php echo $dept_name; ?></strong>
                                            <br><small class="text-muted"><?php echo ucfirst($dept_code); ?></small>
                                        </td>
                                        <td>
                                            <span class="label label-default"><?php echo $queue_prefix; ?></span>
                                        </td>
                                        <td>
                                            <span class="label label-info"><?php echo $counter_current; ?></span>
                                        </td>
                                        <td>
                                            <?php echo $avg_service_time; ?> min
                                        </td>
                                        <td>
                                            <?php echo $max_queue_size; ?>
                                        </td>
                                        <td>
                                            <small><?php echo $working_hours_start; ?> - <?php echo $working_hours_end; ?></small>
                                        </td>
                                        <td>
                                            <?php if ($is_active): ?>
                                                <span class="label label-success">Active</span>
                                            <?php else: ?>
                                                <span class="label label-danger">Inactive</span>
                                            <?php endif; ?>
                                        </td>
                                        <td class="text-right">
                                            <div class="btn-group">
                                                <a href="<?php echo site_url('admin/queue/config/' . $dept_code); ?>" 
                                                   class="btn btn-primary btn-xs" title="Configure">
                                                    <i class="fa fa-cog"></i>
                                                </a>
                                                <a href="<?php echo site_url('admin/queue/department/' . $dept_code); ?>" 
                                                   class="btn btn-info btn-xs" title="View Queue">
                                                    <i class="fa fa-eye"></i>
                                                </a>
                                                <button type="button" class="btn btn-warning btn-xs" 
                                                        onclick="resetCounter('<?php echo $dept_code; ?>')" title="Reset Counter">
                                                    <i class="fa fa-refresh"></i>
                                                </button>
                                            </div>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Actions Row -->
        <div class="row">
            <div class="col-md-6">
                <div class="box box-warning">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-wrench"></i> Queue Maintenance</h3>
                    </div>
                    <div class="box-body">
                        <div class="form-group">
                            <label>Clear Completed Queues Older Than:</label>
                            <div class="input-group">
                                <input type="number" class="form-control" id="clear-days" value="7" min="1" max="365">
                                <span class="input-group-addon">days</span>
                                <span class="input-group-btn">
                                    <button type="button" class="btn btn-warning" onclick="clearOldQueues()">
                                        <i class="fa fa-trash"></i> Clear
                                    </button>
                                </span>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <button type="button" class="btn btn-success btn-block" onclick="resetAllCounters()">
                                <i class="fa fa-refresh"></i> Reset All Department Counters
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <div class="col-md-6">
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-download"></i> Export & Reports</h3>
                    </div>
                    <div class="box-body">
                        <div class="form-group">
                            <label>Export Queue Data:</label>
                            <div class="row">
                                <div class="col-md-6">
                                    <input type="date" class="form-control" id="export-date-from" 
                                           value="<?php echo date('Y-m-d', strtotime('-30 days')); ?>">
                                </div>
                                <div class="col-md-6">
                                    <input type="date" class="form-control" id="export-date-to" 
                                           value="<?php echo date('Y-m-d'); ?>">
                                </div>
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <select class="form-control" id="export-department">
                                <option value="">All Departments</option>
                                <?php foreach ($default_departments as $code => $name): ?>
                                    <option value="<?php echo $code; ?>"><?php echo $name; ?></option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        
                        <button type="button" class="btn btn-info btn-block" onclick="exportQueueData()">
                            <i class="fa fa-download"></i> Download CSV Report
                        </button>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Modals -->
<div class="modal fade" id="confirm-modal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">Confirm Action</h4>
            </div>
            <div class="modal-body">
                <p id="confirm-message">Are you sure you want to proceed?</p>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="confirm-button">Confirm</button>
            </div>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Initialize DataTable
    $('#queue-config-table').DataTable({
        "paging": false,
        "searching": true,
        "ordering": true,
        "info": false,
        "responsive": true,
        "columnDefs": [
            { "orderable": false, "targets": -1 }
        ]
    });
});

/**
 * Reset counter for specific department
 */
function resetCounter(department) {
    showConfirmModal(
        'Reset Counter for ' + department.charAt(0).toUpperCase() + department.slice(1) + '?',
        'This will reset the queue counter to 1. Continue?',
        function() {
            $.ajax({
                url: '<?php echo site_url('admin/queue/reset_counters'); ?>',
                type: 'POST',
                data: { department: department },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        showSuccessToast(response.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        showErrorToast(response.message);
                    }
                },
                error: function() {
                    showErrorToast('Failed to reset counter');
                }
            });
        }
    );
}

/**
 * Reset all department counters
 */
function resetAllCounters() {
    showConfirmModal(
        'Reset All Counters?',
        'This will reset queue counters for ALL departments to 1. This action cannot be undone.',
        function() {
            $.ajax({
                url: '<?php echo site_url('admin/queue/reset_counters'); ?>',
                type: 'POST',
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        showSuccessToast(response.message);
                        setTimeout(function() {
                            location.reload();
                        }, 1500);
                    } else {
                        showErrorToast(response.message);
                    }
                },
                error: function() {
                    showErrorToast('Failed to reset counters');
                }
            });
        }
    );
}

/**
 * Clear old completed queues
 */
function clearOldQueues() {
    const days = $('#clear-days').val() || 7;
    
    showConfirmModal(
        'Clear Old Queues?',
        `This will permanently delete all completed queue records older than ${days} days. Continue?`,
        function() {
            $.ajax({
                url: '<?php echo site_url('admin/queue/clear_old_queues'); ?>',
                type: 'POST',
                data: { days_old: days },
                dataType: 'json',
                success: function(response) {
                    if (response.status === 'success') {
                        showSuccessToast(response.message);
                    } else {
                        showErrorToast(response.message);
                    }
                },
                error: function() {
                    showErrorToast('Failed to clear old queues');
                }
            });
        }
    );
}

/**
 * Export queue data
 */
function exportQueueData() {
    const dateFrom = $('#export-date-from').val();
    const dateTo = $('#export-date-to').val();
    const department = $('#export-department').val();
    
    if (!dateFrom || !dateTo) {
        showErrorToast('Please select date range');
        return;
    }
    
    const params = new URLSearchParams({
        date_from: dateFrom,
        date_to: dateTo
    });
    
    if (department) {
        params.append('department', department);
    }
    
    window.open('<?php echo site_url('admin/queue/export_queue_data'); ?>?' + params.toString(), '_blank');
}

/**
 * Show confirmation modal
 */
function showConfirmModal(title, message, callback) {
    $('#confirm-modal .modal-title').text(title);
    $('#confirm-message').text(message);
    $('#confirm-modal').modal('show');
    
    $('#confirm-button').off('click').on('click', function() {
        $('#confirm-modal').modal('hide');
        callback();
    });
}

/**
 * Toast notifications
 */
function showSuccessToast(message) {
    toastr.success(message);
}

function showErrorToast(message) {
    toastr.error(message);
}

// Initialize toastr if not already done
if (typeof toastr !== 'undefined') {
    toastr.options = {
        "closeButton": true,
        "debug": false,
        "newestOnTop": true,
        "progressBar": true,
        "positionClass": "toast-top-right",
        "preventDuplicates": false,
        "onclick": null,
        "showDuration": "300",
        "hideDuration": "1000",
        "timeOut": "5000",
        "extendedTimeOut": "1000",
        "showEasing": "swing",
        "hideEasing": "linear",
        "showMethod": "fadeIn",
        "hideMethod": "fadeOut"
    };
}
</script>

<style>
.table > tbody > tr > td {
    vertical-align: middle;
}

.btn-group .btn {
    margin-right: 2px;
}

.btn-group .btn:last-child {
    margin-right: 0;
}

.label {
    font-size: 12px;
}

.box-tools .btn {
    margin-left: 5px;
}

@media (max-width: 768px) {
    .btn-group {
        display: flex;
        flex-direction: column;
    }
    
    .btn-group .btn {
        margin-bottom: 2px;
        margin-right: 0;
    }
}
</style>