<?php
/**
 * Enhanced Lab Reports Management Page - Modified Version
 * Path: application/views/admin/lab_reports/index.php
 * 
 * This preserves the original structure and adds structured test results functionality
 */
?>

<div class="content-wrapper">
    <!-- Content Header -->
    <section class="content-header">
        <h1>
            <i class="fa fa-flask"></i> Lab Reports Management
            <small>Upload and manage patient lab reports with structured test results</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Dashboard</a></li>
            <li class="active">Lab Reports</li>
        </ol>
    </section>

    <!-- Main content -->
    <section class="content">
        <!-- Statistics Row - UNCHANGED -->
        <div class="row">
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-aqua">
                    <div class="inner">
                        <h3 id="totalReports">0</h3>
                        <p>Total Reports</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-file-text"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-yellow">
                    <div class="inner">
                        <h3 id="pendingReports">0</h3>
                        <p>Pending Verification</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-clock-o"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-green">
                    <div class="inner">
                        <h3 id="verifiedReports">0</h3>
                        <p>Verified Today</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-check-circle"></i>
                    </div>
                </div>
            </div>
            <div class="col-lg-3 col-xs-6">
                <div class="small-box bg-red">
                    <div class="inner">
                        <h3 id="criticalReports">0</h3>
                        <p>Critical Reports</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-exclamation-triangle"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Patient Search Box - UNCHANGED -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-search"></i> Patient Search</h3>
                    </div>
                    <div class="box-body">
                        <div class="row">
                            <div class="col-md-9">
                                <label>Search Patient</label>
                                <select id="patientSelect" class="form-control select2" style="width: 100%;">
                                    <option value="">Search by patient name, ID, or mobile number...</option>
                                </select>
                            </div>
                            <div class="col-md-3">
                                <label>&nbsp;</label>
                                <button type="button" class="btn btn-primary btn-block" onclick="loadPatientData()">
                                    <i class="fa fa-user"></i> Load Patient
                                </button>
                            </div>
                        </div>
                        
                        <!-- Selected Patient Info - UNCHANGED -->
                        <div id="selectedPatientInfo" class="row" style="display: none; margin-top: 15px;">
                            <div class="col-md-12">
                                <div class="alert alert-info">
                                    <h4><i class="fa fa-info-circle"></i> Selected Patient</h4>
                                    <div class="row">
                                        <div class="col-md-3">
                                            <strong>Name:</strong> <span id="patientName">-</span><br>
                                            <strong>ID:</strong> <span id="patientId">-</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Mobile:</strong> <span id="patientMobile">-</span><br>
                                            <strong>Age:</strong> <span id="patientAge">-</span>
                                        </div>
                                        <div class="col-md-3">
                                            <strong>Gender:</strong> <span id="patientGender">-</span><br>
                                            <strong>Reports:</strong> <span id="patientReportsCount">0</span>
                                        </div>
                                        <div class="col-md-3">
                                            <button class="btn btn-sm btn-info" onclick="viewPatientHistory()">
                                                <i class="fa fa-eye"></i> View All Reports
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Upload and Reports Row - MODIFIED TO INCLUDE TEST RESULTS -->
        <div class="row">
            <!-- Upload Form - ENHANCED -->
            <div class="col-md-8">
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-upload"></i> Upload Lab Report</h3>
                        <!-- ADD: Toggle button for structured results -->
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-sm btn-info" onclick="toggleStructuredResults()">
                                <i class="fa fa-list-alt"></i> <span id="toggleResultsText">Add Structured Results</span>
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <form id="uploadForm" enctype="multipart/form-data">
                            <!-- Basic Information - UNCHANGED -->
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Report Type <span class="text-red">*</span></label>
                                        <select class="form-control" name="report_type" id="reportType" required onchange="handleReportTypeChange()">
                                            <option value="">Select type...</option>
                                            <option value="lab_report">Laboratory Report</option>
                                            <option value="imaging">Imaging/X-Ray</option>
                                            <option value="pathology">Pathology Report</option>
                                            <option value="microbiology">Microbiology</option>
                                            <option value="hematology">Hematology</option>
                                            <option value="biochemistry">Biochemistry</option>
                                            <option value="document">General Document</option>
                                            <option value="other">Other</option>
                                        </select>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Department</label>
                                        <select class="form-control" name="department">
                                            <option value="laboratory">Laboratory</option>
                                            <option value="radiology">Radiology</option>
                                            <option value="pathology">Pathology</option>
                                            <option value="cardiology">Cardiology</option>
                                            <option value="hematology">Hematology</option>
                                            <option value="microbiology">Microbiology</option>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <div class="form-group">
                                <label>Report Title <span class="text-red">*</span></label>
                                <input type="text" class="form-control" name="report_title" 
                                       placeholder="e.g., Complete Blood Count" required>
                            </div>

                            <div class="form-group">
                                <label>Description</label>
                                <textarea class="form-control" name="report_description" rows="3" 
                                          placeholder="Brief description of the report"></textarea>
                            </div>

                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Test Names</label>
                                        <input type="text" class="form-control" name="test_names" 
                                               placeholder="e.g., Hemoglobin, WBC Count, Platelets">
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label>Report Date <span class="text-red">*</span></label>
                                        <input type="date" class="form-control" name="report_date" 
                                               value="<?php echo date('Y-m-d'); ?>" required>
                                    </div>
                                </div>
                            </div>

                            <!-- Critical Flag - UNCHANGED -->
                            <div class="form-group">
                                <div class="checkbox">
                                    <label class="text-red">
                                        <input type="checkbox" name="is_critical" id="isCritical">
                                        <i class="fa fa-exclamation-triangle"></i> Mark as Critical Results
                                    </label>
                                </div>
                            </div>

                            <!-- NEW: STRUCTURED TEST RESULTS SECTION -->
                            <div id="structuredResultsSection" style="display: none;">
                                <div class="form-group">
                                    <h4 class="text-primary">
                                        <i class="fa fa-flask"></i> Structured Test Results
                                        <small class="text-muted">Enter test values for automatic interpretation</small>
                                    </h4>
                                    <hr>
                                </div>

                                <!-- Investigation GHC Section -->
                                <div class="test-category" data-category="investigation_ghc">
                                    <h5 class="bg-primary" style="padding: 10px; margin: 0 0 15px 0; color: white;">
                                        <i class="fa fa-heartbeat"></i> Investigation GHC
                                    </h5>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>HIV</label>
                                                <select class="form-control test-field" name="test_results[hiv]" onchange="interpretResult('hiv', this.value)">
                                                    <option value="">Select...</option>
                                                    <option value="negative">Negative</option>
                                                    <option value="positive">Positive</option>
                                                </select>
                                                <small class="text-muted">Normal: Negative</small>
                                                <div class="interpretation-badge" id="hiv_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Random Blood Sugar (mg/dL)</label>
                                                <input type="number" class="form-control test-field" name="test_results[rbs]" 
                                                       placeholder="Enter value" step="0.1" onchange="interpretResult('rbs', this.value)">
                                                <small class="text-muted">Normal: 70-140 mg/dL</small>
                                                <div class="interpretation-badge" id="rbs_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Haemoglobin (g/dL)</label>
                                                <input type="number" class="form-control test-field" name="test_results[haemoglobin]" 
                                                       placeholder="Enter value" step="0.1" onchange="interpretResult('haemoglobin', this.value)">
                                                <small class="text-muted">M: 13.5-17.5, F: 12-15.5 g/dL</small>
                                                <div class="interpretation-badge" id="haemoglobin_badge"></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>VDRL</label>
                                                <select class="form-control test-field" name="test_results[vdrl]" onchange="interpretResult('vdrl', this.value)">
                                                    <option value="">Select...</option>
                                                    <option value="negative">Negative</option>
                                                    <option value="positive">Positive</option>
                                                </select>
                                                <small class="text-muted">Normal: Negative</small>
                                                <div class="interpretation-badge" id="vdrl_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Creatinine (mg/dL)</label>
                                                <input type="number" class="form-control test-field" name="test_results[creatinine]" 
                                                       placeholder="Enter value" step="0.01" onchange="interpretResult('creatinine', this.value)">
                                                <small class="text-muted">M: 0.7-1.3, F: 0.6-1.1 mg/dL</small>
                                                <div class="interpretation-badge" id="creatinine_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>HBsAg</label>
                                                <select class="form-control test-field" name="test_results[hbsag]" onchange="interpretResult('hbsag', this.value)">
                                                    <option value="">Select...</option>
                                                    <option value="negative">Negative</option>
                                                    <option value="positive">Positive</option>
                                                </select>
                                                <small class="text-muted">Normal: Negative</small>
                                                <div class="interpretation-badge" id="hbsag_badge"></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Hepatitis C Virus Antibody</label>
                                                <select class="form-control test-field" name="test_results[hcv_antibody]" onchange="interpretResult('hcv_antibody', this.value)">
                                                    <option value="">Select...</option>
                                                    <option value="negative">Negative</option>
                                                    <option value="positive">Positive</option>
                                                </select>
                                                <small class="text-muted">Normal: Negative</small>
                                                <div class="interpretation-badge" id="hcv_antibody_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Blood Grouping</label>
                                                <select class="form-control test-field" name="test_results[blood_group]">
                                                    <option value="">Select...</option>
                                                    <option value="A+">A+</option>
                                                    <option value="A-">A-</option>
                                                    <option value="B+">B+</option>
                                                    <option value="B-">B-</option>
                                                    <option value="AB+">AB+</option>
                                                    <option value="AB-">AB-</option>
                                                    <option value="O+">O+</option>
                                                    <option value="O-">O-</option>
                                                </select>
                                                <small class="text-muted">ABO and Rh typing</small>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Pregnancy Test</label>
                                                <select class="form-control test-field" name="test_results[pregnancy_test]" onchange="interpretResult('pregnancy_test', this.value)">
                                                    <option value="">Select...</option>
                                                    <option value="negative">Negative</option>
                                                    <option value="positive">Positive</option>
                                                </select>
                                                <small class="text-muted">Normal: Negative (non-pregnant)</small>
                                                <div class="interpretation-badge" id="pregnancy_test_badge"></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Bs for MPS</label>
                                                <textarea class="form-control test-field" name="test_results[bs_mps]" 
                                                          placeholder="Enter microscopic examination results" rows="2"></textarea>
                                                <small class="text-muted">Microscopic examination</small>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-6">
                                            <div class="form-group">
                                                <label>Stool OC</label>
                                                <textarea class="form-control test-field" name="test_results[stool_oc]" 
                                                          placeholder="Enter stool ova and cysts examination results" rows="2"></textarea>
                                                <small class="text-muted">Ova and cysts examination</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Urinalysis Section -->
                                <div class="test-category" data-category="urinalysis">
                                    <h5 class="bg-info" style="padding: 10px; margin: 15px 0 15px 0; color: white;">
                                        <i class="fa fa-tint"></i> Urinalysis
                                    </h5>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Appearance</label>
                                                <input type="text" class="form-control test-field" name="test_results[urine_appearance]" 
                                                       placeholder="e.g., Clear, Turbid, Cloudy">
                                                <small class="text-muted">Normal: Clear/Yellow</small>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Form</label>
                                                <input type="text" class="form-control test-field" name="test_results[urine_form]" 
                                                       placeholder="e.g., Liquid">
                                                <small class="text-muted">Normal: Liquid</small>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>pH</label>
                                                <input type="number" class="form-control test-field" name="test_results[urine_ph]" 
                                                       placeholder="Enter value" step="0.1" min="0" max="14" onchange="interpretResult('urine_ph', this.value)">
                                                <small class="text-muted">Normal: 4.5-8.0</small>
                                                <div class="interpretation-badge" id="urine_ph_badge"></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Specific Gravity</label>
                                                <input type="number" class="form-control test-field" name="test_results[specific_gravity]" 
                                                       placeholder="Enter value" step="0.001" onchange="interpretResult('specific_gravity', this.value)">
                                                <small class="text-muted">Normal: 1.003-1.030</small>
                                                <div class="interpretation-badge" id="specific_gravity_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Nitrite</label>
                                                <select class="form-control test-field" name="test_results[nitrite]" onchange="interpretResult('nitrite', this.value)">
                                                    <option value="">Select...</option>
                                                    <option value="negative">Negative</option>
                                                    <option value="positive">Positive</option>
                                                </select>
                                                <small class="text-muted">Normal: Negative</small>
                                                <div class="interpretation-badge" id="nitrite_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Protein</label>
                                                <input type="text" class="form-control test-field" name="test_results[protein]" 
                                                       placeholder="e.g., Negative, Trace, 1+, 2+">
                                                <small class="text-muted">Normal: Negative/Trace</small>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label>Microscopy</label>
                                                <textarea class="form-control test-field" name="test_results[microscopy]" 
                                                          placeholder="Enter microscopic examination findings" rows="3"></textarea>
                                                <small class="text-muted">Microscopic examination findings</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Liver Function Test Section -->
                                <div class="test-category" data-category="liver_function">
                                    <h5 class="bg-warning" style="padding: 10px; margin: 15px 0 15px 0; color: white;">
                                        <i class="fa fa-user-md"></i> Liver Function Test
                                    </h5>
                                    
                                    <div class="row">
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>GPT (ALT) U/L</label>
                                                <input type="number" class="form-control test-field" name="test_results[gpt_alt]" 
                                                       placeholder="Enter value" step="0.1" onchange="interpretResult('gpt_alt', this.value)">
                                                <small class="text-muted">Normal: 7-35 U/L</small>
                                                <div class="interpretation-badge" id="gpt_alt_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>GOT (AST) U/L</label>
                                                <input type="number" class="form-control test-field" name="test_results[got_ast]" 
                                                       placeholder="Enter value" step="0.1" onchange="interpretResult('got_ast', this.value)">
                                                <small class="text-muted">Normal: 8-40 U/L</small>
                                                <div class="interpretation-badge" id="got_ast_badge"></div>
                                            </div>
                                        </div>
                                        
                                        <div class="col-md-4">
                                            <div class="form-group">
                                                <label>Total Bilirubin (mg/dL)</label>
                                                <input type="number" class="form-control test-field" name="test_results[total_bilirubin]" 
                                                       placeholder="Enter value" step="0.01" onchange="interpretResult('total_bilirubin', this.value)">
                                                <small class="text-muted">Normal: 0.2-1.2 mg/dL</small>
                                                <div class="interpretation-badge" id="total_bilirubin_badge"></div>
                                            </div>
                                        </div>
                                    </div>

                                    <div class="row">
                                        <div class="col-md-12">
                                            <div class="form-group">
                                                <label>Remarks</label>
                                                <textarea class="form-control test-field" name="test_results[remarks]" 
                                                          placeholder="Additional clinical remarks" rows="2"></textarea>
                                                <small class="text-muted">Additional clinical observations</small>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Test Results Summary -->
                                <div id="testResultsSummary" style="display: none;">
                                    <div class="alert alert-info">
                                        <h4><i class="fa fa-info-circle"></i> Test Results Summary</h4>
                                        <div id="summaryContent"></div>
                                    </div>
                                </div>
                            </div>
                            <!-- END: STRUCTURED TEST RESULTS SECTION -->

                            <!-- File Upload - UNCHANGED -->
                            <div class="form-group">
                                <label>Upload File <span class="text-red">*</span></label>
                                <div class="file-upload-zone" onclick="document.getElementById('fileInput').click()">
                                    <input type="file" id="fileInput" name="lab_report_file" 
                                           accept=".pdf,.jpg,.jpeg,.png,.doc,.docx" style="display: none;" required>
                                    <div class="text-center">
                                        <i class="fa fa-cloud-upload fa-4x text-blue"></i>
                                        <h4>Click to select file or drag and drop</h4>
                                        <p class="text-muted">Supported: PDF, JPG, PNG, DOC, DOCX (Max: 10MB)</p>
                                    </div>
                                </div>
                                <div id="fileInfo" style="display: none; margin-top: 10px;"></div>
                            </div>

                            <!-- Additional Notes - UNCHANGED -->
                            <div class="form-group">
                                <label>Additional Notes</label>
                                <textarea class="form-control" name="notes" rows="2" 
                                          placeholder="Any additional notes or observations"></textarea>
                            </div>

                            <!-- Submit Button - UNCHANGED -->
                            <div class="form-group">
                                <button type="submit" class="btn btn-success btn-lg">
                                    <i class="fa fa-upload"></i> Upload Report
                                </button>
                                <button type="reset" class="btn btn-default btn-lg">
                                    <i class="fa fa-refresh"></i> Reset Form
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Patient Reports List - UNCHANGED -->
            <div class="col-md-4">
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-history"></i> Patient Reports</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-box-tool" onclick="refreshPatientReports()">
                                <i class="fa fa-refresh"></i>
                            </button>
                        </div>
                    </div>
                    <div class="box-body" style="height: 500px; overflow-y: auto;">
                        <div id="patientReportsList">
                            <div class="text-center text-muted" style="padding: 50px 20px;">
                                <i class="fa fa-file-text-o fa-3x"></i>
                                <p style="margin-top: 15px;">Select a patient to view reports</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- All Reports Table - UNCHANGED -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-list"></i> All Lab Reports</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-success btn-sm" onclick="exportReports()">
                                <i class="fa fa-download"></i> Export
                            </button>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- Filters - UNCHANGED -->
                        <div class="row" style="margin-bottom: 15px;">
                            <div class="col-md-2">
                                <select class="form-control input-sm" id="filterStatus">
                                    <option value="">All Status</option>
                                    <option value="pending">Pending</option>
                                    <option value="completed">Completed</option>
                                    <option value="verified">Verified</option>
                                    <option value="cancelled">Cancelled</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <select class="form-control input-sm" id="filterDepartment">
                                    <option value="">All Departments</option>
                                    <option value="laboratory">Laboratory</option>
                                    <option value="radiology">Radiology</option>
                                    <option value="pathology">Pathology</option>
                                    <option value="cardiology">Cardiology</option>
                                </select>
                            </div>
                            <div class="col-md-2">
                                <input type="date" class="form-control input-sm" id="filterDateFrom" placeholder="From Date">
                            </div>
                            <div class="col-md-2">
                                <input type="date" class="form-control input-sm" id="filterDateTo" placeholder="To Date">
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-primary btn-sm" onclick="applyFilters()">
                                    <i class="fa fa-filter"></i> Filter
                                </button>
                            </div>
                            <div class="col-md-2">
                                <button class="btn btn-default btn-sm" onclick="clearFilters()">
                                    <i class="fa fa-times"></i> Clear
                                </button>
                            </div>
                        </div>

                        <!-- Reports Table -->
                        <table id="reportsTable" class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Patient</th>
                                    <th>Report Title</th>
                                    <th>Type</th>
                                    <th>Department</th>
                                    <th>Date</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <!-- Data loaded via AJAX -->
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- View Report Modal - UNCHANGED -->
<div class="modal fade" id="viewReportModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-eye"></i> View Lab Report
                </h4>
            </div>
            <div class="modal-body">
                <div id="reportPreview"></div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" id="downloadReportBtn">
                    <i class="fa fa-download"></i> Download
                </button>
                <button type="button" class="btn btn-success" id="verifyReportBtn" style="display: none;">
                    <i class="fa fa-check"></i> Verify Report
                </button>
            </div>
        </div>
    </div>
</div>

<!-- NEW: Enhanced CSS Styles -->
<style>
/* Original styles preserved */
.file-upload-zone {
    border: 2px dashed #3c8dbc;
    border-radius: 5px;
    padding: 30px;
    text-align: center;
    cursor: pointer;
    transition: all 0.3s ease;
    background: #f9f9f9;
    margin-bottom: 10px;
}

.file-upload-zone:hover {
    border-color: #2c6fbb;
    background: #f0f8ff;
}

.file-upload-zone.dragover {
    border-color: #00a65a;
    background: #f0fff0;
}

.report-item {
    border-left: 4px solid #3c8dbc;
    padding: 10px;
    margin-bottom: 10px;
    background: #fff;
    border-radius: 0 3px 3px 0;
    border: 1px solid #ddd;
}

.report-item.critical {
    border-left-color: #dd4b39;
    background: #fff5f5;
}

.report-item.verified {
    border-left-color: #00a65a;
    background: #f0fff0;
}

.report-item h5 {
    margin: 0 0 5px 0;
    font-size: 14px;
}

.report-item .report-meta {
    font-size: 11px;
    color: #666;
    margin-bottom: 8px;
}

.report-item .report-actions {
    text-align: right;
}

.report-item .report-actions .btn {
    margin-left: 3px;
}

/* NEW: Enhanced styles for structured test results */
.test-category {
    border: 1px solid #ddd;
    border-radius: 5px;
    margin-bottom: 20px;
    padding: 0 15px 15px 15px;
    background: #fafafa;
}

.test-field {
    margin-bottom: 15px;
}

.interpretation-badge {
    margin-top: 5px;
    padding: 2px 8px;
    border-radius: 3px;
    font-size: 11px;
    font-weight: bold;
    text-align: center;
    min-height: 20px;
}

.interpretation-badge.normal {
    background: #d4edda;
    color: #155724;
}

.interpretation-badge.high {
    background: #f8d7da;
    color: #721c24;
}

.interpretation-badge.low {
    background: #fff3cd;
    color: #856404;
}

.interpretation-badge.critical {
    background: #f5c6cb;
    color: #721c24;
    animation: pulse 1s infinite;
}

.interpretation-badge.positive {
    background: #f8d7da;
    color: #721c24;
}

.interpretation-badge.negative {
    background: #d4edda;
    color: #155724;
}

@keyframes pulse {
    0% { opacity: 1; }
    50% { opacity: 0.7; }
    100% { opacity: 1; }
}

.test-field:focus {
    border-color: #3c8dbc;
    box-shadow: 0 0 5px rgba(60, 141, 188, 0.3);
}

#structuredResultsSection {
    border: 2px solid #e3f2fd;
    border-radius: 8px;
    padding: 20px;
    background: #fafbfc;
    margin: 20px 0;
}

.test-category h5 {
    border-radius: 5px;
}

#testResultsSummary {
    margin-top: 20px;
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .test-category .row .col-md-4 {
        margin-bottom: 15px;
    }
}
</style>

<!-- NEW: Enhanced JavaScript -->
<script>
// Enhanced JavaScript for Lab Reports - Fixed Version
let currentPatientId = null;
let currentPatientData = null;
let structuredResultsVisible = false;
let testReferenceRanges = {};

$(document).ready(function() {
    console.log('Initializing Lab Reports page...');
    initializeComponents();
    loadDashboardStats();
    loadReferenceRanges();
});

function initializeComponents() {
    console.log('Initializing components...');
    
    // Initialize Select2 for patient search
    $('#patientSelect').select2({
        placeholder: 'Search patient by name, ID, or mobile number',
        ajax: {
            url: base_url + 'admin/lab_reports/search_patients',
            dataType: 'json',
            delay: 250,
            data: function (params) {
                return { 
                    q: params.term, 
                    page: params.page || 1 
                };
            },
            processResults: function (data, params) {
                console.log('Patient search results:', data);
                params.page = params.page || 1;
                return {
                    results: data.items || [],
                    pagination: { 
                        more: (params.page * 10) < (data.total_count || 0) 
                    }
                };
            },
            cache: true
        },
        minimumInputLength: 2
    });

    // Initialize DataTable with error handling
    try {
        console.log('Initializing DataTable...');
        
        if ($.fn.DataTable.isDataTable('#reportsTable')) {
            $('#reportsTable').DataTable().destroy();
        }
        
        $('#reportsTable').DataTable({
            processing: true,
            serverSide: true,
            ajax: {
                url: base_url + 'admin/lab_reports/get_all_reports',
                type: 'POST',
                data: function(d) {
                    d.status = $('#filterStatus').val();
                    d.department = $('#filterDepartment').val();
                    d.date_from = $('#filterDateFrom').val();
                    d.date_to = $('#filterDateTo').val();
                    console.log('DataTable request data:', d);
                },
                error: function(xhr, error, code) {
                    console.error('DataTable AJAX error:', xhr, error, code);
                    toastr.error('Failed to load reports data');
                }
            },
            columns: [
                { data: 'id', width: '50px' },
                { data: 'patient_name' },
                { data: 'report_title' },
                { data: 'report_type' },
                { data: 'department' },
                { data: 'report_date', width: '100px' },
                { data: 'status', width: '80px' },
                { data: 'actions', orderable: false, width: '120px' }
            ],
            order: [[0, 'desc']],
            pageLength: 25,
            language: {
                processing: 'Loading reports...',
                emptyTable: 'No reports found',
                zeroRecords: 'No matching records found'
            },
            drawCallback: function(settings) {
                console.log('DataTable draw completed. Rows:', settings.aoData.length);
            }
        });
        
        console.log('DataTable initialized successfully');
        
    } catch (error) {
        console.error('DataTable initialization error:', error);
        toastr.error('Failed to initialize reports table');
    }

    setupFileUpload();
    setupFormHandlers();
}

// Load reference ranges from server
function loadReferenceRanges() {
    $.ajax({
        url: base_url + 'admin/lab_reports/get_reference_ranges',
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.success) {
                testReferenceRanges = response.data;
                console.log('Reference ranges loaded:', testReferenceRanges);
            }
        },
        error: function(xhr, status, error) {
            console.log('Failed to load reference ranges, using defaults');
            loadDefaultReferenceRanges();
        }
    });
}

// Default reference ranges fallback
function loadDefaultReferenceRanges() {
    testReferenceRanges = {
        'hiv': { type: 'select', normal: 'negative' },
        'rbs': { type: 'numeric', min: 70, max: 140, unit: 'mg/dL' },
        'haemoglobin': { type: 'numeric', min_male: 13.5, max_male: 17.5, min_female: 12, max_female: 15.5, unit: 'g/dL' },
        'vdrl': { type: 'select', normal: 'negative' },
        'creatinine': { type: 'numeric', min_male: 0.7, max_male: 1.3, min_female: 0.6, max_female: 1.1, unit: 'mg/dL' },
        'hbsag': { type: 'select', normal: 'negative' },
        'hcv_antibody': { type: 'select', normal: 'negative' },
        'pregnancy_test': { type: 'select', normal: 'negative' },
        'urine_ph': { type: 'numeric', min: 4.5, max: 8.0 },
        'specific_gravity': { type: 'numeric', min: 1.003, max: 1.030 },
        'nitrite': { type: 'select', normal: 'negative' },
        'gpt_alt': { type: 'numeric', min: 7, max: 35, unit: 'U/L' },
        'got_ast': { type: 'numeric', min: 8, max: 40, unit: 'U/L' },
        'total_bilirubin': { type: 'numeric', min: 0.2, max: 1.2, unit: 'mg/dL' }
    };
}

// Toggle structured results section
function toggleStructuredResults() {
    structuredResultsVisible = !structuredResultsVisible;
    
    if (structuredResultsVisible) {
        $('#structuredResultsSection').slideDown();
        $('#toggleResultsText').text('Hide Structured Results');
    } else {
        $('#structuredResultsSection').slideUp();
        $('#toggleResultsText').text('Add Structured Results');
    }
}

// Handle report type change
function handleReportTypeChange() {
    const reportType = $('#reportType').val();
    
    // Show structured results by default for lab reports
    if (reportType === 'lab_report' && !structuredResultsVisible) {
        toggleStructuredResults();
    }
}

// Interpret test results dynamically
function interpretResult(testName, value) {
    console.log('Interpreting result for:', testName, 'value:', value);
    
    if (!value || value === '') {
        $('#' + testName + '_badge').html('').removeClass();
        return;
    }

    const range = testReferenceRanges[testName];
    if (!range) {
        console.log('No reference range found for:', testName);
        return;
    }

    let interpretation = '';
    let badgeClass = '';

    if (range.type === 'select') {
        if (value === range.normal) {
            interpretation = 'Normal';
            badgeClass = 'normal';
        } else {
            interpretation = 'Abnormal';
            badgeClass = 'positive';
        }
    } else if (range.type === 'numeric') {
        const numValue = parseFloat(value);
        
        // Check for gender-specific ranges
        let minRange, maxRange;
        const patientGender = $('#patientGender').text().toLowerCase();
        
        if (range.min_male && range.min_female && patientGender) {
            if (patientGender === 'male') {
                minRange = range.min_male;
                maxRange = range.max_male;
            } else {
                minRange = range.min_female;
                maxRange = range.max_female;
            }
        } else {
            minRange = range.min;
            maxRange = range.max;
        }

        if (numValue < minRange) {
            interpretation = 'Low';
            badgeClass = 'low';
        } else if (numValue > maxRange) {
            interpretation = 'High';
            badgeClass = 'high';
            
            // Check for critical values
            if (range.critical_high && numValue > range.critical_high) {
                interpretation = 'Critical High';
                badgeClass = 'critical';
            }
        } else {
            interpretation = 'Normal';
            badgeClass = 'normal';
        }
    }

    $('#' + testName + '_badge')
        .html(interpretation)
        .removeClass()
        .addClass('interpretation-badge ' + badgeClass);

    // Auto-generate summary
    updateTestResultsSummary();
}

// Update test results summary
function updateTestResultsSummary() {
    const abnormalResults = [];
    const criticalResults = [];
    
    $('.interpretation-badge').each(function() {
        const badge = $(this);
        const text = badge.text();
        const testName = badge.attr('id').replace('_badge', '');
        
        if (badge.hasClass('high') || badge.hasClass('low') || badge.hasClass('positive')) {
            abnormalResults.push({
                test: testName.replace(/_/g, ' ').toUpperCase(),
                result: text
            });
        }
        
        if (badge.hasClass('critical')) {
            criticalResults.push({
                test: testName.replace(/_/g, ' ').toUpperCase(),
                result: text
            });
        }
    });

    if (abnormalResults.length > 0 || criticalResults.length > 0) {
        let summaryHtml = '';
        
        if (criticalResults.length > 0) {
            summaryHtml += '<div class="alert alert-danger"><strong>Critical Results:</strong><ul>';
            criticalResults.forEach(function(result) {
                summaryHtml += '<li>' + result.test + ': ' + result.result + '</li>';
            });
            summaryHtml += '</ul></div>';
        }
        
        if (abnormalResults.length > 0) {
            summaryHtml += '<div class="alert alert-warning"><strong>Abnormal Results:</strong><ul>';
            abnormalResults.forEach(function(result) {
                summaryHtml += '<li>' + result.test + ': ' + result.result + '</li>';
            });
            summaryHtml += '</ul></div>';
        }
        
        $('#summaryContent').html(summaryHtml);
        $('#testResultsSummary').show();
        
        // Auto-check critical flag if critical results found
        if (criticalResults.length > 0) {
            $('#isCritical').prop('checked', true);
        }
    } else {
        $('#testResultsSummary').hide();
    }
}

// Setup file upload functionality
function setupFileUpload() {
    const fileInput = document.getElementById('fileInput');
    const fileUploadZone = document.querySelector('.file-upload-zone');
    const fileInfo = document.getElementById('fileInfo');

    if (!fileInput || !fileUploadZone || !fileInfo) {
        console.log('File upload elements not found');
        return;
    }

    fileInput.addEventListener('change', function(e) {
        if (e.target.files.length > 0) {
            displayFileInfo(e.target.files[0]);
        }
    });

    fileUploadZone.addEventListener('dragover', function(e) {
        e.preventDefault();
        fileUploadZone.classList.add('dragover');
    });

    fileUploadZone.addEventListener('dragleave', function(e) {
        e.preventDefault();
        fileUploadZone.classList.remove('dragover');
    });

    fileUploadZone.addEventListener('drop', function(e) {
        e.preventDefault();
        fileUploadZone.classList.remove('dragover');
        
        const files = e.dataTransfer.files;
        if (files.length > 0) {
            fileInput.files = files;
            displayFileInfo(files[0]);
        }
    });

    function displayFileInfo(file) {
        const fileSize = (file.size / 1024 / 1024).toFixed(2);
        fileInfo.innerHTML = `
            <div class="alert alert-info" style="margin: 0;">
                <i class="fa fa-file"></i> <strong>Selected:</strong> ${file.name}<br>
                <i class="fa fa-hdd-o"></i> <strong>Size:</strong> ${fileSize} MB<br>
                <i class="fa fa-file-code-o"></i> <strong>Type:</strong> ${file.type}
            </div>
        `;
        fileInfo.style.display = 'block';
    }
}

// Setup form handlers
function setupFormHandlers() {
    $('#uploadForm').on('submit', function(e) {
        e.preventDefault();
        if (!currentPatientId) {
            toastr.warning('Please select a patient first');
            return;
        }
        uploadLabReport();
    });
}

// Load patient data
function loadPatientData() {
    const selectedPatient = $('#patientSelect').select2('data')[0];
    if (!selectedPatient) {
        toastr.warning('Please select a patient');
        return;
    }

    console.log('Loading patient data:', selectedPatient);

    currentPatientId = selectedPatient.id;
    currentPatientData = selectedPatient;

    // Update patient info display
    $('#patientName').text(selectedPatient.patient_name);
    $('#patientId').text(selectedPatient.id);
    $('#patientMobile').text(selectedPatient.mobileno || 'N/A');
    $('#patientAge').text(selectedPatient.age || 'N/A');
    $('#patientGender').text(selectedPatient.gender || 'N/A');

    $('#selectedPatientInfo').show();
    
    loadPatientReports();
}

// Upload lab report with enhanced error handling
function uploadLabReport() {
    console.log('Starting lab report upload...');
    
    const formData = new FormData($('#uploadForm')[0]);
    formData.append('patient_id', currentPatientId);

    // Add structured test results if present
    if (structuredResultsVisible) {
        const testResults = {};
        const referenceRanges = {};
        const interpretations = {};
        
        $('.test-field').each(function() {
            const field = $(this);
            const name = field.attr('name');
            const value = field.val();
            
            if (name && name.startsWith('test_results[') && value) {
                const testName = name.replace('test_results[', '').replace(']', '');
                testResults[testName] = value;
                
                // Get interpretation if available
                const interpretationBadge = $('#' + testName + '_badge');
                if (interpretationBadge.length && interpretationBadge.text()) {
                    interpretations[testName] = {
                        value: value,
                        interpretation: interpretationBadge.text(),
                        status: interpretationBadge.attr('class').split(' ').pop()
                    };
                }
            }
        });
        
        if (Object.keys(testResults).length > 0) {
            formData.append('test_results', JSON.stringify(testResults));
            formData.append('test_interpretations', JSON.stringify(interpretations));
            formData.append('reference_ranges', JSON.stringify(testReferenceRanges));
            console.log('Added structured test results:', testResults);
        }
    }

    // Show loading
    const submitBtn = $('#uploadForm button[type="submit"]');
    const originalText = submitBtn.html();
    submitBtn.html('<i class="fa fa-spinner fa-spin"></i> Uploading...').prop('disabled', true);

    $.ajax({
        url: base_url + 'admin/lab_reports/upload_report',
        method: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        success: function(response) {
            console.log('Upload response:', response);
            
            if (response.success) {
                toastr.success('Lab report uploaded successfully!');
                $('#uploadForm')[0].reset();
                $('#fileInfo').hide();
                $('#structuredResultsSection').hide();
                structuredResultsVisible = false;
                $('#toggleResultsText').text('Add Structured Results');
                $('.interpretation-badge').html('').removeClass();
                $('#testResultsSummary').hide();
                loadPatientReports();
                loadDashboardStats();
                
                // Reload DataTable
                if ($.fn.DataTable.isDataTable('#reportsTable')) {
                    $('#reportsTable').DataTable().ajax.reload(null, false);
                }
            } else {
                toastr.error('Upload failed: ' + (response.message || 'Unknown error'));
            }
        },
        error: function(xhr, status, error) {
            console.error('Upload error:', xhr, status, error);
            let errorMessage = 'Error uploading file';
            
            try {
                const response = JSON.parse(xhr.responseText);
                errorMessage = response.message || errorMessage;
            } catch (e) {
                // Use default error message
            }
            
            toastr.error(errorMessage);
        },
        complete: function() {
            submitBtn.html(originalText).prop('disabled', false);
        }
    });
}

// Load patient reports
function loadPatientReports() {
    if (!currentPatientId) {
        console.log('No patient selected');
        return;
    }

    console.log('Loading reports for patient:', currentPatientId);

    $.ajax({
        url: base_url + 'admin/lab_reports/get_patient_reports',
        method: 'GET',
        data: { patient_id: currentPatientId },
        dataType: 'json',
        success: function(response) {
            console.log('Patient reports response:', response);
            
            if (response.success) {
                displayPatientReports(response.data);
                $('#patientReportsCount').text(response.data.length);
            } else {
                toastr.error('Failed to load patient reports: ' + (response.message || 'Unknown error'));
            }
        },
        error: function(xhr, status, error) {
            console.error('Error loading patient reports:', xhr, status, error);
            toastr.error('Error loading patient reports');
        }
    });
}

// Display patient reports
function displayPatientReports(reports) {
    const container = $('#patientReportsList');
    
    if (!reports || reports.length === 0) {
        container.html(`
            <div class="text-center text-muted" style="padding: 50px 20px;">
                <i class="fa fa-file-text-o fa-2x"></i>
                <p style="margin-top: 15px;">No reports found</p>
            </div>
        `);
        return;
    }

    let html = '';
    reports.forEach(function(report) {
        const statusClass = getStatusClass(report.status);
        const criticalClass = report.is_critical == 1 ? 'critical' : '';
        const criticalIcon = report.is_critical == 1 ? '<i class="fa fa-exclamation-triangle text-red"></i> ' : '';

        html += `
            <div class="report-item ${criticalClass}">
                <h5>${criticalIcon}${report.report_title}</h5>
                <div class="report-meta">
                    ${report.report_type} • ${report.department} • ${report.report_date}
                </div>
                <div class="report-meta">
                    Status: <span class="label ${statusClass}">${report.status}</span>
                </div>
                <div class="report-actions">
                    <button class="btn btn-xs btn-primary" onclick="viewReport(${report.id})" title="View">
                        <i class="fa fa-eye"></i>
                    </button>
                    <button class="btn btn-xs btn-success" onclick="downloadReport(${report.id})" title="Download">
                        <i class="fa fa-download"></i>
                    </button>
                    <button class="btn btn-xs btn-info" onclick="generateProfessionalReport(${report.id})" title="Professional Report">
                        <i class="fa fa-file-text"></i>
                    </button>
                    ${report.status !== 'verified' ? 
                        `<button class="btn btn-xs btn-warning" onclick="verifyReport(${report.id})" title="Verify">
                            <i class="fa fa-check"></i>
                        </button>` : ''
                    }
                </div>
            </div>
        `;
    });

    container.html(html);
}

// Get status class for labels
function getStatusClass(status) {
    const statusClasses = {
        'pending': 'label-warning',
        'completed': 'label-primary',
        'verified': 'label-success',
        'cancelled': 'label-danger'
    };
    return statusClasses[status] || 'label-default';
}

// View report function
function viewReport(reportId) {
    console.log('Viewing report:', reportId);
    
    $.ajax({
        url: base_url + 'admin/lab_reports/get_report/' + reportId,
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            console.log('Report details:', response);
            
            if (response.success) {
                const report = response.data;
                let previewHtml = '';

                if (report.file_type && report.file_type.includes('image')) {
                    previewHtml = `<img src="${report.file_url}" style="max-width: 100%; height: 400px;" alt="Lab Report">`;
                } else if (report.file_type && report.file_type.includes('pdf')) {
                    previewHtml = `<embed src="${report.file_url}" type="application/pdf" style="width: 100%; height: 400px;">`;
                } else {
                    previewHtml = `
                        <div class="text-center" style="padding: 40px;">
                            <i class="fa fa-file-text-o fa-4x text-muted"></i>
                            <h4>${report.report_title}</h4>
                            <p><strong>File:</strong> ${report.original_filename}</p>
                            <p><strong>Size:</strong> ${(report.file_size / 1024 / 1024).toFixed(2)} MB</p>
                            <p><strong>Type:</strong> ${report.file_type}</p>
                        </div>
                    `;
                }

                $('#reportPreview').html(previewHtml);
                $('#downloadReportBtn').attr('onclick', `downloadReport(${reportId})`);
                
                if (report.status !== 'verified') {
                    $('#verifyReportBtn').show().attr('onclick', `verifyReport(${reportId})`);
                } else {
                    $('#verifyReportBtn').hide();
                }
                
                $('#viewReportModal').modal('show');
            } else {
                toastr.error('Failed to load report: ' + (response.message || 'Unknown error'));
            }
        },
        error: function(xhr, status, error) {
            console.error('Error loading report:', xhr, status, error);
            toastr.error('Error loading report');
        }
    });
}

// Download report function
function downloadReport(reportId) {
    console.log('Downloading report:', reportId);
    window.open(base_url + 'admin/lab_reports/download_report/' + reportId, '_blank');
}

// Verify report function
function verifyReport(reportId) {
    console.log('Verifying report:', reportId);
    
    if (confirm('Are you sure you want to verify this report?')) {
        $.ajax({
            url: base_url + 'admin/lab_reports/verify_report',
            method: 'POST',
            data: { 
                report_id: reportId,
                notes: 'Report verified by staff'
            },
            dataType: 'json',
            success: function(response) {
                console.log('Verification response:', response);
                
                if (response.success) {
                    toastr.success('Report verified successfully!');
                    loadPatientReports();
                    loadDashboardStats();
                    
                    // Reload DataTable
                    if ($.fn.DataTable.isDataTable('#reportsTable')) {
                        $('#reportsTable').DataTable().ajax.reload(null, false);
                    }
                    
                    $('#viewReportModal').modal('hide');
                } else {
                    toastr.error('Verification failed: ' + (response.message || 'Unknown error'));
                }
            },
            error: function(xhr, status, error) {
                console.error('Verification error:', xhr, status, error);
                toastr.error('Error verifying report');
            }
        });
    }
}

// Load dashboard statistics
function loadDashboardStats() {
    console.log('Loading dashboard stats...');
    
    $.ajax({
        url: base_url + 'admin/lab_reports/get_dashboard_stats',
        method: 'GET',
        dataType: 'json',
        success: function(response) {
            console.log('Dashboard stats response:', response);
            
            if (response.success) {
                const stats = response.data;
                $('#totalReports').text(stats.total || 0);
                $('#pendingReports').text(stats.pending || 0);
                $('#verifiedReports').text(stats.verified_today || 0);
                $('#criticalReports').text(stats.critical || 0);
            } else {
                console.log('Failed to load dashboard stats:', response.message);
            }
        },
        error: function(xhr, status, error) {
            console.error('Error loading dashboard stats:', xhr, status, error);
        }
    });
}

// Filter functions
function applyFilters() {
    console.log('Applying filters...');
    if ($.fn.DataTable.isDataTable('#reportsTable')) {
        $('#reportsTable').DataTable().ajax.reload();
    }
}

function clearFilters() {
    console.log('Clearing filters...');
    $('#filterStatus, #filterDepartment, #filterDateFrom, #filterDateTo').val('');
    if ($.fn.DataTable.isDataTable('#reportsTable')) {
        $('#reportsTable').DataTable().ajax.reload();
    }
}

// Export reports function
function exportReports() {
    console.log('Exporting reports...');
    const params = new URLSearchParams({
        status: $('#filterStatus').val(),
        department: $('#filterDepartment').val(),
        date_from: $('#filterDateFrom').val(),
        date_to: $('#filterDateTo').val()
    });
    
    window.open(base_url + 'admin/lab_reports/export_reports?' + params.toString(), '_blank');
}

// Refresh patient reports
function refreshPatientReports() {
    if (currentPatientId) {
        loadPatientReports();
    } else {
        toastr.info('Please select a patient first');
    }
}

// View patient history
function viewPatientHistory() {
    if (currentPatientId) {
        $('html, body').animate({
            scrollTop: $("#reportsTable").offset().top - 100
        }, 1000);
        
        toastr.info('Showing all reports table below');
    }
}

// Generate professional report function
function generateProfessionalReport(reportId) {
    console.log('Generating professional report for:', reportId);
    
    // Open in new window/tab
    const url = base_url + 'admin/lab_reports/generateProfessionalReport/' + reportId;
    window.open(url, '_blank');
}

// Auto-refresh dashboard stats every 5 minutes
setInterval(function() {
    loadDashboardStats();
}, 300000);

// Initialize tooltips
$(document).ready(function() {
    $('[data-toggle="tooltip"]').tooltip();
    
    // Check if base_url is defined
    if (typeof base_url === 'undefined') {
        console.error('base_url is not defined. Please check your view file.');
        window.base_url = window.location.origin + '/'; // Fallback
    }
    
    console.log('Lab Reports JavaScript initialized successfully');
});
</script>