<?php
/**
 * Insurance Companies Management View
 * File: application/views/admin/insurance/companies.php
 */
?>

<div class="content-wrapper">
    <section class="content-header">
        <h1>
            <i class="fa fa-building"></i> Insurance Companies
            <small>Manage insurance company information and settings</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/dashboard"><i class="fa fa-dashboard"></i> Dashboard</a></li>
            <li><a href="<?php echo base_url(); ?>admin/insurance">Insurance</a></li>
            <li class="active">Companies</li>
        </ol>
    </section>

    <section class="content">
        <!-- Quick Stats Row -->
        <div class="row">
            <div class="col-md-3 col-sm-6 col-xs-12">
                <div class="small-box bg-aqua">
                    <div class="inner">
                        <h3><?php echo count(array_filter($companies, function($c) { return $c['status'] == 'active'; })); ?></h3>
                        <p>Active Companies</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-check-circle"></i>
                    </div>
                </div>
            </div>
            <div class="col-md-3 col-sm-6 col-xs-12">
                <div class="small-box bg-green">
                    <div class="inner">
                        <h3><?php echo count($companies); ?></h3>
                        <p>Total Companies</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-building"></i>
                    </div>
                </div>
            </div>
            <div class="col-md-3 col-sm-6 col-xs-12">
                <div class="small-box bg-yellow">
                    <div class="inner">
                        <h3><?php echo count(array_filter($companies, function($c) { return $c['status'] == 'inactive'; })); ?></h3>
                        <p>Inactive Companies</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-pause-circle"></i>
                    </div>
                </div>
            </div>
            <div class="col-md-3 col-sm-6 col-xs-12">
                <div class="small-box bg-red">
                    <div class="inner">
                        <h3><?php echo count(array_filter($companies, function($c) { return $c['status'] == 'suspended'; })); ?></h3>
                        <p>Suspended</p>
                    </div>
                    <div class="icon">
                        <i class="fa fa-ban"></i>
                    </div>
                </div>
            </div>
        </div>

        <!-- Main Companies Table -->
        <div class="row">
            <div class="col-md-12">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-table"></i> Insurance Companies</h3>
                        <div class="box-tools pull-right">
                            <button type="button" class="btn btn-success btn-sm" onclick="showAddCompanyModal()">
                                <i class="fa fa-plus"></i> Add New Company
                            </button>
                            <button type="button" class="btn btn-info btn-sm" onclick="importCompanies()">
                                <i class="fa fa-upload"></i> Import Companies
                            </button>
                            <button type="button" class="btn btn-warning btn-sm" onclick="exportCompanies()">
                                <i class="fa fa-download"></i> Export List
                            </button>
                        </div>
                    </div>
                    
                    <div class="box-body">
                        <!-- Filters -->
                        <div class="row">
                            <div class="col-md-3">
                                <div class="form-group">
                                    <label>Filter by Status</label>
                                    <select id="statusFilter" class="form-control" onchange="filterCompanies()">
                                        <option value="">All Statuses</option>
                                        <option value="active">Active</option>
                                        <option value="inactive">Inactive</option>
                                        <option value="suspended">Suspended</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="form-group">
                                    <label>Search Companies</label>
                                    <input type="text" id="searchCompanies" class="form-control" 
                                           placeholder="Search by name, code, or contact..." onkeyup="filterCompanies()">
                                </div>
                            </div>
                            <div class="col-md-2">
                                <div class="form-group">
                                    <label>&nbsp;</label>
                                    <div>
                                        <button type="button" class="btn btn-default btn-block" onclick="clearFilters()">
                                            <i class="fa fa-refresh"></i> Clear
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Companies Table -->
                        <div class="table-responsive">
                            <table class="table table-striped table-hover" id="companiesTable">
                                <thead>
                                    <tr>
                                        <th width="5%">
                                            <input type="checkbox" id="selectAll" onchange="toggleSelectAll()">
                                        </th>
                                        <th width="20%">Company Name</th>
                                        <th width="10%">Code</th>
                                        <th width="15%">Contact Person</th>
                                        <th width="12%">Phone</th>
                                        <th width="15%">Email</th>
                                        <th width="8%">Commission</th>
                                        <th width="8%">Status</th>
                                        <th width="7%">Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php foreach ($companies as $company): ?>
                                        <tr data-id="<?php echo $company['id']; ?>" class="company-row">
                                            <td>
                                                <input type="checkbox" class="company-checkbox" value="<?php echo $company['id']; ?>">
                                            </td>
                                            <td>
                                                <strong><?php echo htmlspecialchars($company['company_name']); ?></strong>
                                                <?php if (!empty($company['website'])): ?>
                                                    <br><a href="<?php echo $company['website']; ?>" target="_blank" class="text-muted">
                                                        <i class="fa fa-external-link"></i> Website
                                                    </a>
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="label label-default"><?php echo $company['company_code']; ?></span>
                                            </td>
                                            <td>
                                                <?php echo htmlspecialchars($company['contact_person'] ?? '-'); ?>
                                            </td>
                                            <td>
                                                <?php if (!empty($company['phone'])): ?>
                                                    <a href="tel:<?php echo $company['phone']; ?>">
                                                        <?php echo htmlspecialchars($company['phone']); ?>
                                                    </a>
                                                <?php else: ?>
                                                    -
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <?php if (!empty($company['email'])): ?>
                                                    <a href="mailto:<?php echo $company['email']; ?>">
                                                        <?php echo htmlspecialchars($company['email']); ?>
                                                    </a>
                                                <?php else: ?>
                                                    -
                                                <?php endif; ?>
                                            </td>
                                            <td>
                                                <span class="text-success">
                                                    <?php echo number_format($company['commission_rate'], 2); ?>%
                                                </span>
                                            </td>
                                            <td>
                                                <?php
                                                $status_class = '';
                                                switch ($company['status']) {
                                                    case 'active':
                                                        $status_class = 'success';
                                                        break;
                                                    case 'inactive':
                                                        $status_class = 'default';
                                                        break;
                                                    case 'suspended':
                                                        $status_class = 'danger';
                                                        break;
                                                }
                                                ?>
                                                <span class="label label-<?php echo $status_class; ?>">
                                                    <?php echo ucfirst($company['status']); ?>
                                                </span>
                                            </td>
                                            <td>
                                                <div class="btn-group">
                                                    <button type="button" class="btn btn-xs btn-info" 
                                                            onclick="viewCompany(<?php echo $company['id']; ?>)" title="View Details">
                                                        <i class="fa fa-eye"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-xs btn-warning" 
                                                            onclick="editCompany(<?php echo $company['id']; ?>)" title="Edit">
                                                        <i class="fa fa-edit"></i>
                                                    </button>
                                                    <button type="button" class="btn btn-xs btn-danger" 
                                                            onclick="deleteCompany(<?php echo $company['id']; ?>)" title="Delete">
                                                        <i class="fa fa-trash"></i>
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>

                        <!-- Bulk Actions -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="btn-group">
                                    <button type="button" class="btn btn-sm btn-success" onclick="bulkActivate()">
                                        <i class="fa fa-check"></i> Activate Selected
                                    </button>
                                    <button type="button" class="btn btn-sm btn-warning" onclick="bulkDeactivate()">
                                        <i class="fa fa-pause"></i> Deactivate Selected
                                    </button>
                                    <button type="button" class="btn btn-sm btn-danger" onclick="bulkSuspend()">
                                        <i class="fa fa-ban"></i> Suspend Selected
                                    </button>
                                    <button type="button" class="btn btn-sm btn-default" onclick="bulkDelete()">
                                        <i class="fa fa-trash"></i> Delete Selected
                                    </button>
                                </div>
                            </div>
                            <div class="col-md-6 text-right">
                                <small class="text-muted">
                                    Showing <?php echo count($companies); ?> companies
                                </small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Add/Edit Company Modal -->
<div class="modal fade" id="companyModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">&times;</button>
                <h4 class="modal-title">
                    <i class="fa fa-building"></i> 
                    <span id="modalTitle">Add New Insurance Company</span>
                </h4>
            </div>
            <form id="companyForm">
                <div class="modal-body">
                    <input type="hidden" id="companyId" name="company_id">
                    
                    <!-- Basic Information -->
                    <div class="row">
                        <div class="col-md-8">
                            <div class="form-group">
                                <label for="companyName">Company Name <span class="required">*</span></label>
                                <input type="text" id="companyName" name="company_name" class="form-control" 
                                       placeholder="Enter insurance company name" required>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="companyCode">Company Code</label>
                                <input type="text" id="companyCode" name="company_code" class="form-control" 
                                       placeholder="e.g., AAR" maxlength="10">
                                <small class="text-muted">Auto-generated if empty</small>
                            </div>
                        </div>
                    </div>

                    <!-- Contact Information -->
                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="contactPerson">Contact Person</label>
                                <input type="text" id="contactPerson" name="contact_person" class="form-control" 
                                       placeholder="Primary contact name">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="phone">Phone Number</label>
                                <input type="tel" id="phone" name="phone" class="form-control" 
                                       placeholder="+254 700 000 000">
                            </div>
                        </div>
                    </div>

                    <div class="row">
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="email">Email Address</label>
                                <input type="email" id="email" name="email" class="form-control" 
                                       placeholder="contact@insurance.com">
                            </div>
                        </div>
                        <div class="col-md-6">
                            <div class="form-group">
                                <label for="website">Website</label>
                                <input type="url" id="website" name="website" class="form-control" 
                                       placeholder="https://www.insurance.com">
                            </div>
                        </div>
                    </div>

                    <!-- Address -->
                    <div class="form-group">
                        <label for="address">Address</label>
                        <textarea id="address" name="address" class="form-control" rows="2" 
                                  placeholder="Company physical address"></textarea>
                    </div>

                    <!-- Business Information -->
                    <div class="row">
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="commissionRate">Commission Rate (%)</label>
                                <input type="number" id="commissionRate" name="commission_rate" 
                                       class="form-control" min="0" max="100" step="0.01" value="0.00">
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="status">Status</label>
                                <select id="status" name="status" class="form-control">
                                    <option value="active">Active</option>
                                    <option value="inactive">Inactive</option>
                                    <option value="suspended">Suspended</option>
                                </select>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="form-group">
                                <label for="logoUpload">Company Logo</label>
                                <input type="file" id="logoUpload" name="logo_file" class="form-control" 
                                       accept="image/jpeg,image/png,image/gif">
                                <small class="text-muted">Max 2MB, JPG/PNG/GIF</small>
                            </div>
                        </div>
                    </div>

                    <!-- Payment Terms -->
                    <div class="form-group">
                        <label for="paymentTerms">Payment Terms</label>
                        <textarea id="paymentTerms" name="payment_terms" class="form-control" rows="2" 
                                  placeholder="e.g., Net 30 days, 2% discount for early payment"></textarea>
                    </div>

                    <!-- Notes -->
                    <div class="form-group">
                        <label for="notes">Notes</label>
                        <textarea id="notes" name="notes" class="form-control" rows="2" 
                                  placeholder="Additional notes or special instructions"></textarea>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">
                        <i class="fa fa-times"></i> Cancel
                    </button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fa fa-save"></i> <span id="submitText">Save Company</span>
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<!-- View Company Details Modal -->
<div class="modal fade" id="viewCompanyModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">&times;</button>
                <h4 class="modal-title">
                    <i class="fa fa-building"></i> Company Details
                </h4>
            </div>
            <div class="modal-body" id="companyDetailsContent">
                <!-- Content will be loaded via AJAX -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                <button type="button" class="btn btn-warning" onclick="editFromView()">
                    <i class="fa fa-edit"></i> Edit Company
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Import Companies Modal -->
<div class="modal fade" id="importModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal">&times;</button>
                <h4 class="modal-title">
                    <i class="fa fa-upload"></i> Import Insurance Companies
                </h4>
            </div>
            <form id="importForm" enctype="multipart/form-data">
                <div class="modal-body">
                    <div class="form-group">
                        <label for="importFile">Select CSV File</label>
                        <input type="file" id="importFile" name="import_file" class="form-control" 
                               accept=".csv" required>
                        <small class="text-muted">
                            CSV format: Company Name, Company Code, Contact Person, Phone, Email, Website, Address
                        </small>
                    </div>
                    
                    <div class="checkbox">
                        <label>
                            <input type="checkbox" name="update_existing" value="1" checked>
                            Update existing companies with same name
                        </label>
                    </div>
                    
                    <div class="alert alert-info">
                        <i class="fa fa-info-circle"></i>
                        <strong>Expected CSV Format:</strong><br>
                        Company Name, Company Code, Contact Person, Phone, Email, Website, Address<br>
                        <a href="<?php echo base_url(); ?>admin/insurance/download_companies_template" target="_blank">
                            Download Template
                        </a>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary">
                        <i class="fa fa-upload"></i> Import Companies
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
$(document).ready(function() {
    // Initialize tooltips
    $('[title]').tooltip();
    
    // Auto-generate company code when typing company name
    $('#companyName').on('input', function() {
        if ($('#companyCode').val() === '') {
            const name = $(this).val();
            const code = generateCompanyCode(name);
            $('#companyCode').val(code);
        }
    });
});

// Global variables
let currentViewingCompanyId = null;

// Show add company modal
function showAddCompanyModal() {
    $('#modalTitle').text('Add New Insurance Company');
    $('#submitText').text('Save Company');
    $('#companyForm')[0].reset();
    $('#companyId').val('');
    $('#companyModal').modal('show');
}

// Edit company
function editCompany(companyId) {
    currentViewingCompanyId = companyId;
    
    $.ajax({
        url: '<?php echo base_url(); ?>admin/insurance/get_company/' + companyId,
        type: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                const company = response.data;
                
                $('#modalTitle').text('Edit Insurance Company');
                $('#submitText').text('Update Company');
                $('#companyId').val(company.id);
                $('#companyName').val(company.company_name);
                $('#companyCode').val(company.company_code);
                $('#contactPerson').val(company.contact_person);
                $('#phone').val(company.phone);
                $('#email').val(company.email);
                $('#website').val(company.website);
                $('#address').val(company.address);
                $('#commissionRate').val(company.commission_rate);
                $('#status').val(company.status);
                $('#paymentTerms').val(company.payment_terms);
                $('#notes').val(company.notes);
                
                $('#companyModal').modal('show');
            } else {
                showAlert('Error loading company data: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error occurred while loading company data.', 'error');
        }
    });
}

// View company details
function viewCompany(companyId) {
    currentViewingCompanyId = companyId;
    
    $.ajax({
        url: '<?php echo base_url(); ?>admin/insurance/get_company_details/' + companyId,
        type: 'GET',
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                displayCompanyDetails(response.data);
                $('#viewCompanyModal').modal('show');
            } else {
                showAlert('Error loading company details: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error occurred while loading company details.', 'error');
        }
    });
}

// Display company details in view modal
function displayCompanyDetails(company) {
    const statusBadge = company.status === 'active' ? 'success' : 
                       company.status === 'inactive' ? 'default' : 'danger';
    
    const html = `
        <div class="row">
            <div class="col-md-8">
                <h4>${company.company_name}</h4>
                <p class="text-muted">${company.company_code || 'No code assigned'}</p>
            </div>
            <div class="col-md-4 text-right">
                <span class="label label-${statusBadge}">${company.status.toUpperCase()}</span>
            </div>
        </div>
        
        <div class="row">
            <div class="col-md-6">
                <h5>Contact Information</h5>
                <table class="table table-condensed">
                    <tr><td><strong>Contact Person:</strong></td><td>${company.contact_person || '-'}</td></tr>
                    <tr><td><strong>Phone:</strong></td><td>${company.phone ? '<a href="tel:' + company.phone + '">' + company.phone + '</a>' : '-'}</td></tr>
                    <tr><td><strong>Email:</strong></td><td>${company.email ? '<a href="mailto:' + company.email + '">' + company.email + '</a>' : '-'}</td></tr>
                    <tr><td><strong>Website:</strong></td><td>${company.website ? '<a href="' + company.website + '" target="_blank">' + company.website + '</a>' : '-'}</td></tr>
                </table>
            </div>
            <div class="col-md-6">
                <h5>Business Information</h5>
                <table class="table table-condensed">
                    <tr><td><strong>Commission Rate:</strong></td><td>${parseFloat(company.commission_rate).toFixed(2)}%</td></tr>
                    <tr><td><strong>Created:</strong></td><td>${new Date(company.created_at).toLocaleDateString()}</td></tr>
                    <tr><td><strong>Last Updated:</strong></td><td>${new Date(company.updated_at).toLocaleDateString()}</td></tr>
                </table>
            </div>
        </div>
        
        ${company.address ? '<div class="row"><div class="col-md-12"><h5>Address</h5><p>' + company.address + '</p></div></div>' : ''}
        ${company.payment_terms ? '<div class="row"><div class="col-md-12"><h5>Payment Terms</h5><p>' + company.payment_terms + '</p></div></div>' : ''}
        ${company.notes ? '<div class="row"><div class="col-md-12"><h5>Notes</h5><p>' + company.notes + '</p></div></div>' : ''}
        
        <div class="row">
            <div class="col-md-12">
                <h5>Statistics</h5>
                <div class="row">
                    <div class="col-md-4">
                        <div class="info-box bg-aqua">
                            <span class="info-box-icon"><i class="fa fa-list"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Pricing Items</span>
                                <span class="info-box-number">${company.pricing_count || 0}</span>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="info-box bg-green">
                            <span class="info-box-icon"><i class="fa fa-money"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Avg Price</span>
                                <span class="info-box-number">KES ${parseFloat(company.avg_price || 0).toFixed(0)}</span>
                            </div>
                        </div>
                    </div>
                    <div class="col-md-4">
                        <div class="info-box bg-yellow">
                            <span class="info-box-icon"><i class="fa fa-calendar"></i></span>
                            <div class="info-box-content">
                                <span class="info-box-text">Last Import</span>
                                <span class="info-box-number">${company.last_import || 'Never'}</span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    $('#companyDetailsContent').html(html);
}

// Edit from view modal
function editFromView() {
    $('#viewCompanyModal').modal('hide');
    if (currentViewingCompanyId) {
        editCompany(currentViewingCompanyId);
    }
}

// Generate company code from name
function generateCompanyCode(name) {
    if (!name) return '';
    
    const words = name.toUpperCase().split(' ');
    let code = '';
    
    words.forEach(word => {
        if (word.length > 0) {
            code += word.charAt(0);
        }
    });
    
    return code.substring(0, 10);
}

// Form submission
$('#companyForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    const companyId = $('#companyId').val();
    const url = companyId ? 
        '<?php echo base_url(); ?>admin/insurance/update_company/' + companyId :
        '<?php echo base_url(); ?>admin/insurance/create_company';
    
    $.ajax({
        url: url,
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        dataType: 'json',
        success: function(response) {
            if (response.status === 'success') {
                $('#companyModal').modal('hide');
                showAlert(response.message, 'success');
                setTimeout(() => location.reload(), 1500);
            } else {
                showAlert('Error: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error occurred while saving company data.', 'error');
        }
    });
});

// Delete company
function deleteCompany(companyId) {
    if (!confirm('Are you sure you want to delete this insurance company? This action cannot be undone.')) {
        return;
    }
    
    $.ajax({
        url: '<?php echo base_url(); ?>admin/insurance/delete_company',
        type: 'POST',
        dataType: 'json',
        data: { id: companyId },
        success: function(response) {
            if (response.status === 'success') {
                showAlert(response.message, 'success');
                $(`tr[data-id="${companyId}"]`).fadeOut(300, function() {
                    $(this).remove();
                });
            } else {
                showAlert('Error: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error occurred while deleting company.', 'error');
        }
    });
}

// Filtering functions
function filterCompanies() {
    const status = $('#statusFilter').val().toLowerCase();
    const search = $('#searchCompanies').val().toLowerCase();
    
    $('.company-row').each(function() {
        const row = $(this);
        const companyName = row.find('td:nth-child(2)').text().toLowerCase();
        const companyCode = row.find('td:nth-child(3)').text().toLowerCase();
        const contactPerson = row.find('td:nth-child(4)').text().toLowerCase();
        const phone = row.find('td:nth-child(5)').text().toLowerCase();
        const email = row.find('td:nth-child(6)').text().toLowerCase();
        const rowStatus = row.find('td:nth-child(8) .label').text().toLowerCase();
        
        const matchesStatus = !status || rowStatus === status;
        const matchesSearch = !search || 
            companyName.includes(search) || 
            companyCode.includes(search) || 
            contactPerson.includes(search) ||
            phone.includes(search) ||
            email.includes(search);
        
        if (matchesStatus && matchesSearch) {
            row.show();
        } else {
            row.hide();
        }
    });
}

function clearFilters() {
    $('#statusFilter').val('');
    $('#searchCompanies').val('');
    $('.company-row').show();
}

// Select all functionality
function toggleSelectAll() {
    const selectAll = document.getElementById('selectAll');
    const checkboxes = document.querySelectorAll('.company-checkbox');
    
    checkboxes.forEach(checkbox => {
        if ($(checkbox).closest('tr').is(':visible')) {
            checkbox.checked = selectAll.checked;
        }
    });
}

// Get selected company IDs
function getSelectedCompanies() {
    const checkboxes = document.querySelectorAll('.company-checkbox:checked');
    return Array.from(checkboxes).map(cb => cb.value);
}

// Bulk operations
function bulkActivate() {
    const selectedCompanies = getSelectedCompanies();
    if (selectedCompanies.length === 0) {
        alert('Please select at least one company to activate.');
        return;
    }
    
    bulkUpdateStatus(selectedCompanies, 'active', 'activate');
}

function bulkDeactivate() {
    const selectedCompanies = getSelectedCompanies();
    if (selectedCompanies.length === 0) {
        alert('Please select at least one company to deactivate.');
        return;
    }
    
    bulkUpdateStatus(selectedCompanies, 'inactive', 'deactivate');
}

function bulkSuspend() {
    const selectedCompanies = getSelectedCompanies();
    if (selectedCompanies.length === 0) {
        alert('Please select at least one company to suspend.');
        return;
    }
    
    bulkUpdateStatus(selectedCompanies, 'suspended', 'suspend');
}

function bulkUpdateStatus(companyIds, status, action) {
    if (!confirm(`Are you sure you want to ${action} ${companyIds.length} selected companies?`)) {
        return;
    }
    
    $.ajax({
        url: '<?php echo base_url(); ?>admin/insurance/bulk_update_status',
        type: 'POST',
        dataType: 'json',
        data: {
            company_ids: companyIds,
            status: status
        },
        success: function(response) {
            if (response.status === 'success') {
                showAlert(response.message, 'success');
                setTimeout(() => location.reload(), 1500);
            } else {
                showAlert('Error: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error occurred during bulk operation.', 'error');
        }
    });
}

function bulkDelete() {
    const selectedCompanies = getSelectedCompanies();
    if (selectedCompanies.length === 0) {
        alert('Please select at least one company to delete.');
        return;
    }
    
    if (!confirm(`Are you sure you want to delete ${selectedCompanies.length} selected companies? This action cannot be undone.`)) {
        return;
    }
    
    $.ajax({
        url: '<?php echo base_url(); ?>admin/insurance/bulk_delete_companies',
        type: 'POST',
        dataType: 'json',
        data: {
            company_ids: selectedCompanies
        },
        success: function(response) {
            if (response.status === 'success') {
                showAlert(response.message, 'success');
                setTimeout(() => location.reload(), 1500);
            } else {
                showAlert('Error: ' + response.message, 'error');
            }
        },
        error: function() {
            showAlert('Error occurred during bulk deletion.', 'error');
        }
    });
}

// Import companies
function importCompanies() {
    $('#importModal').modal('show');
}

$('#importForm').on('submit', function(e) {
    e.preventDefault();
    
    const formData = new FormData(this);
    
    $.ajax({
        url: '<?php echo base_url(); ?>admin/insurance/import_companies',
        type: 'POST',
        data: formData,
        processData: false,
        contentType: false,
        dataType: 'json',
        success: function(response) {
            $('#importModal').modal('hide');
            if (response.status === 'success') {
                showAlert(`Import completed! ${response.imported_count} companies imported successfully.`, 'success');
                setTimeout(() => location.reload(), 2000);
            } else {
                showAlert('Import failed: ' + response.message, 'error');
            }
        },
        error: function() {
            $('#importModal').modal('hide');
            showAlert('Error occurred during import.', 'error');
        }
    });
});

// Export companies
function exportCompanies() {
    const status = $('#statusFilter').val();
    const search = $('#searchCompanies').val();
    
    let url = '<?php echo base_url(); ?>admin/insurance/export_companies?';
    if (status) url += 'status=' + encodeURIComponent(status) + '&';
    if (search) url += 'search=' + encodeURIComponent(search);
    
    window.open(url, '_blank');
    showAlert('Export started. Download will begin shortly.', 'info');
}

// Utility functions
function showAlert(message, type) {
    const alertClass = type === 'success' ? 'alert-success' : 
                     type === 'warning' ? 'alert-warning' : 
                     type === 'info' ? 'alert-info' : 'alert-danger';
    
    const alertHtml = `
        <div class="alert ${alertClass} alert-dismissible" style="position: fixed; top: 70px; right: 20px; z-index: 9999; max-width: 400px;">
            <button type="button" class="close" data-dismiss="alert">&times;</button>
            ${message}
        </div>
    `;
    
    $('body').append(alertHtml);
    
    setTimeout(() => {
        $('.alert').fadeOut();
    }, 5000);
}
</script>

<style>
.required {
    color: #dd4b39;
}

.company-row:hover {
    background-color: #f5f5f5;
}

.table-responsive {
    overflow-x: auto;
}

.btn-xs {
    padding: 1px 5px;
    font-size: 11px;
}

.info-box {
    margin-bottom: 15px;
}

.company-checkbox {
    margin: 0;
}

.label {
    font-size: 11px;
}

.modal-lg {
    width: 90%;
}

@media (max-width: 768px) {
    .table-responsive {
        border: none;
    }
    
    .btn-group {
        display: block;
    }
    
    .btn-group .btn {
        display: block;
        width: 100%;
        margin-bottom: 5px;
    }
    
    .box-tools {
        display: block;
        text-align: center;
        margin-top: 10px;
    }
    
    .small-box .inner h3 {
        font-size: 20px;
    }
}

.form-group label {
    font-weight: 600;
}

.modal-body {
    max-height: 70vh;
    overflow-y: auto;
}
</style>

<?php
/**
 * Additional Controller Methods for Company Management
 * Add these methods to your Insurance controller
 */

/*
// Add to Insurance.php controller

public function get_company($id) {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company = $this->Insurance_model->getInsuranceCompany($id);
        
        if ($company) {
            echo json_encode([
                'status' => 'success',
                'data' => $company
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Company not found'
            ]);
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function get_company_details($id) {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company = $this->Insurance_model->getCompanyWithStatistics($id);
        
        if ($company) {
            echo json_encode([
                'status' => 'success',
                'data' => $company
            ]);
        } else {
            echo json_encode([
                'status' => 'error',
                'message' => 'Company not found'
            ]);
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function create_company() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_add')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $this->form_validation->set_rules('company_name', 'Company Name', 'required|min_length[2]|max_length[255]');
        $this->form_validation->set_rules('company_code', 'Company Code', 'max_length[50]');
        $this->form_validation->set_rules('contact_person', 'Contact Person', 'max_length[255]');
        $this->form_validation->set_rules('phone', 'Phone', 'max_length[50]');
        $this->form_validation->set_rules('email', 'Email', 'valid_email|max_length[255]');

        if ($this->form_validation->run() === FALSE) {
            throw new Exception(validation_errors());
        }

        // Handle file upload
        $logo_path = $this->handleLogoUpload();

        $company_data = [
            'company_name' => $this->input->post('company_name'),
            'company_code' => $this->input->post('company_code') ?: $this->generateCompanyCode($this->input->post('company_name')),
            'contact_person' => $this->input->post('contact_person'),
            'phone' => $this->input->post('phone'),
            'email' => $this->input->post('email'),
            'website' => $this->input->post('website'),
            'address' => $this->input->post('address'),
            'commission_rate' => $this->input->post('commission_rate') ?: 0,
            'payment_terms' => $this->input->post('payment_terms'),
            'notes' => $this->input->post('notes'),
            'status' => $this->input->post('status') ?: 'active',
            'logo_path' => $logo_path,
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $this->session->userdata('admin_id')
        ];

        $company_id = $this->Insurance_model->createInsuranceCompany($company_data);

        if ($company_id) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Insurance company created successfully',
                'company_id' => $company_id
            ]);
        } else {
            throw new Exception('Failed to create company');
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function update_company($id) {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $this->form_validation->set_rules('company_name', 'Company Name', 'required|min_length[2]|max_length[255]');
        $this->form_validation->set_rules('company_code', 'Company Code', 'max_length[50]');
        $this->form_validation->set_rules('contact_person', 'Contact Person', 'max_length[255]');
        $this->form_validation->set_rules('phone', 'Phone', 'max_length[50]');
        $this->form_validation->set_rules('email', 'Email', 'valid_email|max_length[255]');

        if ($this->form_validation->run() === FALSE) {
            throw new Exception(validation_errors());
        }

        // Handle file upload
        $logo_path = $this->handleLogoUpload();

        $company_data = [
            'company_name' => $this->input->post('company_name'),
            'company_code' => $this->input->post('company_code'),
            'contact_person' => $this->input->post('contact_person'),
            'phone' => $this->input->post('phone'),
            'email' => $this->input->post('email'),
            'website' => $this->input->post('website'),
            'address' => $this->input->post('address'),
            'commission_rate' => $this->input->post('commission_rate') ?: 0,
            'payment_terms' => $this->input->post('payment_terms'),
            'notes' => $this->input->post('notes'),
            'status' => $this->input->post('status'),
            'updated_at' => date('Y-m-d H:i:s'),
            'updated_by' => $this->session->userdata('admin_id')
        ];

        // Only update logo path if new file uploaded
        if ($logo_path) {
            $company_data['logo_path'] = $logo_path;
        }

        if ($this->Insurance_model->updateInsuranceCompany($id, $company_data)) {
            echo json_encode([
                'status' => 'success',
                'message' => 'Insurance company updated successfully'
            ]);
        } else {
            throw new Exception('Failed to update company');
        }

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function bulk_update_status() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_edit')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company_ids = $this->input->post('company_ids');
        $status = $this->input->post('status');

        if (empty($company_ids) || !in_array($status, ['active', 'inactive', 'suspended'])) {
            throw new Exception('Invalid parameters');
        }

        $updated_count = $this->Insurance_model->bulkUpdateCompanyStatus($company_ids, $status, $this->session->userdata('admin_id'));

        echo json_encode([
            'status' => 'success',
            'message' => "{$updated_count} companies updated successfully"
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function bulk_delete_companies() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_delete')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        $company_ids = $this->input->post('company_ids');

        if (empty($company_ids)) {
            throw new Exception('No companies selected');
        }

        $deleted_count = $this->Insurance_model->bulkDeleteCompanies($company_ids);

        echo json_encode([
            'status' => 'success',
            'message' => "{$deleted_count} companies deleted successfully"
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function import_companies() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_add')) {
        access_denied();
    }

    header('Content-Type: application/json');

    try {
        if (!isset($_FILES['import_file']) || $_FILES['import_file']['error'] !== UPLOAD_ERR_OK) {
            throw new Exception('No file uploaded or upload error');
        }

        $file = $_FILES['import_file'];
        $update_existing = $this->input->post('update_existing') === '1';

        // Process CSV file
        $imported_count = $this->Insurance_model->importCompaniesFromCSV(
            $file['tmp_name'], 
            $update_existing, 
            $this->session->userdata('admin_id')
        );

        echo json_encode([
            'status' => 'success',
            'message' => 'Companies imported successfully',
            'imported_count' => $imported_count
        ]);

    } catch (Exception $e) {
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

public function export_companies() {
    if (!$this->rbac->hasPrivilege('insurance', 'can_view')) {
        access_denied();
    }

    $filters = [
        'status' => $this->input->get('status'),
        'search' => $this->input->get('search')
    ];

    $companies = $this->Insurance_model->getInsuranceCompanies('all', $filters);

    // Generate CSV
    $csv_data = "Company Name,Company Code,Contact Person,Phone,Email,Website,Address,Commission Rate,Payment Terms,Status,Created Date\n";
    
    foreach ($companies as $company) {
        $csv_data .= sprintf(
            '"%s","%s","%s","%s","%s","%s","%s",%s,"%s","%s","%s"' . "\n",
            str_replace('"', '""', $company['company_name']),
            $company['company_code'],
            str_replace('"', '""', $company['contact_person'] ?? ''),
            $company['phone'] ?? '',
            $company['email'] ?? '',
            $company['website'] ?? '',
            str_replace('"', '""', $company['address'] ?? ''),
            $company['commission_rate'],
            str_replace('"', '""', $company['payment_terms'] ?? ''),
            $company['status'],
            date('Y-m-d', strtotime($company['created_at']))
        );
    }

    $filename = 'insurance_companies_' . date('Ymd_His') . '.csv';
    
    force_download($filename, $csv_data);
}

public function download_companies_template() {
    $template_data = [
        ['Company Name', 'Company Code', 'Contact Person', 'Phone', 'Email', 'Website', 'Address'],
        ['Sample Insurance Ltd', 'SIL', 'John Doe', '+254700000000', 'contact@sample.com', 'https://sample.com', 'Nairobi, Kenya'],
        ['Example Coverage Co', 'ECC', 'Jane Smith', '+254711111111', 'info@example.co.ke', '', 'Mombasa, Kenya']
    ];

    $csv_content = '';
    foreach ($template_data as $row) {
        $csv_content .= implode(',', array_map(function($field) {
            return '"' . str_replace('"', '""', $field) . '"';
        }, $row)) . "\n";
    }

    force_download('insurance_companies_template.csv', $csv_content);
}

private function handleLogoUpload() {
    if (!isset($_FILES['logo_file']) || $_FILES['logo_file']['error'] === UPLOAD_ERR_NO_FILE) {
        return null;
    }

    $config['upload_path'] = './uploads/insurance_logos/';
    $config['allowed_types'] = 'gif|jpg|png|jpeg';
    $config['max_size'] = 2048; // 2MB
    $config['encrypt_name'] = TRUE;

    if (!is_dir($config['upload_path'])) {
        mkdir($config['upload_path'], 0777, true);
    }

    $this->load->library('upload', $config);

    if ($this->upload->do_upload('logo_file')) {
        $upload_data = $this->upload->data();
        return 'uploads/insurance_logos/' . $upload_data['file_name'];
    } else {
        throw new Exception('Logo upload failed: ' . $this->upload->display_errors());
    }
}

private function generateCompanyCode($company_name) {
    $words = explode(' ', strtoupper($company_name));
    $code = '';
    foreach ($words as $word) {
        if (!empty($word)) {
            $code .= substr($word, 0, 1);
        }
    }
    return substr($code, 0, 10);
}
*/

/**
 * Additional Model Methods for Company Management
 * Add these methods to your Insurance_model.php
 */

/*
// Add to Insurance_model.php

public function getInsuranceCompanies($status = 'active', $filters = []) {
    $this->db->select('*');
    $this->db->from('insurance_companies');
    
    if ($status !== 'all') {
        $this->db->where('status', $status);
    }
    
    // Apply filters
    if (!empty($filters['status'])) {
        $this->db->where('status', $filters['status']);
    }
    
    if (!empty($filters['search'])) {
        $this->db->group_start();
        $this->db->like('company_name', $filters['search']);
        $this->db->or_like('company_code', $filters['search']);
        $this->db->or_like('contact_person', $filters['search']);
        $this->db->or_like('phone', $filters['search']);
        $this->db->or_like('email', $filters['search']);
        $this->db->group_end();
    }
    
    $this->db->order_by('company_name', 'ASC');
    return $this->db->get()->result_array();
}

public function getCompanyWithStatistics($id) {
    // Get company data
    $company = $this->getInsuranceCompany($id);
    if (!$company) {
        return null;
    }
    
    // Get statistics
    $this->db->select('
        COUNT(*) as pricing_count,
        AVG(item_price) as avg_price,
        MAX(created_at) as last_import
    ');
    $this->db->from('insurance_pricing');
    $this->db->where('insurance_company_id', $id);
    $this->db->where('status', 'active');
    $stats = $this->db->get()->row_array();
    
    return array_merge($company, $stats);
}

public function bulkUpdateCompanyStatus($company_ids, $status, $user_id) {
    $data = [
        'status' => $status,
        'updated_at' => date('Y-m-d H:i:s'),
        'updated_by' => $user_id
    ];
    
    $this->db->where_in('id', $company_ids);
    $this->db->update('insurance_companies', $data);
    
    return $this->db->affected_rows();
}

public function bulkDeleteCompanies($company_ids) {
    // Check if any companies have pricing data
    $this->db->select('DISTINCT insurance_company_id');
    $this->db->from('insurance_pricing');
    $this->db->where_in('insurance_company_id', $company_ids);
    $companies_with_pricing = $this->db->get()->result_array();
    
    if (!empty($companies_with_pricing)) {
        $protected_ids = array_column($companies_with_pricing, 'insurance_company_id');
        $deletable_ids = array_diff($company_ids, $protected_ids);
        
        if (!empty($deletable_ids)) {
            $this->db->where_in('id', $deletable_ids);
            $this->db->delete('insurance_companies');
        }
        
        return count($deletable_ids);
    } else {
        $this->db->where_in('id', $company_ids);
        $this->db->delete('insurance_companies');
        
        return $this->db->affected_rows();
    }
}

public function importCompaniesFromCSV($file_path, $update_existing, $user_id) {
    $handle = fopen($file_path, 'r');
    if (!$handle) {
        throw new Exception('Could not open CSV file');
    }
    
    // Skip header row
    fgetcsv($handle);
    
    $imported_count = 0;
    
    while (($data = fgetcsv($handle)) !== FALSE) {
        if (count($data) < 2 || empty($data[0])) {
            continue; // Skip invalid rows
        }
        
        $company_data = [
            'company_name' => trim($data[0]),
            'company_code' => trim($data[1] ?? ''),
            'contact_person' => trim($data[2] ?? ''),
            'phone' => trim($data[3] ?? ''),
            'email' => trim($data[4] ?? ''),
            'website' => trim($data[5] ?? ''),
            'address' => trim($data[6] ?? ''),
            'status' => 'active',
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $user_id
        ];
        
        // Generate code if empty
        if (empty($company_data['company_code'])) {
            $company_data['company_code'] = $this->generateCompanyCode($company_data['company_name']);
        }
        
        // Check if company exists
        $existing = $this->getInsuranceCompanyByName($company_data['company_name']);
        
        if ($existing && $update_existing) {
            // Update existing
            unset($company_data['created_at']);
            unset($company_data['created_by']);
            $company_data['updated_at'] = date('Y-m-d H:i:s');
            $company_data['updated_by'] = $user_id;
            
            $this->updateInsuranceCompany($existing['id'], $company_data);
            $imported_count++;
        } elseif (!$existing) {
            // Create new
            $this->createInsuranceCompany($company_data);
            $imported_count++;
        }
    }
    
    fclose($handle);
    return $imported_count;
}
*/
?>