<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>P9 Tax Certificate Management</title>
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/all.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/css/dataTables.bootstrap.min.css" rel="stylesheet">
    <style>
        .content-wrapper {
            padding: 20px;
            background-color: #f4f4f4;
        }
        
        .box {
            background: #fff;
            border-radius: 3px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.12), 0 1px 2px rgba(0,0,0,0.24);
            margin-bottom: 20px;
        }
        
        .box-header {
            border-bottom: 1px solid #f4f4f4;
            padding: 10px 15px;
            background: #fafafa;
        }
        
        .box-body {
            padding: 15px;
        }
        
        .info-box {
            display: block;
            min-height: 90px;
            background: #fff;
            width: 100%;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
            border-radius: 2px;
            margin-bottom: 15px;
        }
        
        .info-box-icon {
            border-top-left-radius: 2px;
            border-top-right-radius: 0;
            border-bottom-right-radius: 0;
            border-bottom-left-radius: 2px;
            display: block;
            float: left;
            height: 90px;
            width: 90px;
            text-align: center;
            font-size: 45px;
            line-height: 90px;
            background: rgba(0,0,0,0.2);
        }
        
        .info-box-content {
            padding: 5px 10px;
            margin-left: 90px;
        }
        
        .info-box-more {
            padding: 10px;
            background-color: rgba(0,0,0,0.1);
            border-top: 1px solid rgba(0,0,0,0.1);
        }
        
        .bg-blue { background-color: #3c8dbc !important; color: white; }
        .bg-green { background-color: #00a65a !important; color: white; }
        .bg-yellow { background-color: #f39c12 !important; color: white; }
        .bg-red { background-color: #dd4b39 !important; color: white; }
        .bg-aqua { background-color: #00c0ef !important; color: white; }
        
        .small-box {
            border-radius: 2px;
            position: relative;
            display: block;
            margin-bottom: 20px;
            box-shadow: 0 1px 1px rgba(0,0,0,0.1);
        }
        
        .small-box > .inner {
            padding: 10px;
        }
        
        .small-box .icon {
            transition: all .3s linear;
            position: absolute;
            top: -10px;
            right: 10px;
            z-index: 0;
            font-size: 90px;
            color: rgba(0,0,0,0.15);
        }
        
        .small-box h3 {
            font-size: 38px;
            font-weight: bold;
            margin: 0 0 10px 0;
            white-space: nowrap;
            padding: 0;
        }
        
        .small-box p {
            font-size: 15px;
        }
        
        .progress {
            background-color: #f4f4f4;
            border-radius: 4px;
            height: 20px;
            margin-bottom: 5px;
        }
        
        .progress-bar {
            background-color: #fff;
            height: 100%;
            border-radius: 4px;
            transition: width 0.6s ease;
        }
        
        .btn-group-justified {
            display: table;
            width: 100%;
            table-layout: fixed;
            border-collapse: separate;
        }
        
        .btn-group-justified .btn-group {
            display: table-cell;
            float: none;
            width: 1%;
        }
        
        .btn-group-justified .btn {
            float: none;
            display: block;
            width: 100%;
        }
        
        .table-responsive {
            overflow-x: auto;
        }
        
        .p9-status-generated { color: #00a65a; }
        .p9-status-pending { color: #f39c12; }
        .p9-status-error { color: #dd4b39; }
        
        .validation-summary {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-bottom: 20px;
        }
        
        .validation-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin: 10px 0;
        }
        
        .validation-badge {
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 11px;
            font-weight: bold;
        }
        
        .validation-success { background-color: #28a745; }
        .validation-warning { background-color: #ffc107; color: #212529; }
        .validation-danger { background-color: #dc3545; }
        
        .kra-compliance {
            border-left: 4px solid #28a745;
            padding-left: 15px;
            margin: 15px 0;
        }
        
        .processing-indicator {
            text-align: center;
            padding: 40px;
        }
        
        .processing-indicator .spinner {
            font-size: 3em;
            animation: spin 1s infinite linear;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .staff-summary-card {
            background: white;
            border-radius: 8px;
            padding: 15px;
            margin-bottom: 15px;
            box-shadow: 0 2px 4px rgba(0,0,0,0.1);
        }
        
        .monthly-progress {
            background: linear-gradient(45deg, #1e3c72, #2a5298);
            color: white;
            padding: 15px;
            border-radius: 8px;
        }
        
        .generate-controls {
            background: #f8f9fa;
            border: 1px solid #dee2e6;
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 20px;
        }
        
        @media (max-width: 768px) {
            .btn-group-justified .btn-group {
                display: block;
                float: none;
                width: 100%;
                margin-bottom: 5px;
            }
            
            .small-box h3 {
                font-size: 24px;
            }
            
            .info-box-icon {
                font-size: 30px;
                line-height: 60px;
                height: 60px;
                width: 60px;
            }
            
            .info-box-content {
                margin-left: 60px;
            }
        }
    </style>
</head>
<body>
    <div class="content-wrapper">
        <section class="content-header">
            <h1>
                <i class="fa fa-file-text"></i> P9 Tax Certificates
                <small>Generate and manage P9 forms for KRA compliance</small>
            </h1>
            <ol class="breadcrumb">
                <li><a href="#"><i class="fa fa-dashboard"></i> Dashboard</a></li>
                <li><a href="#"><i class="fa fa-users"></i> HR</a></li>
                <li class="active">P9 Forms</li>
            </ol>
        </section>

        <section class="content">
            <!-- Quick Actions Row -->
            <div class="row">
                <div class="col-md-3">
                    <div class="info-box bg-blue">
                        <span class="info-box-icon"><i class="fa fa-file-pdf-o"></i></span>
                        <div class="info-box-content">
                            <span class="info-box-text">Current Year</span>
                            <span class="info-box-number" id="currentYear">2024</span>
                            <div class="progress">
                                <div class="progress-bar" id="yearProgress" style="width: 75%"></div>
                            </div>
                            <span class="progress-description" id="monthProgress">
                                Month 9 of 12 - September
                            </span>
                        </div>
                        <div class="info-box-more">
                            <button class="btn btn-primary btn-block" onclick="generateCurrentYearP9()">
                                <i class="fa fa-plus"></i> Generate Current Year P9s
                            </button>
                        </div>
                    </div>
                </div>

                <div class="col-md-9">
                    <div class="box box-primary">
                        <div class="box-header with-border">
                            <h3 class="box-title"><i class="fa fa-cogs"></i> P9 Generation Controls</h3>
                            <div class="box-tools pull-right">
                                <button type="button" class="btn btn-info btn-sm" onclick="showP9Analytics()">
                                    <i class="fa fa-bar-chart"></i> Analytics
                                </button>
                                <button type="button" class="btn btn-success btn-sm" onclick="exportForKRA()">
                                    <i class="fa fa-download"></i> KRA Export
                                </button>
                            </div>
                        </div>
                        <div class="box-body">
                            <form id="p9FilterForm">
                                <div class="row">
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Year <span class="text-red">*</span></label>
                                            <select class="form-control" name="year" id="filter_year" required>
                                                <option value="2024" selected>2024</option>
                                                <option value="2023">2023</option>
                                                <option value="2022">2022</option>
                                                <option value="2021">2021</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Department</label>
                                            <select class="form-control" name="department_id" id="filter_department">
                                                <option value="">All Departments</option>
                                                <option value="1">Medical</option>
                                                <option value="2">Nursing</option>
                                                <option value="3">Administration</option>
                                                <option value="4">Pharmacy</option>
                                                <option value="5">Laboratory</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>Action</label>
                                            <select class="form-control" id="bulk_action">
                                                <option value="">Select Action</option>
                                                <option value="generate_all">Generate All P9s</option>
                                                <option value="email_all">Email All P9s</option>
                                                <option value="download_all">Download All P9s</option>
                                                <option value="email_download">Email & Download</option>
                                            </select>
                                        </div>
                                    </div>
                                    <div class="col-md-3">
                                        <div class="form-group">
                                            <label>&nbsp;</label>
                                            <div>
                                                <button type="button" class="btn btn-primary btn-block" onclick="loadP9List()">
                                                    <i class="fa fa-search"></i> Load P9 List
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-12">
                                        <div class="btn-group btn-group-justified" role="group">
                                            <div class="btn-group" role="group">
                                                <button type="button" class="btn btn-success" onclick="executeBulkAction()">
                                                    <i class="fa fa-cogs"></i> Execute Action
                                                </button>
                                            </div>
                                            <div class="btn-group" role="group">
                                                <button type="button" class="btn btn-info" onclick="previewP9Generation()">
                                                    <i class="fa fa-eye"></i> Preview Generation
                                                </button>
                                            </div>
                                            <div class="btn-group" role="group">
                                                <button type="button" class="btn btn-warning" onclick="validateP9Data()">
                                                    <i class="fa fa-check-circle"></i> Validate Data
                                                </button>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                </div>
            </div>

            <!-- P9 List -->
            <div class="box box-default">
                <div class="box-header with-border">
                    <h3 class="box-title"><i class="fa fa-list"></i> P9 Forms List</h3>
                    <div class="box-tools pull-right">
                        <div id="p9-summary" class="text-info">
                            <small><strong>Total Staff: <span id="total-staff">0</span> | Total PAYE: KSh <span id="total-paye">0</span></strong></small>
                        </div>
                    </div>
                </div>
                <div class="box-body">
                    <div class="table-responsive">
                        <table class="table table-striped table-bordered table-hover" id="p9-table">
                            <thead>
                                <tr>
                                    <th>#</th>
                                    <th>Employee ID</th>
                                    <th>Name</th>
                                    <th>Department</th>
                                    <th>KRA PIN</th>
                                    <th>Total Gross</th>
                                    <th>Total PAYE</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody id="p9-table-body">
                                <!-- Sample Data -->
                                <tr>
                                    <td>1</td>
                                    <td>EMP001</td>
                                    <td>Dr. John Smith</td>
                                    <td>Medical</td>
                                    <td>A123456789B</td>
                                    <td>KSh 1,200,000.00</td>
                                    <td>KSh 180,000.00</td>
                                    <td><span class="label label-success">Generated</span></td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-info" onclick="viewP9Details(1, 2024)" title="View Details">
                                                <i class="fa fa-eye"></i>
                                            </button>
                                            <button class="btn btn-primary" onclick="downloadP9(1, 2024)" title="Download">
                                                <i class="fa fa-download"></i>
                                            </button>
                                            <button class="btn btn-success" onclick="emailP9(1, 2024)" title="Email">
                                                <i class="fa fa-envelope"></i>
                                            </button>
                                            <button class="btn btn-warning" onclick="regenerateP9(1, 2024)" title="Regenerate">
                                                <i class="fa fa-refresh"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <tr>
                                    <td>2</td>
                                    <td>EMP002</td>
                                    <td>Mary Johnson</td>
                                    <td>Nursing</td>
                                    <td>B987654321C</td>
                                    <td>KSh 800,000.00</td>
                                    <td>KSh 80,000.00</td>
                                    <td><span class="label label-warning">Pending</span></td>
                                    <td>
                                        <div class="btn-group btn-group-sm">
                                            <button class="btn btn-info" onclick="viewP9Details(2, 2024)" title="View Details">
                                                <i class="fa fa-eye"></i>
                                            </button>
                                            <button class="btn btn-primary" onclick="generateP9(2, 2024)" title="Generate">
                                                <i class="fa fa-plus"></i>
                                            </button>
                                            <button class="btn btn-success" onclick="emailP9(2, 2024)" title="Email" disabled>
                                                <i class="fa fa-envelope"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                                <tr>
                                    <td colspan="9" class="text-center" id="loading-row" style="display: none;">
                                        <div class="processing-indicator">
                                            <i class="fa fa-spinner spinner"></i>
                                            <p>Loading P9 data...</p>
                                        </div>
                                    </td>
                                </tr>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Status Display -->
            <div class="row">
                <div class="col-md-12">
                    <div class="box box-default" id="status-box" style="display: none;">
                        <div class="box-header with-border">
                            <h3 class="box-title">Processing Status</h3>
                            <div class="box-tools pull-right">
                                <button type="button" class="btn btn-box-tool" onclick="$('#status-box').hide()">
                                    <i class="fa fa-times"></i>
                                </button>
                            </div>
                        </div>
                        <div class="box-body" id="status-content">
                            <!-- Status content will be loaded here -->
                        </div>
                    </div>
                </div>
            </div>

            <!-- P9 Summary Cards -->
            <div class="row">
                <div class="col-md-3">
                    <div class="small-box bg-aqua">
                        <div class="inner">
                            <h3 id="summary-staff-count">156</h3>
                            <p>Total Staff</p>
                        </div>
                        <div class="icon">
                            <i class="fa fa-users"></i>
                        </div>
                    </div>
                </div>

                <div class="col-md-3">
                    <div class="small-box bg-green">
                        <div class="inner">
                            <h3>KSh <span id="summary-gross-total">12.5M</span></h3>
                            <p>Total Gross Pay</p>
                        </div>
                        <div class="icon">
                            <i class="fa fa-money"></i>
                        </div>
                    </div>
                </div>

                <div class="col-md-3">
                    <div class="small-box bg-yellow">
                        <div class="inner">
                            <h3>KSh <span id="summary-paye-total">1.8M</span></h3>
                            <p>Total PAYE Tax</p>
                        </div>
                        <div class="icon">
                            <i class="fa fa-calculator"></i>
                        </div>
                    </div>
                </div>

                <div class="col-md-3">
                    <div class="small-box bg-red">
                        <div class="inner">
                            <h3><span id="summary-generated">91</span>%</h3>
                            <p>P9s Generated</p>
                        </div>
                        <div class="icon">
                            <i class="fa fa-file-pdf-o"></i>
                        </div>
                    </div>
                </div>
            </div>

            <!-- KRA Compliance Section -->
            <div class="row">
                <div class="col-md-12">
                    <div class="box box-info">
                        <div class="box-header with-border">
                            <h3 class="box-title"><i class="fa fa-shield"></i> KRA Compliance Status</h3>
                        </div>
                        <div class="box-body">
                            <div class="kra-compliance">
                                <h5><i class="fa fa-check-circle text-success"></i> Compliance Checklist</h5>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="validation-item">
                                            <span>All staff have KRA PINs</span>
                                            <span class="validation-badge validation-success">✓ 156/156</span>
                                        </div>
                                        <div class="validation-item">
                                            <span>Payroll data complete</span>
                                            <span class="validation-badge validation-success">✓ 100%</span>
                                        </div>
                                        <div class="validation-item">
                                            <span>P9 forms generated</span>
                                            <span class="validation-badge validation-warning">⚠ 142/156</span>
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="validation-item">
                                            <span>Tax calculations verified</span>
                                            <span class="validation-badge validation-success">✓ Valid</span>
                                        </div>
                                        <div class="validation-item">
                                            <span>Annual returns filed</span>
                                            <span class="validation-badge validation-warning">⚠ Pending</span>
                                        </div>
                                        <div class="validation-item">
                                            <span>Deadline compliance</span>
                                            <span class="validation-badge validation-success">✓ On time</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </section>
    </div>

    <!-- Preview Modal -->
    <div class="modal fade" id="previewModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">P9 Generation Preview</h4>
                </div>
                <div class="modal-body" id="preview-content">
                    <!-- Preview content will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-success" onclick="proceedWithP9Generation()">
                        <i class="fa fa-check"></i> Proceed with Generation
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- P9 Details Modal -->
    <div class="modal fade" id="p9DetailsModal" tabindex="-1" role="dialog">
        <div class="modal-dialog modal-lg" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <button type="button" class="close" data-dismiss="modal">&times;</button>
                    <h4 class="modal-title">P9 Details</h4>
                </div>
                <div class="modal-body" id="p9-details-content">
                    <!-- P9 details content will be loaded here -->
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                    <button type="button" class="btn btn-primary" onclick="downloadP9FromModal()">
                        <i class="fa fa-download"></i> Download P9
                    </button>
                    <button type="button" class="btn btn-success" onclick="emailP9FromModal()">
                        <i class="fa fa-envelope"></i> Email P9
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Progress Modal -->
    <div class="modal fade" id="progressModal" tabindex="-1" role="dialog" data-backdrop="static">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h4 class="modal-title">Processing P9 Forms...</h4>
                </div>
                <div class="modal-body">
                    <div class="progress">
                        <div class="progress-bar progress-bar-striped active" id="progressBar" style="width: 0%"></div>
                    </div>
                    <p id="progressText">Initializing...</p>
                    <div id="progressDetails" class="text-muted" style="font-size: 12px;"></div>
                </div>
            </div>
        </div>
    </div>

    <!-- Scripts -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/twitter-bootstrap/3.4.1/js/bootstrap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/jquery.dataTables.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/dataTables.bootstrap.min.js"></script>

    <script>
        // Global variables
        var p9Table;
        var currentYear = 2024;
        var currentP9Data = null;

        $(document).ready(function() {
            // Initialize DataTable
            initializeP9Table();
            
            // Update current year and progress
            updateYearProgress();
            
            // Filter change events
            $('#filter_year, #filter_department').change(function() {
                loadP9List();
            });

            // Initial load
            loadP9List();
            
            console.log('P9 Tax Certificate Management System Initialized');
        });

        function initializeP9Table() {
            p9Table = $('#p9-table').DataTable({
                processing: true,
                serverSide: false, // Using client-side for demo
                responsive: true,
                order: [[1, 'asc']],
                language: {
                    processing: "Loading P9 data..."
                },
                columnDefs: [
                    { targets: [0, 8], orderable: false }
                ]
            });
        }

        function updateYearProgress() {
            var currentMonth = new Date().getMonth() + 1;
            var currentMonthName = new Date().toLocaleString('default', { month: 'long' });
            var progressPercent = (currentMonth / 12) * 100;
            
            $('#currentYear').text(currentYear);
            $('#yearProgress').css('width', progressPercent + '%');
            $('#monthProgress').text('Month ' + currentMonth + ' of 12 - ' + currentMonthName);
        }

        function loadP9List() {
            showProgress('Loading P9 data...', 30);
            
            setTimeout(function() {
                // Simulate loading P9 data
                updateP9Summary({
                    total_staff: 156,
                    total_paye: 1800000,
                    total_gross: 12500000,
                    generated_count: 142
                });
                
                hideProgress();
                showNotification('info', 'P9 data loaded successfully');
            }, 1000);
        }

        function updateP9Summary(data) {
            if (data) {
                $('#total-staff').text(data.total_staff || 0);
                $('#total-paye').text(numberFormat(data.total_paye || 0));
                $('#summary-staff-count').text(data.total_staff || 0);
                $('#summary-gross-total').text(formatCurrency(data.total_gross || 0));
                $('#summary-paye-total').text(formatCurrency(data.total_paye || 0));
                
                var generatedPercentage = data.total_staff > 0 ? 
                    Math.round((data.generated_count / data.total_staff) * 100) : 0;
                $('#summary-generated').text(generatedPercentage);
            }
        }

        function generateCurrentYearP9() {
            $('#filter_year').val(currentYear);
            $('#bulk_action').val('generate_all');
            executeBulkAction();
        }

        function executeBulkAction() {
            var action = $('#bulk_action').val();
            var year = $('#filter_year').val();
            var department_id = $('#filter_department').val();
            
            if (!action) {
                showNotification('warning', 'Please select an action to execute');
                return;
            }
            
            if (!year) {
                showNotification('warning', 'Please select a year');
                return;
            }

            var actionMessages = {
                'generate_all': 'Generate all P9 forms',
                'email_all': 'Email all P9 forms to staff',
                'download_all': 'Download all P9 forms as ZIP',
                'email_download': 'Email and download all P9 forms'
            };

            if (confirm('Are you sure you want to ' + actionMessages[action] + ' for ' + year + '?')) {
                var deliveryMethod = 'download';
                if (action === 'email_all') deliveryMethod = 'email';
                else if (action === 'email_download') deliveryMethod = 'both';

                showProgress('Processing P9 forms...', 0);
                
                // Simulate P9 processing
                var progress = 0;
                var interval = setInterval(function() {
                    progress += Math.random() * 15;
                    if (progress >= 100) {
                        progress = 100;
                        clearInterval(interval);
                        hideProgress();
                        displayP9Results({
                            total_staff: 156,
                            p9_generated: 156,
                            emails_sent: deliveryMethod === 'email' || deliveryMethod === 'both' ? 156 : 0,
                            errors: [],
                            success_list: [
                                {name: 'Dr. John Smith', action: 'P9 Generated & Emailed'},
                                {name: 'Mary Johnson', action: 'P9 Generated & Emailed'},
                                {name: 'James Wilson', action: 'P9 Generated & Emailed'}
                            ]
                        });
                        showNotification('success', 'P9 processing completed successfully!');
                        loadP9List();
                    }
                    updateProgress(progress, 'Processing staff ' + Math.ceil(progress/100 * 156) + ' of 156...');
                }, 300);
            }
        }

        function previewP9Generation() {
            var year = $('#filter_year').val();
            var department_id = $('#filter_department').val();
            
            if (!year) {
                showNotification('warning', 'Please select a year');
                return;
            }

            showProgress('Generating preview...', 50);
            
            setTimeout(function() {
                var preview = {
                    year: year,
                    total_staff: 156,
                    staff_with_data: 156,
                    total_gross: 12500000,
                    total_paye: 1800000,
                    staff_with_pins: 156,
                    missing_pins: 0,
                    complete_records: 156,
                    incomplete_records: 0,
                    issues: []
                };
                
                displayP9Preview(preview);
                hideProgress();
                $('#previewModal').modal('show');
            }, 800);
        }

        function displayP9Preview(preview) {
            var html = '<div class="row">';
            html += '<div class="col-md-6">';
            html += '<h4><i class="fa fa-info-circle"></i> Generation Summary</h4>';
            html += '<table class="table table-bordered">';
            html += '<tr><td><strong>Year:</strong></td><td>' + preview.year + '</td></tr>';
            html += '<tr><td><strong>Total Staff:</strong></td><td>' + preview.total_staff + '</td></tr>';
            html += '<tr><td><strong>Staff with Payroll Data:</strong></td><td>' + preview.staff_with_data + '</td></tr>';
            html += '<tr><td><strong>Estimated Total Gross:</strong></td><td>KSh ' + numberFormat(preview.total_gross) + '</td></tr>';
            html += '<tr><td><strong>Estimated Total PAYE:</strong></td><td>KSh ' + numberFormat(preview.total_paye) + '</td></tr>';
            html += '</table>';
            html += '</div>';
            
            html += '<div class="col-md-6">';
            html += '<h4><i class="fa fa-check-circle"></i> Validation Results</h4>';
            html += '<table class="table table-bordered">';
            html += '<tr><td><strong>Staff with KRA PINs:</strong></td><td>' + preview.staff_with_pins + '</td></tr>';
            html += '<tr><td><strong>Missing KRA PINs:</strong></td><td class="' + (preview.missing_pins > 0 ? 'text-danger' : 'text-success') + '">' + preview.missing_pins + '</td></tr>';
            html += '<tr><td><strong>Complete Records:</strong></td><td class="text-success">' + preview.complete_records + '</td></tr>';
            html += '<tr><td><strong>Incomplete Records:</strong></td><td class="' + (preview.incomplete_records > 0 ? 'text-warning' : 'text-success') + '">' + preview.incomplete_records + '</td></tr>';
            html += '</table>';
            html += '</div>';
            html += '</div>';

            if (preview.issues && preview.issues.length > 0) {
                html += '<div class="row">';
                html += '<div class="col-md-12">';
                html += '<h4 class="text-warning"><i class="fa fa-warning"></i> Issues Found</h4>';
                html += '<ul class="list-unstyled">';
                preview.issues.forEach(function(issue) {
                    html += '<li class="text-warning"><i class="fa fa-warning"></i> ' + issue + '</li>';
                });
                html += '</ul>';
                html += '</div>';
                html += '</div>';
            } else {
                html += '<div class="alert alert-success">';
                html += '<i class="fa fa-check-circle"></i> <strong>All validations passed!</strong> Ready to generate P9 forms.';
                html += '</div>';
            }

            $('#preview-content').html(html);
        }

        function validateP9Data() {
            var year = $('#filter_year').val();
            var department_id = $('#filter_department').val();
            
            showProgress('Validating P9 data...', 60);
            
            setTimeout(function() {
                var validation = {
                    errors: [] // No errors for demo
                };
                
                displayValidationResults(validation);
                hideProgress();
            }, 1200);
        }

        function displayValidationResults(validation) {
            var html = '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<h4><i class="fa fa-shield"></i> P9 Data Validation Results</h4>';
            
            if (validation.errors.length === 0) {
                html += '<div class="alert alert-success">';
                html += '<i class="fa fa-check-circle"></i> <strong>Validation Passed!</strong> All P9 data is valid and ready for generation';
                html += '</div>';
                
                html += '<div class="row">';
                html += '<div class="col-md-4"><div class="text-center"><h3 class="text-success">156</h3><p>Valid Records</p></div></div>';
                html += '<div class="col-md-4"><div class="text-center"><h3 class="text-success">100%</h3><p>Compliance Rate</p></div></div>';
                html += '<div class="col-md-4"><div class="text-center"><h3 class="text-success">0</h3><p>Issues Found</p></div></div>';
                html += '</div>';
            } else {
                html += '<div class="alert alert-warning">';
                html += '<i class="fa fa-warning"></i> ' + validation.errors.length + ' issues found that need attention';
                html += '</div>';
                
                html += '<div class="table-responsive">';
                html += '<table class="table table-striped">';
                html += '<thead><tr><th>Staff</th><th>Issue</th><th>Severity</th></tr></thead>';
                html += '<tbody>';
                validation.errors.forEach(function(error) {
                    html += '<tr>';
                    html += '<td>' + error.staff_name + '</td>';
                    html += '<td>' + error.issue + '</td>';
                    html += '<td><span class="label label-' + (error.severity === 'high' ? 'danger' : 'warning') + '">' + error.severity + '</span></td>';
                    html += '</tr>';
                });
                html += '</tbody></table>';
                html += '</div>';
            }
            
            html += '</div>';
            html += '</div>';
            
            showStatus(html, 'info');
        }

        function emailP9(staffId, year) {
            if (confirm('Send P9 form via email for ' + year + '?')) {
                showProgress('Sending email...', 70);
                
                setTimeout(function() {
                    hideProgress();
                    showNotification('success', 'P9 form emailed successfully to staff member');
                }, 1500);
            }
        }

        function downloadP9(staffId, year) {
            showProgress('Generating PDF...', 80);
            
            setTimeout(function() {
                hideProgress();
                showNotification('success', 'P9 form downloaded successfully');
                // In real implementation, trigger download
            }, 1000);
        }

        function generateP9(staffId, year) {
            showProgress('Generating P9 form...', 60);
            
            setTimeout(function() {
                hideProgress();
                showNotification('success', 'P9 form generated successfully');
                loadP9List();
            }, 1500);
        }

        function regenerateP9(staffId, year) {
            if (confirm('Regenerate P9 form for ' + year + '? This will overwrite the existing form.')) {
                showProgress('Regenerating P9 form...', 60);
                
                setTimeout(function() {
                    hideProgress();
                    showNotification('success', 'P9 form regenerated successfully');
                    loadP9List();
                }, 1500);
            }
        }

        function viewP9Details(staffId, year) {
            showProgress('Loading P9 details...', 40);
            
            setTimeout(function() {
                var p9Data = {
                    staff: {
                        name: 'John',
                        surname: 'Smith',
                        employee_id: 'EMP001',
                        department_name: 'Medical',
                        kra_pin: 'A123456789B'
                    },
                    totals: {
                        basic_salary: 1000000,
                        gross_pay: 1200000,
                        taxable_pay: 1150000,
                        paye_tax: 180000,
                        monthly_relief: 28800
                    }
                };
                
                displayP9Details(p9Data);
                currentP9Data = {staffId: staffId, year: year};
                hideProgress();
                $('#p9DetailsModal').modal('show');
            }, 800);
        }

        function displayP9Details(p9Data) {
            var staff = p9Data.staff;
            var totals = p9Data.totals;
            
            var html = '<div class="row">';
            html += '<div class="col-md-6">';
            html += '<h4><i class="fa fa-user"></i> Staff Information</h4>';
            html += '<table class="table table-bordered">';
            html += '<tr><td><strong>Name:</strong></td><td>' + staff.name + ' ' + staff.surname + '</td></tr>';
            html += '<tr><td><strong>Employee ID:</strong></td><td>' + staff.employee_id + '</td></tr>';
            html += '<tr><td><strong>Department:</strong></td><td>' + staff.department_name + '</td></tr>';
            html += '<tr><td><strong>KRA PIN:</strong></td><td>' + (staff.kra_pin || 'Not Set') + '</td></tr>';
            html += '</table>';
            html += '</div>';
            
            html += '<div class="col-md-6">';
            html += '<h4><i class="fa fa-calculator"></i> Annual Summary</h4>';
            html += '<table class="table table-bordered">';
            html += '<tr><td><strong>Total Basic Salary:</strong></td><td>KSh ' + numberFormat(totals.basic_salary) + '</td></tr>';
            html += '<tr><td><strong>Total Gross Pay:</strong></td><td>KSh ' + numberFormat(totals.gross_pay) + '</td></tr>';
            html += '<tr><td><strong>Total Taxable Pay:</strong></td><td>KSh ' + numberFormat(totals.taxable_pay) + '</td></tr>';
            html += '<tr><td><strong>Total PAYE Tax:</strong></td><td>KSh ' + numberFormat(totals.paye_tax) + '</td></tr>';
            html += '<tr><td><strong>Total Relief:</strong></td><td>KSh ' + numberFormat(totals.monthly_relief) + '</td></tr>';
            html += '</table>';
            html += '</div>';
            html += '</div>';

            // Add monthly breakdown
            html += '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<h4><i class="fa fa-calendar"></i> Monthly Breakdown</h4>';
            html += '<div class="table-responsive">';
            html += '<table class="table table-striped table-condensed">';
            html += '<thead><tr><th>Month</th><th>Basic Salary</th><th>Gross Pay</th><th>Taxable Pay</th><th>PAYE Tax</th><th>Net Pay</th></tr></thead>';
            html += '<tbody>';
            
            var months = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun', 'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec'];
            for (var i = 0; i < 12; i++) {
                var monthlyBasic = totals.basic_salary / 12;
                var monthlyGross = totals.gross_pay / 12;
                var monthlyTaxable = totals.taxable_pay / 12;
                var monthlyPaye = totals.paye_tax / 12;
                var monthlyNet = monthlyGross - monthlyPaye;
                
                html += '<tr>';
                html += '<td>' + months[i] + '</td>';
                html += '<td>KSh ' + numberFormat(monthlyBasic) + '</td>';
                html += '<td>KSh ' + numberFormat(monthlyGross) + '</td>';
                html += '<td>KSh ' + numberFormat(monthlyTaxable) + '</td>';
                html += '<td>KSh ' + numberFormat(monthlyPaye) + '</td>';
                html += '<td>KSh ' + numberFormat(monthlyNet) + '</td>';
                html += '</tr>';
            }
            
            html += '</tbody></table>';
            html += '</div>';
            html += '</div>';
            html += '</div>';

            $('#p9-details-content').html(html);
        }

        function proceedWithP9Generation() {
            $('#previewModal').modal('hide');
            executeBulkAction();
        }

        function downloadP9FromModal() {
            if (currentP9Data) {
                downloadP9(currentP9Data.staffId, currentP9Data.year);
            }
        }

        function emailP9FromModal() {
            if (currentP9Data) {
                emailP9(currentP9Data.staffId, currentP9Data.year);
            }
        }

        function showP9Analytics() {
            window.open('#', '_blank');
            showNotification('info', 'P9 Analytics page will open in a new tab');
        }

        function exportForKRA() {
            var year = $('#filter_year').val();
            var department_id = $('#filter_department').val();
            
            showProgress('Preparing KRA export...', 70);
            
            setTimeout(function() {
                hideProgress();
                showNotification('success', 'KRA export file downloaded successfully');
            }, 2000);
        }

        function displayP9Results(results) {
            var html = '<div class="row">';
            html += '<div class="col-md-12">';
            html += '<h4><i class="fa fa-check-circle"></i> P9 Generation Results</h4>';
            html += '<div class="row">';
            
            // Statistics
            html += '<div class="col-md-3"><div class="info-box bg-blue">';
            html += '<span class="info-box-icon"><i class="fa fa-users"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">Total Staff</span>';
            html += '<span class="info-box-number">' + results.total_staff + '</span>';
            html += '</div></div></div>';
            
            html += '<div class="col-md-3"><div class="info-box bg-green">';
            html += '<span class="info-box-icon"><i class="fa fa-file-pdf-o"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">P9s Generated</span>';
            html += '<span class="info-box-number">' + results.p9_generated + '</span>';
            html += '</div></div></div>';
            
            html += '<div class="col-md-3"><div class="info-box bg-yellow">';
            html += '<span class="info-box-icon"><i class="fa fa-envelope"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">Emails Sent</span>';
            html += '<span class="info-box-number">' + results.emails_sent + '</span>';
            html += '</div></div></div>';
            
            html += '<div class="col-md-3"><div class="info-box bg-red">';
            html += '<span class="info-box-icon"><i class="fa fa-exclamation-triangle"></i></span>';
            html += '<div class="info-box-content">';
            html += '<span class="info-box-text">Errors</span>';
            html += '<span class="info-box-number">' + results.errors.length + '</span>';
            html += '</div></div></div>';
            
            html += '</div>';
            
            // Success list
            if (results.success_list && results.success_list.length > 0) {
                html += '<h5><i class="fa fa-check"></i> Successfully Processed</h5>';
                html += '<div class="table-responsive">';
                html += '<table class="table table-striped table-sm">';
                html += '<thead><tr><th>Staff Name</th><th>Action</th></tr></thead>';
                html += '<tbody>';
                results.success_list.forEach(function(item) {
                    html += '<tr><td>' + item.name + '</td><td><span class="label label-success">' + item.action + '</span></td></tr>';
                });
                html += '</tbody></table>';
                html += '</div>';
            }
            
            // Errors
            if (results.errors.length > 0) {
                html += '<h5 class="text-red"><i class="fa fa-times"></i> Errors</h5>';
                html += '<ul class="list-unstyled">';
                results.errors.forEach(function(error) {
                    html += '<li class="text-red"><i class="fa fa-times"></i> ' + error + '</li>';
                });
                html += '</ul>';
            }
            
            html += '</div>';
            html += '</div>';
            
            showStatus(html, 'success');
        }

        function showStatus(content, type, loading = false) {
            var boxClass = 'box-' + type;
            if (type === 'danger') boxClass = 'box-danger';
            if (type === 'success') boxClass = 'box-success';
            if (type === 'info') boxClass = 'box-info';
            if (type === 'warning') boxClass = 'box-warning';
            
            $('#status-box').removeClass('box-default box-success box-info box-warning box-danger').addClass(boxClass);
            
            if (loading) {
                content = '<div class="text-center"><i class="fa fa-spinner fa-spin fa-2x"></i><br><br>' + content + '</div>';
            }
            
            $('#status-content').html(content);
            $('#status-box').show();
            
            // Auto-scroll to status box
            $('html, body').animate({
                scrollTop: $('#status-box').offset().top - 100
            }, 500);
        }

        function showProgress(text, percent) {
            $('#progressText').text(text);
            $('#progressBar').css('width', percent + '%');
            $('#progressDetails').text('');
            $('#progressModal').modal('show');
        }

        function updateProgress(percent, text) {
            $('#progressBar').css('width', percent + '%');
            $('#progressText').text(text);
        }

        function hideProgress() {
            $('#progressModal').modal('hide');
        }

        function numberFormat(number) {
            return parseFloat(number).toLocaleString('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
        }

        function formatCurrency(amount) {
            if (amount >= 1000000) {
                return (amount / 1000000).toFixed(1) + 'M';
            } else if (amount >= 1000) {
                return (amount / 1000).toFixed(1) + 'K';
            }
            return amount.toString();
        }

        function showNotification(type, message) {
            var alertClass = 'alert-info';
            if (type === 'success') alertClass = 'alert-success';
            else if (type === 'error') alertClass = 'alert-danger';
            else if (type === 'warning') alertClass = 'alert-warning';
            
            var notification = $('<div class="alert ' + alertClass + ' alert-dismissible" style="position: fixed; top: 20px; right: 20px; z-index: 9999; min-width: 300px;">' +
                '<button type="button" class="close" data-dismiss="alert">&times;</button>' +
                '<i class="fa fa-' + (type === 'success' ? 'check' : type === 'warning' ? 'warning' : type === 'error' ? 'times' : 'info') + '-circle"></i> ' +
                '<strong>' + type.charAt(0).toUpperCase() + type.slice(1) + ':</strong> ' + message +
                '</div>');
            
            $('body').append(notification);
            
            setTimeout(function() {
                notification.fadeOut(function() {
                    $(this).remove();
                });
            }, 5000);
        }

        // Initialize tooltips
        $(function () {
            $('[data-toggle="tooltip"]').tooltip();
        });
    </script>
</body>
</html>