<!DOCTYPE html>
<html>
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Journal Entries</title>
    <!-- AdminLTE styles - matching your system theme -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/AdminLTE.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/admin-lte/2.4.18/css/skins/_all-skins.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/4.7.0/css/font-awesome.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/css/dataTables.bootstrap.min.css">
    
    <!-- jQuery and Bootstrap JS -->
    <script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/bootstrap/3.4.1/js/bootstrap.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/jquery.dataTables.min.js"></script>
    <script src="https://cdnjs.cloudflare.com/ajax/libs/datatables/1.10.21/js/dataTables.bootstrap.min.js"></script>
</head>
<body class="hold-transition skin-blue sidebar-mini">

<div class="content-wrapper">
    <!-- Content Header (Page header) -->
    <section class="content-header">
        <h1>
            <i class="fa fa-book"></i> Journal Entries
            <small>Manual accounting entries and adjustments</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="#"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="#">Financial</a></li>
            <li class="active">Journal Entries</li>
        </ol>
    </section>

    <!-- Main content -->
    <section class="content">
        <!-- Flash Messages -->
        <div id="flashMessages"></div>

        <div class="row">
            <!-- Create Journal Entry Form -->
            <div class="col-md-5">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-plus"></i> Create Journal Entry</h3>
                    </div>
                    <form method="post" id="journalEntryForm">
                        <div class="box-body">
                            <div class="row">
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label for="reference_no">Reference No <span class="text-red">*</span></label>
                                        <input type="text" class="form-control" name="reference_no" id="reference_no" 
                                               value="JE-20250628140000" required>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="form-group">
                                        <label for="date">Date <span class="text-red">*</span></label>
                                        <input type="date" class="form-control" name="date" id="date" 
                                               value="2025-06-28" required>
                                    </div>
                                </div>
                            </div>

                            <div class="form-group">
                                <label for="description">Description <span class="text-red">*</span></label>
                                <textarea class="form-control" name="description" id="description" rows="2" 
                                          placeholder="Brief description of the journal entry" required></textarea>
                            </div>

                            <!-- Journal Lines -->
                            <div class="journal-lines">
                                <div class="form-group">
                                    <label>Journal Lines</label>
                                    <div class="table-responsive">
                                        <table class="table table-bordered" id="journalLinesTable">
                                            <thead>
                                                <tr class="bg-gray">
                                                    <th style="width: 35%;">Account</th>
                                                    <th style="width: 25%;">Debit</th>
                                                    <th style="width: 25%;">Credit</th>
                                                    <th style="width: 10%;">Action</th>
                                                </tr>
                                            </thead>
                                            <tbody id="journalLinesBody">
                                                <tr class="journal-line">
                                                    <td>
                                                        <select class="form-control account-select" name="account_id[]" required>
                                                            <option value="">Select Account</option>
                                                            <option value="1">1001 - Cash</option>
                                                            <option value="2">1002 - Bank</option>
                                                            <option value="3">2001 - Accounts Payable</option>
                                                            <option value="4">3001 - Revenue</option>
                                                            <option value="5">4001 - Expenses</option>
                                                        </select>
                                                    </td>
                                                    <td>
                                                        <input type="number" class="form-control debit-amount" name="debit_amount[]" 
                                                               step="0.01" min="0" placeholder="0.00">
                                                    </td>
                                                    <td>
                                                        <input type="number" class="form-control credit-amount" name="credit_amount[]" 
                                                               step="0.01" min="0" placeholder="0.00">
                                                    </td>
                                                    <td>
                                                        <button type="button" class="btn btn-sm btn-danger remove-line" disabled>
                                                            <i class="fa fa-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                                <tr class="journal-line">
                                                    <td>
                                                        <select class="form-control account-select" name="account_id[]" required>
                                                            <option value="">Select Account</option>
                                                            <option value="1">1001 - Cash</option>
                                                            <option value="2">1002 - Bank</option>
                                                            <option value="3">2001 - Accounts Payable</option>
                                                            <option value="4">3001 - Revenue</option>
                                                            <option value="5">4001 - Expenses</option>
                                                        </select>
                                                    </td>
                                                    <td>
                                                        <input type="number" class="form-control debit-amount" name="debit_amount[]" 
                                                               step="0.01" min="0" placeholder="0.00">
                                                    </td>
                                                    <td>
                                                        <input type="number" class="form-control credit-amount" name="credit_amount[]" 
                                                               step="0.01" min="0" placeholder="0.00">
                                                    </td>
                                                    <td>
                                                        <button type="button" class="btn btn-sm btn-danger remove-line">
                                                            <i class="fa fa-trash"></i>
                                                        </button>
                                                    </td>
                                                </tr>
                                            </tbody>
                                            <tfoot>
                                                <tr>
                                                    <td>
                                                        <button type="button" class="btn btn-sm btn-success" id="addLine">
                                                            <i class="fa fa-plus"></i> Add Line
                                                        </button>
                                                    </td>
                                                    <td>
                                                        <div class="total-debits">
                                                            <strong>Total: $<span id="totalDebits">0.00</span></strong>
                                                        </div>
                                                    </td>
                                                    <td>
                                                        <div class="total-credits">
                                                            <strong>Total: $<span id="totalCredits">0.00</span></strong>
                                                        </div>
                                                    </td>
                                                    <td></td>
                                                </tr>
                                            </tfoot>
                                        </table>
                                    </div>
                                </div>
                            </div>

                            <!-- Balance Check -->
                            <div class="balance-check">
                                <div class="alert alert-info" id="balanceAlert">
                                    <i class="fa fa-info-circle"></i> 
                                    <span id="balanceMessage">Enter amounts to check balance</span>
                                </div>
                            </div>
                        </div>
                        <div class="box-footer">
                            <button type="submit" name="submit" class="btn btn-primary" id="submitEntry" disabled>
                                <i class="fa fa-save"></i> Create Entry
                            </button>
                            <button type="button" class="btn btn-default" id="resetForm">
                                <i class="fa fa-refresh"></i> Reset
                            </button>
                            <button type="button" class="btn btn-info" id="previewEntry">
                                <i class="fa fa-eye"></i> Preview
                            </button>
                        </div>
                    </form>
                </div>

                <!-- Quick Templates -->
                <div class="box box-info">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-flash"></i> Quick Templates</h3>
                    </div>
                    <div class="box-body">
                        <button type="button" class="btn btn-default btn-sm btn-block template-btn" data-template="expense_accrual">
                            Expense Accrual
                        </button>
                        <button type="button" class="btn btn-default btn-sm btn-block template-btn" data-template="depreciation">
                            Depreciation Entry
                        </button>
                        <button type="button" class="btn btn-default btn-sm btn-block template-btn" data-template="prepaid_expense">
                            Prepaid Expense
                        </button>
                        <button type="button" class="btn btn-default btn-sm btn-block template-btn" data-template="bad_debt">
                            Bad Debt Write-off
                        </button>
                        <button type="button" class="btn btn-default btn-sm btn-block template-btn" data-template="inventory_adjustment">
                            Inventory Adjustment
                        </button>
                    </div>
                </div>
            </div>

            <!-- Journal Entries List -->
            <div class="col-md-7">
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-list"></i> Recent Journal Entries</h3>
                        <div class="box-tools pull-right">
                            <div class="btn-group">
                                <button type="button" class="btn btn-success btn-sm" id="exportEntries">
                                    <i class="fa fa-download"></i> Export
                                </button>
                                <button type="button" class="btn btn-default btn-sm" id="printEntries">
                                    <i class="fa fa-print"></i> Print
                                </button>
                            </div>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- No Records State -->
                        <div class="alert alert-info text-center" id="noRecordsAlert">
                            <h4><i class="icon fa fa-info"></i> No Journal Entries</h4>
                            <p>No journal entries found. Create your first entry using the form on the left.</p>
                        </div>
                        
                        <!-- Table (hidden initially until there are records) -->
                        <div class="table-responsive" id="entriesTableContainer" style="display: none;">
                            <table class="table table-bordered table-striped" id="entriesTable">
                                <thead>
                                    <tr>
                                        <th>Date</th>
                                        <th>Reference</th>
                                        <th>Description</th>
                                        <th class="text-right">Amount</th>
                                        <th>Status</th>
                                        <th>Created By</th>
                                        <th class="text-center">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="entriesTableBody">
                                    <!-- Records will be populated here dynamically -->
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Journal Entry Details Modal -->
<div class="modal fade" id="entryModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">Journal Entry Details</h4>
            </div>
            <div class="modal-body" id="entryDetails">
                <!-- Entry details will be loaded here -->
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                <button type="button" class="btn btn-primary" id="printEntry">
                    <i class="fa fa-print"></i> Print
                </button>
            </div>
        </div>
    </div>
</div>

<!-- Reverse Entry Modal -->
<div class="modal fade" id="reverseModal" tabindex="-1" role="dialog">
    <div class="modal-dialog" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">Reverse Journal Entry</h4>
            </div>
            <form id="reverseForm">
                <div class="modal-body">
                    <input type="hidden" id="reverse_entry_id" name="entry_id">
                    
                    <div class="alert alert-warning">
                        <i class="fa fa-warning"></i> 
                        <strong>Warning:</strong> This will create a reversing entry that cancels out the original entry.
                    </div>
                    
                    <div class="form-group">
                        <label>Original Entry</label>
                        <input type="text" class="form-control" id="reverse_reference" readonly>
                    </div>
                    
                    <div class="form-group">
                        <label for="reversal_reason">Reason for Reversal <span class="text-red">*</span></label>
                        <textarea class="form-control" name="reversal_reason" id="reversal_reason" rows="3" 
                                  placeholder="Please provide a reason for reversing this entry..." required></textarea>
                    </div>
                    
                    <div class="form-group">
                        <label for="reversal_date">Reversal Date <span class="text-red">*</span></label>
                        <input type="date" class="form-control" name="reversal_date" id="reversal_date" 
                               value="2025-06-28" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-default" data-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-danger">Reverse Entry</button>
                </div>
            </form>
        </div>
    </div>
</div>

<style>
/* Custom styles matching AdminLTE theme */
.journal-lines .table {
    margin-bottom: 0;
}

.account-select {
    font-size: 12px;
}

.debit-amount, .credit-amount {
    text-align: right;
    font-family: 'Courier New', monospace;
}

.total-debits, .total-credits {
    text-align: right;
    font-family: 'Courier New', monospace;
    color: #333;
}

.balance-check {
    margin-top: 15px;
}

.currency-amount {
    font-family: 'Courier New', monospace;
    font-weight: bold;
}

.journal-line:hover {
    background-color: #f9f9f9;
}

.template-btn {
    margin-bottom: 5px;
}

.bg-gray {
    background-color: #f4f4f4 !important;
}

.text-red {
    color: #dd4b39;
}

/* Validation error styles */
.form-group.has-error .form-control {
    border-color: #dd4b39;
    box-shadow: inset 0 1px 1px rgba(0, 0, 0, 0.075), 0 0 6px #ce8483;
}

.form-group.has-error .help-block {
    color: #dd4b39;
}

/* Print styles */
@media print {
    .box-tools,
    .btn,
    .breadcrumb,
    .modal {
        display: none !important;
    }
    
    .content-wrapper {
        margin: 0 !important;
        padding: 0 !important;
    }
    
    .box {
        box-shadow: none !important;
        border: 1px solid #ddd !important;
    }
    
    .table {
        font-size: 12px;
    }
    
    .col-md-5 {
        display: none !important;
    }
    
    .col-md-7 {
        width: 100% !important;
    }
}

/* Responsive adjustments */
@media (max-width: 768px) {
    .col-md-5, .col-md-7 {
        width: 100% !important;
        margin-bottom: 15px;
    }
}
</style>

<script>
$(document).ready(function() {
    // Initialize variables for data storage
    var journalEntries = JSON.parse(localStorage.getItem('journalEntries') || '[]');
    var accounts = [
        {id: 1, account_code: '1001', account_name: 'Cash', type: 'asset'},
        {id: 2, account_code: '1002', account_name: 'Bank Account', type: 'asset'},
        {id: 3, account_code: '1100', account_name: 'Accounts Receivable', type: 'asset'},
        {id: 4, account_code: '1200', account_name: 'Inventory', type: 'asset'},
        {id: 5, account_code: '1500', account_name: 'Equipment', type: 'asset'},
        {id: 6, account_code: '2001', account_name: 'Accounts Payable', type: 'liability'},
        {id: 7, account_code: '2100', account_name: 'Notes Payable', type: 'liability'},
        {id: 8, account_code: '3001', account_name: 'Owner Equity', type: 'equity'},
        {id: 9, account_code: '4001', account_name: 'Sales Revenue', type: 'revenue'},
        {id: 10, account_code: '5001', account_name: 'Cost of Goods Sold', type: 'expense'},
        {id: 11, account_code: '6001', account_name: 'Rent Expense', type: 'expense'},
        {id: 12, account_code: '6002', account_name: 'Utilities Expense', type: 'expense'},
        {id: 13, account_code: '6003', account_name: 'Office Supplies Expense', type: 'expense'},
        {id: 14, account_code: '6004', account_name: 'Depreciation Expense', type: 'expense'}
    ];
    var entryIdCounter = parseInt(localStorage.getItem('entryIdCounter') || '0');
    var currentEditingId = null;

    // Load initial data
    populateAccountDropdowns();
    loadJournalEntries();
    generateNewReference();
    updateTotals();

    // Function to save data to localStorage
    function saveToStorage() {
        localStorage.setItem('journalEntries', JSON.stringify(journalEntries));
        localStorage.setItem('entryIdCounter', entryIdCounter.toString());
    }

    // Function to get account by ID
    function getAccountById(id) {
        return accounts.find(account => account.id == id);
    }

    // Function to format currency
    function formatCurrency(amount) {
        return parseFloat(amount).toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 });
    }

    // Function to format date
    function formatDate(dateString) {
        return new Date(dateString).toLocaleDateString('en-GB', {
            day: '2-digit',
            month: 'short',
            year: 'numeric'
        });
    }

    // Function to populate account dropdowns
    function populateAccountDropdowns() {
        var accountOptions = '<option value="">Select Account</option>';
        accounts.forEach(function(account) {
            accountOptions += '<option value="' + account.id + '">' +
                            account.account_code + ' - ' + account.account_name + '</option>';
        });
        $('.account-select').html(accountOptions);
    }

    // Function to initialize DataTable
    function initializeDataTable() {
        if ($.fn.DataTable.isDataTable('#entriesTable')) {
            $('#entriesTable').DataTable().destroy();
        }

        if (journalEntries.length > 0) {
            $('#noRecordsAlert').hide();
            $('#entriesTableContainer').show();
            populateEntriesTable();
            $('#entriesTable').DataTable({
                "order": [[ 0, "desc" ]],
                "pageLength": 25,
                "responsive": true,
                "columnDefs": [
                    { "orderable": false, "targets": [6] }
                ]
            });
        } else {
            $('#noRecordsAlert').show();
            $('#entriesTableContainer').hide();
        }
    }

    // Function to populate entries table
    function populateEntriesTable() {
        var tbody = $('#entriesTableBody');
        tbody.empty();

        journalEntries.forEach(function(entry) {
            var statusClass = '';
            switch(entry.status) {
                case 'posted': statusClass = 'label-success'; break;
                case 'pending': statusClass = 'label-warning'; break;
                case 'draft': statusClass = 'label-default'; break;
                case 'reversed': statusClass = 'label-danger'; break;
                default: statusClass = 'label-default';
            }

            var actionsHtml = '<div class="btn-group">';
            actionsHtml += '<button type="button" class="btn btn-xs btn-info view-entry" data-id="' + entry.id + '" title="View Details"><i class="fa fa-eye"></i></button>';

            if (entry.status === 'posted' && !entry.source_module) {
                actionsHtml += '<button type="button" class="btn btn-xs btn-warning reverse-entry" data-id="' + entry.id + '" title="Reverse Entry"><i class="fa fa-undo"></i></button>';
            }

            if (entry.status === 'draft') {
                actionsHtml += '<button type="button" class="btn btn-xs btn-primary edit-entry" data-id="' + entry.id + '" title="Edit"><i class="fa fa-edit"></i></button>';
                actionsHtml += '<button type="button" class="btn btn-xs btn-danger delete-entry" data-id="' + entry.id + '" title="Delete"><i class="fa fa-trash"></i></button>';
            }

            actionsHtml += '</div>';

            var row = '<tr>' +
                '<td>' + formatDate(entry.date) + '</td>' +
                '<td><span class="label label-primary">' + entry.reference_no + '</span></td>' +
                '<td>' + entry.description +
                    (entry.source_module ? '<br><small class="text-muted"><i class="fa fa-tag"></i> ' + entry.source_module + '</small>' :
                     '<br><small class="text-muted"><i class="fa fa-tag"></i> Manual</small>') +
                '</td>' +
                '<td class="text-right"><span class="currency-amount">' + formatCurrency(entry.total_amount) + '</span></td>' +
                '<td><span class="label ' + statusClass + '">' + entry.status.charAt(0).toUpperCase() + entry.status.slice(1) + '</span></td>' +
                '<td>' + entry.created_by + '</td>' +
                '<td class="text-center">' + actionsHtml + '</td>' +
                '</tr>';

            tbody.append(row);
        });
    }

    // Function to load journal entries
    function loadJournalEntries() {
        initializeDataTable();
    }

    // Add new journal line
    $('#addLine').click(function() {
        var newLine = $('.journal-line').first().clone();
        newLine.find('select').val('');
        newLine.find('input').val('');
        newLine.find('.remove-line').prop('disabled', false);
        $('#journalLinesBody').append(newLine);
        updateTotals();
    });

    // Remove journal line
    $(document).on('click', '.remove-line', function() {
        if ($('.journal-line').length > 2) {
            $(this).closest('.journal-line').remove();
            updateTotals();
        }
    });

    // Update totals when amounts change
    $(document).on('input', '.debit-amount, .credit-amount', function() {
        var row = $(this).closest('tr');
        if ($(this).hasClass('debit-amount') && $(this).val()) {
            row.find('.credit-amount').val('');
        } else if ($(this).hasClass('credit-amount') && $(this).val()) {
            row.find('.debit-amount').val('');
        }
        updateTotals();
    });

    // Update totals when account selection or description changes
    $(document).on('change', '.account-select', updateTotals);
    $(document).on('input', '#description', updateTotals);

    // Update totals function
    function updateTotals() {
        var totalDebits = 0;
        var totalCredits = 0;
        var hasValidLines = false;
        var selectedAccounts = new Set();

        $('.journal-line').each(function() {
            var accountId = $(this).find('.account-select').val();
            var debitAmount = parseFloat($(this).find('.debit-amount').val()) || 0;
            var creditAmount = parseFloat($(this).find('.credit-amount').val()) || 0;

            if (accountId && (debitAmount > 0 || creditAmount > 0)) {
                hasValidLines = true;
                totalDebits += debitAmount;
                totalCredits += creditAmount;
                selectedAccounts.add(accountId);
            }
        });

        $('#totalDebits').text(totalDebits.toFixed(2));
        $('#totalCredits').text(totalCredits.toFixed(2));

        var difference = Math.abs(totalDebits - totalCredits);
        var isBalanced = difference < 0.01;
        var hasValidEntry = hasValidLines && totalDebits > 0 && totalCredits > 0;
        var hasDescription = $('#description').val().trim().length > 0;
        var hasUniqueAccounts = selectedAccounts.size === $('.journal-line').filter(function() {
            return $(this).find('.account-select').val() && (
                parseFloat($(this).find('.debit-amount').val()) > 0 ||
                parseFloat($(this).find('.credit-amount').val()) > 0
            );
        }).length;

        if (!hasValidLines || totalDebits === 0) {
            $('#balanceAlert').removeClass('alert-success alert-danger alert-warning').addClass('alert-info');
            $('#balanceMessage').text('Enter amounts and select accounts to check balance');
            $('#submitEntry').prop('disabled', true);
        } else if (!hasDescription) {
            $('#balanceAlert').removeClass('alert-success alert-danger alert-info').addClass('alert-warning');
            $('#balanceMessage').html('<i class="fa fa-exclamation-triangle"></i> Please enter a description');
            $('#submitEntry').prop('disabled', true);
        } else if (!hasUniqueAccounts) {
            $('#balanceAlert').removeClass('alert-success alert-danger alert-info').addClass('alert-warning');
            $('#balanceMessage').html('<i class="fa fa-exclamation-triangle"></i> Each line must use a unique account');
            $('#submitEntry').prop('disabled', true);
        } else if (isBalanced && hasValidEntry) {
            $('#balanceAlert').removeClass('alert-info alert-danger alert-warning').addClass('alert-success');
            $('#balanceMessage').html('<i class="fa fa-check"></i> Entry is balanced and ready to save');
            $('#submitEntry').prop('disabled', false);
        } else {
            $('#balanceAlert').removeClass('alert-info alert-success alert-warning').addClass('alert-danger');
            $('#balanceMessage').html('<i class="fa fa-exclamation-triangle"></i> Entry is not balanced (Difference: $' + difference.toFixed(2) + ')');
            $('#submitEntry').prop('disabled', true);
        }
    }

    // Form submission
    $('#journalEntryForm').on('submit', function(e) {
        e.preventDefault();

        var totalDebits = parseFloat($('#totalDebits').text()) || 0;
        var totalCredits = parseFloat($('#totalCredits').text()) || 0;
        var description = $('#description').val().trim();
        var referenceNo = $('#reference_no').val().trim();
        var date = $('#date').val();

        if (!description) {
            showAlert('danger', 'Please enter a description for the journal entry.');
            $('#description').focus();
            return;
        }

        if (!referenceNo) {
            showAlert('danger', 'Please enter a reference number.');
            $('#reference_no').focus();
            return;
        }

        if (!date) {
            showAlert('danger', 'Please select a date.');
            $('#date').focus();
            return;
        }

        if (Math.abs(totalDebits - totalCredits) > 0.01) {
            showAlert('danger', 'Journal entry is not balanced. Debits must equal credits.');
            return;
        }

        if (totalDebits === 0) {
            showAlert('danger', 'Please enter at least one debit and credit amount.');
            return;
        }

        var formData = {
            reference_no: referenceNo,
            date: date,
            description: description,
            lines: []
        };

        var validLinesCount = 0;
        var selectedAccounts = new Set();
        $('.journal-line').each(function() {
            var accountId = $(this).find('.account-select').val();
            var debitAmount = parseFloat($(this).find('.debit-amount').val()) || 0;
            var creditAmount = parseFloat($(this).find('.credit-amount').val()) || 0;

            if (accountId && (debitAmount > 0 || creditAmount > 0)) {
                var account = getAccountById(accountId);
                if (account) {
                    formData.lines.push({
                        account_id: accountId,
                        account_code: account.account_code,
                        account_name: account.account_name,
                        debit_amount: debitAmount,
                        credit_amount: creditAmount
                    });
                    validLinesCount++;
                    selectedAccounts.add(accountId);
                }
            }
        });

        if (validLinesCount < 2) {
            showAlert('danger', 'Journal entry must have at least 2 lines with accounts selected and amounts entered.');
            return;
        }

        if (selectedAccounts.size < formData.lines.length) {
            showAlert('danger', 'Duplicate accounts detected. Each line must use a unique account.');
            return;
        }

        if (!currentEditingId) {
            var duplicateRef = journalEntries.find(entry =>
                entry.reference_no.toLowerCase() === formData.reference_no.toLowerCase()
            );
            if (duplicateRef) {
                showAlert('danger', 'Reference number already exists. Please use a unique reference number.');
                $('#reference_no').focus();
                return;
            }
        }

        $('#submitEntry').prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> ' + (currentEditingId ? 'Updating...' : 'Creating...'));

        setTimeout(function() {
            try {
                var newEntry = {
                    id: currentEditingId || (++entryIdCounter),
                    reference_no: formData.reference_no,
                    date: formData.date,
                    description: formData.description,
                    lines: formData.lines,
                    total_amount: totalDebits,
                    status: 'posted',
                    created_by: 'Current User',
                    created_at: new Date().toISOString(),
                    source_module: null
                };

                if (currentEditingId) {
                    var index = journalEntries.findIndex(entry => entry.id === currentEditingId);
                    if (index !== -1) {
                        journalEntries[index] = newEntry;
                        showAlert('success', 'Journal entry updated successfully!');
                    }
                    currentEditingId = null;
                } else {
                    journalEntries.unshift(newEntry);
                    showAlert('success', 'Journal entry created successfully! Reference: ' + formData.reference_no);
                }

                saveToStorage();
                resetForm();
                loadJournalEntries();

            } catch (error) {
                showAlert('danger', 'An error occurred while saving the journal entry: ' + error.message);
                console.error('Error saving journal entry:', error);
            }

            $('#submitEntry').prop('disabled', false).html('<i class="fa fa-save"></i> Create Entry');
        }, 1000);
    });

    // View entry details
    $(document).on('click', '.view-entry', function() {
        var entryId = parseInt($(this).data('id'));
        var entry = journalEntries.find(e => e.id === entryId);

        if (entry) {
            var detailsHtml = '<div class="journal-preview">';
            detailsHtml += '<h4>Journal Entry Details</h4>';
            detailsHtml += '<table class="table table-bordered">';
            detailsHtml += '<tr><th style="width:30%">Date:</th><td>' + formatDate(entry.date) + '</td></tr>';
            detailsHtml += '<tr><th>Reference:</th><td>' + entry.reference_no + '</td></tr>';
            detailsHtml += '<tr><th>Description:</th><td>' + entry.description + '</td></tr>';
            detailsHtml += '<tr><th>Status:</th><td><span class="label label-success">' + entry.status.charAt(0).toUpperCase() + entry.status.slice(1) + '</span></td></tr>';
            detailsHtml += '<tr><th>Created By:</th><td>' + entry.created_by + '</td></tr>';
            detailsHtml += '<tr><th>Created At:</th><td>' + formatDate(entry.created_at) + '</td></tr>';
            detailsHtml += '</table>';

            detailsHtml += '<h5>Journal Lines</h5>';
            detailsHtml += '<table class="table table-bordered">';
            detailsHtml += '<thead><tr><th>Account</th><th class="text-right">Debit</th><th class="text-right">Credit</th></tr></thead>';
            detailsHtml += '<tbody>';

            entry.lines.forEach(function(line) {
                detailsHtml += '<tr>';
                detailsHtml += '<td>' + line.account_code + ' - ' + line.account_name + '</td>';
                detailsHtml += '<td class="text-right">' + (line.debit_amount > 0 ? formatCurrency(line.debit_amount) : '-') + '</td>';
                detailsHtml += '<td class="text-right">' + (line.credit_amount > 0 ? formatCurrency(line.credit_amount) : '-') + '</td>';
                detailsHtml += '</tr>';
            });

            detailsHtml += '</tbody>';
            detailsHtml += '<tfoot>';
            detailsHtml += '<tr><th>Total</th><th class="text-right">' + formatCurrency(entry.total_amount) + '</th><th class="text-right">' + formatCurrency(entry.total_amount) + '</th></tr>';
            detailsHtml += '</tfoot>';
            detailsHtml += '</table>';
            detailsHtml += '</div>';

            $('#entryDetails').html(detailsHtml);
            $('#entryModal').modal('show');
        }
    });

    // Edit entry
    $(document).on('click', '.edit-entry', function() {
        var entryId = parseInt($(this).data('id'));
        var entry = journalEntries.find(e => e.id === entryId);

        if (entry) {
            currentEditingId = entryId;
            $('#reference_no').val(entry.reference_no);
            $('#date').val(entry.date);
            $('#description').val(entry.description);

            $('.journal-line').slice(2).remove();
            entry.lines.forEach(function(line, index) {
                if (index >= $('.journal-line').length) {
                    $('#addLine').click();
                }
                var lineRow = $('.journal-line').eq(index);
                lineRow.find('.account-select').val(line.account_id);
                lineRow.find('.debit-amount').val(line.debit_amount || '');
                lineRow.find('.credit-amount').val(line.credit_amount || '');
            });

            updateTotals();
            $('#submitEntry').html('<i class="fa fa-save"></i> Update Entry');
            showAlert('info', 'Entry loaded for editing. Make your changes and click Update Entry.');
            $('html, body').animate({
                scrollTop: $('#journalEntryForm').offset().top - 100
            }, 500);
        }
    });

    // Reverse entry
    $(document).on('click', '.reverse-entry', function() {
        var entryId = parseInt($(this).data('id'));
        var entry = journalEntries.find(e => e.id === entryId);

        if (entry) {
            $('#reverse_entry_id').val(entryId);
            $('#reverse_reference').val(entry.reference_no);
            $('#reverseModal').modal('show');
        }
    });

    // Reverse form submission
    $('#reverseForm').on('submit', function(e) {
        e.preventDefault();

        if (!confirm('Are you sure you want to reverse this journal entry?')) {
            return;
        }

        var entryId = parseInt($('#reverse_entry_id').val());
        var reason = $('#reversal_reason').val();
        var reversalDate = $('#reversal_date').val();
        var originalEntry = journalEntries.find(e => e.id === entryId);

        if (originalEntry) {
            var reversingEntry = {
                id: ++entryIdCounter,
                reference_no: 'REV-' + originalEntry.reference_no,
                date: reversalDate,
                description: 'REVERSAL: ' + originalEntry.description + ' - ' + reason,
                lines: originalEntry.lines.map(line => ({
                    account_id: line.account_id,
                    account_code: line.account_code,
                    account_name: line.account_name,
                    debit_amount: line.credit_amount,
                    credit_amount: line.debit_amount
                })),
                total_amount: originalEntry.total_amount,
                status: 'posted',
                created_by: 'Current User',
                created_at: new Date().toISOString(),
                source_module: 'reversal',
                original_entry_id: entryId
            };

            originalEntry.status = 'reversed';
            journalEntries.unshift(reversingEntry);
            saveToStorage();
            $('#reverseModal').modal('hide');
            showAlert('success', 'Journal entry reversed successfully!');
            loadJournalEntries();
        }
    });

    // Delete entry
    $(document).on('click', '.delete-entry', function() {
        var entryId = parseInt($(this).data('id'));

        if (confirm('Are you sure you want to delete this journal entry?')) {
            var index = journalEntries.findIndex(entry => entry.id === entryId);
            if (index !== -1) {
                journalEntries.splice(index, 1);
                saveToStorage();
                showAlert('success', 'Journal entry deleted successfully!');
                loadJournalEntries();
            }
        }
    });

    // Reset form function
    function resetForm() {
        $('#journalEntryForm')[0].reset();
        generateNewReference();
        $('.journal-line').slice(2).remove();
        $('.journal-line').find('select').val('');
        $('.journal-line').find('input').val('');
        $('.journal-line').first().find('.remove-line').prop('disabled', true);
        $('.journal-line').last().find('.remove-line').prop('disabled', false);
        currentEditingId = null;
        $('#submitEntry').html('<i class="fa fa-save"></i> Create Entry');
        updateTotals();
    }

    // Template buttons
    $('.template-btn').click(function() {
        var template = $(this).data('template');
        loadTemplate(template);
    });

    // Reset form
    $('#resetForm').click(function() {
        resetForm();
    });

    // Preview entry
    $('#previewEntry').click(function() {
        var preview = generatePreview();
        $('#entryDetails').html(preview);
        $('.modal-title').text('Journal Entry Preview');
        $('#entryModal').modal('show');
    });

    // Export entries
    $('#exportEntries').click(function() {
        exportToCSV();
    });

    // Print entries
    $('#printEntries').click(function() {
        window.print();
    });

    // Function to export to CSV
    function exportToCSV() {
        if (journalEntries.length === 0) {
            showAlert('warning', 'No entries to export.');
            return;
        }

        var csv = 'Date,Reference,Description,Amount,Status,Created By\n';
        journalEntries.forEach(function(entry) {
            csv += '"' + entry.date + '","' + entry.reference_no + '","' + entry.description.replace(/"/g, '""') + '","' + entry.total_amount + '","' + entry.status + '","' + entry.created_by + '"\n';
        });

        var blob = new Blob([csv], { type: 'text/csv' });
        var url = window.URL.createObjectURL(blob);
        var a = document.createElement('a');
        a.href = url;
        a.download = 'journal_entries_' + new Date().toISOString().slice(0, 10) + '.csv';
        document.body.appendChild(a);
        a.click();
        document.body.removeChild(a);
        window.URL.revokeObjectURL(url);
        showAlert('success', 'Journal entries exported successfully!');
    }

    // Generate new reference number
    function generateNewReference() {
        var timestamp = new Date().toISOString().replace(/[-:T]/g, '').slice(0, 14);
        $('#reference_no').val('JE-' + timestamp);
    }

    // Load template
    function loadTemplate(templateType) {
        var templates = {
            'expense_accrual': 'Accrual of expenses',
            'depreciation': 'Monthly depreciation expense',
            'prepaid_expense': 'Prepaid expense adjustment',
            'bad_debt': 'Bad debt write-off',
            'inventory_adjustment': 'Inventory quantity adjustment'
        };

        if (templates[templateType]) {
            $('#description').val(templates[templateType]);
            showAlert('info', 'Template loaded. Please select accounts and enter amounts.');
            updateTotals();
        }
    }

    // Generate preview
    function generatePreview() {
        var preview = '<div class="journal-preview">';
        preview += '<h4>Journal Entry Preview</h4>';
        preview += '<table class="table table-bordered">';
        preview += '<tr><th>Date:</th><td>' + ($('#date').val() || '-') + '</td></tr>';
        preview += '<tr><th>Reference:</th><td>' + ($('#reference_no').val() || '-') + '</td></tr>';
        preview += '<tr><th>Description:</th><td>' + ($('#description').val() || '-') + '</td></tr>';
        preview += '</table>';

        preview += '<h5>Journal Lines</h5>';
        preview += '<table class="table table-bordered">';
        preview += '<thead><tr><th>Account</th><th class="text-right">Debit</th><th class="text-right">Credit</th></tr></thead>';
        preview += '<tbody>';

        var totalDebits = 0, totalCredits = 0;

        $('.journal-line').each(function(index) {
            var accountText = $(this).find('.account-select option:selected').text();
            var debit = parseFloat($(this).find('.debit-amount').val()) || 0;
            var credit = parseFloat($(this).find('.credit-amount').val()) || 0;

            if (accountText !== 'Select Account' && (debit > 0 || credit > 0)) {
                preview += '<tr>';
                preview += '<td>' + accountText + '</td>';
                preview += '<td class="text-right">' + (debit > 0 ? formatCurrency(debit) : '-') + '</td>';
                preview += '<td class="text-right">' + (credit > 0 ? formatCurrency(credit) : '-') + '</td>';
                preview += '</tr>';

                totalDebits += debit;
                totalCredits += credit;
            }
        });

        preview += '</tbody>';
        preview += '<tfoot>';
        preview += '<tr><th>Total</th><th class="text-right">' + formatCurrency(totalDebits) + '</th><th class="text-right">' + formatCurrency(totalCredits) + '</th></tr>';
        preview += '</tfoot>';
        preview += '</table>';
        preview += '</div>';

        return preview;
    }

    // Show alert
    function showAlert(type, message) {
        var alertHtml = '<div class="alert alert-' + type + ' alert-dismissible">' +
                       '<button type="button" class="close" data-dismiss="alert">×</button>' +
                       '<i class="fa fa-' + (type === 'success' ? 'check' : type === 'danger' ? 'exclamation-triangle' : 'info') + '"></i> ' +
                       message +
                       '</div>';

        $('#flashMessages').html(alertHtml);

        setTimeout(function() {
            $('.alert').fadeOut();
        }, 5000);
    }

    // Keyboard shortcuts
    $(document).keydown(function(e) {
        if (e.ctrlKey && e.keyCode === 78) {
            e.preventDefault();
            $('#reference_no').focus();
        }

        if (e.ctrlKey && e.keyCode === 76) {
            e.preventDefault();
            $('#addLine').click();
        }

        if (e.ctrlKey && e.keyCode === 83) {
            e.preventDefault();
            if (!$('#submitEntry').prop('disabled')) {
                $('#submitEntry').click();
            }
        }
    });
});
</script>

</body>
</html>