<?php $currency_symbol = $this->customlib->getHospitalCurrencyFormat(); ?>

<div class="content-wrapper">
    <section class="content-header">
        <h1>
            <i class="fa fa-medkit"></i> Medicine Dispensing & Billing
            <small>Process patient medicine dispensing with walk-in support</small>
        </h1>
        <ol class="breadcrumb">
            <li><a href="<?php echo base_url(); ?>admin/admin/dashboard"><i class="fa fa-dashboard"></i> Home</a></li>
            <li><a href="<?php echo base_url(); ?>admin/dispensary">Medical Dispensary</a></li>
            <li class="active">Dispensing</li>
        </ol>
    </section>

    <section class="content">
        <!-- Alert Messages -->
        <div id="alertContainer" style="display: none;">
            <div class="alert alert-dismissible" id="alertBox">
                <button type="button" class="close" data-dismiss="alert" aria-hidden="true">&times;</button>
                <span id="alertMessage"></span>
            </div>
        </div>

        <!-- Dispensing Process Steps -->
        <div class="box box-info">
            <div class="box-header with-border">
                <h3 class="box-title"><i class="fa fa-list-ol"></i> Dispensing Process</h3>
            </div>
            <div class="box-body">
                <div class="row">
                    <div class="col-md-12">
                        <div class="dispensing-workflow">
                            <div class="workflow-step active" id="step1">
                                <div class="step-number">1</div>
                                <div class="step-content">
                                    <h6><i class="fa fa-user-plus"></i> Customer Type</h6>
                                    <small>Select registered patient or walk-in customer</small>
                                </div>
                            </div>
                            <div class="workflow-step" id="step2">
                                <div class="step-number">2</div>
                                <div class="step-content">
                                    <h6><i class="fa fa-search"></i> Customer Details</h6>
                                    <small>Search patient or enter walk-in details</small>
                                </div>
                            </div>
                            <div class="workflow-step" id="step3">
                                <div class="step-number">3</div>
                                <div class="step-content">
                                    <h6><i class="fa fa-plus"></i> Add Medicines</h6>
                                    <small>Select medicines from prescriptions or add new</small>
                                </div>
                            </div>
                            <div class="workflow-step" id="step4">
                                <div class="step-number">4</div>
                                <div class="step-content">
                                    <h6><i class="fa fa-calculator"></i> Calculate Bill</h6>
                                    <small>Review items, apply discounts, and calculate total</small>
                                </div>
                            </div>
                            <div class="workflow-step" id="step5">
                                <div class="step-number">5</div>
                                <div class="step-content">
                                    <h6><i class="fa fa-check-circle"></i> Process Payment</h6>
                                    <small>Complete transaction and print receipt</small>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="row">
            <!-- Customer Selection Section -->
            <div class="col-md-5">
                <div class="box box-primary">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-users"></i> Customer Selection</h3>
                        <div class="box-tools pull-right">
                            <span class="label label-primary">Step 1</span>
                        </div>
                    </div>
                    <div class="box-body">
                        <!-- Customer Type Selection -->
                        <div id="customerTypeSection">
                            <div class="form-group">
                                <label>Customer Type</label>
                                <div class="row">
                                    <div class="col-md-6">
                                        <button type="button" class="btn btn-primary btn-block customer-type-btn" 
                                                data-type="registered" onclick="selectCustomerType('registered')">
                                            <i class="fa fa-user"></i><br>
                                            <strong>Registered Patient</strong><br>
                                            <small>Search existing patient</small>
                                        </button>
                                    </div>
                                    <div class="col-md-6">
                                        <button type="button" class="btn btn-success btn-block customer-type-btn" 
                                                data-type="walkin" onclick="selectCustomerType('walkin')">
                                            <i class="fa fa-walking"></i><br>
                                            <strong>Walk-in Customer</strong><br>
                                            <small>Quick billing without registration</small>
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Patient Search Section -->
                        <div id="patientSearchSection" style="display: none;">
                            <h5><i class="fa fa-search"></i> Patient Search</h5>
                            <form id="patientSearchForm">
                                <div class="form-group">
                                    <label for="patientSearch">Search Patient <span class="text-red">*</span></label>
                                    <div class="input-group">
                                        <input type="text" class="form-control" id="patientSearch" 
                                               placeholder="Enter patient name, ID, or phone number" autocomplete="off">
                                        <span class="input-group-btn">
                                            <button type="button" class="btn btn-primary" onclick="searchPatients()">
                                                <i class="fa fa-search"></i> Search
                                            </button>
                                        </span>
                                    </div>
                                    <small class="help-block">Start typing to search for patients</small>
                                </div>
                            </form>

                            <!-- Patient Search Results -->
                            <div id="patientSearchResults" style="display: none;">
                                <h6><i class="fa fa-users"></i> Search Results</h6>
                                <div class="table-responsive" style="max-height: 300px; overflow-y: auto;">
                                    <table class="table table-striped table-hover">
                                        <thead>
                                            <tr>
                                                <th>Patient ID</th>
                                                <th>Name</th>
                                                <th>Phone</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody id="patientResultsTable">
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Walk-in Customer Section -->
                        <div id="walkinCustomerSection" style="display: none;">
                            <h5><i class="fa fa-walking"></i> Walk-in Customer Details</h5>
                            <form id="walkinCustomerForm">
                                <div class="row">
                                    <div class="col-md-8">
                                        <div class="form-group">
                                            <label for="walkinName">Customer Name</label>
                                            <input type="text" class="form-control" id="walkinName" 
                                                   placeholder="Enter customer name (optional)">
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="form-group">
                                            <label for="walkinAge">Age</label>
                                            <input type="number" class="form-control" id="walkinAge" 
                                                   placeholder="Age" min="1" max="120">
                                        </div>
                                    </div>
                                </div>
                                <div class="row">
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="walkinPhone">Phone Number</label>
                                            <input type="text" class="form-control" id="walkinPhone" 
                                                   placeholder="Phone number (optional)">
                                        </div>
                                    </div>
                                    <div class="col-md-6">
                                        <div class="form-group">
                                            <label for="walkinGender">Gender</label>
                                            <select class="form-control" id="walkinGender">
                                                <option value="">Select Gender</option>
                                                <option value="Male">Male</option>
                                                <option value="Female">Female</option>
                                                <option value="Other">Other</option>
                                            </select>
                                        </div>
                                    </div>
                                </div>
                                <div class="form-group">
                                    <button type="button" class="btn btn-success btn-block" onclick="confirmWalkinCustomer()">
                                        <i class="fa fa-check"></i> Proceed with Walk-in Customer
                                    </button>
                                </div>
                            </form>
                        </div>

                        <!-- Selected Customer Info -->
                        <div id="selectedCustomerInfo" style="display: none;">
                            <hr>
                            <h5><i class="fa fa-user-check"></i> Selected Customer</h5>
                            <div class="customer-details">
                                <div class="row">
                                    <div class="col-md-12">
                                        <div class="info-box" id="customerInfoBox">
                                            <span class="info-box-icon"><i class="fa fa-user"></i></span>
                                            <div class="info-box-content">
                                                <span class="info-box-text" id="customerTypeLabel">Customer Information</span>
                                                <span class="info-box-number" id="selectedCustomerName">-</span>
                                                <div class="progress">
                                                    <div class="progress-bar" style="width: 100%"></div>
                                                </div>
                                                <span class="progress-description" id="customerDetails">
                                                    ID: <span id="selectedCustomerId">-</span> | 
                                                    Phone: <span id="selectedCustomerPhone">-</span>
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <div class="row">
                                    <div class="col-md-6" id="historyButtonContainer">
                                        <button type="button" class="btn btn-info btn-block" onclick="viewCustomerHistory()">
                                            <i class="fa fa-history"></i> View History
                                        </button>
                                    </div>
                                    <div class="col-md-6">
                                        <button type="button" class="btn btn-success btn-block" onclick="proceedToMedicineSelection()">
                                            <i class="fa fa-arrow-right"></i> Add Medicines
                                        </button>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Stored Prescriptions Section (only for registered patients) -->
                        <div id="storedPrescriptionsSection" style="display: none;">
                            <hr>
                            <h5><i class="fa fa-prescription"></i> Stored Prescriptions</h5>
                            <div id="prescriptionsList">
                                <!-- Prescriptions will be loaded here -->
                            </div>
                        </div>

                        <!-- Change Customer Button -->
                        <div id="changeCustomerSection" style="display: none;">
                            <hr>
                            <button type="button" class="btn btn-warning btn-block" onclick="changeCustomer()">
                                <i class="fa fa-exchange"></i> Change Customer
                            </button>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Medicine Selection Section -->
            <div class="col-md-7">
                <div class="box box-success">
                    <div class="box-header with-border">
                        <h3 class="box-title"><i class="fa fa-medkit"></i> Medicine Selection</h3>
                        <div class="box-tools pull-right">
                            <span class="label label-success">Step 2</span>
                        </div>
                    </div>
                    <div class="box-body">
                        <div id="medicineSelectionSection" style="display: none;">
                            <!-- Medicine Search -->
                            <div class="row">
                                <div class="col-md-8">
                                    <div class="form-group">
                                        <label for="medicineSearch">Search Medicine</label>
                                        <input type="text" class="form-control" id="medicineSearch" 
                                               placeholder="Enter medicine name or company" autocomplete="off">
                                    </div>
                                </div>
                                <div class="col-md-4">
                                    <div class="form-group">
                                        <label for="medicineCategory">Category</label>
                                        <select class="form-control" id="medicineCategory">
                                            <option value="">All Categories</option>
                                            <?php if (isset($medicine_categories)): ?>
                                                <?php foreach ($medicine_categories as $category): ?>
                                                    <option value="<?php echo $category->id; ?>">
                                                        <?php echo $category->medicine_category; ?>
                                                    </option>
                                                <?php endforeach; ?>
                                            <?php endif; ?>
                                        </select>
                                    </div>
                                </div>
                            </div>

                            <!-- Medicine Search Results -->
                            <div id="medicineSearchResults" style="display: none;">
                                <h6><i class="fa fa-pills"></i> Available Medicines</h6>
                                <div class="table-responsive" style="max-height: 250px; overflow-y: auto;">
                                    <table class="table table-striped table-condensed">
                                        <thead>
                                            <tr>
                                                <th>Medicine</th>
                                                <th>Company</th>
                                                <th>Stock</th>
                                                <th>Price Range</th>
                                                <th>Action</th>
                                            </tr>
                                        </thead>
                                        <tbody id="medicineResultsTable">
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Show message when no customer selected -->
                        <div id="noCustomerMessage">
                            <div class="callout callout-info">
                                <h4><i class="fa fa-info"></i> Notice</h4>
                                Please select a customer type first to start adding medicines to the bill.
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Bill Items and Calculation -->
        <div class="box box-warning">
            <div class="box-header with-border">
                <h3 class="box-title"><i class="fa fa-shopping-cart"></i> Bill Items & Calculation</h3>
                <div class="box-tools pull-right">
                    <span class="label label-warning">Step 3</span>
                    <button type="button" class="btn btn-sm btn-default" onclick="clearBill()">
                        <i class="fa fa-trash"></i> Clear Bill
                    </button>
                </div>
            </div>
            <div class="box-body">
                <div id="billItemsSection">
                    <!-- Empty Bill Message -->
                    <div id="emptyBillMessage">
                        <div class="callout callout-warning">
                            <h4><i class="fa fa-shopping-cart"></i> Empty Bill</h4>
                            No medicines added to bill yet. Search and select medicines to add them to the bill.
                        </div>
                    </div>

                    <!-- Bill Items Table -->
                    <div id="billItemsTable" style="display: none;">
                        <div class="table-responsive">
                            <table class="table table-bordered table-striped">
                                <thead class="bg-yellow">
                                    <tr>
                                        <th width="5%">#</th>
                                        <th width="30%">Medicine Details</th>
                                        <th width="15%">Batch Info</th>
                                        <th width="10%">Qty</th>
                                        <th width="12%">Unit Price</th>
                                        <th width="15%">Amount</th>
                                        <th width="8%">Action</th>
                                    </tr>
                                </thead>
                                <tbody id="billItemsBody">
                                    <!-- Bill items will be populated here -->
                                </tbody>
                            </table>
                        </div>

                        <!-- Bill Summary -->
                        <div class="row">
                            <div class="col-md-8">
                                <!-- Bill Notes -->
                                <div class="form-group">
                                    <label for="billNotes">Notes (Optional)</label>
                                    <textarea class="form-control" id="billNotes" rows="3" 
                                              placeholder="Add any special instructions or notes..."></textarea>
                                </div>
                            </div>
                            <div class="col-md-4">
                                <div class="bill-summary">
                                    <table class="table table-bordered">
                                        <tr>
                                            <td><strong>Subtotal:</strong></td>
                                            <td class="text-right">
                                                <span id="billSubtotal"><?php echo $currency_symbol; ?>0.00</span>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td>
                                                <div class="input-group input-group-sm">
                                                    <span class="input-group-addon">Discount %:</span>
                                                    <input type="number" class="form-control" id="discountPercentage" 
                                                           value="0" min="0" max="100" onchange="calculateBill()">
                                                </div>
                                            </td>
                                            <td class="text-right">
                                                <span id="billDiscount"><?php echo $currency_symbol; ?>0.00</span>
                                            </td>
                                        </tr>
                                        <tr>
                                            <td>
                                                <div class="input-group input-group-sm">
                                                    <span class="input-group-addon">Tax %:</span>
                                                    <input type="number" class="form-control" id="taxPercentage" 
                                                           value="0" min="0" max="50" onchange="calculateBill()">
                                                </div>
                                            </td>
                                            <td class="text-right">
                                                <span id="billTax"><?php echo $currency_symbol; ?>0.00</span>
                                            </td>
                                        </tr>
                                        <tr class="bg-yellow">
                                            <td><strong>Net Amount:</strong></td>
                                            <td class="text-right">
                                                <strong><span id="billNetAmount"><?php echo $currency_symbol; ?>0.00</span></strong>
                                            </td>
                                        </tr>
                                    </table>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Section -->
                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="paymentMethod">Payment Method</label>
                                    <select class="form-control" id="paymentMethod">
                                        <option value="cash">Cash</option>
                                        <option value="card">Credit/Debit Card</option>
                                        <option value="mpesa">M-Pesa</option>
                                        <option value="bank_transfer">Bank Transfer</option>
                                        <option value="insurance">Insurance</option>
                                    </select>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label for="amountPaid">Amount Paid</label>
                                    <input type="number" class="form-control" id="amountPaid" 
                                           placeholder="Enter amount paid" step="0.01" onchange="calculateChange()">
                                </div>
                            </div>
                        </div>

                        <div class="row">
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>Balance/Change:</label>
                                    <div class="input-group">
                                        <span class="input-group-addon"><?php echo $currency_symbol; ?></span>
                                        <input type="text" class="form-control" id="balanceAmount" readonly>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="form-group">
                                    <label>&nbsp;</label>
                                    <button type="button" class="btn btn-success btn-lg btn-block" onclick="processBill()">
                                        <i class="fa fa-check"></i> Process Bill & Print Receipt
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </section>
</div>

<!-- Medicine Selection Modal -->
<div class="modal fade" id="medicineSelectionModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-medkit"></i> Select Medicine Batch & Quantity
                </h4>
            </div>
            <div class="modal-body">
                <div id="medicineModalContent">
                    <!-- Medicine details will be loaded here -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Customer History Modal -->
<div class="modal fade" id="customerHistoryModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-history"></i> Customer History
                </h4>
            </div>
            <div class="modal-body">
                <div id="customerHistoryContent">
                    <!-- Customer history will be loaded here -->
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Prescription Details Modal -->
<div class="modal fade" id="prescriptionModal" tabindex="-1" role="dialog">
    <div class="modal-dialog modal-lg" role="document">
        <div class="modal-content">
            <div class="modal-header">
                <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                    <span aria-hidden="true">&times;</span>
                </button>
                <h4 class="modal-title">
                    <i class="fa fa-prescription"></i> Prescription Details
                </h4>
            </div>
            <div class="modal-body">
                <div id="prescriptionModalContent">
                    <!-- Prescription details will be loaded here -->
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-default" data-dismiss="modal">Close</button>
                <button type="button" class="btn btn-success" onclick="addAllPrescriptionItems()">
                    <i class="fa fa-plus"></i> Add All to Bill
                </button>
            </div>
        </div>
    </div>
</div>

<style>
/* ===================================================
   CUSTOMER TYPE SELECTION STYLES
   =================================================== */
.customer-type-btn {
    min-height: 80px;
    padding: 15px 10px;
    text-align: center;
    margin-bottom: 10px;
    transition: all 0.3s ease;
}

.customer-type-btn:hover {
    transform: translateY(-2px);
    box-shadow: 0 4px 8px rgba(0,0,0,0.1);
}

.customer-type-btn.active {
    box-shadow: 0 0 0 3px rgba(60,141,188,0.3);
}

/* ===================================================
   WORKFLOW STEPS STYLES
   =================================================== */
.dispensing-workflow {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
    padding: 20px;
    background: #f9f9f9;
    border-radius: 8px;
}

.workflow-step {
    display: flex;
    align-items: center;
    flex: 1;
    position: relative;
    opacity: 0.5;
    transition: all 0.3s ease;
}

.workflow-step.active {
    opacity: 1;
}

.workflow-step.completed {
    opacity: 1;
    color: #00a65a;
}

.workflow-step::after {
    content: '';
    position: absolute;
    top: 50%;
    right: -50%;
    width: 100%;
    height: 2px;
    background: #ddd;
    z-index: -1;
}

.workflow-step:last-child::after {
    display: none;
}

.workflow-step.active::after,
.workflow-step.completed::after {
    background: #00a65a;
}

.step-number {
    width: 40px;
    height: 40px;
    border-radius: 50%;
    background: #ddd;
    display: flex;
    align-items: center;
    justify-content: center;
    font-weight: bold;
    margin-right: 15px;
    transition: all 0.3s ease;
}

.workflow-step.active .step-number {
    background: #3c8dbc;
    color: white;
}

.workflow-step.completed .step-number {
    background: #00a65a;
    color: white;
}

.step-content h6 {
    margin: 0;
    font-weight: 600;
}

.step-content small {
    color: #666;
    font-size: 11px;
}

/* ===================================================
   PRESCRIPTION STYLES
   =================================================== */
.prescription-card {
    border: 1px solid #ddd;
    border-radius: 6px;
    margin-bottom: 10px;
    background: #f9f9f9;
    transition: all 0.3s ease;
}

.prescription-card:hover {
    border-color: #5cb85c;
    background: #f0fff0;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.prescription-header {
    background: linear-gradient(135deg, #5cb85c 0%, #449d44 100%);
    color: white;
    padding: 10px 15px;
    border-radius: 6px 6px 0 0;
    font-weight: 600;
}

.prescription-body {
    padding: 15px;
}

.prescription-meta {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
    font-size: 12px;
    color: #666;
}

.prescription-medicines {
    margin-top: 10px;
}

.medicine-item {
    background: white;
    border: 1px solid #e9ecef;
    border-radius: 4px;
    padding: 8px 12px;
    margin-bottom: 8px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    transition: all 0.3s ease;
}

.medicine-item:hover {
    border-color: #5cb85c;
    background: #f8fff8;
}

.medicine-name {
    font-weight: 600;
    color: #2c3e50;
}

.medicine-dosage {
    font-size: 11px;
    color: #7f8c8d;
    margin-top: 2px;
}

.medicine-duration {
    font-size: 11px;
    color: #34495e;
    background: #ecf0f1;
    padding: 2px 6px;
    border-radius: 3px;
}

.prescription-actions {
    margin-top: 15px;
    text-align: center;
}

.btn-prescription {
    font-size: 11px;
    padding: 5px 10px;
    margin: 0 2px;
}

.prescription-status {
    display: inline-block;
    padding: 3px 8px;
    border-radius: 12px;
    font-size: 10px;
    font-weight: 600;
    text-transform: uppercase;
}

.status-fresh {
    background: #d4edda;
    color: #155724;
}

.status-partial {
    background: #fff3cd;
    color: #856404;
}

.status-dispensed {
    background: #f8d7da;
    color: #721c24;
}

.prescription-empty {
    text-align: center;
    padding: 30px;
    color: #6c757d;
}

.prescription-loading {
    text-align: center;
    padding: 20px;
    color: #007bff;
}

/* ===================================================
   BILL AND SUMMARY STYLES
   =================================================== */
.bill-summary {
    background: #f8f9fa;
    border-radius: 6px;
    padding: 15px;
    margin-top: 10px;
}

.bill-summary table {
    margin-bottom: 0;
}

.bill-summary table tr:last-child {
    font-size: 16px;
    font-weight: bold;
}

.customer-details {
    margin-top: 15px;
}

.info-box {
    border-radius: 8px;
    box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.table-hover tbody tr:hover {
    background-color: #f5f5f5;
    cursor: pointer;
}

.form-control:focus {
   border-color: #3c8dbc;
   box-shadow: 0 0 0 0.2rem rgba(60,141,188,0.25);
}

.alert {
   border-radius: 6px;
   border: none;
   box-shadow: 0 2px 4px rgba(0,0,0,0.1);
}

.btn {
   border-radius: 4px;
   font-weight: 500;
}

.btn:focus {
   box-shadow: 0 0 0 0.2rem rgba(0,123,255,0.25);
}

/* ===================================================
  RESPONSIVE STYLES
  =================================================== */
@media (max-width: 768px) {
   .dispensing-workflow {
       flex-direction: column;
       align-items: stretch;
   }
   
   .workflow-step {
       margin-bottom: 15px;
   }
   
   .workflow-step::after {
       display: none;
   }
   
   .customer-type-btn {
       margin-bottom: 10px;
   }
   
   .prescription-meta {
       flex-direction: column;
       align-items: flex-start;
       gap: 5px;
   }
   
   .medicine-item {
       flex-direction: column;
       align-items: flex-start;
       gap: 5px;
   }
   
   .prescription-actions {
       display: flex;
       flex-direction: column;
       gap: 5px;
   }
}
</style>

<script>
// ===================================================
// GLOBAL VARIABLES
// ===================================================
var selectedCustomer = null;
var customerType = null; // 'registered' or 'walkin'
var billItems = [];
var billSubtotal = 0;
var medicineSearchTimeout = null;
var storedPrescriptions = [];

// ===================================================
// INITIALIZATION FUNCTIONS
// ===================================================
$(document).ready(function() {
   console.log('Enhanced Dispensing page loaded with Walk-in support');
   initializePage();
   setupEventListeners();
   console.log('All functions loaded successfully');
});

function initializePage() {
   // Reset workflow to step 1
   updateWorkflowStep(1);
   
   // Hide sections that should be hidden initially
   $('#patientSearchSection').hide();
   $('#walkinCustomerSection').hide();
   $('#medicineSelectionSection').hide();
   $('#selectedCustomerInfo').hide();
   $('#storedPrescriptionsSection').hide();
   $('#changeCustomerSection').hide();
   $('#billItemsTable').hide();
   
   // Show initial messages
   $('#noCustomerMessage').show();
   $('#emptyBillMessage').show();
   
   // Show customer type selection
   $('#customerTypeSection').show();
}

function setupEventListeners() {
   // Patient search event listeners
   $('#patientSearch').off('input').on('input', function() {
       var searchTerm = $(this).val().trim();
       clearTimeout(medicineSearchTimeout);
       
       if (searchTerm.length >= 2) {
           medicineSearchTimeout = setTimeout(function() {
               searchPatients();
           }, 500);
       } else {
           $('#patientSearchResults').hide();
       }
   });
   
   // Medicine search event listeners
   $('#medicineSearch, #medicineCategory').off('input change').on('input change', function() {
       clearTimeout(medicineSearchTimeout);
       medicineSearchTimeout = setTimeout(function() {
           searchMedicines();
       }, 500);
   });
   
   // Enter key handlers
   $('#patientSearch').off('keypress').on('keypress', function(e) {
       if (e.which == 13) {
           e.preventDefault();
           searchPatients();
       }
   });
   
   $('#medicineSearch').off('keypress').on('keypress', function(e) {
       if (e.which == 13) {
           e.preventDefault();
           searchMedicines();
       }
   });
}

// ===================================================
// CUSTOMER TYPE SELECTION FUNCTIONS
// ===================================================
function selectCustomerType(type) {
   console.log('Customer type selected:', type);
   
   customerType = type;
   
   // Update button states
   $('.customer-type-btn').removeClass('active');
   $('button[data-type="' + type + '"]').addClass('active');
   
   // Hide customer type section
   $('#customerTypeSection').hide();
   
   if (type === 'registered') {
       // Show patient search section
       $('#patientSearchSection').show();
       $('#walkinCustomerSection').hide();
       updateWorkflowStep(2);
       $('#patientSearch').focus();
   } else if (type === 'walkin') {
       // Show walk-in customer section
       $('#walkinCustomerSection').show();
       $('#patientSearchSection').hide();
       updateWorkflowStep(2);
       $('#walkinName').focus();
   }
   
   // Show change customer button
   $('#changeCustomerSection').show();
}

function confirmWalkinCustomer() {
   console.log('Confirming walk-in customer');
   
   // Get walk-in customer details
   var walkinName = $('#walkinName').val().trim() || 'Walk-in Customer';
   var walkinAge = $('#walkinAge').val().trim();
   var walkinPhone = $('#walkinPhone').val().trim();
   var walkinGender = $('#walkinGender').val();
   
   // Create walk-in customer object
   selectedCustomer = {
       id: 'walkin_' + Date.now(), // Temporary ID for walk-in
       customer_type: 'walkin',
       customer_name: walkinName,
       age: walkinAge,
       phone: walkinPhone,
       gender: walkinGender,
       patient_unique_id: 'WALK-IN'
   };
   
   // Update display
   $('#selectedCustomerName').text(walkinName);
   $('#selectedCustomerId').text('WALK-IN');
   $('#selectedCustomerPhone').text(walkinPhone || 'N/A');
   $('#customerTypeLabel').text('Walk-in Customer');
   
   // Style the info box for walk-in
   $('#customerInfoBox').removeClass('bg-green').addClass('bg-aqua');
   $('#customerInfoBox .info-box-icon').html('<i class="fa fa-walking"></i>');
   
   // Hide walk-in form and show customer info
   $('#walkinCustomerSection').hide();
   $('#selectedCustomerInfo').show();
   
   // Hide history button for walk-in customers
   $('#historyButtonContainer').hide();
   
   // Enable medicine selection
   $('#noCustomerMessage').hide();
   $('#medicineSelectionSection').show();
   
   updateWorkflowStep(3);
   showAlert('Walk-in customer confirmed. You can now add medicines.', 'success');
}

function changeCustomer() {
   console.log('Changing customer');
   
   // Reset everything
   selectedCustomer = null;
   customerType = null;
   storedPrescriptions = [];
   
   // Clear forms
   $('#patientSearch').val('');
   $('#walkinName').val('');
   $('#walkinAge').val('');
   $('#walkinPhone').val('');
   $('#walkinGender').val('');
   
   // Reset button states
   $('.customer-type-btn').removeClass('active');
   
   // Hide all sections
   $('#patientSearchSection').hide();
   $('#walkinCustomerSection').hide();
   $('#selectedCustomerInfo').hide();
   $('#storedPrescriptionsSection').hide();
   $('#changeCustomerSection').hide();
   $('#patientSearchResults').hide();
   $('#medicineSelectionSection').hide();
   
   // Show customer type selection
   $('#customerTypeSection').show();
   $('#noCustomerMessage').show();
   
   updateWorkflowStep(1);
   showAlert('Customer selection reset. Please choose customer type.', 'info');
}

// ===================================================
// PATIENT SEARCH FUNCTIONS (for registered patients)
// ===================================================
function searchPatients() {
   var searchTerm = $('#patientSearch').val().trim();
   
   console.log('searchPatients called with term:', searchTerm);
   
   if (searchTerm.length < 2) {
       showAlert('Please enter at least 2 characters to search', 'warning');
       return;
   }
   
   // Show loading
   $('#patientResultsTable').html('<tr><td colspan="4" class="text-center"><i class="fa fa-spinner fa-spin"></i> Searching...</td></tr>');
   $('#patientSearchResults').show();
   
   $.ajax({
       url: '<?php echo base_url(); ?>admin/dispensary/searchPatient',
       type: 'POST',
       data: { search_term: searchTerm },
       dataType: 'json',
       success: function(response) {
           console.log('AJAX response:', response);
           if (response.status === 1) {
               displayPatientResults(response.patients);
           } else {
               $('#patientResultsTable').html('<tr><td colspan="4" class="text-center text-muted">' + (response.message || 'No patients found') + '</td></tr>');
           }
       },
       error: function(xhr, status, error) {
           console.error('AJAX Error:', xhr, status, error);
           showAlert('Error searching patients. Please try again.', 'danger');
           $('#patientSearchResults').hide();
       }
   });
}

function displayPatientResults(patients) {
   var html = '';
   
   if (patients.length === 0) {
       html = '<tr><td colspan="4" class="text-center text-muted">No patients found</td></tr>';
   } else {
       patients.forEach(function(patient) {
           html += '<tr style="cursor: pointer;" onclick="selectPatient(' + patient.id + ', \'' + 
                   patient.patient_name.replace(/'/g, "\\'") + '\', \'' + 
                   (patient.patient_unique_id || 'N/A') + '\', \'' + 
                   (patient.mobileno || 'N/A') + '\')">';
           html += '<td>' + (patient.patient_unique_id || 'N/A') + '</td>';
           html += '<td>' + patient.patient_name + '</td>';
           html += '<td>' + (patient.mobileno || 'N/A') + '</td>';
           html += '<td><button type="button" class="btn btn-xs btn-primary" onclick="selectPatient(' + patient.id + ', \'' + 
                   patient.patient_name.replace(/'/g, "\\'") + '\', \'' + 
                   (patient.patient_unique_id || 'N/A') + '\', \'' + 
                   (patient.mobileno || 'N/A') + '\')"><i class="fa fa-check"></i> Select</button></td>';
           html += '</tr>';
       });
   }
   
   $('#patientResultsTable').html(html);
}

function selectPatient(id, name, unique_id, phone) {
   console.log('Patient selected:', id, name);
   
   selectedCustomer = {
       id: id,
       customer_type: 'registered',
       patient_name: name,
       customer_name: name, // Unified field
       patient_unique_id: unique_id,
       mobileno: phone,
       phone: phone // Unified field
   };
   
   // Update display
   $('#selectedCustomerName').text(name);
   $('#selectedCustomerId').text(unique_id);
   $('#selectedCustomerPhone').text(phone);
   $('#customerTypeLabel').text('Registered Patient');
   
   // Style the info box for registered patient
   $('#customerInfoBox').removeClass('bg-aqua').addClass('bg-green');
   $('#customerInfoBox .info-box-icon').html('<i class="fa fa-user"></i>');
   
   // Hide search results and show customer info
   $('#patientSearchResults').hide();
   $('#patientSearchSection').hide();
   $('#selectedCustomerInfo').show();
   
   // Show history button for registered patients
   $('#historyButtonContainer').show();
   
   // Enable medicine selection
   $('#noCustomerMessage').hide();
   $('#medicineSelectionSection').show();
   
   // Load stored prescriptions for registered patients
   loadPatientPrescriptions(id);
   
   updateWorkflowStep(3);
   showAlert('Patient selected successfully!', 'success');
}

// ===================================================
// PRESCRIPTION LOADING (only for registered patients)
// ===================================================
function loadPatientPrescriptions(patientId) {
   console.log('Loading stored prescriptions for patient ID:', patientId);
   
   // Show loading state
   $('#prescriptionsList').html(`
       <div class="prescription-loading">
           <i class="fa fa-spinner fa-spin"></i> Loading stored prescriptions...
       </div>
   `);
   $('#storedPrescriptionsSection').show();
   
   // Call backend to get patient prescriptions - This is optional and may not exist in your system
   $.ajax({
       url: '<?php echo base_url(); ?>admin/dispensary/getPatientPrescriptions',
       type: 'POST',
       data: { patient_id: patientId },
       dataType: 'json',
       timeout: 10000,
       success: function(response) {
           console.log('Prescriptions response:', response);
           handlePrescriptionsSuccess(response);
       },
       error: function(xhr, status, error) {
           console.error('Error loading prescriptions:', error);
           handlePrescriptionsError(xhr, status, error);
       }
   });
}

function handlePrescriptionsSuccess(response) {
   try {
       if (response && response.status === 'success') {
           const prescriptions = response.prescriptions || [];
           
           if (prescriptions.length > 0) {
               displayPatientPrescriptions(prescriptions);
               storedPrescriptions = prescriptions;
               showAlert(`Found ${prescriptions.length} stored prescription(s)`, 'success');
           } else {
               displayNoPrescriptions();
           }
       } else {
           displayPrescriptionError(response.message || 'Unable to load prescriptions');
       }
   } catch (error) {
       console.error('Error processing prescriptions:', error);
       displayPrescriptionError('Error processing prescription data');
   }
}

function handlePrescriptionsError(xhr, status, error) {
   console.error('Prescription loading error:', xhr, status, error);
   displayPrescriptionError('Prescription service not available');
}

function displayPatientPrescriptions(prescriptions) {
   let html = '';
   
   prescriptions.forEach(function(prescription, index) {
       const status = prescription.dispensed_status || 'fresh';
       const statusClass = status === 'dispensed' ? 'status-dispensed' : 
                         status === 'partial' ? 'status-partial' : 'status-fresh';
       
       html += `
           <div class="prescription-card" data-prescription-id="${prescription.id}">
               <div class="prescription-header">
                   <i class="fa fa-prescription"></i> Prescription #${prescription.prescription_no || prescription.id}
                   <span class="prescription-status ${statusClass}">${status}</span>
               </div>
               <div class="prescription-body">
                   <div class="prescription-meta">
                       <div>
                           <strong>Doctor:</strong> ${prescription.doctor_name || 'N/A'}<br>
                           <strong>Date:</strong> ${formatDate(prescription.prescription_date || prescription.created_at)}
                       </div>
                       <div>
                           <strong>Visit:</strong> ${prescription.visit_type || 'OPD'}<br>
                           <strong>Medicines:</strong> ${prescription.medicines ? prescription.medicines.length : 0}
                       </div>
                   </div>
                   
                   <div class="prescription-medicines">
                       ${displayPrescriptionMedicines(prescription.medicines || [])}
                   </div>
                   
                   <div class="prescription-actions">
                       <button type="button" class="btn btn-info btn-prescription" onclick="viewPrescriptionDetails(${prescription.id})">
                           <i class="fa fa-eye"></i> View Details
                       </button>
                       <button type="button" class="btn btn-success btn-prescription" onclick="addPrescriptionToBill(${prescription.id})">
                           <i class="fa fa-plus"></i> Add to Bill
                       </button>
                   </div>
               </div>
           </div>
       `;
   });
   
   $('#prescriptionsList').html(html);
}

function displayPrescriptionMedicines(medicines) {
   if (!medicines || medicines.length === 0) {
       return '<div class="text-muted" style="font-size: 12px;">No medicines prescribed</div>';
   }
   
   let html = '';
   medicines.slice(0, 3).forEach(function(medicine) {
       html += `
           <div class="medicine-item">
               <div>
                   <div class="medicine-name">${medicine.medicine_name}</div>
                   <div class="medicine-dosage">${medicine.dosage || 'As directed'}</div>
               </div>
               <div class="medicine-duration">${medicine.duration || 'N/A'}</div>
           </div>
       `;
   });
   
   if (medicines.length > 3) {
       html += `<div class="text-center" style="font-size: 11px; color: #666; margin-top: 5px;">
                   +${medicines.length - 3} more medicines
                </div>`;
   }
   
   return html;
}

function displayNoPrescriptions() {
   $('#prescriptionsList').html(`
       <div class="prescription-empty">
           <i class="fa fa-info-circle" style="font-size: 24px; color: #6c757d; margin-bottom: 10px;"></i>
           <p>No stored prescriptions found for this patient.</p>
           <small>Patient can still get medicines through manual search.</small>
       </div>
   `);
}

function displayPrescriptionError(message) {
   $('#prescriptionsList').html(`
       <div class="prescription-empty">
           <i class="fa fa-exclamation-triangle" style="font-size: 24px; color: #f39c12; margin-bottom: 10px;"></i>
           <p style="color: #f39c12;">${message}</p>
           <small>You can still dispense medicines through manual search.</small>
       </div>
   `);
}

// ===================================================
// MEDICINE SEARCH AND SELECTION FUNCTIONS
// ===================================================
function proceedToMedicineSelection() {
   console.log('proceedToMedicineSelection called');
   updateWorkflowStep(4);
   $('#medicineSearch').focus();
   
   if (selectedCustomer.customer_type === 'walkin') {
       showAlert('You can now search and add medicines for the walk-in customer.', 'success');
   } else {
       showAlert('You can now search medicines or use stored prescriptions.', 'success');
   }
}

function searchMedicines() {
   var searchTerm = $('#medicineSearch').val().trim();
   var categoryId = $('#medicineCategory').val();
   
   console.log('Searching medicines:', searchTerm, categoryId);
   
   if (searchTerm.length < 2 && !categoryId) {
       $('#medicineSearchResults').hide();
       return;
   }
   
   // Show loading
   $('#medicineResultsTable').html('<tr><td colspan="5" class="text-center"><i class="fa fa-spinner fa-spin"></i> Searching...</td></tr>');
   $('#medicineSearchResults').show();
   
   $.ajax({
       url: '<?php echo base_url(); ?>admin/dispensary/searchMedicine',
       type: 'POST',
       data: { 
           search_term: searchTerm,
           category_id: categoryId 
       },
       dataType: 'json',
       success: function(response) {
           console.log('Medicine search response:', response);
           if (response.status === 1) {
               displayMedicineResults(response.medicines);
           } else {
               $('#medicineResultsTable').html('<tr><td colspan="5" class="text-center text-muted">No medicines found</td></tr>');
           }
       },
       error: function(xhr, status, error) {
           console.error('Medicine search error:', xhr, status, error);
           showAlert('Error searching medicines', 'danger');
           $('#medicineSearchResults').hide();
       }
   });
}

function displayMedicineResults(medicines) {
   var html = '';
   
   if (medicines.length === 0) {
       html = '<tr><td colspan="5" class="text-center text-muted">No medicines found</td></tr>';
   } else {
       medicines.forEach(function(medicine) {
           var stockClass = medicine.available_stock > 50 ? 'label-success' : 
                          medicine.available_stock > 10 ? 'label-warning' : 'label-danger';
           
           html += '<tr onclick="selectMedicine(' + medicine.id + ')" style="cursor: pointer;">';
           html += '<td><strong>' + medicine.medicine_name + '</strong><br><small class="text-muted">' + (medicine.medicine_category || 'N/A') + '</small></td>';
           html += '<td>' + (medicine.medicine_company || 'N/A') + '</td>';
           html += '<td><span class="label ' + stockClass + '">' + medicine.available_stock + ' ' + (medicine.unit || '') + '</span></td>';
           html += '<td><?php echo $currency_symbol; ?>' + parseFloat(medicine.min_price || 0).toFixed(2) + '</td>';
           html += '<td><button type="button" class="btn btn-xs btn-success" onclick="selectMedicine(' + medicine.id + ')"><i class="fa fa-plus"></i> Add</button></td>';
           html += '</tr>';
       });
   }
   
   $('#medicineResultsTable').html(html);
}

function selectMedicine(medicineId) {
   console.log('Medicine selected:', medicineId);
   
   if (!selectedCustomer) {
       showAlert('Please select a customer first', 'warning');
       return;
   }
   
   // Get medicine details with batches
   $.ajax({
       url: '<?php echo base_url(); ?>admin/dispensary/getMedicineDetails',
       type: 'POST',
       data: { medicine_id: medicineId },
       dataType: 'json',
       success: function(response) {
           if (response.status === 1) {
               showMedicineSelectionModal(response.medicine);
           } else {
               showAlert('Medicine not available or out of stock', 'warning');
           }
       },
       error: function() {
           showAlert('Error loading medicine details', 'danger');
       }
   });
}

function showMedicineSelectionModal(medicineBatches) {
   if (!medicineBatches || medicineBatches.length === 0) {
       showAlert('No available batches for this medicine', 'warning');
       return;
   }
   
   var medicine = medicineBatches[0];
   
   var batchOptions = '';
   medicineBatches.forEach(function(batch) {
       var expiryDate = new Date(batch.expiry);
       
       batchOptions += '<option value="' + batch.batch_id + '" ';
       batchOptions += 'data-price="' + batch.sale_rate + '" ';
       batchOptions += 'data-stock="' + batch.available_quantity + '" ';
       batchOptions += 'data-batch="' + batch.batch_no + '" ';
       batchOptions += 'data-expiry="' + batch.expiry + '">';
       batchOptions += 'Batch: ' + batch.batch_no + ' | Stock: ' + batch.available_quantity + ' | ';
       batchOptions += 'Price: <?php echo $currency_symbol; ?>' + parseFloat(batch.sale_rate).toFixed(2) + ' | ';
       batchOptions += 'Expiry: ' + expiryDate.toLocaleDateString();
       batchOptions += '</option>';
   });
   
   var modalContent = '<form id="addMedicineForm">';
   modalContent += '<input type="hidden" id="selectedMedicineId" value="' + medicine.id + '">';
   modalContent += '<div class="row"><div class="col-md-12">';
   modalContent += '<h5><i class="fa fa-medkit"></i> ' + medicine.medicine_name + '</h5>';
   modalContent += '<p class="text-muted">Company: ' + (medicine.medicine_company || 'N/A') + ' | Category: ' + (medicine.medicine_category || 'N/A') + '</p>';
   modalContent += '</div></div>';
   
   modalContent += '<div class="row"><div class="col-md-12">';
   modalContent += '<div class="form-group"><label for="selectedBatch">Select Batch <span class="text-red">*</span></label>';
   modalContent += '<select class="form-control" id="selectedBatch" required onchange="updateBatchInfo()">';
   modalContent += '<option value="">-- Select Batch --</option>' + batchOptions;
   modalContent += '</select></div></div></div>';
   
   modalContent += '<div class="row">';
   modalContent += '<div class="col-md-6"><div class="form-group">';
   modalContent += '<label for="medicineQuantity">Quantity <span class="text-red">*</span></label>';
   modalContent += '<input type="number" class="form-control" id="medicineQuantity" min="1" required onchange="updateTotal()">';
   modalContent += '<small class="help-block">Available: <span id="availableStock">0</span></small>';
   modalContent += '</div></div>';
   modalContent += '<div class="col-md-6"><div class="form-group">';
   modalContent += '<label for="unitPrice">Unit Price</label>';
   modalContent += '<div class="input-group"><span class="input-group-addon"><?php echo $currency_symbol; ?></span>';
   modalContent += '<input type="number" class="form-control" id="unitPrice" step="0.01" readonly></div>';
   modalContent += '</div></div></div>';
   
   modalContent += '<div class="row"><div class="col-md-12"><div class="form-group">';
   modalContent += '<label>Total Amount</label>';
   modalContent += '<div class="input-group"><span class="input-group-addon"><?php echo $currency_symbol; ?></span>';
   modalContent += '<input type="text" class="form-control" id="totalAmount" readonly></div>';
   modalContent += '</div></div></div>';
   
   modalContent += '<div class="row">';
   modalContent += '<div class="col-md-6"><button type="button" class="btn btn-default btn-block" data-dismiss="modal"><i class="fa fa-times"></i> Cancel</button></div>';
   modalContent += '<div class="col-md-6"><button type="submit" class="btn btn-success btn-block"><i class="fa fa-plus"></i> Add to Bill</button></div>';
   modalContent += '</div></form>';
   
   $('#medicineModalContent').html(modalContent);
   $('#medicineSelectionModal').modal('show');
   
   // Form submission
   $('#addMedicineForm').off('submit').on('submit', function(e) {
       e.preventDefault();
       addMedicineToBill();
   });
}

function updateBatchInfo() {
   var selectedOption = $('#selectedBatch option:selected');
   var price = selectedOption.data('price');
   var stock = selectedOption.data('stock');
   
   $('#unitPrice').val(parseFloat(price).toFixed(2));
   $('#availableStock').text(stock);
   $('#medicineQuantity').attr('max', stock);
   $('#medicineQuantity').val('');
   $('#totalAmount').val('0.00');
}

function updateTotal() {
   var quantity = parseInt($('#medicineQuantity').val()) || 0;
   var unitPrice = parseFloat($('#unitPrice').val()) || 0;
   var total = quantity * unitPrice;
   
   $('#totalAmount').val(total.toFixed(2));
   
   // Validate quantity
   var maxStock = parseInt($('#medicineQuantity').attr('max'));
   if (quantity > maxStock) {
       showAlert('Quantity cannot exceed available stock', 'warning');
       $('#medicineQuantity').val(maxStock);
       updateTotal();
   }
}

function addMedicineToBill() {
   var medicineId = $('#selectedMedicineId').val();
   var batchId = $('#selectedBatch').val();
   var quantity = parseInt($('#medicineQuantity').val());
   var unitPrice = parseFloat($('#unitPrice').val());
   var totalPrice = parseFloat($('#totalAmount').val());
   
   if (!batchId || !quantity || quantity <= 0) {
       showAlert('Please select batch and enter valid quantity', 'warning');
       return;
   }
   
   // Get medicine and batch details
   var selectedOption = $('#selectedBatch option:selected');
   var batchNo = selectedOption.data('batch');
   var expiry = selectedOption.data('expiry');
   var medicineName = $('#medicineModalContent h5').text().replace(/.*\s/, '').trim();
   
   // Check if medicine already exists in bill
   var existingIndex = -1;
   for (var i = 0; i < billItems.length; i++) {
       if (billItems[i].medicine_batch_detail_id == batchId) {
           existingIndex = i;
           break;
       }
   }
   
   if (existingIndex !== -1) {
       // Update existing item
       billItems[existingIndex].quantity += quantity;
       billItems[existingIndex].amount = billItems[existingIndex].quantity * billItems[existingIndex].sale_price;
   } else {
       // Add new item
       billItems.push({
           medicine_id: medicineId,
           medicine_batch_detail_id: batchId,
           medicine_name: medicineName,
           batch_no: batchNo,
           expiry: expiry,
           quantity: quantity,
           sale_price: unitPrice,
           amount: totalPrice,
           unit_price: unitPrice,
           total_price: totalPrice
       });
   }
   
   updateBillDisplay();
   $('#medicineSelectionModal').modal('hide');
   showAlert('Medicine added to bill successfully', 'success');
   updateWorkflowStep(4);
}

// ===================================================
// BILL MANAGEMENT FUNCTIONS
// ===================================================
function updateBillDisplay() {
   if (billItems.length === 0) {
       $('#billItemsTable').hide();
       $('#emptyBillMessage').show();
       return;
   }
   
   $('#emptyBillMessage').hide();
   $('#billItemsTable').show();
   
   var html = '';
   var subtotal = 0;
   
   billItems.forEach(function(item, index) {
       var expiryDate = new Date(item.expiry);
       var monthsToExpiry = Math.ceil((expiryDate - new Date()) / (1000 * 60 * 60 * 24 * 30));
       var expiryClass = monthsToExpiry <= 6 ? 'text-danger' : monthsToExpiry <= 12 ? 'text-warning' : 'text-success';
       
       html += '<tr>';
       html += '<td>' + (index + 1) + '</td>';
       html += '<td><strong>' + item.medicine_name + '</strong>';
       
       if (item.dosage) {
           html += '<br><small class="text-info"><i class="fa fa-prescription"></i> ' + item.dosage + '</small>';
       }
       html += '</td>';
       
       html += '<td><strong>Batch:</strong> ' + item.batch_no + '<br><strong>Expiry:</strong> <span class="' + expiryClass + '">' + expiryDate.toLocaleDateString() + '</span></td>';
       html += '<td><input type="number" class="form-control input-sm" value="' + item.quantity + '" min="1" onchange="updateItemQuantity(' + index + ', this.value)"></td>';
       html += '<td class="text-right"><?php echo $currency_symbol; ?>' + item.unit_price.toFixed(2) + '</td>';
       html += '<td class="text-right"><?php echo $currency_symbol; ?>' + item.total_price.toFixed(2) + '</td>';
       html += '<td class="text-center"><button type="button" class="btn btn-xs btn-danger" onclick="removeBillItem(' + index + ')"><i class="fa fa-trash"></i></button></td>';
       html += '</tr>';
       
       subtotal += item.total_price;
   });
   
   $('#billItemsBody').html(html);
   billSubtotal = subtotal;
   calculateBill();
}

function updateItemQuantity(index, newQuantity) {
   var quantity = parseInt(newQuantity);
   if (quantity > 0) {
       billItems[index].quantity = quantity;
       billItems[index].total_price = quantity * billItems[index].unit_price;
       updateBillDisplay();
   }
}

function removeBillItem(index) {
   if (confirm('Are you sure you want to remove this item from the bill?')) {
       billItems.splice(index, 1);
       updateBillDisplay();
       
       if (billItems.length === 0) {
           updateWorkflowStep(3);
       }
   }
}

function calculateBill() {
   var discountPercentage = parseFloat($('#discountPercentage').val()) || 0;
   var taxPercentage = parseFloat($('#taxPercentage').val()) || 0;
   
   var discountAmount = (billSubtotal * discountPercentage) / 100;
   var afterDiscount = billSubtotal - discountAmount;
   var taxAmount = (afterDiscount * taxPercentage) / 100;
   var netAmount = afterDiscount + taxAmount;
   
   $('#billSubtotal').text('<?php echo $currency_symbol; ?>' + billSubtotal.toFixed(2));
   $('#billDiscount').text('<?php echo $currency_symbol; ?>' + discountAmount.toFixed(2));
   $('#billTax').text('<?php echo $currency_symbol; ?>' + taxAmount.toFixed(2));
   $('#billNetAmount').text('<?php echo $currency_symbol; ?>' + netAmount.toFixed(2));
   
   // Auto-fill amount paid
   $('#amountPaid').val(netAmount.toFixed(2));
   calculateChange();
}

function calculateChange() {
   var netAmount = parseFloat($('#billNetAmount').text().replace('<?php echo $currency_symbol; ?>', ''));
   var amountPaid = parseFloat($('#amountPaid').val()) || 0;
   var balance = amountPaid - netAmount;
   
   $('#balanceAmount').val(balance.toFixed(2));
   
   if (balance < 0) {
       $('#balanceAmount').parent().removeClass('has-success').addClass('has-error');
   } else {
       $('#balanceAmount').parent().removeClass('has-error').addClass('has-success');
   }
}

function clearBill() {
   console.log('clearBill called');
   if (billItems.length > 0) {
       if (confirm('Are you sure you want to clear all items from the bill?')) {
           billItems = [];
           updateBillDisplay();
           updateWorkflowStep(3);
       }
   } else {
       showAlert('Bill is already empty', 'info');
   }
}

function processBill() {
   console.log('processBill called for customer type:', selectedCustomer.customer_type);
   
   if (!selectedCustomer) {
       showAlert('Please select a customer first', 'warning');
       return;
   }
   
   if (billItems.length === 0) {
       showAlert('Please add at least one medicine to the bill', 'warning');
       return;
   }
   
   var amountPaid = parseFloat($('#amountPaid').val()) || 0;
   var netAmount = parseFloat($('#billNetAmount').text().replace('<?php echo $currency_symbol; ?>', ''));
   
   if (amountPaid < netAmount) {
       showAlert('Amount paid cannot be less than net amount', 'warning');
       return;
   }
   
   // Prepare bill data based on customer type
   var billData = {
       customer_type: selectedCustomer.customer_type,
       bill_date: new Date().toISOString().split('T')[0],
       subtotal: billSubtotal,
       discount_percentage: parseFloat($('#discountPercentage').val()) || 0,
       discount_amount: parseFloat($('#billDiscount').text().replace('<?php echo $currency_symbol; ?>', '')),
       tax_percentage: parseFloat($('#taxPercentage').val()) || 0,
       tax_amount: parseFloat($('#billTax').text().replace('<?php echo $currency_symbol; ?>', '')),
       net_amount: netAmount,
       paid_amount: amountPaid,
       balance_amount: parseFloat($('#balanceAmount').val()),
       payment_method: $('#paymentMethod').val(),
       status: 'paid',
       notes: $('#billNotes').val(),
       bill_items: JSON.stringify(billItems)
   };
   
   // Add customer-specific data
   if (selectedCustomer.customer_type === 'walkin') {
       billData.walkin_name = selectedCustomer.customer_name;
       billData.walkin_phone = selectedCustomer.phone;
       billData.walkin_age = selectedCustomer.age;
       billData.walkin_gender = selectedCustomer.gender;
       billData.customer_id = null;
   } else {
       billData.customer_id = selectedCustomer.id;
       billData.from_prescription = storedPrescriptions.length > 0 ? 1 : 0;
   }
   
   console.log('Sending bill data:', billData);
   
   // Show processing message
   var processingBtn = $('button[onclick="processBill()"]');
   var originalText = processingBtn.html();
   processingBtn.prop('disabled', true).html('<i class="fa fa-spinner fa-spin"></i> Processing...');
   
   $.ajax({
       url: '<?php echo base_url(); ?>admin/dispensary/processBill',
       type: 'POST',
       data: billData,
       dataType: 'json',
       success: function(response) {
           console.log('Process bill response:', response);
           processingBtn.prop('disabled', false).html(originalText);
           
           if (response.status === 1) {
               showAlert('Bill processed successfully!', 'success');
               updateWorkflowStep(5);
               
               // Open print window
               var printUrl = '<?php echo base_url(); ?>admin/dispensary/printBill/' + response.bill_id;
               console.log('Opening print URL:', printUrl);
               window.open(printUrl, '_blank');
               
               // Reset form after 2 seconds
               setTimeout(function() {
                   resetForm();
               }, 2000);
           } else {
               showAlert('Error processing bill: ' + (response.message || 'Unknown error'), 'danger');
           }
       },
       error: function(xhr, status, error) {
           console.error('Process bill error:', xhr, status, error);
           processingBtn.prop('disabled', false).html(originalText);
           
           var errorMessage = 'Error processing bill. ';
           try {
               var errorResponse = JSON.parse(xhr.responseText);
               errorMessage += errorResponse.message || errorResponse.error || 'Please try again.';
           } catch (e) {
               if (xhr.status === 404) {
                   errorMessage += 'Endpoint not found. Please check your controller method.';
               } else if (xhr.status === 500) {
                   errorMessage += 'Server error. Please check server logs.';
               } else {
                   errorMessage += 'Please try again.';
               }
           }
           showAlert(errorMessage, 'danger');
       }
   });
}

// ===================================================
// CUSTOMER HISTORY FUNCTIONS
// ===================================================
function viewCustomerHistory() {
   console.log('viewCustomerHistory called for customer type:', selectedCustomer.customer_type);
   
   if (!selectedCustomer) {
       showAlert('No customer selected', 'warning');
       return;
   }
   
   if (selectedCustomer.customer_type === 'walkin') {
       showAlert('History not available for walk-in customers', 'info');
       return;
   }
   
   console.log('Loading history for customer:', selectedCustomer);
   
   // Show loading in modal
   $('#customerHistoryContent').html(`
       <div class="text-center" style="padding: 40px;">
           <i class="fa fa-spinner fa-spin fa-3x text-primary"></i>
           <h4 style="margin-top: 20px;">Loading Customer History...</h4>
           <p class="text-muted">Please wait while we fetch the records</p>
       </div>
   `);
   
   // Show the modal
   $('#customerHistoryModal').modal('show');
   
   // Load history via AJAX
   $.ajax({
       url: '<?php echo base_url(); ?>admin/dispensary/getCustomerHistory',
       type: 'POST',
       data: { 
           customer_id: selectedCustomer.id,
           customer_type: selectedCustomer.customer_type,
           limit: 20
       },
       dataType: 'json',
       timeout: 15000,
       success: function(response) {
           console.log('Customer history response:', response);
           
           if (response.status === 1 && response.bills) {
               displayCustomerHistory(response.bills);
           } else {
               displayNoHistory(response.message || 'No history found');
           }
       },
       error: function(xhr, status, error) {
           console.error('Error loading customer history:', {xhr, status, error});
           displayHistoryError('Error loading customer history');
       }
   });
}

function displayCustomerHistory(bills) {
   let html = `
       <div class="customer-history-container">
           <div class="row">
               <div class="col-md-12">
                   <h5>
                       <i class="fa fa-history"></i> 
                       History for ${selectedCustomer.customer_name}
                       <small class="text-muted">(${bills.length} records found)</small>
                   </h5>
               </div>
           </div>
           
           <div class="table-responsive" style="max-height: 400px; overflow-y: auto;">
               <table class="table table-striped table-hover">
                   <thead style="background: #f8f9fa;">
                       <tr>
                           <th width="15%">Bill No</th>
                           <th width="15%">Date</th>
                           <th width="15%">Items</th>
                           <th width="15%">Amount</th>
                           <th width="15%">Status</th>
                           <th width="25%">Actions</th>
                       </tr>
                   </thead>
                   <tbody>
   `;
   
   bills.forEach(function(bill) {
       const statusClass = getBillStatusClass(bill.status);
       const amount = parseFloat(bill.net_amount || 0);
       
       html += `
           <tr>
               <td>
                   <strong>${bill.bill_no}</strong><br>
                   <small class="text-muted">ID: ${bill.id}</small>
               </td>
               <td>
                   ${bill.bill_date_formatted}<br>
                   <small class="text-muted">${formatTime(bill.bill_date)}</small>
               </td>
               <td>
                   <span class="badge bg-blue">${bill.total_items || 0}</span> item(s)
               </td>
               <td>
                   <strong><?php echo $currency_symbol; ?>${amount.toFixed(2)}</strong>
               </td>
               <td>
                   <span class="label ${statusClass}">${bill.status_label}</span>
               </td>
               <td>
                   <div class="btn-group-vertical btn-group-xs">
                       <button type="button" class="btn btn-info btn-xs" 
                               onclick="viewBillDetails(${bill.id})"
                               title="View Details">
                           <i class="fa fa-eye"></i> View
                       </button>
                       <button type="button" class="btn btn-success btn-xs" 
                               onclick="printBill(${bill.id})"
                               title="Print Bill">
                           <i class="fa fa-print"></i> Print
                       </button>
                   </div>
               </td>
           </tr>
       `;
   });
   
   html += `
                   </tbody>
               </table>
           </div>
       </div>
   `;
   
   $('#customerHistoryContent').html(html);
}

function displayNoHistory(message) {
   $('#customerHistoryContent').html(`
       <div class="text-center" style="padding: 40px;">
           <i class="fa fa-file-text-o fa-3x text-muted" style="margin-bottom: 20px;"></i>
           <h4>No History Found</h4>
           <p class="text-muted">${message}</p>
       </div>
   `);
}

function displayHistoryError(message) {
   $('#customerHistoryContent').html(`
       <div class="text-center" style="padding: 40px;">
           <i class="fa fa-exclamation-triangle fa-3x text-warning" style="margin-bottom: 20px;"></i>
           <h4>Unable to Load History</h4>
           <p class="text-warning">${message}</p>
           <button type="button" class="btn btn-primary btn-sm" onclick="viewCustomerHistory()">
               <i class="fa fa-refresh"></i> Try Again
           </button>
       </div>
   `);
}

// ===================================================
// PRESCRIPTION FUNCTIONS (for registered patients only)
// ===================================================
function viewPrescriptionDetails(prescriptionId) {
   console.log('Viewing prescription details:', prescriptionId);
   
   const prescription = storedPrescriptions.find(p => p.id == prescriptionId);
   if (!prescription) {
       showAlert('Prescription not found', 'error');
       return;
   }
   
   let modalContent = `
       <div class="prescription-details">
           <div class="row">
               <div class="col-md-6">
                   <h5><i class="fa fa-prescription"></i> Prescription #${prescription.prescription_no || prescription.id}</h5>
                   <p><strong>Doctor:</strong> ${prescription.doctor_name || 'N/A'}</p>
                   <p><strong>Date:</strong> ${formatDate(prescription.prescription_date || prescription.created_at)}</p>
                   <p><strong>Visit Type:</strong> ${prescription.visit_type || 'OPD'}</p>
               </div>
               <div class="col-md-6">
                   <p><strong>Patient:</strong> ${selectedCustomer.customer_name}</p>
                   <p><strong>Status:</strong> <span class="label label-info">${prescription.dispensed_status || 'Fresh'}</span></p>
                   <p><strong>Total Medicines:</strong> ${prescription.medicines ? prescription.medicines.length : 0}</p>
               </div>
           </div>
           
           <div class="row">
               <div class="col-md-12">
                   <h6><i class="fa fa-pills"></i> Prescribed Medicines</h6>
                   <div class="table-responsive">
                       <table class="table table-bordered table-striped">
                           <thead>
                               <tr>
                                   <th>Medicine</th>
                                   <th>Dosage</th>
                                   <th>Duration</th>
                                   <th>Instructions</th>
                                   <th>Action</th>
                               </tr>
                           </thead>
                           <tbody>
                               ${buildPrescriptionMedicinesTable(prescription.medicines || [])}
                           </tbody>
                       </table>
                   </div>
               </div>
           </div>
       </div>
   `;
   
   $('#prescriptionModalContent').html(modalContent);
   $('#prescriptionModal').modal('show');
}

function buildPrescriptionMedicinesTable(medicines) {
   if (!medicines || medicines.length === 0) {
       return '<tr><td colspan="5" class="text-center text-muted">No medicines prescribed</td></tr>';
   }
   
   let html = '';
   medicines.forEach(function(medicine, index) {
       html += `
           <tr>
               <td>
                   <strong>${medicine.medicine_name}</strong><br>
                   <small class="text-muted">${medicine.medicine_category || ''}</small>
               </td>
               <td>${medicine.dosage || 'As directed'}</td>
               <td>${medicine.duration || 'N/A'}</td>
               <td><small>${medicine.instruction || 'Take as prescribed'}</small></td>
               <td>
                   <button type="button" class="btn btn-xs btn-success" onclick="addSingleMedicineToBill('${medicine.medicine_name}', '${medicine.dosage}')">
                       <i class="fa fa-plus"></i> Add
                   </button>
               </td>
           </tr>
       `;
   });
   
   return html;
}

function addPrescriptionToBill(prescriptionId) {
   console.log('Adding prescription to bill:', prescriptionId);
   
   if (!prescriptionId) {
       showAlert('Prescription ID is required', 'error');
       return;
   }
   
   if (!selectedCustomer) {
       showAlert('Please select a patient first', 'warning');
       return;
   }
   
   const prescription = storedPrescriptions.find(p => p.id == prescriptionId);
   if (!prescription) {
       showAlert('Prescription not found', 'error');
       return;
   }
   
   if (!prescription.medicines || prescription.medicines.length === 0) {
       showAlert('No medicines found in this prescription', 'warning');
       return;
   }
   
   showAlert('Prescription medicines will be searched and added to bill', 'info');
}

function addSingleMedicineToBill(medicineName, dosage) {
   console.log('Adding single medicine to bill:', medicineName, dosage);
   
   // You can implement medicine search and add functionality here
   showAlert(`${medicineName} search functionality not implemented yet`, 'info');
}

function addAllPrescriptionItems() {
   const prescriptionId = $('.prescription-details').find('h5').text().match(/\#(\d+)/)?.[1];
   if (prescriptionId) {
       addPrescriptionToBill(prescriptionId);
       $('#prescriptionModal').modal('hide');
   }
}

// ===================================================
// UTILITY FUNCTIONS
// ===================================================
function updateWorkflowStep(step) {
   $('.workflow-step').removeClass('active completed');
   
   for (var i = 1; i <= 5; i++) {
       if (i < step) {
           $('#step' + i).addClass('completed');
       } else if (i === step) {
           $('#step' + i).addClass('active');
       }
   }
}

function resetForm() {
   console.log('Resetting form...');
   
   // Reset all variables
   selectedCustomer = null;
   customerType = null;
   billItems = [];
   billSubtotal = 0;
   storedPrescriptions = [];
   
   // Reset form fields
   $('#patientSearch').val('');
   $('#walkinName').val('');
   $('#walkinPhone').val('');
   $('#walkinAge').val('');
   $('#walkinGender').val('');
   $('#medicineSearch').val('');
   $('#medicineCategory').val('');
   $('#discountPercentage').val('0');
   $('#taxPercentage').val('0');
   $('#amountPaid').val('');
   $('#billNotes').val('');
   $('#paymentMethod').val('cash');
   
   // Reset button states
   $('.customer-type-btn').removeClass('active');
   
   // Initialize page
   initializePage();
   
   showAlert('Form reset. You can start a new transaction.', 'info');
}

function showAlert(message, type) {
   var alertClass = 'alert-info';
   if (type === 'success') alertClass = 'alert-success';
   else if (type === 'warning') alertClass = 'alert-warning';
   else if (type === 'danger') alertClass = 'alert-danger';
   
   $('#alertBox').removeClass('alert-success alert-warning alert-danger alert-info').addClass(alertClass);
   $('#alertMessage').text(message);
   $('#alertContainer').show();
   
   // Auto-hide after 5 seconds
   setTimeout(function() {
       $('#alertContainer').fadeOut();
   }, 5000);
}

function formatDate(dateString) {
   if (!dateString) return 'N/A';
   try {
       const date = new Date(dateString);
       return date.toLocaleDateString('en-US', {
           year: 'numeric',
           month: 'short',
           day: 'numeric'
       });
   } catch (e) {
       return dateString;
   }
}

function formatTime(dateString) {
   try {
       const date = new Date(dateString);
       return date.toLocaleTimeString('en-US', {
           hour: '2-digit',
           minute: '2-digit'
       });
   } catch (e) {
       return '';
   }
}

function getBillStatusClass(status) {
   switch(status) {
       case 'paid': return 'label-success';
       case 'pending': return 'label-warning';
       case 'cancelled': return 'label-danger';
       case 'partial': return 'label-info';
       default: return 'label-default';
   }
}

function viewBillDetails(billId) {
   console.log('Viewing bill details:', billId);
   showAlert('Bill details functionality not implemented yet', 'info');
}

function printBill(billId) {
   console.log('Printing bill:', billId);
   const printUrl = '<?php echo base_url(); ?>admin/dispensary/printBill/' + billId;
   window.open(printUrl, '_blank');
}

// Make functions globally available for onclick handlers
window.selectCustomerType = selectCustomerType;
window.confirmWalkinCustomer = confirmWalkinCustomer;
window.changeCustomer = changeCustomer;
window.searchPatients = searchPatients;
window.selectPatient = selectPatient;
window.proceedToMedicineSelection = proceedToMedicineSelection;
window.searchMedicines = searchMedicines;
window.selectMedicine = selectMedicine;
window.updateBatchInfo = updateBatchInfo;
window.updateTotal = updateTotal;
window.addMedicineToBill = addMedicineToBill;
window.updateItemQuantity = updateItemQuantity;
window.removeBillItem = removeBillItem;
window.calculateBill = calculateBill;
window.calculateChange = calculateChange;
window.clearBill = clearBill;
window.processBill = processBill;
window.viewCustomerHistory = viewCustomerHistory;
window.viewPrescriptionDetails = viewPrescriptionDetails;
window.addPrescriptionToBill = addPrescriptionToBill;
window.addSingleMedicineToBill = addSingleMedicineToBill;
window.addAllPrescriptionItems = addAllPrescriptionItems;
window.viewBillDetails = viewBillDetails;
window.printBill = printBill;

console.log('Enhanced dispensary functions loaded successfully with Walk-in support');
</script>