<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Supplier_model extends CI_Model {
    
    public function __construct() {
        parent::__construct();
        $this->load->database();
        $this->createTableIfNotExists();
    }
    
    private function createTableIfNotExists() {
        if (!$this->db->table_exists('suppliers')) {
            $sql = "CREATE TABLE `suppliers` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `supplier_name` varchar(200) NOT NULL,
                `supplier_code` varchar(50) UNIQUE NOT NULL,
                `contact_person` varchar(100) DEFAULT NULL,
                `email` varchar(100) DEFAULT NULL,
                `phone` varchar(20) DEFAULT NULL,
                `address` text,
                `city` varchar(100) DEFAULT NULL,
                `supplier_category` enum('medical','pharmaceutical','equipment','general','services') DEFAULT 'general',
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
            $this->insertSampleData();
        }
    }
    
    private function insertSampleData() {
        $suppliers = [
            ['supplier_name' => 'Meditech Kenya Ltd', 'supplier_code' => 'MED001', 'contact_person' => 'John Kamau', 'email' => 'sales@meditech.co.ke', 'phone' => '+254722123456', 'supplier_category' => 'medical'],
            ['supplier_name' => 'PharmaCare Distributors', 'supplier_code' => 'PHARM001', 'contact_person' => 'Mary Wanjiku', 'email' => 'orders@pharmacare.co.ke', 'phone' => '+254733987654', 'supplier_category' => 'pharmaceutical'],
            ['supplier_name' => 'Office Solutions Kenya', 'supplier_code' => 'OFF001', 'contact_person' => 'Peter Mwangi', 'email' => 'info@officesolutions.co.ke', 'phone' => '+254711222333', 'supplier_category' => 'general']
        ];
        $this->db->insert_batch('suppliers', $suppliers);
    }
    
    public function saveSupplier($data) {
        if (empty($data['supplier_code'])) {
            $data['supplier_code'] = $this->generateSupplierCode($data['supplier_name']);
        }
        $this->db->insert('suppliers', $data);
        return $this->db->insert_id();
    }
    
    public function updateSupplier($supplier_id, $data) {
        $this->db->where('id', $supplier_id);
        return $this->db->update('suppliers', $data);
    }
    
    public function getActiveSuppliers() {
        $this->db->select('*');
        $this->db->from('suppliers');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('supplier_name', 'asc');
        $query = $this->db->get();
        return $query->result_array();
    }
    
    public function getSupplierDetails($supplier_id) {
        $this->db->select('*');
        $this->db->from('suppliers');
        $this->db->where('id', $supplier_id);
        $query = $this->db->get();
        return $query->row_array();
    }
    
    public function getSuppliersList($start = 0, $length = 10, $search = '', $orderColumn = 0, $orderDir = 'desc') {
        $this->db->select('*');
        $this->db->from('suppliers');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('supplier_name', $search);
            $this->db->or_like('supplier_code', $search);
            $this->db->or_like('contact_person', $search);
            $this->db->group_end();
        }
        
        $total = $this->db->count_all_results('', FALSE);
        $filtered = $total;
        
        $this->db->order_by('supplier_name', $orderDir);
        $this->db->limit($length, $start);
        $query = $this->db->get();
        
        return [
            'data' => $query->result_array(),
            'total' => $total,
            'filtered' => $filtered
        ];
    }
    
    private function generateSupplierCode($supplier_name) {
        $prefix = strtoupper(substr(preg_replace('/[^A-Za-z]/', '', $supplier_name), 0, 3));
        $this->db->select('COUNT(*) + 1 as next_seq');
        $this->db->from('suppliers');
        $this->db->like('supplier_code', $prefix, 'after');
        $query = $this->db->get();
        $sequence = $query->row()->next_seq;
        return $prefix . str_pad($sequence, 3, '0', STR_PAD_LEFT);
    }
}