<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Reports Model
 * 
 * Handles all database operations for reports
 * Location: application/models/Reports_model.php
 */
class Reports_model extends CI_Model {
    
    public function __construct() {
        parent::__construct();
    }
    
    /**
     * Revenue Report Methods
     */
    
    public function getTotalRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    COALESCE(SUM(amount), 0) as total_revenue,
                    COUNT(DISTINCT patient_id) as unique_patients,
                    COUNT(*) as total_transactions
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND type = 'payment'
                    AND status != 'cancelled'";
        
        return $this->db->query($sql, array($date_from, $date_to))->row();
    }
    
    public function getRevenueByPaymentMethod($date_from, $date_to) {
        $sql = "SELECT 
                    payment_mode,
                    COUNT(*) as transaction_count,
                    COALESCE(SUM(amount), 0) as total_amount,
                    ROUND((COALESCE(SUM(amount), 0) / (
                        SELECT SUM(amount) 
                        FROM transactions 
                        WHERE DATE(payment_date) BETWEEN ? AND ? 
                        AND type = 'payment'
                        AND status != 'cancelled'
                    )) * 100, 2) as percentage
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND type = 'payment'
                    AND status != 'cancelled'
                GROUP BY payment_mode
                ORDER BY total_amount DESC";
        
        return $this->db->query($sql, array($date_from, $date_to, $date_from, $date_to))->result();
    }
    
    public function getRevenueTrend($date_from, $date_to, $group_by = 'day') {
        $date_format = '%Y-%m-%d';
        $interval = '1 DAY';
        
        switch($group_by) {
            case 'week':
                $date_format = '%Y-%u';
                $interval = '1 WEEK';
                break;
            case 'month':
                $date_format = '%Y-%m';
                $interval = '1 MONTH';
                break;
        }
        
        $sql = "SELECT 
                    DATE_FORMAT(payment_date, ?) as period,
                    COALESCE(SUM(amount), 0) as revenue,
                    COUNT(*) as transaction_count
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND type = 'payment'
                    AND status != 'cancelled'
                GROUP BY period
                ORDER BY payment_date ASC";
        
        return $this->db->query($sql, array($date_format, $date_from, $date_to))->result();
    }
    
    public function getRevenueComparison($date_from, $date_to) {
        // Calculate previous period
        $date1 = new DateTime($date_from);
        $date2 = new DateTime($date_to);
        $interval = $date1->diff($date2);
        $days = $interval->days + 1;
        
        $prev_date_to = date('Y-m-d', strtotime($date_from . ' -1 day'));
        $prev_date_from = date('Y-m-d', strtotime($prev_date_to . ' -' . ($days - 1) . ' days'));
        
        // Current period
        $current = $this->getTotalRevenue($date_from, $date_to);
        
        // Previous period
        $sql = "SELECT 
                    COALESCE(SUM(amount), 0) as total_revenue
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND type = 'payment'
                    AND status != 'cancelled'";
        
        $previous = $this->db->query($sql, array($prev_date_from, $prev_date_to))->row();
        
        $growth = 0;
        if ($previous->total_revenue > 0) {
            $growth = (($current->total_revenue - $previous->total_revenue) / $previous->total_revenue) * 100;
        }
        
        return array(
            'current_period' => $current->total_revenue,
            'previous_period' => $previous->total_revenue,
            'growth_percentage' => round($growth, 2),
            'growth_amount' => $current->total_revenue - $previous->total_revenue
        );
    }
    
    public function getDepartmentRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    department,
                    SUM(revenue) as total_revenue,
                    SUM(patient_count) as total_patients,
                    ROUND(SUM(revenue) / NULLIF(SUM(patient_count), 0), 2) as avg_per_patient
                FROM (
                    -- OPD Revenue
                    SELECT 
                        'OPD' as department,
                        COALESCE(SUM(t.amount), 0) as revenue,
                        COUNT(DISTINCT o.patient_id) as patient_count
                    FROM opd_details o
                    LEFT JOIN transactions t ON t.opd_id = o.id
                    WHERE DATE(o.appointment_date) BETWEEN ? AND ?
                        AND t.type = 'payment'
                        AND t.status != 'cancelled'
                    
                    UNION ALL
                    
                    -- IPD Revenue
                    SELECT 
                        'IPD' as department,
                        COALESCE(SUM(t.amount), 0) as revenue,
                        COUNT(DISTINCT i.patient_id) as patient_count
                    FROM ipd_details i
                    LEFT JOIN transactions t ON t.ipd_id = i.id
                    WHERE DATE(i.date_of_admission) BETWEEN ? AND ?
                        AND t.type = 'payment'
                        AND t.status != 'cancelled'
                    
                    UNION ALL
                    
                    -- Pharmacy Revenue
                    SELECT 
                        'Pharmacy' as department,
                        COALESCE(SUM(p.net_amount), 0) as revenue,
                        COUNT(DISTINCT p.patient_id) as patient_count
                    FROM pharmacy_bill_basic p
                    WHERE DATE(p.date) BETWEEN ? AND ?
                        AND p.payment_status != 'cancelled'
                    
                    UNION ALL
                    
                    -- Pathology Revenue
                    SELECT 
                        'Pathology' as department,
                        COALESCE(SUM(pb.net_amount), 0) as revenue,
                        COUNT(DISTINCT pb.patient_id) as patient_count
                    FROM pathology_billing pb
                    WHERE DATE(pb.reporting_date) BETWEEN ? AND ?
                    
                    UNION ALL
                    
                    -- Radiology Revenue
                    SELECT 
                        'Radiology' as department,
                        COALESCE(SUM(rb.net_amount), 0) as revenue,
                        COUNT(DISTINCT rb.patient_id) as patient_count
                    FROM radiology_billing rb
                    WHERE DATE(rb.reporting_date) BETWEEN ? AND ?
                    
                    UNION ALL
                    
                    -- Blood Bank Revenue
                    SELECT 
                        'Blood Bank' as department,
                        COALESCE(SUM(bb.net_amount), 0) as revenue,
                        COUNT(DISTINCT bb.patient_id) as patient_count
                    FROM blood_issue bb
                    WHERE DATE(bb.date_of_issue) BETWEEN ? AND ?
                    
                    UNION ALL
                    
                    -- Ambulance Revenue
                    SELECT 
                        'Ambulance' as department,
                        COALESCE(SUM(ac.amount), 0) as revenue,
                        COUNT(DISTINCT ac.patient_id) as patient_count
                    FROM ambulance_call ac
                    WHERE DATE(ac.date) BETWEEN ? AND ?
                ) as department_revenue
                GROUP BY department
                ORDER BY total_revenue DESC";
        
        $params = array_fill(0, 14, '');
        for ($i = 0; $i < 14; $i += 2) {
            $params[$i] = $date_from;
            $params[$i + 1] = $date_to;
        }
        
        return $this->db->query($sql, $params)->result();
    }
    
    /**
     * Patient Visit Report Methods
     */
    
    public function getTotalVisits($date_from, $date_to) {
        $sql = "SELECT 
                    COUNT(*) as total_visits,
                    COUNT(DISTINCT patient_id) as unique_patients,
                    COUNT(CASE WHEN appointment_date = DATE(appointment_date) THEN 1 END) as scheduled,
                    COUNT(CASE WHEN appointment_date != DATE(appointment_date) THEN 1 END) as walk_ins
                FROM opd_details
                WHERE DATE(appointment_date) BETWEEN ? AND ?";
        
        return $this->db->query($sql, array($date_from, $date_to))->row();
    }
    
    public function getVisitBreakdown($date_from, $date_to) {
        $sql = "SELECT 
                    COUNT(CASE WHEN p.patient_type = 'walk-in' THEN 1 END) as walk_in_count,
                    COUNT(CASE WHEN p.patient_type != 'walk-in' THEN 1 END) as registered_count,
                    COUNT(CASE WHEN p.patient_type = 'walk-in' THEN 1 END) * 100.0 / COUNT(*) as walk_in_percentage,
                    COUNT(CASE WHEN p.patient_type != 'walk-in' THEN 1 END) * 100.0 / COUNT(*) as registered_percentage
                FROM opd_details o
                JOIN patients p ON o.patient_id = p.id
                WHERE DATE(o.appointment_date) BETWEEN ? AND ?";
        
        return $this->db->query($sql, array($date_from, $date_to))->row();
    }
    
    public function getPatientTypeBreakdown($date_from, $date_to) {
        $sql = "SELECT 
                    COUNT(CASE WHEN visit_count = 1 THEN 1 END) as new_patients,
                    COUNT(CASE WHEN visit_count > 1 THEN 1 END) as returning_patients
                FROM (
                    SELECT 
                        patient_id,
                        COUNT(*) as visit_count
                    FROM opd_details
                    WHERE DATE(appointment_date) <= ?
                    GROUP BY patient_id
                ) as patient_visits
                WHERE patient_id IN (
                    SELECT DISTINCT patient_id 
                    FROM opd_details 
                    WHERE DATE(appointment_date) BETWEEN ? AND ?
                )";
        
        return $this->db->query($sql, array($date_to, $date_from, $date_to))->row();
    }
    
    public function getHourlyVisitDistribution($date_from, $date_to) {
        $sql = "SELECT 
                    HOUR(appointment_date) as hour,
                    COUNT(*) as visit_count,
                    AVG(TIMESTAMPDIFF(MINUTE, appointment_date, consultation_start_time)) as avg_wait_time
                FROM opd_details
                WHERE DATE(appointment_date) BETWEEN ? AND ?
                GROUP BY HOUR(appointment_date)
                ORDER BY hour";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    public function getDepartmentVisits($date_from, $date_to) {
        $sql = "SELECT 
                    d.department_name,
                    COUNT(o.id) as visit_count,
                    COUNT(DISTINCT o.patient_id) as unique_patients,
                    AVG(TIMESTAMPDIFF(MINUTE, o.appointment_date, o.consultation_start_time)) as avg_wait_time
                FROM opd_details o
                JOIN staff s ON o.cons_doctor = s.id
                JOIN department d ON s.department_id = d.id
                WHERE DATE(o.appointment_date) BETWEEN ? AND ?
                GROUP BY d.id, d.department_name
                ORDER BY visit_count DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    public function getAverageWaitTimes($date_from, $date_to) {
        $sql = "SELECT 
                    AVG(TIMESTAMPDIFF(MINUTE, appointment_date, consultation_start_time)) as overall_avg,
                    MIN(TIMESTAMPDIFF(MINUTE, appointment_date, consultation_start_time)) as min_wait,
                    MAX(TIMESTAMPDIFF(MINUTE, appointment_date, consultation_start_time)) as max_wait,
                    STD(TIMESTAMPDIFF(MINUTE, appointment_date, consultation_start_time)) as std_deviation
                FROM opd_details
                WHERE DATE(appointment_date) BETWEEN ? AND ?
                    AND consultation_start_time IS NOT NULL";
        
        return $this->db->query($sql, array($date_from, $date_to))->row();
    }
    
    /**
     * Insurance Report Methods
     */
    
    public function getInsuranceClaimsSummary($date_from, $date_to, $insurance_id = null) {
        $sql = "SELECT 
                    o.organisation as insurance_company,
                    o.id as insurance_id,
                    COUNT(DISTINCT t.id) as total_claims,
                    COALESCE(SUM(t.amount), 0) as total_amount,
                    COUNT(CASE WHEN t.payment_mode = 'Approved' THEN 1 END) as approved_claims,
                    COUNT(CASE WHEN t.payment_mode = 'Pending' THEN 1 END) as pending_claims,
                    COUNT(CASE WHEN t.payment_mode = 'Rejected' THEN 1 END) as rejected_claims
                FROM transactions t
                JOIN organisation o ON t.organisation_id = o.id
                WHERE DATE(t.payment_date) BETWEEN ? AND ?
                    AND t.payment_mode IN ('Insurance', 'Approved', 'Pending', 'Rejected')";
        
        $params = array($date_from, $date_to);
        
        if ($insurance_id) {
            $sql .= " AND o.id = ?";
            $params[] = $insurance_id;
        }
        
        $sql .= " GROUP BY o.id, o.organisation ORDER BY total_amount DESC";
        
        return $this->db->query($sql, $params)->result();
    }
    
    public function getPendingInsuranceClaims($insurance_id = null) {
        $sql = "SELECT 
                    t.id as claim_id,
                    p.patient_name,
                    p.patient_unique_id,
                    o.organisation as insurance_company,
                    t.amount as claim_amount,
                    t.payment_date as claim_date,
                    DATEDIFF(CURDATE(), t.payment_date) as days_pending
                FROM transactions t
                JOIN patients p ON t.patient_id = p.id
                JOIN organisation o ON t.organisation_id = o.id
                WHERE t.payment_mode = 'Pending'";
        
        $params = array();
        
        if ($insurance_id) {
            $sql .= " AND o.id = ?";
            $params[] = $insurance_id;
        }
        
        $sql .= " ORDER BY days_pending DESC";
        
        return $this->db->query($sql, $params)->result();
    }
    
    public function getClaimStatusBreakdown($date_from, $date_to, $insurance_id = null) {
        $sql = "SELECT 
                    payment_mode as status,
                    COUNT(*) as count,
                    COALESCE(SUM(amount), 0) as total_amount
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND payment_mode IN ('Approved', 'Pending', 'Rejected')";
        
        $params = array($date_from, $date_to);
        
        if ($insurance_id) {
            $sql .= " AND organisation_id = ?";
            $params[] = $insurance_id;
        }
        
        $sql .= " GROUP BY payment_mode";
        
        return $this->db->query($sql, $params)->result();
    }
    
    public function getInsuranceCompanyBreakdown($date_from, $date_to) {
        $sql = "SELECT 
                    o.organisation,
                    o.code,
                    COUNT(DISTINCT t.patient_id) as patient_count,
                    COUNT(t.id) as claim_count,
                    COALESCE(SUM(t.amount), 0) as total_claimed,
                    COALESCE(SUM(CASE WHEN t.payment_mode = 'Approved' THEN t.amount ELSE 0 END), 0) as approved_amount,
                    ROUND((COALESCE(SUM(CASE WHEN t.payment_mode = 'Approved' THEN t.amount ELSE 0 END), 0) / 
                           NULLIF(COALESCE(SUM(t.amount), 0), 0)) * 100, 2) as approval_rate
                FROM organisation o
                LEFT JOIN transactions t ON o.id = t.organisation_id 
                    AND DATE(t.payment_date) BETWEEN ? AND ?
                WHERE o.organisation_type = 'Insurance'
                GROUP BY o.id
                ORDER BY total_claimed DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    /**
     * Cash Payment Report Methods
     */
    
    public function getTotalCashCollected($date_from, $date_to, $cashier_id = null) {
        $sql = "SELECT 
                    COALESCE(SUM(amount), 0) as total_cash,
                    COUNT(*) as transaction_count,
                    COUNT(DISTINCT patient_id) as patient_count,
                    COUNT(DISTINCT received_by) as cashier_count
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND payment_mode = 'Cash'
                    AND type = 'payment'
                    AND status != 'cancelled'";
        
        $params = array($date_from, $date_to);
        
        if ($cashier_id) {
            $sql .= " AND received_by = ?";
            $params[] = $cashier_id;
        }
        
        return $this->db->query($sql, $params)->row();
    }
    
    public function getCashierWiseCollection($date_from, $date_to) {
        $sql = "SELECT 
                    s.id as cashier_id,
                    s.name as cashier_name,
                    s.employee_id,
                    COUNT(t.id) as transaction_count,
                    COALESCE(SUM(t.amount), 0) as total_collected,
                    COUNT(DISTINCT DATE(t.payment_date)) as days_worked
                FROM staff s
                JOIN transactions t ON s.id = t.received_by
                WHERE DATE(t.payment_date) BETWEEN ? AND ?
                    AND t.payment_mode = 'Cash'
                    AND t.type = 'payment'
                    AND t.status != 'cancelled'
                GROUP BY s.id
                ORDER BY total_collected DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    public function getDenominationBreakdown($date_from, $date_to) {
        // This would require a denomination tracking table
        // Returning placeholder data structure
        return array(
            array('denomination' => 1000, 'count' => 0, 'total' => 0),
            array('denomination' => 500, 'count' => 0, 'total' => 0),
            array('denomination' => 100, 'count' => 0, 'total' => 0),
            array('denomination' => 50, 'count' => 0, 'total' => 0),
            array('denomination' => 20, 'count' => 0, 'total' => 0),
            array('denomination' => 10, 'count' => 0, 'total' => 0)
        );
    }
    
    public function getHourlyCashFlow($date_from, $date_to) {
        $sql = "SELECT 
                    HOUR(payment_date) as hour,
                    COUNT(*) as transaction_count,
                    COALESCE(SUM(amount), 0) as total_amount
                FROM transactions
                WHERE DATE(payment_date) BETWEEN ? AND ?
                    AND payment_mode = 'Cash'
                    AND type = 'payment'
                    AND status != 'cancelled'
                GROUP BY HOUR(payment_date)
                ORDER BY hour";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    /**
     * Outstanding Payment Report Methods
     */
    
    public function getTotalOutstanding($department = 'all') {
        $sql = "SELECT 
                    COALESCE(SUM(balance_amount), 0) as total_outstanding,
                    COUNT(DISTINCT patient_id) as patient_count,
                    AVG(DATEDIFF(CURDATE(), bill_date)) as avg_days_outstanding
                FROM (";
        
        $union_parts = array();
        
        if ($department == 'all' || $department == 'OPD') {
            $union_parts[] = "SELECT 
                patient_id,
                total_amount - paid_amount as balance_amount,
                date as bill_date
            FROM opd_billing
            WHERE total_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'IPD') {
            $union_parts[] = "SELECT 
                patient_id,
                total_amount - paid_amount as balance_amount,
                date as bill_date
            FROM ipd_billing
            WHERE total_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'Pharmacy') {
            $union_parts[] = "SELECT 
                patient_id,
                net_amount - paid_amount as balance_amount,
                date as bill_date
            FROM pharmacy_bill_basic
            WHERE net_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'Pathology') {
            $union_parts[] = "SELECT 
                patient_id,
                net_amount - paid_amount as balance_amount,
                reporting_date as bill_date
            FROM pathology_billing
            WHERE net_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'Radiology') {
            $union_parts[] = "SELECT 
                patient_id,
                net_amount - paid_amount as balance_amount,
                reporting_date as bill_date
            FROM radiology_billing
            WHERE net_amount > paid_amount";
        }
        
        $sql .= implode(' UNION ALL ', $union_parts) . ") as outstanding_bills";
        
        return $this->db->query($sql)->row();
    }
    
    public function getAgingAnalysis($aging_days = 30, $department = 'all') {
        $sql = "SELECT 
                    CASE 
                        WHEN days_outstanding <= ? THEN CONCAT('0-', ?, ' days')
                        WHEN days_outstanding <= ? * 2 THEN CONCAT(? + 1, '-', ? * 2, ' days')
                        WHEN days_outstanding <= ? * 3 THEN CONCAT(? * 2 + 1, '-', ? * 3, ' days')
                        ELSE CONCAT('Over ', ? * 3, ' days')
                    END as aging_bucket,
                    COUNT(*) as bill_count,
                    COUNT(DISTINCT patient_id) as patient_count,
                    COALESCE(SUM(balance_amount), 0) as total_amount
                FROM (";
        
        $union_parts = array();
        
        if ($department == 'all' || $department == 'OPD') {
            $union_parts[] = "SELECT 
                patient_id,
                total_amount - paid_amount as balance_amount,
                DATEDIFF(CURDATE(), date) as days_outstanding
            FROM opd_billing
            WHERE total_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'IPD') {
            $union_parts[] = "SELECT 
                patient_id,
                total_amount - paid_amount as balance_amount,
                DATEDIFF(CURDATE(), date) as days_outstanding
            FROM ipd_billing
            WHERE total_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'Pharmacy') {
            $union_parts[] = "SELECT 
                patient_id,
                net_amount - paid_amount as balance_amount,
                DATEDIFF(CURDATE(), date) as days_outstanding
            FROM pharmacy_bill_basic
            WHERE net_amount > paid_amount";
        }
        
        $sql .= implode(' UNION ALL ', $union_parts) . ") as outstanding_bills
                GROUP BY aging_bucket
                ORDER BY MIN(days_outstanding)";
        
        $params = array_fill(0, 9, $aging_days);
        
        return $this->db->query($sql, $params)->result();
    }
    
    public function getPatientOutstanding($department = 'all') {
        $sql = "SELECT 
                    p.id as patient_id,
                    p.patient_name,
                    p.patient_unique_id,
                    p.mobileno,
                    COALESCE(SUM(o.balance_amount), 0) as total_outstanding,
                    MIN(o.bill_date) as oldest_bill_date,
                    MAX(o.bill_date) as latest_bill_date,
                    COUNT(o.bill_id) as bill_count
                FROM patients p
                JOIN (";
        
        $union_parts = array();
        
        if ($department == 'all' || $department == 'OPD') {
            $union_parts[] = "SELECT 
                id as bill_id,
                patient_id,
                total_amount - paid_amount as balance_amount,
                date as bill_date
            FROM opd_billing
            WHERE total_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'IPD') {
            $union_parts[] = "SELECT 
                id as bill_id,
                patient_id,
                total_amount - paid_amount as balance_amount,
                date as bill_date
            FROM ipd_billing
            WHERE total_amount > paid_amount";
        }
        
        if ($department == 'all' || $department == 'Pharmacy') {
            $union_parts[] = "SELECT 
                id as bill_id,
                patient_id,
                net_amount - paid_amount as balance_amount,
                date as bill_date
            FROM pharmacy_bill_basic
            WHERE net_amount > paid_amount";
        }
        
        $sql .= implode(' UNION ALL ', $union_parts) . ") as o ON p.id = o.patient_id
                GROUP BY p.id
                ORDER BY total_outstanding DESC
                LIMIT 100";
        
        return $this->db->query($sql)->result();
    }
    
    public function getInsuranceOutstanding() {
        $sql = "SELECT 
                    o.id as insurance_id,
                    o.organisation as insurance_company,
                    o.code as insurance_code,
                    COUNT(DISTINCT t.patient_id) as patient_count,
                    COUNT(t.id) as claim_count,
                    COALESCE(SUM(t.amount), 0) as total_outstanding,
                    MIN(t.payment_date) as oldest_claim,
                    AVG(DATEDIFF(CURDATE(), t.payment_date)) as avg_days_pending
                FROM organisation o
                JOIN transactions t ON o.id = t.organisation_id
                WHERE t.payment_mode = 'Pending'
                    AND o.organisation_type = 'Insurance'
                GROUP BY o.id
                ORDER BY total_outstanding DESC";
        
        return $this->db->query($sql)->result();
    }
    
    /**
     * Department Detailed Revenue Methods
     */
    
    public function getDepartmentDetailedRevenue($department, $date_from, $date_to) {
        switch($department) {
            case 'OPD':
                return $this->getOPDRevenue($date_from, $date_to);
            case 'IPD':
                return $this->getIPDRevenue($date_from, $date_to);
            case 'Pharmacy':
                return $this->getPharmacyRevenue($date_from, $date_to);
            case 'Pathology':
                return $this->getPathologyRevenue($date_from, $date_to);
            case 'Radiology':
                return $this->getRadiologyRevenue($date_from, $date_to);
            case 'Blood Bank':
                return $this->getBloodBankRevenue($date_from, $date_to);
            case 'Ambulance':
                return $this->getAmbulanceRevenue($date_from, $date_to);
            default:
                return array();
        }
    }
    
    private function getOPDRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(o.appointment_date) as date,
                    COUNT(DISTINCT o.patient_id) as patient_count,
                    COUNT(o.id) as visit_count,
                    COALESCE(SUM(t.amount), 0) as revenue,
                    COUNT(DISTINCT o.cons_doctor) as doctor_count
                FROM opd_details o
                LEFT JOIN transactions t ON t.opd_id = o.id AND t.type = 'payment'
                WHERE DATE(o.appointment_date) BETWEEN ? AND ?
                GROUP BY DATE(o.appointment_date)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    private function getIPDRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(i.date_of_admission) as date,
                    COUNT(DISTINCT i.patient_id) as patient_count,
                    COUNT(i.id) as admission_count,
                    COALESCE(SUM(t.amount), 0) as revenue,
                    AVG(DATEDIFF(i.date_of_discharge, i.date_of_admission)) as avg_stay_days
                FROM ipd_details i
                LEFT JOIN transactions t ON t.ipd_id = i.id AND t.type = 'payment'
                WHERE DATE(i.date_of_admission) BETWEEN ? AND ?
                    OR DATE(i.date_of_discharge) BETWEEN ? AND ?
                GROUP BY DATE(i.date_of_admission)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to, $date_from, $date_to))->result();
    }
    
    private function getPharmacyRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(p.date) as date,
                    COUNT(DISTINCT p.patient_id) as patient_count,
                    COUNT(p.id) as bill_count,
                    COALESCE(SUM(p.net_amount), 0) as revenue,
                    COUNT(DISTINCT pd.medicine_name) as unique_medicines
                FROM pharmacy_bill_basic p
                LEFT JOIN pharmacy_bill_detail pd ON pd.pharmacy_bill_basic_id = p.id
                WHERE DATE(p.date) BETWEEN ? AND ?
                    AND p.payment_status != 'cancelled'
                GROUP BY DATE(p.date)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    private function getPathologyRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(pb.reporting_date) as date,
                    COUNT(DISTINCT pb.patient_id) as patient_count,
                    COUNT(pb.id) as test_count,
                    COALESCE(SUM(pb.net_amount), 0) as revenue,
                    COUNT(DISTINCT pr.pathology_id) as unique_tests
                FROM pathology_billing pb
                LEFT JOIN pathology_report pr ON pr.pathology_bill_id = pb.id
                WHERE DATE(pb.reporting_date) BETWEEN ? AND ?
                GROUP BY DATE(pb.reporting_date)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    private function getRadiologyRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(rb.reporting_date) as date,
                    COUNT(DISTINCT rb.patient_id) as patient_count,
                    COUNT(rb.id) as test_count,
                    COALESCE(SUM(rb.net_amount), 0) as revenue,
                    COUNT(DISTINCT rr.radiology_id) as unique_tests
                FROM radiology_billing rb
                LEFT JOIN radiology_report rr ON rr.radiology_bill_id = rb.id
                WHERE DATE(rb.reporting_date) BETWEEN ? AND ?
                GROUP BY DATE(rb.reporting_date)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    private function getBloodBankRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(bi.date_of_issue) as date,
                    COUNT(DISTINCT bi.patient_id) as patient_count,
                    COUNT(bi.id) as issue_count,
                    COALESCE(SUM(bi.net_amount), 0) as revenue,
                    SUM(bi.quantity) as total_units
                FROM blood_issue bi
                WHERE DATE(bi.date_of_issue) BETWEEN ? AND ?
                GROUP BY DATE(bi.date_of_issue)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    private function getAmbulanceRevenue($date_from, $date_to) {
        $sql = "SELECT 
                    DATE(ac.date) as date,
                    COUNT(DISTINCT ac.patient_id) as patient_count,
                    COUNT(ac.id) as call_count,
                    COALESCE(SUM(ac.amount), 0) as revenue,
                    AVG(ac.distance) as avg_distance
                FROM ambulance_call ac
                WHERE DATE(ac.date) BETWEEN ? AND ?
                GROUP BY DATE(ac.date)
                ORDER BY date DESC";
        
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    /**
     * Quick Stats Methods
     */
    
    public function getTodayRevenue() {
        $sql = "SELECT COALESCE(SUM(amount), 0) as revenue
                FROM transactions
                WHERE DATE(payment_date) = CURDATE()
                    AND type = 'payment'
                    AND status != 'cancelled'";
        
        $result = $this->db->query($sql)->row();
        return $result->revenue;
    }
    
    public function getTodayPatients() {
        $sql = "SELECT COUNT(DISTINCT patient_id) as count
                FROM opd_details
                WHERE DATE(appointment_date) = CURDATE()";
        
        $result = $this->db->query($sql)->row();
        return $result->count;
    }
    
    public function getTodayAppointments() {
        $sql = "SELECT COUNT(*) as count
                FROM appointment
                WHERE DATE(date) = CURDATE()";
        
        $result = $this->db->query($sql)->row();
        return $result->count;
    }
    
    public function getTodayAdmissions() {
        $sql = "SELECT COUNT(*) as count
                FROM ipd_details
                WHERE DATE(date_of_admission) = CURDATE()";
        
        $result = $this->db->query($sql)->row();
        return $result->count;
    }
    
    public function getMonthRevenue() {
        $sql = "SELECT COALESCE(SUM(amount), 0) as revenue
                FROM transactions
                WHERE MONTH(payment_date) = MONTH(CURDATE())
                    AND YEAR(payment_date) = YEAR(CURDATE())
                    AND type = 'payment'
                    AND status != 'cancelled'";
        
        $result = $this->db->query($sql)->row();
        return $result->revenue;
    }
    
    public function getMonthPatients() {
        $sql = "SELECT COUNT(DISTINCT patient_id) as count
                FROM opd_details
                WHERE MONTH(appointment_date) = MONTH(CURDATE())
                    AND YEAR(appointment_date) = YEAR(CURDATE())";
        
        $result = $this->db->query($sql)->row();
        return $result->count;
    }
    
    public function getMonthGrowth() {
        // Current month revenue
        $current = $this->getMonthRevenue();
        
        // Previous month revenue
        $sql = "SELECT COALESCE(SUM(amount), 0) as revenue
                FROM transactions
                WHERE MONTH(payment_date) = MONTH(DATE_SUB(CURDATE(), INTERVAL 1 MONTH))
                    AND YEAR(payment_date) = YEAR(DATE_SUB(CURDATE(), INTERVAL 1 MONTH))
                    AND type = 'payment'
                    AND status != 'cancelled'";
        
        $result = $this->db->query($sql)->row();
        $previous = $result->revenue;
        
        if ($previous > 0) {
            return round((($current - $previous) / $previous) * 100, 2);
        }
        
        return 0;
    }
    
    /**
     * Custom Report Methods
     */
    
    public function getReportableTables() {
        // Return list of tables that can be used in custom reports
        return array(
            'patients' => array('id', 'patient_name', 'patient_unique_id', 'age', 'gender', 'mobileno'),
            'opd_details' => array('id', 'patient_id', 'appointment_date', 'cons_doctor', 'amount'),
            'ipd_details' => array('id', 'patient_id', 'date_of_admission', 'date_of_discharge'),
            'pharmacy_bill_basic' => array('id', 'patient_id', 'date', 'net_amount'),
            'pathology_billing' => array('id', 'patient_id', 'reporting_date', 'net_amount'),
            'radiology_billing' => array('id', 'patient_id', 'reporting_date', 'net_amount'),
            'transactions' => array('id', 'patient_id', 'payment_date', 'amount', 'payment_mode')
        );
    }
    
    public function getSavedCustomReports() {
        $sql = "SELECT * FROM report_configurations 
                WHERE is_active = 1 
                ORDER BY created_at DESC";
        
        return $this->db->query($sql)->result();
    }
    
    public function saveCustomReport($data) {
        $this->db->insert('report_configurations', $data);
        return $this->db->insert_id();
    }
    
    /**
     * Report Access Logging
     */
    
    public function logReportAccess($log_data) {
        $this->db->insert('report_access_log', $log_data);
    }
    
    /**
     * Department Comparison
     */
    
    public function getDepartmentComparison($date_from, $date_to) {
        $sql = "SELECT 
                    d.department,
                    d.revenue,
                    d.patient_count,
                    ROUND((d.revenue / t.total_revenue) * 100, 2) as revenue_percentage,
                    ROUND((d.patient_count / t.total_patients) * 100, 2) as patient_percentage
                FROM (
                    SELECT 
                        department,
                        SUM(revenue) as revenue,
                        SUM(patient_count) as patient_count
                    FROM (
                        -- Department revenue subquery (same as getDepartmentRevenue)
                        -- ... (implementation details)
                    ) as dept_data
                    GROUP BY department
                ) as d
                CROSS JOIN (
                    SELECT 
                        SUM(revenue) as total_revenue,
                        SUM(patient_count) as total_patients
                    FROM (
                        -- Total revenue subquery
                        -- ... (implementation details)
                    ) as total_data
                ) as t
                ORDER BY d.revenue DESC";
        
        // Implementation would include the full queries
        return $this->db->query($sql, array($date_from, $date_to))->result();
    }
    
    /**
     * Top Services by Department
     */
    
    public function getTopServicesByDepartment($department, $date_from, $date_to, $limit = 10) {
        // Implementation would vary by department
        // This is a placeholder structure
        $sql = "SELECT 
                    service_name,
                    COUNT(*) as service_count,
                    SUM(amount) as total_revenue
                FROM service_table
                WHERE department = ?
                    AND DATE(service_date) BETWEEN ? AND ?
                GROUP BY service_name
                ORDER BY total_revenue DESC
                LIMIT ?";
        
        return $this->db->query($sql, array($department, $date_from, $date_to, $limit))->result();
    }
}