<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Radiology_model extends CI_Model {

    function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // PATIENT MANAGEMENT METHODS
    // =============================================
        /**
     * Search patients for radiology services (Required by JavaScript)
     */
public function searchPatients($search_term) {
    try {
        $this->db->select('
            id,
            patient_name,
            patient_unique_id,
            guardian_name,
            gender,
            age,
            month,
            mobileno,
            dob,
            patient_workflow,
            patient_type,
            fingerprint_template,
            fingerprint_verified,
            created_at
        ');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        
        if (!empty($search_term)) {
            $this->db->group_start();
            $this->db->like('patient_name', $search_term);
            $this->db->or_like('patient_unique_id', $search_term);
            $this->db->or_like('mobileno', $search_term);
            $this->db->or_like('guardian_name', $search_term);
            $this->db->group_end();
        }
        
        $this->db->limit(20);
        $this->db->order_by('patient_name');
        
        $query = $this->db->get();
        $patients = $query->result_array();
        
        // Process patient data with proper workflow handling
        foreach ($patients as &$patient) {
            $patient['has_fingerprint'] = !empty($patient['fingerprint_template']);
            
            // CRITICAL FIX: Ensure patient_workflow is properly set
            $workflow = $patient['patient_workflow'];
            
            // Handle NULL, empty, or invalid workflow values
            if (empty($workflow) || $workflow === 'null') {
                $workflow = 'general';
            }
            
            // Normalize workflow value
            $workflow = strtolower(trim($workflow));
            
            // Ensure valid workflow
            if (!in_array($workflow, ['gcc', 'general', 'specialized'])) {
                $workflow = 'general';
            }
            
            // Set normalized values
            $patient['patient_workflow'] = $workflow;
            $patient['patient_type'] = strtoupper($workflow); // This is what gets passed to selectPatient
            
            // Remove sensitive data
            unset($patient['fingerprint_template']);
            
            // Debug info
            $patient['debug_workflow'] = $workflow;
        }
        
        return $patients;
        
    } catch (Exception $e) {
        log_message('error', 'Error searching patients: ' . $e->getMessage());
        return [];
    }
}


    /**
     * Get all patients available for radiology selection
     */
    public function getAllPatientsForRadiology() {
        try {
            $this->db->select('
                p.*,
                (SELECT MAX(ro.order_date) FROM radiology_orders ro WHERE ro.patient_id = p.id) as last_visit,
                (SELECT COUNT(*) FROM radiology_orders ro WHERE ro.patient_id = p.id) as total_orders
            ');
            $this->db->from('patients p');
            $this->db->where('p.is_active', 'yes');
            $this->db->order_by('p.created_at', 'DESC');
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting patients for radiology: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get patient by ID with comprehensive information
     */
    public function getPatientById($patient_id) {
        try {
            $this->db->select('
                p.*,
                (SELECT COUNT(*) FROM radiology_orders ro WHERE ro.patient_id = p.id) as radiology_count,
                (SELECT MAX(ro.order_date) FROM radiology_orders ro WHERE ro.patient_id = p.id) as last_radiology_date
            ');
            $this->db->from('patients p');
            $this->db->where('p.id', $patient_id);
            $this->db->where('p.is_active', 'yes');
            
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting patient by ID: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get patient's comprehensive radiology history
     */
    public function getPatientRadiologyHistory($patient_id) {
        try {
            // Check if radiology_billing table exists
            $billing_exists = $this->tableExists('radiology_billing');
            
            if ($billing_exists) {
                $this->db->select('
                    ro.id as order_id,
                    ro.order_date,
                    ro.priority,
                    ro.status,
                    ro.clinical_notes,
                    ro.completed_at,
                    s.name as doctor_name,
                    s.surname as doctor_surname,
                    tech.name as technician_name,
                    tech.surname as technician_surname,
                    GROUP_CONCAT(rt.test_name SEPARATOR ", ") as test_names,
                    COUNT(rot.id) as test_count,
                    COALESCE(rb.net_amount, 0) as net_amount,
                    COALESCE(rb.payment_status, "pending") as payment_status
                ');
            } else {
                $this->db->select('
                    ro.id as order_id,
                    ro.order_date,
                    ro.priority,
                    ro.status,
                    ro.clinical_notes,
                    ro.completed_at,
                    s.name as doctor_name,
                    s.surname as doctor_surname,
                    tech.name as technician_name,
                    tech.surname as technician_surname,
                    GROUP_CONCAT(rt.test_name SEPARATOR ", ") as test_names,
                    COUNT(rot.id) as test_count,
                    0 as net_amount,
                    "pending" as payment_status
                ');
            }
            
            $this->db->from('radiology_orders ro');
            $this->db->join('radiology_order_tests rot', 'rot.radiology_order_id = ro.id', 'left');
            $this->db->join('radiology_tests rt', 'rt.id = rot.test_id', 'left');
            $this->db->join('staff s', 's.id = ro.referring_doctor_id', 'left');
            $this->db->join('staff tech', 'tech.id = ro.technician_id', 'left');
            
            if ($billing_exists) {
                $this->db->join('radiology_billing rb', 'rb.order_id = ro.id', 'left');
            }
            
            $this->db->where('ro.patient_id', $patient_id);
            $this->db->group_by('ro.id');
            $this->db->order_by('ro.order_date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            
            if ($query === FALSE) {
                $error = $this->db->error();
                throw new Exception('Database query failed: ' . (isset($error['message']) ? $error['message'] : 'Unknown database error'));
            }
            
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting patient radiology history: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Check if a table exists in the database
     */
    private function tableExists($table_name) {
        try {
            return $this->db->table_exists($table_name);
        } catch (Exception $e) {
            log_message('error', 'Error checking if table exists: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Search patients for radiology with filters
     */
    public function searchPatientsForRadiology($search_term, $filters = array()) {
        try {
            $this->db->select('
                p.*,
                (SELECT MAX(ro.order_date) FROM radiology_orders ro WHERE ro.patient_id = p.id) as last_visit
            ');
            $this->db->from('patients p');
            $this->db->where('p.is_active', 'yes');
            
            if (!empty($search_term)) {
                $this->db->group_start();
                $this->db->like('p.patient_name', $search_term);
                $this->db->or_like('p.id', $search_term);
                $this->db->or_like('p.mobileno', $search_term);
                $this->db->or_like('p.guardian_name', $search_term);
                $this->db->group_end();
            }
            
            if (!empty($filters['gender'])) {
                $this->db->where('p.gender', $filters['gender']);
            }
            
            if (!empty($filters['age_min'])) {
                $this->db->where('p.age >=', $filters['age_min']);
            }
            
            if (!empty($filters['age_max'])) {
                $this->db->where('p.age <=', $filters['age_max']);
            }
            
            $this->db->order_by('p.patient_name', 'ASC');
            $this->db->limit(50);
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error searching patients: ' . $e->getMessage());
            return array();
        }
    }
    
    /**
     * Get patient workflow status for radiology compliance
     */
    public function getPatientWorkflowStatus($patient_id, $patient_type) {
        if ($patient_type !== 'GCC') {
            return array('status' => 'general', 'can_proceed' => true);
        }

        // Check GCC workflow compliance if workflow_step_history table exists
        if (!$this->tableExists('workflow_step_history')) {
            return array('status' => 'gcc', 'can_proceed' => true, 'message' => 'Workflow tracking not available');
        }

        try {
            $this->db->select('step_name, to_status, completion_time');
            $this->db->from('workflow_step_history');
            $this->db->where('patient_id', $patient_id);
            $this->db->where('workflow_type', 'gcc');
            $this->db->order_by('completion_time DESC');
            
            $steps = $this->db->get()->result_array();
            
            $completed_steps = array();
            foreach ($steps as $step) {
                if ($step['to_status'] === 'completed') {
                    $completed_steps[] = $step['step_name'];
                }
            }

            // Check if triage is completed (required for radiology)
            $triage_completed = in_array('triage', $completed_steps);
            
            return array(
                'status' => 'gcc',
                'completed_steps' => $completed_steps,
                'can_proceed' => $triage_completed,
                'message' => $triage_completed ? 'Ready for radiology' : 'Triage required before radiology'
            );
        } catch (Exception $e) {
            log_message('error', 'Error checking workflow status: ' . $e->getMessage());
            return array('status' => 'gcc', 'can_proceed' => true, 'message' => 'Workflow check failed');
        }
    }

    /**
     * Get pending radiology orders
     */
    public function getPendingOrders($limit = 20) {
        if (!$this->tableExists('radiology_orders')) {
            return array();
        }

        try {
            $this->db->select('ro.*, p.patient_name, p.patient_unique_id, p.mobileno, 
                              p.fingerprint_template, p.patient_type, s.name as doctor_name');
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'ro.patient_id = p.id');
            $this->db->join('staff s', 'ro.referring_doctor_id = s.id', 'left');
            $this->db->where('ro.status', 'pending');
            $this->db->order_by('ro.priority DESC, ro.order_date ASC');
            $this->db->limit($limit);
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting pending orders: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get today's appointments
     */
    public function getTodayAppointments() {
        if (!$this->tableExists('radiology_scheduling')) {
            return array();
        }

        try {
            $today = date('Y-m-d');
            
            $this->db->select('rs.*, ro.patient_id, p.patient_name, p.patient_unique_id, 
                              re.equipment_name, s.name as technician_name');
            $this->db->from('radiology_scheduling rs');
            $this->db->join('radiology_orders ro', 'rs.order_id = ro.id');
            $this->db->join('patients p', 'ro.patient_id = p.id');
            $this->db->join('radiology_equipment re', 'rs.equipment_id = re.id', 'left');
            $this->db->join('staff s', 'rs.technician_id = s.id', 'left');
            $this->db->where('rs.scheduled_date', $today);
            $this->db->where_in('rs.status', ['scheduled', 'confirmed', 'in_progress']);
            $this->db->order_by('rs.scheduled_time ASC');
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting today appointments: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get active radiology tests
     */
    public function getActiveTests() {
        try {
            $this->db->select('r.*, rc.category_name, c.charge_category, c.charge_type, c.standard_charge');
            $this->db->from('radio r');
            $this->db->join('radiology_category rc', 'r.radiology_category_id = rc.id', 'left');
            $this->db->join('charges c', 'r.charge_id = c.id', 'left');
            $this->db->where('c.is_active', 'yes');
            $this->db->order_by('rc.category_name, r.test_name');
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting active tests: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get active equipment
     */
    public function getActiveEquipment() {
        if (!$this->tableExists('radiology_equipment')) {
            return array();
        }

        try {
            $this->db->select('*');
            $this->db->from('radiology_equipment');
            $this->db->where('status', 'active');
            $this->db->order_by('equipment_name');
            
            return $this->db->get()->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting active equipment: ' . $e->getMessage());
            return array();
        }
    }

    // =============================================
    // RADIOLOGY ORDER MANAGEMENT
    // =============================================

    /**
     * Create new radiology order
     */
    public function createRadiologyOrder($order_data) {
        try {
            $this->db->trans_start();
            
            // Only set order_number if the column exists
            if ($this->db->field_exists('order_number', 'radiology_orders') && empty($order_data['order_number'])) {
                $order_data['order_number'] = $this->generateOrderNumber();
            }
            
            // Remove order_number from data if column doesn't exist
            if (!$this->db->field_exists('order_number', 'radiology_orders')) {
                unset($order_data['order_number']);
            }
            
            $this->db->insert('radiology_orders', $order_data);
            $order_id = $this->db->insert_id();
            
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Failed to create radiology order');
            }
            
            return $order_id;
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Error creating radiology order: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Add test to radiology order
     */
    public function addOrderTest($test_data) {
        try {
            $this->db->insert('radiology_order_tests', $test_data);
            return $this->db->insert_id();
        } catch (Exception $e) {
            log_message('error', 'Error adding order test: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Add multiple tests to radiology order
     */
    public function addMultipleOrderTests($order_id, $test_ids, $additional_data = array()) {
        try {
            $this->db->trans_start();
            
            foreach ($test_ids as $test_id) {
                $test_data = array_merge(array(
                    'radiology_order_id' => $order_id,
                    'test_id' => $test_id,
                    'status' => 'pending',
                    'created_at' => date('Y-m-d H:i:s')
                ), $additional_data);
                
                $this->db->insert('radiology_order_tests', $test_data);
            }
            
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Failed to add tests to order');
            }
            
            return true;
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Error adding multiple order tests: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Update radiology order
     */
    public function updateRadiologyOrder($order_id, $update_data) {
        try {
            $update_data['updated_at'] = date('Y-m-d H:i:s');
            
            $this->db->where('id', $order_id);
            $result = $this->db->update('radiology_orders', $update_data);
            
            if ($result) {
                // Log the update
                $this->logActivity(array(
                    'order_id' => $order_id,
                    'activity_type' => 'order_updated',
                    'description' => 'Radiology order updated',
                    'timestamp' => date('Y-m-d H:i:s')
                ));
            }
            
            return $result;
        } catch (Exception $e) {
            log_message('error', 'Error updating radiology order: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get radiology order by ID with full details
     */
    public function getRadiologyOrderDetails($order_id) {
        try {
            $this->db->select('
                ro.*,
                p.patient_name,
                p.id,
                p.age,
                p.gender,
                p.mobileno,
                p.address,
                p.known_allergies,
                s.name as doctor_name,
                s.surname as doctor_surname,
                s.employee_id as doctor_employee_id,
                tech.name as technician_name,
                tech.surname as technician_surname,
                rad.name as radiologist_name,
                rad.surname as radiologist_surname
            ');
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'p.id = ro.patient_id', 'inner');
            $this->db->join('staff s', 's.id = ro.referring_doctor_id', 'left');
            $this->db->join('staff tech', 'tech.id = ro.technician_id', 'left');
            $this->db->join('staff rad', 'rad.id = ro.radiologist_id', 'left');
            $this->db->where('ro.id', $order_id);
            
            $query = $this->db->get();
            $order = $query->row_array();
            
            if ($order) {
                // Get tests for this order
                $order['tests'] = $this->getOrderTests($order_id);
                
                // Get scheduling information
                $order['scheduling'] = $this->getOrderScheduling($order_id);
                
                // Get billing information
                $order['billing'] = $this->getOrderBilling($order_id);
                
                // Get quality metrics
                $order['quality_metrics'] = $this->getOrderQualityMetrics($order_id);
            }
            
            return $order;
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology order details: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get tests for a specific order
     */
    public function getOrderTests($order_id) {
        try {
            $this->db->select('
                rot.*,
                rt.test_name,
                rt.test_code,
                rt.category,
                rt.description,
                rt.preparation_instructions,
                rt.estimated_duration,
                rt.price,
                rt.requires_contrast,
                rt.requires_fasting
            ');
            $this->db->from('radiology_order_tests rot');
            $this->db->join('radiology_tests rt', 'rt.id = rot.test_id', 'inner');
            $this->db->where('rot.radiology_order_id', $order_id);
            $this->db->order_by('rot.id', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting order tests: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Cancel radiology order
     */
    public function cancelRadiologyOrder($order_id, $reason = '', $cancelled_by = null) {
        try {
            $this->db->trans_start();
            
            // Update order status
            $update_data = array(
                'status' => 'cancelled',
                'cancellation_reason' => $reason,
                'cancelled_by' => $cancelled_by,
                'cancelled_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            );
            
            $this->db->where('id', $order_id);
            $this->db->update('radiology_orders', $update_data);
            
            // Cancel all associated tests
            $this->db->where('radiology_order_id', $order_id);
            $this->db->update('radiology_order_tests', array(
                'status' => 'cancelled',
                'updated_at' => date('Y-m-d H:i:s')
            ));
            
            // Cancel scheduling
            $this->db->where('order_id', $order_id);
            $this->db->update('radiology_scheduling', array(
                'status' => 'cancelled',
                'updated_at' => date('Y-m-d H:i:s')
            ));
            
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Failed to cancel radiology order');
            }
            
            return true;
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Error cancelling radiology order: ' . $e->getMessage());
            return false;
        }
    }

    // =============================================
    // QUEUE MANAGEMENT
    // =============================================

    /**
     * Get radiology queue for specified date
     */
    public function getRadiologyQueue($date = null, $status_filter = null) {
        try {
            if (!$date) {
                $date = date('Y-m-d');
            }
            
            // Check if optional tables exist
            $billing_exists = $this->tableExists('radiology_billing');
            $scheduling_exists = $this->tableExists('radiology_scheduling');
            $equipment_exists = $this->tableExists('radiology_equipment');
            
            $select_fields = '
                ro.id as order_id,
                COALESCE(ro.id, 0) as order_number,
                ro.order_date,
                ro.priority,
                ro.status,
                ro.clinical_notes,
                COALESCE(ro.verification_status, "pending") as verification_status,
                p.id as patient_id,
                p.patient_name,
                p.id,
                p.age,
                p.gender,
                p.mobileno,
                CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as doctor_name,
                CONCAT(COALESCE(tech.name, ""), " ", COALESCE(tech.surname, "")) as technician_name,
                GROUP_CONCAT(rt.test_name SEPARATOR ", ") as test_names,
                COUNT(rot.id) as test_count
            ';
            
            // Check if order_number column exists and use it if available
            if ($this->db->field_exists('order_number', 'radiology_orders')) {
                $select_fields = str_replace(
                    'COALESCE(ro.id, 0) as order_number',
                    'COALESCE(ro.order_number, CONCAT("RAD", ro.id)) as order_number',
                    $select_fields
                );
            }
            
            if ($scheduling_exists) {
                $select_fields .= ',
                    rs.scheduled_date,
                    rs.scheduled_time,
                    COALESCE(rs.estimated_duration, 30) as estimated_duration
                ';
                
                if ($equipment_exists) {
                    $select_fields .= ',
                        re.equipment_name,
                        re.location as equipment_location
                    ';
                }
            } else {
                $select_fields .= ',
                    NULL as scheduled_date,
                    NULL as scheduled_time,
                    30 as estimated_duration,
                    NULL as equipment_name,
                    NULL as equipment_location
                ';
            }
            
            if ($billing_exists) {
                $select_fields .= ',
                    COALESCE(rb.payment_status, "pending") as payment_status
                ';
            } else {
                $select_fields .= ',
                    "pending" as payment_status
                ';
            }
            
            $this->db->select($select_fields);
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'p.id = ro.patient_id', 'inner');
            $this->db->join('staff s', 's.id = ro.referring_doctor_id', 'left');
            $this->db->join('staff tech', 'tech.id = ro.technician_id', 'left');
            $this->db->join('radiology_order_tests rot', 'rot.radiology_order_id = ro.id', 'left');
            $this->db->join('radiology_tests rt', 'rt.id = rot.test_id', 'left');
            
            if ($scheduling_exists) {
                $this->db->join('radiology_scheduling rs', 'rs.order_id = ro.id', 'left');
                if ($equipment_exists) {
                    $this->db->join('radiology_equipment re', 're.id = rs.equipment_id', 'left');
                }
            }
            
            if ($billing_exists) {
                $this->db->join('radiology_billing rb', 'rb.order_id = ro.id', 'left');
            }
            
            // Date filter
            $this->db->where('DATE(ro.order_date)', $date);
            
            // Status filter
            if ($status_filter) {
                if (is_array($status_filter)) {
                    $this->db->where_in('ro.status', $status_filter);
                } else {
                    $this->db->where('ro.status', $status_filter);
                }
            } else {
                $this->db->where_in('ro.status', array('pending', 'in_progress'));
            }
            
            $this->db->group_by('ro.id');
            $this->db->order_by('FIELD(ro.priority, "stat", "urgent", "routine")', '', FALSE);
            $this->db->order_by('ro.order_date', 'ASC');
            
            $query = $this->db->get();
            
            if ($query === FALSE) {
                $error = $this->db->error();
                throw new Exception('Database query failed: ' . (isset($error['message']) ? $error['message'] : 'Unknown database error'));
            }
            
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology queue: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get queue statistics
     */
    public function getQueueStatistics($date = null) {
        try {
            if (!$date) {
                $date = date('Y-m-d');
            }
            
            $stats = array();
            
            // Total orders for the day
            $this->db->select('COUNT(*) as total');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date)', $date);
            $query = $this->db->get();
            $stats['total_orders'] = $query->row()->total;
            
            // Orders by status
            $this->db->select('status, COUNT(*) as count');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date)', $date);
            $this->db->group_by('status');
            $query = $this->db->get();
            $status_counts = $query->result_array();
            
            foreach ($status_counts as $status) {
                $stats['status_' . $status['status']] = $status['count'];
            }
            
            // Orders by priority
            $this->db->select('priority, COUNT(*) as count');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date)', $date);
            $this->db->group_by('priority');
            $query = $this->db->get();
            $priority_counts = $query->result_array();
            
            foreach ($priority_counts as $priority) {
                $stats['priority_' . $priority['priority']] = $priority['count'];
            }
            
            // Average completion time
            $this->db->select('AVG(TIMESTAMPDIFF(MINUTE, started_at, completed_at)) as avg_time');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date)', $date);
            $this->db->where('status', 'completed');
            $this->db->where('started_at IS NOT NULL');
            $this->db->where('completed_at IS NOT NULL');
            $query = $this->db->get();
            $result = $query->row();
            $stats['avg_completion_time'] = $result->avg_time ? round($result->avg_time, 2) : 0;
            
            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Error getting queue statistics: ' . $e->getMessage());
            return array();
        }
    }

    // =============================================
    // RADIOLOGY TESTS MANAGEMENT
    // =============================================

    /**
     * Get all available radiology tests
     */
public function getRadiologyTests() {
    try {
        $this->db->select('
            id,
            test_name,
            test_code,
            category,
            description,
            preparation_instructions,
            estimated_duration,
            price,
            is_gcc,
            requires_contrast,
            requires_fasting,
            is_active
        ');
        $this->db->from('radiology_tests');
        $this->db->where('is_active', 1);
        $this->db->order_by('category, test_name');
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting radiology tests: ' . $e->getMessage());
        return [];
    }
}

    /**
     * Get radiology test by ID
     */
    public function getRadiologyTestById($test_id) {
        try {
            $this->db->select('*');
            $this->db->from('radiology_tests');
            $this->db->where('id', $test_id);
            $this->db->where('is_active', 1);
            
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology test by ID: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get tests grouped by category
     */
    public function getTestsByCategory() {
        try {
            $this->db->select('*');
            $this->db->from('radiology_tests');
            $this->db->where('is_active', 1);
            $this->db->order_by('category', 'ASC');
            $this->db->order_by('test_name', 'ASC');
            
            $query = $this->db->get();
            $tests = $query->result_array();
            
            $grouped = array();
            foreach ($tests as $test) {
                $grouped[$test['category']][] = $test;
            }
            
            return $grouped;
        } catch (Exception $e) {
            log_message('error', 'Error getting tests by category: ' . $e->getMessage());
            return array();
        }
    }

    // =============================================
    // VERIFICATION AND SECURITY
    // =============================================

    /**
     * Log patient verification attempt
     */
    public function logVerificationAttempt($verification_data) {
        try {
            $verification_data['created_at'] = date('Y-m-d H:i:s');
            $this->db->insert('patient_verifications', $verification_data);
            return $this->db->insert_id();
        } catch (Exception $e) {
            log_message('error', 'Error logging verification attempt: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get patient verification history
     */
    public function getPatientVerificationHistory($patient_id, $department = 'radiology', $limit = 10) {
        try {
            $this->db->select('
                pv.*,
                s.name as staff_name,
                s.surname as staff_surname,
                s.employee_id as staff_employee_id
            ');
            $this->db->from('patient_verifications pv');
            $this->db->join('staff s', 's.id = pv.verified_by', 'left');
            $this->db->where('pv.patient_id', $patient_id);
            $this->db->where('pv.department', $department);
            $this->db->order_by('pv.verification_timestamp', 'DESC');
            $this->db->limit($limit);
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting verification history: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Check if patient requires verification
     */
    public function requiresVerification($patient_id, $order_id = null) {
        try {
            // Check if patient has fingerprint template
            $this->db->select('fingerprint_template');
            $this->db->from('patients');
            $this->db->where('id', $patient_id);
            $query = $this->db->get();
            $patient = $query->row();
            
            if (!$patient || empty($patient->fingerprint_template)) {
                return false; // No biometric available
            }
            
            // Check if verification is required for this order type
            if ($order_id) {
                $this->db->select('verification_required');
                $this->db->from('radiology_orders');
                $this->db->where('id', $order_id);
                $query = $this->db->get();
                $order = $query->row();
                
                return $order ? (bool)$order->verification_required : true;
            }
            
            return true; // Default to requiring verification
        } catch (Exception $e) {
            log_message('error', 'Error checking verification requirement: ' . $e->getMessage());
            return false;
        }
    }

    // =============================================
    // SCHEDULING METHODS
    // =============================================

    /**
     * Get order scheduling information
     */
    public function getOrderScheduling($order_id) {
        try {
            $this->db->select('
                rs.*,
                re.equipment_name,
                re.location,
                s.name as technician_name,
                s.surname as technician_surname
            ');
            $this->db->from('radiology_scheduling rs');
            $this->db->join('radiology_equipment re', 're.id = rs.equipment_id', 'left');
            $this->db->join('staff s', 's.id = rs.technician_id', 'left');
            $this->db->where('rs.order_id', $order_id);
            
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting order scheduling: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Schedule radiology appointment
     */
    public function scheduleAppointment($scheduling_data) {
        try {
            $this->db->trans_start();
            
            // Insert scheduling record
            $this->db->insert('radiology_scheduling', $scheduling_data);
            $schedule_id = $this->db->insert_id();
            
            // Update order status to scheduled
            $this->db->where('id', $scheduling_data['order_id']);
            $this->db->update('radiology_orders', array(
                'status' => 'scheduled',
                'updated_at' => date('Y-m-d H:i:s')
            ));
            
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Failed to schedule appointment');
            }
            
            return $schedule_id;
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Error scheduling appointment: ' . $e->getMessage());
            throw $e;
        }
    }

    // =============================================
    // BILLING METHODS
    // =============================================

    /**
     * Get order billing information
     */
    public function getOrderBilling($order_id) {
        try {
            $this->db->select('*');
            $this->db->from('radiology_billing');
            $this->db->where('order_id', $order_id);
            
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting order billing: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Create billing record for order
     */
    public function createBillingRecord($billing_data) {
        try {
            $billing_data['created_at'] = date('Y-m-d H:i:s');
            $this->db->insert('radiology_billing', $billing_data);
            return $this->db->insert_id();
        } catch (Exception $e) {
            log_message('error', 'Error creating billing record: ' . $e->getMessage());
            throw $e;
        }
    }

    /**
     * Calculate order total amount
     */
    public function calculateOrderTotal($order_id) {
        try {
            $this->db->select('SUM(rt.price) as total_amount');
            $this->db->from('radiology_order_tests rot');
            $this->db->join('radiology_tests rt', 'rt.id = rot.test_id', 'inner');
            $this->db->where('rot.radiology_order_id', $order_id);
            
            $query = $this->db->get();
            $result = $query->row();
            
            return $result ? floatval($result->total_amount) : 0.00;
        } catch (Exception $e) {
            log_message('error', 'Error calculating order total: ' . $e->getMessage());
            return 0.00;
        }
    }

    // =============================================
    // QUALITY METRICS
    // =============================================

    /**
     * Get order quality metrics
     */
    public function getOrderQualityMetrics($order_id) {
        try {
            $this->db->select('*');
            $this->db->from('radiology_quality_metrics');
            $this->db->where('order_id', $order_id);
            
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting quality metrics: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Save quality metrics
     */
    public function saveQualityMetrics($metrics_data) {
        try {
            $metrics_data['created_at'] = date('Y-m-d H:i:s');
            $this->db->insert('radiology_quality_metrics', $metrics_data);
            return $this->db->insert_id();
        } catch (Exception $e) {
            log_message('error', 'Error saving quality metrics: ' . $e->getMessage());
            throw $e;
        }
    }

    // =============================================
    // ACTIVITY LOGGING
    // =============================================

    /**
     * Log radiology activity
     */
    public function logActivity($log_data) {
        try {
            $log_data['timestamp'] = date('Y-m-d H:i:s');
            $log_data['department'] = 'radiology';
            
            $this->db->insert('radiology_activity_log', $log_data);
            return $this->db->insert_id();
        } catch (Exception $e) {
            log_message('error', 'Error logging radiology activity: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get activity log for patient/order
     */
    public function getActivityLog($patient_id = null, $order_id = null, $limit = 20) {
        try {
            $this->db->select('
                ral.*,
                s.name as staff_name,
                s.surname as staff_surname,
                s.employee_id as staff_employee_id,
                p.patient_name,
                ro.order_number
            ');
            $this->db->from('radiology_activity_log ral');
            $this->db->join('staff s', 's.id = ral.staff_id', 'left');
            $this->db->join('patients p', 'p.id = ral.patient_id', 'left');
            $this->db->join('radiology_orders ro', 'ro.id = ral.order_id', 'left');
            
            if ($patient_id) {
                $this->db->where('ral.patient_id', $patient_id);
            }
            
            if ($order_id) {
                $this->db->where('ral.order_id', $order_id);
            }
            
            $this->db->order_by('ral.timestamp', 'DESC');
            $this->db->limit($limit);
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting activity log: ' . $e->getMessage());
            return array();
        }
    }

    // =============================================
    // STATISTICS AND REPORTING
    // =============================================

    /**
     * Get comprehensive radiology statistics
     */
    public function getRadiologyStatistics($date_from = null, $date_to = null) {
        try {
            if (!$date_from) $date_from = date('Y-m-d', strtotime('-30 days'));
            if (!$date_to) $date_to = date('Y-m-d');
            
            $stats = array();
            
            // Total orders in period
            $this->db->select('COUNT(*) as total_orders');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date) >=', $date_from);
            $this->db->where('DATE(order_date) <=', $date_to);
            $query = $this->db->get();
            $stats['total_orders'] = $query->row()->total_orders;
            
            // Orders by status
            $this->db->select('status, COUNT(*) as count');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date) >=', $date_from);
            $this->db->where('DATE(order_date) <=', $date_to);
            $this->db->group_by('status');
            $query = $this->db->get();
            $stats['by_status'] = $query->result_array();
            
            // Orders by priority
            $this->db->select('priority, COUNT(*) as count');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date) >=', $date_from);
            $this->db->where('DATE(order_date) <=', $date_to);
            $this->db->group_by('priority');
            $query = $this->db->get();
            $stats['by_priority'] = $query->result_array();
            
            // Most common tests
            $this->db->select('rt.test_name, rt.category, COUNT(*) as count');
            $this->db->from('radiology_order_tests rot');
            $this->db->join('radiology_tests rt', 'rt.id = rot.test_id', 'inner');
            $this->db->join('radiology_orders ro', 'ro.id = rot.radiology_order_id', 'inner');
            $this->db->where('DATE(ro.order_date) >=', $date_from);
            $this->db->where('DATE(ro.order_date) <=', $date_to);
            $this->db->group_by('rot.test_id');
            $this->db->order_by('count', 'DESC');
            $this->db->limit(10);
            $query = $this->db->get();
            $stats['common_tests'] = $query->result_array();
            
            // Daily order counts
            $this->db->select('DATE(order_date) as order_date, COUNT(*) as count');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date) >=', $date_from);
            $this->db->where('DATE(order_date) <=', $date_to);
            $this->db->group_by('DATE(order_date)');
            $this->db->order_by('order_date', 'ASC');
            $query = $this->db->get();
            $stats['daily_counts'] = $query->result_array();
            
            // Revenue statistics
            $this->db->select('
                SUM(net_amount) as total_revenue,
                AVG(net_amount) as avg_order_value,
                COUNT(*) as billed_orders
            ');
            $this->db->from('radiology_billing rb');
            $this->db->join('radiology_orders ro', 'ro.id = rb.order_id', 'inner');
            $this->db->where('DATE(ro.order_date) >=', $date_from);
            $this->db->where('DATE(ro.order_date) <=', $date_to);
            $query = $this->db->get();
            $stats['revenue'] = $query->row_array();
            
            // Performance metrics
            $this->db->select('
                AVG(TIMESTAMPDIFF(MINUTE, started_at, completed_at)) as avg_completion_time,
                MIN(TIMESTAMPDIFF(MINUTE, started_at, completed_at)) as min_completion_time,
                MAX(TIMESTAMPDIFF(MINUTE, started_at, completed_at)) as max_completion_time
            ');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date) >=', $date_from);
            $this->db->where('DATE(order_date) <=', $date_to);
            $this->db->where('status', 'completed');
            $this->db->where('started_at IS NOT NULL');
            $this->db->where('completed_at IS NOT NULL');
            $query = $this->db->get();
            $stats['performance'] = $query->row_array();
            
            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology statistics: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get department dashboard statistics
     */
    public function getDepartmentStats() {
        try {
            $today = date('Y-m-d');
            $stats = array();
            
            // Today's orders
            $this->db->select('COUNT(*) as today_orders');
            $this->db->from('radiology_orders');
            $this->db->where('DATE(order_date)', $today);
            $query = $this->db->get();
            $stats['today_orders'] = $query->row()->today_orders;
            
            // Pending orders
            $this->db->select('COUNT(*) as pending_orders');
            $this->db->from('radiology_orders');
            $this->db->where('status', 'pending');
            $query = $this->db->get();
            $stats['pending_orders'] = $query->row()->pending_orders;
            
            // In progress orders
            $this->db->select('COUNT(*) as in_progress_orders');
            $this->db->from('radiology_orders');
            $this->db->where('status', 'in_progress');
            $query = $this->db->get();
            $stats['in_progress_orders'] = $query->row()->in_progress_orders;
            
            // Completed today
            $this->db->select('COUNT(*) as completed_today');
            $this->db->from('radiology_orders');
            $this->db->where('status', 'completed');
            $this->db->where('DATE(completed_at)', $today);
            $query = $this->db->get();
            $stats['completed_today'] = $query->row()->completed_today;
            
            // STAT orders
            $this->db->select('COUNT(*) as stat_orders');
            $this->db->from('radiology_orders');
            $this->db->where('priority', 'stat');
            $this->db->where_in('status', array('pending', 'scheduled', 'in_progress'));
            $query = $this->db->get();
            $stats['stat_orders'] = $query->row()->stat_orders;
            
            // Equipment utilization
            $this->db->select('
                COUNT(DISTINCT rs.equipment_id) as active_equipment,
                COUNT(*) as total_scheduled
            ');
            $this->db->from('radiology_scheduling rs');
            $this->db->where('rs.scheduled_date', $today);
            $this->db->where('rs.status', 'scheduled');
            $query = $this->db->get();
            $equipment_stats = $query->row_array();
            $stats['active_equipment'] = $equipment_stats['active_equipment'];
            $stats['total_scheduled'] = $equipment_stats['total_scheduled'];
            
            // Revenue today
            $this->db->select('SUM(net_amount) as today_revenue');
            $this->db->from('radiology_billing rb');
            $this->db->join('radiology_orders ro', 'ro.id = rb.order_id', 'inner');
            $this->db->where('DATE(ro.order_date)', $today);
            $query = $this->db->get();
            $stats['today_revenue'] = $query->row()->today_revenue ?: 0;
            
            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Error getting department stats: ' . $e->getMessage());
            return array();
        }
    }

    // =============================================
    // SEARCH AND FILTERING
    // =============================================

    /**
     * Search radiology orders with advanced filters
     */
    public function searchRadiologyOrders($search_params) {
        try {
            // Check if billing table exists
            $billing_exists = $this->tableExists('radiology_billing');
            
            $select_fields = '
                ro.*,
                p.patient_name,
                p.id,
                p.age,
                p.gender,
                CONCAT(COALESCE(s.name, ""), " ", COALESCE(s.surname, "")) as doctor_name,
                GROUP_CONCAT(rt.test_name SEPARATOR ", ") as test_names
            ';
            
            if ($billing_exists) {
                $select_fields .= ',
                    COALESCE(rb.net_amount, 0.00) as net_amount,
                    COALESCE(rb.payment_status, "pending") as payment_status
                ';
            } else {
                $select_fields .= ',
                    0.00 as net_amount,
                    "pending" as payment_status
                ';
            }
            
            $this->db->select($select_fields);
            $this->db->from('radiology_orders ro');
            $this->db->join('patients p', 'p.id = ro.patient_id', 'inner');
            $this->db->join('staff s', 's.id = ro.referring_doctor_id', 'left');
            $this->db->join('radiology_order_tests rot', 'rot.radiology_order_id = ro.id', 'left');
            $this->db->join('radiology_tests rt', 'rt.id = rot.test_id', 'left');
            
            if ($billing_exists) {
                $this->db->join('radiology_billing rb', 'rb.order_id = ro.id', 'left');
            }
            
            // Apply search filters
            if (!empty($search_params['patient_name'])) {
                $this->db->like('p.patient_name', $search_params['patient_name']);
            }
            
            if (!empty($search_params['patient_id'])) {
                $this->db->like('p.id', $search_params['patient_id']);
            }
            
            if (!empty($search_params['order_number']) && $this->db->field_exists('order_number', 'radiology_orders')) {
                $this->db->like('ro.order_number', $search_params['order_number']);
            }
            
            if (!empty($search_params['status'])) {
                if (is_array($search_params['status'])) {
                    $this->db->where_in('ro.status', $search_params['status']);
                } else {
                    $this->db->where('ro.status', $search_params['status']);
                }
            }
            
            if (!empty($search_params['priority'])) {
                if (is_array($search_params['priority'])) {
                    $this->db->where_in('ro.priority', $search_params['priority']);
                } else {
                    $this->db->where('ro.priority', $search_params['priority']);
                }
            }
            
            if (!empty($search_params['date_from'])) {
                $this->db->where('DATE(ro.order_date) >=', $search_params['date_from']);
            }
            
            if (!empty($search_params['date_to'])) {
                $this->db->where('DATE(ro.order_date) <=', $search_params['date_to']);
            }
            
            if (!empty($search_params['doctor_id'])) {
                $this->db->where('ro.referring_doctor_id', $search_params['doctor_id']);
            }
            
            if (!empty($search_params['test_category'])) {
                $this->db->where('rt.category', $search_params['test_category']);
            }
            
            if (!empty($search_params['payment_status']) && $billing_exists) {
                $this->db->where('rb.payment_status', $search_params['payment_status']);
            }
            
            $this->db->group_by('ro.id');
            $this->db->order_by('ro.order_date', 'DESC');
            
            // Pagination
            if (!empty($search_params['limit'])) {
                $this->db->limit($search_params['limit']);
                if (!empty($search_params['offset'])) {
                    $this->db->offset($search_params['offset']);
                }
            }
            
            $query = $this->db->get();
            
            if ($query === FALSE) {
                throw new Exception('Database query failed: ' . $this->db->_error_message());
            }
            
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error searching radiology orders: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get search filters options
     */
    public function getSearchFilterOptions() {
        try {
            $options = array();
            
            // Get available statuses
            $this->db->select('DISTINCT status');
            $this->db->from('radiology_orders');
            $this->db->where('status IS NOT NULL');
            $query = $this->db->get();
            $options['statuses'] = array_column($query->result_array(), 'status');
            
            // Get available priorities
            $this->db->select('DISTINCT priority');
            $this->db->from('radiology_orders');
            $this->db->where('priority IS NOT NULL');
            $query = $this->db->get();
            $options['priorities'] = array_column($query->result_array(), 'priority');
            
            // Get test categories
            $this->db->select('DISTINCT category');
            $this->db->from('radiology_tests');
            $this->db->where('is_active', 1);
            $this->db->order_by('category', 'ASC');
            $query = $this->db->get();
            $options['test_categories'] = array_column($query->result_array(), 'category');
            
            // Get referring doctors
            $this->db->select('DISTINCT s.id, CONCAT(s.name, " ", s.surname) as doctor_name');
            $this->db->from('radiology_orders ro');
            $this->db->join('staff s', 's.id = ro.referring_doctor_id', 'inner');
            $this->db->where('s.name IS NOT NULL');
            $this->db->order_by('s.name', 'ASC');
            $query = $this->db->get();
            $options['doctors'] = $query->result_array();
            
            return $options;
        } catch (Exception $e) {
            log_message('error', 'Error getting search filter options: ' . $e->getMessage());
            return array();
        }
    }

    // =============================================
    // EQUIPMENT MANAGEMENT
    // =============================================

    /**
     * Get all radiology equipment
     */
    public function getRadiologyEquipment($status = null) {
        try {
            $this->db->select('*');
            $this->db->from('radiology_equipment');
            $this->db->where('is_active', 1);
            
            if ($status) {
                $this->db->where('status', $status);
            }
            
            $this->db->order_by('equipment_name', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting radiology equipment: ' . $e->getMessage());
            return array();
        }
    }

    /**
     * Get equipment availability for scheduling
     */
    public function getEquipmentAvailability($equipment_id, $date, $time_slot = null) {
        try {
            $this->db->select('*');
            $this->db->from('radiology_scheduling');
            $this->db->where('equipment_id', $equipment_id);
            $this->db->where('scheduled_date', $date);
            $this->db->where_in('status', array('scheduled', 'in_progress'));
            
            if ($time_slot) {
                $this->db->where('scheduled_time', $time_slot);
            }
            
            $query = $this->db->get();
            $bookings = $query->result_array();
            
            return array(
                'is_available' => empty($bookings),
                'bookings' => $bookings
            );
        } catch (Exception $e) {
            log_message('error', 'Error checking equipment availability: ' . $e->getMessage());
            return array('is_available' => false, 'bookings' => array());
        }
    }

    // =============================================
    // UTILITY METHODS
    // =============================================

    /**
     * Generate unique order number
     */
    private function generateOrderNumber() {
        try {
            $prefix = 'RAD';
            $date = date('Ymd');
            
            // Check if order_number column exists
            if (!$this->db->field_exists('order_number', 'radiology_orders')) {
                // If column doesn't exist, return a simple format based on ID
                return $prefix . $date . '0001';
            }
            
            // Get last order number for today
            $this->db->select('order_number');
            $this->db->from('radiology_orders');
            $this->db->like('order_number', $prefix . $date, 'after');
            $this->db->order_by('id', 'DESC');
            $this->db->limit(1);
            $query = $this->db->get();
            
            if ($query === FALSE) {
                log_message('error', 'Database query failed in generateOrderNumber: ' . $this->db->error()['message']);
                return $prefix . $date . '0001';
            }
            
            if ($query->num_rows() > 0) {
                $last_order = $query->row();
                $last_number = intval(substr($last_order->order_number, -4));
                $new_number = $last_number + 1;
            } else {
                $new_number = 1;
            }
            
            return $prefix . $date . str_pad($new_number, 4, '0', STR_PAD_LEFT);
        } catch (Exception $e) {
            log_message('error', 'Error generating order number: ' . $e->getMessage());
            return 'RAD' . date('Ymd') . '0001';
        }
    }

    /**
     * Validate order data
     */
    public function validateOrderData($order_data) {
        $errors = array();
        
        // Required fields validation
        if (empty($order_data['patient_id'])) {
            $errors[] = 'Patient ID is required';
        }
        
        if (empty($order_data['referring_doctor_id'])) {
            $errors[] = 'Referring doctor is required';
        }
        
        // Validate patient exists
        if (!empty($order_data['patient_id'])) {
            $this->db->select('id');
            $this->db->from('patients');
            $this->db->where('id', $order_data['patient_id']);
            $this->db->where('is_active', 'yes');
            $query = $this->db->get();
            
            if ($query->num_rows() === 0) {
                $errors[] = 'Invalid patient ID';
            }
        }
        
        // Validate doctor exists
        if (!empty($order_data['referring_doctor_id'])) {
            $this->db->select('id');
            $this->db->from('staff');
            $this->db->where('id', $order_data['referring_doctor_id']);
            $this->db->where('is_active', 'yes');
            $query = $this->db->get();
            
            if ($query->num_rows() === 0) {
                $errors[] = 'Invalid referring doctor ID';
            }
        }
        
        // Validate priority
        if (!empty($order_data['priority'])) {
            $valid_priorities = array('routine', 'urgent', 'stat');
            if (!in_array($order_data['priority'], $valid_priorities)) {
                $errors[] = 'Invalid priority value';
            }
        }
        
        return $errors;
    }

    /**
     * Get order status options
     */
    public function getOrderStatusOptions() {
        return array(
            'pending' => 'Pending',
            'scheduled' => 'Scheduled',
            'in_progress' => 'In Progress',
            'completed' => 'Completed',
            'cancelled' => 'Cancelled',
            'reported' => 'Reported'
        );
    }

    /**
     * Get priority options
     */
    public function getPriorityOptions() {
        return array(
            'routine' => 'Routine',
            'urgent' => 'Urgent',
            'stat' => 'STAT'
        );
    }

    /**
     * Check if order can be cancelled
     */
    public function canCancelOrder($order_id) {
        try {
            $this->db->select('status');
            $this->db->from('radiology_orders');
            $this->db->where('id', $order_id);
            $query = $this->db->get();
            
            if ($query->num_rows() === 0) {
                return false;
            }
            
            $order = $query->row();
            $cancellable_statuses = array('pending', 'scheduled');
            
            return in_array($order->status, $cancellable_statuses);
        } catch (Exception $e) {
            log_message('error', 'Error checking if order can be cancelled: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get test preparation instructions
     */
    public function getTestPreparationInstructions($test_ids) {
        try {
            if (empty($test_ids)) {
                return array();
            }
            
            $this->db->select('test_name, preparation_instructions, requires_fasting, requires_contrast');
            $this->db->from('radiology_tests');
            $this->db->where_in('id', $test_ids);
            $this->db->where('is_active', 1);
            
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Error getting test preparation instructions: ' . $e->getMessage());
            return array();
        }
    }
    
    public function getTestsByWorkflow($patient_workflow) {
    try {
        $this->db->select('*');
        $this->db->from('radiology_tests');
        $this->db->where('is_active', 1);
        
        // Filter based on workflow
        if (strtoupper($patient_workflow) === 'GCC') {
            // GCC patients only see GCC tests
            $this->db->where('is_gcc', 1);
        }
        // General/Specialized patients see all tests (no additional filter)
        
        $this->db->order_by('category, test_name');
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Error getting tests by workflow: ' . $e->getMessage());
        return [];
    }
}

    /**
     * Clean up old activity logs
     */
    public function cleanupOldLogs($days = 90) {
        try {
            $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days} days"));
            
            $this->db->where('timestamp <', $cutoff_date);
            $this->db->delete('radiology_activity_log');
            
            $deleted = $this->db->affected_rows();
            log_message('info', "Cleaned up {$deleted} old radiology activity log entries");
            
            return $deleted;
        } catch (Exception $e) {
            log_message('error', 'Error cleaning up old logs: ' . $e->getMessage());
            return 0;
        }
    }
}