<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Queue Management Model
 * Handles all queue operations for the hospital management system
 */
class Queue_model extends CI_Model {
    
    public function __construct() {
        parent::__construct();
        $this->load->database();
        $this->initializeTables();
    }

    // =============================================
    // QUEUE CREATION AND MANAGEMENT
    // =============================================

    /**
     * Add patient to queue after registration
     */
    public function addPatientToQueue($patient_id, $department = 'registration', $priority = 'normal', $workflow_type = 'general') {
        try {
            // Generate queue number
            $queue_number = $this->generateQueueNumber($department);
            
            $queue_data = [
                'patient_id' => $patient_id,
                'queue_number' => $queue_number,
                'department' => $department,
                'status' => 'waiting',
                'priority' => $priority,
                'queue_type' => $department,
                'estimated_wait_time' => $this->calculateEstimatedWaitTime($department),
                'created_at' => date('Y-m-d H:i:s')
            ];

            $this->db->insert('patient_queue', $queue_data);
            $queue_id = $this->db->insert_id();

            // Auto-route to next queue if configured
            $this->checkAutoRouting($patient_id, $department, $workflow_type);

            return [
                'status' => 'success',
                'queue_id' => $queue_id,
                'queue_number' => $queue_number,
                'estimated_wait' => $queue_data['estimated_wait_time']
            ];

        } catch (Exception $e) {
            log_message('error', 'Queue creation failed: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Generate unique queue number for department
     */
    private function generateQueueNumber($department) {
        // Get queue configuration
        $config = $this->getQueueConfig($department);
        $prefix = $config['queue_prefix'] ?? strtoupper(substr($department, 0, 3));
        $current_number = $config['counter_current'] ?? 1;

        // Check if number already exists today
        $today = date('Y-m-d');
        $queue_number = $prefix . sprintf('%03d', $current_number);
        
        $this->db->where('queue_number', $queue_number);
        $this->db->where('DATE(created_at)', $today);
        $exists = $this->db->count_all_results('patient_queue');

        while ($exists > 0) {
            $current_number++;
            $queue_number = $prefix . sprintf('%03d', $current_number);
            
            $this->db->where('queue_number', $queue_number);
            $this->db->where('DATE(created_at)', $today);
            $exists = $this->db->count_all_results('patient_queue');
        }

        // Update counter in config
        $this->updateQueueCounter($department, $current_number + 1);

        return $queue_number;
    }

    /**
     * Calculate estimated wait time based on queue position and service time
     */
    private function calculateEstimatedWaitTime($department) {
        // Get number of people waiting
        $this->db->where('department', $department);
        $this->db->where('status', 'waiting');
        $waiting_count = $this->db->count_all_results('patient_queue');

        // Get average service time
        $config = $this->getQueueConfig($department);
        $avg_service_time = $config['average_service_time'] ?? 15;

        return $waiting_count * $avg_service_time;
    }

    // =============================================
    // QUEUE OPERATIONS
    // =============================================

    /**
     * Call next patient in queue
     */
    public function callNextPatient($department, $served_by = null) {
        try {
            // Get next patient in queue (priority order)
            $this->db->select('*');
            $this->db->from('patient_queue');
            $this->db->where('department', $department);
            $this->db->where('status', 'waiting');
            $this->db->order_by("FIELD(priority, 'urgent', 'high', 'normal', 'low')", '', FALSE);
            $this->db->order_by('created_at', 'ASC');
            $this->db->limit(1);
            
            $query = $this->db->get();
            $next_patient = $query->row_array();

            if (!$next_patient) {
                return ['status' => 'error', 'message' => 'No patients in queue'];
            }

            // Update status to in_progress
            $update_data = [
                'status' => 'in_progress',
                'called_at' => date('Y-m-d H:i:s'),
                'served_by' => $served_by,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $next_patient['id']);
            $this->db->update('patient_queue', $update_data);

            // Get patient details
            $patient_details = $this->getPatientDetails($next_patient['patient_id']);

            return [
                'status' => 'success',
                'patient' => array_merge($next_patient, $update_data, $patient_details)
            ];

        } catch (Exception $e) {
            log_message('error', 'Call next patient failed: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Complete service for current patient
     */
    public function completeService($queue_id, $next_department = null, $notes = null) {
        try {
            // Get current queue record
            $this->db->where('id', $queue_id);
            $current_queue = $this->db->get('patient_queue')->row_array();

            if (!$current_queue) {
                return ['status' => 'error', 'message' => 'Queue record not found'];
            }

            // Calculate actual wait time
            $called_time = strtotime($current_queue['called_at']);
            $actual_wait_time = $called_time ? round((time() - $called_time) / 60) : null;

            // Update current queue as completed
            $update_data = [
                'status' => 'completed',
                'completed_at' => date('Y-m-d H:i:s'),
                'actual_wait_time' => $actual_wait_time,
                'notes' => $notes,
                'next_queue' => $next_department,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $queue_id);
            $this->db->update('patient_queue', $update_data);

            // Add to next department queue if specified
            if ($next_department) {
                $priority = $this->determinePriority($current_queue['patient_id'], $next_department);
                $result = $this->addPatientToQueue($current_queue['patient_id'], $next_department, $priority);
                
                return [
                    'status' => 'success',
                    'message' => 'Service completed',
                    'next_queue' => $result
                ];
            }

            return ['status' => 'success', 'message' => 'Service completed'];

        } catch (Exception $e) {
            log_message('error', 'Complete service failed: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Skip/cancel patient in queue
     */
    public function skipPatient($queue_id, $reason = null) {
        try {
            $update_data = [
                'status' => 'skipped',
                'notes' => $reason,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $queue_id);
            $result = $this->db->update('patient_queue', $update_data);

            if ($result) {
                return ['status' => 'success', 'message' => 'Patient skipped'];
            } else {
                return ['status' => 'error', 'message' => 'Failed to skip patient'];
            }

        } catch (Exception $e) {
            log_message('error', 'Skip patient failed: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    // =============================================
    // QUEUE DISPLAY AND MONITORING
    // =============================================

    /**
     * Get current queue status for department
     */
    public function getDepartmentQueue($department, $limit = 50) {
        try {
            $this->db->select('pq.*, p.patient_name, p.mobileno, p.age, p.gender');
            $this->db->from('patient_queue pq');
            $this->db->join('patients p', 'p.id = pq.patient_id', 'left');
            $this->db->where('pq.department', $department);
            $this->db->where_in('pq.status', ['waiting', 'in_progress']);
            $this->db->order_by("FIELD(pq.priority, 'urgent', 'high', 'normal', 'low')", '', FALSE);
            $this->db->order_by('pq.created_at', 'ASC');
            $this->db->limit($limit);

            $query = $this->db->get();
            $queue_list = $query->result_array();

            // Get current serving patient
            $this->db->where('department', $department);
            $this->db->where('status', 'in_progress');
            $current_serving = $this->db->get('patient_queue')->row_array();

            // Get queue statistics
            $stats = $this->getQueueStatistics($department);

            return [
                'status' => 'success',
                'department' => $department,
                'current_serving' => $current_serving,
                'queue_list' => $queue_list,
                'statistics' => $stats
            ];

        } catch (Exception $e) {
            log_message('error', 'Get department queue failed: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Get queue statistics for department
     */
    public function getQueueStatistics($department) {
        $today = date('Y-m-d');

        // Waiting count
        $this->db->where('department', $department);
        $this->db->where('status', 'waiting');
        $waiting_count = $this->db->count_all_results('patient_queue');

        // Today's total
        $this->db->where('department', $department);
        $this->db->where('DATE(created_at)', $today);
        $today_total = $this->db->count_all_results('patient_queue');

        // Completed today
        $this->db->where('department', $department);
        $this->db->where('status', 'completed');
        $this->db->where('DATE(completed_at)', $today);
        $completed_today = $this->db->count_all_results('patient_queue');

        // Average wait time today
        $this->db->select('AVG(actual_wait_time) as avg_wait');
        $this->db->where('department', $department);
        $this->db->where('status', 'completed');
        $this->db->where('DATE(completed_at)', $today);
        $this->db->where('actual_wait_time IS NOT NULL');
        $avg_wait_result = $this->db->get('patient_queue')->row();
        $avg_wait_time = $avg_wait_result ? round($avg_wait_result->avg_wait, 1) : 0;

        return [
            'waiting_count' => $waiting_count,
            'today_total' => $today_total,
            'completed_today' => $completed_today,
            'avg_wait_time' => $avg_wait_time,
            'pending_count' => $today_total - $completed_today
        ];
    }

    /**
     * Get all department queues overview
     */
    public function getAllQueuesOverview() {
        $departments = $this->getAllDepartments();
        $overview = [];

        foreach ($departments as $dept) {
            $stats = $this->getQueueStatistics($dept['department']);
            $overview[] = [
                'department' => $dept['department'],
                'display_name' => $dept['display_name'],
                'theme_color' => $dept['theme_color'],
                'statistics' => $stats
            ];
        }

        return $overview;
    }

    // =============================================
    // CONFIGURATION AND UTILITIES
    // =============================================

    /**
     * Get queue configuration for department
     */
    public function getQueueConfig($department) {
        $this->db->where('department', $department);
        $config = $this->db->get('queue_config')->row_array();
        
        return $config ?: [
            'department' => $department,
            'queue_prefix' => strtoupper(substr($department, 0, 3)),
            'counter_current' => 1,
            'average_service_time' => 15
        ];
    }

    /**
     * Update queue counter
     */
    private function updateQueueCounter($department, $new_counter) {
        $this->db->where('department', $department);
        $this->db->update('queue_config', ['counter_current' => $new_counter]);
    }

    /**
     * Get all departments with display settings
     */
    private function getAllDepartments() {
        return $this->db->get('queue_display_settings')->result_array();
    }

    /**
     * Get patient details
     */
    private function getPatientDetails($patient_id) {
        $this->db->where('id', $patient_id);
        $patient = $this->db->get('patients')->row_array();
        
        return $patient ?: ['patient_name' => 'Unknown Patient'];
    }

    /**
     * Determine priority based on triage or department rules
     */
    private function determinePriority($patient_id, $department) {
        // Check if patient has triage record
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit(1);
        $triage = $this->db->get('triage')->row();

        if ($triage && isset($triage->priority)) {
            return $triage->priority;
        }

        return 'normal';
    }

    /**
     * Check auto-routing rules
     */
    private function checkAutoRouting($patient_id, $current_department, $workflow_type) {
        $this->db->where('from_department', $current_department);
        $this->db->where('condition_type', 'automatic');
        $this->db->where('workflow_type', $workflow_type);
        $this->db->where('is_active', 1);
        $rules = $this->db->get('queue_flow_rules')->result_array();

        foreach ($rules as $rule) {
            // Auto-add to next queue
            $priority = $this->determinePriority($patient_id, $rule['to_department']);
            $this->addPatientToQueue($patient_id, $rule['to_department'], $priority, $workflow_type);
        }
    }

    /**
     * Initialize required tables
     */
    private function initializeTables() {
        // This would typically run the schema creation
        // For production, run the SQL schema separately
        if (!$this->db->table_exists('patient_queue')) {
            log_message('info', 'Queue tables need to be created. Please run the database schema.');
        }
    }

    /**
     * Clear completed queues (maintenance function)
     */
    public function clearOldQueues($days_old = 7) {
        $cutoff_date = date('Y-m-d H:i:s', strtotime("-{$days_old} days"));
        
        $this->db->where('status', 'completed');
        $this->db->where('completed_at <', $cutoff_date);
        $deleted = $this->db->delete('patient_queue');

        return [
            'status' => 'success',
            'message' => "Cleared old queue records",
            'deleted_count' => $this->db->affected_rows()
        ];
    }
}