<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * P9 Model
 * Handles P9 tax certificate generation and calculations
 */
class P9_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
        $this->load->model('hr_payroll_model');
    }

    // =============================================
    // P9 DATA GENERATION
    // =============================================

    public function generateP9Data($staff_id, $year) {
        // Get staff details
        $staff = $this->getStaffDetailsForP9($staff_id);
        if (!$staff) {
            return false;
        }

        // Get monthly payroll data for the year
        $monthly_data = $this->getMonthlyPayrollData($staff_id, $year);
        
        // Calculate P9 totals
        $totals = $this->calculateP9Totals($monthly_data);
        
        // Process monthly breakdown for P9 format
        $processed_monthly = $this->processMonthlyDataForP9($monthly_data);

        return [
            'staff' => $staff,
            'year' => $year,
            'monthly_data' => $processed_monthly,
            'totals' => $totals,
            'generated_date' => date('Y-m-d H:i:s')
        ];
    }

    private function getStaffDetailsForP9($staff_id) {
        $this->db->select('s.*, d.name as department_name, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');
        $this->db->where('s.id', $staff_id);
        $this->db->where('s.is_active', 'yes');
        
        return $this->db->get()->row();
    }

    private function getMonthlyPayrollData($staff_id, $year) {
        $this->db->select('sp.*, s.name, s.surname, s.employee_id');
        $this->db->from('staff_payroll sp');
        $this->db->join('staff s', 's.id = sp.staff_id', 'left');
        $this->db->where('sp.staff_id', $staff_id);
        $this->db->where('sp.year', $year);
        $this->db->where('sp.status !=', 'draft');
        $this->db->order_by('sp.month', 'ASC');
        
        $results = $this->db->get()->result();
        
        // Index by month for easier access
        $monthly_data = [];
        foreach ($results as $record) {
            $monthly_data[$record->month] = $record;
        }
        
        return $monthly_data;
    }

    private function processMonthlyDataForP9($monthly_data) {
        $processed = [];
        
        foreach ($monthly_data as $month => $data) {
            $earnings = json_decode($data->earnings_json, true) ?: [];
            $deductions = json_decode($data->deductions_json, true) ?: [];
            
            $processed[$month] = [
                'basic_salary' => $earnings['BASIC'] ?? 0,
                'benefits_non_cash' => $this->calculateBenefitsNonCash($earnings),
                'value_of_quarters' => $earnings['HOUSE'] ?? 0, // House allowance can be considered as quarters
                'gross_pay' => $data->gross_salary,
                'thirty_percent_cash' => $this->calculateThirtyPercent($data->gross_salary),
                'actual_contribution' => $deductions['NSSF'] ?? 0,
                'owner_occupier' => 0, // Usually 0 unless staff has mortgage interest
                'retirement_contribution' => $deductions['NSSF'] ?? 0,
                'insurance_relief' => $this->calculateInsuranceRelief($deductions),
                'taxable_pay' => $this->calculateTaxablePay($data->gross_salary, $deductions),
                'tax_charged' => $this->calculateTaxCharged($this->calculateTaxablePay($data->gross_salary, $deductions)),
                'monthly_relief' => 1408, // Standard monthly relief in Kenya
                'insurance_relief_month' => $this->calculateMonthlyInsuranceRelief($deductions),
                'paye_tax' => $deductions['PAYE'] ?? 0
            ];
        }
        
        return $processed;
    }

    private function calculateBenefitsNonCash($earnings) {
        // Calculate non-cash benefits (excluding basic salary and cash allowances)
        $non_cash_benefits = 0;
        $cash_components = ['BASIC', 'HOUSE', 'TRANSPORT', 'OVERTIME'];
        
        foreach ($earnings as $component => $amount) {
            if (!in_array($component, $cash_components)) {
                $non_cash_benefits += $amount;
            }
        }
        
        return $non_cash_benefits;
    }

    private function calculateThirtyPercent($gross_pay) {
        // 30% of cash pay for mortgage interest relief
        return $gross_pay * 0.30;
    }

    private function calculateInsuranceRelief($deductions) {
        // Insurance premium relief calculation
        $insurance_premium = ($deductions['NHIF'] ?? 0) + ($deductions['INSURANCE'] ?? 0);
        return min($insurance_premium * 0.15, 60000 / 12); // 15% of premium or max KSh 5,000 per month
    }

    private function calculateMonthlyInsuranceRelief($deductions) {
        return $this->calculateInsuranceRelief($deductions);
    }

    private function calculateTaxablePay($gross_salary, $deductions) {
        $nssf = $deductions['NSSF'] ?? 0;
        $nhif = $deductions['NHIF'] ?? 0;
        $insurance_relief = $this->calculateInsuranceRelief($deductions);
        
        return $gross_salary - $nssf - $nhif - $insurance_relief;
    }

    private function calculateTaxCharged($taxable_pay) {
        // Kenya PAYE calculation with tax relief
        $tax_before_relief = $this->hr_payroll_model->calculatePAYE($taxable_pay + 1408);
        return max(0, $tax_before_relief);
    }

    private function calculateP9Totals($monthly_data) {
        $totals = [
            'basic_salary' => 0,
            'benefits_non_cash' => 0,
            'value_of_quarters' => 0,
            'gross_pay' => 0,
            'thirty_percent_cash' => 0,
            'actual_contribution' => 0,
            'owner_occupier' => 0,
            'retirement_contribution' => 0,
            'insurance_relief' => 0,
            'taxable_pay' => 0,
            'tax_charged' => 0,
            'monthly_relief' => 0,
            'insurance_relief_total' => 0,
            'paye_tax' => 0
        ];

        foreach ($monthly_data as $data) {
            $earnings = json_decode($data->earnings_json, true) ?: [];
            $deductions = json_decode($data->deductions_json, true) ?: [];
            
            $totals['basic_salary'] += $earnings['BASIC'] ?? 0;
            $totals['benefits_non_cash'] += $this->calculateBenefitsNonCash($earnings);
            $totals['value_of_quarters'] += $earnings['HOUSE'] ?? 0;
            $totals['gross_pay'] += $data->gross_salary;
            $totals['thirty_percent_cash'] += $this->calculateThirtyPercent($data->gross_salary);
            $totals['actual_contribution'] += $deductions['NSSF'] ?? 0;
            $totals['retirement_contribution'] += $deductions['NSSF'] ?? 0;
            $totals['insurance_relief'] += $this->calculateInsuranceRelief($deductions);
            $totals['taxable_pay'] += $this->calculateTaxablePay($data->gross_salary, $deductions);
            $totals['tax_charged'] += $this->calculateTaxCharged($this->calculateTaxablePay($data->gross_salary, $deductions));
            $totals['monthly_relief'] += 1408; // KSh 1,408 per month
            $totals['insurance_relief_total'] += $this->calculateInsuranceRelief($deductions);
            $totals['paye_tax'] += $deductions['PAYE'] ?? 0;
        }

        return $totals;
    }

    // =============================================
    // BULK P9 GENERATION
    // =============================================

    public function generateBulkP9($year, $department_id = null, $delivery_method = 'download') {
        $this->db->trans_start();

        // Get staff list
        $this->db->select('s.id, s.employee_id, s.name, s.surname, s.email, d.name as department_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('s.is_active', 'yes');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        // Only include staff who have payroll records for the year
        $this->db->where('s.id IN (
            SELECT DISTINCT staff_id 
            FROM staff_payroll 
            WHERE year = ' . $year . ' AND status != "draft"
        )');
        
        $staff_list = $this->db->get()->result();

        $results = [
            'total_staff' => count($staff_list),
            'p9_generated' => 0,
            'emails_sent' => 0,
            'errors' => [],
            'success_list' => []
        ];

        // Create temporary directory for PDFs if needed
        $temp_dir = null;
        if ($delivery_method === 'download' || $delivery_method === 'both') {
            $temp_dir = APPPATH . 'temp/p9_' . $year . '_' . date('YmdHis') . '/';
            if (!is_dir($temp_dir)) {
                mkdir($temp_dir, 0755, true);
            }
        }

        foreach ($staff_list as $staff) {
            try {
                $p9_data = $this->generateP9Data($staff->id, $year);
                
                if (!$p9_data) {
                    $results['errors'][] = 'No payroll data found for ' . $staff->name . ' ' . $staff->surname;
                    continue;
                }

                // Generate PDF
                if ($delivery_method === 'download' || $delivery_method === 'both') {
                    $pdf_content = $this->generateP9PDFContent($p9_data);
                    $filename = 'P9_' . $staff->employee_id . '_' . $year . '.pdf';
                    
                    if ($temp_dir) {
                        file_put_contents($temp_dir . $filename, $pdf_content);
                        $results['p9_generated']++;
                    }
                }

                // Email P9
                if (($delivery_method === 'email' || $delivery_method === 'both') && $staff->email) {
                    $email_result = $this->emailP9($staff->id, $year);
                    if ($email_result['status'] === 'success') {
                        $results['emails_sent']++;
                        $results['success_list'][] = [
                            'name' => $staff->name . ' ' . $staff->surname,
                            'action' => 'P9 emailed to ' . $staff->email
                        ];
                    } else {
                        $results['errors'][] = 'Failed to email P9 to ' . $staff->name . ': ' . $email_result['message'];
                    }
                }

                // Log P9 generation
                $this->logP9Generation($staff->id, $year, $delivery_method);

            } catch (Exception $e) {
                $results['errors'][] = 'Error processing P9 for ' . $staff->name . ': ' . $e->getMessage();
            }
        }

        // Create ZIP file if PDFs were generated
        if ($temp_dir && $results['p9_generated'] > 0) {
            $this->load->library('zip');
            $this->zip->read_dir($temp_dir, FALSE);
            
            $zip_filename = 'P9_Forms_' . $year . '.zip';
            $this->zip->download($zip_filename);
            
            // Clean up temporary files
            $this->deleteDirectory($temp_dir);
        }

        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            return [
                'status' => 'error',
                'message' => 'Bulk P9 generation failed due to database error'
            ];
        }

        return [
            'status' => 'success',
            'message' => 'Bulk P9 generation completed',
            'results' => $results
        ];
    }

    // =============================================
    // EMAIL FUNCTIONALITY
    // =============================================

    public function emailP9($staff_id, $year) {
        try {
            $p9_data = $this->generateP9Data($staff_id, $year);
            
            if (!$p9_data) {
                return [
                    'status' => 'error',
                    'message' => 'No P9 data found for the specified year'
                ];
            }

            $staff = $p9_data['staff'];
            
            if (!$staff->email) {
                return [
                    'status' => 'error',
                    'message' => 'Staff email address not found'
                ];
            }

            // Generate PDF content
            $pdf_content = $this->generateP9PDFContent($p9_data);
            
            // Load email library
            $this->load->library('email');
            
            $subject = 'P9 Tax Certificate for ' . $year . ' - ' . $staff->name . ' ' . $staff->surname;
            
            $message = "Dear " . $staff->name . ",\n\n";
            $message .= "Please find attached your P9 Tax Certificate for the year " . $year . ".\n\n";
            $message .= "This document contains your annual tax deduction summary as required by the Kenya Revenue Authority (KRA).\n\n";
            $message .= "Key Information:\n";
            $message .= "- Employee ID: " . $staff->employee_id . "\n";
            $message .= "- Year: " . $year . "\n";
            $message .= "- Total Gross Pay: KSh " . number_format($p9_data['totals']['gross_pay'], 2) . "\n";
            $message .= "- Total PAYE Tax: KSh " . number_format($p9_data['totals']['paye_tax'], 2) . "\n\n";
            $message .= "Please retain this document for your tax filing purposes.\n\n";
            $message .= "If you have any questions, please contact the HR Department.\n\n";
            $message .= "Best regards,\n";
            $message .= "HR Department\n";
            $message .= $this->config->item('hospital_name') ?: 'Hospital Management System';
            
            $this->email->from($this->config->item('smtp_from_email'), $this->config->item('hospital_name'));
            $this->email->to($staff->email);
            $this->email->subject($subject);
            $this->email->message($message);
            
            // Attach PDF
            $filename = 'P9_' . $staff->employee_id . '_' . $year . '.pdf';
            $this->email->attach($pdf_content, 'attachment', $filename, 'application/pdf');
            
            if ($this->email->send()) {
                return [
                    'status' => 'success',
                    'message' => 'P9 emailed successfully to ' . $staff->email
                ];
            } else {
                return [
                    'status' => 'error',
                    'message' => 'Failed to send email'
                ];
            }

        } catch (Exception $e) {
            return [
                'status' => 'error',
                'message' => 'Email error: ' . $e->getMessage()
            ];
        }
    }

    private function generateP9PDFContent($p9_data) {
        // This would use the same PDF generation logic as in the controller
        // For now, returning a placeholder
        return 'PDF content would be generated here';
    }

    // =============================================
    // P9 LISTING AND STATUS
    // =============================================

    public function getP9ListDatatables($year, $department_id = null) {
        $this->_getP9Query($year, $department_id);
        if($_POST['length'] != -1) {
            $this->db->limit($_POST['length'], $_POST['start']);
        }
        return $this->db->get()->result();
    }

    public function countAllP9($year, $department_id = null) {
        $this->_getP9BaseQuery($year, $department_id);
        return $this->db->count_all_results();
    }

    public function countFilteredP9($year, $department_id = null) {
        $this->_getP9Query($year, $department_id);
        return $this->db->count_all_results();
    }

    private function _getP9Query($year, $department_id = null) {
        $this->_getP9BaseQuery($year, $department_id);

        // Search functionality
        if(isset($_POST['search']['value']) && !empty($_POST['search']['value'])) {
            $search = $_POST['search']['value'];
            $this->db->group_start();
            $this->db->like('s.name', $search);
            $this->db->or_like('s.surname', $search);
            $this->db->or_like('s.employee_id', $search);
            $this->db->or_like('s.kra_pin', $search);
            $this->db->or_like('d.name', $search);
            $this->db->group_end();
        }

        // Ordering
        if(isset($_POST['order'])) {
            $column_order = ['s.employee_id', 's.name', 'd.name', 's.kra_pin'];
            $this->db->order_by($column_order[$_POST['order']['0']['column']], $_POST['order']['0']['dir']);
        } else {
            $this->db->order_by('s.employee_id', 'ASC');
        }
    }

    private function _getP9BaseQuery($year, $department_id = null) {
        $this->db->select('
            s.id, s.employee_id, s.name, s.surname, s.email, s.kra_pin,
            d.name as department_name,
            COALESCE(SUM(sp.gross_salary), 0) as total_gross,
            COALESCE(SUM(JSON_EXTRACT(sp.deductions_json, "$.PAYE")), 0) as total_paye
        ');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('staff_payroll sp', 's.id = sp.staff_id AND sp.year = ' . $year . ' AND sp.status != "draft"', 'left');
        $this->db->where('s.is_active', 'yes');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->group_by('s.id');
    }

    public function getP9Status($staff_id, $year) {
        // Check if P9 has been generated for this staff and year
        $this->db->where('staff_id', $staff_id);
        $this->db->where('year', $year);
        $p9_log = $this->db->get('p9_generation_log')->row();
        
        return [
            'generated' => $p9_log ? true : false,
            'generation_date' => $p9_log ? $p9_log->created_at : null,
            'generated_by' => $p9_log ? $p9_log->generated_by : null
        ];
    }

    // =============================================
    // KRA EXPORT
    // =============================================

    public function generateKRAExport($year, $department_id = null) {
        $this->db->select('
            s.employee_id,
            s.kra_pin,
            CONCAT(s.name, " ", s.surname) as full_name,
            s.id_number,
            d.name as department,
            SUM(sp.gross_salary) as annual_gross,
            SUM(JSON_EXTRACT(sp.deductions_json, "$.PAYE")) as annual_paye,
            SUM(JSON_EXTRACT(sp.deductions_json, "$.NSSF")) as annual_nssf,
            SUM(JSON_EXTRACT(sp.deductions_json, "$.NHIF")) as annual_nhif
        ');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('staff_payroll sp', 's.id = sp.staff_id AND sp.year = ' . $year, 'inner');
        $this->db->where('s.is_active', 'yes');
        $this->db->where('sp.status !=', 'draft');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->group_by('s.id');
        $this->db->order_by('s.employee_id', 'ASC');
        
        $data = $this->db->get()->result_array();
        
        // Add headers for CSV export
        $headers = [
            'Employee ID',
            'KRA PIN',
            'Full Name',
            'ID Number',
            'Department',
            'Annual Gross Salary',
            'Annual PAYE Tax',
            'Annual NSSF',
            'Annual NHIF'
        ];
        
        return [
            'headers' => $headers,
            'data' => $data,
            'summary' => [
                'total_staff' => count($data),
                'total_gross' => array_sum(array_column($data, 'annual_gross')),
                'total_paye' => array_sum(array_column($data, 'annual_paye')),
                'total_nssf' => array_sum(array_column($data, 'annual_nssf')),
                'total_nhif' => array_sum(array_column($data, 'annual_nhif'))
            ]
        ];
    }

    // =============================================
    // UTILITIES
    // =============================================

    public function getAvailableYears() {
        $this->db->select('DISTINCT year');
        $this->db->from('staff_payroll');
        $this->db->where('status !=', 'draft');
        $this->db->order_by('year', 'DESC');
        
        $results = $this->db->get()->result();
        $years = [];
        
        foreach ($results as $result) {
            $years[] = $result->year;
        }
        
        // Add current year if not in list
        $current_year = date('Y');
        if (!in_array($current_year, $years)) {
            array_unshift($years, $current_year);
        }
        
        return $years;
    }

    private function logP9Generation($staff_id, $year, $method) {
        $log_data = [
            'staff_id' => $staff_id,
            'year' => $year,
            'generation_method' => $method,
            'generated_by' => $this->session->userdata('admin_id'),
            'ip_address' => $this->input->ip_address(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        return $this->db->insert('p9_generation_log', $log_data);
    }

    private function deleteDirectory($dir) {
        if (is_dir($dir)) {
            $files = array_diff(scandir($dir), array('.', '..'));
            foreach ($files as $file) {
                (is_dir("$dir/$file")) ? $this->deleteDirectory("$dir/$file") : unlink("$dir/$file");
            }
            rmdir($dir);
        }
    }

    // =============================================
    // DATABASE INITIALIZATION
    // =============================================

    public function createP9Tables() {
        // P9 Generation Log Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `p9_generation_log` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `year` int(4) NOT NULL,
                `generation_method` varchar(50) DEFAULT 'manual',
                `generated_by` int(11) DEFAULT NULL,
                `ip_address` varchar(45) DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_year` (`staff_id`,`year`),
                KEY `generated_by` (`generated_by`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Add KRA PIN column to staff table if not exists
        $this->db->query("
            ALTER TABLE `staff` 
            ADD COLUMN IF NOT EXISTS `kra_pin` varchar(11) DEFAULT NULL,
            ADD COLUMN IF NOT EXISTS `id_number` varchar(20) DEFAULT NULL,
            ADD COLUMN IF NOT EXISTS `nssf_number` varchar(20) DEFAULT NULL,
            ADD COLUMN IF NOT EXISTS `nhif_number` varchar(20) DEFAULT NULL;
        ");

        // Add indexes for better performance
        $this->db->query("CREATE INDEX IF NOT EXISTS idx_staff_kra_pin ON staff(kra_pin);");
        $this->db->query("CREATE INDEX IF NOT EXISTS idx_payroll_year_staff ON staff_payroll(year, staff_id);");
        
        return true;
    }
}