<?php

defined('BASEPATH') OR exit('No direct script access allowed');

class Nursing_model extends CI_Model {

    private $tables = [
        'vital_signs' => 'nursing_vital_signs',
        'procedures' => 'nursing_procedures',
        'assessments' => 'nursing_assessments',
        'care_plans' => 'nursing_care_plans',
        'templates' => 'nursing_procedure_templates'
    ];

    public function __construct() {
        parent::__construct();
        $this->load->database();
        $this->load->helper(['date', 'url', 'string']);
        $this->initializeNursingSystem();
    }

    private function initializeNursingSystem() {
        try {
            $this->createEssentialTables();
            log_message('info', 'Nursing system initialized');
        } catch (Exception $e) {
            log_message('error', 'Nursing system initialization error: ' . $e->getMessage());
        }
    }

    private function createEssentialTables() {
        if (!$this->db->table_exists($this->tables['vital_signs'])) {
            $this->createVitalSignsTable();
        }
        if (!$this->db->table_exists($this->tables['procedures'])) {
            $this->createProceduresTable();
        }
        if (!$this->db->table_exists($this->tables['assessments'])) {
            $this->createAssessmentsTable();
        }
        if (!$this->db->table_exists($this->tables['care_plans'])) {
            $this->createCarePlansTable();
        }
        if (!$this->db->table_exists($this->tables['templates'])) {
            $this->createTemplatesTable();
        }
    }

    private function createVitalSignsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->tables['vital_signs']}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `ipd_id` int(11) DEFAULT NULL,
            `opd_id` int(11) DEFAULT NULL,
            `temperature` decimal(4,2) DEFAULT NULL,
            `temperature_unit` enum('C','F') DEFAULT 'C',
            `systolic_bp` int(3) DEFAULT NULL,
            `diastolic_bp` int(3) DEFAULT NULL,
            `pulse_rate` int(3) DEFAULT NULL,
            `respiratory_rate` int(3) DEFAULT NULL,
            `oxygen_saturation` int(3) DEFAULT NULL,
            `pain_scale` int(2) DEFAULT NULL,
            `weight` decimal(5,2) DEFAULT NULL,
            `height` decimal(5,2) DEFAULT NULL,
            `bmi` decimal(4,2) DEFAULT NULL,
            `blood_glucose` decimal(5,2) DEFAULT NULL,
            `notes` text DEFAULT NULL,
            `recorded_by` int(11) NOT NULL,
            `recorded_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_patient_id` (`patient_id`),
            KEY `idx_recorded_at` (`recorded_at`),
            KEY `idx_recorded_by` (`recorded_by`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        $this->db->query($sql);
    }

    private function createProceduresTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->tables['procedures']}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `ipd_id` int(11) DEFAULT NULL,
            `opd_id` int(11) DEFAULT NULL,
            `procedure_type` varchar(100) NOT NULL,
            `procedure_name` varchar(255) NOT NULL,
            `procedure_code` varchar(50) DEFAULT NULL,
            `procedure_description` text DEFAULT NULL,
            `description` text DEFAULT NULL,
            `pre_procedure_notes` text DEFAULT NULL,
            `post_procedure_notes` text DEFAULT NULL,
            `procedure_notes` text DEFAULT NULL,
            `complications` text DEFAULT NULL,
            `outcome` enum('successful','partial','failed','complications','cancelled') DEFAULT 'successful',
            `status` enum('scheduled','in_progress','completed','cancelled','on_hold') DEFAULT 'scheduled',
            `priority` enum('routine','urgent','emergent','stat') DEFAULT 'routine',
            `performed_by` int(11) NOT NULL,
            `performed_at` timestamp NULL DEFAULT NULL,
            `duration_minutes` int(4) DEFAULT NULL,
            `equipment_used` text DEFAULT NULL,
            `patient_response` text DEFAULT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_patient_id` (`patient_id`),
            KEY `idx_procedure_type` (`procedure_type`),
            KEY `idx_status` (`status`),
            KEY `idx_performed_by` (`performed_by`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        $this->db->query($sql);
    }

    private function createAssessmentsTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->tables['assessments']}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `ipd_id` int(11) DEFAULT NULL,
            `assessment_type` varchar(100) NOT NULL,
            `assessment_category` enum('initial','ongoing','discharge','emergency','admission','routine','pre_operative','post_operative') DEFAULT 'ongoing',
            `assessment_details` text DEFAULT NULL,
            `findings` text DEFAULT NULL,
            `recommendations` text DEFAULT NULL,
            `assessment_score` int(3) DEFAULT NULL,
            `risk_level` enum('low','medium','high','critical') DEFAULT 'low',
            `risk_factors` text DEFAULT NULL,
            `follow_up_required` tinyint(1) DEFAULT 0,
            `follow_up_date` datetime DEFAULT NULL,
            `assessed_by` int(11) NOT NULL,
            `assessment_date` timestamp DEFAULT CURRENT_TIMESTAMP,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_patient_id` (`patient_id`),
            KEY `idx_assessment_type` (`assessment_type`),
            KEY `idx_risk_level` (`risk_level`),
            KEY `idx_assessed_by` (`assessed_by`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        $this->db->query($sql);
    }

    private function createCarePlansTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->tables['care_plans']}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `patient_id` int(11) NOT NULL,
            `ipd_id` int(11) DEFAULT NULL,
            `plan_name` varchar(255) NOT NULL,
            `nursing_diagnosis` text DEFAULT NULL,
            `nursing_diagnoses` text DEFAULT NULL,
            `goals_outcomes` text DEFAULT NULL,
            `goals_short_term` text DEFAULT NULL,
            `goals_long_term` text DEFAULT NULL,
            `interventions` text DEFAULT NULL,
            `evaluation_criteria` text DEFAULT NULL,
            `priority` enum('low','normal','medium','high','urgent') DEFAULT 'medium',
            `status` enum('active','completed','discontinued','on_hold','revised') DEFAULT 'active',
            `start_date` date NOT NULL,
            `target_date` date DEFAULT NULL,
            `completion_date` date DEFAULT NULL,
            `created_by` int(11) NOT NULL,
            `assigned_to` int(11) DEFAULT NULL,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_patient_id` (`patient_id`),
            KEY `idx_status` (`status`),
            KEY `idx_priority` (`priority`),
            KEY `idx_created_by` (`created_by`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        $this->db->query($sql);
    }

    private function createTemplatesTable() {
        $sql = "CREATE TABLE IF NOT EXISTS `{$this->tables['templates']}` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `procedure_name` varchar(255) NOT NULL,
            `procedure_type` varchar(100) NOT NULL,
            `procedure_code` varchar(50) DEFAULT NULL,
            `category` varchar(100) DEFAULT NULL,
            `description` text DEFAULT NULL,
            `indication` text DEFAULT NULL,
            `pre_procedure_checklist` text DEFAULT NULL,
            `procedure_steps` text DEFAULT NULL,
            `post_procedure_checklist` text DEFAULT NULL,
            `equipment_needed` text DEFAULT NULL,
            `supplies_needed` text DEFAULT NULL,
            `estimated_duration` int(4) DEFAULT NULL,
            `skill_level_required` enum('basic','intermediate','advanced','expert') DEFAULT 'basic',
            `safety_measures` text DEFAULT NULL,
            `documentation_requirements` text DEFAULT NULL,
            `is_active` tinyint(1) DEFAULT 1,
            `usage_count` int(11) DEFAULT 0,
            `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
            `updated_at` timestamp DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `idx_procedure_type` (`procedure_type`),
            KEY `idx_category` (`category`),
            KEY `idx_is_active` (`is_active`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4";
        $this->db->query($sql);
    }

    public function forceInitialization() {
        try {
            $this->createEssentialTables();
            $this->insertBasicProcedureTemplates();
            return true;
        } catch (Exception $e) {
            log_message('error', 'Force initialization failed: ' . $e->getMessage());
            throw $e;
        }
    }
    
    /**
 * Get patient consultations with doctor notes
 */
public function getPatientConsultations($patient_id, $limit = 20) {
    try {
        $this->db->select('
            c.id,
            c.date as consultation_date,
            c.symptoms,
            c.diagnosis,
            c.notes as consultation_notes,
            c.verification_status,
            c.verification_score,
            c.has_prescription,
            c.prescription_count,
            c.prescription_total_cost,
            c.general_prescription_notes,
            s.name as doctor_name,
            s.surname as doctor_surname,
            s.employee_id as doctor_employee_id
        ');
        $this->db->from('consultations c');
        $this->db->join('staff s', 's.id = c.doctor', 'left');
        $this->db->where('c.patient_id', $patient_id);
        $this->db->order_by('c.date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting consultations: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient prescriptions
 */
public function getPatientPrescriptions($patient_id, $limit = 15) {
    try {
        // Try modern prescriptions table first
        if ($this->db->table_exists('prescriptions')) {
            $this->db->select('
                p.id,
                p.prescription_date,
                p.prescription_time,
                p.status,
                p.total_cost,
                p.general_instructions,
                p.pharmacy_notes,
                p.dispensed_at,
                s1.name as doctor_name,
                s1.surname as doctor_surname,
                s2.name as dispensed_by_name,
                s2.surname as dispensed_by_surname
            ');
            $this->db->from('prescriptions p');
            $this->db->join('staff s1', 's1.id = p.doctor_id', 'left');
            $this->db->join('staff s2', 's2.id = p.dispensed_by', 'left');
            $this->db->where('p.patient_id', $patient_id);
            $this->db->order_by('p.prescription_date', 'DESC');
            $this->db->limit($limit);
            
            $query = $this->db->get();
            $prescriptions = $query->result_array();
            
            // Get prescription items for each prescription
            foreach ($prescriptions as &$prescription) {
                $prescription['items'] = $this->getPrescriptionItems($prescription['id']);
            }
            
            return $prescriptions;
        }
        
        // Fallback to consultation_prescriptions if available
        if ($this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.id,
                cp.created_at as prescription_date,
                cp.status,
                cp.total_cost,
                cp.dosage_instructions,
                cp.duration,
                cp.special_instructions,
                cp.dispensed_at,
                m.medicine_name,
                m.medicine_category,
                s1.name as doctor_name,
                s1.surname as doctor_surname,
                c.date as consultation_date
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->join('consultations c', 'c.id = cp.consultation_id', 'left');
            $this->db->join('staff s1', 's1.id = c.doctor', 'left');
            $this->db->where('cp.patient_id', $patient_id);
            $this->db->order_by('cp.created_at', 'DESC');
            $this->db->limit($limit);
            
            $query = $this->db->get();
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting prescriptions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get prescription items for a prescription
 */
public function getPrescriptionItems($prescription_id) {
    try {
        if ($this->db->table_exists('prescription_items')) {
            $this->db->select('
                pi.id,
                pi.quantity,
                pi.dosage_instructions,
                pi.duration_days,
                pi.special_instructions,
                pi.unit_cost,
                pi.total_cost,
                m.medicine_name,
                m.medicine_category,
                m.medicine_type
            ');
            $this->db->from('prescription_items pi');
            $this->db->join('medicine m', 'm.id = pi.medicine_id', 'left');
            $this->db->where('pi.prescription_id', $prescription_id);
            $this->db->order_by('pi.id', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting prescription items: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient clinical notes from various sources
 */
public function getPatientClinicalNotes($patient_id, $limit = 20) {
    try {
        $clinical_notes = [];
        
        // Get OPD visit notes
        if ($this->db->table_exists('opd_details')) {
            $this->db->select('
                od.id,
                od.opd_no,
                od.case_reference_id,
                od.appointment_date,
                od.symptoms,
                od.any_known_allergies,
                od.note,
                od.refference,
                od.cons_doctor,
                s.name as doctor_name,
                s.surname as doctor_surname,
                "opd_visit" as note_type
            ');
            $this->db->from('opd_details od');
            $this->db->join('staff s', 's.id = od.cons_doctor', 'left');
            $this->db->where('od.patient_id', $patient_id);
            $this->db->order_by('od.appointment_date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            $opd_notes = $query->result_array();
            
            foreach ($opd_notes as $note) {
                $clinical_notes[] = [
                    'id' => $note['id'],
                    'date' => $note['appointment_date'],
                    'type' => 'OPD Visit',
                    'content' => $note['symptoms'],
                    'additional_info' => $note['note'],
                    'doctor_name' => ($note['doctor_name'] ?? '') . ' ' . ($note['doctor_surname'] ?? ''),
                    'reference' => $note['opd_no'],
                    'allergies' => $note['any_known_allergies']
                ];
            }
        }
        
        // Get IPD consultant notes
        if ($this->db->table_exists('consultant_register')) {
            $this->db->select('
                cr.id,
                cr.date,
                cr.instruction,
                s.name as doctor_name,
                s.surname as doctor_surname,
                ipd.case_reference_id,
                "consultant_note" as note_type
            ');
            $this->db->from('consultant_register cr');
            $this->db->join('ipd_details ipd', 'ipd.id = cr.ipd_id', 'left');
            $this->db->join('staff s', 's.id = cr.cons_doctor', 'left');
            $this->db->where('ipd.patient_id', $patient_id);
            $this->db->order_by('cr.date', 'DESC');
            $this->db->limit(10);
            
            $query = $this->db->get();
            $consultant_notes = $query->result_array();
            
            foreach ($consultant_notes as $note) {
                $clinical_notes[] = [
                    'id' => $note['id'],
                    'date' => $note['date'],
                    'type' => 'Consultant Note',
                    'content' => $note['instruction'],
                    'doctor_name' => ($note['doctor_name'] ?? '') . ' ' . ($note['doctor_surname'] ?? ''),
                    'reference' => $note['case_reference_id']
                ];
            }
        }
        
        // Sort all notes by date descending
        usort($clinical_notes, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });
        
        return array_slice($clinical_notes, 0, $limit);
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting clinical notes: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient OPD visits
 */
public function getPatientOPDVisits($patient_id, $limit = 10) {
    try {
        if (!$this->db->table_exists('opd_details')) {
            return [];
        }
        
        $this->db->select('
            od.id,
            od.opd_no,
            od.case_reference_id,
            od.appointment_date,
            od.appointment_time,
            od.symptoms,
            od.any_known_allergies,
            od.note,
            od.height,
            od.weight,
            od.bp,
            od.pulse,
            od.temperature,
            od.respiration,
            s.name as doctor_name,
            s.surname as doctor_surname
        ');
        $this->db->from('opd_details od');
        $this->db->join('staff s', 's.id = od.cons_doctor', 'left');
        $this->db->where('od.patient_id', $patient_id);
        $this->db->order_by('od.appointment_date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting OPD visits: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get patient IPD visits
 */
public function getPatientIPDVisits($patient_id, $limit = 10) {
    try {
        if (!$this->db->table_exists('ipd_details')) {
            return [];
        }
        
        $this->db->select('
            ipd.id,
            ipd.case_reference_id,
            ipd.date as admission_date,
            ipd.discharged,
            ipd.discharge_date,
            ipd.symptoms,
            ipd.any_known_allergies,
            ipd.note,
            ipd.height,
            ipd.weight,
            ipd.bp,
            ipd.pulse,
            ipd.temperature,
            ipd.respiration,
            ipd.bed,
            ipd.bed_group_id,
            s.name as doctor_name,
            s.surname as doctor_surname
        ');
        $this->db->from('ipd_details ipd');
        $this->db->join('staff s', 's.id = ipd.cons_doctor', 'left');
        $this->db->where('ipd.patient_id', $patient_id);
        $this->db->order_by('ipd.date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        return $query->result_array();
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting IPD visits: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get consultation details by ID
 */
public function getConsultationDetails($consultation_id) {
    try {
        $this->db->select('
            c.*,
            s.name as doctor_name,
            s.surname as doctor_surname,
            s.employee_id as doctor_employee_id,
            p.patient_name,
            p.patient_unique_id
        ');
        $this->db->from('consultations c');
        $this->db->join('staff s', 's.id = c.doctor', 'left');
        $this->db->join('patients p', 'p.id = c.patient_id', 'left');
        $this->db->where('c.id', $consultation_id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() == 0) {
            return null;
        }
        
        $consultation = $query->row_array();
        
        // Get related prescriptions if any
        $consultation['prescriptions'] = $this->getPrescriptionsByConsultation($consultation_id);
        
        return $consultation;
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting consultation details: ' . $e->getMessage());
        return null;
    }
}

/**
 * Get prescriptions by consultation ID
 */
public function getPrescriptionsByConsultation($consultation_id) {
    try {
        if ($this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.*,
                m.medicine_name,
                m.medicine_category
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->where('cp.consultation_id', $consultation_id);
            $this->db->order_by('cp.id', 'ASC');
            
            $query = $this->db->get();
            return $query->result_array();
        }
        
        return [];
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting consultation prescriptions: ' . $e->getMessage());
        return [];
    }
}

/**
 * Get prescription details by ID
 */
public function getPrescriptionDetails($prescription_id) {
    try {
        $this->db->select('
            p.*,
            s1.name as doctor_name,
            s1.surname as doctor_surname,
            s2.name as dispensed_by_name,
            s2.surname as dispensed_by_surname,
            pt.patient_name
        ');
        $this->db->from('prescriptions p');
        $this->db->join('staff s1', 's1.id = p.doctor_id', 'left');
        $this->db->join('staff s2', 's2.id = p.dispensed_by', 'left');
        $this->db->join('patients pt', 'pt.id = p.patient_id', 'left');
        $this->db->where('p.id', $prescription_id);
        
        $query = $this->db->get();
        
        if ($query->num_rows() == 0) {
            return null;
        }
        
        $prescription = $query->row_array();
        
        // Get prescription items
        $prescription['items'] = $this->getPrescriptionItems($prescription_id);
        
        return $prescription;
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting prescription details: ' . $e->getMessage());
        return null;
    }
}

/**
 * Get patient's recent medical history summary
 */
public function getPatientMedicalSummary($patient_id) {
    try {
        $summary = [
            'consultations_count' => 0,
            'prescriptions_count' => 0,
            'opd_visits_count' => 0,
            'ipd_visits_count' => 0,
            'last_consultation_date' => null,
            'last_prescription_date' => null,
            'active_prescriptions' => 0,
            'known_allergies' => [],
            'chronic_conditions' => []
        ];
        
        // Count consultations
        if ($this->db->table_exists('consultations')) {
            $this->db->where('patient_id', $patient_id);
            $summary['consultations_count'] = $this->db->count_all_results('consultations');
            
            // Get last consultation date
            $this->db->select('date');
            $this->db->from('consultations');
            $this->db->where('patient_id', $patient_id);
            $this->db->order_by('date', 'DESC');
            $this->db->limit(1);
            $query = $this->db->get();
            if ($query->num_rows() > 0) {
                $summary['last_consultation_date'] = $query->row()->date;
            }
        }
        
        // Count prescriptions
        if ($this->db->table_exists('prescriptions')) {
            $this->db->where('patient_id', $patient_id);
            $summary['prescriptions_count'] = $this->db->count_all_results('prescriptions');
            
            // Count active prescriptions
            $this->db->where('patient_id', $patient_id);
            $this->db->where_in('status', ['pending', 'partially_dispensed']);
            $summary['active_prescriptions'] = $this->db->count_all_results('prescriptions');
            
            // Get last prescription date
            $this->db->select('prescription_date');
            $this->db->from('prescriptions');
            $this->db->where('patient_id', $patient_id);
            $this->db->order_by('prescription_date', 'DESC');
            $this->db->limit(1);
            $query = $this->db->get();
            if ($query->num_rows() > 0) {
                $summary['last_prescription_date'] = $query->row()->prescription_date;
            }
        }
        
        // Count OPD visits
        if ($this->db->table_exists('opd_details')) {
            $this->db->where('patient_id', $patient_id);
            $summary['opd_visits_count'] = $this->db->count_all_results('opd_details');
        }
        
        // Count IPD visits
        if ($this->db->table_exists('ipd_details')) {
            $this->db->where('patient_id', $patient_id);
            $summary['ipd_visits_count'] = $this->db->count_all_results('ipd_details');
        }
        
        // Get known allergies from recent visits
        $allergies = [];
        if ($this->db->table_exists('opd_details')) {
            $this->db->select('any_known_allergies');
            $this->db->from('opd_details');
            $this->db->where('patient_id', $patient_id);
            $this->db->where('any_known_allergies IS NOT NULL');
            $this->db->where('any_known_allergies !=', '');
            $this->db->order_by('appointment_date', 'DESC');
            $this->db->limit(5);
            $query = $this->db->get();
            foreach ($query->result() as $row) {
                if (!empty($row->any_known_allergies)) {
                    $allergies[] = $row->any_known_allergies;
                }
            }
        }
        $summary['known_allergies'] = array_unique($allergies);
        
        return $summary;
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error getting medical summary: ' . $e->getMessage());
        return [];
    }
}

/**
 * Search patient medical records
 */
public function searchPatientMedicalRecords($patient_id, $search_term, $record_types = ['consultations', 'prescriptions', 'clinical_notes']) {
    try {
        $results = [];
        
        if (in_array('consultations', $record_types) && $this->db->table_exists('consultations')) {
            $this->db->select('
                c.id,
                c.date,
                c.symptoms,
                c.diagnosis,
                c.notes,
                s.name as doctor_name,
                s.surname as doctor_surname,
                "consultation" as record_type
            ');
            $this->db->from('consultations c');
            $this->db->join('staff s', 's.id = c.doctor', 'left');
            $this->db->where('c.patient_id', $patient_id);
            $this->db->group_start();
            $this->db->like('c.symptoms', $search_term);
            $this->db->or_like('c.diagnosis', $search_term);
            $this->db->or_like('c.notes', $search_term);
            $this->db->group_end();
            $this->db->order_by('c.date', 'DESC');
            
            $query = $this->db->get();
            $results = array_merge($results, $query->result_array());
        }
        
        if (in_array('prescriptions', $record_types) && $this->db->table_exists('consultation_prescriptions')) {
            $this->db->select('
                cp.id,
                cp.created_at as date,
                cp.dosage_instructions,
                cp.special_instructions,
                m.medicine_name,
                s.name as doctor_name,
                s.surname as doctor_surname,
                "prescription" as record_type
            ');
            $this->db->from('consultation_prescriptions cp');
            $this->db->join('medicine m', 'm.id = cp.medicine_id', 'left');
            $this->db->join('consultations c', 'c.id = cp.consultation_id', 'left');
            $this->db->join('staff s', 's.id = c.doctor', 'left');
            $this->db->where('cp.patient_id', $patient_id);
            $this->db->group_start();
            $this->db->like('m.medicine_name', $search_term);
            $this->db->or_like('cp.dosage_instructions', $search_term);
            $this->db->or_like('cp.special_instructions', $search_term);
            $this->db->group_end();
            $this->db->order_by('cp.created_at', 'DESC');
            
            $query = $this->db->get();
            $results = array_merge($results, $query->result_array());
        }
        
        // Sort results by date
        usort($results, function($a, $b) {
            return strtotime($b['date']) - strtotime($a['date']);
        });
        
        return $results;
        
    } catch (Exception $e) {
        log_message('error', 'Nursing model error searching medical records: ' . $e->getMessage());
        return [];
    }
}

    private function insertBasicProcedureTemplates() {
        if ($this->db->table_exists($this->tables['templates'])) {
            $this->db->select('COUNT(*) as count');
            $this->db->from($this->tables['templates']);
            $query = $this->db->get();
            $result = $query->row();
            if ($result->count > 0) return;
        }

        $templates = [
            [
                'procedure_name' => 'Blood Pressure Measurement',
                'procedure_type' => 'vital_signs',
                'procedure_code' => 'VS-001',
                'category' => 'Vital Signs',
                'description' => 'Standard blood pressure measurement using sphygmomanometer',
                'indication' => 'Routine vital signs monitoring, hypertension screening',
                'equipment_needed' => 'Sphygmomanometer, stethoscope',
                'estimated_duration' => 5,
                'skill_level_required' => 'basic',
                'is_active' => 1
            ],
            [
                'procedure_name' => 'Temperature Measurement',
                'procedure_type' => 'vital_signs',
                'procedure_code' => 'VS-002',
                'category' => 'Vital Signs',
                'description' => 'Body temperature measurement using digital thermometer',
                'indication' => 'Routine monitoring, fever assessment',
                'equipment_needed' => 'Digital thermometer, probe covers',
                'estimated_duration' => 3,
                'skill_level_required' => 'basic',
                'is_active' => 1
            ],
            [
                'procedure_name' => 'Wound Dressing',
                'procedure_type' => 'wound_care',
                'procedure_code' => 'WC-001',
                'category' => 'Wound Care',
                'description' => 'Sterile wound dressing change',
                'indication' => 'Post-operative wound care, chronic wound management',
                'equipment_needed' => 'Sterile gloves, gauze, tape, antiseptic',
                'estimated_duration' => 15,
                'skill_level_required' => 'intermediate',
                'is_active' => 1
            ],
            [
                'procedure_name' => 'IV Insertion',
                'procedure_type' => 'iv_therapy',
                'procedure_code' => 'IV-001',
                'category' => 'IV Therapy',
                'description' => 'Peripheral intravenous catheter insertion',
                'indication' => 'Medication administration, fluid therapy',
                'equipment_needed' => 'IV catheter, tourniquet, alcohol swabs, tape',
                'estimated_duration' => 10,
                'skill_level_required' => 'intermediate',
                'is_active' => 1
            ],
            [
                'procedure_name' => 'Catheter Care',
                'procedure_type' => 'catheter_care',
                'procedure_code' => 'CC-001',
                'category' => 'Catheter Care',
                'description' => 'Urinary catheter maintenance and care',
                'indication' => 'Urinary retention, post-operative care',
                'equipment_needed' => 'Sterile gloves, antiseptic, catheter bag',
                'estimated_duration' => 10,
                'skill_level_required' => 'basic',
                'is_active' => 1
            ]
        ];

        foreach ($templates as $template) {
            $this->db->insert($this->tables['templates'], $template);
        }
        log_message('info', 'Basic procedure templates inserted');
    }

    private function formatPatientAge($patient) {
        try {
            if (!isset($this->customlib)) {
                $this->load->library('customlib');
            }
            if (method_exists($this->customlib, 'getPatientAge')) {
                $age = $patient['age'] ?? 0;
                $month = $patient['month'] ?? 0;
                $day = $patient['day'] ?? 0;
                return $this->customlib->getPatientAge($age, $month, $day);
            }
            if (!empty($patient['age'])) {
                return $patient['age'] . ' years';
            }
            return 'N/A';
        } catch (Exception $e) {
            log_message('error', 'Format patient age error: ' . $e->getMessage());
            return !empty($patient['age']) ? $patient['age'] . ' years' : 'N/A';
        }
    }

    private function checkIPDStatus($patient_id) {
        try {
            if ($this->db->table_exists('ipd_details')) {
                $this->db->select('id');
                $this->db->from('ipd_details');
                $this->db->where('patient_id', $patient_id);
                $this->db->where('discharged', 'no');
                $query = $this->db->get();
                return $query->num_rows() > 0;
            }
            return false;
        } catch (Exception $e) {
            return false;
        }
    }

    public function getPatientIPDDetails($patient_id) {
        try {
            // Check what fields actually exist in ipd_details table
            $ipd_fields = $this->db->list_fields('ipd_details');
            
            // Build select based on available fields
            $select_fields = ['ipd.*', 'p.patient_name', 'p.mobileno', 'p.age', 'p.gender'];
            
            // Add doctor fields if cons_doctor field exists
            if (in_array('cons_doctor', $ipd_fields)) {
                $select_fields[] = 's.name as doctor_name';
                $select_fields[] = 's.surname as doctor_surname';
            }
            
            $this->db->select(implode(', ', $select_fields));
            $this->db->from('ipd_details ipd');
            $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
            
            // Only join staff if cons_doctor field exists
            if (in_array('cons_doctor', $ipd_fields)) {
                $this->db->join('staff s', 's.id = ipd.cons_doctor', 'left');
            }
            
            $this->db->where('ipd.patient_id', $patient_id);
            $this->db->where('ipd.discharged', 'no');
            $this->db->order_by('ipd.date', 'DESC');
            $this->db->limit(1);
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'IPD details error: ' . $e->getMessage());
            return null;
        }
    }

    public function addVitalSigns($data) {
        try {
            if (!empty($data['height']) && !empty($data['weight'])) {
                $height_m = $data['height'] / 100;
                $data['bmi'] = round($data['weight'] / ($height_m * $height_m), 2);
            }
            if (empty($data['patient_id']) || empty($data['recorded_by'])) {
                throw new Exception('Patient ID and recorded by staff ID are required');
            }
            $this->db->insert($this->tables['vital_signs'], $data);
            $insert_id = $this->db->insert_id();
            if (!$insert_id) {
                throw new Exception('Failed to insert vital signs record');
            }
            return $insert_id;
        } catch (Exception $e) {
            log_message('error', 'Vital signs error: ' . $e->getMessage());
            throw new Exception('Failed to record vital signs: ' . $e->getMessage());
        }
    }

    public function getPatientVitalSigns($patient_id, $limit = 20) {
        try {
            if (!$this->db->table_exists($this->tables['vital_signs'])) {
                return [];
            }
            $this->db->select('
                vs.*, s.name as recorded_by_name, s.surname as recorded_by_surname
            ');
            $this->db->from($this->tables['vital_signs'] . ' vs');
            $this->db->join('staff s', 's.id = vs.recorded_by', 'left');
            $this->db->where('vs.patient_id', $patient_id);
            $this->db->order_by('vs.recorded_at', 'DESC');
            $this->db->limit($limit);
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Get vital signs error: ' . $e->getMessage());
            return [];
        }
    }

    public function getVitalSignsById($vital_id) {
        try {
            if (!$this->db->table_exists($this->tables['vital_signs'])) {
                return null;
            }
            $this->db->select('
                vs.*, 
                s.name as recorded_by_name, 
                s.surname as recorded_by_surname,
                p.patient_name
            ');
            $this->db->from($this->tables['vital_signs'] . ' vs');
            $this->db->join('staff s', 's.id = vs.recorded_by', 'left');
            $this->db->join('patients p', 'p.id = vs.patient_id', 'left');
            $this->db->where('vs.id', $vital_id);
            $query = $this->db->get();
            return $query->row_array();
        } catch (Exception $e) {
            log_message('error', 'Get vital signs by ID error: ' . $e->getMessage());
            return null;
        }
    }

    public function getAllRecentVitalSigns($limit = 50) {
        try {
            if (!$this->db->table_exists($this->tables['vital_signs'])) {
                return [];
            }
            $this->db->select('
                vs.*, 
                s.name as recorded_by_name, 
                s.surname as recorded_by_surname,
                p.patient_name, 
                p.gender,
                p.age
            ');
            $this->db->from($this->tables['vital_signs'] . ' vs');
            $this->db->join('staff s', 's.id = vs.recorded_by', 'left');
            $this->db->join('patients p', 'p.id = vs.patient_id', 'left');
            $this->db->order_by('vs.recorded_at', 'DESC');
            $this->db->limit($limit);
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Get all recent vital signs error: ' . $e->getMessage());
            return [];
        }
    }

    public function deleteVitalSigns($vital_id) {
        try {
            if (!$this->db->table_exists($this->tables['vital_signs'])) {
                return false;
            }
            $this->db->where('id', $vital_id);
            $this->db->delete($this->tables['vital_signs']);
            return $this->db->affected_rows() > 0;
        } catch (Exception $e) {
            log_message('error', 'Delete vital signs error: ' . $e->getMessage());
            return false;
        }
    }

public function addNurseNote($data) {
    try {
        log_message('info', '=== NURSING MODEL addNurseNote START ===');
        log_message('info', 'Input data: ' . json_encode($data));
        
        if (!$this->db->table_exists('nurse_note')) {
            throw new Exception('Nurse note table does not exist');
        }
        
        $fields = $this->db->list_fields('nurse_note');
        log_message('info', 'Nurse note table fields: ' . json_encode($fields));
        
        // Validate required fields
        if (empty($data['staff_id'])) {
            throw new Exception('Staff ID is required');
        }
        if (empty($data['note'])) {
            throw new Exception('Note content is required');
        }
        
        // CRITICAL: Ensure patient_id is set if available
        if (empty($data['patient_id']) && !empty($data['ipd_id'])) {
            // Try to get patient_id from IPD record
            $this->db->select('patient_id');
            $this->db->from('ipd_details');
            $this->db->where('id', $data['ipd_id']);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $ipd_row = $query->row();
                $data['patient_id'] = $ipd_row->patient_id;
                log_message('info', 'Retrieved patient_id from IPD: ' . $data['patient_id']);
            }
        }
        
        // Build the note data with only existing fields
        $note_data = [];
        
        // Essential fields mapping
        $field_mappings = [
            'staff_id' => $data['staff_id'],
            'date' => $data['date'] ?: date('Y-m-d H:i:s'),
            'note' => $data['note'],
            'comment' => $data['comment'] ?? ''
        ];
        
        // Add patient_id if field exists and data is available
        if (in_array('patient_id', $fields) && !empty($data['patient_id'])) {
            $field_mappings['patient_id'] = $data['patient_id'];
            log_message('info', 'Adding patient_id to note: ' . $data['patient_id']);
        }
        
        // Add IPD ID if field exists and data is available
        if (in_array('ipd_id', $fields) && !empty($data['ipd_id'])) {
            $field_mappings['ipd_id'] = $data['ipd_id'];
            log_message('info', 'Adding ipd_id to note: ' . $data['ipd_id']);
        }
        
        // Add optional fields if they exist in the table
        $optional_fields = ['created_at', 'created_by', 'updated_at', 'note_type'];
        foreach ($optional_fields as $field) {
            if (in_array($field, $fields) && isset($data[$field])) {
                $field_mappings[$field] = $data[$field];
            }
        }
        
        // Set created_at if field exists and not set
        if (in_array('created_at', $fields) && !isset($field_mappings['created_at'])) {
            $field_mappings['created_at'] = date('Y-m-d H:i:s');
        }
        
        // Set created_by if field exists and not set
        if (in_array('created_by', $fields) && !isset($field_mappings['created_by'])) {
            $field_mappings['created_by'] = $data['staff_id'];
        }
        
        // Only include fields that exist in the table
        foreach ($field_mappings as $field => $value) {
            if (in_array($field, $fields)) {
                $note_data[$field] = $value;
            } else {
                log_message('info', "Skipping field '{$field}' - not found in table");
            }
        }
        
        log_message('info', 'Final note data for insertion: ' . json_encode($note_data));
        
        // Validate that we have the minimum required data
        if (empty($note_data['staff_id']) || empty($note_data['note'])) {
            throw new Exception('Missing required fields after processing');
        }
        
        // Insert the note
        $this->db->insert('nurse_note', $note_data);
        $insert_id = $this->db->insert_id();
        
        if (!$insert_id) {
            $db_error = $this->db->error();
            log_message('error', 'Database insert failed: ' . json_encode($db_error));
            log_message('error', 'Failed query: ' . $this->db->last_query());
            throw new Exception('Failed to insert nursing note. Database error: ' . $db_error['message']);
        }
        
        log_message('info', 'Nurse note inserted successfully with ID: ' . $insert_id);
        log_message('info', 'Insert query: ' . $this->db->last_query());
        
        // Verify the insertion
        $this->db->select('*');
        $this->db->from('nurse_note');
        $this->db->where('id', $insert_id);
        $verify_query = $this->db->get();
        
        if ($verify_query && $verify_query->num_rows() > 0) {
            $inserted_row = $verify_query->row_array();
            log_message('info', 'Verification - inserted row: ' . json_encode($inserted_row));
            
            // Check if patient_id was properly saved..
            if (in_array('patient_id', $fields)) {
                if (empty($inserted_row['patient_id'])) {
                    log_message('warning', 'WARNING: patient_id is NULL in inserted record');
                } else {
                    log_message('info', 'SUCCESS: patient_id properly saved: ' . $inserted_row['patient_id']);
                }
            }
        }
        
        log_message('info', '=== NURSING MODEL addNurseNote END ===');
        return $insert_id;
        
    } catch (Exception $e) {
        log_message('error', 'Add nurse note error: ' . $e->getMessage());
        log_message('error', 'Stack trace: ' . $e->getTraceAsString());
        throw new Exception('Failed to add nursing note: ' . $e->getMessage());
    }
}

public function getPatientNurseNotes($patient_id, $limit = 50) {
    try {
        log_message('debug', 'Getting nurse notes for patient: ' . $patient_id);
        
        if (!$this->db->table_exists('nurse_note')) {
            log_message('error', 'nurse_note table does not exist');
            return [];
        }
        
        $note_fields = $this->db->list_fields('nurse_note');
        log_message('debug', 'nurse_note table fields: ' . json_encode($note_fields));
        
        $this->db->select('
            nn.*, s.name as nurse_name, s.surname as nurse_surname
        ');
        $this->db->from('nurse_note nn');
        $this->db->join('staff s', 's.id = nn.staff_id', 'left');
        
        // Prioritize direct patient_id link if available
        if (in_array('patient_id', $note_fields)) {
            log_message('debug', 'Using direct patient_id link');
            // Get notes with direct patient_id OR through IPD
            $this->db->group_start();
            $this->db->where('nn.patient_id', $patient_id);
            if (in_array('ipd_id', $note_fields)) {
                $this->db->or_where("nn.ipd_id IN (SELECT id FROM ipd_details WHERE patient_id = {$patient_id})");
            }
            $this->db->group_end();
        } elseif (in_array('ipd_id', $note_fields)) {
            // Fallback to IPD link only
            log_message('debug', 'Using IPD link only');
            $this->db->join('ipd_details ipd', 'ipd.id = nn.ipd_id', 'left');
            $this->db->where('ipd.patient_id', $patient_id);
        } else {
            log_message('error', 'No patient linking fields available in nurse_note table');
            return [];
        }
        
        $this->db->order_by('nn.date', 'DESC');
        $this->db->limit($limit);
        
        $query = $this->db->get();
        $notes = $query->result_array();
        
        log_message('debug', 'SQL Query: ' . $this->db->last_query());
        log_message('debug', 'Notes found: ' . count($notes));
        
        return $notes;
        
    } catch (Exception $e) {
        log_message('error', 'Get nurse notes error: ' . $e->getMessage());
        return [];
    }
}

    public function getPatientMedications($patient_id) {
        try {
            if (!$this->db->table_exists('medication_report')) {
                return [];
            }
            
            // Check what fields exist in medication_report
            $med_fields = $this->db->list_fields('medication_report');
            
            // Build select based on available fields
            $select_fields = ['mr.*'];
            $join_staff_field = 'generated_by'; // Default
            
            if (in_array('given_by', $med_fields)) {
                $join_staff_field = 'given_by';
            } elseif (in_array('administered_by', $med_fields)) {
                $join_staff_field = 'administered_by';
            }
            
            $select_fields[] = 's.name as given_by_name';
            $select_fields[] = 's.surname as given_by_surname';
            
            $this->db->select(implode(', ', $select_fields));
            $this->db->from('medication_report mr');
            $this->db->join('staff s', "s.id = mr.{$join_staff_field}", 'left');
            $this->db->join('ipd_details ipd', 'ipd.id = mr.ipd_id', 'left');
            $this->db->where('ipd.patient_id', $patient_id);
            $this->db->order_by('mr.date', 'DESC');
            $this->db->limit(50);
            $query = $this->db->get();
            return $query->result_array();
        } catch (Exception $e) {
            log_message('error', 'Get medications error: ' . $e->getMessage());
            return [];
        }
    }

    // NEW METHODS ADDED FOR CONTROLLER COMPATIBILITY

public function addAssessment($data) {
    try {
        if (!$this->db->table_exists($this->tables['assessments'])) {
            throw new Exception('Assessments table not available');
        }
        if (empty($data['patient_id']) || empty($data['assessed_by'])) {
            throw new Exception('Patient ID and assessed by staff ID are required');
        }
        
        // Get actual table fields to avoid field errors
        $table_fields = $this->db->list_fields($this->tables['assessments']);
        log_message('info', 'Assessment table fields: ' . json_encode($table_fields));
        
        // Filter data to only include fields that exist in the table
        $filtered_data = [];
        foreach ($data as $key => $value) {
            if (in_array($key, $table_fields)) {
                $filtered_data[$key] = $value;
            } else {
                log_message('info', "Skipping field '{$key}' - not found in table");
            }
        }
        
        // Handle field compatibility - map to existing fields
        if (isset($data['assessment_details']) && !in_array('assessment_details', $table_fields)) {
            if (in_array('findings', $table_fields)) {
                $filtered_data['findings'] = $data['assessment_details'];
                log_message('info', 'Mapped assessment_details to findings');
            } elseif (in_array('description', $table_fields)) {
                $filtered_data['description'] = $data['assessment_details'];
                log_message('info', 'Mapped assessment_details to description');
            }
        }
        
        // Remove assessment_category if it doesn't exist
        if (isset($filtered_data['assessment_category']) && !in_array('assessment_category', $table_fields)) {
            unset($filtered_data['assessment_category']);
            log_message('info', 'Removed assessment_category - field does not exist');
        }
        
        // Ensure required fields are present
        if (!isset($filtered_data['patient_id'])) {
            $filtered_data['patient_id'] = $data['patient_id'];
        }
        if (!isset($filtered_data['assessed_by'])) {
            $filtered_data['assessed_by'] = $data['assessed_by'];
        }
        
        log_message('info', 'Final assessment data: ' . json_encode($filtered_data));
        
        $this->db->insert($this->tables['assessments'], $filtered_data);
        $insert_id = $this->db->insert_id();
        
        if (!$insert_id) {
            $db_error = $this->db->error();
            log_message('error', 'Database insert failed: ' . json_encode($db_error));
            throw new Exception('Failed to insert assessment record. Query error: ' . $db_error['message'] . ' - ' . $this->db->last_query());
        }
        
        return $insert_id;
    } catch (Exception $e) {
        log_message('error', 'Add assessment error: ' . $e->getMessage());
        throw new Exception('Failed to add assessment: ' . $e->getMessage());
    }
}

    public function getAssessmentById($assessment_id) {
        try {
            if (!$this->db->table_exists($this->tables['assessments'])) {
                return null;
            }
            
            $this->db->select('a.*, p.patient_name, CONCAT(s.name, " ", s.surname) as assessed_by_name');
            $this->db->from($this->tables['assessments'] . ' a');
            $this->db->join('patients p', 'p.id = a.patient_id', 'left');
            $this->db->join('staff s', 's.id = a.assessed_by', 'left');
            $this->db->where('a.id', $assessment_id);
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                return $query->row_array();
            }
            return null;
        } catch (Exception $e) {
            log_message('error', 'Get assessment by ID error: ' . $e->getMessage());
            return null;
        }
    }

    public function addProcedure($data) {
        try {
            if (!$this->db->table_exists($this->tables['procedures'])) {
                throw new Exception('Procedures table not available');
            }
            if (empty($data['patient_id']) || empty($data['performed_by'])) {
                throw new Exception('Patient ID and performed by staff ID are required');
            }
            
            // Check what fields exist and filter data accordingly
            $fields = $this->db->list_fields($this->tables['procedures']);
            $filtered_data = [];
            
            foreach ($data as $key => $value) {
                if (in_array($key, $fields)) {
                    $filtered_data[$key] = $value;
                }
            }
            
            // Handle field compatibility
            if (isset($data['procedure_description']) && !isset($filtered_data['description'])) {
                $filtered_data['description'] = $data['procedure_description'];
            }
            if (isset($data['procedure_notes']) && !isset($filtered_data['post_procedure_notes'])) {
                $filtered_data['post_procedure_notes'] = $data['procedure_notes'];
            }
            
            if (isset($filtered_data['status']) && $filtered_data['status'] == 'completed' && empty($filtered_data['performed_at'])) {
                $filtered_data['performed_at'] = date('Y-m-d H:i:s');
            }
            
            $this->db->insert($this->tables['procedures'], $filtered_data);
            $insert_id = $this->db->insert_id();
            if (!$insert_id) {
                throw new Exception('Failed to insert procedure record');
            }
            return $insert_id;
        } catch (Exception $e) {
            log_message('error', 'Add procedure error: ' . $e->getMessage());
            throw new Exception('Failed to record procedure: ' . $e->getMessage());
        }
    }

    public function getProcedureById($procedure_id) {
        try {
            if (!$this->db->table_exists($this->tables['procedures'])) {
                return null;
				}
           
           $this->db->select('p.*, pat.patient_name, CONCAT(s.name, " ", s.surname) as performed_by_name');
           $this->db->from($this->tables['procedures'] . ' p');
           $this->db->join('patients pat', 'pat.id = p.patient_id', 'left');
           $this->db->join('staff s', 's.id = p.performed_by', 'left');
           $this->db->where('p.id', $procedure_id);
           $query = $this->db->get();
           
           if ($query && $query->num_rows() > 0) {
               return $query->row_array();
           }
           return null;
       } catch (Exception $e) {
           log_message('error', 'Get procedure by ID error: ' . $e->getMessage());
           return null;
       }
   }

public function addCarePlan($data) {
    try {
        if (!$this->db->table_exists($this->tables['care_plans'])) {
            throw new Exception('Care plans table not available');
        }
        if (empty($data['patient_id']) || empty($data['created_by'])) {
            throw new Exception('Patient ID and created by staff ID are required');
        }
        
        // Get actual table fields to avoid field errors
        $table_fields = $this->db->list_fields($this->tables['care_plans']);
        log_message('info', 'Care plan table fields: ' . json_encode($table_fields));
        
        // Filter data to only include fields that exist in the table
        $filtered_data = [];
        foreach ($data as $key => $value) {
            if (in_array($key, $table_fields)) {
                $filtered_data[$key] = $value;
            } else {
                log_message('info', "Skipping field '{$key}' - not found in care plan table");
            }
        }
        
        // Handle field compatibility - map to existing fields
        if (isset($data['nursing_diagnosis']) && !in_array('nursing_diagnosis', $table_fields)) {
            if (in_array('nursing_diagnoses', $table_fields)) {
                $filtered_data['nursing_diagnoses'] = $data['nursing_diagnosis'];
                log_message('info', 'Mapped nursing_diagnosis to nursing_diagnoses');
            }
        }
        
        if (isset($data['goals_outcomes']) && !in_array('goals_outcomes', $table_fields)) {
            if (in_array('goals_short_term', $table_fields)) {
                $filtered_data['goals_short_term'] = $data['goals_outcomes'];
                log_message('info', 'Mapped goals_outcomes to goals_short_term');
            } elseif (in_array('goals', $table_fields)) {
                $filtered_data['goals'] = $data['goals_outcomes'];
                log_message('info', 'Mapped goals_outcomes to goals');
            }
        }
        
        // Ensure required fields are present
        if (!isset($filtered_data['patient_id'])) {
            $filtered_data['patient_id'] = $data['patient_id'];
        }
        if (!isset($filtered_data['created_by'])) {
            $filtered_data['created_by'] = $data['created_by'];
        }
        
        log_message('info', 'Final care plan data: ' . json_encode($filtered_data));
        
        $this->db->insert($this->tables['care_plans'], $filtered_data);
        $insert_id = $this->db->insert_id();
        
        if (!$insert_id) {
            $db_error = $this->db->error();
            log_message('error', 'Database insert failed: ' . json_encode($db_error));
            throw new Exception('Failed to insert care plan. Query error: ' . $db_error['message'] . ' - ' . $this->db->last_query());
        }
        
        return $insert_id;
    } catch (Exception $e) {
        log_message('error', 'Add care plan error: ' . $e->getMessage());
        throw new Exception('Failed to create care plan: ' . $e->getMessage());
    }
}

   public function getCarePlanById($plan_id) {
       try {
           if (!$this->db->table_exists($this->tables['care_plans'])) {
               return null;
           }
           
           $this->db->select('cp.*, p.patient_name, CONCAT(s.name, " ", s.surname) as created_by_name');
           $this->db->from($this->tables['care_plans'] . ' cp');
           $this->db->join('patients p', 'p.id = cp.patient_id', 'left');
           $this->db->join('staff s', 's.id = cp.created_by', 'left');
           $this->db->where('cp.id', $plan_id);
           $query = $this->db->get();
           
           if ($query && $query->num_rows() > 0) {
               return $query->row_array();
           }
           return null;
       } catch (Exception $e) {
           log_message('error', 'Get care plan by ID error: ' . $e->getMessage());
           return null;
       }
   }

   public function getMedicationById($medication_id) {
       try {
           if (!$this->db->table_exists('medication_report')) {
               return null;
           }
           
           // Check what fields exist in medication_report
           $med_fields = $this->db->list_fields('medication_report');
           $join_staff_field = 'generated_by'; // Default
           
           if (in_array('given_by', $med_fields)) {
               $join_staff_field = 'given_by';
           } elseif (in_array('administered_by', $med_fields)) {
               $join_staff_field = 'administered_by';
           }
           
           $this->db->select('mr.*, p.patient_name, CONCAT(s.name, " ", s.surname) as given_by_name');
           $this->db->from('medication_report mr');
           $this->db->join('ipd_details ipd', 'ipd.id = mr.ipd_id', 'left');
           $this->db->join('patients p', 'p.id = ipd.patient_id', 'left');
           $this->db->join('staff s', "s.id = mr.{$join_staff_field}", 'left');
           $this->db->where('mr.id', $medication_id);
           $query = $this->db->get();
           
           if ($query && $query->num_rows() > 0) {
               return $query->row_array();
           }
           return null;
       } catch (Exception $e) {
           log_message('error', 'Get medication by ID error: ' . $e->getMessage());
           return null;
       }
   }

   public function getPatientProcedures($patient_id, $limit = 50) {
       try {
           if (!$this->db->table_exists($this->tables['procedures'])) {
               return [];
           }

           // Check what fields exist in procedures table
           $fields = $this->db->list_fields($this->tables['procedures']);
           
           // Build select statement based on available fields
           $select_fields = ['p.*'];
           
           // Add staff name if staff table exists
           if ($this->db->table_exists('staff')) {
               $select_fields[] = 's.name as performed_by_name';
               $select_fields[] = 's.surname as performed_by_surname';
           }
           
           // Only add template name if template_id field exists and templates table exists
           $join_templates = in_array('template_id', $fields) && $this->db->table_exists($this->tables['templates']);
           if ($join_templates) {
               $select_fields[] = 't.procedure_name as template_name';
           }
           
           $this->db->select(implode(', ', $select_fields));
           $this->db->from($this->tables['procedures'] . ' p');
           
           // Join staff table if it exists
           if ($this->db->table_exists('staff')) {
               $this->db->join('staff s', 's.id = p.performed_by', 'left');
           }
           
           // Only join templates if template_id field exists
           if ($join_templates) {
               $this->db->join($this->tables['templates'] . ' t', 't.id = p.template_id', 'left');
           }
           
           $this->db->where('p.patient_id', $patient_id);
           $this->db->order_by('p.created_at', 'DESC');
           $this->db->limit($limit);
           
           $query = $this->db->get();
           
           // Check if query was successful
           if (!$query) {
               log_message('error', 'Database query failed in getPatientProcedures');
               return [];
           }
           
           $procedures = $query->result_array();
           
           // Add missing fields with default values
           foreach ($procedures as &$procedure) {
               if (!isset($procedure['performed_by_name'])) {
                   $procedure['performed_by_name'] = 'Unknown';
               }
               if (!isset($procedure['performed_by_surname'])) {
                   $procedure['performed_by_surname'] = '';
               }
               if (!isset($procedure['template_name'])) {
                   $procedure['template_name'] = $procedure['procedure_name'] ?? 'Unknown Procedure';
               }
           }
           
           return $procedures;
           
       } catch (Exception $e) {
           log_message('error', 'Get procedures error: ' . $e->getMessage());
           return [];
       }
   }

   public function getProcedureTemplates($type = null) {
       try {
           if (!$this->db->table_exists($this->tables['templates'])) {
               return [];
           }
           $this->db->select('*');
           $this->db->from($this->tables['templates']);
           $this->db->where('is_active', 1);
           if ($type) {
               $this->db->where('procedure_type', $type);
           }
           $this->db->order_by('procedure_type, procedure_name', 'ASC');
           $query = $this->db->get();
           
           if (!$query) {
               return [];
           }
           
           return $query->result_array();
       } catch (Exception $e) {
           log_message('error', 'Get templates error: ' . $e->getMessage());
           return [];
       }
   }

   public function getPatientAssessments($patient_id, $limit = 20) {
       try {
           if (!$this->db->table_exists($this->tables['assessments'])) {
               return [];
           }
           $this->db->select('
               a.*, s.name as assessed_by_name, s.surname as assessed_by_surname
           ');
           $this->db->from($this->tables['assessments'] . ' a');
           $this->db->join('staff s', 's.id = a.assessed_by', 'left');
           $this->db->where('a.patient_id', $patient_id);
           $this->db->order_by('a.assessment_date', 'DESC');
           $this->db->limit($limit);
           $query = $this->db->get();
           return $query->result_array();
       } catch (Exception $e) {
           log_message('error', 'Get assessments error: ' . $e->getMessage());
           return [];
       }
   }

   public function getPatientCarePlans($patient_id, $limit = 20) {
       try {
           if (!$this->db->table_exists($this->tables['care_plans'])) {
               return [];
           }
           $this->db->select('
               cp.*, s1.name as created_by_name, s1.surname as created_by_surname,
               s2.name as assigned_to_name, s2.surname as assigned_to_surname
           ');
           $this->db->from($this->tables['care_plans'] . ' cp');
           $this->db->join('staff s1', 's1.id = cp.created_by', 'left');
           $this->db->join('staff s2', 's2.id = cp.assigned_to', 'left');
           $this->db->where('cp.patient_id', $patient_id);
           $this->db->where('cp.status !=', 'discontinued');
           $this->db->order_by('cp.priority', 'DESC');
           $this->db->order_by('cp.created_at', 'DESC');
           $this->db->limit($limit);
           $query = $this->db->get();
           return $query->result_array();
       } catch (Exception $e) {
           log_message('error', 'Get care plans error: ' . $e->getMessage());
           return [];
       }
   }

   // ALIAS METHODS FOR BACKWARD COMPATIBILITY
   
   public function createCarePlan($data) {
       return $this->addCarePlan($data);
   }

   public function getDailyNursingStats($date = null) {
       if (!$date) $date = date('Y-m-d');
       try {
           $stats = [
               'vital_signs' => 0,
               'procedures' => 0,
               'nursing_notes' => 0,
               'assessments' => 0
           ];
           if ($this->db->table_exists($this->tables['vital_signs'])) {
               $this->db->select('COUNT(*) as count');
               $this->db->from($this->tables['vital_signs']);
               $this->db->where('DATE(recorded_at)', $date);
               $query = $this->db->get();
               $stats['vital_signs'] = intval($query->row()->count ?? 0);
           }
           if ($this->db->table_exists($this->tables['procedures'])) {
               $this->db->select('COUNT(*) as count');
               $this->db->from($this->tables['procedures']);
               $this->db->where('DATE(performed_at)', $date);
               $this->db->where('status', 'completed');
               $query = $this->db->get();
               $stats['procedures'] = intval($query->row()->count ?? 0);
           }
           if ($this->db->table_exists('nurse_note')) {
               $this->db->select('COUNT(*) as count');
               $this->db->from('nurse_note');
               $this->db->where('DATE(date)', $date);
               $query = $this->db->get();
               $stats['nursing_notes'] = intval($query->row()->count ?? 0);
           }
           if ($this->db->table_exists($this->tables['assessments'])) {
               $this->db->select('COUNT(*) as count');
               $this->db->from($this->tables['assessments']);
               $this->db->where('DATE(assessment_date)', $date);
               $query = $this->db->get();
               $stats['assessments'] = intval($query->row()->count ?? 0);
           }
           return $stats;
       } catch (Exception $e) {
           log_message('error', 'Get daily nursing stats error: ' . $e->getMessage());
           return ['vital_signs' => 0, 'procedures' => 0, 'nursing_notes' => 0, 'assessments' => 0];
       }
   }

public function getProcedureStats() {
    try {
        if (!$this->db->table_exists($this->tables['procedures'])) {
            return [];
        }
        
        $this->db->select('procedure_type, status, COUNT(*) as count');
        $this->db->from($this->tables['procedures']);
        $this->db->where('DATE(created_at) >=', date('Y-m-d', strtotime('-30 days')));
        $this->db->group_by(['procedure_type', 'status']); // Use array for multiple group by
        $this->db->order_by('procedure_type, status');
        
        $query = $this->db->get();
        return $query->result_array();
    } catch (Exception $e) {
        log_message('error', 'Get procedure stats error: ' . $e->getMessage());
        return [];
    }
}

   public function getStaffWorkloadStats() {
       try {
           $stats = [
               'procedures' => [],
               'nursing_notes' => [],
               'assessments' => []
           ];
           if ($this->db->table_exists($this->tables['procedures'])) {
               $this->db->select('s.name, s.surname, COUNT(*) as procedure_count');
               $this->db->from($this->tables['procedures'] . ' p');
               $this->db->join('staff s', 's.id = p.performed_by');
               $this->db->where('DATE(p.created_at) >=', date('Y-m-d', strtotime('-7 days')));
               $this->db->group_by('s.id');
               $this->db->group_by('s.name');
               $this->db->group_by('s.surname');
               $this->db->order_by('procedure_count', 'DESC');
               $this->db->limit(10);
               $query = $this->db->get();
               $stats['procedures'] = $query->result_array();
           }
           if ($this->db->table_exists('nurse_note')) {
               $this->db->select('s.name, s.surname, COUNT(*) as note_count');
               $this->db->from('nurse_note nn');
               $this->db->join('staff s', 's.id = nn.staff_id');
               $this->db->where('DATE(nn.date) >=', date('Y-m-d', strtotime('-7 days')));
               $this->db->group_by('s.id');
               $this->db->group_by('s.name');
               $this->db->group_by('s.surname');
               $this->db->order_by('note_count', 'DESC');
               $this->db->limit(10);
               $query = $this->db->get();
               $stats['nursing_notes'] = $query->result_array();
           }
           if ($this->db->table_exists($this->tables['assessments'])) {
               $this->db->select('s.name, s.surname, COUNT(*) as assessment_count');
               $this->db->from($this->tables['assessments'] . ' a');
               $this->db->join('staff s', 's.id = a.assessed_by');
               $this->db->where('DATE(a.assessment_date) >=', date('Y-m-d', strtotime('-7 days')));
               $this->db->group_by('s.id');
               $this->db->group_by('s.name');
               $this->db->group_by('s.surname');
               $this->db->order_by('assessment_count', 'DESC');
               $this->db->limit(10);
               $query = $this->db->get();
               $stats['assessments'] = $query->result_array();
           }
           return $stats;
       } catch (Exception $e) {
           log_message('error', 'Get staff workload stats error: ' . $e->getMessage());
           return ['procedures' => [], 'nursing_notes' => [], 'assessments' => []];
       }
   }

   public function isSystemReady() {
       try {
           $required_tables = [$this->tables['vital_signs'], $this->tables['procedures']];
           foreach ($required_tables as $table) {
               if (!$this->db->table_exists($table)) {
                   return false;
               }
           }
           return true;
       } catch (Exception $e) {
           log_message('error', 'System ready check error: ' . $e->getMessage());
           return false;
       }
   }

   // Additional helper methods for database compatibility

   public function checkTableStructure($table_name) {
       try {
           if (!$this->db->table_exists($table_name)) {
               return [
                   'exists' => false,
                   'fields' => [],
                   'sample_data' => null
               ];
           }

           $fields = $this->db->list_fields($table_name);
           
           // Get sample data
           $this->db->limit(1);
           $query = $this->db->get($table_name);
           $sample_data = $query->num_rows() > 0 ? $query->row_array() : null;

           return [
               'exists' => true,
               'fields' => $fields,
               'sample_data' => $sample_data
           ];

       } catch (Exception $e) {
           log_message('error', 'Check table structure error: ' . $e->getMessage());
           return [
               'exists' => false,
               'fields' => [],
               'sample_data' => null,
               'error' => $e->getMessage()
           ];
       }
   }

   public function getCompatibleFieldName($table_name, $field_variations) {
       try {
           if (!$this->db->table_exists($table_name)) {
               return null;
           }

           $fields = $this->db->list_fields($table_name);
           
           foreach ($field_variations as $variation) {
               if (in_array($variation, $fields)) {
                   return $variation;
               }
           }

           return null;

       } catch (Exception $e) {
           log_message('error', 'Get compatible field name error: ' . $e->getMessage());
           return null;
       }
   }

   public function safeQuery($table_name, $select_fields, $joins = [], $conditions = [], $order_by = null, $limit = null) {
       try {
           if (!$this->db->table_exists($table_name)) {
               return [];
           }

           // Verify all fields exist before using them
           $table_fields = $this->db->list_fields($table_name);
           $safe_select = [];

           foreach ($select_fields as $field) {
               // Handle aliased fields (e.g., "table.field as alias")
               if (strpos($field, ' as ') !== false) {
                   $safe_select[] = $field; // Trust aliased fields for now
               } else {
                   // Check if field exists in main table
                   $field_name = str_replace($table_name . '.', '', $field);
                   if (in_array($field_name, $table_fields)) {
                       $safe_select[] = $field;
                   }
               }
           }

           if (empty($safe_select)) {
               $safe_select = [$table_name . '.*'];
           }

           $this->db->select(implode(', ', $safe_select));
           $this->db->from($table_name);

           // Apply joins
           foreach ($joins as $join) {
               if (isset($join['table']) && isset($join['on']) && isset($join['type'])) {
                   if ($this->db->table_exists($join['table'])) {
                       $this->db->join($join['table'], $join['on'], $join['type']);
                   }
               }
           }

           // Apply conditions
           foreach ($conditions as $condition) {
               if (isset($condition['field']) && isset($condition['value'])) {
                   $method = $condition['method'] ?? 'where';
                   $this->db->$method($condition['field'], $condition['value']);
               }
           }

           if ($order_by) {
               $this->db->order_by($order_by);
           }

           if ($limit) {
               $this->db->limit($limit);
           }

           $query = $this->db->get();
           return $query ? $query->result_array() : [];

       } catch (Exception $e) {
           log_message('error', 'Safe query error: ' . $e->getMessage());
           return [];
       }
   }

   public function diagnoseTableIssues() {
       $diagnosis = [];
       
       foreach ($this->tables as $key => $table_name) {
           $structure = $this->checkTableStructure($table_name);
           $diagnosis[$key] = [
               'table_name' => $table_name,
               'exists' => $structure['exists'],
               'field_count' => count($structure['fields']),
               'fields' => $structure['fields'],
               'has_sample_data' => !empty($structure['sample_data']),
               'issues' => []
           ];

           // Check for common issues
           if ($key === 'procedures' && $structure['exists']) {
               if (!in_array('template_id', $structure['fields'])) {
                   $diagnosis[$key]['issues'][] = 'Missing template_id field';
               }
           }

           if ($key === 'vital_signs' && $structure['exists']) {
               if (!in_array('recorded_by', $structure['fields'])) {
                   $diagnosis[$key]['issues'][] = 'Missing recorded_by field';
               }
           }
       }

       // Check external tables
       $external_tables = ['nurse_note', 'staff', 'patients', 'ipd_details', 'medication_report'];
       foreach ($external_tables as $table) {
           $structure = $this->checkTableStructure($table);
           $diagnosis['external_' . $table] = [
               'table_name' => $table,
               'exists' => $structure['exists'],
               'field_count' => count($structure['fields']),
               'fields' => $structure['fields']
           ];
       }

       return $diagnosis;
   }

   // DEBUG AND TESTING METHODS

   public function testVitalSignsInsert() {
       try {
           $test_data = [
               'patient_id' => 1,
               'temperature' => 37.5,
               'systolic_bp' => 120,
               'diastolic_bp' => 80,
               'pulse_rate' => 72,
               'respiratory_rate' => 16,
               'recorded_by' => 1,
               'recorded_at' => date('Y-m-d H:i:s')
           ];
           
           return $this->addVitalSigns($test_data);
       } catch (Exception $e) {
           log_message('error', 'Test vital signs insert error: ' . $e->getMessage());
           throw $e;
       }
   }

   public function testNursingNoteInsert() {
       try {
           $test_data = [
               'ipd_id' => 1,
               'note' => 'Test nursing note from model',
               'comment' => 'Test comment',
               'staff_id' => 1,
               'date' => date('Y-m-d H:i:s')
           ];
           
           return $this->addNurseNote($test_data);
       } catch (Exception $e) {
           log_message('error', 'Test nursing note insert error: ' . $e->getMessage());
           throw $e;
       }
   }

   public function getSystemStatus() {
       try {
           $status = [
               'system_ready' => $this->isSystemReady(),
               'tables' => [],
               'sample_counts' => [],
               'last_activities' => []
           ];

           // Check each table
           foreach ($this->tables as $key => $table_name) {
               $status['tables'][$key] = [
                   'name' => $table_name,
                   'exists' => $this->db->table_exists($table_name),
                   'fields' => $this->db->table_exists($table_name) ? count($this->db->list_fields($table_name)) : 0
               ];

               // Get record counts
               if ($this->db->table_exists($table_name)) {
                   $this->db->select('COUNT(*) as count');
                   $this->db->from($table_name);
                   $query = $this->db->get();
                   $status['sample_counts'][$key] = $query->row()->count ?? 0;
               }
           }

           // Check external tables
           $external_tables = ['nurse_note', 'staff', 'patients', 'ipd_details'];
           foreach ($external_tables as $table) {
               $status['tables']['external_' . $table] = [
                   'name' => $table,
                   'exists' => $this->db->table_exists($table),
                   'fields' => $this->db->table_exists($table) ? count($this->db->list_fields($table)) : 0
               ];
           }

           return $status;

       } catch (Exception $e) {
           log_message('error', 'Get system status error: ' . $e->getMessage());
           return [
               'system_ready' => false,
               'error' => $e->getMessage()
           ];
       }
   }

}
?>