<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Marketing Model
 * Handles all database operations for marketing analytics
 */
class Marketing_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get KPI metrics for dashboard
     */
    public function get_kpi_metrics($date_from, $date_to) {
        $kpi = [];
        
        // New patients acquired
        $this->db->where('created_at >=', $date_from);
        $this->db->where('created_at <=', $date_to . ' 23:59:59');
        $kpi['new_patients'] = $this->db->count_all_results('patients');
        
        // Total revenue
        $this->db->select('SUM(amount) as total');
        $this->db->where('payment_date >=', $date_from);
        $this->db->where('payment_date <=', $date_to);
        $this->db->where('type', 'payment');
        $revenue = $this->db->get('transactions')->row();
        $kpi['total_revenue'] = $revenue->total ?: 0;
        
        // Average patient value
        $kpi['avg_patient_value'] = $kpi['new_patients'] > 0 ? 
            round($kpi['total_revenue'] / $kpi['new_patients'], 2) : 0;
        
        // Conversion rate (inquiries to patients)
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $inquiries = $this->db->count_all_results('patient_inquiries');
        $kpi['conversion_rate'] = $inquiries > 0 ? 
            round(($kpi['new_patients'] / $inquiries) * 100, 2) : 0;
        
        // Patient retention rate
        $kpi['retention_rate'] = $this->calculate_retention_rate($date_from, $date_to);
        
        // Service utilization rate
        $kpi['utilization_rate'] = $this->calculate_utilization_rate($date_from, $date_to);
        
        return $kpi;
    }

    /**
     * Get patient acquisition sources
     */
    public function get_acquisition_sources($date_from, $date_to) {
        $this->db->select('referral_source, COUNT(*) as count, 
                          AVG(DATEDIFF(NOW(), created_at)) as avg_days_since_acquisition');
        $this->db->from('patients');
        $this->db->where('created_at >=', $date_from);
        $this->db->where('created_at <=', $date_to . ' 23:59:59');
        $this->db->group_by('referral_source');
        $this->db->order_by('count', 'DESC');
        
        $sources = $this->db->get()->result();
        
        // Calculate revenue per source
        foreach ($sources as &$source) {
            $source->revenue = $this->get_revenue_by_source($source->referral_source, $date_from, $date_to);
            $source->percentage = 0; // Will be calculated in view
        }
        
        return $sources;
    }

    /**
     * Get revenue by department
     */
    public function get_revenue_by_department($date_from, $date_to) {
        $departments = [];
        
        // OPD Revenue
        $this->db->select('SUM(paid_amount) as total');
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $opd = $this->db->get('opd_billing')->row();
        $departments['OPD'] = $opd->total ?: 0;
        
        // IPD Revenue
        $this->db->select('SUM(paid_amount) as total');
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $ipd = $this->db->get('ipd_billing')->row();
        $departments['IPD'] = $ipd->total ?: 0;
        
        // Pathology Revenue
        $this->db->select('SUM(net_amount) as total');
        $this->db->where('reporting_date >=', $date_from);
        $this->db->where('reporting_date <=', $date_to);
        $pathology = $this->db->get('pathology_billing')->row();
        $departments['Laboratory'] = $pathology->total ?: 0;
        
        // Radiology Revenue
        $this->db->select('SUM(net_amount) as total');
        $this->db->where('reporting_date >=', $date_from);
        $this->db->where('reporting_date <=', $date_to);
        $radiology = $this->db->get('radiology_billing')->row();
        $departments['Radiology'] = $radiology->total ?: 0;
        
        // Pharmacy Revenue
        $this->db->select('SUM(net_amount) as total');
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $pharmacy = $this->db->get('pharmacy_bill_basic')->row();
        $departments['Pharmacy'] = $pharmacy->total ?: 0;
        
        return $departments;
    }

    /**
     * Get active marketing campaigns
     */
    public function get_all_campaigns() {
        $this->db->select('*');
        $this->db->from('marketing_campaigns');
        $this->db->order_by('created_at', 'DESC');
        return $this->db->get()->result();
    }

    /**
     * Get campaign metrics
     */
    public function get_campaign_metrics($campaign_id) {
        $metrics = new stdClass();
        
        // Get campaign details
        $campaign = $this->db->get_where('marketing_campaigns', ['id' => $campaign_id])->row();
        
        if (!$campaign) {
            return $metrics;
        }
        
        // Impressions/Reach
        $this->db->where('campaign_id', $campaign_id);
        $metrics->impressions = $this->db->count_all_results('campaign_impressions');
        
        // Conversions
        $this->db->where('campaign_id', $campaign_id);
        $metrics->conversions = $this->db->count_all_results('campaign_conversions');
        
        // Conversion rate
        $metrics->conversion_rate = $metrics->impressions > 0 ? 
            round(($metrics->conversions / $metrics->impressions) * 100, 2) : 0;
        
        // Cost per acquisition
        $metrics->cpa = $metrics->conversions > 0 && $campaign->budget > 0 ? 
            round($campaign->budget / $metrics->conversions, 2) : 0;
        
        // ROI
        $revenue = $this->get_campaign_revenue($campaign_id);
        $metrics->roi = $campaign->budget > 0 ? 
            round((($revenue - $campaign->budget) / $campaign->budget) * 100, 2) : 0;
        
        return $metrics;
    }

    /**
     * Create new campaign
     */
    public function create_campaign($data) {
        $this->db->insert('marketing_campaigns', $data);
        return $this->db->insert_id();
    }

    /**
     * Get patient demographics
     */
    public function get_patient_demographics() {
        $demographics = [];
        
        // Age distribution
        $this->db->select("
            CASE 
                WHEN TIMESTAMPDIFF(YEAR, dob, CURDATE()) < 18 THEN '0-17'
                WHEN TIMESTAMPDIFF(YEAR, dob, CURDATE()) BETWEEN 18 AND 30 THEN '18-30'
                WHEN TIMESTAMPDIFF(YEAR, dob, CURDATE()) BETWEEN 31 AND 45 THEN '31-45'
                WHEN TIMESTAMPDIFF(YEAR, dob, CURDATE()) BETWEEN 46 AND 60 THEN '46-60'
                ELSE '60+' 
            END as age_group,
            COUNT(*) as count
        ");
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->group_by('age_group');
        $demographics['age_distribution'] = $this->db->get()->result();
        
        // Gender distribution
        $this->db->select('gender, COUNT(*) as count');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->group_by('gender');
        $demographics['gender_distribution'] = $this->db->get()->result();
        
        // Geographic distribution
        $this->db->select('city, COUNT(*) as count');
        $this->db->from('patients');
        $this->db->where('is_active', 'yes');
        $this->db->where('city !=', '');
        $this->db->group_by('city');
        $this->db->order_by('count', 'DESC');
        $this->db->limit(10);
        $demographics['geographic_distribution'] = $this->db->get()->result();
        
        return $demographics;
    }

    /**
     * Get referral statistics
     */
    public function get_referral_statistics($date_from, $date_to) {
        $stats = [];
        
        // Top referring doctors
        $this->db->select('r.name as referrer, COUNT(p.id) as referral_count');
        $this->db->from('patients p');
        $this->db->join('referral_person r', 'p.refference = r.id', 'left');
        $this->db->where('p.created_at >=', $date_from);
        $this->db->where('p.created_at <=', $date_to . ' 23:59:59');
        $this->db->where('r.name IS NOT NULL');
        $this->db->group_by('r.id');
        $this->db->order_by('referral_count', 'DESC');
        $this->db->limit(10);
        $stats['top_referrers'] = $this->db->get()->result();
        
        // Referral conversion rate
        $this->db->where('created_at >=', $date_from);
        $this->db->where('created_at <=', $date_to . ' 23:59:59');
        $this->db->where('refference IS NOT NULL');
        $referred = $this->db->count_all_results('patients');
        
        $this->db->where('created_at >=', $date_from);
        $this->db->where('created_at <=', $date_to . ' 23:59:59');
        $total = $this->db->count_all_results('patients');
        
        $stats['referral_rate'] = $total > 0 ? round(($referred / $total) * 100, 2) : 0;
        
        return $stats;
    }

    /**
     * Get service utilization
     */
    public function get_service_utilization($date_from, $date_to) {
        $utilization = [];
        
        // OPD visits
        $this->db->where('appointment_date >=', $date_from);
        $this->db->where('appointment_date <=', $date_to);
        $utilization['opd_visits'] = $this->db->count_all_results('opd_details');
        
        // IPD admissions
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $utilization['ipd_admissions'] = $this->db->count_all_results('ipd_details');
        
        // Lab tests
        $this->db->where('reporting_date >=', $date_from);
        $this->db->where('reporting_date <=', $date_to);
        $utilization['lab_tests'] = $this->db->count_all_results('pathology_report');
        
        // Radiology scans
        $this->db->where('reporting_date >=', $date_from);
        $this->db->where('reporting_date <=', $date_to);
        $utilization['radiology_scans'] = $this->db->count_all_results('radiology_report');
        
        // Pharmacy prescriptions
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $utilization['prescriptions'] = $this->db->count_all_results('pharmacy_bill_basic');
        
        return $utilization;
    }

    /**
     * Get monthly trends
     */
    public function get_monthly_trends() {
        $trends = [];
        
        // Get last 12 months
        for ($i = 11; $i >= 0; $i--) {
            $month_start = date('Y-m-01', strtotime("-$i months"));
            $month_end = date('Y-m-t', strtotime("-$i months"));
            $month_name = date('M Y', strtotime("-$i months"));
            
            // New patients
            $this->db->where('created_at >=', $month_start);
            $this->db->where('created_at <=', $month_end . ' 23:59:59');
            $new_patients = $this->db->count_all_results('patients');
            
            // Revenue
            $this->db->select('SUM(amount) as total');
            $this->db->where('payment_date >=', $month_start);
            $this->db->where('payment_date <=', $month_end);
            $this->db->where('type', 'payment');
            $revenue = $this->db->get('transactions')->row();
            
            $trends[] = [
                'month' => $month_name,
                'new_patients' => $new_patients,
                'revenue' => $revenue->total ?: 0
            ];
        }
        
        return $trends;
    }

    /**
     * Segment patients for targeted marketing
     */
    public function segment_patients() {
        $segments = [];
        
        // High-value patients
        $this->db->select('p.*, SUM(t.amount) as total_spent');
        $this->db->from('patients p');
        $this->db->join('transactions t', 'p.id = t.patient_id', 'left');
        $this->db->where('t.type', 'payment');
        $this->db->group_by('p.id');
        $this->db->having('total_spent >', 50000);
        $this->db->order_by('total_spent', 'DESC');
        $segments['high_value'] = $this->db->get()->result();
        
        // Frequent visitors
        $this->db->select('p.*, COUNT(o.id) as visit_count');
        $this->db->from('patients p');
        $this->db->join('opd_details o', 'p.id = o.patient_id', 'left');
        $this->db->where('o.appointment_date >= DATE_SUB(NOW(), INTERVAL 6 MONTH)');
        $this->db->group_by('p.id');
        $this->db->having('visit_count >', 5);
        $segments['frequent_visitors'] = $this->db->get()->result();
        
        // At-risk patients (haven't visited in 3+ months)
        $this->db->select('p.*, MAX(o.appointment_date) as last_visit');
        $this->db->from('patients p');
        $this->db->join('opd_details o', 'p.id = o.patient_id', 'left');
        $this->db->group_by('p.id');
        $this->db->having('last_visit < DATE_SUB(NOW(), INTERVAL 3 MONTH)');
        $segments['at_risk'] = $this->db->get()->result();
        
        // New patients (joined in last 30 days)
        $this->db->where('created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)');
        $segments['new_patients'] = $this->db->get('patients')->result();
        
        return $segments;
    }

    /**
     * Calculate patient lifetime value
     */
    public function calculate_patient_lifetime_value() {
        $this->db->select('
            AVG(total_revenue) as avg_ltv,
            MAX(total_revenue) as max_ltv,
            MIN(total_revenue) as min_ltv,
            AVG(visit_count) as avg_visits,
            AVG(months_active) as avg_retention_months
        ');
        $this->db->from('(
            SELECT 
                p.id,
                SUM(t.amount) as total_revenue,
                COUNT(DISTINCT o.id) as visit_count,
                TIMESTAMPDIFF(MONTH, p.created_at, NOW()) as months_active
            FROM patients p
            LEFT JOIN transactions t ON p.id = t.patient_id AND t.type = "payment"
            LEFT JOIN opd_details o ON p.id = o.patient_id
            WHERE p.is_active = "yes"
            GROUP BY p.id
        ) as patient_stats');
        
        return $this->db->get()->row();
    }

    /**
     * Analyze service profitability
     */
    public function analyze_service_profitability($date_from, $date_to) {
        $profitability = [];
        
        // Get revenue and costs for each service
        $services = [
            'consultation' => 'opd_billing',
            'laboratory' => 'pathology_billing',
            'radiology' => 'radiology_billing',
            'pharmacy' => 'pharmacy_bill_basic'
        ];
        
        foreach ($services as $service => $table) {
            $this->db->select('
                COUNT(*) as transaction_count,
                SUM(net_amount) as revenue,
                AVG(net_amount) as avg_transaction_value
            ');
            $this->db->from($table);
            $this->db->where('date >=', $date_from);
            $this->db->where('date <=', $date_to);
            
            $result = $this->db->get()->row();
            
            $profitability[$service] = [
                'revenue' => $result->revenue ?: 0,
                'transactions' => $result->transaction_count ?: 0,
                'avg_value' => $result->avg_transaction_value ?: 0,
                'margin' => $this->calculate_service_margin($service)
            ];
        }
        
        return $profitability;
    }

    /**
     * Get acquisition funnel data
     */
    public function get_acquisition_funnel($date_from, $date_to) {
        $funnel = [];
        
        // Website visitors (if tracking)
        $funnel['visitors'] = $this->get_website_visitors($date_from, $date_to);
        
        // Inquiries
        $this->db->where('date >=', $date_from);
        $this->db->where('date <=', $date_to);
        $funnel['inquiries'] = $this->db->count_all_results('patient_inquiries');
        
        // Appointments scheduled
        $this->db->where('appointment_date >=', $date_from);
        $this->db->where('appointment_date <=', $date_to);
        $funnel['appointments'] = $this->db->count_all_results('appointment');
        
        // New patients registered
        $this->db->where('created_at >=', $date_from);
        $this->db->where('created_at <=', $date_to . ' 23:59:59');
        $funnel['registrations'] = $this->db->count_all_results('patients');
        
        // First visit completed
        $this->db->select('COUNT(DISTINCT patient_id) as count');
        $this->db->from('opd_details');
        $this->db->where('appointment_date >=', $date_from);
        $this->db->where('appointment_date <=', $date_to);
        $result = $this->db->get()->row();
        $funnel['first_visits'] = $result->count;
        
        return $funnel;
    }

    /**
     * Calculate channel ROI
     */
    public function calculate_channel_roi($date_from, $date_to) {
        $channels = [];
        
        // Get all marketing channels
        $this->db->select('DISTINCT channels');
        $this->db->from('marketing_campaigns');
        $this->db->where('start_date >=', $date_from);
        $this->db->where('start_date <=', $date_to);
        $campaign_channels = $this->db->get()->result();
        
        foreach ($campaign_channels as $channel_data) {
            $channel_list = json_decode($channel_data->channels);
            
            foreach ($channel_list as $channel) {
                if (!isset($channels[$channel])) {
                    $channels[$channel] = [
                        'spend' => 0,
                        'revenue' => 0,
                        'conversions' => 0,
                        'roi' => 0
                    ];
                }
                
                // Calculate spend and revenue for this channel
                $channel_stats = $this->get_channel_performance($channel, $date_from, $date_to);
                $channels[$channel] = array_merge($channels[$channel], $channel_stats);
            }
        }
        
        return $channels;
    }

    /**
     * Generate comprehensive report
     */
    public function generate_report($type, $date_from, $date_to) {
        $report = [
            'metadata' => [
                'type' => $type,
                'date_from' => $date_from,
                'date_to' => $date_to,
                'generated_at' => date('Y-m-d H:i:s'),
                'generated_by' => $this->session->userdata('admin')['name']
            ]
        ];
        
        switch ($type) {
            case 'executive_summary':
                $report['kpi'] = $this->get_kpi_metrics($date_from, $date_to);
                $report['revenue'] = $this->get_revenue_by_department($date_from, $date_to);
                $report['trends'] = $this->get_monthly_trends();
                break;
                
            case 'acquisition_report':
                $report['sources'] = $this->get_acquisition_sources($date_from, $date_to);
                $report['funnel'] = $this->get_acquisition_funnel($date_from, $date_to);
                $report['demographics'] = $this->get_patient_demographics();
                break;
                
            case 'campaign_performance':
                $report['campaigns'] = $this->get_campaign_report($date_from, $date_to);
                $report['channel_roi'] = $this->calculate_channel_roi($date_from, $date_to);
                break;
                
            case 'revenue_analysis':
                $report['department_revenue'] = $this->get_revenue_by_department($date_from, $date_to);
                $report['service_profitability'] = $this->analyze_service_profitability($date_from, $date_to);
                $report['patient_ltv'] = $this->calculate_patient_lifetime_value();
                break;
        }
        
        return $report;
    }

    /**
     * Helper Methods
     */
    
    private function calculate_retention_rate($date_from, $date_to) {
        // Patients who visited in the period
        $this->db->select('COUNT(DISTINCT patient_id) as count');
        $this->db->from('opd_details');
        $this->db->where('appointment_date >=', $date_from);
        $this->db->where('appointment_date <=', $date_to);
        $current_patients = $this->db->get()->row()->count;
        
        // Patients who visited in previous period and current period
        $prev_date_from = date('Y-m-d', strtotime($date_from . ' -1 month'));
        $prev_date_to = date('Y-m-d', strtotime($date_to . ' -1 month'));
        
        $this->db->select('COUNT(DISTINCT o1.patient_id) as count');
        $this->db->from('opd_details o1');
        $this->db->join('opd_details o2', 'o1.patient_id = o2.patient_id', 'inner');
        $this->db->where('o1.appointment_date >=', $prev_date_from);
        $this->db->where('o1.appointment_date <=', $prev_date_to);
        $this->db->where('o2.appointment_date >=', $date_from);
        $this->db->where('o2.appointment_date <=', $date_to);
        $retained = $this->db->get()->row()->count;
        
        return $current_patients > 0 ? round(($retained / $current_patients) * 100, 2) : 0;
    }

    private function calculate_utilization_rate($date_from, $date_to) {
        // Implementation would depend on your capacity metrics
        return rand(65, 85); // Placeholder
    }

    private function get_revenue_by_source($source, $date_from, $date_to) {
        $this->db->select('SUM(t.amount) as total');
        $this->db->from('transactions t');
        $this->db->join('patients p', 't.patient_id = p.id');
        $this->db->where('p.referral_source', $source);
        $this->db->where('t.payment_date >=', $date_from);
        $this->db->where('t.payment_date <=', $date_to);
        $this->db->where('t.type', 'payment');
        $result = $this->db->get()->row();
        
        return $result->total ?: 0;
    }

    private function get_campaign_revenue($campaign_id) {
        $this->db->select('SUM(t.amount) as total');
        $this->db->from('campaign_conversions c');
        $this->db->join('transactions t', 'c.patient_id = t.patient_id');
        $this->db->where('c.campaign_id', $campaign_id);
        $this->db->where('t.type', 'payment');
        $result = $this->db->get()->row();
        
        return $result->total ?: 0;
    }

    private function calculate_service_margin($service) {
        // This would need actual cost data
        // Placeholder margins
        $margins = [
            'consultation' => 65,
            'laboratory' => 45,
            'radiology' => 40,
            'pharmacy' => 25
        ];
        
        return $margins[$service] ?? 0;
    }

    /**
     * Log marketing activity
     */
    public function log_activity($type, $reference_id = null, $data = []) {
        $log = [
            'activity_type' => $type,
            'reference_id' => $reference_id,
            'data' => json_encode($data),
            'user_id' => $this->session->userdata('admin')['id'],
            'ip_address' => $this->input->ip_address(),
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        return $this->db->insert('marketing_activity_log', $log);
    }
}