<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Lab_setup_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    // =============================================
    // DATABASE TABLE CREATION
    // =============================================

    public function createLabTables() {
        // Create lab_test_categories table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `lab_test_categories` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `category_name` varchar(100) NOT NULL,
                `description` text,
                `is_active` tinyint(1) DEFAULT 1,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `category_name` (`category_name`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        // Create lab_units table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `lab_units` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `unit_name` varchar(50) NOT NULL,
                `unit_symbol` varchar(10) NOT NULL,
                `description` text,
                `is_active` tinyint(1) DEFAULT 1,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `unit_name` (`unit_name`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        // Create lab_tests table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `lab_tests` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `test_name` varchar(150) NOT NULL,
                `short_name` varchar(50) NOT NULL,
                `category_id` int(11) NOT NULL,
                `unit_id` int(11) DEFAULT NULL,
                `cost` decimal(10,2) NOT NULL DEFAULT 0.00,
                `report_days` int(3) DEFAULT NULL,
                `method` varchar(100) DEFAULT NULL,
                `sample_type` varchar(100) DEFAULT NULL,
                `test_type` enum('quantitative','qualitative','descriptive') DEFAULT 'qualitative',
                `description` text,
                `is_active` tinyint(1) DEFAULT 1,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `test_name` (`test_name`),
                KEY `category_id` (`category_id`),
                KEY `unit_id` (`unit_id`),
                CONSTRAINT `lab_tests_category_fk` FOREIGN KEY (`category_id`) REFERENCES `lab_test_categories` (`id`) ON DELETE CASCADE,
                CONSTRAINT `lab_tests_unit_fk` FOREIGN KEY (`unit_id`) REFERENCES `lab_units` (`id`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        // Create lab_reference_ranges table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `lab_reference_ranges` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `test_id` int(11) NOT NULL,
                `min_value` decimal(15,6) DEFAULT NULL,
                `max_value` decimal(15,6) DEFAULT NULL,
                `gender` enum('male','female','both') DEFAULT 'both',
                `age_min` int(3) DEFAULT NULL,
                `age_max` int(3) DEFAULT NULL,
                `conditions` text,
                `interpretation` text,
                `is_active` tinyint(1) DEFAULT 1,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `test_id` (`test_id`),
                CONSTRAINT `lab_reference_ranges_test_fk` FOREIGN KEY (`test_id`) REFERENCES `lab_tests` (`id`) ON DELETE CASCADE
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        // Create laboratory table (if not exists)
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `laboratory` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `patient_id` int(11) NOT NULL,
                `test_name` varchar(150) NOT NULL,
                `test_result` text,
                `notes` text,
                `tested_by` int(11) DEFAULT NULL,
                `test_date` date DEFAULT NULL,
                `verification_status` enum('pending','verified','rejected') DEFAULT 'pending',
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `patient_id` (`patient_id`),
                KEY `tested_by` (`tested_by`),
                KEY `test_date` (`test_date`),
                KEY `verification_status` (`verification_status`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        // Create lab_test_parameters table (for complex tests with multiple parameters)
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `lab_test_parameters` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `test_id` int(11) NOT NULL,
                `parameter_name` varchar(100) NOT NULL,
                `unit_id` int(11) DEFAULT NULL,
                `reference_range` varchar(100) DEFAULT NULL,
                `display_order` int(3) DEFAULT 1,
                `is_active` tinyint(1) DEFAULT 1,
                `created_at` timestamp NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `test_id` (`test_id`),
                KEY `unit_id` (`unit_id`),
                CONSTRAINT `lab_test_parameters_test_fk` FOREIGN KEY (`test_id`) REFERENCES `lab_tests` (`id`) ON DELETE CASCADE,
                CONSTRAINT `lab_test_parameters_unit_fk` FOREIGN KEY (`unit_id`) REFERENCES `lab_units` (`id`) ON DELETE SET NULL
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
        ");

        return true;
    }

    // =============================================
    // DEFAULT DATA INSERTION
    // =============================================

    public function addDefaultCategories() {
        $categories = array(
            array('category_name' => 'Hematology', 'description' => 'Blood-related tests'),
            array('category_name' => 'Clinical Chemistry', 'description' => 'Chemical analysis tests'),
            array('category_name' => 'Microbiology', 'description' => 'Bacterial and viral tests'),
            array('category_name' => 'Immunology', 'description' => 'Immune system tests'),
            array('category_name' => 'Urine Analysis', 'description' => 'Urine-based tests'),
            array('category_name' => 'Serology', 'description' => 'Blood serum tests'),
            array('category_name' => 'Endocrinology', 'description' => 'Hormone tests'),
            array('category_name' => 'Cardiology', 'description' => 'Heart-related tests'),
            array('category_name' => 'Parasitology', 'description' => 'Parasite detection tests'),
            array('category_name' => 'Molecular Biology', 'description' => 'DNA/RNA tests')
        );

        foreach ($categories as $category) {
            // Check if category already exists
            $this->db->where('category_name', $category['category_name']);
            $existing = $this->db->get('lab_test_categories')->num_rows();
            
            if ($existing == 0) {
                $category['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_test_categories', $category);
            }
        }
    }

    public function addDefaultUnits() {
        $units = array(
            array('unit_name' => 'Grams per deciliter', 'unit_symbol' => 'g/dL'),
            array('unit_name' => 'Milligrams per deciliter', 'unit_symbol' => 'mg/dL'),
            array('unit_name' => 'Micrograms per deciliter', 'unit_symbol' => 'μg/dL'),
            array('unit_name' => 'International Units per liter', 'unit_symbol' => 'IU/L'),
            array('unit_name' => 'Units per liter', 'unit_symbol' => 'U/L'),
            array('unit_name' => 'Millimoles per liter', 'unit_symbol' => 'mmol/L'),
            array('unit_name' => 'Micromoles per liter', 'unit_symbol' => 'μmol/L'),
            array('unit_name' => 'Nanograms per milliliter', 'unit_symbol' => 'ng/mL'),
            array('unit_name' => 'Picograms per milliliter', 'unit_symbol' => 'pg/mL'),
            array('unit_name' => 'Cells per microliter', 'unit_symbol' => 'cells/μL'),
            array('unit_name' => 'Percentage', 'unit_symbol' => '%'),
            array('unit_name' => 'Ratio', 'unit_symbol' => 'ratio'),
            array('unit_name' => 'Milliseconds', 'unit_symbol' => 'ms'),
            array('unit_name' => 'Millimeters per hour', 'unit_symbol' => 'mm/hr'),
            array('unit_name' => 'Count', 'unit_symbol' => 'count')
        );

        foreach ($units as $unit) {
            // Check if unit already exists
            $this->db->where('unit_name', $unit['unit_name']);
            $existing = $this->db->get('lab_units')->num_rows();
            
            if ($existing == 0) {
                $unit['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_units', $unit);
            }
        }
    }

    public function addDefaultTests() {
        // Get category and unit IDs
        $hematology_id = $this->getCategoryIdByName('Hematology');
        $chemistry_id = $this->getCategoryIdByName('Clinical Chemistry');
        $urine_id = $this->getCategoryIdByName('Urine Analysis');
        $serology_id = $this->getCategoryIdByName('Serology');
        
        $gdl_unit = $this->getUnitIdBySymbol('g/dL');
        $mgdl_unit = $this->getUnitIdBySymbol('mg/dL');
        $percent_unit = $this->getUnitIdBySymbol('%');
        $count_unit = $this->getUnitIdBySymbol('count');

        $tests = array(
            // Hematology Tests
            array(
                'test_name' => 'Complete Blood Count (CBC)',
                'short_name' => 'CBC',
                'category_id' => $hematology_id,
                'unit_id' => $count_unit,
                'cost' => 25.00,
                'report_days' => 1,
                'method' => 'Automated cell counter',
                'sample_type' => 'EDTA Blood',
                'test_type' => 'quantitative'
            ),
            array(
                'test_name' => 'Hemoglobin',
                'short_name' => 'Hb',
                'category_id' => $hematology_id,
                'unit_id' => $gdl_unit,
                'cost' => 10.00,
                'report_days' => 1,
                'method' => 'Spectrophotometry',
                'sample_type' => 'EDTA Blood',
                'test_type' => 'quantitative'
            ),
            array(
                'test_name' => 'Hematocrit',
                'short_name' => 'Hct',
                'category_id' => $hematology_id,
                'unit_id' => $percent_unit,
                'cost' => 8.00,
                'report_days' => 1,
                'method' => 'Microhematocrit',
                'sample_type' => 'EDTA Blood',
                'test_type' => 'quantitative'
            ),
            
            // Clinical Chemistry Tests
            array(
                'test_name' => 'Blood Glucose (Fasting)',
                'short_name' => 'FBS',
                'category_id' => $chemistry_id,
                'unit_id' => $mgdl_unit,
                'cost' => 15.00,
                'report_days' => 1,
                'method' => 'Enzymatic',
                'sample_type' => 'Serum',
                'test_type' => 'quantitative'
            ),
            array(
                'test_name' => 'Blood Glucose (Random)',
                'short_name' => 'RBS',
                'category_id' => $chemistry_id,
                'unit_id' => $mgdl_unit,
                'cost' => 12.00,
                'report_days' => 1,
                'method' => 'Enzymatic',
                'sample_type' => 'Serum',
                'test_type' => 'quantitative'
            ),
            array(
                'test_name' => 'Total Cholesterol',
                'short_name' => 'CHOL',
                'category_id' => $chemistry_id,
                'unit_id' => $mgdl_unit,
                'cost' => 20.00,
                'report_days' => 1,
                'method' => 'Enzymatic',
                'sample_type' => 'Serum',
                'test_type' => 'quantitative'
            ),
            
            // Urine Analysis
            array(
                'test_name' => 'Urinalysis (Complete)',
                'short_name' => 'UA',
                'category_id' => $urine_id,
                'unit_id' => null,
                'cost' => 18.00,
                'report_days' => 1,
                'method' => 'Microscopy and Dipstick',
                'sample_type' => 'Fresh Urine',
                'test_type' => 'descriptive'
            ),
            
            // Serology Tests
            array(
                'test_name' => 'Hepatitis B Surface Antigen',
                'short_name' => 'HBsAg',
                'category_id' => $serology_id,
                'unit_id' => null,
                'cost' => 35.00,
                'report_days' => 2,
                'method' => 'ELISA',
                'sample_type' => 'Serum',
                'test_type' => 'qualitative'
            ),
            array(
                'test_name' => 'HIV Screening',
                'short_name' => 'HIV',
                'category_id' => $serology_id,
                'unit_id' => null,
                'cost' => 30.00,
                'report_days' => 2,
                'method' => 'Rapid Test/ELISA',
                'sample_type' => 'Serum',
                'test_type' => 'qualitative'
            )
        );

        foreach ($tests as $test) {
            // Check if test already exists
            $this->db->where('test_name', $test['test_name']);
            $existing = $this->db->get('lab_tests')->num_rows();
            
            if ($existing == 0) {
                $test['created_at'] = date('Y-m-d H:i:s');
                $this->db->insert('lab_tests', $test);
            }
        }
    }

    // =============================================
    // CATEGORY MANAGEMENT
    // =============================================

    public function getTestCategories($active_only = true) {
        if ($active_only) {
            $this->db->where('is_active', 1);
        }
        $this->db->order_by('category_name', 'ASC');
        $query = $this->db->get('lab_test_categories');
        return $query->result_array();
    }

    public function getTestCategory($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('lab_test_categories');
        return $query->row_array();
    }

    public function addTestCategory($data) {
        return $this->db->insert('lab_test_categories', $data);
    }

    public function updateTestCategory($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('lab_test_categories', $data);
    }

    public function deleteTestCategory($id) {
        $this->db->where('id', $id);
        return $this->db->delete('lab_test_categories');
    }

    public function checkCategoryExists($category_name, $exclude_id = null) {
        $this->db->where('category_name', $category_name);
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        $query = $this->db->get('lab_test_categories');
        return $query->num_rows() > 0;
    }

    public function getTestsByCategoryCount($category_id) {
        $this->db->where('category_id', $category_id);
        return $this->db->count_all_results('lab_tests');
    }

    private function getCategoryIdByName($name) {
        $this->db->select('id');
        $this->db->where('category_name', $name);
        $query = $this->db->get('lab_test_categories');
        $result = $query->row_array();
        return $result ? $result['id'] : null;
    }

    // =============================================
    // UNITS MANAGEMENT
    // =============================================

    public function getUnits($active_only = true) {
        if ($active_only) {
            $this->db->where('is_active', 1);
        }
        $this->db->order_by('unit_name', 'ASC');
        $query = $this->db->get('lab_units');
        return $query->result_array();
    }

    public function getUnit($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('lab_units');
        return $query->row_array();
    }

    public function addUnit($data) {
        return $this->db->insert('lab_units', $data);
    }

    public function updateUnit($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('lab_units', $data);
    }

    public function deleteUnit($id) {
        $this->db->where('id', $id);
        return $this->db->delete('lab_units');
    }

    private function getUnitIdBySymbol($symbol) {
        $this->db->select('id');
        $this->db->where('unit_symbol', $symbol);
        $query = $this->db->get('lab_units');
        $result = $query->row_array();
        return $result ? $result['id'] : null;
    }

    // =============================================
    // TESTS MANAGEMENT
    // =============================================

    public function getTestsForDataTable($start = 0, $length = 10, $search = '') {
        $this->db->select('lt.*, ltc.category_name, lu.unit_name');
        $this->db->from('lab_tests lt');
        $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
        $this->db->join('lab_units lu', 'lu.id = lt.unit_id', 'left');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('lt.test_name', $search);
            $this->db->or_like('lt.short_name', $search);
            $this->db->or_like('ltc.category_name', $search);
            $this->db->or_like('lt.sample_type', $search);
            $this->db->group_end();
        }
        
        $this->db->order_by('lt.test_name', 'ASC');
        $this->db->limit($length, $start);
        
        $query = $this->db->get();
        return $query->result_array();
    }

    public function getTestsCount($search = '') {
        $this->db->from('lab_tests lt');
        $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
        
        if (!empty($search)) {
            $this->db->group_start();
            $this->db->like('lt.test_name', $search);
            $this->db->or_like('lt.short_name', $search);
            $this->db->or_like('ltc.category_name', $search);
            $this->db->or_like('lt.sample_type', $search);
            $this->db->group_end();
        }
        
        return $this->db->count_all_results();
    }

    public function getTest($id) {
        $this->db->where('id', $id);
        $query = $this->db->get('lab_tests');
        return $query->row_array();
    }

    public function getTestDetails($id) {
        $this->db->select('lt.*, ltc.category_name, lu.unit_name, lu.unit_symbol');
        $this->db->from('lab_tests lt');
        $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
        $this->db->join('lab_units lu', 'lu.id = lt.unit_id', 'left');
        $this->db->where('lt.id', $id);
        
        $query = $this->db->get();
        return $query->row_array();
    }

    public function addTest($data) {
        return $this->db->insert('lab_tests', $data) ? $this->db->insert_id() : false;
    }

    public function updateTest($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('lab_tests', $data);
    }

    public function deleteTest($id) {
        $this->db->where('id', $id);
        return $this->db->delete('lab_tests');
    }

    public function checkTestExists($test_name, $exclude_id = null) {
        $this->db->where('test_name', $test_name);
        if ($exclude_id) {
            $this->db->where('id !=', $exclude_id);
        }
        $query = $this->db->get('lab_tests');
        return $query->num_rows() > 0;
    }

    public function getTestUsageCount($test_id) {
        // Check if test is used in laboratory results
        $test = $this->getTest($test_id);
        if (!$test) return 0;
        
        $this->db->where('test_name', $test['test_name']);
        return $this->db->count_all_results('laboratory');
    }

    public function getTestsByCategory($category_id, $active_only = true) {
        $this->db->select('id, test_name, short_name, cost');
        $this->db->where('category_id', $category_id);
        if ($active_only) {
            $this->db->where('is_active', 1);
        }
        $this->db->order_by('test_name', 'ASC');
        $query = $this->db->get('lab_tests');
        return $query->result_array();
    }

    public function searchTests($search_term, $limit = 20) {
        $this->db->select('id, test_name, short_name, cost');
        $this->db->where('is_active', 1);
        $this->db->group_start();
        $this->db->like('test_name', $search_term);
        $this->db->or_like('short_name', $search_term);
        $this->db->group_end();
        $this->db->order_by('test_name', 'ASC');
        $this->db->limit($limit);
        
        $query = $this->db->get('lab_tests');
        return $query->result_array();
    }

    public function getAllTestsForExport() {
        $this->db->select('lt.*, ltc.category_name, lu.unit_name');
        $this->db->from('lab_tests lt');
        $this->db->join('lab_test_categories ltc', 'ltc.id = lt.category_id', 'left');
        $this->db->join('lab_units lu', 'lu.id = lt.unit_id', 'left');
        $this->db->order_by('ltc.category_name, lt.test_name', 'ASC');
        
        $query = $this->db->get();
        return $query->result_array();
    }

    // =============================================
    // REFERENCE RANGES MANAGEMENT
    // =============================================

    public function getReferenceRanges($test_id) {
        $this->db->where('test_id', $test_id);
        $this->db->where('is_active', 1);
        $this->db->order_by('gender, age_min', 'ASC');
        $query = $this->db->get('lab_reference_ranges');
        return $query->result_array();
    }

    public function addReferenceRange($data) {
        return $this->db->insert('lab_reference_ranges', $data);
    }

    public function updateReferenceRange($id, $data) {
        $this->db->where('id', $id);
        return $this->db->update('lab_reference_ranges', $data);
    }

    public function deleteReferenceRange($id) {
        $this->db->where('id', $id);
        return $this->db->delete('lab_reference_ranges');
    }

    // =============================================
    // STATISTICS AND REPORTS
    // =============================================

    public function getSetupStats() {
        $stats = array();
        
        // Count categories
        $this->db->where('is_active', 1);
        $stats['total_categories'] = $this->db->count_all_results('lab_test_categories');
        
        // Count tests
        $this->db->where('is_active', 1);
        $stats['total_tests'] = $this->db->count_all_results('lab_tests');
        
        // Count units
        $this->db->where('is_active', 1);
        $stats['total_units'] = $this->db->count_all_results('lab_units');
        
        // Count reference ranges
        $this->db->where('is_active', 1);
        $stats['total_reference_ranges'] = $this->db->count_all_results('lab_reference_ranges');
        
        // Count laboratory results (if table exists)
        if ($this->db->table_exists('laboratory')) {
            $stats['total_lab_results'] = $this->db->count_all_results('laboratory');
        } else {
            $stats['total_lab_results'] = 0;
        }
        
        // Most expensive tests
        $this->db->select('test_name, cost');
        $this->db->where('is_active', 1);
        $this->db->order_by('cost', 'DESC');
        $this->db->limit(5);
        $query = $this->db->get('lab_tests');
        $stats['expensive_tests'] = $query->result_array();
        
        // Tests by category
        $this->db->select('ltc.category_name, COUNT(lt.id) as test_count');
        $this->db->from('lab_test_categories ltc');
        $this->db->join('lab_tests lt', 'lt.category_id = ltc.id', 'left');
        $this->db->where('ltc.is_active', 1);
        $this->db->group_by('ltc.id, ltc.category_name');
        $this->db->order_by('test_count', 'DESC');
        $query = $this->db->get();
        $stats['tests_by_category'] = $query->result_array();
        
        return $stats;
    }

    // =============================================
    // HELPER METHODS
    // =============================================

    public function getActiveTestsForDropdown() {
        $this->db->select('id, test_name, cost');
        $this->db->where('is_active', 1);
        $this->db->order_by('test_name', 'ASC');
        $query = $this->db->get('lab_tests');
        
        $tests = array();
        foreach ($query->result_array() as $test) {
            //$tests[$test['id']] = $test['test_name'] . ' -  . number_format($test['cost'], 2);
            $tests[$test['id']] = $test['test_name'] . ' - $' . number_format($test['cost'], 2);
        }
        
        return $tests;
    }

    public function getTestCost($test_id) {
        $this->db->select('cost');
        $this->db->where('id', $test_id);
        $query = $this->db->get('lab_tests');
        $result = $query->row_array();
        
        return $result ? $result['cost'] : 0;
    }

    public function checkSystemInitialized() {
        // Check if tables exist and have data
        $tables = array('lab_test_categories', 'lab_tests', 'lab_units');
        
        foreach ($tables as $table) {
            if (!$this->db->table_exists($table)) {
                return false;
            }
            
            if ($this->db->count_all_results($table) == 0) {
                return false;
            }
        }
        
        return true;
    }
}