<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

/**
 * SIMPLIFIED Lab Reports Model - Fixed Version
 * Removes complex queries that might cause issues
 */
class Lab_reports_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get report by ID - Simplified version
     */
    public function getReportById($report_id) {
        try {
            $this->db->select('pr.*, p.patient_name, p.mobileno, p.age, p.gender, p.dob');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'pr.patient_id = p.id', 'left');
            $this->db->where('pr.id', $report_id);
            
            $query = $this->db->get();
            
            if ($query && $query->num_rows() > 0) {
                $result = $query->row_array();
                
                // Add computed fields
                $result['has_structured_results'] = (!empty($result['test_results']) && $result['test_results'] != '') ? 1 : 0;
                
                return $result;
            }
            
            return null;
            
        } catch (Exception $e) {
            log_message('error', 'Error getting report by ID: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get patient reports - Simplified version
     */
    public function getPatientReports($patient_id) {
        try {
            $this->db->select('*');
            $this->db->from('patient_lab_reports');
            $this->db->where('patient_id', $patient_id);
            $this->db->order_by('report_date', 'DESC');
            $this->db->order_by('created_at', 'DESC');
            
            $query = $this->db->get();
            
            if ($query) {
                $reports = $query->result_array();
                
                // Add computed fields
                foreach ($reports as &$report) {
                    $report['has_structured_results'] = (!empty($report['test_results']) && $report['test_results'] != '') ? 1 : 0;
                }
                
                return $reports;
            }
            
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Error getting patient reports: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Insert new lab report - Simplified version
     */
    public function insertReport($data) {
        try {
            // Ensure required fields have defaults
            if (!isset($data['status'])) {
                $data['status'] = 'pending';
            }
            
            if (!isset($data['is_critical'])) {
                $data['is_critical'] = 0;
            }
            
            if (!isset($data['created_at'])) {
                $data['created_at'] = date('Y-m-d H:i:s');
            }
            
            if (!isset($data['updated_at'])) {
                $data['updated_at'] = date('Y-m-d H:i:s');
            }
            
            $this->db->insert('patient_lab_reports', $data);
            
            if ($this->db->affected_rows() > 0) {
                return $this->db->insert_id();
            }
            
            return false;
            
        } catch (Exception $e) {
            log_message('error', 'Error inserting lab report: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Update lab report
     */
    public function updateReport($report_id, $data) {
        try {
            $data['updated_at'] = date('Y-m-d H:i:s');
            
            $this->db->where('id', $report_id);
            $result = $this->db->update('patient_lab_reports', $data);
            
            return $this->db->affected_rows() > 0;
            
        } catch (Exception $e) {
            log_message('error', 'Error updating report: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Verify report - Simplified version
     */
    public function verifyReport($report_id, $staff_id, $notes = null) {
        try {
            $update_data = [
                'status' => 'verified',
                'verified_by' => $staff_id,
                'verified_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s')
            ];

            if ($notes) {
                $update_data['verification_notes'] = $notes;
            }

            $this->db->where('id', $report_id);
            $this->db->update('patient_lab_reports', $update_data);

            return $this->db->affected_rows() > 0;

        } catch (Exception $e) {
            log_message('error', 'Error verifying report: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Mark report as critical
     */
    public function markAsCritical($report_id, $is_critical = 1) {
        try {
            $update_data = [
                'is_critical' => $is_critical,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $report_id);
            $this->db->update('patient_lab_reports', $update_data);

            return $this->db->affected_rows() > 0;

        } catch (Exception $e) {
            log_message('error', 'Error marking report as critical: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get dashboard statistics - Simplified version
     */
    public function getDashboardStats() {
        try {
            $stats = [];

            // Total reports
            $this->db->from('patient_lab_reports');
            $stats['total'] = $this->db->count_all_results();

            // Pending reports
            $this->db->from('patient_lab_reports');
            $this->db->where('status', 'pending');
            $stats['pending'] = $this->db->count_all_results();

            // Verified today
            $this->db->from('patient_lab_reports');
            $this->db->where('status', 'verified');
            $this->db->where('DATE(verified_at)', date('Y-m-d'));
            $stats['verified_today'] = $this->db->count_all_results();

            // Critical reports
            $this->db->from('patient_lab_reports');
            $this->db->where('is_critical', 1);
            $this->db->where('status !=', 'cancelled');
            $stats['critical'] = $this->db->count_all_results();

            return $stats;

        } catch (Exception $e) {
            log_message('error', 'Error getting dashboard stats: ' . $e->getMessage());
            return [
                'total' => 0,
                'pending' => 0,
                'verified_today' => 0,
                'critical' => 0
            ];
        }
    }

    /**
     * Search patients for dropdown - Simplified version
     */
    public function searchPatients($search_term, $limit = 10) {
        try {
            $this->db->select('id, patient_name, mobileno, age, gender');
            $this->db->from('patients');
            
            // Check if is_active column exists
            if ($this->db->field_exists('is_active', 'patients')) {
                $this->db->where('is_active', 'yes');
            }
            
            if (!empty($search_term)) {
                $this->db->group_start();
                $this->db->like('patient_name', $search_term);
                $this->db->or_like('mobileno', $search_term);
                $this->db->or_like('id', $search_term);
                $this->db->group_end();
            }
            
            $this->db->order_by('patient_name', 'ASC');
            $this->db->limit($limit);

            $query = $this->db->get();
            return $query ? $query->result_array() : [];
            
        } catch (Exception $e) {
            log_message('error', 'Error searching patients: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get reports for DataTables - Simplified version
     */
    public function getReportsForDataTables($request) {
        try {
            // Base query
            $this->db->select('pr.id, pr.patient_id, pr.report_title, pr.report_type, 
                              pr.department, pr.report_date, pr.status, pr.is_critical,
                              pr.created_at, pr.updated_at,
                              p.patient_name, p.mobileno');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'pr.patient_id = p.id', 'left');

            // Apply basic filters
            if (!empty($request['status'])) {
                $this->db->where('pr.status', $request['status']);
            }

            if (!empty($request['department'])) {
                $this->db->where('pr.department', $request['department']);
            }

            if (!empty($request['date_from'])) {
                $this->db->where('pr.report_date >=', $request['date_from']);
            }

            if (!empty($request['date_to'])) {
                $this->db->where('pr.report_date <=', $request['date_to']);
            }

            // Search functionality
            if (!empty($request['search']['value'])) {
                $search_value = $request['search']['value'];
                $this->db->group_start();
                $this->db->like('pr.report_title', $search_value);
                $this->db->or_like('p.patient_name', $search_value);
                $this->db->or_like('pr.report_type', $search_value);
                $this->db->or_like('pr.department', $search_value);
                $this->db->group_end();
            }

            // Get total count before limit
            $total_query = clone $this->db;
            $recordsTotal = $total_query->count_all_results();

            // Apply ordering
            $columns = ['pr.id', 'p.patient_name', 'pr.report_title', 'pr.report_type', 'pr.department', 'pr.report_date', 'pr.status'];
            if (!empty($request['order'])) {
                $order_column = $columns[$request['order'][0]['column']] ?? 'pr.id';
                $order_dir = $request['order'][0]['dir'] ?? 'DESC';
                $this->db->order_by($order_column, $order_dir);
            } else {
                $this->db->order_by('pr.id', 'DESC');
            }

            // Apply pagination
            if (isset($request['length']) && $request['length'] != -1) {
                $this->db->limit($request['length'], $request['start'] ?? 0);
            }

            $query = $this->db->get();
            $data = $query ? $query->result_array() : [];

            // Add computed fields
            foreach ($data as &$row) {
                $row['has_structured_results'] = (!empty($row['test_results']) && $row['test_results'] != '') ? 1 : 0;
            }

            return [
                'draw' => intval($request['draw'] ?? 1),
                'recordsTotal' => $recordsTotal,
                'recordsFiltered' => $recordsTotal,
                'data' => $data
            ];

        } catch (Exception $e) {
            log_message('error', 'Error getting reports for DataTables: ' . $e->getMessage());
            return [
                'draw' => 0,
                'recordsTotal' => 0,
                'recordsFiltered' => 0,
                'data' => []
            ];
        }
    }

    /**
     * Get all reports for export
     */
    public function getAllReports($limit = 1000, $offset = 0, $order_by = 'id DESC', $filters = []) {
        try {
            $this->db->select('pr.*, p.patient_name, p.mobileno');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'pr.patient_id = p.id', 'left');

            // Apply filters
            if (!empty($filters['status'])) {
                $this->db->where('pr.status', $filters['status']);
            }

            if (!empty($filters['department'])) {
                $this->db->where('pr.department', $filters['department']);
            }

            if (!empty($filters['date_from'])) {
                $this->db->where('pr.report_date >=', $filters['date_from']);
            }

            if (!empty($filters['date_to'])) {
                $this->db->where('pr.report_date <=', $filters['date_to']);
            }

            if (!empty($filters['is_critical'])) {
                $this->db->where('pr.is_critical', 1);
            }

            if ($order_by) {
                $order_parts = explode(' ', $order_by);
                $column = $order_parts[0] ?? 'id';
                $direction = $order_parts[1] ?? 'DESC';
                $this->db->order_by('pr.' . $column, $direction);
            }

            if ($limit > 0) {
                $this->db->limit($limit, $offset);
            }

            $query = $this->db->get();
            return $query ? $query->result_array() : [];

        } catch (Exception $e) {
            log_message('error', 'Error getting all reports: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get patient reports count
     */
    public function getPatientReportsCount($patient_id) {
        try {
            $this->db->select('COUNT(*) as total');
            $this->db->from('patient_lab_reports');
            $this->db->where('patient_id', $patient_id);
            $query = $this->db->get();
            $result = $query->row();
            return $result ? $result->total : 0;
        } catch (Exception $e) {
            log_message('error', 'Error getting patient reports count: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get reports by status
     */
    public function getReportsByStatus($status) {
        try {
            $this->db->select('pr.*, p.patient_name, p.id as patient_id');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'p.id = pr.patient_id', 'left');
            $this->db->where('pr.status', $status);
            $this->db->order_by('pr.created_at', 'DESC');

            $query = $this->db->get();
            return $query ? $query->result_array() : [];
        } catch (Exception $e) {
            log_message('error', 'Error getting reports by status: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get recent reports
     */
    public function getRecentReports($limit = 10) {
        try {
            $this->db->select('pr.*, p.patient_name, p.id as patient_id');
            $this->db->from('patient_lab_reports pr');
            $this->db->join('patients p', 'p.id = pr.patient_id', 'left');
            $this->db->order_by('pr.created_at', 'DESC');
            $this->db->limit($limit);

            $query = $this->db->get();
            return $query ? $query->result_array() : [];
        } catch (Exception $e) {
            log_message('error', 'Error getting recent reports: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Delete report
     */
    public function deleteReport($report_id) {
        try {
            // Get report details for file cleanup
            $report = $this->getReportById($report_id);
            
            if (!$report) {
                return false;
            }

            // Delete main report
            $this->db->where('id', $report_id);
            $this->db->delete('patient_lab_reports');

            if ($this->db->affected_rows() > 0) {
                // Clean up file if it exists
                if (!empty($report['file_path']) && file_exists($report['file_path'])) {
                    unlink($report['file_path']);
                }
                return true;
            }

            return false;

        } catch (Exception $e) {
            log_message('error', 'Error deleting report: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if table exists
     */
    public function checkTable() {
        return $this->db->table_exists('patient_lab_reports');
    }

    /**
     * Update test results
     */
    public function updateTestResults($report_id, $test_results, $interpretations = null, $reference_ranges = null) {
        try {
            $update_data = [
                'test_results' => is_array($test_results) ? json_encode($test_results) : $test_results,
                'updated_at' => date('Y-m-d H:i:s')
            ];

            if ($interpretations) {
                $update_data['test_interpretations'] = is_array($interpretations) ? json_encode($interpretations) : $interpretations;
            }

            if ($reference_ranges) {
                $update_data['reference_ranges'] = is_array($reference_ranges) ? json_encode($reference_ranges) : $reference_ranges;
            }

            $this->db->where('id', $report_id);
            $this->db->update('patient_lab_reports', $update_data);

            return $this->db->affected_rows() > 0;

        } catch (Exception $e) {
            log_message('error', 'Error updating test results: ' . $e->getMessage());
            return false;
        }
    }
}