<?php
defined('BASEPATH') OR exit('No direct script access allowed');

class Items_model extends CI_Model {
    
    public function __construct() {
        parent::__construct();
        $this->load->database();
        $this->createTablesIfNotExists();
    }
    
    private function createTablesIfNotExists() {
        if (!$this->db->table_exists('item_categories')) {
            $sql = "CREATE TABLE `item_categories` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `category_name` varchar(100) NOT NULL,
                `category_code` varchar(20) UNIQUE NOT NULL,
                `description` text,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
            $this->insertSampleCategories();
        }
        
        if (!$this->db->table_exists('items')) {
            $sql = "CREATE TABLE `items` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `item_code` varchar(50) UNIQUE NOT NULL,
                `item_name` varchar(200) NOT NULL,
                `description` text,
                `category_id` int(11) NOT NULL,
                `unit_of_measure` varchar(50) NOT NULL DEFAULT 'pieces',
                `unit_cost` decimal(10,2) DEFAULT 0.00,
                `minimum_stock` int(11) DEFAULT 0,
                `reorder_level` int(11) DEFAULT 0,
                `current_stock` int(11) DEFAULT 0,
                `item_type` enum('consumable','non_consumable','service') DEFAULT 'consumable',
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8";
            
            $this->db->query($sql);
            $this->insertSampleItems();
        }
    }
    
    private function insertSampleCategories() {
        $categories = [
            ['category_name' => 'Medical Supplies', 'category_code' => 'MED', 'description' => 'General medical supplies'],
            ['category_name' => 'Pharmaceuticals', 'category_code' => 'PHARM', 'description' => 'Medicines and drugs'],
            ['category_name' => 'Laboratory Supplies', 'category_code' => 'LAB', 'description' => 'Laboratory consumables'],
            ['category_name' => 'Office Supplies', 'category_code' => 'OFFICE', 'description' => 'General office supplies']
        ];
        $this->db->insert_batch('item_categories', $categories);
    }
    
    private function insertSampleItems() {
        $items = [
            ['item_code' => 'SYR001', 'item_name' => 'Disposable Syringes 5ml', 'category_id' => 1, 'unit_cost' => 15.00, 'minimum_stock' => 100, 'reorder_level' => 200, 'current_stock' => 500],
            ['item_code' => 'GLV001', 'item_name' => 'Latex Gloves Medium', 'category_id' => 1, 'unit_of_measure' => 'box', 'unit_cost' => 800.00, 'minimum_stock' => 10, 'reorder_level' => 20, 'current_stock' => 45],
            ['item_code' => 'PAR001', 'item_name' => 'Paracetamol 500mg', 'category_id' => 2, 'unit_of_measure' => 'tablets', 'unit_cost' => 2.50, 'minimum_stock' => 500, 'reorder_level' => 1000, 'current_stock' => 2500]
        ];
        $this->db->insert_batch('items', $items);
    }
    
    public function saveItem($data) {
        if (empty($data['item_code'])) {
            $data['item_code'] = $this->generateItemCode($data['item_name']);
        }
        $this->db->insert('items', $data);
        return $this->db->insert_id();
    }
    
    public function getActiveCategories() {
        $this->db->select('*');
        $this->db->from('item_categories');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('category_name', 'asc');
        $query = $this->db->get();
        return $query->result_array();
    }
    
    public function getItemsForSelect() {
        $this->db->select('id, item_name, item_code, unit_cost, unit_of_measure');
        $this->db->from('items');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('item_name', 'asc');
        $query = $this->db->get();
        return $query->result_array();
    }
    
    public function countLowStockItems() {
        $this->db->from('items');
        $this->db->where('current_stock <= reorder_level');
        $this->db->where('is_active', 'yes');
        return $this->db->count_all_results();
    }
    
    public function getUrgentItems($limit = 10) {
        $this->db->select('i.*, ic.category_name');
        $this->db->from('items i');
        $this->db->join('item_categories ic', 'ic.id = i.category_id', 'left');
        $this->db->where('i.current_stock <= i.reorder_level');
        $this->db->where('i.is_active', 'yes');
        $this->db->limit($limit);
        $query = $this->db->get();
        return $query->result_array();
    }
    
    public function updateStock($item_id, $quantity) {
        $this->db->set('current_stock', 'current_stock + ' . (int)$quantity, FALSE);
        $this->db->where('id', $item_id);
        return $this->db->update('items');
    }
    
    private function generateItemCode($item_name) {
        $prefix = strtoupper(substr(preg_replace('/[^A-Za-z]/', '', $item_name), 0, 3));
        $this->db->select('COUNT(*) + 1 as next_seq');
        $this->db->from('items');
        $query = $this->db->get();
        $sequence = $query->row()->next_seq;
        return $prefix . str_pad($sequence, 3, '0', STR_PAD_LEFT);
    }
}