<?php
/**
 * Insurance Model - FIXED VERSION
 * File: application/models/Insurance_model.php
 */

class Insurance_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    /**
     * Get all insurance companies
     */
    public function getInsuranceCompanies($status = 'active') {
        $this->db->select('*');
        $this->db->from('insurance_companies');
        if ($status !== 'all') {
            $this->db->where('status', $status);
        }
        $this->db->order_by('company_name', 'ASC');
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getInsuranceCompanies query failed: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
    }

    /**
     * Get insurance company by ID
     */
    public function getInsuranceCompany($id) {
        $this->db->select('*');
        $this->db->from('insurance_companies');
        $this->db->where('id', $id);
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getInsuranceCompany query failed: ' . $this->db->error()['message']);
            return null;
        }
        
        return $query->row_array();
    }

    /**
     * Get insurance company by name
     */
    public function getInsuranceCompanyByName($name) {
        $this->db->select('*');
        $this->db->from('insurance_companies');
        $this->db->where('company_name', $name);
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getInsuranceCompanyByName query failed: ' . $this->db->error()['message']);
            return null;
        }
        
        return $query->row_array();
    }

    /**
     * Create or get insurance company
     */
    public function getOrCreateInsuranceCompany($companyName, $created_by = null) {
        // Check if company exists
        $existing = $this->getInsuranceCompanyByName($companyName);
        if ($existing) {
            return $existing['id'];
        }

        // Create new company
        $company_code = $this->generateCompanyCode($companyName);
        $insurance_data = [
            'company_name' => $companyName,
            'company_code' => $company_code,
            'status' => 'active',
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $created_by
        ];

        $this->db->insert('insurance_companies', $insurance_data);
        return $this->db->insert_id();
    }

    /**
     * Generate company code from name
     */
    private function generateCompanyCode($companyName) {
        // Create code from first letters of words
        $words = explode(' ', strtoupper($companyName));
        $code = '';
        foreach ($words as $word) {
            if (!empty($word)) {
                $code .= substr($word, 0, 1);
            }
        }
        
        // Limit to 10 characters
        $code = substr($code, 0, 10);
        
        // Check if code already exists
        $this->db->where('company_code', $code);
        $existing = $this->db->get('insurance_companies')->num_rows();
        
        if ($existing > 0) {
            $code .= rand(10, 99);
        }
        
        return $code;
    }

    /**
     * Get all item families
     */
    public function getItemFamilies($status = 'active') {
        // Check if item_families table exists
        if (!$this->db->table_exists('item_families')) {
            // Create basic families array if table doesn't exist
            return [
                ['id' => 1, 'family_name' => 'Analgesics', 'status' => 'active'],
                ['id' => 2, 'family_name' => 'Antibiotics', 'status' => 'active'],
                ['id' => 3, 'family_name' => 'Diagnostic', 'status' => 'active'],
                ['id' => 4, 'family_name' => 'Laboratory', 'status' => 'active'],
                ['id' => 5, 'family_name' => 'Consultation', 'status' => 'active'],
                ['id' => 6, 'family_name' => 'Monitoring', 'status' => 'active']
            ];
        }

        $this->db->select('*');
        $this->db->from('item_families');
        if ($status !== 'all') {
            $this->db->where('status', $status);
        }
        $this->db->order_by('family_name', 'ASC');
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getItemFamilies query failed: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
    }

    /**
     * Get or create item family
     */
    public function getOrCreateItemFamily($familyName, $created_by = null) {
        // Check if item_families table exists
        if (!$this->db->table_exists('item_families')) {
            // Return a default family ID if table doesn't exist
            return 1;
        }

        // Check if family exists
        $this->db->select('id');
        $this->db->from('item_families');
        $this->db->where('family_name', $familyName);
        $existing = $this->db->get()->row_array();
        
        if ($existing) {
            return $existing['id'];
        }

        // Create new family
        $family_code = strtoupper(substr(str_replace(' ', '_', $familyName), 0, 10));
        $family_data = [
            'family_name' => $familyName,
            'family_code' => $family_code,
            'status' => 'active',
            'created_at' => date('Y-m-d H:i:s'),
            'created_by' => $created_by
        ];

        $this->db->insert('item_families', $family_data);
        return $this->db->insert_id();
    }

    /**
     * FIXED: Get insurance pricing with proper pagination
     */
    public function getInsurancePricing($filters = []) {
        $this->db->select('
            ip.*,
            ic.company_name,
            ic.company_code
        ');
        $this->db->from('insurance_pricing ip');
        $this->db->join('insurance_companies ic', 'ip.insurance_company_id = ic.id', 'left');
        
        // Only join master_items if the table exists
        if ($this->db->table_exists('master_items')) {
            $this->db->select('mi.item_code, mi.generic_name, mi.brand_name', false);
            $this->db->join('master_items mi', 'ip.master_item_id = mi.id', 'left');
        }

        // Apply filters
        if (!empty($filters['insurance_id'])) {
            $this->db->where('ip.insurance_company_id', $filters['insurance_id']);
        }

        if (!empty($filters['item_type'])) {
            $this->db->where('ip.item_type', $filters['item_type']);
        }

        if (!empty($filters['item_family'])) {
            $this->db->where('ip.item_family', $filters['item_family']);
        }

        if (!empty($filters['status'])) {
            $this->db->where('ip.status', $filters['status']);
        } else {
            // Default to active items only
            $this->db->where('ip.status', 'active');
        }

        if (!empty($filters['search'])) {
            $this->db->like('ip.item_name', $filters['search']);
        }

        if (!empty($filters['approval_status'])) {
            $this->db->where('ip.approval_status', $filters['approval_status']);
        }

        // FIXED: Proper pagination handling
        if (!empty($filters['limit'])) {
            $limit = (int) $filters['limit'];
            $offset = isset($filters['offset']) ? max(0, (int) $filters['offset']) : 0;
            
            // Ensure limit is positive
            if ($limit > 0) {
                $this->db->limit($limit, $offset);
            }
        }

        $this->db->order_by('ic.company_name ASC, ip.item_name ASC');
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getInsurancePricing query failed: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
    }

    /**
     * FIXED: Count insurance pricing records with proper error handling
     */
    public function countInsurancePricing($filters = []) {
        $this->db->from('insurance_pricing ip');
        $this->db->join('insurance_companies ic', 'ip.insurance_company_id = ic.id', 'left');

        // Apply same filters as getInsurancePricing
        if (!empty($filters['insurance_id'])) {
            $this->db->where('ip.insurance_company_id', $filters['insurance_id']);
        }

        if (!empty($filters['item_type'])) {
            $this->db->where('ip.item_type', $filters['item_type']);
        }

        if (!empty($filters['item_family'])) {
            $this->db->where('ip.item_family', $filters['item_family']);
        }

        if (!empty($filters['status'])) {
            $this->db->where('ip.status', $filters['status']);
        } else {
            // Default to active items only
            $this->db->where('ip.status', 'active');
        }

        if (!empty($filters['search'])) {
            $this->db->like('ip.item_name', $filters['search']);
        }

        if (!empty($filters['approval_status'])) {
            $this->db->where('ip.approval_status', $filters['approval_status']);
        }

        $count = $this->db->count_all_results();
        
        if ($count === false) {
            log_message('error', 'countInsurancePricing query failed: ' . $this->db->error()['message']);
            return 0;
        }
        
        return $count;
    }

    /**
     * Check if pricing item exists
     */
    public function checkExistingPricingItem($insurance_id, $item_name, $item_type) {
        $this->db->select('*');
        $this->db->from('insurance_pricing');
        $this->db->where('insurance_company_id', $insurance_id);
        $this->db->where('item_name', $item_name);
        $this->db->where('item_type', $item_type);
        $this->db->where('status !=', 'inactive');
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'checkExistingPricingItem query failed: ' . $this->db->error()['message']);
            return null;
        }
        
        return $query->row_array();
    }

    /**
     * Insert pricing item
     */
    public function insertPricingItem($data) {
        $result = $this->db->insert('insurance_pricing', $data);
        if (!$result) {
            log_message('error', 'insertPricingItem failed: ' . $this->db->error()['message']);
        }
        return $result;
    }

    /**
     * Update pricing item
     */
    public function updatePricingItem($id, $data) {
        $this->db->where('id', $id);
        $result = $this->db->update('insurance_pricing', $data);
        if (!$result) {
            log_message('error', 'updatePricingItem failed: ' . $this->db->error()['message']);
        }
        return $result;
    }

    /**
     * Bulk insert pricing items
     */
    public function bulkInsertPricingItems($items) {
        if (empty($items)) {
            return true;
        }
        
        $result = $this->db->insert_batch('insurance_pricing', $items);
        if (!$result) {
            log_message('error', 'bulkInsertPricingItems failed: ' . $this->db->error()['message']);
        }
        return $result;
    }

    /**
     * Get master item by name and type
     */
    public function getMasterItemByName($item_name, $item_type) {
        // Check if master_items table exists
        if (!$this->db->table_exists('master_items')) {
            return null;
        }

        $this->db->select('*');
        $this->db->from('master_items');
        $this->db->where('item_name', $item_name);
        $this->db->where('item_type', $item_type);
        $this->db->where('is_active', 'yes');
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getMasterItemByName query failed: ' . $this->db->error()['message']);
            return null;
        }
        
        return $query->row_array();
    }

    /**
     * Create master item
     */
    public function createMasterItem($data) {
        // Check if master_items table exists
        if (!$this->db->table_exists('master_items')) {
            return null;
        }

        $this->db->insert('master_items', $data);
        return $this->db->insert_id();
    }

    /**
     * FIXED: Process import batch with enhanced error handling
     */
    public function processImportBatch($batch_id, $items, $options = []) {
        $this->db->trans_start();

        $results = [
            'total' => count($items),
            'created' => 0,
            'updated' => 0,
            'failed' => 0,
            'errors' => []
        ];

        foreach ($items as $index => $item) {
            try {
                // Validate required fields
                if (empty($item['itemName']) || empty($item['itemType']) || empty($item['insuranceCompany'])) {
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $index + 1,
                        'item' => $item['itemName'] ?? 'Unknown',
                        'error' => 'Missing required fields (itemName, itemType, or insuranceCompany)'
                    ];
                    continue;
                }

                // Get or create insurance company
                $insurance_id = $this->getOrCreateInsuranceCompany(
                    $item['insuranceCompany'],
                    $options['user_id'] ?? null
                );

                // Get or create item family (handle gracefully if table doesn't exist)
                $family_id = null;
                if (!empty($item['itemFamily'])) {
                    $family_id = $this->getOrCreateItemFamily(
                        $item['itemFamily'],
                        $options['user_id'] ?? null
                    );
                }

                // Check if item exists
                $existing_item = $this->checkExistingPricingItem(
                    $insurance_id,
                    $item['itemName'],
                    $item['itemType']
                );

                $pricing_data = [
                    'insurance_company_id' => $insurance_id,
                    'item_name' => $item['itemName'],
                    'item_type' => $item['itemType'],
                    'item_family' => $item['itemFamily'] ?? 'General',
                    'item_price' => floatval($item['itemPrice'] ?? 0),
                    'import_batch_id' => $batch_id,
                    'import_source' => 'excel_upload',
                    'effective_date' => !empty($item['effectiveDate']) ? date('Y-m-d', strtotime($item['effectiveDate'])) : date('Y-m-d'),
                    'status' => !empty($item['status']) ? $item['status'] : 'active',
                    'approval_status' => 'approved', // Auto-approve imports
                    'approved_by' => $options['user_id'] ?? null,
                    'approved_at' => date('Y-m-d H:i:s'),
                    'updated_at' => date('Y-m-d H:i:s'),
                    'updated_by' => $options['user_id'] ?? null
                ];

                if ($existing_item && ($options['update_existing'] ?? false)) {
                    // Update existing item
                    if ($this->updatePricingItem($existing_item['id'], $pricing_data)) {
                        $results['updated']++;
                    } else {
                        $results['failed']++;
                        $results['errors'][] = [
                            'row' => $index + 1,
                            'item' => $item['itemName'],
                            'error' => 'Failed to update existing item'
                        ];
                    }
                } elseif (!$existing_item) {
                    // Create new item
                    $pricing_data['created_at'] = date('Y-m-d H:i:s');
                    $pricing_data['created_by'] = $options['user_id'] ?? null;
                    
                    if ($this->insertPricingItem($pricing_data)) {
                        $results['created']++;
                    } else {
                        $results['failed']++;
                        $results['errors'][] = [
                            'row' => $index + 1,
                            'item' => $item['itemName'],
                            'error' => 'Failed to create new item'
                        ];
                    }
                } else {
                    // Item exists but update not allowed
                    $results['failed']++;
                    $results['errors'][] = [
                        'row' => $index + 1,
                        'item' => $item['itemName'],
                        'error' => 'Item already exists (update not enabled)'
                    ];
                }

            } catch (Exception $e) {
                $results['failed']++;
                $results['errors'][] = [
                    'row' => $index + 1,
                    'item' => $item['itemName'] ?? 'Unknown',
                    'error' => $e->getMessage()
                ];
                log_message('error', 'Import batch processing error: ' . $e->getMessage());
            }
        }

        $this->db->trans_complete();

        if ($this->db->trans_status() === FALSE) {
            log_message('error', 'Database transaction failed during import batch processing');
            throw new Exception('Database transaction failed during import');
        }

        return $results;
    }

    /**
     * Create import history record
     */
    public function createImportHistory($data) {
        // Check if import_history table exists
        if (!$this->db->table_exists('import_history')) {
            log_message('error', 'import_history table does not exist');
            return false;
        }

        $result = $this->db->insert('import_history', $data);
        if (!$result) {
            log_message('error', 'createImportHistory failed: ' . $this->db->error()['message']);
            return false;
        }
        
        return $this->db->insert_id();
    }

    /**
     * Update import history
     */
    public function updateImportHistory($id, $data) {
        // Check if import_history table exists
        if (!$this->db->table_exists('import_history')) {
            log_message('error', 'import_history table does not exist');
            return false;
        }

        $this->db->where('id', $id);
        $result = $this->db->update('import_history', $data);
        if (!$result) {
            log_message('error', 'updateImportHistory failed: ' . $this->db->error()['message']);
        }
        return $result;
    }

    /**
     * FIXED: Get import history with error handling
     */
    public function getImportHistory($limit = 20, $offset = 0) {
        // Check if import_history table exists
        if (!$this->db->table_exists('import_history')) {
            log_message('info', 'import_history table does not exist, returning empty array');
            return [];
        }

        // Ensure valid pagination values
        $limit = max(1, (int) $limit);
        $offset = max(0, (int) $offset);

        $this->db->select('*');
        $this->db->from('import_history');
        $this->db->where('import_type', 'insurance_pricing');
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit, $offset);
        
        $query = $this->db->get();
        if (!$query) {
            log_message('error', 'getImportHistory query failed: ' . $this->db->error()['message']);
            return [];
        }
        
        return $query->result_array();
    }

    /**
     * Log import with error handling
     */
    public function logImport($data) {
        // Check if import_history table exists
        if (!$this->db->table_exists('import_history')) {
            log_message('warning', 'Cannot log import: import_history table does not exist');
            return false;
        }

        $import_data = array(
            'batch_id' => $data['batch_id'],
            'file_name' => $data['file_name'],
            'file_size' => $data['file_size'] ?? null,
            'file_path' => $data['file_path'] ?? null,
            'total_records' => $data['total_records'] ?? 0,
            'import_type' => 'insurance_pricing',
            'import_status' => 'pending',
            'imported_by' => $data['imported_by'] ?? null,
            'created_at' => date('Y-m-d H:i:s')
        );
        
        $result = $this->db->insert('import_history', $import_data);
        if (!$result) {
            log_message('error', 'logImport failed: ' . $this->db->error()['message']);
        }
        
        return $result;
    }

    /**
     * Generate unique batch ID
     */
    public function generateBatchId() {
        return 'BATCH_' . date('Ymd_His') . '_' . mt_rand(1000, 9999);
    }

    /**
     * Validate pricing data
     */
    public function validatePricingData($item) {
        $errors = [];

        // Required fields
        if (empty($item['itemName']) || strlen(trim($item['itemName'])) < 2) {
            $errors[] = 'Item name is required and must be at least 2 characters';
        }

        if (empty($item['itemType'])) {
            $errors[] = 'Item type is required';
        }

        if (!in_array($item['itemType'], ['Stock', 'Service', 'Medicine', 'Equipment', 'Procedure'])) {
            $errors[] = 'Invalid item type';
        }

        if (empty($item['insuranceCompany'])) {
            $errors[] = 'Insurance company is required';
        }

        // Price validation
        if (!isset($item['itemPrice']) || !is_numeric($item['itemPrice'])) {
            $errors[] = 'Item price must be a valid number';
        } elseif ($item['itemPrice'] < 0) {
            $errors[] = 'Item price cannot be negative';
        } elseif ($item['itemPrice'] > 1000000) {
            $errors[] = 'Item price seems unreasonably high (over 1,000,000)';
        }

        // Item name length validation
        if (strlen($item['itemName']) > 500) {
            $errors[] = 'Item name is too long (maximum 500 characters)';
        }

        return $errors;
    }

    /**
     * Delete pricing item
     */
    public function deletePricingItem($id) {
        $this->db->where('id', $id);
        $result = $this->db->delete('insurance_pricing');
        if (!$result) {
            log_message('error', 'deletePricingItem failed: ' . $this->db->error()['message']);
        }
        return $result;
    }

    /**
     * Get pricing statistics with error handling
     */
    public function getPricingStatistics() {
        $statistics = [
            'by_company' => [],
            'by_type' => [],
            'by_family' => [],
            'overall' => [
                'total_items' => 0,
                'total_companies' => 0,
                'total_families' => 0,
                'average_price' => 0,
                'min_price' => 0,
                'max_price' => 0
            ]
        ];

        try {
            // Total items by insurance company
            $this->db->select('ic.company_name, COUNT(ip.id) as total_items');
            $this->db->from('insurance_pricing ip');
            $this->db->join('insurance_companies ic', 'ip.insurance_company_id = ic.id');
            $this->db->where('ip.status', 'active');
            $this->db->group_by('ic.id, ic.company_name');
            $this->db->order_by('total_items', 'DESC');
            $query = $this->db->get();
            if ($query) {
                $statistics['by_company'] = $query->result_array();
            }

            // Total items by type
            $this->db->select('item_type, COUNT(*) as total_items');
            $this->db->from('insurance_pricing');
            $this->db->where('status', 'active');
            $this->db->group_by('item_type');
            $query = $this->db->get();
            if ($query) {
                $statistics['by_type'] = $query->result_array();
            }

            // Total items by family
            $this->db->select('item_family, COUNT(*) as total_items');
            $this->db->from('insurance_pricing');
            $this->db->where('status', 'active');
            $this->db->group_by('item_family');
            $this->db->order_by('total_items', 'DESC');
            $this->db->limit(10);
            $query = $this->db->get();
            if ($query) {
                $statistics['by_family'] = $query->result_array();
            }

            // Overall statistics
            $this->db->select('
                COUNT(*) as total_items,
                COUNT(DISTINCT insurance_company_id) as total_companies,
                COUNT(DISTINCT item_family) as total_families,
                AVG(item_price) as average_price,
                MIN(item_price) as min_price,
                MAX(item_price) as max_price
            ');
            $this->db->from('insurance_pricing');
            $this->db->where('status', 'active');
            $query = $this->db->get();
            if ($query) {
                $overall = $query->row_array();
                if ($overall) {
                    $statistics['overall'] = $overall;
                }
            }

        } catch (Exception $e) {
            log_message('error', 'getPricingStatistics failed: ' . $e->getMessage());
        }

        return $statistics;
    }

    /**
     * FIXED: Check if table exists before operations
     */
    private function tableExists($table_name) {
        return $this->db->table_exists($table_name);
    }

    /**
     * Enhanced error handling for all database operations
     */
    private function handleDatabaseError($operation = 'Database operation') {
        $error = $this->db->error();
        $message = $operation . ' failed: ' . $error['message'];
        log_message('error', $message);
        
        // In development, you might want to show the error
        if (ENVIRONMENT === 'development') {
            show_error($message);
        }
        
        return false;
    }

    /**
     * Debug method to check system status
     */
    public function getSystemStatus() {
        $status = [
            'database_connected' => $this->db->conn_id ? true : false,
            'tables' => []
        ];

        $required_tables = [
            'insurance_companies',
            'insurance_pricing',
            'item_families',
            'master_items',
            'import_history',
            'price_change_log'
        ];

        foreach ($required_tables as $table) {
            $status['tables'][$table] = [
                'exists' => $this->db->table_exists($table),
                'count' => 0
            ];
            
            if ($status['tables'][$table]['exists']) {
                try {
                    $status['tables'][$table]['count'] = $this->db->count_all($table);
                } catch (Exception $e) {
                    $status['tables'][$table]['error'] = $e->getMessage();
                }
            }
        }

        return $status;
    }
}

?>