<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * HR Training Model
 * Handles training and learning management operations
 */
class Hr_training_model extends CI_Model {

    public function getStaffSkillMatrix($staff_id) {
        $this->db->where('staff_id', $staff_id);
        $this->db->order_by('skill_category', 'ASC');
        return $this->db->get('staff_skill_matrix')->result();
    }

    public function getExpiringCertifications($days = 30) {
        $expiry_date = date('Y-m-d', strtotime('+' . $days . ' days'));
        
        $this->db->select('ssm.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('staff_skill_matrix ssm');
        $this->db->join('staff s', 's.id = ssm.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('ssm.certification_expiry <=', $expiry_date);
        $this->db->where('ssm.certification_expiry >=', date('Y-m-d'));
        $this->db->order_by('ssm.certification_expiry', 'ASC');
        
        return $this->db->get()->result();
    }

    // =============================================
    // TRAINING SESSIONS MANAGEMENT
    // =============================================

    public function scheduleTrainingSession($data) {
        return $this->db->insert('training_sessions', $data);
    }

    public function getTrainingSessions($course_id = null, $start_date = null, $end_date = null) {
        $this->db->select('ts.*, tc.name as course_name, tc.duration_hours, 
                          trainer.name as trainer_name, trainer.surname as trainer_surname');
        $this->db->from('training_sessions ts');
        $this->db->join('training_courses tc', 'tc.id = ts.course_id', 'left');
        $this->db->join('staff trainer', 'trainer.id = ts.trainer_id', 'left');
        
        if ($course_id) {
            $this->db->where('ts.course_id', $course_id);
        }
        
        if ($start_date && $end_date) {
            $this->db->where('ts.session_date >=', $start_date);
            $this->db->where('ts.session_date <=', $end_date);
        }
        
        $this->db->order_by('ts.session_date', 'ASC');
        return $this->db->get()->result();
    }

    public function getUpcomingTraining($limit = 5) {
        $this->db->select('ts.*, tc.name as course_name');
        $this->db->from('training_sessions ts');
        $this->db->join('training_courses tc', 'tc.id = ts.course_id', 'left');
        $this->db->where('ts.session_date >=', date('Y-m-d'));
        $this->db->where('ts.status', 'scheduled');
        $this->db->order_by('ts.session_date', 'ASC');
        $this->db->limit($limit);
        return $this->db->get()->result();
    }

    public function getSessionAttendance($session_id) {
        $this->db->select('tsa.*, s.name, s.surname, s.employee_id, d.name as department_name');
        $this->db->from('training_session_attendance tsa');
        $this->db->join('staff s', 's.id = tsa.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('tsa.session_id', $session_id);
        $this->db->order_by('s.name', 'ASC');
        return $this->db->get()->result();
    }

    public function markSessionAttendance($session_id, $staff_id, $attendance_status, $notes = '') {
        $attendance_data = [
            'session_id' => $session_id,
            'staff_id' => $staff_id,
            'attendance_status' => $attendance_status,
            'notes' => $notes,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        // Check if attendance already recorded
        $this->db->where('session_id', $session_id);
        $this->db->where('staff_id', $staff_id);
        $existing = $this->db->get('training_session_attendance')->row();
        
        if ($existing) {
            $this->db->where('id', $existing->id);
            unset($attendance_data['created_at']);
            $attendance_data['updated_at'] = date('Y-m-d H:i:s');
            return $this->db->update('training_session_attendance', $attendance_data);
        } else {
            return $this->db->insert('training_session_attendance', $attendance_data);
        }
    }

    // =============================================
    // TRAINING HISTORY & RECORDS
    // =============================================

    public function getStaffTrainingHistory($staff_id) {
        $this->db->select('te.*, tc.name as course_name, tc.duration_hours, tcat.name as category_name,
                          ts.session_date, ts.location as session_location');
        $this->db->from('training_enrollments te');
        $this->db->join('training_courses tc', 'tc.id = te.course_id', 'left');
        $this->db->join('training_categories tcat', 'tcat.id = tc.category_id', 'left');
        $this->db->join('training_sessions ts', 'ts.course_id = te.course_id', 'left');
        $this->db->where('te.staff_id', $staff_id);
        $this->db->order_by('te.completion_date', 'DESC');
        return $this->db->get()->result();
    }

    public function getTrainingCompletionRate($department_id = null, $course_id = null) {
        $this->db->select('
            COUNT(te.id) as total_enrollments,
            SUM(CASE WHEN te.status = "completed" THEN 1 ELSE 0 END) as completed_enrollments
        ');
        $this->db->from('training_enrollments te');
        $this->db->join('staff s', 's.id = te.staff_id', 'left');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        if ($course_id) {
            $this->db->where('te.course_id', $course_id);
        }
        
        $result = $this->db->get()->row();
        
        if ($result && $result->total_enrollments > 0) {
            return round(($result->completed_enrollments / $result->total_enrollments) * 100, 2);
        }
        
        return 0;
    }

    public function getTrainingDueCount() {
        // Get staff who haven't completed mandatory trainings
        $this->db->select('COUNT(DISTINCT s.id) as count');
        $this->db->from('staff s');
        $this->db->join('training_courses tc', 'tc.is_mandatory = "yes" AND tc.is_active = "yes"', 'inner');
        $this->db->join('training_enrollments te', 'te.staff_id = s.id AND te.course_id = tc.id AND te.status = "completed"', 'left');
        $this->db->where('s.is_active', 'yes');
        $this->db->where('te.id IS NULL');
        
        $result = $this->db->get()->row();
        return $result ? $result->count : 0;
    }

    // =============================================
    // PERFORMANCE TRACKING
    // =============================================

    public function getTrainingPerformanceReport($start_date, $end_date, $department_id = null) {
        $this->db->select('
            d.name as department,
            COUNT(te.id) as total_enrollments,
            SUM(CASE WHEN te.status = "completed" THEN 1 ELSE 0 END) as completed_trainings,
            AVG(CASE WHEN te.status = "completed" THEN te.score ELSE NULL END) as avg_score,
            SUM(tc.duration_hours) as total_training_hours
        ');
        $this->db->from('training_enrollments te');
        $this->db->join('training_courses tc', 'tc.id = te.course_id', 'left');
        $this->db->join('staff s', 's.id = te.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('te.enrollment_date >=', $start_date);
        $this->db->where('te.enrollment_date <=', $end_date);
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        $this->db->group_by('s.department_id');
        $this->db->order_by('completed_trainings', 'DESC');
        
        return $this->db->get()->result();
    }

    public function getTopPerformingStaff($limit = 10) {
        $this->db->select('
            s.name, s.surname, s.employee_id, d.name as department_name,
            COUNT(te.id) as completed_trainings,
            AVG(te.score) as avg_score,
            SUM(tc.duration_hours) as total_hours
        ');
        $this->db->from('training_enrollments te');
        $this->db->join('training_courses tc', 'tc.id = te.course_id', 'left');
        $this->db->join('staff s', 's.id = te.staff_id', 'left');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->where('te.status', 'completed');
        $this->db->where('te.completion_date >=', date('Y-01-01')); // This year
        $this->db->group_by('te.staff_id');
        $this->db->order_by('avg_score', 'DESC');
        $this->db->order_by('completed_trainings', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }

    // =============================================
    // TRAINING BUDGET & COSTS
    // =============================================

    public function getTrainingBudget($year = null) {
        if (!$year) {
            $year = date('Y');
        }
        
        $this->db->select('
            SUM(tc.cost_per_participant * (
                SELECT COUNT(*) FROM training_enrollments te2 
                WHERE te2.course_id = tc.id AND YEAR(te2.enrollment_date) = ' . $year . '
            )) as total_budget_used,
            COUNT(DISTINCT te.staff_id) as staff_trained,
            SUM(tc.duration_hours) as total_training_hours
        ');
        $this->db->from('training_courses tc');
        $this->db->join('training_enrollments te', 'te.course_id = tc.id AND YEAR(te.enrollment_date) = ' . $year, 'left');
        
        return $this->db->get()->row();
    }

    // =============================================
    // TRAINING REMINDERS & NOTIFICATIONS
    // =============================================

    public function getTrainingReminders() {
        $reminders = [];
        
        // Mandatory training overdue
        $this->db->select('s.id, s.name, s.surname, s.email, tc.name as course_name');
        $this->db->from('staff s');
        $this->db->join('training_courses tc', 'tc.is_mandatory = "yes" AND tc.is_active = "yes"', 'inner');
        $this->db->join('training_enrollments te', 'te.staff_id = s.id AND te.course_id = tc.id', 'left');
        $this->db->where('s.is_active', 'yes');
        $this->db->where('te.id IS NULL OR (te.status != "completed" AND te.enrollment_date <= DATE_SUB(NOW(), INTERVAL 30 DAY))');
        
        $overdue_training = $this->db->get()->result();
        
        if ($overdue_training) {
            $reminders[] = [
                'type' => 'overdue_mandatory_training',
                'title' => 'Overdue Mandatory Training',
                'count' => count($overdue_training),
                'data' => $overdue_training
            ];
        }
        
        // Expiring certifications
        $expiring_certs = $this->getExpiringCertifications(30);
        
        if ($expiring_certs) {
            $reminders[] = [
                'type' => 'expiring_certifications',
                'title' => 'Expiring Certifications',
                'count' => count($expiring_certs),
                'data' => $expiring_certs
            ];
        }
        
        return $reminders;
    }

    // =============================================
    // DATABASE INITIALIZATION
    // =============================================

    public function createTrainingTables() {
        // Training Categories Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `training_categories` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `name` varchar(100) NOT NULL,
                `description` text,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Training Courses Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `training_courses` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `name` varchar(255) NOT NULL,
                `description` text,
                `category_id` int(11) DEFAULT NULL,
                `duration_hours` int(11) NOT NULL DEFAULT 0,
                `is_mandatory` enum('yes','no') DEFAULT 'no',
                `skill_category` varchar(100) DEFAULT NULL,
                `certification_validity_months` int(11) DEFAULT NULL,
                `cost_per_participant` decimal(10,2) DEFAULT 0.00,
                `max_participants` int(11) DEFAULT NULL,
                `prerequisites` text,
                `learning_objectives` text,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `category_id` (`category_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Training Enrollments Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `training_enrollments` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `course_id` int(11) NOT NULL,
                `enrollment_date` date NOT NULL,
                `status` enum('enrolled','in_progress','completed','dropped','cancelled') DEFAULT 'enrolled',
                `completion_date` date DEFAULT NULL,
                `score` decimal(5,2) DEFAULT NULL,
                `grade` varchar(10) DEFAULT NULL,
                `certificate_number` varchar(100) DEFAULT NULL,
                `trainer_notes` text,
                `enrolled_by` int(11) DEFAULT NULL,
                `completed_by` int(11) DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_course` (`staff_id`,`course_id`),
                KEY `staff_id` (`staff_id`),
                KEY `course_id` (`course_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Staff Skill Matrix Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_skill_matrix` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `skill_category` varchar(100) NOT NULL,
                `competency_level` enum('beginner','basic','intermediate','advanced','expert') DEFAULT 'beginner',
                `last_training_date` date DEFAULT NULL,
                `certification_expiry` date DEFAULT NULL,
                `notes` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `staff_skill` (`staff_id`,`skill_category`),
                KEY `staff_id` (`staff_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Training Sessions Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `training_sessions` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `course_id` int(11) NOT NULL,
                `session_name` varchar(255) NOT NULL,
                `session_date` date NOT NULL,
                `start_time` time NOT NULL,
                `end_time` time NOT NULL,
                `location` varchar(255) DEFAULT NULL,
                `trainer_id` int(11) DEFAULT NULL,
                `external_trainer` varchar(255) DEFAULT NULL,
                `max_participants` int(11) DEFAULT NULL,
                `status` enum('scheduled','in_progress','completed','cancelled') DEFAULT 'scheduled',
                `notes` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `course_id` (`course_id`),
                KEY `trainer_id` (`trainer_id`),
                KEY `session_date` (`session_date`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Training Session Attendance Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `training_session_attendance` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `session_id` int(11) NOT NULL,
                `staff_id` int(11) NOT NULL,
                `attendance_status` enum('present','absent','late','excused') DEFAULT 'present',
                `notes` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `session_staff` (`session_id`,`staff_id`),
                KEY `session_id` (`session_id`),
                KEY `staff_id` (`staff_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        return true;
    }

    public function insertDefaultTrainingData() {
        // Insert default training categories
        $categories = [
            ['name' => 'Medical Training', 'description' => 'Medical and clinical training courses'],
            ['name' => 'Safety & Compliance', 'description' => 'Health and safety compliance training'],
            ['name' => 'Leadership', 'description' => 'Leadership and management development'],
            ['name' => 'Technical Skills', 'description' => 'Technical and specialized skills training'],
            ['name' => 'Customer Service', 'description' => 'Patient care and customer service training'],
            ['name' => 'IT & Systems', 'description' => 'Information technology and systems training']
        ];

        foreach ($categories as $category) {
            // Check if category already exists
            $this->db->where('name', $category['name']);
            $existing = $this->db->get('training_categories')->row();
            
            if (!$existing) {
                $this->db->insert('training_categories', $category);
            }
        }

        // Insert some default mandatory courses
        $courses = [
            [
                'name' => 'Health and Safety Orientation',
                'description' => 'Basic health and safety training for all staff',
                'category_id' => 2, // Safety & Compliance
                'duration_hours' => 4,
                'is_mandatory' => 'yes',
                'skill_category' => 'Safety',
                'certification_validity_months' => 12
            ],
            [
                'name' => 'Infection Control',
                'description' => 'Infection prevention and control procedures',
                'category_id' => 1, // Medical Training
                'duration_hours' => 6,
                'is_mandatory' => 'yes',
                'skill_category' => 'Infection Control',
                'certification_validity_months' => 12
            ],
            [
                'name' => 'Patient Confidentiality',
                'description' => 'Patient privacy and confidentiality requirements',
                'category_id' => 2, // Safety & Compliance
                'duration_hours' => 2,
                'is_mandatory' => 'yes',
                'skill_category' => 'Compliance',
                'certification_validity_months' => 24
            ]
        ];

        foreach ($courses as $course) {
            // Check if course already exists
            $this->db->where('name', $course['name']);
            $existing = $this->db->get('training_courses')->row();
            
            if (!$existing) {
                $this->db->insert('training_courses', $course);
            }
        }

        return true;
    }
} function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // TRAINING CATALOG MANAGEMENT
    // =============================================

    public function getTrainingCategories() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('name', 'ASC');
        return $this->db->get('training_categories')->result();
    }

    public function addTrainingCategory($data) {
        return $this->db->insert('training_categories', $data);
    }

    public function getTrainingCourses($category_id = null) {
        $this->db->select('tc.*, tcat.name as category_name');
        $this->db->from('training_courses tc');
        $this->db->join('training_categories tcat', 'tcat.id = tc.category_id', 'left');
        $this->db->where('tc.is_active', 'yes');
        
        if ($category_id) {
            $this->db->where('tc.category_id', $category_id);
        }
        
        $this->db->order_by('tc.name', 'ASC');
        return $this->db->get()->result();
    }

    public function addTrainingCourse($data) {
        return $this->db->insert('training_courses', $data);
    }

    public function updateTrainingCourse($course_id, $data) {
        $this->db->where('id', $course_id);
        return $this->db->update('training_courses', $data);
    }

    public function getCourseById($course_id) {
        $this->db->select('tc.*, tcat.name as category_name');
        $this->db->from('training_courses tc');
        $this->db->join('training_categories tcat', 'tcat.id = tc.category_id', 'left');
        $this->db->where('tc.id', $course_id);
        return $this->db->get()->row();
    }

    public function getMandatoryTrainings() {
        $this->db->select('tc.*, tcat.name as category_name');
        $this->db->from('training_courses tc');
        $this->db->join('training_categories tcat', 'tcat.id = tc.category_id', 'left');
        $this->db->where('tc.is_mandatory', 'yes');
        $this->db->where('tc.is_active', 'yes');
        $this->db->order_by('tc.name', 'ASC');
        return $this->db->get()->result();
    }

    // =============================================
    // TRAINING ENROLLMENT
    // =============================================

    public function enrollStaffInTraining($data) {
        $this->db->trans_start();
        
        // Check if already enrolled
        $this->db->where('staff_id', $data['staff_id']);
        $this->db->where('course_id', $data['course_id']);
        $existing = $this->db->get('training_enrollments')->row();
        
        if ($existing) {
            $this->db->trans_rollback();
            return ['status' => 'error', 'message' => 'Staff already enrolled in this course'];
        }
        
        $enrollment_data = [
            'staff_id' => $data['staff_id'],
            'course_id' => $data['course_id'],
            'enrollment_date' => date('Y-m-d'),
            'status' => 'enrolled',
            'enrolled_by' => $data['enrolled_by'] ?? null,
            'created_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->insert('training_enrollments', $enrollment_data);
        $enrollment_id = $this->db->insert_id();
        
        $this->db->trans_complete();
        
        if ($this->db->trans_status() === FALSE) {
            return ['status' => 'error', 'message' => 'Failed to enroll staff'];
        }
        
        return ['status' => 'success', 'enrollment_id' => $enrollment_id];
    }

    public function bulkEnrollStaff($course_id, $staff_ids, $enrolled_by) {
        $this->db->trans_start();
        
        $successful_enrollments = 0;
        $failed_enrollments = 0;
        
        foreach ($staff_ids as $staff_id) {
            $result = $this->enrollStaffInTraining([
                'staff_id' => $staff_id,
                'course_id' => $course_id,
                'enrolled_by' => $enrolled_by
            ]);
            
            if ($result['status'] == 'success') {
                $successful_enrollments++;
            } else {
                $failed_enrollments++;
            }
        }
        
        $this->db->trans_complete();
        
        return [
            'successful' => $successful_enrollments,
            'failed' => $failed_enrollments,
            'total' => count($staff_ids)
        ];
    }

    public function getStaffEnrollments($staff_id = null, $course_id = null) {
        $this->db->select('te.*, tc.name as course_name, tc.duration_hours, tc.is_mandatory, 
                          s.name as staff_name, s.surname as staff_surname, s.employee_id,
                          tcat.name as category_name');
        $this->db->from('training_enrollments te');
        $this->db->join('training_courses tc', 'tc.id = te.course_id', 'left');
        $this->db->join('training_categories tcat', 'tcat.id = tc.category_id', 'left');
        $this->db->join('staff s', 's.id = te.staff_id', 'left');
        
        if ($staff_id) {
            $this->db->where('te.staff_id', $staff_id);
        }
        
        if ($course_id) {
            $this->db->where('te.course_id', $course_id);
        }
        
        $this->db->order_by('te.enrollment_date', 'DESC');
        return $this->db->get()->result();
    }

    // =============================================
    // TRAINING COMPLETION
    // =============================================

    public function markTrainingComplete($enrollment_id, $completion_data) {
        $this->db->trans_start();
        
        // Update enrollment status
        $update_data = [
            'status' => 'completed',
            'completion_date' => $completion_data['completion_date'],
            'score' => $completion_data['score'] ?? null,
            'grade' => $completion_data['grade'] ?? null,
            'certificate_number' => $completion_data['certificate_number'] ?? null,
            'trainer_notes' => $completion_data['trainer_notes'] ?? null,
            'completed_by' => $completion_data['completed_by'] ?? null,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $enrollment_id);
        $this->db->update('training_enrollments', $update_data);
        
        // Get enrollment details for skill matrix update
        $enrollment = $this->getEnrollmentById($enrollment_id);
        
        if ($enrollment && $enrollment->course_id) {
            // Update staff skill matrix
            $this->updateStaffSkillMatrix($enrollment->staff_id, $enrollment->course_id, $completion_data);
        }
        
        $this->db->trans_complete();
        return $this->db->trans_status();
    }

    public function getEnrollmentById($enrollment_id) {
        $this->db->select('te.*, tc.name as course_name, tc.skill_category, s.name as staff_name, s.surname as staff_surname');
        $this->db->from('training_enrollments te');
        $this->db->join('training_courses tc', 'tc.id = te.course_id', 'left');
        $this->db->join('staff s', 's.id = te.staff_id', 'left');
        $this->db->where('te.id', $enrollment_id);
        return $this->db->get()->row();
    }

    // =============================================
    // SKILL MATRIX MANAGEMENT
    // =============================================

    public function updateStaffSkillMatrix($staff_id, $course_id, $completion_data) {
        // Get course skill information
        $course = $this->getCourseById($course_id);
        
        if (!$course || !$course->skill_category) {
            return false;
        }
        
        // Check if skill matrix entry exists
        $this->db->where('staff_id', $staff_id);
        $this->db->where('skill_category', $course->skill_category);
        $existing_skill = $this->db->get('staff_skill_matrix')->row();
        
        $skill_data = [
            'staff_id' => $staff_id,
            'skill_category' => $course->skill_category,
            'competency_level' => $this->calculateCompetencyLevel($completion_data['score'] ?? 0),
            'last_training_date' => $completion_data['completion_date'],
            'certification_expiry' => $this->calculateCertificationExpiry($course->certification_validity_months ?? 0),
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        if ($existing_skill) {
            $this->db->where('id', $existing_skill->id);
            return $this->db->update('staff_skill_matrix', $skill_data);
        } else {
            $skill_data['created_at'] = date('Y-m-d H:i:s');
            return $this->db->insert('staff_skill_matrix', $skill_data);
        }
    }

    private function calculateCompetencyLevel($score) {
        if ($score >= 90) return 'expert';
        elseif ($score >= 80) return 'advanced';
        elseif ($score >= 70) return 'intermediate';
        elseif ($score >= 60) return 'basic';
        else return 'beginner';
    }

    private function calculateCertificationExpiry($validity_months) {
        if ($validity_months > 0) {
            return date('Y-m-d', strtotime('+' . $validity_months . ' months'));
        }
        return null;
    }

    public