<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * HR Model
 * Handles database operations for HR module
 */
class Hr_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // STAFF MANAGEMENT
    // =============================================

    public function getTotalStaffCount() {
        return $this->db->count_all('staff');
    }

    public function getActiveStaffCount() {
        $this->db->where('is_active', 'yes');
        return $this->db->count_all_results('staff');
    }

    public function getStaffDatatables() {
        $this->_getStaffQuery();
        if($_POST['length'] != -1) {
            $this->db->limit($_POST['length'], $_POST['start']);
        }
        $query = $this->db->get();
        return $query->result();
    }

    public function countAllStaff() {
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');
        return $this->db->count_all_results();
    }

    public function countFilteredStaff() {
        $this->_getStaffQuery();
        return $this->db->count_all_results();
    }

    private function _getStaffQuery() {
        $this->db->select('s.*, d.name as department_name, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');

        // Search functionality
        if(isset($_POST['search']['value']) && !empty($_POST['search']['value'])) {
            $search = $_POST['search']['value'];
            $this->db->group_start();
            $this->db->like('s.name', $search);
            $this->db->or_like('s.surname', $search);
            $this->db->or_like('s.employee_id', $search);
            $this->db->or_like('s.email', $search);
            $this->db->or_like('d.name', $search);
            $this->db->or_like('r.name', $search);
            $this->db->group_end();
        }

        // Ordering
        if(isset($_POST['order'])) {
            $column_order = ['s.employee_id', 's.name', 'd.name', 'r.name', 's.employment_type', 's.date_of_joining', 's.is_active'];
            $this->db->order_by($column_order[$_POST['order']['0']['column']], $_POST['order']['0']['dir']);
        } else {
            $this->db->order_by('s.id', 'DESC');
        }
    }

    public function addStaff($data) {
        $this->db->insert('staff', $data);
        return $this->db->insert_id();
    }

    public function updateStaff($staff_id, $data) {
        $this->db->where('id', $staff_id);
        return $this->db->update('staff', $data);
    }

    public function getStaffById($staff_id) {
        $this->db->where('id', $staff_id);
        return $this->db->get('staff')->row();
    }

    public function getStaffDetailsById($staff_id) {
        $this->db->select('s.*, d.name as department_name, r.name as role_name, r.id as role_id, d.id as department_id');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');
        $this->db->where('s.id', $staff_id);
        return $this->db->get()->row();
    }

    public function getRecentHires($limit = 5) {
        $this->db->select('s.*, d.name as department_name, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');
        $this->db->where('s.date_of_joining >=', date('Y-m-d', strtotime('-30 days')));
        $this->db->order_by('s.date_of_joining', 'DESC');
        $this->db->limit($limit);
        return $this->db->get()->result();
    }

    public function getExpiringContracts($days = 30) {
        $this->db->where('contract_end_date <=', date('Y-m-d', strtotime('+' . $days . ' days')));
        $this->db->where('contract_end_date >=', date('Y-m-d'));
        $this->db->where('is_active', 'yes');
        return $this->db->count_all_results('staff');
    }

    // =============================================
    // DEPARTMENTS & ROLES
    // =============================================

    public function getDepartments() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('name', 'ASC');
        return $this->db->get('departments')->result();
    }

    public function getRoles() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('name', 'ASC');
        return $this->db->get('roles')->result();
    }

    public function getEmploymentTypes() {
        return [
            'permanent' => 'Permanent',
            'contract' => 'Contract',
            'temporary' => 'Temporary',
            'internship' => 'Internship',
            'consultant' => 'Consultant'
        ];
    }

    // =============================================
    // STAFF DOCUMENTS
    // =============================================

    public function getStaffDocuments($staff_id) {
        $this->db->where('staff_id', $staff_id);
        $this->db->order_by('created_at', 'DESC');
        return $this->db->get('staff_documents')->result();
    }

    public function addStaffDocument($data) {
        $this->db->insert('staff_documents', $data);
        return $this->db->insert_id();
    }

    public function deleteStaffDocument($doc_id) {
        $this->db->where('id', $doc_id);
        return $this->db->delete('staff_documents');
    }

    // =============================================
    // EMPLOYMENT HISTORY
    // =============================================

    public function getEmploymentHistory($staff_id) {
        $this->db->select('eh.*, d.name as department_name, r.name as role_name');
        $this->db->from('employment_history eh');
        $this->db->join('departments d', 'd.id = eh.department_id', 'left');
        $this->db->join('roles r', 'r.id = eh.role_id', 'left');
        $this->db->where('eh.staff_id', $staff_id);
        $this->db->order_by('eh.start_date', 'DESC');
        return $this->db->get()->result();
    }

    public function addEmploymentHistory($data) {
        $this->db->insert('employment_history', $data);
        return $this->db->insert_id();
    }

    // =============================================
    // COMPLIANCE & ALERTS
    // =============================================

    public function getComplianceAlerts() {
        $alerts = [];
        
        // Expiring contracts
        $this->db->select('COUNT(*) as count');
        $this->db->where('contract_end_date <=', date('Y-m-d', strtotime('+30 days')));
        $this->db->where('contract_end_date >=', date('Y-m-d'));
        $this->db->where('is_active', 'yes');
        $expiring_contracts = $this->db->get('staff')->row()->count;
        
        if ($expiring_contracts > 0) {
            $alerts[] = [
                'type' => 'warning',
                'title' => 'Expiring Contracts',
                'message' => $expiring_contracts . ' staff contracts expiring in next 30 days',
                'count' => $expiring_contracts
            ];
        }

        // Missing documents
        $this->db->select('s.id, s.name, s.surname');
        $this->db->from('staff s');
        $this->db->where('s.is_active', 'yes');
        $this->db->where('s.id NOT IN (SELECT DISTINCT staff_id FROM staff_documents WHERE document_type = "contract")', NULL, FALSE);
        $missing_contracts = $this->db->get()->result();
        
        if (count($missing_contracts) > 0) {
            $alerts[] = [
                'type' => 'danger',
                'title' => 'Missing Contracts',
                'message' => count($missing_contracts) . ' staff members missing employment contracts',
                'count' => count($missing_contracts)
            ];
        }

        return $alerts;
    }

    // =============================================
    // ACTIVITY LOGGING
    // =============================================

    public function logActivity($data) {
        $data['created_at'] = date('Y-m-d H:i:s');
        $this->db->insert('hr_activity_log', $data);
        return $this->db->insert_id();
    }

    public function getRecentActivities($limit = 10) {
        $this->db->select('hal.*, s.name as staff_name, s.surname as staff_surname, u.name as performed_by_name');
        $this->db->from('hr_activity_log hal');
        $this->db->join('staff s', 's.id = hal.staff_id', 'left');
        $this->db->join('staff u', 'u.id = hal.performed_by', 'left');
        $this->db->order_by('hal.created_at', 'DESC');
        $this->db->limit($limit);
        return $this->db->get()->result();
    }

    // =============================================
    // SALARY STRUCTURES
    // =============================================

    public function getSalaryComponents() {
        $this->db->where('is_active', 'yes');
        $this->db->order_by('display_order', 'ASC');
        return $this->db->get('salary_components')->result();
    }

    public function getStaffSalaryStructure($staff_id) {
        $this->db->select('sss.*, sc.name as component_name, sc.type as component_type');
        $this->db->from('staff_salary_structure sss');
        $this->db->join('salary_components sc', 'sc.id = sss.component_id', 'left');
        $this->db->where('sss.staff_id', $staff_id);
        $this->db->where('sss.is_active', 'yes');
        $this->db->order_by('sc.display_order', 'ASC');
        return $this->db->get()->result();
    }

    public function updateStaffSalaryStructure($staff_id, $components) {
        // Deactivate existing components
        $this->db->where('staff_id', $staff_id);
        $this->db->update('staff_salary_structure', ['is_active' => 'no', 'updated_at' => date('Y-m-d H:i:s')]);

        // Insert new components
        foreach ($components as $component) {
            $data = [
                'staff_id' => $staff_id,
                'component_id' => $component['component_id'],
                'amount' => $component['amount'],
                'is_active' => 'yes',
                'created_at' => date('Y-m-d H:i:s')
            ];
            $this->db->insert('staff_salary_structure', $data);
        }

        return true;
    }

    // =============================================
    // PERFORMANCE MANAGEMENT
    // =============================================

    public function getPerformanceReviews($staff_id = null) {
        $this->db->select('pr.*, s.name as staff_name, s.surname as staff_surname, r.name as reviewer_name');
        $this->db->from('performance_reviews pr');
        $this->db->join('staff s', 's.id = pr.staff_id', 'left');
        $this->db->join('staff r', 'r.id = pr.reviewer_id', 'left');
        
        if ($staff_id) {
            $this->db->where('pr.staff_id', $staff_id);
        }
        
        $this->db->order_by('pr.review_period_start', 'DESC');
        return $this->db->get()->result();
    }

    public function addPerformanceReview($data) {
        $this->db->insert('performance_reviews', $data);
        return $this->db->insert_id();
    }

    public function updatePerformanceReview($review_id, $data) {
        $this->db->where('id', $review_id);
        return $this->db->update('performance_reviews', $data);
    }

    // =============================================
    // PROBATION TRACKING
    // =============================================

    public function getProbationaryStaff() {
        $this->db->select('s.*, d.name as department_name, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');
        $this->db->where('s.probation_end_date >=', date('Y-m-d'));
        $this->db->where('s.is_active', 'yes');
        $this->db->order_by('s.probation_end_date', 'ASC');
        return $this->db->get()->result();
    }

    public function getExpiringProbations($days = 7) {
        $this->db->select('s.*, d.name as department_name, r.name as role_name');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->join('roles r', 'r.id = s.role_id', 'left');
        $this->db->where('s.probation_end_date <=', date('Y-m-d', strtotime('+' . $days . ' days')));
        $this->db->where('s.probation_end_date >=', date('Y-m-d'));
        $this->db->where('s.is_active', 'yes');
        $this->db->order_by('s.probation_end_date', 'ASC');
        return $this->db->get()->result();
    }

    // =============================================
    // REPORTS
    // =============================================

    public function getHeadcountReport($department_id = null, $start_date = null, $end_date = null) {
        $this->db->select('d.name as department, COUNT(s.id) as total_staff, 
                          SUM(CASE WHEN s.is_active = "yes" THEN 1 ELSE 0 END) as active_staff,
                          SUM(CASE WHEN s.gender = "male" THEN 1 ELSE 0 END) as male_count,
                          SUM(CASE WHEN s.gender = "female" THEN 1 ELSE 0 END) as female_count');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        
        if ($department_id) {
            $this->db->where('s.department_id', $department_id);
        }
        
        if ($start_date && $end_date) {
            $this->db->where('s.date_of_joining >=', $start_date);
            $this->db->where('s.date_of_joining <=', $end_date);
        }
        
        $this->db->group_by('s.department_id');
        $this->db->order_by('d.name', 'ASC');
        return $this->db->get()->result();
    }

    public function getAttritionReport($year = null) {
        if (!$year) {
            $year = date('Y');
        }

        $this->db->select('d.name as department, 
                          COUNT(s.id) as total_staff,
                          SUM(CASE WHEN s.date_of_leaving IS NOT NULL AND YEAR(s.date_of_leaving) = ' . $year . ' THEN 1 ELSE 0 END) as left_staff');
        $this->db->from('staff s');
        $this->db->join('departments d', 'd.id = s.department_id', 'left');
        $this->db->group_by('s.department_id');
        $this->db->order_by('d.name', 'ASC');
        return $this->db->get()->result();
    }

    // =============================================
    // DATABASE INITIALIZATION
    // =============================================

    public function createHRTables() {
        // Staff Documents Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_documents` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `document_type` varchar(100) NOT NULL,
                `document_title` varchar(255) NOT NULL,
                `file_name` varchar(255) NOT NULL,
                `file_path` varchar(500) NOT NULL,
                `file_size` int(11) DEFAULT NULL,
                `mime_type` varchar(100) DEFAULT NULL,
                `upload_date` date NOT NULL,
                `expiry_date` date DEFAULT NULL,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Employment History Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `employment_history` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `department_id` int(11) DEFAULT NULL,
                `role_id` int(11) DEFAULT NULL,
                `employment_type` varchar(50) DEFAULT NULL,
                `start_date` date NOT NULL,
                `end_date` date DEFAULT NULL,
                `salary` decimal(10,2) DEFAULT NULL,
                `reason_for_change` text,
                `notes` text,
                `is_current` enum('yes','no') DEFAULT 'no',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // HR Activity Log Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `hr_activity_log` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `activity_type` varchar(100) NOT NULL,
                `staff_id` int(11) DEFAULT NULL,
                `description` text NOT NULL,
                `performed_by` int(11) DEFAULT NULL,
                `ip_address` varchar(45) DEFAULT NULL,
                `user_agent` text,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `performed_by` (`performed_by`),
                KEY `activity_type` (`activity_type`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Salary Components Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `salary_components` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `name` varchar(100) NOT NULL,
                `short_code` varchar(20) NOT NULL,
                `type` enum('earning','deduction') NOT NULL,
                `calculation_type` enum('fixed','percentage','formula') DEFAULT 'fixed',
                `is_statutory` enum('yes','no') DEFAULT 'no',
                `is_taxable` enum('yes','no') DEFAULT 'yes',
                `display_order` int(11) DEFAULT 0,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                UNIQUE KEY `short_code` (`short_code`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Staff Salary Structure Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `staff_salary_structure` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `component_id` int(11) NOT NULL,
                `amount` decimal(10,2) NOT NULL DEFAULT 0.00,
                `percentage` decimal(5,2) DEFAULT NULL,
                `effective_from` date DEFAULT NULL,
                `effective_to` date DEFAULT NULL,
                `is_active` enum('yes','no') DEFAULT 'yes',
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `component_id` (`component_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        // Performance Reviews Table
        $this->db->query("
            CREATE TABLE IF NOT EXISTS `performance_reviews` (
                `id` int(11) NOT NULL AUTO_INCREMENT,
                `staff_id` int(11) NOT NULL,
                `reviewer_id` int(11) NOT NULL,
                `review_period_start` date NOT NULL,
                `review_period_end` date NOT NULL,
                `review_type` varchar(50) DEFAULT 'annual',
                `overall_rating` decimal(3,2) DEFAULT NULL,
                `goals_achievement` text,
                `strengths` text,
                `areas_for_improvement` text,
                `development_plan` text,
                `reviewer_comments` text,
                `staff_comments` text,
                `status` enum('draft','submitted','approved','completed') DEFAULT 'draft',
                `review_date` date DEFAULT NULL,
                `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
                `updated_at` timestamp NULL DEFAULT NULL ON UPDATE CURRENT_TIMESTAMP,
                PRIMARY KEY (`id`),
                KEY `staff_id` (`staff_id`),
                KEY `reviewer_id` (`reviewer_id`)
            ) ENGINE=InnoDB DEFAULT CHARSET=utf8;
        ");

        return true;
    }

    public function insertDefaultSalaryComponents() {
        $components = [
            ['name' => 'Basic Salary', 'short_code' => 'BASIC', 'type' => 'earning', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'yes', 'display_order' => 1],
            ['name' => 'House Allowance', 'short_code' => 'HOUSE', 'type' => 'earning', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'yes', 'display_order' => 2],
            ['name' => 'Transport Allowance', 'short_code' => 'TRANSPORT', 'type' => 'earning', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'yes', 'display_order' => 3],
            ['name' => 'Medical Allowance', 'short_code' => 'MEDICAL', 'type' => 'earning', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'no', 'display_order' => 4],
            ['name' => 'Overtime', 'short_code' => 'OVERTIME', 'type' => 'earning', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'yes', 'display_order' => 5],
            ['name' => 'PAYE Tax', 'short_code' => 'PAYE', 'type' => 'deduction', 'calculation_type' => 'percentage', 'is_statutory' => 'yes', 'is_taxable' => 'no', 'display_order' => 6],
            ['name' => 'NHIF', 'short_code' => 'NHIF', 'type' => 'deduction', 'calculation_type' => 'fixed', 'is_statutory' => 'yes', 'is_taxable' => 'no', 'display_order' => 7],
            ['name' => 'NSSF', 'short_code' => 'NSSF', 'type' => 'deduction', 'calculation_type' => 'percentage', 'is_statutory' => 'yes', 'is_taxable' => 'no', 'display_order' => 8],
            ['name' => 'Affordable Housing Levy', 'short_code' => 'AHL', 'type' => 'deduction', 'calculation_type' => 'percentage', 'is_statutory' => 'yes', 'is_taxable' => 'no', 'display_order' => 9],
            ['name' => 'Staff Loan', 'short_code' => 'LOAN', 'type' => 'deduction', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'no', 'display_order' => 10],
            ['name' => 'Advance Salary', 'short_code' => 'ADVANCE', 'type' => 'deduction', 'calculation_type' => 'fixed', 'is_statutory' => 'no', 'is_taxable' => 'no', 'display_order' => 11]
        ];

        foreach ($components as $component) {
            // Check if component already exists
            $this->db->where('short_code', $component['short_code']);
            $existing = $this->db->get('salary_components')->row();
            
            if (!$existing) {
                $this->db->insert('salary_components', $component);
            }
        }

        return true;
    }
}