<?php
defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Front Office Model
 * Handles all front office related database operations including copay, cash management, and dashboard stats
 */
class Front_office_model extends CI_Model {

    public function __construct() {
        parent::__construct();
        $this->load->database();
    }

    // =============================================
    // DASHBOARD STATISTICS METHODS
    // =============================================

    /**
     * Get comprehensive dashboard statistics for a staff member
     */
    public function getDashboardStats($staff_id, $date = null) {
        if (!$date) {
            $date = date('Y-m-d');
        }

        try {
            $stats = [];

            // Get patients registered today by this staff
            $stats['patients_registered'] = $this->getPatientsRegisteredCount($staff_id, $date);
            
            // Get cash collected today
            $stats['cash_collected'] = $this->getCashCollectedAmount($staff_id, $date);
            
            // Get copay collections count
            $stats['copay_collections'] = $this->getCopayCollectionsCount($staff_id, $date);
            
            // Get average processing time
            $stats['avg_processing_time'] = $this->calculateAverageProcessingTime($staff_id, $date);
            
            // Get payment method breakdown
            $payment_breakdown = $this->getPaymentMethodBreakdown($staff_id, $date);
            $stats = array_merge($stats, $payment_breakdown);
            
            // Get shift information
            $stats['shift_info'] = $this->getCurrentShiftInfo($staff_id, $date);
            
            return $stats;
            
        } catch (Exception $e) {
            log_message('error', 'Front Office Model - getDashboardStats error: ' . $e->getMessage());
            return $this->getEmptyStats();
        }
    }

    /**
     * Get patients registered count for staff and date
     */
    private function getPatientsRegisteredCount($staff_id, $date) {
        $this->db->where('registered_by_staff_id', $staff_id);
        $this->db->where('DATE(created_at)', $date);
        return $this->db->count_all_results('patients');
    }

    /**
     * Get cash collected amount for staff and date
     */
    private function getCashCollectedAmount($staff_id, $date) {
        $this->db->select('COALESCE(SUM(collected_amount), 0) as total_cash');
        $this->db->where('collected_by_staff_id', $staff_id);
        $this->db->where('DATE(collection_date)', $date);
        $this->db->where('payment_method', 'cash');
        $this->db->where('status', 'collected');
        
        $query = $this->db->get('copay_collections');
        if ($query && $query->num_rows() > 0) {
            $result = $query->row();
            return floatval($result->total_cash);
        }
        return 0.00;
    }

    /**
     * Get copay collections count for staff and date
     */
    private function getCopayCollectionsCount($staff_id, $date) {
        $this->db->where('collected_by_staff_id', $staff_id);
        $this->db->where('DATE(collection_date)', $date);
        $this->db->where('status', 'collected');
        return $this->db->count_all_results('copay_collections');
    }

    /**
     * Calculate average processing time
     */
    private function calculateAverageProcessingTime($staff_id, $date) {
        // Check if copay_collections table exists
        if (!$this->db->table_exists('copay_collections')) {
            return 0;
        }

        $this->db->select('AVG(TIMESTAMPDIFF(MINUTE, created_at, updated_at)) as avg_time');
        $this->db->where('collected_by_staff_id', $staff_id);
        $this->db->where('DATE(collection_date)', $date);
        $this->db->where('status', 'collected');
        $this->db->where('updated_at IS NOT NULL');
        
        $query = $this->db->get('copay_collections');
        if ($query && $query->num_rows() > 0) {
            $result = $query->row();
            return round(floatval($result->avg_time ?: 0));
        }
        return 0;
    }

    /**
     * Get payment method breakdown
     */
    private function getPaymentMethodBreakdown($staff_id, $date) {
        $breakdown = [
            'cash_count' => 0, 'cash_total' => 0.00,
            'card_count' => 0, 'card_total' => 0.00,
            'mobile_money_count' => 0, 'mpesa_total' => 0.00,
            'cheque_count' => 0, 'cheque_total' => 0.00
        ];

        // Check if copay_collections table exists
        if (!$this->db->table_exists('copay_collections')) {
            return $breakdown;
        }

        $payment_methods = ['cash', 'card', 'mobile_money', 'cheque'];
        
        foreach ($payment_methods as $method) {
            $this->db->select('COUNT(*) as count, COALESCE(SUM(collected_amount), 0) as total');
            $this->db->where('collected_by_staff_id', $staff_id);
            $this->db->where('DATE(collection_date)', $date);
            $this->db->where('payment_method', $method);
            $this->db->where('status', 'collected');
            
            $query = $this->db->get('copay_collections');
            if ($query && $query->num_rows() > 0) {
                $result = $query->row();
                $breakdown[$method . '_count'] = intval($result->count);
                $breakdown[$method . '_total'] = floatval($result->total);
            }
            
            // Handle mobile_money as mpesa for display
            if ($method === 'mobile_money') {
                $breakdown['mpesa_count'] = $breakdown['mobile_money_count'];
            }
        }

        return $breakdown;
    }

    /**
     * Get current shift information
     */
    private function getCurrentShiftInfo($staff_id, $date) {
        // Check if table exists
        if (!$this->db->table_exists('front_office_cash_tracking')) {
            return null;
        }

        $this->db->where('staff_id', $staff_id);
        $this->db->where('shift_date', $date);
        $query = $this->db->get('front_office_cash_tracking');
        
        if ($query && $query->num_rows() > 0) {
            return $query->row_array();
        }
        return null;
    }

    /**
     * Return empty stats structure
     */
    private function getEmptyStats() {
        return [
            'patients_registered' => 0,
            'cash_collected' => 0.00,
            'copay_collections' => 0,
            'avg_processing_time' => 0,
            'cash_count' => 0, 'cash_total' => 0.00,
            'card_count' => 0, 'card_total' => 0.00,
            'mpesa_count' => 0, 'mpesa_total' => 0.00,
            'cheque_count' => 0, 'cheque_total' => 0.00,
            'shift_info' => null
        ];
    }

    // =============================================
    // COPAY COLLECTION METHODS
    // =============================================

    /**
     * Get recent copay collections
     */
    public function getRecentCopays($staff_id, $date = null, $limit = 10) {
        if (!$date) {
            $date = date('Y-m-d');
        }

        // Check if table exists
        if (!$this->db->table_exists('copay_collections')) {
            return [];
        }

        try {
            $this->db->select('cc.*, p.patient_name, o.organisation_name');
            $this->db->from('copay_collections cc');
            $this->db->join('patients p', 'p.id = cc.patient_id', 'left');
            $this->db->join('organisation o', 'o.id = p.organisation_id', 'left');
            $this->db->where('cc.collected_by_staff_id', $staff_id);
            $this->db->where('DATE(cc.collection_date)', $date);
            $this->db->where('cc.status', 'collected');
            $this->db->order_by('cc.collection_date', 'DESC');
            $this->db->limit($limit);

            $query = $this->db->get();
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Front Office Model - getRecentCopays error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Process copay collection
     */
    public function processCopayCollection($patient_id, $amount_received, $payment_method, $staff_id, $notes = null) {
        try {
            $this->db->trans_start();

            // Get patient details
            $patient = $this->getPatientDetails($patient_id);
            if (!$patient) {
                throw new Exception('Patient not found');
            }

            // Generate receipt number
            $receipt_number = $this->generateReceiptNumber($patient_id);

            // Prepare copay data
            $copay_data = [
                'patient_id' => $patient_id,
                'total_service_amount' => $patient->total_service_amount ?: 0,
                'insurance_coverage_amount' => $patient->insurance_amount ?: 0,
                'copay_amount' => $patient->copay_amount ?: $amount_received,
                'collected_amount' => $amount_received,
                'payment_method' => $payment_method,
                'receipt_number' => $receipt_number,
                'collection_date' => date('Y-m-d H:i:s'),
                'collected_by_staff_id' => $staff_id,
                'status' => 'collected',
                'notes' => $notes
            ];

            // Insert copay collection record if table exists
            if ($this->db->table_exists('copay_collections')) {
                $this->db->insert('copay_collections', $copay_data);
            }

            // Update patient copay status
            $this->db->where('id', $patient_id);
            $this->db->update('patients', ['copay_status' => 'collected']);

            // Update cash tracking if cash payment
            if ($payment_method === 'cash') {
                $this->updateCashTracking($staff_id, $amount_received, 'received');
            }

            // Update payment method summary
            $this->updatePaymentMethodSummary($staff_id, $payment_method, $amount_received);

            // Log activity
            $this->logActivity($staff_id, $patient_id, 'copay_collection', 
                'Copay collected via ' . $payment_method . ' for ' . $patient->patient_name, 
                $amount_received, $payment_method, $receipt_number);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Database transaction failed');
            }

            return [
                'status' => 'success',
                'receipt_number' => $receipt_number,
                'amount' => $amount_received
            ];
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Process copay collection failed: ' . $e->getMessage());
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Generate unique receipt number
     */
    private function generateReceiptNumber($patient_id) {
        return 'CPY-' . date('Ymd') . '-' . $patient_id . '-' . substr(uniqid(), -6);
    }

    /**
     * Search patients for copay collection
     */
    public function searchPatientsForCopay($search_term) {
        try {
            $this->db->select('p.id, p.patient_name, p.mobileno, p.copay_amount, p.total_service_amount, p.insurance_amount, o.organisation_name');
            $this->db->from('patients p');
            $this->db->join('organisation o', 'o.id = p.organisation_id', 'left');
            
            // Search conditions
            $this->db->group_start();
            $this->db->like('p.patient_name', $search_term);
            $this->db->or_like('p.mobileno', $search_term);
            $this->db->or_like('p.patient_unique_id', $search_term);
            $this->db->group_end();
            
            // Optional: filter for pending copay only if column exists
            if ($this->db->field_exists('copay_status', 'patients')) {
                $this->db->where('p.copay_status', 'pending');
            }
            
            $this->db->order_by('p.created_at', 'DESC');
            $this->db->limit(10);

            $query = $this->db->get();
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Search patients for copay error: ' . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // CASH MANAGEMENT METHODS
    // =============================================

    /**
     * Start staff shift
     */
    public function startShift($staff_id, $opening_cash) {
        try {
            $this->db->trans_start();

            // Check if shift table exists
            if (!$this->db->table_exists('front_office_cash_tracking')) {
                log_message('info', 'Front office cash tracking table does not exist, creating shift record in alternative way');
                return $this->createSimpleShiftRecord($staff_id, $opening_cash);
            }

            // Check if shift already exists for today
            $this->db->where('staff_id', $staff_id);
            $this->db->where('shift_date', date('Y-m-d'));
            $existing_shift = $this->db->get('front_office_cash_tracking')->row();

            if ($existing_shift) {
                throw new Exception('Shift already started for today');
            }

            // Insert new shift record
            $shift_data = [
                'staff_id' => $staff_id,
                'shift_date' => date('Y-m-d'),
                'shift_start_time' => date('H:i:s'),
                'opening_cash_amount' => $opening_cash,
                'shift_status' => 'active',
                'created_at' => date('Y-m-d H:i:s')
            ];

            $this->db->insert('front_office_cash_tracking', $shift_data);

            // Log activity
            $this->logActivity($staff_id, null, 'cash_reconciliation', 
                'Shift started with opening cash: KES ' . number_format($opening_cash, 2), 
                $opening_cash);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Database transaction failed');
            }

            return true;
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Start shift failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Create simple shift record when table doesn't exist
     */
    private function createSimpleShiftRecord($staff_id, $opening_cash) {
        // Log in activity log as alternative
        $this->logActivity($staff_id, null, 'cash_reconciliation', 
            'Shift started with opening cash: KES ' . number_format($opening_cash, 2), 
            $opening_cash);
        return true;
    }

    /**
     * End staff shift
     */
    public function endShift($staff_id, $closing_cash) {
        try {
            $this->db->trans_start();

            // Check if table exists
            if (!$this->db->table_exists('front_office_cash_tracking')) {
                return $this->endSimpleShift($staff_id, $closing_cash);
            }

            // Get current shift data
            $this->db->where('staff_id', $staff_id);
            $this->db->where('shift_date', date('Y-m-d'));
            $this->db->where('shift_status', 'active');
            $shift = $this->db->get('front_office_cash_tracking')->row();

            if (!$shift) {
                throw new Exception('No active shift found');
            }

            // Calculate variance
            $expected_cash = $shift->opening_cash_amount + $shift->total_cash_received - $shift->total_cash_dispensed;
            $variance = $closing_cash - $expected_cash;

            // Update shift record
            $update_data = [
                'shift_end_time' => date('H:i:s'),
                'closing_cash_amount' => $closing_cash,
                'variance_amount' => $variance,
                'shift_status' => 'completed',
                'reconciled_by' => $staff_id,
                'reconciled_at' => date('Y-m-d H:i:s')
            ];

            $this->db->where('id', $shift->id);
            $this->db->update('front_office_cash_tracking', $update_data);

            // Log activity
            $this->logActivity($staff_id, null, 'cash_reconciliation', 
                'Shift ended. Closing cash: KES ' . number_format($closing_cash, 2) . ', Variance: KES ' . number_format($variance, 2), 
                $variance);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Database transaction failed');
            }

            return [
                'status' => true,
                'variance' => $variance,
                'report_url' => base_url('admin/front_office/shift_report/' . $shift->id)
            ];
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'End shift failed: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * End simple shift when table doesn't exist
     */
    private function endSimpleShift($staff_id, $closing_cash) {
        $this->logActivity($staff_id, null, 'cash_reconciliation', 
            'Shift ended. Closing cash: KES ' . number_format($closing_cash, 2), 
            $closing_cash);
        return ['status' => true, 'variance' => 0];
    }

    /**
     * Update cash tracking
     */
    private function updateCashTracking($staff_id, $amount, $type = 'received') {
        if (!$this->db->table_exists('front_office_cash_tracking')) {
            return;
        }

        $field = ($type === 'received') ? 'total_cash_received' : 'total_cash_dispensed';
        
        $this->db->where('staff_id', $staff_id);
        $this->db->where('shift_date', date('Y-m-d'));
        $this->db->set($field, $field . ' + ' . $amount, FALSE);
        
        if ($type === 'received') {
            $this->db->set('copay_collections_amount', 'copay_collections_amount + ' . $amount, FALSE);
            $this->db->set('copay_collections_count', 'copay_collections_count + 1', FALSE);
        }
        
        $this->db->update('front_office_cash_tracking');
    }

    // =============================================
    // ACTIVITY LOG METHODS
    // =============================================

    /**
     * Get activity log for staff
     */
    public function getActivityLog($staff_id, $date = null, $limit = 20) {
        if (!$date) {
            $date = date('Y-m-d');
        }

        try {
            // Check if activity log table exists
            if (!$this->db->table_exists('front_office_activity_log')) {
                return $this->getAlternativeActivityLog($staff_id, $date, $limit);
            }

            $this->db->select('fal.*, p.patient_name');
            $this->db->from('front_office_activity_log fal');
            $this->db->join('patients p', 'p.id = fal.patient_id', 'left');
            $this->db->where('fal.staff_id', $staff_id);
            $this->db->where('DATE(fal.activity_date)', $date);
            $this->db->order_by('fal.activity_date', 'DESC');
            $this->db->limit($limit);

            $query = $this->db->get();
            if ($query && $query->num_rows() > 0) {
                $results = $query->result_array();
                return $this->formatActivityResults($results);
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Get activity log error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get alternative activity log from other sources
     */
    private function getAlternativeActivityLog($staff_id, $date, $limit) {
        $activities = [];
        
        // Get recent patient registrations
        if ($this->db->table_exists('patients')) {
            $this->db->select('patient_name, created_at');
            $this->db->where('registered_by_staff_id', $staff_id);
            $this->db->where('DATE(created_at)', $date);
            $this->db->order_by('created_at', 'DESC');
            $this->db->limit($limit);
            
            $registrations = $this->db->get('patients');
            if ($registrations && $registrations->num_rows() > 0) {
                foreach ($registrations->result_array() as $reg) {
                    $activities[] = [
                        'time' => date('H:i:s', strtotime($reg['created_at'])),
                        'activity' => 'Patient Registration',
                        'patient_name' => $reg['patient_name'],
                        'amount' => 0,
                        'payment_method' => 'N/A',
                        'status' => 'Completed'
                    ];
                }
            }
        }
        
        return array_slice($activities, 0, $limit);
    }

    /**
     * Format activity results for display
     */
    private function formatActivityResults($results) {
        foreach ($results as &$activity) {
            $activity['time'] = date('H:i:s', strtotime($activity['activity_date']));
            $activity['activity'] = $activity['activity_description'];
            $activity['patient_name'] = $activity['patient_name'] ?: 'N/A';
            $activity['status'] = 'Completed';
            $activity['amount'] = $activity['amount'] ?: 0;
            $activity['payment_method'] = $activity['payment_method'] ?: 'N/A';
        }
        return $results;
    }

    /**
     * Log front office activity
     */
    private function logActivity($staff_id, $patient_id, $activity_type, $description, $amount = 0, $payment_method = null, $reference_number = null) {
        try {
            // Check if activity log table exists
            if (!$this->db->table_exists('front_office_activity_log')) {
                log_message('info', 'Activity logged: ' . $description);
                return;
            }

            $activity_data = [
                'staff_id' => $staff_id,
                'patient_id' => $patient_id,
                'activity_type' => $activity_type,
                'activity_description' => $description,
                'amount' => $amount,
                'payment_method' => $payment_method,
                'reference_number' => $reference_number,
                'activity_date' => date('Y-m-d H:i:s'),
                'ip_address' => $this->input->ip_address(),
                'user_agent' => substr($this->input->user_agent(), 0, 255)
            ];

            $this->db->insert('front_office_activity_log', $activity_data);
            
        } catch (Exception $e) {
            log_message('error', 'Log activity error: ' . $e->getMessage());
        }
    }

    // =============================================
    // HELPER METHODS
    // =============================================

    /**
     * Get patient details
     */
    private function getPatientDetails($patient_id) {
        try {
            $this->db->select('p.*, o.organisation_name');
            $this->db->from('patients p');
            $this->db->join('organisation o', 'o.id = p.organisation_id', 'left');
            $this->db->where('p.id', $patient_id);
            
            $query = $this->db->get();
            if ($query && $query->num_rows() > 0) {
                return $query->row();
            }
            return null;
            
        } catch (Exception $e) {
            log_message('error', 'Get patient details error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Update payment method summary
     */
    private function updatePaymentMethodSummary($staff_id, $payment_method, $amount) {
        try {
            // Check if table exists
            if (!$this->db->table_exists('payment_method_summary')) {
                return;
            }

            $summary_data = [
                'staff_id' => $staff_id,
                'transaction_date' => date('Y-m-d'),
                'payment_method' => $payment_method,
                'transaction_count' => 1,
                'total_amount' => $amount
            ];

            // Check if record exists
            $this->db->where('staff_id', $staff_id);
            $this->db->where('transaction_date', date('Y-m-d'));
            $this->db->where('payment_method', $payment_method);
            $existing = $this->db->get('payment_method_summary')->row();

            if ($existing) {
                // Update existing record
                $this->db->where('id', $existing->id);
                $this->db->set('transaction_count', 'transaction_count + 1', FALSE);
                $this->db->set('total_amount', 'total_amount + ' . $amount, FALSE);
                $this->db->update('payment_method_summary');
            } else {
                // Insert new record
                $this->db->insert('payment_method_summary', $summary_data);
            }
            
        } catch (Exception $e) {
            log_message('error', 'Update payment method summary error: ' . $e->getMessage());
        }
    }

    /**
     * Get active insurance providers
     */
    public function getActiveInsuranceProviders() {
        try {
            $this->db->select('id, organisation_name, code');
            
            // Check if new columns exist
            if ($this->db->field_exists('default_coverage_percentage', 'organisation')) {
                $this->db->select('id, organisation_name, code, default_coverage_percentage, schemes_offered');
                $this->db->where('is_active', 1);
            }
            
            $this->db->order_by('organisation_name', 'ASC');
            
            $query = $this->db->get('organisation');
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Get active insurance providers error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get front office staff list
     */
    public function getFrontOfficeStaff() {
        try {
            $this->db->select('s.id, s.name, s.employee_id, r.name as role_name');
            $this->db->from('staff s');
            $this->db->join('roles r', 'r.id = s.role_id', 'left');
            $this->db->where('s.is_active', 1);
            $this->db->order_by('s.name', 'ASC');
            
            $query = $this->db->get();
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Get front office staff error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Generate daily report
     */
    public function generateDailyReport($date, $staff_id = null) {
        try {
            $report = [];
            $staff_condition = $staff_id ? " AND staff_id = " . intval($staff_id) : "";

            // Cash tracking summary
            if ($this->db->table_exists('front_office_cash_tracking')) {
                $cash_query = "
                    SELECT 
                        s.name as staff_name,
                        s.employee_id,
                        foct.*
                    FROM front_office_cash_tracking foct
                    LEFT JOIN staff s ON s.id = foct.staff_id
                    WHERE foct.shift_date = ? {$staff_condition}
                    ORDER BY s.name
                ";
                
                $report['cash_tracking'] = $this->db->query($cash_query, [$date])->result_array();
            } else {
                $report['cash_tracking'] = [];
            }

            // Patient registrations
            $registration_query = "
                SELECT 
                    COUNT(*) as total_registrations,
                    s.name as staff_name
                FROM patients p
                LEFT JOIN staff s ON s.id = p.registered_by_staff_id
                WHERE DATE(p.created_at) = ? {$staff_condition}
                GROUP BY p.registered_by_staff_id, s.name
                ORDER BY s.name
            ";
            
            $report['registrations'] = $this->db->query($registration_query, [$date])->result_array();

            return $report;
            
        } catch (Exception $e) {
            log_message('error', 'Generate daily report error: ' . $e->getMessage());
            return ['cash_tracking' => [], 'registrations' => []];
        }
    }

    /**
     * Check if required tables exist
     */
    public function checkTableStructure() {
        $required_tables = [
            'front_office_cash_tracking',
            'copay_collections',
            'front_office_activity_log',
            'payment_method_summary'
        ];

        $table_status = [];
        foreach ($required_tables as $table) {
            $table_status[$table] = $this->db->table_exists($table);
        }

        return $table_status;
    }

    /**
     * Update cash dispensed amount
     */
    public function updateCashDispensed($staff_id, $cash_dispensed) {
        try {
            if (!$this->db->table_exists('front_office_cash_tracking')) {
                return false;
            }

            $this->db->where('staff_id', $staff_id);
            $this->db->where('shift_date', date('Y-m-d'));
            $this->db->update('front_office_cash_tracking', [
                'total_cash_dispensed' => $cash_dispensed
            ]);

            return $this->db->affected_rows() > 0;
            
        } catch (Exception $e) {
            log_message('error', 'Update cash dispensed error: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Get cash reconciliation data
     */
    public function getCashReconciliationData($staff_id = null, $date = null) {
        if (!$date) {
            $date = date('Y-m-d');
        }

        try {
            if (!$this->db->table_exists('front_office_cash_tracking')) {
                return [];
            }

            $this->db->select('foct.*, s.name as staff_name, s.employee_id');
            $this->db->from('front_office_cash_tracking foct');
            $this->db->join('staff s', 's.id = foct.staff_id', 'left');
            $this->db->where('foct.shift_date', $date);
            
            if ($staff_id) {
                $this->db->where('foct.staff_id', $staff_id);
            }
            
            $this->db->order_by('s.name', 'ASC');
            
            $query = $this->db->get();
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Get cash reconciliation data error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get shift summary for specific staff and date
     */
    public function getShiftSummary($staff_id, $date = null) {
        if (!$date) {
            $date = date('Y-m-d');
        }

        try {
            if (!$this->db->table_exists('front_office_cash_tracking')) {
                return null;
            }

            $this->db->where('staff_id', $staff_id);
            $this->db->where('shift_date', $date);
            $shift = $this->db->get('front_office_cash_tracking')->row();

            if (!$shift) {
                return null;
            }

            // Get detailed breakdown
            $summary = [
                'shift_info' => $shift,
                'copay_breakdown' => $this->getCopayBreakdown($staff_id, $date),
                'payment_methods' => $this->getPaymentMethodBreakdown($staff_id, $date),
                'activity_count' => $this->getActivityCount($staff_id, $date)
            ];

            return $summary;
            
        } catch (Exception $e) {
            log_message('error', 'Get shift summary error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get copay breakdown for staff and date
     */
    private function getCopayBreakdown($staff_id, $date) {
        try {
            if (!$this->db->table_exists('copay_collections')) {
                return [];
            }

            $this->db->select('COUNT(*) as count, SUM(collected_amount) as total, payment_method');
            $this->db->where('collected_by_staff_id', $staff_id);
            $this->db->where('DATE(collection_date)', $date);
            $this->db->where('status', 'collected');
            $this->db->group_by('payment_method');
            
            $query = $this->db->get('copay_collections');
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Get copay breakdown error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get activity count for staff and date
     */
    private function getActivityCount($staff_id, $date) {
        try {
            if (!$this->db->table_exists('front_office_activity_log')) {
                return [];
            }

            $this->db->select('activity_type, COUNT(*) as count');
            $this->db->where('staff_id', $staff_id);
            $this->db->where('DATE(activity_date)', $date);
            $this->db->group_by('activity_type');
            
            $query = $this->db->get('front_office_activity_log');
            if ($query && $query->num_rows() > 0) {
                return $query->result_array();
            }
            return [];
            
        } catch (Exception $e) {
            log_message('error', 'Get activity count error: ' . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // INSURANCE VERIFICATION METHODS
    // =============================================

    /**
     * Verify insurance coverage
     */
    public function verifyInsurance($patient_id, $organisation_id, $insurance_id, $staff_id) {
        try {
            $this->db->trans_start();

            // Simulate insurance verification (replace with actual API call)
            $verification_status = $this->performInsuranceVerification($organisation_id, $insurance_id);

            // Log verification attempt if table exists
            if ($this->db->table_exists('insurance_verification_log')) {
                $verification_data = [
                    'patient_id' => $patient_id,
                    'organisation_id' => $organisation_id,
                    'insurance_id' => $insurance_id,
                    'verification_date' => date('Y-m-d H:i:s'),
                    'verification_status' => $verification_status['status'],
                    'coverage_percentage' => $verification_status['coverage_percentage'],
                    'verification_response' => json_encode($verification_status),
                    'verified_by_staff_id' => $staff_id
                ];

                $this->db->insert('insurance_verification_log', $verification_data);
            }

            // Update patient insurance status if columns exist
            $update_data = [];
            if ($this->db->field_exists('insurance_verification_status', 'patients')) {
                $update_data['insurance_verification_status'] = $verification_status['status'];
            }
            if ($this->db->field_exists('insurance_coverage_percentage', 'patients')) {
                $update_data['insurance_coverage_percentage'] = $verification_status['coverage_percentage'];
            }

            if (!empty($update_data)) {
                $this->db->where('id', $patient_id);
                $this->db->update('patients', $update_data);
            }

            // Log activity
            $this->logActivity($staff_id, $patient_id, 'insurance_verification', 
                'Insurance verification: ' . $verification_status['status']);

            $this->db->trans_complete();

            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Database transaction failed');
            }

            return [
                'status' => 'success',
                'verification_result' => $verification_status
            ];
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            log_message('error', 'Insurance verification failed: ' . $e->getMessage());
            return [
                'status' => 'error',
                'message' => $e->getMessage()
            ];
        }
    }

    /**
     * Perform actual insurance verification (placeholder)
     */
    private function performInsuranceVerification($organisation_id, $insurance_id) {
        try {
            // Get organisation details
            $organisation = $this->db->get_where('organisation', ['id' => $organisation_id])->row();
            
            if (!$organisation) {
                return [
                    'status' => 'rejected',
                    'coverage_percentage' => 0,
                    'message' => 'Invalid insurance provider'
                ];
            }

            // Get default coverage percentage
            $default_coverage = 70; // Default fallback
            if (isset($organisation->default_coverage_percentage)) {
                $default_coverage = $organisation->default_coverage_percentage;
            }

            // Simulate verification logic (replace with actual API integration)
            $verification_results = [
                'verified' => [
                    'status' => 'verified',
                    'coverage_percentage' => $default_coverage,
                    'message' => 'Insurance coverage verified successfully'
                ],
                'expired' => [
                    'status' => 'expired',
                    'coverage_percentage' => 0,
                    'message' => 'Insurance policy has expired'
                ],
                'rejected' => [
                    'status' => 'rejected',
                    'coverage_percentage' => 0,
                    'message' => 'Insurance policy not found or invalid'
                ]
            ];

            // For demo: 70% verified, 20% expired, 10% rejected
            $outcomes = array_merge(
                array_fill(0, 7, 'verified'),
                array_fill(0, 2, 'expired'),
                array_fill(0, 1, 'rejected')
            );
            $selected_outcome = $outcomes[array_rand($outcomes)];
            
            return $verification_results[$selected_outcome];
            
        } catch (Exception $e) {
            log_message('error', 'Perform insurance verification error: ' . $e->getMessage());
            return [
                'status' => 'rejected',
                'coverage_percentage' => 0,
                'message' => 'Verification service unavailable'
            ];
        }
    }

    // =============================================
    // UTILITY AND VALIDATION METHODS
    // =============================================

    /**
     * Validate staff access
     */
    public function validateStaffAccess($staff_id) {
        try {
            $this->db->select('id, name, employee_id, is_active');
            $this->db->where('id', $staff_id);
            $this->db->where('is_active', 1);
            
            $query = $this->db->get('staff');
            if ($query && $query->num_rows() > 0) {
                return $query->row();
            }
            return null;
            
        } catch (Exception $e) {
            log_message('error', 'Validate staff access error: ' . $e->getMessage());
            return null;
        }
    }

    /**
     * Get database table status for troubleshooting
     */
    public function getDatabaseStatus() {
        $status = [
            'tables' => $this->checkTableStructure(),
            'patients_table' => $this->db->table_exists('patients'),
            'staff_table' => $this->db->table_exists('staff'),
            'organisation_table' => $this->db->table_exists('organisation'),
            'database_connection' => $this->db->conn_id ? true : false
        ];

        // Check for new columns in patients table
        if ($status['patients_table']) {
            $status['patient_columns'] = [
                'copay_amount' => $this->db->field_exists('copay_amount', 'patients'),
                'copay_status' => $this->db->field_exists('copay_status', 'patients'),
                'insurance_coverage_percentage' => $this->db->field_exists('insurance_coverage_percentage', 'patients'),
                'registered_by_staff_id' => $this->db->field_exists('registered_by_staff_id', 'patients')
            ];
        }

        return $status;
    }

    /**
     * Initialize missing tables with basic structure
     */
    public function initializeMissingTables() {
        $results = [];
        
        try {
            // Create basic copay collections table if missing
            if (!$this->db->table_exists('copay_collections')) {
                $sql = "
                CREATE TABLE `copay_collections` (
                    `id` INT(11) NOT NULL AUTO_INCREMENT,
                    `patient_id` INT(11) NOT NULL,
                    `collected_amount` DECIMAL(10,2) NOT NULL DEFAULT 0.00,
                    `payment_method` VARCHAR(20) NOT NULL DEFAULT 'cash',
                    `receipt_number` VARCHAR(50) NULL,
                    `collection_date` DATETIME NOT NULL,
                    `collected_by_staff_id` INT(11) NOT NULL,
                    `status` VARCHAR(20) DEFAULT 'collected',
                    `notes` TEXT NULL,
                    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
                ";
                
                if ($this->db->query($sql)) {
                    $results['copay_collections'] = 'Created successfully';
                } else {
                    $results['copay_collections'] = 'Failed to create';
                }
            }

            // Create basic front office cash tracking table if missing
            if (!$this->db->table_exists('front_office_cash_tracking')) {
                $sql = "
                CREATE TABLE `front_office_cash_tracking` (
                    `id` INT(11) NOT NULL AUTO_INCREMENT,
                    `staff_id` INT(11) NOT NULL,
                    `shift_date` DATE NOT NULL,
                    `shift_start_time` TIME NULL,
                    `shift_end_time` TIME NULL,
                    `opening_cash_amount` DECIMAL(10,2) DEFAULT 0.00,
                    `closing_cash_amount` DECIMAL(10,2) DEFAULT 0.00,
                    `total_cash_received` DECIMAL(10,2) DEFAULT 0.00,
                    `total_cash_dispensed` DECIMAL(10,2) DEFAULT 0.00,
                    `variance_amount` DECIMAL(10,2) DEFAULT 0.00,
                    `copay_collections_count` INT(11) DEFAULT 0,
                    `copay_collections_amount` DECIMAL(10,2) DEFAULT 0.00,
                    `shift_status` VARCHAR(20) DEFAULT 'active',
                    `created_at` TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                    PRIMARY KEY (`id`)
                ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
                ";
                
                if ($this->db->query($sql)) {
                    $results['front_office_cash_tracking'] = 'Created successfully';
                } else {
                    $results['front_office_cash_tracking'] = 'Failed to create';
                }
            }

            return $results;
            
        } catch (Exception $e) {
            log_message('error', 'Initialize missing tables error: ' . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }

    /**
     * Add missing columns to existing tables
     */
    public function addMissingColumns() {
        $results = [];
        
        try {
            // Add columns to patients table if missing
            if ($this->db->table_exists('patients')) {
                $columns_to_add = [
                    'copay_amount' => 'DECIMAL(10,2) DEFAULT 0.00',
                    'copay_status' => 'VARCHAR(20) DEFAULT "pending"',
                    'insurance_coverage_percentage' => 'DECIMAL(5,2) DEFAULT 0.00',
                    'total_service_amount' => 'DECIMAL(10,2) DEFAULT 0.00',
                    'insurance_amount' => 'DECIMAL(10,2) DEFAULT 0.00',
                    'registered_by_staff_id' => 'INT(11) NULL'
                ];

                foreach ($columns_to_add as $column => $definition) {
                    if (!$this->db->field_exists($column, 'patients')) {
                        $sql = "ALTER TABLE `patients` ADD COLUMN `{$column}` {$definition}";
                        if ($this->db->query($sql)) {
                            $results[$column] = 'Added successfully';
                        } else {
                            $results[$column] = 'Failed to add';
                        }
                    } else {
                        $results[$column] = 'Already exists';
                    }
                }
            }

            return $results;
            
        } catch (Exception $e) {
            log_message('error', 'Add missing columns error: ' . $e->getMessage());
            return ['error' => $e->getMessage()];
        }
    }
}

// End of Front_office_model class