<?php

if (!defined('BASEPATH'))
    exit('No direct script access allowed');

class Dispensary_model extends CI_Model {

    public function __construct() {
        parent::__construct();
    }

    /**
     * Count total medicines
     */
    public function countMedicines() {
        return $this->db->count_all_results('medicines');
    }

    /**
     * Get low stock medicines
     */
    public function getLowStockMedicines($limit = 10) {
        $this->db->select('m.id, m.medicine_name, m.medicine_company, 
                          SUM(mbd.available_quantity) as total_stock, 
                          COUNT(mbd.id) as batch_count');
        $this->db->from('medicines m');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.id');
        $this->db->having('total_stock < 50 OR total_stock IS NULL');
        $this->db->order_by('total_stock', 'ASC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }

    /**
     * Get expired medicines
     */
    public function getExpiredMedicines($limit = 10) {
        $this->db->select('m.medicine_name, m.medicine_company, mbd.batch_no, 
                          mbd.expiry, mbd.available_quantity');
        $this->db->from('medicine_batch_details mbd');
        $this->db->join('medicines m', 'm.id = mbd.medicine_id');
        $this->db->where('mbd.expiry <', date('Y-m-d'));
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('m.is_active', 'yes');
        $this->db->order_by('mbd.expiry', 'ASC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }

    /**
     * Get today's sales
     */
   public function getTodaysSales() {
    $today = date('Y-m-d');
    
    $this->db->select('COUNT(*) as total_bills, COALESCE(SUM(net_amount), 0) as total_sales');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('DATE(date)', $today);
    $this->db->where('payment_status !=', 'cancelled');
    
    $result = $this->db->get()->row();
    
    if (!$result) {
        return (object)array('total_bills' => 0, 'total_sales' => 0);
    }
    
    return $result;
}

    /**
     * Get pending bills
     */
public function getPendingBills() {
    $this->db->select('COUNT(*) as count, COALESCE(SUM(net_amount), 0) as total_balance');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('payment_status', 'pending');
    
    $result = $this->db->get()->row();
    
    if (!$result) {
        return (object)array('count' => 0, 'total_balance' => 0);
    }
    
    return $result;
}

    /**
     * Get recent transactions
     */
public function getRecentTransactions($limit = 10) {
    $this->db->select('pbb.id, pbb.date as bill_date, pbb.net_amount, pbb.payment_status as status,
                      p.patient_name, p.patient_unique_id');
    $this->db->from('pharmacy_bill_basic pbb');
    $this->db->join('patients p', 'p.id = pbb.patient_id', 'left');
    $this->db->order_by('pbb.date', 'DESC');
    $this->db->limit($limit);
    
    $result = $this->db->get()->result();
    
    // Add bill numbers to the results
    foreach ($result as $key => $transaction) {
        $result[$key]->bill_no = 'MDB' . str_pad($transaction->id, 6, '0', STR_PAD_LEFT);
    }
    
    return $result;
}

/**
 * Get weekly sales data for charts
 */
public function getWeeklySales() {
    $week_start = date('Y-m-d', strtotime('-7 days'));
    $today = date('Y-m-d');
    
    $this->db->select('DATE(date) as sale_date, COUNT(*) as bills_count, COALESCE(SUM(net_amount), 0) as total_sales');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('DATE(date) >=', $week_start);
    $this->db->where('DATE(date) <=', $today);
    $this->db->where('payment_status !=', 'cancelled');
    $this->db->group_by('DATE(date)');
    $this->db->order_by('DATE(date)', 'ASC');
    
    return $this->db->get()->result();
}

/**
 * Get monthly sales data
 */
public function getMonthlySales() {
    $month_start = date('Y-m-01');
    $today = date('Y-m-d');
    
    $this->db->select('COUNT(*) as total_bills, COALESCE(SUM(net_amount), 0) as total_sales');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('DATE(date) >=', $month_start);
    $this->db->where('DATE(date) <=', $today);
    $this->db->where('payment_status !=', 'cancelled');
    
    $result = $this->db->get()->row();
    
    if (!$result) {
        return (object)array('total_bills' => 0, 'total_sales' => 0);
    }
    
    return $result;
}

/**
 * Get comprehensive dashboard statistics
 */
public function getDashboardStats() {
    $stats = array();
    
    // Today's sales
    $stats['todays_sales'] = $this->getTodaysSales();
    
    // Monthly sales
    $stats['monthly_sales'] = $this->getMonthlySales();
    
    // Pending bills
    $stats['pending_bills'] = $this->getPendingBills();
    
    // Stock statistics
    $stats['total_medicines'] = $this->countMedicines();
    $stats['low_stock_count'] = count($this->getLowStockMedicines(0)); // 0 = no limit
    $stats['expired_count'] = count($this->getExpiredMedicines(0)); // 0 = no limit
    $stats['expiring_soon_count'] = count($this->getMedicinesExpiringSoon(30, 0)); // 0 = no limit
    
    // Recent data
    $stats['recent_transactions'] = $this->getRecentTransactions(5);
    $stats['top_medicines'] = $this->getTopSellingMedicines(5);
    $stats['low_stock_medicines'] = $this->getLowStockMedicines(5);
    $stats['expired_medicines'] = $this->getExpiredMedicines(5);
    
    // Weekly sales for charts
    $stats['weekly_sales'] = $this->getWeeklySales();
    
    return $stats;
}

/**
 * Get sales summary for date range
 */
public function getSalesSummary($date_from, $date_to) {
    $this->db->select('COUNT(*) as total_bills, 
                      COALESCE(SUM(net_amount), 0) as total_sales,
                      COALESCE(AVG(net_amount), 0) as average_bill_amount');
    $this->db->from('pharmacy_bill_basic');
    $this->db->where('DATE(date) >=', $date_from);
    $this->db->where('DATE(date) <=', $date_to);
    $this->db->where('payment_status !=', 'cancelled');
    
    return $this->db->get()->row();
}

    /**
     * Get medicine categories
     */
    public function getMedicineCategories() {
        $this->db->select('*');
        $this->db->from('medicine_category');
        $this->db->order_by('medicine_category', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get all medicines with stock information
     */
    public function getAllMedicinesWithStock() {
        $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                          mc.medicine_category,
                          SUM(mbd.available_quantity) as total_stock,
                          COUNT(mbd.id) as batch_count,
                          MIN(mbd.expiry) as nearest_expiry,
                          AVG(mbd.sale_rate) as avg_price');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.id');
        $this->db->order_by('m.medicine_name', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get available medicines for dispensing
     */
    public function getAvailableMedicines() {
        $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                          mc.medicine_category,
                          SUM(mbd.available_quantity) as available_stock');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id');
        $this->db->where('m.is_active', 'yes');
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('mbd.expiry >', date('Y-m-d'));
        $this->db->group_by('m.id');
        $this->db->having('available_stock > 0');
        $this->db->order_by('m.medicine_name', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Search patients
     */
    public function searchPatients($search_term) {
        $this->db->select('id, patient_unique_id, patient_name, guardian_name, 
                          mobileno, age, gender');
        $this->db->from('patients');
        $this->db->group_start();
            $this->db->like('patient_name', $search_term);
            $this->db->or_like('patient_unique_id', $search_term);
            $this->db->or_like('mobileno', $search_term);
        $this->db->group_end();
        $this->db->where('is_active', 'yes');
        $this->db->order_by('patient_name', 'ASC');
        $this->db->limit(20);
        
        return $this->db->get()->result();
    }

    /**
     * Get patient by ID
     */
    public function getPatientById($patient_id) {
        $this->db->select('id, patient_unique_id, patient_name, guardian_name, 
                          mobileno, age, gender, address, email');
        $this->db->from('patients');
        $this->db->where('id', $patient_id);
        $this->db->where('is_active', 'yes');
        
        return $this->db->get()->row();
    }

    /**
     * Search medicines
     */
    public function searchMedicines($search_term = '', $category_id = '') {
        $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                          mc.medicine_category,
                          SUM(mbd.available_quantity) as available_stock,
                          MIN(mbd.sale_rate) as min_price,
                          MAX(mbd.sale_rate) as max_price');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id');
        
        if (!empty($search_term)) {
            $this->db->group_start();
                $this->db->like('m.medicine_name', $search_term);
                $this->db->or_like('m.medicine_company', $search_term);
                $this->db->or_like('m.medicine_composition', $search_term);
            $this->db->group_end();
        }
        
        if (!empty($category_id)) {
            $this->db->where('m.medicine_category_id', $category_id);
        }
        
        $this->db->where('m.is_active', 'yes');
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('mbd.expiry >', date('Y-m-d'));
        $this->db->group_by('m.id');
        $this->db->having('available_stock > 0');
        $this->db->order_by('m.medicine_name', 'ASC');
        $this->db->limit(50);
        
        return $this->db->get()->result();
    }

    /**
     * Get medicine with stock details
     */
    public function getMedicineWithStock($medicine_id) {
        $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                          m.medicine_composition, mc.medicine_category,
                          mbd.id as batch_id, mbd.batch_no, mbd.expiry,
                          mbd.available_quantity, mbd.sale_rate, mbd.mrp');
        $this->db->from('medicines m');
        $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id');
        $this->db->where('m.id', $medicine_id);
        $this->db->where('m.is_active', 'yes');
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('mbd.expiry >', date('Y-m-d'));
        $this->db->order_by('mbd.expiry', 'ASC'); // FIFO - First Expiry First Out
        
        return $this->db->get()->result();
    }

    /**
     * Generate bill number
     */
    public function generateBillNumber() {
        // Get prefix from prefixes table
        $this->db->select('prefix');
        $this->db->from('prefixes');
        $this->db->where('type', 'pharmacy_billing');
        $prefix_result = $this->db->get()->row();
        $prefix = $prefix_result ? $prefix_result->prefix : 'MDB';
        
        // Get the last bill number
        $this->db->select('bill_no');
        $this->db->from('pharmacy_bills');
        $this->db->like('bill_no', $prefix, 'after');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        $last_bill = $this->db->get()->row();
        
        if ($last_bill) {
            $last_number = (int) str_replace($prefix, '', $last_bill->bill_no);
            $new_number = $last_number + 1;
        } else {
            $new_number = 1;
        }
        
        return $prefix . str_pad($new_number, 6, '0', STR_PAD_LEFT);
    }
    
    /**
     * Insert new bill
     */
    public function insertBill($bill_data) {
        $this->db->insert('pharmacy_bills', $bill_data);
        return $this->db->insert_id();
    }
    
    /**
     * Insert bill item
     */
    public function insertBillItem($item_data) {
        return $this->db->insert('pharmacy_bill_items', $item_data);
    }

    /**
     * Update medicine stock
     */
    public function updateStock($batch_id, $quantity) {
        $this->db->set('available_quantity', 'available_quantity - ' . (int)$quantity, FALSE);
        $this->db->where('id', $batch_id);
        return $this->db->update('medicine_batch_details');
    }

    /**
     * Get bill details
     */
    public function getBillDetails($bill_id) {
        $this->db->select('pb.*, p.patient_name, p.patient_unique_id, p.age, 
                          p.gender, p.mobileno, s.name as generated_by_name');
        $this->db->from('pharmacy_bills pb');
        $this->db->join('patients p', 'p.id = pb.patient_id', 'left');
        $this->db->join('staff s', 's.id = pb.created_by', 'left');
        $this->db->where('pb.id', $bill_id);
        
        return $this->db->get()->row();
    }

    /**
     * Get bill items
     */
    public function getBillItems($bill_id) {
        $this->db->select('pbi.*, m.medicine_name, m.medicine_company,
                          mbd.batch_no, mbd.expiry');
        $this->db->from('pharmacy_bill_items pbi');
        $this->db->join('medicines m', 'm.id = pbi.medicine_id');
        $this->db->join('medicine_batch_details mbd', 'mbd.id = pbi.batch_id');
        $this->db->where('pbi.bill_id', $bill_id);
        $this->db->order_by('pbi.id', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get patient bills
     */
    public function getPatientBills($patient_id, $limit = 10) {
        $this->db->select('id, bill_no, bill_date, net_amount, status');
        $this->db->from('pharmacy_bills');
        $this->db->where('patient_id', $patient_id);
        $this->db->order_by('created_at', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }

    /**
     * Check medicine availability
     */
    public function checkMedicineAvailability($medicine_id, $quantity) {
        $this->db->select('SUM(available_quantity) as total_available');
        $this->db->from('medicine_batch_details');
        $this->db->where('medicine_id', $medicine_id);
        $this->db->where('available_quantity >', 0);
        $this->db->where('expiry >', date('Y-m-d'));
        
        $result = $this->db->get()->row();
        $available = $result ? $result->total_available : 0;
        
        return array(
            'available' => $available,
            'requested' => $quantity,
            'sufficient' => ($available >= $quantity),
            'shortage' => max(0, $quantity - $available)
        );
    }

    /**
     * Get low stock count
     */
    public function getLowStockCount() {
        $this->db->select('COUNT(DISTINCT m.id) as count');
        $this->db->from('medicines m');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id', 'left');
        $this->db->where('m.is_active', 'yes');
        $this->db->group_by('m.id');
        $this->db->having('SUM(mbd.available_quantity) < 50 OR SUM(mbd.available_quantity) IS NULL');
        
        return $this->db->count_all_results();
    }

    /**
     * Get expired medicines count
     */
    public function getExpiredCount() {
        $this->db->select('COUNT(*) as count');
        $this->db->from('medicine_batch_details mbd');
        $this->db->join('medicines m', 'm.id = mbd.medicine_id');
        $this->db->where('mbd.expiry <', date('Y-m-d'));
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('m.is_active', 'yes');
        
        $result = $this->db->get()->row();
        return $result ? $result->count : 0;
    }

    /**
     * Get pharmacy units
     */
    public function getPharmacyUnits() {
        $this->db->select('*');
        $this->db->from('pharmacy_unit');
        $this->db->where('is_active', 'yes');
        $this->db->order_by('unit_name', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get sales report data
     */
    public function getSalesReport($date_from, $date_to, $medicine_id = null) {
        $this->db->select('pb.bill_date, pb.bill_no, pb.net_amount, pb.status,
                          p.patient_name, p.patient_unique_id,
                          COUNT(pbi.id) as item_count');
        $this->db->from('pharmacy_bills pb');
        $this->db->join('patients p', 'p.id = pb.patient_id', 'left');
        $this->db->join('pharmacy_bill_items pbi', 'pbi.bill_id = pb.id');
        
        if ($medicine_id) {
            $this->db->where('pbi.medicine_id', $medicine_id);
        }
        
        $this->db->where('pb.bill_date >=', $date_from);
        $this->db->where('pb.bill_date <=', $date_to);
        $this->db->where('pb.status !=', 'cancelled');
        $this->db->group_by('pb.id');
        $this->db->order_by('pb.bill_date', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get inventory valuation
     */
    public function getInventoryValuation() {
        $this->db->select('m.medicine_name, m.medicine_company,
                          SUM(mbd.available_quantity) as total_qty,
                          AVG(mbd.purchase_price) as avg_cost,
                          AVG(mbd.sale_rate) as avg_sale_price,
                          SUM(mbd.available_quantity * mbd.purchase_price) as cost_value,
                          SUM(mbd.available_quantity * mbd.sale_rate) as sale_value');
        $this->db->from('medicines m');
        $this->db->join('medicine_batch_details mbd', 'm.id = mbd.medicine_id');
        $this->db->where('m.is_active', 'yes');
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->group_by('m.id');
        $this->db->order_by('sale_value', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get expiring medicines (within next 30 days)
     */
    public function getExpiringMedicines($days = 30) {
        $expiry_date = date('Y-m-d', strtotime("+{$days} days"));
        
        $this->db->select('m.medicine_name, m.medicine_company, mbd.batch_no,
                          mbd.expiry, mbd.available_quantity, mbd.sale_rate,
                          (mbd.available_quantity * mbd.sale_rate) as value_at_risk');
        $this->db->from('medicine_batch_details mbd');
        $this->db->join('medicines m', 'm.id = mbd.medicine_id');
        $this->db->where('mbd.expiry <=', $expiry_date);
        $this->db->where('mbd.expiry >', date('Y-m-d'));
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('m.is_active', 'yes');
        $this->db->order_by('mbd.expiry', 'ASC');
        
        return $this->db->get()->result();
    }

    /**
     * Get top selling medicines
     */
    public function getTopSellingMedicines($limit = 10, $days = 30) {
        $date_from = date('Y-m-d', strtotime("-{$days} days"));
        
        $this->db->select('m.medicine_name, m.medicine_company,
                          SUM(pbi.quantity) as total_sold,
                          SUM(pbi.total_price) as total_revenue,
                          COUNT(DISTINCT pb.id) as bill_count');
        $this->db->from('pharmacy_bill_items pbi');
        $this->db->join('pharmacy_bills pb', 'pb.id = pbi.bill_id');
        $this->db->join('medicines m', 'm.id = pbi.medicine_id');
        $this->db->where('pb.bill_date >=', $date_from);
        $this->db->where('pb.status !=', 'cancelled');
        $this->db->group_by('pbi.medicine_id');
        $this->db->order_by('total_sold', 'DESC');
        $this->db->limit($limit);
        
        return $this->db->get()->result();
    }
    
  /**
 * Get Medicine Full Details with all batches
 */
public function getMedicineFullDetails($medicine_id) {
    $this->db->select('m.*, mc.medicine_category');
    $this->db->from('medicines m');
    $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
    $this->db->where('m.id', $medicine_id);
    $this->db->where('m.is_active', 'yes');
    
    $medicine = $this->db->get()->row();
    
    if ($medicine) {
        // Get all batches for this medicine
        $this->db->select('mbd.*, 
                          (mbd.available_quantity) as current_stock');
        $this->db->from('medicine_batch_details mbd');
        $this->db->where('mbd.medicine_id', $medicine_id);
        $this->db->order_by('mbd.expiry', 'ASC');
        
        $medicine->batches = $this->db->get()->result();
        
        // Calculate total stock
        $medicine->total_available = array_sum(array_column($medicine->batches, 'available_quantity'));
    }
    
    return $medicine;
}

/**
 * Export stock data
 */
public function exportStock($format = 'excel') {
    $medicines = $this->getAllMedicinesWithStock();
    
    if ($format === 'excel') {
        return $this->generateExcelExport($medicines);
    } else {
        return $this->generateCSVExport($medicines);
    }
}

/**
 * Mark batch as expired
 */
public function markBatchExpired($batch_id) {
    // Move to bad stock table
    $this->db->select('*');
    $this->db->from('medicine_batch_details');
    $this->db->where('id', $batch_id);
    $batch = $this->db->get()->row();
    
    if ($batch && $batch->available_quantity > 0) {
        // Insert into bad stock
        $bad_stock_data = array(
            'medicine_batch_details_id' => $batch_id,
            'outward_date' => date('Y-m-d'),
            'expiry_date' => $batch->expiry,
            'batch_no' => $batch->batch_no,
            'quantity' => $batch->available_quantity,
            'note' => 'Marked as expired on ' . date('Y-m-d')
        );
        
        $this->db->insert('medicine_bad_stock', $bad_stock_data);
        
        // Update available quantity to 0
        $this->db->set('available_quantity', 0);
        $this->db->where('id', $batch_id);
        $this->db->update('medicine_batch_details');
        
        return true;
    }
    
    return false;
}

/**
 * Get medicines with batches for selection modal
 */
public function getMedicineWithBatches($medicine_id) {
    $this->db->select('m.id, m.medicine_name, m.medicine_company, m.unit,
                      m.medicine_composition, mc.medicine_category');
    $this->db->from('medicines m');
    $this->db->join('medicine_category mc', 'm.medicine_category_id = mc.id', 'left');
    $this->db->where('m.id', $medicine_id);
    $this->db->where('m.is_active', 'yes');
    
    $medicine = $this->db->get()->row();
    
    if ($medicine) {
        // Get available batches
        $this->db->select('mbd.id as batch_id, mbd.batch_no, mbd.expiry,
                          mbd.available_quantity, mbd.sale_rate, mbd.mrp');
        $this->db->from('medicine_batch_details mbd');
        $this->db->where('mbd.medicine_id', $medicine_id);
        $this->db->where('mbd.available_quantity >', 0);
        $this->db->where('mbd.expiry >', date('Y-m-d'));
        $this->db->order_by('mbd.expiry', 'ASC');
        
        $medicine->batches = $this->db->get()->result();
    }
    
    return $medicine;
}

/**
 * Validate stock before processing bill
 */
public function validateBillItems($bill_items) {
    $validation_errors = array();
    
    foreach ($bill_items as $item) {
        // Check if batch exists and has sufficient stock
        $this->db->select('available_quantity, medicine_id');
        $this->db->from('medicine_batch_details');
        $this->db->where('id', $item['medicine_batch_detail_id']);
        $batch = $this->db->get()->row();
        
        if (!$batch) {
            $validation_errors[] = "Batch not found for medicine";
            continue;
        }
        
        if ($batch->available_quantity < $item['quantity']) {
            $validation_errors[] = "Insufficient stock. Available: {$batch->available_quantity}, Requested: {$item['quantity']}";
        }
        
        // Check if medicine is active
        $this->db->select('is_active');
        $this->db->from('medicines');
        $this->db->where('id', $batch->medicine_id);
        $medicine = $this->db->get()->row();
        
        if (!$medicine || $medicine->is_active !== 'yes') {
            $validation_errors[] = "Medicine is not active";
        }
    }
    
    return $validation_errors;
}

/**
 * Generate CSV export
 */
private function generateCSVExport($medicines) {
    $filename = 'medicine_stock_' . date('Y_m_d') . '.csv';
    
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    $output = fopen('php://output', 'w');
    
    // CSV headers
    fputcsv($output, array(
        'Medicine Name', 'Company', 'Category', 'Total Stock', 
        'Batch Count', 'Nearest Expiry', 'Average Price'
    ));
    
    // CSV data
    foreach ($medicines as $medicine) {
        fputcsv($output, array(
            $medicine->medicine_name,
            $medicine->medicine_company ?: 'N/A',
            $medicine->medicine_category ?: 'N/A',
            $medicine->total_stock ?: 0,
            $medicine->batch_count ?: 0,
            $medicine->nearest_expiry ? date('d/m/Y', strtotime($medicine->nearest_expiry)) : 'N/A',
            $medicine->avg_price ? number_format($medicine->avg_price, 2) : '0.00'
        ));
    }
    
    fclose($output);
}

/**
 * Generate Excel export (simplified)
 */
private function generateExcelExport($medicines) {
    $filename = 'medicine_stock_' . date('Y_m_d') . '.xls';
    
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="' . $filename . '"');
    
    echo "<table border='1'>";
    echo "<tr><th>Medicine Name</th><th>Company</th><th>Category</th><th>Total Stock</th><th>Batch Count</th><th>Nearest Expiry</th><th>Average Price</th></tr>";
    
    foreach ($medicines as $medicine) {
        echo "<tr>";
        echo "<td>" . $medicine->medicine_name . "</td>";
        echo "<td>" . ($medicine->medicine_company ?: 'N/A') . "</td>";
        echo "<td>" . ($medicine->medicine_category ?: 'N/A') . "</td>";
        echo "<td>" . ($medicine->total_stock ?: 0) . "</td>";
        echo "<td>" . ($medicine->batch_count ?: 0) . "</td>";
        echo "<td>" . ($medicine->nearest_expiry ? date('d/m/Y', strtotime($medicine->nearest_expiry)) : 'N/A') . "</td>";
        echo "<td>" . ($medicine->avg_price ? number_format($medicine->avg_price, 2) : '0.00') . "</td>";
        echo "</tr>";
    }
    
    echo "</table>";
}

/**
 * Get stock adjustment form data
 */
public function getStockAdjustmentData($medicine_id) {
    $medicine = $this->getMedicineWithBatches($medicine_id);
    
    if ($medicine && !empty($medicine->batches)) {
        return array(
            'medicine' => $medicine,
            'adjustment_types' => array(
                'add' => 'Add Stock',
                'reduce' => 'Reduce Stock',
                'correct' => 'Stock Correction'
            )
        );
    }
    
    return false;
}

/**
 * Process stock adjustment
 */
public function processStockAdjustment($adjustment_data) {
    $batch_id = $adjustment_data['batch_id'];
    $adjustment_type = $adjustment_data['adjustment_type'];
    $quantity = (int)$adjustment_data['quantity'];
    $reason = $adjustment_data['reason'];
    $staff_id = $adjustment_data['staff_id'];
    
    // Get current batch details
    $this->db->select('*');
    $this->db->from('medicine_batch_details');
    $this->db->where('id', $batch_id);
    $batch = $this->db->get()->row();
    
    if (!$batch) {
        return array('status' => false, 'message' => 'Batch not found');
    }
    
    $old_quantity = $batch->available_quantity;
    $new_quantity = $old_quantity;
    
    switch ($adjustment_type) {
        case 'add':
            $new_quantity = $old_quantity + $quantity;
            break;
        case 'reduce':
            $new_quantity = max(0, $old_quantity - $quantity);
            break;
        case 'correct':
            $new_quantity = $quantity;
            break;
    }
    
    // Update stock
    $this->db->set('available_quantity', $new_quantity);
    $this->db->where('id', $batch_id);
    $success = $this->db->update('medicine_batch_details');
    
    if ($success) {
        // Log the adjustment
        $log_data = array(
            'medicine_batch_detail_id' => $batch_id,
            'adjustment_type' => $adjustment_type,
            'old_quantity' => $old_quantity,
            'new_quantity' => $new_quantity,
            'adjustment_quantity' => abs($new_quantity - $old_quantity),
            'reason' => $reason,
            'staff_id' => $staff_id,
            'created_at' => date('Y-m-d H:i:s')
        );
        
        // Create stock adjustment log table if it doesn't exist
        $this->createStockAdjustmentLogTable();
        $this->db->insert('stock_adjustment_log', $log_data);
        
        return array(
            'status' => true, 
            'message' => 'Stock adjusted successfully',
            'old_quantity' => $old_quantity,
            'new_quantity' => $new_quantity
        );
    }
    
    return array('status' => false, 'message' => 'Failed to update stock');
}

/**
 * Create stock adjustment log table if it doesn't exist
 */
private function createStockAdjustmentLogTable() {
    $table_exists = $this->db->table_exists('stock_adjustment_log');
    
    if (!$table_exists) {
        $sql = "CREATE TABLE `stock_adjustment_log` (
            `id` int(11) NOT NULL AUTO_INCREMENT,
            `medicine_batch_detail_id` int(11) NOT NULL,
            `adjustment_type` varchar(20) NOT NULL,
            `old_quantity` int(11) NOT NULL,
            `new_quantity` int(11) NOT NULL,
            `adjustment_quantity` int(11) NOT NULL,
            `reason` text,
            `staff_id` int(11) DEFAULT NULL,
            `created_at` timestamp NOT NULL DEFAULT CURRENT_TIMESTAMP,
            PRIMARY KEY (`id`),
            KEY `medicine_batch_detail_id` (`medicine_batch_detail_id`),
            KEY `staff_id` (`staff_id`),
            KEY `created_at` (`created_at`)
        ) ENGINE=InnoDB DEFAULT CHARSET=utf8mb3";
        
        $this->db->query($sql);
    }
}

/**
 * Get stock movement history
 */
public function getStockMovementHistory($medicine_id = null, $limit = 50) {
    $this->db->select('sal.*, m.medicine_name, mbd.batch_no, s.name as staff_name');
    $this->db->from('stock_adjustment_log sal');
    $this->db->join('medicine_batch_details mbd', 'mbd.id = sal.medicine_batch_detail_id');
    $this->db->join('medicines m', 'm.id = mbd.medicine_id');
    $this->db->join('staff s', 's.id = sal.staff_id', 'left');
    
    if ($medicine_id) {
        $this->db->where('mbd.medicine_id', $medicine_id);
    }
    
    $this->db->order_by('sal.created_at', 'DESC');
    $this->db->limit($limit);
    
    return $this->db->get()->result();
}

/**
 * Get dashboard analytics data
 */
public function getDashboardAnalytics() {
    $analytics = array();
    
    // Total medicines
    $analytics['total_medicines'] = $this->countMedicines();
    
    // Low stock count
    $analytics['low_stock_count'] = $this->getLowStockCount();
    
    // Expired count
    $analytics['expired_count'] = $this->getExpiredCount();
    
    // Today's sales
    $analytics['todays_sales'] = $this->getTodaysSales();
    
    // This month's sales
    $analytics['monthly_sales'] = $this->getMonthlySales();
    
    // Top selling medicines
    $analytics['top_medicines'] = $this->getTopSellingMedicines(5);
    
    // Recent transactions
    $analytics['recent_transactions'] = $this->getRecentTransactions(5);
    
    // Stock value
    $analytics['stock_value'] = $this->getTotalStockValue();
    
    return $analytics;
}

/**
 * Get total stock value
 */
public function getTotalStockValue() {
    $this->db->select('SUM(available_quantity * sale_rate) as total_value');
    $this->db->from('medicine_batch_details mbd');
    $this->db->join('medicines m', 'm.id = mbd.medicine_id');
    $this->db->where('m.is_active', 'yes');
    $this->db->where('mbd.available_quantity >', 0);
    
    $result = $this->db->get()->row();
    return $result ? $result->total_value : 0;
}
}