<?php

if (!defined('BASEPATH')) {
    exit('No direct script access allowed');
}

/**
 * Control Center Model - Complete Rewrite
 * Hospital Management System - Comprehensive Dashboard Data Management
 * 
 * This model provides all data operations for the Control Center dashboard
 * with proper error handling, database field validation, and performance optimization.
 */
class Control_center_model extends CI_Model {

    private $cache_timeout = 300; // 5 minutes cache
    private $valid_tables = [];
    private $table_fields = [];

    function __construct() {
        parent::__construct();
        $this->_initializeValidTables();
    }

    // =============================================
    // INITIALIZATION AND VALIDATION
    // =============================================

    /**
     * Initialize valid tables and their field mappings
     */
    private function _initializeValidTables() {
        try {
            $this->valid_tables = $this->db->list_tables();
            
            // Cache field mappings for critical tables
            $critical_tables = [
                'patients', 'staff', 'pharmacy_bill', 'pathology_billing', 
                'radiology_billing', 'medicines', 'appointment', 'opd_details',
                'ipd_details', 'nurse_note', 'blood_bank', 'ambulance_call'
            ];

            foreach ($critical_tables as $table) {
                if ($this->_tableExists($table)) {
                    $this->table_fields[$table] = $this->db->list_fields($table);
                }
            }
        } catch (Exception $e) {
            log_message('error', 'Control Center model initialization error: ' . $e->getMessage());
        }
    }

    /**
     * Safe table existence check
     */
    private function _tableExists($table_name) {
        try {
            return in_array($table_name, $this->valid_tables);
        } catch (Exception $e) {
            log_message('error', 'Table exists check error for ' . $table_name . ': ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Check if field exists in table
     */
    private function _fieldExists($table, $field) {
        try {
            if (!isset($this->table_fields[$table])) {
                $this->table_fields[$table] = $this->db->list_fields($table);
            }
            return in_array($field, $this->table_fields[$table]);
        } catch (Exception $e) {
            return false;
        }
    }

    /**
     * Get the best available field from a list of possible fields
     */
    private function _getBestField($table, $possible_fields) {
        foreach ($possible_fields as $field) {
            if ($this->_fieldExists($table, $field)) {
                return $field;
            }
        }
        return null;
    }

    // =============================================
    // MAIN DASHBOARD DATA METHODS
    // =============================================

    /**
     * Get comprehensive system overview
     */
    public function getSystemOverview() {
        try {
            return [
                'patients' => $this->_getPatientsOverview(),
                'appointments' => $this->_getAppointmentsOverview(),
                'staff' => $this->_getStaffOverview(),
                'financial' => $this->_getFinancialOverview(),
                'operational' => $this->_getOperationalOverview(),
                'system_health' => $this->getSystemHealth()
            ];
        } catch (Exception $e) {
            log_message('error', 'System overview error: ' . $e->getMessage());
            return $this->_getDefaultOverview();
        }
    }

    /**
     * Get real-time metrics for dashboard
     */
    public function getRealTimeMetrics() {
        try {
            return [
                'current_queues' => $this->getCurrentQueueStatus(),
                'today_revenue' => $this->getTodayRevenue(),
                'active_sessions' => $this->_getActiveSessions(),
                'system_load' => $this->_getSystemLoad(),
                'critical_alerts' => $this->getCriticalNotifications(),
                'inventory_status' => $this->_getInventoryStatus()
            ];
        } catch (Exception $e) {
            log_message('error', 'Real-time metrics error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get current queue status across all departments
     */
    public function getCurrentQueueStatus() {
        try {
            $queues = [];

            // OPD Queue
            if ($this->_tableExists('opd_details')) {
                $this->db->where('DATE(created_at)', date('Y-m-d'));
                $queues['opd'] = $this->db->count_all_results('opd_details');
            }

            // IPD Active Patients
            if ($this->_tableExists('ipd_details')) {
                $date_field = $this->_getBestField('ipd_details', ['date', 'created_at', 'admission_date']);
                if ($date_field) {
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $queues['ipd'] = $this->db->count_all_results('ipd_details');
                }
            }

            // Pharmacy Queue
            if ($this->_tableExists('pharmacy_bill')) {
                $date_field = $this->_getBestField('pharmacy_bill', ['date', 'created_at']);
                if ($date_field) {
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $queues['pharmacy'] = $this->db->count_all_results('pharmacy_bill');
                }
            }

            // Laboratory Queue
            if ($this->_tableExists('pathology_billing')) {
                $date_field = $this->_getBestField('pathology_billing', ['date', 'created_at']);
                if ($date_field) {
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    if ($this->_fieldExists('pathology_billing', 'parameter_update')) {
                        $this->db->where('parameter_update', 'no');
                    }
                    $queues['laboratory'] = $this->db->count_all_results('pathology_billing');
                }
            }

            // Radiology Queue
            if ($this->_tableExists('radiology_billing')) {
                $date_field = $this->_getBestField('radiology_billing', ['date', 'created_at']);
                if ($date_field) {
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    if ($this->_fieldExists('radiology_billing', 'parameter_update')) {
                        $this->db->where('parameter_update', 'no');
                    }
                    $queues['radiology'] = $this->db->count_all_results('radiology_billing');
                }
            }

            return $queues;
        } catch (Exception $e) {
            log_message('error', 'Queue status error: ' . $e->getMessage());
            return ['opd' => 0, 'ipd' => 0, 'pharmacy' => 0, 'laboratory' => 0, 'radiology' => 0];
        }
    }

    // =============================================
    // FINANCIAL METRICS
    // =============================================

    /**
     * Get today's revenue from all sources
     */
    public function getTodayRevenue() {
        try {
            $total_revenue = 0;
            $today = date('Y-m-d');

            $revenue_tables = [
                'pharmacy_bill',
                'pathology_billing', 
                'radiology_billing',
                'opd_billing',
                'ipd_billing'
            ];

            foreach ($revenue_tables as $table) {
                if ($this->_tableExists($table)) {
                    $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount', 'paid_amount']);
                    $date_field = $this->_getBestField($table, ['date', 'created_at', 'payment_date']);

                    if ($amount_field && $date_field) {
                        $sql = "SELECT SUM({$amount_field}) as revenue FROM {$table} WHERE DATE({$date_field}) = ?";
                        $query = $this->db->query($sql, [$today]);
                        
                        if ($query && $query->num_rows() > 0) {
                            $row = $query->row();
                            $total_revenue += ($row && isset($row->revenue)) ? (float)$row->revenue : 0;
                        }
                    }
                }
            }

            return round($total_revenue, 2);
        } catch (Exception $e) {
            log_message('error', 'Today revenue error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get monthly revenue
     */
public function getMonthlyRevenue() {
        try {
            $total_revenue = 0;
            $current_year = date('Y');
            $current_month = date('n'); // 1-12

            $revenue_tables = ['pharmacy_bill', 'pathology_billing', 'radiology_billing'];

            foreach ($revenue_tables as $table) {
                if (!$this->_tableExists($table)) {
                    continue;
                }

                $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount']);
                $date_field = $this->_getBestField($table, ['date', 'created_at']);

                if ($amount_field && $date_field) {
                    // Use YEAR() and MONTH() instead of DATE_FORMAT to avoid syntax errors
                    $sql = "SELECT SUM({$amount_field}) as revenue 
                            FROM {$table} 
                            WHERE YEAR({$date_field}) = ? AND MONTH({$date_field}) = ?";
                    
                    $query = $this->db->query($sql, [$current_year, $current_month]);
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $total_revenue += ($row && $row->revenue) ? (float)$row->revenue : 0;
                    }
                }
            }

            return round($total_revenue, 2);
        } catch (Exception $e) {
            log_message('error', 'Monthly revenue error: ' . $e->getMessage());
            return 0;
        }
    }


    /**
     * Get yearly revenue
     */
public function getYearlyRevenue() {
        try {
            $total_revenue = 0;
            $current_year = date('Y');

            $revenue_tables = ['pharmacy_bill', 'pathology_billing', 'radiology_billing'];

            foreach ($revenue_tables as $table) {
                if (!$this->_tableExists($table)) {
                    continue;
                }

                $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount']);
                $date_field = $this->_getBestField($table, ['date', 'created_at']);

                if ($amount_field && $date_field) {
                    $sql = "SELECT SUM({$amount_field}) as revenue FROM {$table} WHERE YEAR({$date_field}) = ?";
                    $query = $this->db->query($sql, [$current_year]);
                    
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $total_revenue += ($row && $row->revenue) ? (float)$row->revenue : 0;
                    }
                }
            }

            return round($total_revenue, 2);
        } catch (Exception $e) {
            log_message('error', 'Yearly revenue error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get comprehensive financial summary
     */
    public function getFinancialSummary() {
        try {
            return [
                'today_revenue' => $this->getTodayRevenue(),
                'monthly_revenue' => $this->getMonthlyRevenue(),
                'yearly_revenue' => $this->getYearlyRevenue(),
                'revenue_trend' => $this->_getRevenueTrend(),
                'top_sources' => $this->_getTopRevenueSources(),
                'collection_rate' => $this->_getCollectionRate(),
                'outstanding_amount' => $this->_getOutstandingAmounts(),
                'payment_methods' => $this->_getPaymentMethodsBreakdown()
            ];
        } catch (Exception $e) {
            log_message('error', 'Financial summary error: ' . $e->getMessage());
            return $this->_getDefaultFinancialSummary();
        }
    }

    // =============================================
    // DEPARTMENT STATISTICS
    // =============================================

    /**
     * Get department statistics
     */
    public function getDepartmentStats() {
        try {
            $stats = [];

            // Pharmacy Department
            if ($this->_tableExists('pharmacy_bill')) {
                $stats['pharmacy'] = $this->_getPharmacyStats();
            }

            // Laboratory Department
            if ($this->_tableExists('pathology_billing')) {
                $stats['laboratory'] = $this->_getLaboratoryStats();
            }

            // Radiology Department
            if ($this->_tableExists('radiology_billing')) {
                $stats['radiology'] = $this->_getRadiologyStats();
            }

            // OPD Department
            if ($this->_tableExists('opd_details')) {
                $stats['opd'] = $this->_getOPDStats();
            }

            // IPD Department
            if ($this->_tableExists('ipd_details')) {
                $stats['ipd'] = $this->_getIPDStats();
            }

            // Nursing Department
            if ($this->_tableExists('nurse_note')) {
                $stats['nursing'] = $this->_getNursingStats();
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Department stats error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get detailed department statistics for specific department
     */
    public function getDepartmentDetailedStats($department) {
        try {
            switch (strtolower($department)) {
                case 'pharmacy':
                    return $this->_getDetailedPharmacyStats();
                case 'laboratory':
                case 'pathology':
                    return $this->_getDetailedLaboratoryStats();
                case 'radiology':
                    return $this->_getDetailedRadiologyStats();
                case 'opd':
                    return $this->_getDetailedOPDStats();
                case 'ipd':
                    return $this->_getDetailedIPDStats();
                default:
                    return [];
            }
        } catch (Exception $e) {
            log_message('error', "Detailed stats error for {$department}: " . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // SYSTEM HEALTH AND PERFORMANCE
    // =============================================

    /**
     * Get comprehensive system health metrics
     */
    public function getSystemHealth() {
        try {
            return [
                'overall_status' => $this->_getOverallSystemStatus(),
                'database' => $this->_getDatabaseHealth(),
                'memory' => $this->_getMemoryHealth(),
                'storage' => $this->_getStorageHealth(),
                'performance' => $this->_getPerformanceHealth(),
                'connectivity' => $this->_getConnectivityHealth()
            ];
        } catch (Exception $e) {
            log_message('error', 'System health error: ' . $e->getMessage());
            return $this->_getDefaultSystemHealth();
        }
    }

    /**
     * Get performance metrics
     */
    public function getPerformanceMetrics() {
        try {
            return [
                'response_times' => $this->_getResponseTimes(),
                'query_performance' => $this->_getQueryPerformance(),
                'memory_usage' => $this->_getMemoryUsage(),
                'cpu_usage' => $this->_getCPUUsage(),
                'cache_hit_rate' => $this->_getCacheHitRate(),
                'error_rates' => $this->_getErrorRates()
            ];
        } catch (Exception $e) {
            log_message('error', 'Performance metrics error: ' . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // INVENTORY AND ALERTS
    // =============================================

    /**
     * Get inventory alerts with dynamic field detection
     */
    public function getInventoryAlerts() {
        try {
            $alerts = [];

            if ($this->_tableExists('medicines')) {
                $alerts['medicines'] = $this->_getMedicineInventoryAlerts();
            }

            if ($this->_tableExists('medical_supplies')) {
                $alerts['supplies'] = $this->_getSupplyInventoryAlerts();
            }

            if ($this->_tableExists('blood_bank')) {
                $alerts['blood_bank'] = $this->_getBloodBankAlerts();
            }

            return $alerts;
        } catch (Exception $e) {
            log_message('error', 'Inventory alerts error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get system alerts and notifications
     */
    public function getSystemAlerts() {
        try {
            $alerts = [];

            // Inventory alerts
            $inventory_alerts = $this->getInventoryAlerts();
            foreach ($inventory_alerts as $type => $type_alerts) {
                if (is_array($type_alerts)) {
                    foreach ($type_alerts as $alert) {
                        $alerts[] = [
                            'type' => $alert['severity'] ?? 'info',
                            'category' => 'inventory',
                            'title' => $alert['title'] ?? 'Inventory Alert',
                            'message' => $alert['message'] ?? '',
                            'timestamp' => date('Y-m-d H:i:s'),
                            'department' => ucfirst($type)
                        ];
                    }
                }
            }

            // System performance alerts
            $performance_alerts = $this->_getPerformanceAlerts();
            $alerts = array_merge($alerts, $performance_alerts);

            // Critical system alerts
            $critical_alerts = $this->getCriticalNotifications();
            $alerts = array_merge($alerts, $critical_alerts);

            // Sort by priority and timestamp
            usort($alerts, function($a, $b) {
                $priority_order = ['critical' => 1, 'danger' => 2, 'warning' => 3, 'info' => 4];
                $a_priority = $priority_order[$a['type']] ?? 5;
                $b_priority = $priority_order[$b['type']] ?? 5;
                
                if ($a_priority === $b_priority) {
                    return strtotime($b['timestamp']) - strtotime($a['timestamp']);
                }
                return $a_priority - $b_priority;
            });

            return array_slice($alerts, 0, 50); // Limit to 50 alerts
        } catch (Exception $e) {
            log_message('error', 'System alerts error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get critical notifications that require immediate attention
     */
    public function getCriticalNotifications() {
        try {
            $notifications = [];

            // Check memory usage
            $memory_usage = memory_get_usage(true);
            $memory_limit = $this->_convertToBytes(ini_get('memory_limit'));
            
            if ($memory_limit > 0 && ($memory_usage / $memory_limit) > 0.9) {
                $notifications[] = [
                    'type' => 'critical',
                    'category' => 'system',
                    'title' => 'Critical Memory Usage',
                    'message' => 'System is using more than 90% of available memory',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }

            // Check database connection
            if (!$this->db->conn_id) {
                $notifications[] = [
                    'type' => 'critical',
                    'category' => 'database',
                    'title' => 'Database Connection Lost',
                    'message' => 'Unable to connect to the database',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }

            // Check critical inventory levels
            if ($this->_tableExists('medicines')) {
                $critical_medicines = $this->_getCriticalMedicineStock();
                foreach ($critical_medicines as $medicine) {
                    $notifications[] = [
                        'type' => 'critical',
                        'category' => 'inventory',
                        'title' => 'Critical Medicine Stock',
                        'message' => "Medicine '{$medicine['name']}' is critically low or out of stock",
                        'timestamp' => date('Y-m-d H:i:s')
                    ];
                }
            }

            return $notifications;
        } catch (Exception $e) {
            log_message('error', 'Critical notifications error: ' . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // REPORTS AND ANALYTICS
    // =============================================

    /**
     * Get report categories available in the system
     */
    public function getReportCategories() {
        try {
            return [
                'financial' => [
                    'name' => 'Financial Reports',
                    'icon' => 'fa-chart-line',
                    'reports' => [
                        'revenue_summary' => 'Revenue Summary',
                        'department_revenue' => 'Department Revenue',
                        'payment_methods' => 'Payment Methods Analysis',
                        'outstanding_payments' => 'Outstanding Payments'
                    ]
                ],
                'clinical' => [
                    'name' => 'Clinical Reports',
                    'icon' => 'fa-user-md',
                    'reports' => [
                        'patient_demographics' => 'Patient Demographics',
                        'diagnosis_trends' => 'Diagnosis Trends',
                        'treatment_outcomes' => 'Treatment Outcomes',
                        'readmission_rates' => 'Readmission Rates'
                    ]
                ],
                'operational' => [
                    'name' => 'Operational Reports',
                    'icon' => 'fa-cogs',
                    'reports' => [
                        'bed_occupancy' => 'Bed Occupancy',
                        'staff_productivity' => 'Staff Productivity',
                        'queue_performance' => 'Queue Performance',
                        'equipment_utilization' => 'Equipment Utilization'
                    ]
                ],
                'inventory' => [
                    'name' => 'Inventory Reports',
                    'icon' => 'fa-boxes',
                    'reports' => [
                        'stock_levels' => 'Stock Levels',
                        'expiry_tracking' => 'Expiry Tracking',
                        'consumption_analysis' => 'Consumption Analysis',
                        'supplier_performance' => 'Supplier Performance'
                    ]
                ]
            ];
        } catch (Exception $e) {
            log_message('error', 'Report categories error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Generate report based on parameters
     */
    public function generateReport($params) {
        try {
            $report_type = $params['type'] ?? '';
            $date_from = $params['date_from'] ?? date('Y-m-01');
            $date_to = $params['date_to'] ?? date('Y-m-d');
            $department = $params['department'] ?? 'all';
            
            switch ($report_type) {
                case 'revenue_summary':
                    return $this->_generateRevenueSummaryReport($date_from, $date_to, $department);
                case 'patient_demographics':
                    return $this->_generatePatientDemographicsReport($date_from, $date_to);
                case 'inventory_status':
                    return $this->_generateInventoryStatusReport();
                case 'staff_productivity':
                    return $this->_generateStaffProductivityReport($date_from, $date_to, $department);
                default:
                    throw new Exception('Unknown report type: ' . $report_type);
            }
        } catch (Exception $e) {
            log_message('error', 'Generate report error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'data' => []
            ];
        }
    }

    // =============================================
    // MAINTENANCE AND SYSTEM ADMINISTRATION
    // =============================================

    /**
     * Get cache information
     */
    public function getCacheInfo() {
        try {
            $cache_info = [
                'status' => 'unknown',
                'type' => 'file',
                'size' => 0,
                'files' => 0,
                'hit_rate' => 0
            ];

            $cache_path = APPPATH . 'cache/';
            if (is_dir($cache_path)) {
                $cache_info['status'] = 'active';
                $files = glob($cache_path . '*');
                $cache_info['files'] = count($files);
                
                $total_size = 0;
                foreach ($files as $file) {
                    if (is_file($file)) {
                        $total_size += filesize($file);
                    }
                }
                $cache_info['size'] = $this->_formatBytes($total_size);
            }

            return $cache_info;
        } catch (Exception $e) {
            log_message('error', 'Cache info error: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Get log files information
     */
    public function getLogFiles() {
        try {
            $log_files = [];
            $log_path = APPPATH . 'logs/';
            
            if (is_dir($log_path)) {
                $files = glob($log_path . '*.php');
                foreach ($files as $file) {
                    $log_files[] = [
                        'name' => basename($file),
                        'size' => $this->_formatBytes(filesize($file)),
                        'modified' => date('Y-m-d H:i:s', filemtime($file)),
                        'lines' => $this->_countFileLines($file)
                    ];
                }
                
                // Sort by modification time
                usort($log_files, function($a, $b) {
                    return strtotime($b['modified']) - strtotime($a['modified']);
                });
            }

            return $log_files;
        } catch (Exception $e) {
            log_message('error', 'Log files info error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get database information
     */
    public function getDatabaseInfo() {
        try {
            $info = [
                'version' => $this->db->version(),
                'size_mb' => $this->_getDatabaseSize(),
                'table_count' => count($this->valid_tables),
                'charset' => $this->db->char_set,
                'collation' => $this->db->dbcollat
            ];

            // Get largest tables
            $db_name = $this->db->database;
            $sql = "
                SELECT 
                    table_name,
                    table_rows,
                    ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb
                FROM information_schema.tables 
                WHERE table_schema = ?
                ORDER BY (data_length + index_length) DESC
                LIMIT 10
            ";
            
            $query = $this->db->query($sql, [$db_name]);
            if ($query && $query->num_rows() > 0) {
                $info['largest_tables'] = $query->result_array();
            }

            return $info;
        } catch (Exception $e) {
            log_message('error', 'Database info error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get server information
     */
    public function getServerInfo() {
        try {
            return [
                'php_version' => phpversion(),
                'server_software' => $_SERVER['SERVER_SOFTWARE'] ?? 'Unknown',
                'operating_system' => php_uname('s') . ' ' . php_uname('r'),
                'memory_limit' => ini_get('memory_limit'),
                'max_execution_time' => ini_get('max_execution_time'),
                'upload_max_filesize' => ini_get('upload_max_filesize'),
                'post_max_size' => ini_get('post_max_size'),
                'timezone' => date_default_timezone_get(),
                'disk_free_space' => $this->_formatBytes(disk_free_space('.')),
                'disk_total_space' => $this->_formatBytes(disk_total_space('.'))
            ];
        } catch (Exception $e) {
            log_message('error', 'Server info error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Create system backup
     */
    public function createBackup($options = []) {
        try {
            $backup_type = $options['type'] ?? 'database';
            $include_files = $options['include_files'] ?? false;
            $created_by = $options['created_by'] ?? 1;

            $backup_dir = APPPATH . '../backups/';
            if (!is_dir($backup_dir)) {
                mkdir($backup_dir, 0755, true);
            }

            $timestamp = date('Y_m_d_H_i_s');
            $filename = "backup_{$backup_type}_{$timestamp}";

            if ($backup_type === 'database') {
                $result = $this->_createDatabaseBackup($backup_dir . $filename . '.sql');
            } else {
                $result = $this->_createFullBackup($backup_dir . $filename . '.zip', $include_files);
            }

            if ($result['success']) {
                // Log backup creation
                $this->logMaintenanceAction('backup_created', 
                    "Backup created: {$filename} by user ID {$created_by}");
            }

            return $result;
        } catch (Exception $e) {
            log_message('error', 'Create backup error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }

    /**
     * Optimize database tables
     */
    public function optimizeDatabase($tables = 'all', $operations = ['optimize']) {
        try {
            $results = [
                'success' => false,
                'tables_processed' => 0,
                'operations_performed' => [],
                'errors' => []
            ];

            if ($tables === 'all') {
                $tables = $this->valid_tables;
            } elseif (is_string($tables)) {
                $tables = [$tables];
            }

            foreach ($tables as $table) {
                if (!$this->_tableExists($table)) {
                    $results['errors'][] = "Table {$table} does not exist";
                    continue;
                }

                foreach ($operations as $operation) {
                    try {
                        switch (strtolower($operation)) {
                            case 'optimize':
                                $this->db->query("OPTIMIZE TABLE `{$table}`");
                                break;
                            case 'repair':
                                $this->db->query("REPAIR TABLE `{$table}`");
                                break;
                            case 'analyze':
                                $this->db->query("ANALYZE TABLE `{$table}`");
                                break;
                            case 'check':
                                $this->db->query("CHECK TABLE `{$table}`");
                                break;
                            default:
                                throw new Exception("Unknown operation: {$operation}");
                        }

                        $results['operations_performed'][] = "{$operation} on {$table}";

                    } catch (Exception $e) {
                        $results['errors'][] = "Failed to {$operation} table {$table}: " . $e->getMessage();
                    }
                }

                $results['tables_processed']++;
            }

            $results['success'] = empty($results['errors']);
            
            // Log the optimization
            if ($results['success']) {
                $this->logMaintenanceAction('database_optimize', 
                    "Database optimized: {$results['tables_processed']} tables processed");
            }

            return $results;

        } catch (Exception $e) {
            log_message('error', 'Database optimization error: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage(),
                'tables_processed' => 0
            ];
        }
    }

    /**
     * Get maintenance tasks and their status
     */
    public function getMaintenanceTasks() {
        try {
            return [
                'database_optimization' => [
                    'name' => 'Database Optimization',
                    'last_run' => $this->_getLastMaintenanceRun('db_optimize'),
                    'status' => 'pending',
                    'frequency' => 'weekly'
                ],
                'cache_cleanup' => [
                    'name' => 'Cache Cleanup',
                    'last_run' => $this->_getLastMaintenanceRun('cache_clear'),
                    'status' => 'pending',
                    'frequency' => 'daily'
                ],
                'log_rotation' => [
                    'name' => 'Log File Rotation',
                    'last_run' => $this->_getLastMaintenanceRun('log_rotation'),
                    'status' => 'pending',
                    'frequency' => 'weekly'
                ],
                'backup_creation' => [
                    'name' => 'System Backup',
                    'last_run' => $this->_getLastMaintenanceRun('backup_created'),
                    'status' => 'pending',
                    'frequency' => 'daily'
                ]
            ];
        } catch (Exception $e) {
            log_message('error', 'Maintenance tasks error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Log maintenance action
     */
    public function logMaintenanceAction($action, $description) {
        try {
            if ($this->_tableExists('maintenance_log')) {
                $data = [
                    'action' => $action,
                    'description' => $description,
                    'performed_at' => date('Y-m-d H:i:s'),
                    'performed_by' => $this->_getCurrentUserId()
                ];
                $this->db->insert('maintenance_log', $data);
            } else {
                // Log to file if table doesn't exist
                log_message('info', "Maintenance: {$action} - {$description}");
            }
        } catch (Exception $e) {
            log_message('error', 'Log maintenance action error: ' . $e->getMessage());
        }
    }

    // =============================================
    // RECENT ACTIVITIES AND AUDIT TRAIL
    // =============================================

    /**
     * Get recent activities across the system
     */
    public function getRecentActivities($limit = 20) {
        try {
            $activities = [];

            // Recent patient registrations
            if ($this->_tableExists('patients')) {
                $date_field = $this->_getBestField('patients', ['created_at', 'date']);
                $name_field = $this->_getBestField('patients', ['patient_name', 'name']);
                
                if ($date_field && $name_field) {
                    $this->db->select("id, {$name_field} as name, {$date_field} as created_at, 'patient_registration' as type");
                    $this->db->from('patients');
                    $this->db->order_by($date_field, 'DESC');
                    $this->db->limit(5);
                    $query = $this->db->get();
                    if ($query && $query->num_rows() > 0) {
                        $activities = array_merge($activities, $query->result_array());
                    }
                }
            }

            // Recent appointments
            if ($this->_tableExists('appointment')) {
                $date_field = $this->_getBestField('appointment', ['date', 'created_at']);
                if ($date_field) {
                    $this->db->select("id, patient_id as name, {$date_field} as created_at, 'appointment' as type");
                    $this->db->from('appointment');
                    $this->db->order_by($date_field, 'DESC');
                    $this->db->limit(5);
                    $query = $this->db->get();
                    if ($query && $query->num_rows() > 0) {
                        $activities = array_merge($activities, $query->result_array());
                    }
                }
            }

            // Recent pharmacy transactions
            if ($this->_tableExists('pharmacy_bill')) {
                $date_field = $this->_getBestField('pharmacy_bill', ['date', 'created_at']);
                $ref_field = $this->_getBestField('pharmacy_bill', ['case_reference_id', 'patient_id', 'id']);
                
                if ($date_field && $ref_field) {
                    $this->db->select("id, {$ref_field} as name, {$date_field} as created_at, 'pharmacy_transaction' as type");
                    $this->db->from('pharmacy_bill');
                    $this->db->order_by($date_field, 'DESC');
                    $this->db->limit(3);
                    $query = $this->db->get();
                    if ($query && $query->num_rows() > 0) {
                        $activities = array_merge($activities, $query->result_array());
                    }
                }
            }

            // Recent lab tests
            if ($this->_tableExists('pathology_billing')) {
                $date_field = $this->_getBestField('pathology_billing', ['date', 'created_at']);
                $ref_field = $this->_getBestField('pathology_billing', ['case_reference_id', 'patient_id', 'id']);
                
                if ($date_field && $ref_field) {
                    $this->db->select("id, {$ref_field} as name, {$date_field} as created_at, 'lab_test' as type");
                    $this->db->from('pathology_billing');
                    $this->db->order_by($date_field, 'DESC');
                    $this->db->limit(3);
                    $query = $this->db->get();
                    if ($query && $query->num_rows() > 0) {
                        $activities = array_merge($activities, $query->result_array());
                    }
                }
            }

            // Sort by timestamp and limit
            if (!empty($activities)) {
                usort($activities, function($a, $b) {
                    return strtotime($b['created_at']) - strtotime($a['created_at']);
                });
            }

            return array_slice($activities, 0, $limit);

        } catch (Exception $e) {
            log_message('error', 'Recent activities error: ' . $e->getMessage());
            return [];
        }
    }

    // =============================================
    // PRIVATE HELPER METHODS
    // =============================================

    /**
     * Get patients overview statistics
     */
private function _getPatientsOverview() {
        try {
            $overview = ['total' => 0, 'today' => 0, 'this_month' => 0, 'active' => 0];

            if (!$this->_tableExists('patients')) {
                return $overview;
            }

            // Total patients
            $overview['total'] = $this->db->count_all('patients');

            // Today's registrations - Use DATE() function instead
            $today_sql = "SELECT COUNT(*) as count FROM patients WHERE DATE(created_at) = ?";
            $today_query = $this->db->query($today_sql, [date('Y-m-d')]);
            if ($today_query && $today_query->num_rows() > 0) {
                $overview['today'] = (int)$today_query->row()->count;
            }

            // This month's registrations - Use YEAR() and MONTH() instead of DATE_FORMAT
            $month_sql = "SELECT COUNT(*) as count FROM patients WHERE YEAR(created_at) = ? AND MONTH(created_at) = ?";
            $month_query = $this->db->query($month_sql, [date('Y'), date('n')]);
            if ($month_query && $month_query->num_rows() > 0) {
                $overview['this_month'] = (int)$month_query->row()->count;
            }

            $overview['active'] = $overview['this_month'];

            return $overview;
        } catch (Exception $e) {
            log_message('error', 'Patients overview error: ' . $e->getMessage());
            return ['total' => 0, 'today' => 0, 'this_month' => 0, 'active' => 0];
        }
    }

    /**
     * Get appointments overview
     */
    private function _getAppointmentsOverview() {
        try {
            $overview = ['total_today' => 0, 'completed' => 0, 'pending' => 0, 'cancelled' => 0];

            if ($this->_tableExists('appointment')) {
                $date_field = $this->_getBestField('appointment', ['date', 'appointment_date', 'created_at']);
                
                if ($date_field) {
                    // Today's appointments
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $overview['total_today'] = $this->db->count_all_results('appointment');

                    // Status-based counts if status field exists
                    if ($this->_fieldExists('appointment', 'status')) {
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('status', 'approved');
                        $overview['completed'] = $this->db->count_all_results('appointment');

                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('status', 'pending');
                        $overview['pending'] = $this->db->count_all_results('appointment');

                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('status', 'cancelled');
                        $overview['cancelled'] = $this->db->count_all_results('appointment');
                    }
                }
            }

            return $overview;
        } catch (Exception $e) {
            log_message('error', 'Appointments overview error: ' . $e->getMessage());
            return ['total_today' => 0, 'completed' => 0, 'pending' => 0, 'cancelled' => 0];
        }
    }

    /**
     * Get staff overview
     */
    private function _getStaffOverview() {
        try {
            $overview = ['total' => 0, 'active' => 0, 'on_duty' => 0];

            if ($this->_tableExists('staff')) {
                // Total staff
                $overview['total'] = $this->db->count_all('staff');

                // Active staff
                if ($this->_fieldExists('staff', 'is_active')) {
                    $this->db->where('is_active', 'yes');
                    $overview['active'] = $this->db->count_all_results('staff');
                } else {
                    $overview['active'] = $overview['total'];
                }

                // On duty (if duty roster exists)
                if ($this->_tableExists('duty_roster')) {
                    $this->db->where('date', date('Y-m-d'));
                    $overview['on_duty'] = $this->db->count_all_results('duty_roster');
                } else {
                    $overview['on_duty'] = $overview['active'];
                }
            }

            return $overview;
        } catch (Exception $e) {
            log_message('error', 'Staff overview error: ' . $e->getMessage());
            return ['total' => 0, 'active' => 0, 'on_duty' => 0];
        }
    }

    /**
     * Get financial overview
     */
    private function _getFinancialOverview() {
        try {
            return [
                'today_revenue' => $this->getTodayRevenue(),
                'monthly_revenue' => $this->getMonthlyRevenue(),
                'yearly_revenue' => $this->getYearlyRevenue(),
              //  'growth_rate' => $this->_getRevenueGrowthRate(),
                'top_departments' => $this->_getTopRevenueDepartments()
            ];
        } catch (Exception $e) {
            log_message('error', 'Financial overview error: ' . $e->getMessage());
            return [
                'today_revenue' => 0,
                'monthly_revenue' => 0,
                'yearly_revenue' => 0,
            //    'growth_rate' => 0,
                'top_departments' => []
            ];
        }
    }

    /**
     * Get operational overview
     */
    private function _getOperationalOverview() {
        try {
            return [
                'bed_occupancy' => $this->_getBedOccupancyRate(),
                'queue_efficiency' => $this->_getQueueEfficiency(),
                'staff_productivity' => $this->_getStaffProductivityRate(),
                'patient_satisfaction' => $this->_getPatientSatisfactionScore()
            ];
        } catch (Exception $e) {
            log_message('error', 'Operational overview error: ' . $e->getMessage());
            return [
                'bed_occupancy' => 0,
                'queue_efficiency' => 0,
                'staff_productivity' => 0,
                'patient_satisfaction' => 0
            ];
        }
    }

    /**
     * Get medicine inventory alerts with dynamic field detection
     */
private function _getMedicineInventoryAlerts() {
        try {
            $alerts = [];

            if (!$this->_tableExists('medicines')) {
                return $alerts;
            }

            // Get actual field names from the table
            $fields = $this->db->list_fields('medicines');
            if (!$fields) {
                return $alerts;
            }

            // Find the correct quantity field
            $quantity_field = null;
            $possible_qty_fields = ['quantity', 'available_quantity', 'stock_qty', 'current_stock'];
            foreach ($possible_qty_fields as $field) {
                if (in_array($field, $fields)) {
                    $quantity_field = $field;
                    break;
                }
            }

            // Find the correct minimum level field
            $min_level_field = null;
            $possible_min_fields = ['min_level', 'minimum_level', 'reorder_level', 'minimum_stock'];
            foreach ($possible_min_fields as $field) {
                if (in_array($field, $fields)) {
                    $min_level_field = $field;
                    break;
                }
            }

            // Find the name field
            $name_field = null;
            $possible_name_fields = ['medicine_name', 'name', 'drug_name', 'item_name'];
            foreach ($possible_name_fields as $field) {
                if (in_array($field, $fields)) {
                    $name_field = $field;
                    break;
                }
            }

            if (!$quantity_field || !$min_level_field || !$name_field) {
                log_message('debug', 'Medicine table missing required fields. Available: ' . implode(', ', $fields));
                return $alerts;
            }

            // Low stock alerts
            $sql = "SELECT {$name_field} as name, {$quantity_field} as quantity, {$min_level_field} as min_level 
                    FROM medicines 
                    WHERE {$quantity_field} < {$min_level_field} AND {$quantity_field} > 0";
            
            if (in_array('is_active', $fields)) {
                $sql .= " AND is_active = 'yes'";
            }

            $query = $this->db->query($sql);
            if ($query && $query->num_rows() > 0) {
                foreach ($query->result_array() as $item) {
                    $alerts[] = [
                        'severity' => 'warning',
                        'title' => 'Low Stock Alert',
                        'message' => "{$item['name']} is running low (Current: {$item['quantity']}, Min: {$item['min_level']})",
                        'item' => $item['name'],
                        'current_stock' => $item['quantity'],
                        'min_level' => $item['min_level']
                    ];
                }
            }

            // Out of stock alerts
            $sql_out = "SELECT {$name_field} as name FROM medicines WHERE {$quantity_field} = 0";
            if (in_array('is_active', $fields)) {
                $sql_out .= " AND is_active = 'yes'";
            }

            $query_out = $this->db->query($sql_out);
            if ($query_out && $query_out->num_rows() > 0) {
                foreach ($query_out->result_array() as $item) {
                    $alerts[] = [
                        'severity' => 'danger',
                        'title' => 'Out of Stock Alert',
                        'message' => "{$item['name']} is out of stock",
                        'item' => $item['name'],
                        'current_stock' => 0
                    ];
                }
            }

            return $alerts;
        } catch (Exception $e) {
            log_message('error', 'Medicine inventory alerts error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get pharmacy department statistics
     */
    private function _getPharmacyStats() {
        try {
            $stats = ['bills_today' => 0, 'revenue_today' => 0, 'items_sold' => 0];

            if ($this->_tableExists('pharmacy_bill')) {
                $date_field = $this->_getBestField('pharmacy_bill', ['date', 'created_at']);
                $amount_field = $this->_getBestField('pharmacy_bill', ['net_amount', 'total_amount', 'amount']);

                if ($date_field) {
                    // Today's bills count
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['bills_today'] = $this->db->count_all_results('pharmacy_bill');

                    // Today's revenue
                    if ($amount_field) {
                        $this->db->select("SUM({$amount_field}) as total");
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $query = $this->db->get('pharmacy_bill');
                        if ($query && $query->num_rows() > 0) {
                            $stats['revenue_today'] = (float)$query->row()->total;
                        }
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Pharmacy stats error: ' . $e->getMessage());
            return ['bills_today' => 0, 'revenue_today' => 0, 'items_sold' => 0];
        }
    }

    /**
     * Get laboratory department statistics
     */
    private function _getLaboratoryStats() {
        try {
            $stats = ['tests_today' => 0, 'revenue_today' => 0, 'completed_tests' => 0, 'pending_tests' => 0];

            if ($this->_tableExists('pathology_billing')) {
                $date_field = $this->_getBestField('pathology_billing', ['date', 'created_at']);
                $amount_field = $this->_getBestField('pathology_billing', ['net_amount', 'total_amount', 'amount']);

                if ($date_field) {
                    // Today's tests count
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['tests_today'] = $this->db->count_all_results('pathology_billing');

                    // Today's revenue
                    if ($amount_field) {
                        $this->db->select("SUM({$amount_field}) as total");
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $query = $this->db->get('pathology_billing');
                        if ($query && $query->num_rows() > 0) {
                            $stats['revenue_today'] = (float)$query->row()->total;
                        }
                    }

                    // Completed vs pending tests (if status field exists)
                    if ($this->_fieldExists('pathology_billing', 'parameter_update')) {
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('parameter_update', 'yes');
                        $stats['completed_tests'] = $this->db->count_all_results('pathology_billing');

                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('parameter_update', 'no');
                        $stats['pending_tests'] = $this->db->count_all_results('pathology_billing');
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Laboratory stats error: ' . $e->getMessage());
            return ['tests_today' => 0, 'revenue_today' => 0, 'completed_tests' => 0, 'pending_tests' => 0];
        }
    }

    /**
     * Get radiology department statistics
     */
    private function _getRadiologyStats() {
        try {
            $stats = ['scans_today' => 0, 'revenue_today' => 0, 'completed_scans' => 0, 'pending_scans' => 0];

            if ($this->_tableExists('radiology_billing')) {
                $date_field = $this->_getBestField('radiology_billing', ['date', 'created_at']);
                $amount_field = $this->_getBestField('radiology_billing', ['net_amount', 'total_amount', 'amount']);

                if ($date_field) {
                    // Today's scans count
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['scans_today'] = $this->db->count_all_results('radiology_billing');

                    // Today's revenue
                    if ($amount_field) {
                        $this->db->select("SUM({$amount_field}) as total");
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $query = $this->db->get('radiology_billing');
                        if ($query && $query->num_rows() > 0) {
                            $stats['revenue_today'] = (float)$query->row()->total;
                        }
                    }

                    // Completed vs pending scans
                    if ($this->_fieldExists('radiology_billing', 'parameter_update')) {
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('parameter_update', 'yes');
                        $stats['completed_scans'] = $this->db->count_all_results('radiology_billing');

                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('parameter_update', 'no');
                        $stats['pending_scans'] = $this->db->count_all_results('radiology_billing');
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Radiology stats error: ' . $e->getMessage());
            return ['scans_today' => 0, 'revenue_today' => 0, 'completed_scans' => 0, 'pending_scans' => 0];
        }
    }

    /**
     * Get OPD department statistics
     */
    private function _getOPDStats() {
        try {
            $stats = ['patients_today' => 0, 'consultations' => 0, 'waiting' => 0];

            if ($this->_tableExists('opd_details')) {
                $date_field = $this->_getBestField('opd_details', ['appointment_date', 'date', 'created_at']);

                if ($date_field) {
                    // Today's OPD patients
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['patients_today'] = $this->db->count_all_results('opd_details');

                    // Consultations completed (if doctor field exists)
                    if ($this->_fieldExists('opd_details', 'cons_doctor')) {
                        $this->db->where("DATE({$date_field})", date('Y-m-d'));
                        $this->db->where('cons_doctor !=', '');
                        $this->db->where('cons_doctor !=', 0);
                        $stats['consultations'] = $this->db->count_all_results('opd_details');

                        $stats['waiting'] = $stats['patients_today'] - $stats['consultations'];
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'OPD stats error: ' . $e->getMessage());
            return ['patients_today' => 0, 'consultations' => 0, 'waiting' => 0];
        }
    }

    /**
     * Get IPD department statistics
     */
    private function _getIPDStats() {
        try {
            $stats = ['admissions_today' => 0, 'occupied_beds' => 0, 'discharges_today' => 0];

            if ($this->_tableExists('ipd_details')) {
                $date_field = $this->_getBestField('ipd_details', ['date', 'created_at', 'admission_date']);

                if ($date_field) {
                    // Today's admissions
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['admissions_today'] = $this->db->count_all_results('ipd_details');

                    // Occupied beds (patients without discharge date)
                    if ($this->_fieldExists('ipd_details', 'discharge_date')) {
                        $this->db->where('discharge_date IS NULL OR discharge_date = ""');
                        $stats['occupied_beds'] = $this->db->count_all_results('ipd_details');

                        // Today's discharges
                        $this->db->where('DATE(discharge_date)', date('Y-m-d'));
                        $stats['discharges_today'] = $this->db->count_all_results('ipd_details');
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'IPD stats error: ' . $e->getMessage());
            return ['admissions_today' => 0, 'occupied_beds' => 0, 'discharges_today' => 0];
        }
    }

    /**
     * Get nursing department statistics
     */
    private function _getNursingStats() {
        try {
            $stats = ['notes_today' => 0, 'patients_monitored' => 0, 'vital_signs' => 0];

            if ($this->_tableExists('nurse_note')) {
                $date_field = $this->_getBestField('nurse_note', ['date', 'created_at']);

                if ($date_field) {
                    // Today's nursing notes
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['notes_today'] = $this->db->count_all_results('nurse_note');

                    // Unique patients with notes today
                    $this->db->distinct();
                    $this->db->select('patient_id');
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $query = $this->db->get('nurse_note');
                    $stats['patients_monitored'] = $query ? $query->num_rows() : 0;
                }
            }

            // Vital signs if table exists
            if ($this->_tableExists('vital_signs')) {
                $date_field = $this->_getBestField('vital_signs', ['recorded_at', 'date', 'created_at']);
                if ($date_field) {
                    $this->db->where("DATE({$date_field})", date('Y-m-d'));
                    $stats['vital_signs'] = $this->db->count_all_results('vital_signs');
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Nursing stats error: ' . $e->getMessage());
            return ['notes_today' => 0, 'patients_monitored' => 0, 'vital_signs' => 0];
        }
    }

    /**
     * Get revenue trend for the last 7 days
     */
    private function _getRevenueTrend() {
        try {
            $trend = [];

            for ($i = 6; $i >= 0; $i--) {
                $date = date('Y-m-d', strtotime("-{$i} days"));
                $daily_revenue = 0;

                $revenue_tables = ['pharmacy_bill', 'pathology_billing', 'radiology_billing'];

                foreach ($revenue_tables as $table) {
                    if ($this->_tableExists($table)) {
                        $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount']);
                        $date_field = $this->_getBestField($table, ['date', 'created_at']);

                        if ($amount_field && $date_field) {
                            $this->db->select("SUM({$amount_field}) as revenue");
                            $this->db->where("DATE({$date_field})", $date);
                            $query = $this->db->get($table);

                            if ($query && $query->num_rows() > 0) {
                                $row = $query->row();
                                $daily_revenue += ($row && isset($row->revenue)) ? (float)$row->revenue : 0;
                            }
                        }
                    }
                }

                $trend[] = [
                    'date' => $date,
                    'revenue' => round($daily_revenue, 2),
                    'day' => date('D', strtotime($date))
                ];
            }

            return $trend;
        } catch (Exception $e) {
            log_message('error', 'Revenue trend error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get top revenue sources/departments
     */
private function _getTopRevenueSources() {
        try {
            $sources = [];
            $current_year = date('Y');
            $current_month = date('n');

            $departments = [
                'Pharmacy' => 'pharmacy_bill',
                'Laboratory' => 'pathology_billing',
                'Radiology' => 'radiology_billing',
                'OPD' => 'opd_billing',
                'IPD' => 'ipd_billing'
            ];

            foreach ($departments as $dept_name => $table) {
                if (!$this->_tableExists($table)) {
                    continue;
                }

                $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount']);
                $date_field = $this->_getBestField($table, ['date', 'created_at']);

                if ($amount_field && $date_field) {
                    $sql = "SELECT SUM({$amount_field}) as revenue 
                            FROM {$table} 
                            WHERE YEAR({$date_field}) = ? AND MONTH({$date_field}) = ?";
                    
                    $query = $this->db->query($sql, [$current_year, $current_month]);

                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $revenue = ($row && $row->revenue) ? (float)$row->revenue : 0;

                        if ($revenue > 0) {
                            $sources[] = [
                                'department' => $dept_name,
                                'revenue' => round($revenue, 2),
                                'percentage' => 0
                            ];
                        }
                    }
                }
            }

            // Calculate percentages
            $total_revenue = array_sum(array_column($sources, 'revenue'));
            if ($total_revenue > 0) {
                foreach ($sources as &$source) {
                    $source['percentage'] = round(($source['revenue'] / $total_revenue) * 100, 1);
                }
            }

            usort($sources, function($a, $b) {
                return $b['revenue'] - $a['revenue'];
            });

            return array_slice($sources, 0, 5);
        } catch (Exception $e) {
            log_message('error', 'Top revenue sources error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get collection rate (percentage of billed amount collected)
     */
private function _getCollectionRate() {
        try {
            $total_billed = 0;
            $total_collected = 0;
            $current_year = date('Y');
            $current_month = date('n');

            $revenue_tables = ['pharmacy_bill', 'pathology_billing', 'radiology_billing'];

            foreach ($revenue_tables as $table) {
                if (!$this->_tableExists($table)) {
                    continue;
                }

                $billed_field = $this->_getBestField($table, ['total_amount', 'amount', 'net_amount']);
                $collected_field = $this->_getBestField($table, ['net_amount', 'paid_amount', 'amount']);
                $date_field = $this->_getBestField($table, ['date', 'created_at']);

                if ($billed_field && $collected_field && $date_field) {
                    $sql = "SELECT SUM({$billed_field}) as billed, SUM({$collected_field}) as collected 
                            FROM {$table} 
                            WHERE YEAR({$date_field}) = ? AND MONTH({$date_field}) = ?";
                    
                    $query = $this->db->query($sql, [$current_year, $current_month]);

                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $total_billed += ($row && $row->billed) ? (float)$row->billed : 0;
                        $total_collected += ($row && $row->collected) ? (float)$row->collected : 0;
                    }
                }
            }

            return $total_billed > 0 ? round(($total_collected / $total_billed) * 100, 1) : 0;
        } catch (Exception $e) {
            log_message('error', 'Collection rate error: ' . $e->getMessage());
            return 0;
        }
    }


    /**
     * Get outstanding amounts (unpaid bills)
     */
    private function _getOutstandingAmounts() {
        try {
            $outstanding = 0;

            $revenue_tables = ['pharmacy_bill', 'pathology_billing', 'radiology_billing'];

            foreach ($revenue_tables as $table) {
                if ($this->_tableExists($table)) {
                    $total_field = $this->_getBestField($table, ['total_amount', 'amount']);
                    $paid_field = $this->_getBestField($table, ['net_amount', 'paid_amount']);

                    if ($total_field && $paid_field && $total_field !== $paid_field) {
                        $sql = "SELECT SUM({$total_field} - {$paid_field}) as outstanding 
                                FROM {$table} WHERE {$total_field} > {$paid_field}";
                        $query = $this->db->query($sql);

                        if ($query && $query->num_rows() > 0) {
                            $row = $query->row();
                            $outstanding += ($row && isset($row->outstanding)) ? (float)$row->outstanding : 0;
                        }
                    }
                }
            }

            return round($outstanding, 2);
        } catch (Exception $e) {
            log_message('error', 'Outstanding amounts error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get payment methods breakdown
     */
    private function _getPaymentMethodsBreakdown() {
        try {
            $methods = [];

            if ($this->_tableExists('transactions')) {
                $amount_field = $this->_getBestField('transactions', ['amount', 'paid_amount', 'net_amount']);
                $method_field = $this->_getBestField('transactions', ['payment_mode', 'payment_method', 'method']);
                $date_field = $this->_getBestField('transactions', ['payment_date', 'date', 'created_at']);

                if ($amount_field && $method_field && $date_field) {
                    $sql = "SELECT {$method_field} as method, SUM({$amount_field}) as total 
                            FROM transactions 
                            WHERE DATE_FORMAT({$date_field}, '%Y-%m') = ? 
                            GROUP BY {$method_field}
                            ORDER BY total DESC";
                    
                    $query = $this->db->query($sql, [date('Y-m')]);
                    if ($query && $query->num_rows() > 0) {
                        $methods = $query->result_array();
                    }
                }
            }

            return $methods;
        } catch (Exception $e) {
            log_message('error', 'Payment methods breakdown error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get system health metrics
     */
    private function _getOverallSystemStatus() {
        try {
            $issues = 0;
            $total_checks = 4;

            // Database connectivity
            if (!$this->db->conn_id) {
                $issues++;
            }

            // Memory usage check
            $memory_usage = memory_get_usage(true);
            $memory_limit = $this->_convertToBytes(ini_get('memory_limit'));
            if ($memory_limit > 0 && ($memory_usage / $memory_limit) > 0.8) {
                $issues++;
            }

            // Critical inventory check
            $critical_medicines = $this->_getCriticalMedicineStock();
            if (count($critical_medicines) > 5) {
                $issues++;
            }

            // Error log check (simplified)
            $error_log_size = $this->_getErrorLogSize();
            if ($error_log_size > 50 * 1024 * 1024) { // 50MB
                $issues++;
            }

            if ($issues === 0) {
                return 'excellent';
            } elseif ($issues <= 1) {
                return 'good';
            } elseif ($issues <= 2) {
                return 'fair';
            } else {
                return 'poor';
            }
        } catch (Exception $e) {
            return 'unknown';
        }
    }

    /**
     * Get database health metrics
     */
    private function _getDatabaseHealth() {
        try {
            $health = [
                'status' => 'operational',
                'connection_time' => 0,
                'query_time' => 0,
                'size_mb' => 0
            ];

            // Test connection time
            $start_time = microtime(true);
            if ($this->db->conn_id) {
                $health['connection_time'] = round((microtime(true) - $start_time) * 1000, 2);
            } else {
                $health['status'] = 'error';
                return $health;
            }

            // Test query time
            $start_time = microtime(true);
            $this->db->query('SELECT 1');
            $health['query_time'] = round((microtime(true) - $start_time) * 1000, 2);

            // Database size
            $health['size_mb'] = $this->_getDatabaseSize();

            return $health;
        } catch (Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Get memory health metrics
     */
    private function _getMemoryHealth() {
        try {
            $current = memory_get_usage(true);
            $peak = memory_get_peak_usage(true);
            $limit = $this->_convertToBytes(ini_get('memory_limit'));

            return [
                'current_mb' => round($current / 1024 / 1024, 2),
                'peak_mb' => round($peak / 1024 / 1024, 2),
                'limit_mb' => round($limit / 1024 / 1024, 2),
                'usage_percentage' => $limit > 0 ? round(($current / $limit) * 100, 1) : 0,
                'status' => $limit > 0 && ($current / $limit) > 0.8 ? 'warning' : 'good'
            ];
        } catch (Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Get storage health metrics
     */
    private function _getStorageHealth() {
        try {
            $free_space = disk_free_space('.');
            $total_space = disk_total_space('.');
            $used_percentage = $total_space > 0 ? round((1 - ($free_space / $total_space)) * 100, 1) : 0;

            return [
                'free_space' => $this->_formatBytes($free_space),
                'total_space' => $this->_formatBytes($total_space),
                'used_percentage' => $used_percentage,
                'status' => $used_percentage > 90 ? 'critical' : ($used_percentage > 80 ? 'warning' : 'good')
            ];
        } catch (Exception $e) {
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    // =============================================
    // UTILITY AND HELPER METHODS
    // =============================================

    /**
     * Convert memory string to bytes
     */
    private function _convertToBytes($val) {
        try {
            $val = trim($val);
            if (empty($val)) return 0;
            
            $last = strtolower($val[strlen($val)-1]);
            $val = (int)$val;
            
            switch($last) {
                case 'g': $val *= 1024;
                case 'm': $val *= 1024;
                case 'k': $val *= 1024;
            }
            
            return $val;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Format bytes to human readable format
     */
    private function _formatBytes($bytes, $precision = 2) {
        $units = array('B', 'KB', 'MB', 'GB', 'TB');
        
        for ($i = 0; $bytes > 1024 && $i < count($units) - 1; $i++) {
            $bytes /= 1024;
        }
        
        return round($bytes, $precision) . ' ' . $units[$i];
    }

    /**
     * Get database size in MB
     */
    private function _getDatabaseSize() {
        try {
            $db_name = $this->db->database;
            $sql = "
                SELECT ROUND(SUM(data_length + index_length) / 1024 / 1024, 2) AS size_mb 
                FROM information_schema.tables 
                WHERE table_schema = ?
            ";
            
            $query = $this->db->query($sql, [$db_name]);
            if ($query && $query->num_rows() > 0) {
                return (float)$query->row()->size_mb;
            }
            return 0;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Count lines in a file
     */
    private function _countFileLines($filename) {
        try {
            $line_count = 0;
            $handle = fopen($filename, 'r');
            
            if ($handle) {
                while (fgets($handle) !== false) {
                    $line_count++;
                }
                fclose($handle);
            }
            
            return $line_count;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Get current user ID
     */
    private function _getCurrentUserId() {
        try {
            // Try different methods to get current user ID
            if (isset($this->session)) {
                $user_id = $this->session->userdata('admin_id') ?: 
                          $this->session->userdata('staff_id') ?: 
                          $this->session->userdata('user_id');
                if ($user_id) {
                    return $user_id;
                }
            }
            
            // If using customlib
            if (class_exists('Customlib') && method_exists($this->customlib, 'getStaffID')) {
                return $this->customlib->getStaffID();
            }
            
            return 1; // Default system user
        } catch (Exception $e) {
            return 1;
        }
    }

    /**
     * Get active sessions count
     */
    private function _getActiveSessions() {
        try {
            if ($this->_tableExists('ci_sessions')) {
                $this->db->where('timestamp >', time() - 1800); // Last 30 minutes
                return $this->db->count_all_results('ci_sessions');
            }
            return 0;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Get system load metrics
     */
    private function _getSystemLoad() {
        try {
            return [
                'memory_usage' => memory_get_usage(true),
                'memory_peak' => memory_get_peak_usage(true),
                'execution_time' => round((microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) * 1000, 2),
                'cpu_usage' => function_exists('sys_getloadavg') ? sys_getloadavg()[0] : 0
            ];
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Get critical medicine stock items
     */
    private function _getCriticalMedicineStock() {
        try {
            $critical_items = [];

            if (!$this->_tableExists('medicines')) {
                return $critical_items;
            }

            $quantity_field = $this->_getBestField('medicines', ['available_quantity', 'quantity', 'stock_qty']);
            $name_field = $this->_getBestField('medicines', ['medicine_name', 'name', 'drug_name']);

            if ($quantity_field && $name_field) {
                $this->db->select("{$name_field} as name, {$quantity_field} as quantity");
                $this->db->where("{$quantity_field} <=", 5); // Very low stock
                if ($this->_fieldExists('medicines', 'is_active')) {
                    $this->db->where('is_active', 'yes');
                }
                $this->db->limit(10);
                $query = $this->db->get('medicines');

                if ($query && $query->num_rows() > 0) {
                    $critical_items = $query->result_array();
                }
            }

            return $critical_items;
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Get error log size
     */
    private function _getErrorLogSize() {
        try {
            $log_file = APPPATH . 'logs/log-' . date('Y-m-d') . '.php';
            return file_exists($log_file) ? filesize($log_file) : 0;
        } catch (Exception $e) {
            return 0;
        }
    }

    /**
     * Get inventory status overview
     */
    private function _getInventoryStatus() {
        try {
            $status = [
                'medicines' => $this->_getMedicineInventoryAlerts(),
                'supplies' => [],
                'blood_bank' => []
            ];

            // Add blood bank status if exists
            if ($this->_tableExists('blood_bank')) {
                $this->db->select('blood_group, COUNT(*) as count');
                $this->db->group_by('blood_group');
                $query = $this->db->get('blood_bank');
                if ($query && $query->num_rows() > 0) {
                    $status['blood_bank'] = $query->result_array();
                }
            }

            return $status;
        } catch (Exception $e) {
            return [];
        }
    }

    /**
     * Create database backup
     */
    private function _createDatabaseBackup($filename) {
        try {
            $backup_content = '';
            $backup_content .= "-- Hospital Management System Database Backup\n";
            $backup_content .= "-- Generated on: " . date('Y-m-d H:i:s') . "\n\n";
            
            foreach ($this->valid_tables as $table) {
                if (!$this->_tableExists($table)) continue;
                
                // Get create table statement
                $create_query = $this->db->query("SHOW CREATE TABLE `{$table}`");
                if ($create_query && $create_query->num_rows() > 0) {
                    $create_row = $create_query->row_array();
                    $backup_content .= "\n-- Table structure for table `{$table}`\n";
                    $backup_content .= "DROP TABLE IF EXISTS `{$table}`;\n";
                    $backup_content .= $create_row['Create Table'] . ";\n\n";
                }
                
                // Get table data
                $data_query = $this->db->get($table);
                if ($data_query && $data_query->num_rows() > 0) {
                    $backup_content .= "-- Dumping data for table `{$table}`\n";
                    foreach ($data_query->result_array() as $row) {
                        $values = array_map(function($value) {
                            return $value === null ? 'NULL' : "'" . addslashes($value) . "'";
                        }, $row);
                        $backup_content .= "INSERT INTO `{$table}` VALUES (" . implode(',', $values) . ");\n";
                    }
                    $backup_content .= "\n";
                }
            }
            
            if (file_put_contents($filename, $backup_content)) {
                return [
                    'success' => true,
                    'filename' => basename($filename),
                    'size' => $this->_formatBytes(filesize($filename))
                ];
            }
            
            return ['success' => false, 'error' => 'Could not write backup file'];
            
        } catch (Exception $e) {
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Get last maintenance run timestamp
     */
    private function _getLastMaintenanceRun($action) {
        try {
            if ($this->_tableExists('maintenance_log')) {
                $this->db->select('MAX(performed_at) as last_run');
                $this->db->where('action', $action);
                $query = $this->db->get('maintenance_log');
                
                if ($query && $query->num_rows() > 0) {
                    $result = $query->row();
                    return $result->last_run ?: 'Never';
                }
            }
            return 'Never';
        } catch (Exception $e) {
            return 'Unknown';
        }
    }

    /**
     * Get default overview when errors occur
     */
    private function _getDefaultOverview() {
        return [
            'patients' => ['total' => 0, 'today' => 0, 'this_month' => 0, 'active' => 0],
            'appointments' => ['total_today' => 0, 'completed' => 0, 'pending' => 0, 'cancelled' => 0],
            'staff' => ['total' => 0, 'active' => 0, 'on_duty' => 0],
            'financial' => ['today_revenue' => 0, 'monthly_revenue' => 0, 'yearly_revenue' => 0],
            'operational' => ['bed_occupancy' => 0, 'queue_efficiency' => 0],
            'system_health' => ['overall_status' => 'unknown']
        ];
    }

    /**
     * Get default financial summary when errors occur
     */
    private function _getDefaultFinancialSummary() {
        return [
            'today_revenue' => 0,
            'monthly_revenue' => 0,
            'yearly_revenue' => 0,
            'revenue_trend' => [],
            'top_sources' => [],
            'collection_rate' => 0,
            'outstanding_amount' => 0,
            'payment_methods' => []
        ];
    }

    /**
     * Get default system health when errors occur
     */
    private function _getDefaultSystemHealth() {
        return [
            'overall_status' => 'unknown',
            'database' => ['status' => 'unknown'],
            'memory' => ['status' => 'unknown'],
            'storage' => ['status' => 'unknown'],
            'performance' => ['status' => 'unknown'],
            'connectivity' => ['status' => 'unknown']
        ];
    }

    /**
     * Get top revenue-generating departments
     */
    private function _getTopRevenueDepartments() {
        try {
            $sources = [];
            $current_year = date('Y');
            $current_month = date('n');

            $departments = [
                'Pharmacy' => 'pharmacy_bill',
                'Laboratory' => 'pathology_billing',
                'Radiology' => 'radiology_billing',
                'OPD' => 'opd_billing',
                'IPD' => 'ipd_billing'
            ];

            foreach ($departments as $dept_name => $table) {
                if (!$this->_tableExists($table)) {
                    continue;
                }

                $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount', 'paid_amount']);
                $date_field = $this->_getBestField($table, ['date', 'created_at', 'payment_date']);

                if ($amount_field && $date_field) {
                    $sql = "SELECT SUM({$amount_field}) as revenue 
                            FROM {$table} 
                            WHERE YEAR({$date_field}) = ? AND MONTH({$date_field}) = ?";
                    $query = $this->db->query($sql, [$current_year, $current_month]);

                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $revenue = ($row && $row->revenue) ? (float)$row->revenue : 0;

                        if ($revenue > 0) {
                            $sources[] = [
                                'department' => $dept_name,
                                'revenue' => round($revenue, 2)
                            ];
                        }
                    }
                }
            }

            // Sort by revenue and calculate percentages
            $total_revenue = array_sum(array_column($sources, 'revenue'));
            foreach ($sources as &$source) {
                $source['percentage'] = $total_revenue > 0 ? round(($source['revenue'] / $total_revenue) * 100, 1) : 0;
            }

            usort($sources, function($a, $b) {
                return $b['revenue'] - $a['revenue'];
            });

            return array_slice($sources, 0, 5);
        } catch (Exception $e) {
            log_message('error', 'Top revenue departments error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get bed occupancy rate
     */
    private function _getBedOccupancyRate() {
        try {
            if (!$this->_tableExists('ipd_details')) {
                return 0;
            }

            $occupied_beds = 0;
            $total_beds = 0;

            // Count occupied beds (patients without discharge date)
            if ($this->_fieldExists('ipd_details', 'discharge_date')) {
                $this->db->where('discharge_date IS NULL OR discharge_date = ""');
                $occupied_beds = $this->db->count_all_results('ipd_details');
            }

            // Get total beds from a configuration table if available
            if ($this->_tableExists('bed')) {
                $total_beds = $this->db->count_all('bed');
            } else {
                // Fallback: Assume a default number of beds (configurable)
                $total_beds = 100; // Adjust based on hospital size
            }

            return $total_beds > 0 ? round(($occupied_beds / $total_beds) * 100, 1) : 0;
        } catch (Exception $e) {
            log_message('error', 'Bed occupancy rate error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get queue efficiency
     */
    private function _getQueueEfficiency() {
        try {
            $queues = $this->getCurrentQueueStatus();
            $total_patients = array_sum($queues);

            if ($total_patients === 0) {
                return 100; // No patients, assume perfect efficiency
            }

            // Calculate average wait time (simplified, assumes wait_time field exists)
            $avg_wait_time = 0;
            if ($this->_tableExists('opd_details') && $this->_fieldExists('opd_details', 'wait_time')) {
                $this->db->select('AVG(wait_time) as avg_wait');
                $this->db->where('DATE(appointment_date)', date('Y-m-d'));
                $query = $this->db->get('opd_details');
                if ($query && $query->num_rows() > 0) {
                    $avg_wait_time = (float)$query->row()->avg_wait;
                }
            }

            // Normalize efficiency: Lower wait times = higher efficiency
            // Assume 60 minutes as maximum acceptable wait time
            $max_wait_time = 60;
            $efficiency = $avg_wait_time > 0 ? max(0, 100 - (($avg_wait_time / $max_wait_time) * 100)) : 100;

            return round($efficiency, 1);
        } catch (Exception $e) {
            log_message('error', 'Queue efficiency error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get staff productivity rate
     */
    private function _getStaffProductivityRate() {
        try {
            if (!$this->_tableExists('staff') || !$this->_tableExists('appointment')) {
                return 0;
            }

            $active_staff = 0;
            $appointments_handled = 0;

            // Count active staff
            if ($this->_fieldExists('staff', 'is_active')) {
                $this->db->where('is_active', 'yes');
                $active_staff = $this->db->count_all_results('staff');
            } else {
                $active_staff = $this->db->count_all('staff');
            }

            // Count appointments handled today
            $date_field = $this->_getBestField('appointment', ['date', 'appointment_date', 'created_at']);
            if ($date_field) {
                $this->db->where('DATE(' . $date_field . ')', date('Y-m-d'));
                if ($this->_fieldExists('appointment', 'status')) {
                    $this->db->where('status', 'approved');
                }
                $appointments_handled = $this->db->count_all_results('appointment');
            }

            // Calculate productivity: appointments per active staff
            $productivity = $active_staff > 0 ? ($appointments_handled / $active_staff) * 100 : 0;

            return round($productivity, 1);
        } catch (Exception $e) {
            log_message('error', 'Staff productivity rate error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get patient satisfaction score
     */
    private function _getPatientSatisfactionScore() {
        try {
            if (!$this->_tableExists('patient_feedback')) {
                return 0;
            }

            $score_field = $this->_getBestField('patient_feedback', ['rating', 'score', 'satisfaction_score']);
            $date_field = $this->_getBestField('patient_feedback', ['date', 'created_at', 'feedback_date']);

            if ($score_field && $date_field) {
                $sql = "SELECT AVG({$score_field}) as avg_score 
                        FROM patient_feedback 
                        WHERE DATE({$date_field}) >= ? 
                        AND {$score_field} IS NOT NULL";
                $query = $this->db->query($sql, [date('Y-m-d', strtotime('-30 days'))]);

                if ($query && $query->num_rows() > 0) {
                    $row = $query->row();
                    return $row && $row->avg_score ? round((float)$row->avg_score, 1) : 0;
                }
            }

            return 0;
        } catch (Exception $e) {
            log_message('error', 'Patient satisfaction score error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get performance alerts
     */
    private function _getPerformanceAlerts() {
        try {
            $alerts = [];

            // Check database query performance
            $query_time = $this->_getQueryPerformance();
            if ($query_time['avg_query_time'] > 500) { // 500ms threshold
                $alerts[] = [
                    'type' => 'warning',
                    'category' => 'performance',
                    'title' => 'Slow Database Queries',
                    'message' => 'Average query time exceeds 500ms (' . round($query_time['avg_query_time'], 2) . 'ms)',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }

            // Check CPU usage
            $cpu_usage = $this->_getCPUUsage();
            if ($cpu_usage > 80) { // 80% threshold
                $alerts[] = [
                    'type' => 'warning',
                    'category' => 'performance',
                    'title' => 'High CPU Usage',
                    'message' => 'CPU usage is at ' . round($cpu_usage, 1) . '%',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }

            // Check cache hit rate
            $cache_hit_rate = $this->_getCacheHitRate();
            if ($cache_hit_rate < 50) { // 50% threshold
                $alerts[] = [
                    'type' => 'warning',
                    'category' => 'performance',
                    'title' => 'Low Cache Hit Rate',
                    'message' => 'Cache hit rate is low at ' . round($cache_hit_rate, 1) . '%',
                    'timestamp' => date('Y-m-d H:i:s')
                ];
            }

            return $alerts;
        } catch (Exception $e) {
            log_message('error', 'Performance alerts error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get query performance metrics
     */
    private function _getQueryPerformance() {
        try {
            $metrics = [
                'avg_query_time' => 0,
                'total_queries' => 0
            ];

            // CodeIgniter's database debugging must be enabled
            if ($this->db->db_debug) {
                $queries = $this->db->queries;
                $times = $this->db->query_times;

                $metrics['total_queries'] = count($queries);
                if ($metrics['total_queries'] > 0) {
                    $total_time = array_sum($times);
                    $metrics['avg_query_time'] = ($total_time / $metrics['total_queries']) * 1000; // Convert to ms
                }
            }

            return $metrics;
        } catch (Exception $e) {
            log_message('error', 'Query performance error: ' . $e->getMessage());
            return ['avg_query_time' => 0, 'total_queries' => 0];
        }
    }

    /**
     * Get CPU usage
     */
    private function _getCPUUsage() {
        try {
            if (function_exists('sys_getloadavg')) {
                $load = sys_getloadavg();
                return isset($load[0]) ? round($load[0] * 100, 1) : 0;
            }
            return 0;
        } catch (Exception $e) {
            log_message('error', 'CPU usage error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get cache hit rate
     */
    private function _getCacheHitRate() {
        try {
            // Placeholder: Implement actual cache hit rate calculation
            // This requires a caching system to track hits and misses
            return 75.0; // Dummy value for demonstration
        } catch (Exception $e) {
            log_message('error', 'Cache hit rate error: ' . $e->getMessage());
            return 0;
        }
    }

    /**
     * Get error rates
     */
    private function _getErrorRates() {
        try {
            $error_count = 0;
            $log_path = APPPATH . 'logs/log-' . date('Y-m-d') . '.php';
            
            if (file_exists($log_path)) {
                $content = file_get_contents($log_path);
                $error_count = substr_count($content, 'ERROR');
            }

            return [
                'error_count' => $error_count,
                'error_rate' => 0 // Requires request count for accurate rate
            ];
        } catch (Exception $e) {
            log_message('error', 'Error rates error: ' . $e->getMessage());
            return ['error_count' => 0, 'error_rate' => 0];
        }
    }

    /**
     * Get supply inventory alerts
     */
    private function _getSupplyInventoryAlerts() {
        try {
            $alerts = [];

            if (!$this->_tableExists('medical_supplies')) {
                return $alerts;
            }

            $quantity_field = $this->_getBestField('medical_supplies', ['quantity', 'available_quantity', 'stock_qty']);
            $min_level_field = $this->_getBestField('medical_supplies', ['min_level', 'minimum_level', 'reorder_level']);
            $name_field = $this->_getBestField('medical_supplies', ['supply_name', 'name', 'item_name']);

            if ($quantity_field && $min_level_field && $name_field) {
                // Low stock alerts
                $sql = "SELECT {$name_field} as name, {$quantity_field} as quantity, {$min_level_field} as min_level 
                        FROM medical_supplies 
                        WHERE {$quantity_field} < {$min_level_field} AND {$quantity_field} > 0";
                
                if ($this->_fieldExists('medical_supplies', 'is_active')) {
                    $sql .= " AND is_active = 'yes'";
                }

                $query = $this->db->query($sql);
                if ($query && $query->num_rows() > 0) {
                    foreach ($query->result_array() as $item) {
                        $alerts[] = [
                            'severity' => 'warning',
                            'title' => 'Low Supply Stock Alert',
                            'message' => "{$item['name']} is running low (Current: {$item['quantity']}, Min: {$item['min_level']})",
                            'item' => $item['name'],
                            'current_stock' => $item['quantity'],
                            'min_level' => $item['min_level']
                        ];
                    }
                }

                // Out of stock alerts
                $sql_out = "SELECT {$name_field} as name FROM medical_supplies WHERE {$quantity_field} = 0";
                if ($this->_fieldExists('medical_supplies', 'is_active')) {
                    $sql_out .= " AND is_active = 'yes'";
                }

                $query_out = $this->db->query($sql_out);
                if ($query_out && $query_out->num_rows() > 0) {
                    foreach ($query_out->result_array() as $item) {
                        $alerts[] = [
                            'severity' => 'danger',
                            'title' => 'Out of Supply Stock Alert',
                            'message' => "{$item['name']} is out of stock",
                            'item' => $item['name'],
                            'current_stock' => 0
                        ];
                    }
                }
            }

            return $alerts;
        } catch (Exception $e) {
            log_message('error', 'Supply inventory alerts error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get blood bank alerts
     */
    private function _getBloodBankAlerts() {
        try {
            $alerts = [];

            if (!$this->_tableExists('blood_bank')) {
                return $alerts;
            }

            $quantity_field = $this->_getBestField('blood_bank', ['quantity', 'available_units', 'stock']);
            $min_level_field = $this->_getBestField('blood_bank', ['min_level', 'minimum_level', 'threshold']);
            $blood_group_field = $this->_getBestField('blood_bank', ['blood_group', 'type']);

            if ($quantity_field && $min_level_field && $blood_group_field) {
                // Low stock alerts
                $sql = "SELECT {$blood_group_field} as blood_group, {$quantity_field} as quantity, {$min_level_field} as min_level 
                        FROM blood_bank 
                        WHERE {$quantity_field} < {$min_level_field} AND {$quantity_field} > 0";
                
                $query = $this->db->query($sql);
                if ($query && $query->num_rows() > 0) {
                    foreach ($query->result_array() as $item) {
                        $alerts[] = [
                            'severity' => 'warning',
                            'title' => 'Low Blood Stock Alert',
                            'message' => "Blood group {$item['blood_group']} is running low (Current: {$item['quantity']}, Min: {$item['min_level']})",
                            'item' => $item['blood_group'],
                            'current_stock' => $item['quantity'],
                            'min_level' => $item['min_level']
                        ];
                    }
                }

                // Out of stock alerts
                $sql_out = "SELECT {$blood_group_field} as blood_group FROM blood_bank WHERE {$quantity_field} = 0";
                $query_out = $this->db->query($sql_out);
                if ($query_out && $query_out->num_rows() > 0) {
                    foreach ($query_out->result_array() as $item) {
                        $alerts[] = [
                            'severity' => 'danger',
                            'title' => 'Out of Blood Stock Alert',
                            'message' => "Blood group {$item['blood_group']} is out of stock",
                            'item' => $item['blood_group'],
                            'current_stock' => 0
                        ];
                    }
                }
            }

            return $alerts;
        } catch (Exception $e) {
            log_message('error', 'Blood bank alerts error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get detailed pharmacy statistics
     */
    private function _getDetailedPharmacyStats() {
        try {
            $stats = [
                'daily_bills' => [],
                'top_medicines' => [],
                'revenue_by_category' => [],
                'pending_orders' => 0
            ];

            if ($this->_tableExists('pharmacy_bill')) {
                $date_field = $this->_getBestField('pharmacy_bill', ['date', 'created_at']);
                $amount_field = $this->_getBestField('pharmacy_bill', ['net_amount', 'total_amount', 'amount']);

                // Daily bills for last 7 days
                for ($i = 6; $i >= 0; $i--) {
                    $date = date('Y-m-d', strtotime("-{$i} days"));
                    $this->db->where("DATE({$date_field})", $date);
                    $count = $this->db->count_all_results('pharmacy_bill');
                    $stats['daily_bills'][] = [
                        'date' => $date,
                        'count' => $count
                    ];
                }

                // Top medicines (if pharmacy bill items table exists)
                if ($this->_tableExists('pharmacy_bill_items')) {
                    $medicine_field = $this->_getBestField('pharmacy_bill_items', ['medicine_id', 'item_id']);
                    $quantity_field = $this->_getBestField('pharmacy_bill_items', ['quantity', 'qty']);
                    if ($medicine_field && $quantity_field) {
                        $sql = "SELECT m.medicine_name, SUM(pbi.{$quantity_field}) as total_sold
                                FROM pharmacy_bill_items pbi
                                JOIN medicines m ON pbi.{$medicine_field} = m.id
                                WHERE DATE(pbi.created_at) >= ?
                                GROUP BY pbi.{$medicine_field}
                                ORDER BY total_sold DESC
                                LIMIT 5";
                        $query = $this->db->query($sql, [date('Y-m-d', strtotime('-30 days'))]);
                        if ($query && $query->num_rows() > 0) {
                            $stats['top_medicines'] = $query->result_array();
                        }
                    }
                }

                // Pending orders
                if ($this->_fieldExists('pharmacy_bill', 'status')) {
                    $this->db->where('status', 'pending');
                    $stats['pending_orders'] = $this->db->count_all_results('pharmacy_bill');
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Detailed pharmacy stats error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get detailed laboratory statistics
     */
    private function _getDetailedLaboratoryStats() {
        try {
            $stats = [
                'daily_tests' => [],
                'top_tests' => [],
                'pending_results' => 0
            ];

            if ($this->_tableExists('pathology_billing')) {
                $date_field = $this->_getBestField('pathology_billing', ['date', 'created_at']);

                // Daily tests for last 7 days
                for ($i = 6; $i >= 0; $i--) {
                    $date = date('Y-m-d', strtotime("-{$i} days"));
                    $this->db->where("DATE({$date_field})", $date);
                    $count = $this->db->count_all_results('pathology_billing');
                    $stats['daily_tests'][] = [
                        'date' => $date,
                        'count' => $count
                    ];
                }

                // Top tests
                if ($this->_fieldExists('pathology_billing', 'test_id') && $this->_tableExists('pathology_tests')) {
                    $sql = "SELECT pt.test_name, COUNT(pb.id) as test_count
                            FROM pathology_billing pb
                            JOIN pathology_tests pt ON pb.test_id = pt.id
                            WHERE DATE(pb.{$date_field}) >= ?
                            GROUP BY pb.test_id
                            ORDER BY test_count DESC
                            LIMIT 5";
                    $query = $this->db->query($sql, [date('Y-m-d', strtotime('-30 days'))]);
                    if ($query && $query->num_rows() > 0) {
                        $stats['top_tests'] = $query->result_array();
                    }
                }

                // Pending results
                if ($this->_fieldExists('pathology_billing', 'parameter_update')) {
                    $this->db->where('parameter_update', 'no');
                    $stats['pending_results'] = $this->db->count_all_results('pathology_billing');
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Detailed laboratory stats error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get detailed radiology statistics
     */
    private function _getDetailedRadiologyStats() {
        try {
            $stats = [
                'daily_scans' => [],
                'top_scans' => [],
                'pending_results' => 0
            ];

            if ($this->_tableExists('radiology_billing')) {
                $date_field = $this->_getBestField('radiology_billing', ['date', 'created_at']);

                // Daily scans for last 7 days
                for ($i = 6; $i >= 0; $i--) {
                    $date = date('Y-m-d', strtotime("-{$i} days"));
                    $this->db->where("DATE({$date_field})", $date);
                    $count = $this->db->count_all_results('radiology_billing');
                    $stats['daily_scans'][] = [
                        'date' => $date,
                        'count' => $count
                    ];
                }

                // Top scans
                if ($this->_fieldExists('radiology_billing', 'test_id') && $this->_tableExists('radiology_tests')) {
                    $sql = "SELECT rt.test_name, COUNT(rb.id) as scan_count
                            FROM radiology_billing rb
                            JOIN radiology_tests rt ON rb.test_id = rt.id
                            WHERE DATE(rb.{$date_field}) >= ?
                            GROUP BY rb.test_id
                            ORDER BY scan_count DESC
                            LIMIT 5";
                    $query = $this->db->query($sql, [date('Y-m-d', strtotime('-30 days'))]);
                    if ($query && $query->num_rows() > 0) {
                        $stats['top_scans'] = $query->result_array();
                    }
                }

                // Pending results
                if ($this->_fieldExists('radiology_billing', 'parameter_update')) {
                    $this->db->where('parameter_update', 'no');
                    $stats['pending_results'] = $this->db->count_all_results('radiology_billing');
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Detailed radiology stats error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get detailed OPD statistics
     */
    private function _getDetailedOPDStats() {
        try {
            $stats = [
                'daily_patients' => [],
                'top_doctors' => [],
                'waiting_times' => []
            ];

            if ($this->_tableExists('opd_details')) {
                $date_field = $this->_getBestField('opd_details', ['appointment_date', 'date', 'created_at']);

                if ($date_field) {
                    // Daily patients for last 7 days
                    for ($i = 6; $i >= 0; $i--) {
                        $date = date('Y-m-d', strtotime("-{$i} days"));
                        $this->db->where("DATE({$date_field})", $date);
                        $count = $this->db->count_all_results('opd_details');
                        $stats['daily_patients'][] = [
                            'date' => $date,
                            'count' => $count
                        ];
                    }

                    // Top doctors by consultations
                    if ($this->_fieldExists('opd_details', 'cons_doctor') && $this->_tableExists('staff')) {
                        $sql = "SELECT s.name, COUNT(o.id) as consultation_count
                                FROM opd_details o
                                JOIN staff s ON o.cons_doctor = s.id
                                WHERE DATE(o.{$date_field}) >= ?
                                GROUP BY o.cons_doctor
                                ORDER BY consultation_count DESC
                                LIMIT 5";
                        $query = $this->db->query($sql, [date('Y-m-d', strtotime('-30 days'))]);
                        if ($query && $query->num_rows() > 0) {
                            $stats['top_doctors'] = $query->result_array();
                        }
                    }

                    // Average waiting times
                    if ($this->_fieldExists('opd_details', 'wait_time')) {
                        $sql = "SELECT DATE({$date_field}) as date, AVG(wait_time) as avg_wait
                                FROM opd_details
                                WHERE DATE({$date_field}) >= ?
                                GROUP BY DATE({$date_field})
                                ORDER BY date DESC";
                        $query = $this->db->query($sql, [date('Y-m-d', strtotime('-7 days'))]);
                        if ($query && $query->num_rows() > 0) {
                            $stats['waiting_times'] = $query->result_array();
                        }
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Detailed OPD stats error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get detailed IPD statistics
     */
    private function _getDetailedIPDStats() {
        try {
            $stats = [
                'daily_admissions' => [],
                'average_stay' => 0,
                'bed_occupancy' => []
            ];

            if ($this->_tableExists('ipd_details')) {
                $date_field = $this->_getBestField('ipd_details', ['date', 'created_at', 'admission_date']);

                // Daily admissions for last 7 days
                for ($i = 6; $i >= 0; $i--) {
                    $date = date('Y-m-d', strtotime("-{$i} days"));
                    $this->db->where("DATE({$date_field})", $date);
                    $count = $this->db->count_all_results('ipd_details');
                    $stats['daily_admissions'][] = [
                        'date' => $date,
                        'count' => $count
                    ];
                }

                // Average length of stay
                if ($this->_fieldExists('ipd_details', 'admission_date') && $this->_fieldExists('ipd_details', 'discharge_date')) {
                    $sql = "SELECT AVG(DATEDIFF(discharge_date, admission_date)) as avg_stay
                            FROM ipd_details
                            WHERE discharge_date IS NOT NULL AND discharge_date != ''
                            AND DATE(discharge_date) >= ?";
                    $query = $this->db->query($sql, [date('Y-m-d', strtotime('-30 days'))]);
                    if ($query && $query->num_rows() > 0) {
                        $row = $query->row();
                        $stats['average_stay'] = $row && $row->avg_stay ? round((float)$row->avg_stay, 1) : 0;
                    }
                }

                // Bed occupancy by ward (if ward field exists)
                if ($this->_fieldExists('ipd_details', 'ward_id') && $this->_tableExists('wards')) {
                    $sql = "SELECT w.ward_name, COUNT(i.id) as occupied_beds
                            FROM ipd_details i
                            JOIN wards w ON i.ward_id = w.id
                            WHERE i.discharge_date IS NULL OR i.discharge_date = ''
                            GROUP BY i.ward_id";
                    $query = $this->db->query($sql);
                    if ($query && $query->num_rows() > 0) {
                        $stats['bed_occupancy'] = $query->result_array();
                    }
                }
            }

            return $stats;
        } catch (Exception $e) {
            log_message('error', 'Detailed IPD stats error: ' . $e->getMessage());
            return [];
        }
    }

    /**
     * Get performance health metrics
     */
    private function _getPerformanceHealth() {
        try {
            return [
                'response_time' => $this->_getResponseTimes(),
                'query_performance' => $this->_getQueryPerformance(),
                'cpu_usage' => $this->_getCPUUsage(),
                'cache_hit_rate' => $this->_getCacheHitRate(),
                'status' => 'good' // Simplified status
            ];
        } catch (Exception $e) {
            log_message('error', 'Performance health error: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Get connectivity health metrics
     */
    private function _getConnectivityHealth() {
        try {
            $health = [
                'database' => $this->db->conn_id ? 'connected' : 'disconnected',
                'internet' => 'unknown', // Requires external check
                'status' => $this->db->conn_id ? 'good' : 'critical'
            ];

            return $health;
        } catch (Exception $e) {
            log_message('error', 'Connectivity health error: ' . $e->getMessage());
            return ['status' => 'error', 'message' => $e->getMessage()];
        }
    }

    /**
     * Get response times
     */
    private function _getResponseTimes() {
        try {
            return [
                'current_request' => round((microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) * 1000, 2),
                'average' => 0 // Requires tracking of historical requests
            ];
        } catch (Exception $e) {
            log_message('error', 'Response times error: ' . $e->getMessage());
            return ['current_request' => 0, 'average' => 0];
        }
    }

    /**
     * Get memory usage
     */
    private function _getMemoryUsage() {
        try {
            return round(memory_get_usage(true) / 1024 / 1024, 2);
        } catch (Exception $e) {
            log_message('error', 'Memory usage error: ' . $e->getMessage());
            return 0;
        }
    }


    /**
     * Create full system backup
     */
    private function _createFullBackup($filename, $include_files = false) {
        try {
            if (!extension_loaded('zip')) {
                return ['success' => false, 'error' => 'Zip extension not loaded'];
            }

            $zip = new ZipArchive();
            if ($zip->open($filename, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
                return ['success' => false, 'error' => 'Could not create zip file'];
            }

            // Add database backup
            $db_backup_file = APPPATH . '../backups/db_temp_' . date('Y_m_d_H_i_s') . '.sql';
            $db_result = $this->_createDatabaseBackup($db_backup_file);
            if ($db_result['success']) {
                $zip->addFile($db_backup_file, 'database.sql');
            } else {
                $zip->close();
                return $db_result;
            }

            // Add files if requested
            if ($include_files) {
                $paths = [
                    APPPATH . 'config' => 'config/',
                    APPPATH . 'controllers' => 'controllers/',
                    APPPATH . 'models' => 'models/',
                    APPPATH . 'views' => 'views/'
                ];

                foreach ($paths as $path => $zip_path) {
                    if (is_dir($path)) {
                        $files = new RecursiveIteratorIterator(
                            new RecursiveDirectoryIterator($path),
                            RecursiveIteratorIterator::LEAVES_ONLY
                        );

                        foreach ($files as $file) {
                            if (!$file->isDir()) {
                                $file_path = $file->getRealPath();
                                $relative_path = $zip_path . substr($file_path, strlen($path));
                                $zip->addFile($file_path, $relative_path);
                            }
                        }
                    }
                }
            }

            $zip->close();

            // Clean up temporary database backup
            if (file_exists($db_backup_file)) {
                unlink($db_backup_file);
            }

            return [
                'success' => true,
                'filename' => basename($filename),
                'size' => $this->_formatBytes(filesize($filename))
            ];
        } catch (Exception $e) {
            log_message('error', 'Full backup error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Generate revenue summary report
     */
    private function _generateRevenueSummaryReport($date_from, $date_to, $department = 'all') {
        try {
            $report = [
                'success' => true,
                'data' => [],
                'total_revenue' => 0,
                'department' => $department
            ];

            $tables = $department === 'all' ? 
                ['pharmacy_bill', 'pathology_billing', 'radiology_billing', 'opd_billing', 'ipd_billing'] : 
                [$department === 'pharmacy' ? 'pharmacy_bill' : 
                 ($department === 'laboratory' ? 'pathology_billing' : 
                 ($department === 'radiology' ? 'radiology_billing' : 
                 ($department === 'opd' ? 'opd_billing' : 'ipd_billing')))];

            foreach ($tables as $table) {
                if (!$this->_tableExists($table)) {
                    continue;
                }

                $amount_field = $this->_getBestField($table, ['net_amount', 'amount', 'total_amount']);
                $date_field = $this->_getBestField($table, ['date', 'created_at', 'payment_date']);

                if ($amount_field && $date_field) {
                    $sql = "SELECT DATE({$date_field}) as date, SUM({$amount_field}) as revenue
                            FROM {$table}
                            WHERE DATE({$date_field}) BETWEEN ? AND ?
                            GROUP BY DATE({$date_field})
                            ORDER BY DATE({$date_field})";
                    $query = $this->db->query($sql, [$date_from, $date_to]);

                    if ($query && $query->num_rows() > 0) {
                        foreach ($query->result_array() as $row) {
                            $report['data'][] = [
                                'date' => $row['date'],
                                'revenue' => round((float)$row['revenue'], 2),
                                'department' => $table
                            ];
                            $report['total_revenue'] += (float)$row['revenue'];
                        }
                    }
                }
            }

            $report['total_revenue'] = round($report['total_revenue'], 2);
            return $report;
        } catch (Exception $e) {
            log_message('error', 'Revenue summary report error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage(), 'data' => []];
        }
    }

    /**
     * Generate patient demographics report
     */
    private function _generatePatientDemographicsReport($date_from, $date_to) {
        try {
            $report = [
                'success' => true,
                'data' => [
                    'age_groups' => [],
                    'gender_distribution' => [],
                    'new_vs_returning' => []
                ]
            ];

            if ($this->_tableExists('patients')) {
                $date_field = $this->_getBestField('patients', ['created_at', 'date']);

                // Age groups
                if ($this->_fieldExists('patients', 'age')) {
                    $sql = "SELECT 
                            CASE 
                                WHEN age < 18 THEN '0-17'
                                WHEN age BETWEEN 18 AND 35 THEN '18-35'
                                WHEN age BETWEEN 36 AND 60 THEN '36-60'
                                ELSE '60+'
                            END as age_group,
                            COUNT(*) as count
                            FROM patients
                            WHERE DATE({$date_field}) BETWEEN ? AND ?
                            GROUP BY age_group";
                    $query = $this->db->query($sql, [$date_from, $date_to]);
                    if ($query && $query->num_rows() > 0) {
                        $report['data']['age_groups'] = $query->result_array();
                    }
                }

                // Gender distribution
                if ($this->_fieldExists('patients', 'gender')) {
                    $sql = "SELECT gender, COUNT(*) as count
                            FROM patients
                            WHERE DATE({$date_field}) BETWEEN ? AND ?
                            GROUP BY gender";
                    $query = $this->db->query($sql, [$date_from, $date_to]);
                    if ($query && $query->num_rows() > 0) {
                        $report['data']['gender_distribution'] = $query->result_array();
                    }
                }

                // New vs returning patients
                if ($this->_tableExists('opd_details')) {
                    $sql = "SELECT 
                            CASE 
                                WHEN COUNT(o.id) > 1 THEN 'Returning'
                                ELSE 'New'
                            END as patient_type,
                            COUNT(DISTINCT p.id) as count
                            FROM patients p
                            LEFT JOIN opd_details o ON p.id = o.patient_id
                            WHERE DATE(p.{$date_field}) BETWEEN ? AND ?
                            GROUP BY patient_type";
                    $query = $this->db->query($sql, [$date_from, $date_to]);
                    if ($query && $query->num_rows() > 0) {
                        $report['data']['new_vs_returning'] = $query->result_array();
                    }
                }
            }

            return $report;
        } catch (Exception $e) {
            log_message('error', 'Patient demographics report error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage(), 'data' => []];
        }
    }

    /**
     * Generate inventory status report
     */
    private function _generateInventoryStatusReport() {
        try {
            $report = [
                'success' => true,
                'data' => [
                    'medicines' => [],
                    'supplies' => [],
                    'blood_bank' => []
                ]
            ];

            // Medicines inventory
            if ($this->_tableExists('medicines')) {
                $quantity_field = $this->_getBestField('medicines', ['quantity', 'available_quantity', 'stock_qty']);
                $name_field = $this->_getBestField('medicines', ['medicine_name', 'name']);
                if ($quantity_field && $name_field) {
                    $sql = "SELECT {$name_field} as name, {$quantity_field} as quantity
                            FROM medicines
                            WHERE {$quantity_field} <= ? OR {$quantity_field} IS NULL
                            ORDER BY {$quantity_field} ASC
                            LIMIT 10";
                    $query = $this->db->query($sql, [10]); // Low stock threshold
                    if ($query && $query->num_rows() > 0) {
                        $report['data']['medicines'] = $query->result_array();
                    }
                }
            }

            // Medical supplies
            if ($this->_tableExists('medical_supplies')) {
                $quantity_field = $this->_getBestField('medical_supplies', ['quantity', 'available_quantity']);
                $name_field = $this->_getBestField('medical_supplies', ['supply_name', 'name']);
                if ($quantity_field && $name_field) {
                    $sql = "SELECT {$name_field} as name, {$quantity_field} as quantity
                            FROM medical_supplies
                            WHERE {$quantity_field} <= ? OR {$quantity_field} IS NULL
                            ORDER BY {$quantity_field} ASC
                            LIMIT 10";
                    $query = $this->db->query($sql, [10]);
                    if ($query && $query->num_rows() > 0) {
                        $report['data']['supplies'] = $query->result_array();
                    }
                }
            }

            // Blood bank
            if ($this->_tableExists('blood_bank')) {
                $quantity_field = $this->_getBestField('blood_bank', ['quantity', 'available_units']);
                $blood_group_field = $this->_getBestField('blood_bank', ['blood_group', 'type']);
                if ($quantity_field && $blood_group_field) {
                    $sql = "SELECT {$blood_group_field} as blood_group, {$quantity_field} as quantity
                            FROM blood_bank
                            WHERE {$quantity_field} <= ? OR {$quantity_field} IS NULL
                            ORDER BY {$quantity_field} ASC
                            LIMIT 10";
                    $query = $this->db->query($sql, [5]); // Low blood stock threshold
                    if ($query && $query->num_rows() > 0) {
                        $report['data']['blood_bank'] = $query->result_array();
                    }
                }
            }

            return $report;
        } catch (Exception $e) {
            log_message('error', 'Inventory status report error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage(), 'data' => []];
        }
    }

    /**
     * Generate staff productivity report
     */
    private function _generateStaffProductivityReport($date_from, $date_to, $department = 'all') {
        try {
            $report = [
                'success' => true,
                'data' => []
            ];

            if (!$this->_tableExists('staff') || !$this->_tableExists('appointment')) {
                return ['success' => false, 'error' => 'Required tables not found', 'data' => []];
            }

            $date_field = $this->_getBestField('appointment', ['date', 'appointment_date', 'created_at']);
            $dept_condition = $department !== 'all' && $this->_fieldExists('staff', 'department') ? 
                "AND s.department = ?" : "";

            $sql = "SELECT s.name, s.id as staff_id, COUNT(a.id) as appointment_count
                    FROM staff s
                    LEFT JOIN appointment a ON s.id = a.doctor_id
                    WHERE DATE(a.{$date_field}) BETWEEN ? AND ?
                    {$dept_condition}
                    GROUP BY s.id
                    ORDER BY appointment_count DESC
                    LIMIT 10";

            $params = $department !== 'all' ? [$date_from, $date_to, $department] : [$date_from, $date_to];
            $query = $this->db->query($sql, $params);

            if ($query && $query->num_rows() > 0) {
                $report['data'] = $query->result_array();
            }

            return $report;
        } catch (Exception $e) {
            log_message('error', 'Staff productivity report error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage(), 'data' => []];
        }
    }
}

/* End of file Control_center_model.php */