<?php

defined('BASEPATH') OR exit('No direct script access allowed');

/**
 * Consumable Store Model
 * 
 * Handles all database operations for the consumable store module
 * 
 * @author Hospital Management System
 * @version 1.0
 */
class Consumable_model extends CI_Model
{
    
    public function __construct()
    {
        parent::__construct();
        $this->load->database();
    }

    // ========================================================
    // CONSUMABLE CATEGORIES METHODS
    // ========================================================

    /**
     * Get all consumable categories
     */
    public function getConsumableCategories($active_only = true)
    {
        $this->db->select('cc.*, pc.category_name as parent_category_name');
        $this->db->from('consumable_categories cc');
        $this->db->join('consumable_categories pc', 'cc.parent_category_id = pc.id', 'left');
        
        if ($active_only) {
            $this->db->where('cc.is_active', 'yes');
        }
        
        $this->db->order_by('cc.category_name', 'ASC');
        return $this->db->get()->result();
    }

    /**
     * Get category by ID
     */
    public function getConsumableCategoryById($id)
    {
        $this->db->select('cc.*, pc.category_name as parent_category_name');
        $this->db->from('consumable_categories cc');
        $this->db->join('consumable_categories pc', 'cc.parent_category_id = pc.id', 'left');
        $this->db->where('cc.id', $id);
        return $this->db->get()->row();
    }

    /**
     * Add new consumable category
     */
    public function addConsumableCategory($data)
    {
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['created_by'] = $this->session->userdata('admin')['id'] ?? null;
        
        return $this->db->insert('consumable_categories', $data);
    }

    /**
     * Update consumable category
     */
    public function updateConsumableCategory($id, $data)
    {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('id', $id);
        return $this->db->update('consumable_categories', $data);
    }

    // ========================================================
    // CONSUMABLES METHODS
    // ========================================================

    /**
     * Get all consumables with category information
     */
    public function getConsumables($filters = [])
    {
        $this->db->select('
            c.*,
            cc.category_name,
            cc.category_code,
            CASE 
                WHEN c.current_stock = 0 THEN "Out of Stock"
                WHEN c.current_stock <= c.reorder_level THEN "Low Stock"
                WHEN c.current_stock >= c.maximum_stock THEN "Overstock"
                ELSE "Normal"
            END as stock_status,
            (c.current_stock * c.unit_cost) as total_stock_value
        ');
        $this->db->from('consumables c');
        $this->db->join('consumable_categories cc', 'c.category_id = cc.id', 'left');
        
        // Apply filters
        if (!empty($filters['category_id'])) {
            $this->db->where('c.category_id', $filters['category_id']);
        }
        
        if (!empty($filters['active_only']) && $filters['active_only'] === true) {
            $this->db->where('c.is_active', 'yes');
        }
        
        if (!empty($filters['low_stock']) && $filters['low_stock'] === true) {
            $this->db->where('c.current_stock <=', 'c.reorder_level', false);
        }
        
        if (!empty($filters['search'])) {
            $this->db->group_start();
            $this->db->like('c.consumable_name', $filters['search']);
            $this->db->or_like('c.generic_name', $filters['search']);
            $this->db->or_like('c.consumable_code', $filters['search']);
            $this->db->group_end();
        }
        
        $this->db->order_by('c.consumable_name', 'ASC');
        return $this->db->get()->result();
    }

    /**
     * Get consumable by ID
     */
    public function getConsumableById($id)
    {
        $this->db->select('c.*, cc.category_name, cc.category_code');
        $this->db->from('consumables c');
        $this->db->join('consumable_categories cc', 'c.category_id = cc.id', 'left');
        $this->db->where('c.id', $id);
        return $this->db->get()->row();
    }

    /**
     * Add new consumable
     */
    public function addConsumable($data)
    {
        // Generate consumable code if not provided
        if (empty($data['consumable_code'])) {
            $data['consumable_code'] = $this->generateConsumableCode($data['category_id']);
        }
        
        $data['created_at'] = date('Y-m-d H:i:s');
        $data['created_by'] = $this->session->userdata('admin')['id'] ?? null;
        
        return $this->db->insert('consumables', $data);
    }

    /**
     * Update consumable
     */
    public function updateConsumable($id, $data)
    {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('id', $id);
        return $this->db->update('consumables', $data);
    }

    /**
     * Generate unique consumable code
     */
    private function generateConsumableCode($category_id)
    {
        // Get category code
        $category = $this->db->get_where('consumable_categories', ['id' => $category_id])->row();
        $prefix = $category ? $category->category_code : 'CON';
        
        // Get next sequence number
        $this->db->select('SUBSTRING(consumable_code, ' . (strlen($prefix) + 1) . ') as sequence');
        $this->db->from('consumables');
        $this->db->where('consumable_code LIKE', $prefix . '%');
        $this->db->order_by('id', 'DESC');
        $this->db->limit(1);
        
        $result = $this->db->get()->row();
        $next_sequence = $result ? (intval($result->sequence) + 1) : 1;
        
        return $prefix . str_pad($next_sequence, 3, '0', STR_PAD_LEFT);
    }

    // ========================================================
    // DEPARTMENT STOCK METHODS
    // ========================================================

    /**
     * Get department stock for a specific department
     */
    public function getDepartmentStock($department_id, $filters = [])
    {
        $this->db->select('
            dcs.*,
            c.consumable_name,
            c.consumable_code,
            c.unit_of_measure,
            c.unit_cost,
            cc.category_name,
            CASE 
                WHEN dcs.current_stock = 0 THEN "Out of Stock"
                WHEN dcs.current_stock <= dcs.minimum_stock THEN "Low Stock"
                ELSE "Normal"
            END as dept_stock_status,
            (dcs.current_stock * c.unit_cost) as dept_stock_value
        ');
        $this->db->from('department_consumable_stock dcs');
        $this->db->join('consumables c', 'dcs.consumable_id = c.id');
        $this->db->join('consumable_categories cc', 'c.category_id = cc.id', 'left');
        $this->db->where('dcs.department_id', $department_id);
        $this->db->where('dcs.is_active', 'yes');
        
        if (!empty($filters['low_stock']) && $filters['low_stock'] === true) {
            $this->db->where('dcs.current_stock <=', 'dcs.minimum_stock', false);
        }
        
        $this->db->order_by('c.consumable_name', 'ASC');
        return $this->db->get()->result();
    }

    /**
     * Update department stock
     */
    public function updateDepartmentStock($department_id, $consumable_id, $data)
    {
        $data['updated_at'] = date('Y-m-d H:i:s');
        $this->db->where('department_id', $department_id);
        $this->db->where('consumable_id', $consumable_id);
        return $this->db->update('department_consumable_stock', $data);
    }

    /**
     * Release consumables to department
     */
    public function releaseToDepar tment($department_id, $consumable_id, $quantity, $staff_id, $notes = '')
    {
        $this->db->trans_start();
        
        try {
            // Check available stock
            $consumable = $this->getConsumableById($consumable_id);
            if (!$consumable || $consumable->available_stock < $quantity) {
                throw new Exception('Insufficient stock available');
            }
            
            // Generate transaction number
            $transaction_number = $this->generateTransactionNumber('ISS');
            
            // Create transaction record
            $transaction_data = [
                'transaction_number' => $transaction_number,
                'transaction_type' => 'issued',
                'consumable_id' => $consumable_id,
                'to_department_id' => $department_id,
                'quantity' => $quantity,
                'unit_cost' => $consumable->unit_cost,
                'total_value' => $quantity * $consumable->unit_cost,
                'reference_type' => 'manual',
                'staff_id' => $staff_id,
                'transaction_date' => date('Y-m-d H:i:s'),
                'notes' => $notes
            ];
            
            $this->db->insert('consumable_transactions', $transaction_data);
            
            // Update or insert department stock
            $existing_dept_stock = $this->db->get_where('department_consumable_stock', [
                'department_id' => $department_id,
                'consumable_id' => $consumable_id
            ])->row();
            
            if ($existing_dept_stock) {
                $this->updateDepartmentStock($department_id, $consumable_id, [
                    'current_stock' => $existing_dept_stock->current_stock + $quantity,
                    'issued_stock' => $existing_dept_stock->issued_stock + $quantity,
                    'last_restocked_date' => date('Y-m-d H:i:s'),
                    'last_restocked_by' => $staff_id
                ]);
            } else {
                $dept_stock_data = [
                    'department_id' => $department_id,
                    'consumable_id' => $consumable_id,
                    'current_stock' => $quantity,
                    'minimum_stock' => 0,
                    'maximum_stock' => $quantity * 2,
                    'issued_stock' => $quantity,
                    'last_restocked_date' => date('Y-m-d H:i:s'),
                    'last_restocked_by' => $staff_id
                ];
                $this->db->insert('department_consumable_stock', $dept_stock_data);
            }
            
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Transaction failed');
            }
            
            return ['success' => true, 'transaction_number' => $transaction_number];
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    // ========================================================
    // TRANSACTION METHODS
    // ========================================================

    /**
     * Get transactions with filters
     */
    public function getTransactions($filters = [])
    {
        $this->db->select('
            ct.*,
            c.consumable_name,
            c.consumable_code,
            fd.department_name as from_department,
            td.department_name as to_department,
            p.patient_name,
            s.name as staff_name,
            s.surname as staff_surname
        ');
        $this->db->from('consumable_transactions ct');
        $this->db->join('consumables c', 'ct.consumable_id = c.id');
        $this->db->join('departments fd', 'ct.from_department_id = fd.id', 'left');
        $this->db->join('departments td', 'ct.to_department_id = td.id', 'left');
        $this->db->join('patients p', 'ct.patient_id = p.id', 'left');
        $this->db->join('staff s', 'ct.staff_id = s.id', 'left');
        
        // Apply filters
        if (!empty($filters['transaction_type'])) {
            $this->db->where('ct.transaction_type', $filters['transaction_type']);
        }
        
        if (!empty($filters['department_id'])) {
            $this->db->group_start();
            $this->db->where('ct.from_department_id', $filters['department_id']);
            $this->db->or_where('ct.to_department_id', $filters['department_id']);
            $this->db->group_end();
        }
        
        if (!empty($filters['consumable_id'])) {
            $this->db->where('ct.consumable_id', $filters['consumable_id']);
        }
        
        if (!empty($filters['date_from'])) {
            $this->db->where('DATE(ct.transaction_date) >=', $filters['date_from']);
        }
        
        if (!empty($filters['date_to'])) {
            $this->db->where('DATE(ct.transaction_date) <=', $filters['date_to']);
        }
        
        $this->db->order_by('ct.transaction_date', 'DESC');
        
        if (!empty($filters['limit'])) {
            $this->db->limit($filters['limit']);
        }
        
        return $this->db->get()->result();
    }

    /**
     * Add transaction
     */
    public function addTransaction($data)
    {
        // Generate transaction number if not provided
        if (empty($data['transaction_number'])) {
            $prefix = strtoupper(substr($data['transaction_type'], 0, 3));
            $data['transaction_number'] = $this->generateTransactionNumber($prefix);
        }
        
        $data['created_at'] = date('Y-m-d H:i:s');
        return $this->db->insert('consumable_transactions', $data);
    }

    /**
     * Generate unique transaction number
     */
    private function generateTransactionNumber($prefix)
    {
        $date = date('Ymd');
        $pattern = $prefix . '-' . $date . '-%';
        
        $this->db->select('COUNT(*) + 1 as next_number');
        $this->db->from('consumable_transactions');
        $this->db->where('transaction_number LIKE', $pattern);
        
        $result = $this->db->get()->row();
        $next_number = $result ? $result->next_number : 1;
        
        return $prefix . '-' . $date . '-' . str_pad($next_number, 4, '0', STR_PAD_LEFT);
    }

    // ========================================================
    // REQUISITION METHODS
    // ========================================================

    /**
     * Get requisitions
     */
    public function getRequisitions($filters = [])
    {
        $this->db->select('
            cr.*,
            d.department_name,
            s.name as requested_by_name,
            s.surname as requested_by_surname,
            a.name as approved_by_name,
            a.surname as approved_by_surname
        ');
        $this->db->from('consumable_requisitions cr');
        $this->db->join('departments d', 'cr.requesting_department_id = d.id');
        $this->db->join('staff s', 'cr.requested_by = s.id');
        $this->db->join('staff a', 'cr.approved_by = a.id', 'left');
        
        // Apply filters
        if (!empty($filters['department_id'])) {
            $this->db->where('cr.requesting_department_id', $filters['department_id']);
        }
        
        if (!empty($filters['status'])) {
            $this->db->where('cr.status', $filters['status']);
        }
        
        if (!empty($filters['priority'])) {
            $this->db->where('cr.priority', $filters['priority']);
        }
        
        $this->db->order_by('cr.request_date', 'DESC');
        return $this->db->get()->result();
    }

    /**
     * Get requisition by ID with items
     */
    public function getRequisitionById($id)
    {
        // Get requisition header
        $this->db->select('
            cr.*,
            d.department_name,
            s.name as requested_by_name,
            s.surname as requested_by_surname,
            a.name as approved_by_name,
            a.surname as approved_by_surname
        ');
        $this->db->from('consumable_requisitions cr');
        $this->db->join('departments d', 'cr.requesting_department_id = d.id');
        $this->db->join('staff s', 'cr.requested_by = s.id');
        $this->db->join('staff a', 'cr.approved_by = a.id', 'left');
        $this->db->where('cr.id', $id);
        
        $requisition = $this->db->get()->row();
        
        if ($requisition) {
            // Get requisition items
            $this->db->select('
                cri.*,
                c.consumable_name,
                c.consumable_code,
                c.unit_of_measure,
                c.unit_cost,
                c.current_stock
            ');
            $this->db->from('consumable_requisition_items cri');
            $this->db->join('consumables c', 'cri.consumable_id = c.id');
            $this->db->where('cri.requisition_id', $id);
            $this->db->order_by('c.consumable_name', 'ASC');
            
            $requisition->items = $this->db->get()->result();
        }
        
        return $requisition;
    }

    /**
     * Add new requisition
     */
    public function addRequisition($requisition_data, $items_data)
    {
        $this->db->trans_start();
        
        try {
            // Generate requisition number
            $requisition_data['requisition_number'] = $this->generateRequisitionNumber();
            $requisition_data['created_at'] = date('Y-m-d H:i:s');
            
            $this->db->insert('consumable_requisitions', $requisition_data);
            $requisition_id = $this->db->insert_id();
            
            // Add requisition items
            foreach ($items_data as $item) {
                $item['requisition_id'] = $requisition_id;
                $this->db->insert('consumable_requisition_items', $item);
            }
            
            $this->db->trans_complete();
            
            if ($this->db->trans_status() === FALSE) {
                throw new Exception('Failed to create requisition');
            }
            
            return ['success' => true, 'requisition_id' => $requisition_id];
            
        } catch (Exception $e) {
            $this->db->trans_rollback();
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Approve requisition
     */
    public function approveRequisition($id, $approved_by, $comments = '')
    {
        $data = [
            'status' => 'approved',
            'approved_by' => $approved_by,
            'approved_date' => date('Y-m-d H:i:s'),
            'approval_comments' => $comments,
            'updated_at' => date('Y-m-d H:i:s')
        ];
        
        $this->db->where('id', $id);
        return $this->db->update('consumable_requisitions', $data);
    }

    /**
     * Generate unique requisition number
     */
    private function generateRequisitionNumber()
    {
        $year = date('Y');
        $month = date('m');
        $pattern = 'REQ' . $year . $month . '%';
        
        $this->db->select('COUNT(*) + 1 as next_number');
        $this->db->from('consumable_requisitions');
        $this->db->where('requisition_number LIKE', $pattern);
        
        $result = $this->db->get()->row();
        $next_number = $result ? $result->next_number : 1;
        
        return 'REQ' . $year . $month . str_pad($next_number, 4, '0', STR_PAD_LEFT);
    }

    // ========================================================
    // ALERTS AND REPORTING METHODS
    // ========================================================

    /**
     * Get active alerts
     */
    public function getActiveAlerts($filters = [])
    {
        $this->db->select('
            ca.*,
            c.consumable_name,
            d.department_name,
            cb.batch_number
        ');
        $this->db->from('consumable_alerts ca');
        $this->db->join('consumables c', 'ca.consumable_id = c.id', 'left');
        $this->db->join('departments d', 'ca.department_id = d.id', 'left');
        $this->db->join('consumable_batches cb', 'ca.batch_id = cb.id', 'left');
        $this->db->where('ca.is_resolved', 0);
        
        if (!empty($filters['alert_level'])) {
            $this->db->where('ca.alert_level', $filters['alert_level']);
        }
        
        if (!empty($filters['alert_type'])) {
            $this->db->where('ca.alert_type', $filters['alert_type']);
        }
        
        $this->db->order_by('ca.alert_level', 'DESC');
        $this->db->order_by('ca.created_at', 'DESC');
        
        return $this->db->get()->result();
    }

    /**
     * Get stock summary report
     */
    public function getStockSummaryReport()
    {
        return $this->db->query("
            SELECT 
                cc.category_name,
                COUNT(c.id) as total_items,
                SUM(c.current_stock) as total_stock,
                SUM(c.current_stock * c.unit_cost) as total_value,
                SUM(CASE WHEN c.current_stock <= c.reorder_level THEN 1 ELSE 0 END) as low_stock_items,
                SUM(CASE WHEN c.current_stock = 0 THEN 1 ELSE 0 END) as out_of_stock_items
            FROM consumable_categories cc
            LEFT JOIN consumables c ON cc.id = c.category_id AND c.is_active = 'yes'
            WHERE cc.is_active = 'yes'
            GROUP BY cc.id, cc.category_name
            ORDER BY cc.category_name
        ")->result();
    }

    /**
     * Get expiring items report
     */
    public function getExpiringItemsReport($days_ahead = 90)
    {
        return $this->db->query("
            SELECT 
                c.consumable_name,
                cb.batch_number,
                cb.expiry_date,
                cb.quantity_remaining,
                DATEDIFF(cb.expiry_date, CURRENT_DATE) as days_to_expiry,
                CASE 
                    WHEN cb.expiry_date < CURRENT_DATE THEN 'Expired'
                    WHEN DATEDIFF(cb.expiry_date, CURRENT_DATE) <= 30 THEN 'Critical'
                    WHEN DATEDIFF(cb.expiry_date, CURRENT_DATE) <= 90 THEN 'Warning'
                    ELSE 'Monitor'
                END as expiry_status,
                (cb.quantity_remaining * c.unit_cost) as value_at_risk
            FROM consumable_batches cb
            JOIN consumables c ON cb.consumable_id = c.id
            WHERE cb.quantity_remaining > 0 
            AND cb.quality_status = 'approved'
            AND DATEDIFF(cb.expiry_date, CURRENT_DATE) <= ?
            ORDER BY cb.expiry_date ASC
        ", [$days_ahead])->result();
    }

    /**
     * Get departments list
     */
    public function getDepartments($active_only = true)
    {
        $this->db->select('*');
        $this->db->from('departments');
        
        if ($active_only) {
            $this->db->where('is_active', 'yes');
        }
        
        $this->db->order_by('department_name', 'ASC');
        return $this->db->get()->result();
    }

    /**
     * Get suppliers list
     */
    public function getSuppliers($active_only = true)
    {
        $this->db->select('*');
        $this->db->from('suppliers');
        
        if ($active_only) {
            $this->db->where('is_active', 'yes');
        }
        
        $this->db->order_by('supplier_name', 'ASC');
        return $this->db->get()->result();
    }
}

?>